/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  skunique.c
**
**    This is an attempt to make the bulk of rwuniq into a stand-alone
**    library.
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skunique.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/rwascii.h>
#include <silk/hashlib.h>
#include <silk/iptree.h>
#include <silk/sktempfile.h>
#include "skunique.h"


#define HASH_MAX_NODE_BYTES  (HASHLIB_MAX_KEY_WIDTH + HASHLIB_MAX_VALUE_WIDTH)

#define HASH_INITIAL_SIZE    500000

#define MAX_MERGE_FILES 1024



typedef enum {
    /* compute the distinct count by comparing hash keys */
    DISTINCT_COUNT,
    /* compute the distinct count by comparing the current value with
     * the previous value.  requires that we cache the previous value.
     * DISTINCT_CACHE says the value is stored inline;
     * DISTINCT_CACHE_ALLOC says a separate buffer is used to cache
     * the value.  */
    DISTINCT_CACHE,
    DISTINCT_CACHE_ALLOC,
    /* compute the dintinct count by keeping track of each value we
     * see.  DISTINCT_BITMAP is used for values up to 16bits;
     * DISTINCT_IPTREE is used for values up to 32 bits,
     * DISTINCT_HASHSET is used for larger values.
     * DISTINCT_HASHSET_FILES indicates file backing for the
     * hashset. */
    DISTINCT_BITMAP,
    DISTINCT_IPTREE,
    DISTINCT_HASHSET,
    DISTINCT_HASHSET_FILES
} distinct_type_t;




/* typedef struct sk_fieldentry_st sk_fieldentry_t; */
struct sk_fieldentry_st {
    sk_fieldlist_rec_to_bin_fn_t    rec_to_bin;
    sk_fieldlist_bin_cmp_fn_t       bin_compare;
    sk_fieldlist_rec_to_bin_fn_t    add_rec_to_bin;
    sk_fieldlist_bin_merge_fn_t     bin_merge;
    sk_fieldlist_output_fn_t        bin_output;

    int                             id;

    /* the byte-offset where this field begins in the binary key used
     * for binning. */
    size_t                          offset;
    size_t                          octets;
    void                           *context;

    uint8_t                        *initial_value;

    sk_fieldlist_t                 *parent_list;
};


/* typedef struct sk_fieldlist_st struct sk_fieldlist_t; */
struct sk_fieldlist_st {
    sk_fieldentry_t   *fields;
    size_t             capacity;
    size_t             num_fields;
    size_t             total_octets;
};



/* LOCAL FUNCTION DECLARATIONS */


static int uniqueDumpHashToTemp(
    sk_unique_t            *uniq);
static int uniqueCreateHashTable(
    sk_unique_t            *uniq);


/* FUNCTION DEFINITIONS */


/* ************************************************************ */
/* ************************************************************ */


#define COMPARE(cmp_a, cmp_b)                                   \
    (((cmp_a) < (cmp_b)) ? -1 : (((cmp_a) > (cmp_b)) ? 1 : 0))

#define WARN_OVERFLOW(wo_max, wo_a, wo_b)                       \
    if (wo_max - wo_b >= wo_a) { /* ok */ } else {              \
        skAppPrintErr("Overflow at %s:%d", __FILE__, __LINE__); \
    }

#if !SKUNIQ_USE_MEMCPY

#define CMP_INT_PTRS(cmp_out, cmp_type, cmp_a, cmp_b)                   \
    {                                                                   \
        cmp_out = COMPARE(*(cmp_type *)(cmp_a), *(cmp_type *)(cmp_b));  \
    }

#define MERGE_INT_PTRS(mrg_max, mrg_type, mrg_a, mrg_b)                 \
    {                                                                   \
        WARN_OVERFLOW(mrg_max, *(mrg_type*)(mrg_a), *(mrg_type*)(mrg_b)); \
        *(mrg_type*)(mrg_a) += *(mrg_type*)(mrg_b);                     \
    }

#define ADD_TO_INT_PTR(mrg_type, mrg_ptr, mrg_val)      \
    {                                                   \
        *((mrg_type*)(mrg_ptr)) += mrg_val;             \
    }

#else

#define CMP_INT_PTRS(cmp_out, cmp_type, cmp_a, cmp_b)   \
    {                                                   \
        cmp_type cip_val_a;                             \
        cmp_type cip_val_b;                             \
                                                        \
        memcpy(&cip_val_a, (cmp_a), sizeof(cmp_type));  \
        memcpy(&cip_val_b, (cmp_b), sizeof(cmp_type));  \
                                                        \
        cmp_out = COMPARE(cip_val_a, cip_val_b);        \
    }

#define MERGE_INT_PTRS(mrg_max, mrg_type, mrg_a, mrg_b) \
    {                                                   \
        mrg_type mip_val_a;                             \
        mrg_type mip_val_b;                             \
                                                        \
        memcpy(&mip_val_a, (mrg_a), sizeof(mrg_type));  \
        memcpy(&mip_val_b, (mrg_b), sizeof(mrg_type));  \
                                                        \
        WARN_OVERFLOW(mrg_max, mip_val_a, mip_val_b);   \
        mip_val_a += mip_val_b;                         \
        memcpy((mrg_a), &mip_val_a, sizeof(mrg_type));  \
    }

#define ADD_TO_INT_PTR(mrg_type, mrg_ptr, mrg_val)              \
    {                                                           \
        mrg_type atip_val_a;                                    \
                                                                \
        memcpy(&atip_val_a, (mrg_ptr), sizeof(mrg_type));       \
        atip_val_a += mrg_val;                                  \
        memcpy((mrg_ptr), &atip_val_a, sizeof(mrg_type));       \
    }

#endif  /* SKUNIQ_USE_MEMCPY */



int skFieldCompareMemcmp(const void *a, const void *b, void *len)
{
    /* FIXME.  size_t or unit32_t? */
    return memcmp(a, b, *(size_t*)len);
}


int skFieldCompareUint8(const void *a, const void *b, void UNUSED(*ctx))
{
    return COMPARE(*(uint8_t*)a, *(uint8_t*)b);
}

void skFieldMergeUint8(void *a, const void *b, void UNUSED(*ctx))
{
    WARN_OVERFLOW(UINT8_MAX, *(uint8_t*)a, *(uint8_t*)b);
    *(uint8_t*)a += *(uint8_t*)b;
}


int skFieldCompareUint16(const void *a, const void *b, void UNUSED(*ctx))
{
    int rv;
    CMP_INT_PTRS(rv, uint16_t, a, b);
    return rv;
}

void skFieldMergeUint16(void *a, const void *b, void UNUSED(*ctx))
{
    MERGE_INT_PTRS(UINT16_MAX, uint16_t, a, b);
}


int skFieldCompareUint32(const void *a, const void *b, void UNUSED(*ctx))
{
    int rv;
    CMP_INT_PTRS(rv, uint32_t, a, b);
    return rv;
}

void skFieldMergeUint32(void *a, const void *b, void UNUSED(*ctx))
{
    MERGE_INT_PTRS(UINT32_MAX, uint32_t, a, b);
}


int skFieldCompareUint64(const void *a, const void *b, void UNUSED(*ctx))
{
    int rv;
    CMP_INT_PTRS(rv, uint64_t, a, b);
    return rv;
}

void skFieldMergeUint64(void *a, const void *b, void UNUSED(*ctx))
{
    MERGE_INT_PTRS(UINT64_MAX, uint64_t, a, b);
}



/* create a new field list */
int skFieldListCreate(
    sk_fieldlist_t                **field_list)
{
    sk_fieldlist_t *fl;

    fl = calloc(1, sizeof(sk_fieldlist_t));
    if (NULL == fl) {
        return -1;
    }

    /* FIXME.  etc */

    *field_list = fl;
    return 0;
}


/* destroy a field list */
void skFieldListDestroy(
    sk_fieldlist_t                **field_list)
{
    sk_fieldlist_t *fl;

    if (NULL == field_list || NULL == *field_list) {
        return;
    }

    fl = *field_list;
    *field_list = NULL;

    free(fl->fields);
    free(fl);
}


static int fieldlistGrow(
    sk_fieldlist_t                *field_list)
{
    size_t new_size = field_list->capacity + 8;
    void *old_list = field_list->fields;

    field_list->fields = realloc(old_list, new_size * sizeof(sk_fieldentry_t));
    if (field_list->fields) {
        field_list->capacity = new_size;
    } else {
        field_list->fields = old_list;
        return -1;
    }
    return 0;
}


/* add a field to a field list */
sk_fieldentry_t *skFieldListAddField(
    sk_fieldlist_t                 *field_list,
    const sk_fieldlist_entrydata_t *regdata,
    void                           *ctx)
{
    sk_fieldentry_t *field = NULL;
    size_t i;

    /* FIXME. add error checking */

    if (field_list->capacity == field_list->num_fields) {
        if (fieldlistGrow(field_list)) {
            return NULL;
        }
    }
    field = &field_list->fields[field_list->num_fields];
    ++field_list->num_fields;

    memset(field, 0, sizeof(sk_fieldentry_t));
    field->offset = field_list->total_octets;
    field->context = ctx;
    field->parent_list = field_list;
    field->id = SK_FIELD_CALLER;

    field->octets = regdata->bin_octets;
    field->rec_to_bin = regdata->rec_to_bin;
    field->bin_compare = regdata->bin_compare;
    field->add_rec_to_bin = regdata->add_rec_to_bin;
    field->bin_merge = regdata->bin_merge;
    field->bin_output = regdata->bin_output;
    if (regdata->initial_value) {
        /* only create space for value if it contains non-NUL */
        for (i = 0; i < field->octets; ++i) {
            if ('\0' != regdata->initial_value[i]) {
                field->initial_value = malloc(field->octets);
                if (NULL == field->initial_value) {
                    --field_list->num_fields;
                    return NULL;
                }
                memcpy(field->initial_value, regdata->initial_value,
                       field->octets);
                break;
            }
        }
    }

    field_list->total_octets += field->octets;

    return field;
}


sk_fieldentry_t *skFieldListAddKnownField(
    sk_fieldlist_t                 *field_list,
    int                             field_id,
    void                           *ctx)
{
    sk_fieldentry_t *field = NULL;
    int bin_octets = 0;

    /* FIXME.  add error checking */

    switch (field_id) {
      case SK_FIELD_SIPv4:
      case SK_FIELD_DIPv4:
      case SK_FIELD_NHIPv4:
      case SK_FIELD_PACKETS:
      case SK_FIELD_BYTES:
      case SK_FIELD_STARTTIME:
      case SK_FIELD_STARTTIME_MSEC:
      case SK_FIELD_ELAPSED:
      case SK_FIELD_ELAPSED_MSEC:
      case SK_FIELD_ENDTIME:
      case SK_FIELD_ENDTIME_MSEC:
      case SK_FIELD_RECORDS:
      case SK_FIELD_SUM_PACKETS:
      case SK_FIELD_SUM_ELAPSED:
      case SK_FIELD_MIN_STARTTIME:
      case SK_FIELD_MAX_ENDTIME:
        bin_octets = 4;
        break;

      case SK_FIELD_SPORT:
      case SK_FIELD_DPORT:
      case SK_FIELD_SID:
      case SK_FIELD_INPUT:
      case SK_FIELD_OUTPUT:
      case SK_FIELD_APPLICATION:
      case SK_FIELD_ICMP_TYPE_CODE:
        bin_octets = 2;
        break;

      case SK_FIELD_PROTO:
      case SK_FIELD_FLAGS:
      case SK_FIELD_INIT_FLAGS:
      case SK_FIELD_REST_FLAGS:
      case SK_FIELD_TCP_STATE:
      case SK_FIELD_FTYPE_CLASS:
      case SK_FIELD_FTYPE_TYPE:
        bin_octets = 1;
        break;

      case SK_FIELD_SUM_BYTES:
        bin_octets = 8;
        break;

      case SK_FIELD_SIPv6:
      case SK_FIELD_DIPv6:
      case SK_FIELD_NHIPv6:
        bin_octets = 16;
        break;

      case SK_FIELD_CALLER:
        break;
    }

    if (bin_octets == 0) {
        skAppPrintErr("Unknown field id %d", field_id);
        return NULL;
    }

    /* make space for the new field */
    if (field_list->capacity == field_list->num_fields) {
        if (fieldlistGrow(field_list)) {
            return NULL;
        }
    }

    field = &field_list->fields[field_list->num_fields];
    ++field_list->num_fields;

    memset(field, 0, sizeof(sk_fieldentry_t));
    field->offset = field_list->total_octets;
    field->octets = bin_octets;
    field->parent_list = field_list;
    field->id = field_id;
    field->context = ctx;

    field_list->total_octets += bin_octets;

    return field;
}


void *skFieldListEntryGetContext(const sk_fieldentry_t *field)
{
    assert(field);
    return field->context;
}


uint32_t skFieldListEntryGetId(const sk_fieldentry_t *field)
{
    assert(field);
    return field->id;
}


size_t skFieldListEntryGetBinOctets(const sk_fieldentry_t *field)
{
    assert(field);
    return field->octets;
}


/* returns (binary) size of all fields */
size_t skFieldListGetBufferSize(
    const sk_fieldlist_t           *field_list)
{
    assert(field_list);
    return field_list->total_octets;
}


size_t skFieldListGetFieldCount(
    const sk_fieldlist_t           *field_list)
{
    assert(field_list);
    return field_list->num_fields;
}


#define FIELD_PTR(all_fields_buffer, flent)     \
    ((all_fields_buffer) + (flent)->offset)

#if !SKUNIQ_USE_MEMCPY

#define REC_TO_KEY_SZ(rtk_type, rtk_val, rtk_buf, rtk_flent)    \
    { *((rtk_type*)FIELD_PTR(rtk_buf, rtk_flent)) = rtk_val; }

#else

#define REC_TO_KEY_SZ(rtk_type, rtk_val, rtk_buf, rtk_flent)    \
    {                                                           \
        rtk_type rtk_tmp = rtk_val;                             \
        memcpy(FIELD_PTR(rtk_buf, rtk_flent), &rtk_tmp,         \
               sizeof(rtk_type));                               \
    }

#endif


#define REC_TO_KEY_64(val, all_fields_buffer, flent)            \
    REC_TO_KEY_SZ(uint64_t, val, all_fields_buffer, flent)

#define REC_TO_KEY_32(val, all_fields_buffer, flent)              \
    REC_TO_KEY_SZ(uint32_t, val, all_fields_buffer, flent)

#define REC_TO_KEY_16(val, all_fields_buffer, flent)              \
    REC_TO_KEY_SZ(uint16_t, val, all_fields_buffer, flent)

#define REC_TO_KEY_08(val, all_fields_buffer, flent)      \
    { *(FIELD_PTR(all_fields_buffer, flent)) = val; }



/* gets the binary value for each field in 'field_list' and sets that
 * value in 'all_fields_buffer' */
void skFieldListRecToBinary(
    const sk_fieldlist_t           *field_list,
    const rwRec                    *rwrec,
    uint8_t                        *bin_buffer)
{
    sk_fieldentry_t *f;
    size_t i;

    for (i = 0, f = field_list->fields; i < field_list->num_fields; ++i, ++f) {
        if (f->rec_to_bin) {
            f->rec_to_bin(rwrec, FIELD_PTR(bin_buffer, f), f->context);
        } else {
            switch (f->id) {
#if SK_ENABLE_IPV6
              case SK_FIELD_SIPv6:
                rwRecMemGetSIPv6(rwrec, FIELD_PTR(bin_buffer, f));
                break;
              case SK_FIELD_DIPv6:
                rwRecMemGetDIPv6(rwrec, FIELD_PTR(bin_buffer, f));
                break;
              case SK_FIELD_NHIPv6:
                rwRecMemGetNhIPv6(rwrec, FIELD_PTR(bin_buffer, f));
                break;
#endif  /* SK_ENABLE_IPV6 */

              case SK_FIELD_SIPv4:
                REC_TO_KEY_32(rwRecGetSIPv4(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_DIPv4:
                REC_TO_KEY_32(rwRecGetDIPv4(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_NHIPv4:
                REC_TO_KEY_32(rwRecGetNhIPv4(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_SPORT:
                REC_TO_KEY_16(rwRecGetSPort(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_DPORT:
              case SK_FIELD_ICMP_TYPE_CODE:
                REC_TO_KEY_16(rwRecGetDPort(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_PROTO:
                REC_TO_KEY_08(rwRecGetProto(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_PACKETS:
                REC_TO_KEY_32(rwRecGetPkts(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_BYTES:
                REC_TO_KEY_32(rwRecGetBytes(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_FLAGS:
                REC_TO_KEY_08(rwRecGetFlags(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_SID:
                REC_TO_KEY_16(rwRecGetSensor(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_INPUT:
                REC_TO_KEY_16(rwRecGetInput(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_OUTPUT:
                REC_TO_KEY_16(rwRecGetOutput(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_INIT_FLAGS:
                REC_TO_KEY_08(rwRecGetInitFlags(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_REST_FLAGS:
                REC_TO_KEY_08(rwRecGetRestFlags(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_TCP_STATE:
                REC_TO_KEY_08(rwRecGetTcpState(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_APPLICATION:
                REC_TO_KEY_16(rwRecGetApplication(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_FTYPE_CLASS:
              case SK_FIELD_FTYPE_TYPE:
                REC_TO_KEY_08(rwRecGetFlowType(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_STARTTIME:
              case SK_FIELD_STARTTIME_MSEC:
                REC_TO_KEY_32(rwRecGetStartSeconds(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_ELAPSED:
              case SK_FIELD_ELAPSED_MSEC:
                REC_TO_KEY_32(rwRecGetElapsedSeconds(rwrec), bin_buffer, f);
                break;
              case SK_FIELD_ENDTIME:
              case SK_FIELD_ENDTIME_MSEC:
                REC_TO_KEY_32(rwRecGetEndSeconds(rwrec), bin_buffer, f);
                break;
              default:
                break;
            }
        }
    }
}


/* adds the binary value for each field in 'field_list' to the values
 * in 'all_fields_buffer' */
void skFieldListAddRecToBuffer(
    const sk_fieldlist_t           *field_list,
    const rwRec                    *rwrec,
    uint8_t                        *summed)
{
#if !SKUNIQ_USE_MEMCPY
    uint32_t *val_ptr;
#else
    uint32_t val_a;
#endif
    sk_fieldentry_t *f;
    size_t i;

    for (i = 0, f = field_list->fields; i < field_list->num_fields; ++i, ++f) {
        if (f->add_rec_to_bin) {
            f->add_rec_to_bin(rwrec, FIELD_PTR(summed, f), f->context);
        } else {
            switch (f->id) {
              case SK_FIELD_RECORDS:
                ADD_TO_INT_PTR(uint32_t, FIELD_PTR(summed, f), 1);
                break;

              case SK_FIELD_SUM_BYTES:
                ADD_TO_INT_PTR(uint64_t, FIELD_PTR(summed, f),
                               rwRecGetBytes(rwrec));
                break;

              case SK_FIELD_SUM_PACKETS:
                ADD_TO_INT_PTR(uint32_t, FIELD_PTR(summed, f),
                               rwRecGetPkts(rwrec));
                break;

              case SK_FIELD_SUM_ELAPSED:
                ADD_TO_INT_PTR(uint32_t, FIELD_PTR(summed, f),
                               rwRecGetElapsedSeconds(rwrec));
                break;

#if !SKUNIQ_USE_MEMCPY
              case SK_FIELD_MIN_STARTTIME:
                val_ptr = (uint32_t*)FIELD_PTR(summed, f);
                if (rwRecGetStartSeconds(rwrec) < *val_ptr) {
                    *val_ptr = rwRecGetStartSeconds(rwrec);
                }
                break;

              case SK_FIELD_MAX_ENDTIME:
                val_ptr = (uint32_t*)FIELD_PTR(summed, f);
                if (rwRecGetEndSeconds(rwrec) > *val_ptr) {
                    *val_ptr = rwRecGetEndSeconds(rwrec);
                }
                break;

#else  /* SKUNIQ_USE_MEMCPY */
              case SK_FIELD_MIN_STARTTIME:
                memcpy(&val_a, FIELD_PTR(summed, f), f->octets);
                if (rwRecGetStartSeconds(rwrec) < val_a) {
                    val_a = rwRecGetStartSeconds(rwrec);
                    memcpy(FIELD_PTR(summed, f), &val_a, f->octets);
                }
                break;

              case SK_FIELD_MAX_ENDTIME:
                memcpy(&val_a, FIELD_PTR(summed, f), f->octets);
                if (rwRecGetEndSeconds(rwrec) > val_a) {
                    val_a = rwRecGetEndSeconds(rwrec);
                    memcpy(FIELD_PTR(summed, f), &val_a, f->octets);
                }
                break;
#endif  /* SKUNIQ_USE_MEMCPY */

              case SK_FIELD_CALLER:
                break;

              default:
                break;
            }
        }
    }
}


/* sets 'all_fields_buffer' to the initial value for each field in the
 * field list. */
void skFieldListInitializeBuffer(
    const sk_fieldlist_t           *field_list,
    uint8_t                        *all_fields_buffer)
{
    sk_fieldentry_t *f;
    size_t i;

    memset(all_fields_buffer, 0, field_list->total_octets);
    for (i = 0, f = field_list->fields; i < field_list->num_fields; ++i, ++f) {
        if (f->initial_value) {
            memcpy(FIELD_PTR(all_fields_buffer, f), f->initial_value,
                   f->octets);
        } else {
            switch (f->id) {
              case SK_FIELD_MIN_STARTTIME:
                memset(FIELD_PTR(all_fields_buffer, f), 0xFF, f->octets);
                break;
              default:
                break;
            }
        }
    }
}


/* merges (e.g., add) two buffers for a field list */
void skFieldListMergeBuffers(
    const sk_fieldlist_t           *field_list,
    uint8_t                        *all_fields_buffer1,
    const uint8_t                  *all_fields_buffer2)
{
#if !SKUNIQ_USE_MEMCPY
    uint32_t *a_ptr, *b_ptr;
#else
    uint32_t val_a, val_b;
#endif
    sk_fieldentry_t *f;
    size_t i;

    for (i = 0, f = field_list->fields; i < field_list->num_fields; ++i, ++f) {
        if (f->bin_merge) {
            f->bin_merge(FIELD_PTR(all_fields_buffer1, f),
                         FIELD_PTR(all_fields_buffer2, f),
                         f->context);
        } else {
            switch (f->id) {
              case SK_FIELD_RECORDS:
              case SK_FIELD_SUM_PACKETS:
              case SK_FIELD_SUM_ELAPSED:
                MERGE_INT_PTRS(UINT32_MAX, uint32_t,
                               FIELD_PTR(all_fields_buffer1, f),
                               FIELD_PTR(all_fields_buffer2, f));
                break;

              case SK_FIELD_SUM_BYTES:
                MERGE_INT_PTRS(UINT64_MAX, uint64_t,
                               FIELD_PTR(all_fields_buffer1, f),
                               FIELD_PTR(all_fields_buffer2, f));
                break;

#if !SKUNIQ_USE_MEMCPY
              case SK_FIELD_MIN_STARTTIME:
                /* put smallest value into a */
                a_ptr = (uint32_t*)FIELD_PTR(all_fields_buffer1, f);
                b_ptr = (uint32_t*)FIELD_PTR(all_fields_buffer2, f);
                if (*b_ptr < *a_ptr) {
                    *a_ptr = *b_ptr;
                }
                break;

              case SK_FIELD_MAX_ENDTIME:
                /* put largest value into a */
                a_ptr = (uint32_t*)FIELD_PTR(all_fields_buffer1, f);
                b_ptr = (uint32_t*)FIELD_PTR(all_fields_buffer2, f);
                if (*b_ptr > *a_ptr) {
                    *a_ptr = *b_ptr;
                }
                break;

#else  /* SKUNIQ_USE_MEMCPY */
              case SK_FIELD_MIN_STARTTIME:
                memcpy(&val_a, FIELD_PTR(all_fields_buffer1, f), f->octets);
                memcpy(&val_b, FIELD_PTR(all_fields_buffer2, f), f->octets);
                if (val_b < val_a) {
                    val_a = val_b;
                    memcpy(FIELD_PTR(all_fields_buffer1, f), &val_a, f->octets);
                }
                break;

              case SK_FIELD_MAX_ENDTIME:
                memcpy(&val_a, FIELD_PTR(all_fields_buffer1, f), f->octets);
                memcpy(&val_b, FIELD_PTR(all_fields_buffer2, f), f->octets);
                if (val_b > val_a) {
                    val_a = val_b;
                    memcpy(FIELD_PTR(all_fields_buffer1, f), &val_a, f->octets);
                }
                break;
#endif  /* SKUNIQ_USE_MEMCPY */

              default:
                break;
            }
        }
    }
}


/* compare two field buffers, return -1, 0, 1, if 'all_fields_buffer1'
 * is <, ==, > 'all_fields_buffer2' */
int skFieldListCompareBuffers(
    const sk_fieldlist_t           *field_list,
    const uint8_t                  *all_fields_buffer1,
    const uint8_t                  *all_fields_buffer2)
{
    sk_fieldentry_t *f;
    size_t i;
    int rv = 0;

    for (i = 0, f = field_list->fields;
         rv == 0 && i < field_list->num_fields;
         ++i, ++f)
    {
        if (f->bin_compare) {
            rv = f->bin_compare(&all_fields_buffer1[f->offset],
                                &all_fields_buffer2[f->offset],
                                f->context);
        } else {
            switch (f->id) {
              case SK_FIELD_SIPv6:
              case SK_FIELD_DIPv6:
              case SK_FIELD_NHIPv6:
                rv = memcmp(FIELD_PTR(all_fields_buffer1, f),
                            FIELD_PTR(all_fields_buffer2, f),
                            f->octets);
                break;

              case SK_FIELD_SIPv4:
              case SK_FIELD_DIPv4:
              case SK_FIELD_NHIPv4:
              case SK_FIELD_PACKETS:
              case SK_FIELD_BYTES:
              case SK_FIELD_STARTTIME:
              case SK_FIELD_STARTTIME_MSEC:
              case SK_FIELD_ELAPSED:
              case SK_FIELD_ELAPSED_MSEC:
              case SK_FIELD_ENDTIME:
              case SK_FIELD_ENDTIME_MSEC:
              case SK_FIELD_RECORDS:
              case SK_FIELD_SUM_PACKETS:
              case SK_FIELD_SUM_ELAPSED:
              case SK_FIELD_MIN_STARTTIME:
              case SK_FIELD_MAX_ENDTIME:
                CMP_INT_PTRS(rv, uint32_t, FIELD_PTR(all_fields_buffer1, f),
                             FIELD_PTR(all_fields_buffer2, f));
                break;

              case SK_FIELD_SPORT:
              case SK_FIELD_DPORT:
              case SK_FIELD_SID:
              case SK_FIELD_INPUT:
              case SK_FIELD_OUTPUT:
              case SK_FIELD_APPLICATION:
              case SK_FIELD_ICMP_TYPE_CODE:
                CMP_INT_PTRS(rv, uint16_t, FIELD_PTR(all_fields_buffer1, f),
                             FIELD_PTR(all_fields_buffer2, f));
                break;

              case SK_FIELD_PROTO:
              case SK_FIELD_FLAGS:
              case SK_FIELD_INIT_FLAGS:
              case SK_FIELD_REST_FLAGS:
              case SK_FIELD_TCP_STATE:
              case SK_FIELD_FTYPE_CLASS:
              case SK_FIELD_FTYPE_TYPE:
                rv = COMPARE(*(FIELD_PTR(all_fields_buffer1, f)),
                             *(FIELD_PTR(all_fields_buffer2, f)));
                break;

              case SK_FIELD_SUM_BYTES:
                CMP_INT_PTRS(rv, uint64_t, FIELD_PTR(all_fields_buffer1, f),
                             FIELD_PTR(all_fields_buffer2, f));
                break;

              default:
                break;
            }
        }
    }


    return rv;
}





/* calls the output callback for each field */
void skFieldListOutputBuffer(
    const sk_fieldlist_t           *field_list,
    const uint8_t                  *all_fields_buffer);


/* Do we still need the field iterators (as public)? */

/* bind an iterator to a field list */
void skFieldListIteratorBind(
    const sk_fieldlist_t           *field_list,
    sk_fieldlist_iterator_t        *iter)
{
    assert(field_list);
    assert(iter);

    memset(iter, 0, sizeof(sk_fieldlist_iterator_t));
    iter->field_list = field_list;
    iter->field_idx = 0;
}

/* reset the fieldlist iterator */
void skFieldListIteratorReset(
    sk_fieldlist_iterator_t        *iter)
{
    assert(iter);
    iter->field_idx = 0;
}

sk_fieldentry_t *skFieldListIteratorNext(
    sk_fieldlist_iterator_t        *iter)
{
    sk_fieldentry_t *f = NULL;

    assert(iter);
    if (iter->field_idx < iter->field_list->num_fields) {
        f = &iter->field_list->fields[iter->field_idx];
        ++iter->field_idx;
    }
    return f;
}


/* copy the value associated with 'field_id' from 'all_fields_buffer'
 * and into 'one_field_buf' */
void skFieldListExtractFromBuffer(
    const sk_fieldlist_t    UNUSED(*field_list),
    const uint8_t                  *all_fields_buffer,
    sk_fieldentry_t                *field_id,
    uint8_t                        *one_field_buf)
{
    assert(field_id->parent_list == field_list);
    memcpy(one_field_buf, FIELD_PTR(all_fields_buffer, field_id),
           field_id->octets);
}



/* ************************************************************ */


/* LOCAL DEFINES AND TYPEDEFS */

typedef struct HashSet_st {
    HashTable   *table;
    uint8_t      is_sorted;
    uint8_t      key_width;
    uint8_t      mod_key;
} HashSet;

typedef struct hashset_iter {
    HASH_ITER    table_iter;
    uint8_t      key[HASHLIB_MAX_KEY_WIDTH];
    uint8_t      val;
} hashset_iter;


/* LOCAL VARIABLE DEFINITIONS */

/* position of least significant bit, as in 1<<N */
static const uint8_t lowest_bit_in_val[] = {
    /*   0- 15 */  8, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /*  16- 31 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /*  32- 47 */  5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /*  48- 63 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /*  64- 79 */  6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /*  80- 95 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /*  96-111 */  5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 112-127 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 128-143 */  7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 144-159 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 160-175 */  5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 176-191 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 192-207 */  6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 208-223 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 224-239 */  5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 240-255 */  4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0
};

/* number of high bits in each value */
static const uint8_t bits_in_value[] = {
    /*   0- 15 */  0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
    /*  16- 31 */  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    /*  32- 47 */  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    /*  48- 63 */  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    /*  64- 79 */  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    /*  80- 95 */  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    /*  96-111 */  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    /* 112-127 */  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    /* 128-143 */  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
    /* 144-159 */  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    /* 160-175 */  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    /* 176-191 */  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    /* 192-207 */  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
    /* 208-223 */  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    /* 224-239 */  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
    /* 240-255 */  4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8
};



static HashSet *hashset_create_set(
    uint8_t             key_width,
    uint32_t            estimated_count,
    uint8_t             load_factor)
{
    uint8_t no_value = 0;
    HashSet *hash_set;

    hash_set = calloc(1, sizeof(HashSet));
    if (NULL == hash_set) {
        return NULL;
    }
    hash_set->key_width = key_width;
    hash_set->mod_key = key_width - 1;
    hash_set->table = hashlib_create_table(key_width, 1, HTT_INPLACE,
                                           &no_value, NULL, 0,
                                           estimated_count, load_factor);
    if (hash_set->table == NULL) {
        free(hash_set);
        return NULL;
    }
    return hash_set;
}


static int hashset_insert(
    HashSet            *set_ptr,
    const uint8_t      *key_ptr)
{
    uint8_t tmp_key[HASHLIB_MAX_KEY_WIDTH];
    uint8_t *value_ptr;
    uint8_t bit;
    int rv;

    /* make a new key, masking off the lowest three bits */
    memcpy(tmp_key, key_ptr, set_ptr->key_width);
    tmp_key[set_ptr->mod_key] &= 0xF8;

    /* determine which bit to check/set */
    bit = 1 << (key_ptr[set_ptr->mod_key] & 0x7);

    rv = hashlib_insert(set_ptr->table, tmp_key, &value_ptr);
    switch (rv) {
      case OK_DUPLICATE:
        if (0 == (*value_ptr & bit)) {
            rv = OK;
        }
        /* FALLTHROUGH */
      case OK:
        *value_ptr |= bit;
        break;
    }
    return rv;
}


#if 0                           /* currently unused; #if 0 to avoid warning */
static int hashset_lookup(
    const HashSet      *set_ptr,
    const uint8_t      *key_ptr)
{
    uint8_t tmp_key[HASHLIB_MAX_KEY_WIDTH];
    uint8_t *value_ptr;
    uint8_t bit;
    int rv;

    /* make a new key, masking off the lowest three bits */
    memcpy(tmp_key, key_ptr, set_ptr->key_width);
    tmp_key[set_ptr->mod_key] &= 0xF8;

    /* determine which bit to check/set */
    bit = 1 << (key_ptr[set_ptr->mod_key] & 0x7);

    rv = hashlib_lookup(set_ptr->table, tmp_key, &value_ptr);
    if (rv == OK && (*value_ptr & bit)) {
        return OK;
    }
    return ERR_NOTFOUND;
}
#endif  /* 0 */


static hashset_iter hashset_create_iterator(const HashSet   *set_ptr)
{
    hashset_iter iter;

    memset(&iter, 0, sizeof(iter));
    iter.table_iter = hashlib_create_iterator(set_ptr->table);
    return iter;
}

#if 0
static int hashset_sort_entries(
    HashSet        *set_ptr)
{
    set_ptr->is_sorted = 1;
    return hashlib_sort_entries(set_ptr->table);
}
#endif

static int hashset_iterate(
    const HashSet      *set_ptr,
    hashset_iter       *iter,
    uint8_t           **key_pptr)
{
    uint8_t *hash_key;
    uint8_t *hash_value;
    int rv;

    if (iter->val == 0) {
        /* need to get a key/value pair */
        rv = hashlib_iterate(set_ptr->table, &iter->table_iter,
                             &hash_key, &hash_value);
        if (rv != OK) {
            return rv;
        }
        memcpy(&iter->key, hash_key, set_ptr->key_width);
        iter->val = hash_value[0];
    }

    switch (lowest_bit_in_val[iter->val]) {
      case 0:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8);
        iter->val &= 0xFE;
        break;
      case 1:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8) | 1;
        iter->val &= 0xFD;
        break;
      case 2:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8) | 2;
        iter->val &= 0xFB;
        break;
      case 3:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8) | 3;
        iter->val &= 0xF7;
        break;
      case 4:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8) | 4;
        iter->val &= 0xEF;
        break;
      case 5:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8) | 5;
        iter->val &= 0xDF;
        break;
      case 6:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8) | 6;
        iter->val &= 0xBF;
        break;
      case 7:
        iter->key[set_ptr->mod_key] = (iter->key[set_ptr->mod_key] & 0xF8) | 7;
        iter->val &= 0x7F;
      default:
        skAbortBadCase(lowest_bit_in_val[iter->val]);
    }

    *key_pptr = iter->key;
    return OK;
}


static void hashset_free_table(HashSet *set_ptr)
{
    if (set_ptr) {
        if (set_ptr->table) {
            hashlib_free_table(set_ptr->table);
            set_ptr->table = NULL;
        }
        free(set_ptr);
    }
}


#if 0                           /* currently unused; #if 0 to avoid warning */
static uint32_t hashset_count_entries(const HashSet *set_ptr)
{
    HASH_ITER iter;
    uint8_t *key_ptr;
    uint8_t *val_ptr;
    uint32_t count = 0;

    iter = hashlib_create_iterator(set_ptr->table);

    while (hashlib_iterate(set_ptr->table, &iter, &key_ptr, &val_ptr)) {
        count += bits_in_value[*val_ptr];
    }

    return count;
}
#endif /* 0 */


/* ************************************************************ */


/* structure for field info; used by sk_unique_t and sk_sort_unique_t */
typedef struct sk_uniq_field_info_st {
    const sk_fieldlist_t   *key_fields;
    const sk_fieldlist_t   *value_fields;
    const sk_fieldlist_t   *distinct_fields;

    uint8_t                 key_num_fields;
    uint8_t                 key_octets;

    uint8_t                 value_num_fields;
    uint8_t                 value_octets;

    /* number of distinct fields */
    uint8_t                 distinct_num_fields;
    uint8_t                 distinct_octets;
} sk_uniq_field_info_t;



/* structure for binning records */

/* typedef struct sk_unique_st sk_unique_t; */
struct sk_unique_st {
    /* information about the fields */
    sk_uniq_field_info_t    fi;

    /* where to write temporary files */
    char                   *temp_dir;

    /* the hash table */
    HashTable              *ht;

    /* the byte length of the key-value pair in the hash table, which
     * is comprised of the key and the value */
    size_t                  hash_node_octets;

    /* the byte count of the key passed to the hash table, which is
     * the length of the "key" and any "distinct" fields  */
    uint8_t                 hash_key_octets;

    /* the byte length of the value used by the hash table, which has
     * a minimum length of 1  */
    uint8_t                 hash_value_octets;

    /* highest index of temporary file that has been used */
    int                     temp_file_idx;

    /* whether the output should be sorted */
    unsigned                sort_output :1;

    /* whether PrepareForInput()/PrepareForOutput() have been called */
    unsigned                ready_for_input:1;
    unsigned                ready_for_output:1;
};


#define KEY_ONLY            1
#define VALUE_ONLY          2
#define DISTINCT_ONLY       4
#define KEY_VALUE           (KEY_ONLY | VALUE_ONLY)
#define KEY_DISTINCT        (KEY_ONLY | DISTINCT_ONLY)
#define VALUE_DISTINCT      (VALUE_ONLY | DISTINCT_ONLY)
#define KEY_VALUE_DISTINCT  (KEY_ONLY | VALUE_ONLY | DISTINCT_ONLY)


static struct allowed_fieldid_st {
    sk_fieldid_t    fieldid;
    uint8_t         kvd;
} allowed_fieldid[] = {
    {SK_FIELD_SIPv4,            KEY_DISTINCT},
    {SK_FIELD_DIPv4,            KEY_DISTINCT},
    {SK_FIELD_SPORT,            KEY_DISTINCT},
    {SK_FIELD_DPORT,            KEY_DISTINCT},
    {SK_FIELD_PROTO,            KEY_DISTINCT},
    {SK_FIELD_PACKETS,          KEY_DISTINCT},
    {SK_FIELD_BYTES,            KEY_DISTINCT},
    {SK_FIELD_FLAGS,            KEY_DISTINCT},
    {SK_FIELD_STARTTIME,        KEY_DISTINCT},
    {SK_FIELD_ELAPSED,          KEY_DISTINCT},
    {SK_FIELD_ENDTIME,          KEY_DISTINCT},
    {SK_FIELD_SID,              KEY_DISTINCT},
    {SK_FIELD_INPUT,            KEY_DISTINCT},
    {SK_FIELD_OUTPUT,           KEY_DISTINCT},
    {SK_FIELD_NHIPv4,           KEY_DISTINCT},
    {SK_FIELD_INIT_FLAGS,       KEY_DISTINCT},
    {SK_FIELD_REST_FLAGS,       KEY_DISTINCT},
    {SK_FIELD_TCP_STATE,        KEY_DISTINCT},
    {SK_FIELD_APPLICATION,      KEY_DISTINCT},
    {SK_FIELD_FTYPE_CLASS,      KEY_DISTINCT},
    {SK_FIELD_FTYPE_TYPE,       KEY_DISTINCT},
    {SK_FIELD_STARTTIME_MSEC,   KEY_DISTINCT},
    {SK_FIELD_ENDTIME_MSEC,     KEY_DISTINCT},
    {SK_FIELD_ELAPSED_MSEC,     KEY_DISTINCT},
    {SK_FIELD_ICMP_TYPE_CODE,   KEY_DISTINCT},
    {SK_FIELD_SIPv6,            KEY_DISTINCT},
    {SK_FIELD_DIPv6,            KEY_DISTINCT},
    {SK_FIELD_NHIPv6,           KEY_DISTINCT},
    {SK_FIELD_RECORDS,          VALUE_ONLY},
    {SK_FIELD_SUM_PACKETS,      VALUE_ONLY},
    {SK_FIELD_SUM_BYTES,        VALUE_ONLY},
    {SK_FIELD_SUM_ELAPSED,      VALUE_ONLY},
    {SK_FIELD_MIN_STARTTIME,    VALUE_ONLY},
    {SK_FIELD_MAX_ENDTIME,      VALUE_ONLY},
    {SK_FIELD_CALLER,           KEY_VALUE}
};


static int uniqCheckFields(
    sk_uniq_field_info_t           *field_info,
    sk_msg_fn_t                     err_fn)
{
#define ERR_FN  if (!err_fn) { } else err_fn

#define SAFE_SET(variable, value)               \
    {                                           \
        size_t sz = (value);                    \
        if (sz > UINT8_MAX) {                   \
            ERR_FN("Overflow");                 \
            return -1;                          \
        }                                       \
        variable = (uint8_t)value;              \
    }

    sk_fieldlist_iterator_t fl_iter;
    sk_fieldlist_iterator_t fl_iter2;
    sk_fieldentry_t *field;
    sk_fieldentry_t *field2;
    size_t num_allowed;
    uint8_t field_type;
    uint32_t field_id;
    size_t i;

    assert(field_info);

    num_allowed = sizeof(allowed_fieldid)/sizeof(struct allowed_fieldid_st);

    /* must have at least one key field */
    if (NULL == field_info->key_fields) {
        ERR_FN("No key fields were specified");
        return -1;
    }
    /* must have at least one value or one distinct field */
    if (NULL == field_info->value_fields
        && NULL == field_info->distinct_fields)
    {
        ERR_FN("Neither value nor distinct fields were specified");
        return -1;
    }

    /* handle key fields */
    skFieldListIteratorBind(field_info->key_fields, &fl_iter);
    while (NULL != (field = skFieldListIteratorNext(&fl_iter))) {
        field_type = 0;
        field_id = skFieldListEntryGetId(field);
        for (i = 0; i < num_allowed; ++i) {
            if (field_id == allowed_fieldid[i].fieldid) {
                field_type = allowed_fieldid[i].kvd;
                break;
            }
        }
        if (field_type == 0) {
            ERR_FN("Unknown field %d", field->id);
            return -1;
        }
        if (!(field_type & KEY_ONLY)) {
            ERR_FN("Field %d is not allowed in the key", field->id);
            return -1;
        }
    }
    SAFE_SET(field_info->key_num_fields,
             skFieldListGetFieldCount(field_info->key_fields));
    SAFE_SET(field_info->key_octets,
             skFieldListGetBufferSize(field_info->key_fields));
    if (field_info->key_num_fields == 0 || field_info->key_octets == 0) {
        ERR_FN("No key fields were specified");
        return -1;
    }

    /* handle value fields */
    if (field_info->value_fields) {
        skFieldListIteratorBind(field_info->value_fields, &fl_iter);
        while (NULL != (field = skFieldListIteratorNext(&fl_iter))) {
            field_type = 0;
            field_id = skFieldListEntryGetId(field);
            for (i = 0; i < num_allowed; ++i) {
                if (field_id == allowed_fieldid[i].fieldid) {
                    field_type = allowed_fieldid[i].kvd;
                    break;
                }
            }
            if (field_type == 0) {
                ERR_FN("Unknown field %d", field->id);
                return -1;
            }
            if (!(field_type & VALUE_ONLY)) {
                ERR_FN("Field %d is not allowed in the value", field->id);
                return -1;
            }
        }

        SAFE_SET(field_info->value_num_fields,
                 skFieldListGetFieldCount(field_info->value_fields));
        SAFE_SET(field_info->value_octets,
                 skFieldListGetBufferSize(field_info->value_fields));
    }

    /* handle distinct fields */
    if (field_info->distinct_fields) {
        skFieldListIteratorBind(field_info->distinct_fields, &fl_iter);
        while (NULL != (field = skFieldListIteratorNext(&fl_iter))) {
            field_type = 0;
            field_id = skFieldListEntryGetId(field);
            for (i = 0; i < num_allowed; ++i) {
                if (field_id == allowed_fieldid[i].fieldid) {
                    field_type = allowed_fieldid[i].kvd;
                    break;
                }
            }
            if (field_type == 0) {
                ERR_FN("Unknown field %d", field->id);
                return -1;
            }
            if (!(field_type & DISTINCT_ONLY)) {
                ERR_FN("Field %d is not allowed in the distinct", field->id);
                return -1;
            }

            /* ensure distinct field is not part of key */
            skFieldListIteratorBind(field_info->key_fields, &fl_iter2);
            while (NULL != (field2 = skFieldListIteratorNext(&fl_iter2))) {
                if (skFieldListEntryGetId(field2) == field_id) {
                    ERR_FN("Will not count distinct"
                           " value that are also part of key");
                    return -1;
                }
            }
        }

        SAFE_SET(field_info->distinct_num_fields,
                 skFieldListGetFieldCount(field_info->distinct_fields));
        SAFE_SET(field_info->distinct_octets,
                 skFieldListGetBufferSize(field_info->distinct_fields));
    }

    /* ensure either values or distincts are specified */
    if (((field_info->value_num_fields + field_info->distinct_num_fields) == 0)
        || ((field_info->value_octets + field_info->distinct_octets) == 0))
    {
        ERR_FN("No value or distinct fields were specified");
        return -1;
    }

    return 0;
}


/* creates a new unique object */
int skUniqueCreate(
    sk_unique_t                   **uniq)
{
    *uniq = calloc(1, sizeof(sk_unique_t));
    if (NULL == *uniq) {
        return -1;
    }

    (*uniq)->temp_file_idx = -1;

    return 0;
}


/* destroys a unique object; cleans up any temporary files; etc. */
void skUniqueDestroy(
    sk_unique_t                   **uniq)
{
    sk_unique_t *u;

    if (NULL == uniq || NULL == *uniq) {
        return;
    }

    u = *uniq;
    *uniq = NULL;

    skTempFileTeardown();
    u->temp_file_idx = -1;
    if (u->ht) {
        hashlib_free_table(u->ht);
    }
    if (u->temp_dir) {
        free(u->temp_dir);
    }

    free(u);
}


/* sort our output */
int skUniqueSetSortedOutput(
    sk_unique_t                    *uniq)
{
    assert(uniq);

    if (uniq->ready_for_input) {
        return -1;
    }
    uniq->sort_output = 1;
    return 0;
}


/* specifies that unique object will be fed records that have been
 * pre-sorted by the key.  Must be called before
 * skUniquePrepareForInput().
 *
 * One big issue with this: currently when processing sorted data, we
 * can print/output rows as we read them.  If we now have separate
 * "read-everything" then "output-everything" steps, we lose a lot of
 * the advantages of processing sorted data.
 *
 * With the V2 version of this API, this is no longer an issue since
 * we have an output callback function.
 *
 * With V3 of the interface, we have the user specify a callback to
 * use when processing presorted input.  If the data is not presorted,
 * the user can iterate over the contents of the unique object as much
 * as they want.
 */


/* specifies the temporary directory. */
void skUniqueSetTempDirectory(
    sk_unique_t                    *uniq,
    const char                     *temp_dir)
{
    assert(uniq);

    if (uniq->ready_for_input) {
        return;
    }

    if (uniq->temp_dir) {
        free(uniq->temp_dir);
        uniq->temp_dir = NULL;
    }
    if (temp_dir) {
        uniq->temp_dir = strdup(temp_dir);
    }
}


/* sets the fields. */
int skUniqueSetFields(
    sk_unique_t                    *uniq,
    const sk_fieldlist_t           *key_fields,
    const sk_fieldlist_t           *distinct_fields,
    const sk_fieldlist_t           *agg_value_fields)
{
    assert(uniq);

    if (uniq->ready_for_input) {
        return -1;
    }

    memset(&uniq->fi, 0, sizeof(sk_uniq_field_info_t));
    uniq->fi.key_fields = key_fields;
    uniq->fi.distinct_fields = distinct_fields;
    uniq->fi.value_fields = agg_value_fields;

    return 0;
}


/* tells the unique object that initialization is complete.  returns
 * an error if the object is not completely specified. */
int skUniquePrepareForInput(
    sk_unique_t                    *uniq,
    sk_msg_fn_t                     err_fn)
{
    assert(uniq);

    if (uniq->ready_for_input) {
        return 0;
    }

    if (uniqCheckFields(&uniq->fi, err_fn)) {
        return -1;
    }

    /* set sizes for the hash table */
    SAFE_SET(uniq->hash_key_octets,
             uniq->fi.key_octets + uniq->fi.distinct_octets);
    uniq->hash_value_octets = uniq->fi.value_octets;
    if (uniq->hash_value_octets == 0) {
        /* need at least one byte for the value */
        uniq->hash_value_octets = 1;
    }
    uniq->hash_node_octets
        = uniq->hash_key_octets + uniq->hash_value_octets;

    /* Create the hash table */
    if (uniqueCreateHashTable(uniq)) {
        ERR_FN("Unable to create hash table");
        return -1;
    }

    if (skTempFileInitialize(uniq->temp_dir, err_fn)) {
        ERR_FN("Unable to initialize temporary file space");
        return -1;
    }

    uniq->ready_for_input = 1;
    return 0;
}


static int uniqueCreateHashTable(
    sk_unique_t        *uniq)
{
    uint8_t no_val[HASHLIB_MAX_VALUE_WIDTH];

    memset(no_val, 0xFF, sizeof(no_val));

    uniq->ht = hashlib_create_table(uniq->hash_key_octets,
                                    uniq->hash_value_octets,
                                    HTT_INPLACE,
                                    no_val,
                                    NULL,
                                    0,
                                    HASH_INITIAL_SIZE,
                                    DEFAULT_LOAD_FACTOR);
    if (NULL == uniq->ht) {
        return -1;
    }
    return 0;
}


/* adds a record to a unique object */
int skUniqueAddRecord(
    sk_unique_t                    *uniq,
    const rwRec                    *rwrec)
{
    uint8_t hash_key[HASHLIB_MAX_KEY_WIDTH];
    uint8_t *hash_val;
    int rv;

    assert(uniq);
    assert(uniq->ht);
    assert(rwrec);

    if (!uniq->ready_for_input) {
        return -1;
    }

    skFieldListRecToBinary(uniq->fi.key_fields, rwrec, hash_key);
    if (uniq->fi.distinct_num_fields) {
        skFieldListRecToBinary(uniq->fi.distinct_fields, rwrec,
                               hash_key + uniq->fi.key_octets);
    }

    /* the 'insert' will set 'hash_val' to the memory to use to
     * store the values. either fresh memory or the existing
     * value(s). */
    rv = hashlib_insert(uniq->ht, hash_key, &hash_val);
    switch (rv) {
      case OK:
        /* new value */
        if (0 == uniq->fi.value_num_fields) {
            /* only doing distinct */
            hash_val[0] = 0;
        } else {
            skFieldListInitializeBuffer(uniq->fi.value_fields, hash_val);
            skFieldListAddRecToBuffer(uniq->fi.value_fields, rwrec, hash_val);
        }
        break;

      case OK_DUPLICATE:
        /* existing value; add new value */
        if (uniq->fi.value_num_fields) {
            skFieldListAddRecToBuffer(uniq->fi.value_fields, rwrec, hash_val);
        }
        break;

      case ERR_OUTOFMEMORY:
      case ERR_NOMOREBLOCKS:
        /* out of memory */
        if (uniqueDumpHashToTemp(uniq)) {
            skAppPrintErr("Unable to create temporary file");
            return -1;
        }
        /* destroy and re-create the hash table */
        hashlib_free_table(uniq->ht);
        if (uniqueCreateHashTable(uniq)) {
            return -1;
        }
        rv = hashlib_insert(uniq->ht, hash_key, &hash_val);
        if (OK != rv) {
            skAppPrintErr(("Unexpected return code '%d'"
                           " from hash table insert on new hash table"),
                          rv);
            return -1;
        }
        if (0 == uniq->fi.value_num_fields) {
            /* only doing distinct */
            hash_val[0] = 0;
        } else {
            skFieldListInitializeBuffer(uniq->fi.value_fields, hash_val);
            skFieldListAddRecToBuffer(uniq->fi.value_fields, rwrec, hash_val);
        }
        break;

      default:
        skAppPrintErr(("Unexpected return code '%d'"
                       " from hash table insert"),
                      rv);
        return -1;
    }

    return 0;
}


static int uniqueCompareKeys(
    const void *a,
    const void *b,
    void       *v_field_list)
{
    return skFieldListCompareBuffers(v_field_list, a, b);
}


static int uniqueCompareKeysDistinct(
    const void *a,
    const void *b,
    void       *v_uniq)
{
    sk_unique_t *uniq = (sk_unique_t*)v_uniq;
    int rv;

    rv = skFieldListCompareBuffers(uniq->fi.key_fields, a, b);
    if (0 == rv) {
        rv = memcmp((uint8_t*)a + uniq->fi.key_octets,
                    (uint8_t*)b + uniq->fi.key_octets,
                    uniq->fi.distinct_octets);
    }
    return rv;
}


/* Get ready to return records to the caller. */
int skUniquePrepareForOutput(
    sk_unique_t                    *uniq)
{
    if (uniq->ready_for_output) {
        return 0;
    }
    if (!uniq->ready_for_input) {
        return -1;
    }

    if (uniq->temp_file_idx >= 0) {
        /* dump the current/final hash entries to a file */
        if (uniqueDumpHashToTemp(uniq)) {
            skAppPrintErr("Unable to create temporary file");
            return -1;
        }
        /* destroy the hash table; we're done with it */
        hashlib_free_table(uniq->ht);
        uniq->ht = NULL;

    } else if (uniq->fi.distinct_num_fields) {
        if (uniq->sort_output) {
            /* need to sort using skFieldListCompareBuffers for keys,
             * and memcmp for the distinct part of the key */
            hashlib_sort_entries_usercmp(uniq->ht, uniqueCompareKeysDistinct,
                                         (void*)uniq);
        } else {
            /* the order doesn't really matter, we just have to group
             * items that have the same key.  Sort by memcmp-ing the
             * keys */
            hashlib_sort_entries(uniq->ht);
        }

    } else if (uniq->sort_output) {
        /* need to sort using the skFieldListCompareBuffers function */
        hashlib_sort_entries_usercmp(uniq->ht, uniqueCompareKeys,
                                     (void*)uniq->fi.key_fields);
    }

    uniq->ready_for_output = 1;
    return 0;
}


#define DISTINCT_PTR(d_buffer, d_array, d_index)        \
    ((d_buffer) + (d_array)[(d_index)].dv_offset)


typedef struct hashset_file_st {
    /* table into which to write new values */
    HashSet        *hf_hashset;
    /* file in which to store old values; every time the HashSet
     * fills, a new block of records is appended to the file */
    FILE           *hf_file;
    /* offsets in the 'hf_file' where each block ends/begins.  does
     * not include an entry for the first block, which is 0. */
    sk_vector_t    *hf_offsets;
    /* index of hf_file in the temporary file API */
    int             hf_tmp_index;
} hashset_file_t;

typedef union distinct_tracker_un {
    hashset_file_t     *dv_hf;
    HashSet            *dv_hashset;
    sk_bitmap_t        *dv_bitmap;
    skIPTree_t         *dv_iptree;
    uint8_t            *dv_cache_alloc;
    uint8_t             dv_cache[SK_SIZEOF_VOIDP];
} distinct_tracker_t;

typedef struct distinct_value_st {
    uint64_t            dv_count;
    distinct_type_t     dv_type;
    uint8_t             dv_octets;
    uint8_t             dv_offset;
    int                 dv_ip;
    distinct_tracker_t  dv_v;
} distinct_value_t;


/*
 *
 *    Free all memory that was allocated by uniqDistinctAlloc().
 */
static void uniqDistinctFree(
    const sk_uniq_field_info_t *field_info,
    distinct_value_t           *distincts)
{
    distinct_value_t *dist;
    uint8_t i;

    for (i = 0; i < field_info->distinct_num_fields; ++i) {
        dist = &distincts[i];
        switch (dist->dv_type) {
          case DISTINCT_COUNT:
          case DISTINCT_CACHE:
            break;
          case DISTINCT_CACHE_ALLOC:
            if (dist->dv_v.dv_cache_alloc) {
                free(dist->dv_v.dv_cache_alloc);
            }
            break;
          case DISTINCT_BITMAP:
            skBitmapDestroy(&dist->dv_v.dv_bitmap);
            break;
          case DISTINCT_IPTREE:
            skIPTreeDelete(&dist->dv_v.dv_iptree);
            break;
          case DISTINCT_HASHSET_FILES:
            if (dist->dv_v.dv_hf) {
                if (dist->dv_v.dv_hf->hf_hashset) {
                    hashset_free_table(dist->dv_v.dv_hf->hf_hashset);
                }
                if (dist->dv_v.dv_hf->hf_offsets) {
                    skVectorDestroy(dist->dv_v.dv_hf->hf_offsets);
                }
                if (dist->dv_v.dv_hf->hf_file) {
                    skTempFileRemove(dist->dv_v.dv_hf->hf_tmp_index);
                }
                memset(dist->dv_v.dv_hf, 0, sizeof(hashset_file_t));
                free(dist->dv_v.dv_hf);
            }
            break;
          case DISTINCT_HASHSET:
            if (dist->dv_v.dv_hashset) {
                hashset_free_table(dist->dv_v.dv_hashset);
                dist->dv_v.dv_hashset = NULL;
            }
            break;
        }
    }
    free(distincts);
}


typedef enum {
    DISTINCT_KEYS_RANDOM,
    DISTINCT_KEYS_UNIQUE,
    DISTINCT_KEYS_SORTED
} distinct_key_property_t;



/*
 *  ok = uniqDistinctAlloc(field_info, iter, keys_are_uniq);
 *
 *    Create the data structures required by 'iter' to count distinct
 *    values.
 *
 *    When the value 'keys_are_uniq' is 1 and only 1 distinct field is
 *    being counted, the code will optimize the counting by counting
 *    the number of distinct keys.
 */
static int uniqDistinctAlloc(
    const sk_uniq_field_info_t *field_info,
    distinct_value_t          **new_distincts,
    distinct_key_property_t     key_prop)
{
    sk_fieldlist_iterator_t fl_iter;
    sk_fieldentry_t *field;
    distinct_value_t *distincts;
    distinct_value_t *dist;
    uint8_t total_octets = 0;
    uint8_t i;

    if (0 == field_info->distinct_num_fields) {
        return 0;
    }

    distincts = calloc(field_info->distinct_num_fields,
                       sizeof(distinct_value_t));
    if (NULL == distincts) {
        goto ERROR;
    }

    dist = distincts;

    /* determine how each field maps into the single buffer */
    skFieldListIteratorBind(field_info->distinct_fields, &fl_iter);
    while (NULL != (field = skFieldListIteratorNext(&fl_iter))) {
        dist->dv_octets = skFieldListEntryGetBinOctets(field);
        dist->dv_offset = total_octets;
        total_octets += dist->dv_octets;
        ++dist;
        assert(dist <= distincts + field_info->distinct_num_fields);
    }

    dist = distincts;
    i = 0;

    if (DISTINCT_KEYS_UNIQUE == key_prop
        && 1 == field_info->distinct_num_fields)
    {
        /* number of distinct values equals number of hash entries
         * that have the same key (ignoring distinct part of key) */
        dist->dv_type = DISTINCT_COUNT;

        *new_distincts = distincts;
        return 0;
    }

    if (DISTINCT_KEYS_RANDOM != key_prop) {
        /* the first field gets cached so we can watch for it to change */
        if (dist->dv_octets <= SK_SIZEOF_VOIDP) {
            dist->dv_type = DISTINCT_CACHE;
        } else {
            dist->dv_type = DISTINCT_CACHE_ALLOC;
            dist->dv_v.dv_cache_alloc = malloc(dist->dv_octets);
            if (NULL == dist->dv_v.dv_cache_alloc) {
                goto ERROR;
            }
        }
        ++dist;
        ++i;
    }

    /* for the other fields, we must keep track of each distinct value
     * we see */
    for ( ; i < field_info->distinct_num_fields; ++dist, ++i) {
        if (dist->dv_octets <= 2) {
            dist->dv_type = DISTINCT_BITMAP;
            if (skBitmapCreate(&dist->dv_v.dv_bitmap, 8 * dist->dv_octets)) {
                goto ERROR;
            }
        } else if (dist->dv_octets <= 4) {
            dist->dv_type = DISTINCT_IPTREE;
            if (skIPTreeCreate(&dist->dv_v.dv_iptree)) {
                goto ERROR;
            }
        } else {
            /* create a hashset for keeping track of values we see */
            dist->dv_type = DISTINCT_HASHSET;
            dist->dv_v.dv_hashset =
                hashset_create_set(dist->dv_octets,
                                   1 << 4 * dist->dv_octets,
                                   DEFAULT_LOAD_FACTOR);
            if (NULL == dist->dv_v.dv_hashset) {
                goto ERROR;
            }
        }
    }

    *new_distincts = distincts;
    return 0;

  ERROR:
    uniqDistinctFree(field_info, distincts);
    return -1;
}


/*
 *    Increment the distinct counters given 'key'.
 */
static int uniqDistinctIncrement(
    const sk_uniq_field_info_t *field_info,
    distinct_value_t           *distincts,
    const uint8_t              *key)
{
    distinct_value_t *dist;
    uint8_t i;
    int rv;

    for (i = 0; i < field_info->distinct_num_fields; ++i) {
        dist = &distincts[i];
        switch (dist->dv_type) {
          case DISTINCT_COUNT:
            ++dist->dv_count;
            break;
          case DISTINCT_CACHE:
            if (0 != memcmp(dist->dv_v.dv_cache,
                            DISTINCT_PTR(key, distincts, i),
                            dist->dv_octets))
            {
                ++dist->dv_count;
                memcpy(dist->dv_v.dv_cache,
                       DISTINCT_PTR(key, distincts, i),
                       dist->dv_octets);
            }
            break;
          case DISTINCT_CACHE_ALLOC:
            if (0 != memcmp(dist->dv_v.dv_cache_alloc,
                            DISTINCT_PTR(key, distincts, i),
                            dist->dv_octets))
            {
                ++dist->dv_count;
                memcpy(dist->dv_v.dv_cache_alloc,
                       DISTINCT_PTR(key, distincts, i),
                       dist->dv_octets);
            }
            break;
          case DISTINCT_BITMAP:
            skBitmapSetBit(dist->dv_v.dv_bitmap,
                           *(uint16_t*)DISTINCT_PTR(key, distincts, i));
            dist->dv_count
                = skBitmapGetHighCount(dist->dv_v.dv_bitmap);
            break;
          case DISTINCT_IPTREE:
            if (!skIPTreeCheckAddress(dist->dv_v.dv_iptree,
                                      *(uint32_t*)DISTINCT_PTR(key, distincts,
                                                               i)))
            {
                skIPTreeAddAddress(dist->dv_v.dv_iptree,
                                   *(uint32_t*)DISTINCT_PTR(key,distincts,i));
                ++dist->dv_count;
            }
            break;
          case DISTINCT_HASHSET_FILES:
#if 0
            for (j = 0, beg = end = 0;
                 j < skVectorGetCount(dist->dv_v.dv_hf->hf_offsets);
                 ++j, beg = end)
            {
                skVectorGetValue(&end, dist->dv_v.dv_hf->hf_offsets, j);
                bot = beg;
                top = end;
                while (top > bot) {
                    pos = (bot + top) >> 1;
                    cmp = memcmp(DISTINCT_PTR(key, distincts, i),



               , dist->dv_octets);


key_in, GET_KEY_PTR(bt, node->key, pos),
                     bt->cmp_size);
        if (cmp < 0) {
            top = pos - 1;
        } else if (cmp > 0) {
            bot = pos + 1;
        } else {
            *position = pos;
            return SKBTREE_OK;
        }
    }



            rv = hashset_insert(dist->dv_v.dv_hashset,
                                DISTINCT_PTR(key, distincts, i));
            switch (rv) {
              case OK:
                ++dist->dv_count;
                break;
              case OK_DUPLICATE:
                break;
              default:
                return -1;
            }
            break;




            if (dist->dv_v.dv_hf) {
                if (dist->dv_v.dv_hf->hf_hashset) {
                    hashset_free_table(dist->dv_v.dv_hf->hf_hashset);
                }
                if (dist->dv_v.dv_hf->hf_offsets) {
                    skVectorDestroy(dist->dv_v.dv_hf->hf_offsets);
                }
                if (dist->dv_v.dv_hf->hf_file) {
                    skTempFileRemove(dist->dv_v.dv_hf->hf_tmp_index);
                }
                memset(dist->dv_v.dv_hf, 0, sizeof(hashset_file_t));
                free(dist->dv_v.dv_hf);
            }
#endif  /* 0 */
            break;
          case DISTINCT_HASHSET:
            rv = hashset_insert(dist->dv_v.dv_hashset,
                                DISTINCT_PTR(key, distincts, i));
            switch (rv) {
              case OK:
                ++dist->dv_count;
                break;
              case OK_DUPLICATE:
                break;
              default:
                return -1;
            }
            break;
        }
    }

    return 0;
}


/*
 *    Set the outbuf buffer to contain the number of distinct values
 */
static int uniqDistinctSetOutputBuf(
    const sk_uniq_field_info_t   *field_info,
    const distinct_value_t       *distincts,
    uint8_t                      *out_buf)
{
    const distinct_value_t *dist;
    uint8_t i;

    for (i = 0; i < field_info->distinct_num_fields; ++i) {
        dist = &distincts[i];
        switch (dist->dv_octets) {
          case 1:
            *((uint8_t*)DISTINCT_PTR(out_buf, distincts, i))
                = (uint8_t)(dist->dv_count);
            break;
#if !SKUNIQ_USE_MEMCPY
          case 2:
            *((uint16_t*)DISTINCT_PTR(out_buf, distincts, i))
                = (uint16_t)(dist->dv_count);
            break;
          case 4:
            *((uint32_t*)DISTINCT_PTR(out_buf, distincts, i))
                = (uint32_t)(dist->dv_count);
            break;
          case 8:
            *((uint64_t*)DISTINCT_PTR(out_buf, distincts, i))
                = dist->dv_count;
            break;
          default:
            *((uint64_t*)DISTINCT_PTR(out_buf, distincts, i))
                = dist->dv_count;
            break;
#else
          case 2:
            {
                uint16_t val16 = (uint16_t)(dist->dv_count);
                memcpy(DISTINCT_PTR(out_buf, distincts, i),
                       &val16, sizeof(val16));
            }
            break;
          case 4:
            {
                uint32_t val32 = (uint32_t)(dist->dv_count);
                memcpy(DISTINCT_PTR(out_buf, distincts, i),
                       &val32, sizeof(val32));
            }
            break;
          case 8:
            memcpy(DISTINCT_PTR(out_buf, distincts, i),
                   &dist->dv_count, sizeof(uint64_t));
            break;
          default:
            memcpy(DISTINCT_PTR(out_buf, distincts, i),
                   &dist->dv_count, sizeof(uint64_t));
            break;
#endif  /* SKUNIQ_USE_MEMCPY */
        }
    }

    return 0;
}


/*
 *    Reset the distinct counters.
 */
static int uniqDistinctReset(
    const sk_uniq_field_info_t *field_info,
    distinct_value_t           *distincts,
    const uint8_t              *key)
{
    distinct_value_t *dist;
    uint8_t i;
    int rv;

    if (NULL == key) {
        for (i = 0; i < field_info->distinct_num_fields; ++i) {
            dist = &distincts[i];
            switch (dist->dv_type) {
              case DISTINCT_COUNT:
              case DISTINCT_CACHE:
              case DISTINCT_CACHE_ALLOC:
                break;
              case DISTINCT_BITMAP:
                skBitmapClearAllBits(dist->dv_v.dv_bitmap);
                break;
              case DISTINCT_IPTREE:
                skIPTreeRemoveAll(dist->dv_v.dv_iptree);
                break;
              case DISTINCT_HASHSET_FILES:
                if (dist->dv_v.dv_hf) {
                    if (dist->dv_v.dv_hf->hf_hashset) {
                        hashset_free_table(dist->dv_v.dv_hf->hf_hashset);
                    }
                    if (dist->dv_v.dv_hf->hf_offsets) {
                        skVectorDestroy(dist->dv_v.dv_hf->hf_offsets);
                    }
                    if (dist->dv_v.dv_hf->hf_file) {
                        skTempFileRemove(dist->dv_v.dv_hf->hf_tmp_index);
                    }
                    memset(dist->dv_v.dv_hf, 0, sizeof(hashset_file_t));
                    free(dist->dv_v.dv_hf);
                }
                break;
              case DISTINCT_HASHSET:
                if (dist->dv_v.dv_hashset) {
                    hashset_free_table(dist->dv_v.dv_hashset);
                }
                dist->dv_v.dv_hashset =
                    hashset_create_set(dist->dv_octets,
                                       1 << 4 * dist->dv_octets,
                                       DEFAULT_LOAD_FACTOR);
                if (NULL == dist->dv_v.dv_hashset) {
                    return -1;
                }
                break;
            }
            dist->dv_count = 0;
        }
        return 0;
    }


    for (i = 0; i < field_info->distinct_num_fields; ++i) {
        dist = &distincts[i];
        switch (dist->dv_type) {
          case DISTINCT_COUNT:
            break;
          case DISTINCT_CACHE:
            memcpy(dist->dv_v.dv_cache,
                   DISTINCT_PTR(key, distincts, i),
                   dist->dv_octets);
            break;
          case DISTINCT_CACHE_ALLOC:
            memcpy(dist->dv_v.dv_cache_alloc,
                   DISTINCT_PTR(key, distincts, i),
                   dist->dv_octets);
            break;
          case DISTINCT_BITMAP:
            skBitmapClearAllBits(dist->dv_v.dv_bitmap);
            skBitmapSetBit(dist->dv_v.dv_bitmap,
                           *(uint16_t*)DISTINCT_PTR(key, distincts, i));
            break;
          case DISTINCT_IPTREE:
            skIPTreeRemoveAll(dist->dv_v.dv_iptree);
            skIPTreeAddAddress(dist->dv_v.dv_iptree,
                               *(uint32_t*)DISTINCT_PTR(key, distincts, i));
            break;
          case DISTINCT_HASHSET_FILES:
            if (dist->dv_v.dv_hf) {
                if (dist->dv_v.dv_hf->hf_hashset) {
                    hashset_free_table(dist->dv_v.dv_hf->hf_hashset);
                }
                if (dist->dv_v.dv_hf->hf_offsets) {
                    skVectorDestroy(dist->dv_v.dv_hf->hf_offsets);
                }
                if (dist->dv_v.dv_hf->hf_file) {
                    skTempFileRemove(dist->dv_v.dv_hf->hf_tmp_index);
                }
                memset(dist->dv_v.dv_hf, 0, sizeof(hashset_file_t));
                free(dist->dv_v.dv_hf);
            }
            break;
          case DISTINCT_HASHSET:
            if (dist->dv_v.dv_hashset) {
                hashset_free_table(dist->dv_v.dv_hashset);
            }
            dist->dv_v.dv_hashset =
                hashset_create_set(dist->dv_octets,
                                   1 << (4 * dist->dv_octets),
                                   DEFAULT_LOAD_FACTOR);
            if (NULL == dist->dv_v.dv_hashset) {
                return -1;
            }
            rv = hashset_insert(dist->dv_v.dv_hashset,
                                DISTINCT_PTR(key, distincts, i));
            if (rv != OK) {
                return -1;
            }
            break;
        }
        dist->dv_count = 1;
    }
    return 0;
}



/****************************************************************
 * Iterator for hanlding one hash table, no distinct counts
 ***************************************************************/

typedef struct uniqiter_simple_st {
    sk_uniqiter_reset_fn_t  reset_fn;
    sk_uniqiter_next_fn_t   next_fn;
    sk_uniqiter_free_fn_t   free_fn;
    sk_unique_t            *uniq;
    HASH_ITER               ithash;
} uniqiter_simple_t;


static int uniqIterSimpleReset(
    sk_unique_iterator_t           *v_iter)
{
    uniqiter_simple_t *iter = (uniqiter_simple_t*)v_iter;

    /* create the iterator */
    iter->ithash = hashlib_create_iterator(iter->uniq->ht);
    return 0;
}


static int uniqIterSimpleNext(
    sk_unique_iterator_t           *v_iter,
    uint8_t                       **key_fields_buffer,
    uint8_t                UNUSED(**distinct_fields_buffer),
    uint8_t                       **value_fields_buffer)
{
    uniqiter_simple_t *iter = (uniqiter_simple_t*)v_iter;

    if (hashlib_iterate(iter->uniq->ht, &iter->ithash,
                        key_fields_buffer, value_fields_buffer)
        == ERR_NOMOREENTRIES)
    {
        return SK_ITERATOR_NO_MORE_ENTRIES;
    }
    return SK_ITERATOR_OK;
}


static void uniqIterSimpleDestroy(
    sk_unique_iterator_t          **v_iter)
{
    uniqiter_simple_t *iter;

    if (v_iter && *v_iter) {
        iter = *(uniqiter_simple_t**)v_iter;
        memset(iter, 0, sizeof(uniqiter_simple_t));
        free(iter);
        *v_iter = NULL;
    }
}


static int uniqIterSimpleCreate(
    sk_unique_t                    *uniq,
    sk_unique_iterator_t          **new_iter)
{
    uniqiter_simple_t *iter;

    iter = calloc(1, sizeof(uniqiter_simple_t));
    if (NULL == iter) {
        return -1;
    }

    iter->uniq = uniq;
    iter->reset_fn = uniqIterSimpleReset;
    iter->next_fn = uniqIterSimpleNext;
    iter->free_fn = uniqIterSimpleDestroy;

    if (uniqIterSimpleReset((sk_unique_iterator_t*)iter)) {
        uniqIterSimpleDestroy((sk_unique_iterator_t**)&iter);
        return -1;
    }

    *new_iter = (sk_unique_iterator_t*)iter;
    return 0;
}



/****************************************************************
 * Iterator for hanlding distinct values in one hash table
 ***************************************************************/

typedef struct uniqiter_distinct_st {
    sk_uniqiter_reset_fn_t  reset_fn;
    sk_uniqiter_next_fn_t   next_fn;
    sk_uniqiter_free_fn_t   free_fn;
    sk_unique_t            *uniq;
    distinct_value_t       *distincts;
    HASH_ITER               ithash;
    uint8_t                 returned_buf[HASH_MAX_NODE_BYTES];
    uint8_t                 scratch[HASH_MAX_NODE_BYTES];
    /* next two values are pointers into 'scratch' */
    uint8_t                *cached_key;
    uint8_t                *merged_values;
    int                     no_more_entries;
} uniqiter_distinct_t;


static int uniqIterDistinctReset(
    sk_unique_iterator_t           *v_iter)
{
    uniqiter_distinct_t *iter = (uniqiter_distinct_t*)v_iter;
    uint8_t *hash_key;
    uint8_t *hash_val;

    /* create the iterator */
    iter->ithash = hashlib_create_iterator(iter->uniq->ht);

    /* get first key/value from hash table */
    if (hashlib_iterate(iter->uniq->ht, &iter->ithash, &hash_key, &hash_val)
        == ERR_NOMOREENTRIES)
    {
        /* no data.  done. */
        iter->no_more_entries = 1;
        return 0;
    }

    /* cache the key and start summing the values */
    memcpy(iter->cached_key, hash_key, iter->uniq->fi.key_octets);
    memcpy(iter->merged_values, hash_val, iter->uniq->fi.value_octets);

    /* handle the distinct fields */
    uniqDistinctReset(&iter->uniq->fi, iter->distincts,
                      hash_key + iter->uniq->fi.key_octets);

    return 0;
}


static int uniqIterDistinctNext(
    sk_unique_iterator_t   *v_iter,
    uint8_t               **key_fields_buffer,
    uint8_t               **distinct_fields_buffer,
    uint8_t               **value_fields_buffer)
{
    uniqiter_distinct_t *iter = (uniqiter_distinct_t*)v_iter;
    uint8_t *hash_key;
    uint8_t *hash_val;

    if (iter->no_more_entries) {
        return SK_ITERATOR_NO_MORE_ENTRIES;
    }

    /* process the remaining hash entries */
    while (hashlib_iterate(iter->uniq->ht, &iter->ithash, &hash_key, &hash_val)
           != ERR_NOMOREENTRIES)
    {
        /* compare keys */
        if (0 == memcmp(iter->cached_key, hash_key,
                        iter->uniq->fi.key_octets))
        {
            /* keys match; add current 'hash_val' to the
             * 'merged_values' array */
            skFieldListMergeBuffers(iter->uniq->fi.value_fields,
                                    iter->merged_values, hash_val);
            uniqDistinctIncrement(&iter->uniq->fi, iter->distincts,
                                  hash_key + iter->uniq->fi.key_octets);

        } else {
            /* keys differ. write the current key, values, and
             * distincts into the buffers that get returned to the
             * caller */
            memcpy(iter->returned_buf, iter->cached_key,
                   iter->uniq->fi.key_octets);
            memcpy(iter->returned_buf + iter->uniq->hash_key_octets,
                   iter->merged_values, iter->uniq->fi.value_octets);
            uniqDistinctSetOutputBuf(&iter->uniq->fi, iter->distincts,
                                     (iter->returned_buf
                                      + iter->uniq->fi.key_octets));

            /* remember the key and reset values and distincts */
            memcpy(iter->cached_key, hash_key, iter->uniq->fi.key_octets);

            if (iter->uniq->fi.value_num_fields) {
                skFieldListInitializeBuffer(iter->uniq->fi.value_fields,
                                            iter->merged_values);
                skFieldListMergeBuffers(iter->uniq->fi.value_fields,
                                        iter->merged_values,
                                        hash_val);
            }
            uniqDistinctReset(&iter->uniq->fi, iter->distincts,
                              hash_key + iter->uniq->fi.key_octets);
            goto END;
        }
    }

    iter->no_more_entries = 1;

    /* put final keys, values, distincts into the return buffers */
    memcpy(iter->returned_buf, iter->cached_key,
           iter->uniq->fi.key_octets);
    memcpy(iter->returned_buf + iter->uniq->hash_key_octets,
           iter->merged_values, iter->uniq->fi.value_octets);
    uniqDistinctSetOutputBuf(&iter->uniq->fi, iter->distincts,
                             (iter->returned_buf + iter->uniq->fi.key_octets));

  END:
    /* set user's pointers to the buffers */
    *key_fields_buffer = iter->returned_buf;
    *value_fields_buffer = iter->returned_buf + iter->uniq->hash_key_octets;
    *distinct_fields_buffer = iter->returned_buf + iter->uniq->fi.key_octets;
    return SK_ITERATOR_OK;
}


static void uniqIterDistinctDestroy(
    sk_unique_iterator_t          **v_iter)
{
    uniqiter_distinct_t *iter;

    if (v_iter && *v_iter) {
        iter = *(uniqiter_distinct_t**)v_iter;

        uniqDistinctFree(&iter->uniq->fi, iter->distincts);

        memset(iter, 0, sizeof(uniqiter_distinct_t));
        free(iter);
        *v_iter = NULL;
    }
}


static int uniqIterDistinctCreate(
    sk_unique_t                    *uniq,
    sk_unique_iterator_t          **new_iter)
{
    uniqiter_distinct_t *iter;

    assert(uniq);
    assert(uniq->fi.distinct_num_fields > 0);

    iter = calloc(1, sizeof(uniqiter_distinct_t));
    if (NULL == iter) {
        return -1;
    }

    iter->uniq = uniq;
    iter->reset_fn = uniqIterDistinctReset;
    iter->next_fn = uniqIterDistinctNext;
    iter->free_fn = uniqIterDistinctDestroy;

    iter->cached_key = iter->scratch;
    iter->merged_values = iter->scratch + uniq->hash_key_octets;

    /* set up the handling of distinct field(s) */
    if (0 == uniq->fi.distinct_num_fields) {
        /* this should never happen */
        skAppPrintErr("distinct_num_fields is 0");
        skAbort();
    }
    if (uniqDistinctAlloc(&uniq->fi, &iter->distincts, DISTINCT_KEYS_UNIQUE)) {
        uniqIterDistinctDestroy((sk_unique_iterator_t**)&iter);
        return -1;
    }

    if (uniqIterDistinctReset((sk_unique_iterator_t*)iter)) {
        uniqIterDistinctDestroy((sk_unique_iterator_t**)&iter);
        return -1;
    }

    *new_iter = (sk_unique_iterator_t*)iter;
    return 0;
}



/****************************************************************
 * Iterator for hanlding temporary files
 ***************************************************************/

#define UNIQ_TEMPFILE_CMP(tfc_node_a, tfc_node_b, tfc_iter)             \
    ((!(tfc_iter)->uniq->sort_output)                                   \
     ? memcmp((tfc_node_a), (tfc_node_b),                               \
              (tfc_iter)->uniq->hash_key_octets)                        \
     : (((tfc_iter)->uniq->fi.distinct_num_fields)                      \
        ? uniqueCompareKeysDistinct((tfc_node_a), (tfc_node_b),         \
                                    (void*)(tfc_iter)->uniq)            \
        : uniqueCompareKeys((tfc_node_a), (tfc_node_b),                 \
                            (void*)(tfc_iter)->uniq->fi.key_fields)))   \


typedef struct uniqiter_tempfiles_st {
    sk_uniqiter_reset_fn_t  reset_fn;
    sk_uniqiter_next_fn_t   next_fn;
    sk_uniqiter_free_fn_t   free_fn;
    sk_unique_t            *uniq;
    distinct_value_t       *distincts;
    uint16_t                reading[MAX_MERGE_FILES];
    FILE                   *fps[MAX_MERGE_FILES];
    uint8_t                 node[MAX_MERGE_FILES][HASH_MAX_NODE_BYTES];
    uint8_t                 returned_buf[HASH_MAX_NODE_BYTES];
    uint8_t                 scratch[HASH_MAX_NODE_BYTES];
    /* next three values are pointers into 'scratch' */
    uint8_t                *cached_key;
    uint8_t                *key_distincts;
    uint8_t                *merged_values;
    int                     no_more_entries;
    uint16_t                open_count;
    uint16_t                read_count;
} uniqiter_tempfiles_t;


static int uniqIterOpenAllTempFiles(
    uniqiter_tempfiles_t   *iter);


static int uniqIterTempfilesReset(
    sk_unique_iterator_t   *v_iter)
{
    uniqiter_tempfiles_t *iter = (uniqiter_tempfiles_t*)v_iter;
    uint16_t lowest = 0;
    uint16_t read_idx = 0;
    uint16_t j;

    /* if files are already open (e.g., caller has reset the
     * iterator), we need to close them first */
    for (j = 0; j < iter->open_count; ++j) {
        if (iter->fps[j]) {
            fclose(iter->fps[j]);
        }
        iter->fps[j] = NULL;
    }

    /* open all temp files */
    iter->open_count = uniqIterOpenAllTempFiles(iter);
    if (iter->open_count == (uint16_t)-1) {
        return -1;
    }

    /* read the first key/value pair from each temp file, put into the
     * work buffer, and find the one with the lowest key.  Use the
     * entire hash_key---i.e., the user's key and any distinct
     * columns. */
    for (j = 0; j < iter->open_count; ++j) {
        if (!fread(iter->node[j], iter->uniq->hash_node_octets,
                   1, iter->fps[j]))
        {
            PRINTDEBUG((stderr, "Cannot read first record from file"));
            continue;
        }

        if ((iter->read_count == 0)
            || (0 > UNIQ_TEMPFILE_CMP(iter->node[j], iter->node[lowest],iter)))
        {
            /* either this is the first node we've seen, or the node
             * from j is lower than current lowest; record the index
             * of this temp file as being lowest */
            lowest = j;
            read_idx = iter->read_count;
        }

        iter->reading[iter->read_count] = j;
        ++iter->read_count;
    }

    if (iter->read_count == 0) {
        skAppPrintErr("Could not read records from any temp files");
        return -1;
    }

    /* found the lowest; cache it */
    memcpy(iter->scratch, iter->node[lowest], iter->uniq->hash_node_octets);

    /* replace the record we just cached */
    if (!fread(iter->node[lowest], iter->uniq->hash_node_octets,
               1, iter->fps[lowest]))
    {
        /* finished with this file; replace with last file is list */
        --iter->read_count;
        iter->reading[read_idx] = iter->reading[iter->read_count];
    }

    uniqDistinctReset(&iter->uniq->fi, iter->distincts, iter->key_distincts);

    return 0;
}


static int uniqIterTempfilesNext(
    sk_unique_iterator_t   *v_iter,
    uint8_t               **key_fields_buffer,
    uint8_t               **distinct_fields_buffer,
    uint8_t               **value_fields_buffer)
{
    uniqiter_tempfiles_t *iter = (uniqiter_tempfiles_t*)v_iter;
    uint16_t lowest;
    uint16_t read_idx;
    uint16_t i;
    int return_to_caller = 0;

    if (iter->no_more_entries) {
        return SK_ITERATOR_NO_MORE_ENTRIES;
    }

    /* process all remaining key/value pairs in all opened files */
    while (iter->read_count) {
        /* set "lowest" to first file with data */
        lowest = iter->reading[0];
        read_idx = 0;

        /* compare "lowest" with every other file to find the
         * actual lowest */
        for (i = 1; i < iter->read_count; ++i) {
            /* If the temp file's key is lower than the current
             * lowest... */
            if (0 > UNIQ_TEMPFILE_CMP(iter->node[iter->reading[i]],
                                      iter->node[lowest], iter))
            {
                /* ...record the index of this temp file as having
                 * the lowest key */
                lowest = iter->reading[i];
                read_idx = i;
            }
        }

        /* found the lowest key; compare this key with the cached_key;
         * using only the user's key (ie, ignoring distincts) */
        if (0 == memcmp(iter->cached_key, iter->node[lowest],
                        iter->uniq->fi.key_octets))
        {
            /* keys are same; add the current values */
            skFieldListMergeBuffers(iter->uniq->fi.value_fields,
                                    iter->merged_values,
                                    (iter->node[lowest]
                                     + iter->uniq->hash_key_octets));
            uniqDistinctIncrement(&iter->uniq->fi, iter->distincts,
                                  (iter->node[lowest]
                                   + iter->uniq->fi.key_octets));

        } else {
            /* keys differ. write the current key, values, and
             * distincts into the buffers that get returned to the
             * caller.  we'll return once we read another node */
            memcpy(iter->returned_buf, iter->cached_key,
                   iter->uniq->fi.key_octets);
            memcpy(iter->returned_buf + iter->uniq->hash_key_octets,
                   iter->merged_values, iter->uniq->fi.value_octets);
            uniqDistinctSetOutputBuf(&iter->uniq->fi, iter->distincts,
                                     (iter->returned_buf
                                      + iter->uniq->fi.key_octets));
            return_to_caller = 1;

            /* cache the node (key and values) and reset distincts */
            memcpy(iter->scratch, iter->node[lowest],
                   iter->uniq->hash_node_octets);
            uniqDistinctReset(&iter->uniq->fi, iter->distincts,
                              iter->key_distincts);
        }

        /* replace the record we just processed */
        if (!fread(iter->node[lowest], iter->uniq->hash_node_octets,
                   1, iter->fps[lowest]))
        {
            /* file is empty; replac with last file in the list */
            --iter->read_count;
            iter->reading[read_idx] = iter->reading[iter->read_count];
        }

        /* maybe return */
        if (return_to_caller) {
            goto END;
        }
    }

    iter->no_more_entries = 1;

    /* output the final entry */
    memcpy(iter->returned_buf, iter->cached_key,
           iter->uniq->fi.key_octets);
    memcpy(iter->returned_buf + iter->uniq->hash_key_octets,
           iter->merged_values, iter->uniq->fi.value_octets);
    uniqDistinctSetOutputBuf(&iter->uniq->fi, iter->distincts,
                             (iter->returned_buf + iter->uniq->fi.key_octets));

  END:
    /* set user's pointers to the buffers */
    *key_fields_buffer = iter->returned_buf;
    *value_fields_buffer = iter->returned_buf + iter->uniq->hash_key_octets;
    *distinct_fields_buffer = iter->returned_buf + iter->uniq->fi.key_octets;
    return SK_ITERATOR_OK;
}


static void uniqIterTempfilesDestroy(
    sk_unique_iterator_t          **v_iter)
{
    uniqiter_tempfiles_t *iter;
    size_t i;

    if (v_iter && *v_iter) {
        iter = *(uniqiter_tempfiles_t**)v_iter;

        for (i = 0; i < iter->open_count; ++i) {
            if (iter->fps[i]) {
                fclose(iter->fps[i]);
            }
            iter->fps[i] = NULL;
        }

        uniqDistinctFree(&iter->uniq->fi, iter->distincts);

        memset(iter, 0, sizeof(uniqiter_tempfiles_t));
        free(iter);
        *v_iter = NULL;
    }
}


static int uniqIterTempfilesCreate(
    sk_unique_t                    *uniq,
    sk_unique_iterator_t          **new_iter)
{
    uniqiter_tempfiles_t *iter;

    iter = calloc(1, sizeof(uniqiter_tempfiles_t));
    if (NULL == iter) {
        return -1;
    }

    iter->uniq = uniq;
    iter->reset_fn = uniqIterTempfilesReset;
    iter->next_fn = uniqIterTempfilesNext;
    iter->free_fn = uniqIterTempfilesDestroy;

    iter->cached_key = iter->scratch;
    iter->key_distincts = iter->cached_key + iter->uniq->fi.key_octets;
    iter->merged_values = iter->scratch + uniq->hash_key_octets;

    /* set up the handling of distinct field(s) */
    if (uniq->fi.distinct_num_fields) {
        if (uniqDistinctAlloc(&uniq->fi,&iter->distincts,DISTINCT_KEYS_SORTED))
        {
            free(iter);
            return -1;
        }
    }

    if (uniqIterTempfilesReset((sk_unique_iterator_t*)iter)) {
        uniqIterTempfilesDestroy((sk_unique_iterator_t**)&iter);
        return -1;
    }

    *new_iter = (sk_unique_iterator_t*)iter;
    return 0;
}




/* used to get bins from the unique object */
int skUniqueIteratorCreate(
    sk_unique_t                    *uniq,
    sk_unique_iterator_t          **new_iter)
{
    if (!uniq->ready_for_output) {
        return -1;
    }

    if (uniq->temp_file_idx >= 0) {
        return uniqIterTempfilesCreate(uniq, new_iter);
    }

    if (uniq->fi.distinct_num_fields) {
        return uniqIterDistinctCreate(uniq, new_iter);
    }

    return uniqIterSimpleCreate(uniq, new_iter);
}


/*
 *  count = uniqIterOpenAllTempFiles(iter);
 *
 *    Open all temporary files numbered between 'idx_1' and 'idx_2',
 *    put the file handles in the array 'fp', and return the number of
 *    files opened.  Normally, the number of returned file handles is
 *    given by (1 + idx_2 - idx_1).
 *
 *    If it is impossible to open all files due to a lack of file
 *    handles, the existing temporary files will be merged into new
 *    temporary files, and then another attempt will be made to open
 *    all files.
 *
 *    This function will only return when it is possible to return a
 *    file handle to every existing temporary file.  If it is unable
 *    to create a new temporary file, it returns -1.
 */
static int uniqIterOpenAllTempFiles(
    uniqiter_tempfiles_t   *iter)
{
    uint16_t i;
    uint16_t lowest;
    uint16_t read_idx;
    int j;
    int tmp_idx_a;
    int tmp_idx_b;
    FILE *fp_intermediate = NULL;
    int tmp_idx_intermediate;
    int open_count;


    /* index at which to start the merge */
    tmp_idx_a = 0;

    /* This loop repeats as long as we haven't opened all of the temp
     * files generated while reading the flows. */
    for (;;) {
        /* index at which to stop the merge */
        tmp_idx_b = iter->uniq->temp_file_idx;

        PRINTDEBUG((stderr, "opening tmp files #%d through #%d\n",
                    tmp_idx_a, tmp_idx_b));

        /* open an intermediate temp file.  The merge-sort will have
         * to write nodes here if there are not enough file handles
         * available to open all the temporary files we wrote while
         * reading the data. */
        fp_intermediate = skTempFileCreate(&tmp_idx_intermediate,  NULL);
        if (fp_intermediate == NULL) {
            return -1;
        }

        open_count = 0;

        /* Attempt to open all temp files, but stop after
         * MAX_MERGE_FILES files, or if we fail due to lack of
         * resources (EMFILE or ENOMEM) */
        for (j = tmp_idx_a; j <= tmp_idx_b; ++j) {
            iter->fps[open_count] = skTempFileOpen(j);
            if (iter->fps[open_count] == NULL) {
                if ((open_count > 0)
                    && ((errno == EMFILE) || (errno == ENOMEM)))
                {
                    /* Blast!  We can't open any more temp files.  So,
                     * we rewind by one to catch this one the next
                     * time around. */
                    tmp_idx_b = j - 1;
                    PRINTDEBUG((stderr, ("EMFILE limit hit--"
                                         "merging %d through %d to %d\n"),
                                tmp_idx_a, tmp_idx_b, tmp_idx_intermediate));
                    break;
                } else {
                    PRINTDEBUG((stderr, "couldn't open tmp file %s\n",
                                skTempFileGetName(j)));
                    return -1;
                }
            }

            ++open_count;
            if (open_count == MAX_MERGE_FILES) {
                /* We've reached the limit for this pass.  Set
                 * tmp_idx_b to the file we just opened. */
                tmp_idx_b = j;
                PRINTDEBUG((stderr, ("MAX_MERGE_FILES limit hit--"
                                     "merging %d through %d to %d\n"),
                            tmp_idx_a, tmp_idx_b, tmp_idx_intermediate));
                break;
            }
        }

        PRINTDEBUG((stderr,"opened %d temp files\n", open_count));

        /* Check to see if we've opened all temp files.  If so, close
         * the intermediate file and return */
        if (tmp_idx_b == iter->uniq->temp_file_idx) {
            PRINTDEBUG((stderr, "successfully opened all temp files\n"));
            if (EOF == fclose(fp_intermediate)) {
                skAppPrintSyserror("Error closing file %s",
                                   skTempFileGetName(tmp_idx_intermediate));
                return -1;
            }
            return open_count;
        }
        /* Else, we could not open all temp files, so merge all opened
         * temp files into the intermediate file */

        iter->uniq->temp_file_idx = tmp_idx_intermediate;

        iter->read_count = 0;

        /* Read the first key/value pair from each temp file into the
         * work buffer. */
        for (i = 0; i < open_count; ++i) {
            if (!fread(iter->node[i], iter->uniq->hash_node_octets,
                       1, iter->fps[i]))
            {
                PRINTDEBUG((stderr, "couldn't get first record from file %s\n",
                            skTempFileGetName(tmp_idx_a + i)));
                continue;
            }
            /* 'reading' holds the indexes of the files from which we
             * are reading records */
            iter->reading[iter->read_count] = i;
            ++iter->read_count;
        }

        PRINTDEBUG((stderr,
                    "open_count: %" PRIu16 "; read_count: %" PRIu16 "\n",
                    open_count, iter->read_count));

        /* exit this while() once we are only processing a single
         * file */
        while (iter->read_count > 1) {
            /* set "lowest" to first file with data */
            lowest = iter->reading[0];
            read_idx = 0;

            /* compare "lowest" with every other file to find the
             * actual lowest */
            for (i = 1; i < iter->read_count; ++i) {
                /* If the temp file's key is lower than the current
                 * lowest... */
                if (0 > UNIQ_TEMPFILE_CMP(iter->node[iter->reading[i]],
                                          iter->node[lowest], iter))
                {
                    /* ...record the index of this temp file as having
                     * the lowest key */
                    lowest = iter->reading[i];
                    read_idx = i;
                }
            }

            /* write the lowest key/value pair to the intermediate
             * temp file */
            if (!fwrite(iter->node[lowest], iter->uniq->hash_node_octets,
                        1, fp_intermediate))
            {
                skAppPrintErr("Could not write to tmpfile '%s'",
                              skTempFileGetName(tmp_idx_intermediate));
                return -1;
            }

            /* replace the node we just wrote */
            if (!fread(iter->node[lowest], iter->uniq->hash_node_octets,
                      1, iter->fps[lowest]))
            {
                /* no more data for this file; replace file with last
                 * file in 'reading' array */
                --iter->read_count;
                iter->reading[read_idx] = iter->reading[iter->read_count];
            }
        }

        /* read records from the remaining file */
        if (iter->read_count) {
            assert(1 == iter->read_count);
            lowest = iter->reading[0];
            do {
                if (!fwrite(iter->node[lowest], iter->uniq->hash_node_octets,
                            1, fp_intermediate))
                {
                    skAppPrintErr("Could not write to tmpfile '%s'",
                                  skTempFileGetName(tmp_idx_intermediate));
                    return -1;
                }
            } while (fread(iter->node[lowest], iter->uniq->hash_node_octets,
                           1, iter->fps[lowest]) == 1);
        }

        /* Close all the temp files that we processed this time. */
        for (i = 0; i < open_count; ++i) {
            fclose(iter->fps[i]);
        }
        /* Delete all the temp files that we opened */
        for (j = tmp_idx_a; j <= tmp_idx_b; ++j) {
            skTempFileRemove(j);
        }

        /* Close the intermediate temp file. */
        if (fp_intermediate) {
            PRINTDEBUG((stderr, "Finished writing '%s'\n",
                        skTempFileGetName(tmp_idx_intermediate)));
            if (EOF == fclose(fp_intermediate)) {
                skAppPrintSyserror("Error closing file '%s'",
                                   skTempFileGetName(tmp_idx_intermediate));
                return -1;
            }
            fp_intermediate = NULL;
        }

        /* Start the next merge with the next input temp file */
        tmp_idx_a = tmp_idx_b + 1;
    }

    return -1;    /* NOT REACHED */
}


/*
 *  status = uniqueDumpHashToTemp(uniq);
 *
 *    Write the entries in the current hash table on 'uniq' to a
 *    temporary file..  The entries are written in sorted order, where
 *    the sort algorithm will depend on whether the user requested
 *    sorted output and whether distinct fields are present.  Return 0
 *    on success, or -1 on failure.
 */
static int uniqueDumpHashToTemp(
    sk_unique_t    *uniq)
{
    HASH_ITER ithash;
    uint8_t *hash_key;
    uint8_t *hash_val;
    FILE *temp_filep;
    char *temp_name;
    int rv = -1; /* return value */

    /* sort the hash entries. */
    if (0 == uniq->sort_output) {
        /* order doesn't matter; we can sort using memcmp() */
        hashlib_sort_entries(uniq->ht);
    } else if (uniq->fi.distinct_num_fields) {
        /* need to sort the key using the skFieldListCompareBuffers
         * function and the distinct fields using memcmp */
        hashlib_sort_entries_usercmp(uniq->ht, uniqueCompareKeysDistinct,
                                     (void*)uniq);
    } else {
        /* sort key using skFieldListCompareBuffers */
        hashlib_sort_entries_usercmp(uniq->ht, uniqueCompareKeys,
                                     (void*)uniq->fi.key_fields);
    }

    /* create an iterator for the hash table */
    ithash = hashlib_create_iterator(uniq->ht);

    /* create temp file */
    temp_filep = skTempFileCreate(&uniq->temp_file_idx, &temp_name);
    if (temp_filep == NULL) {
        return rv;
    }

    PRINTDEBUG((stderr, "Writing %u key/value pairs to '%s'...",
                hashlib_count_entries(uniq->ht), temp_name));

    /* write the sorted entries to disk */
    while (hashlib_iterate(uniq->ht, &ithash, &hash_key, &hash_val)
           != ERR_NOMOREENTRIES)
    {
        if (!fwrite(hash_key, uniq->hash_key_octets, 1, temp_filep)
            || !fwrite(hash_val, uniq->hash_value_octets, 1, temp_filep))
        {
            /* error writing, errno may or may not be set */
            skAppPrintSyserror("Error writing records to temp file '%s'",
                               temp_name);
            PRINTDEBUG((stderr, "ERROR!\n"));
            goto END;
        }
    }

    /* Success so far */
    PRINTDEBUG((stderr, "done\n"));
    rv = 0;

  END:
    /* close the file */
    if (fclose(temp_filep) == EOF) {
        /* error closing file; only report error if status so far
         * is good. */
        if (rv == 0) {
            skAppPrintSyserror("Error closing temp file '%s'", temp_name);
            PRINTDEBUG((stderr, "error on close\n"));
            rv = -1;
        }
    }

    return rv;
}



/* ******************************************************************** */


/*    PRESORTED INPUT */


/* ******************************************************************** */


/* structure for binning records */
/* typedef struct sk_sort_unique_st sk_sort_unique_t; */
/* typedef struct sk_unique_st sk_unique_t; */
struct sk_sort_unique_st {
    sk_uniq_field_info_t    fi;

    int                   (*post_open_fn)(skstream_t *);
    int                   (*read_rec_fn)(skstream_t *, rwRec *);

    /* vector containing the names of files to process */
    sk_vector_t            *files;

    /* where to write temporary files */
    char                   *temp_dir;

    /* the skstream_t or FILE* that are being read */
    void                   *fps[MAX_MERGE_FILES];

    /* array of records, one for each open file */
    rwRec                  *rec;

    /* memory to hold the key for each open file */
    uint8_t                *key_data;

    /* array of keys, one for each open file, holds pointers into
     * 'key_data' */
    uint8_t               **key;

    /* array of indexes specifying which of the 'fps' values still
     * have data to process */
    uint16_t               *reading;

    /* array holding information required to count distinct fields */
    distinct_value_t       *distincts;

    /* current position in the 'files' vector */
    int                     files_position;

    /* flag to detect recursive calls to skPresortedUniqueProcess() */
    unsigned                processing : 1;
};


#define SORTUNIQ_CMP(tfc_node_a, tfc_node_b, tfc_suniq)         \
    skFieldListCompareBuffers((tfc_suniq)->fi.key_fields,       \
                              (tfc_node_a), (tfc_node_b))



/*
 *  status = sortuniqOpenNextInput(uniq, &stream);
 *
 *    Get the name of the next SiLK Flow record file to open, and set
 *    'stream' to that stream.
 *
 *    Return 0 on success.  Return 1 if no more files to open.  Return
 *    -2 if the file cannot be opened due to lack of memory or file
 *    handles.  Return -1 on other error.
 */
static int sortuniqOpenNextInput(
    sk_sort_unique_t   *uniq,
    skstream_t        **out_stream)
{
    skstream_t *stream = NULL;
    const char *filename;
    int rv;

    do {
        rv = skVectorGetValue(&filename, uniq->files,
                              uniq->files_position);
        if (rv != 0) {
            /* no more files */
            return 1;
        }
        ++uniq->files_position;

        errno = 0;
        rv = skStreamOpenSilkFlow(&stream, filename, SK_IO_READ);
        if (rv) {
            if (errno == EMFILE || errno == ENOMEM) {
                rv = -2;
                /* decrement counter to try this file next time */
                --uniq->files_position;
                PRINTDEBUG((stderr, "unable to open '%s': %s\n",
                            filename, strerror(errno)));
            } else {
                skStreamPrintLastErr(stream, rv, &skAppPrintErr);
                rv = -1;
            }
            skStreamDestroy(&stream);
            return rv;
        }

        /* call the user's PostOpenFn if they provided one. */
        if (uniq->post_open_fn) {
            rv = uniq->post_open_fn(stream);
            if (rv == 1 || rv == -1) {
                skStreamDestroy(&stream);
                return rv;
            }
            if (rv != 0) {
                skStreamDestroy(&stream);
            }
        }
    } while (0 != rv);

    *out_stream = stream;
    return 0;
}


/*
 *  ok = sortuniqFillRecordAndKey(uniq, idx);
 *
 *    Read a record from a stream and compute the key for that record.
 *    The stream to read and the destinations for the record and key
 *    are determined by the index 'idx'.
 *
 *    Return 1 if a record was read; 0 otherwise.
 */
static int sortuniqFillRecordAndKey(
    sk_sort_unique_t  *uniq,
    uint16_t           idx)
{
    int rv;

    rv = uniq->read_rec_fn((skstream_t*)uniq->fps[idx], &uniq->rec[idx]);
    if (rv) {
        if (rv != SKSTREAM_ERR_EOF) {
            skStreamPrintLastErr((skstream_t*)uniq->fps[idx], rv,
                                 &skAppPrintErr);
        }
        return 0;
    }

    skFieldListRecToBinary(uniq->fi.key_fields, &uniq->rec[idx],
                           uniq->key[idx]);
    return 1;
}


/*
 *  status = sortuniqAddNodeToTemp(uniq, fp, key_buffer, value_buffer);
 *
 *    Write the values from 'key_buffer', 'value_buffer', and any
 *    distinct fields (located on the 'uniq' object) to the file
 *    handle 'fp'.  Return 0 on success, or -1 on failure.
 *
 *    Data is written as follows:
 *
 *      the key_buffer
 *      the value_buffer
 *      for each distinct field:
 *          number of distinct values
 *          distinct value 1, distinct value 2, ...
 */
static int sortuniqAddNodeToTemp(
    const sk_sort_unique_t *uniq,
    FILE                   *fp,
    const uint8_t          *key_buffer,
    const uint8_t          *value_buffer)
{
    const distinct_value_t *dist;
    sk_bitmap_iter_t b_iter;
    skIPTreeIterator_t ipt_iter;
    hashset_iter h_iter;
    uint8_t *hash_key;
    uint16_t i;
    uint32_t tmp32;
    uint16_t val16;
    uint8_t val8;

    /* write keys and values */
    if (!fwrite(key_buffer, uniq->fi.key_octets, 1, fp)
        || !fwrite(value_buffer, uniq->fi.value_octets, 1, fp))
    {
        return -1;
    }

    /* handle all the distinct fields */
    dist = uniq->distincts;
    for (i = 0; i < uniq->fi.distinct_num_fields; ++i, ++dist) {
        /* write the count */
        if (!fwrite(&dist->dv_count, sizeof(uint64_t), 1, fp)) {
            return -1;
        }
        /* write each value */
        switch (dist->dv_type) {
          case DISTINCT_COUNT:
          case DISTINCT_CACHE:
          case DISTINCT_CACHE_ALLOC:
            skAbortBadCase(dist->dv_type);

          case DISTINCT_BITMAP:
            assert(skBitmapGetHighCount(dist->dv_v.dv_bitmap)
                   == dist->dv_count);
            skBitmapIteratorBind(dist->dv_v.dv_bitmap, &b_iter);
            if (1 == dist->dv_octets) {
                while (SK_ITERATOR_OK==skBitmapIteratorNext(&b_iter, &tmp32)) {
                    val8 = (uint8_t)tmp32;
                    if (!fwrite(&val8, sizeof(uint8_t), 1, fp)) {
                        return -1;
                    }
                }
            } else {
                assert(2 == dist->dv_octets);
                while (SK_ITERATOR_OK==skBitmapIteratorNext(&b_iter, &tmp32)) {
                    val16 = (uint16_t)tmp32;
                    if (!fwrite(&val16, sizeof(uint16_t), 1, fp)) {
                        return -1;
                    }
                }
            }
            break;

          case DISTINCT_IPTREE:
            assert(skIPTreeCountIPs(dist->dv_v.dv_iptree)
                   == dist->dv_count);
            skIPTreeIteratorBind(&ipt_iter, dist->dv_v.dv_iptree);
            while (SK_ITERATOR_OK == skIPTreeIteratorNext(&tmp32, &ipt_iter)) {
                if (!fwrite(&tmp32, sizeof(uint32_t), 1, fp)) {
                    return -1;
                }
            }
            break;

          case DISTINCT_HASHSET_FILES:
          case DISTINCT_HASHSET:
            h_iter = hashset_create_iterator(dist->dv_v.dv_hashset);
            while (OK == hashset_iterate(dist->dv_v.dv_hashset,
                                         &h_iter, &hash_key))
            {
                if (!fwrite(hash_key, dist->dv_octets, 1, fp)) {
                    return -1;
                }
            }
            break;
        }
    }

    return 0;
}


/*
 *  status = sortuniqMergeDataFromTemp(uniq, fp, value_buffer);
 *
 *    Read everything---except the key---which was written by the call
 *    to sortuniqAddNodeToTemp(), and merge those value into the
 *    current values for the 'value_buffer' and the distinct fields
 *    (located on the 'uniq' object).  Return 0 on success, or -1 on
 *    failure.
 */
static int sortuniqMergeDataFromTemp(
    sk_sort_unique_t       *uniq,
    FILE                   *fp,
    uint8_t                *value_buffer)
{
    distinct_value_t *dist;
    uint8_t buf[4096];
    uint64_t count;
    uint16_t i;
    int rv;

    /* read the value and merge it into the current value */
    if (!fread(buf, uniq->fi.value_octets, 1, fp)) {
        return -1;
    }
    skFieldListMergeBuffers(uniq->fi.value_fields, value_buffer, buf);

    /* handle the distinct fields */
    dist = uniq->distincts;
    for (i = 0; i < uniq->fi.distinct_num_fields; ++i, ++dist) {
        /* read the count */
        if (!fread(&count, sizeof(uint64_t), 1, fp)) {
            return -1;
        }
        /* read each value and add to the distinct object */
        switch (dist->dv_type) {
          case DISTINCT_COUNT:
          case DISTINCT_CACHE:
          case DISTINCT_CACHE_ALLOC:
            skAbortBadCase(dist->dv_type);

          case DISTINCT_BITMAP:
            if (1 == dist->dv_octets) {
                while (count > 0) {
                    --count;
                    if (!fread(buf, sizeof(uint8_t), 1, fp)) {
                        return -1;
                    }
                    skBitmapSetBit(dist->dv_v.dv_bitmap, *buf);
                }
            } else {
                assert(2 == dist->dv_octets);
                while (count > 0) {
                    --count;
                    if (!fread(buf, sizeof(uint16_t), 1, fp)) {
                        return -1;
                    }
                    skBitmapSetBit(dist->dv_v.dv_bitmap,
                                   *(uint16_t*)buf);
                }
            }
            dist->dv_count = skBitmapGetHighCount(dist->dv_v.dv_bitmap);
            break;

          case DISTINCT_IPTREE:
            while (count > 0) {
                --count;
                if (!fread(buf, sizeof(uint32_t), 1, fp)) {
                    return -1;
                }
                skIPTreeAddAddress(dist->dv_v.dv_iptree, *(uint32_t*)buf);
            }
            dist->dv_count = skIPTreeCountIPs(dist->dv_v.dv_iptree);
            break;

          case DISTINCT_HASHSET_FILES:
          case DISTINCT_HASHSET:
            while (count > 0) {
                --count;
                if (!fread(buf, dist->dv_octets, 1, fp)) {
                    return -1;
                }
                rv = hashset_insert(dist->dv_v.dv_hashset, buf);
                switch (rv) {
                  case OK:
                    ++dist->dv_count;
                    break;
                  case OK_DUPLICATE:
                    break;
                  default:
                    return -1;
                }
            }
            break;
        }
    }

    return 0;
}


/*
 *  open_count = sortuniqOpenAllTempFiles(uniq, max_merge, temp_file_idx);
 *
 *    Return once there is a file handle to every existing temporary
 *    file that was created when reading the SiLK Flow input files.
 *    'temp_file_idx' is the index of the last temp file.
 *
 *    The function attempts to open all the temp files, but no more
 *    than 'max_merge' files.  If all the temp files can be opened,
 *    the function returns.
 *
 *    However, if temporary files remain, the opened temp files are
 *    merged into a new temporary file, and then another attempt is
 *    made to open all temp files.  This repeats until there is a
 *    handle to every temp file.
 */
static int sortuniqOpenAllTempFiles(
    sk_sort_unique_t       *uniq,
    uint16_t                max_merge,
    int                     temp_file_idx)
{
    distinct_value_t *dist;
    uint8_t buf[4096];
    uint64_t count;
    size_t sz;
    uint16_t open_count;
    uint16_t read_count;
    uint16_t tmp_idx_a;
    uint16_t tmp_idx_b;
    uint16_t lowest;
    uint16_t read_idx;
    uint16_t i;
    uint16_t j;
    FILE *fp_out;

    /* index at which to start the merge */
    tmp_idx_a = 0;

    /* This loop repeats as long as we haven't opened all of the temp
     * files generated while reading the flows. */
    for (;;) {
        /* index at which to stop the merge */
        tmp_idx_b = temp_file_idx;

        PRINTDEBUG((stderr, "opening tmp files #%d through #%d\n",
                    tmp_idx_a, tmp_idx_b));

        /* open an intermediate temp file.  The merge-sort will have
         * to write nodes here if there are not enough file handles
         * available to open all the temporary files we wrote while
         * reading the data. */
        fp_out = skTempFileCreate(&temp_file_idx,  NULL);
        if (fp_out == NULL) {
            return -1;
        }

        /* count number of files we open */
        open_count = 0;

        /* Attempt to open up to max_merge, though we an open may fail
         * due to lack of resources (EMFILE or ENOMEM) */
        for (j = tmp_idx_a; j <= tmp_idx_b; ++j) {
            uniq->fps[open_count] = skTempFileOpen(j);
            if (uniq->fps[open_count] == NULL) {
                if ((open_count > 0)
                    && ((errno == EMFILE) || (errno == ENOMEM)))
                {
                    /* Blast!  We can't open any more temp files.  So,
                     * we rewind by one to catch this one the next
                     * time around. */
                    tmp_idx_b = j - 1;
                    PRINTDEBUG((stderr, ("EMFILE limit hit--"
                                         "merging %d through %d to %d\n"),
                                tmp_idx_a, tmp_idx_b, temp_file_idx));
                    break;
                } else {
                    PRINTDEBUG((stderr, "couldn't open tmp file %s\n",
                                skTempFileGetName(j)));
                    return -1;
                }
            }

            ++open_count;
            if (open_count == max_merge) {
                /* We've reached the limit for this pass.  Set
                 * tmp_idx_b to the file we just opened. */
                tmp_idx_b = j;
                PRINTDEBUG((stderr, ("MAX_MERGE_FILES limit hit--"
                                     "merging %d through %d to %d\n"),
                            tmp_idx_a, tmp_idx_b, temp_file_idx));
                break;
            }
        }

        /* Here, we check to see if we've opened all temp files.  If
         * so, set a flag so we write data to final destination and
         * break out of the loop after we're done. */
        if (tmp_idx_b == temp_file_idx - 1) {
            /* no longer need the intermediate temp file */
            PRINTDEBUG((stderr, "opened all (remaining) temp files\n"));
            fclose(fp_out);
            fp_out = NULL;
            return open_count;
        }

        /* this is the number of files with data to read */
        read_count = 0;

        /* read the first key from each file into the work buffer */
        for (i = 0; i < open_count; ++i) {
            if (!fread(uniq->key[i], uniq->fi.key_octets,
                       1, (FILE*)uniq->fps[i]))
            {
                PRINTDEBUG((stderr, "couldn't get first record from file %s\n",
                            skTempFileGetName(tmp_idx_a + i)));
                continue;
            }
            /* 'reading' holds the indexes of the files from which we
             * are reading records */
            uniq->reading[read_count] = i;
            ++read_count;
        }

        PRINTDEBUG((stderr,
                    "open_count: %" PRIu16 "; read_count: %" PRIu16 "\n",
                    open_count, read_count));

        /* exit this while() once all records for all opened files
         * have been read */
        while (read_count) {
            /* set "lowest" to first file with data */
            lowest = uniq->reading[0];
            read_idx = 0;

            /* compare "lowest" with every other file to find the
             * actual lowest */
            for (i = 1; i < read_count; ++i) {
                /* If the temp file's key is lower than the current
                 * lowest... */
                if (0 > SORTUNIQ_CMP(uniq->key[uniq->reading[i]],
                                     uniq->key[lowest], uniq))
                {
                    /* ...record the index of this temp file as having
                     * the lowest key */
                    lowest = uniq->reading[i];
                    read_idx = i;
                }
            }

            /* write the key */
            if (!fwrite(&uniq->key[lowest], uniq->fi.key_octets, 1, fp_out)) {
                return -1;
            }
            /* read and write the values and the distincts */
            if (!fread(buf, uniq->fi.value_octets, 1, uniq->fps[lowest])
                || !fwrite(buf, uniq->fi.value_octets, 1, fp_out))
            {
                return -1;
            }
            for (i = 0; i < uniq->fi.distinct_num_fields; ++i) {
                dist = &uniq->distincts[i];
                if (!fread(&count, sizeof(uint64_t), 1, uniq->fps[lowest])
                    || !fwrite(&count, sizeof(uint64_t), 1, fp_out))
                {
                    return -1;
                }
                count *= dist->dv_octets;
                while (count) {
                    sz = ((count < sizeof(buf)) ? count : sizeof(buf));
                    if (!fread(buf, sz, 1, uniq->fps[lowest])
                        || !fwrite(buf, sz, 1, fp_out))
                    {
                        return -1;
                    }
                    count -= sz;
                }
            }

            /* replace the key we just processed */
            if (!fread(uniq->key[lowest], uniq->fi.key_octets,
                       1, uniq->fps[lowest]))
            {
                /* no more data for this file; replace file with last
                 * file in 'reading' array */
                --read_count;
                uniq->reading[read_idx] = uniq->reading[read_count];
            }
        }

        /* Close all the temp files that we processed this time. */
        for (i = 0; i < open_count; ++i) {
            fclose((FILE*)uniq->fps[i]);
        }
        /* Delete all the temp files that we opened */
        for (j = tmp_idx_a; j <= tmp_idx_b; ++j) {
            skTempFileRemove(j);
        }

        /* Close the intermediate temp file. */
        if (fp_out) {
            if (EOF == fclose(fp_out)) {
                skAppPrintSyserror("Error closing file '%s'",
                                   skTempFileGetName(temp_file_idx));
                return -1;
            }
            fp_out = NULL;
        }

        /* start the next merge with the next temp file */
        tmp_idx_a = tmp_idx_b + 1;
    }
}


/*
 */
int skPresortedUniqueCreate(
    sk_sort_unique_t  **uniq)
{
    *uniq = calloc(1, sizeof(sk_sort_unique_t));
    if (NULL == *uniq) {
        return -1;
    }
    (*uniq)->files = skVectorNew(sizeof(char*));
    if (NULL == (*uniq)->files) {
        free(*uniq);
        return -1;
    }

    (*uniq)->read_rec_fn = &skStreamReadRecord;

    return 0;

}


void skPresortedUniqueDestroy(
    sk_sort_unique_t  **uniq)
{
    sk_sort_unique_t *u;
    char *filename;
    size_t i;

    if (NULL == uniq || NULL == *uniq) {
        return;
    }

    u = *uniq;
    *uniq = NULL;

    skTempFileTeardown();
    if (u->temp_dir) {
        free(u->temp_dir);
    }
    if (u->files) {
        for (i = 0; 0 == skVectorGetValue(&filename, u->files, i); ++i) {
            free(filename);
        }
        skVectorDestroy(u->files);
    }

    if (u->rec) {
        free(u->rec);
    }
    if (u->key) {
        free(u->key);
    }
    if (u->key_data) {
        free(u->key_data);
    }
    if (u->reading) {
        free(u->reading);
    }
    if (u->distincts) {
        uniqDistinctFree(&u->fi, u->distincts);
    }

    free(u);
}


int skPresortedUniqueAddInputFile(
    sk_sort_unique_t               *uniq,
    const char                     *filename)
{
    char *copy;

    assert(uniq);
    assert(filename);

    if (uniq->processing) {
        return -1;
    }

    copy = strdup(filename);
    if (NULL == copy) {
        return -1;
    }
    if (skVectorAppendValue(uniq->files, &copy)) {
        free(copy);
        return -1;
    }

    return 0;
}


void skPresortedUniqueSetTempDirectory(
    sk_sort_unique_t   *uniq,
    const char         *temp_dir)
{
    if (uniq->temp_dir) {
        free(uniq->temp_dir);
        uniq->temp_dir = NULL;
    }
    if (temp_dir) {
        uniq->temp_dir = strdup(temp_dir);
    }
}


int skPresortedUniqueSetPostOpenFn(
    sk_sort_unique_t   *uniq,
    int               (*stream_post_open)(skstream_t *))
{
    assert(uniq);

    if (uniq->processing) {
        return -1;
    }

    uniq->post_open_fn = stream_post_open;
    return 0;
}


int skPresortedUniqueSetReadFn(
    sk_sort_unique_t   *uniq,
    int               (*stream_read)(skstream_t *, rwRec *))
{
    assert(uniq);

    if (uniq->processing) {
        return -1;
    }

    if (NULL == stream_read) {
        uniq->read_rec_fn = &skStreamReadRecord;
    } else {
        uniq->read_rec_fn = stream_read;
    }
    return 0;
}


int skPresortedUniqueSetFields(
    sk_sort_unique_t               *uniq,
    const sk_fieldlist_t           *key_fields,
    const sk_fieldlist_t           *distinct_fields,
    const sk_fieldlist_t           *agg_value_fields)
{
    assert(uniq);

    if (uniq->processing) {
        return -1;
    }

    memset(&uniq->fi, 0, sizeof(sk_uniq_field_info_t));
    uniq->fi.key_fields = key_fields;
    uniq->fi.value_fields = agg_value_fields;
    uniq->fi.distinct_fields = distinct_fields;

    return 0;
}


int skPresortedUniqueProcess(
    sk_sort_unique_t               *uniq,
    sk_unique_output_fn_t           output_fn,
    void                           *callback_data)
{
    uint16_t max_merge = MAX_MERGE_FILES;
    uint16_t step = 1;
    uint16_t lowest;
    uint16_t read_idx;
    FILE *fp_intermediate;
    int temp_file_idx = -1;
    uint16_t open_count;
    uint16_t read_count;
    uint8_t cached_key[HASHLIB_MAX_KEY_WIDTH];
    uint8_t distinct_buffer[HASHLIB_MAX_KEY_WIDTH];
    uint8_t merged_values[HASHLIB_MAX_VALUE_WIDTH];
    uint16_t i;
    uint16_t j;
    int no_more_inputs = 0;
    int rv = -1;

    assert(uniq);
    assert(output_fn);

    /* no recursive processing */
    if (uniq->processing) {
        return -1;
    }
    uniq->processing = 1;

    if (uniqCheckFields(&uniq->fi, NULL)) {
        return -1;
    }

    if (skTempFileInitialize(uniq->temp_dir, NULL)) {
        /* FIXME.  ERR_FN("Unable to initialize temporary file space");*/
        return -1;
    }

    /* set up distinct fields */
    if (uniq->fi.distinct_num_fields) {
        if (uniqDistinctAlloc(&uniq->fi, &uniq->distincts,
                              DISTINCT_KEYS_RANDOM))
        {
            return -1;
        }
    }

    /* This outer loop is over the SiLK Flow input files and it
     * repeats as long as we haven't read all the input files */
    do {
        /* open an intermediate temp file that we will use if there
         * are not enough file handles available to open all the input
         * files. */
        fp_intermediate = skTempFileCreate(&temp_file_idx, NULL);
        if (fp_intermediate == NULL) {
            rv = -1;
            goto END;
        }

        /* Attempt to open up to max_merge files, though an open may
         * fail due to lack of resources (EMFILE or ENOMEM) */
        for (open_count = 0; open_count < max_merge; ++open_count) {
            rv = sortuniqOpenNextInput(uniq,
                                       (skstream_t**)&uniq->fps[open_count]);
            if (rv != 0) {
                break;
            }
        }
        switch (rv) {
          case 1:
            /* successfully opened all (remaining) input files */
            PRINTDEBUG((stderr, "opened all (remaining) input files\n"));
            no_more_inputs = 1;
            if (temp_file_idx == 0) {
                /* we opened all the input files in a single pass.  we
                 * no longer need the intermediate temp file */
                fclose(fp_intermediate);
                fp_intermediate = NULL;
                temp_file_idx = -1;
            }
            break;
          case -1:
            /* unexpected error opening a file */
            return -1;
          case -2:
            /* ran out of memory or file descriptors */
            PRINTDEBUG((stderr, ("unable to open all inputs---"
                                 "out of memory or file handles\n")));
            break;
          case 0:
            if (open_count == max_merge) {
                /* ran out of pointers for this run */
                PRINTDEBUG((stderr, ("unable to open all inputs---"
                                     "max_merge (%d) limit reached\n"),
                            max_merge));
                break;
            }
            /* no other way that rv == 0 */
            PRINTDEBUG((stderr, "rv == 0 but open_count == %d;"
                        " max_merge == %d.  Abort.\n",
                        open_count, max_merge));
            skAbort();
          default:
            /* unexpected error */
            PRINTDEBUG((stderr, "got unexpected rv value = %d\n", rv));
            skAbortBadCase(rv);
        }

        /* if this is the first iteration, allocate space for the
         * records and keys we will use while processing the files */
        if (NULL == uniq->rec) {
            uint8_t *n;
            max_merge = open_count;
            uniq->rec = malloc(max_merge * sizeof(rwRec));
            if (NULL == uniq->rec) {
                return -1;
            }
            uniq->key_data = malloc(max_merge * uniq->fi.key_octets);
            if (NULL == uniq->key_data) {
                return -1;
            }
            uniq->key = malloc(max_merge * sizeof(uint8_t*));
            if (NULL == uniq->key) {
                return -1;
            }
            for (i = 0, n = uniq->key_data;
                 i < max_merge;
                 ++i, n += uniq->fi.key_octets)
            {
                uniq->key[i] = n;
            }
            uniq->reading = malloc(max_merge * sizeof(uint16_t));
            if (NULL == uniq->reading) {
                return -1;
            }
        }

        /* this is the number of files with data to read */
        read_count = 0;

        /* Read the first record from each file into the work buffer */
        for (i = 0; i < open_count; ++i) {
            if (sortuniqFillRecordAndKey(uniq, i)) {
                /* 'reading' holds the indexes of the files from which
                 * we are reading records */
                uniq->reading[read_count] = i;
                ++read_count;
            }
        }

        PRINTDEBUG((stderr,
                    "open_count: %" PRIu16 "; read_count: %" PRIu16 "\n",
                    open_count, read_count));

        /* exit this while() once all records for all opened files
         * have been read */
        while (read_count) {
            /* set "lowest" to first file with data. use 'read_idx'
             * to remember this position in the 'reading' array */
            lowest = uniq->reading[0];
            read_idx = 0;

            /* compare 'lowest' with every other file to find actual
             * lowest */
            for (i = 1; i < read_count; ++i) {
                /* If the file's record is lower than the current
                 * lowest... */
                if (0 > SORTUNIQ_CMP(uniq->key[uniq->reading[i]],
                                     uniq->key[lowest], uniq))
                {
                    /* ...record the index of this file as having the
                     * lowest record */
                    lowest = uniq->reading[i];
                    read_idx = i;
                }
            }

            /* cache this low key, initialize the values, then add the
             * values and the distincts from this record */
            memcpy(cached_key, uniq->key[lowest], uniq->fi.key_octets);

            skFieldListInitializeBuffer(uniq->fi.value_fields,
                                        merged_values);
            skFieldListAddRecToBuffer(uniq->fi.value_fields, &uniq->rec[lowest],
                                      merged_values);
            if (uniq->fi.distinct_num_fields) {
                skFieldListRecToBinary(uniq->fi.distinct_fields,
                                       &uniq->rec[lowest], distinct_buffer);
                uniqDistinctReset(&uniq->fi, uniq->distincts,
                                  distinct_buffer);
            }

            /* replace the record and key we just processed */
            if (!sortuniqFillRecordAndKey(uniq, lowest)) {
                /* no more data for this file; replace file with last
                 * file in 'reading' array */
                --read_count;
                uniq->reading[read_idx] = uniq->reading[read_count];
            }

            /* process all entries in all open input files that match
             * the current key */
            for (i = 0; i < read_count; i += step) {
                step = 1;
                lowest = uniq->reading[i];
                while (0==SORTUNIQ_CMP(cached_key, uniq->key[lowest], uniq))
                {
                    /* keys are the same: add this record's values */
                    skFieldListAddRecToBuffer(uniq->fi.value_fields,
                                              &uniq->rec[lowest],
                                              merged_values);
                    if (uniq->fi.distinct_num_fields) {
                        skFieldListRecToBinary(uniq->fi.distinct_fields,
                                               &uniq->rec[lowest],
                                               distinct_buffer);
                        uniqDistinctIncrement(&uniq->fi, uniq->distincts,
                                              distinct_buffer);
                    }

                    /* replace the record */
                    if (!sortuniqFillRecordAndKey(uniq, lowest)) {
                        /* no more data for this file; replace file
                         * with last file, do not increment 'i' */
                        --read_count;
                        uniq->reading[i] = uniq->reading[read_count];
                        step = 0;
                        break;
                    }
                }
            }

            /* output this key and its values.  If we opened all input
             * files, call the output callback.  Else, write the key,
             * values, and distincts to a temp file.  The temp files
             * will be merged after all input files have been
             * processed. */
            if (fp_intermediate) {
                sortuniqAddNodeToTemp(uniq, fp_intermediate,
                                      cached_key, merged_values);
            } else {
                if (uniq->fi.distinct_num_fields) {
                    uniqDistinctSetOutputBuf(&uniq->fi, uniq->distincts,
                                             distinct_buffer);
                }
                rv = output_fn(cached_key, distinct_buffer, merged_values,
                               callback_data);
                if (rv != 0) {
                    goto END;
                }
            }
        } /* inner-while */

        /* Close the input files that we processed this time. */
        for (j = 0; j < open_count; ++j) {
            skStreamDestroy((skstream_t**)&uniq->fps[j]);
        }

        /* Close the intermediate temp file. */
        if (fp_intermediate) {
            PRINTDEBUG((stderr, "Finished writing '%s'\n",
                        skTempFileGetName(temp_file_idx)));
            if (EOF == fclose(fp_intermediate)) {
                skAppPrintErr("Error closing file %s: %s",
                              skTempFileGetName(temp_file_idx),
                              strerror(errno));
                return -1;
            }
            fp_intermediate = NULL;
        }

    } while (!no_more_inputs);

    /* If any temporary files were written, we now have to merge them.
     * Otherwise, we didn't write any temporary files, and we are
     * done. */
    if (-1 == temp_file_idx) {
        goto END;
    }

    /* we are finished processing records; free the 'rec' array */
    free(uniq->rec);
    uniq->rec = NULL;

    /* open all the temporary files we created */
    open_count = sortuniqOpenAllTempFiles(uniq, max_merge, temp_file_idx);
    if (open_count == (uint16_t)-1) {
        goto END;
    }

    /* read the first key from each file into the work buffer */
    for (i = 0; i < open_count; ++i) {
        if (!fread(uniq->key[i], uniq->fi.key_octets, 1, (FILE*)uniq->fps[i])){
            PRINTDEBUG((stderr, "couldn't get first record from file\n"));
            continue;
        }
        /* 'uniq->reading' holds the indexes of the files from which we
         * are reading records */
        uniq->reading[read_count] = i;
        ++read_count;
    }

    PRINTDEBUG((stderr,
                "open_count: %" PRIu16 "; read_count: %" PRIu16 "\n",
                open_count, read_count));

    /* exit this while() once all records for all opened files
     * have been read */
    while (read_count) {
        /* set "lowest" to first file with data */
        lowest = uniq->reading[0];
        read_idx = 0;

        /* compare "lowest" with every other file to find the
         * actual lowest */
        for (i = 1; i < read_count; ++i) {
            /* If the temp file's key is lower than the current
             * lowest... */
            if (0 > SORTUNIQ_CMP(uniq->key[uniq->reading[i]],
                                 uniq->key[lowest], uniq))
            {
                /* ...record the index of this temp file as having
                 * the lowest key */
                lowest = uniq->reading[i];
                read_idx = i;
            }
        }

        /* cache this low key, initialize the values and distincts */
        memcpy(cached_key, uniq->key[lowest], uniq->fi.key_octets);
        skFieldListInitializeBuffer(uniq->fi.value_fields, merged_values);
        if (uniq->fi.distinct_num_fields) {
            uniqDistinctReset(&uniq->fi, uniq->distincts, NULL);
        }

        /* read data from the file */
        sortuniqMergeDataFromTemp(uniq, uniq->fps[lowest], merged_values);

        /* replace the node we just processed */
        if (!fread(uniq->key[lowest], uniq->fi.key_octets,
                   1, (FILE*)uniq->fps[lowest]))
        {
            /* no more data for this file; replace file with last
             * file in 'reading' array */
            --read_count;
            uniq->reading[read_idx] = uniq->reading[read_count];
        }

        /* process all entries in all open input files that match
         * the current key */
        for (i = 0; i < read_count; i += step) {
            step = 1;
            lowest = uniq->reading[i];
            while (0 == SORTUNIQ_CMP(cached_key, uniq->key[lowest], uniq)) {
                /* keys are the same: add this record's values */
                sortuniqMergeDataFromTemp(uniq, uniq->fps[lowest],
                                          merged_values);
                /* replace the key */
                if (!fread(uniq->key[lowest], uniq->fi.key_octets,
                           1, (FILE*)uniq->fps[lowest]))
                {
                    /* no more data for this file; replace file
                     * with last file, do not increment 'i' */
                    --read_count;
                    uniq->reading[i] = uniq->reading[read_count];
                    step = 0;
                    break;
                }
            }
        }

        /* output this key and its values */
        if (uniq->fi.distinct_num_fields) {
            uniqDistinctSetOutputBuf(&uniq->fi, uniq->distincts,
                                     distinct_buffer);
        }
        rv = output_fn(cached_key, distinct_buffer, merged_values,
                       callback_data);
        if (rv != 0) {
            goto END;
        }
    }

    /* Close all the temp files that we processed this time. */
    for (i = 0; i < open_count; ++i) {
        fclose((FILE*)uniq->fps[i]);
    }

  END:

    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
