/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**
**  num2doc.c
**
**  Suresh L Konda
**  7/31/2002
**      "filter" to convert numeric ip to dotted ip. The default field
**      delimiter is '|' in deference to our internal default.  The
**      default field is 1 (numbering starts at 1).  Changes can be
**      provided via options --ip-fields=<range> and
**      --delimiter=<char>.
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: num2dot.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* where to send usage output */
#define USAGE_FH stdout

/* max fields (columns) we support in each line of the output. */
#define MAX_FIELD_COUNT         1024

/* max length of input line */
#define MAX_LINE_LENGTH         2048

/* default field(s) to convert. this gets passed to parseIPFields() if
 * the user does not provide an --ip-fields value */
#define N2D_DEFAULT_FIELDS      "1"

/* the type of each field on a line of input (from rwresolve.c) */
typedef enum ip_field_type_en {
    /* the final field to handle; nothing but text remains */
    N2D_TEXT_FINAL,
    /* field is surrounded by IP fields */
    N2D_TEXT_SINGLE,
    /* field begins a list of contiguous text fields */
    N2D_TEXT_OPEN,
    /* field is in middle of contiguous text fields */
    N2D_TEXT_CONTINUE,
    /* field closes a list of contiguous text fields */
    N2D_TEXT_CLOSE,
    /* field contains an IP to convert */
    N2D_CONVERT
} ip_field_type_t;



/* LOCAL VARIABLES */

/* what each field of the input contains */
static ip_field_type_t ip_fields[MAX_FIELD_COUNT];

/* number of expected "parts" on a line when all contiguous non-IP
 * fields become a single "part" */
static int line_part_count = 0;

/* input stream (stdin) */
static skstream_t *in_stream = NULL;

/* where to send output (stdout) */
static FILE *outf;

/* width of IP columns */
static const int column_width = 15;

/* delimiter between the fields */
static char delimiter = '|';



/* OPTIONS SETUP */

typedef enum {
    OPT_IP_FIELDS, OPT_DELIMITER
} appOptionsEnum;

static struct option appOptions[] = {
    {"ip-fields", REQUIRED_ARG, 0, OPT_IP_FIELDS},
    {"delimiter", REQUIRED_ARG, 0, OPT_DELIMITER},
    {0,0,0,0}     /* sentinel entry */
};

static const char *appHelp[] = {
    ("Convert numbers to dotted-decimal IP addresses in these\n"
     "\tinput columns.  Column numbers begin with 1."
     " Def. " N2D_DEFAULT_FIELDS),
    "Specify the delimiter to expect between fields. Def. '|'",
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static int  parseIPFields(const char *b);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                            \
    ("[SWITCHES]\n"                                                          \
     "\tRead pipe (|) delimited text from the standard input, convert\n"     \
     "\tinteger values in the specified column(s) (default first column)\n"  \
     "\tto dotted-decimal IP addresss, and print result to standard output.\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    skStreamDestroy(&in_stream);

    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    int arg_index;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    outf = stdout;

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        skAppUsage();           /* never returns */
    }

    /* check for extra arguments */
    if (arg_index != argc) {
        skAppPrintErr("Unexpected argument '%s'", argv[arg_index]);
        skAppUsage();             /* never returns */
    }

    /* set the default fields if none specified */
    if (0 == line_part_count) {
        parseIPFields(N2D_DEFAULT_FIELDS);
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    /* open input */
    if ((rv = skStreamCreate(&in_stream, SK_IO_READ, SK_CONTENT_TEXT))
        || (rv = skStreamBind(in_stream, "stdin"))
        || (rv = skStreamOpen(in_stream)))
    {
        skStreamPrintLastErr(in_stream, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    return;                     /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    switch (opt_index) {
      case OPT_IP_FIELDS:
        /* ip fields */
        parseIPFields(opt_arg);          /* never returns on error */
        break;

      case OPT_DELIMITER:
        /* delimiter */
        delimiter = *opt_arg;
        if ('\0' == delimiter) {
            skAppPrintErr("Empty string not valid argument for --delimiter");
            return 1;
        }
        break;

      default:
        skAbortBadCase(opt_index);
    }
    return 0;                   /* OK */
}


/* comparison function used to sort the user's field list */
static int compr32(const void *a, const void* b)
{
    const uint32_t a32 = *((uint32_t*)a);
    const uint32_t b32 = *((uint32_t*)b);

    if (a32 < b32) {
        return -1;
    }
    if (a32 > b32) {
        return 1;
    }
    return 0;
}


/*
 *  is_ok = parseIPFields(arg);
 *
 *    Given a comma-separated list of numbers and/or ranges, set the
 *    appropriate flags in the 'ip_fields' array.
 */
static int parseIPFields(const char *arg)
{
    uint32_t *list;
    uint32_t count;
    uint32_t i;
    uint32_t j;
    int rv;

    /* have we been here before? */
    if (line_part_count) {
        skAppPrintErr("Invalid %s: Option given multiple times",
                      appOptions[OPT_IP_FIELDS].name);
        return 1;
    }

    /* parse the values into an array */
    rv = skStringParseNumberList(&list, &count, arg, 1, MAX_FIELD_COUNT, 0);
    if (rv) {
        skAppPrintErr("Invalid %s '%s': %s",
                      appOptions[OPT_IP_FIELDS].name, arg,
                      skStringParseStrerror(rv));
        return 1;
    }

    /* the list of fields must be in ascending order and contain no
     * duplicates */
    skQSort(list, count, sizeof(uint32_t), &compr32);
    for (i = 0, j = 1; j < count; ++j) {
        if (list[i] < list[j]) {
            ++i;
            if (i != j) {
                list[i] = list[j];
            }
        }
    }
    count = i + 1;

    /* initialize values */
    memset(ip_fields, 0, sizeof(ip_fields));
    i = 0;
    j = 0;

    /* handle the first field */
    if (i + 1 == list[j]) {
        /* this field is a number to convert to an IP */
        ip_fields[i] = N2D_CONVERT;
        ++line_part_count;
        ++j;
        if (j == count) {
            ip_fields[i + 1] = N2D_TEXT_FINAL;
            ++line_part_count;
            goto END;
        }
    } else {
        ip_fields[i] = N2D_TEXT_OPEN;
        ++line_part_count;
    }
    ++i;

    /* handle all remaining fields */
    for ( ; i < MAX_FIELD_COUNT; ++i) {
        if (i + 1 == list[j]) {
            /* this field is an IP to lookup */

            /* properly close previous field */
            switch (ip_fields[i - 1]) {
              case N2D_CONVERT:
                break;

              case N2D_TEXT_OPEN:
                ip_fields[i - 1] = N2D_TEXT_SINGLE;
                break;

              case N2D_TEXT_CONTINUE:
                ip_fields[i - 1] = N2D_TEXT_CLOSE;
                break;

              default:
                skAbortBadCase(ip_fields[i-1]);
            }

            ip_fields[i] = N2D_CONVERT;
            ++line_part_count;
            ++j;
            if (j == count) {
                ip_fields[i + 1] = N2D_TEXT_FINAL;
                ++line_part_count;
                break;
            }
        } else {
            /* this is a text field. set its type based on the
             * previous field */
            switch (ip_fields[i - 1]) {
              case N2D_CONVERT:
                ip_fields[i] = N2D_TEXT_OPEN;
                ++line_part_count;
                break;

              case N2D_TEXT_OPEN:
              case N2D_TEXT_CONTINUE:
                ip_fields[i] = N2D_TEXT_CONTINUE;
                break;

              default:
                skAbortBadCase(ip_fields[i-1]);
            }
        }
    }

  END:
    free(list);
    return 0;                   /* OK */
}


int main(int argc, char **argv)
{
    char line_buf[MAX_LINE_LENGTH];
    char *cp, *sp, *ep;
    int lc = 0;
    uint32_t num;
    uint32_t field;
    int rv;

    appSetup(argc, argv);                       /* never returns */

    /* read until end of file */
    while ((rv = skStreamGetLine(in_stream, line_buf, sizeof(line_buf), &lc))
           != SKSTREAM_ERR_EOF)
    {
        switch (rv) {
          case SKSTREAM_OK:
            /* good, we got our line */
            break;
          case SKSTREAM_ERR_LONG_LINE:
            /* bad: line was longer than sizeof(line_buf) */
            skAppPrintErr("Input line %d too long. ignored",
                          lc);
            continue;
          default:
            /* unexpected error */
            skStreamPrintLastErr(in_stream, rv, &skAppPrintErr);
            goto END;
        }

        /* process each field in the line */
        for (cp = line_buf, sp = cp, field = 0; *cp != '\0'; ++field) {
            switch (ip_fields[field]) {
              case N2D_CONVERT:
                /* field contains a number to convert to IP */
                ep = strchr(cp, delimiter);
                if (ep) {
                    *ep = '\0';
                }
                if (0 == skStringParseUint32(&num, cp, 0, 0)) {
                    fprintf(outf, "%*s", column_width, num2dot(num));
                } else {
                    while (isspace((int)*cp)) {
                        ++cp;
                    }
                    fprintf(outf, "%*s", column_width, cp);
                }
                if (ep) {
                    fprintf(outf, "%c", delimiter);
                    cp = ep + 1;
                } else {
                    cp += strlen(cp);
                }
                break;

              case N2D_TEXT_FINAL:
                /* the final field to handle; nothing but text remains */
                fprintf(outf, "%s", cp);
                cp += strlen(cp);
                break;

              case N2D_TEXT_OPEN:
                /* field begins contiguous text fields */
                sp = cp;
                /* FALLTHROUGH */

              case N2D_TEXT_CONTINUE:
                /* field is in middle of contiguous text fields */
                ep = strchr(cp, delimiter);
                if (ep) {
                    cp = ep + 1;
                    if (!*cp) {
                        /* unexpected end of input */
                        fprintf(outf, "%s", sp);
                    }
                } else {
                    /* unexpected end of input */
                    fprintf(outf, "%s", sp);
                    cp += strlen(cp);
                }
                break;

              case N2D_TEXT_SINGLE:
                /* field is surrounded by IP fields */
                sp = cp;
                /* FALLTHROUGH */

              case N2D_TEXT_CLOSE:
                /* field closes a list of contiguous text fields */
                ep = strchr(cp, delimiter);
                if (ep) {
                    *ep = '\0';
                    fprintf(outf, "%s%c", sp, delimiter);
                    cp = ep + 1;
                } else {
                    fprintf(outf, "%s", sp);
                    cp += strlen(cp);
                }
                break;
            }
        }
        fprintf(outf, "\n");
    }

  END:
    appTeardown();
    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
