/*
** Copyright (C) 2007-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Python wrappers
**
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: pysilk.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/rwrec.h>
#include <silk/iptree.h>
#include <silk/bagtree.h>
#include <silk/skprefixmap.h>
#include <silk/sksite.h>
#include <silk/skcountry.h>
#include <Python.h>
#include <datetime.h>
#include <structmember.h>


/* LOCAL DEFINES AND TYPEDEFS */

#ifndef PyMODINIT_FUNC        /* declarations for DLL import/export */
#define PyMODINIT_FUNC void
#endif

/* In Python 2.4, length()-type functions are 'inquiry's that return
 * int.  In Python 2.5, they are 'lenfunc's that return Py_ssize_t. */
#if PY_VERSION_HEX < 0x02050000
#define Py_ssize_t     int
#define PY_SSIZE_T_MAX INT_MAX
#endif

#define NOT_SET -9999

#define CHECK_SITE                              \
    if (!site_configured) {                     \
        init_site();                            \
        site_configured = 1;                    \
    }

#if SK_ENABLE_IPV6
#define UNUSED_NOv6(x) x
#else
#define UNUSED_NOv6(x) UNUSED(x)
#endif

/* EXPORTED VARIABLE DEFINITIONS */



/* LOCAL VARIABLE DEFINITIONS */

static PyObject *silkmod    = NULL;
static PyObject *timedelta  = NULL;
static PyObject *datetime   = NULL;
static PyObject *maxelapsed = NULL;
static PyObject *minelapsed = NULL;
static PyObject *zerotime   = NULL;
static PyObject *thousand   = NULL;
static PyObject *maxintipv4 = NULL;
#if SK_ENABLE_IPV6
static PyObject *maxintipv6 = NULL;
#endif
static PyObject *havesite   = NULL;
static PyObject *sensors    = NULL;
static PyObject *classes    = NULL;
static PyObject *flowtypes  = NULL;

static int site_configured = 0;

static char error_buffer[1024];

static int init_site(void);

/* EXPORTED FUNCTION PROTOTYPES */

PyMODINIT_FUNC initpysilk(void);
PyMODINIT_FUNC initpysilk_nl(void);


/* FUNCTION DEFINITIONS */

#ifdef TEST_PRINTF_FORMATS
#define error_printf  printf
#else  /* TEST_PRINTF_FORMATS */
static int error_printf(const char *fmt, ...)
    SK_CHECK_PRINTF(1, 2);

static int error_printf(const char *fmt, ...)
{
    int rv;

    va_list args;
    va_start(args, fmt);
    rv = vsnprintf(error_buffer, sizeof(error_buffer), fmt, args);
    error_buffer[sizeof(error_buffer) - 1] = '\0';
    va_end(args);

    return rv;
}
#endif  /* TEST_PRINTF_FORMATS */

static PyObject *any_obj_error(
    PyObject   *exc,
    const char *format,
    PyObject   *obj)
{
    PyObject *pformat = PyString_FromString(format);
    PyObject *msg = PyString_Format(pformat, obj);
    PyErr_SetObject(exc, msg);
    Py_DECREF(pformat);
    Py_DECREF(msg);
    return NULL;
}

static PyObject *obj_error(const char *format, PyObject *obj)
{
    return any_obj_error(PyExc_ValueError, format, obj);
}

static void obj_dealloc(PyObject *obj)
{
    obj->ob_type->tp_free(obj);
}

static PyObject *iter_iter(
    PyObject *self)
{
    Py_INCREF(self);
    return self;
}

#if !SK_ENABLE_IPV6
static PyObject *silkPyNotImplemented(
    PyObject UNUSED(*self),
    PyObject UNUSED(*args),
    PyObject UNUSED(*kwds))
{
    return PyErr_Format(PyExc_NotImplementedError,
                        "SiLK was not built with IPv6 support.");
}
#endif  /* !SK_ENABLE_IPV6 */

typedef struct silkPyIPAddr_st {
    PyObject_HEAD
    skipaddr_t addr;
} silkPyIPAddr;

static PyTypeObject silkPyIPAddrType;
static PyTypeObject silkPyIPv4AddrType;
static PyTypeObject silkPyIPv6AddrType;

#define silkPyIPAddr_Check(op) PyObject_TypeCheck(op, &silkPyIPAddrType)
#define silkPyIPv4Addr_Check(op) PyObject_TypeCheck(op, &silkPyIPv4AddrType)
#define silkPyIPv6Addr_Check(op) PyObject_TypeCheck(op, &silkPyIPv6AddrType)

static PyObject *silkPyIPAddr_repr(silkPyIPAddr *obj)
{
    char buf[SK_NUM2DOT_STRLEN];
    PyTypeObject *type;
    PyObject *retval;

    type = (PyTypeObject *)PyObject_Type((PyObject *)obj);
    if (type == NULL) {
        return NULL;
    }

    skipaddrString(buf, &obj->addr, SKIPADDR_CANONICAL);

    retval = PyString_FromFormat("%s('%s')", type->tp_name, buf);
    Py_DECREF(type);

    return retval;
}

static PyObject *silkPyIPAddr_str(silkPyIPAddr *obj)
{
    char buf[SK_NUM2DOT_STRLEN];

    skipaddrString(buf, &obj->addr, SKIPADDR_CANONICAL);
    return PyString_FromString(buf);
}

static PyObject *silkPyIPAddr_padded(silkPyIPAddr *obj)
{
    char buf[SK_NUM2DOT_STRLEN];

    skipaddrString(buf, &obj->addr, SKIPADDR_ZEROPAD);
    return PyString_FromString(buf);
}

static PyObject *silkPyIPAddr_to_ipv4(PyObject *self)
{
#if SK_ENABLE_IPV6
    PyObject *obj;

    obj = PyObject_CallFunctionObjArgs((PyObject *)&silkPyIPv4AddrType,
                                       self, NULL);
    if (obj == NULL) {
        if (PyErr_ExceptionMatches(PyExc_ValueError)) {
            PyErr_Clear();
            Py_RETURN_NONE;
        }
    }

    return obj;
#else  /* SK_ENABLE_IPV6 */
    Py_INCREF(self);
    return self;
#endif  /* SK_ENABLE_IPV6 */
}

#if SK_ENABLE_IPV6
static PyObject *silkPyIPAddr_to_ipv6(PyObject *self)
{
    return PyObject_CallFunctionObjArgs((PyObject *)&silkPyIPv6AddrType,
                                        self, NULL);
}
#endif  /* SK_ENABLE_IPV6 */

static int silkPyIPAddr_compare(silkPyIPAddr *a, silkPyIPAddr *b)
{
    int rv = skipaddrCompare(&a->addr, &b->addr);

    if (rv < 0) {
        return -1;
    }
    if (rv > 0) {
        return 1;
    }
    return 0;
}

static long silkPyIPAddr_hash(silkPyIPAddr *obj)
{
    long retval;
#if SK_ENABLE_IPV6
    if (skipaddrIsV6(&obj->addr)) {
        uint8_t v6[16];
        skipaddrGetAsV6(&obj->addr, v6);
#if SK_SIZEOF_LONG == 8
        retval = *(long *)(&v6[8]);
#else  /* SK_SIZEOF_LONG */
        /* Assume long is 4 bytes */
        retval = *(long *)(&v6[12]);
#endif  /* SK_SIZEOF_LONG */
        return retval;
    }
#endif  /* SK_ENABLE_IPV6 */
    retval = skipaddrGetV4(&obj->addr);
    if (retval == -1) {
        retval = 0;
    }
    return retval;
}

static PyObject *silkPyIPAddr_int(silkPyIPAddr *obj)
{
    PyObject *result;

#if SK_ENABLE_IPV6
    if (skipaddrIsV6(&obj->addr)) {
        char      buf[33];
        char     *p;
        int       i;
        uint32_t *v6;

        p = buf;
        v6 = (uint32_t *)obj->addr.ip_ip.ipu_ipv6;
        for (i = 0; i < 4; i++) {
            sprintf(p, "%08" PRIx32, ntohl(*v6));
            p += 8;
            ++v6;
        }
        result = PyLong_FromString(buf, NULL, 16);
    } else
#endif  /* SK_ENABLE_IPV6 */
    {
        result = PyLong_FromUnsignedLong(skipaddrGetV4(&obj->addr));
    }

    return result;
}

static PyObject *silkPyIPAddr_mask(
    silkPyIPAddr *self,
    PyObject     *mask)
{
    silkPyIPAddr *retval;
    skipaddr_t addr;
    PyTypeObject *type;

    if (!silkPyIPAddr_Check(mask)) {
        PyErr_SetString(PyExc_TypeError, "Argument must be an IPAddr");
        return NULL;
    }

    skipaddrCopy(&addr, &self->addr);
    skipaddrMask(&addr, &((silkPyIPAddr *)mask)->addr);
    if (skipaddrIsV6(&addr)) {
        type = &silkPyIPv6AddrType;
    } else {
        type = &silkPyIPv4AddrType;
    }
    retval = PyObject_New(silkPyIPAddr, type);
    if (retval == NULL) {
        return NULL;
    }
    skipaddrCopy(&retval->addr, &addr);

    return (PyObject *)retval;
}

static PyObject *silkPyIPAddr_mask_prefix(
    silkPyIPAddr *self,
    PyObject     *prefix)
{
    silkPyIPAddr *retval;
    PyTypeObject *type;
    long p;
    int max;

    if (!PyInt_Check(prefix) && !PyLong_Check(prefix)) {
        PyErr_SetString(PyExc_TypeError, "Prefix must be an integer");
        return NULL;
    }

    if (skipaddrIsV6(&self->addr)) {
        type = &silkPyIPv6AddrType;
        max = 128;
    } else {
        type = &silkPyIPv4AddrType;
        max = 32;
    }

    p = PyInt_AsLong(prefix);
    if (PyErr_Occurred()) {
        return NULL;
    }

    if (p < 0 || p > max) {
        return PyErr_Format(PyExc_ValueError,
                            "Prefix must be between 0 and %d", max);
    }

    if (skipaddrIsV6(&self->addr)) {
        type = &silkPyIPv6AddrType;
    } else {
        type = &silkPyIPv4AddrType;
    }
    retval = PyObject_New(silkPyIPAddr, type);
    if (retval == NULL) {
        return NULL;
    }

    skipaddrCopy(&retval->addr, &self->addr);
    /* skipaddrApplyCIDR fails to work on IPv4 addresses when p >= * 32 */
    if (type != &silkPyIPv4AddrType || p < 32) {
        skipaddrApplyCIDR(&retval->addr, p);
    }

    return (PyObject *)retval;
}

static PyObject *silkPyIPAddr_new(
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    static char  *kwlist[] = {"address", NULL};
    silkPyIPAddr *self;
    PyObject     *o;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O", kwlist, &o)) {
        return NULL;
    }

    if (o->ob_type == &silkPyIPv4AddrType ||
        o->ob_type == &silkPyIPv6AddrType)
    {
        /* IPv{4,6}Addr objects are immutable, so just incref*/
        Py_INCREF(o);
        return o;
    }
    if (silkPyIPAddr_Check(o)) {
        /* Unknown subclass of IPAddr, do a real copy */
        if (type == &silkPyIPAddrType) {
            if (skipaddrIsV6(&((silkPyIPAddr *)o)->addr)) {
                type = &silkPyIPv6AddrType;
            } else {
                type = &silkPyIPv4AddrType;
            }
        }
    } else if (PyString_Check(o)) {
        char *straddr;
        char *c;

        straddr = PyString_AS_STRING(o);
        c = strchr(straddr, ':');

        if (c) {
            type = &silkPyIPv6AddrType;
        } else {
            type = &silkPyIPv4AddrType;
        }
    } else if (PyInt_Check(o) || PyLong_Check(o)) {
        /* The IPAddr(int) constructor is deprecated */
        int rv = PyErr_Warn(PyExc_DeprecationWarning,
                            ("IPAddr(int) is deprecated.  Use IPv4Addr(int) "
                             "or IPv6Addr(int) instead."));
        if (rv) {
            return NULL;
        }
        type = &silkPyIPv4AddrType;

    } else {
        return PyErr_Format(PyExc_TypeError, "Must be a string or IPAddr");
    }

    /* Allocate the object */
    self = (silkPyIPAddr *)type->tp_alloc(type, 0);
    if (self == NULL) {
        return NULL;
    }

    return (PyObject *)self;
}

static PyObject *silkPyIPvXAddr_new(
    PyTypeObject *basetype,
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    static char  *kwlist[] = {"address", NULL};
    PyObject   *self;
    PyObject   *o;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O", kwlist, &o)) {
        return NULL;
    }

    if (type == basetype && o->ob_type == basetype) {
        Py_INCREF(o);
        return o;
    }

    self = type->tp_alloc(type, 0);
    return self;
}

static PyObject *silkPyIPv4Addr_octets(
    silkPyIPAddr *self)
{
    PyObject *retval;
    PyObject *octet;
    uint32_t v4 = skipaddrGetV4(&self->addr);
    int i;

    retval = PyTuple_New(4);
    if (retval == NULL) {
        return NULL;
    }

    for (i = 3; i >= 0; i--) {
        octet = PyInt_FromLong(v4 & 0xff);
        if (octet == NULL) {
            Py_DECREF(retval);
            return NULL;
        }
        PyTuple_SET_ITEM(retval, i, octet);
        v4 >>= 8;
    }

    return retval;
}

static PyObject *silkPyIPv4Addr_new(
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    return silkPyIPvXAddr_new(&silkPyIPv4AddrType, type, args, kwds);
}

static int silkPyIPv4Addr_init(
    silkPyIPAddr *self,
    PyObject     *args,
    PyObject     *kwds)
{
    static char  *kwlist[] = {"address", NULL};
    PyObject     *addr;
    int           rv;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O", kwlist, &addr)) {
        return -1;
    }

    if (addr == (PyObject *)self) {
        /* We were initialized by new */
        return 0;
    }

    if (PyString_Check(addr)) {
        rv = skStringParseIP(&self->addr, PyString_AS_STRING(addr));
        if (rv != 0) {
            PyErr_SetString(PyExc_ValueError,
                            "String is not a valid IP address");
            return -1;
        }
        if (skipaddrIsV6(&self->addr)) {
            PyErr_SetString(PyExc_ValueError,
                            "String is not a valid IPv4 address");
            return -1;
        }
    } else if (PyLong_Check(addr) || PyInt_Check(addr)) {
        uint32_t value;
        PyObject *num;

        num = PyLong_FromLong(0);
        rv = PyObject_Compare(addr, num);
        Py_DECREF(num);
        if (rv < 0) {
            /* Negative */
            PyErr_SetString(PyExc_ValueError,
                            "Illegal IPv4 address (negative)");
            return -1;
        }
        rv = PyObject_Compare(addr, maxintipv4);
        if (rv > 0) {
            PyErr_SetString(PyExc_ValueError,
                            "Illegal IPv4 address (integer too large)");
            return -1;
        }

        value = PyLong_AsUnsignedLong(addr);
        skipaddrSetV4(&self->addr, &value);

#if SK_ENABLE_IPV6
    } else if (silkPyIPv6Addr_Check(addr)) {
        /* Convert to v4 */
        silkPyIPAddr *v6addr = (silkPyIPAddr *)addr;

        if (skipaddrV6toV4(&v6addr->addr, &self->addr)) {
            PyErr_SetString(PyExc_ValueError,
                            "IP address not convertable to IPv4.");
            return -1;
        }
#endif  /* SK_ENABLE_IPV6 */

    } else if (silkPyIPv4Addr_Check(addr)) {
        /* Copy */
        skipaddrCopy(&self->addr, &((silkPyIPAddr *)addr)->addr);
    } else {
        PyErr_SetString(PyExc_TypeError, "Must be a string or integer");
        return -1;
    }

    return 0;
}

#if SK_ENABLE_IPV6
static PyObject *silkPyIPv6Addr_new(
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    return silkPyIPvXAddr_new(&silkPyIPv6AddrType, type, args, kwds);
}
#endif  /* SK_ENABLE_IPV6 */

static int silkPyIPv6Addr_init(
    silkPyIPAddr *self,
    PyObject     *args,
    PyObject     *kwds)
{
#if !SK_ENABLE_IPV6
    silkPyNotImplemented((PyObject *)self, args, kwds);
    return -1;
#else  /* if SK_ENABLE_IPV6 */
    static char  *kwlist[] = {"address", NULL};
    PyObject     *addr;
    int           rv;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O", kwlist, &addr)) {
        return -1;
    }

    if (addr == (PyObject *)self) {
        /* We were initialized by new */
        return 0;
    }

    if (PyString_Check(addr)) {
        rv = skStringParseIP(&self->addr, PyString_AS_STRING(addr));
        if (rv != 0) {
            PyErr_SetString(PyExc_ValueError,
                            "String is not a valid IP address");
            return -1;
        }
        if (!skipaddrIsV6(&self->addr)) {
            PyErr_SetString(PyExc_ValueError,
                            "String is not a valid IPv6 address");
            return -1;
        }
    } else if (PyLong_Check(addr) || PyInt_Check(addr)) {
        uint8_t   v6[16];
        uint32_t *v6_32;
        PyObject *next;
        PyObject *shift;
        PyObject *num;
        int i;

        num = PyLong_FromLong(0);
        rv = PyObject_Compare(addr, num);
        Py_DECREF(num);
        if (rv < 0) {
            /* Negative */
            PyErr_SetString(PyExc_ValueError,
                            "Illegal IPv6 address (negative)");
            return -1;
        }
        rv = PyObject_Compare(addr, maxintipv6);
        if (rv > 0) {
            PyErr_SetString(PyExc_ValueError,
                            "Illegal IPv6 address (integer too large)");
            return -1;
        }

        /* Set IP */
        shift = PyLong_FromLong(32);
        v6_32 = (uint32_t *)v6 + 3;

        next = addr;
        Py_INCREF(next);
        for (i = 0; i < 4; i++, v6_32--) {
            PyObject *tmp;

            tmp = PyNumber_And(next, maxintipv4);
            *v6_32 = htonl(PyLong_AsUnsignedLong(tmp));
            Py_DECREF(tmp);
            tmp = next;
            next = PyNumber_Rshift(tmp, shift);
            Py_DECREF(tmp);
        }
        Py_DECREF(shift);
        skipaddrSetV6(&self->addr, v6);

    } else if (silkPyIPv4Addr_Check(addr)) {
        /* Convert to v6 */
        silkPyIPAddr *v4addr = (silkPyIPAddr *)addr;

        if (skipaddrIsV6(&v4addr->addr)) {
            skipaddrCopy(&self->addr, &v4addr->addr);
        } else {
            skipaddrV4toV6(&v4addr->addr, &self->addr);
        }

    } else if (silkPyIPv6Addr_Check(addr)) {
        /* Copy */
        skipaddrCopy(&self->addr, &((silkPyIPAddr *)addr)->addr);
    } else {
        PyErr_SetString(PyExc_TypeError, "Must be a string or integer");
        return -1;
    }

    return 0;
#endif  /* else of !SK_ENABLE_IPV6 */
}

static PyObject *silkPyIPAddr_ipv6(
    silkPyIPAddr *UNUSED_NOv6(self))
{
#if SK_ENABLE_IPV6
    return PyBool_FromLong(skipaddrIsV6(&self->addr));
#else
    Py_RETURN_FALSE;
#endif
}

static PyObject *silkPyIPAddr_country_code(
    silkPyIPAddr *self)
{
    char name[3];
    sk_countrycode_t code;
    int rv;

    rv = skCountrySetup(NULL, error_printf);
    if (rv != 0) {
        PyErr_SetString(PyExc_RuntimeError, error_buffer);
        return NULL;
    }

    code = skCountryLookupCode(&self->addr);
    if (code == SK_COUNTRYCODE_INVALID) {
        Py_RETURN_NONE;
    }

    return PyString_FromString(skCountryCodeToName(code, name, sizeof(name)));
}


static PyMethodDef silkPyIPAddr_methods[] = {

    {"isipv6", (PyCFunction)silkPyIPAddr_ipv6, METH_NOARGS,
     "addr.isipv6() -> bool -- return whether addr is an IPv6 address"},

    {"is_ipv6", (PyCFunction)silkPyIPAddr_ipv6, METH_NOARGS,
     "addr.isipv6() -> bool -- return whether addr is an IPv6 address"},

    {"to_ipv6", (PyCFunction)
#if SK_ENABLE_IPV6
     silkPyIPAddr_to_ipv6, METH_NOARGS,
#else
     silkPyNotImplemented, METH_VARARGS | METH_KEYWORDS,
#endif
     "addr.to_ipv6() -- return addr converted to an IPv6 address"},

    {"to_ipv4", (PyCFunction)silkPyIPAddr_to_ipv4, METH_NOARGS,
     "addr.to_ipv4() -- return addr converted to an IPv4 address"},

    {"padded", (PyCFunction)silkPyIPAddr_padded, METH_NOARGS,
     "addr.padded() -> str -- return zero-padded IP string"},

    {"mask", (PyCFunction)silkPyIPAddr_mask, METH_O,
     "addr.mask(addr2) -> addr3 -- return addr masked by addr2"},

    {"mask_prefix", (PyCFunction)silkPyIPAddr_mask_prefix, METH_O,
     ("addr.mask(prefix) -> addr2 -- "
      "return addr masked by the top prefix bits")},

    {"country_code", (PyCFunction)silkPyIPAddr_country_code, METH_NOARGS,
     "addr.country_code() -> string -- 2-character country code"},

    {NULL, NULL, 0, NULL}       /* Sentinel */
};


static PyMethodDef silkPyIPv4Addr_methods[] = {
    {"octets", (PyCFunction)silkPyIPv4Addr_octets, METH_NOARGS,
     ("addr.octets() = (o1, o2 ...) -- "
      "return the octets of addr as a tuple")},

    {NULL, NULL, 0, NULL}       /* Sentinel */
};


static char silkPyIPAddr_doc[] =
    ("IPAddr(string) -> ip address\n"
     "IPAddr(ipaddr) -> copy of ip address");

static char silkPyIPv4Addr_doc[] =
    ("IPv4Addr(string) -> IPv4 address\n"
     "IPv4Addr(int) -> IPv4 address\n"
     "IPv4Addr(IPV6Addr) -> IPv4 from IPv4 in IPv6 address\n"
     "IPv4Addr(IPv4Addr) -> copy of ip address");

static char silkPyIPv6Addr_doc[] =
    ("IPv6Addr(string) -> IPv6 address\n"
     "IPv6Addr(int) -> IPv6 address\n"
     "IPv6Addr(IPV4Addr) -> IPv4 in IPv6 address\n"
     "IPv6Addr(IPv6Addr) -> copy of ip address");

static PyNumberMethods silkPyIPAddr_number_methods = {
    0,                          /* nb_add */
    0,                          /* nb_subtract */
    0,                          /* nb_multiply */
    0,                          /* nb_divide */
    0,                          /* nb_remainder */
    0,                          /* nb_divmod */
    0,                          /* nb_power */
    0,                          /* nb_negative */
    0,                          /* nb_positive */
    0,                          /* nb_absolute */
    0,                          /* nb_nonzero */
    0,                          /* nb_invert */
    0,                          /* nb_lshift */
    0,                          /* nb_rshift */
    0,                          /* nb_and */
    0,                          /* nb_xor */
    0,                          /* nb_or */
    0,                          /* nb_coerce */
    (unaryfunc)silkPyIPAddr_int, /* nb_int */
    (unaryfunc)silkPyIPAddr_int, /* nb_long */
    0,                          /* nb_float */
    0,                          /* nb_oct */
    0,                          /* nb_hex */
    0,                          /* nb_inplace_add */
    0,                          /* nb_inplace_subtract */
    0,                          /* nb_inplace_multiply */
    0,                          /* nb_inplace_divide */
    0,                          /* nb_inplace_remainder */
    0,                          /* nb_inplace_power */
    0,                          /* nb_inplace_lshift */
    0,                          /* nb_inplace_rshift */
    0,                          /* nb_inplace_and */
    0,                          /* nb_inplace_xor */
    0,                          /* nb_inplace_or */
    0,                          /* nb_floor_divide */
    0,                          /* nb_true_divide */
    0,                          /* nb_inplace_floor_divide */
    0                           /* nb_inplace_true_divide */
#if PY_VERSION_HEX >= 0x02050000
    ,0                          /* nb_index */
#endif
};

static PyTypeObject silkPyIPAddrType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.IPAddr",              /* tp_name */
    sizeof(silkPyIPAddr),       /* tp_basicsize */
    0,                          /* tp_itemsize */
    obj_dealloc,                /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    (cmpfunc)silkPyIPAddr_compare, /* tp_compare */
    (reprfunc)silkPyIPAddr_repr, /* tp_repr */
    &silkPyIPAddr_number_methods, /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    (hashfunc)silkPyIPAddr_hash, /* tp_hash  */
    0,                          /* tp_call */
    (reprfunc)silkPyIPAddr_str, /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT,         /* tp_flags */
    silkPyIPAddr_doc,           /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    silkPyIPAddr_methods,       /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    silkPyIPAddr_new,           /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

static PyTypeObject silkPyIPv4AddrType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.IPv4Addr",            /* tp_name */
    0,                          /* tp_basicsize */
    0,                          /* tp_itemsize */
    0,                          /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    silkPyIPv4Addr_doc,         /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    silkPyIPv4Addr_methods,     /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    &silkPyIPAddrType,          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyIPv4Addr_init, /* tp_init */
    0,                          /* tp_alloc */
    silkPyIPv4Addr_new,         /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

static PyTypeObject silkPyIPv6AddrType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.IPv6Addr",            /* tp_name */
    0,                          /* tp_basicsize */
    0,                          /* tp_itemsize */
    0,                          /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    silkPyIPv6Addr_doc,         /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    0,                          /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    &silkPyIPAddrType,          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyIPv6Addr_init, /* tp_init */
    0,                          /* tp_alloc */
#if SK_ENABLE_IPV6
    silkPyIPv6Addr_new,         /* tp_new */
#else
    (newfunc)silkPyNotImplemented, /* tp_new */
#endif
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyIPWildcard_st {
    PyObject_HEAD
    skIPWildcard_t  wildcard;
    PyObject       *name;
} silkPyIPWildcard;

static PyTypeObject silkPyIPWildcardType;

#define silkPyIPWildcard_Check(op) \
    PyObject_TypeCheck(op, &silkPyIPWildcardType)

typedef struct silkPyIPWildcardIter_st {
    PyObject_HEAD
    silkPyIPWildcard       *wildcard;
    skIPWildcardIterator_t  iter;
} silkPyIPWildcardIter;

static PyTypeObject silkPyIPWildcardIterType;

#define silkPyIPWildcardIter_Check(op) \
    PyObject_TypeCheck(op, &silkPyIPWildcardIterType)


static PyObject *silkPyIPWildcard_repr(silkPyIPWildcard *obj)
{
    PyObject *format;
    PyObject *arg;
    PyObject *retval;

    format = PyString_FromString("silk.IPWildcard(\"%s\")");
    if (format == NULL) {
        return NULL;
    }
    arg = Py_BuildValue("(O)", obj->name);
    if (arg == NULL) {
        Py_DECREF(format);
        return NULL;
    }
    retval = PyString_Format(format, arg);
    Py_DECREF(format);
    Py_DECREF(arg);

    return retval;
}

static PyObject *silkPyIPWildcard_str(silkPyIPWildcard *obj)
{
    Py_INCREF(obj->name);
    return obj->name;
}

static void silkPyIPWildcard_dealloc(silkPyIPWildcard *obj)
{
    Py_XDECREF(obj->name);
    obj->ob_type->tp_free((PyObject *)obj);
}

static PyObject *silkPyIPWildcard_new(
    PyTypeObject *type,
    PyObject     *args,
    PyObject     *kwds)
{
    silkPyIPWildcard *self;

    self = (silkPyIPWildcard *)type->tp_alloc(type, 0);

    if (self != NULL) {
        static char *kwlist[] = {"wildcard", NULL};
        PyObject    *wildcard;
        int          rv;

        if (!PyArg_ParseTupleAndKeywords(args, kwds, "S", kwlist, &wildcard)) {
            Py_DECREF(self);
            return NULL;
        }
        rv = skStringParseIPWildcard(
            &self->wildcard, PyString_AS_STRING(wildcard));
        if (rv != 0) {
            Py_DECREF(self);
            return PyErr_Format(PyExc_ValueError, "Illegal IP wildcard: %s",
                                PyString_AS_STRING(wildcard));
        }
        Py_INCREF(wildcard);
        self->name = wildcard;
    }

    return (PyObject *)self;
}

static PyObject *silkPyIPWildcard_iter(
    silkPyIPWildcard *self)
{
    int                   rv;
    silkPyIPWildcardIter *iter;

    iter = (silkPyIPWildcardIter *)silkPyIPWildcardIterType.tp_alloc(
        &silkPyIPWildcardIterType, 0);
    if (iter) {
        rv = skIPWildcardIteratorBind(&iter->iter, &self->wildcard);
        assert(rv == 0);
        Py_INCREF(self);
        iter->wildcard = self;
    }
    return (PyObject *)iter;
}

static int silkPyIPWildcard_contains(
    silkPyIPWildcard *self,
    PyObject         *obj)
{
    int           retval;
    silkPyIPAddr *silkaddr;

    if (PyString_Check(obj)) {
        obj = PyObject_CallFunctionObjArgs((PyObject *)&silkPyIPAddrType,
                                           obj, NULL);
        if (obj == NULL) {
            return -1;
        }
    } else if (silkPyIPAddr_Check(obj)) {
        Py_INCREF(obj);
    } else {
        PyErr_SetString(PyExc_TypeError, "Must be a string or silk.IPAddr");
        return -1;
    }

    silkaddr = (silkPyIPAddr *)obj;
    retval = skIPWildcardCheckIp(&self->wildcard, &silkaddr->addr);
    Py_DECREF(obj);

    return retval ? 1 : 0;
}

static PyObject *silkPyIPWildcard_ipv6(silkPyIPWildcard *UNUSED_NOv6(self))
{
#if SK_ENABLE_IPV6
    return PyBool_FromLong(skIPWildcardIsV6(&self->wildcard));
#else
    Py_RETURN_FALSE;
#endif
}

static PyMethodDef silkPyIPWildcard_methods[] = {
    {"isipv6", (PyCFunction)silkPyIPWildcard_ipv6, METH_NOARGS,
     "wild.isipv6() -> bool -- return whether wild is an IPv6 wildcard"},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static PySequenceMethods silkPyIPWildcard_sequence_methods = {
    0,                          /* sq_length */
    0,                          /* sq_concat */
    0,                          /* sq_repeat */
    0,                          /* sq_item */
    0,                          /* sq_slice */
    0,                          /* sq_ass_item */
    0,                          /* sq_ass_slice */
    (objobjproc)silkPyIPWildcard_contains, /* sq_contains */
    0,                          /* sq_inplace_concat */
    0                           /* sq_inplace_repeat */
};

static char silkPyIPWildcard_doc[] =
    ("IPWildcard(string) -> IP Wildcard address");

static PyTypeObject silkPyIPWildcardType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.IPWildcard",          /* tp_name */
    sizeof(silkPyIPWildcard),   /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyIPWildcard_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    (reprfunc)silkPyIPWildcard_repr, /* tp_repr */
    0,                          /* tp_as_number */
    &silkPyIPWildcard_sequence_methods, /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    (reprfunc)silkPyIPWildcard_str, /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,
                                /* tp_flags */
    silkPyIPWildcard_doc,       /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    (getiterfunc)silkPyIPWildcard_iter, /* tp_iter */
    0,                          /* tp_iternext */
    silkPyIPWildcard_methods,   /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    silkPyIPWildcard_new,       /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

static void silkPyIPWildcardIter_dealloc(
    silkPyIPWildcardIter *self)
{
    Py_XDECREF(self->wildcard);
    self->ob_type->tp_free((PyObject *)self);
}

static PyObject *silkPyIPWildcardIter_iternext(
    silkPyIPWildcardIter *self)
{
    silkPyIPAddr       *addr;
    skIteratorStatus_t  rv;
    skipaddr_t          raw_addr;

    rv = skIPWildcardIteratorNext(&self->iter, &raw_addr);
    if (rv == SK_ITERATOR_NO_MORE_ENTRIES) {
        PyErr_SetNone(PyExc_StopIteration);
        return NULL;
    }
    addr = (silkPyIPAddr *)silkPyIPv4AddrType.tp_alloc(&silkPyIPv4AddrType, 0);
    if (addr == NULL) {
        return NULL;
    }
    addr->addr = raw_addr;

    return (PyObject *)addr;
}

static PyTypeObject silkPyIPWildcardIterType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.IPWildcardIter", /* tp_name */
    sizeof(silkPyIPWildcardIter), /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyIPWildcardIter_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    "IP Wildcard iterator object", /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    iter_iter,                  /* tp_iter */
    (iternextfunc)silkPyIPWildcardIter_iternext, /* tp_iternext */
    0,                          /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyIPSet_st {
    PyObject_HEAD
    skIPTree_t *ipset;
} silkPyIPSet;

static PyTypeObject silkPyIPSetType;

#define silkPyIPSet_Check(op) \
    PyObject_TypeCheck(op, &silkPyIPSetType)


typedef struct silkPyIPSetIter_st {
    PyObject_HEAD
    silkPyIPSet *set;
    union {
        skIPTreeIterator_t          *ip;
        skIPTreeCIDRBlockIterator_t *cidr;
    } iter;
    unsigned is_cidr : 1;
} silkPyIPSetIter;

static PyTypeObject silkPyIPSetIterType;

#define silkPyIPSetIter_Check(op) \
    PyObject_TypeCheck(op, &silkPyIPSetIterType)

static void silkPyIPSet_dealloc(silkPyIPSet *obj)
{
    if (obj->ipset) {
        skIPTreeDelete(&obj->ipset);
    }
    obj->ob_type->tp_free((PyObject *)obj);
}

static int silkPyIPSet_init(
    silkPyIPSet *self,
    PyObject    *args,
    PyObject    *kwds)
{
    static char *kwlist[] = {"filename", NULL};
    PyObject    *fname = NULL;
    int          rv;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "|S", kwlist, &fname)) {
        return -1;
    }

    if (fname) {
        rv = skIPTreeLoad(&self->ipset, PyString_AS_STRING(fname));
        if (rv != 0) {
            PyErr_Format(PyExc_ValueError, "Unable to read IPSet from %s: %s",
                         PyString_AS_STRING(fname), skIPTreeStrError(rv));
            return -1;
        }
    } else {
        rv = skIPTreeCreate(&self->ipset);
        if (rv == SKIP_ERR_ALLOC) {
            PyErr_NoMemory();
            return -1;
        }
        assert(rv == SKIP_OK);
    }

    return 0;
}

static int silkPyIPSet_contains(
    silkPyIPSet *self,
    PyObject    *obj)
{
    int           retval;
    silkPyIPAddr *silkaddr;

    if (PyString_Check(obj)) {
        obj = PyObject_CallFunctionObjArgs((PyObject *)&silkPyIPv4AddrType,
                                           obj, NULL);
        if (obj == NULL) {
            return -1;
        }
    } else if (silkPyIPv4Addr_Check(obj)) {
        Py_INCREF(obj);
    } else {
        PyErr_SetString(PyExc_TypeError, "Must be a string or silk.IPv4Addr");
        return -1;
    }

    silkaddr = (silkPyIPAddr *)obj;
    if (skipaddrIsV6(&silkaddr->addr)) {
        retval = 0;
    } else {
        retval = skIPTreeCheckAddress(self->ipset,
                                      skipaddrGetV4(&silkaddr->addr));
    }
    Py_DECREF(obj);

    return retval ? 1 : 0;
}

static Py_ssize_t silkPyIPSet_len(
    silkPyIPSet *self)
{
    uint64_t count = skIPTreeCountIPs(self->ipset);
    if (count > PY_SSIZE_T_MAX) {
        PyErr_SetString(PyExc_OverflowError, "IPSet too long for integer");
        return -1;
    }
    return (Py_ssize_t)count;
}

static PyObject *silkPyIPSet_cardinality(
    silkPyIPSet *self)
{
    uint64_t count = skIPTreeCountIPs(self->ipset);
    return PyLong_FromUnsignedLongLong(count);
}

static PyObject *silkPyIPSet_union_update(
    silkPyIPSet *self,
    silkPyIPSet *obj)
{
    int rv;

    if (!silkPyIPSet_Check(obj)) {
        PyErr_SetString(PyExc_NotImplementedError,
                        "Argument must be a silk.IPSet");
        return NULL;
    }

    rv = skIPTreeUnion(self->ipset, obj->ipset);
    if (rv != 0) {
        return PyErr_NoMemory();
    }

    Py_INCREF(self);
    return (PyObject *)self;
}

static PyObject *silkPyIPSet_intersection_update(
    silkPyIPSet *self,
    silkPyIPSet *obj)
{
    if (!silkPyIPSet_Check(obj)) {
        PyErr_SetString(PyExc_NotImplementedError,
                        "Argument must be a silk.IPSet");
        return NULL;
    }

    skIPTreeIntersect(self->ipset, obj->ipset);

    Py_INCREF(self);
    return (PyObject *)self;
}

static PyObject *silkPyIPSet_difference_update(
    silkPyIPSet *self,
    silkPyIPSet *obj)
{
    if (!silkPyIPSet_Check(obj)) {
        PyErr_SetString(PyExc_NotImplementedError,
                        "Argument must be a silk.IPSet");
        return NULL;
    }

    skIPTreeSubtract(self->ipset, obj->ipset);

    Py_INCREF(self);
    return (PyObject *)self;
}


static PyObject *silkPyIPSet_clear(
    silkPyIPSet *self)
{
    skIPTreeRemoveAll(self->ipset);

    Py_INCREF(self);
    return (PyObject *)self;
}


static PyObject *silkPyIPSet_save(
    silkPyIPSet *self,
    PyObject    *name)
{
    int rv;

    if (!PyString_Check(name)) {
        PyErr_SetString(PyExc_TypeError, "Filename required");
        return NULL;
    }

    rv = skIPTreeSave(self->ipset, PyString_AS_STRING(name));
    if (rv != SKIP_OK) {
        PyErr_SetString(PyExc_IOError, skIPTreeStrError(rv));
        return NULL;
    }

    Py_RETURN_NONE;
}

static PyObject *silkPyIPSet_add(
    silkPyIPSet *self,
    PyObject *obj)
{
    int           rv;

    if (silkPyIPv4Addr_Check(obj)) {
        silkPyIPAddr *addr = (silkPyIPAddr *)obj;

        rv = skIPTreeAddAddress(self->ipset, skipaddrGetV4(&addr->addr));
    } else if silkPyIPWildcard_Check(obj) {
        silkPyIPWildcard *wild;

        assert(silkPyIPWildcard_Check(obj));

        wild = (silkPyIPWildcard *)obj;

        rv = skIPTreeAddIPWildcard(self->ipset, &wild->wildcard);
        if (rv == SKIP_ERR_IPV6) {
            PyErr_SetString(PyExc_ValueError,
                            "Must only include IPv4 addresses");
            return NULL;
        }
    } else {
        PyErr_SetString(PyExc_TypeError,
                        "Must be an IPv4 Address or IP Wildcard");
        return NULL;
    }

    if (rv == SKIP_ERR_ALLOC) {
        return PyErr_NoMemory();
    }
    assert(rv == SKIP_OK);

    Py_INCREF(self);
    return (PyObject *)self;
}

static PyObject *silkPyIPSet_iter(
    silkPyIPSet *self)
{
    int              rv;
    silkPyIPSetIter *iter;

    iter = (silkPyIPSetIter *)silkPyIPSetIterType.tp_alloc(
        &silkPyIPSetIterType, 0);
    if (iter) {
        rv = skIPTreeIteratorCreate(&iter->iter.ip, self->ipset);
        if (rv != 0) {
            Py_DECREF(iter);
            return PyErr_NoMemory();
        }
        Py_INCREF(self);
        iter->set = self;
    }
    return (PyObject *)iter;
}

static PyObject *silkPyIPSet_cidr_iter(
    silkPyIPSet *self)
{
    int              rv;
    silkPyIPSetIter *iter;

    iter = (silkPyIPSetIter *)silkPyIPSetIterType.tp_alloc(
        &silkPyIPSetIterType, 0);
    if (iter) {
        rv = skIPTreeCIDRBlockIteratorCreate(&iter->iter.cidr, self->ipset);
        if (rv != 0) {
            Py_DECREF(iter);
            return PyErr_NoMemory();
        }
        Py_INCREF(self);
        iter->set = self;
        iter->is_cidr = 1;
    }
    return (PyObject *)iter;
}

static PyMethodDef silkPyIPSet_methods[] = {
    {"cardinality", (PyCFunction)silkPyIPSet_cardinality, METH_NOARGS,
     "ipset.cardinality() -> long -- number of IP Addresses in the IPSet"},
    {"intersection_update", (PyCFunction)silkPyIPSet_intersection_update,
     METH_O,
     ("Return the intersection of two IPSets as a new IPSet.\n"
      "\n"
      "(i.e. all elements that are in both IPSets.)")},
    {"update", (PyCFunction)silkPyIPSet_union_update,
     METH_O,
     ("Update an IPSet with the union of itself and another.")},
    {"difference_update", (PyCFunction)silkPyIPSet_difference_update,
     METH_O,
     ("Remove all elements of another IPSet from this IPSet.")},
    {"clear", (PyCFunction)silkPyIPSet_clear, METH_NOARGS,
     ("Remove all elements from this IPSet.")},
    {"save", (PyCFunction)silkPyIPSet_save, METH_O,
     "ipset.save(filename) -- Saves the set to a file."},
    {"cidr_iter", (PyCFunction)silkPyIPSet_cidr_iter, METH_NOARGS,
     "Return an iterator over IPAddr/prefix tuples."},
    {"add", (PyCFunction)silkPyIPSet_add, METH_O,
     ("Add an element to an IPSet.  The element can be an IP address, an\n"
      "IP wildcard, or the string representation of either.\n"
      "\n"
      "This has no effect for any element already present.")},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static PySequenceMethods silkPyIPSet_sequence_methods = {
#if PY_VERSION_HEX < 0x02050000
    (inquiry)silkPyIPSet_len,   /* sq_length */
#else
    (lenfunc)silkPyIPSet_len,   /* sq_length */
#endif
    0,                          /* sq_concat */
    0,                          /* sq_repeat */
    0,                          /* sq_item */
    0,                          /* sq_slice */
    0,                          /* sq_ass_item */
    0,                          /* sq_ass_slice */
    (objobjproc)silkPyIPSet_contains, /* sq_contains */
    0,                          /* sq_inplace_concat */
    0                           /* sq_inplace_repeat */
};

static char silkPyIPSet_doc[] =
    ("IPSetBase() -> empty IPset\n"
     "IPSetBase(filename) -> IPset from file");

static PyTypeObject silkPyIPSetType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.IPSetBase",    /* tp_name */
    sizeof(silkPyIPSet),        /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyIPSet_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    &silkPyIPSet_sequence_methods, /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    silkPyIPSet_doc,            /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    (getiterfunc)silkPyIPSet_iter, /* tp_iter */
    0,                          /* tp_iternext */
    silkPyIPSet_methods,        /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyIPSet_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

static void silkPyIPSetIter_dealloc(
    silkPyIPSetIter *self)
{
    if (self->is_cidr) {
        skIPTreeCIDRBlockIteratorDestroy(&self->iter.cidr);
    } else {
        skIPTreeIteratorDestroy(&self->iter.ip);
    }
    Py_XDECREF(self->set);
    self->ob_type->tp_free((PyObject *)self);
}

static PyObject *silkPyIPSetIter_iternext(
    silkPyIPSetIter *self)
{
    silkPyIPAddr       *addr;
    skIteratorStatus_t  rv;
    PyObject           *retval;

    if (self->is_cidr) {
        skIPTreeCIDRBlock_t  block;
        PyObject            *pair;
        PyObject            *len;

        rv = skIPTreeCIDRBlockIteratorNext(&block, self->iter.cidr);
        if (rv == SK_ITERATOR_NO_MORE_ENTRIES) {
            PyErr_SetNone(PyExc_StopIteration);
            return NULL;
        }
        addr = (silkPyIPAddr *)silkPyIPv4AddrType.
               tp_alloc(&silkPyIPv4AddrType, 0);
        if (addr == NULL) {
            return NULL;
        }
        skipaddrSetV4(&addr->addr, &block.addr);
        len = PyInt_FromLong(block.mask);
        if (len == NULL) {
            Py_DECREF(addr);
            return NULL;
        }

        pair = PyTuple_New(2);
        if (pair == NULL) {
            Py_DECREF(addr);
            Py_DECREF(len);
        }
        PyTuple_SET_ITEM(pair, 0, (PyObject *)addr);
        PyTuple_SET_ITEM(pair, 1, len);

        retval = pair;
    } else {
        uint32_t raw_addr;

        rv = skIPTreeIteratorNext(&raw_addr, self->iter.ip);
        if (rv == SK_ITERATOR_NO_MORE_ENTRIES) {
            PyErr_SetNone(PyExc_StopIteration);
            return NULL;
        }
        addr = (silkPyIPAddr *)silkPyIPv4AddrType.
               tp_alloc(&silkPyIPv4AddrType, 0);
        if (addr == NULL) {
            return NULL;
        }
        skipaddrSetV4(&addr->addr, &raw_addr);

        retval = (PyObject *)addr;
    }

    return retval;
}


static PyTypeObject silkPyIPSetIterType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.IPSetIter",    /* tp_name */
    sizeof(silkPyIPSetIter),    /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyIPSetIter_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    "IP Set iterator object",   /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    iter_iter,                  /* tp_iter */
    (iternextfunc)silkPyIPSetIter_iternext, /* tp_iternext */
    0,                          /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyPmap_st {
    PyObject_HEAD
    skPrefixMap_t *map;
} silkPyPmap;

static PyTypeObject silkPyPmapType;

#define silkPyPmap_Check(op) \
    PyObject_TypeCheck(op, &silkPyPmapType)

typedef struct silkPyPmapIter_st {
    PyObject_HEAD
    silkPyPmap *map;
    skPrefixMapIterator_t iter;
} silkPyPmapIter;

static PyTypeObject silkPyPmapIterType;

#define silkPyPmapIter_Check(op) \
    PyObject_TypeCheck(op, &silkPyPmapIterType)

static void silkPyPmap_dealloc(silkPyPmap *obj)
{
    if (obj->map) {
        skPrefixMapDelete(obj->map);
    }
    obj->ob_type->tp_free((PyObject *)obj);
}

static int silkPyPmap_init(
    silkPyPmap *self,
    PyObject   *args,
    PyObject   *kwds)
{
    static char      *kwlist[] = {"filename", NULL};
    PyObject         *fname;
    skPrefixMapErr_t  rv;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "S", kwlist, &fname)) {
        return -1;
    }

    rv = skPrefixMapLoad(&self->map, PyString_AS_STRING(fname));
    if (rv != SKPREFIXMAP_OK) {
        PyErr_Format(PyExc_ValueError, "Unable to read prefix map from %s: %s",
                     PyString_AS_STRING(fname), skPrefixMapStrerror(rv));
        return -1;
    }

    return 0;
}


static PyObject *silkPyPmap_get_value_string(
    silkPyPmap *self,
    PyObject   *value)
{
    uint32_t  val;
    uint32_t  size;
    char     *buf;
    int       rv;
    PyObject *retval;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return NULL;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred()) {
        return NULL;
    }

    size = skPrefixMapDictionaryGetMaxWordSize(self->map) + 1;
    buf = malloc(size);
    if (buf == NULL) {
        return PyErr_NoMemory();
    }

    rv = skPrefixMapDictionaryGetEntry(self->map, val, buf, size);
    assert(rv < (int32_t)size);

    retval = PyString_FromStringAndSize(buf, rv);
    free(buf);

    return retval;
}

static PyObject *silkPyPmap_subscript(
    silkPyPmap *self,
    PyObject   *sub)
{
    unsigned long key;
    uint32_t value;

    if (!PyInt_Check(sub) && !PyLong_Check(sub)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return NULL;
    }

    key = PyLong_AsUnsignedLong(sub);
    if (PyErr_Occurred()) {
        if (PyErr_ExceptionMatches(PyExc_OverflowError)) {
            PyErr_Clear();
            PyErr_SetString(PyExc_IndexError, "Subscript out of range");
        }
        return NULL;
    }

    if (key > UINT32_MAX) {
        PyErr_SetString(PyExc_IndexError, "Subscript out of range");
        return NULL;
    }

    value = skPrefixMapGet(self->map, key);

    return PyLong_FromUnsignedLong(value);
}

static PyObject *silkPyPmap_get_content(
    silkPyPmap *self,
    void *UNUSED(cbdata))
{
    return PyString_FromString(
        skPrefixMapGetContentName(skPrefixMapGetContentType(self->map)));
}

static PyObject *silkPyPmap_get_name(
    silkPyPmap *self,
    void *UNUSED(cbdata))
{
    const char *name = skPrefixMapGetMapName(self->map);
    if (name == NULL) {
        Py_RETURN_NONE;
    }

    return PyString_FromString(name);
}

static PyObject *silkPyPmap_get_num_values(
    silkPyPmap *self,
    void *UNUSED(cbdata))
{
    return PyInt_FromLong(skPrefixMapDictionaryGetWordCount(self->map));
}

static PyObject *silkPyPmap_iter(
    silkPyPmap *self)
{
    int             rv;
    silkPyPmapIter *iter;

    iter = (silkPyPmapIter *)silkPyPmapIterType.tp_alloc(
        &silkPyPmapIterType, 0);
    if (iter) {
        rv = skPrefixMapIteratorBind(&iter->iter, self->map);
        if (rv != 0) {
            Py_DECREF(iter);
            PyErr_SetString(PyExc_RuntimeError,
                            "Failed to create prefix map iterator");
            return NULL;
        }
        Py_INCREF(self);
        iter->map = self;
    }
    return (PyObject *)iter;
}

static PyMethodDef silkPyPmap_methods[] = {
    {"get_value_string", (PyCFunction)silkPyPmap_get_value_string, METH_O,
     ("Get the string associated with an integer value")},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static PyGetSetDef silkPyPmap_getsetters[] = {
    {"content", (getter)silkPyPmap_get_content, NULL, "Content type", NULL},
    {"name", (getter)silkPyPmap_get_name, NULL, "Prefix map name", NULL},
    {"num_values", (getter)silkPyPmap_get_num_values, NULL,
     "Prefix map number of values", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyMappingMethods silkPyPmap_mapping_methods = {
    0,
    (binaryfunc)silkPyPmap_subscript,
    0
};

static char silkPyPmap_doc[] =
    ("PMapBase(filename) -> Prefix map from file");

static PyTypeObject silkPyPmapType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.PMapBase",     /* tp_name */
    sizeof(silkPyPmap),         /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyPmap_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    &silkPyPmap_mapping_methods, /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    silkPyPmap_doc,             /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    (getiterfunc)silkPyPmap_iter, /* tp_iter */
    0,                          /* tp_iternext */
    silkPyPmap_methods,         /* tp_methods */
    0,                          /* tp_members */
    silkPyPmap_getsetters,      /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyPmap_init,  /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

static void silkPyPmapIter_dealloc(
    silkPyPmapIter *self)
{
    Py_XDECREF(self->map);
    self->ob_type->tp_free((PyObject *)self);
}

static PyObject *silkPyPmapIter_iternext(
    silkPyPmapIter *self)
{
    skIteratorStatus_t  rv;
    PyObject           *retval;
    uint32_t            start, end, value;

    rv = skPrefixMapIteratorNext(&self->iter, &start, &end, &value);
    if (rv == SK_ITERATOR_NO_MORE_ENTRIES) {
        PyErr_SetNone(PyExc_StopIteration);
        return NULL;
    }

    retval = Py_BuildValue("kkk", start, end, value);

    return retval;
}

static PyTypeObject silkPyPmapIterType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.PMapBaseIter", /* tp_name */
    sizeof(silkPyPmapIterType), /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyPmapIter_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    "Prefix map iterator object", /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    iter_iter,                  /* tp_iter */
    (iternextfunc)silkPyPmapIter_iternext, /* tp_iternext */
    0,                          /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyBag_st {
    PyObject_HEAD
    skBag_t *bag;
} silkPyBag;

static PyTypeObject silkPyBagType;

#define silkPyBag_Check(op) \
    PyObject_TypeCheck(op, &silkPyBagType)

typedef struct silkPyBagIter_st {
    PyObject_HEAD
    silkPyBag       *bag;
    skBagIterator_t *iter;
} silkPyBagIter;

static PyTypeObject silkPyBagIterType;

#define silkPyBagIter_Check(op) \
    PyObject_TypeCheck(op, &silkPyBagIterType)

static void silkPyBag_dealloc(silkPyBag *obj)
{
    if (obj->bag) {
        skBagFree(obj->bag);
    }
    obj->ob_type->tp_free((PyObject *)obj);
}

static int silkPyBag_init(
    silkPyBag *self,
    PyObject  *args,
    PyObject  *kwds)
{
    static char *kwlist[] = {"copy", "filename", NULL};
    PyObject   *fname     = NULL;
    silkPyBag  *copy      = NULL;
    skBagErr_t  rv;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "|O!S", kwlist,
                                     &silkPyBagType,
                                     (PyObject *)&copy, &fname))
    {
        return -1;
    }

    if (copy && fname) {
        PyErr_SetString(PyExc_ValueError, "Can't copy and use a filename");
        return -1;
    }

    if (fname) {
        rv = skBagLoad(&self->bag, PyString_AS_STRING(fname));
        if (rv != SKBAG_OK) {
            PyErr_Format(PyExc_ValueError, "Unable to read IPSet from %s: %s",
                         PyString_AS_STRING(fname), skBagStrerror(rv));
            return -1;
        }
    } else if (copy) {
        rv = skBagCopy(&self->bag, copy->bag);
    } else {
        rv = skBagCreate(&self->bag);
    }
    if (rv == SKBAG_ERR_MEMORY) {
        PyErr_NoMemory();
        return -1;
    }
    assert(rv == SKBAG_OK);

    return 0;
}

static PyObject *silkPyBag_save(
    silkPyBag *self,
    PyObject  *name)
{
    skBagErr_t rv;

    if (!PyString_Check(name)) {
        PyErr_SetString(PyExc_TypeError, "Filename required");
        return NULL;
    }

    rv = skBagSave(self->bag, PyString_AS_STRING(name));
    if (rv != SKBAG_OK) {
        PyErr_SetString(PyExc_IOError, skBagStrerror(rv));
        return NULL;
    }

    Py_RETURN_NONE;
}

static PyObject *silkPyBag_clear(
    silkPyBag *self)
{
    skBagErr_t rv;

    skBagFree(self->bag);
    rv = skBagCreate(&self->bag);
    if (rv == SKBAG_ERR_MEMORY) {
        return PyErr_NoMemory();
    }
    assert(rv == SKBAG_OK);

    Py_RETURN_NONE;
}

static PyObject *silkPyBag_subscript(
    silkPyBag *self,
    PyObject  *sub)
{
    unsigned long  key;
    skBagKey_t     bagkey;
    skBagCounter_t value;
    skBagErr_t     rv;

    if (!PyInt_Check(sub) && !PyLong_Check(sub)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return NULL;
    }

    key = PyLong_AsUnsignedLong(sub);
    if (PyErr_Occurred()) {
        if (PyErr_ExceptionMatches(PyExc_OverflowError)) {
            PyErr_Clear();
            PyErr_SetString(PyExc_IndexError, "Subscript out of range");
        }
        return NULL;
    }

    if (key > UINT32_MAX) {
        PyErr_SetString(PyExc_IndexError, "Subscript out of range");
        return NULL;
    }

    bagkey = key;
    rv = skBagGetCounter(self->bag, &bagkey, &value);
    assert(rv != SKBAG_ERR_KEY_NOT_FOUND);
    if (rv != SKBAG_OK) {
        PyErr_SetString(PyExc_RuntimeError, skBagStrerror(rv));
        return NULL;
    }

    return PyLong_FromUnsignedLongLong(value);
}

static int silkPyBag_modify(
    silkPyBag *self,
    PyObject  *sub,
    PyObject  *value,
    skBagErr_t (*fn)(skBag_t *, const skBagKey_t *, const skBagCounter_t *))
{
    skBagKey_t         bagkey;
    skBagCounter_t     bagvalue;
    unsigned long      key;
    unsigned long long v;
    skBagErr_t         rv;

    if (!PyInt_Check(sub) && !PyLong_Check(sub)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer subscript");
        return -1;
    }
    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer value");
        return -1;
    }

    key = PyLong_AsUnsignedLong(sub);
    if (PyErr_Occurred()) {
        if (PyErr_ExceptionMatches(PyExc_OverflowError)) {
            PyErr_Clear();
            PyErr_SetString(PyExc_IndexError, "Subscript out of range");
        }
        return -1;
    }

    if (key > UINT32_MAX) {
        PyErr_SetString(PyExc_IndexError, "Subscript out of range");
        return -1;
    }

    if (PyLong_Check(value)) {
        v = PyLong_AsUnsignedLongLong(value);
    } else {
        v = PyInt_AsLong(value);
    }
    if (PyErr_Occurred()) {
        return -1;
    }

    bagkey = key;
    bagvalue = v;

    rv = fn(self->bag, &bagkey, &bagvalue);
    switch (rv) {
      case SKBAG_OK:
        break;
      case SKBAG_ERR_MEMORY:
        PyErr_NoMemory();
        return -1;
      case SKBAG_ERR_OP_BOUNDS:
        PyErr_SetString(PyExc_OverflowError, skBagStrerror(rv));
        return -1;
      case SKBAG_ERR_KEY_NOT_FOUND:
        /* Fall through */
      default:
        skAbortBadCase(rv);
    }

    return 0;
}

static int silkPyBag_ass_subscript(
    silkPyBag *self,
    PyObject  *sub,
    PyObject  *value)
{
    return silkPyBag_modify(self, sub, value, skBagSetCounter);
}

static PyObject *silkPyBag_incr(
    silkPyBag *self,
    PyObject  *args,
    PyObject  *kwds)
{
    static char *kwlist[] = {"key", "value", NULL};
    PyObject *sub;
    PyObject *value;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "OO", kwlist, &sub, &value)) {
        return NULL;
    }

    if (silkPyBag_modify(self, sub, value, skBagAddToCounter)) {
        return NULL;
    }

    Py_RETURN_NONE;
}

static PyObject *silkPyBag_decr(
    silkPyBag *self,
    PyObject  *args,
    PyObject  *kwds)
{
    static char *kwlist[] = {"key", "value", NULL};
    PyObject *sub;
    PyObject *value;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "OO", kwlist, &sub, &value)) {
        return NULL;
    }

    if (silkPyBag_modify(self, sub, value, skBagSubtractFromCounter)) {
        return NULL;
    }

    Py_RETURN_NONE;
}

static Py_ssize_t silkPyBag_count(
    silkPyBag *self)
{
    uint64_t count = skBagCountKeys(self->bag);
    return (Py_ssize_t)count;
}


static PyObject *silkPyBag_iter(
    silkPyBag *self)
{
    skBagErr_t     rv;
    silkPyBagIter *iter;

    iter = (silkPyBagIter *)silkPyBagIterType.tp_alloc(
        &silkPyBagIterType, 0);
    if (iter) {
        rv = skBagIteratorCreate(self->bag, &iter->iter);
        if (rv == SKBAG_ERR_MEMORY) {
            Py_DECREF(iter);
            return PyErr_NoMemory();
        }
        if (rv != SKBAG_OK) {
            Py_DECREF(iter);
            PyErr_SetString(PyExc_RuntimeError,
                            "Failed to create prefix map iterator");
            return NULL;
        }
        Py_INCREF(self);
        iter->bag = self;
    }
    return (PyObject *)iter;
}


static PyMethodDef silkPyBag_methods[] = {
    {"incr", (PyCFunction)silkPyBag_incr, METH_KEYWORDS,
     ("bag.incr(key, value) -- increments bag[key] by value")},
    {"decr", (PyCFunction)silkPyBag_decr, METH_KEYWORDS,
     ("bag.decr(key, value) -- decrements bag[key] by value")},
    {"save", (PyCFunction)silkPyBag_save, METH_O,
     "bag.save(filename) -- saves the bag to a file"},
    {"clear", (PyCFunction)silkPyBag_clear, METH_NOARGS,
     "bag.clear() -- empties the bag"},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static PyMappingMethods silkPyBag_mapping_methods = {
#if PY_VERSION_HEX < 0x02050000
    (inquiry)silkPyBag_count,
#else
    (lenfunc)silkPyBag_count,
#endif
    (binaryfunc)silkPyBag_subscript,
    (objobjargproc)silkPyBag_ass_subscript
};

static char silkPyBag_doc[] =
    ("BagBase(filename) -> Bag from file");

static PyTypeObject silkPyBagType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.BagBase",      /* tp_name */
    sizeof(silkPyBag),          /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyBag_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    &silkPyBag_mapping_methods, /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    silkPyBag_doc,              /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    (getiterfunc)silkPyBag_iter, /* tp_iter */
    0,                          /* tp_iternext */
    silkPyBag_methods,          /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyBag_init,   /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

static void silkPyBagIter_dealloc(
    silkPyBagIter *self)
{
    Py_XDECREF(self->bag);
    skBagIteratorDestroy(self->iter);
    self->ob_type->tp_free((PyObject *)self);
}

static PyObject *silkPyBagIter_iternext(
    silkPyBagIter *self)
{
    skBagErr_t      rv;
    PyObject       *retval;
    skBagKey_t      key;
    skBagCounter_t  counter;

    rv = skBagIteratorNext(self->iter, &key, &counter);
    if (rv == SKBAG_ERR_KEY_NOT_FOUND) {
        PyErr_SetNone(PyExc_StopIteration);
        return NULL;
    }

    retval = Py_BuildValue("kK", key, counter);

    return retval;
}

static PyTypeObject silkPyBagIterType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.BagBaseIter",  /* tp_name */
    sizeof(silkPyBagIterType),  /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyBagIter_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    "Bag iterator object",      /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    iter_iter,                  /* tp_iter */
    (iternextfunc)silkPyBagIter_iternext, /* tp_iternext */
    0,                          /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    0,                          /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyTCPFlags_st {
    PyObject_HEAD
    uint8_t val;
} silkPyTCPFlags;

static PyTypeObject silkPyTCPFlagsType;

#define silkPyTCPFlags_Check(op) \
    PyObject_TypeCheck(op, &silkPyTCPFlagsType)

static PyObject *silkPyTCPFlags_new(
    PyTypeObject *type,
    PyObject     *UNUSED(args),
    PyObject     *UNUSED(kwds))
{
    silkPyTCPFlags *self;

    self = (silkPyTCPFlags *)type->tp_alloc(type, 0);

    if (self != NULL) {
        self->val = 0;
    }

    return (PyObject *)self;
}

static int silkPyTCPFlags_init(
    silkPyTCPFlags *self,
    PyObject       *args,
    PyObject       *kwds)
{
    static char *kwlist[] = {"value", NULL};
    PyObject *val;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "O", kwlist, &val)) {
        return -1;
    }

    if (silkPyTCPFlags_Check(val)) {
        silkPyTCPFlags *oflags = (silkPyTCPFlags *)val;
        self->val = oflags->val;
    } else if (PyInt_Check(val) || PyLong_Check(val)) {
        long intval = PyLong_AsLong(val);
        if (intval < 0 || intval > (long)UINT8_MAX) {
            PyErr_Format(PyExc_OverflowError,
                         "Illegal TCP flag value: %ld", intval);
            return -1;
        }
        self->val = intval;
    } else if (PyString_Check(val)) {
        char *strval = PyString_AsString(val);
        if (skStringParseTCPFlags(&self->val, strval)) {
            PyErr_Format(PyExc_ValueError,
                         "Illegal TCP flag value: %s", strval);
            return -1;
        }
    } else {
        obj_error("Illegal value: %s", val);
        return -1;
    }

    return 0;
}

static PyObject *silkPyTCPFlags_repr(silkPyTCPFlags *obj)
{
    char flags[SK_TCPFLAGS_STRLEN];

    tcpflags_string_r(obj->val, flags);

    return PyString_FromFormat("silk.TCPFlags('%s')", flags);
}

static PyObject *silkPyTCPFlags_padded(silkPyTCPFlags *obj)
{
    char flags[SK_TCPFLAGS_STRLEN];

    tcpflags_string_r(obj->val, flags);

    return PyString_FromString(flags);
}

static PyObject *silkPyTCPFlags_str(silkPyTCPFlags *obj)
{
    char flags[SK_TCPFLAGS_STRLEN];
    char unpadded[SK_TCPFLAGS_STRLEN];
    char *f, *u;

    tcpflags_string_r(obj->val, flags);
    for (f = flags, u = unpadded; *f; f++) {
        if (*f != ' ') {
            *u++ = *f;
        }
    }
    *u = '\0';

    return PyString_FromString(unpadded);
}

static long silkPyTCPFlags_hash(silkPyTCPFlags *obj)
{
    return obj->val;
}

static PyObject *silkPyTCPFlags_richcompare(
    silkPyTCPFlags *self,
    PyObject       *obj,
    int             cmp)
{
    if (cmp != Py_EQ && cmp != Py_NE) {
        Py_INCREF(Py_NotImplemented);
        return Py_NotImplemented;
    }

    if (!silkPyTCPFlags_Check(obj)) {
        PyErr_SetString(PyExc_TypeError, "Expected silk.TCPFlags");
        return NULL;
    }

    if (self->val == ((silkPyTCPFlags *)obj)->val) {
        if (cmp == Py_EQ) {
            Py_RETURN_TRUE;
        } else {
            Py_RETURN_FALSE;
        }
    }
    if (cmp == Py_NE) {
        Py_RETURN_TRUE;
    }

    Py_RETURN_FALSE;
}

static silkPyTCPFlags *silkPyTCPFlags_invert(silkPyTCPFlags *obj)
{
    silkPyTCPFlags *new_obj =
        (silkPyTCPFlags *)silkPyTCPFlagsType.tp_alloc(&silkPyTCPFlagsType, 0);

    if (new_obj != NULL) {
        new_obj->val = ~obj->val;
    }
    return new_obj;
}

static silkPyTCPFlags *silkPyTCPFlags_and(
    silkPyTCPFlags *a,
    silkPyTCPFlags *b)
{
    silkPyTCPFlags *new_obj =
        (silkPyTCPFlags *)silkPyTCPFlagsType.tp_alloc(&silkPyTCPFlagsType, 0);

    if (new_obj != NULL) {
        new_obj->val = a->val & b->val;
    }
    return new_obj;
}

static silkPyTCPFlags *silkPyTCPFlags_or(
    silkPyTCPFlags *a,
    silkPyTCPFlags *b)
{
    silkPyTCPFlags *new_obj =
        (silkPyTCPFlags *)silkPyTCPFlagsType.tp_alloc(&silkPyTCPFlagsType, 0);

    if (new_obj != NULL) {
        new_obj->val = a->val | b->val;
    }
    return new_obj;
}

static silkPyTCPFlags *silkPyTCPFlags_xor(
    silkPyTCPFlags *a,
    silkPyTCPFlags *b)
{
    silkPyTCPFlags *new_obj =
        (silkPyTCPFlags *)silkPyTCPFlagsType.tp_alloc(&silkPyTCPFlagsType, 0);

    if (new_obj != NULL) {
        new_obj->val = a->val ^ b->val;
    }
    return new_obj;
}

static PyObject *silkPyTCPFlags_int(silkPyTCPFlags *obj)
{
    return PyInt_FromLong(obj->val);
}

static int silkPyTCPFlags_nonzero(silkPyTCPFlags *a)
{
    return a->val ? 1 : 0;
}

static PyObject *silkPyTCPFlags_matches(silkPyTCPFlags *self,
                                        PyObject *arg)
{
    char *repr;
    uint8_t high, mask;
    int rv;

    if (!PyString_Check(arg)) {
        PyErr_SetString(PyExc_TypeError, "Expected string");
        return NULL;
    }

    repr = PyString_AS_STRING(arg);
    rv = skStringParseTCPFlagsHighMask(&high, &mask, repr);
    if (rv == SKUTILS_ERR_SHORT) {
        mask = high;
    } else if (rv != 0) {
        PyErr_SetString(PyExc_ValueError, "Illegal flag/mask");
        return NULL;
    }

    return PyBool_FromLong((self->val & mask) == high);
}

static PyMethodDef silkPyTCPFlags_methods[] = {
    {"matches", (PyCFunction)silkPyTCPFlags_matches, METH_O,
     "Return whether the flags match the high/mask flagstring"},
    {"padded", (PyCFunction)silkPyTCPFlags_padded, METH_NOARGS,
     "Returns the flags string padded with spaces, so flags line up"},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static PyObject *silkPyTCPFlags_getflag(silkPyTCPFlags *obj, void *bit)
{
    return PyBool_FromLong(obj->val & *(uint8_t *)bit);
}

static uint8_t flags_array[] = {
    FIN_FLAG, SYN_FLAG, RST_FLAG, PSH_FLAG,
    ACK_FLAG, URG_FLAG, ECE_FLAG, CWR_FLAG};

static PyGetSetDef silkPyTCPFlags_getsetters[] = {
    {"FIN", (getter)silkPyTCPFlags_getflag, NULL,
     "FIN flag", (void *)&flags_array[0]},
    {"SYN", (getter)silkPyTCPFlags_getflag, NULL,
     "SYN flag", (void *)&flags_array[1]},
    {"RST", (getter)silkPyTCPFlags_getflag, NULL,
     "RST flag", (void *)&flags_array[2]},
    {"PSH", (getter)silkPyTCPFlags_getflag, NULL,
     "PSH flag", (void *)&flags_array[3]},
    {"ACK", (getter)silkPyTCPFlags_getflag, NULL,
     "ACK flag", (void *)&flags_array[4]},
    {"URG", (getter)silkPyTCPFlags_getflag, NULL,
     "URG flag", (void *)&flags_array[5]},
    {"ECE", (getter)silkPyTCPFlags_getflag, NULL,
     "ECE flag", (void *)&flags_array[6]},
    {"CWR", (getter)silkPyTCPFlags_getflag, NULL,
     "CWR flag", (void *)&flags_array[7]},
    {NULL, NULL, NULL, NULL, NULL}
};

static int init_tcpflags_constants(PyObject *mod)
{
    PyGetSetDef *def;
    int rv;

    for (def = silkPyTCPFlags_getsetters; def->name; def++) {
        PyObject     *constant;
        unsigned int  flag;

        flag = *((uint8_t *)def->closure);
        constant = PyObject_CallFunction(
            (PyObject *)&silkPyTCPFlagsType, "I", flag);
        rv = PyModule_AddObject(mod, def->name, constant);
        if (rv != 0) {
            return -1;
        }
    }

    return 0;
}

static PyNumberMethods silkPyTCPFlags_number_methods = {
    0,                          /* nb_add */
    0,                          /* nb_subtract */
    0,                          /* nb_multiply */
    0,                          /* nb_divide */
    0,                          /* nb_remainder */
    0,                          /* nb_divmod */
    0,                          /* nb_power */
    0,                          /* nb_negative */
    0,                          /* nb_positive */
    0,                          /* nb_absolute */
    (inquiry)silkPyTCPFlags_nonzero, /* nb_nonzero */
    (unaryfunc)silkPyTCPFlags_invert, /* nb_invert */
    0,                          /* nb_lshift */
    0,                          /* nb_rshift */
    (binaryfunc)silkPyTCPFlags_and, /* nb_and */
    (binaryfunc)silkPyTCPFlags_xor, /* nb_xor */
    (binaryfunc)silkPyTCPFlags_or, /* nb_or */
    0,                          /* nb_coerce */
    (unaryfunc)silkPyTCPFlags_int, /* nb_int */
    0,                          /* nb_long */
    0,                          /* nb_float */
    0,                          /* nb_oct */
    0,                          /* nb_hex */
    0,                          /* nb_inplace_add */
    0,                          /* nb_inplace_subtract */
    0,                          /* nb_inplace_multiply */
    0,                          /* nb_inplace_divide */
    0,                          /* nb_inplace_remainder */
    0,                          /* nb_inplace_power */
    0,                          /* nb_inplace_lshift */
    0,                          /* nb_inplace_rshift */
    0,                          /* nb_inplace_and */
    0,                          /* nb_inplace_xor */
    0,                          /* nb_inplace_or */
    0,                          /* nb_floor_divide */
    0,                          /* nb_true_divide */
    0,                          /* nb_inplace_floor_divide */
    0                           /* nb_inplace_true_divide */
#if PY_VERSION_HEX >= 0x02050000
    ,0                          /* nb_index */
#endif
};

static char silkPyTCPFlags_doc[] =
    ("TCPFlags(string)   -> TCPFlags based on flag string\n"
     "TCPFlags(int)      -> TCPFlags based on integer representation\n"
     "TCPFlags(TCPFlags) -> Copy of TCPFlags");

static PyTypeObject silkPyTCPFlagsType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.TCPFlags",            /* tp_name */
    sizeof(silkPyTCPFlags),     /* tp_basicsize */
    0,                          /* tp_itemsize */
    obj_dealloc,                /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    (reprfunc)silkPyTCPFlags_repr, /* tp_repr */
    &silkPyTCPFlags_number_methods, /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    (hashfunc)silkPyTCPFlags_hash, /* tp_hash  */
    0,                          /* tp_call */
    (reprfunc)silkPyTCPFlags_str, /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_RICHCOMPARE |
    Py_TPFLAGS_CHECKTYPES,
                                /* tp_flags */
    silkPyTCPFlags_doc,         /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    (richcmpfunc)silkPyTCPFlags_richcompare, /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    silkPyTCPFlags_methods,     /* tp_methods */
    0,                          /* tp_members */
    silkPyTCPFlags_getsetters,  /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyTCPFlags_init, /* tp_init */
    0,                          /* tp_alloc */
    silkPyTCPFlags_new,         /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyRawRWRec_st {
    PyObject_HEAD
    rwRec *rec;
} silkPyRawRWRec;

static PyTypeObject silkPyRawRWRecType;

#define silkPyRawRWRec_Check(op) \
    PyObject_TypeCheck(op, &silkPyRawRWRecType)

static PyObject *silkPyRawRWRec_new(
    PyTypeObject *type,
    PyObject     *UNUSED(args),
    PyObject     *UNUSED(kwds))
{
    silkPyRawRWRec *self;

    self = (silkPyRawRWRec *)type->tp_alloc(type, 0);

    if (self != NULL) {
        self->rec = PyMem_Malloc(sizeof(*self->rec));
        if (self->rec == NULL) {
            Py_DECREF((PyObject *)self);
            return PyErr_NoMemory();
        }
        RWREC_CLEAR(self->rec);
    }

    return (PyObject *)self;
}

static int silkPyRawRWRec_init(
    silkPyRawRWRec *self,
    PyObject    *args,
    PyObject    *kwds)
{
    static char *kwlist[] = {"rec", NULL};
    PyObject *copy = NULL;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "|O!", kwlist,
                                     &silkPyRawRWRecType, &copy))
    {
        return -1;
    }

    if (copy) {
        RWREC_COPY(self->rec, ((silkPyRawRWRec *)copy)->rec);
    }
    return 0;
}

static void silkPyRawRWRec_dealloc(silkPyRawRWRec *obj)
{
    if (obj->rec) {
        PyMem_Free(obj->rec);
    }
    obj->ob_type->tp_free(obj);
}

static PyTypeObject silkPyRawRWRecType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.RWRawRec",     /* tp_name */
    sizeof(silkPyRawRWRec),     /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyRawRWRec_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    "Raw RW Record",            /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    0,                          /* tp_methods */
    0,                          /* tp_members */
    0,                          /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyRawRWRec_init, /* tp_init */
    0,                          /* tp_alloc */
    silkPyRawRWRec_new,         /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyRWRec_st {
    PyObject_HEAD
    silkPyRawRWRec *raw;
} silkPyRWRec;

static PyTypeObject silkPyRWRecType;

#define silkPyRWRec_Check(op) \
    PyObject_TypeCheck(op, &silkPyRWRecType)

static int silkPyRWRec_init(
    silkPyRWRec *self,
    PyObject    *args,
    PyObject    *kwds)
{
    static char *kwlist[] = {"clone", "copy", NULL};
    silkPyRawRWRec *clne = NULL;
    silkPyRWRec    *copy  = NULL;

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "|O!O!", kwlist,
                                     &silkPyRawRWRecType, (PyObject **)&clne,
                                     &silkPyRWRecType, (PyObject **)&copy))
    {
        return -1;
    }

    if (clne && copy) {
        PyErr_SetString(PyExc_RuntimeError, "Cannot clone and copy");
        return -1;
    }

    Py_XDECREF((PyObject *)self->raw);
    if (clne) {
        Py_INCREF(clne);
        self->raw = clne;
    } else if (copy) {
        self->raw = (silkPyRawRWRec *)PyObject_CallFunctionObjArgs(
            (PyObject *)&silkPyRawRWRecType, copy->raw, NULL);
    } else {
        self->raw = (silkPyRawRWRec *)PyObject_CallFunctionObjArgs(
            (PyObject *)&silkPyRawRWRecType, NULL);
    }

    if (self->raw == NULL) {
        return -1;
    }

    return 0;
}

static void silkPyRWRec_dealloc(silkPyRWRec *obj)
{
    Py_XDECREF((PyObject *)obj->raw);
    obj->ob_type->tp_free(obj);
}

static PyObject *silkPyRWRec_get_application(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyInt_FromLong(rwRecGetApplication(obj->raw->rec));
}

static int silkPyRWRec_set_application(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT16_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The application value must be a 16-bit integer");
        return -1;
    }

    rwRecSetApplication(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_bytes(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyLong_FromUnsignedLong(rwRecGetBytes(obj->raw->rec));
}

static int silkPyRWRec_set_bytes(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    unsigned long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsUnsignedLong(value);
    if (PyErr_Occurred() || val > UINT32_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The bytes value must be a 32-bit integer");
        return -1;
    }

    rwRecSetBytes(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_class(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    char          classname[SK_MAX_STRLEN_FLOWTYPE+1];
    flowtypeID_t  flowtype = rwRecGetFlowType(obj->raw->rec);

    CHECK_SITE;

    sksiteFlowtypeGetClass(classname, sizeof(classname), flowtype);

    return PyString_InternFromString(classname);
}

static PyObject *silkPyRWRec_get_classtype(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    char          classname[SK_MAX_STRLEN_FLOWTYPE+1];
    char          typename[SK_MAX_STRLEN_FLOWTYPE+1];
    flowtypeID_t  flowtype = rwRecGetFlowType(obj->raw->rec);
    PyObject     *pair     = PyTuple_New(2);

    if (pair == NULL) {
        return NULL;
    }

    CHECK_SITE;

    sksiteFlowtypeGetClass(classname, sizeof(classname), flowtype);
    sksiteFlowtypeGetType(typename, sizeof(typename), flowtype);

    PyTuple_SET_ITEM(pair, 0, PyString_InternFromString(classname));
    PyTuple_SET_ITEM(pair, 1, PyString_InternFromString(typename));

    return pair;
}

static int silkPyRWRec_set_classtype(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    char *class, *type;
    flowtypeID_t flowtype;


    if (!PyArg_ParseTuple(value, "ss", &class, &type)) {
        return -1;
    }

    CHECK_SITE;

    flowtype = sksiteFlowtypeLookupByClassType(class, type);

    if (flowtype == SK_INVALID_FLOWTYPE) {
        PyErr_SetString(PyExc_ValueError, "Invalid (class, type) pair");
        return -1;
    }

    rwRecSetFlowType(obj->raw->rec, flowtype);
    return 0;
}

static PyObject *silkPyRWRec_get_dip(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    silkPyIPAddr *addr;
    PyTypeObject *type;

    if (rwRecIsIPv6(obj->raw->rec)) {
        type = &silkPyIPv6AddrType;
    } else {
        type = &silkPyIPv4AddrType;
    }

    addr = PyObject_New(silkPyIPAddr, type);
    if (addr != NULL) {
        rwRecMemGetDIP(obj->raw->rec, &addr->addr);
    }

    return (PyObject *)addr;
}

static int silkPyRWRec_set_dip(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    if (PyString_Check(value)) {
        skipaddr_t  addr;
        char       *repr = PyString_AS_STRING(value);
        int         rv   = skStringParseIP(&addr, repr);
        if (rv != 0) {
            PyErr_Format(PyExc_ValueError, "Illegal IP address: %s", repr);
            return -1;
        }
        rwRecMemSetDIP(obj->raw->rec, &addr);
        return 0;
    }

    if (silkPyIPAddr_Check(value)) {
        silkPyIPAddr *addr = (silkPyIPAddr *)value;
        rwRecMemSetDIP(obj->raw->rec, &addr->addr);
        return 0;
    }

    PyErr_SetString(PyExc_TypeError, "The dip must be a valid IP address");
    return -1;
}

static PyObject *silkPyRWRec_get_dport(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyInt_FromLong(rwRecGetDPort(obj->raw->rec));
}

static int silkPyRWRec_set_dport(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT16_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The dport value must be a 16-bit integer");
        return -1;
    }

    rwRecSetDPort(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_duration(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyObject_CallFunction(timedelta, "IIII", 0, 0, 0,
                                 rwRecGetElapsed(obj->raw->rec));
}

static int silkPyRWRec_set_duration(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    PyObject *days;
    PyObject *secs;
    PyObject *usecs;
    uint32_t millisecs;

    if (!PyDelta_Check(value)) {
        PyErr_SetString(PyExc_TypeError,
                        "The duration value must be a datetime.timedelta");
        return -1;
    }
    if (PyObject_RichCompareBool(value, minelapsed, Py_LT) ||
        PyObject_RichCompareBool(value, maxelapsed, Py_GT))
    {
        PyErr_SetString(PyExc_OverflowError,
                        ("The duration must be in the range [0,4294967295] "
                         "milliseconds"));
        return -1;
    }
    days = PyObject_GetAttrString(value, "days");
    secs = PyObject_GetAttrString(value, "seconds");
    usecs = PyObject_GetAttrString(value, "microseconds");
    millisecs = PyLong_AsLong(days) * 1e3 * 3600 * 24 +
                PyLong_AsLong(secs) * 1e3 +
                PyLong_AsLong(usecs) / 1e3;
    Py_DECREF(secs);
    Py_DECREF(usecs);

    rwRecSetElapsed(obj->raw->rec, millisecs);
    return 0;
}

static PyObject *silkPyRWRec_get_duration_secs(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    double elapsed = ((double)rwRecGetElapsed(obj->raw->rec)) / 1.0e3;

    return PyFloat_FromDouble(elapsed);
}

static int silkPyRWRec_set_duration_secs(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    const char *errstr = "The duration_secs value must be a positive number";
    PyObject *pyfloat_val;
    PyObject *pyint_val;
    int64_t long_val;

    if (!PyNumber_Check(value)) {
        PyErr_SetString(PyExc_TypeError, errstr);
        return -1;
    }

    pyfloat_val = PyNumber_Multiply(value, thousand);
    if (pyfloat_val == NULL) {
        PyErr_SetString(PyExc_TypeError, errstr);
        return -1;
    }
    pyint_val = PyNumber_Int(pyfloat_val);
    Py_DECREF(pyfloat_val);
    if (pyint_val == NULL) {
        PyErr_SetString(PyExc_TypeError, errstr);
        return -1;
    }
    long_val = PyLong_AsLongLong(pyint_val);
    Py_DECREF(pyint_val);
    if (long_val < 0) {
        PyErr_SetString(PyExc_OverflowError, errstr);
        return -1;
    }

    if (long_val > UINT32_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The duration_secs value must be less than "
                        "4294967.295");
        return -1;
    }

    rwRecSetElapsed(obj->raw->rec, long_val);
    return 0;
}


static PyObject *silkPyRWRec_get_finnoack(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    uint8_t state;

    state = rwRecGetTcpState(obj->raw->rec);
    if (!(state & SK_TCPSTATE_EXPANDED)) {
        Py_RETURN_NONE;
    }
    return PyBool_FromLong(state & SK_TCPSTATE_FIN_FOLLOWED_NOT_ACK);
}

static int silkPyRWRec_set_finnoack(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    int     rv;
    uint8_t state;

    rv = PyObject_IsTrue(value);
    if (rv == -1) {
        return -1;
    }
    state = rwRecGetTcpState(obj->raw->rec) | SK_TCPSTATE_EXPANDED;
    if (rv) {
        state |= SK_TCPSTATE_FIN_FOLLOWED_NOT_ACK;
    } else {
        state &= ~SK_TCPSTATE_FIN_FOLLOWED_NOT_ACK;
    }
    rwRecSetTcpState(obj->raw->rec, state);

    return 0;
}

static PyObject *silkPyRWRec_get_icmpcode(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyInt_FromLong(rwRecGetIcmpCode(obj->raw->rec));
}

static int silkPyRWRec_set_icmpcode(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT8_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The icmpcode value must be a 8-bit integer");
        return -1;
    }

    rwRecSetIcmpCode(obj->raw->rec, (uint8_t)val);
    return 0;
}

static PyObject *silkPyRWRec_get_icmptype(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyInt_FromLong(rwRecGetIcmpType(obj->raw->rec));
}

static int silkPyRWRec_set_icmptype(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT8_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The icmptype value must be a 8-bit integer");
        return -1;
    }

    rwRecSetIcmpType(obj->raw->rec, (uint8_t)val);
    return 0;
}

static PyObject *silkPyRWRec_get_initflags(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    silkPyTCPFlags *flags;

    if (!(rwRecGetTcpState(obj->raw->rec) & SK_TCPSTATE_EXPANDED)) {
        Py_RETURN_NONE;
    }
    flags = (silkPyTCPFlags *)silkPyTCPFlagsType.tp_alloc(&silkPyTCPFlagsType,
                                                          0);
    if (flags != NULL) {
        flags->val = rwRecGetInitFlags(obj->raw->rec);
    }

    return (PyObject *)flags;
}

static int silkPyRWRec_set_initflags(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long    val;
    uint8_t state;

    state = rwRecGetTcpState(obj->raw->rec) | SK_TCPSTATE_EXPANDED;

    if (PyString_Check(value)) {
        uint8_t  flagval;
        char    *repr = PyString_AS_STRING(value);
        int      rv   = skStringParseTCPFlags(&flagval, repr);
        if (rv != 0) {
            PyErr_Format(PyExc_ValueError, "Illegal TCP flags: %s", repr);
            return -1;
        }
        rwRecSetTcpState(obj->raw->rec, state);
        rwRecSetInitFlags(obj->raw->rec, flagval);
        return 0;
    }

    if (silkPyTCPFlags_Check(value)) {
        silkPyTCPFlags *flags = (silkPyTCPFlags *)value;
        rwRecSetTcpState(obj->raw->rec, state);
        rwRecSetInitFlags(obj->raw->rec, flags->val);
        return 0;
    }

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT8_MAX) {
        obj_error("Illegal TCP flags value: %s", value);
        return -1;
    }

    rwRecSetTcpState(obj->raw->rec, state);
    rwRecSetInitFlags(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_input(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return  PyInt_FromLong(rwRecGetInput(obj->raw->rec));
}

static int silkPyRWRec_set_input(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT16_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The input value must be a 16-bit integer");
        return -1;
    }

    rwRecSetInput(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_nhip(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    silkPyIPAddr *addr;
    PyTypeObject *type;

    if (rwRecIsIPv6(obj->raw->rec)) {
        type = &silkPyIPv6AddrType;
    } else {
        type = &silkPyIPv4AddrType;
    }

    addr = PyObject_New(silkPyIPAddr, type);
    if (addr != NULL) {
        rwRecMemGetNhIP(obj->raw->rec, &addr->addr);
    }
    return (PyObject *)addr;
}

static int silkPyRWRec_set_nhip(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    if (PyString_Check(value)) {
        skipaddr_t  addr;
        char       *repr = PyString_AS_STRING(value);
        int         rv   = skStringParseIP(&addr, repr);
        if (rv != 0) {
            PyErr_Format(PyExc_ValueError, "Illegal IP address: %s", repr);
            return -1;
        }
        rwRecMemSetNhIP(obj->raw->rec, &addr);
        return 0;
    }

    if (silkPyIPAddr_Check(value)) {
        silkPyIPAddr *addr = (silkPyIPAddr *)value;
        rwRecMemSetNhIP(obj->raw->rec, &addr->addr);
        return 0;
    }

    PyErr_SetString(PyExc_TypeError, "The nhip must be a valid IP address");
    return -1;
}

static PyObject *silkPyRWRec_get_output(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyInt_FromLong(rwRecGetOutput(obj->raw->rec));
}

static int silkPyRWRec_set_output(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT16_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The output value must be a 16-bit integer");
        return -1;
    }

    rwRecSetOutput(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_packets(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyLong_FromUnsignedLong(rwRecGetPkts(obj->raw->rec));
}

static int silkPyRWRec_set_packets(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    unsigned long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsUnsignedLong(value);
    if (PyErr_Occurred() || val > UINT32_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The packets value must be a 32-bit integer");
        return -1;
    }

    rwRecSetPkts(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_protocol(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyInt_FromLong(rwRecGetProto(obj->raw->rec));
}

static int silkPyRWRec_set_protocol(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT8_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The protocol value must be an 8-bit integer");
        return -1;
    }

    rwRecSetProto(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_restflags(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    silkPyTCPFlags *flags;

    if (!(rwRecGetTcpState(obj->raw->rec) & SK_TCPSTATE_EXPANDED)) {
        Py_RETURN_NONE;
    }
    flags = (silkPyTCPFlags *)silkPyTCPFlagsType.tp_alloc(&silkPyTCPFlagsType,
                                                          0);
    if (flags != NULL) {
        flags->val = rwRecGetRestFlags(obj->raw->rec);
    }

    return (PyObject *)flags;
}

static int silkPyRWRec_set_restflags(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long    val;
    uint8_t state;

    state = rwRecGetTcpState(obj->raw->rec) | SK_TCPSTATE_EXPANDED;

    if (PyString_Check(value)) {
        uint8_t  flagval;
        char    *repr = PyString_AS_STRING(value);
        int      rv   = skStringParseTCPFlags(&flagval, repr);
        if (rv != 0) {
            PyErr_Format(PyExc_ValueError, "Illegal TCP flags: %s", repr);
            return -1;
        }
        rwRecSetTcpState(obj->raw->rec, state);
        rwRecSetRestFlags(obj->raw->rec, flagval);
        return 0;
    }

    if (silkPyTCPFlags_Check(value)) {
        silkPyTCPFlags *flags = (silkPyTCPFlags *)value;
        rwRecSetTcpState(obj->raw->rec, state);
        rwRecSetRestFlags(obj->raw->rec, flags->val);
        return 0;
    }

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT8_MAX) {
        obj_error("Illegal TCP flags value: %s", value);
        return -1;
    }

    rwRecSetTcpState(obj->raw->rec, state);
    rwRecSetRestFlags(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_sensor(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    char name[SK_MAX_STRLEN_SENSOR+1];

    CHECK_SITE;

    sksiteSensorGetName(name, sizeof(name), rwRecGetSensor(obj->raw->rec));
    return PyString_InternFromString(name);
}

static int silkPyRWRec_set_sensor(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    char *repr;
    sensorID_t sensor;

    repr = PyString_AsString(value);
    if (repr == NULL) {
        PyErr_SetString(PyExc_TypeError, "The sensor value must be a string");
        return -1;
    }

    CHECK_SITE;

    sensor = sksiteSensorLookup(repr);
    if (sensor == SK_INVALID_SENSOR) {
        PyErr_SetString(PyExc_ValueError, "Invalid sensor name");
        return -1;
    }

    rwRecSetSensor(obj->raw->rec, sensor);
    return 0;
}

static PyObject *silkPyRWRec_get_sip(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    silkPyIPAddr *addr;
    PyTypeObject *type;

    if (rwRecIsIPv6(obj->raw->rec)) {
        type = &silkPyIPv6AddrType;
    } else {
        type = &silkPyIPv4AddrType;
    }

    addr = PyObject_New(silkPyIPAddr, type);
    if (addr != NULL) {
        rwRecMemGetSIP(obj->raw->rec, &addr->addr);
    }
    return (PyObject *)addr;
}

static int silkPyRWRec_set_sip(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    if (PyString_Check(value)) {
        skipaddr_t  addr;
        char       *repr = PyString_AS_STRING(value);
        int         rv   = skStringParseIP(&addr, repr);
        if (rv != 0) {
            PyErr_Format(PyExc_ValueError, "Illegal IP address: %s", repr);
            return -1;
        }
        rwRecMemSetSIP(obj->raw->rec, &addr);
        return 0;
    }

    if (silkPyIPAddr_Check(value)) {
        silkPyIPAddr *addr = (silkPyIPAddr *)value;
        rwRecMemSetSIP(obj->raw->rec, &addr->addr);
        return 0;
    }

    PyErr_SetString(PyExc_TypeError, "The sip must be a valid IP address");
    return -1;
}

static PyObject *silkPyRWRec_get_sport(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    return PyInt_FromLong(rwRecGetSPort(obj->raw->rec));
}

static int silkPyRWRec_set_sport(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT16_MAX) {
        PyErr_SetString(PyExc_OverflowError,
                        "The sport value must be a 16-bit integer");
        return -1;
    }

    rwRecSetSPort(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_stime(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    PyObject *seconds, *millisecs, *final;
    imaxdiv_t d = imaxdiv(rwRecGetStartTime(obj->raw->rec), 1e3);

    seconds = PyObject_CallMethod(datetime, "utcfromtimestamp", "L", d.quot);
    millisecs = PyObject_CallFunction(timedelta, "IIIL", 0, 0, 0, d.rem);
    final = PyNumber_Add(seconds, millisecs);
    Py_DECREF(seconds);
    Py_DECREF(millisecs);
    return final;
}

static int silkPyRWRec_set_stime(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    PyObject *delta;
    PyObject *days;
    PyObject *secs;
    PyObject *usecs;
    int64_t millisecs;

    if (!PyDateTime_Check(value)) {
        PyErr_SetString(PyExc_TypeError,
                        "The stime value must be a datetime.datetime");
        return -1;
    }
    if (PyObject_RichCompareBool(value, zerotime, Py_LT)) {
        PyErr_SetString(PyExc_OverflowError, "Minimum stime is Jan 1, 1970");
        return -1;
    }
    delta = PyNumber_Subtract(value, zerotime);
    days = PyObject_GetAttrString(delta, "days");
    secs = PyObject_GetAttrString(delta, "seconds");
    usecs = PyObject_GetAttrString(delta, "microseconds");
    millisecs = PyLong_AsLong(days) * (int64_t)1e3 * 24 * 3600 +
                PyLong_AsLong(secs) * (int64_t)1e3 +
                PyLong_AsLong(usecs) / 1e3;
    if (PyErr_Occurred()) {
        return -1;
    }
    Py_DECREF(delta);
    Py_DECREF(days);
    Py_DECREF(secs);
    Py_DECREF(usecs);

    rwRecSetStartTime(obj->raw->rec, millisecs);
    return 0;
}

static PyObject *silkPyRWRec_get_stime_secs(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    double s_time = ((double)rwRecGetStartTime(obj->raw->rec)) / 1.0e3;

    return PyFloat_FromDouble(s_time);
}

static int silkPyRWRec_set_stime_secs(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    const char *errstr = ("The stime_epoch_secs value must be a "
                          "positive number");
    PyObject *pyfloat_val;
    PyObject *pyint_val;
    int64_t long_val;

    if (!PyNumber_Check(value)) {
        PyErr_SetString(PyExc_TypeError, errstr);
        return -1;
    }

    pyfloat_val = PyNumber_Multiply(value, thousand);
    if (pyfloat_val == NULL) {
        PyErr_SetString(PyExc_TypeError, errstr);
        return -1;
    }
    pyint_val = PyNumber_Int(pyfloat_val);
    Py_DECREF(pyfloat_val);
    if (pyint_val == NULL) {
        PyErr_SetString(PyExc_TypeError, errstr);
        return -1;
    }
    long_val = PyLong_AsLongLong(pyint_val);
    Py_DECREF(pyint_val);
    if (long_val < 0) {
        PyErr_SetString(PyExc_OverflowError, errstr);
        return -1;
    }

    rwRecSetStartTime(obj->raw->rec, long_val);
    return 0;
}

static PyObject *silkPyRWRec_get_tcpflags(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    silkPyTCPFlags *flags;

    flags = (silkPyTCPFlags *)silkPyTCPFlagsType.tp_alloc(&silkPyTCPFlagsType,
                                                          0);
    if (flags != NULL) {
        flags->val = rwRecGetFlags(obj->raw->rec);
    }

    return (PyObject *)flags;
}

static int silkPyRWRec_set_tcpflags(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    long val;

    if (PyString_Check(value)) {
        uint8_t  flagval;
        char    *repr = PyString_AS_STRING(value);
        int      rv   = skStringParseTCPFlags(&flagval, repr);
        if (rv != 0) {
            PyErr_Format(PyExc_ValueError, "Illegal TCP flags: %s", repr);
            return -1;
        }
        rwRecSetFlags(obj->raw->rec, flagval);
        return 0;
    }

    if (silkPyTCPFlags_Check(value)) {
        silkPyTCPFlags *flags = (silkPyTCPFlags *)value;
        rwRecSetFlags(obj->raw->rec, flags->val);
        return 0;
    }

    if (!PyInt_Check(value) && !PyLong_Check(value)) {
        PyErr_SetString(PyExc_TypeError, "Expected an integer");
        return -1;
    }

    val = PyLong_AsLong(value);
    if (PyErr_Occurred() || val < 0 || val > (long)UINT8_MAX) {
        obj_error("Illegal TCP flags value: %s", value);
        return -1;
    }

    rwRecSetFlags(obj->raw->rec, val);
    return 0;
}

static PyObject *silkPyRWRec_get_timeout_killed(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    uint8_t state;

    state = rwRecGetTcpState(obj->raw->rec);
    if (!(state & SK_TCPSTATE_EXPANDED)) {
        Py_RETURN_NONE;
    }
    return PyBool_FromLong(state & SK_TCPSTATE_TIMEOUT_KILLED);
}

static int silkPyRWRec_set_timeout_killed(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    int     rv;
    uint8_t state;

    rv = PyObject_IsTrue(value);
    if (rv == -1) {
        return -1;
    }
    state = rwRecGetTcpState(obj->raw->rec) | SK_TCPSTATE_EXPANDED;
    if (rv) {
        state |= SK_TCPSTATE_TIMEOUT_KILLED;
    } else {
        state &= ~SK_TCPSTATE_TIMEOUT_KILLED;
    }
    rwRecSetTcpState(obj->raw->rec, state);

    return 0;
}

static PyObject *silkPyRWRec_get_timeout_started(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    uint8_t state;

    state = rwRecGetTcpState(obj->raw->rec);
    if (!(state & SK_TCPSTATE_EXPANDED)) {
        Py_RETURN_NONE;
    }
    return PyBool_FromLong(state & SK_TCPSTATE_TIMEOUT_STARTED);
}

static int silkPyRWRec_set_timeout_started(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    int     rv;
    uint8_t state;

    rv = PyObject_IsTrue(value);
    if (rv == -1) {
        return -1;
    }
    state = rwRecGetTcpState(obj->raw->rec) | SK_TCPSTATE_EXPANDED;
    if (rv) {
        state |= SK_TCPSTATE_TIMEOUT_STARTED;
    } else {
        state &= ~SK_TCPSTATE_TIMEOUT_STARTED;
    }
    rwRecSetTcpState(obj->raw->rec, state);

    return 0;
}

static PyObject *silkPyRWRec_get_type(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    char          typename[SK_MAX_STRLEN_FLOWTYPE+1];
    flowtypeID_t  flowtype = rwRecGetFlowType(obj->raw->rec);

    CHECK_SITE;

    sksiteFlowtypeGetType(typename, sizeof(typename), flowtype);
    return PyString_InternFromString(typename);
}

static PyObject *silkPyRWRec_get_etime(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    PyObject *s_time, *dur;
    PyObject *retval;

    s_time = silkPyRWRec_get_stime(obj, NULL);
    if (s_time == NULL) {
        return NULL;
    }
    dur = silkPyRWRec_get_duration(obj, NULL);
    if (dur == NULL) {
        Py_DECREF(s_time);
        return NULL;
    }

    retval = PyNumber_Add(s_time, dur);

    Py_DECREF(s_time);

    return retval;
}

static int silkPyRWRec_set_etime(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    PyObject *s_time, *dur;
    int       retval;

    s_time = silkPyRWRec_get_stime(obj, NULL);
    if (s_time == NULL) {
        return -1;
    }

    dur = PyNumber_Subtract(value, s_time);
    Py_DECREF(s_time);
    if (dur == NULL) {
        return -1;
    }

    retval = silkPyRWRec_set_duration(obj, dur, NULL);
    Py_DECREF(dur);

    return retval;
}

static PyObject *silkPyRWRec_get_etime_secs(
    silkPyRWRec *obj,
    void        *UNUSED(closure))
{
    double duration = (double)(rwRecGetStartTime(obj->raw->rec) +
                               rwRecGetElapsed(obj->raw->rec)) / 1.0e3;

    return PyFloat_FromDouble(duration);
}

static int silkPyRWRec_set_etime_secs(
    silkPyRWRec *obj,
    PyObject    *value,
    void        *UNUSED(closure))
{
    PyObject *s_time, *dur;
    int       retval;

    s_time = silkPyRWRec_get_stime_secs(obj, NULL);
    if (s_time == NULL) {
        return -1;
    }

    dur = PyNumber_Subtract(value, s_time);
    Py_DECREF(s_time);
    if (dur == NULL) {
        return -1;
    }

    retval = silkPyRWRec_set_duration_secs(obj, dur, NULL);
    Py_DECREF(dur);

    return retval;
}

static PyObject *silkPyRWRec_is_web(silkPyRWRec *obj)
{
    uint16_t sport = rwRecGetSPort(obj->raw->rec);
    uint16_t dport = rwRecGetDPort(obj->raw->rec);
    return PyBool_FromLong(
        rwRecGetProto(obj->raw->rec) == 6 &&
        (sport == 80 || dport == 80 ||
         sport == 443 || dport == 443 ||
         sport == 8080 || dport == 8080));
}

static PyObject *silkPyRWRec_richcompare(
    silkPyRWRec *self,
    PyObject    *obj,
    int          cmp)
{
    int rv;

    if ((cmp != Py_EQ && cmp != Py_NE) ||
        !silkPyRWRec_Check(obj))
    {
        Py_INCREF(Py_NotImplemented);
        return Py_NotImplemented;
    }

    rv = memcmp(self->raw->rec, ((silkPyRWRec *)obj)->raw->rec, sizeof(*self->raw->rec));
    rv = (rv == 0) ? 1 : 0;
    if (cmp == Py_NE) {
        rv = !rv;
    }

    return PyBool_FromLong(rv);
}

static PyGetSetDef silkPyRWRec_getseters[] = {
    {"application",
     (getter)silkPyRWRec_get_application, (setter)silkPyRWRec_set_application,
     "\"service\" port set by the collector", NULL},
    {"bytes",
     (getter)silkPyRWRec_get_bytes,       (setter)silkPyRWRec_set_bytes,
     "Count of bytes", NULL},
    {"classname",
     (getter)silkPyRWRec_get_class,       NULL,
     "class name (read-only)", NULL},
    {"classtype",
     (getter)silkPyRWRec_get_classtype,   (setter)silkPyRWRec_set_classtype,
     "class name, type name pair", NULL},
    {"dip",
     (getter)silkPyRWRec_get_dip,         (setter)silkPyRWRec_set_dip,
     "destination IP", NULL},
    {"dport",
     (getter)silkPyRWRec_get_dport,       (setter)silkPyRWRec_set_dport,
     "Destination port", NULL},
    {"duration",
     (getter)silkPyRWRec_get_duration,    (setter)silkPyRWRec_set_duration,
     "duration of flow as datetime.timedelta", NULL},
    {"duration_secs",
     (getter)silkPyRWRec_get_duration_secs,
     (setter)silkPyRWRec_set_duration_secs,
     "duration of flow in seconds", NULL},
    {"etime",
     (getter)silkPyRWRec_get_etime,       (setter)silkPyRWRec_set_etime,
     "end time of flow as datetime.timedelta", NULL},
    {"etime_epoch_secs",
     (getter)silkPyRWRec_get_etime_secs,  (setter)silkPyRWRec_set_etime_secs,
     "end time of flow as a number of seconds since the epoch time", NULL},
    {"finnoack",
     (getter)silkPyRWRec_get_finnoack,    (setter)silkPyRWRec_set_finnoack,
     "FIN followed by not ACK", NULL},
    {"icmpcode",
     (getter)silkPyRWRec_get_icmpcode,    (setter)silkPyRWRec_set_icmpcode,
     "ICMP code", NULL},
    {"icmptype",
     (getter)silkPyRWRec_get_icmptype,    (setter)silkPyRWRec_set_icmptype,
     "ICMP type", NULL},
    {"initflags",
     (getter)silkPyRWRec_get_initflags,   (setter)silkPyRWRec_set_initflags,
     "TCP flags of first packet", NULL},
    {"input",
     (getter)silkPyRWRec_get_input,       (setter)silkPyRWRec_set_input,
     "router incoming SNMP interface", NULL},
    {"nhip",
     (getter)silkPyRWRec_get_nhip,        (setter)silkPyRWRec_set_nhip,
     "router next hop IP", NULL},
    {"output",
     (getter)silkPyRWRec_get_output,      (setter)silkPyRWRec_set_output,
     "router outgoing SNMP interface", NULL},
    {"packets",
     (getter)silkPyRWRec_get_packets,     (setter)silkPyRWRec_set_packets,
     "count of packets", NULL},
    {"protocol",
     (getter)silkPyRWRec_get_protocol,    (setter)silkPyRWRec_set_protocol,
     "IP protocol", NULL},
    {"restflags",
     (getter)silkPyRWRec_get_restflags,   (setter)silkPyRWRec_set_restflags,
     "TCP flags on non-initial packets", NULL},
    {"sensor",
     (getter)silkPyRWRec_get_sensor,      (setter)silkPyRWRec_set_sensor,
     "sensor ID", NULL},
    {"sip",
     (getter)silkPyRWRec_get_sip,         (setter)silkPyRWRec_set_sip,
     "source IP", NULL},
    {"sport",
     (getter)silkPyRWRec_get_sport,       (setter)silkPyRWRec_set_sport,
     "source port", NULL},
    {"stime",
     (getter)silkPyRWRec_get_stime,       (setter)silkPyRWRec_set_stime,
     "start time of flow as datetime.datetime", NULL},
    {"stime_epoch_secs",
     (getter)silkPyRWRec_get_stime_secs,  (setter)silkPyRWRec_set_stime_secs,
     "start time of flow as a number of seconds since the epoch time", NULL},
    {"tcpflags",
     (getter)silkPyRWRec_get_tcpflags,    (setter)silkPyRWRec_set_tcpflags,
     "OR of all tcpflags", NULL},
    {"timeout_killed",
     (getter)silkPyRWRec_get_timeout_killed,
     (setter)silkPyRWRec_set_timeout_killed,
     "flow ended prematurely due to timeout by the collector", NULL},
    {"timeout_started",
     (getter)silkPyRWRec_get_timeout_started,
     (setter)silkPyRWRec_set_timeout_started,
     "flow is a continuation of a flow timed-out by the collector", NULL},
    {"typename",
     (getter)silkPyRWRec_get_type,        NULL,
     "type name (read-only)", NULL},
    {NULL, NULL, NULL, NULL, NULL}
};

static PyMethodDef silkPyRWRec_methods[] = {
    {"is_web", (PyCFunction)silkPyRWRec_is_web, METH_NOARGS,
     "Returns whether IPv6 can be stored in a WWW file format"},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static PyTypeObject silkPyRWRecType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.RWRecBase",    /* tp_name */
    sizeof(silkPyRWRec),        /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyRWRec_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_RICHCOMPARE,
                                /* tp_flags */
    "Base RW Record",           /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    (richcmpfunc)silkPyRWRec_richcompare, /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    silkPyRWRec_methods,        /* tp_methods */
    0,                          /* tp_members */
    silkPyRWRec_getseters,      /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyRWRec_init, /* tp_init */
    0,                          /* tp_alloc */
    0,                          /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

typedef struct silkPyRWIO_st {
    PyObject_HEAD
    skstream_t *io;
} silkPyRWIO;

static PyTypeObject silkPyRWIOType;

#define silkPyRWIO_Check(op) \
    PyObject_TypeCheck(op, &silkPyRWIOType)

static void silkPyRWIO_dealloc(silkPyRWIO *obj)
{
    if (obj->io) {
        skStreamDestroy(&obj->io);
    }
    obj->ob_type->tp_free(obj);
}

static PyObject *silkPyRWIO_new(
    PyTypeObject *type,
    PyObject     *UNUSED(args),
    PyObject     *UNUSED(kwds))
{
    silkPyRWIO *self;

    self = (silkPyRWIO *)type->tp_alloc(type, 0);

    if (self != NULL) {
        self->io = NULL;
    }

    return (PyObject *)self;
}

static PyObject *throw_ioerror(silkPyRWIO *obj, int errcode)
{
    skStreamPrintLastErr(obj->io, errcode, error_printf);
    PyErr_SetString(PyExc_IOError, error_buffer);
    return NULL;
}

static int silkPyRWIO_init(
    silkPyRWIO *self,
    PyObject   *args,
    PyObject   *kwds)
{
    char             *filename;
    int               mode;
    int               format      = NOT_SET;
    int               policy      = NOT_SET;
    int               compr       = NOT_SET;
    PyObject         *annotations = NULL;
    PyObject         *invocations = NULL;
    sk_file_header_t *hdr;
    int               rv;

    static char *kwlist[] = {"filename", "mode", "compression",
                             "format", "policy", "invocations",
                             "notes", NULL};

    if (!PyArg_ParseTupleAndKeywords(args, kwds, "si|iiiO!O!", kwlist,
                                     &filename, &mode, &compr,
                                     &format, &policy,
                                     &PyList_Type, &invocations,
                                     &PyList_Type, &annotations))
    {
        return -1;
    }

    if (mode != SK_IO_READ && mode != SK_IO_WRITE && mode != SK_IO_APPEND) {
        PyErr_SetString(PyExc_ValueError, "Illegal mode");
        Py_DECREF(self);
        return -1;
    }
    if (self->io) {
        skStreamDestroy(&self->io);
    }
    rv = skStreamCreate(&self->io, mode, SK_CONTENT_SILK_FLOW);
    if (rv != 0) {
        throw_ioerror(self, rv);
        return -1;
    }
    rv = skStreamBind(self->io, filename);
    if (rv != 0) {
        throw_ioerror(self, rv);
        return -1;
    }

    hdr = skStreamGetSilkHeader(self->io);

    if (policy != NOT_SET) {
        rv = skStreamSetIPv6Policy(self->io, policy);
        if (rv != 0) {
            throw_ioerror(self, rv);
            return -1;
        }
    }

    if (compr != NOT_SET) {
        if (mode != SK_IO_WRITE) {
            PyErr_SetString(PyExc_ValueError,
                            "Cannot set compression unless in WRITE mode");
            return -1;
        }
        rv = skHeaderSetCompressionMethod(hdr, compr);
        if (rv != 0) {
            throw_ioerror(self, rv);
            return -1;
        }
    }

    if (format != NOT_SET) {
        if (mode != SK_IO_WRITE) {
            PyErr_SetString(PyExc_ValueError,
                            "Cannot set file format unless in WRITE mode");
            return -1;
        }
        rv = skHeaderSetFileFormat(hdr, format);
        if (rv != 0) {
            throw_ioerror(self, rv);
            return -1;
        }
    }

    if (annotations != NULL) {
        if (mode != SK_IO_WRITE) {
            PyErr_SetString(PyExc_ValueError,
                            "Cannot set file format unless in WRITE mode");
            return -1;
        }
        if (hdr != NULL) {
            ssize_t len = PyList_GET_SIZE(annotations);
            ssize_t i;

            for (i = 0; i < len; i++) {
                PyObject *item = PyList_GET_ITEM(annotations, i);

                if (!PyString_Check(item)) {
                    PyErr_SetString(PyExc_TypeError,
                                    "Annotation was not a string");
                    return -1;
                }

                rv = skHeaderAddAnnotation(hdr, PyString_AsString(item));
                if (rv != 0) {
                    throw_ioerror(self, rv);
                }
            }
        }
    }

    if (invocations != NULL) {
        if (mode != SK_IO_WRITE) {
            PyErr_SetString(PyExc_ValueError,
                            "Cannot set file format unless in WRITE mode");
            return -1;
        }
        if (hdr != NULL) {
            ssize_t len = PyList_GET_SIZE(invocations);
            ssize_t i;

            for (i = 0; i < len; i++) {
                PyObject *item = PyList_GET_ITEM(invocations, i);
                char     *value;

                if (!PyString_Check(item)) {
                    PyErr_SetString(PyExc_TypeError,
                                    "Invocation was not a string");
                    return -1;
                }

                value = PyString_AsString(item);
                rv = skHeaderAddInvocation(hdr, 0, 1, &value);
                if (rv != 0) {
                    throw_ioerror(self, rv);
                }
            }
        }
    }

    rv = skStreamOpen(self->io);
    if (rv != 0) {
        throw_ioerror(self, rv);
        return -1;
    }

    if (mode == SK_IO_WRITE) {
        rv = skStreamWriteSilkHeader(self->io);
        if (rv != 0) {
            throw_ioerror(self, rv);
            return -1;
        }
    } else {
        rv = skStreamReadSilkHeader(self->io, NULL);
        if (rv != 0) {
            throw_ioerror(self, rv);
            return -1;
        }
    }

    return 0;
}

static PyObject *silkPyRWIO_read(silkPyRWIO *obj)
{
    int rv;
    rwRec *rec;

    rec = PyMem_Malloc(sizeof(*rec));
    if (rec == NULL) {
        return PyErr_NoMemory();
    }

    rv = skStreamReadRecord(obj->io, rec);
    if (rv == 0) {
        PyObject *pyrec = silkPyRawRWRecType.tp_alloc(&silkPyRawRWRecType, 0);
        if (pyrec != NULL) {
            ((silkPyRawRWRec *)pyrec)->rec = rec;
        }
        return pyrec;
    }

    PyMem_Free(rec);

    if (rv == SKSTREAM_ERR_EOF) {
        Py_RETURN_NONE;
    }

    return throw_ioerror(obj, rv);
}

static PyObject *silkPyRWIO_write(silkPyRWIO *obj, PyObject *rec)
{
    int rv;

    if (!silkPyRWRec_Check(rec)) {
        PyErr_SetString(PyExc_TypeError, "Argument not a RWRec");
        return NULL;
    }

    rv = skStreamWriteRecord(obj->io, ((silkPyRWRec *)rec)->raw->rec);
    if (rv == 0) {
        Py_RETURN_NONE;
    }

    return throw_ioerror(obj, rv);
}


static PyObject *silkPyRWIO_close(silkPyRWIO *obj)
{
    int rv;

    rv = skStreamClose(obj->io);
    if (rv == 0) {
        Py_RETURN_NONE;
    }

    return throw_ioerror(obj, rv);
}

static PyObject *silkPyRWIO_get_annotations(silkPyRWIO *obj)
{
    sk_file_header_t     *hdr;
    sk_header_entry_t    *entry;
    sk_hentry_iterator_t  iter;
    PyObject             *list;
    PyObject             *annot;
    int                   rv;

    list = PyList_New(0);
    if (list == NULL) {
        return NULL;
    }
    hdr = skStreamGetSilkHeader(obj->io);
    if (hdr != NULL) {
        skHeaderIteratorBindType(&iter, hdr, SK_HENTRY_ANNOTATION_ID);
        while ((entry = skHeaderIteratorNext(&iter)) != NULL) {
            annot = PyString_FromString(
                ((sk_hentry_annotation_t *)entry)->annotation);
            if (annot == NULL) {
                Py_DECREF(list);
                return NULL;
            }
            rv = PyList_Append(list, annot);
            Py_DECREF(annot);
            if (rv != 0) {
                Py_DECREF(list);
                return NULL;
            }
        }
    }

    return list;
}

static PyObject *silkPyRWIO_get_invocations(silkPyRWIO *obj)
{
    sk_file_header_t     *hdr;
    sk_header_entry_t    *entry;
    sk_hentry_iterator_t  iter;
    PyObject             *list;
    PyObject             *invoc;
    int                   rv;

    list = PyList_New(0);
    if (list == NULL) {
        return NULL;
    }
    hdr = skStreamGetSilkHeader(obj->io);
    if (hdr != NULL) {
        skHeaderIteratorBindType(&iter, hdr, SK_HENTRY_INVOCATION_ID);
        while ((entry = skHeaderIteratorNext(&iter)) != NULL) {
            invoc = PyString_FromString(
                ((sk_hentry_invocation_t *)entry)->command_line);
            if (invoc == NULL) {
                Py_DECREF(list);
                return NULL;
            }
            rv = PyList_Append(list, invoc);
            Py_DECREF(invoc);
            if (rv != 0) {
                Py_DECREF(list);
                return NULL;
            }
        }
    }

    return list;
}

static PyObject *silkPyRWIO_get_name(
    silkPyRWIO *obj,
    void       *UNUSED(closure))
{
    const char *name = skStreamGetPathname(obj->io);
    if (name) {
        return PyString_FromString(name);
    }
    Py_RETURN_NONE;
}

static PyMethodDef silkPyRWIO_methods[] = {
    {"read", (PyCFunction)silkPyRWIO_read, METH_NOARGS,
     "Read a RWRec from a RW File"},
    {"write", (PyCFunction)silkPyRWIO_write, METH_O,
     "Write a RWRec to a RW File"},
    {"close", (PyCFunction)silkPyRWIO_close, METH_NOARGS,
     "Read a RWRec from a RW File"},
    {"notes", (PyCFunction)silkPyRWIO_get_annotations, METH_NOARGS,
     "Get the file's annotations"},
    {"invocations", (PyCFunction)silkPyRWIO_get_invocations, METH_NOARGS,
     "Get the file's invocations"},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};

static PyGetSetDef silkPyRWIO_getseters[] = {
    {"name", (getter)silkPyRWIO_get_name, NULL,
     "name of file associated with SilkFile", NULL},
    {NULL, NULL, NULL, NULL, NULL}    /* Sentinel */
};

static PyTypeObject silkPyRWIOType = {
    PyObject_HEAD_INIT(NULL)
    0,                          /* ob_size */
    "silk.pysilk.SilkFileBase", /* tp_name */
    sizeof(silkPyRWIO),         /* tp_basicsize */
    0,                          /* tp_itemsize */
    (destructor)silkPyRWIO_dealloc, /* tp_dealloc */
    0,                          /* tp_print */
    0,                          /* tp_getattr */
    0,                          /* tp_setattr */
    0,                          /* tp_compare */
    0,                          /* tp_repr */
    0,                          /* tp_as_number */
    0,                          /* tp_as_sequence */
    0,                          /* tp_as_mapping */
    0,                          /* tp_hash  */
    0,                          /* tp_call */
    0,                          /* tp_str */
    0,                          /* tp_getattro */
    0,                          /* tp_setattro */
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
    "Base Silk File",           /* tp_doc */
    0,                          /* tp_traverse */
    0,                          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
    0,                          /* tp_iter */
    0,                          /* tp_iternext */
    silkPyRWIO_methods,         /* tp_methods */
    0,                          /* tp_members */
    silkPyRWIO_getseters,       /* tp_getset */
    0,                          /* tp_base */
    0,                          /* tp_dict */
    0,                          /* tp_descr_get */
    0,                          /* tp_descr_set */
    0,                          /* tp_dictoffset */
    (initproc)silkPyRWIO_init,  /* tp_init */
    0,                          /* tp_alloc */
    silkPyRWIO_new,             /* tp_new */
    0,                          /* tp_free */
    0,                          /* tp_is_gc */
    0,                          /* tp_bases */
    0,                          /* tp_mro */
    0,                          /* tp_cache */
    0,                          /* tp_subclasses */
    0,                          /* tp_weaklist */
    0                           /* tp_del */
#if PY_VERSION_HEX >= 0x02060000
    ,0                          /* tp_version_tag */
#endif
};

static int init_rwfile_module(PyObject *mod)
{
    int rv;

    PyModule_AddIntConstant(mod, "IGNORE", SK_IPV6POLICY_IGNORE);
    PyModule_AddIntConstant(mod, "ASV4", SK_IPV6POLICY_ASV4);
    PyModule_AddIntConstant(mod, "MIX", SK_IPV6POLICY_MIX);
    PyModule_AddIntConstant(mod, "FORCE", SK_IPV6POLICY_FORCE);
    PyModule_AddIntConstant(mod, "ONLY", SK_IPV6POLICY_ONLY);

    PyModule_AddIntConstant(mod, "READ", SK_IO_READ);
    PyModule_AddIntConstant(mod, "WRITE", SK_IO_WRITE);
    PyModule_AddIntConstant(mod, "APPEND", SK_IO_APPEND);

    PyModule_AddIntConstant(mod, "DEFAULT", NOT_SET);
    PyModule_AddIntConstant(mod, "NO_COMPRESSION", SK_COMPMETHOD_NONE);
    PyModule_AddIntConstant(mod, "ZLIB", SK_COMPMETHOD_ZLIB);
    PyModule_AddIntConstant(mod, "LZO1X", SK_COMPMETHOD_LZO1X);

    PyModule_AddObject(mod, "BAG_COUNTER_MAX",
                       PyLong_FromUnsignedLongLong(SKBAG_COUNTER_MAX));

    if (PyType_Ready(&silkPyRWIOType) < 0) {
        return -1;
    }
    Py_INCREF(&silkPyRWIOType);
    rv = PyModule_AddObject(mod, "SilkFileBase", (PyObject *)&silkPyRWIOType);
    if (rv != 0) {
        return -1;
    }

    return 0;
}

static silkPyRawRWRec *rwrec_steal_from_c(
    PyObject *UNUSED(self),
    PyObject *c_rec)
{
    silkPyRawRWRec *pyrec;

    if (!PyCObject_Check(c_rec)) {
        PyErr_SetString(PyExc_TypeError, "Illegal argument type");
        return NULL;
    }

    pyrec = (silkPyRawRWRec *)silkPyRawRWRecType.tp_alloc(
        &silkPyRawRWRecType, 0);
    if (pyrec != NULL) {
        pyrec->rec = (rwRec *)PyCObject_AsVoidPtr(c_rec);
    }

    return pyrec;
}

static silkPyRawRWRec *rwrec_copy_from_c(
    PyObject *UNUSED(self),
    PyObject *c_rec)
{
    silkPyRawRWRec *pyrec;
    rwRec       *rec;

    if (!PyCObject_Check(c_rec)) {
        PyErr_SetString(PyExc_TypeError, "Illegal argument type");
        return NULL;
    }

    pyrec = (silkPyRawRWRec *)silkPyRawRWRecType.tp_alloc(
        &silkPyRawRWRecType, 0);
    if (pyrec != NULL) {
        rec = PyCObject_AsVoidPtr(c_rec);
        if (rec != NULL) {
            pyrec->rec = PyMem_Malloc(sizeof(*rec));
            if (pyrec->rec == NULL) {
                Py_DECREF(pyrec);
                PyErr_NoMemory();
                return NULL;
            }
            RWREC_COPY(pyrec->rec, rec);
        }
    }

    return pyrec;
}


static PyObject *ipv6_enabled(void)
{
#if SK_ENABLE_IPV6
    Py_RETURN_TRUE;
#else
    Py_RETURN_FALSE;
#endif
}

static PyObject *initial_tcpflags_enabled(void)
{
    Py_RETURN_TRUE;
}

static PyObject *have_site_config(void)
{
    CHECK_SITE;

    Py_INCREF(havesite);
    return havesite;
}

static PyObject *sensor_list(void)
{
    CHECK_SITE;

    Py_INCREF(sensors);
    return sensors;
}

static PyObject *flowtype_list(void)
{
    CHECK_SITE;

    Py_INCREF(flowtypes);
    return flowtypes;
}

static PyObject *class_list(void)
{
    CHECK_SITE;

    Py_INCREF(classes);
    return classes;
}

static PyObject *silk_version(void)
{
    return PyString_InternFromString(SK_PACKAGE_VERSION);
}

static PyObject *silk_init_site(
    PyObject *UNUSED(self),
    PyObject *args)
{
    int   rv;
    char *filename = NULL;

    rv = PyArg_ParseTuple(args, "|s", &filename);
    if (!rv) {
        return NULL;
    }

    if (site_configured) {
        PyErr_SetString(PyExc_RuntimeError, "Site already initialized");
        return NULL;
    }

    if (filename) {
        assert(!site_configured);
        rv = sksiteSetConfigPath(filename);
        assert(rv == 0);
    }

    CHECK_SITE;

    if (havesite == Py_False) {
        PyErr_SetString(PyExc_RuntimeError, "Site file does not exist");
        return NULL;
    }

    if (filename) {
        /* This is needed for subprocess calls to silk, such as from
         * silk.FGlob */
        PyObject *os;
        static char env_buf[101 + PATH_MAX];

        /* setenv() does not exist on Solaris 9, so we use putenv()
         * instead */
        rv = snprintf(env_buf, sizeof(env_buf),
                      "%s=%s", SILK_CONFIG_FILE_ENVAR, filename);
        if (rv >= (int)sizeof(env_buf) || putenv(env_buf) != 0) {
            PyErr_SetString(PyExc_RuntimeError,
                            ("Could not set " SILK_CONFIG_FILE_ENVAR));
            return NULL;
        }

        /* Attempt to add the environment variable to Python's
         * environment list as well */
        os = PyImport_ImportModule("os");
        if (os != NULL) {
            PyObject *env = PyObject_GetAttrString(os, "environ");
            if (env != NULL) {
                PyObject *s = PyString_FromString(filename);
                if (s != NULL) {
                    PyMapping_SetItemString(env, SILK_CONFIG_FILE_ENVAR, s);
                    Py_DECREF(s);
                }
                Py_DECREF(env);
            }
            Py_DECREF(os);
        }
    }

    Py_RETURN_NONE;
}

static PyObject *init_country_codes(
    PyObject *UNUSED(self),
    PyObject *args)
{
    int   rv;
    char *filename = NULL;

    rv = PyArg_ParseTuple(args, "|s", &filename);
    if (!rv) {
        return NULL;
    }

    skCountryTeardown();
    rv = skCountrySetup(filename, error_printf);
    if (rv != 0) {
        PyErr_SetString(PyExc_RuntimeError, error_buffer);
        return NULL;
    }

    Py_RETURN_NONE;
}


static PyMethodDef silk_methods[] = {
    {"ipv6_enabled", (PyCFunction)ipv6_enabled, METH_NOARGS,
     "Return whether IPv6 was enabled at compile-time"},
    {"initial_tcpflags_enabled", (PyCFunction)initial_tcpflags_enabled,
     METH_NOARGS,
     "Return whether initial tcpflagewere enabled at compile-time"},
    {"init_site", (PyCFunction)silk_init_site, METH_VARARGS,
     "Initialize the silk site using the given filename.\n"
     "If the filename is not supplied, it will use the value of\n"
     "the environment variable " SILK_CONFIG_FILE_ENVAR " if available,\n"
     "or the value compiled into the silk libraries otherwise.\n"
     "Will throw an exception if the site is already initialized."},
    {"have_site_config", (PyCFunction)have_site_config, METH_NOARGS,
     "Return whether the site configuration file has been loaded"},
    {"sensor_info", (PyCFunction)sensor_list, METH_NOARGS,
     "Returns a list of information for configured sensors"},
    {"class_info", (PyCFunction)class_list, METH_NOARGS,
     "Return a list of information for configured classes"},
    {"flowtype_info", (PyCFunction)flowtype_list, METH_NOARGS,
     "Return a list of information for configured flowtypes"},
    {"silk_version", (PyCFunction)silk_version, METH_NOARGS,
     "Return the version of SiLK that PySilk was linked against"},
    {"init_country_codes", (PyCFunction)init_country_codes, METH_VARARGS,
     "Initialize the country codes from the given file (can be left blank)"},
    {"_raw_rwrec_steal", (PyCFunction)rwrec_steal_from_c, METH_O,
     "Create a RawRWRec from a C rwrec PyCObject, stealing the reference"},
    {"_raw_rwrec_copy", (PyCFunction)rwrec_copy_from_c, METH_O,
     "Create a RawRWRec from a C rwrec PyCObject, copying the value"},
    {NULL, NULL, 0, NULL}       /* Sentinel */
};


static int init_sensors(void)
{
    sensor_iter_t  sensor_iter;
    sensorID_t     sensor;
    char           name[SK_MAX_STRLEN_SENSOR+1];
    PyObject      *list = NULL;
    PyObject      *dict = NULL;
    int            rv;

    if (sensors != NULL) {
        return 0;
    }

    sensors = PyDict_New();
    if (sensors == NULL) {
        goto error;
    }

    sksiteSensorIterator(&sensor_iter);
    while (sksiteSensorIteratorNext(&sensor_iter, &sensor)) {
        class_iter_t  class_iter;
        classID_t     class;
        PyObject     *val;

        dict = PyDict_New();
        if (dict == NULL) {
            goto error;
        }
        val = PyInt_FromLong(sensor);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItem(sensors, val, dict);
        if (rv != 0) {
            Py_DECREF(val);
            goto error;
        }
        rv = PyDict_SetItemString(dict, "id", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        sksiteSensorGetName(name, sizeof(name), sensor);
        val = PyString_InternFromString(name);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "name", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        list = PyList_New(0);
        if (list == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "classes", list);
        if (rv != 0) {
            goto error;
        }
        sksiteSensorClassIterator(sensor, &class_iter);
        while (sksiteClassIteratorNext(&class_iter, &class)) {
            val = PyInt_FromLong(class);
            if (val == NULL) {
                goto error;
            }
            rv = PyList_Append(list, val);
            Py_DECREF(val);
            if (rv != 0) {
                goto error;
            }
        }
        Py_CLEAR(list);

        Py_CLEAR(dict);
    }

    return 0;

  error:
    Py_XDECREF(list);
    Py_XDECREF(dict);
    Py_CLEAR(sensors);

    return -1;
}

static int init_flowtypes(void)
{
    flowtype_iter_t  flowtype_iter;
    flowtypeID_t     flowtype;
    char             name[SK_MAX_STRLEN_SENSOR+1];
    PyObject        *dict = NULL;
    int              rv;

    if (flowtypes != NULL) {
        return 0;
    }

    flowtypes = PyDict_New();
    if (flowtypes == NULL) {
        goto error;
    }

    sksiteFlowtypeIterator(&flowtype_iter);
    while (sksiteFlowtypeIteratorNext(&flowtype_iter, &flowtype)) {
        classID_t     class;
        PyObject     *val;

        dict = PyDict_New();
        if (dict == NULL) {
            goto error;
        }
        val = PyInt_FromLong(flowtype);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItem(flowtypes, val, dict);
        if (rv != 0) {
            Py_DECREF(val);
            goto error;
        }
        rv = PyDict_SetItemString(dict, "id", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        sksiteFlowtypeGetName(name, sizeof(name), flowtype);
        val = PyString_InternFromString(name);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "name", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        sksiteFlowtypeGetType(name, sizeof(name), flowtype);
        val = PyString_InternFromString(name);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "type", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        class = sksiteFlowtypeGetClassID(flowtype);
        val = PyInt_FromLong(class);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "class", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        Py_CLEAR(dict);
    }

    return 0;

  error:
    Py_XDECREF(dict);
    Py_CLEAR(sensors);

    return -1;
}

static int init_classes(void)
{
    class_iter_t  iter;
    classID_t     class;
    char          name[SK_MAX_STRLEN_FLOWTYPE+1];
    PyObject     *list = NULL;
    PyObject     *dict = NULL;
    int           rv;

    if (classes != NULL) {
        return 0;
    }

    classes = PyDict_New();
    if (classes == NULL) {
        return -1;
    }

    sksiteClassIterator(&iter);
    while (sksiteClassIteratorNext(&iter, &class)) {
        sensor_iter_t    sensor_iter;
        sensorID_t       sensor;
        flowtype_iter_t  flowtype_iter;
        flowtypeID_t     flowtype;
        PyObject        *val;

        dict = PyDict_New();
        if (dict == NULL) {
            goto error;
        }
        val = PyInt_FromLong(class);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItem(classes, val, dict);
        if (rv != 0) {
            Py_DECREF(val);
            goto error;
        }
        rv = PyDict_SetItemString(dict, "id", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        sksiteClassGetName(name, sizeof(name), class);
        val = PyString_InternFromString(name);
        if (val == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "name", val);
        Py_DECREF(val);
        if (rv != 0) {
            goto error;
        }

        list = PyList_New(0);
        if (list == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "sensors", list);
        if (rv != 0) {
            goto error;
        }
        sksiteClassSensorIterator(class, &sensor_iter);
        while (sksiteSensorIteratorNext(&sensor_iter, &sensor)) {
            val = PyInt_FromLong(sensor);
            if (val == NULL) {
                goto error;
            }
            rv = PyList_Append(list, val);
            Py_DECREF(val);
            if (rv != 0) {
                goto error;
            }
        }
        Py_DECREF(list);

        list = PyList_New(0);
        if (list == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "flowtypes", list);
        if (rv != 0) {
            goto error;
        }
        sksiteClassFlowtypeIterator(class, &flowtype_iter);
        while (sksiteFlowtypeIteratorNext(&flowtype_iter, &flowtype)) {
            val = PyInt_FromLong(flowtype);
            if (val == NULL) {
                goto error;
            }
            rv = PyList_Append(list, val);
            Py_DECREF(val);
            if (rv != 0) {
                goto error;
            }
        }
        Py_DECREF(list);

        list = PyList_New(0);
        if (list == NULL) {
            goto error;
        }
        rv = PyDict_SetItemString(dict, "default_flowtypes", list);
        if (rv != 0) {
            goto error;
        }
        sksiteClassDefaultFlowtypeIterator(class, &flowtype_iter);
        while (sksiteFlowtypeIteratorNext(&flowtype_iter, &flowtype)) {
            val = PyInt_FromLong(flowtype);
            if (val == NULL) {
                goto error;
            }
            rv = PyList_Append(list, val);
            Py_DECREF(val);
            if (rv != 0) {
                goto error;
            }
        }
        Py_CLEAR(list);

        Py_CLEAR(dict);
    }

    return 0;

  error:
    Py_XDECREF(list);
    Py_XDECREF(dict);
    Py_CLEAR(sensors);

    return -1;
}

static int init_site(void)
{
    int rv = sksiteConfigure(0);
    if (rv == 0) {
        havesite = Py_True;
    } else if (rv == -2) {
        havesite = Py_False;
    } else {
        skAppPrintErr("error parsing site configuration file");
        return -1;
    }
    Py_INCREF(havesite);
    assert(havesite != NULL);
    rv = PyModule_AddObject(silkmod, "_havesite", havesite);
    assert(rv == 0);

    rv = init_sensors();
    if (rv != 0) {
        return -1;
    }
    rv = PyModule_AddObject(silkmod, "_sensors", sensors);
    assert(rv == 0);

    rv = init_classes();
    if (rv != 0) {
        return -1;
    }
    rv = PyModule_AddObject(silkmod, "_classes", classes);
    assert(rv == 0);

    rv = init_flowtypes();
    if (rv != 0) {
        return -1;
    }
    rv = PyModule_AddObject(silkmod, "_flowtypes", flowtypes);
    assert(rv == 0);

    return 0;
}

static void initpysilkbase(char* name)
{
    PyObject *tmp;
    int       rv;

    PyDateTime_IMPORT;

    silkmod = Py_InitModule3(name, silk_methods, "SiLK extension module");
    if (silkmod == NULL) {
        skAppPrintErr("Could not create module silk");
        goto err;
    }

    if (init_rwfile_module(silkmod)) {
        goto err;
    }

    if (PyType_Ready(&silkPyIPAddrType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyIPAddrType);
    rv = PyModule_AddObject(silkmod, "IPAddr", (PyObject *)&silkPyIPAddrType);
    assert(rv == 0);

    if (PyType_Ready(&silkPyIPv4AddrType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyIPv4AddrType);
    rv = PyModule_AddObject(silkmod, "IPv4Addr",
                            (PyObject *)&silkPyIPv4AddrType);
    assert(rv == 0);

    if (PyType_Ready(&silkPyIPv6AddrType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyIPv6AddrType);
    rv = PyModule_AddObject(silkmod, "IPv6Addr",
                            (PyObject *)&silkPyIPv6AddrType);
    assert(rv == 0);

    if (PyType_Ready(&silkPyIPWildcardType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyIPWildcardType);
    rv = PyModule_AddObject(
        silkmod, "IPWildcard", (PyObject *)&silkPyIPWildcardType);
    assert(rv == 0);

    silkPyIPWildcardIterType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyIPWildcardIterType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyIPWildcardIterType);
    rv = PyModule_AddObject(silkmod, "IPWildcardIter",
                            (PyObject *)&silkPyIPWildcardIterType);
    assert(rv == 0);

    silkPyIPSetType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyIPSetType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyIPSetType);
    rv = PyModule_AddObject(silkmod, "IPSetBase",
                            (PyObject *)&silkPyIPSetType);
    assert(rv == 0);

    silkPyIPSetIterType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyIPSetIterType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyIPSetIterType);
    rv = PyModule_AddObject(silkmod, "IPSetIter",
                            (PyObject *)&silkPyIPSetIterType);
    assert(rv == 0);

    silkPyPmapType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyPmapType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyPmapType);
    rv = PyModule_AddObject(silkmod, "PMapBase", (PyObject *)&silkPyPmapType);
    assert(rv == 0);

    silkPyPmapIterType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyPmapIterType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyPmapIterType);
    rv = PyModule_AddObject(silkmod, "PMapBaseIter",
                            (PyObject *)&silkPyPmapIterType);
    assert(rv == 0);

    silkPyBagType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyBagType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyBagType);
    rv = PyModule_AddObject(silkmod, "BagBase", (PyObject *)&silkPyBagType);
    assert(rv == 0);

    silkPyBagIterType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyBagIterType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyBagIterType);
    rv = PyModule_AddObject(silkmod, "BagBaseIter",
                            (PyObject *)&silkPyBagIterType);
    assert(rv == 0);

    if (PyType_Ready(&silkPyTCPFlagsType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyTCPFlagsType);
    rv = PyModule_AddObject(silkmod, "TCPFlags",
                            (PyObject *)&silkPyTCPFlagsType);
    assert(rv == 0);

    if (init_tcpflags_constants(silkmod)) {
        goto err;
    }

    if (PyType_Ready(&silkPyRawRWRecType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyRawRWRecType);
    rv = PyModule_AddObject(silkmod, "RawRWRec",
                            (PyObject *)&silkPyRawRWRecType);
    assert(rv == 0);

    silkPyRWRecType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&silkPyRWRecType) < 0) {
        goto err;
    }
    Py_INCREF(&silkPyRWRecType);
    rv = PyModule_AddObject(silkmod, "RWRecBase",
                            (PyObject *)&silkPyRWRecType);
    assert(rv == 0);

    tmp = PyImport_ImportModule("datetime");
    if (tmp == NULL) {
        skAppPrintErr("Failed to import datetime module");
        goto err;
    }

    timedelta = PyObject_GetAttrString(tmp, "timedelta");
    assert(timedelta != NULL);
    rv = PyModule_AddObject(silkmod, "_timedelta", timedelta);
    assert(rv == 0);

    datetime = PyObject_GetAttrString(tmp, "datetime");
    assert(datetime != NULL);
    rv = PyModule_AddObject(silkmod, "_datetime", datetime);
    assert(rv == 0);

    Py_DECREF(tmp);

    maxelapsed = PyObject_CallFunction(timedelta, "iiik", 0, 0, 0, UINT32_MAX);
    assert(maxelapsed != NULL);
    rv = PyModule_AddObject(silkmod, "_maxelapsed", maxelapsed);
    assert(rv == 0);

    minelapsed = PyObject_CallObject(timedelta, NULL);
    assert(minelapsed != NULL);
    rv = PyModule_AddObject(silkmod, "_minelapsed", minelapsed);
    assert(rv == 0);

    zerotime = PyObject_CallFunction(datetime, "iii", 1970, 1, 1);
    assert(zerotime != NULL);
    rv = PyModule_AddObject(silkmod, "_zerotime", zerotime);
    assert(rv == 0);

    thousand = PyFloat_FromDouble(1000.0);
    assert(thousand != NULL);
    rv = PyModule_AddObject(silkmod, "_thousand", thousand);
    assert(rv == 0);

    maxintipv4 = PyLong_FromString("0xffffffff", NULL, 0);
    assert(maxintipv4 != NULL);
    rv = PyModule_AddObject(silkmod, "_maxintipv4", maxintipv4);
    assert(rv == 0);

#if SK_ENABLE_IPV6
    maxintipv6 = PyLong_FromString("0xffffffffffffffffffffffffffffffff",
                                   NULL, 0);
    assert(maxintipv6 != NULL);
    rv = PyModule_AddObject(silkmod, "_maxintipv6", maxintipv6);
    assert(rv == 0);
#endif  /* SK_ENABLE_IPV6 */

    return;

  err:
    if (PyErr_Occurred()) {
        PyErr_Print();
    }
    exit(EXIT_FAILURE);
}

PyMODINIT_FUNC initpysilk(void)
{
    skAppRegister(Py_GetProgramName());
    initpysilkbase("pysilk");
}

PyMODINIT_FUNC initpysilk_nl(void)
{
    initpysilkbase("pysilk_nl");
}

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
