/*
** Copyright (C) 2003-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  Reads PDUs from a router and writes the flows into packed files.
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwflowpack.c 96187c93d013 2012-06-26 20:45:55Z mthomas $");

#include <dlfcn.h>

#include <silk/sksite.h>
#include "rwflowpack_priv.h"
#include "stream-cache.h"
#include <silk/sktimer.h>
#include <silk/skvector.h>
#include <silk/skdaemon.h>
#include <silk/skplugin.h>
#include <silk/skpolldir.h>

/*
 *    MAX FILE HANDLE NOTES
 *
 *    In response to attempts to use 100+ probes that polled
 *    directories which caused us to run out of file handles, we tried
 *    to make some of the code smarter about the number of files
 *    handles we use.
 *
 *    However, currently we only look at polldir numbers, and we do
 *    not consider the number of file handles that we have open to
 *    read from the network.  One issue is we don't know how many that
 *    is until after we start.
 *
 *    We could be smarter and set the number of poll dir handles after
 *    we see how many polldirs we are actually using.
 *
 *    We could use sysconf(_SC_OPEN_MAX) to get the max number of file
 *    handles available and set our values based on that.
 */


/* MACROS AND DATA TYPES */

/* Where to write usage (--help) information */
#define USAGE_FH stdout

/* The maximum number of open output files to support, which is the
 * size of the stream_cache.  This default may be changed with the
 * --file-cache-size switch.  Also specify the minimum size of stream
 * cache. */
#define STREAM_CACHE_SIZE 128
#define STREAM_CACHE_MIN  4

/* These next two values are used when rwflowpack is using probes that
 * poll directories, and they specify fractions of the
 * stream_cache_size.
 *
 * The first is the maximum number of input files to read from
 * simultaneously.  The second is the maximum number of simultaneous
 * directory polls to perform.
 *
 * In addition, specify the absolute minumum for those values. */
#define INPUT_FILEHANDLES_FRACTION   (1.0/8.0)
#define POLLDIR_FILEHANDLES_FRACTION (1.0/16.0)

#define INPUT_FILEHANDLES_MIN      2
#define POLLDIR_FILEHANDLES_MIN    1

/* How often, in seconds, to flush the files in the stream_cache.
 * This default may be changed with the --flush-timeout switch. */
#define FLUSH_TIMEOUT  120

/* Number of seconds to wait between polling the incoming directory or
 * the poll-directory's specified in the sensor.conf file.  This
 * default may be changed with the --polling-interval switch. */
#define POLLING_INTERVAL 15

/* Helper macro for messages */
#define CHECK_PLURAL(cp_x) ((1 == (cp_x)) ? "" : "s")

/* The number of reader types to allow.  Used to set array sizes. */
#define MAX_READER_TYPE_COUNT 9

/* Default record version to write. */
#ifndef RWFLOWPACK_DEFAULT_VERSION
#  define RWFLOWPACK_DEFAULT_VERSION SK_RECORD_VERSION_ANY
#endif

/* The signal the reader thread (the manageProcessor() function) sends
 * to main thread to indicate that the reader thread is done.
 * Normally SIGUSR2, but you may need to use a different value when
 * running under a debugger, valgrind, etc. */
#ifndef READER_DONE_SIGNAL
#  define READER_DONE_SIGNAL  SIGUSR2
#endif

/* Must be greater than the number of switches defined */
#define MAX_OPTION_COUNT 32

/* In each of the above mode, an option can be required, optional,
 * illegal, or non-sensical.  This enumerates those values; the
 * mode_options[][] array holds the values for each option for each
 * mode. */
typedef enum {
    MODOPT_ILLEGAL = 0, MODOPT_REQUIRED, MODOPT_OPTIONAL, MODOPT_NONSENSE
} mode_option_t;

typedef void (*cleanupHandler)(void *);

/* signal list */
struct siglist_st {
    int signal;
    char *name;
};

/* Create an array of these to cache the options we get from the
 * user */
typedef struct opt_cache_st {
    int     seen;
    char   *value;
} opt_cache_t;


/* IDs for the reader types.  Keep in sync with reader_type_inits[]. */
typedef enum {
    READER_TYPE_FLOWCAP_FILES,
#if SK_ENABLE_IPFIX
    READER_TYPE_IPFIX,
#endif
    READER_TYPE_PDU,
    READER_TYPE_PDU_FILE,
    READER_TYPE_DIRECTORY,
    READER_TYPE_RESPOOL,
    _READER_TYPE_MAX_
} reader_type_id_t;

/* LOCAL VARIABLES */

/* initialize functions for the various flow readers.  Keep in sync
 * with reader_type_id_t. */
static fr_initialize_fn_t reader_type_inits[] = {
    &fcFilesReaderInitialize,
#if SK_ENABLE_IPFIX
    &ipfixReaderInitialize,
#endif
    &pduReaderInitialize,
    &pduFileReaderInitialize,
    &dirReaderInitialize,
    &respoolReaderInitialize
};

/* The possible reader types */
static reader_type_t reader_types[MAX_READER_TYPE_COUNT];

/* Get a total count of the reader types actually implemented. */
static size_t num_reader_types = (sizeof(reader_type_inits)
                                  / sizeof(fr_initialize_fn_t));

/* The flow processors; one per probe */
static flow_proc_t *flow_processors = NULL;

/* The number of flow_processors */
static size_t num_flow_processors = 0;

/* The number of flow_processor threads currently running */
static int fproc_thread_count = 0;

/* Mutex controlling access to 'fproc_thread_count' */
static pthread_mutex_t fproc_thread_count_mutex = PTHREAD_MUTEX_INITIALIZER;

/* The packing logic to use to categorize flow records */
static packlogic_plugin_t packlogic;

/* the compression method to use when writing the file.
 * sksiteCompmethodOptionsRegister() will set this to the default or
 * to the value the user specifies. */
static sk_compmethod_t comp_method;

/* Command line option giving the sensor---from the probeconf
 * configuration file---that this rwflowpack is packing.  If none
 * given, use first probe in the probeconf file. */
static const char *sensor_name = NULL;

/* True as long as we are reading. */
static uint8_t reading = 0;

/* non-zero when rwflowpack is shutting down */
static volatile int shuttingDown = 0;

/* non-zero when file locking is disabled */
static int no_file_locking = 0;

/* control thread */
static pthread_t main_thread;

/* Timer that flushes files every so often */
static skTimer_t timing_thread = NULL;

/* Number of seconds between cache flushes */
static uint32_t flush_timeout = FLUSH_TIMEOUT;

/* All open files to which we are writing */
static stream_cache_t *stream_cache;

/* Size of the cache for writing output files.  Can be modified by
 * --file-cache-size switch. */
static uint32_t stream_cache_size = STREAM_CACHE_SIZE;

/* Maximum number of input file handles and the number remaining.
 * They are computed as a fraction of the stream_cache_size.  */
static int input_filehandles_max;
static int input_filehandles_left;

/* Mutex controlling access to the 'input_filehandles' variables */
static pthread_mutex_t input_filehandles_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t  input_filehandles_cond  = PTHREAD_COND_INITIALIZER;

/* Byte order for newly packed files.  When appending to existing
 * files, use the files' existing byte order. */
static silk_endian_t byte_order = SILK_ENDIAN_NATIVE;

/* When not in sending mode, rwflowpack writes the flows to files
 * located under the root directory (sksiteGetRootDir()). */

/* When in sending mode, rwflowpack uses this directory temporarily to
 * create the packed files while it is processing the incoming
 * records.  Files in this directory are incomplete. */
static const char *incremental_directory = NULL;

/* When in sending mode, rwflowpack copies the complete incremental
 * files to this directory and informs the sender about them. */
static const char *sender_directory = NULL;

static reader_options_t reader_opts;

/* options from the user's command line */
static opt_cache_t *opt_cache = NULL;

/* How to run: Input and Output Modes. Must be kept in sync with the
 * available_modes[] array */
typedef enum {
    INPUT_STREAM,
    INPUT_PDUFILE,
    INPUT_FLOWCAP_FILES,
    INPUT_RESPOOL,
    OUTPUT_LOCAL_STORAGE, OUTPUT_SENDING
} io_mode_t;

/* The number of modes */
#define NUM_MODES 6

/* default input and output modes */
static io_mode_t input_mode = INPUT_STREAM;
static io_mode_t output_mode = OUTPUT_LOCAL_STORAGE;

/* The index of the first Output Mode */
static const io_mode_t first_output_mode = OUTPUT_LOCAL_STORAGE;

/* Keep in sync with values in io_mode_t enumeration */
static const struct available_modes_st {
    const io_mode_t iomode;
    const char     *name;
    const char     *title;
    const char     *description;
} available_modes[NUM_MODES] = {
    {INPUT_STREAM, "stream", "Stream Input",
     ("\tRead flow data from the network and/or poll directories for files\n"
      "\tcontaining NetFlow v5 PDUs.  The --polling-interval switch applies\n"
      "\tonly when polling directories.\n")},
    {INPUT_PDUFILE, "pdufile", "PDU-File Input",
     ("\tProcess a single file containing NetFlow v5 PDUs and exit.  The\n"
      "\t--sensor-name switch is required unless the sensor configuration\n"
      "\tfile contains a single sensor.\n")},
    {INPUT_FLOWCAP_FILES, "fcfiles", "Flowcap Files Input",
     ("\tContinually poll a directory for files created by flowcap and\n"
      "\tprocess the data those files contain.\n")},
    {INPUT_RESPOOL, "respool", "Respool SiLK Files Input",
     ("\tContinually poll a directory for SiLK Flow files.  Store the SiLK\n"
      "\tFlow records in each file in the repository, keeping the existing\n"
      "\tsensor ID and flowtype values on each record unchanged.\n")},
    {OUTPUT_LOCAL_STORAGE, "local-storage", "Local-Storage Output",
     ("\tWrite the SiLK Flow records to their final location.\n")},
    {OUTPUT_SENDING, "sending", "Sending Output",
     ("\tWrite the SiLK Flow records to temporary files and have rwsender\n"
      "\tmove the files to another machine for final storage.\n")}
};

/* which options are valid in which modes. */
static mode_option_t mode_options[NUM_MODES][MAX_OPTION_COUNT];

/* Options for byte-order switch */
static struct {
    const char     *name;
    silk_endian_t   value;
} byte_order_opts[] = {
    {"native", SILK_ENDIAN_NATIVE},
    {"little", SILK_ENDIAN_LITTLE},
    {"big",    SILK_ENDIAN_BIG},
    {NULL,     SILK_ENDIAN_ANY} /* sentinel */
};


/* OPTIONS SETUP */

typedef enum {
    OPT_INPUT_MODE, OPT_OUTPUT_MODE,
    OPT_NO_FILE_LOCKING,
    OPT_FLUSH_TIMEOUT,
    OPT_STREAM_CACHE_SIZE,
    OPT_PACK_INTERFACES, OPT_BYTE_ORDER,
    OPT_ERROR_DIRECTORY,
    OPT_ARCHIVE_DIRECTORY, OPT_FLAT_ARCHIVE, OPT_POST_ARCHIVE_COMMAND,
    OPT_SENSOR_CONFIG, OPT_VERIFY_SENSOR_CONFIG,
#ifndef SK_PACKING_LOGIC_PATH
    OPT_PACKING_LOGIC,
#endif
    OPT_SENSOR_NAME,
    OPT_INCOMING_DIRECTORY, OPT_POLLING_INTERVAL,
    OPT_NETFLOW_FILE,
    OPT_ROOT_DIRECTORY,
    OPT_SENDER_DIRECTORY, OPT_INCREMENTAL_DIRECTORY
} appOptionsEnum;

static struct option appOptions[] = {
    {"input-mode",              REQUIRED_ARG, 0, OPT_INPUT_MODE},
    {"output-mode",             REQUIRED_ARG, 0, OPT_OUTPUT_MODE},

    {"no-file-locking",         NO_ARG,       0, OPT_NO_FILE_LOCKING},
    {"flush-timeout",           REQUIRED_ARG, 0, OPT_FLUSH_TIMEOUT},
    {"file-cache-size",         REQUIRED_ARG, 0, OPT_STREAM_CACHE_SIZE},
    {"pack-interfaces",         NO_ARG,       0, OPT_PACK_INTERFACES},
    {"byte-order",              REQUIRED_ARG, 0, OPT_BYTE_ORDER},

    {"error-directory",         REQUIRED_ARG, 0, OPT_ERROR_DIRECTORY},
    {"archive-directory",       REQUIRED_ARG, 0, OPT_ARCHIVE_DIRECTORY},
    {"flat-archive",            NO_ARG,       0, OPT_FLAT_ARCHIVE},
    {"post-archive-command",    REQUIRED_ARG, 0, OPT_POST_ARCHIVE_COMMAND},

    {"sensor-configuration",    REQUIRED_ARG, 0, OPT_SENSOR_CONFIG},
    {"verify-sensor-config",    OPTIONAL_ARG, 0, OPT_VERIFY_SENSOR_CONFIG},
#ifndef SK_PACKING_LOGIC_PATH
    {"packing-logic",           REQUIRED_ARG, 0, OPT_PACKING_LOGIC},
#endif

    {"sensor-name",             REQUIRED_ARG, 0, OPT_SENSOR_NAME},

    {"incoming-directory",      REQUIRED_ARG, 0, OPT_INCOMING_DIRECTORY},
    {"polling-interval",        REQUIRED_ARG, 0, OPT_POLLING_INTERVAL},

    {"netflow-file",            REQUIRED_ARG, 0, OPT_NETFLOW_FILE},

    {"root-directory",          REQUIRED_ARG, 0, OPT_ROOT_DIRECTORY},

    {"sender-directory",        REQUIRED_ARG, 0, OPT_SENDER_DIRECTORY},
    {"incremental-directory",   REQUIRED_ARG, 0, OPT_INCREMENTAL_DIRECTORY},

    {0,0,0,0}                   /* sentinel entry */
};

static const char *appHelp[] = {
    ("Select the source of flow records"),
    ("Select the destination for SiLK flow records"),
    ("Do not attempt to lock the files prior to writing\n"
     "\trecords to them. Def. Use locking"),
    ("Time (in seconds) between periodic flushes of open\n"
     "\tSiLK Flow files to disk"),
    ("Maximum number of SiLK Flow files to have open for\n"
     "\twriting simultaneously"),
    ("Include SNMP interface indexes in packed records\n"
     "\t(useful for debugging the router configuration). Def. No"),
    ("Byte order to use for newly packed files:\n"
     "\tChoices: 'native', 'little', or 'big'. Def. native"),

    ("Move input files that are NOT successfully processed\n"
     "\tinto this directory.  If not specified, rwflowpack exits when it\n"
     "\tencounters one of these problem files. Def. None"),
    ("Move input files into this directory after\n"
     "\tprocessing successfully.  When not specified, input files are\n"
     "\tdeleted unless the input mode is 'pdufile'.  In 'stream' input mode,\n"
     "\tthis switch affects 'poll-directory' probes.  Def. None"),
    ("Store files in the root of the archive-directory.\n"
     "\tWhen not given, incremental files are stored in subdirectories of\n"
     "\tthe archive-directory. Def. Use subdirectories"),
    ("Run this command on each input file once it has\n"
     "\tbeen successfully processed and moved to the archive-directory.\n"
     "\tDef. None.  Each \"%s\" in the command will be replaced by the full\n"
     "\tpath to the archived file.  Requires use of --archive-directory."),

    ("Read sensor configuration from named file"),
    ("Verify that the sensor configuration file is\n"
     "\tcorrect and immediately exit.  If argument provided, print the names\n"
     "\tof the probes and sensors defined in the file. Def. no"),
#ifndef SK_PACKING_LOGIC_PATH
    ("Specify path to the plug-in that provides functions\n"
     "\tto determine into which class and type each flow record will be\n"
     "\tcategorized and the format of the output files"),
#endif

    ("Ignore all sensors in the sensor-configuration file\n"
     "\texcept this sensor"),

    ("Directory to monitor for input files to process"),
    ("Interval (in seconds) between checks of\n"
     "\tdirectories for new input files to process"),

    ("Read NetFlow v5 flow records from the named file,\n"
     "\tpack the flows, and exit rwflowpack"),
    ("Store the packed files locally under the directory\n"
     "\ttree tree rooted at this location"),

    ("Move the packed files into this directory for\n"
     "\tprocessing by rwsender"),
    ("Temporary directory to use while generating\n"
     "\tpacked SiLK data files before moving the completed files to the\n"
     "\tsender-directory"),

    (char *)NULL  /* sentinel entry */
};

/* First option that is mode-specific, i.e., not common to all modes */
static const appOptionsEnum first_mode_option = OPT_SENSOR_CONFIG;


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static int  appOptionsProcessOpt(int opt_index, char *opt_arg);
static int  byteOrderParse(const char *endian_string);
static int  validateOptions(opt_cache_t *ocache, size_t arg_count);
static int  flowpackSetMaximumFileHandles(int new_max_fh);
static int  startAllProcessors(void);
static void stopAllProcessors(void);
static void printReaderStats(void);
static int  writeRecord(const rwRec *rwrec, const skpc_probe_t *probe);
static int  getProbes(sk_vector_t *probe_vec);
static int  createFlowProcessorsFlowcap(void);
static int  createFlowProcessorsRespool(void);
static int  createFlowProcessorsPduFile(void);
static int  createFlowProcessorsStream(void);
static void nullSigHandler(int sig);
static void sendFiles(void);
static int  defineRunModeOptions(void);
static int  verifySensorConfig(const char *sensor_conf, int verbose);
static int  initPackingLogic(const char *user_path);
static fileFormat_t defaultDetermineFileFormat(
    const skpc_probe_t *probe,
    flowtypeID_t        ftype);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                       \
    ("<SWITCHES>\n"                                                     \
     "\tRead flow records generated by NetFlow(v5), IPFIX, or flowcap\n"  \
     "\tfrom a socket or from a file and pack the flow records into\n"  \
     "\thourly flat-files organized in a time-based directory structure.\n")

    FILE *fh = USAGE_FH;
    unsigned int i, j;

    fprintf(fh, "%s %s", skAppName(), USAGE_MSG);

    fprintf(fh, "\nGeneral switches:\n");
    skOptionsDefaultUsage(fh);

    /* print the "common" (non-mode-specific) options */
    for (i = 0; i <= OPT_BYTE_ORDER; ++i) {
        fprintf(fh, "--%s %s. ", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]));
        switch (appOptions[i].val) {
          case OPT_FLUSH_TIMEOUT:
            fprintf(fh, "%s. Def. %d", appHelp[i], FLUSH_TIMEOUT);
            break;

          case OPT_STREAM_CACHE_SIZE:
            fprintf(fh, "%s. Range %d-%d. Def. %d",
                    appHelp[i], STREAM_CACHE_MIN,
                    UINT16_MAX, STREAM_CACHE_SIZE);
            break;

          case OPT_INPUT_MODE:
            fprintf(fh, "%s\n\tChoices: %s",
                    appHelp[i], available_modes[0].name);
            for (j = 1; j < first_output_mode; ++j) {
                fprintf(fh, ", %s", available_modes[j].name);
            }
            for (j = 0; j < first_output_mode; ++j) {
                if (j == input_mode) {
                    fprintf(fh, ". Def. %s", available_modes[j].name);
                    break;
                }
            }
            break;

          case OPT_OUTPUT_MODE:
            fprintf(fh, "%s\n\tChoices: %s",
                    appHelp[i], available_modes[first_output_mode].name);
            for (j = 1+first_output_mode; j < NUM_MODES; ++j) {
                fprintf(fh, ", %s", available_modes[j].name);
            }
            for (j = first_output_mode; j < NUM_MODES; ++j) {
                if (j == output_mode) {
                    fprintf(fh, ". Def. %s", available_modes[j].name);
                    break;
                }
            }
            break;

          default:
            fprintf(fh, "%s", appHelp[i]);
            break;
        }
        fprintf(fh, "\n");
    }

    sksiteCompmethodOptionsUsage(fh);
    sksiteOptionsUsage(fh);

    fprintf(fh, "\nSwitches for disposal of input flow files:\n");
    for ( ; i < first_mode_option; ++i) {
        fprintf(fh, "--%s %s. %s\n", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]), appHelp[i]);
    }

    fprintf(fh, "\nLogging and daemon switches:\n");
    skdaemonOptionsUsage(fh);

    /* print options that are used by each input/output mode */
    for (j = 0; j < NUM_MODES; ++j) {
        assert(j == available_modes[j].iomode);
        fprintf(fh, "\n%s Mode (--%s=%s)",
                available_modes[j].title,
                ((j < first_output_mode)
                 ? appOptions[OPT_INPUT_MODE].name
                 : appOptions[OPT_OUTPUT_MODE].name),
                available_modes[j].name);
        if ((j == input_mode) || (j == output_mode)) {
            fprintf(fh, " [default]");
        }
        fprintf(fh, "\n%s", available_modes[j].description);

        for (i = first_mode_option; appOptions[i].name; ++i) {
            switch (mode_options[j][appOptions[i].val]) {
              case MODOPT_REQUIRED:
              case MODOPT_OPTIONAL:
                fprintf(fh, "--%s %s. ", appOptions[i].name,
                        SK_OPTION_HAS_ARG(appOptions[i]));

                switch (appOptions[i].val) {
                  case OPT_POLLING_INTERVAL:
                    fprintf(fh, "%s. Def. %d", appHelp[i], POLLING_INTERVAL);
                    break;

                  default:
                    fprintf(fh, "%s", appHelp[i]);
                    break;
                }
                fprintf(fh, "\n");
                break;

              case MODOPT_ILLEGAL:
              case MODOPT_NONSENSE:
                break;
            }
        }
    }
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;
    size_t i;
    reader_type_t *rt;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    if (input_mode == INPUT_PDUFILE) {
       INFOMSG("Finishing rwflowpack...");
    } else {
       INFOMSG("Begin shutting down...");
    }
    shuttingDown = 1;

    /* broadcast so any poll-dir probes that are waiting for a file
     * handle will wake up and begin shutting down. */
    pthread_cond_broadcast(&input_filehandles_cond);

    printReaderStats();
    stopAllProcessors();

    if (stream_cache) {
        /* Destroy the cache, flushing, closing and freeing all the
         * open streams.  We're in shutdown, so ignore the return
         * code. */
        INFOMSG("Closing all files.");
        (void)skCacheDestroy(stream_cache);
        stream_cache = NULL;
    }

    DEBUGMSG("Destroying the readers.");
    for (i = 0; i < num_reader_types; ++i) {
        rt = &reader_types[i];
        if (rt->teardown_fn != NULL) {
            rt->teardown_fn();
            rt->teardown_fn = NULL;
        }

        if (rt->probes != NULL) {
            skVectorDestroy(rt->probes);
            rt->probes = NULL;
        }
    }

    if (flow_processors) {
        free(flow_processors);
        flow_processors = NULL;
    }
    if (opt_cache) {
        free(opt_cache);
        opt_cache = NULL;
    }

    /* clean up any site-specific memory */
    DEBUGMSG("Unloading the packing logic");
    if (packlogic.teardown_fn != NULL) {
        packlogic.teardown_fn();
    }
    if (packlogic.handle) {
        dlclose(packlogic.handle);
    }
    if (packlogic.path) {
        free(packlogic.path);
    }

    /* teardown the probe configuration */
    skpcTeardown();

    if (input_mode == INPUT_PDUFILE) {
       INFOMSG("Finished processing PDU file.");
    } else {
       INFOMSG("Finished shutting down.");
    }
    skdaemonTeardown();
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    int arg_index;
    size_t i;
    int max_fh;
    reader_type_t *rt;
    size_t arg_count = sizeof(appOptions)/sizeof(struct option);
    struct sigaction action;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) == arg_count);
    assert(arg_count < MAX_OPTION_COUNT);

    /* verify reader_types array is big enough */
    assert(num_reader_types <= MAX_READER_TYPE_COUNT);
    assert(_READER_TYPE_MAX_ <= MAX_READER_TYPE_COUNT);

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    memset(reader_types, 0, sizeof(reader_types));
    memset(&reader_opts, 0, sizeof(reader_opts));
    memset(&packlogic,   0, sizeof(packlogic));


    /* create an array to hold the user's command line options */
    opt_cache = calloc(arg_count, sizeof(opt_cache_t));
    if (NULL == opt_cache) {
        skAppPrintErr(FMT_MEM_FAILURE);
        exit(EXIT_FAILURE);
    }

    /* Set which switches are valid for which modes */
    if (defineRunModeOptions()) {
        skAppPrintErr("Unable to initialize modes");
        exit(EXIT_FAILURE);
    }

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler,(clientData)opt_cache)
        || sksiteCompmethodOptionsRegister(&comp_method)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* rwflowpack runs as a daemon; use the threaded logger */
    if (skdaemonSetup((SKLOG_FEATURE_LEGACY | SKLOG_FEATURE_SYSLOG),
                      argc, argv)
        || sklogEnableThreadedLogging())
    {
        exit(EXIT_FAILURE);
    }

    /* Allow each reader to do its initial setup and register its options */
    for (i = 0; i < num_reader_types; ++i) {
        rt = &reader_types[i];
        if (0 != ((*reader_type_inits[i])(rt))) {
            /* initialize failed.  print error and exit. */
            if (rt->reader_name) {
                skAppPrintErr("Unable to setup the %s flow reader",
                              reader_types[i].reader_name);
            } else {
                skAppPrintErr("Unable to setup the flow reader number %u",
                              (unsigned int)i);
            }
            exit(EXIT_FAILURE);
        }
    }

    /* parse the options.  This first pass does very little parsing; it
     * primarily fills the 'opt_cache[]' array with the arguments.  The
     * appOptionsProcessOpt() fucntion called by validateOptions() is
     * where most of the parsing actually occurs. */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options handler has printed error */
        skAppUsage();
    }

    /* check for additional arguments */
    if (argc != arg_index) {
        skAppPrintErr("Too many or unrecognized argument specified '%s'",
                      argv[arg_index]);
        skAppUsage();
    }

    /* validate the options */
    if (validateOptions(opt_cache, arg_count)) {
        skAppUsage();  /* never returns */
    }

    /* set input file handles based on stream_cache_size */
    max_fh = (int)((double)stream_cache_size * INPUT_FILEHANDLES_FRACTION);
    if (max_fh < INPUT_FILEHANDLES_MIN) {
        max_fh = INPUT_FILEHANDLES_MIN;
    }
    if (flowpackSetMaximumFileHandles(max_fh)) {
        skAppPrintErr("Cannot set maximum input files to %d", max_fh);
        exit(EXIT_FAILURE);
    }

    /* set polldir file handles based on stream_cache_size */
    max_fh = (int)((double)stream_cache_size * POLLDIR_FILEHANDLES_FRACTION);
    if (max_fh < POLLDIR_FILEHANDLES_MIN) {
        max_fh = POLLDIR_FILEHANDLES_MIN;
    }
    if (skPollDirSetMaximumFileHandles(max_fh)) {
        skAppPrintErr("Cannot set maximum polldirs to %d", max_fh);
        exit(EXIT_FAILURE);
    }

    /* set the mask so that the mode is 0644 */
    (void)umask((mode_t)0022);

    /* Have the main thread handle the signal sent by readers to
     * indicate the reader (i.e., the manageProcessor() function) is
     * exiting.  Normally this is SIGUSR2.  */
    memset(&action, 0, sizeof(action));
    /* mask any further signals while we're inside the handler */
    sigfillset(&action.sa_mask);
    action.sa_handler = &nullSigHandler;
    if (sigaction(READER_DONE_SIGNAL, &action, NULL) == -1) {
        skAppPrintErr("Could not handle SIG%s: %s",
                      skSignalToName(READER_DONE_SIGNAL), strerror(errno));
        exit(EXIT_FAILURE);
    }

    /* who am I? */
    main_thread = pthread_self();
    skthread_init("main");

    return;                       /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  cData,
    int         opt_index,
    char       *opt_arg)
{
    static int arg_count = 0;
    opt_cache_t *ocache = (opt_cache_t*)cData;
    unsigned int i;
    int found_mode;

    switch ((appOptionsEnum)opt_index) {
      case OPT_INPUT_MODE:
        found_mode = 0;
        for (i = 0; i < first_output_mode; ++i) {
            if (0 == strcmp(opt_arg, available_modes[i].name)) {
                found_mode = 1;
                input_mode = (io_mode_t)i;
                break;
            }
        }
        if (!found_mode) {
            skAppPrintErr("Invalid %s '%s'",
                          appOptions[opt_index].name, opt_arg);
            return 1;
        }
        break;

      case OPT_OUTPUT_MODE:
        found_mode = 0;
        for (i = first_output_mode; i < NUM_MODES; ++i) {
            if (0 == strcmp(opt_arg, available_modes[i].name)) {
                found_mode = 1;
                output_mode = (io_mode_t)i;
                break;
            }
        }
        if (!found_mode) {
            skAppPrintErr("Invalid %s '%s'",
                          appOptions[opt_index].name, opt_arg);
            return 1;
        }
        break;

#ifndef SK_PACKING_LOGIC_PATH
      case OPT_PACKING_LOGIC:
        if (initPackingLogic(opt_arg)) {
            skAppPrintErr("Unable to load %s '%s'",
                          appOptions[opt_index].name, opt_arg);
            return 1;
        }
        break;
#endif  /* SK_PACKING_LOGIC_PATH */

      default:
        if (ocache[opt_index].seen) {
            skAppPrintErr("Switch %s already seen",
                          appOptions[opt_index].name);
            return 1;
        }

        ++arg_count;
        ocache[opt_index].seen = arg_count;
        ocache[opt_index].value = opt_arg;
        break;
    }

    return 0;
}


static int appOptionsProcessOpt(
    int         opt_index,
    char       *opt_arg)
{
    uint32_t opt_val;
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_ROOT_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        sksiteSetRootDir(opt_arg);
        break;

      case OPT_ERROR_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        errorDirectorySetPath(opt_arg);
        break;

      case OPT_ARCHIVE_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        archiveDirectorySetPath(opt_arg);
        break;

      case OPT_POST_ARCHIVE_COMMAND:
        archiveDirectorySetPostCommand(opt_arg);
        break;

      case OPT_FLAT_ARCHIVE:
        archiveDirectorySetFlat();
        break;

      case OPT_SENDER_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        sender_directory = opt_arg;
        break;

      case OPT_INCREMENTAL_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        incremental_directory = opt_arg;
        break;

      case OPT_BYTE_ORDER:
        return byteOrderParse(opt_arg);

      case OPT_PACK_INTERFACES:
        /* use function that includes SNMP information */
        packlogic.determine_fileformat_fn = &defaultDetermineFileFormat;
        break;

      case OPT_NO_FILE_LOCKING:
        no_file_locking = 1;
        break;

      case OPT_SENSOR_CONFIG:
        if (verifySensorConfig(opt_arg, 0)) {
            exit(EXIT_FAILURE);
        }
        break;

      case OPT_SENSOR_NAME:
        sensor_name = opt_arg;
        break;

      case OPT_INCOMING_DIRECTORY:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        switch (input_mode) {
          case INPUT_FLOWCAP_FILES:
            reader_opts.fcfiles.incoming_directory = opt_arg;
            break;
          case INPUT_RESPOOL:
            reader_opts.respool.incoming_directory = opt_arg;
            break;
          default:
            skAbortBadCase(input_mode);
        }
        break;

      case OPT_POLLING_INTERVAL:
        rv = skStringParseUint32(&opt_val, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        switch (input_mode) {
          case INPUT_FLOWCAP_FILES:
            reader_opts.fcfiles.polling_interval = opt_val;
            break;
          case INPUT_STREAM:
            reader_opts.stream_polldir.polling_interval = opt_val;
            break;
          case INPUT_RESPOOL:
            reader_opts.respool.polling_interval = opt_val;
            break;
          default:
            skAbortBadCase(input_mode);
        }
        break;

      case OPT_FLUSH_TIMEOUT:
        rv = skStringParseUint32(&opt_val, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        flush_timeout = opt_val;
        break;

      case OPT_STREAM_CACHE_SIZE:
        rv = skStringParseUint32(&opt_val, opt_arg,
                                 STREAM_CACHE_MIN, INT16_MAX);
        if (rv) {
            goto PARSE_ERROR;
        }
        stream_cache_size = (int)opt_val;
        break;

      case OPT_NETFLOW_FILE:
        if (opt_arg[0] == '\0') {
            skAppPrintErr("Empty %s supplied", appOptions[opt_index].name);
            return 1;
        }
        reader_opts.pdu_file.netflow_file = opt_arg;
        break;

      case OPT_INPUT_MODE:
      case OPT_OUTPUT_MODE:
      case OPT_VERIFY_SENSOR_CONFIG:
#ifndef SK_PACKING_LOGIC_PATH
      case OPT_PACKING_LOGIC:
#endif
        /* ain't supposed to happen */
        skAbortBadCase(opt_index);
    }

    return 0;                     /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/*
 *  ok = byteOrderParse(argument)
 *
 *    parse the argument to the --byte-order switch
 */
static int byteOrderParse(const char *endian_string)
{
    static int option_seen = 0;
    int i;
    size_t len;

    /* only process option one time */
    if (option_seen != 0) {
        skAppPrintErr("Option %s given multiple times",
                      appOptions[OPT_BYTE_ORDER].name);
        return 1;
    }
    option_seen = 1;

    len = strlen(endian_string);
    if (len == 0) {
        skAppPrintErr("Empty string given for %s option",
                      appOptions[OPT_BYTE_ORDER].name);
        return 1;
    }

    /* initialize byte order */
    byte_order = SILK_ENDIAN_ANY;

    /* parse user's input */
    for (i = 0; byte_order_opts[i].name != NULL; ++i) {
        if ((len <= strlen(byte_order_opts[i].name))
            && (0 == strncmp(byte_order_opts[i].name, endian_string, len)))
        {
            if (byte_order != SILK_ENDIAN_ANY) {
                skAppPrintErr("Ambiguous %s value '%s'",
                              byte_order_opts[i].name, endian_string);
                return 1;
            }
            byte_order = byte_order_opts[i].value;
        }
    }

    if (byte_order == SILK_ENDIAN_ANY) {
        skAppPrintErr("Cannot parse %s value '%s'",
                      appOptions[OPT_BYTE_ORDER].name, endian_string);
        return 1;
    }

    return 0;
}


/*
 *  ok = validateOptions(argc, argv);
 *
 *    Call the options parser.  If options parsing succeeds, validate
 *    that all the required arguments are present, and that the user
 *    didn't give inconsistent arguments.  Returns 0 on success, or -1
 *    otherwise.
 */
static int validateOptions(opt_cache_t *ocache, size_t arg_count)
{
    fp_daemon_mode_t old_deamon_mode = FP_DAEMON_OFF;
    fp_daemon_mode_t deamon_mode = FP_DAEMON_OFF;
    int daemon_seen;
    reader_type_t *rt;
    int options_error = 0;
    size_t i;

    /* Process the root directory first */
    if (ocache[OPT_ROOT_DIRECTORY].seen) {
        if (appOptionsProcessOpt(OPT_ROOT_DIRECTORY,
                                 ocache[OPT_ROOT_DIRECTORY].value))
        {
            return -1;
        }
    }

    /* ensure the site config is available; do this after setting the
     * root directory */
    if (sksiteConfigure(1)) {
        exit(EXIT_FAILURE);
    }

    /* setup the probe configuration parser */
    if (skpcSetup()) {
        skAppPrintErr("Unable to setup probe config file parser");
        exit(EXIT_FAILURE);
    }

    /* if the packing logic plug-in was not loaded, check whether the
     * plug-in is given statically and initialize it if it is.
     * Otherwise, check whether the site configuration defines a
     * packing-logic value, and use that.  If that also fails, tell
     * the user the --packing-logic switch is required. */
#ifdef SK_PACKING_LOGIC_PATH
    /* call the initialize function on the linked-in pack-logic */
    if (initPackingLogic(NULL)) {
        skAppPrintErr("Error loading statically linked packer logic");
        exit(EXIT_FAILURE);
    }
#else
    if (packlogic.determine_flowtype_fn == NULL) {
        char pack_path[PATH_MAX];

        if (sksiteGetPackingLogicPath(pack_path, sizeof(pack_path)) == NULL) {
            /* error if no site-specific packing logic */
            skAppPrintErr("The --%s switch is required",
                          appOptions[OPT_PACKING_LOGIC].name);
            options_error = 1;
        } else if (initPackingLogic(pack_path)) {
            skAppPrintErr("Unable to load packing logic from '%s'",
                          pack_path);
            exit(EXIT_FAILURE);
        }
    }
#endif  /* SK_PACKING_LOGIC_PATH */

    /* setup the site packing logic */
    if ((packlogic.setup_fn != NULL) && (packlogic.setup_fn() != 0)) {
        skAppPrintErr("Unable to setup packing logic plugin");
        exit(EXIT_FAILURE);
    }

    /* if we're only checking the syntax of the sensor.conf file, do
     * that and exit. */
    if (ocache[OPT_VERIFY_SENSOR_CONFIG].seen) {
        int verbose = 0;

        /* any value except empty string and "0" turns on verbose */
        if ((ocache[OPT_VERIFY_SENSOR_CONFIG].value)
            && (ocache[OPT_VERIFY_SENSOR_CONFIG].value[0] != '\0')
            && (0 != strcmp(ocache[OPT_VERIFY_SENSOR_CONFIG].value, "0")))
        {
            verbose = 1;
        }

        /* make certain we have the sensor-config */
        if (!ocache[OPT_SENSOR_CONFIG].seen) {
            skAppPrintErr("The --%s switch is required",
                          appOptions[OPT_SENSOR_CONFIG].name);
            exit(EXIT_FAILURE);
        }
        if (verifySensorConfig(ocache[OPT_SENSOR_CONFIG].value, verbose)) {
            exit(EXIT_FAILURE);
        }
        appTeardown();
        exit(EXIT_SUCCESS);
    }

    /*
     * Do the real options parsing.
     *
     * Given the input and output modes, check for or missing
     * inconsistent switches: Make sure the required options were
     * provided, and that no illegal options were given.  If the
     * option is given and we were supposed to see it, call the "real"
     * options handler to process it.
     */
    for (i = 0; i < arg_count; ++i) {
        if (ocache[i].seen == 0) {
            /* option 'i' not given; see if it is required */
            if (mode_options[input_mode][i] == MODOPT_REQUIRED) {
                skAppPrintErr("The --%s switch is required in %s Mode",
                              appOptions[i].name,
                              available_modes[input_mode].title);
                options_error = 1;
            } else if (mode_options[output_mode][i] == MODOPT_REQUIRED) {
                skAppPrintErr("The --%s switch is required in %s Mode",
                              appOptions[i].name,
                              available_modes[output_mode].title);
                options_error = 1;
            }
        } else {
            /* option 'i' was given; see if it is illegal */
            if (mode_options[input_mode][i] == MODOPT_ILLEGAL) {
                skAppPrintErr("The --%s switch is illegal in %s Mode",
                              appOptions[i].name,
                              available_modes[input_mode].title);
                options_error = 1;
            } else if (mode_options[output_mode][i] == MODOPT_ILLEGAL) {
                skAppPrintErr("The --%s switch is illegal in %s Mode",
                              appOptions[i].name,
                              available_modes[output_mode].title);
                options_error = 1;
            } else if (i == OPT_ROOT_DIRECTORY) {
                /* don't process this, we did it above */
            }  else if (appOptionsProcessOpt(i, ocache[i].value)) {
                options_error = 1;
            }
        }
    }

    /* verify the required options for logging */
    if (skdaemonOptionsVerify()) {
        options_error = 1;
    }

    /* --post-archive-command requires --archive-dir */
    if (archiveDirectoryIsSet() == -1) {
        skAppPrintErr("The --%s switch is required when using --%s",
                      appOptions[OPT_ARCHIVE_DIRECTORY].name,
                      appOptions[OPT_POST_ARCHIVE_COMMAND].name);
        options_error = -1;
    }

    /* return if we have options problems */
    if (options_error) {
        return -1;
    }

    /* in pdufile mode, the archiveDirectoryInsertOrRemove() function
     * should not remove the input file when --archive-dir was not
     * given */
    if (INPUT_PDUFILE == input_mode) {
        archiveDirectorySetNoRemove();
    }

    /* Set the polling-interval default value */
    if (ocache[OPT_POLLING_INTERVAL].seen == 0) {
        switch (input_mode) {
          case INPUT_FLOWCAP_FILES:
            reader_opts.fcfiles.polling_interval = POLLING_INTERVAL;
            break;
          case INPUT_STREAM:
            reader_opts.stream_polldir.polling_interval = POLLING_INTERVAL;
            break;
          case INPUT_RESPOOL:
            reader_opts.respool.polling_interval = POLLING_INTERVAL;
            break;
          default:
            break;
        }
    }

    /* make certain we have at least one sensor from the configuration
     * file. */
    if ((mode_options[input_mode][OPT_SENSOR_CONFIG] == MODOPT_REQUIRED)
        && (skpcCountSensors() == 0))
    {
        skAppPrintErr("No sensors were read from the configuration file.");
        return -1;
    }

    /* how we create the flow_processors depends on the input mode. */
    switch (input_mode) {
      case INPUT_FLOWCAP_FILES:
        if (createFlowProcessorsFlowcap()) {
            return -1;
        }
        break;

      case INPUT_PDUFILE:
        if (createFlowProcessorsPduFile()) {
            return -1;
        }
        break;

      case INPUT_STREAM:
        if (createFlowProcessorsStream()) {
            return -1;
        }
        break;

      case INPUT_RESPOOL:
        if (createFlowProcessorsRespool()) {
            return -1;
        }
        break;

      case OUTPUT_LOCAL_STORAGE:
      case OUTPUT_SENDING:
        skAbortBadCase(input_mode);
    }

    /* Call the setup function for each active reader; ignore readers
     * that have no probes. */
    daemon_seen = 0;
    for (i = 0; i < num_reader_types; ++i) {
        rt = &reader_types[i];
        if (NULL == rt->probes) {
            /* Call teardown_fn for this reader now? */
            continue;
        }

        if (rt->setup_fn(&deamon_mode, rt->probes, &reader_opts)) {
            return -1;
        }

        /* All the active reader_types must have same "daemon-ness" */
        if (daemon_seen == 0) {
            daemon_seen = 1;
            old_deamon_mode = deamon_mode;
        } else if (old_deamon_mode != deamon_mode) {
            skAppPrintErr("Cannot mix probes that work as daemons with\n"
                          "\tprobes that do not.");
            return -1;
        }
    }

    assert(daemon_seen);
    if (deamon_mode == FP_DAEMON_OFF) {
        skdaemonDontFork();
    }

    return 0;
}


/*
 *  status = verifySensorConfig(sensor_conf, verbose);
 *
 *    Verify that the 'sensor_conf' file is valid.  If verbose is
 *    non-zero, print the probes and sensors that were found in the
 *    file.
 *
 *    Return 0 if the file is valid, -1 otherwise.
 */
static int verifySensorConfig(
    const char *sensor_conf,
    int         verbose)

{
    skpc_probe_iter_t probe_iter;
    const skpc_probe_t *probe;
    skpc_sensor_iter_t sensor_iter;
    const skpc_sensor_t *sensor;
    uint32_t count;
    int first;

    /* parse it */
    if (skpcParse(sensor_conf, packlogic.verify_sensor_fn)) {
        skAppPrintErr("Errors while parsing %s file '%s'",
                      appOptions[OPT_SENSOR_CONFIG].name, sensor_conf);
        return -1;
    }

    /* make certain we have at least one sensor from the configuration
     * file. */
    if (skpcCountSensors() == 0) {
        skAppPrintErr("No sensor definitions exist in '%s'", sensor_conf);
        return -1;
    }

    /* if a value was provided to the --verify-sensor switch, be verbose */
    if (verbose) {
        /* print the probes */
        count = skpcCountProbes();
        printf("%s: Successfully parsed %" PRIu32 " probe%s:\n",
               skAppName(), count, CHECK_PLURAL(count));
        if (count) {
            skpcProbeIteratorBind(&probe_iter);
            first = 1;
            while (skpcProbeIteratorNext(&probe_iter, &probe)) {
                if (first) {
                    first = 0;
                    printf("\t%s", skpcProbeGetName(probe));
                } else {
                    printf(", %s", skpcProbeGetName(probe));
                }
            }
            printf("\n");
        }

        /* print the sensors */
        count = skpcCountSensors();
        printf("%s: Successfully parsed %" PRIu32 " sensor%s:\n",
               skAppName(), count, CHECK_PLURAL(count));
        skpcSensorIteratorBind(&sensor_iter);
        first = 1;
        while (skpcSensorIteratorNext(&sensor_iter, &sensor)) {
            if (first) {
                first = 0;
                printf("\t%s", skpcSensorGetName(sensor));
            } else {
                printf(", %s", skpcSensorGetName(sensor));
            }
        }
        printf("\n");
    }

    return 0;
}


/*
 *  status = initPackingLogic(path);
 *
 *    If 'path' is not NULL, treat it as the name of a plug-in to
 *    load.  Return an error code if the plug-in cannot be loaded or
 *    does not define the necessary functions.
 *
 *    If 'path' is NULL, verify that the packing-logic has been
 *    statically linked.
 *
 *    Call the initalize function on the packing-logic "plug-in", and
 *    verify that the required fucntion pointers are set.  Return an
 *    error code if they are not.
 *
 *    If all is well, return 0.
 */
static int initPackingLogic(const char *user_path)
{
    int (*init_fn)(packlogic_plugin_t *) = NULL;
    void *dl_handle = NULL;
    const char *debug = NULL;
    char *env_value;

    if (INPUT_RESPOOL == input_mode) {
        /* user is respooling */
        init_fn = &packLogicRespoolInitialize;
        packlogic.path = strdup("respool");
        if (NULL == packlogic.path) {
            skAppPrintErr(FMT_MEM_FAILURE);
            return -1;
        }

#ifndef SK_PACKING_LOGIC_PATH
    } else if (user_path == NULL) {
        /* packing-logic is NOT compiled in, yet the path is NULL. */
        skAppPrintErr("Bad argument to --%s switch or bad parse of silk.conf",
                      appOptions[OPT_PACKING_LOGIC].name);
        return -1;

#else
    } else if (user_path == NULL) {
        /* assumes the packing-logic is compiled in. */
        const char *packlogic_path = SK_PACKING_LOGIC_PATH;
        const char *plp = strrchr(packlogic_path, '/');
        if (plp) {
            ++plp;
        } else {
            plp = packlogic_path;
        }
        packlogic.path = strdup(plp);
        if (NULL == packlogic.path) {
            skAppPrintErr(FMT_MEM_FAILURE);
            goto ERROR;
        }

        init_fn = &packLogicInitialize;
        if (init_fn == NULL) {
            /* linker should not have allowed this this program to compile */
            skAppPrintErr("No static packing logic function");
            skAbort();
        }
#endif  /* SK_PACKING_LOGIC_PATH */

    } else {
        char dl_path[PATH_MAX];

        /* check if debug is enabled */
        env_value = getenv(SKPLUGIN_DEBUG_ENVAR);
        if ((NULL != env_value) && ('\0' != env_value[0])) {
            debug = (SKPLUGIN_DEBUG_ENVAR ": ");
        }

        /* Attempt to find the full path to the plug-in, and set
         * 'dl_path' to its path; if we cannot find the path, copy the
         * plug-in's name to 'dl_path' and we'll let dlopen() handle
         * finding the plug-in. */
        if ( !skFindPluginPath(user_path, dl_path, sizeof(dl_path), debug)) {
            strncpy(dl_path, user_path, sizeof(dl_path));
            dl_path[sizeof(dl_path)-1] = '\0';
        }

        /* try to dlopen() the plug-in  */
        if (debug) {
            skAppPrintErr("%sdlopen'ing '%s'", debug, dl_path);
        }
        dl_handle = dlopen(dl_path, RTLD_NOW | RTLD_GLOBAL);
        if (NULL == dl_handle) {
            if (debug) {
                skAppPrintErr("%sdlopen warning: %s",
                              debug, dlerror());
            }
            goto ERROR;
        }
        if (debug) {
            skAppPrintErr("%sdlopen() successful", debug);
        }

        /* get pointer to the initialization function */
        *(void**)&(init_fn) = dlsym(dl_handle, SK_PACKLOGIC_INIT);
        if (NULL == init_fn) {
            if (debug) {
                skAppPrintErr("%sfunction %s not found",
                              debug, SK_PACKLOGIC_INIT);
            }
            goto ERROR;
        }

        /* stash the pathname */
        packlogic.path = strdup(dl_path);
        if (NULL == packlogic.path) {
            skAppPrintErr(FMT_MEM_FAILURE);
            goto ERROR;
        }
    }

    /* invoke the initialization function */
    if (init_fn(&packlogic)) {
        if (debug) {
            skAppPrintErr("%sfunction %s returned error",
                          debug, SK_PACKLOGIC_INIT);
        }
        goto ERROR;
    }

    /* make certain required functions were provided */
    if (NULL == packlogic.determine_flowtype_fn) {
        skAppPrintErr(("Cannot find function to determine flowtype in the\n"
                       "\npacking logic plugin %s"),
                      (packlogic.path ? packlogic.path : ""));
        goto ERROR;
    }
    if (NULL == packlogic.verify_sensor_fn) {
        skAppPrintErr(("Cannot find function to verify sensor in the\n"
                       "\npacking logic plugin %s"),
                      (packlogic.path ? packlogic.path : ""));
        goto ERROR;
    }

    /* if not provided, use a default function for determining the
     * fileformat */
    if (NULL == packlogic.determine_fileformat_fn) {
        packlogic.determine_fileformat_fn = &defaultDetermineFileFormat;
    }

    /* fill out the remaining variables on the packlogic object */
    packlogic.handle = dl_handle;
    packlogic.initialize_fn = init_fn;

    /* all OK */
    return 0;

  ERROR:
    if (dl_handle) {
        dlclose(dl_handle);
    }
    if (packlogic.path) {
        free(packlogic.path);
    }
    return -1;
}



/*
 *  status = defineRunModeOptions();
 *
 *    Set the values mode_options[][] array.
 */
static int defineRunModeOptions(void)
{
    unsigned int i, j;

    memset(mode_options, MODOPT_ILLEGAL, sizeof(mode_options));

    /* common options; all are optional */
    for (i = 0; i < NUM_MODES; ++i) {
        for (j = 0; j < first_mode_option; ++j) {
            mode_options[i][j] = MODOPT_OPTIONAL;
        }
    }

    /* for the input modes, we don't care whether the output options
     * were specified. */
    for (i = 0; i < first_output_mode; ++i) {
        mode_options[i][OPT_SENDER_DIRECTORY] = MODOPT_NONSENSE;
        mode_options[i][OPT_INCREMENTAL_DIRECTORY] = MODOPT_NONSENSE;
        mode_options[i][OPT_ROOT_DIRECTORY] = MODOPT_NONSENSE;
    }

    /* for the output modes, we don't care about input-only options */
    for (i = first_output_mode; i < NUM_MODES; ++i) {
        mode_options[i][OPT_SENSOR_CONFIG] = MODOPT_NONSENSE;
        mode_options[i][OPT_VERIFY_SENSOR_CONFIG] = MODOPT_NONSENSE;
        mode_options[i][OPT_INCOMING_DIRECTORY] = MODOPT_NONSENSE;
        mode_options[i][OPT_POLLING_INTERVAL] = MODOPT_NONSENSE;
        mode_options[i][OPT_NETFLOW_FILE] = MODOPT_NONSENSE;
        mode_options[i][OPT_SENSOR_NAME] = MODOPT_NONSENSE;
    }

    /* for all input modes except respool, set --sensor-conf as
     * required, and make --verify-sensor and --packing-logic
     * optional  */
    for (i = 0; i < first_output_mode; ++i) {
        if (INPUT_RESPOOL == i) {
            continue;
        }
        mode_options[i][OPT_SENSOR_CONFIG] = MODOPT_REQUIRED;
        mode_options[i][OPT_VERIFY_SENSOR_CONFIG] = MODOPT_OPTIONAL;
#ifndef SK_PACKING_LOGIC_PATH
        mode_options[i][OPT_PACKING_LOGIC] = MODOPT_OPTIONAL;
#endif
    }

    /* handle other mode-specific fields. */
    mode_options[INPUT_FLOWCAP_FILES][OPT_INCOMING_DIRECTORY]= MODOPT_REQUIRED;
    mode_options[INPUT_FLOWCAP_FILES][OPT_POLLING_INTERVAL] = MODOPT_OPTIONAL;

    mode_options[INPUT_PDUFILE][OPT_NETFLOW_FILE] = MODOPT_REQUIRED;
    mode_options[INPUT_PDUFILE][OPT_SENSOR_NAME] = MODOPT_OPTIONAL;

    mode_options[INPUT_STREAM][OPT_SENSOR_NAME] = MODOPT_OPTIONAL;
    mode_options[INPUT_STREAM][OPT_POLLING_INTERVAL] = MODOPT_OPTIONAL;

    mode_options[INPUT_RESPOOL][OPT_INCOMING_DIRECTORY]= MODOPT_REQUIRED;
    mode_options[INPUT_RESPOOL][OPT_POLLING_INTERVAL] = MODOPT_OPTIONAL;

    mode_options[OUTPUT_SENDING][OPT_SENDER_DIRECTORY] = MODOPT_REQUIRED;
    mode_options[OUTPUT_SENDING][OPT_INCREMENTAL_DIRECTORY] = MODOPT_REQUIRED;

    mode_options[OUTPUT_LOCAL_STORAGE][OPT_ROOT_DIRECTORY] = MODOPT_REQUIRED;

    return 0;
}


/*
 *  nullSigHandler(signal);
 *
 *    Do nothing.  Called when the reader thread---that is, the
 *    manageProcessor() function---sends a signal (normally SIGUSR2)
 *    indicating the reader is done.
 */
void nullSigHandler(int UNUSED(s))
{
    return;
}


/* Acquire a file handle.  Return 0 on success, or -1 if we have
 * started shutting down.  */
int flowpackAcquireFileHandle(void)
{
    int rv = -1;

    pthread_mutex_lock(&input_filehandles_mutex);
    while (input_filehandles_left <= 0 && !shuttingDown) {
        pthread_cond_wait(&input_filehandles_cond, &input_filehandles_mutex);
    }
    if (!shuttingDown) {
        --input_filehandles_left;
        rv = 0;
    }
    pthread_mutex_unlock(&input_filehandles_mutex);
    return rv;
}


/* Release the file handle. */
void flowpackReleaseFileHandle(void)
{
    pthread_mutex_lock(&input_filehandles_mutex);
    ++input_filehandles_left;
    pthread_cond_signal(&input_filehandles_cond);
    pthread_mutex_unlock(&input_filehandles_mutex);
}


/* Change the maximum number of input filehandles we can use. */
static int flowpackSetMaximumFileHandles(
    int new_max_fh)
{
    if (new_max_fh < 1) {
        return -1;
    }

    pthread_mutex_lock(&input_filehandles_mutex);
    input_filehandles_left += new_max_fh - input_filehandles_max;
    input_filehandles_max = new_max_fh;
    pthread_mutex_unlock(&input_filehandles_mutex);

    return 0;
}


/*
 *  printReaderStats();
 *
 *    Calls the print_stats_fn for each reader, assuming it supports
 *    that functionality.
 */
static void printReaderStats(void)
{
    unsigned i;

    /* Call stats functions for each flow processor */
    for (i = 0; i < num_flow_processors; i++) {
        flow_proc_t *fproc = &flow_processors[i];
        if (fproc->reader_type->print_stats_fn) {
            fproc->reader_type->print_stats_fn(fproc);
        }
    }
}


/*
 *  flushFiles(NULL);
 *
 *  THREAD ENTRY POINT
 *
 *    Flushes all the files in global stream cache stream_cache.
 *
 *    Called every 'flush_timeout' seconds by the timing_thread.
 */
static skTimerRepeat_t flushFiles(void *UNUSED(dummy))
{
    /* Flush the stream cache */
    NOTICEMSG("Flushing files after %" PRIu32 " seconds.", flush_timeout);
    if (skCacheFlush(stream_cache)) {
        CRITMSG("Error flushing files -- shutting down");
        exit(EXIT_FAILURE);
    }
    printReaderStats();

    return SK_TIMER_REPEAT;
}


/*
 *  timedSendFiles(NULL);
 *
 *  THREAD ENTRY POINT
 *
 *    Closes all the open files in the stream cache and moves all the
 *    files in the incremental-directory to the sender-directory.
 *
 *    Called every 'flush_timeout' seconds by the timing_thread.
 */
static skTimerRepeat_t timedSendFiles(void *UNUSED(dummy))
{
    sendFiles();
    printReaderStats();

    return SK_TIMER_REPEAT;
}


/*
 *  status = createFlowProcessorsFlowcap()
 *
 *    When processing flowcap files (in fcfiles mode), the reader_type
 *    handles all probes, and there is a single flow_processor.
 */
static int createFlowProcessorsFlowcap(void)
{
    reader_type_t *rt;

    /* get the correct reader */
    assert(INPUT_FLOWCAP_FILES == input_mode);
    rt = &reader_types[READER_TYPE_FLOWCAP_FILES];

    /* only one flow_processor is required. */
    num_flow_processors = 1;
    flow_processors = calloc(num_flow_processors, sizeof(flow_proc_t));
    if (!flow_processors) {
        skAppPrintErr(FMT_MEM_FAILURE);
        return -1;
    }
    flow_processors[0].probe = NULL;
    flow_processors[0].reader_type = rt;

    /* need to set the 'probes' field on the reader_type to a non-NULL
     * value, so we know the flowcap reader_type is in use.  Create an
     * empty vector; no need to fill it. */
    rt->probes = skVectorNew(sizeof(skpc_probe_t*));
    if (rt->probes == NULL) {
        skAppPrintErr(FMT_MEM_FAILURE);
        free(flow_processors);
        return -1;
    }

    return 0;
}


/*
 *  status = createFlowProcessorsRespool()
 *
 *    When processing SiLK flow files for respooling, there are no
 *    probes, and there is a single flow_processor.
 */
static int createFlowProcessorsRespool(void)
{
    reader_type_t *rt;

    /* get the correct reader */
    assert(INPUT_RESPOOL == input_mode);
    rt = &reader_types[READER_TYPE_RESPOOL];

    /* only one flow_processor is required. */
    num_flow_processors = 1;
    flow_processors = calloc(num_flow_processors, sizeof(flow_proc_t));
    if (!flow_processors) {
        skAppPrintErr(FMT_MEM_FAILURE);
        return -1;
    }
    flow_processors[0].probe = NULL;
    flow_processors[0].reader_type = rt;

    /* need to set the 'probes' field on the reader_type to a non-NULL
     * value, so we know the respool reader_type is in use.  Create an
     * empty vector; no need to fill it. */
    rt->probes = skVectorNew(sizeof(skpc_probe_t*));
    if (rt->probes == NULL) {
        skAppPrintErr(FMT_MEM_FAILURE);
        free(flow_processors);
        return -1;
    }

    return 0;
}


/*
 *  status = createFlowProcessorsPduFile()
 *
 *    Search through the sensors to find the probe that will handle
 *    the PDU files.  It is an error if there is more than one.
 *    Create the single flow processor that will handle this probe.
 *
 *    Return 0 if everything is correct, or non-zero otherwise.
 */
static int createFlowProcessorsPduFile(void)
{
    sk_vector_t *probe_vec;
    skpc_probe_t *have_probe = NULL;
    skpc_probe_t **p;
    reader_type_t *rt;
    size_t j;
    int rv = -1;

    assert(INPUT_PDUFILE == input_mode);

    /* get the reader, we know which reader we are using */
    rt = &reader_types[READER_TYPE_PDU_FILE];

    /* create vector to hold probes */
    probe_vec = skVectorNew(sizeof(skpc_probe_t*));
    if (NULL == probe_vec) {
        skAppPrintErr(FMT_MEM_FAILURE);
        goto END;
    }

    /* get the probes; either all probes or those listed in the global
     * 'sensor_name' */
    if (getProbes(probe_vec)) {
        goto END;
    }
    if (0 == skVectorGetCount(probe_vec)) {
        skAbort();
    }

    /*
     *  There should only be a single probe that we want to process
     *  for this invocation.  In reality, we should support an
     *  unlimited number of probes as long as they all map to the same
     *  sensor.  Also, forcing the user to specify a probe when we
     *  know we are reading from the file system is kludgy; we should
     *  just be able to cons-up the proper probe on the fly....
     */

    for (j = 0; NULL != (p = skVectorGetValuePointer(probe_vec, j)); ++j) {
        if (rt->want_probe_fn(*p)) {
            if (NULL != have_probe) {
                /* ERROR: more than one probe */
                skAppPrintErr(("Multiple %s probes specified.  The %s input\n"
                               "\tmode requires a single probe"
                               " that reads from a file."),
                              skpcProbetypeEnumtoName(PROBE_ENUM_NETFLOW_V5),
                              available_modes[INPUT_PDUFILE].name);
                goto END;
            }
            have_probe = *p;
        }
    }

    if (NULL == have_probe) {
        /* ERROR: no valid probe */
        skAppPrintErr("Could not find any probes to use for %s input mode",
                      available_modes[INPUT_PDUFILE].name);
        goto END;
    }

    /* looks good.  create the flow processor */
    num_flow_processors = 1;
    flow_processors = calloc(num_flow_processors, sizeof(flow_proc_t));
    if (!flow_processors) {
        skAppPrintErr(FMT_MEM_FAILURE);
        return -1;
    }
    flow_processors[0].probe = have_probe;
    flow_processors[0].reader_type = rt;

    /* re-use the existing probe vector and attach it to the
     * reader-type */
    if (skVectorGetCount(probe_vec) != 1) {
        /* need to clean it out */
        skVectorSetCapacity(probe_vec, 0);
        if (skVectorAppendValue(probe_vec, &have_probe)) {
            skAppPrintErr(FMT_MEM_FAILURE);
            goto END;
        }
    }
    rt->probes = probe_vec;

    /* success */
    rv = 0;

  END:
    if (rv != 0) {
        /* failure.  clean up the vectors and the flow_processors[] */
        if (flow_processors) {
            free(flow_processors);
            flow_processors = NULL;
        }
        if (probe_vec) {
            skVectorDestroy(probe_vec);
        }
    }
    return rv;
}


/*
 *  status = createFlowProcessorsStream()
 *
 *    Search through the sensors to find probes that the stream-based
 *    reader-types (PDU, IPFIX, etc) can use.  Create a flow processor
 *    for each probe, and add the probe to the reader-type's
 *    probe-list.
 *
 *    Return 0 if everything is correct, or non-zero otherwise.
 */
static int createFlowProcessorsStream(void)
{
    sk_vector_t *probe_vec;
    size_t count;
    size_t i, j;
    reader_type_t *rt;
    reader_type_t *probe_rt;
    skpc_probe_t **p;
    int have_poll_dir = 0;
    int rv = -1;
#if SK_ENABLE_IPFIX
    int initialized_ipfix = 0;
#endif

    /* create vector to hold the probes */
    probe_vec = skVectorNew(sizeof(skpc_probe_t*));
    if (NULL == probe_vec) {
        skAppPrintErr(FMT_MEM_FAILURE);
        goto END;
    }

    /* get the probes; either all probes or those specified in the
     * global 'sensor_name' */
    if (getProbes(probe_vec)) {
        goto END;
    }
    count = skVectorGetCount(probe_vec);
    if (0 == count) {
        skAbort();
    }

    /* as an upper limit, each probe will be used and will have its
     * own flow_processor. */
    num_flow_processors = 0;
    flow_processors = calloc(count, sizeof(flow_proc_t));
    if (!flow_processors) {
        skAppPrintErr(FMT_MEM_FAILURE);
        goto END;
    }

    /* attempt to assign each probe to a reader_type */
    for (j = 0; NULL != (p = skVectorGetValuePointer(probe_vec, j)); ++j) {
        /* find the reader that will process the probe.  there can be
         * no more than one. */
        probe_rt = NULL;
        for (i = 0; i < num_reader_types; ++i) {
            /* ignore non-stream reader-types */
            switch ((reader_type_id_t)i) {
              case READER_TYPE_FLOWCAP_FILES:
              case READER_TYPE_PDU_FILE:
                continue;
              default:
                break;
            }

            rt = &reader_types[i];
            if (rt->want_probe_fn(*p)) {
                /* reader 'rt' can process probe 'p'.  If 'probe_rt'
                 * is not-NULL, multiple readers are attempting to
                 * claim the probe.  This shouldn't happen, and
                 * probably indicates a programming error. */
                if (NULL != probe_rt) {
                    skAppPrintErr("Multiple readers can process probe %s",
                                  skpcProbeGetName(*p));
                    goto END;
                }
                probe_rt = rt;
            }
        }
        if (probe_rt == NULL) {
            /* no reader wants to process this probe */
            skAppPrintErr("Warning: Ignoring probe '%s' in stream mode",
                          skpcProbeGetName(*p));
            continue;
        }

        assert(num_flow_processors < count);
        flow_processors[num_flow_processors].probe = *p;
        flow_processors[num_flow_processors].reader_type = probe_rt;
        num_flow_processors++;

        /* if we haven't seen any probes with a poll-directory yet,
         * check if this probe has one. */
        if (!have_poll_dir) {
            if (skpcProbeGetPollDirectory(*p)) {
                have_poll_dir = 1;
            }
        }

        /* add the probe to the 'probes' vector on the reader_type,
         * creating the vector if it does not exist. */
        if (probe_rt->probes == NULL) {
            probe_rt->probes = skVectorNew(sizeof(skpc_probe_t*));
            if (probe_rt->probes == NULL) {
                skAppPrintErr(FMT_MEM_FAILURE);
                goto END;
            }
        }
        if (skVectorAppendValue(probe_rt->probes, p)) {
            skAppPrintErr(FMT_MEM_FAILURE);
            goto END;
        }

#if SK_ENABLE_IPFIX
        /* if we haven't initialized the IPFIX source, do that now */
        if (0 == initialized_ipfix
            && (PROBE_ENUM_IPFIX == skpcProbeGetType(*p)
                || PROBE_ENUM_NETFLOW_V9 == skpcProbeGetType(*p)))
        {
            if (ipfixSourcesSetup()) {
                skAppPrintErr(("Cannot use %s probes: "
                               "GLib2 does not support multiple threads"),
                              skpcProbetypeEnumtoName(skpcProbeGetType(*p)));
                goto END;
            }
        }
#endif /* SK_ENABLE_IPFIX */
    }

    /* sanity check that we processed every probe. */
    if (j != count) {
        skAppPrintErr("Error getting probe %u from vector",
                      (unsigned int)j);
        goto END;
    }

    /* if no probes are using a poll-directory, warn if --archive-dir,
     * --post-archive-command, --error-dir, or --polling-interval were
     * given. */
    if (!have_poll_dir) {
#define NUM_IGNORED_OPTS 4
        int ignored_opts[NUM_IGNORED_OPTS]
            = {OPT_ARCHIVE_DIRECTORY, OPT_POST_ARCHIVE_COMMAND,
               OPT_ERROR_DIRECTORY, OPT_POLLING_INTERVAL};
        for (i = 0; i < NUM_IGNORED_OPTS; ++i) {
            if (opt_cache[ignored_opts[i]].seen) {
                skAppPrintErr(("Ignoring --%s since no probes"
                               " use directory polling"),
                              appOptions[ignored_opts[i]].name);
            }
        }
    }

    /* we could realloc flow_processors[] here */

    /* success */
    rv = 0;

  END:
    if (rv != 0) {
        /* failure.  clean up the vectors and the flow_processors[] */
        for (i = 0; i < num_reader_types; ++i) {
            if (reader_types[i].probes != NULL) {
                skVectorDestroy(reader_types[i].probes);
                reader_types[i].probes = NULL;
            }
        }
        if (flow_processors) {
            free(flow_processors);
            flow_processors = NULL;
        }
    }
    if (probe_vec) {
        skVectorDestroy(probe_vec);
    }
    return rv;
}


/*
 *  status = getProbes(out_probe_vector);
 *
 *    If the global 'sensor_name' is NULL, then append each verified
 *    probe specified in the sensor.conf file to the
 *    'out_probe_vector' if the probe is associated with a sensor.
 *
 *    If 'sensor_name' is non-NULL, treat it as a C string containing
 *    a comma separated list of Sensor Names.  Append all the verified
 *    probes defined in the sensor-configuration file that appear in
 *    the sensor-list to the 'out_probe_vector'.
 *
 *    In either case, return 0 if probes were found and they were
 *    added to the vector.  Return -1 if no verified probes were
 *    found, if probes do not exist for a specified sensor, if there
 *    is an error parsing the 'sensor_list', or for a memory
 *    allocation error.
 */
static int getProbes(sk_vector_t *probe_vec)
{
    sk_vector_t *tmp_vec = NULL;
    skpc_probe_iter_t p_iter;
    const skpc_probe_t *p;
    const skpc_probe_t **p_ptr;
    const skpc_probe_t **p_ptr2;
    skpc_sensor_iter_t s_iter;
    const skpc_sensor_t *sensor;
    char *sensor_name_copy = NULL;
    char *sensor_name_tokens;
    char *sensor_token;
    int found_sensor;
    size_t i;
    size_t j;
    int seen;
    int rv = -1;

    /* If the global 'sensor-name' is NULL, return all verified probes
     * that are associated with a sensor. */
    if (NULL == sensor_name) {
        /* get each probe from probeconf and append to 'probe_vec' */
        skpcProbeIteratorBind(&p_iter);
        while (skpcProbeIteratorNext(&p_iter, &p)) {
            if (0 == skpcProbeGetSensorCount(p)) {
                continue;
            }
            if (0 == skpcProbeIsVerified(p)) {
                continue;
            }
            if (skVectorAppendValue(probe_vec, &p)) {
                skAppPrintErr(FMT_MEM_FAILURE);
                return -1;
            }
        }

        if (0 == skVectorGetCount(probe_vec)) {
            skAppPrintErr("No probes are associated with the sensors");
            return -1;
        }
        return 0;
    }

    /* create a temporary vector to hold the probes */
    tmp_vec = skVectorNew(sizeof(skpc_probe_t*));
    if (NULL == probe_vec) {
        skAppPrintErr(FMT_MEM_FAILURE);
        goto END;
    }

    /* create modifiable copy of the 'sensor_name' */
    sensor_name_copy = strdup(sensor_name);
    sensor_name_tokens = sensor_name_copy;
    if (NULL == sensor_name_copy) {
        goto END;
    }

    /* loop over list of sensor names; for each name find the
     * corresponding sensor object; for each object, find the probes
     * for that sensor. */
    while ((sensor_token = strsep(&sensor_name_tokens, ",")) != NULL) {
        /* check for empty token (e.g., double comma) */
        if ('\0' == *sensor_token) {
            continue;
        }

        /* loop over all sensors */
        found_sensor = 0;
        skpcSensorIteratorBind(&s_iter);
        while (skpcSensorIteratorNext(&s_iter, &sensor)) {
            /* find the named sensor */
            if (0 != strcmp(sensor_token, skpcSensorGetName(sensor))) {
                continue;
            }
            ++found_sensor;

            /* get the probes for the sensor and add to tmp_vec */
            skpcSensorGetProbes(sensor, tmp_vec);
        }

        if (0 == found_sensor) {
            skAppPrintErr("Sensor configuration does not define sensor '%s'",
                          sensor_token);
            goto END;
        }
    }

    /* filter any unverified probes out of tmp_vec, and ensure that
     * each probe only gets added to probe_vec one time. */
    for (i = 0; NULL != (p_ptr = skVectorGetValuePointer(tmp_vec, i)); ++i) {
        if (0 == skpcProbeIsVerified(*p_ptr)) {
            continue;
        }
        seen = 0;
        for (j=0; NULL!=(p_ptr2 = skVectorGetValuePointer(probe_vec, j)); ++j){
            if (*p_ptr == *p_ptr2) {
                seen = 1;
                break;
            }
        }
        if (!seen) {
            if (skVectorAppendValue(probe_vec, p_ptr)) {
                skAppPrintErr(FMT_MEM_FAILURE);
                goto END;
            }
        }
    }

    if (0 == skVectorGetCount(probe_vec)) {
        skAppPrintErr("No probes founds for sensor '%s'", sensor_name);
        goto END;
    }

    /* success */
    rv = 0;

  END:
    if (tmp_vec != NULL) {
        skVectorDestroy(tmp_vec);
    }
    if (sensor_name_copy != NULL) {
        free(sensor_name_copy);
    }
    return rv;
}


/*
 *  format = defaultDetermineFileFormat(probe, ftype);
 *
 *    This function is used when either the --pack-interfaces switch
 *    is given or when the packing logic plug-in (normally
 *    probeconf-<SITE>.c) does not set a function to return the file
 *    format.
 *
 *    This function returns the most compact file format capable of
 *    holding all the infomation that this installation of SiLK
 *    supports.  All these file formats provide space for the next-hop
 *    IP address and SNMP interface information.
 */
static fileFormat_t defaultDetermineFileFormat(
    const skpc_probe_t  UNUSED(*probe),
    flowtypeID_t        UNUSED(ftype))
{
    /* Use a format with SNMP interface information */
#if   SK_ENABLE_IPV6
    return FT_RWIPV6ROUTING;
#else
    switch (skpcProbeGetType(probe)) {
      case PROBE_ENUM_NETFLOW_V5:
        return FT_RWROUTED;

#if   SK_ENABLE_ASA_ZERO_PACKET_HACK && SK_ENABLE_NETFLOW9
      case PROBE_ENUM_NETFLOW_V9:
      case PROBE_ENUM_SILK:
        return FT_RWGENERIC;
#endif

      default:
        return FT_RWAUGROUTING;
    }
#endif  /* #else of #if SK_ENABLE_IPV6 */
}


/*
 *  rwios = openOutputStream(timestamp, sensor_id, flowtype, probe);
 *
 *    Creates an rwIO Stream to store flow records of the specified
 *    'flowtype', hourly 'timestamp', and 'sensor_id'.
 *
 *    The file's location is determined by using the file location
 *    rules that are specified in the silk.conf file.
 *
 *    Gets a write lock on the file.
 *
 *    Returns the stream on success, or NULL on error.
 */
static skstream_t *openOutputStream(
    const cache_key_t  *key,
    void               *v_probe)
{
    const skpc_probe_t *probe = (skpc_probe_t*)v_probe;
    char filename[PATH_MAX];
    skstream_t *rwIOS = NULL;
    sk_file_header_t *hdr;
    fileFormat_t file_format;
    int errnum = 0;
    int creating_file = 0;
    int rv;

    /* Build the file name--WHERE the records will be written onto
     * disk. */
    if (output_mode == OUTPUT_SENDING) {
        char tmpbuf[PATH_MAX];
        char *fname;

        sksiteGeneratePathname(tmpbuf, sizeof(tmpbuf),
                               key->flowtype_id, key->sensor_id,
                               key->time_stamp, "", NULL, &fname);
        snprintf(filename, sizeof(filename), "%s/%s",
                 incremental_directory, fname);
    } else {
        sksiteGeneratePathname(filename, sizeof(filename),
                               key->flowtype_id, key->sensor_id,
                               key->time_stamp, "", NULL, NULL);
    }

    if (skFileExists(filename)) {
        /* Open existing file for append, lock it, and read its header */
        DEBUGMSG("Opening existing output file %s", filename);

        if ((rv = skStreamCreate(&rwIOS, SK_IO_APPEND, SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(rwIOS, filename))
            || (rv = skStreamOpen(rwIOS)))
        {
            goto END;
        }
        if (!no_file_locking) {
            rv = skStreamLockFile(rwIOS);
            if (rv) {
                errnum = skStreamGetLastErrno(rwIOS);
                goto END;
            }
        }
        rv = skStreamReadSilkHeader(rwIOS, NULL);

    } else {
        /* Open a new file, lock it, create and write its header */
        INFOMSG("Opening new output file %s", filename);
        creating_file = 1;

        if ((rv = skStreamCreate(&rwIOS, SK_IO_WRITE, SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(rwIOS, filename))
            || (rv = skStreamMakeDirectory(rwIOS))
            || (rv = skStreamOpen(rwIOS)))
        {
            goto END;
        }
        if (!no_file_locking) {
            rv = skStreamLockFile(rwIOS);
            if (rv) {
                errnum = skStreamGetLastErrno(rwIOS);
                goto END;
            }
        }

        /* Get the file output format---HOW the records will be written to
         * disk. */
        file_format = packlogic.determine_fileformat_fn(probe,
                                                        key->flowtype_id);

        /* Get file's header and fill it in */
        hdr = skStreamGetSilkHeader(rwIOS);
        if ((rv = skHeaderSetFileFormat(hdr, file_format))
            || (rv = skHeaderSetRecordVersion(hdr, RWFLOWPACK_DEFAULT_VERSION))
            || (rv = skHeaderSetCompressionMethod(hdr, comp_method))
            || (rv = skHeaderSetByteOrder(hdr, byte_order))
            || (rv = skHeaderAddPackedfile(hdr, key->time_stamp,
                                           key->flowtype_id, key->sensor_id))
            || (rv = skStreamWriteSilkHeader(rwIOS)))
        {
            goto END;
        }
    }

  END:
    if (rv) {
        skStreamPrintLastErr(rwIOS, rv, &CRITMSG);
        if (ENOLCK == errnum) {
            INFOMSG("Unable to get write lock; consider using the --%s switch",
                    appOptions[OPT_NO_FILE_LOCKING].name);
        }
        skStreamDestroy(&rwIOS);
        if (creating_file) {
            /* remove the file if we were creating it, so as to not
             * leave invalid files in the data store */
            unlink(filename);
        }
        rwIOS = NULL;
    }

    return rwIOS;
}


/*
 *  ok = writeRecord(&rwrec, probe);
 *
 *    Given the time, sensor, and flowtype values as set in the SiLK
 *    flow record 'rwrec', pack the record into the correct file using
 *    the appropriate file output format.  Return 0 on success; return
 *    -1 to indicate a problem with the record, or -2 to indicate a
 *    problem with the file, or -3 to indicate an issue flushing a
 *    file that was removed from the cache.
 */
static int writeRecord(const rwRec *rwrec, const skpc_probe_t *probe)
{
    cache_entry_t *entry;
    cache_key_t key;
    skstream_t *rwIOS;
    int rv;

    /* The flowtype (class/type) and sensor says where the flow was
     * collected and where to write the flow. */
    key.flowtype_id = rwRecGetFlowType(rwrec);
    key.sensor_id = rwRecGetSensor(rwrec);

    /* Determine the hour this data is associated with.  This is a UTC
     * value expressed in milliseconds since the unix epoch, rounded
     * (down) to the hour. */
    key.time_stamp = rwRecGetStartTime(rwrec);
    key.time_stamp -= key.time_stamp % 3600000;

    /* Get the file from the cache, opening an existing file or
     * creating a new file as required.  This function may invoke
     * openOutputStream() as a callback.  */
    switch (skCacheLookupOrOpenAdd(stream_cache, &key, (void*)probe, &entry)) {
      case 0:
        break;
      case -1:
        /* problem opening file or adding file to cache */
        return -2;
      case 1:
        /* problem closing existing cache entry */
        return -3;
    }

    rwIOS = skCacheEntryGetStream(entry);

    /* Write record */
    rv = skStreamWriteRecord(rwIOS, rwrec);
    if (rv != SKSTREAM_OK) {
        if (SKSTREAM_ERROR_IS_FATAL(rv)) {
            skStreamPrintLastErr(rwIOS, rv, &ERRMSG);
            rv = -2;
        } else {
            skStreamPrintLastErr(rwIOS, rv, &WARNINGMSG);
            rv = -1;
        }
    }

    /* unlock stream */
    skCacheEntryRelease(entry);
    return rv;
}


/*
 *  manageProcessor(fproc);
 *
 *  THREAD ENTRY POINT for the 'process_thread'.
 *
 *    Gets a flow record from the flowsource object and passes the
 *    record to writeRecord() for storage.  Runs until the flow
 *    source stream is exhausted (for file-based readers), until the
 *    global variable 'reading' is set to 0, or until an error occurs.
 */
static void *manageProcessor(void *vp_fproc)
{
    flow_proc_t *fproc = (flow_proc_t*)vp_fproc;
    reader_type_t *reader_type = fproc->reader_type;
    flowtypeID_t ftypes[MAX_SPLIT_FLOWTYPES];
    sensorID_t sensorids[MAX_SPLIT_FLOWTYPES];
    sigset_t sigs;
    rwRec rec;
    const skpc_probe_t *probe;
    int count;
    int rec_is_bad;
    int i;

    DEBUGMSG("Started manager thread for %s", reader_type->reader_name);

    sigfillset(&sigs);
    pthread_sigmask(SIG_SETMASK, &sigs, NULL);

    for (;;) {
        switch (reader_type->get_record_fn(&rec, &probe, fproc)) {
          case FP_FILE_BREAK:
            /* We've processed one input file; there may be more input
             * files.  This is a safe place to quit, so if we are no
             * longer reading, break out of the while().  Otherwise we
             * try again to get a record---we didn't get a record this
             * time. */
            if (!reading) {
                goto END;
            }
            continue;

          case FP_END_STREAM:
            /* We've processed all the input; there is no more.  Tell
             * the sender to send the packed files we've created.
             * Set 'shuttingDown' to begin the shutdown process. */
            sendFiles();
            shuttingDown = 1;
            goto END;

          case FP_GET_ERROR:
            /* An error occurred and we did not get a record.  Break
             * out of the while() if we are no longer reading,
             * otherwise try again to get a record. */
            if (!reading) {
                goto END;
            }
            continue;

          case FP_FATAL_ERROR:
            /* Exit the program.  The get_record_fn should have logged
             * an error. */
            shuttingDown = 1;
            goto END;

          case FP_BREAK_POINT:
            /* We got a record, and this is a safe place to quit if we
             * are no longer reading.  If not shutting down, process
             * the record. */
            if (!reading) {
                goto END;
            }
            /* FALLTHROUGH */

          case FP_RECORD:
            /* We got a record and we may NOT stop processing.
             * Process the record. */
            ++fproc->rec_count_total;

            /* Get the record's sensor(s) and flow_type(s) */
            count = packlogic.determine_flowtype_fn(probe, &rec,
                                                    ftypes, sensorids);
            assert(count >= -1);
            assert(count < MAX_SPLIT_FLOWTYPES);
            if (count == -1) {
                NOTICEMSG(("Cannot determine flowtype of record from"
                           " probe %s: input %d; output %d"),
                          skpcProbeGetName(probe), rwRecGetInput(&rec),
                          rwRecGetOutput(&rec));
                ++fproc->rec_count_bad;
                continue;
            }

            /* have we logged this record as bad? */
            rec_is_bad = 0;

            /* Store the record in each flowtype/sensor file. */
            for (i = 0; i < count; ++i) {
                rwRecSetFlowType(&rec, ftypes[i]);
                rwRecSetSensor(&rec, sensorids[i]);
                switch (writeRecord(&rec, probe)) {
                  case 0:
                    break;
                  case -1:
                    if (0 == rec_is_bad) {
                        /* Count bad records, but only once */
                        ++fproc->rec_count_bad;
                        rec_is_bad = 1;
                    }
                    break;
                  case -2:
                    /* error writing to disk--shut down */
                    CRITMSG(("Error writing records for probe '%s' -- "
                            " shutting down"),
                           skpcProbeGetName(probe));
                    shuttingDown = 1;
                    goto END;
                  case -3:
                    CRITMSG("Error closing file -- shutting down");
                    shuttingDown = 1;
                    goto END;
                  default:
                    CRITMSG("Unspecified error -- shutting down");
                    shuttingDown = 1;
                    goto END;
                }
            }
            break;

        } /*switch*/
    } /*while*/

  END:
    DEBUGMSG("Stopping manager thread for %s", reader_type->reader_name);

    /* thread is ending, decrement the count */
    pthread_mutex_lock(&fproc_thread_count_mutex);
    --fproc_thread_count;
    pthread_mutex_unlock(&fproc_thread_count_mutex);

    /* send a signal (normally SIGUSR2) to the main thread to tell it
     * to check the thread count */
    pthread_kill(main_thread, READER_DONE_SIGNAL);
    return NULL;
}


/*
 *  status = startAllProcessors();
 *
 *    For each flow-processor, call its start function and spawn a
 *    thread to manage that processor.
 *
 *    Start the timing_thread to flush/send files when appropriate.
 *
 *    Return 1 if something fails, or 0 for success.
 */
static int startAllProcessors(void)
{
    skTimerRepeat_t (*timer_func)(void*) = &flushFiles;
    flow_proc_t *fproc;
    size_t i;

    assert(stream_cache);

    /* Start each flow_processor, but don't start reading records
     * until every processor is running */
    for (i = 0; i < num_flow_processors; ++i) {
        fproc = &flow_processors[i];

        DEBUGMSG("Starting flow processor #%lu for %s",
                 ((unsigned long)i + 1u), fproc->reader_type->reader_name);

        if (fproc->reader_type->start_fn(fproc) != 0) {
            ERRMSG("Unable to start flow processor #%lu for %s",
                   ((unsigned long)i + 1u), fproc->reader_type->reader_name);
            return 1;
        }
    }

    reading = 1;

    /* Spawn threads to read records from each processor */
    for (i = 0; i < num_flow_processors; ++i) {
        fproc = &flow_processors[i];

        pthread_mutex_lock(&fproc_thread_count_mutex);
        ++fproc_thread_count;
        pthread_mutex_unlock(&fproc_thread_count_mutex);

        if (skthread_create(fproc->reader_type->reader_name,
                            &fproc->thread, &manageProcessor, fproc))
        {
            ERRMSG("Unable to create manager thread #%lu for %s",
                   ((unsigned long)i + 1u), fproc->reader_type->reader_name);
            reading = 0;
            return 1;
        }
    }

    if (output_mode == OUTPUT_SENDING) {
        if (input_mode != INPUT_PDUFILE) {
            timer_func = &timedSendFiles;
        } else {
            /* disable timer */
            timer_func = NULL;
        }
    }

    /* Start timer */
    if (timer_func) {
        INFOMSG("Starting flush timer");
        if (skTimerCreate(&timing_thread, flush_timeout, timer_func, NULL)
            == -1)
        {
            ERRMSG("Unable to start flush timer.");
            return 1;
        }
    }

    return 0;
}


static void stopAllProcessors(void)
{
    flow_proc_t *fproc;
    size_t i;

    if (reading) {
        INFOMSG("Stopping processors...");
        reading = 0;

        /* Give the threads a chance to quit on their own---by their
         * checking the 'reading' variable. */
        sleep(2);

        if (timing_thread != NULL) {
            DEBUGMSG("Stopping timer");
            skTimerDestroy(timing_thread);
        }

        /* stop each flow processor and join its thread */
        INFOMSG("Waiting for record handlers...");
        for (i = 0; i < num_flow_processors; ++i) {
            fproc = &flow_processors[i];
            DEBUGMSG("Stopping flow processor #%lu: %s",
                     ((unsigned long)i + 1u), fproc->reader_type->reader_name);
            fproc->reader_type->stop_fn(fproc);
            pthread_join(fproc->thread, NULL); /* join */
        }

        INFOMSG("Stopped processors.");
    }
}


/*
 *  sendFiles();
 *
 *    Closes all the open files in the stream cache and moves all the
 *    files in the incremental-directory to the sender-directory.
 */
static void sendFiles(void)
{
    int tmp_fd;
    char path[PATH_MAX];
    char newpath[PATH_MAX];
    struct dirent *entry;
    DIR *dir;
    int file_count = 0;
    int successfully_sent = 0;
    int rv;

    /* Return if sending mode not specified */
    if (output_mode != OUTPUT_SENDING) {
        return;
    }

    NOTICEMSG("Preparing to move files for sending...");

    /* Close all the output files. */
    INFOMSG("Closing incremental files...");
    if (skCacheLockAndCloseAll(stream_cache)) {
        skCacheUnlock(stream_cache);
        CRITMSG("Error closing incremental files -- shutting down");
        exit(EXIT_FAILURE);
    }

    /* Open the directory holding the files to send and loop over the
     * files in the directory. */
    dir = opendir(incremental_directory);
    if (NULL == dir) {
        CRITMSG("Fatal error: Unable to open incremental directory %s: %s",
                incremental_directory, strerror(errno));
        skCacheUnlock(stream_cache);
        exit(EXIT_FAILURE);
    }

    INFOMSG("Moving files to %s...", sender_directory);
    while ((entry = readdir(dir)) != NULL) {
        /* ignore dot-files */
        if ('.' == entry->d_name[0]) {
            continue;
        }
        ++file_count;

        /* Copy each file to a unique name */
        snprintf(path, sizeof(path), "%s/%s",
                 incremental_directory, entry->d_name);
        snprintf(newpath, sizeof(newpath), "%s/%s.XXXXXX",
                 sender_directory, entry->d_name);
        tmp_fd = mkstemp(newpath);
        if (tmp_fd == -1) {
            ERRMSG("Could not create and open temporary file %s: %s",
                   newpath, strerror(errno));
            continue;
        }
        close(tmp_fd);
        rv = skMoveFile(path, newpath);
        if (rv != 0) {
            ERRMSG("Could not move file '%s' to '%s' for sending: %s",
                   path, newpath, strerror(rv));
            continue;
        }

        /* Moving a file to the sender_directory is considered a
         * successful send, even when no sender is running. */
        ++successfully_sent;
        INFOMSG("%s", newpath);
    }

    closedir(dir);

    /* Print status message */
    if (file_count == 0) {
        NOTICEMSG("No files to send.");
    } else {
        NOTICEMSG("Successfully moved %d/%d file%s.",
                  successfully_sent, file_count, CHECK_PLURAL(file_count));
    }

    skCacheUnlock(stream_cache);
}


int main(int argc, char **argv)
{
    appSetup(argc, argv);                 /* never returns on error */

    /* start the logger and become a daemon */
    if (skdaemonize(&shuttingDown, &appTeardown) == -1) {
        exit(EXIT_FAILURE);
    }

    /* Log a message about the packing logic we are using */
    INFOMSG("Using packing logic from %s", packlogic.path);

    /* Create a cache of streams (file handles) so we don't have to
     * incur the expense of reopening files */
    INFOMSG("Creating stream cache");
    stream_cache = skCacheCreate(stream_cache_size, &openOutputStream);
    if (NULL == stream_cache) {
        CRITMSG("Unable to create stream cache.");
        exit(EXIT_FAILURE);
    }

    if (output_mode == OUTPUT_SENDING) {
        /* If there are any files in the incremental-directory from
         * the previous run, move them to the sender-directory.  We
         * don't want to attempt to append to them, since they may be
         * corrupt or incomplete due to a previously bad shutdown.  If
         * they are corrupt, we'll let rwflowappend handle it. */
        INFOMSG("Preparing to send old incremental files (if any)");
        sendFiles();
    }

    /* Choose which processing thread to run based on user choice */
    if (startAllProcessors() != 0) {
        CRITMSG("Unable to start flow processor");
        exit(EXIT_FAILURE);
    }

    /* We now run forever, excepting signals, until the shuttingDown
     * flag is set or until all flow-processor threads exit. */
    while (!shuttingDown && fproc_thread_count > 0) {
        pause();
    }

    /* done */
    appTeardown();

    return 0;
}

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
