#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-ipfix-ipv6.pl 72e0a2f73883 2010-09-09 21:30:31Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = $ENV{RWFLOWPACK} || './rwflowpack';

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsilk2ipfx = check_silk_app('rwsilk2ipfix');

# find the data files we use as sources, or exit 77
my %file;
$file{v6data} = get_data_or_exit77('v6data');

# verify that we have FIXBUF support
exit 77 unless $SiLKTests::SK_ENABLE_IPFIX;

# verify we have IPv6 support
exit 77 unless $SiLKTests::SK_ENABLE_IPV6;

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or exit 77;

# create our tempdir.  clean up unless the appropriate environment
# variable is set.
my $tmpdir = File::Temp::tempdir(CLEANUP => !$ENV{SK_TESTS_SAVEOUTPUT});
exit 77 unless -d $tmpdir;

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'ipfix', 'polldir', 'ipv6');

# Generate the test data
my $ipfixdata = "$tmpdir/v6data.ipfix";
unlink $ipfixdata;
system "$rwsilk2ipfx --ipfix-output=$ipfixdata $file{v6data}"
    and die "ERROR: Failed running rwsilk2ipfx\n";

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     #"--log-level=debug",
                     "--sensor-conf=$sensor_conf",
                     "--move $ipfixdata:incoming",
                     "--limit=501876",
                     "--basedir=$tmpdir",
                     "--",
                     "--polling-interval=5",
                     "--byte-order=big",
                     "--compression-method=none",
    );

# run it and check the MD5 hash of its output
check_md5_output('a78a286719574389a972724d761c931e', $cmd);

# the following directories should be empty
my @empty_dirs = qw(error incoming incremental sender);
for my $d (map {"$tmpdir/$_"} @empty_dirs) {
    next unless -d $d;
    unless (opendir D, $d) {
        warn "ERROR: Cannot open directory '$d': $!\n";
        next;
    }
    while (my $f = readdir(D)) {
        next if $f =~ /^\.\.?$/;
        die "ERROR: Found unexpected file '$d/$f'\n";
    }
}

# path to the data directory
my $data_dir = "$tmpdir/root";
die "ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# number of files to find in the data directory
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every packed file we expect to find.  Read
# these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $data_dir);

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files in root; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename
    s,^.*/,,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};
    ++$file_count;

    # do the MD5 sums match?  IPv6 files do not use the bytes/packet
    # ratio, so no rounding issue to work-around
    check_md5_file($md5_map{$_}, $path);
#                     "$rwcat --compression=none --byte-order=little $path");
}


__DATA__
f974c3ede806e5ebbe23da765f2ce56b in-S0_20090212.00
a6b31cfb7872c8c9082dea1902d980af in-S0_20090212.01
50995b8129ad5aaba14523514c0b72f1 in-S0_20090212.02
8eeba1f6b8718920826ec3042cd1dba6 in-S0_20090212.03
10cb73815275159c7760f04c7777c6e5 in-S0_20090212.04
6944b87403dda665dbb8f7e85db71e06 in-S0_20090212.05
c0ad598ced07f862118f3df3825f983b in-S0_20090212.06
d7f24b824b7bcaa47fbc3cf02431f581 in-S0_20090212.07
6eeed1a32fd070a78529a809e63a6566 in-S0_20090212.08
83f332da6a02058a765e91ec38155dc1 in-S0_20090212.09
4918837e4008ea93b56b04b892e2a04b in-S0_20090212.10
1996a4ef4ab8e9cd2ba56249029e43cf in-S0_20090212.11
ab0fe0a6d19f6be273543ac1879fd7bf in-S0_20090212.12
0d685db8834a763a4dbe41fd1702f430 in-S0_20090212.13
fe41d592d0bd521451e136ed95ae2a45 in-S0_20090212.14
a3be6dbd55c0e6bbedffdef052d767ef in-S0_20090212.15
93aadec9acffc81599b0048b66e9e801 in-S0_20090212.16
257cbd840fe0033a3a7c5e75bb9e8916 in-S0_20090212.17
39bef590ad550248eb4d15d4a6c0dada in-S0_20090212.18
80fae9e6b7df2e4b6d1c259faecf9b42 in-S0_20090212.19
13e2d5dbd15a77a16abc6b0effaef269 in-S0_20090212.20
ee733fbe06d7f183ae9a662a71b266a8 in-S0_20090212.21
6512434c9d2b7145868aa352ce5df628 in-S0_20090212.22
7a3e216c763c14cced0a57238da1e8df in-S0_20090212.23
685a245f1244afea32226620e2e76b03 in-S0_20090213.00
e3f13e885e6b6c0d820a9674f157d223 in-S0_20090213.01
d1b8f6f4af329fa94f8a9129b13ab2cb in-S0_20090213.02
7d88ee9a617708ddbd39958cbfe05372 in-S0_20090213.03
1cabed5c7fc7cdfbc2d9e6b91e598210 in-S0_20090213.04
ed11b0755762e746ac11ad2aedf99577 in-S0_20090213.05
f597a5fe8e4ce06efcdbba1509e60f59 in-S0_20090213.06
d11daaf213ec73a00d47b61c1ba0fea9 in-S0_20090213.07
fab3e822e7bece1fdfd1d87645426a2a in-S0_20090213.08
c3956f2cd6e2de70bc640c1ca919d965 in-S0_20090213.09
b08aa9c0cec941d09c0092ec58c999e5 in-S0_20090213.10
29c403c3047518ec3b3b0342020ede11 in-S0_20090213.11
f25f06e69821601c158f492915a9b27d in-S0_20090213.12
98fe739792f9bc4383067c816397eb81 in-S0_20090213.13
399cfeb49a4bb16a7bc115a17c3fa7aa in-S0_20090213.14
96ab55c4afc76b7424895462c871b82f in-S0_20090213.15
d33426bf4889664a4626c57e87ff644a in-S0_20090213.16
e6ff5bc44186664f4eb9b91a4e32ac1f in-S0_20090213.17
b6d2ecd3076d7a89f9291a1dfcf813d0 in-S0_20090213.18
5526656e570763c2a55bebd01a24d4b4 in-S0_20090213.19
3e936d045f6c3395fad1fcb3cd1e72fe in-S0_20090213.20
2edb9824df3ca0f52993afaa025fc5d9 in-S0_20090213.21
43237926d7a31baff7f5fc10e67db676 in-S0_20090213.22
c175c34c4631280c491ff71f45f90151 in-S0_20090213.23
404a5d56bbaf2f315fe3ccb5eb20c92d in-S0_20090214.00
5e156ddeabecdeda8f0ed2f40a5f2ff7 in-S0_20090214.01
18e1a22a6c915893041fa8d599379cad in-S0_20090214.02
e530fd1861f7df243a0350bc64bb9bf8 in-S0_20090214.03
bff87a974c359237b3f61ab6e9ecc728 in-S0_20090214.04
da835617b2f7e043d3c0e3190c7dfe27 in-S0_20090214.05
32c1874b33ae6b08a7152d0a406eb93a in-S0_20090214.06
5605194505de386a514303aec863ea53 in-S0_20090214.07
5e50ed5dfc43a9c9854f1fa548a0a7af in-S0_20090214.08
0464c5355e85ea599f91dd3a8b720526 in-S0_20090214.09
72cd413a126d30f559b137f1df293fea in-S0_20090214.10
5da9a756be1f324cebfc8370f1562c70 in-S0_20090214.11
b61d1448255d4b1059ad8fcc4283bca8 in-S0_20090214.12
0b92e47f92d5443f8cd43fca6bd9befc in-S0_20090214.13
ff698a7c6712ef6d7d97f05af9829396 in-S0_20090214.14
28f03a63adab92ab8279663477a50b31 in-S0_20090214.15
0afba0434ae6be3aeed3c13e92190609 in-S0_20090214.16
8c5bf43399e067cfc7461c40788281e0 in-S0_20090214.17
1cac4ddf5063b735b39bb56e9d0f16c0 in-S0_20090214.18
6f6011ab23c38f2407bf9f9433bff91a in-S0_20090214.19
2244f0889aed3423ad79de0dd09049dd in-S0_20090214.20
77fea57bf43695d01773d4852ef87efc in-S0_20090214.21
cae03b1b36b2fd36408d02d1f7a3090b in-S0_20090214.22
b2ef13d47be14c689e8321ad345fd68a in-S0_20090214.23
45a91a83a3e1b6242bb82d76a6090a5c iw-S0_20090212.00
712871074081c50ecdb7a2a52b447001 iw-S0_20090212.01
1c7b41719c068f55048f673f5aa630a2 iw-S0_20090212.02
e6c4eee31c8c2e65021ddcc99bbec417 iw-S0_20090212.03
032b31b01d5be1226b419c71c3771cda iw-S0_20090212.04
2697c0ce15c94d451ed8f2e9f2eb7210 iw-S0_20090212.05
4fb0a63e3bbc6309f9d5d8bb248a153d iw-S0_20090212.06
bc7d7593e07951f0d8995c52631727b3 iw-S0_20090212.07
eed411dd35ed1dbd367638c7e065d9a5 iw-S0_20090212.08
335c339f4a2a1063e4998ceb55b8ef08 iw-S0_20090212.09
06b8e22086734618b41a8872cc92b66a iw-S0_20090212.10
ccd56e7604fa5eca87ef9c6d4d66701d iw-S0_20090212.11
5da1412b509c91289eb6b961f4135f22 iw-S0_20090212.12
101e1ea94d1a0b5a3dc600b20bd5717d iw-S0_20090212.13
bcd7445258d6bf5a64687f88a81dd919 iw-S0_20090212.14
ce287240a6c581ae0095a404129ffdda iw-S0_20090212.15
cf4cb2e6421b0e22db1f64442ace3f22 iw-S0_20090212.16
4d33dcd7bb8a4e38f8f2081cc37c41ed iw-S0_20090212.17
212cdc1df504f58959334171292f0294 iw-S0_20090212.18
095765e73751c3bbc9f7e6a69b514f4a iw-S0_20090212.19
33562345399d1e83a5027a899d24b87f iw-S0_20090212.20
eb9e4964a9f8f49c39077eed7371decd iw-S0_20090212.21
51aae8896321de2534a04ce99dc848ca iw-S0_20090212.22
c5f6ecfbd951d36c52d77800d42386fa iw-S0_20090212.23
8f7261e36916a04076d7c5822614192b iw-S0_20090213.00
0bab07fd85f2dbe0b9455c6a37737240 iw-S0_20090213.01
f0f6375f04405d901eba482ccd3621b7 iw-S0_20090213.02
cda4db7daff7516193eec55690bf4bb3 iw-S0_20090213.03
2fdcd5e9f48ed230d6de14ecbe676a05 iw-S0_20090213.04
1afdee232eab87eebedc0b4505c95884 iw-S0_20090213.05
f24cef36b1b8acc47b73b1feae133560 iw-S0_20090213.06
2a0d2c8cca5b2db5f06867f7ae1a7969 iw-S0_20090213.07
607eb3705456d60894f60879815b9c26 iw-S0_20090213.08
a9b54e0896e0b99f68f85f6ab719941a iw-S0_20090213.09
a66229681e85b3161dd67b363a014ff0 iw-S0_20090213.10
fb68a3ea109a85276ab1a78135a85adf iw-S0_20090213.11
3f10088f3ee34c8e4a62875868b8df30 iw-S0_20090213.12
656057b5e4205090af2088f844b9a0b7 iw-S0_20090213.13
3229d26b4860db10ef7c9c8b879bd790 iw-S0_20090213.14
2507d885662b26cb2d2bee5d3f9b0103 iw-S0_20090213.15
8a5d069136b72c46a164c801a00384db iw-S0_20090213.16
26ee294d0f4739c85213e1a5f8c4edfe iw-S0_20090213.17
c39d77255c73bba754c7c24d6243a6bf iw-S0_20090213.18
09ce729e77768e5ded7e90fd3ddbc8f2 iw-S0_20090213.19
7ba3908ff101f148dfcb24b4c8a74235 iw-S0_20090213.20
d14fffb96d0dcc7702310a330e8a71b1 iw-S0_20090213.21
90e0a175441926b6588a61789c32bc34 iw-S0_20090213.22
6113d4d3b8f3187b7bf40cda15314cfb iw-S0_20090213.23
77b890285473a624d6c0c88a99558048 iw-S0_20090214.00
573d24d4ad3f69b4df2438ab56edb6ca iw-S0_20090214.01
93fb96f5c3db8ba66e204e8188ee2a04 iw-S0_20090214.02
2512303d6e1512f80de825bb114b4c0b iw-S0_20090214.03
e129ffec91a5680d5a3bfbabb188ce73 iw-S0_20090214.04
1613e0849ebdd4640afa4fd8649f4077 iw-S0_20090214.05
4414d7b0c72c74cf7260c61ceda9fc8b iw-S0_20090214.06
d9bd99c82c7e2e9f164daa8fe6406212 iw-S0_20090214.07
80fa3311b685df8c9cb93bc2fc101fe5 iw-S0_20090214.08
287912370577cf4fe8ad32dcde934dfa iw-S0_20090214.09
153bffc8df8fee2a5d8981b23dc6c5c9 iw-S0_20090214.10
c75d657c23fb34386ce45b188b433ed0 iw-S0_20090214.11
4b2f5fdf4030fa595fb0c0a97227a498 iw-S0_20090214.12
c1abd33ba4bfeab0a17152a6b74ba535 iw-S0_20090214.13
d21d5964746ec35ce07303b61a175aa7 iw-S0_20090214.14
4dfb8ffc7d737f2797c180939a2fc674 iw-S0_20090214.15
0e5f9f081f80dcd63d819f0c5289a492 iw-S0_20090214.16
884e722f09619552816b2e3a5347c259 iw-S0_20090214.17
6d2cd50f8f678882d0394659cd692641 iw-S0_20090214.18
b9a5c3fa26d206d18c657867f48ad0d8 iw-S0_20090214.19
5e77130428270fa3e0904912c4d47037 iw-S0_20090214.20
47d03860f8e4c0c2918a4a6e49837c51 iw-S0_20090214.21
1c520425807199d67816e62ac66224b0 iw-S0_20090214.22
fc44d34c64d88bed699d6b099141fa49 iw-S0_20090214.23
d6cfad0e8b16fa1ddacf0af29f15f3e8 other-S0_20090212.00
d4aae743a1ba6654071ade3bfaa09762 other-S0_20090212.01
8c3468fb5f624b0f138e56c902448770 other-S0_20090212.02
1e7b92c8601de86c1dd15522e3cccef4 other-S0_20090212.03
cdfb19b72ab2b2a618fff1386c0d5434 other-S0_20090212.04
6919806a54dc58793fff5505d197e4e2 other-S0_20090212.05
9d3714f01b9d16d753807711288d88ea other-S0_20090212.06
40ab36d2dfb335f2efba2a9e3fb1d344 other-S0_20090212.07
f144e5588f2976d40b8a93a6e8ce6d09 other-S0_20090212.08
6564d33e46db38b4ab08c62c54259234 other-S0_20090212.09
7d79d08af28ba72a5107304ed6106cc5 other-S0_20090212.10
37fb87e54c50d1f6f6ab57238bcf8356 other-S0_20090212.11
da153282c032672b6e0f28507a24550d other-S0_20090212.12
378c975f384fd8ba090276031661716f other-S0_20090212.13
76d2c39bbfa6bc1f97b2005af1c70102 other-S0_20090212.14
0691cb08deec09ee7b5b402c9129515f other-S0_20090212.15
f9747e1f6de171159c3386700d3bae16 other-S0_20090212.16
fda50dd259260c87de33296e3fd2a18d other-S0_20090212.17
fd10cd1ee58a86db20161c92f2db8d75 other-S0_20090212.18
7d00bd4b667cede4f8d365f89d8d8aca other-S0_20090212.19
570812f0fad674348b0edadbfd12a232 other-S0_20090212.20
ff26bfd364e2369e7a789ce0458be94e other-S0_20090212.21
644fc7a3fd5edf99b01a61e6c3ade555 other-S0_20090212.22
b1042bf5a47dac769c82c902ab443e71 other-S0_20090212.23
e79827eaf614666a800cb06e8f69d7d5 other-S0_20090213.00
d598167680adc13be2a6732fee5472d7 other-S0_20090213.01
8db46471321518fdf8344d30729a0b5a other-S0_20090213.02
07b3cf74f30077c3ead5fbfc73be5ef9 other-S0_20090213.03
5f3d525dbf9181714bc8b98db4bbdbfb other-S0_20090213.04
0fe1c027daca35e824b171daf04f3850 other-S0_20090213.05
32ccea73fb22db05241518a7f09360e2 other-S0_20090213.06
82c0d29cf28020e785808562512f93e9 other-S0_20090213.07
390d39d38403f1c7879c7592468d0d77 other-S0_20090213.08
798cfdcd7bbd0657c45c187df9deb6cc other-S0_20090213.09
5513ee76a4ffb564bb8502f3cb6de482 other-S0_20090213.10
1099fecbcdbc65dd28601642593851fb other-S0_20090213.11
d95da1ceecb541adfd24f1e39c5f6c6b other-S0_20090213.12
e5fb0fc41b2cba427d297583ff7cb287 other-S0_20090213.13
b8644e80519908e86ba92c9e0945ea50 other-S0_20090213.14
7c55aec75fb12d54ea97392121b34268 other-S0_20090213.15
13c7f9077d893b1d162b9e34fce8ab8a other-S0_20090213.16
495b2bc9764fdf144529c121acb60dde other-S0_20090213.17
2d630759ac8675ca2448a101af4d8ad8 other-S0_20090213.18
a9e1028253bdaf34fd8bcab732b1af76 other-S0_20090213.19
8ae6229753a1b457bcb2feee50f0c75b other-S0_20090213.20
2ed66acebb32c62cd63958e10ab3df8f other-S0_20090213.21
59c6936aa54850647c3e58983aab3fc2 other-S0_20090213.22
1bc1014a2886cadf4ac26cb743f2478d other-S0_20090213.23
625c47defa92d4ad5a9d992ecc09d486 other-S0_20090214.00
5942eb564a1076e69fd550bdf3bc518c other-S0_20090214.01
054fad516de02f394279d4c1397e5c91 other-S0_20090214.02
6b7b97b5e4370c778e862c7b35de35d5 other-S0_20090214.03
ff19db2b5e1c493e3d2296a294bc697b other-S0_20090214.04
a6725aa63e94ee5757627cc47a4d3bea other-S0_20090214.05
90c6483e0a99f8e4545dc79e1c73fec7 other-S0_20090214.06
8efec844fb2122f627f61eba393d30af other-S0_20090214.07
e07006f83532773008b617c32d8d5daf other-S0_20090214.08
7591bcac38b1d247d2144c7c67bbc1b0 other-S0_20090214.09
2109d4a64a3943a100ef9be3935f4399 other-S0_20090214.10
436ce1396992294775727e09a230c2c6 other-S0_20090214.11
8661c3f4c27647b0a8a4765df3007b1e other-S0_20090214.12
23f810de65a3cadd8dd14ed5e9b2f89b other-S0_20090214.13
a53916a2997ba509bafb51ad9dec3996 other-S0_20090214.14
da312786a23c256bab0251f81a11a349 other-S0_20090214.15
2efa392009365ef701140ada49174fe3 other-S0_20090214.16
04ac727c05f969b17906931669e3a623 other-S0_20090214.17
a32d8a4465b7fc4f9bc822a6f9810964 other-S0_20090214.18
a8009b77b1d405f7623001cba3ce92c3 other-S0_20090214.19
a5c49da4183361300b0fd5ee464ea293 other-S0_20090214.20
aaabdfc31064884be9b7c1d454fa5f57 other-S0_20090214.21
b460204941916f912d51211ada19ef54 other-S0_20090214.22
1bd77dbe03acbe12bfc8fbd8d234028b other-S0_20090214.23
efc892e6e6a094661ff8e1e2cf532512 out-S0_20090212.00
7416a000b1f4b325eb5c55e0007ed99f out-S0_20090212.01
7f301a3c6904fa25895dc2ce6a283108 out-S0_20090212.02
5fe73841cac6d114967bddcf667a6d0a out-S0_20090212.03
1d79d4318eef1b68a54acf4b890bf763 out-S0_20090212.04
426f020f71d1da01bbe27612c16b0904 out-S0_20090212.05
6f69e0e4cf3c99dea6e7b8526dd2698f out-S0_20090212.06
0e026c3301e3126ab4fe38bed47ce5f9 out-S0_20090212.07
8118b739348ca64e4cd196fc3c42a393 out-S0_20090212.08
d1922908ada0e96f91119123acc48d2e out-S0_20090212.09
86d9284910438ee284a6456262e86ffb out-S0_20090212.10
d86aa8aa469c1b93aaae0dce62aae6a4 out-S0_20090212.11
8e25cec58d518b3c490b3de11a8cf6a6 out-S0_20090212.12
49912b91c6820a1e31d91fbaf93d37fc out-S0_20090212.13
4500a5802d8ed93558c0bff78f174e64 out-S0_20090212.14
cc6db6ebf8d976cfe067a4268250d7c3 out-S0_20090212.15
f4ae226d50b6cf489303cd9185f35c9d out-S0_20090212.16
60d8516b3611f654528e90289c8b9877 out-S0_20090212.17
393267e5ab9ad37b359c2389e57a5717 out-S0_20090212.18
4249f90cc477694cc024a9d8aa16f97a out-S0_20090212.19
3b46601bfd99366435eb6e1925c10ea5 out-S0_20090212.20
695c33a239333129dd69cb897e00efef out-S0_20090212.21
46e58ca68824e9c52bcd0ccdd013d984 out-S0_20090212.22
b5b70cc1ec34323a6e0df6080a277abd out-S0_20090212.23
fadc7bce8270387c8eef5258e790086d out-S0_20090213.00
5218474f00bd80fc528818e15c5b36ea out-S0_20090213.01
ffb31991676606e728ae804778835643 out-S0_20090213.02
ae2da800d08119ff77193c04a4ce2438 out-S0_20090213.03
0c332b18618b4abfe05cef7e23d17616 out-S0_20090213.04
3b7b29234b98eeb9daf1d8ed6c6cd9b2 out-S0_20090213.05
6bf4bd07caa6699c1ecedf49f6673348 out-S0_20090213.06
5ac5b879330637bd86a408ef0f0df0de out-S0_20090213.07
3ff424d1b644b58871c9e93f4bb6fa62 out-S0_20090213.08
e6b51158a44022e5647cccb72db9b57f out-S0_20090213.09
f02d9c7b56dc09e66d01d1d059dcf9f9 out-S0_20090213.10
0a6693a63bd818580796422de6cd3261 out-S0_20090213.11
d62c3004228fafc1ee02342883b001f9 out-S0_20090213.12
6a6a3f0f2a1079aaad512e46bce6e013 out-S0_20090213.13
1716636b566ce9ba7eb4c96a4d99bf90 out-S0_20090213.14
9a455ae392bad85ac7111acda4566091 out-S0_20090213.15
03f857f463871a5057938aab902f360f out-S0_20090213.16
97d2e73a76be813c0f0fd67969d48e49 out-S0_20090213.17
2cea62908869becf790ff56edfe1c120 out-S0_20090213.18
a2baf97a7bb5080f4cd4f3f9c72ff58e out-S0_20090213.19
8d206995c4838214a0e08272adff1e3b out-S0_20090213.20
2faabbd6efb80dd95bf13bcde44279f6 out-S0_20090213.21
6d4dc20b7afb7387063ef5e934c312c0 out-S0_20090213.22
84d655cefc0ba51abd7f9726322e2753 out-S0_20090213.23
b46396190242c5fe33a455771c36e08b out-S0_20090214.00
be277c5825f9a3f8bb38c89fe11a2506 out-S0_20090214.01
23bcab3415a4394a78b8febb6e74bc48 out-S0_20090214.02
8a39532be3da002fdf756e8e83cb64bc out-S0_20090214.03
3737e03c6354eddec3d86291c78c23bd out-S0_20090214.04
f15a433e4462ca98b7a3402d40cccdd4 out-S0_20090214.05
f964cf12de6541e1df41e91f0188b0ad out-S0_20090214.06
75adcee6b392b3674532d50e0a6d4e18 out-S0_20090214.07
ad36b466960c2cd5cb4bf2e6b5f67bb5 out-S0_20090214.08
a3cc2c01160cf052c3c771fa464aed79 out-S0_20090214.09
8fb60ebf0f30155a8ae493157a4799a9 out-S0_20090214.10
fa6a24a0ca4d3c954dd9e6fdd6577f88 out-S0_20090214.11
5056e218f681e9e15bb1367c6d0ea384 out-S0_20090214.12
bfa26780fa8b432a5e7dfdc9125f9372 out-S0_20090214.13
d148d80969c1fcfd7cdbd790a10f0ca0 out-S0_20090214.14
ae9f0d442a7a0704dee08fe0ccd13794 out-S0_20090214.15
93c60d7ca350819a06826a17ecc4df58 out-S0_20090214.16
359950cc4ae48229a726ff35cd376e2c out-S0_20090214.17
56cbcb967dbd8eddc6f4a551e4588abe out-S0_20090214.18
4e0c7ad9ecefc39f4b1e17b75600a79e out-S0_20090214.19
c440557eb31b37881c2188a72cedab47 out-S0_20090214.20
1ac329cea8c657834e1f81534902a004 out-S0_20090214.21
57ea7aad372a9bc76d78dd85a21046f0 out-S0_20090214.22
c1f9a543a1c63c2a6678477f9893fa85 out-S0_20090214.23
dbe2bb89d387b4c130d4547cd4d841b4 outnull-S0_20090212.00
6b02ce30f899eaa27cb36d86f420d7b7 outnull-S0_20090212.01
bdf956636ba54b313a1c7b581698a484 outnull-S0_20090212.02
4029e1ba272737be684331161db42993 outnull-S0_20090212.03
27a0d2d0f67af7d7623968918730731d outnull-S0_20090212.04
ab96e62a40555203a4b9a7fee25976fe outnull-S0_20090212.05
590a30bc3e666ab54ebbbd5ab876ef4e outnull-S0_20090212.06
6b3bb44027f2644646a930b5adb9b7ab outnull-S0_20090212.07
50ff617ef1f26e5d7117366a2e05cfcc outnull-S0_20090212.08
740113c887e9d06fa7f7be654df2ed05 outnull-S0_20090212.09
25b45edbd802845b962e33a9f5e2f65a outnull-S0_20090212.10
6cc91f2afb539af49525355a545fb881 outnull-S0_20090212.11
228f032ccd62dac722a912faf7d62ed9 outnull-S0_20090212.12
fccbd9103b96cb7b60c4e4604072519b outnull-S0_20090212.13
98b7f3531503b21b1100db14ef8cdfc4 outnull-S0_20090212.14
ba115438fb24362aab955df4ebd66a76 outnull-S0_20090212.15
7c2a58aa5780f488a7e93121602e386f outnull-S0_20090212.16
f5ed9908db91ceb7ef3a0464f1a0db84 outnull-S0_20090212.17
79a278ba5e2b27560df1a39db941f857 outnull-S0_20090212.18
965ab022b9c5ef6180b154c0697c8789 outnull-S0_20090212.19
f0c2908415c5adbfb4e5e495f68e2d74 outnull-S0_20090212.20
81ea4bb067031a474149d7a4683b02b4 outnull-S0_20090212.21
3bdee6dd1b63d8e875d2cb0a398cbcfc outnull-S0_20090212.22
ac7c447770417dd8ca8ac1dd88aa754f outnull-S0_20090212.23
0f9c87bf8ce17d5140a8f3f1bb8166d9 outnull-S0_20090213.00
297c90247e8887ff3240e25ef07323ae outnull-S0_20090213.01
57e0d9071229383b664f21ddda408d8e outnull-S0_20090213.02
bdca9062f68aab1112d3c6eb2c6645cd outnull-S0_20090213.03
27edc3934b86adf54ab840a4eded562a outnull-S0_20090213.04
51aef37be2165dd88f3c47771279ebb8 outnull-S0_20090213.05
4413601c362430700c50ae195344eade outnull-S0_20090213.06
5ec50aae0fda002d5feb95675b4e6920 outnull-S0_20090213.07
0e4a6fa24b45abe80b0d8e262e4bc93e outnull-S0_20090213.08
1aa3cd4769b10853c721e758661b8bd5 outnull-S0_20090213.09
fcf54d0db0620bf953bd51501d9e5b80 outnull-S0_20090213.10
1b4c42c81189410fa4844be7c4cc450d outnull-S0_20090213.11
bd9ab58b1478be4c8df44a2e6efda73d outnull-S0_20090213.12
d624239581ad7da4aad5a6df82bca7b1 outnull-S0_20090213.13
0c0ab43c287d5347213a733482bad121 outnull-S0_20090213.14
7061ea18fdfd4d0bf64a846450a3685d outnull-S0_20090213.15
cb9e2e9f9ad1c0d9980c62797a4fe0c2 outnull-S0_20090213.16
e827c6e3ada513da6ab2710783d6b6a0 outnull-S0_20090213.17
e758b9e2214efbd197e9ea6315af2d9f outnull-S0_20090213.18
19d16fd1de200bb081782c3f66e6618b outnull-S0_20090213.19
e0a9be2a65f4906825b03572eafb37ff outnull-S0_20090213.20
45035ceb53c998d1293f8485cf21adca outnull-S0_20090213.21
5b62d29698b6f245ce3cf9a45d716d94 outnull-S0_20090213.22
8829849f3b5ff34495f249846cb298dc outnull-S0_20090213.23
698c6dcbcac2bbc6569ef6cd2eb790f9 outnull-S0_20090214.00
62deda803c40a92507e5e1c4ff891db1 outnull-S0_20090214.01
efea66651aaaa7c64b4560ce5e373736 outnull-S0_20090214.02
027d3e12e6d2b3152de32151bb137927 outnull-S0_20090214.03
36031412e6580db3b91177c1f5e94287 outnull-S0_20090214.04
4be3b5dd2f89a2acc6f9aee95dddb144 outnull-S0_20090214.05
9003173153cf202f0de250b906d64f05 outnull-S0_20090214.06
03386d8603305a52b5e6523f922c2537 outnull-S0_20090214.07
325f2f9f1f36690bd5ee06e5d730cbc0 outnull-S0_20090214.08
439fd3451ac0dcb237c39b0b960fbb51 outnull-S0_20090214.09
03bb18e52046b2c89f9340001d0afc3c outnull-S0_20090214.10
8a9c0e7d5abaf834974981a919e5cba7 outnull-S0_20090214.11
bb4714f54c18ea1170edf983614f3215 outnull-S0_20090214.12
80fa597c35d08551848b876988234ffa outnull-S0_20090214.13
f24f2fef31ec49a0c4d6971eb522bcc6 outnull-S0_20090214.14
a82fa83714fd3b86d99629dbeb324f7e outnull-S0_20090214.15
0470ead38c93bd4a74b2b99cf099c026 outnull-S0_20090214.16
457ac5f5864b80bed86aad764eaadbf1 outnull-S0_20090214.17
4b27333890eb056bfd8d295b46a35aae outnull-S0_20090214.18
cf386fa4d4a9da837f356aad177339a0 outnull-S0_20090214.19
d3fdeda2a82d22df86904660448493b3 outnull-S0_20090214.20
50a112c3a5793aaf5adca3e55e518bda outnull-S0_20090214.21
2a596d16db2828bfac65a48913ee4d1b outnull-S0_20090214.22
7874f557e01033bb98e8fe8280fca68d outnull-S0_20090214.23
0833b79ec05445b70a585e8141400e8d ow-S0_20090212.00
07f90d75f73b3735a97f2c86bc25bc2b ow-S0_20090212.01
feecfbcf6191f1bd6c3db84d31eb8044 ow-S0_20090212.02
3bd2d830fac1ba98a79bcc02592968a5 ow-S0_20090212.03
09cee513afb80bf0b63f28eda7c8c99c ow-S0_20090212.04
08c9a5f84d93205b71ba0e7728a666a7 ow-S0_20090212.05
6fa3263153efc78e44077d874a4f381b ow-S0_20090212.06
29efd3b9b73e9f35d8c19d7b2cfc64a1 ow-S0_20090212.07
31d8a378c5e1e396748456a99581642f ow-S0_20090212.08
14f8f73425061d24c4f8e19a0dfc6305 ow-S0_20090212.09
d18d5c2fbb6a11ebafae4e2a65cff51e ow-S0_20090212.10
eccd7e3421c9c9b467b25b954fa1cd18 ow-S0_20090212.11
5a24c9c9a36842db3e7f04c6cc299bb3 ow-S0_20090212.12
aef58535dd9d97f8b712b5d38c603c0c ow-S0_20090212.13
d77cdd9e4753bb5dfa422f03c099d186 ow-S0_20090212.14
727a24c2f167486cc4b30db2a7057128 ow-S0_20090212.15
ed76ba320f60bb3b258bfe1684287378 ow-S0_20090212.16
72632bb815da20bb26fd2f96cab28a07 ow-S0_20090212.17
ea8a833f55665721c13e7aade5c57010 ow-S0_20090212.18
5e2d323f27eb8a7a2dd156a85bd2aaaf ow-S0_20090212.19
a4acc0d965c26325960d63096c948bf1 ow-S0_20090212.20
f4474de7b630cda444b471aabe6c026e ow-S0_20090212.21
7608f2deccb84e1bcac965a1ce7bc5e7 ow-S0_20090212.22
5c98a79fb1beca84173f1265750e662e ow-S0_20090212.23
31d5892630ae59117480ec70f0dddc06 ow-S0_20090213.00
43c541c93ec3e27d08cfdb9f5d8566c4 ow-S0_20090213.01
2db6e17511707f78d9078e1510371ecb ow-S0_20090213.02
3dac91633725e1cca02910ab474c9a67 ow-S0_20090213.03
e62be48ea8f7c69a751c4924d9e3acca ow-S0_20090213.04
a3257959edd258268e362b55edd8be96 ow-S0_20090213.05
7ce40d2751bb4348bb3bc5141d02a5d9 ow-S0_20090213.06
675c4a9aefbe4ea4facb0ca2244640fb ow-S0_20090213.07
56b11e6e036646d6a93597a5e5bf4c14 ow-S0_20090213.08
9308734dec4f89753c8666cd9fa6fc39 ow-S0_20090213.09
197cd7cb3cba46d44cecc3be4f1bd544 ow-S0_20090213.10
04868dce36a4cbbf56e14acfbab98d5c ow-S0_20090213.11
6f62771d2466a0815d047dc2e6a4a79b ow-S0_20090213.12
3ce1272b6adb21fc3245b837d34dd780 ow-S0_20090213.13
a5dd3675055c1c96a84897e9b888880a ow-S0_20090213.14
c20070bfcf5e235c566875edc2ee2026 ow-S0_20090213.15
5b2a483ccab444e6d3ade78edb16a539 ow-S0_20090213.16
0fddf69cdb3193a07d9e3f118b7d7ab6 ow-S0_20090213.17
ef1219e34673e90a3aa3dadef3ef0b54 ow-S0_20090213.18
8686bed8e77ba1d3d48521f7060d7203 ow-S0_20090213.19
db0b7ec66d3a71954f43dc811169fd40 ow-S0_20090213.20
ffad367b75bf34ce1af65dce6d2bf16c ow-S0_20090213.21
554808e653147605df392d4d76b19912 ow-S0_20090213.22
29f6f6f2e3f01a3105b742b6dd9bf6ba ow-S0_20090213.23
ca68434fd4ce22a8b7b6b2923fa913e5 ow-S0_20090214.00
ea887a738936eca8100542dd6f233362 ow-S0_20090214.01
7b87814680ba425adaa6f79ddb26f73a ow-S0_20090214.02
33ae794754a32bbf2db643e9eca0a17e ow-S0_20090214.03
2882b4098d0d95363bb136f0b7d3f52b ow-S0_20090214.04
8bcf7481dcd6446b4fd998db7885e055 ow-S0_20090214.05
67a530ac9ed04572c14b2bffec4f9f81 ow-S0_20090214.06
4b48c4225cfd76aa4694b0f87a08d158 ow-S0_20090214.07
fec050c3e70d9927532c59b095892a60 ow-S0_20090214.08
a60a78523ab7035ff8eee4afce39f3b3 ow-S0_20090214.09
969fa9487bc017875095272bef383f15 ow-S0_20090214.10
cdd7d39e38597e7bd1b500b89bf12ce9 ow-S0_20090214.11
6a1db4322505280b57e232ee5b8418a0 ow-S0_20090214.12
58bb0024d67b0d5408ec549bbb6e9aeb ow-S0_20090214.13
0d5c70ea854ec3d89ee8d69d9cc763b8 ow-S0_20090214.14
c0a27b6076d9281f9143789cabca09cc ow-S0_20090214.15
f917de62cfa3e0dec1be418c5d422a82 ow-S0_20090214.16
10c922109bf215431114b7c47eb98d9d ow-S0_20090214.17
90e73d49745e29ad05b3f0dc9ada11ce ow-S0_20090214.18
9bd50e1e3278d25db6fcd374e514c0e4 ow-S0_20090214.19
21ff60734cbb5085dc929ec897f9ea01 ow-S0_20090214.20
f18b633c03ca12fb877f85b0666cb3ae ow-S0_20090214.21
1c1490fa40e028b48539bf7c4f73f0b6 ow-S0_20090214.22
8700fe19de6925bb8fb44113d51f236d ow-S0_20090214.23
