/*
** Copyright (C) 2007-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  rwsilk2ipfix.c
**
**  SiLK to IPFIX translation application
**
**  Authors: Brian Trammell <bht@cert.org>
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: rwsilk2ipfix.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skipfix.h>
#include <silk/rwrec.h>
#include <silk/utils.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* where to write --help output */
#define USAGE_FH stdout

/* where to write --print-stat output */
#define STATS_FH stderr


/* LOCAL VARIABLE DEFINITIONS */

/* index into argv for looping over input */
static int arg_index;

/* name of the IPFIX output file; use stdout if value is NULL */
static const char *ipfix_output = NULL;

/* whether to print statistics */
static int print_statistics = 0;


/* OPTIONS SETUP */

typedef enum {
    OPT_IPFIX_OUTPUT,
    OPT_PRINT_STATISTICS
} appOptionsEnum;

static struct option appOptions[] = {
    {"ipfix-output",            REQUIRED_ARG, 0, OPT_IPFIX_OUTPUT},
    {"print-statistics",        NO_ARG,       0, OPT_PRINT_STATISTICS},
    {0,0,0,0}                   /* sentinel entry */
};

static const char *appHelp[] = {
    ("Write IPFIX records to the specified path. Def. stdout"),
    ("Print the count of processed records. Def. No"),
    (char *)NULL
};



/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                             \
    ("[SWITCHES] [SILK_FILES]\n"                                              \
     "\tReads SiLK Flow records from files named on the command line or\n"    \
     "\tfrom the standard input, converts them to an IPFIX format, and\n"     \
     "\twrites the IPFIX records to the named file or the standard output.\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse the options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options parsing should print error */
        skAppUsage();           /* never returns */
    }

    /* arg_index is looking at first file name to process */
    if (arg_index == argc) {
        if (FILEIsATty(stdin)) {
            skAppPrintErr("No input files on command line and"
                          " stdin is connected to a terminal");
            skAppUsage();       /* never returns */
        }
    }

    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    return;  /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    static int saw_ipfix_output = 0;

    switch ((appOptionsEnum)opt_index) {
      case OPT_IPFIX_OUTPUT:
        if (saw_ipfix_output) {
            skAppPrintErr("--%s specified multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        if (skFileExists(opt_arg)) {
            skAppPrintErr("IPFIX output file '%s' already exists", opt_arg);
            return 1;
        }
        saw_ipfix_output = 1;
        /* store the name unless "stdout" or "-" is specified, which
         * is the default */
        if ((0 != strcmp(opt_arg, "stdout")) || (0 == strcmp(opt_arg, "-"))) {
            ipfix_output = opt_arg;
        }
        break;

      case OPT_PRINT_STATISTICS:
        print_statistics = 1;
        break;
    }

    return 0;  /* OK */
}


/*
 *  rwios = appNextInput(argc, argv);
 *
 *    Open and return the next input file from the command line or the
 *    standard input if no files were given on the command line.
 */
static skstream_t *appNextInput(int argc, char **argv)
{
    static int initialized = 0;
    skstream_t *rwios = NULL;
    const char *fname = NULL;
    int rv;

    if (arg_index < argc) {
        /* get current file and prepare to get next */
        fname = argv[arg_index];
        ++arg_index;
    } else {
        if (initialized) {
            /* no more input */
            return NULL;
        }
        /* input is from stdin */
        fname = "stdin";
    }

    initialized = 1;

    /* create rwios and open file */
    rv = skStreamOpenSilkFlow(&rwios, fname, SK_IO_READ);
    if (rv) {
        skStreamPrintLastErr(rwios, rv, NULL);
        skStreamDestroy(&rwios);
    }

    return rwios;
}


int main(int argc, char **argv)
{
    FILE *ipfix_stream = stdout;
    fBuf_t *fbuf = NULL;
    GError *err = NULL;
    skstream_t *rwios_in = NULL;
    rwRec rwrec;
    uint64_t rec_count = 0;
    int rv;

    appSetup(argc, argv);                       /* never returns on error */

    /* open output file if provided */
    if (!ipfix_output) {
        ipfix_output = "stdout";
        ipfix_stream = stdout;
    } else {
        ipfix_stream = fopen(ipfix_output, "wb");
        if (ipfix_stream == NULL) {
            skAppPrintSyserror("Could not open IPFIX output file '%s'",
                               ipfix_output);
            exit(EXIT_FAILURE);
        }
    }

    if (FILEIsATty(ipfix_stream)) {
        skAppPrintErr("Will not binary data to the terminal '%s'",
                      ipfix_output);
        exit(EXIT_FAILURE);
    }

    /* Create IPFIX buffer */
    fbuf = skiCreateWriteBufferForFP(ipfix_stream, 0, &err);
    if (!fbuf) {
        g_error("Could not open '%s' for IPFIX: %s",
                ipfix_output, err->message);
        exit(EXIT_FAILURE);
    }

    /* For each input, process each record */
    while (NULL != (rwios_in = appNextInput(argc, argv))) {
        while ((rv = skStreamReadRecord(rwios_in, &rwrec)) == SKSTREAM_OK) {
            /* process record */
            ++rec_count;
            if (!skiRwAppendRecord(fbuf, &rwrec, &err)) {
                skAppPrintErr("Could not write IPFIX record: %s", err->message);
                exit(EXIT_FAILURE);
            }
        }
        if (rv != SKSTREAM_ERR_EOF) {
            skStreamPrintLastErr(rwios_in, rv, &skAppPrintErr);
        }
        skStreamDestroy(&rwios_in);
    }

    /* finalize the output */
    if (!fBufEmit(fbuf, &err)) {
        skAppPrintErr("Could not write final IPFIX message: %s", err->message);
        exit(EXIT_FAILURE);
    }
    fbExporterClose(fBufGetExporter(fbuf));

    if (fbuf) {
        fBufFree(fbuf);
    }
    if (stdout == ipfix_stream) {
        if (fflush(ipfix_stream)) {
            skAppPrintSyserror("Error writing IPFIX to stdout");
        }
    } else if (fclose(ipfix_stream)) {
        skAppPrintSyserror("Error closing IPFIX stream '%s'", ipfix_output);
    }

    /* print record count */
    if (print_statistics) {
        fprintf(STATS_FH, ("%s: Wrote %" PRIu64 " IPFIX records to '%s'\n"),
                skAppName(), rec_count, ipfix_output);
    }

    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
