/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
 * Rwset.c
 *
 * Michael Collins
 * May 6th
 *
 * rwset is an application which takes filter data and generates a
 * tree (rwset) of ip addresses which come out of a filter file.  This
 * tree can then be used to generate aggregate properties; rwsets will
 * be used for filtering large sets of ip addresses, aggregate properties
 * per element can be counted, &c.
 */

#include <silk/silk.h>

RCSIDENT("$SiLK: rwset.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/skipset.h>
#include <silk/skstream.h>
#include <silk/utils.h>
#include <silk/iochecks.h>
#include <silk/skstream.h>
#include <silk/sksite.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* where to write output from --help */
#define USAGE_FH stdout

/* Where to write filenames if --print-file specified */
#define PRINT_FILENAMES_FH  stderr

/* What to do when malloc() fails */
#define EXIT_NO_MEMORY                                               \
    do {                                                             \
        skAppPrintErr("Out of memory at %s:%d", __FILE__, __LINE__); \
        exit(EXIT_FAILURE);                                          \
    } while(0)

/* number of set files that can be created (sip, dip, nhip) */
#define SET_FILE_TYPES 3

/* structure to hold a set and the stream to write it to */
typedef struct setfile_st {
    skipset_t  *ipset;
    skstream_t *stream;
} setfile_t;


/* LOCAL VARIABLES */

/* streams to store the IPsets in */
static setfile_t sets[SET_FILE_TYPES];

/* the compression method to use when writing the files.
 * sksiteCompmethodOptionsRegister() will set this to the default or
 * to the value the user specifies. */
static sk_compmethod_t comp_method;

/* whether to print the names of files as they are opened */
static int print_filenames = 0;

/* for input processing */
static iochecksInfoStruct_t *ioISP;



/* OPTIONS SETUP */

typedef enum {
    OPT_SIP_FILE,
    OPT_DIP_FILE,
    OPT_NHIP_FILE,
    OPT_PRINT_FILENAMES,
    OPT_COPY_INPUT
} appOptionsEnum;

static struct option appOptions[] = {
    {"sip-file",            REQUIRED_ARG, 0, OPT_SIP_FILE},
    {"dip-file",            REQUIRED_ARG, 0, OPT_DIP_FILE},
    {"nhip-file",           REQUIRED_ARG, 0, OPT_NHIP_FILE},
    {"print-filenames",     NO_ARG,       0, OPT_PRINT_FILENAMES},
    {"copy-input",          REQUIRED_ARG, 0, OPT_COPY_INPUT},
    {0,0,0,0}               /* sentinel entry */
};

static const char *appHelp[] = {
    ("Create an IPset from source addresses\n"
     "\tand write it to the named file (file must not exist)"),
    ("Create an IPset from destination addresses\n"
     "\tand write it to the named file (file must not exist)"),
    ("Create an IPset from next-hop addresses\n"
     "\tand write it to the named file (file must not exist)"),
    "Print names of input files as they are opened. Def. No",
    "Copy all input SiLK Flows to given pipe or file. Def. No",
    (char *)NULL
};



/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                             \
    ("{--sip-file=FILE | --dip-file=FILE | --nhip-file=FILE} [FILES]\n"       \
     "\tRead SiLK Flow records and generate binary IPset file(s).  When no\n" \
     "\tfiles are given on command line, flows are read from STDIN.\n")

    FILE *fh = USAGE_FH;

    skAppStandardUsage(fh, USAGE_MSG, appOptions, appHelp);
    skOptionsNotesUsage(fh);
    sksiteCompmethodOptionsUsage(fh);
    sksiteOptionsUsage(fh);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;
    int i;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    for (i = 0; i < SET_FILE_TYPES; ++i) {
        if (sets[i].ipset) {
            skIPSetDestroy(&sets[i].ipset);
            sets[i].ipset = NULL;
        }
        if (sets[i].stream) {
            skStreamDestroy(&sets[i].stream);
        }
    }

    iochecksTeardown(ioISP);
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    sk_file_header_t *out_hdr;
    int i;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    memset(sets, 0, sizeof(sets));

    /* initialize input checker */
    ioISP = iochecksSetup(0, 0, argc, argv);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || skOptionsNotesRegister(NULL)
        || sksiteCompmethodOptionsRegister(&comp_method)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* Parse options */
    ioISP->firstFile = skOptionsParse(argc, argv);
    if (ioISP->firstFile < 0) {
        skAppUsage();           /* never returns */
    }

    /* try to load site config file; if it fails, we will not be able
     * to resolve flowtype and sensor from input file names */
    sksiteConfigure(0);

    /* Make certain the user has requested some output. */
    for (i = 0; i < SET_FILE_TYPES; ++i) {
        if (sets[i].stream) {
            break;
        }
    }
    if (SET_FILE_TYPES == i) {
        skAppPrintErr("No output specified; must specify file(s) to create");
        skAppUsage();
    }


    /* Use STDIN as an input stream if it is not a TTY; make certain
     * we have some input and we are either reading from STDIN or
     * using files listed the command line, but not both. */
    if (iochecksAcceptFromStdin(ioISP) || iochecksInputs(ioISP, 0)) {
        skAppUsage();
    }

    /* All arguments look good.  Set the atexit() handler */
    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        appTeardown();
        exit(EXIT_FAILURE);
    }

    /* For each output file, set the compression method, add the
     * notes (if given), and open the file */
    for (i = 0; i < SET_FILE_TYPES; ++i) {
        if (sets[i].stream) {
            out_hdr = skStreamGetSilkHeader(sets[i].stream);
            rv = skHeaderSetCompressionMethod(out_hdr, comp_method);
            if (rv) {
                skStreamPrintLastErr(sets[i].stream, rv, &skAppPrintErr);
                exit(EXIT_FAILURE);
            }

            rv = skHeaderAddInvocation(out_hdr, 1, argc, argv);
            if (rv) {
                skStreamPrintLastErr(sets[i].stream, rv, &skAppPrintErr);
                exit(EXIT_FAILURE);
            }

            rv = skOptionsNotesAddToStream(sets[i].stream);
            if (rv) {
                skStreamPrintLastErr(sets[i].stream, rv, &skAppPrintErr);
                exit(EXIT_FAILURE);
            }

            rv = skStreamOpen(sets[i].stream);
            if (rv) {
                skStreamPrintLastErr(sets[i].stream, rv, &skAppPrintErr);
                exit(EXIT_FAILURE);
            }
        }
    }

    skOptionsNotesTeardown();

    /* Create IPsets */
    for (i = 0; i < SET_FILE_TYPES; ++i) {
        if (sets[i].stream) {
            if (skIPSetCreate(&(sets[i].ipset), 0)) {
                EXIT_NO_MEMORY;
            }
        }
    }

    /* looks good, open the --copy-input destination */
    if (iochecksOpenCopyDest(ioISP)) {
        exit(EXIT_FAILURE);
    }

    return;                       /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int  appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    static int stdout_used = 0;
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_SIP_FILE:
      case OPT_DIP_FILE:
      case OPT_NHIP_FILE:
        if (sets[opt_index].stream) {
            skAppPrintErr("The --%s switch was given multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        if (0 == strcmp(opt_arg, "stdout")) {
            if (stdout_used) {
                skAppPrintErr("Multiple outputs are trying to use stdout");
                return 1;
            }
            stdout_used = 1;
        }
        if ((rv = skStreamCreate(&(sets[opt_index].stream), SK_IO_WRITE,
                                 SK_CONTENT_SILK))
            || (rv = skStreamBind(sets[opt_index].stream, opt_arg)))
        {
            skStreamPrintLastErr(sets[opt_index].stream, rv, &skAppPrintErr);
            skStreamDestroy(&(sets[opt_index].stream));
            return 1;
        }
        break;

      case OPT_PRINT_FILENAMES:
        print_filenames = 1;
        break;

      case OPT_COPY_INPUT:
        if (0 == strcmp(opt_arg, "stdout")) {
            if (stdout_used) {
                skAppPrintErr("Multiple outputs are trying to use stdout");
                return 1;
            }
            stdout_used = 1;
        }
        if (iochecksAllDestinations(ioISP, opt_arg)) {
            return 1;
        }
        break;
    }

    return 0;                     /* OK */
}


/*
 *  status = rwsetProcessFile(filename);
 *
 *    Read the records from 'filename' and add them to the appropriate
 *    IPSet files.  Return 0 on success, or -1 on failure.
 */
static int rwsetProcessFile(const char *filename)
{
    static rwRec rwrec;
    static skipaddr_t ip;
    skstream_t *rwIOS;
    int rv = SKIPSET_OK;
    int rw_rv;
    int bad_set = 0;

    /* Open file */
    rw_rv = skStreamOpenSilkFlow(&rwIOS, filename, SK_IO_READ);
    if (rw_rv) {
        skStreamPrintLastErr(rwIOS, rw_rv, &skAppPrintErr);
        skStreamDestroy(&rwIOS);
        return -1;
    }
    (void)skStreamSetCopyInput(rwIOS, ioISP->inputCopyFD);
    if (print_filenames) {
        fprintf(PRINT_FILENAMES_FH, "%s\n", skStreamGetPathname(rwIOS));
    }
    /* ignore IPv6 flows */
    skStreamSetIPv6Policy(rwIOS, SK_IPV6POLICY_ASV4);

    /* Read in records */
    while ((rw_rv = skStreamReadRecord(rwIOS, &rwrec)) == SKSTREAM_OK) {
        if (sets[OPT_SIP_FILE].ipset) {
            rwRecMemGetSIP(&rwrec, &ip);
            rv = skIPSetInsertAddress(sets[OPT_SIP_FILE].ipset, &ip, 0);
            if (rv) {
                bad_set = OPT_SIP_FILE;
#if SK_ENABLE_IPV6
                if (SKIPSET_ERR_IPV6 == rv) {
                    rv = skIPSetConvert(sets[OPT_SIP_FILE].ipset, 6);
                    if (rv) {
                        break;
                    }
                    rv = skIPSetInsertAddress(sets[OPT_SIP_FILE].ipset, &ip,0);
                }
#endif  /* SK_ENABLE_IPV6 */
                if (rv) {
                    break;
                }
            }
        }
        if (sets[OPT_DIP_FILE].ipset) {
            rwRecMemGetDIP(&rwrec, &ip);
            rv = skIPSetInsertAddress(sets[OPT_DIP_FILE].ipset, &ip, 0);
            if (rv) {
                bad_set = OPT_DIP_FILE;
#if SK_ENABLE_IPV6
                if (SKIPSET_ERR_IPV6 == rv) {
                    rv = skIPSetConvert(sets[OPT_DIP_FILE].ipset, 6);
                    if (rv) {
                        break;
                    }
                    rv = skIPSetInsertAddress(sets[OPT_DIP_FILE].ipset, &ip,0);
                }
#endif  /* SK_ENABLE_IPV6 */
                if (rv) {
                    break;
                }
            }
        }
        if (sets[OPT_NHIP_FILE].ipset) {
            rwRecMemGetNhIP(&rwrec, &ip);
            rv = skIPSetInsertAddress(sets[OPT_NHIP_FILE].ipset, &ip, 0);
            if (rv) {
                bad_set = OPT_NHIP_FILE;
#if SK_ENABLE_IPV6
                if (SKIPSET_ERR_IPV6 == rv) {
                    rv = skIPSetConvert(sets[OPT_NHIP_FILE].ipset, 6);
                    if (rv) {
                        break;
                    }
                    rv = skIPSetInsertAddress(sets[OPT_NHIP_FILE].ipset,&ip,0);
                }
#endif  /* SK_ENABLE_IPV6 */
                if (rv) {
                    break;
                }
            }
        }
    }
    if (rw_rv != SKSTREAM_ERR_EOF && rw_rv != SKSTREAM_OK) {
        skStreamPrintLastErr(rwIOS, rw_rv, &skAppPrintErr);
    }

    if (rv) {
        skAppPrintErr("Error adding IP to %s: %s",
                      appOptions[bad_set].name, skIPSetStrerror(rv));
        return -1;
    }

    skStreamDestroy(&rwIOS);
    return 0;
}


int main(int argc, char **argv)
{
    int had_err = 0;
    int i;
    int counter;

    appSetup(argc, argv);                 /* never returns */

    /* Process files */
    for (counter = ioISP->firstFile; counter < ioISP->fileCount ; counter++) {
        if (rwsetProcessFile(ioISP->fnArray[counter])) {
            exit(EXIT_FAILURE);
        }
    }

    /* Generate output */
    for (i = 0; i < SET_FILE_TYPES; ++i) {
        if (sets[i].ipset) {
            if (sets[i].stream) {
                skIPSetClean(sets[i].ipset);
                if ((0 != skIPSetWrite(sets[i].ipset, sets[i].stream))
                    || (0 != skStreamClose(sets[i].stream)))
                {
                    skAppPrintErr("Error writing IPset to '%s'",
                                  skStreamGetPathname(sets[i].stream));
                    had_err = 1;
                }
                skStreamDestroy(&sets[i].stream);
            }

            skIPSetDestroy(&sets[i].ipset);
            sets[i].ipset = NULL;
        }
    }

    /* done */
    appTeardown();

    return had_err;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
