/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/
#ifndef _RWSTATS_H
#define _RWSTATS_H

#include <silk/silk.h>

RCSIDENTVAR(rcsID_RWSTATS_H, "$SiLK: rwstats.h 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");


/*
**  rwstats.h
**
**    Header file for the rwstats application.  See rwstats.c for a
**    full explanation.
**
*/

#include <silk/utils.h>
#include <silk/hashlib.h>
#include <silk/rwascii.h>
#include <silk/sksite.h>
#include <silk/skstream.h>
#include <silk/skvector.h>
#include <silk/iptree.h>
#include <silk/skplugin.h>
#include <silk/sktempfile.h>

/* use TRACEMSG_LEVEL as our tracing variable */
#define TRACEMSG(msg) TRACEMSG_TO_TRACEMSGLVL(1, msg)
#include <silk/sktracemsg.h>


/* TYPEDEFS AND DEFINES */

/* For the compound key version of rwstats, the maximum number of bytes
 * we allow in the key.  An arbitrary number that should be big enough
 * for almost all cases. */
#define MAX_KEY_BYTE_WIDTH 256

/* For ASCII output, maximum size of a single value column. Must be
 * big enough to hold printed uint64_t, sktimestamp(), class, type,
 * sensor, or arbitrary pmap value. */
#define VALUE_OUT_BUF_SIZE  256

/* Max size of a value in the hash table.  Must be able to hold the
 * values: bytes(8), packets(4), flows(4), sTime(4), and eTime(4), and
 * any plugin values.  */
#define MAX_VALUE_BYTE_WIDTH 256

/* Max size of a key/value pair in the hash table.  Must be able to
 * hold the max_key, distinct sIP(4) & dIP(4), and the max_value byte
 * width. */
#define MAX_HASH_NODE_BYTES (MAX_KEY_BYTE_WIDTH + 8 + MAX_VALUE_BYTE_WIDTH)

/* Value to use as exit code when we run out of memory */
#define RWSTATS_NO_MEMORY_EXIT_CODE 16

/* default sTime bin size to use when --bin-time is requested */
#define DEFAULT_TIME_BIN  60

/* whether we need to use memcpy() or can access unaligned memory */
#ifndef RWSTATS_USE_MEMCPY
#ifdef SK_HAVE_ALIGNED_ACCESS_REQUIRED
#define RWSTATS_USE_MEMCPY 1
#else
#define RWSTATS_USE_MEMCPY 0
#endif
#endif

/* symbol names for whether this is a top-N or bottom-N */
typedef enum {
    RWSTATS_DIR_TOP, RWSTATS_DIR_BOTTOM
} rwstats_direction_t;

/* what type of cutoff to use; keep these in same order as appOptionsEnum */
typedef enum {
    /* specify the N for a Top-N or Bottom-N */
    RWSTATS_COUNT = 0,
    /* output bins whose value is at-least/no-more-than this value */
    RWSTATS_THRESHOLD = 1,
    /* output bins whose value relative to the total across all bins
     * is at-least/no-more-than this percentage */
    RWSTATS_PERCENTAGE = 2
} rwstats_limit_type_t;

/* used to convert a percentage or threshold limit to a number of bins */
typedef struct rwstats_limit_st {
    /* three values that correspond to rwstats_limit_type_t */
    uint64_t                value[3];
    /* number of entries in the hash table */
    uint64_t                entries;
    rwstats_limit_type_t    type;
    unsigned                seen :1;
} rwstats_limit_t;

/* function that gets the local part of the key from an rwrec */
typedef void (*rec2key_fn_t)(const rwRec *rec, void *key_pos);

/* function that puts the key into the rwrec */
typedef void (*key2rec_fn_t)(rwRec *rec, const void *key_pos);

/* struct to hold the columns/fields that make up our key for this run */
typedef struct key_field_st {
    /* a buffer to hold the textual value for this field when printing
     * the results */
    char              kf_textbuf[VALUE_OUT_BUF_SIZE];
    /* the functions to call to fill in this field.  May be NULL.
     * When these are set, they are used to convert a member of an
     * rwRec into a binary value, and to convert the binary value back
     * to members of the rwRec. */
    rec2key_fn_t      kf_r2k;
    key2rec_fn_t      kf_k2r;
    /* If non-null, the plugin field handle to use for this field */
    skplugin_field_t *kf_field_handle;
    /* the ID of the column, from parseFields(), that makes up this
     * key_field */
    uint32_t          kf_id;
    /* the byte-offset where this field begins in the binary key used
     * for binning. */
    size_t            kf_offset;
    /* the byte-width of this field in the binary key */
    size_t            kf_width;
} key_field_t;

/*
 *  The built-in value fields to count w.r.t. the key.  Keep this in
 *  sync with the potential_values[] array in rwstatssetup.c.
 */
typedef enum {
    VALUE_BYTES = 0,
    VALUE_PACKETS,
    VALUE_FLOWS,
    DISTINCT_SIP,
    DISTINCT_DIP,
    VALUE_PLUGIN
} value_types_t;


/* struct to hold information about the value fields */
typedef struct value_field_st {
    /* the title of this field */
    const char       *vf_title;
    /* the byte-width of this field */
    size_t            vf_byte_width;
    /* the byte-offset for each value field */
    size_t            vf_offset;
    /* the text width of the field for columnar output */
    int               vf_text_len;
    /* the id for this column */
    value_types_t     vf_id;
    /* the plugin information for this field (if any) */
    skplugin_field_t *vf_plugin_field;
} value_field_t;

/* The number of distinct fields that are supported */
#define NUM_DISTINCTS 2

/* struct to hold information about distinct fields */
typedef struct distinct_field_st {
    /* function to call to pack this field into key */
    rec2key_fn_t        df_r2k;
    /* the byte-width of this field when used in the key */
    size_t              df_byte_width;
    /* the byte-offset for this field in the key */
    size_t              df_offset;
    /* the id for this column */
    value_types_t       df_id;
    /* the location of this field in the value_fields[] */
    int                 df_val_field_pos;
} distinct_field_t;

/* group input/output variables into a single struct */
typedef struct output_st {
    FILE                 *fp;
    const char           *pathname;
    int                   ispipe;
} app_output_t;

/* flags set by user options */
typedef struct app_flags_st {
    unsigned presorted_input    :1;      /* Assume input is sorted */
    unsigned no_percents        :1;      /* Whether to include the % cols */
    unsigned print_filenames    :1;
    unsigned no_columns         :1;
    unsigned no_titles          :1;
    unsigned no_final_delimiter :1;
    unsigned integer_sensors    :1;
    unsigned integer_tcp_flags  :1;
    unsigned integer_ips        :1;
    unsigned zero_pad_ips       :1;
} app_flags_t;

/* names for the columns */
enum width_type {
    WIDTH_KEY, WIDTH_VAL, WIDTH_INTVL, WIDTH_PCT
};

#define RWSTATS_COLUMN_WIDTH_COUNT 4

/* Option indentifiers.  Keep in sync with appOptions.  Need option
 * identifiers in the header so legacy options can invoke them. */
typedef enum {
    OPT_OVERALL_STATS, OPT_DETAIL_PROTO_STATS,

    OPT_FIELDS, OPT_VALUES, OPT_PLUGIN,

    /* keep these in same order as stat_stat_type_t */
    OPT_COUNT, OPT_THRESHOLD, OPT_PERCENTAGE,

    OPT_TOP, OPT_BOTTOM,

    OPT_PRESORTED_INPUT,
    OPT_NO_PERCENTS,
    OPT_BIN_TIME,
    OPT_EPOCH_TIME,
    OPT_INTEGER_IPS,
    OPT_ZERO_PAD_IPS,
    OPT_INTEGER_SENSORS,
    OPT_INTEGER_TCP_FLAGS,
    OPT_NO_TITLES,
    OPT_NO_COLUMNS,
    OPT_COLUMN_SEPARATOR,
    OPT_NO_FINAL_DELIMITER,
    OPT_DELIMITED,
    OPT_PRINT_FILENAMES,
    OPT_COPY_INPUT,
    OPT_OUTPUT_PATH,
    OPT_PAGER,
    OPT_LEGACY_TIMESTAMPS,
    OPT_LEGACY_HELP
} appOptionsEnum;

/* used to handle legacy switches */
typedef struct rwstats_legacy_st {
    const char *fields;
    const char *values;
} rwstats_legacy_t;


/* VARIABLE DECLARATIONSS */

/* non-zero when --overall-stats or --detail-proto-stats is given */
extern int proto_stats;

/* whether this is a top-n or bottom-n */
extern rwstats_direction_t direction;

/* hold the value of the N for top-N,bottom-N */
extern rwstats_limit_t limit;

/* the columns/fields that make up our key for this run */
extern key_field_t *key_fields;

/* the number of columns/fields that make up the key */
extern size_t key_num_fields;

/* the total byte count of the key */
extern size_t key_byte_width;

/* The value fields that we are computing */
extern value_field_t *value_fields;

/* number of value fields the user has requested */
extern size_t value_num_fields;

/* byte count for all value_fields that are being computed, except
 * that it does not include the byte width(s) for distinct fields. */
extern size_t value_byte_width;

/* the initial values as a byte-encoded array */
extern uint8_t *value_initial_vals;

/* Information about each distinct field. */
extern distinct_field_t distinct_fields[NUM_DISTINCTS];

/* number of distinct fields the user has requested */
extern size_t distinct_num_fields;

/* the byte length of the key used by the hash table, which is the
 * length of the "key" and any "distinct" fields  */
extern size_t hash_key_bytes;

/* the byte length of the value used by the hash table, which has a
 * minimum length of 1  */
extern size_t hash_value_bytes;

/* the byte length of the key-value pair in the hash table, which is
 * comprised of the key and the value */
extern size_t hash_node_bytes;

/* the byte length of the value used by the heap. this is
 * value_byte_width plus space for distinct fields */
extern size_t heap_value_bytes;

/* the byte length of the key-value pair in the heap */
extern size_t heap_node_bytes;

/* to convert the key fields (as an rwRec) to ascii */
extern rwAsciiStream_t *ascii_str;

/* the output */
extern app_output_t output;

/* flags set by the user options */
extern app_flags_t app_flags;

/* output column widths.  mapped to width_type */
extern int width[RWSTATS_COLUMN_WIDTH_COUNT];

/* delimiter between output columns */
extern char delimiter;

/* the final delimiter on each line */
extern char final_delim[];

/* how to print timestamps */
extern uint32_t time_flags;

/* how to handle IPv6 flows */
extern sk_ipv6policy_t ipv6_policy;

/* CIDR block mask for sIPs and dIPs.  If 0, use all bits; otherwise,
 * the IP address should be bitwised ANDed with this value. */
extern uint32_t cidr_sip;
extern uint32_t cidr_dip;


/* FUNCTION DECLARATIONS */

/* rwstatssetup.c */

void appSetup(int argc, char **argv);
void appTeardown(void);
void appExit(int status);
int  appNextInput(skstream_t **rwios);
int  appOptionsHandler(
    clientData  cData,
    int         opt_index,
    char       *opt_arg);


/* rwstatsproto.c: Functions for detailed protocol statistics */

int  protoStatsParse(const char *arg);
int  protoStatsMain(void);


/* from rwstatslegacy.c */

int  legacyOptionsSetup(clientData cData);
void legacyOptionsUsage(FILE *fh);

#endif /* _RWSTATS_H */

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
