/*
** Copyright (C) 2001-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  rwstatsutils.c
**
**  utility functions for the rwstats application.  See rwstats.c for
**  a full explanation.
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwstatslegacy.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include "rwstats.h"


/* OPTIONS SETUP */

typedef enum legacyOptions_en {
    LEGOPT_SIP, LEGOPT_DIP, LEGOPT_SPORT, LEGOPT_DPORT, LEGOPT_PROTOCOL,
    LEGOPT_ICMP,

    LEGOPT_FLOWS, LEGOPT_PACKETS, LEGOPT_BYTES
} legacyOptionsEnum;

static struct option legacyOptions[] = {
    {"sip",                    OPTIONAL_ARG, 0, LEGOPT_SIP},
    {"dip",                    OPTIONAL_ARG, 0, LEGOPT_DIP},
    {"sport",                  NO_ARG,       0, LEGOPT_SPORT},
    {"dport",                  NO_ARG,       0, LEGOPT_DPORT},
    {"protocol",               NO_ARG,       0, LEGOPT_PROTOCOL},
    {"icmp",                   NO_ARG,       0, LEGOPT_ICMP},

    {"flows",                  NO_ARG,       0, LEGOPT_FLOWS},
    {"packets",                NO_ARG,       0, LEGOPT_PACKETS},
    {"bytes",                  NO_ARG,       0, LEGOPT_BYTES},

    {0,0,0,0}                  /* sentinel entry */
};

static const char *legacyHelp[] = {
    ("Use: --fields=sip\n"
     "\tUse the source address as (part of) the key"),
    ("Use: --fields=dip\n"
     "\tUse the destination address as (part of) the key"),
    ("Use: --fields=sport\n"
     "\tUse the source port as (part of) the key"),
    ("Use: --fields=dport\n"
     "\tUse the destination port as (part of) the key"),
    ("Use: --fields=proto\n"
     "\tUse the protocol as the key"),
    ("Use: --fields=icmp\n"
     "\tUse the ICMP type and code as the key"),

    ("Use: --values=flows\n"
     "\tUse the flow count as the value"),
    ("Use: --values=packets\n"
     "\tUse the packet count as the value"),
    ("Use: --values=bytes\n"
     "\tUse the byte count as the value"),

    (char *)NULL
};


/* very very old options that should really just go away */
typedef enum ancientOptions_en {
    LEGOPT_SIP_TOP_N, LEGOPT_SIP_TOP_THRESHOLD, LEGOPT_SIP_TOP_PCT,
    LEGOPT_SIP_BTM_N, LEGOPT_SIP_BTM_THRESHOLD, LEGOPT_SIP_BTM_PCT,

    LEGOPT_DIP_TOP_N, LEGOPT_DIP_TOP_THRESHOLD, LEGOPT_DIP_TOP_PCT,
    LEGOPT_DIP_BTM_N, LEGOPT_DIP_BTM_THRESHOLD, LEGOPT_DIP_BTM_PCT,

    LEGOPT_PAIR_TOP_N, LEGOPT_PAIR_TOP_THRESHOLD, LEGOPT_PAIR_TOP_PCT,
    LEGOPT_PAIR_BTM_N, LEGOPT_PAIR_BTM_THRESHOLD, LEGOPT_PAIR_BTM_PCT,

    LEGOPT_SPORT_TOP_N, LEGOPT_SPORT_TOP_THRESHOLD, LEGOPT_SPORT_TOP_PCT,
    LEGOPT_SPORT_BTM_N, LEGOPT_SPORT_BTM_THRESHOLD, LEGOPT_SPORT_BTM_PCT,

    LEGOPT_DPORT_TOP_N, LEGOPT_DPORT_TOP_THRESHOLD, LEGOPT_DPORT_TOP_PCT,
    LEGOPT_DPORT_BTM_N, LEGOPT_DPORT_BTM_THRESHOLD, LEGOPT_DPORT_BTM_PCT,

    LEGOPT_PORTPAIR_TOP_N, LEGOPT_PORTPAIR_TOP_THRESHOLD,
    LEGOPT_PORTPAIR_TOP_PCT, LEGOPT_PORTPAIR_BTM_N,
    LEGOPT_PORTPAIR_BTM_THRESHOLD, LEGOPT_PORTPAIR_BTM_PCT,

    LEGOPT_PROTO_TOP_N, LEGOPT_PROTO_TOP_THRESHOLD, LEGOPT_PROTO_TOP_PCT,
    LEGOPT_PROTO_BTM_N, LEGOPT_PROTO_BTM_THRESHOLD, LEGOPT_PROTO_BTM_PCT,

    LEGOPT_CIDR_SRC, LEGOPT_CIDR_DEST
} ancientOptionsEnum;

static struct option ancientOptions[] = {
    {"sip-topn",               REQUIRED_ARG, 0, LEGOPT_SIP_TOP_N},
    {"sip-top-threshold",      REQUIRED_ARG, 0, LEGOPT_SIP_TOP_THRESHOLD},
    {"sip-top-pct",            REQUIRED_ARG, 0, LEGOPT_SIP_TOP_PCT},
    {"sip-btmn",               REQUIRED_ARG, 0, LEGOPT_SIP_BTM_N},
    {"sip-btm-threshold",      REQUIRED_ARG, 0, LEGOPT_SIP_BTM_THRESHOLD},
    {"sip-btm-pct",            REQUIRED_ARG, 0, LEGOPT_SIP_BTM_PCT},

    {"dip-topn",               REQUIRED_ARG, 0, LEGOPT_DIP_TOP_N},
    {"dip-top-threshold",      REQUIRED_ARG, 0, LEGOPT_DIP_TOP_THRESHOLD},
    {"dip-top-pct",            REQUIRED_ARG, 0, LEGOPT_DIP_TOP_PCT},
    {"dip-btmn",               REQUIRED_ARG, 0, LEGOPT_DIP_BTM_N},
    {"dip-btm-threshold",      REQUIRED_ARG, 0, LEGOPT_DIP_BTM_THRESHOLD},
    {"dip-btm-pct",            REQUIRED_ARG, 0, LEGOPT_DIP_BTM_PCT},

    {"pair-topn",              REQUIRED_ARG, 0, LEGOPT_PAIR_TOP_N},
    {"pair-top-threshold",     REQUIRED_ARG, 0, LEGOPT_PAIR_TOP_THRESHOLD},
    {"pair-top-pct",           REQUIRED_ARG, 0, LEGOPT_PAIR_TOP_PCT},
    {"pair-btmn",              REQUIRED_ARG, 0, LEGOPT_PAIR_BTM_N},
    {"pair-btm-threshold",     REQUIRED_ARG, 0, LEGOPT_PAIR_BTM_THRESHOLD},
    {"pair-btm-pct",           REQUIRED_ARG, 0, LEGOPT_PAIR_BTM_PCT},

    {"sport-topn",             REQUIRED_ARG, 0, LEGOPT_SPORT_TOP_N},
    {"sport-top-threshold",    REQUIRED_ARG, 0, LEGOPT_SPORT_TOP_THRESHOLD},
    {"sport-top-pct",          REQUIRED_ARG, 0, LEGOPT_SPORT_TOP_PCT},
    {"sport-btmn",             REQUIRED_ARG, 0, LEGOPT_SPORT_BTM_N},
    {"sport-btm-threshold",    REQUIRED_ARG, 0, LEGOPT_SPORT_BTM_THRESHOLD},
    {"sport-btm-pct",          REQUIRED_ARG, 0, LEGOPT_SPORT_BTM_PCT},

    {"dport-topn",             REQUIRED_ARG, 0, LEGOPT_DPORT_TOP_N},
    {"dport-top-threshold",    REQUIRED_ARG, 0, LEGOPT_DPORT_TOP_THRESHOLD},
    {"dport-top-pct",          REQUIRED_ARG, 0, LEGOPT_DPORT_TOP_PCT},
    {"dport-btmn",             REQUIRED_ARG, 0, LEGOPT_DPORT_BTM_N},
    {"dport-btm-threshold",    REQUIRED_ARG, 0, LEGOPT_DPORT_BTM_THRESHOLD},
    {"dport-btm-pct",          REQUIRED_ARG, 0, LEGOPT_DPORT_BTM_PCT},

    {"portpair-topn",          REQUIRED_ARG, 0, LEGOPT_PORTPAIR_TOP_N},
    {"portpair-top-threshold", REQUIRED_ARG, 0, LEGOPT_PORTPAIR_TOP_THRESHOLD},
    {"portpair-top-pct",       REQUIRED_ARG, 0, LEGOPT_PORTPAIR_TOP_PCT},
    {"portpair-btmn",          REQUIRED_ARG, 0, LEGOPT_PORTPAIR_BTM_N},
    {"portpair-btm-threshold", REQUIRED_ARG, 0, LEGOPT_PORTPAIR_BTM_THRESHOLD},
    {"portpair-btm-pct",       REQUIRED_ARG, 0, LEGOPT_PORTPAIR_BTM_PCT},

    {"proto-topn",             REQUIRED_ARG, 0, LEGOPT_PROTO_TOP_N},
    {"proto-top-threshold",    REQUIRED_ARG, 0, LEGOPT_PROTO_TOP_THRESHOLD},
    {"proto-top-pct",          REQUIRED_ARG, 0, LEGOPT_PROTO_TOP_PCT},
    {"proto-btmn",             REQUIRED_ARG, 0, LEGOPT_PROTO_BTM_N},
    {"proto-btm-threshold",    REQUIRED_ARG, 0, LEGOPT_PROTO_BTM_THRESHOLD},
    {"proto-btm-pct",          REQUIRED_ARG, 0, LEGOPT_PROTO_BTM_PCT},

    {"cidr-src",               REQUIRED_ARG, 0, LEGOPT_CIDR_SRC},
    {"cidr-dest",              REQUIRED_ARG, 0, LEGOPT_CIDR_DEST},

    {0,0,0,0}                  /* sentinel entry */
};

static const char *ancientHelp[] = {
    ("Use: --fields=sip --values=flows --top --count=N\n"
     "\tPrint topN (by rec-count) src ips"),
    ("Use: --fields=sip --values=flows --top --threshold=N\n"
     "\tPrint src ips with rec-count >= top-threshold"),
    ("Use: --fields=sip --values=flows --top --percentage=N\n"
     "\tPrint src ips with %of-rec-count >= top-pct"),
    ("Use: --fields=sip --values=flows --bottom --count=N\n"
     "\tPrint bottomN (by rec-count) src ips."),
    ("Use: --fields=sip --values=flows --bottom --threshold=N\n"
     "\tPrint src ips with rec-count <= btm-threshold"),
    ("Use: --fields=sip --values=flows --bottom --percentage=N\n"
     "\tPrint src ips with %of-rec-count <= btm-pct"),

    ("Use: --fields=dip --values=flows --top --count=N\n"
     "\tPrint topN (by rec-count) dest ips"),
    ("Use: --fields=dip --values=flows --top --threshold=N\n"
     "\tPrint dest ips with rec-count >= top-threshold"),
    ("Use: --fields=dip --values=flows --top --percentage=N\n"
     "\tPrint dest ips with %of-rec-count >= top-pct"),
    ("Use: --fields=dip --values=flows --bottom --count=N\n"
     "\tPrint bottomN (by rec-count) dest ips."),
    ("Use: --fields=dip --values=flows --bottom --threshold=N\n"
     "\tPrint dest ips with rec-count <= btm-threshold"),
    ("Use: --fields=dip --values=flows --bottom --percentage=N\n"
     "\tPrint dest ips with %of-rec-count <= btm-pct"),

    ("Use: --fields=sip,dip --values=flows --top --count=N\n"
     "\tPrint topN (by rec-count) src+dest ip pairs"),
    ("Use: --fields=sip,dip --values=flows --top --threshold=N\n"
     "\tPrint src+dest ip pairs with rec-count >= top-threshold"),
    ("Use: --fields=sip,dip --values=flows --top --percentage=N\n"
     "\tPrint src+dest ip pairs with %of-rec-count >= top-pct"),
    ("Use: --fields=sip,dip --values=flows --bottom --count=N\n"
     "\tPrint bottomN (by rec-count) src+dest ip pairs."),
    ("Use: --fields=sip,dip --values=flows --bottom --threshold=N\n"
     "\tPrint src+dest ip pairs with rec-count <= btm-threshold"),
    ("Use: --fields=sip,dip --values=flows --bottom --percentage=N\n"
     "\tPrint src+dest ip pairs with %of-rec-count <= btm-pct"),

    ("Use: --fields=sport --values=flows --top --count=N\n"
     "\tPrint topN (by rec-count) src ports"),
    ("Use: --fields=sport --values=flows --top --threshold=N\n"
     "\tPrint src ports with rec-count >= top-threshold"),
    ("Use: --fields=sport --values=flows --top --percentage=N\n"
     "\tPrint src ports with %of-rec-count >= top-pct"),
    ("Use: --fields=sport --values=flows --bottom --count=N\n"
     "\tPrint bottomN (by rec-count) src ports."),
    ("Use: --fields=sport --values=flows --bottom --threshold=N\n"
     "\tPrint src ports with rec-count <= btm-threshold"),
    ("Use: --fields=sport --values=flows --bottom --percentage=N\n"
     "\tPrint src ports with %of-rec-count <= btm-pct"),

    ("Use: --fields=dport --values=flows --top --count=N\n"
     "\tPrint topN (by rec-count) dest ports"),
    ("Use: --fields=dport --values=flows --top --threshold=N\n"
     "\tPrint dest ports with rec-count >= top-threshold"),
    ("Use: --fields=dport --values=flows --top --percentage=N\n"
     "\tPrint dest ports with %of-rec-count >= top-pct"),
    ("Use: --fields=dport --values=flows --bottom --count=N\n"
     "\tPrint bottomN (by rec-count) dest ports."),
    ("Use: --fields=dport --values=flows --bottom --threshold=N\n"
     "\tPrint dest ports with rec-count <= btm-threshold"),
    ("Use: --fields=dport --values=flows --bottom --percentage=N\n"
     "\tPrint dest ports with %of-rec-count <= btm-pct"),

    ("Use: --fields=sport,dport --top --count=N\n"
     "\tPrint topN (by rec-count) sPort+dPort pairs"),
    ("Use: --fields=sport,dport --top --threshold=N\n"
     "\tPrint sPort+dPort pairs with rec-count >= top-threshold"),
    ("Use: --fields=sport,dport --top --percentage=N\n"
     "\tPrint sPort+dPort pairs with %of-rec-count >= top-pct"),
    ("Use: --fields=sport,dport --bottom --count=N\n"
     "\tPrint bottomN (by rec-count) sPort+dPort pairs."),
    ("Use: --fields=sport,dport --bottom --threshold=N\n"
     "\tPrint sPort+dPort pairs with rec-count <= btm-threshold"),
    ("Use: --fields=sport,dport --bottom --percentage=N\n"
     "\tPrint sPort+dPort pairs with %of-rec-count <= btm-pct"),

    ("Use: --fields=proto --values=flows --top --count=N\n"
     "\tPrint topN (by rec-count) protos"),
    ("Use: --fields=proto --values=flows --top --threshold=N\n"
     "\tPrint protos with rec-count >= top-threshold"),
    ("Use: --fields=proto --values=flows --top --percentage=N\n"
     "\tPrint protos with %of-rec-count >= top-pct"),
    ("Use: --fields=proto --values=flows --bottom --count=N\n"
     "\tPrint bottomN (by rec-count) protos."),
    ("Use: --fields=proto --values=flows --bottom --threshold=N\n"
     "\tPrint protos with rec-count <= btm-threshold"),
    ("Use: --fields=proto --values=flows --bottom --percentage=N\n"
     "\tPrint protos with %of-rec-count <= btm-pct"),

    ("Use: --sip=N\n"
     "\tConsider first N bits of src ip (sip-* & pair-* opts). Def. 32"),
    ("Use: --dip=N\n"
     "\tConsider first N bits of dest ip (dip-* & pair-* opts). Def. 32"),

    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static int legacyOptionsHandler(
    clientData  cData,
    int         opt_index,
    char       *opt_arg);
static int ancientOptionsHandler(
    clientData  cData,
    int         opt_index,
    char       *opt_arg);


/* FUNCTION DEFINITIONS */

/*
 *  legacyOptionsSetup(void);
 *
 *    Register the legacy options.
 */
int legacyOptionsSetup(clientData cData)
{
    assert((sizeof(legacyHelp)/sizeof(char *)) ==
           (sizeof(legacyOptions)/sizeof(struct option)));
    assert((sizeof(ancientHelp)/sizeof(char *)) ==
           (sizeof(ancientOptions)/sizeof(struct option)));

    /* register the options */
    if (skOptionsRegister(legacyOptions, &legacyOptionsHandler, cData)
        || skOptionsRegister(ancientOptions, &ancientOptionsHandler, cData))
    {
        skAppPrintErr("Unable to register legacy options");
        return 1;
    }

    return 0;
}


/*
 *  legacyOptionsUsage(fh);
 *
 *    Print the usage information for the legacy options to the named
 *    file handle.
 */
void legacyOptionsUsage(FILE *fh)
{
    int i;

    fprintf(fh, "\nLEGACY SWITCHES:\n");
    for (i = 0; legacyOptions[i].name; i++ ) {
        fprintf(fh, "--%s %s. %s\n", legacyOptions[i].name,
                SK_OPTION_HAS_ARG(legacyOptions[i]), legacyHelp[i]);
    }
    for (i = 0; ancientOptions[i].name; i++ ) {
        fprintf(fh, "--%s %s. %s\n", ancientOptions[i].name,
                SK_OPTION_HAS_ARG(ancientOptions[i]), ancientHelp[i]);
    }
}


/*
 *  status = legacyOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Process the legacy versions of the switches by calling the real
 *    appOptionsHandler().
 */
static int legacyOptionsHandler(
    clientData  cData,
    int         opt_index,
    char       *opt_arg)
{
#define KEY_COMBO_ERR(a, b)                                             \

    static int old_id = -1;
    rwstats_legacy_t *leg = (rwstats_legacy_t*)cData;
    const char *val_type = NULL;
    uint32_t val;
    int rv;

    switch ((legacyOptionsEnum)opt_index) {
      case LEGOPT_SIP:
        if (opt_arg) {
            rv = skStringParseUint32(&val, opt_arg, 1, 31);
            if (rv) {
                goto PARSE_ERROR;
            }
            cidr_sip = ~0 << (32 - val);
        }
        break;

      case LEGOPT_DIP:
        if (opt_arg) {
            rv = skStringParseUint32(&val, opt_arg, 1, 31);
            if (rv) {
                goto PARSE_ERROR;
            }
            cidr_dip = ~0 << (32 - val);
        }
        break;

      case LEGOPT_SPORT:
      case LEGOPT_DPORT:
      case LEGOPT_PROTOCOL:
      case LEGOPT_ICMP:
        break;

      case LEGOPT_FLOWS:
        val_type = "Records";
        break;

      case LEGOPT_PACKETS:
      case LEGOPT_BYTES:
        val_type = legacyOptions[opt_index].name;
        break;
    }

    if (opt_index <= LEGOPT_ICMP) {
        if (NULL == leg->fields) {
            old_id = opt_index;
            leg->fields = legacyOptions[opt_index].name;
        } else if (((1 << LEGOPT_SIP) | (1 << LEGOPT_DIP))
                   == ((1 << old_id) | (1 << opt_index)))
        {
            leg->fields = "sip,dip";
        } else if (((1 << LEGOPT_SPORT) | (1 << LEGOPT_DPORT))
                   == ((1 << old_id) | (1 << opt_index)))
        {
            leg->fields = "sport,dport";
        } else {
            skAppPrintErr(("Key combination --%s and --%s is not supported.\n"
                           "\tUse the --fields switch for this combination"),
                          legacyOptions[opt_index].name,
                          legacyOptions[old_id].name);
            return 1;
        }
    } else {
        assert(val_type != NULL);
        if (leg->values) {
            skAppPrintErr(("May only specify one of --%s, --%s or --%s.\n"
                           "Use the --values switch for multiple values"),
                          legacyOptions[LEGOPT_FLOWS].name,
                          legacyOptions[LEGOPT_PACKETS].name,
                          legacyOptions[LEGOPT_BYTES].name);
            return 1;
        }
        leg->values = val_type;
    }

    return 0;                   /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  legacyOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}




/*
 *  status = ancientOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Process the ancient versions of the switches by calling the real
 *    appOptionsHandler().
 */
static int ancientOptionsHandler(
    clientData  cData,
    int         opt_index,
    char       *opt_arg)
{
    static int been_here = -1;
    rwstats_legacy_t *leg = (rwstats_legacy_t*)cData;

    /* Handle CIDR blocks since they don't affect been_here */
    if (opt_index == LEGOPT_CIDR_SRC) {
        return legacyOptionsHandler(cData, LEGOPT_SIP, opt_arg);
    } else if (opt_index == LEGOPT_CIDR_DEST) {
        return legacyOptionsHandler(cData, LEGOPT_DIP, opt_arg);
    }

    if (been_here != -1) {
        skAppPrintErr(("%s no longer supports multiple analyses"
                       " in one run.\n"
                       "\tMay only specify one of "
                       "--sip-*, --dip-*, --sport-*, --dport-*,\n"
                       "\t--proto-*, --pair-*, or --portpair*."),
                      skAppName());
        return 1;
    }
    been_here = opt_index;

    /* verify no other keys or values */
    if (leg->fields) {
        skAppPrintErr(("Cannot use --%s with --sip, --dip, --sport, --dport,"
                       " --proto, or --icmp"),
                      ancientOptions[opt_index].name);
        return 1;
    }
    if (leg->values) {
        skAppPrintErr("Cannot use --%s with --bytes, --packets, or --flows",
                      ancientOptions[opt_index].name);
        return 1;
    }
    leg->values = "Records";

    if (FILEIsATty(stderr)) {
        skAppPrintErr(("The --%s switch will be removed in 2010."
                       "  Please use the newer command line switches"),
                      ancientOptions[opt_index].name);
    }

    switch ((ancientOptionsEnum)opt_index) {
      case LEGOPT_SIP_TOP_N:
        leg->fields = "sip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SIP_TOP_THRESHOLD:
        leg->fields = "sip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SIP_TOP_PCT:
        leg->fields = "sip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SIP_BTM_N:
        leg->fields = "sip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SIP_BTM_THRESHOLD:
        leg->fields = "sip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SIP_BTM_PCT:
        leg->fields = "sip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DIP_TOP_N:
        leg->fields = "dip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DIP_TOP_THRESHOLD:
        leg->fields = "dip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DIP_TOP_PCT:
        leg->fields = "dip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DIP_BTM_N:
        leg->fields = "dip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DIP_BTM_THRESHOLD:
        leg->fields = "dip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DIP_BTM_PCT:
        leg->fields = "dip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PAIR_TOP_N:
        leg->fields = "sip,dip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PAIR_TOP_THRESHOLD:
        leg->fields = "sip,dip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PAIR_TOP_PCT:
        leg->fields = "sip,dip";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PAIR_BTM_N:
        leg->fields = "sip,dip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PAIR_BTM_THRESHOLD:
        leg->fields = "sip,dip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PAIR_BTM_PCT:
        leg->fields = "sip,dip";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SPORT_TOP_N:
        leg->fields = "sport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SPORT_TOP_THRESHOLD:
        leg->fields = "sport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SPORT_TOP_PCT:
        leg->fields = "sport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SPORT_BTM_N:
        leg->fields = "sport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SPORT_BTM_THRESHOLD:
        leg->fields = "sport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_SPORT_BTM_PCT:
        leg->fields = "sport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DPORT_TOP_N:
        leg->fields = "dport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DPORT_TOP_THRESHOLD:
        leg->fields = "dport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DPORT_TOP_PCT:
        leg->fields = "dport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DPORT_BTM_N:
        leg->fields = "dport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DPORT_BTM_THRESHOLD:
        leg->fields = "dport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_DPORT_BTM_PCT:
        leg->fields = "dport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PORTPAIR_TOP_N:
        leg->fields = "sport,dport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PORTPAIR_TOP_THRESHOLD:
        leg->fields = "sport,dport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PORTPAIR_TOP_PCT:
        leg->fields = "sport,dport";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PORTPAIR_BTM_N:
        leg->fields = "sport,dport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PORTPAIR_BTM_THRESHOLD:
        leg->fields = "sport,dport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PORTPAIR_BTM_PCT:
        leg->fields = "sport,dport";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PROTO_TOP_N:
        leg->fields = "proto";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PROTO_TOP_THRESHOLD:
        leg->fields = "proto";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PROTO_TOP_PCT:
        leg->fields = "proto";
        if (appOptionsHandler(cData, OPT_TOP, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PROTO_BTM_N:
        leg->fields = "proto";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_COUNT, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PROTO_BTM_THRESHOLD:
        leg->fields = "proto";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_THRESHOLD, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_PROTO_BTM_PCT:
        leg->fields = "proto";
        if (appOptionsHandler(cData, OPT_BOTTOM, NULL)
            || appOptionsHandler(cData, OPT_PERCENTAGE, opt_arg))
        {
            return 1;
        }
        break;

      case LEGOPT_CIDR_SRC:
      case LEGOPT_CIDR_DEST:
        /* should have been handled above */
        skAbortBadCase(opt_index);
    }

    return 0;                   /* OK */
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
