/*
** Copyright (C) 2006-2012 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  SiLK file transfer program (receiver)
**
**  Michael Welsh Duggan
**  December 2006
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: rwreceiver.c 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $");

#include <silk/utils.h>
#include <silk/skdaemon.h>
#include <silk/sklog.h>
#include <silk/skmsg.h>
#include <silk/skdllist.h>
#include "rwtransfer.h"

/* LOCAL DEFINES AND TYPEDEFS */

/* where to write --help output */
#define USAGE_FH stdout

#define RWRECEIVER_PASSWORD_ENV ("RWRECEIVER" PASSWORD_ENV_POSTFIX)

/* Implement certain error conditions differently for version 1 */
#define SEND_CONN_REJECT(sndr) \
    (((sndr)->remote_version > 1) \
     ? CONN_REJECT_FILE \
     : CONN_DISCONNECT)
#define SEND_CONN_DUPLICATE(sndr) \
    (((sndr)->remote_version > 1) \
     ? CONN_DUPLICATE_FILE \
     : CONN_DISCONNECT)
#define FILE_INFO_ERROR_STATE(sndr) \
    (sndr->remote_version > 1 ? File_info : Error)


/* EXPORTED VARIABLE DEFINITIONS */

/* Set to non-zero when shutting down. */
int shuttingdown;

/* Per-sender data */
struct rbtree *transfers;

/* Local-side and remote-side version type identifiers */
skm_type_t local_version_check = CONN_RECEIVER_VERSION;
skm_type_t remote_version_check = CONN_SENDER_VERSION;

/* Password environment variable name */
const char *password_env = RWRECEIVER_PASSWORD_ENV;


/* LOCAL VARIABLE DEFINITIONS */

/* Destination directories */
static char *destination_dir;

/* Destination directories */
static sk_dllist_t *duplicate_dirs;

/* command, supplied by user, to run whenever a file is received */
static const char *post_command = NULL;


/* OPTIONS SETUP */

typedef enum {
    /* App specific options */
    OPT_DESTINATION_DIR,
    OPT_DUPLICATE_DEST,
    OPT_POST_COMMAND
} appOptionsEnum;

static struct option appOptions[] = {
    {"destination-directory", REQUIRED_ARG, 0, OPT_DESTINATION_DIR},
    {"duplicate-destination", REQUIRED_ARG, 0, OPT_DUPLICATE_DEST},
    {"post-command",          REQUIRED_ARG, 0, OPT_POST_COMMAND},
    {0,0,0,0}           /* sentinel entry */
};

static const char *appHelp[] = {
    ("Directory in which to drop incoming files"),
    ("Directory in which to stash a copy of\n"
     "\tincoming files. Repeatable. (optional)"),
    ("Run this command on each file after having successfully\n"
     "\treceived it. Def. None.  \"%s\" in the command will be substituted\n"
     "\tby the incremental file's path, and \"%I\" will be substituted by\n"
     "\tthe identifier of the rwsender that sent the file."),
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static void appUsageLong(void);
static void appTeardown(void);
static void appSetup(int argc, char **argv);
static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static void rwreceiverVerifyOptions(void);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  This function is
 *    passed to optionsSetup(); skOptionsParse() will call this funciton
 *    and then exit the program when the --help option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                       \
    ("<SWITCHES>\n"                                                     \
     "\tAccepts files from one or more sender daemons (rwsender)\n"     \
     "\tand places them in a given directory.\n")

    transferUsageLong(USAGE_FH, USAGE_MSG, appOptions, appHelp);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
static void appTeardown(void)
{
    static int teardownFlag = 0;
    RBLIST *iter;
    transfer_t *sndr;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    NOTICEMSG("Shutting down");

    shuttingdown = 1;

    transferShutdown();
    transferTeardown();

    /* Destroy stuff */
    iter = rbopenlist(transfers);
    CHECK_ALLOC(iter);
    while ((sndr = (transfer_t *)rbreadlist(iter)) != NULL) {

        if (sndr->ident != NULL) {
            free(sndr->ident);
        }
        free(sndr);
    }
    rbcloselist(iter);
    rbdestroy(transfers);

    skDLListDestroy(duplicate_dirs);

    NOTICEMSG("Finished shutting down");

    skdaemonTeardown();
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
static void appSetup(int argc, char **argv)
{
    int arg_index;

    /* check that we have the same number of options entries and help*/
    assert((sizeof(appHelp)/sizeof(char *)) ==
           (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize globals */
    shuttingdown    = 0;
    destination_dir = NULL;
    duplicate_dirs  = skDLListCreate(NULL);
    if (duplicate_dirs == NULL) {
        skAppPrintErr("Memory error creating list");
        exit(EXIT_FAILURE);
    }

    transfers = transferIdentTreeCreate();
    if (transfers == NULL) {
        skAppPrintErr("Unable to allocate receiver data structure");
        exit(EXIT_FAILURE);
    }

    /* register the options and handler */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL))
    {
        skAppPrintErr("Unable to register application options");
        exit(EXIT_FAILURE);
    }

    /* Register the other transfer options */
    if (transferSetup()) {
        exit(EXIT_FAILURE);
    }

    /* senddaemon runs as a daemon */
    if (skdaemonSetup((SKLOG_FEATURE_LEGACY | SKLOG_FEATURE_SYSLOG),
                      argc, argv)
        || sklogEnableThreadedLogging())
    {
        exit(EXIT_FAILURE);
    }

    /* parse the options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        /* options parsing should print error */
        skAppUsage();           /* never returns */
    }

    /* Verify the options */
    rwreceiverVerifyOptions();

    /* verify the required options for logging */
    if (skdaemonOptionsVerify()) {
        skAppUsage();           /* never returns */
    }

    /* check for extraneous arguments */
    if (arg_index != argc) {
        skAppPrintErr("Too many arguments or unrecognized switch --%s",
                      argv[arg_index]);
        skAppUsage();           /* never returns */
    }

    /* Identify the main thread */
    skthread_init("main");

    return;  /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;

    switch ((appOptionsEnum)opt_index) {

      case OPT_DESTINATION_DIR:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }
        destination_dir = opt_arg;
        break;

      case OPT_DUPLICATE_DEST:
        if (skOptionsCheckDirectory(opt_arg, appOptions[opt_index].name)) {
            return 1;
        }

        rv = skDLListPushTail(duplicate_dirs, opt_arg);
        if (rv != 0) {
            skAppPrintErr("Memory error adding directory");
            return 1;
        }
        break;

      case OPT_POST_COMMAND:
        post_command = opt_arg;
        break;
    }

    return 0;  /* OK */
}



static void rwreceiverVerifyOptions(void)
{
    int rv;

    /* Verify the transfer options */
    rv = transferVerifyOptions();
    if (rv != 0) {
        skAppUsage();           /* never returns */
    }

    /* Check for the existance of destination_dir */
    if (destination_dir == NULL) {
        skAppPrintErr("A destination directory is required");
        skAppUsage();           /* never returns */
    }
}


int transferUnblock(transfer_t *UNUSED(item))
{
    return 0;
}


/*
 *  runPostCommand(command, filename, ident);
 *
 *    Spawn a new subprocess to run 'command'.  Formatting directives
 *    in 'command' may be expanded to hold to the 'filename' that has
 *    just been received and the 'ident' of the rwsender that sent the
 *    file.
 */
static void runPostCommand(
    const char       *command,
    const char       *file,
    const char       *ident)
{
    sigset_t sigs;
    pid_t pid;
    size_t len;
    size_t file_len;
    size_t ident_len;
    const char *cp;
    const char *sp;
    char *expanded_cmd;
    char *exp_cp;

    /* Parent (original process) forks to create Child 1 */
    pid = fork();
    if (-1 == pid) {
        ERRMSG("Could not fork to run command: %s", strerror(errno));
        return;
    }

    /* Parent reaps Child 1 and returns */
    if (0 != pid) {
        /* Wait for Child 1 to exit. */
        while (waitpid(pid, NULL, 0) == -1) {
            if (EINTR != errno) {
                NOTICEMSG("Error waiting for child %ld: %s",
                          (long)pid, strerror(errno));
                break;
            }
        }
        return;
    }

    /* Disable log rotation in Child 1 */
    sklogDisableRotation();

    /* Child 1 forks to create Child 2 */
    pid = fork();
    if (pid == -1) {
        ERRMSG("Child could not fork for to run command: %s", strerror(errno));
        _exit(EXIT_FAILURE);
    }

    /* Child 1 immediately exits, so Parent can stop waiting */
    if (pid != 0) {
        _exit(EXIT_SUCCESS);
    }

    /* Only Child 2 makes it here */

    /* Unmask signals */
    sigemptyset(&sigs);
    sigprocmask(SIG_SETMASK, &sigs, NULL);

    /* Determine length of buffer needed for the expanded command
     * string and allocate it.  */
    len = strlen(command);
    file_len = strlen(file);
    ident_len = strlen(ident);
    cp = command;
    while (NULL != (sp = strchr(cp, (int)'%'))) {
        switch (*(sp + 1)) {
          case 's':
            len += file_len - 2;
            cp = sp + 2;
            break;
          case 'I':
            len += ident_len - 2;
            cp = sp + 2;
            break;
          default:
            cp = sp + 1;
        }
    }
    expanded_cmd = malloc(len + 1);
    if (expanded_cmd == NULL) {
        WARNINGMSG("Unable to allocate memory to create command string");
        _exit(EXIT_FAILURE);
    }

    /* Copy command into buffer, handling %-expansions */
    cp = command;
    exp_cp = expanded_cmd;
    while (NULL != (sp = strchr(cp, (int)'%'))) {
        strncpy(exp_cp, cp, sp - cp);
        exp_cp += (sp - cp);
        switch (*(sp + 1)) {
          case 's':
            strcpy(exp_cp, file);
            exp_cp += file_len;
            cp = sp + 2;
            break;
          case 'I':
            strcpy(exp_cp, ident);
            exp_cp += ident_len;
            cp = sp + 2;
            break;
          default:
            *exp_cp = *sp;
            exp_cp++;
            cp = sp + 1;
        }
    }
    strcpy(exp_cp, cp);
    expanded_cmd[len] = '\0';

    /* Execute the command */
    if (execl("/bin/sh", "sh", "-c", expanded_cmd, (char*)NULL)
        == -1)
    {
        ERRMSG(("Error invoking /bin/sh: %s"),
               strerror(errno));
        _exit(EXIT_FAILURE);
    }

    /* Should never get here. */
    skAbort();
}


void transferFiles(
    sk_msg_queue_t *q,
    skm_channel_t   channel,
    transfer_t     *sndr)
{
    int fd = -1;
    int dotfd = -1;
    uint64_t size = 0;
    uint8_t *map = NULL;
    char *name = NULL;
    char *dotname = NULL;
    char dotpath[PATH_MAX];
    char destpath[sizeof(dotpath)-1];
    int proto_err;
    int rv;
    sk_dll_iter_t iter;
    const char *duplicate_dir;
    enum transfer_state {File_info, File_info_ack,
                         Send_file, Complete_ack, Error} state;
    int thread_exit = 0;

    state = File_info;
    proto_err = 0;
    destpath[0] = '\0';
    dotpath[0] = '\0';

    while (!shuttingdown && !proto_err && !sndr->disconnect
           && (state != Error))
    {
        sk_msg_t *msg;

        /* Handle reads */
        switch (state) {
          case File_info:
          case Send_file:
            rv = skMsgQueueGetMessage(q, &msg);
            if (rv == -1) {
                ASSERT_ABORT(shuttingdown);
                continue;
            }
            if (handleDisconnect(msg, sndr->ident)) {
                state = Error;
                continue;
            }
            break;
          case Error:
            ASSERT_ABORT(0);
            break;
          default:
            msg = NULL;
        }

        /* Handle all states */
        switch (state) {
          case File_info:
            {
                file_info_t *finfo;
                uint32_t len;
                mode_t mode;
                off_t offrv;

                if ((proto_err = checkMsg(msg, q, CONN_NEW_FILE))) {
                    break;
                }
                DEBUG_PRINT1("Received CONN_NEW_FILE");
                finfo = (file_info_t *)skMsgMessage(msg);
                size = (uint64_t)ntohl(finfo->high_filesize) << 32 |
                       ntohl(finfo->low_filesize);
                /* blocksize = ntohl(finfo->block_size); --- UNUSED */
                mode = ntohl(finfo->mode) & 0777;
                len = skMsgLength(msg) - offsetof(file_info_t, filename);
                dotname = (char *)calloc(1, len + 1);
                CHECK_ALLOC(dotname);
                name = dotname + 1;
                dotname[0] = '.';
                memcpy(name, finfo->filename, len);
                if (!memchr(name, '\0', len)) {
                    sendString(q, channel, EXTERNAL, SEND_CONN_REJECT(sndr),
                               LOG_WARNING, "Illegal filename (from %s)",
                               sndr->ident);
                    state = FILE_INFO_ERROR_STATE(sndr);
                    break;
                }

                INFOMSG("Receiving from %s: %s", sndr->ident, name);

                /* Create the filename and .filename */
                rv = snprintf(destpath, sizeof(destpath), "%s/%s",
                              destination_dir, name);
                if ((size_t)rv >= sizeof(destpath)) {
                    sendString(q, channel, EXTERNAL,
                               SEND_CONN_REJECT(sndr),
                               LOG_WARNING, "Filename too long (from %s)",
                               sndr->ident);
                    state = FILE_INFO_ERROR_STATE(sndr);
                    break;
                }

                assert((size_t)rv < sizeof(destpath));

              reopen:
                fd = open(destpath, O_CREAT | O_EXCL | O_WRONLY, 0);
                if (fd == -1) {
                    struct stat st;

                    if (errno != EEXIST) {
                        CRITMSG("Could not create %s: %s",
                                destpath, strerror(errno));
                        thread_exit = 1;
                        break;
                    }

                    rv = stat(destpath, &st);
                    if (!rv && S_ISREG(st.st_mode)
                        && ((st.st_mode & 0777) == 0)
                        && ((st.st_size == 0)))
                    {
                        WARNINGMSG(("Filename %s already exists "
                                    "(from a previous run?). Removing"),
                                   destpath);
                        rv = unlink(destpath);
                        if (rv == 0) {
                            goto reopen;
                        }
                        WARNINGMSG("Failed to unlink %s: %s",
                                   destpath, strerror(errno));
                    }
                    destpath[0] = dotpath[0] = '\0';
                    sendString(q, channel, EXTERNAL,
                           SEND_CONN_DUPLICATE(sndr),
                           LOG_WARNING,
                           "Filename already exists (from %s)",
                           sndr->ident);
                    state = FILE_INFO_ERROR_STATE(sndr);
                    break;
                }

                DEBUGMSG("Created %s", destpath);

                rv = close(fd);
                fd = -1;
                if (rv == -1) {
                    CRITMSG("Could not close file descriptor for %s: %s",
                            destpath, strerror(errno));
                    thread_exit = 1;
                    break;
                }

                rv = snprintf(dotpath, sizeof(dotpath), "%s/%s",
                              destination_dir, dotname);

              reopen2:
                dotfd = open(dotpath, O_RDWR | O_CREAT | O_EXCL, mode);
                if (dotfd == -1) {
                    int saveerrno = errno;
                    if (errno == EEXIST) {
                        WARNINGMSG(("Filename %s already exists. "
                                    "Removing"), dotpath);
                        rv = unlink(dotpath);
                        if (rv == 0) {
                            goto reopen2;
                        }
                        WARNINGMSG("Failed to unlink %s: %s",
                                   dotpath, strerror(errno));
                    }
                    CRITMSG("Could not create %s: %s", dotpath,
                            strerror(saveerrno));
                    dotpath[0] = '\0';
                    break;
                }
                DEBUGMSG("Created %s", dotpath);

                /* Allocate space */
                offrv = lseek(dotfd, size - 1, SEEK_SET);
                if (offrv == -1) {
                    CRITMSG("Could not allocate disk space for %s: %s",
                            dotpath, strerror(errno));
                    thread_exit = 1;
                    break;
                }
                rv = write(dotfd, "", 1);
                if (rv == -1) {
                    CRITMSG("Could not allocate disk space for %s: %s",
                            dotpath, strerror(errno));
                    thread_exit = 1;
                    break;
                }

                /* Map space */
                map = (uint8_t *)mmap(0, size, PROT_READ | PROT_WRITE,
                                      MAP_SHARED, dotfd, 0);
                if ((void *)map == MAP_FAILED) {
                    CRITMSG("Could not map %s: %s",
                            dotpath, strerror(errno));
                    thread_exit = 1;
                    break;
                }
                rv = close(dotfd);
                dotfd = -1;
                if (rv == -1) {
                    CRITMSG(("Could not close file descriptor "
                             "for %s: %s"),
                            dotpath, strerror(errno));
                    thread_exit = 1;
                    break;
                }

                state = File_info_ack;
            }
            break;

          case File_info_ack:
            DEBUG_PRINT1("Sending CONN_NEW_FILE_READY");
            proto_err = skMsgQueueSendMessage(q, channel,
                                              CONN_NEW_FILE_READY, NULL, 0);
            state = Send_file;
            break;

          case Send_file:
            {
                block_info_t *block;
                uint64_t offset;
                uint32_t len;

                if (skMsgType(msg) != CONN_FILE_BLOCK) {
                    if ((proto_err = checkMsg(msg, q, CONN_FILE_COMPLETE))) {
                        break;
                    }
                    DEBUG_PRINT1("Received CONN_FILE_COMPLETE");
                    state = Complete_ack;
                    break;
                }

                block = (block_info_t *)skMsgMessage(msg);
                len = skMsgLength(msg) - offsetof(block_info_t, block);
                offset = (uint64_t)ntohl(block->high_offset) << 32 |
                         ntohl(block->low_offset);
                DEBUG_MSG_PRINT("Receiving offset == %" PRIu64
                                "  len == %" PRIu32,
                                offset, len);
                if (offset + len > size) {
                    sendString(q, channel, EXTERNAL, CONN_DISCONNECT,
                               LOG_WARNING,
                               ("Illegal block (offset/size %" PRIu64
                                "/%" PRIu32 ")"), offset, len);
                    state = Error;
                    break;
                }
                memcpy(map + offset, block->block, len);
            }
            break;

          case Complete_ack:
            rv = munmap(map, size);
            map = NULL;
            if (rv == -1) {
                CRITMSG("Could not unmap file %s: %s",
                        dotpath, strerror(errno));
                thread_exit = 1;
                break;
            }

            skDLLAssignIter(&iter, duplicate_dirs);
            while (skDLLIterForward(&iter, (void **)&duplicate_dir) == 0) {
                char path[sizeof(destpath)];

                snprintf(path, sizeof(path), "%s/%s", duplicate_dir, name);
                DEBUGMSG("Linking %s as %s", dotpath, path);
                rv = link(dotpath, path);
                if (rv == EXDEV) {
                    DEBUGMSG("Link failed EXDEV; copying %s to %s",
                             dotpath, path);
                    rv = skCopyFile(dotpath, path);
                    if (rv != 0) {
                        WARNINGMSG("Could not copy %s to %s: %s",
                                   dotpath, path, strerror(rv));
                        continue;
                    }
                } else if (rv != 0) {
                    WARNINGMSG("Could not link %s as %s: %s",
                               dotpath, path, strerror(errno));
                    continue;
                }
            }

            DEBUGMSG("Renaming %s to %s", dotpath, destpath);
            rv = rename(dotpath, destpath);
            if (rv != 0) {
                CRITMSG("Failed rename of %s to %s: %s",
                        dotpath, destpath, strerror(errno));
                thread_exit = 1;
                break;
            }

            DEBUG_PRINT1("Sending CONN_FILE_COMPLETE");
            proto_err = skMsgQueueSendMessage(q, channel,
                                              CONN_FILE_COMPLETE, NULL, 0);
            if (proto_err == 0) {
                /* Run the post command on the file */
                if (post_command) {
                    runPostCommand(post_command, destpath, sndr->ident);
                }
                destpath[0] = '\0';
                INFOMSG("Finished receiving from %s: %s", sndr->ident, name);
                free(dotname);
                dotname = NULL;
            }

            destpath[0] = dotpath[0] = '\0';

            state = File_info;
            break;

          case Error:
            ASSERT_ABORT(0);
        }

        if (msg != NULL) {
            skMsgDestroy(msg);
        }
    }

    if (fd != -1) {
        close(fd);
    }
    if (dotfd != -1) {
        close(fd);
    }
    if (map != NULL) {
        munmap(map, size);
    }
    if (dotname != NULL) {
        free(dotname);
    }
    if (dotpath[0] != '\0') {
        DEBUGMSG("Removing %s", dotpath);
        unlink(dotpath);
    }
    if (destpath[0] != '\0') {
        DEBUGMSG("Removing %s", destpath);
        unlink(destpath);
    }
    if (thread_exit) {
        threadExit(EXIT_FAILURE, NULL);
    }
}


int main(int argc, char **argv)
{
    int rv;

    appSetup(argc, argv);       /* never returns on error */

    /* start the logger and become a daemon */
    rv = skdaemonize(&shuttingdown, &appTeardown);
    if (rv == -1) {
        exit(EXIT_FAILURE);
    }

    /* Run in client or server mode */
    rv = startTransferDaemon();
    if (rv != 0) {
        exit(EXIT_FAILURE);
    }

    /* We now run forever, accepting signals */
    while (!shuttingdown) {
        pause();
    }

    /* done */
    appTeardown();
    return main_retval;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
