#!/usr/bin/env python
#######################################################################
# Copyright (C) 2008-2012 by Carnegie Mellon University.
#
# @OPENSOURCE_HEADER_START@
#
# Use of the SILK system and related source code is subject to the terms
# of the following licenses:
#
# GNU Public License (GPL) Rights pursuant to Version 2, June 1991
# Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
#
# NO WARRANTY
#
# ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
# PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
# PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
# "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
# KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
# LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
# MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
# OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
# SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
# TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
# WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
# LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
# CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
# CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
# DELIVERABLES UNDER THIS LICENSE.
#
# Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
# Mellon University, its trustees, officers, employees, and agents from
# all claims or demands made against them (and any related losses,
# expenses, or attorney's fees) arising out of, or relating to Licensee's
# and/or its sub licensees' negligent use or willful misuse of or
# negligent conduct or willful misconduct regarding the Software,
# facilities, or other rights or assistance granted by Carnegie Mellon
# University under this License, including, but not limited to, any
# claims of product liability, personal injury, death, damage to
# property, or violation of any laws or regulations.
#
# Carnegie Mellon University Software Engineering Institute authored
# documents are sponsored by the U.S. Department of Defense under
# Contract FA8721-05-C-0003. Carnegie Mellon University retains
# copyrights in all material produced under this contract. The U.S.
# Government retains a non-exclusive, royalty-free license to publish or
# reproduce these documents, or allow others to do so, for U.S.
# Government purposes only pursuant to the copyright license under the
# contract clause at 252.227.7013.
#
# @OPENSOURCE_HEADER_END@
#
#######################################################################

#######################################################################
# $SiLK: sendrcv_tests.py 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $
#######################################################################

import unittest
import sys
import re
import gc
import os
from time import sleep
srcdir = os.environ.get("srcdir")
if srcdir:
    sys.path.insert(0, os.path.join(srcdir, "tests"))
from sendrcv_daemon_test import *
from gencerts import reset_all_certs_and_keys
from config_vars import config_vars
from optparse import OptionParser

VERBOSE_LOGS = True
OVERWRITE    = False
LOG_LEVEL    = "info"

rfiles = None

def setup():
    global rfiles
    rfiles = []
    if VERBOSE_LOGS:
        sys.stdout.write("Creating temporary files\n")
    for x in xrange(0,10):
        rfiles.append(
            create_random_file(suffix=("." + chr(ord('a') + x)),
                               size=(500000, 600000)))

def teardown():
    global rfiles
    if VERBOSE_LOGS:
        sys.stdout.write("Removing temporary files\n")
    for file, data in rfiles:
        os.remove(file)
    rfiles = None

def tls_supported():
    return eval(config_vars.get('SK_ENABLE_GNUTLS', None))

def Sender(**kwds):
    return Rwsender(overwrite=OVERWRITE, log_level=LOG_LEVEL, **kwds)

def Receiver(**kwds):
    return Rwreceiver(overwrite=OVERWRITE, log_level=LOG_LEVEL, **kwds)

class TestSenderReceiver(unittest.TestCase):

    def _testConnectClose(self, tls=False):
        if tls and not tls_supported():
            return None
        reset_all_certs_and_keys()
        s1 = Sender()
        r1 = Receiver()
        sy = System()
        try:
            sy.connect(s1, r1, tls=tls)
            if VERBOSE_LOGS:
                sy.logs.tee(sys.stdout)
            sy.start()
            self.assert_(sy.logs.timed_trigger(
                    15, s1.name, "Connected to remote %s" % r1.name))
            self.assert_(sy.logs.timed_trigger(
                    1, s1.name, "Connected to remote %s" % r1.name))
            status = sy.stop()
            self.assert_(sy.logs.timed_trigger(15, s1.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, r1.name, "Stopped logging"))
        except:
            sy.logs.stop()
            sy.stop()
            raise
        self.assert_(sy.logs.timed_trigger(10, s1.name,
                                          "Finished shutting down"))
        self.assert_(sy.logs.timed_trigger(1, r1.name,
                                          "Finished shutting down"))
        rv = sy.logs.sequence_match(
            [(s1.name, "Attempting to connect to %s" % r1.name),
             (s1.name, "Connected to remote %s" % r1.name),
             (s1.name, "Shutting down due to SIGTERM signal"),
             (s1.name, "Finished shutting down")])
        self.assert_(not rv)

        rv = sy.logs.sequence_match(
            [(r1.name, "Connected to remote %s" % s1.name),
             (r1.name, "Shutting down due to SIGTERM signal"),
             (r1.name, "Finished shutting down")])
        self.assert_(not rv)

    def test010_Simple(self):
        """
        Test to see if we can start a sender/receiver pair, that they
        connect, and that they shut down properly.
        """
        self._testConnectClose()

    def test020_TLS(self):
        """
        Test to see if we can start a sender/receiver pair using TLS,
        that they connect, and that they shut down properly.
        """
        self._testConnectClose(tls=True)

    def _testSendRcv(self, tls=False,
                     sender_client=True,
                     stop_sender=False,
                     kill=False):
        if tls and not tls_supported():
            return None
        global rfiles
        reset_all_certs_and_keys()
        s1 = Sender()
        r1 = Receiver()
        if stop_sender:
            if kill:
                stop = s1.kill
            else:
                stop = s1.stop
            start = s1.start
        else:
            if kill:
                stop = r1.kill
            else:
                stop = r1.stop
            start = r1.start
        s1.create_dirs()
        s1.send_files(rfiles)
        sy = System()
        try:
            if VERBOSE_LOGS:
                sy.logs.tee(sys.stdout)
            if sender_client:
                sy.connect(s1, r1, tls=tls)
            else:
                sy.connect(r1, s1, tls=tls)
            sy.start()
            self.assert_(sy.logs.timed_trigger(
                    15, s1.name, "Connected to remote %s" % r1.name))
            self.assert_(sy.logs.timed_trigger(
                    30, s1.name, "Succeeded sending .* to %s" % r1.name))
            stop()
            sleep(10)
            start()
            self.assert_(sy.logs.timed_trigger(
                    40, s1.name, "Connected to remote %s" % r1.name))
            for path, data in rfiles:
                self.assert_(sy.logs.timed_trigger(
                        30, (s1.name, r1.name),
                        ("Succeeded sending .*/%(name)s to %(rname)s|"
                         "Finished receiving from %(sname)s: %(name)s|"
                         "Remote side %(rname)s rejected .*/%(name)s") %
                        {"name": re.escape(os.path.basename(path)),
                         "rname": r1.name, "sname": s1.name}))
            for f in rfiles:
                (error, path) = r1.check_sent(f)
                self.assert_(not error)
            status = sy.stop()
            self.assert_(sy.logs.timed_trigger(15, s1.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, r1.name, "Stopped logging"))
        except:
            sy.logs.stop()
            sy.stop()
            raise

    def test030_SendRcvStopReceiverServer(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        stopping the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=False, kill=False)

    def test040_SendRcvStopSenderServer(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        stopping the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=True, kill=False)

    def test050_SendRcvStopReceiverClient(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        stopping the receiver.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=False, kill=False)

    def test060_SendRcvStopSenderClient(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        stopping the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=True, kill=False)

    def test070_SendRcvKillReceiverServer(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        killing the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=False, kill=True)

    def test080_SendRcvKillSenderServer(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        killing the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=True, kill=True)

    def test090_SendRcvKillReceiverClient(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        killing the receiver.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=False, kill=True)

    def test100_SendRcvKillSenderClient(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        killing the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=True, kill=True)

    def test110_SendRcvStopReceiverServerTLS(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        stopping the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=False, kill=False,
                          tls=True)

    def test120_SendRcvStopSenderServerTLS(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        stopping the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=True, kill=False,
                          tls=True)

    def test130_SendRcvStopReceiverClientTLS(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        stopping the receiver.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=False, kill=False,
                          tls=True)

    def test140_SendRcvStopSenderClientTLS(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        stopping the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=True, kill=False,
                          tls=True)

    def test150_SendRcvKillReceiverServerTLS(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        killing the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=False, kill=True,
                          tls=True)

    def test160_SendRcvKillSenderServerTLS(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        killing the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=True, kill=True,
                          tls=True)

    def test170_SendRcvKillReceiverClientTLS(self):
        """
        Test a sender/receiver connection, with sender as server,
        sending files.  Midway the connection is terminated by
        killing the receiver.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=False, stop_sender=False, kill=True,
                          tls=True)

    def test180_SendRcvKillSenderClientTLS(self):
        """
        Test a sender/receiver connection, with receiver as server,
        sending files.  Midway the connection is terminated by
        killing the sender.  The connection is restarted and resumed.
        """
        self._testSendRcv(sender_client=True, stop_sender=True, kill=True,
                          tls=True)

    def _testMultiple(self, tls=False):
        global rfiles
        if tls and not tls_supported():
            return None
        reset_all_certs_and_keys()
        s1 = Sender()
        s2 = Sender()
        r1 = Receiver()
        r2 = Receiver()
        sy = System()
        try:
            if VERBOSE_LOGS:
                sy.logs.tee(sys.stdout)
            sy.connect([r1, r2], [s1, s2], tls=tls)
            sy.start()
            self.assert_(sy.logs.timed_trigger(
                    60, s1.name, "Connected to remote %s" % r1.name))
            self.assert_(sy.logs.timed_trigger(
                    60, s1.name, "Connected to remote %s" % r2.name))
            self.assert_(sy.logs.timed_trigger(
                    60, s2.name, "Connected to remote %s" % r1.name))
            self.assert_(sy.logs.timed_trigger(
                    60, s2.name, "Connected to remote %s" % r2.name))
            filea = rfiles[0]
            fileb = rfiles[1]
            s1.send_files([filea])
            s2.send_files([fileb])
            params = {"filea": re.escape(os.path.basename(filea[0])),
                      "fileb": re.escape(os.path.basename(fileb[0])),
                      "rnamec": r1.name, "rnamed": r2.name}
            self.assert_(sy.logs.timed_trigger(
                    30, s1.name, "Succeeded sending .*/%(filea)s to %(rnamec)s"
                    % params))
            self.assert_(sy.logs.timed_trigger(
                    30, s1.name, "Succeeded sending .*/%(filea)s to %(rnamed)s"
                    % params))
            self.assert_(sy.logs.timed_trigger(
                    30, s2.name, "Succeeded sending .*/%(fileb)s to %(rnamec)s"
                    % params))
            self.assert_(sy.logs.timed_trigger(
                    30, s2.name, "Succeeded sending .*/%(fileb)s to %(rnamed)s"
                    % params))
            for f in [filea, fileb]:
                for r in [r1, r2]:
                    (error, path) = r.check_sent(f)
                    self.assert_(not error)
            status = sy.stop()
            self.assert_(sy.logs.timed_trigger(15, s1.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, s2.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, r1.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, r2.name, "Stopped logging"))
        except:
            sy.logs.stop()
            sy.stop()
            raise

    def test190_Multiple(self):
        """
        Test two senders connected to two receivers.  Each sender
        sends a file to both receivers.
        """
        self._testMultiple()

    def test200_MultipleTLS(self):
        """
        Test two senders connected to two receivers via TLS.  Each
        sender sends a file to both receivers.
        """
        self._testMultiple(tls=True)

    def _testFilter(self, tls=False):
        global rfiles
        if tls and not tls_supported():
            return None
        reset_all_certs_and_keys()
        r1 = Receiver()
        r2 = Receiver()
        s1 = Sender(filters=[(r1.name, "[a-g]$"), (r2.name, "[d-j]$")])
        sy = System()
        try:
            if VERBOSE_LOGS:
                sy.logs.tee(sys.stdout)
            sy.connect([r1, r2], s1, tls=tls)
            sy.start()
            self.assert_(sy.logs.timed_trigger(
                    30, s1.name, "Connected to remote %s" % r1.name))
            self.assert_(sy.logs.timed_trigger(
                    30, s1.name, "Connected to remote %s" % r2.name))
            s1.send_files(rfiles)
            cfiles = [x for x in rfiles if 'a' <= x[0][-1] <= 'g']
            dfiles = [x for x in rfiles if 'd' <= x[0][-1] <= 'j']
            for (f, data) in cfiles:
                self.assert_(sy.logs.timed_trigger(
                    15, s1.name, "Succeeded sending .*/%(file)s to %(name)s"
                    % {"file": re.escape(os.path.basename(f)),
                       "name" : r1.name}))
            for (f, data) in dfiles:
                self.assert_(sy.logs.timed_trigger(
                    15, s1.name, "Succeeded sending .*/%(file)s to %(name)s"
                    % {"file": re.escape(os.path.basename(f)),
                       "name" : r2.name}))
            for f in cfiles:
                (error, path) = r1.check_sent(f)
                self.assert_(not error)
            for f in dfiles:
                (error, path) = r2.check_sent(f)
                self.assert_(not error)
            cset = set(cfiles)
            dset = set(dfiles)
            for f in cset - dset:
                (error, path) = r2.check_sent(f)
                self.assert_(error)
            for f in dset - cset:
                (error, path) = r1.check_sent(f)
                self.assert_(error)
            status = sy.stop()
            self.assert_(sy.logs.timed_trigger(15, s1.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, r1.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, r2.name, "Stopped logging"))
        except:
            sy.logs.stop()
            sy.stop()
            raise

    def test210_Filter(self):
        """
        Test filtering with a sender and two receivers.  Using
        filters, some files get sent to receiver A, some to receiver
        B, and some to both.
        """
        self._testFilter()

    def test220_PostCommand(self):
        """
        Test the post command option.
        """
        global rfiles
        if srcdir:
            cmddir = os.path.join(srcdir, "tests")
        else:
            cmddir = os.path.join(".", "tests")
        command = os.path.join(cmddir, "post-command.sh")
        post_command = command + " %I %s"
        s1 = Sender()
        r1 = Receiver(post_command=post_command)
        s1.create_dirs()
        s1.send_files(rfiles)
        sy = System()
        try:
            if VERBOSE_LOGS:
                sy.logs.tee(sys.stdout)
            sy.connect(s1, r1)
            sy.start()
            self.assert_(sy.logs.timed_trigger(
                    15, s1.name, "Connected to remote %s" % r1.name))
            for path, data in rfiles:
                self.assert_(sy.logs.timed_trigger(
                        30, r1.name,
                        ("Post command: Ident: %(sname)s  "
                         "Filename: .*/%(file)s") %
                        {"file": re.escape(os.path.basename(path)),
                         "sname": s1.name}))
            status = sy.stop()
            self.assert_(sy.logs.timed_trigger(15, s1.name, "Stopped logging"))
            self.assert_(sy.logs.timed_trigger(15, r1.name, "Stopped logging"))

        except:
            sy.logs.stop()
            sy.stop()
            raise

def suite():
    suite = unittest.TestSuite()
    suite.addTest(unittest.TestLoader().
                  loadTestsFromTestCase(TestSenderReceiver))
    return suite

if __name__ == '__main__':
    parser = OptionParser()
    parser.add_option("--verbose", action="store_true", dest="verbose",
                      default=False)
    parser.add_option("--overwrite-dirs", action="store_true", dest="overwrite",
                      default=False)
    parser.add_option("--log-level", action="store", type="string",
                      dest="log_level", default="info")
    (options, args) = parser.parse_args()
    VERBOSE_LOGS = options.verbose
    OVERWRITE = options.overwrite
    LOG_LEVEL = options.log_level
    setup()
    retval = 1
    try:
        result = unittest.TextTestRunner(verbosity=3).run(suite())
    except:
        teardown()
        raise
    if result.wasSuccessful():
        retval = 0
    teardown()
    sys.exit(retval)
