#######################################################################
# Copyright (C) 2009-2012 by Carnegie Mellon University.
#
# @OPENSOURCE_HEADER_START@
#
# Use of the SILK system and related source code is subject to the terms
# of the following licenses:
#
# GNU Public License (GPL) Rights pursuant to Version 2, June 1991
# Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
#
# NO WARRANTY
#
# ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
# PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
# PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
# "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
# KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
# LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
# MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
# OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
# SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
# TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
# WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
# LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
# CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
# CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
# DELIVERABLES UNDER THIS LICENSE.
#
# Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
# Mellon University, its trustees, officers, employees, and agents from
# all claims or demands made against them (and any related losses,
# expenses, or attorney's fees) arising out of, or relating to Licensee's
# and/or its sub licensees' negligent use or willful misuse of or
# negligent conduct or willful misconduct regarding the Software,
# facilities, or other rights or assistance granted by Carnegie Mellon
# University under this License, including, but not limited to, any
# claims of product liability, personal injury, death, damage to
# property, or violation of any laws or regulations.
#
# Carnegie Mellon University Software Engineering Institute authored
# documents are sponsored by the U.S. Department of Defense under
# Contract FA8721-05-C-0003. Carnegie Mellon University retains
# copyrights in all material produced under this contract. The U.S.
# Government retains a non-exclusive, royalty-free license to publish or
# reproduce these documents, or allow others to do so, for U.S.
# Government purposes only pursuant to the copyright license under the
# contract clause at 252.227.7013.
#
# @OPENSOURCE_HEADER_END@
#
#######################################################################

#######################################################################
# $SiLK: daemon_test.py 372a8bc31d8a 2012-02-10 21:55:28Z mthomas $
#######################################################################
import socket
import os
import os.path
import re
import tempfile
import shutil
import sys
import traceback
from config_vars import config_vars
from signal import SIGTERM, SIGKILL
from select import select
from threading import Thread, Lock, Timer, Event
from subprocess import Popen, PIPE, call


kill_delay = 20
global_int = 0

V5PDU_LEN  = 1464
TCPBUF     = 2048
LINEBUF    = 1024

class TimedReadline(object):
    def __init__(self, fd):
        self.buf = ""
        if isinstance(fd, file):
            self.fd = fd.fileno()
        else:
            self.fd = fd

    def __call__(self, timeout):
        while True:
            x = self.buf.find('\n')
            if x >= 0:
                retval = self.buf[:x+1]
                self.buf = self.buf[x+1:]
                return retval
            (r, w, x) = select([self.fd], [], [], timeout)
            if r:
                more = os.read(self.fd, LINEBUF)
                if more:
                    self.buf += more
                else:
                    return None
            else:
                return ""

def check_call(*popenargs, **kwargs):
    sys.stderr.write('Calling "%s"\n' % ' '.join(*popenargs))
    if call(*popenargs, **kwargs):
        raise RuntimeError, ("Failed to execute %s" % ' '.join(*popenargs))

class Dirobject(object):

    def __init__(self, overwrite=False, basedir=None):
        self.dirs = list()
        self.dirname = dict()
        self.basedir = basedir
        self.dirs_created = False
        self.overwrite = overwrite

    def create_basedir(self):
        if self.basedir:
            if not os.path.exists(self.basedir):
                os.mkdir(self.basedir)
        else:
            self.basedir = tempfile.mkdtemp()

    def remove_basedir(self):
        if self.basedir and self.dirs_created:
            shutil.rmtree(self.basedir)
            self.dirs_created = False

    def create_dirs(self):
        if not self.dirs_created:
            self.create_basedir()
            for name in self.dirs:
                self.dirname[name] = os.path.abspath(
                    os.path.join(self.basedir, name))
                if os.path.exists(self.dirname[name]):
                    if self.overwrite:
                        shutil.rmtree(self.dirname[name])
                        os.mkdir(self.dirname[name])
                else:
                    os.mkdir(self.dirname[name])
            self.dirs_created = True

    def get_path(self, name, path):
        return os.path.join(self.dirname[name], path)


class PduSender(object):
    def __init__(self, num, port, log=None):
        self._file = file
        self._port = port
        self._num = num
        self._log = log
        self.process = None

    def start(self):
        prog = os.path.join(os.environ["top_srcdir"], "tests", "make-data.pl")
        args = [config_vars["PERL"], prog,
                "--pdu-network", "localhost:" + str(self._port),
                "--max-records", str(self._num)]
        self._log("Starting: %s" % args)
        self.process = Popen(args)
        return self.process

    def stop(self):
        if self.process is None:
            return None
        self.process.poll()
        if self.process.returncode is None:
            try:
                os.kill(self.process.pid, SIGTERM)
            except OSError:
                pass
        return self.process.returncode


class TcpSender(object):
    def __init__(self, file, port):
        self._file = file
        self._address = ('localhost', port)
        self._running = False

    def start(self):
        thread = Thread(target = self.go)
        thread.daemon = True
        thread.start()

    def go(self):
        self._running = True
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.connect(self._address)
        sock.settimeout(1)
        while self._running:
            pdu = self._file.read(TCPBUF)
            if not pdu:
                self._running = False
                continue
            count = len(pdu)
            while self._running and count:
                try:
                    num_read = sock.send(pdu)
                    pdu = pdu[num_read:]
                    count -= num_read
                except socket.timeout:
                    pass

    def stop(self):
        self._running = False

class Daemon(Dirobject):

    def __init__(self, name=None, log_level="info", prog_env=None, **kwds):
        global global_int
        Dirobject.__init__(self, **kwds)
        if name:
            self.name = name
        else:
            self.name = type(self).__name__ + str(global_int)
            global_int += 1
        self.process = None
        self.log_manager = None
        self.log_level = log_level
        self._prog_env = prog_env
        self.daemon = True
        self.verbose = False

    def printv(self, *args):
        if self.verbose:
            sys.stderr.write(' '.join(str(x) for x in args) + '\n')

    def get_executable(self):
        return os.environ.get(self._prog_env,
                              os.path.join(".", self.exe_name))

    def get_args(self):
        args = [self.get_executable(),
                '--no-daemon',
                '--log-dest', 'stderr',
                '--log-level', self.log_level]
        return args

    def init(self):
        pass

    def start(self):
        args = self.get_args()
        self.printv("Starting: %s" % args)
        self.process = Popen(args, stderr=PIPE)
        self.printv("Started")
        if self.log_manager:
            self.log_manager.add_client(self.name, self.process.stderr)
        return self.process

    def kill(self):
        self.printv("Killing")
        if self.process and self.process.returncode is None:
            try:
                self.printv("Sending SIGKILL")
                os.kill(self.process.pid, SIGKILL)
            except OSError:
                pass

    def stop(self):
        self.printv("Stopping")
        if self.process is None:
            return None
        if self.process.returncode is None:
            try:
                self.printv("Sending SIGTERM")
                os.kill(self.process.pid, SIGTERM)
                timer = Timer(kill_delay, self.kill)
                timer.start()
                (pid, status) = os.waitpid(self.process.pid, 0)
                self.printv("Stopped: status =", status)
                timer.cancel()
            except OSError:
                pass
        return self.process.returncode


class SafeThread(Thread):
    def run(self, *args, **kwds):
        try:
            self.go(*args, **kwds)
        except SystemExit:
            raise
        except:
            traceback.print_exc(file=sys.stdout)
            os._exit(os.EX_SOFTWARE)

class Log_manager(SafeThread):

    def __init__(self, verbose=False):
        Thread.__init__(self)
        self.name = "Log Manager"
        self._lock = Lock()
        self._tee_locs = list()
        self._running = False
        self.clients = dict()
        self.rdict = dict()
        self._triggers = dict()
        self._rtriggers = dict()
        self.logs = list()
        self._uniqlock = Lock()
        self._uniqnum = 0
        self._verbose = verbose
        self.daemon = True

    def lock(self):
        self._lock.acquire()

    def unlock(self):
        self._lock.release()

    def add_client(self, name, output):
        self.lock()
        self.clients[name] = output
        self.rdict[output] = name
        self.unlock()

    def go(self):
        self._running = True
        while True:
            self.lock()
            readset = self.clients.values()
            self.unlock()
            (readers, writers, x) = select(readset, [], [], 1)
            for reader in readers:
                name = self.rdict[reader]
                self.lock()
                line = reader.readline()
                if not line:
                    del self.rdict[reader]
                    del self.clients[name]
                else:
                    for loc in self._tee_locs:
                        loc.write(": ".join([name, line]))
                        loc.flush()
                    self.logs.append((name, line))
                    if name in self._triggers:
                        dellist = []
                        for (ident,
                             (checkre, callback, args, kwds,
                              delete)) in (self._triggers[name].iteritems()):
                            match = checkre.search(line)
                            if match:
                                callback(match, *args, **kwds)
                                if delete:
                                    dellist.append(ident)
                        for ident in dellist:
                            self.remove_from_triggers(ident)
                self.unlock()
            if not readers and not self._running:
                return None

    def _get_uniq(self):
        self._uniqlock.acquire()
        ident = self._uniqnum
        self._uniqnum += 1
        self._uniqlock.release()
        return ident

    def _add_to_triggers(self, names, value):
        ident = self._get_uniq()
        for name in names:
            if name not in self._triggers:
                self._triggers[name] = dict([(ident, value)])
            else:
                self._triggers[name][ident] = value
        self._rtriggers[ident] = names
        return ident

    def remove_from_triggers(self, ident):
        names = self._rtriggers.pop(ident)
        for name in names:
            d = self._triggers[name]
            del d[ident]
            if not d:
                del self._triggers[name]

    def trigger_callback(self, checkname, checkline, callback, delete=True,
                         *args, **kwds):
        if isinstance(checkname, basestring):
            checkname = [checkname]
        checkre = re.compile(checkline)
        self.lock()
        for (name, line) in self.logs:
            if name in checkname:
                match = checkre.search(line)
                if match:
                    callback(match, *args, **kwds)
                    self.unlock()
                    return None
        ident = self._add_to_triggers(checkname,
                                      (checkre, callback, args, kwds, delete))
        self.unlock()
        return ident

    def timed_trigger(self, seconds, name, match):
        event = Event()
        ident = self.trigger_callback(name, match, lambda x: event.set())
        event.wait(seconds)
        if not event.isSet():
            self.lock()
            try:
                self.remove_from_triggers(ident)
            except KeyError:
                pass
            self.unlock()
            if self._verbose:
                sys.stderr.write("trigger timed out %s\n" % match)
        elif self._verbose:
            sys.stderr.write("trigger succeeded %s\n" % match)
        return event.isSet()

    def stop(self):
        self._running = False

    def tee(self, loc):
        self.lock()
        self._tee_locs.append(loc)
        self.unlock()

    def sequence_match(self, checklist):
        self.lock()
        checklist = iter(checklist)
        try:
            (checkname, checkline) = checklist.next()
            if isinstance(checkname, basestring):
                checkname = [checkname]
        except StopIteration:
            self.unlock()
            return None
        checkre = re.compile(checkline)
        for (name, line) in self.logs:
            if ((not checkname or name in checkname) and
                checkre.search(line)):
                try:
                    (checkname, checkline) = checklist.next()
                    if isinstance(checkname, basestring):
                        checkname = [checkname]
                    checkre = re.compile(checkline)
                except StopIteration:
                    self.unlock()
                    return None
        self.unlock()
        return (checkname, checkline)


def get_ephemeral_port():
    sock = socket.socket()
    sock.bind(("", 0))
    (addr, port) = sock.getsockname()
    sock.close()
    return port
