#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-ipfix-ipv6.pl 288d52a2e0d1 2019-10-17 18:29:26Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsilk2ipfix = check_silk_app('rwsilk2ipfix');

# find the data files we use as sources, or exit 77
my %file;
$file{v6data} = get_data_or_exit77('v6data');

# verify that required features are available
check_features(qw(ipv6 ipfix));

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'ipfix', 0, 'polldir', 'ipv6');

# Generate the test data
my $ipfixdata = "$tmpdir/v6data.ipfix";
unlink $ipfixdata;
system "$rwsilk2ipfix --ipfix-output=$ipfixdata $file{v6data}"
    and die "ERROR: Failed running rwsilk2ipfix\n";

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--sensor-conf=$sensor_conf",
                     "--move $ipfixdata:incoming",
                     "--limit=501876",
                     "--basedir=$tmpdir",
                     "--",
                     "--polling-interval=5",
                     "--byte-order=big",
                     "--compression-method=none",
    );

# run it and check the MD5 hash of its output
check_md5_output('a78a286719574389a972724d761c931e', $cmd);

# the following directories should be empty
verify_empty_dirs($tmpdir, qw(error incoming incremental sender));

# path to the data directory
my $data_dir = "$tmpdir/root";
die "ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# number of files to find in the data directory
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every packed file we expect to find.  Read
# these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $data_dir);

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files in root; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename
    s,^.*/,,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};
    ++$file_count;

    # do the MD5 sums match?  IPv6 files do not use the bytes/packet
    # ratio, so no rounding issue to work-around
    check_md5_file($md5_map{$_}, $path);
#                     "$rwcat --compression=none --byte-order=little $path");
}


__DATA__
268e51e4c695ea814ff20b7290c6dacf  in-S0_20090212.00
79c64fbc09168a8910087c2a91046e8a  in-S0_20090212.01
c3ff84a942c3b4830020cccdd1f1f758  in-S0_20090212.02
2d84834569f750de3cfc11845211a64a  in-S0_20090212.03
388e2d5f5d45e7f6ddda2c5488c56c1d  in-S0_20090212.04
c7279a2c2dc20f4f0e08e4146fc8d9bc  in-S0_20090212.05
05bc7b188fdc2a889c9ad5dc8cb4e187  in-S0_20090212.06
a861a66e214ff2d56ad5c0bb3b823c87  in-S0_20090212.07
d003f03943752aea94fdda983d09f611  in-S0_20090212.08
ce71239283af9ac8b4a7dd948565fc1f  in-S0_20090212.09
a2e3d161716ebbcd36a8b16bce5225bf  in-S0_20090212.10
7558fcab6103a62f963f302fec5ed0d0  in-S0_20090212.11
d844c00a6a90c4ef9cf89ffb62cee277  in-S0_20090212.12
03208534bb4b736ec2d11242e3d08234  in-S0_20090212.13
63a0bbf1ded8765a11a23cea9f5ad2e5  in-S0_20090212.14
22587df7033f8092c30ebdebe9dc9403  in-S0_20090212.15
741c23c09a155874b37ab64e69d80dd4  in-S0_20090212.16
543742bff5f67338c337cf39086db4c8  in-S0_20090212.17
94bb4fef0c527ae8e4e5789ce41c686a  in-S0_20090212.18
ce13ce6228f26e4ca76f51492fcae8c6  in-S0_20090212.19
85522bf8b398febcb3018adbf89b66d1  in-S0_20090212.20
1c5b7691a47367fb2f75a0f9d35b312c  in-S0_20090212.21
acd6218e2927c134d0d3a20de9b405d1  in-S0_20090212.22
a33bdfa882aaf014c187d1a0c850fca2  in-S0_20090212.23
55248d074fb8d503842388a7be47dfba  in-S0_20090213.00
bd0639ccc5e45867e2f919d1470360ed  in-S0_20090213.01
ff96c5d2510f483afdf00feacfbd6a44  in-S0_20090213.02
10b6d1ed04436b38c595e59e7d10eaa3  in-S0_20090213.03
347335cfe57fdcd040a6972c749a36a4  in-S0_20090213.04
0ff471d6080ac5ddd2f87b7d6bbb5f5f  in-S0_20090213.05
8b8b8a241fafe9f0784bfe1bb771cfe8  in-S0_20090213.06
1d7dd8a8551917fa1048b3cd24501b86  in-S0_20090213.07
a0b1720857f62cd0d467c0d2a2488a3d  in-S0_20090213.08
d2ea5c422db712cac195acb71ce21741  in-S0_20090213.09
d1f1ab5f071989fe0a53b91c67aafce6  in-S0_20090213.10
bdd80448fa2e393e34b9c10eb147d12a  in-S0_20090213.11
aaec8c9fb7ac138da09e9ebf5c66f410  in-S0_20090213.12
d029d840323986c716a8eb0144677ae5  in-S0_20090213.13
e5133be72f60a0a66d74fce5c5b0fac8  in-S0_20090213.14
4c9f42d8ef1726047f44b6d7316c0bf8  in-S0_20090213.15
d6ece351da5c568ddc2261e3124d222c  in-S0_20090213.16
9cf6733a44f8af93c552a9d27bd01ab8  in-S0_20090213.17
b34cba2338f2a0dfcd868720158b3a3d  in-S0_20090213.18
e0deced498abbdbc27c7d99d5aff1797  in-S0_20090213.19
6989468aa75edffe94c82180b7e118cc  in-S0_20090213.20
1897572bc71b366cd44cd09a7dc6618b  in-S0_20090213.21
cd5b0d6a6560355706c8c19e0ae24470  in-S0_20090213.22
38cf92a975c76994a7b5c21d33135c8f  in-S0_20090213.23
91ce5cc42f5b7519818980dbe256d65b  in-S0_20090214.00
2b5aab45b3fbdf40b017c0246e2576ec  in-S0_20090214.01
14e85bb971a6ea1202ffb925f79f0269  in-S0_20090214.02
184a1261b85250dffe79dfeac73fe392  in-S0_20090214.03
87d256e50247711d77b0aa4f0fecbc25  in-S0_20090214.04
2badcd3b13751bfe0ef8372047bbd57c  in-S0_20090214.05
0e8eec70bb71c5ae36a3a2f4781fdbf8  in-S0_20090214.06
de25becac2f7b2c430a93206bb43ed08  in-S0_20090214.07
0b4a572764fe92fa67dfec0d42c5cf0e  in-S0_20090214.08
17cd55210a3e1985f7ad4650011c4bad  in-S0_20090214.09
9d7951f5d23373adb4f9a307862442db  in-S0_20090214.10
65771ea69f15d1b8d2a688fc97c08d55  in-S0_20090214.11
88a5c2ea8c0fc2364734d98e0a263770  in-S0_20090214.12
191c08d0901f1ed561e7f86c31fdbbb2  in-S0_20090214.13
cdb5cb673b6280e7e1497df5b72557ba  in-S0_20090214.14
2bbae19d3cf988b65d039de2e12d17a9  in-S0_20090214.15
d3f61668beac037c4df16533de6f84c7  in-S0_20090214.16
53dd131a3ccbc7a4571cd293e3da7788  in-S0_20090214.17
04a3dd965e4926ef47bdb227a67baee0  in-S0_20090214.18
b32f69522eb8b6f83782d7acbe5846e4  in-S0_20090214.19
e4813d2a0e176272d61820697b056f48  in-S0_20090214.20
f65b31c8796276ebffc83be730f03cf9  in-S0_20090214.21
e5cd9605d202f368161fdbdd89886479  in-S0_20090214.22
bd1a5bcdc77e9e03e441d72e3654069e  in-S0_20090214.23
45a91a83a3e1b6242bb82d76a6090a5c  iw-S0_20090212.00
712871074081c50ecdb7a2a52b447001  iw-S0_20090212.01
1c7b41719c068f55048f673f5aa630a2  iw-S0_20090212.02
e6c4eee31c8c2e65021ddcc99bbec417  iw-S0_20090212.03
032b31b01d5be1226b419c71c3771cda  iw-S0_20090212.04
2697c0ce15c94d451ed8f2e9f2eb7210  iw-S0_20090212.05
4fb0a63e3bbc6309f9d5d8bb248a153d  iw-S0_20090212.06
bc7d7593e07951f0d8995c52631727b3  iw-S0_20090212.07
eed411dd35ed1dbd367638c7e065d9a5  iw-S0_20090212.08
335c339f4a2a1063e4998ceb55b8ef08  iw-S0_20090212.09
06b8e22086734618b41a8872cc92b66a  iw-S0_20090212.10
ccd56e7604fa5eca87ef9c6d4d66701d  iw-S0_20090212.11
5da1412b509c91289eb6b961f4135f22  iw-S0_20090212.12
101e1ea94d1a0b5a3dc600b20bd5717d  iw-S0_20090212.13
bcd7445258d6bf5a64687f88a81dd919  iw-S0_20090212.14
ce287240a6c581ae0095a404129ffdda  iw-S0_20090212.15
cf4cb2e6421b0e22db1f64442ace3f22  iw-S0_20090212.16
4d33dcd7bb8a4e38f8f2081cc37c41ed  iw-S0_20090212.17
212cdc1df504f58959334171292f0294  iw-S0_20090212.18
095765e73751c3bbc9f7e6a69b514f4a  iw-S0_20090212.19
33562345399d1e83a5027a899d24b87f  iw-S0_20090212.20
eb9e4964a9f8f49c39077eed7371decd  iw-S0_20090212.21
51aae8896321de2534a04ce99dc848ca  iw-S0_20090212.22
c5f6ecfbd951d36c52d77800d42386fa  iw-S0_20090212.23
8f7261e36916a04076d7c5822614192b  iw-S0_20090213.00
0bab07fd85f2dbe0b9455c6a37737240  iw-S0_20090213.01
f0f6375f04405d901eba482ccd3621b7  iw-S0_20090213.02
cda4db7daff7516193eec55690bf4bb3  iw-S0_20090213.03
2fdcd5e9f48ed230d6de14ecbe676a05  iw-S0_20090213.04
1afdee232eab87eebedc0b4505c95884  iw-S0_20090213.05
f24cef36b1b8acc47b73b1feae133560  iw-S0_20090213.06
2a0d2c8cca5b2db5f06867f7ae1a7969  iw-S0_20090213.07
607eb3705456d60894f60879815b9c26  iw-S0_20090213.08
a9b54e0896e0b99f68f85f6ab719941a  iw-S0_20090213.09
a66229681e85b3161dd67b363a014ff0  iw-S0_20090213.10
fb68a3ea109a85276ab1a78135a85adf  iw-S0_20090213.11
3f10088f3ee34c8e4a62875868b8df30  iw-S0_20090213.12
656057b5e4205090af2088f844b9a0b7  iw-S0_20090213.13
3229d26b4860db10ef7c9c8b879bd790  iw-S0_20090213.14
2507d885662b26cb2d2bee5d3f9b0103  iw-S0_20090213.15
8a5d069136b72c46a164c801a00384db  iw-S0_20090213.16
26ee294d0f4739c85213e1a5f8c4edfe  iw-S0_20090213.17
c39d77255c73bba754c7c24d6243a6bf  iw-S0_20090213.18
09ce729e77768e5ded7e90fd3ddbc8f2  iw-S0_20090213.19
7ba3908ff101f148dfcb24b4c8a74235  iw-S0_20090213.20
d14fffb96d0dcc7702310a330e8a71b1  iw-S0_20090213.21
90e0a175441926b6588a61789c32bc34  iw-S0_20090213.22
6113d4d3b8f3187b7bf40cda15314cfb  iw-S0_20090213.23
77b890285473a624d6c0c88a99558048  iw-S0_20090214.00
573d24d4ad3f69b4df2438ab56edb6ca  iw-S0_20090214.01
93fb96f5c3db8ba66e204e8188ee2a04  iw-S0_20090214.02
2512303d6e1512f80de825bb114b4c0b  iw-S0_20090214.03
e129ffec91a5680d5a3bfbabb188ce73  iw-S0_20090214.04
1613e0849ebdd4640afa4fd8649f4077  iw-S0_20090214.05
4414d7b0c72c74cf7260c61ceda9fc8b  iw-S0_20090214.06
d9bd99c82c7e2e9f164daa8fe6406212  iw-S0_20090214.07
80fa3311b685df8c9cb93bc2fc101fe5  iw-S0_20090214.08
287912370577cf4fe8ad32dcde934dfa  iw-S0_20090214.09
153bffc8df8fee2a5d8981b23dc6c5c9  iw-S0_20090214.10
c75d657c23fb34386ce45b188b433ed0  iw-S0_20090214.11
4b2f5fdf4030fa595fb0c0a97227a498  iw-S0_20090214.12
c1abd33ba4bfeab0a17152a6b74ba535  iw-S0_20090214.13
d21d5964746ec35ce07303b61a175aa7  iw-S0_20090214.14
4dfb8ffc7d737f2797c180939a2fc674  iw-S0_20090214.15
0e5f9f081f80dcd63d819f0c5289a492  iw-S0_20090214.16
884e722f09619552816b2e3a5347c259  iw-S0_20090214.17
6d2cd50f8f678882d0394659cd692641  iw-S0_20090214.18
b9a5c3fa26d206d18c657867f48ad0d8  iw-S0_20090214.19
5e77130428270fa3e0904912c4d47037  iw-S0_20090214.20
47d03860f8e4c0c2918a4a6e49837c51  iw-S0_20090214.21
1c520425807199d67816e62ac66224b0  iw-S0_20090214.22
fc44d34c64d88bed699d6b099141fa49  iw-S0_20090214.23
bbe2db4f1420444364143b3186b9145f  other-S0_20090212.00
f380d176e8402b81f9059016ba3ed613  other-S0_20090212.01
bb558fbdf40d417b6f99d287479978a4  other-S0_20090212.02
e07d0ed84536144aa3aeefec77b3e65e  other-S0_20090212.03
5d72696f8f39cac3ba76471f07b4ec64  other-S0_20090212.04
6708de2da4cff516dcbe967ddebdc3a2  other-S0_20090212.05
90884542314e611e3f226defae14d37e  other-S0_20090212.06
c771511872499a4fcdc25a9766763ec8  other-S0_20090212.07
8bf0e46feede12f4e9ed5d5aad879a4c  other-S0_20090212.08
929c43cc40eba79474c7b0eba75627ae  other-S0_20090212.09
19f6c175274cb54a02633cb5869d3ac9  other-S0_20090212.10
f54dedca462c5485b2c53d87fb0a8bce  other-S0_20090212.11
44a568f220715492cf4e8ac2e8a138e5  other-S0_20090212.12
0e5655dc1d64b041df85ba9cb1a775bd  other-S0_20090212.13
a7f99a9a7205c7d3ccbadf0abf63169c  other-S0_20090212.14
7e9ada85fd471e8d14ef7deb5d799a05  other-S0_20090212.15
a963ac365696baa8bd5c9a4600ccbe46  other-S0_20090212.16
a857c595d2293d027528ded5fc295547  other-S0_20090212.17
d1ed4d1182a66b8ace38c002a27df595  other-S0_20090212.18
7f80bef5ec23d25233b024c2eccba3b7  other-S0_20090212.19
c78935f50e4356fddad278dbe43bcc65  other-S0_20090212.20
4bbaa50370fd9bcf5ad26fdc352a3d2f  other-S0_20090212.21
fc2b68d8f477775794676601bcb59f56  other-S0_20090212.22
5f5d5205b5e6fe6a6e1737e925302c37  other-S0_20090212.23
7b203b9d446a8ec3d8c720f205498e8c  other-S0_20090213.00
603ea9e7d75de37446f9443d7b27db7f  other-S0_20090213.01
1ede498b1460eb03c77172f0a225fa97  other-S0_20090213.02
b72563603cf70ced21816b972c0ad325  other-S0_20090213.03
f2c7805ed73826c00db9c17eb621a572  other-S0_20090213.04
0d71521f66e3cfa8f969c377bdf62ba0  other-S0_20090213.05
343d87a6395f600757eafa7017b5a3b7  other-S0_20090213.06
ef8530a47876653ecd6668fb77491453  other-S0_20090213.07
a13a214943f371d2c5dd79f0978ee75d  other-S0_20090213.08
e9fc7ea632a535745189ad8f1f136cc3  other-S0_20090213.09
5d9b965d7b705251f2b3df3ae6745630  other-S0_20090213.10
fdc83ab0675ac0a81843489dbe8024d0  other-S0_20090213.11
2876747b8322ae287122e726f65be27e  other-S0_20090213.12
a3c2b90f56e1444881b8c32df2443db0  other-S0_20090213.13
bf64a47305e4c716decc70e8047d5e22  other-S0_20090213.14
c4149d52dfa77bfc02581a7bc438925a  other-S0_20090213.15
647d923aabd448c334cae1ccf9f756be  other-S0_20090213.16
c4f5698a2bf61982ebe4dca513a6f1c1  other-S0_20090213.17
58706ce329661e4336ade93e04d35c84  other-S0_20090213.18
c01bd24f302882ed1dd8ab915a914d8a  other-S0_20090213.19
efa50bcf610c6b77894283042eea8aca  other-S0_20090213.20
c32bc124fa1309c2358c7a618f2da9b7  other-S0_20090213.21
3e5fbf153a4796bb70c592765c596bef  other-S0_20090213.22
a4b4c1fb4530bbca81b5f73500db4030  other-S0_20090213.23
e73c5cb4b0ddf681c6489ba5fda1cd68  other-S0_20090214.00
ff056242cc9571a8a8baba3059860d48  other-S0_20090214.01
9c82a0202b852b22d540654f5bb320c2  other-S0_20090214.02
7820140506eb33eb6e2ccaf912b66516  other-S0_20090214.03
a1a01f8f6e39dbcf065bf6cd693bcabc  other-S0_20090214.04
15f7349a9f93954b66b5bd178e7990ff  other-S0_20090214.05
c2c9ebcb79fdca875cf653767d6a0c03  other-S0_20090214.06
4941c2468efe9c9b89c58160f9ffefe5  other-S0_20090214.07
b4feb9c5dd253caf08161e9360d90602  other-S0_20090214.08
d81f0925da5088943e933b4393afac6d  other-S0_20090214.09
9e2f0cf53c635d707794538d5569b329  other-S0_20090214.10
7b166a9d3ea57c0855a3358ef4c79f7b  other-S0_20090214.11
bc6619381ed335b4684f26cceca05000  other-S0_20090214.12
7d6b83f9a4d51465449e02520b61551f  other-S0_20090214.13
51c2ea9f4caeaa4d16bc4dd67776ac5d  other-S0_20090214.14
97ec5dc6c68d3e3f76c3565c80c387c7  other-S0_20090214.15
29b7d36a589ebe29ba54748a79843a56  other-S0_20090214.16
fb51f6de6af6170dad28a5a83a3bc4c5  other-S0_20090214.17
b4bad3e9d818e09acb1849dda014d269  other-S0_20090214.18
5533a0550a2acde4eea8a01b09b81dc8  other-S0_20090214.19
53ba5e79942ccb468303069415a458fc  other-S0_20090214.20
a9cb008367b219e62549c8e173857476  other-S0_20090214.21
bdf9c7805235c21496d26f7a6ef1cf2f  other-S0_20090214.22
30c22d91061bdeff2b9d8cdd4d5bce02  other-S0_20090214.23
900f0abd5cc5bbf3bbd29606947ca1f7  out-S0_20090212.00
88d0f82080b95fcaef40cbbe952f9d85  out-S0_20090212.01
aa49adc4cfe507fa122612031ff12ebe  out-S0_20090212.02
086dd5e7483c0858491982bf369af3ba  out-S0_20090212.03
c9d51c4c21571a8171d08c01ca4e375d  out-S0_20090212.04
9de73b2f2e378f0508c55211ca974e75  out-S0_20090212.05
57586028f3db1b9569011f3ecd567be9  out-S0_20090212.06
c6b180327c9b46c97ca0cc146b721dc1  out-S0_20090212.07
a8495657388cdcc7c29f676d93b43058  out-S0_20090212.08
9cfb0f911acc07e02da92983097ca890  out-S0_20090212.09
2d876b4cb8433fa46a65bb1dff622654  out-S0_20090212.10
49e8b1b5510f42c09150627a1dd383b5  out-S0_20090212.11
ad515ae4de379a2293baf99a5ae91283  out-S0_20090212.12
6183bf577c0c42ba5f7387ac0b3b1fea  out-S0_20090212.13
0d9449d12339abdb64d20a7d96182daf  out-S0_20090212.14
085c5cbcc8cbe5043edee838aab56528  out-S0_20090212.15
9fa048e85d0d88744fd13e63aca9c308  out-S0_20090212.16
8ba17905a32803481cf61b55de4b44c4  out-S0_20090212.17
d9d86aafcaaf62d845427d132f6593bb  out-S0_20090212.18
76c1a64983a99543d127945b97153d5e  out-S0_20090212.19
eb667970096a863b5ed98b8de0f1450f  out-S0_20090212.20
e4c4a02fb0a18d0a472233490eb7f3b2  out-S0_20090212.21
2eea23e4f8a3542688ac18f1ea837236  out-S0_20090212.22
96ff665bb4528addcebdc600226b1937  out-S0_20090212.23
9aac0a57f0654e68ba12ed5d552e6936  out-S0_20090213.00
cab4f89a16d06c38bf7072d95ae520c1  out-S0_20090213.01
73a629971b10c4e196376c53bfcc2a41  out-S0_20090213.02
70f1cccd0b326e6251ad75808b36cf8c  out-S0_20090213.03
871efe75779e5edbf44aab21e6dfaff3  out-S0_20090213.04
d02e6d8927bca739fcd5c1c54f1c000f  out-S0_20090213.05
7ace277920c09d4de574150f320393ac  out-S0_20090213.06
fca86c505eba6647617a35f8cd0f2e30  out-S0_20090213.07
c624c4db35e78cbdb2a7842bb0099136  out-S0_20090213.08
70f84f438894756eb5d730f58f8e486a  out-S0_20090213.09
2d1dc9a9a3cca81c47f9a95425e0ffa1  out-S0_20090213.10
b0d8a4b5e0f38b1defdbb6fb51f4cdf5  out-S0_20090213.11
d6b69d40dfb6ff818cb9dfba8fec11c2  out-S0_20090213.12
59b2659f78c675f01cc14f0c8aa50487  out-S0_20090213.13
25d4187e7ce01acd523c96b9be73bd74  out-S0_20090213.14
1c04877f9566e559cb8ba21b7b3c283b  out-S0_20090213.15
e48bfc3e270ba9ac8d917922153f73b5  out-S0_20090213.16
1481d18d3648d78ee373e3837012f42f  out-S0_20090213.17
d4a62b6d9be49614b7c3b0a000e8ce24  out-S0_20090213.18
da1883938e0d8b53685149df32af7d81  out-S0_20090213.19
a7d6908e13319c149d718f32b5fe591c  out-S0_20090213.20
52898dc9e46008c176b957d2b30fc9e2  out-S0_20090213.21
c84271ae6e287e7fc9322d4f7674ad89  out-S0_20090213.22
258932ffae7e86238d49c64619bea2f5  out-S0_20090213.23
c433e5d160bcb848ec13c61e53eac971  out-S0_20090214.00
3fe0c39b1d260e37e5578b075d51fb40  out-S0_20090214.01
bf0ae898a77bdaaf6ce850b7c3b4d6b0  out-S0_20090214.02
d41e61b20f8871ddaaf38744376a6b97  out-S0_20090214.03
7490bd37e132b0aaa301a2a2eeb753a3  out-S0_20090214.04
314d8ad8dbcbae6d0b6c114bfafd03b2  out-S0_20090214.05
d6545c4997bae4131966b01717e3e6c1  out-S0_20090214.06
88150fc5da99d4ed2cfabe87b25c2f19  out-S0_20090214.07
f9f2a5c384b8eebf6445741d46d47b02  out-S0_20090214.08
1aacc1f896579491fc0ce20754772730  out-S0_20090214.09
4f3bc9ff38da6319a5157b6dd14d325a  out-S0_20090214.10
e694f523318315212989a87f2b344599  out-S0_20090214.11
487b1e739d6a5486226444ccbaa47767  out-S0_20090214.12
e204015b58f6c394f825f74ec7af11e1  out-S0_20090214.13
db951d687539dbba6055b95ac560d72e  out-S0_20090214.14
33b4f719279f2b295ed1ac75a0619a45  out-S0_20090214.15
c7745ce11b6625258d0b4bee0408626f  out-S0_20090214.16
a9cd25fd9d22db90f13bb3f2b4264101  out-S0_20090214.17
d213a4e0a70e83956ea1433cdd487845  out-S0_20090214.18
4a75e04d81fbe2d4ba42f05194b19526  out-S0_20090214.19
21961bb0ad313867682568cbeff7cee2  out-S0_20090214.20
7e545ba2a0b1954a8cd18929f1c4476c  out-S0_20090214.21
f01cf05e8c2d3f63620b4df4c3aba8d6  out-S0_20090214.22
e261d92ea205aa92df42a47f95403d93  out-S0_20090214.23
c68fb03afd17ebbe7e49b78953f1ae2b  outnull-S0_20090212.00
beb99eba4040200d596a888d4544664e  outnull-S0_20090212.01
ce1f0ee0e9d3ff84ff344545524d95f7  outnull-S0_20090212.02
c023ac3ecec0a38efc74c4c68a4d1913  outnull-S0_20090212.03
8f591ec0a55f9560b2061c05940d81bc  outnull-S0_20090212.04
3b663948b46cae460f52bff601126572  outnull-S0_20090212.05
5a1a0c865c7108ac6fd913ba30f6f7a2  outnull-S0_20090212.06
662e45f9c73dea6c92733f79cd50a16b  outnull-S0_20090212.07
49ab05648f46e898a2e209b97a5f9da4  outnull-S0_20090212.08
3a0374e34ecec94642164950cb77b948  outnull-S0_20090212.09
9be51f644d4ecb7a496a65b9611ad781  outnull-S0_20090212.10
ab448f5b3302401c76ea8fc98d203b1b  outnull-S0_20090212.11
ec6802ccf3916052a58cd53521eb2b57  outnull-S0_20090212.12
e1a510e12453bf8d35ba7260917177be  outnull-S0_20090212.13
ef6490e64abd2e926fb769c4137e550d  outnull-S0_20090212.14
531505719281794c5f07526d04c103a6  outnull-S0_20090212.15
f63b99e82ade72397dabea632da9bc42  outnull-S0_20090212.16
63ffc46ab2ee4d09a7405bc1e143f19b  outnull-S0_20090212.17
b64fdee01989c10fad49bf7efb0f51b3  outnull-S0_20090212.18
5674bd945dfeb97d5ea90d5479b6a401  outnull-S0_20090212.19
1200cff29a8b44e96c6261f6b04ede1d  outnull-S0_20090212.20
4849f2f384db599dda28d264e61e46ea  outnull-S0_20090212.21
b738c4dbc5c0cba16149b114d8e34f73  outnull-S0_20090212.22
e5b9950657e5ba8f181f6d7f7a847cf8  outnull-S0_20090212.23
b3bc807c6013b16a43b5074385546539  outnull-S0_20090213.00
079c6e655dc94f10b28d0c99c193c738  outnull-S0_20090213.01
64d70599a38dd21028b8874948c351e3  outnull-S0_20090213.02
9bcc6288b68d5d7c21eac3e4795fc28f  outnull-S0_20090213.03
8fd60c6d92d2c338c9a52fe9ad7d2015  outnull-S0_20090213.04
9c736f0f60e9b3d703666d4d8c20b847  outnull-S0_20090213.05
c676ac8a448c39adb5440390f2cfa0ba  outnull-S0_20090213.06
ca958fa48b3bc15b9f884ca1f6de2066  outnull-S0_20090213.07
e44a888cea9d4217f0d10d45dd5fcdf8  outnull-S0_20090213.08
a357fe09d459dc5b5b1d94e97afeb55e  outnull-S0_20090213.09
2a7cd75ff7504822918290eedccc2f46  outnull-S0_20090213.10
089c164a6029d04b2e116d5df6f543fa  outnull-S0_20090213.11
611dbb2f14b8f2ad7c0c24115a937f64  outnull-S0_20090213.12
d9ec992415cf8e69e45d062f8b44532a  outnull-S0_20090213.13
63641cc669f891f9b2704ac0b38ad137  outnull-S0_20090213.14
cd1a1182cd131df987065b6e438e7ac3  outnull-S0_20090213.15
82ff3fbd95fc6ce3ca6a8c05d8bc9ddb  outnull-S0_20090213.16
f3c5aed6de25c8f04bb28731e0099986  outnull-S0_20090213.17
1efda1e2f6edfbcb224bcfb03fecd35a  outnull-S0_20090213.18
b8318182aa939c5fac106bce73d72ebf  outnull-S0_20090213.19
13bd2ad3c7e781b854e3290edbd2bdb5  outnull-S0_20090213.20
3d4973b9ecc3c9caacb776cfe892b58b  outnull-S0_20090213.21
2d960fb0380b7d9a116e4366c437e1d0  outnull-S0_20090213.22
f463e3c6c1e2d7756142256669287445  outnull-S0_20090213.23
d797c1596e233c254b411d564108cd56  outnull-S0_20090214.00
9da2b07c64fa567fa3aec6431a8c1ef1  outnull-S0_20090214.01
5ef5b09031c08dc6ec40f93ac940e05a  outnull-S0_20090214.02
2d860260129bb17995d1c2f470dafdb7  outnull-S0_20090214.03
a8744155aa800e11ac2aac177d6542b3  outnull-S0_20090214.04
bb101252510d2acee840b83ad4e46b94  outnull-S0_20090214.05
c6b1a29fa03da78fef981927f065ed28  outnull-S0_20090214.06
677fd4a91833bbfbe741c538404fab26  outnull-S0_20090214.07
77bade8dabd1349f724410b998156bc2  outnull-S0_20090214.08
90cdfd1c277dac48fc2de797f8818aad  outnull-S0_20090214.09
8764c4e49917831612119010cc2f78f8  outnull-S0_20090214.10
bf0a269b4ded04e6199c83d56f9fb86f  outnull-S0_20090214.11
12d1607cabce733b8bf2a55113be798a  outnull-S0_20090214.12
6473e7c58e068ecccfed718bae47481b  outnull-S0_20090214.13
b48e7d76beff2c681c4566bd2a92d0ab  outnull-S0_20090214.14
05c64c09cb28491a8045b9cf2224e43b  outnull-S0_20090214.15
3a64514e4e81c45988a11a7f6f32ac1d  outnull-S0_20090214.16
c1e32645b9c3fa61ed900729e19032a5  outnull-S0_20090214.17
4c072360b87fd4499328b6deb4d93882  outnull-S0_20090214.18
69f45a98233e517d998b0893f0bdf340  outnull-S0_20090214.19
cbb2e6113ceb8f5dffd5b8eeddef9e4e  outnull-S0_20090214.20
014b3a4aaa6ced9f4b51b2e2ed485c2f  outnull-S0_20090214.21
abff3ccf96f58bafcbd6e7e7c6bd4c19  outnull-S0_20090214.22
00819f54ae77818e4cdc37cefbd7f5b6  outnull-S0_20090214.23
0833b79ec05445b70a585e8141400e8d  ow-S0_20090212.00
07f90d75f73b3735a97f2c86bc25bc2b  ow-S0_20090212.01
feecfbcf6191f1bd6c3db84d31eb8044  ow-S0_20090212.02
3bd2d830fac1ba98a79bcc02592968a5  ow-S0_20090212.03
09cee513afb80bf0b63f28eda7c8c99c  ow-S0_20090212.04
08c9a5f84d93205b71ba0e7728a666a7  ow-S0_20090212.05
6fa3263153efc78e44077d874a4f381b  ow-S0_20090212.06
29efd3b9b73e9f35d8c19d7b2cfc64a1  ow-S0_20090212.07
31d8a378c5e1e396748456a99581642f  ow-S0_20090212.08
14f8f73425061d24c4f8e19a0dfc6305  ow-S0_20090212.09
d18d5c2fbb6a11ebafae4e2a65cff51e  ow-S0_20090212.10
eccd7e3421c9c9b467b25b954fa1cd18  ow-S0_20090212.11
5a24c9c9a36842db3e7f04c6cc299bb3  ow-S0_20090212.12
aef58535dd9d97f8b712b5d38c603c0c  ow-S0_20090212.13
d77cdd9e4753bb5dfa422f03c099d186  ow-S0_20090212.14
727a24c2f167486cc4b30db2a7057128  ow-S0_20090212.15
ed76ba320f60bb3b258bfe1684287378  ow-S0_20090212.16
72632bb815da20bb26fd2f96cab28a07  ow-S0_20090212.17
ea8a833f55665721c13e7aade5c57010  ow-S0_20090212.18
5e2d323f27eb8a7a2dd156a85bd2aaaf  ow-S0_20090212.19
a4acc0d965c26325960d63096c948bf1  ow-S0_20090212.20
f4474de7b630cda444b471aabe6c026e  ow-S0_20090212.21
7608f2deccb84e1bcac965a1ce7bc5e7  ow-S0_20090212.22
5c98a79fb1beca84173f1265750e662e  ow-S0_20090212.23
31d5892630ae59117480ec70f0dddc06  ow-S0_20090213.00
43c541c93ec3e27d08cfdb9f5d8566c4  ow-S0_20090213.01
2db6e17511707f78d9078e1510371ecb  ow-S0_20090213.02
3dac91633725e1cca02910ab474c9a67  ow-S0_20090213.03
e62be48ea8f7c69a751c4924d9e3acca  ow-S0_20090213.04
a3257959edd258268e362b55edd8be96  ow-S0_20090213.05
7ce40d2751bb4348bb3bc5141d02a5d9  ow-S0_20090213.06
675c4a9aefbe4ea4facb0ca2244640fb  ow-S0_20090213.07
56b11e6e036646d6a93597a5e5bf4c14  ow-S0_20090213.08
9308734dec4f89753c8666cd9fa6fc39  ow-S0_20090213.09
197cd7cb3cba46d44cecc3be4f1bd544  ow-S0_20090213.10
04868dce36a4cbbf56e14acfbab98d5c  ow-S0_20090213.11
6f62771d2466a0815d047dc2e6a4a79b  ow-S0_20090213.12
3ce1272b6adb21fc3245b837d34dd780  ow-S0_20090213.13
a5dd3675055c1c96a84897e9b888880a  ow-S0_20090213.14
c20070bfcf5e235c566875edc2ee2026  ow-S0_20090213.15
5b2a483ccab444e6d3ade78edb16a539  ow-S0_20090213.16
0fddf69cdb3193a07d9e3f118b7d7ab6  ow-S0_20090213.17
ef1219e34673e90a3aa3dadef3ef0b54  ow-S0_20090213.18
8686bed8e77ba1d3d48521f7060d7203  ow-S0_20090213.19
db0b7ec66d3a71954f43dc811169fd40  ow-S0_20090213.20
ffad367b75bf34ce1af65dce6d2bf16c  ow-S0_20090213.21
554808e653147605df392d4d76b19912  ow-S0_20090213.22
29f6f6f2e3f01a3105b742b6dd9bf6ba  ow-S0_20090213.23
ca68434fd4ce22a8b7b6b2923fa913e5  ow-S0_20090214.00
ea887a738936eca8100542dd6f233362  ow-S0_20090214.01
7b87814680ba425adaa6f79ddb26f73a  ow-S0_20090214.02
33ae794754a32bbf2db643e9eca0a17e  ow-S0_20090214.03
2882b4098d0d95363bb136f0b7d3f52b  ow-S0_20090214.04
8bcf7481dcd6446b4fd998db7885e055  ow-S0_20090214.05
67a530ac9ed04572c14b2bffec4f9f81  ow-S0_20090214.06
4b48c4225cfd76aa4694b0f87a08d158  ow-S0_20090214.07
fec050c3e70d9927532c59b095892a60  ow-S0_20090214.08
a60a78523ab7035ff8eee4afce39f3b3  ow-S0_20090214.09
969fa9487bc017875095272bef383f15  ow-S0_20090214.10
cdd7d39e38597e7bd1b500b89bf12ce9  ow-S0_20090214.11
6a1db4322505280b57e232ee5b8418a0  ow-S0_20090214.12
58bb0024d67b0d5408ec549bbb6e9aeb  ow-S0_20090214.13
0d5c70ea854ec3d89ee8d69d9cc763b8  ow-S0_20090214.14
c0a27b6076d9281f9143789cabca09cc  ow-S0_20090214.15
f917de62cfa3e0dec1be418c5d422a82  ow-S0_20090214.16
10c922109bf215431114b7c47eb98d9d  ow-S0_20090214.17
90e73d49745e29ad05b3f0dc9ada11ce  ow-S0_20090214.18
9bd50e1e3278d25db6fcd374e514c0e4  ow-S0_20090214.19
21ff60734cbb5085dc929ec897f9ea01  ow-S0_20090214.20
f18b633c03ca12fb877f85b0666cb3ae  ow-S0_20090214.21
1c1490fa40e028b48539bf7c4f73f0b6  ow-S0_20090214.22
8700fe19de6925bb8fb44113d51f236d  ow-S0_20090214.23
