#! /usr/bin/perl -w
# MD5: multiple
# TEST: ../rwfilter/rwfilter --stime=2009/02/13:20:00-2009/02/13:20 --sensor=S2 --proto=6 --aport=80,8080,443 --pass=stdout ../../tests/data.rwf | ./rwallformats --no-invocation --basename=/tmp/sk-teststmp && md5 /tmp/sk-teststmp*

use strict;
use SiLKTests;
use File::Find;

# name of this script
my $NAME = $0;
$NAME =~ s,.*/,,;

my $rwallformats = check_silk_app('rwallformats');
my $rwcat = check_silk_app('rwcat');
my $rwfilter = check_silk_app('rwfilter');
my $rwfileinfo = check_silk_app('rwfileinfo');
my %file;
$file{data} = get_data_or_exit77('data');

my $base_name = 'sk-teststmp';

# create our tempdir
my $tmpdir = make_tempdir();

# create subdirs for writing and reading with rwcat
my $write_dir = "$tmpdir/write";
my $rwcat_dir = "$tmpdir/rwcat";
for my $d ($write_dir, $rwcat_dir) {
    mkdir $d
        or die "$NAME: Cannot mkdir '$d': $!";
}

my $cmd = "$rwfilter --stime=2009/02/13:20:00-2009/02/13:20 --sensor=S2 --proto=6 --aport=80,8080,443 --pass=stdout $file{data} | $rwallformats --no-invocation --basename='$write_dir/$base_name'";
if (!check_exit_status($cmd)) {
    exit 1;
}

# get list of expected MD5s for each file from the end of this file;
# these lists include the MD5 of the file itself (stored in
# %checksums) and the result of running rwcat on the file (stored in
# %readback).  some files have multiple checksums due to differences
# in LZO or other options
my %checksums;
my %readback;
while (<DATA>) {
    next unless /\w/;
    next if /^\#/;

    my ($expect, $tail_name) = split " ";

    # Handle checksums for reading the data with rwcat
    if ($tail_name !~ /-c\d+-[BL]/) {
        push @{$readback{$tail_name}}, $expect;
        next;
    }

    if (exists $checksums{$tail_name}) {
        push @{$checksums{$tail_name}}, $expect;
    }
    elsif (!$SiLKTests::SK_ENABLE_ZLIB && $tail_name =~ /-c1-/) {
        # skip
    }
    elsif (!$SiLKTests::SK_ENABLE_LZO && $tail_name =~ /-c2-/) {
        # skip
    }
    elsif (!$SiLKTests::SK_ENABLE_SNAPPY && $tail_name =~ /-c3-/) {
        # skip
    }
    else {
        $checksums{$tail_name} = [$expect];
    }
}

# hash to store names of files that had an issue; value is an array of
# possible isues: 'missing', 'file-mismatch', 'read-mismatch',
# 'unknown-file', 'unknown-readback'
#
my %mismatch;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $write_dir);

# handle files we did not find by adding missing entries to %mismatch
for my $missing (keys %checksums) {
    push @{$mismatch{"$base_name-$missing"}}, 'missing';
}

# print results: Files are sorted by the reverse of the name so all
# differences across a wide range of file formats and versions are
# seen first.
#
my @badfiles = sort { reverse($a) cmp reverse($b) } keys %mismatch;
if (@badfiles) {

    my %issues;

    my $msg = sprintf("%s: Found %u file%s with problems:",
                      $NAME, scalar(@badfiles),
                      ((scalar(@badfiles) == 1) ? "" : "s"));
    print STDERR $msg, "\n";

    for my $file (@badfiles) {
        for my $i (@{$mismatch{$file}}) {
            ++$issues{$i};
        }
        print STDERR join(" ", "$NAME:", "$file -", @{$mismatch{$file}}), "\n";
    }

    my $issuemsg = join ", ", map { "$_: ".$issues{$_} } keys %issues;

    die($msg, " ", $issuemsg, "\n");
}

# successful!
exit 0;


# This function is called by File::Find::find.  The full path to the
# file is in the $_ variable.
#
# The function checks for the file in the %checksums hash.  If not
# present, an entry is added to %mismatch for an unknown file and the
# function returns.
#
# Files in %checksums hash have their values removed from the hash so
# mising files may be reported at shutdown.
#
# The function computes the MD5 of the file and compares it to the
# expected value; if the values do not match, the function puts the
# file name into the %mismatch hash with a file-mismatch.
#
# The function runs rwcat on the file and computes the MD5 of the
# result.  If that does not match the expected value, the function
# puts the file name into the %mismatch hash with a read-mismatch.
#
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;

    my $path = $_;

    # file name without $write_dir
    my $file = $_;
    $file =~ s,^\Q$write_dir/\E,,;

    # set $tail_name to be the varying part of the filename; that is,
    # remove the directory and base_name
    my $tail_name = $_;
    $tail_name =~ s,^$write_dir/$base_name-,,;

    # set $read_name to be $tail_name truncated at the version (that
    # is, minus the compresssion and byte-order)
    my $read_name = $tail_name;
    $read_name =~ s,^(\w+-v\d+)-.+,$1,;

    unless (exists $checksums{$tail_name}) {
        push @{$mismatch{$file}}, 'unknown-file';
        return;
    }

    # file checksums are stored as an array; delete the checksums for
    # files we find so that we may report missing files
    my @sums = @{$checksums{$tail_name}};
    delete $checksums{$tail_name};

    # check file itselef
    my $file_md5;
    compute_md5_file(\$file_md5, $path, 0);
    if (! grep {$_ eq $file_md5} @sums ) {
        print STDERR "$NAME: $tail_name: file checksum mismatch [$file_md5]",
            " (expected ", join(",", @sums), ")\n";
        push @{$mismatch{$file}}, 'file-mismatch';
    }

    unless (exists $readback{$read_name}) {
        push @{$mismatch{$file}}, 'unknown-readback';
        return;
    }

    # check reading
    my $read_md5;
    my $read_cmd = ("$rwcat --byte-order=little --compression=none"
                    ." --ipv4-output --output-path=$rwcat_dir/$file $path");
    check_exit_status($read_cmd, 1);
    compute_md5_file(\$read_md5, "$rwcat_dir/$file");
    if (! grep {$_ eq $read_md5} @{$readback{$read_name}}) {
        print STDERR "$NAME: $read_name: read checksum mismatch [$read_md5]",
            " (expected ", join(",", @{$readback{$read_name}}), ")\n";
        push @{$mismatch{$file}}, 'read-mismatch';
    }
}



__DATA__
b44a4d16273f868f907b081519eacb0c FT_FLOWCAP-v2-c0-B.dat
aad19fce22233c7766afcb5bacc72148 FT_FLOWCAP-v2-c0-L.dat
62d46f91eec4f70bac9cc5dfe0882714 FT_FLOWCAP-v2-c1-B.dat
735b483a42705c8ed0338290af7058fd FT_FLOWCAP-v2-c1-L.dat
17328ebc0908190bd40d55f25bc23dc6 FT_FLOWCAP-v2-c2-B.dat
17452fc1de6627ec5bee7c9943bd08b3 FT_FLOWCAP-v2-c2-L.dat
fe6d1d23029632ebf082705a16332077 FT_FLOWCAP-v2-c3-B.dat
25ad9a986d62feaf6062cf02fe0f6733 FT_FLOWCAP-v2-c3-L.dat
cee1b1f74822b31a7441a8211fa5bfed FT_FLOWCAP-v3-c0-B.dat
a93c68d3143fe1f5c341f56ab860492b FT_FLOWCAP-v3-c0-L.dat
86dca7d5924c99c676f2613b9a655fdc FT_FLOWCAP-v3-c1-B.dat
ffe123fdb7dd8e01099575c3f9219ba3 FT_FLOWCAP-v3-c1-L.dat
2aab9a2a193d09ac293e5a60c0853a73 FT_FLOWCAP-v3-c2-B.dat
085b796186e4392ff8df1436bf5b57f0 FT_FLOWCAP-v3-c2-L.dat
d805f91ec2f50f173df4804fccaf7c13 FT_FLOWCAP-v3-c3-B.dat
5efaf2bdb2c3bc24f41a936464d53208 FT_FLOWCAP-v3-c3-L.dat
ca48fd33ad7dd52b39b2fd7845c5068c FT_FLOWCAP-v4-c0-B.dat
be8884450fa255feafb1c29237543abe FT_FLOWCAP-v4-c0-L.dat
c18e37c1d921c4085a9a468c034d4cec FT_FLOWCAP-v4-c1-B.dat
a816fd63f0e0679318620f6df349ff1a FT_FLOWCAP-v4-c1-L.dat
cc5788bf9bd9170c8e1866133b9db791 FT_FLOWCAP-v4-c2-B.dat
7268776400d7f2e33b9be03dd119a989 FT_FLOWCAP-v4-c2-L.dat
5dab0633d7c8ba725d655571924fba00 FT_FLOWCAP-v4-c3-B.dat
d6ca894b19d402c8edfda2f50d82a694 FT_FLOWCAP-v4-c3-L.dat
7528be67c52e6f0dd5752ef39d9a8c8e FT_FLOWCAP-v5-c0-B.dat
a85ad2f2bdb42130027602e743bdc04c FT_FLOWCAP-v5-c0-L.dat
adb12ec96e46e3d7afd7cbb8dc2b0260 FT_FLOWCAP-v5-c1-B.dat
345a487a83ff21b06dd5c4b50446998a FT_FLOWCAP-v5-c1-L.dat
375dbf3a0b1d7eb32d2873405ed78f20 FT_FLOWCAP-v5-c2-B.dat
f52996762d83f050fe0e95bbb47c0a16 FT_FLOWCAP-v5-c2-L.dat
cad0d5f8e7e3a70f2699252932d8b132 FT_FLOWCAP-v5-c3-B.dat
911b7081ec8fd2c6a9cac2a5159d59a3 FT_FLOWCAP-v5-c3-L.dat
c1cc1ca72b1732d6ffb2f2c2085d4c89 FT_FLOWCAP-v6-c0-B.dat
02580d776eb78efbc269641247589afd FT_FLOWCAP-v6-c0-L.dat
259634b5542115cd8fb92ff156eaca2c FT_FLOWCAP-v6-c1-B.dat
16c8f6e3d0792de285a86af6b00ed723 FT_FLOWCAP-v6-c1-L.dat
8cf761c5a5f90a5ee76e2302d61946c4 FT_FLOWCAP-v6-c2-B.dat
64329ca995c82e4f2df92af5338c4693 FT_FLOWCAP-v6-c2-L.dat
e96c4ee5ebe6fed9279d411d282f2bec FT_FLOWCAP-v6-c3-B.dat
63ae160c0b59d1b4c7a239730661bead FT_FLOWCAP-v6-c3-L.dat
4a50279ac5644e9303abed48e86932e9 FT_RWAUGMENTED-v1-c0-B.dat
3bf9a754b721f9baed5430524a81367c FT_RWAUGMENTED-v1-c0-L.dat
a5e5d3717c3ff3392677b15620167b9f FT_RWAUGMENTED-v1-c1-B.dat
45cac95bc7c791923aab9feaabfa2c5b FT_RWAUGMENTED-v1-c1-L.dat
b1f2ef65432af17c8c23eae012e5e677 FT_RWAUGMENTED-v1-c2-B.dat
8055ba43f57bccb4237d21abd815949b FT_RWAUGMENTED-v1-c2-L.dat
cf6d053719435ef62cd666966be412da FT_RWAUGMENTED-v1-c3-B.dat
36e77f1bb6b8fd3d2d26d3436e53468e FT_RWAUGMENTED-v1-c3-L.dat
373b4efb3dcd689d43bd1a872d8a0b82 FT_RWAUGMENTED-v2-c0-B.dat
e49b4d08689b220153101d6a306fa144 FT_RWAUGMENTED-v2-c0-L.dat
cd2058d07fdf08190ddd1ba70dc9ed19 FT_RWAUGMENTED-v2-c1-B.dat
49476b3038a4dec8aad8de2bb731d2e9 FT_RWAUGMENTED-v2-c1-L.dat
42377ff91107814405349029e54c96a5 FT_RWAUGMENTED-v2-c2-B.dat
f21bcd9349a304f81195f947b151880b FT_RWAUGMENTED-v2-c2-L.dat
30ab357f7685d8e8c3308a7a6b7cb422 FT_RWAUGMENTED-v2-c3-B.dat
7eaa5ba20e024652e53b13d9da0cfed8 FT_RWAUGMENTED-v2-c3-L.dat
b2bd70fb60cfcb7f61e928508b2fcbb0 FT_RWAUGMENTED-v3-c0-B.dat
dd3828107a232b67e074d3b2883d2ca1 FT_RWAUGMENTED-v3-c0-L.dat
564cadb0881704e57e641acdca8d2505 FT_RWAUGMENTED-v3-c1-B.dat
55febe0cad038fc447683285a22dffda FT_RWAUGMENTED-v3-c1-L.dat
028414eca3693c58658c21a5d74a0756 FT_RWAUGMENTED-v3-c2-B.dat
5b3da2eb1dbb9720ae439bbaaaa8d2d7 FT_RWAUGMENTED-v3-c2-L.dat
c5d01f2efe0f0d4029cb6ad6c8cb2baa FT_RWAUGMENTED-v3-c3-B.dat
61200d95966950d63c156876ad2d3b8b FT_RWAUGMENTED-v3-c3-L.dat
262f51d3ae0f756e0127aa08538519b1 FT_RWAUGMENTED-v4-c0-B.dat
144aba6d77cd68ea8370460f74a4461c FT_RWAUGMENTED-v4-c0-L.dat
92093094295a885840f97b1c836ea23d FT_RWAUGMENTED-v4-c1-B.dat
0c76232bc7fdb51a19d95ccec8677d18 FT_RWAUGMENTED-v4-c1-L.dat
15e7a96c911a2c0a7c369a62612d4ce0 FT_RWAUGMENTED-v4-c2-B.dat
345545ae3004e0db2b2a8f352e079876 FT_RWAUGMENTED-v4-c2-L.dat
e8102fc1bb038ab258c10cc95074cf5c FT_RWAUGMENTED-v4-c3-B.dat
740778a2db383b4f7e4351bf0abd9c6d FT_RWAUGMENTED-v4-c3-L.dat
a837739733c2ff3df2148bb5cd4947af FT_RWAUGMENTED-v5-c0-B.dat
d2480d6252e5efe1ec04b1a1bde56241 FT_RWAUGMENTED-v5-c0-L.dat
df4441021aaf12ce64736a4bce232359 FT_RWAUGMENTED-v5-c1-B.dat
0faa9c6d01d0a335f94dcc6aabfac87d FT_RWAUGMENTED-v5-c1-L.dat
4fe6788bdd83da5e25723649ace4aca4 FT_RWAUGMENTED-v5-c2-B.dat
8da2469163cd8f18965ad1cccfedb138 FT_RWAUGMENTED-v5-c2-L.dat
c90f360d0df2c5ade0807c33b02412fb FT_RWAUGMENTED-v5-c3-B.dat
44e712f5762c5ae44b0dbe939db90eb7 FT_RWAUGMENTED-v5-c3-L.dat
b455d0de077aafcf307de297276081b1 FT_RWAUGMENTED-v6-c0-B.dat
1ccfbf9923c3fef8e20882e12ab67832 FT_RWAUGMENTED-v6-c0-L.dat
7143116525bf86ca2f6b2f17a885d487 FT_RWAUGMENTED-v6-c1-B.dat
313297d15813d45e94f43a151fb47e9f FT_RWAUGMENTED-v6-c1-L.dat
76a477528ee41201ec2d4de3435d358a FT_RWAUGMENTED-v6-c2-B.dat
5565e1b423ff4e8b9cee915ea510bd08 FT_RWAUGMENTED-v6-c2-L.dat
bb714d066b8f642a66514f8716eb41d7 FT_RWAUGMENTED-v6-c3-B.dat
29f2f299188812f5d637e111a9cb0c0b FT_RWAUGMENTED-v6-c3-L.dat
98f941dd108eb9adb6cd2345e84f3446 FT_RWAUGROUTING-v1-c0-B.dat
142535ea5803b315fe462c577f489981 FT_RWAUGROUTING-v1-c0-L.dat
385a009413348bd9907bc4ee9b42971d FT_RWAUGROUTING-v1-c1-B.dat
9e1a77905d37ab7b0b09e0451c8957ba FT_RWAUGROUTING-v1-c1-L.dat
f0345fe39fa5ba8c2850e1d0fc7e3512 FT_RWAUGROUTING-v1-c2-B.dat
b2ce0a2060dd7394e18e44ab8479739d FT_RWAUGROUTING-v1-c2-L.dat
f3f0b7602ca707bc192eff99253b7dbf FT_RWAUGROUTING-v1-c3-B.dat
1f8fbfe9192ee25162945c1657e5d235 FT_RWAUGROUTING-v1-c3-L.dat
6c15e70f39c472ce767cb980b60f0906 FT_RWAUGROUTING-v2-c0-B.dat
75399a83966d3aaee25fab55e6880670 FT_RWAUGROUTING-v2-c0-L.dat
83c2f84eff7b0d9bf60d2ce8c754cf0b FT_RWAUGROUTING-v2-c1-B.dat
6378188823a34aaf9e410ec3ebca6662 FT_RWAUGROUTING-v2-c1-L.dat
fee5b7e3606874f0547bf24980cec47e FT_RWAUGROUTING-v2-c2-B.dat
b0e1615a33b13a784c88fe4bc5d4760d FT_RWAUGROUTING-v2-c2-L.dat
79adb620cd8ea27e516d36c12273e2ba FT_RWAUGROUTING-v2-c3-B.dat
5e657b4449207187e5f70a7b8af6585c FT_RWAUGROUTING-v2-c3-L.dat
38de6be29e51e2ae9612ad38ad13c38d FT_RWAUGROUTING-v3-c0-B.dat
f6de88739278985c7b8f3b4e3b359d0f FT_RWAUGROUTING-v3-c0-L.dat
c4ff7dc4cb28933647ca1ad9a2383850 FT_RWAUGROUTING-v3-c1-B.dat
bcfd975c66e9166a95b28b76cd1c49a9 FT_RWAUGROUTING-v3-c1-L.dat
2b2187ed21d638f88432e31441a9b8be FT_RWAUGROUTING-v3-c2-B.dat
af234bd2185ae67b6ee988bf7b3ffb25 FT_RWAUGROUTING-v3-c2-L.dat
bdcf6fb78713040ceb518c72d49b15b4 FT_RWAUGROUTING-v3-c3-B.dat
c9550388429c76a9eb22246ac2af3e86 FT_RWAUGROUTING-v3-c3-L.dat
c971232239be750b0ce8ef6078955f9a FT_RWAUGROUTING-v4-c0-B.dat
bac0c79592ffa00c6cfe37af3aee49a6 FT_RWAUGROUTING-v4-c0-L.dat
053445f125f46dc694c60bb8981f7b92 FT_RWAUGROUTING-v4-c1-B.dat
ed5bdbdddf969a7717cc6c4e29d6b1a7 FT_RWAUGROUTING-v4-c1-L.dat
1958a0255faea25ff0309ac0c33f2e6f FT_RWAUGROUTING-v4-c2-B.dat
378a8c3a4ce5507ff77ed4205eb2e1b9 FT_RWAUGROUTING-v4-c2-L.dat
5c8f763e8ebb546d02ab4d26e47180f5 FT_RWAUGROUTING-v4-c3-B.dat
8e52de08624fe717944eff1295058039 FT_RWAUGROUTING-v4-c3-L.dat
f5eb6d15777dd5ce35a31b5869f0b3bc FT_RWAUGROUTING-v5-c0-B.dat
0ccdac22beb624a192afca1b1dbb6e48 FT_RWAUGROUTING-v5-c0-L.dat
1f927fbb39227e6787d8d5c9d980fb89 FT_RWAUGROUTING-v5-c1-B.dat
9327907535e2f22f3be63ac80bb50c6e FT_RWAUGROUTING-v5-c1-L.dat
c280bc817bd45b18a4d215c6a42ca6d8 FT_RWAUGROUTING-v5-c2-B.dat
040ee413492170a753c1bc77d98c34c6 FT_RWAUGROUTING-v5-c2-L.dat
be3106f3e37ccef53e31be43e3b709a0 FT_RWAUGROUTING-v5-c3-B.dat
758857a1139310bcb82583b261364924 FT_RWAUGROUTING-v5-c3-L.dat
571e1996b6f92963675bb77d6a1e7906 FT_RWAUGROUTING-v6-c0-B.dat
0227744425ca6a54005729b1d5d58b35 FT_RWAUGROUTING-v6-c0-L.dat
4f4a2bb21804de56986af0a7c8fba5b3 FT_RWAUGROUTING-v6-c1-B.dat
9a7e914a89b7779a779a0b651738b22a FT_RWAUGROUTING-v6-c1-L.dat
aed016d0f023b80253ec18311bae5b10 FT_RWAUGROUTING-v6-c2-B.dat
04e9b53b2221ef844b2a7f9e63cb8dee FT_RWAUGROUTING-v6-c2-L.dat
8a5121e17ccb78d815054c067e28d70b FT_RWAUGROUTING-v6-c3-B.dat
a8b13e0b815c0c7d09b876aa8e6a2b54 FT_RWAUGROUTING-v6-c3-L.dat
149c0f889c54d264053645ed0f02931a FT_RWAUGSNMPOUT-v1-c0-B.dat
0cd1a08bdc4ddc2aac2192a84d7141d9 FT_RWAUGSNMPOUT-v1-c0-L.dat
7b252a5581fd720576dd48b707720c4a FT_RWAUGSNMPOUT-v1-c1-B.dat
0753e9733faa189a22f75f7190a5e29e FT_RWAUGSNMPOUT-v1-c1-L.dat
d2956ea1e10a511257265490d46cdec0 FT_RWAUGSNMPOUT-v1-c2-B.dat
5d3b367607de68d1a2de072bdecfb870 FT_RWAUGSNMPOUT-v1-c2-L.dat
c276530855a82a5b5018dacf0c51958f FT_RWAUGSNMPOUT-v1-c3-B.dat
013c933cf10011bb6cd00474a60a90b7 FT_RWAUGSNMPOUT-v1-c3-L.dat
ae2efb2041e4b72ef737de7c22a0018d FT_RWAUGSNMPOUT-v2-c0-B.dat
d736603959b74a839365489823f37354 FT_RWAUGSNMPOUT-v2-c0-L.dat
3a7a9ee77c5572e850274c82d2caa238 FT_RWAUGSNMPOUT-v2-c1-B.dat
adc9b15e3e734d3786172a192b36ad22 FT_RWAUGSNMPOUT-v2-c1-L.dat
85ab41120bc52410764b37d5331e0df7 FT_RWAUGSNMPOUT-v2-c2-B.dat
d7d2d6dfd254167b8c866bea1d713340 FT_RWAUGSNMPOUT-v2-c2-L.dat
02cb6e2c4d24825f2e621a119db16dd0 FT_RWAUGSNMPOUT-v2-c3-B.dat
2de3078ec7192efb93fff01ef5ff4d4b FT_RWAUGSNMPOUT-v2-c3-L.dat
a3bd51c268884138d6264aaa648fb345 FT_RWAUGSNMPOUT-v3-c0-B.dat
3143664e6fa8a9f234d1aad9af982ece FT_RWAUGSNMPOUT-v3-c0-L.dat
c5896cb89ebb0f8f2d0283beb8f870f6 FT_RWAUGSNMPOUT-v3-c1-B.dat
96ddce6ce8f6c771813c37e25ecd7823 FT_RWAUGSNMPOUT-v3-c1-L.dat
e9b386334dcdbf368301499546d1ac21 FT_RWAUGSNMPOUT-v3-c2-B.dat
2e42052b89abfb6807e28bca0c5773f6 FT_RWAUGSNMPOUT-v3-c2-L.dat
0161cf2f82c1b79280a6c947e94c9f55 FT_RWAUGSNMPOUT-v3-c3-B.dat
8acb3c5a8256dbbf1c8b04841f6b9cec FT_RWAUGSNMPOUT-v3-c3-L.dat
2c877ee05351789df6aa8059311278c1 FT_RWAUGSNMPOUT-v4-c0-B.dat
8bd3676ddb668cb8024ef9a8dc2bb2f3 FT_RWAUGSNMPOUT-v4-c0-L.dat
5d42754bbf97415a1c20954b4416e615 FT_RWAUGSNMPOUT-v4-c1-B.dat
e3e4ca6136c8541600b1178c508dde73 FT_RWAUGSNMPOUT-v4-c1-L.dat
28ff462463d46cdf67eae9d1c4cab296 FT_RWAUGSNMPOUT-v4-c2-B.dat
eb30d1b37ef7b04f0a019339e890d1a4 FT_RWAUGSNMPOUT-v4-c2-L.dat
6a4253ac078f942ca98a9d10b494fef8 FT_RWAUGSNMPOUT-v4-c3-B.dat
3cc69f0bc614ecbe2bb20494a14850be FT_RWAUGSNMPOUT-v4-c3-L.dat
518c0691f788ef76fb71fe58d3d921b3 FT_RWAUGSNMPOUT-v5-c0-B.dat
0fe6d5e27c443f63a5a31e7045189e19 FT_RWAUGSNMPOUT-v5-c0-L.dat
d4b8d59f94c25fb0ed1c75c5ffd333a1 FT_RWAUGSNMPOUT-v5-c1-B.dat
34ebc0473675b006a1c324276b5bf40c FT_RWAUGSNMPOUT-v5-c1-L.dat
9ff94772e1145cd86e74d6097284c86a FT_RWAUGSNMPOUT-v5-c2-B.dat
56620fa7b835ef1a9259876be4a9d5c5 FT_RWAUGSNMPOUT-v5-c2-L.dat
9acb3dfb44fde9a574ed3f7b62983b10 FT_RWAUGSNMPOUT-v5-c3-B.dat
86940744902547ea20ba44c8c385ca43 FT_RWAUGSNMPOUT-v5-c3-L.dat
3e8a44d3508c1759901f66681433b28a FT_RWAUGWEB-v1-c0-B.dat
242d72f8797f1bfd645f95d0863dda01 FT_RWAUGWEB-v1-c0-L.dat
258e9e083d17f50ff5254e768f8ae1b3 FT_RWAUGWEB-v1-c1-B.dat
4d51c8409541472060562b96e0b88b3d FT_RWAUGWEB-v1-c1-L.dat
2af82b075b071d77133155e417a7838f FT_RWAUGWEB-v1-c2-B.dat
fc29b6b2b7d8073abeb4f2e44b16e16a FT_RWAUGWEB-v1-c2-L.dat
6c4b063e3b025c74a456e57224db227a FT_RWAUGWEB-v1-c3-B.dat
4d34c817ff3a08ba3c266069f901d766 FT_RWAUGWEB-v1-c3-L.dat
fb8842cf8fc017d74ddf481bcfeccc3c FT_RWAUGWEB-v2-c0-B.dat
10a7545e53e6359186ec43e9625911df FT_RWAUGWEB-v2-c0-L.dat
9e5906f38d5ac464bce779618b26adac FT_RWAUGWEB-v2-c1-B.dat
1af8a48b20bb309da7f099535a02faad FT_RWAUGWEB-v2-c1-L.dat
0db77a6c2e5d9503c5150656b19b2dc4 FT_RWAUGWEB-v2-c2-B.dat
147f54af0fe2c71c574a541e41caba89 FT_RWAUGWEB-v2-c2-L.dat
2f0e694c6258f120b1da32bad2d385c5 FT_RWAUGWEB-v2-c3-B.dat
62f017729111163cdb27bb46f69ac088 FT_RWAUGWEB-v2-c3-L.dat
0dbcf4148f9ef1d5134b60032d8f955c FT_RWAUGWEB-v3-c0-B.dat
1b60b8c8a9d5d7208956e4b0a0e65c50 FT_RWAUGWEB-v3-c0-L.dat
8917f6d3683d6a8eea94bf2c24d16387 FT_RWAUGWEB-v3-c1-B.dat
56e77c0ddee1e255c861b7c404447878 FT_RWAUGWEB-v3-c1-L.dat
3369dd55203b75c770a2b6a6b7c3ec54 FT_RWAUGWEB-v3-c2-B.dat
1701fbb1ac43031ceb4ffe3828b48c05 FT_RWAUGWEB-v3-c2-L.dat
817db6a6dd2be1f1d6324131b56f7b73 FT_RWAUGWEB-v3-c3-B.dat
a070b5986ff94a850b08e22abd569a91 FT_RWAUGWEB-v3-c3-L.dat
d6b90b5d7a3c9a97a03d0e2c9f583f42 FT_RWAUGWEB-v4-c0-B.dat
539b181b898acd59e463ff779a7c059e FT_RWAUGWEB-v4-c0-L.dat
8d7523adafeb5364c66eba36dca77c80 FT_RWAUGWEB-v4-c1-B.dat
f60be431bbee22799abe42c0990ff8b4 FT_RWAUGWEB-v4-c1-L.dat
6bd8f17373a7eb9b203123618435058b FT_RWAUGWEB-v4-c2-B.dat
d899792cdebb3ec7017326ea9c07ce81 FT_RWAUGWEB-v4-c2-L.dat
36aa859db50bee9cc021a7262389de81 FT_RWAUGWEB-v4-c3-B.dat
60af440bdf464554e4fb1fd7459eb260 FT_RWAUGWEB-v4-c3-L.dat
1ff83fc66e0f959967fbe7fc94068071 FT_RWAUGWEB-v5-c0-B.dat
f958d7a2d8f5522802a714934a0fc38e FT_RWAUGWEB-v5-c0-L.dat
be5e60a075f8371045a66157b34271f0 FT_RWAUGWEB-v5-c1-B.dat
a4609989704d4904e94ad83ed79b55d0 FT_RWAUGWEB-v5-c1-L.dat
84a277ff0adebe0d72221f3c03770605 FT_RWAUGWEB-v5-c2-B.dat
de709ae76c4326f845b4f9c908a93f93 FT_RWAUGWEB-v5-c2-L.dat
41ec757064a1b24eed1d88428824ca94 FT_RWAUGWEB-v5-c3-B.dat
a882f4687ef89f8dbc589e61cd35ac00 FT_RWAUGWEB-v5-c3-L.dat
f3c7436cb9be19af7ff2fa656060cd99 FT_RWAUGWEB-v6-c0-B.dat
0380415f9582caa0f6615ae9f3294231 FT_RWAUGWEB-v6-c0-L.dat
0ea91565f474baa7c7cb32649c716506 FT_RWAUGWEB-v6-c1-B.dat
6deb4738b88c48860eb0cd1669126971 FT_RWAUGWEB-v6-c1-L.dat
3665528667fdc2e437c2ca26fbc72603 FT_RWAUGWEB-v6-c2-B.dat
698c90c2d8f68cde5c1b4bff5216ceec FT_RWAUGWEB-v6-c2-L.dat
b933a49ba17b2442ae79d870b7811807 FT_RWAUGWEB-v6-c3-B.dat
6afa5dd9ecf657f9a2d29e77b7258919 FT_RWAUGWEB-v6-c3-L.dat
6986dfb3ca505b391d5f9fa9be1cda30 FT_RWFILTER-v1-c0-B.dat
cea530e72ae2b25052dbc5e96d9d0aa5 FT_RWFILTER-v1-c0-L.dat
091cf3a98cf8c1ec3d30e1d3f4936b7a FT_RWFILTER-v1-c1-B.dat
78ca148b710d9cb00105aa4f8c0a1cc1 FT_RWFILTER-v1-c1-L.dat
f714e7a5380e16cb1fd7cb81d7c1a20b FT_RWFILTER-v1-c2-B.dat
103f8d51add43fe1cf1f1e6e5ec0b765 FT_RWFILTER-v1-c2-L.dat
746c67b5a512be05c5fdf1c6eca18bfe FT_RWFILTER-v1-c3-B.dat
f457f61d48f8bddced3203e9d9c64ae4 FT_RWFILTER-v1-c3-L.dat
c1fc3a70265e72fbebe417ef8d675093 FT_RWFILTER-v2-c0-B.dat
d64003c3c6d3abed3c0df4db0529fd7b FT_RWFILTER-v2-c0-L.dat
26c4e838cdeb201d791727e112fb96d8 FT_RWFILTER-v2-c1-B.dat
3987ba7f0b485dce4b65ff0d6fbe2d2a FT_RWFILTER-v2-c1-L.dat
264217bbf4768eb9b5a7b2304ed71c21 FT_RWFILTER-v2-c2-B.dat
9838648a479719d6c891832cb8b8d42d FT_RWFILTER-v2-c2-L.dat
058ac1ebd00160d8b7dfafb8fa53cf30 FT_RWFILTER-v2-c3-B.dat
bf6ea3f45df1cc0aab338812f1c59661 FT_RWFILTER-v2-c3-L.dat
82e819749752c149e6517bcf4c8c2e61 FT_RWFILTER-v3-c0-B.dat
0fd4b025dcb2d0adcdcc86b79eba6bb0 FT_RWFILTER-v3-c0-L.dat
29c6934e0d4cd6df4a34cb1225080257 FT_RWFILTER-v3-c1-B.dat
0ecc5cd71f91388ca3e8565046ad7e73 FT_RWFILTER-v3-c1-L.dat
cc3bc63da36bab23aeaaa58ee69d54d0 FT_RWFILTER-v3-c2-B.dat
fab739247996547627ce22e31d4bac89 FT_RWFILTER-v3-c2-L.dat
993bf373f1d4142872ac81dfe0e713c0 FT_RWFILTER-v3-c3-B.dat
81973b698cfd39d615d51aec5b99ded1 FT_RWFILTER-v3-c3-L.dat
4d67c6ff508ee230c297195531fdf9a6 FT_RWFILTER-v4-c0-B.dat
8e0dccab479e7f961961e5f41155c255 FT_RWFILTER-v4-c0-L.dat
2aa64ffe5dc55c6808f1028961bf1fff FT_RWFILTER-v4-c1-B.dat
ebb415f850e0831a0b0fb986382e10d9 FT_RWFILTER-v4-c1-L.dat
e96b80f6ca5903cedea61d001bec5008 FT_RWFILTER-v4-c2-B.dat
dc5d2ffe74ff1b7ec6843a63abc23536 FT_RWFILTER-v4-c2-L.dat
1b5e29bc71c834a296ad106b6a83be5e FT_RWFILTER-v4-c3-B.dat
03d387c3612a061f3ab4c756e4a4c0e0 FT_RWFILTER-v4-c3-L.dat
e5b1e8db7a40a8f0ddb1982168a8181f FT_RWFILTER-v5-c0-B.dat
8b6c7f61e15b9226e1ceaf899aed253e FT_RWFILTER-v5-c0-L.dat
1e17cf9578609c27e1b9ca574caa4d9a FT_RWFILTER-v5-c1-B.dat
7cde0994c0758c2240b2b63b7d312a1f FT_RWFILTER-v5-c1-L.dat
3ca1de6d51d1b892d2e061ae08792cae FT_RWFILTER-v5-c2-B.dat
5e00a4bde82f084cf12bd7f179a33d36 FT_RWFILTER-v5-c2-L.dat
41a6470d76eb57ce10c6717631c9ff75 FT_RWFILTER-v5-c3-B.dat
048cdc23f96b7313f741d847b3325a44 FT_RWFILTER-v5-c3-L.dat
7dcaa52987c38d63b84a5b7c5ce33941 FT_RWGENERIC-v0-c0-B.dat
fd656c86111fe47d0cbf13ed392bed67 FT_RWGENERIC-v0-c0-L.dat
27893661440063991af0c88614fed945 FT_RWGENERIC-v0-c1-B.dat
f3122e81b9b23638187aa3376cc65ac5 FT_RWGENERIC-v0-c1-L.dat
5f7e5c8de1a6266b77f42d0ac4d10d73 FT_RWGENERIC-v0-c2-B.dat
bc415d8b1b19e9390c165e8d35a1a6bf FT_RWGENERIC-v0-c2-L.dat
04bf5772661751dff7298bc5c8b4dcf9 FT_RWGENERIC-v0-c3-B.dat
2628e85d1ed676e359fea7ac4a593252 FT_RWGENERIC-v0-c3-L.dat
513984dc560dde040158a1971320680c FT_RWGENERIC-v1-c0-B.dat
1640b04c5156437dcb9c3f7395fe98c1 FT_RWGENERIC-v1-c0-L.dat
099d36f38bfebec3ec35952b327ce13c FT_RWGENERIC-v1-c1-B.dat
e3d4609ba3e3729f4f0d7b7335f3ed44 FT_RWGENERIC-v1-c1-L.dat
4ba13eb429dfcc94d73386c3215249cd FT_RWGENERIC-v1-c2-B.dat
d51f3cb16cb0830019bfca4b1cd70247 FT_RWGENERIC-v1-c2-L.dat
54869a671c6ecdc2d7cf9585d230c8a2 FT_RWGENERIC-v1-c3-B.dat
b56688150cfcc6410c075fe0157022d6 FT_RWGENERIC-v1-c3-L.dat
947f1421d65b62e736c08d7a79f0ff42 FT_RWGENERIC-v2-c0-B.dat
d05b78204f16fb235fc150b1be56d0b7 FT_RWGENERIC-v2-c0-L.dat
a490c651a771b764ab218a6bb5e3a6e8 FT_RWGENERIC-v2-c1-B.dat
35742b3af08059c433ad65ae8e295105 FT_RWGENERIC-v2-c1-L.dat
4c1ed99c498491657b6105525b21ab67 FT_RWGENERIC-v2-c2-B.dat
bf959c2fb40352f6234df8b9f3717e1f FT_RWGENERIC-v2-c2-L.dat
851780723e7a5f0279716f118e8ed7f3 FT_RWGENERIC-v2-c3-B.dat
e66e5c63ef892c361269061fb9c41cf3 FT_RWGENERIC-v2-c3-L.dat
d6a93bcf2a9a6020bed0835ceb646ea2 FT_RWGENERIC-v3-c0-B.dat
0f0add9723e5b1fc24330730297c1b34 FT_RWGENERIC-v3-c0-L.dat
f57e0dd0e62b5c927f117101c88caeaa FT_RWGENERIC-v3-c1-B.dat
0e849cf5ab4fb0df06f07e614d305991 FT_RWGENERIC-v3-c1-L.dat
152de2e564e5c7177ae6defeff387bf7 FT_RWGENERIC-v3-c2-B.dat
aee6d908907c55f7c430bfb7aa5c1bcb FT_RWGENERIC-v3-c2-L.dat
aca3d2a9bba4c109e8ff371e371bea8d FT_RWGENERIC-v3-c3-B.dat
b1bf034389cb31bcc66c5ddca45b2d7e FT_RWGENERIC-v3-c3-L.dat
71d4e55cd818777ee5438be65f7ccac5 FT_RWGENERIC-v4-c0-B.dat
14f984703521151b77b2a63669abefe4 FT_RWGENERIC-v4-c0-L.dat
d8b9226e8477614aebbb376a78dc5576 FT_RWGENERIC-v4-c1-B.dat
8291163b094e4a2ce441446ee6d6b4d0 FT_RWGENERIC-v4-c1-L.dat
069bacaccc25fadc2e2bbd1432857d16 FT_RWGENERIC-v4-c2-B.dat
dd18b1eabe121da6f8ac7c323e504e04 FT_RWGENERIC-v4-c2-L.dat
c961dd9662deb11f4887a7388ef7e66b FT_RWGENERIC-v4-c3-B.dat
b9e5250ddc6498937417c4d2d8d81add FT_RWGENERIC-v4-c3-L.dat
dd89f00bd51f03caf9de1e7812108005 FT_RWGENERIC-v5-c0-B.dat
223ab374329314618fe9d8a1d2f300e8 FT_RWGENERIC-v5-c0-L.dat
0884d019bfed192007a770275cfc52a7 FT_RWGENERIC-v5-c1-B.dat
518368e2e59f58adf0d21c796c73aa7e FT_RWGENERIC-v5-c1-L.dat
3fa208bcfe8d3aaf76111ed9d7b154a1 FT_RWGENERIC-v5-c2-B.dat
aeb77d2fcb5bbd7d8d13c48de54d36ff FT_RWGENERIC-v5-c2-L.dat
52915d901799aca23f5798491c53e0f6 FT_RWGENERIC-v5-c3-B.dat
19e2a823b0f729a71d462e78d2aa4d68 FT_RWGENERIC-v5-c3-L.dat
bf06e20bbd653ab9605bfc4ae116f3d2 FT_RWGENERIC-v6-c0-B.dat
b6675fd7d53471e1b125118e75a0a7f2 FT_RWGENERIC-v6-c0-L.dat
9ef592a7423c3c49ef7674ef7cae45f1 FT_RWGENERIC-v6-c1-B.dat
49b06c5afb303ac01335aff37b5778fd FT_RWGENERIC-v6-c1-L.dat
a227d9be2f8ccc14b7b61646a75e84fd FT_RWGENERIC-v6-c2-B.dat
10451b1ead529cd20d5fd7e34b7c0d46 FT_RWGENERIC-v6-c2-L.dat
577adf036cc267519df464cf45ccffaa FT_RWGENERIC-v6-c3-B.dat
a6f01f6a09bd062dfedc8086fa92c346 FT_RWGENERIC-v6-c3-L.dat
2f9f457aa09dadb79ce6e6b9cac7aa05 FT_RWIPV6-v1-c0-B.dat
9c18ef017e67ee89fa0daabc31daa233 FT_RWIPV6-v1-c0-L.dat
699212297d2621249b08d909de8b80a3 FT_RWIPV6-v1-c1-B.dat
0dfa265492d6a1c62d78882ef32a9964 FT_RWIPV6-v1-c1-L.dat
893d07912f8754ab1f611826f93a8cfa FT_RWIPV6-v1-c2-B.dat
9584e20a7b9641afd78422afbd81a09b FT_RWIPV6-v1-c2-L.dat
34820ff7361b78cda507f060d7674e58 FT_RWIPV6-v1-c3-B.dat
23638d77118b8cd4435fc3d75318b2f5 FT_RWIPV6-v1-c3-L.dat
acc4780b215db74a91c541c98811e77f FT_RWIPV6-v2-c0-B.dat
1c69b295749a55894504e9d5d00042e5 FT_RWIPV6-v2-c0-L.dat
6ff220d018769f8caa8aa3d96b9c92e5 FT_RWIPV6-v2-c1-B.dat
2b4d41ee54c55a1dec7d0f01e4eef410 FT_RWIPV6-v2-c1-L.dat
d3f5dcce303ef4395641299a4ea316cb FT_RWIPV6-v2-c2-B.dat
f147e9fa29126c13c715aba56df5d842 FT_RWIPV6-v2-c2-L.dat
c9bcf6015ed77a86747f8e327152d936 FT_RWIPV6-v2-c3-B.dat
f51b84442edfbe75a70afcd6fc204661 FT_RWIPV6-v2-c3-L.dat
f80718f00f7ed871040d66735431be78 FT_RWIPV6-v3-c0-B.dat
8d151b9caed32eeabe2e0102eb7153fe FT_RWIPV6-v3-c0-L.dat
d02066ecc7d0c645f34029ff9439ce1a FT_RWIPV6-v3-c1-B.dat
db4cea2fc390cc674705fd38d2fab0fa FT_RWIPV6-v3-c1-L.dat
c091abc91efe5f90b40ddeb306764494 FT_RWIPV6-v3-c2-B.dat
503caa091d9a1ee9e32d2171d28dffba FT_RWIPV6-v3-c2-L.dat
ca82c3b1f486b4b04a2033500bf21dd4 FT_RWIPV6-v3-c3-B.dat
62e148efc9688b8d19c79812ecdbaef6 FT_RWIPV6-v3-c3-L.dat
c603a50b2fe0f71073449bb00ffdcd7b FT_RWIPV6ROUTING-v1-c0-B.dat
5ce7f5f312d053e719a09de2bdf4dd48 FT_RWIPV6ROUTING-v1-c0-L.dat
01db8a0206495ffa9cb3853b6a263565 FT_RWIPV6ROUTING-v1-c1-B.dat
78ad670f0cf3c3d67cea0e9ceda29be4 FT_RWIPV6ROUTING-v1-c1-L.dat
f2c65c5fabba2f150acb324ebdb26b51 FT_RWIPV6ROUTING-v1-c2-B.dat
58f02ba420d0860f0f5dd860827e7d61 FT_RWIPV6ROUTING-v1-c2-L.dat
41b963254853c12657905e5d14e87586 FT_RWIPV6ROUTING-v1-c3-B.dat
a177c82eed9c423cea2f89296aeb2ec9 FT_RWIPV6ROUTING-v1-c3-L.dat
0655e3c8de4049f43adf2f9eceff540b FT_RWIPV6ROUTING-v2-c0-B.dat
5ed32a580fcf09eefd1016f68b3eab3b FT_RWIPV6ROUTING-v2-c0-L.dat
59aefe73f0b406ac0c5b2dae4db48aec FT_RWIPV6ROUTING-v2-c1-B.dat
95207e7656f6e575ba0363f9db673592 FT_RWIPV6ROUTING-v2-c1-L.dat
ba749a6064c4b274960437ca1b1d11c8 FT_RWIPV6ROUTING-v2-c2-B.dat
7d0990a104def93b4350ed12bab74e69 FT_RWIPV6ROUTING-v2-c2-L.dat
110fe7b5b54a778bea340e51d8517d7a FT_RWIPV6ROUTING-v2-c3-B.dat
e15e835a64c7940743dd4c738cf84363 FT_RWIPV6ROUTING-v2-c3-L.dat
df514250db00d79242f13463b7245c70 FT_RWIPV6ROUTING-v3-c0-B.dat
041296f90b30ede7da5f81d7c3864074 FT_RWIPV6ROUTING-v3-c0-L.dat
d68e6034f3621bfad96967aec9e753c9 FT_RWIPV6ROUTING-v3-c1-B.dat
f695cab19a9a761ffcf918db10d936d8 FT_RWIPV6ROUTING-v3-c1-L.dat
4d7af74d80435e84e6775f09ea41ddb6 FT_RWIPV6ROUTING-v3-c2-B.dat
aa9f2303e3c62bb5ac60e4bd404b41bf FT_RWIPV6ROUTING-v3-c2-L.dat
86c78131cd4bde9200c61421553abe73 FT_RWIPV6ROUTING-v3-c3-B.dat
d39a9dd77f8cfe88f20a7182d62ea465 FT_RWIPV6ROUTING-v3-c3-L.dat
8c9fec9b9c99ef79c686b4f49730f783 FT_RWIPV6ROUTING-v4-c0-B.dat
daaf307bf65d1693be3550f8b972a388 FT_RWIPV6ROUTING-v4-c0-L.dat
a7d5c58a274060feab29aae3e9a8adc8 FT_RWIPV6ROUTING-v4-c1-B.dat
382a6074005b54f94b69741125c3e14b FT_RWIPV6ROUTING-v4-c1-L.dat
8ef5921ba9f1245efa187d3bcd8b6aa9 FT_RWIPV6ROUTING-v4-c2-B.dat
66000963b28721dd6efe0d3507bf83de FT_RWIPV6ROUTING-v4-c2-L.dat
1349e2d3ee659b4488b6d2a5a4f3b9eb FT_RWIPV6ROUTING-v4-c3-B.dat
e53a6a77138e239b3c124d2a74dde489 FT_RWIPV6ROUTING-v4-c3-L.dat
63ab6740566b8c6af7d43c2948ee8e82 FT_RWNOTROUTED-v1-c0-B.dat
c0fe797de574e9d2fd85447032910ff6 FT_RWNOTROUTED-v1-c0-L.dat
d305bcb80c26979cfa5aaaa19c74530c FT_RWNOTROUTED-v1-c1-B.dat
4d54be7afa877687d4ec504f6a3ad060 FT_RWNOTROUTED-v1-c1-L.dat
ba6c890bd067b4459ddb411bc04f547e FT_RWNOTROUTED-v1-c2-B.dat
bf52e8553b91ac3f5eae64583c6dabb9 FT_RWNOTROUTED-v1-c2-L.dat
804e4bfef6d18a74972aac43f1cc2175 FT_RWNOTROUTED-v1-c3-B.dat
a1aaa711c8ad5c30020d1eb37a60f77f FT_RWNOTROUTED-v1-c3-L.dat
c68a35d9f1f8536fda5b27b7a2e7fc9b FT_RWNOTROUTED-v2-c0-B.dat
2389313df8dab0ae4c0c732cedbb5ea1 FT_RWNOTROUTED-v2-c0-L.dat
500b1718ee443086107097de48f9ca77 FT_RWNOTROUTED-v2-c1-B.dat
8a81a6828631802676faa183ad16a117 FT_RWNOTROUTED-v2-c1-L.dat
48d90f0351183137fb0db6bb24b79a8a FT_RWNOTROUTED-v2-c2-B.dat
e321d9f627262e931838c6df4ed0209e FT_RWNOTROUTED-v2-c2-L.dat
2fcab32ffe1f358f471b033c4a923507 FT_RWNOTROUTED-v2-c3-B.dat
8a34fcffb89ab2b05f76b8a6018084fb FT_RWNOTROUTED-v2-c3-L.dat
8a5362f0ab8f6bcbd48463878c23c94d FT_RWNOTROUTED-v3-c0-B.dat
b572af9ccec4743025b7c28d66cf2d2d FT_RWNOTROUTED-v3-c0-L.dat
540025825039fae3208d91d3567881c2 FT_RWNOTROUTED-v3-c1-B.dat
68ceda0aff331556818f48f0c35e20a8 FT_RWNOTROUTED-v3-c1-L.dat
60f5962b9d2d1de58aa530934473d004 FT_RWNOTROUTED-v3-c2-B.dat
044eec2be252b10c34adc4fe32d6cfa0 FT_RWNOTROUTED-v3-c2-L.dat
3afc036af530ad061c3dbb1724bf45f5 FT_RWNOTROUTED-v3-c3-B.dat
4f28efc9ed6a1c33318cbaf86f805273 FT_RWNOTROUTED-v3-c3-L.dat
72e1c54caaa54243661c0a0c1a9de78b FT_RWNOTROUTED-v4-c0-B.dat
96f05809a0f851cd5baf40ef450e514e FT_RWNOTROUTED-v4-c0-L.dat
da068228b2ba1390c5fa065529ed65d1 FT_RWNOTROUTED-v4-c1-B.dat
5aa97d6d58b9dbc40b0e1c4880a42cd1 FT_RWNOTROUTED-v4-c1-L.dat
342ad55b44619fa4258e9857936b3c29 FT_RWNOTROUTED-v4-c2-B.dat
ad7338ea337bcfb54177950e83f34ece FT_RWNOTROUTED-v4-c2-L.dat
8837f213217d4a6e593004eb05a67f97 FT_RWNOTROUTED-v4-c3-B.dat
7bf22de122856f329718fc99125615fe FT_RWNOTROUTED-v4-c3-L.dat
db20862725dc8c2b7ae234eb74f2e6dd FT_RWNOTROUTED-v5-c0-B.dat
5a5fdb68fa587d361ad47ab3071a0150 FT_RWNOTROUTED-v5-c0-L.dat
152f9f6672ef4556c5b611459b6cdbee FT_RWNOTROUTED-v5-c1-B.dat
29642a8c73b50035ca5e2b0f6fae2dd4 FT_RWNOTROUTED-v5-c1-L.dat
c1a0505018059c833fb3d1b41e0b7ce7 FT_RWNOTROUTED-v5-c2-B.dat
f8bf559b9a06dc7dd3cbeeef8c45716d FT_RWNOTROUTED-v5-c2-L.dat
4174d5c48ca071991c60355da3340750 FT_RWNOTROUTED-v5-c3-B.dat
e320776b2770cfb4a34cfa6644469cd1 FT_RWNOTROUTED-v5-c3-L.dat
45e616c2ea42dfc32ea7f238e2e2cfe5 FT_RWROUTED-v1-c0-B.dat
82c16c2a864c206de65bc5119767dff1 FT_RWROUTED-v1-c0-L.dat
fa9d7a6c0e8267b967c128349d010392 FT_RWROUTED-v1-c1-B.dat
1a1b604369c33b30b1322353328c13e6 FT_RWROUTED-v1-c1-L.dat
2f9184009ad2b56a4df53f3d40796352 FT_RWROUTED-v1-c2-B.dat
3ce7398124bccbb727ee3ad436b4e25e FT_RWROUTED-v1-c2-L.dat
6fbb998631eb33e25e29ce441ea9b1fe FT_RWROUTED-v1-c3-B.dat
a1e9596481448fd42f1f1e053dadb55a FT_RWROUTED-v1-c3-L.dat
cb581ce9f132280ca46394fdaae64e63 FT_RWROUTED-v2-c0-B.dat
5e56afa64d9351d69c2ae973d5f663ed FT_RWROUTED-v2-c0-L.dat
576e4ce47503cf80d7992db86b99d2dc FT_RWROUTED-v2-c1-B.dat
36493734239e466cb9c892b1864a197f FT_RWROUTED-v2-c1-L.dat
deeddd3127643705a1fd0b50c1b8ca7b FT_RWROUTED-v2-c2-B.dat
cbbe239c3af9e33ed6212a55b527a45e FT_RWROUTED-v2-c2-L.dat
d6e95b66a68449c2bf42765c69cce513 FT_RWROUTED-v2-c3-B.dat
1ced1b2871256a8a2bc3dc0ca3586bc2 FT_RWROUTED-v2-c3-L.dat
077f02f2ee0d4a6ec5ac3b50e231a4d2 FT_RWROUTED-v3-c0-B.dat
f3a099444867927403f4fdd06a10a8ea FT_RWROUTED-v3-c0-L.dat
b99deaad26c4199c189ab21111d3c698 FT_RWROUTED-v3-c1-B.dat
f67ea6d04eff01384f2809999e9dda53 FT_RWROUTED-v3-c1-L.dat
5b4c3aec0f06c80a67a2d2975720d6cc FT_RWROUTED-v3-c2-B.dat
a5b89aa73ac64b9d988e8b7d0f96a301 FT_RWROUTED-v3-c2-L.dat
8fe039cf744c49f9e274384610d96378 FT_RWROUTED-v3-c3-B.dat
3bca18712738a656cf1a79b69f8d3957 FT_RWROUTED-v3-c3-L.dat
f17d7d2aa5427eaff43c4fac1574857f FT_RWROUTED-v4-c0-B.dat
5034cea9260da8a6dad733b5f7c78d5a FT_RWROUTED-v4-c0-L.dat
9e755deb6b1aad2aebed1cec1ab39ea1 FT_RWROUTED-v4-c1-B.dat
0dd46efea48cd2d08c14ea3f6987ea1e FT_RWROUTED-v4-c1-L.dat
a89778891e1ba5f86b2cc095465284ed FT_RWROUTED-v4-c2-B.dat
8ed64ad91f5c7c98f8fa4e03ff8873e4 FT_RWROUTED-v4-c2-L.dat
cfbaadc084411a4363d0864b42aab35d FT_RWROUTED-v4-c3-B.dat
93a5241b99ab85af3f0a4016ddf72c51 FT_RWROUTED-v4-c3-L.dat
e9e16d7a47a3ba266bc5df8584d1b88f FT_RWROUTED-v5-c0-B.dat
e5d2153a62b3691069e49dfb15ac71b7 FT_RWROUTED-v5-c0-L.dat
53ca49143ecb4ebd9ec5513c6fe1e62e FT_RWROUTED-v5-c1-B.dat
27bb27f62bd0bd74e4dc5c6e664fb95a FT_RWROUTED-v5-c1-L.dat
0668ecd5cd56a9c8bd94dc92dc021123 FT_RWROUTED-v5-c2-B.dat
f946ee0f1b13a987c555f0aab85bf411 FT_RWROUTED-v5-c2-L.dat
f335fe9494e3313da07b93233c4a8904 FT_RWROUTED-v5-c3-B.dat
aa6e59633699ba5c43965f77512761c4 FT_RWROUTED-v5-c3-L.dat
6be74a9eff6b196a51e97dfcc8e31d0a FT_RWSPLIT-v1-c0-B.dat
ead8c12cd4ceed70f9d1cdbde78b37de FT_RWSPLIT-v1-c0-L.dat
11a9157a43acfc36a5d6837cdf46899c FT_RWSPLIT-v1-c1-B.dat
db25ad2e4c631bd22a4e55b829102ba0 FT_RWSPLIT-v1-c1-L.dat
2981a349f762c728dc94d06d6db0f0a3 FT_RWSPLIT-v1-c2-B.dat
9f668bae733c50a8991ee3cec869cbed FT_RWSPLIT-v1-c2-L.dat
c3636cccae4a563430b2b9a9dbaac5a4 FT_RWSPLIT-v1-c3-B.dat
f00fa0ffffc97473cb9a0f5975d48e21 FT_RWSPLIT-v1-c3-L.dat
d2210f9a7d15e606b4a9bdadcb05e017 FT_RWSPLIT-v2-c0-B.dat
3a796fa45a437dcb01eeb16cf0bcc6d4 FT_RWSPLIT-v2-c0-L.dat
79bd8edb052c5644d92f402a13805f68 FT_RWSPLIT-v2-c1-B.dat
1e6bcff7d83ce045b27fcaafc305a34d FT_RWSPLIT-v2-c1-L.dat
b933981892054094ab04faa9dceb1f34 FT_RWSPLIT-v2-c2-B.dat
104c0dee9f0dfae12ed6ec08d4fd9519 FT_RWSPLIT-v2-c2-L.dat
53528851aae4b1ad6584bc496f51b29a FT_RWSPLIT-v2-c3-B.dat
7377659ec4c05b1dde5e1c197e8595a0 FT_RWSPLIT-v2-c3-L.dat
d43215bde7090e7badc8c9fb4f133095 FT_RWSPLIT-v3-c0-B.dat
366b18ddbd8d7d5ea7a3bcdea35cebdf FT_RWSPLIT-v3-c0-L.dat
4a09747c160f6e23c085e2c2bcf60151 FT_RWSPLIT-v3-c1-B.dat
a76c1ef9e575c1b84030a7158d57acb2 FT_RWSPLIT-v3-c1-L.dat
22e485baeb50cee6ef7315399bd0dbf9 FT_RWSPLIT-v3-c2-B.dat
99f031786758815819964b0d42388fc2 FT_RWSPLIT-v3-c2-L.dat
84d8ab1418fff30ffdc1330a2bd1103f FT_RWSPLIT-v3-c3-B.dat
a1f71249450c07d93ddd7d2023d300f5 FT_RWSPLIT-v3-c3-L.dat
6ea55e3e335e04f9b1a4a845ad7b3223 FT_RWSPLIT-v4-c0-B.dat
0fa608aacfea2014dc4556072853fdf0 FT_RWSPLIT-v4-c0-L.dat
9c9528e2f6bff9acaf3b8347d9dd2c0c FT_RWSPLIT-v4-c1-B.dat
244f5c16ea72549132f67ae5506f02d4 FT_RWSPLIT-v4-c1-L.dat
be1126439a83f8b295fab6b355d92430 FT_RWSPLIT-v4-c2-B.dat
b1d788cec04da73ac18c0df4fc858db5 FT_RWSPLIT-v4-c2-L.dat
bf6e7460317944cda8b366e2470473a1 FT_RWSPLIT-v4-c3-B.dat
82ed30caa69da391b7e14e13de634336 FT_RWSPLIT-v4-c3-L.dat
a4405165075cb9286a941c90bb68019d FT_RWSPLIT-v5-c0-B.dat
cd3c855001d584165bc8ea62afa352c9 FT_RWSPLIT-v5-c0-L.dat
298e49f809f86d6d265a725383a2aa9c FT_RWSPLIT-v5-c1-B.dat
554014f72af33a165fc5ede025f3d160 FT_RWSPLIT-v5-c1-L.dat
3690b7caf4c2e17960f8b8e3eb632c29 FT_RWSPLIT-v5-c2-B.dat
5a47281757ea51957f0b9c8d9514b9ca FT_RWSPLIT-v5-c2-L.dat
178130efb5e30c94449ce30c4ce5005b FT_RWSPLIT-v5-c3-B.dat
04318041701f5385ad2e52511a768522 FT_RWSPLIT-v5-c3-L.dat
0e1b1e8228b57886008a97cb765c331b FT_RWWWW-v1-c0-B.dat
6c123a6c98d83c8cf317d0254c002cb6 FT_RWWWW-v1-c0-L.dat
089f3ea77e04dab201087a299f5d3ae0 FT_RWWWW-v1-c1-B.dat
4a857fcec93a5f9b283cb80b12d69f17 FT_RWWWW-v1-c1-L.dat
fa07a51f375a495e8ae5efa2ec5115fb FT_RWWWW-v1-c2-B.dat
be8c53ead9dbe25bbe4158c87844db72 FT_RWWWW-v1-c2-L.dat
94d07a9a5f89f4c9d8a44ab4b20cc45d FT_RWWWW-v1-c3-B.dat
1ad58acc9fb4d750f14e2e5f2b5d47df FT_RWWWW-v1-c3-L.dat
9f1f0176f53e5cc804b1bddfc2421b98 FT_RWWWW-v2-c0-B.dat
4a9b9dbc6f98dc0d5df5b9f83b1b140f FT_RWWWW-v2-c0-L.dat
27a3e17dcc5525c9d200b4d61ae5f487 FT_RWWWW-v2-c1-B.dat
8b3b01eb20974cf574b60aec367e29e8 FT_RWWWW-v2-c1-L.dat
3237bae27c7c20ff91760dfea299b690 FT_RWWWW-v2-c2-B.dat
3e060aeb0fd815f70ccdb9a36191c73b FT_RWWWW-v2-c2-L.dat
8a268afdc2e04c5f3ee60642cb1786b3 FT_RWWWW-v2-c3-B.dat
0b9ff55c9d5f2c689677feb4118745d5 FT_RWWWW-v2-c3-L.dat
a726e5fc2c493da0856695a85fd11298 FT_RWWWW-v3-c0-B.dat
cfdd861442b0a4a49723d57595681bf7 FT_RWWWW-v3-c0-L.dat
7a027dd8366cf4b8d681d5a06ca37d8c FT_RWWWW-v3-c1-B.dat
e15f6c43b9b10b6d9da58329efe12bd8 FT_RWWWW-v3-c1-L.dat
b55cb6891fcea6f5249ba629f8631e9b FT_RWWWW-v3-c2-B.dat
abc4c8b10b22ba1277c1a04343e2a874 FT_RWWWW-v3-c2-L.dat
a6b489645e4802df7bdca98b18761adf FT_RWWWW-v3-c3-B.dat
43fa802816383cd671671ecde04386d4 FT_RWWWW-v3-c3-L.dat
b67d537959ff93208eafb81469c5a09c FT_RWWWW-v4-c0-B.dat
4c348bb299e4dd82ed24223aa0ed9a14 FT_RWWWW-v4-c0-L.dat
cd0c4dca54ac9922397da0bd398dfacd FT_RWWWW-v4-c1-B.dat
3f406a8506b555adab4d8cd379a9d508 FT_RWWWW-v4-c1-L.dat
b0ff24fded4b760dc6fcb008761749b0 FT_RWWWW-v4-c2-B.dat
da6711570db3fcdfe2f9a9b82c6e51ff FT_RWWWW-v4-c2-L.dat
920a1d3a6c5888f4e45f09a834f837b2 FT_RWWWW-v4-c3-B.dat
cc7e778a78f81c85e863573193cb2cec FT_RWWWW-v4-c3-L.dat
123deb72a005b827d60a8e211a13f077 FT_RWWWW-v5-c0-B.dat
eed0098880ea791ccf34c4e1311887b4 FT_RWWWW-v5-c0-L.dat
32810191c8b2c8265b4bafc5a3d3181a FT_RWWWW-v5-c1-B.dat
63355fc37c6141d131e5ad29a788d2f4 FT_RWWWW-v5-c1-L.dat
224fe676195b48daa49fded2045bd983 FT_RWWWW-v5-c2-B.dat
fed04907fc9b1b72e4c7c7955f491abe FT_RWWWW-v5-c2-L.dat
694b30c27dee268f7ac6837ecc3a2ad0 FT_RWWWW-v5-c3-B.dat
b8468a4747f90849cd260a7b3d2740fa FT_RWWWW-v5-c3-L.dat

# these are for LZ0 2.05
a4e0ac1c437310d6c17be507c559afeb  FT_FLOWCAP-v2-c2-B.dat
c017f5d40f061cef2f618dfd506d95bb  FT_FLOWCAP-v2-c2-L.dat
782d711b70e70ef4e31103b86201d7d1  FT_FLOWCAP-v3-c2-B.dat
0f11857a2049cded66be1ec05391861e  FT_FLOWCAP-v3-c2-L.dat
89b31c6dcd6ccc3afca8560c464b51b8  FT_FLOWCAP-v4-c2-B.dat
ef4e3a4b98833f7d58e1e2b350d4509c  FT_FLOWCAP-v4-c2-L.dat
a36dcc30de049968c726c5a072297a3e  FT_FLOWCAP-v5-c2-B.dat
c375b8a2324d17f9ab3b3e25787d89c6  FT_FLOWCAP-v5-c2-L.dat
7d6233dc312713f204419a79688b34af  FT_FLOWCAP-v6-c2-B.dat
48f8b9beda2dcfc4e126ccbe0055f4b8  FT_FLOWCAP-v6-c2-L.dat
bb68cf8288b2a886fa8834f11fb8487a  FT_RWAUGMENTED-v1-c2-B.dat
5a4b1818ff457804de2cc42a82b13648  FT_RWAUGMENTED-v1-c2-L.dat
795bb874ddd7aa5204e4cea56dbdf0f7  FT_RWAUGMENTED-v2-c2-B.dat
5471237c27e2298e8224965d4ae965ed  FT_RWAUGMENTED-v2-c2-L.dat
b6ea73eb6e88c390a8e88c24e02fd773  FT_RWAUGMENTED-v3-c2-B.dat
b492b042e977ca3e95befb60576cfc24  FT_RWAUGMENTED-v3-c2-L.dat
63db7feef7a0dc5fac87e79b9c4e8426  FT_RWAUGMENTED-v4-c2-B.dat
def543fdc6eb4a010dc615244ee798f9  FT_RWAUGMENTED-v4-c2-L.dat
ea070efc91e0c4bc635d27f7aa7db261  FT_RWAUGMENTED-v5-c2-B.dat
f0b16ae7aaf31fcae3f88e5c414298cb  FT_RWAUGMENTED-v5-c2-L.dat
b248346dd161f1086b12124c21d96fc2  FT_RWAUGROUTING-v1-c2-B.dat
d48b29f70badf448460665506cdb8105  FT_RWAUGROUTING-v1-c2-L.dat
e92eccf9e0b6e692bc800a24e7592af9  FT_RWAUGROUTING-v2-c2-B.dat
611b5eb73ff88c2d5196913c10094470  FT_RWAUGROUTING-v2-c2-L.dat
aef9494014304e7039339fc2b1ec1e7b  FT_RWAUGROUTING-v3-c2-B.dat
1536239303fb0ab51764b7b86890f122  FT_RWAUGROUTING-v3-c2-L.dat
ac1266bda8f769d8af130f999cd38b64  FT_RWAUGROUTING-v4-c2-B.dat
02cd539d7781b69ed8c8b9fa5e316665  FT_RWAUGROUTING-v4-c2-L.dat
982171f7fa917eef2ac67153037dfb23  FT_RWAUGROUTING-v4-c2-L.dat
21d7f13b1205f5f99c54bfe6456f4ff9  FT_RWAUGROUTING-v5-c2-B.dat
841880810bcb005f54720eda57ff68b5  FT_RWAUGROUTING-v5-c2-L.dat
4a64b53e543ca0404a3d86cf63f5704e  FT_RWAUGSNMPOUT-v1-c2-B.dat
e27d2d5d3f7290c0438af58887770c4f  FT_RWAUGSNMPOUT-v1-c2-L.dat
c8def06c781fd0dc54e9d2ad17761a4d  FT_RWAUGSNMPOUT-v2-c2-B.dat
c4fdd886c699a7bbfa0da43db1bbed1d  FT_RWAUGSNMPOUT-v2-c2-L.dat
81821692f45c2ebd1492be720326d12e  FT_RWAUGSNMPOUT-v3-c2-B.dat
ca5c63f9b25399502783b5a6cb638517  FT_RWAUGSNMPOUT-v3-c2-L.dat
7776dbfc6cc30f8e4d45a1a0cec825b9  FT_RWAUGSNMPOUT-v4-c2-B.dat
77517cec816e2f0595b79330e09bb3cb  FT_RWAUGSNMPOUT-v4-c2-L.dat
a418fa84e005719d291ea094d09a2ee8  FT_RWAUGSNMPOUT-v4-c2-L.dat
abb72022b214182e74a4a83f33d9e0a0  FT_RWAUGSNMPOUT-v5-c2-B.dat
735a1aaaaa731e4ca09459f27a267a06  FT_RWAUGSNMPOUT-v5-c2-L.dat
e13c9ddbf88db7c164fa35a16fd42ed4  FT_RWAUGWEB-v1-c2-B.dat
75dd3b0e091b8b47bdf8aeef273cb885  FT_RWAUGWEB-v1-c2-L.dat
8c9ceee6ea01dccef2aadb04724d93d8  FT_RWAUGWEB-v2-c2-B.dat
3395f89ec192cef4cc2c480ff2cf5655  FT_RWAUGWEB-v2-c2-L.dat
ac7dd539cfef7dd1ec1cf7a3ed09ed71  FT_RWAUGWEB-v3-c2-B.dat
22098b52166e172e4ff3dd3b041b5d2d  FT_RWAUGWEB-v3-c2-L.dat
4f726e586357bd9ad368b271088cd19b  FT_RWAUGWEB-v4-c2-B.dat
55e088874c8a13d3bd4b0fa6eff7631e  FT_RWAUGWEB-v4-c2-L.dat
617e97e4083e5cdee40ea8681ee41ba6  FT_RWAUGWEB-v5-c2-B.dat
24735d528f6caaf2472817e4fbf1dbf0  FT_RWAUGWEB-v5-c2-L.dat
175297a49afe93c8e776d326e70c1a94  FT_RWFILTER-v1-c2-B.dat
45b67cf1370043ceac3a1990b3b11523  FT_RWFILTER-v1-c2-L.dat
50f7a64b64ab8a66f990f940e79acf32  FT_RWFILTER-v1-c2-L.dat
840d91d68c7fcaffdad266ea572e070c  FT_RWFILTER-v2-c2-B.dat
4d1f6e9a0d6cf1b5257cc9da08bfac46  FT_RWFILTER-v2-c2-L.dat
4d823f4f740170e0735f873a376a6a33  FT_RWFILTER-v2-c2-L.dat
9ff9fff30a8966510a1ace5225dcb13b  FT_RWFILTER-v3-c2-B.dat
54c445ea4d0720491f9a27751c4fdf0e  FT_RWFILTER-v3-c2-L.dat
9177d1a168fb7d7f5e2bce317fb70681  FT_RWFILTER-v4-c2-B.dat
74fd5636f9e60feea81afff77de8e5c0  FT_RWFILTER-v4-c2-L.dat
b1e599acbd38297668a26111e9a1a5ae  FT_RWFILTER-v5-c2-B.dat
03f10f91b38d587d3f637957f6da6ecb  FT_RWFILTER-v5-c2-L.dat
1113491db59268294475c1f17f2a3b43  FT_RWGENERIC-v0-c2-B.dat
ed644a6334b6539162bb2666e185b6b6  FT_RWGENERIC-v0-c2-B.dat
ebb5e5cd03075df9d16c4cdedfdff201  FT_RWGENERIC-v0-c2-L.dat
03b3626ea799141bcb04f7c8bb81729e  FT_RWGENERIC-v1-c2-B.dat
3cac6c8c86141be002e09f2a64f162b3  FT_RWGENERIC-v1-c2-B.dat
565bcd86bee05b2568a8e6b032d2cfde  FT_RWGENERIC-v1-c2-L.dat
cdbb0f75a7cc26655984cde91a93fba5  FT_RWGENERIC-v1-c2-L.dat
e4a92ceb39512bf4083ad5feb50c3c51  FT_RWGENERIC-v2-c2-B.dat
2341f474272c4757a43c8af83d10d22e  FT_RWGENERIC-v2-c2-L.dat
b9eebbab145c24a5b3a9319f2c08f1b4  FT_RWGENERIC-v3-c2-B.dat
cfcbb41b0943775e67f1ae0a79899386  FT_RWGENERIC-v3-c2-L.dat
e8ddb9c4f05b74741207c9857e24ecc1  FT_RWGENERIC-v3-c2-L.dat
4002bdb8d625cec9c2a2f4ff5df53d78  FT_RWGENERIC-v4-c2-B.dat
21461364e5d98839f80ef8748187a3f3  FT_RWGENERIC-v4-c2-L.dat
3d105fb623442acef5713822e40ee1e0  FT_RWGENERIC-v4-c2-L.dat
1f5c8950a3d830662dec12e39df8178d  FT_RWGENERIC-v5-c2-B.dat
4fd0fce59fc8842d8323ed0ecd912143  FT_RWGENERIC-v5-c2-L.dat
04290bbd4e3792378f4dff4a1f5a7ffe  FT_RWIPV6-v1-c2-B.dat
120d260058331f00a70c22d9c7336436  FT_RWIPV6-v1-c2-L.dat
9fac70d8c01c31593f9aebacbe162ec2  FT_RWIPV6-v2-c2-B.dat
44173848504cbd453a2879f1fe9dc928  FT_RWIPV6-v2-c2-L.dat
5159a43177c18c05b73173404a3fa582  FT_RWIPV6ROUTING-v1-c2-B.dat
36feaefacfeae8f026d3d99c6e987331  FT_RWIPV6ROUTING-v1-c2-L.dat
ab52c004814044775ace0d261355a2c7  FT_RWIPV6ROUTING-v2-c2-B.dat
c9fb85772fa528ac9ed27e20b6790c28  FT_RWIPV6ROUTING-v2-c2-L.dat
900f4e183068efc6e799d3462995ea6a  FT_RWIPV6ROUTING-v3-c2-B.dat
12c3ca82a05a3929d25fb0375a374f38  FT_RWIPV6ROUTING-v3-c2-L.dat
83e009f3e2ccade295f10b0d059936b6  FT_RWNOTROUTED-v1-c2-B.dat
0e255acc3bed430a78f894f092db26de  FT_RWNOTROUTED-v1-c2-L.dat
1dab0016c7378a3ae899c437c098b402  FT_RWNOTROUTED-v2-c2-B.dat
e93c395441b58b1ecfc265f4fc89fca1  FT_RWNOTROUTED-v2-c2-L.dat
802cd817a803c4444a5f19c83c641efd  FT_RWNOTROUTED-v3-c2-B.dat
1f985b55009ef1537d4043f1c2c069f0  FT_RWNOTROUTED-v3-c2-L.dat
7c5ae989965d10f340d66ebcc5b34142  FT_RWNOTROUTED-v4-c2-B.dat
0883071253b0d411a4999958a2de9b98  FT_RWNOTROUTED-v4-c2-L.dat
a3e1ec14056374ed4655e243caa8813a  FT_RWNOTROUTED-v5-c2-B.dat
0dbde685f35e61ac591acfe38eee5a3a  FT_RWNOTROUTED-v5-c2-L.dat
0b9db379ac08cbb91aa0cc9012d8e7d7  FT_RWROUTED-v1-c2-B.dat
8ff6912cb01fa853c3c2cae5618aa4cb  FT_RWROUTED-v1-c2-L.dat
4ae4cbc09fbb9f7f8d50918d9601fe6b  FT_RWROUTED-v2-c2-B.dat
5e00ae2d4a38399df53514f54f36caef  FT_RWROUTED-v2-c2-L.dat
594da013877ccded5e06ede473343806  FT_RWROUTED-v3-c2-B.dat
090182ea16bc33f0637c23b15f49b804  FT_RWROUTED-v3-c2-L.dat
342dcdfae3e7a564fd75e896efa86c9f  FT_RWROUTED-v4-c2-B.dat
af7b0cc4065f25f9f6636d37f9abc812  FT_RWROUTED-v4-c2-L.dat
57bff52f73267c20b7ea572be077cd5d  FT_RWROUTED-v5-c2-B.dat
ed24b39a73700af62b613aab278290eb  FT_RWROUTED-v5-c2-L.dat
4badb225c7c836289fed62a6f0f0caa7  FT_RWSPLIT-v1-c2-B.dat
119920643f6e44636f0522146e5456a3  FT_RWSPLIT-v1-c2-L.dat
9c4da6cca34b50ffccbbf02e121cbe32  FT_RWSPLIT-v2-c2-B.dat
3b0a1b69dfaa03686e504546d4dd523d  FT_RWSPLIT-v2-c2-L.dat
e24303f6099c208d48d3bb12847c0908  FT_RWSPLIT-v3-c2-B.dat
2c5f4c3ae6ce2be3ebf3497052a10df4  FT_RWSPLIT-v3-c2-L.dat
5dbcce6cb27c8b0bfacaae80be30cc15  FT_RWSPLIT-v4-c2-B.dat
2a8d703b45520b8e6b689002e67c3c65  FT_RWSPLIT-v4-c2-L.dat
66261ff71674601dbb26669f8bb78a6d  FT_RWSPLIT-v5-c2-B.dat
38e1540440f9b59e02859d971b23c148  FT_RWSPLIT-v5-c2-L.dat
b04aedb14475fb0ac4895791f4634483  FT_RWWWW-v1-c2-B.dat
05d8689cc5b0394f71bd5f3ee3913dde  FT_RWWWW-v1-c2-L.dat
5b85282b92664f3cbcef5452c9858d42  FT_RWWWW-v2-c2-B.dat
7aff9fcfda792e584d619442f974abe7  FT_RWWWW-v2-c2-L.dat
60f07f703982330978b3843b18304bb5  FT_RWWWW-v3-c2-B.dat
6d8e8d225daf544ddcb8b3b2f69590e6  FT_RWWWW-v3-c2-L.dat
de98d83fa8969ae3e26af2ebfe85abf2  FT_RWWWW-v4-c2-B.dat
8fffb563e2eb43257450e25f098afe7b  FT_RWWWW-v4-c2-L.dat
b0d369d8c9ba2d73dda356fb9abdd35e  FT_RWWWW-v5-c2-B.dat
a5bb40388a14669608958dcec52e9b11  FT_RWWWW-v5-c2-L.dat

# LZO-2.06 on 32-bit Ubuntu gives different results for two files
eec70dcca953c0dcefeeafbffae6bd97  FT_RWIPV6ROUTING-v3-c2-B.dat
a3345cae9921e02bb0e858552edd961b  FT_RWIPV6ROUTING-v3-c2-L.dat

# Snappy-1.1.8 gives different results on some files
bcef4f3584d4acb429c2f128307f108e FT_RWAUGMENTED-v4-c3-B.dat
1ae21aa47b260eebb76215194388c6b9 FT_RWAUGMENTED-v5-c3-B.dat
39bd9090b448aa8ed9da59706700b237 FT_RWAUGMENTED-v5-c3-L.dat
72d6ad86b427706c86ce91bb2ac3327e FT_RWAUGROUTING-v4-c3-L.dat
dcec039fa1b5638e3adb2c61738e781b FT_RWAUGROUTING-v5-c3-L.dat
3098f088feb82363e39571f8077b2ed5 FT_RWAUGSNMPOUT-v4-c3-B.dat
5ae5e26ee52af80b3b081ef894588bd9 FT_RWAUGSNMPOUT-v4-c3-L.dat
4e001b4aabe8eb0939edb49c0a22d661 FT_RWAUGWEB-v4-c3-B.dat
900f9eceaf5a25abc722b03cafc0435c FT_RWNOTROUTED-v3-c3-L.dat
e50c25f8cde5410837ea85951296b779 FT_RWNOTROUTED-v4-c3-L.dat
c983aa4a59066a037d0d1403c6884fbb FT_RWNOTROUTED-v5-c3-B.dat
ba7a218ec09e74c3998333e830e7ea51 FT_RWNOTROUTED-v5-c3-L.dat
02e5eed3f0fd24cd5150e8fa98bcebac FT_RWROUTED-v5-c3-B.dat
5e7dc83712b152ea1f474eddb2b007a4 FT_RWROUTED-v5-c3-L.dat
c4085b9be37d03e3f1be8324a6b2ff11 FT_RWSPLIT-v1-c3-B.dat
a2af50eb668775733d5951ab21f012ea FT_RWSPLIT-v1-c3-L.dat
91a10997453bf01ab0e7b16ca6efa843 FT_RWSPLIT-v2-c3-B.dat
6900422b6f37d6b0d7d075b4207872b6 FT_RWSPLIT-v2-c3-L.dat
866e1ec9b4b698d5494343b9df20df72 FT_RWSPLIT-v3-c3-B.dat
6b590a1ee21ec269f14bb22616eb7668 FT_RWSPLIT-v3-c3-L.dat
dcb85544b639af041a08cbd4601104ee FT_RWSPLIT-v4-c3-B.dat
b66a6f0da28bdaf28d8f52945b110a5a FT_RWSPLIT-v4-c3-L.dat
20ec45f9df3471b577cec9dc3a80cedd FT_RWSPLIT-v5-c3-B.dat
961578b0d4862a0fd89adf9bdc184799 FT_RWSPLIT-v5-c3-L.dat
7d9958a6d7c770a8b6fe4f09f9538dce FT_RWWWW-v1-c3-B.dat
475c154323c71b0739d4f0a2163cf39a FT_RWWWW-v1-c3-L.dat
405f7b49edab37af5edef6f9a6b92a9a FT_RWWWW-v2-c3-B.dat
cff69f6f22e17a01d983a63a905ffc72 FT_RWWWW-v2-c3-L.dat
13974fb774095db3d12362d7f3cad973 FT_RWWWW-v3-c3-B.dat
53c9c3b4bc0dd7229d504a42cb15671b FT_RWWWW-v3-c3-L.dat
a0d2e37ac04924b4aae6b1664d5808fa FT_RWWWW-v4-c3-B.dat
8d47f2317af93f0f29b5952d4bacbb94 FT_RWWWW-v4-c3-L.dat
8edb88cd8c3603ebe23d463657d2c97b FT_RWWWW-v5-c3-B.dat
9b5f66a63005daeffb9b55c9da979ef2 FT_RWWWW-v5-c3-L.dat

# Snappy-1.2.1 gives yet other results
0de5a4ba5a46ad4d3699a0925de5da89 FT_FLOWCAP-v2-c3-B.dat
923621611f4947440e0918eff641c02d FT_FLOWCAP-v2-c3-L.dat
d1afc4d96f10434fcfc7a130e7847d37 FT_FLOWCAP-v3-c3-B.dat
6e5f84e170ef37161650a98d71d96a7c FT_FLOWCAP-v3-c3-L.dat
d580ca3c5cc0f59cbae31ca1b4e0b5b7 FT_FLOWCAP-v4-c3-B.dat
383b43f310391a612219cc32d755fe73 FT_FLOWCAP-v4-c3-L.dat
8a060ba62d603f5e90c1dcab8c4f66f2 FT_FLOWCAP-v5-c3-B.dat
a150853e59862c47fcf97145fd4217ba FT_FLOWCAP-v5-c3-L.dat
7a5335113105a6851c520ae164faad7f FT_FLOWCAP-v6-c3-B.dat
616c0eebf104ae2c742c2eac7449a3f4 FT_FLOWCAP-v6-c3-L.dat
85d1f7f09f1eb42175d017f381d68f8a FT_RWAUGMENTED-v1-c3-B.dat
08d4009a2ba1afc1c1d8e54939961d26 FT_RWAUGMENTED-v1-c3-L.dat
98fba49910e379fbb4266b9e55af2dd1 FT_RWAUGMENTED-v2-c3-B.dat
bde6446acc4c681da4b17e98b5c3b874 FT_RWAUGMENTED-v2-c3-L.dat
fbe5b8972d267a90d3da9cbd337c5b18 FT_RWAUGMENTED-v3-c3-B.dat
11709e5dbb4da15c68e26f07145e7efc FT_RWAUGMENTED-v3-c3-L.dat
8c7be77fe081c04f7e5e2683f03153a9 FT_RWAUGMENTED-v4-c3-B.dat
b600cd2def1c077fb01bac08c81ea664 FT_RWAUGMENTED-v4-c3-L.dat
d61b674d216e0fa778ea60e6aa49f1e0 FT_RWAUGMENTED-v5-c3-B.dat
2886f05cd2b279bc44d1f3702f11f91c FT_RWAUGMENTED-v5-c3-L.dat
4049bc26596cd5af16875f89bfa77b03 FT_RWAUGMENTED-v6-c3-B.dat
340ee296997e0a7ed6ac67a20527717d FT_RWAUGMENTED-v6-c3-L.dat
621a74606ad113275477db61c2aeae1c FT_RWAUGROUTING-v1-c3-B.dat
535ba88b04023b812ebface2a9c56666 FT_RWAUGROUTING-v1-c3-L.dat
ecf3a5c48553f4d07c8c29c5d4adafc2 FT_RWAUGROUTING-v2-c3-B.dat
f2e0775fe0183fa71d1196954c04483f FT_RWAUGROUTING-v2-c3-L.dat
6514f223657e150a94cc1af721365a5c FT_RWAUGROUTING-v3-c3-B.dat
0801ec58130a708ecef8a9c0874e1795 FT_RWAUGROUTING-v3-c3-L.dat
0ce9b27162990bfa5fbaf8187549db90 FT_RWAUGROUTING-v4-c3-B.dat
30c28a6c52cd5adc8b171df2df5741c0 FT_RWAUGROUTING-v4-c3-L.dat
c65bdca87787dd09b26f66bbad7103cb FT_RWAUGROUTING-v5-c3-B.dat
78a4201fd958525dfe01bea35e0906f0 FT_RWAUGROUTING-v5-c3-L.dat
863911f282b4c63a4fde8439ae3344cb FT_RWAUGROUTING-v6-c3-B.dat
53dc712f71d4af79ac289a1119cdde00 FT_RWAUGROUTING-v6-c3-L.dat
6e5026a8d5aeeabe6ad2e57d54b440ac FT_RWAUGSNMPOUT-v1-c3-B.dat
18de887601fd71b023a6129745723ed7 FT_RWAUGSNMPOUT-v1-c3-L.dat
f827dde75af6b305ddc847da0c3b7ee2 FT_RWAUGSNMPOUT-v2-c3-B.dat
c34438875c0fac7e78fc3c06cf6e1055 FT_RWAUGSNMPOUT-v2-c3-L.dat
1e710dd59c8c59e52239770bb7b46ecb FT_RWAUGSNMPOUT-v3-c3-B.dat
65e3b1991b0392ffc2673359c228f5ee FT_RWAUGSNMPOUT-v3-c3-L.dat
258209af27cdb41564a88ca1afccf5ba FT_RWAUGSNMPOUT-v4-c3-B.dat
41097184eca9a9610ca1957e76bb23a5 FT_RWAUGSNMPOUT-v4-c3-L.dat
6b856e635a51c8db064fd4b888b5db70 FT_RWAUGSNMPOUT-v5-c3-B.dat
5b45a95e9dcbfac4ad0809cd7cb683e9 FT_RWAUGSNMPOUT-v5-c3-L.dat
acc5179ed3326ed85293ab398e9576c5 FT_RWAUGWEB-v1-c3-B.dat
068cd4d85a5b8ac871962e1d4bd08258 FT_RWAUGWEB-v1-c3-L.dat
599389280bce48045957da4a6afe5471 FT_RWAUGWEB-v2-c3-B.dat
df5ccd6a0dabe7c51fbadb3f370313b1 FT_RWAUGWEB-v2-c3-L.dat
2fb046497e6ebdfb4300fd9e5fc0cad6 FT_RWAUGWEB-v3-c3-B.dat
a20c7c9d69eb30ddf489cd34606ad4f5 FT_RWAUGWEB-v3-c3-L.dat
9d681af470eacdeae03e80ad43c6aa2f FT_RWAUGWEB-v4-c3-B.dat
abbaebacb3ffbf91a0c368a84ba95b2b FT_RWAUGWEB-v4-c3-L.dat
16936b96b29a54f7d9fd52cc0d3c8260 FT_RWAUGWEB-v5-c3-B.dat
cdb0d42c425bc4034a5bee427eeb3b3d FT_RWAUGWEB-v5-c3-L.dat
b965c0a66dc6bf85e13d3cdb868c53ae FT_RWAUGWEB-v6-c3-B.dat
3a97fabb43da259edca309237eddc5f1 FT_RWAUGWEB-v6-c3-L.dat
d9a59f92a6653bc63106cc183a3392ec FT_RWFILTER-v1-c3-B.dat
d41f449b0d9bca9589e8e81a367131a8 FT_RWFILTER-v1-c3-L.dat
24cd5b91512869ee61614bf50d82e1c1 FT_RWFILTER-v2-c3-B.dat
f522e3f1c6b27729a4a62bf505624dc8 FT_RWFILTER-v2-c3-L.dat
b2730c8f6a19e7e971fc2c33bde10bf7 FT_RWFILTER-v3-c3-B.dat
301150fd3a9d4fffcbc9bf20552cc31d FT_RWFILTER-v3-c3-L.dat
5def1650dfa50f279d7e03949bb8d004 FT_RWFILTER-v4-c3-B.dat
6ac5d861869caf7be3ee45b299f623ec FT_RWFILTER-v4-c3-L.dat
deb0393dd66332c3ad1cb14c39a5418e FT_RWFILTER-v5-c3-B.dat
335e27caf6828bb8cf55f8b8ddc3cb8d FT_RWFILTER-v5-c3-L.dat
bc43fa06bcae07d58a31274a4fe5df9a FT_RWGENERIC-v0-c3-B.dat
2eaef4713f2167ab20253b8d0341effa FT_RWGENERIC-v0-c3-L.dat
359ec81eca0acfb93ab6b8930e9ab479 FT_RWGENERIC-v1-c3-B.dat
7dc276a5eac7a2516a0c41747a6aef8c FT_RWGENERIC-v1-c3-L.dat
4fa41381748dde2f23345cd756bf7e9c FT_RWGENERIC-v2-c3-B.dat
2c481452bf89da0aa28180f639e92bfb FT_RWGENERIC-v2-c3-L.dat
445fbf6a50878e42373b431e23bfe3a1 FT_RWGENERIC-v3-c3-B.dat
6eb9d5a2fc2abe7fd4692b354a5b22d8 FT_RWGENERIC-v3-c3-L.dat
54f633a46e24917bf5a78fd36590e38c FT_RWGENERIC-v4-c3-B.dat
5ee4fd3755f6197c5952387145088836 FT_RWGENERIC-v4-c3-L.dat
52fda6fa7d5eac5664b4a9657c3738db FT_RWGENERIC-v5-c3-B.dat
8de7f653a2e8524849f865928eec23f5 FT_RWGENERIC-v5-c3-L.dat
c7edf306c0b7d7145a56d8df1edb97e3 FT_RWGENERIC-v6-c3-B.dat
17b937984ab373eb4c533a06ecf00bbc FT_RWGENERIC-v6-c3-L.dat
4f7c9cb862ab3479463993b87220825a FT_RWIPV6-v1-c3-B.dat
3ff8bc5b7e6b88fc13a652da01b6ccec FT_RWIPV6-v1-c3-L.dat
8c07eb99c0e80e0bc7b7234c750d8114 FT_RWIPV6-v2-c3-B.dat
6bc3b461bb07e9f2906f5258e8a4ef73 FT_RWIPV6-v2-c3-L.dat
f12fa379d0a03e916c785529b1a72b5e FT_RWIPV6-v3-c3-B.dat
14769bb3dd2c12dd0c41415667d4b51d FT_RWIPV6-v3-c3-L.dat
e3f88a96bc9d660a714e7d4ea107e2c3 FT_RWIPV6ROUTING-v1-c3-B.dat
98da961d7b9c3a9468700ee21a4d0989 FT_RWIPV6ROUTING-v1-c3-L.dat
38446cfb7c9f1621f6c3c3422a371f7a FT_RWIPV6ROUTING-v2-c3-B.dat
ef1d696f2e23ef4151974895f0d91c5c FT_RWIPV6ROUTING-v2-c3-L.dat
6676f90ba17722a22c9918d432928493 FT_RWIPV6ROUTING-v3-c3-B.dat
0eeffc8561b5eaf17ad9002e68a8f8ff FT_RWIPV6ROUTING-v3-c3-L.dat
2d6c165dffcc0e4166d1f90b66e90be5 FT_RWIPV6ROUTING-v4-c3-B.dat
8ecadb65bd7937e416de8376e3756e2e FT_RWIPV6ROUTING-v4-c3-L.dat
2c0c4461e462a8529774272780cfd35b FT_RWNOTROUTED-v1-c3-B.dat
3a9262e0750b5def4d806c0be98aa954 FT_RWNOTROUTED-v1-c3-L.dat
5dd87f4dea04f7cbfe4558bc81b97cca FT_RWNOTROUTED-v2-c3-B.dat
e9a62a14168f33d922bc788145821ee1 FT_RWNOTROUTED-v2-c3-L.dat
233f25c53d390b3df62b52c84f466308 FT_RWNOTROUTED-v3-c3-B.dat
393b67e2acc6c363ec7b5fc4c2667dca FT_RWNOTROUTED-v4-c3-B.dat
4221a3eee295ef1c375acc5b9460e138 FT_RWNOTROUTED-v5-c3-L.dat
44570312b4d933c0e4e25cd43d981e11 FT_RWROUTED-v1-c3-B.dat
b5aedac84bfae05b8cbe1972bc188f3b FT_RWROUTED-v1-c3-L.dat
5a6eb896b3430bbbe618c3c7c3810bd3 FT_RWROUTED-v2-c3-B.dat
a833ac28292fa9f1467dd0054124dcf1 FT_RWROUTED-v2-c3-L.dat
e275ad0a4eff38bc99e373f0fe1f179b FT_RWROUTED-v3-c3-B.dat
203ceb5ab836eba049fc2ce4d9c2c21a FT_RWROUTED-v3-c3-L.dat
4028135cb2fb652f607088fae37bbaa5 FT_RWROUTED-v4-c3-B.dat
fc75436f251a6cf742d7e3fa6420f8bf FT_RWROUTED-v4-c3-L.dat
abb8d7fd1a0fba5403d21ed120c38ba4 FT_RWROUTED-v5-c3-B.dat
0590320a7f706ae51c8291fd1291538e FT_RWROUTED-v5-c3-L.dat
046a31c2f57b2ca65e4baaf42bd13203 FT_RWSPLIT-v1-c3-B.dat
40ef6a6257552dc557c70a7331fc6532 FT_RWSPLIT-v2-c3-B.dat
46413c76145a858593afc28de8292288 FT_RWSPLIT-v5-c3-L.dat
f8bc6c24d9036aa6539e698aea409755 FT_RWWWW-v1-c3-B.dat
e30cc93f6c81bd3774c4f007fb0987f0 FT_RWWWW-v1-c3-L.dat
be57f3e782982ed989448930e77fb99a FT_RWWWW-v2-c3-B.dat
d5ef1b8b1022f0def3481cdb5af47c1e FT_RWWWW-v2-c3-L.dat

# Snappy-1.2.1 on Apple Silicon
9c77b593198fcbd77ea20598edf964f5 FT_FLOWCAP-v2-c3-B.dat
006cbca4b316cd852f77a1f7e61d7333 FT_FLOWCAP-v2-c3-L.dat
3196e531b5a7c2847560ecc7128e682e FT_FLOWCAP-v3-c3-B.dat
040cbe9e053bd059da5d7d3b2b1f460a FT_FLOWCAP-v3-c3-L.dat
3952ae7b9037138b910a6bcc0cc28e32 FT_FLOWCAP-v4-c3-B.dat
340446d230496efe0d154d734428f453 FT_FLOWCAP-v4-c3-L.dat
7f9002f8f6f407a9970b07804a0c7113 FT_FLOWCAP-v5-c3-B.dat
4bc380a2fa2aab426f5617c07e2e9283 FT_FLOWCAP-v5-c3-L.dat
2a54e1b6e41bfa1abd77bf6c6ff86a63 FT_FLOWCAP-v6-c3-B.dat
cf066fca6f8c1de3787a7537f18ef96e FT_FLOWCAP-v6-c3-L.dat
3b95eb25d6b8c180aeaea0b0e88fa3b0 FT_RWAUGMENTED-v1-c3-B.dat
cb0bf9f9f0ae8ff4153d6403793b57e3 FT_RWAUGMENTED-v1-c3-L.dat
b2821de850b356cd07ac3182ea54e9de FT_RWAUGMENTED-v2-c3-B.dat
0c92a34365c60a17a004a605f33f2e21 FT_RWAUGMENTED-v2-c3-L.dat
312e5e5bd4419807c31f4ebe255a99ca FT_RWAUGMENTED-v3-c3-B.dat
a00b2b55610d58a7426a9609d45adca8 FT_RWAUGMENTED-v3-c3-L.dat
5eb8bc4363fabf6663b2ac32888ee6b9 FT_RWAUGMENTED-v4-c3-B.dat
2dbe3111217925d7e4a1d43fc7a62ffa FT_RWAUGMENTED-v4-c3-L.dat
fe57a2b6363ba8014218f52e70053d4b FT_RWAUGMENTED-v5-c3-B.dat
0c7989e9d888b6231bc96ddfc7a725db FT_RWAUGMENTED-v5-c3-L.dat
14c8a5863230be1ada96ae6301b645f1 FT_RWAUGMENTED-v6-c3-B.dat
9fff0fecb58ac664713e65131891334d FT_RWAUGMENTED-v6-c3-L.dat
1e33aa62dacc3b5935e5fd9cc3262253 FT_RWAUGROUTING-v1-c3-B.dat
ed49ef6514ddd09b4be057e3f33006b5 FT_RWAUGROUTING-v1-c3-L.dat
9f207608cd36028f41b688ad19901593 FT_RWAUGROUTING-v2-c3-B.dat
406974bc90e6893b888ab5b0e089b7ea FT_RWAUGROUTING-v2-c3-L.dat
5e453b1625fada09f2178499322cdae5 FT_RWAUGROUTING-v3-c3-B.dat
4c5dd7d21b76d4da8082b0a06c444de9 FT_RWAUGROUTING-v3-c3-L.dat
9280ff3c59d327c7778b61a488d27821 FT_RWAUGROUTING-v4-c3-B.dat
7b03c57e6009b48d0a61501d2f346cea FT_RWAUGROUTING-v4-c3-L.dat
e989a5e7129062458f7e5bb95b39f376 FT_RWAUGROUTING-v5-c3-B.dat
1b8841348ed932419a40665420644ad3 FT_RWAUGROUTING-v5-c3-L.dat
ee9e8222980a3063d89a74cd133870b7 FT_RWAUGROUTING-v6-c3-B.dat
eabc50028cf1f6a1e304b8556f8f16a6 FT_RWAUGROUTING-v6-c3-L.dat
9c7970df33f9bb3d5f74db36ee79d8b7 FT_RWAUGSNMPOUT-v1-c3-B.dat
1c4c524401c4038982eba4abd1ff863a FT_RWAUGSNMPOUT-v1-c3-L.dat
6ee526bf7f81dfc1c53f01368762fa2c FT_RWAUGSNMPOUT-v2-c3-B.dat
5cdb2d158ce8978fec1eb9245129591c FT_RWAUGSNMPOUT-v2-c3-L.dat
637317846eac49744caedab9ee8ce516 FT_RWAUGSNMPOUT-v3-c3-B.dat
56ee5998f9832e0e269647ec30e3eaa0 FT_RWAUGSNMPOUT-v3-c3-L.dat
54986410b7f2ab92d3cf3de26dadf71d FT_RWAUGSNMPOUT-v4-c3-B.dat
d936f197bcc79893c48cad0bd197f1d2 FT_RWAUGSNMPOUT-v4-c3-L.dat
545939c3c5acda33e40d301e62c8174c FT_RWAUGSNMPOUT-v5-c3-B.dat
4de8dbad0c688e515ea78fbede01e432 FT_RWAUGSNMPOUT-v5-c3-L.dat
a84d4c49c04f022ed23964d768057f7c FT_RWAUGWEB-v1-c3-B.dat
9b1cafbd7f58831e96156a2d1ac86913 FT_RWAUGWEB-v1-c3-L.dat
2148fd58f0d25dd7c581373d341b1a43 FT_RWAUGWEB-v2-c3-B.dat
99683ff58e31e4af273a3b2d29edb033 FT_RWAUGWEB-v2-c3-L.dat
771481730c1e9ca986eee9148968ed6e FT_RWAUGWEB-v3-c3-B.dat
0c20c44c43e3fbd2a0a2ddc06c7b423d FT_RWAUGWEB-v3-c3-L.dat
573994d7628c884c2aa02fe7ec96f8e9 FT_RWAUGWEB-v4-c3-B.dat
d3a9810c9332a4554436ed60ce4abb07 FT_RWAUGWEB-v4-c3-L.dat
2d65011605152b64c1d2de5f09cbbb68 FT_RWAUGWEB-v5-c3-B.dat
2fc1d969c08ff5ab3b2874a24c07227f FT_RWAUGWEB-v5-c3-L.dat
740a9a9b40ab474b4c2694bd313d5a80 FT_RWAUGWEB-v6-c3-B.dat
ca8df3735d8b6e8602a10deb4a699c5d FT_RWAUGWEB-v6-c3-L.dat
89df4ca8c0b992b012a6d23078e83d21 FT_RWFILTER-v1-c3-B.dat
417018da6c4680e6783153d08f38fcc8 FT_RWFILTER-v1-c3-L.dat
215bfef24a668264c97d8d7e555ed66a FT_RWFILTER-v2-c3-B.dat
9ecab48cce99fe4e921c36dbd1c0867c FT_RWFILTER-v2-c3-L.dat
00f4fd343b8224af8f3139fb261af847 FT_RWFILTER-v3-c3-B.dat
51f54c0846f3cc635ec384b1a6d82a99 FT_RWFILTER-v3-c3-L.dat
01d801bd7e8e382aa82377830a81b18e FT_RWFILTER-v4-c3-B.dat
e6eab505fcb2cb0e2a56c8a39c0bdcbc FT_RWFILTER-v4-c3-L.dat
f39e3d8017fbf1ef360cc37805e2c3e5 FT_RWFILTER-v5-c3-B.dat
446c977e847b7373d95085b4d30a306a FT_RWFILTER-v5-c3-L.dat
e4e83d0b2c859e1d3aba421bd38348ab FT_RWGENERIC-v0-c3-B.dat
b81a0a6354910004064643931638da50 FT_RWGENERIC-v0-c3-L.dat
52264300a3a9f979616376a8170794f6 FT_RWGENERIC-v1-c3-B.dat
11b36f49e07dc7cb065eb4e1129ec406 FT_RWGENERIC-v1-c3-L.dat
48bce19dcc08406a90d31506aba13e99 FT_RWGENERIC-v2-c3-B.dat
4515152da824a619906b24df16085c05 FT_RWGENERIC-v2-c3-L.dat
8b060d883d755ed5817ef1347aee49b1 FT_RWGENERIC-v3-c3-B.dat
bd9d5ad6f22433398465fc8f3ea6529a FT_RWGENERIC-v3-c3-L.dat
96e9be3ec48b709b007e144c023038a3 FT_RWGENERIC-v4-c3-B.dat
27104c000ce98f1e217974d85d0516fe FT_RWGENERIC-v4-c3-L.dat
56a63084f89cef32502348d2196992ce FT_RWGENERIC-v5-c3-B.dat
7040016c3a4f7754c2dbb5e1ffb0d395 FT_RWGENERIC-v5-c3-L.dat
343b72798b94fbca434703254112eb4d FT_RWGENERIC-v6-c3-B.dat
22b77bd2dd16f7691ec2c4b1b7f52581 FT_RWGENERIC-v6-c3-L.dat
ca169969e8e0ba8e5da19bc7f8768191 FT_RWIPV6-v1-c3-B.dat
847ef08edc0d2efe5d4cdf543801dd70 FT_RWIPV6-v1-c3-L.dat
6e2ac10206ddc2eba06963c711bf58ba FT_RWIPV6-v2-c3-B.dat
ffa06ad8f0ef28adc68c3b1b95a0376d FT_RWIPV6-v2-c3-L.dat
191b5272fc84b9dc0b93e8131cff15ba FT_RWIPV6-v3-c3-B.dat
66bbb7092e5a14bd5de82fae98e5afcf FT_RWIPV6-v3-c3-L.dat
8a16d060ae63f7f2b3e8f15fee9aa2e7 FT_RWIPV6ROUTING-v1-c3-B.dat
79be33358f21ea771a3adfc3741483b6 FT_RWIPV6ROUTING-v1-c3-L.dat
eb16a8ad99e81635302de6c1a0098c37 FT_RWIPV6ROUTING-v2-c3-B.dat
e47f063ca462d82a3331273579e83572 FT_RWIPV6ROUTING-v2-c3-L.dat
c345685927abec406d5dad15f6c057dd FT_RWIPV6ROUTING-v3-c3-B.dat
605137e165e7823ef893775ac563e2a8 FT_RWIPV6ROUTING-v3-c3-L.dat
0fb8fd00fd4f6380b4e572aa324ac22d FT_RWIPV6ROUTING-v4-c3-B.dat
12740dbe5dbd4601db0102b6907eea5d FT_RWIPV6ROUTING-v4-c3-L.dat
24f78829e58497a4879432c6a055c9a4 FT_RWNOTROUTED-v1-c3-B.dat
133ec4bacc789e6e3a87bba82be9fa8b FT_RWNOTROUTED-v1-c3-L.dat
523658154358da4173a03faaea05d338 FT_RWNOTROUTED-v2-c3-B.dat
4a45dbef0d1b93385dbba5c12d255658 FT_RWNOTROUTED-v2-c3-L.dat
ec7168fcd1ee0db94b6df535ee7e9a19 FT_RWNOTROUTED-v3-c3-B.dat
20b5e20f4c0d156143dcad07d76ee859 FT_RWNOTROUTED-v4-c3-B.dat
ae5f4c0a325c2f2e1453d4646407c389 FT_RWNOTROUTED-v5-c3-L.dat
467a0ff8f44007ed6824f74fd59da723 FT_RWROUTED-v1-c3-B.dat
366e0974455d329cbb1222e009632492 FT_RWROUTED-v1-c3-L.dat
7d6df643b8964c92428ca63022762cdf FT_RWROUTED-v2-c3-B.dat
f25aeb2069e74dcaa10ed7453ce3200a FT_RWROUTED-v2-c3-L.dat
007bdf8cc0b2bb5cc964868dea257ac8 FT_RWROUTED-v3-c3-B.dat
3aa4ce7a12a6b9cccaeafe9ac2c7cc45 FT_RWROUTED-v3-c3-L.dat
4786772f6ac8bfc8ed833853c35d3011 FT_RWROUTED-v4-c3-B.dat
8af000623a01a5f279d02bbf6d331aef FT_RWROUTED-v4-c3-L.dat
ec82a1dbf7dd3c2cb2573740fe35b98a FT_RWROUTED-v5-c3-B.dat
c4ef8d0391066f2fc9a49397e36e6ca9 FT_RWROUTED-v5-c3-L.dat
64ebe936c00d059026f7c3cc05ae0250 FT_RWSPLIT-v1-c3-B.dat
7122329ab7d10976df5e7a48cf916ccd FT_RWSPLIT-v2-c3-B.dat
1c4b4d7583f03c86cbe03dc092cbf832 FT_RWSPLIT-v5-c3-L.dat
9e0edaa98db89b898144b484dd58dc1a FT_RWWWW-v1-c3-B.dat
7fe66369ccb48c6fe0d79e06c48dc827 FT_RWWWW-v1-c3-L.dat
134229a22d74afdc955831305ea5a6b8 FT_RWWWW-v2-c3-B.dat
f392a9a016e011eac085138c8d57bb7b FT_RWWWW-v2-c3-L.dat

#########################################################################
#
#   These are the expected MD5 values when reading the data with rwcat
#
#   Values should not depend on compression or byte-order
#
f3159654d08c88aeac35ad14ff3ae97b FT_FLOWCAP-v2
920deb0e2989c6efedd08fb04315a4fa FT_FLOWCAP-v3
920deb0e2989c6efedd08fb04315a4fa FT_FLOWCAP-v4
920deb0e2989c6efedd08fb04315a4fa FT_FLOWCAP-v5
de1fa25faa867028b7bb5dc3adc15652 FT_FLOWCAP-v6
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGMENTED-v1
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGMENTED-v2
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGMENTED-v3
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGMENTED-v4
65cac56a571d430a404b04ac9f7501df FT_RWAUGMENTED-v5
65cac56a571d430a404b04ac9f7501df FT_RWAUGMENTED-v6
7c34164b346174045c02758c0dc71452 FT_RWAUGROUTING-v1
7c34164b346174045c02758c0dc71452 FT_RWAUGROUTING-v2
7c34164b346174045c02758c0dc71452 FT_RWAUGROUTING-v3
7c34164b346174045c02758c0dc71452 FT_RWAUGROUTING-v4
64e11be15c96dcad26d74c669601aafb FT_RWAUGROUTING-v5
64e11be15c96dcad26d74c669601aafb FT_RWAUGROUTING-v6
2689712e2cf13408179b382aed4ae794 FT_RWAUGSNMPOUT-v1
2689712e2cf13408179b382aed4ae794 FT_RWAUGSNMPOUT-v2
2689712e2cf13408179b382aed4ae794 FT_RWAUGSNMPOUT-v3
2689712e2cf13408179b382aed4ae794 FT_RWAUGSNMPOUT-v4
7581bb6f9ceebeecf8050e32f63af540 FT_RWAUGSNMPOUT-v5
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGWEB-v1
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGWEB-v2
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGWEB-v3
13aecd2926f13bb8673c65fa22c1c5a6 FT_RWAUGWEB-v4
65cac56a571d430a404b04ac9f7501df FT_RWAUGWEB-v5
65cac56a571d430a404b04ac9f7501df FT_RWAUGWEB-v6
a690170760b44a66faa52244d05db0d9 FT_RWFILTER-v1
a690170760b44a66faa52244d05db0d9 FT_RWFILTER-v2
64af2c4729f3cbbb7f8e7fb183deca61 FT_RWFILTER-v3
d4636ac309e6d3eebbc7fb2b701fd60b FT_RWFILTER-v4
d4636ac309e6d3eebbc7fb2b701fd60b FT_RWFILTER-v5
f3159654d08c88aeac35ad14ff3ae97b FT_RWGENERIC-v0
f3159654d08c88aeac35ad14ff3ae97b FT_RWGENERIC-v1
481d119bb59765639182e3c0ded5de69 FT_RWGENERIC-v2
79b68daea450dfcc27e68022e2bac97c FT_RWGENERIC-v3
79b68daea450dfcc27e68022e2bac97c FT_RWGENERIC-v4
79b68daea450dfcc27e68022e2bac97c FT_RWGENERIC-v5
79b68daea450dfcc27e68022e2bac97c FT_RWGENERIC-v6
974858603dd918ad2ca89e38c3f54a3e FT_RWIPV6-v1
65cac56a571d430a404b04ac9f7501df FT_RWIPV6-v2
65cac56a571d430a404b04ac9f7501df FT_RWIPV6-v3
79b68daea450dfcc27e68022e2bac97c FT_RWIPV6ROUTING-v1
781caa1168810c78a692912631ffa457 FT_RWIPV6ROUTING-v2
79b68daea450dfcc27e68022e2bac97c FT_RWIPV6ROUTING-v3
79b68daea450dfcc27e68022e2bac97c FT_RWIPV6ROUTING-v4
3cb74d758bf8ff9bade6aa8157c38cad FT_RWNOTROUTED-v1
3cb74d758bf8ff9bade6aa8157c38cad FT_RWNOTROUTED-v2
e0e4e88791fb9bdd0e2b7854275018af FT_RWNOTROUTED-v3
e0e4e88791fb9bdd0e2b7854275018af FT_RWNOTROUTED-v4
e0e4e88791fb9bdd0e2b7854275018af FT_RWNOTROUTED-v5
5b75a83af9c46fd4a5ee183cc56cb242 FT_RWROUTED-v1
5b75a83af9c46fd4a5ee183cc56cb242 FT_RWROUTED-v2
b842fc7d6e3115e856251b17d12fc2fe FT_RWROUTED-v3
b842fc7d6e3115e856251b17d12fc2fe FT_RWROUTED-v4
b842fc7d6e3115e856251b17d12fc2fe FT_RWROUTED-v5
d392190c5ae0c1504f9e1b2abeb55d1c FT_RWSPLIT-v1
d392190c5ae0c1504f9e1b2abeb55d1c FT_RWSPLIT-v2
3af293c57a4d7ff119048cb97ce1488a FT_RWSPLIT-v3
3af293c57a4d7ff119048cb97ce1488a FT_RWSPLIT-v4
3af293c57a4d7ff119048cb97ce1488a FT_RWSPLIT-v5
d392190c5ae0c1504f9e1b2abeb55d1c FT_RWWWW-v1
d392190c5ae0c1504f9e1b2abeb55d1c FT_RWWWW-v2
3af293c57a4d7ff119048cb97ce1488a FT_RWWWW-v3
3af293c57a4d7ff119048cb97ce1488a FT_RWWWW-v4
3af293c57a4d7ff119048cb97ce1488a FT_RWWWW-v5
