/*
** Copyright (C) 2006-2024 by Carnegie Mellon University.
**
** @OPENSOURCE_LICENSE_START@
**
** SiLK 3.23
**
** Copyright 2024 Carnegie Mellon University.
**
** NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
** INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
** UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
** IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
** FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
** OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
** MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
** TRADEMARK, OR COPYRIGHT INFRINGEMENT.
**
** Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
** contact permission@sei.cmu.edu for full terms.
**
** [DISTRIBUTION STATEMENT A] This material has been approved for public
** release and unlimited distribution.  Please see Copyright notice for
** non-US Government use and distribution.
**
** This Software includes and/or makes use of Third-Party Software each
** subject to its own license.
**
** DM24-1064
**
** @OPENSOURCE_LICENSE_END@
*/

/*
**  Test program for skiobuf.c
*/


#include <silk/silk.h>

RCSIDENT("$SiLK: skiobuf-test.c 11854b6e2830 2024-06-06 20:15:57Z mthomas $");

#include <silk/skstream.h>
#include <silk/utils.h>
#include "skiobuf.h"


#define FAIL do {assert(0); skAbort();} while (0)

/* where to send --help output */
#define USAGE_FH stdout

/* Linux exits when using a block size larger than 1m */
#define TEST_MAX_BLOCKSIZE   ((SKIOBUF_MAX_BLOCKSIZE < 0x100000)        \
                              ? SKIOBUF_MAX_BLOCKSIZE : 0x100000)

/* Minimum block size */
#define TEST_MIN_BLOCKSIZE   0x200

/* Characters used to create the data */
static char g_data[] =
    "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

/* this will be set to strlen(g_data) */
static int datalen;

/* template for temporary file */
static char template[4096];

/* whether to run the timing test */
static int timing = 0;

/* min block size to use for timing test */
static uint32_t min_blk = TEST_MIN_BLOCKSIZE;

/* max block size to use for timing test */
static uint32_t max_blk = TEST_MAX_BLOCKSIZE;


/* OPTIONS SETUP */

enum {
    OPT_TIMING,
    OPT_MAX_BLOCKSIZE,
    OPT_MIN_BLOCKSIZE
};

static struct option appOptions[] = {
    {"timing",          NO_ARG,       0, OPT_TIMING},
    {"max-blocksize",   REQUIRED_ARG, 0, OPT_MAX_BLOCKSIZE},
    {"min-blocksize",   REQUIRED_ARG, 0, OPT_MIN_BLOCKSIZE},
    {0,0,0,0}           /* sentinel entry */
};

static const char *appHelp[] = {
    "Run a timing test. Def. No",
    "Set maximum block size for timing test. Def. ",
    "Set minimum block size for timing test. Def. ",
    NULL
};


/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.
 */
static void
appUsageLong(
    void)
{
    FILE *fh = USAGE_FH;
    int i;

#define USAGE_MSG                                                       \
    ("[SWITCHES]\n"                                                     \
     "\tRun tests to check skiobuf\n")

    fprintf(fh, "%s %s", skAppName(), USAGE_MSG);
    fprintf(fh, "\nSWITCHES:\n");
    skOptionsDefaultUsage(fh);

    for (i = 0; appOptions[i].name; ++i) {
        fprintf(fh, "--%s %s. ", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]));
        switch (appOptions[i].val) {
          case OPT_MAX_BLOCKSIZE:
            fprintf(fh, "%s%u\n", appHelp[i], TEST_MAX_BLOCKSIZE);
            break;
          case OPT_MIN_BLOCKSIZE:
            fprintf(fh, "%s%u\n", appHelp[i], TEST_MIN_BLOCKSIZE);
            break;
          default:
            fprintf(fh, "%s\n", appHelp[i]);
            break;
        }
    }
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    Run tests for value of switch specified.
 */
static int
appOptionsHandler(
    clientData          cData,
    int                 opt_index,
    char               *opt_arg)
{
    int rv;

    (void)cData;

    switch (opt_index) {
      case OPT_TIMING:
        timing = 1;
        break;

      case OPT_MAX_BLOCKSIZE:
        rv = skStringParseUint32(&max_blk, opt_arg, 0, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_MIN_BLOCKSIZE:
        rv = skStringParseUint32(&min_blk, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;
    }

    return 0;                     /* OK */


  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/* select a random character */
static char
rval(
    void)
{
    int i = (int)((float)datalen * (rand() / (RAND_MAX + 1.0)));
    return g_data[i];
}


/*
 *    Create a memory map, fill it, and return a pointer to the memory.
 *
 *    Create a temporary file using `filename` as the template for the
 *    filename.  Use the temporary file to create an mmap and write
 *    `numrecs` `recsize`-byte records to the mmap.
 */
static char *
create_test_data(
    char              **filename,
    unsigned            recsize,
    unsigned            numrecs)
{
    int fd;
    unsigned i, j;
    ssize_t rv;
    off_t   ov;
    char c;
    void *map;
    char *name = strdup(template);
    char *buf;

    fd = mkstemp(name);
    if (fd == -1) {
        skAppPrintSyserror("Unable to open temporary file");
        FAIL;
    }
    ov = lseek(fd, recsize * numrecs, SEEK_SET);
    if (ov == -1) {
        skAppPrintSyserror("Unable to seek in temporary file");
        FAIL;
    }
    rv = write(fd, "", 1);
    if (rv != 1) {
        skAppPrintSyserror("Unable to write to temporary file");
        FAIL;
    }
    map = mmap(0, recsize * numrecs, PROT_READ | PROT_WRITE, MAP_SHARED,
               fd, 0);
    if (map == MAP_FAILED) {
        skAppPrintSyserror("Unable to create memory map");
        FAIL;
    }
    close(fd);

    buf = (char *)map;
    for (i = 0; i < numrecs; i++) {
        c = rval();
        for (j = 0; j < recsize; j++) {
            *buf++ = c;
        }
    }

    *filename = name;
    return (char *)map;
}


/*
 *    Delete (unmap) the test data file.
 */
static void
delete_test_data(
    char               *name,
    char               *data,
    unsigned            recsize,
    unsigned            numrecs)
{
    munmap(data, recsize * numrecs);
    unlink(name);
    free(name);
}

/*
 *    Writes the buffer `data` to disk and reads it back and compares the
 *    results.
 *
 *    Processes the data as `numrecs` records each of size `recsize`.  The
 *    skIOBuf uses a block size of `blocksize` and the compression method
 *    `method.
 *
 *    If `skipafter` is non-zero and less than `numrecs`, the first
 *    `skipafter` records are read and compared for accuracy, and the next
 *    `skipfor` records are read but not checked.  The next `skipafter` are
 *    compared and then the next `skipfor` ignored, repeating until the end of
 *    the data.
 */
static void
test(
    int                 method,
    uint32_t            blocksize,
    unsigned            skipafter,
    unsigned            skipfor,
    const char         *data,
    unsigned            recsize,
    unsigned            numrecs)
{
    char *name = strdup(template);
    int fd;
    sk_iobuf_t *buf = NULL;
    int rv;
    off_t ov;
    unsigned i;
    const char *p;
    uint32_t i32;
    ssize_t out;
    off_t off;
    char *readbuf;
    struct timeval t[4];
    char methname[128];
    off_t sz;

    if (skipafter >= numrecs) {
        skipafter = 0;
    }

    readbuf = malloc(1 + recsize);
    if (NULL == readbuf) {
        FAIL;
    }

    skCompMethodGetName(methname, sizeof(methname), method);

    fd = mkstemp(name);
    if (fd == -1) {
        FAIL;
    }

    gettimeofday(&t[0], NULL);

    buf = skIOBufCreate(SK_IO_WRITE);
    if (buf == NULL) {
        FAIL;
    }

    rv = skIOBufSetRecordSize(buf, recsize);
    if (rv == -1) {
        FAIL;
    }

    rv = skIOBufSetBlockSize(buf, blocksize);
    if (rv == -1) {
        FAIL;
    }

    rv = skIOBufBind(buf, fd, method);
    if (rv == -1) {
        FAIL;
    }

    /* accept a max limit of 24MB */
    i32 = skIOBufUpperCompBlockSize(buf);
    /* fprintf(stderr, "%" PRIu32 "\n", i32); */
    if (i32 > 24 * 1024 * 1024) {
        FAIL;
    }

    p = data;
    for (i = 0; i < numrecs; i++) {
        out = skIOBufWrite(buf, p, recsize);
        if (out != (int)recsize) {
            FAIL;
        }
        p += recsize;
    }
    off = skIOBufFlush(buf);
    if (off == -1) {
        FAIL;
    }

    gettimeofday(&t[1], NULL);

    off = skIOBufTotalUpperBound(buf);
    if (off == -1) {
        FAIL;
    }

    skIOBufDestroy(buf);
    buf = NULL;

    sz = lseek(fd, 0, SEEK_CUR);

    ov = lseek(fd, 0, SEEK_SET);
    if (ov == -1) {
        FAIL;
    }

    buf = skIOBufCreate(SK_IO_READ);
    if (buf == NULL) {
        FAIL;
    }

    if (0 == skipafter) {

        gettimeofday(&t[2], NULL);

        rv = skIOBufBind(buf, fd, method);
        if (rv == -1) {
            FAIL;
        }

        p = data;
        for (i = 0; i < numrecs; i++) {
            out = skIOBufRead(buf, readbuf, recsize);
            if (out != recsize) {
                FAIL;
            }
            if (0 != memcmp(p, readbuf, recsize)) {
                FAIL;
            }
            p += recsize;
        }

    } else {
        unsigned skip;
        int skipping;

        gettimeofday(&t[2], NULL);

        rv = skIOBufBind(buf, fd, method);
        if (rv == -1) {
            FAIL;
        }

        p = data;
        skip = 1;
        skipping = 0;
        for (i = 0; i < numrecs; i++) {
            if (skipping && (skip == skipfor)) {
                skip = 1;
                skipping = 0;
            } else if (skip == skipafter) {
                skip = 1;
                skipping = 1;
            }
            if (skipping) {
                out = skIOBufRead(buf, NULL, recsize);
                if (out != recsize) {
                    FAIL;
                }
            } else {
                out = skIOBufRead(buf, readbuf, recsize);
                if (out != recsize) {
                    FAIL;
                }
                if (0 != memcmp(p, readbuf, recsize)) {
                    FAIL;
                }
            }
            p += recsize;
            skip++;
        }

    }

    out = skIOBufRead(buf, readbuf, 1);
    if (out != 0) {
        FAIL;
    }

    gettimeofday(&t[3], NULL);

    skIOBufDestroy(buf);
    buf = NULL;

    close(fd);

    unlink(name);
    free(name);
    free(readbuf);

    for (i = 1; i < 4; i+= 2) {
        t[i].tv_sec -= t[i - 1].tv_sec;
        t[i].tv_usec -= t[i - 1].tv_usec;
        if (t[i].tv_usec < 0) {
            t[i].tv_usec += 1000000;
            --t[i].tv_sec;
        }
    }

    printf("%3ld.%06ld|%3ld.%06ld|%6s|%12ld|%4u|%10u|%10u|%3u|%3u|\n",
           t[1].tv_sec, (long)t[1].tv_usec, t[3].tv_sec, (long)t[3].tv_usec,
           methname, (long)sz, recsize, numrecs, blocksize, skipafter, skipfor);
}


static void
timing_test(
    const char   *data,
    unsigned      recsize,
    unsigned      numrecs)
{
    int meth;
    unsigned blk_size;

    /*
     *  Args to the test() function
     *
     *  int                 method
     *  uint32_t            blocksize
     *  unsigned            skipafter
     *  unsigned            skipfor
     *  char               *data
     *  unsigned            recsize
     *  unsigned            numrecs
     */

    for (meth = 0; skCompMethodCheck(meth) > 0; ++meth) {
        if (skCompMethodCheck(meth) != SK_COMPMETHOD_IS_AVAIL) {
            continue;
        }

        for (blk_size = min_blk;
             blk_size <= max_blk;
             blk_size <<= 1)
        {
            test(meth, blk_size, 0, 0, data, recsize, numrecs);

            /* if the default blocksize falls between the next value and this
             * one, run a test using the default size */
            if ((blk_size << 1) > SKIOBUF_DEFAULT_BLOCKSIZE
                && blk_size < SKIOBUF_DEFAULT_BLOCKSIZE)
            {
                test(meth, SKIOBUF_DEFAULT_BLOCKSIZE, 0, 0,
                     data, recsize, numrecs);
            }
        }

        printf("\n");
    }
}


static void
simple_test(
    const char   *data,
    unsigned      recsize,
    unsigned      numrecs)
{
    test(SK_COMPMETHOD_NONE, SKIOBUF_DEFAULT_BLOCKSIZE, 0, 0,
         data, recsize, numrecs);
    test(SK_COMPMETHOD_NONE, 100, 50, 200,
         data, recsize, numrecs);

#if SK_ENABLE_ZLIB
    test(SK_COMPMETHOD_ZLIB, SKIOBUF_DEFAULT_BLOCKSIZE, 0, 0,
         data, recsize, numrecs);
    test(SK_COMPMETHOD_ZLIB, 100, 50, 200,
         data, recsize, numrecs);
#endif

#if SK_ENABLE_LZO
    test(SK_COMPMETHOD_LZO1X, SKIOBUF_DEFAULT_BLOCKSIZE, 0, 0,
         data, recsize, numrecs);
    test(SK_COMPMETHOD_LZO1X, 100, 50, 200,
         data, recsize, numrecs);
#endif

#if SK_ENABLE_SNAPPY
    test(SK_COMPMETHOD_SNAPPY, SKIOBUF_DEFAULT_BLOCKSIZE, 0, 0,
         data, recsize, numrecs);
    test(SK_COMPMETHOD_SNAPPY, 100, 50, 200,
         data, recsize, numrecs);
#endif
}


int main(int argc, char *argv[])
{
    SILK_FEATURES_DEFINE_STRUCT(features);
    const char *tmpdir;
    char *testfname;
    char *testfile;
    int rec_size;
    unsigned num_rec;
    int arg_index;

    datalen = strlen(g_data);

    skAppRegister(argv[0]);
    skAppVerifyFeatures(&features, NULL);
    skOptionsSetUsageCallback(&appUsageLong);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)) {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* parse options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        skAppUsage();           /* never returns */
    }

    /* Get the temporary directory and use it to create a template for
     * temporary files */
    tmpdir = skTempDir(NULL, NULL);
    if (NULL == tmpdir) {
        skAppPrintErr("Unable to get temporary directory");
        FAIL;
    }
    snprintf(template, sizeof(template)-1, "%s/skiobuf-test.XXXXXX", tmpdir);

    /* Choose how much data to process */
    if (timing) {
        rec_size = 100;
        num_rec = 10000000;
    } else {
        rec_size = 10;
        num_rec = 2 * (SKIOBUF_DEFAULT_BLOCKSIZE + rec_size);
    }

    /* Create the test data using an mmap in the temp dir */
    testfile = create_test_data(&testfname, rec_size, num_rec);

    printf("%10s|%10s|%6s|%12s|%4s|%10s|%10s|%3s|%3s|\n",
           "write time", "read time", "method", "filesize", "rec", "numrecs",
           "blocksize", "aft", "for");

    if (timing) {
        timing_test(testfile, rec_size, num_rec);
    } else {
        simple_test(testfile, rec_size, num_rec);
    }

    delete_test_data(testfname, testfile, rec_size, num_rec);

    skAppUnregister();
    return 0;
}

/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
