#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-silk-discard-when-ipset-v6.pl 323d2cb74c99 2024-05-30 18:17:09Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsetbuild = check_silk_app('rwsetbuild');
my $rwsort = check_silk_app('rwsort');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');
$file{v6data} = get_data_or_exit77('v6data');

# verify that required features are available
check_features(qw(ipv6));

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the IPset files
my %ipset = (
    internal_v4 => "$tmpdir/internal-v4.set",
    external_v4 => "$tmpdir/external-v4.set",
    null_v4     => "$tmpdir/null-v4.set",
    internal_v6 => "$tmpdir/internal-v6.set",
    external_v6 => "$tmpdir/external-v6.set",
    null_v6     => "$tmpdir/null-v6.set",
    );
my %blocks = (
    internal_v4 => '192.168.x.x',
    external_v4 => '10.0.0.0/8',
    null_v4     => '172.16.0.0/13',
    internal_v6 => '2001:db8:c0:a8::x:x',
    external_v6 => '2001:db8:a:x::x:x',
    null_v6     => '2001:db8:ac:10-17::x:x',
    );

for my $k (keys %ipset) {
    my $cmd = "echo '$blocks{$k}' | $rwsetbuild - '$ipset{$k}'";
    check_md5_output('d41d8cd98f00b204e9800998ecf8427e', $cmd);
}

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
{
    my $regex = join "|", keys %ipset;
    my $sensor_conf_text;
    get_config_body(\$sensor_conf_text);
    $sensor_conf_text =~ s,\$\{($regex)\},$ipset{$1},g;
    make_config_file($sensor_conf, \$sensor_conf_text);
}

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--sensor-conf=$sensor_conf",
                     "--copy $file{data}:incoming",
                     "--copy $file{v6data}:incoming",
                     "--limit=972788",
                     "--basedir=$tmpdir",
                     "--",
                     "--polling-interval=5",
                     "--flat-archive",
    );

# run it and check the MD5 hash of its output
check_md5_output('69f68cb84613ff60f57979dc63605dfd', $cmd);

# the following directories should be empty
verify_empty_dirs($tmpdir, qw(error incoming incremental sender));

# input files should now be in the archive directory
verify_directory_files("$tmpdir/archive", $file{data}, $file{v6data});

# path to the data directory
my $data_dir = "$tmpdir/root";
die "ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# number of files to find in the data directory
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every packed file we expect to find.  Read
# these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $data_dir);

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files in root; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename
    s,^.*/,,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};
    ++$file_count;

    # do the MD5 sums match?  Use rwsort since input has two files,
    # and we don't know which was processed first.  IPv6 files do not
    # use the bytes/packet ratio, so no rounding issue to work-around
    check_md5_output($md5_map{$_},
                     ("$rwsort --fields=stime,sip $path "
                      ." | $rwcat --compression=none --byte-order=little"));
}

sub get_config_body
{
    my ($string_ref) = @_;

    $$string_ref = <<'EOF';
# the sensor.conf file for this test
probe P0 silk
    poll-directory ${incoming}
    quirks zero-packets
end probe

group internal
    ipsets ${internal_v4}
    ipsets "${internal_v6}"
end group
group external
    ipsets "${external_v6}" ${external_v4}
end group
group null
    ipsets "${null_v4}"
    ipsets ${null_v6}
end group

sensor S0
    silk-probes P0

    discard-when source-ipsets @null
    discard-when destination-ipsets ${null_v6} ${null_v4}

    internal-ipsets  @internal
    external-ipsets  @external
    null-ipsets      ${null_v4} "${null_v6}"
end sensor
EOF
}

__END__
5183a9d82976c6e01ebb86fc6e485095 in-S0_20090212.00
b207b958735c508876d11076a583d24c in-S0_20090212.01
d6e4d27e2b6825fcce08bd873627f0fe in-S0_20090212.02
0daaeea2e8941236bcf49611daf9cb7c in-S0_20090212.03
20689138ee0f2c9139f6cfeda94510ed in-S0_20090212.04
2386a66b36e7d8e5e6317e17f109e53a in-S0_20090212.05
96165bdae630b41585e912c5570db131 in-S0_20090212.06
e7db3ba10780d317249e585548cfcbef in-S0_20090212.07
4555982bb21317c6822209075ea1447e in-S0_20090212.08
abdb8ee9df3cd81f92c192fb99a5a5c7 in-S0_20090212.09
bae6fae358e65ec16cf36eb72be844fe in-S0_20090212.10
bd5cdfe3bfe2f46f90ef5592291f6098 in-S0_20090212.11
d35886691e6f4fc1d637f50ddd1eff32 in-S0_20090212.12
6eec3db5b97bfb1ee6b5f6e830aaf19b in-S0_20090212.13
26fbe66574938670341b541a21486de9 in-S0_20090212.14
5346c77f13e5e5e35bf307bef2e1cab5 in-S0_20090212.15
f5338cfd38fc270becb5cb513fc27401 in-S0_20090212.16
f93b24130ed2218379c473a3e14b9334 in-S0_20090212.17
84c21b69ad4deee4868f1fe16ac59af3 in-S0_20090212.18
973c891399c169aab1a0ec15e3d93ad4 in-S0_20090212.19
bd7bdc0dfdcfb851bd4f2b5fa8f1e32f in-S0_20090212.20
6f73c6e3e185af0d3f1374139198bf93 in-S0_20090212.21
97eadd9c181f1a90184272a2ef7980bf in-S0_20090212.22
d94895d7979e592e10ef3b72d23389bc in-S0_20090212.23
01e75e94d3b067095f3958211f0c1a8f in-S0_20090213.00
a4e4dff88719c5dc8aeaf4020705e74e in-S0_20090213.01
485a6d96e5c1a124466282d57cd57447 in-S0_20090213.02
e68687af075577ac1ccd00ba550fca83 in-S0_20090213.03
80b8ead86a1c3da8b5d83452b32d33fe in-S0_20090213.04
66523d4f93c057d578ca7ac564ef0f0c in-S0_20090213.05
cd27a84d09fdcb22d5a80544dda36b48 in-S0_20090213.06
87ba5662c239ee7ebf03c12a20d0164d in-S0_20090213.07
9f699c10398654f4c4f7409ac228d52e in-S0_20090213.08
da68201697a0589b945cbd37eb8072da in-S0_20090213.09
283cd7c506ae973cc30d326ea2e3db02 in-S0_20090213.10
bad1aac973e60fb89597a49ae443ba2c in-S0_20090213.11
6f9686146d84727fbb596b94b064148a in-S0_20090213.12
01637cd55dd95040dcf187fc70aa3d2a in-S0_20090213.13
5c5bd8f4a07501c753da1cd18d1e0902 in-S0_20090213.14
f1bfb4d58a73dd57a49a270121ff3e2f in-S0_20090213.15
5c0bc0ddb319f24616e57b7a18cc64d6 in-S0_20090213.16
3a20ddaee1d69dcfd36f6bc2a2dd49f2 in-S0_20090213.17
815728ed36b9197cf1b635bc9cf2485c in-S0_20090213.18
2f9d6ae762e35c0a35de90f039c0ddfc in-S0_20090213.19
f452378ebc0b6457121c8dce4e634d0e in-S0_20090213.20
d3eae49109559a621ebd3ad164e17a40 in-S0_20090213.21
ac5ca2e0cf1bcca8978afb26c49842f2 in-S0_20090213.22
153379021b0a647c491a31fba67a2aa5 in-S0_20090213.23
d81feb1e2ae7b012215fb45e7453f591 in-S0_20090214.00
00d87b210892d3ef12471b04e6b15842 in-S0_20090214.01
711d72979f2b9d3c59be7aa5cec3db21 in-S0_20090214.02
6fafd111e1f791299f86577e4e37c832 in-S0_20090214.03
0d148a6c9ae270cdc18b1d80b1e8fec9 in-S0_20090214.04
96978a43e035dee2630ed92212a02828 in-S0_20090214.05
2e4e0793dad5b01ee7aa55e79395bf8e in-S0_20090214.06
ea4dced0c4cdae34e14fae1c2221c53a in-S0_20090214.07
53cb873286342b9a134e23a8dd9f9690 in-S0_20090214.08
020be76b39a229a15d931f370ecd3937 in-S0_20090214.09
1969dba0adda199f9f699a4f098fc39e in-S0_20090214.10
bbc4da5a09cc8a243d9255e6dcfa76c5 in-S0_20090214.11
efbfaaca5bd4fb7dbd20944d34dae59b in-S0_20090214.12
b108116a9ab86d69e6c69450cadf687f in-S0_20090214.13
381dea87ed0f508208e363e2856ee98b in-S0_20090214.14
24d261c5ea663eaf39e34c0da2e2767e in-S0_20090214.15
1b930c6d6a0cab7fe8779e5a772e5269 in-S0_20090214.16
acdbd1a798306a8019a47439c5ddfa99 in-S0_20090214.17
5b26e006aef13917a97fa7319a76ec6e in-S0_20090214.18
34c30ffabd644ead3ca71e76e7b1e3a5 in-S0_20090214.19
c6ed0b30da0b481ddfd0606e698e7101 in-S0_20090214.20
269ae41558a5c7d86c592ba90134a51f in-S0_20090214.21
9382c66c52c2bd13edf8ec089f8575ba in-S0_20090214.22
1378bfb5055d9ca7b8e5ddd5f9b2d9ef in-S0_20090214.23
29523d925540ae1d9c5b2468eff64c40 iw-S0_20090212.00
c21c90ea1622bafb3e295bac58c48716 iw-S0_20090212.01
0cca1eb272f2bcdca465618f423989fc iw-S0_20090212.02
2d26e78087a7e471b783a18f5d7733a9 iw-S0_20090212.03
bab1662b50238b29f4be11e454ad088e iw-S0_20090212.04
82ea4cd19d26a5bda14015be7a62423b iw-S0_20090212.05
6197a58d1163261ad82a5a74ccd21e7f iw-S0_20090212.06
a311715a8edd00132071a5a8f33cc5da iw-S0_20090212.07
2ccfd5324a8f922d5f7f9e22b5f425b7 iw-S0_20090212.08
e1428fc6e8acdcb33bf29c78d84563fb iw-S0_20090212.09
6b7efa0659ba410d39bb363342f196f6 iw-S0_20090212.10
dca8e01e1c3022e6b2c0a569733c70db iw-S0_20090212.11
2b2b496443a6a624cf460cfc1a2d4d78 iw-S0_20090212.12
ea7fdb01df80b04375b98aceeed619e5 iw-S0_20090212.13
82f3a64a7e55814237f0fa2aa05f90b8 iw-S0_20090212.14
cf64df13abde70b396239d0caafbea1c iw-S0_20090212.15
0fcbaeaa2d797401a52fbed3870b9de1 iw-S0_20090212.16
20a84a3a68ec251b7eea75c0abdbf2c5 iw-S0_20090212.17
16d62a48917ad4809158ade8a0598634 iw-S0_20090212.18
cd47022ad4368123fef23f23e3814433 iw-S0_20090212.19
b919dfa40e6346beb41e6ae9ecb44cd3 iw-S0_20090212.20
2fcdb32874d230450b5f505418543df6 iw-S0_20090212.21
1b44b76e5f54925facd3f8041ce42a62 iw-S0_20090212.22
a7d55b94b2d603a3e462b920f46b9347 iw-S0_20090212.23
fe3e4e2ed14fc62dfad7e3dddcd1d631 iw-S0_20090213.00
4460dddd6bb0fe70c7c772c0ecc2fc8e iw-S0_20090213.01
ac37f5e9fa314c5e5af4ef065f24128a iw-S0_20090213.02
7e7daaae01aa1564c631e069d2500e0d iw-S0_20090213.03
32dbc89c099c692489149fdcaf014e31 iw-S0_20090213.04
758bc43723462bdb84dfd8edcd815e32 iw-S0_20090213.05
99d452ea63f85762ef8e0489d1688561 iw-S0_20090213.06
768339c0db0853c96f082bdee13f219d iw-S0_20090213.07
14b5f49eec30e7f5226b1588fac3b3da iw-S0_20090213.08
a90e22f7eb9c40fd0f1102e88dcfd8c3 iw-S0_20090213.09
83daa0821597b49be22a4fc849d011ee iw-S0_20090213.10
de869c001adbfc6a2fd0097f358357a2 iw-S0_20090213.11
1ba5dce53638ee01d30c261ad786ad5e iw-S0_20090213.12
dc9b7ccfedf251029ee1053288fe4db2 iw-S0_20090213.13
2ae89d1e91f4ef28345e7f164e813dac iw-S0_20090213.14
4b0e611add386d1132675239bbbd7bce iw-S0_20090213.15
a274b162cc4d39389a279033925c8a5f iw-S0_20090213.16
5b107b812d3b650cdf9dd90ac5e39c6f iw-S0_20090213.17
31952931a179cd759427f9f59637f6a8 iw-S0_20090213.18
1ad0249e58eff981765714f4ba15746f iw-S0_20090213.19
007dea425cd85071919a69a6fd01c690 iw-S0_20090213.20
9389eee9f703cb258c120b4d84f78ab9 iw-S0_20090213.21
667505a4fb87b2aa04def139c328908d iw-S0_20090213.22
936eb95c3c16f34c68dae4cb950da5e7 iw-S0_20090213.23
0a511cebd4f50fc8b70f2acc792ff165 iw-S0_20090214.00
93bd95fab9bf77b0bd507578713d8f5e iw-S0_20090214.01
4a2f873370272d6d2a70de4bbc6ce751 iw-S0_20090214.02
1fbc37220f9c0d6bbe8e6e45c9004d41 iw-S0_20090214.03
af30b3b2ceba34b6c9d8423d4315c560 iw-S0_20090214.04
aeb6bb6163d8feb452b86d5b35373195 iw-S0_20090214.05
45718d5286609a7c650ac11a9810f98b iw-S0_20090214.06
5c0c1ad221a33cf48325a46e640b03cc iw-S0_20090214.07
507d35731208931184be30824a4132ca iw-S0_20090214.08
603e42b1033de3265856959518edefe9 iw-S0_20090214.09
9c072011eba65af7dfb391d5e6604347 iw-S0_20090214.10
eb1a5b5596b5f93ef2b9a25a1ffa75f3 iw-S0_20090214.11
d7aa980b48141bee21119a9cb263df6e iw-S0_20090214.12
74e72b79853d4b616dfcaf0ec6071dba iw-S0_20090214.13
7226eb58fb678eb6f1243d601860807a iw-S0_20090214.14
5dc14d4860e922effc504398c407eb62 iw-S0_20090214.15
8a36df5ecc0ec98a0c596b515180b3f3 iw-S0_20090214.16
1a1304a31d707a7b705a5896af051eaa iw-S0_20090214.17
2400dbefce9e8213b5ce6c2ede8ad05a iw-S0_20090214.18
19bd9670f509b3b2e67374c3dc816620 iw-S0_20090214.19
49a794b7776a6ec0bfbcace258eeb229 iw-S0_20090214.20
482758eea3bf19f0e291f866b2ef7a38 iw-S0_20090214.21
a03e859b12f835e95ffc1f2c89eade05 iw-S0_20090214.22
59d4a22ec958901077b59035aa883cfa iw-S0_20090214.23
f1124b07c02cef9350ac5f4db13a3b7f other-S0_20090212.00
0d242feccc45df691622475b3fa3f6c9 other-S0_20090212.01
b315f3500ee1bb0360dbf46db85e47be other-S0_20090212.02
ac750f5c60135b41d3f9f79421bb9528 other-S0_20090212.03
26f69c8a882f5592dc1d5263ca71e8e6 other-S0_20090212.04
df271354fe2c96a6af679413c002d859 other-S0_20090212.05
02f10b206557b68213684da33e5d3e37 other-S0_20090212.06
f20d73af615cb649ea0c7adc66209648 other-S0_20090212.07
a59459ad7c78c8b6349246c3dec6ac9d other-S0_20090212.08
deb2cd1bc11150d138545e6a575699fb other-S0_20090212.09
f9a8dd075523204059b327c5401e8810 other-S0_20090212.10
1e7af106946d98fb9a69d2313fb49e07 other-S0_20090212.11
3fc157db269ab9dd1a5e85b257662c61 other-S0_20090212.12
54039354326025b8a78ecac715d56436 other-S0_20090212.13
20fd02bd1474ce8d646afb8af797786b other-S0_20090212.14
24ff2c42a7f7a1edf0cd4f81f2d754f8 other-S0_20090212.15
a6325c4ca4dabf85939beded390a2e1e other-S0_20090212.16
291f4db2cb8e1760c9aec28edeec0135 other-S0_20090212.17
54b3b60dd1c9e422c0c9e162235e2acd other-S0_20090212.18
7a934907065e560e5a032c8409639349 other-S0_20090212.19
5c2002cfe0d5cc1e1e91581a8bd1174d other-S0_20090212.20
0237d878829a55d1fddb3c1d89f169df other-S0_20090212.21
54e906e22bf0e40a68cd239a8ea18b42 other-S0_20090212.22
83eb537de772678221109b5d456f4d75 other-S0_20090212.23
af1da994f4c04e4dfdc5912ed3b83644 other-S0_20090213.00
26270d2b85600af927f8da447fc0864a other-S0_20090213.01
2715f70c39ed07a6724b1aa794c20e16 other-S0_20090213.02
68396789f522bd76976fea357b1435d1 other-S0_20090213.03
a795f79e748ffc2d31ae4ea9d519dfa8 other-S0_20090213.04
2c7087e87217ea43cb8031a1af0c421d other-S0_20090213.05
1eb1aab1c4399a87f7beaa1150b918d8 other-S0_20090213.06
f49efa81028c9137cc8137e813bd66e5 other-S0_20090213.07
6bb10a7a5a52eaf28c20a7467c6255ea other-S0_20090213.08
a0a7f75e81758a2127145f456980080d other-S0_20090213.09
2f4a1e673675d87fef12c94b7d8cb51a other-S0_20090213.10
c955337631976b67a08531c3c709a43b other-S0_20090213.11
491178592965813df3e7f2fedb4c1327 other-S0_20090213.12
9f7374b76e76dab06f1e77d1c85aa976 other-S0_20090213.13
91b3b7c9dfb9fadef6a385b7aa635a22 other-S0_20090213.14
48c972fc8af2cf1b467c8ad93faf17e4 other-S0_20090213.15
da6de4e5f99176517d5c41cd06bb0858 other-S0_20090213.16
e667ae058ee149f1c2d970b9b16abbd9 other-S0_20090213.17
faff49c86eb69e5444de74576ef26ccb other-S0_20090213.18
abbc2301e810fb9872e8f6623e0636f1 other-S0_20090213.19
cd5268ebb07d3ddc72d38cdbf7daaf6f other-S0_20090213.20
ca3f67443740b25fd8634c68aecec862 other-S0_20090213.21
4fcd743df4c67b14ddd9725aaef5c120 other-S0_20090213.22
e817a9b64ed18e3de99ad08dadcc32e6 other-S0_20090213.23
a833f5a95a4f2bee34ae7d484b842462 other-S0_20090214.00
f65c0279b0569a9b4bda66b0dc94699c other-S0_20090214.01
b8c99b13af2cd929dbfbaad8333004d9 other-S0_20090214.02
eab85b030a265dad87c536662dcd052e other-S0_20090214.03
cea4cd7bc9b21ec2e434552c0ca5c13b other-S0_20090214.04
1187768f8df545a4f8e7543b9603ba9d other-S0_20090214.05
4c14d2e2622b9b39b949b5643442d59d other-S0_20090214.06
8199d7f2fcf57349373802802e6d5c6b other-S0_20090214.07
af0b2505b929fde26834577aed5cc7d1 other-S0_20090214.08
583e977a6b53ea7b3ee756d5b3be204c other-S0_20090214.09
ced6e28476dc5acb63cff09d27840672 other-S0_20090214.10
be22bab4a734658c11de1ca183774faa other-S0_20090214.11
c79c4d2fa3fc25af68ca06d9cbe48d27 other-S0_20090214.12
ce31a818401d47ae3e2423c800e76221 other-S0_20090214.13
2635c084570d454216783caa767a8a11 other-S0_20090214.14
fe5394bd75e3781ff2db576c9bddc7c5 other-S0_20090214.15
894b889dc17c135b46276cd65b6b4c5c other-S0_20090214.16
2077f15c643ddbe55c076caa07ea2e93 other-S0_20090214.17
5460dd34950626aabe002ea3440d07c4 other-S0_20090214.18
a6e9bcd4b6945132076ea26e083d79ee other-S0_20090214.19
14537b2c29bd421ebf5d3ac205151aac other-S0_20090214.20
ed83e005c8df4b705ed83cadfdae31af other-S0_20090214.21
a1639ba677cbdb0f5624f6c1fdec59f5 other-S0_20090214.22
e8594c08eb7325e752488d0376f40832 other-S0_20090214.23
60f7597a3e55b9f745197b0060e478f2 out-S0_20090212.00
5692d0ee737e058df3bb7cb5f6ff73a6 out-S0_20090212.01
d8d62287262b84dc0b076aedc6e9f203 out-S0_20090212.02
ddb33e0418eec4f7d2f9fd657cb28a83 out-S0_20090212.03
fa910f28bb8d338ffaac655b3629022b out-S0_20090212.04
aa6b09bfb00f1ae48f32561d6c9ae092 out-S0_20090212.05
4e7778df1f79ce1161ef3b466dbe6262 out-S0_20090212.06
8994884e1d654537c0bc5a0976584f77 out-S0_20090212.07
fab8ec5cc4954c322503c90e83768944 out-S0_20090212.08
049cc66dcaeac78bb856dcc4ba515e0b out-S0_20090212.09
ca3a8651a59e0e006113acdec9e76555 out-S0_20090212.10
d42130431ab2457b01dd23189b8c8b76 out-S0_20090212.11
8593dcb41972ed714c475ae2655c0e0f out-S0_20090212.12
ee3d92a3b19d17e679541712246eb5c7 out-S0_20090212.13
48dac75b887298f729a48b5629b9e34a out-S0_20090212.14
a016df031baa41bfdfd784ed7faffe31 out-S0_20090212.15
4fc4717fd7d02f8d73930be2c715f9c7 out-S0_20090212.16
a8a9fb1c48f10b4e3bd8c809e5992600 out-S0_20090212.17
d4f46c3f3494b221fcdebf41807d00c5 out-S0_20090212.18
fb37e66f556ecaec30f25a5ae14c29ae out-S0_20090212.19
ded5c1e627c34c4cf5664e058ec06fbc out-S0_20090212.20
1527b22c96ecff44ff0e1962b2e2dd88 out-S0_20090212.21
53897aa40779d516f3f601f490b259fe out-S0_20090212.22
a06763af353140766fa103ab7fbb633d out-S0_20090212.23
8d634c813cf3f4bcf925096ac99523ab out-S0_20090213.00
a31148c96d6155ae08ce6f85d65ec6c4 out-S0_20090213.01
462f8fa8fcd3af3942313994863c552e out-S0_20090213.02
87d85a3543b08f08f5a79a297839f576 out-S0_20090213.03
aa76f57f709c7978543140919958c4ed out-S0_20090213.04
c1a3ee7cc5edb70aa13d66ffe8059536 out-S0_20090213.05
4efb3e3b7729cb9cb73ee28201e59266 out-S0_20090213.06
52c356c0c23e05c89ebbe52b124440ed out-S0_20090213.07
73a03a9b8fcd8c9671fab2623e0c0a13 out-S0_20090213.08
2a8beef13989a523b828d41f8d79a370 out-S0_20090213.09
d0e0aea0fb39b13624d9c0335122e811 out-S0_20090213.10
7c03f5b95ba31a7276c4da636eb577df out-S0_20090213.11
6b0af2a0993e2f9da83f9e54c7d74f93 out-S0_20090213.12
6c91b185eed48cda75d39e1dfb8a618c out-S0_20090213.13
df050bf942d6df96166b5adc388ce617 out-S0_20090213.14
4d26af88f90a76437474675ec726b629 out-S0_20090213.15
70fc19207aa989df446e7bb17120bd01 out-S0_20090213.16
a51363a61f256a27b5235376939aed69 out-S0_20090213.17
5defceac7915d5ab52b0504b9ef4767f out-S0_20090213.18
a694d4d68388f93ae5ed955e31037129 out-S0_20090213.19
3b24cb6319abb96e53b973f262b6a0dc out-S0_20090213.20
5b7dacd4b3a1c9fe4047404f164ea789 out-S0_20090213.21
b675323fd5942058286424cbbdb03132 out-S0_20090213.22
2f476e0d851c8e362191d13feb79e2f4 out-S0_20090213.23
990808b779e502930ea85345675398d6 out-S0_20090214.00
220a0ce382de5769570c525cd3d4e041 out-S0_20090214.01
e29860550bfc83e7fb0185be97c01d61 out-S0_20090214.02
9ed33bd1a86604efad5a14feb7f5d2a7 out-S0_20090214.03
464cacbe113a9e2b48dfa6b864218614 out-S0_20090214.04
8365e8c3521f953d4ab87bcdc0eebd27 out-S0_20090214.05
1608f290a389e63b13f509fc1a70c84a out-S0_20090214.06
9ab456724fe46879623bf3546a48919e out-S0_20090214.07
0df29921c8176bb4da0f4cc49cb3de82 out-S0_20090214.08
1bb0ed1ba490ba372fe46dedeb2ce6aa out-S0_20090214.09
a28cca8f5063c45e646316d919e22930 out-S0_20090214.10
0ed973e358557a7ce68214005d027f78 out-S0_20090214.11
db70ce7a9599a5b294c9c4d3e062e921 out-S0_20090214.12
f12c375513e77a06fdf51c6efe5e508d out-S0_20090214.13
f8744f00cd6cd59cbff159496b5f45ae out-S0_20090214.14
7b3679787f9734a7aaa64139d64730ac out-S0_20090214.15
a6d80008e466800d1e51668352ce4e61 out-S0_20090214.16
0e90a0e5dbf3447eb19563dcabaf2213 out-S0_20090214.17
fab9d161fc315d2f4e08a6fc5e318432 out-S0_20090214.18
782bbc2240ec7bbeebe6db6536218f7f out-S0_20090214.19
03dd1a5f7eedebcbefb5314659d425c2 out-S0_20090214.20
66f3e3d16a8c9d3b7f6c03275d87fe31 out-S0_20090214.21
f8965cecd3b22556053a93633a6d1f19 out-S0_20090214.22
e6b88efc3252e3c4f30a90364d92d485 out-S0_20090214.23
516c52caae1571961ff74703557d42f2 ow-S0_20090212.00
060a95a5545bee6deb3bea6cfff531b7 ow-S0_20090212.01
49b0eaf820ea0e1332e65029c4a6b4da ow-S0_20090212.02
e6c8784d1e25b4492ff275fee0047e7e ow-S0_20090212.03
253dfd848776b30004789ef7705ef78f ow-S0_20090212.04
cb59b1bf8c88456116e3604c61e6db57 ow-S0_20090212.05
881d2710cb61ed6d7892106231f636d3 ow-S0_20090212.06
b2d6468548a41aac10d3e220706c8f0a ow-S0_20090212.07
d3348e199fe7940c05075999b0694229 ow-S0_20090212.08
ff853dd30fe410142f44d9ea06c160db ow-S0_20090212.09
90f795c2dbaeb6276dc253bad60dd997 ow-S0_20090212.10
939e858bd86e694ef11d6f9b3a36262b ow-S0_20090212.11
a3b4726dd623396be2550f893696bd2c ow-S0_20090212.12
e5965b183c3b478e89b4f2e7f0ce3221 ow-S0_20090212.13
82c332cd5205ffd3ab4d8238ad6b5d99 ow-S0_20090212.14
32dc81d4c80700419ccb718ac35df91c ow-S0_20090212.15
5bb36a40f177cb3768e7896bedd4d1f7 ow-S0_20090212.16
37258ffe2675fb9a0a8a892a604523d6 ow-S0_20090212.17
3f2e4707023859028c8ef1fea8e264b8 ow-S0_20090212.18
3484aa7a6fdc906bce1748523fd62ca7 ow-S0_20090212.19
dde85ffec754972709566e4086b18a16 ow-S0_20090212.20
572987c4d448617d482a444890a0c460 ow-S0_20090212.21
67b119524631773f87fb3a2fb16c43e1 ow-S0_20090212.22
27bf04040645cdd5dbd35fdc5ba158e7 ow-S0_20090212.23
cf2491e6a077550264bb260e9f72a2f0 ow-S0_20090213.00
7d3efdab0c732e29389964f34997273c ow-S0_20090213.01
7aaa089a2badba5def1ac551dd6072d3 ow-S0_20090213.02
05c2b5d4cdfc042e5f31abb5574f9eb9 ow-S0_20090213.03
0b02925ef647817278e346d0f019d4ef ow-S0_20090213.04
19faad768c844281516c5dfb0d50649a ow-S0_20090213.05
b1d0c0aedb09252b6d99e99b203f363a ow-S0_20090213.06
c31fb8e230c3ee54ceb848f491962d2a ow-S0_20090213.07
73233246967fea92e63cd8f574908c8c ow-S0_20090213.08
0512e5d804b79f9e1da81d90ef0dc9bf ow-S0_20090213.09
bd906a05688778419ee24f492858d821 ow-S0_20090213.10
aa4714f30832b75c6805624e763cb23c ow-S0_20090213.11
f16fc870825ca358c9be2b875413df98 ow-S0_20090213.12
531d99221a7857af16e8a914bd3b36d7 ow-S0_20090213.13
c1bf2c4a5831a4ac8f0e8252b22b6ddd ow-S0_20090213.14
7b0c740a74315f3c7e398342511c1868 ow-S0_20090213.15
42a2bfb1fed335624756121130fd46b4 ow-S0_20090213.16
c8d4b37620da44f6d3795dff865f064f ow-S0_20090213.17
cd4ad1fbc8bc07c76e7b7de5131005db ow-S0_20090213.18
73e0cdb3f84792da9b50ee98ecf51035 ow-S0_20090213.19
c52e8731d038abdd06384a31eb6af67b ow-S0_20090213.20
d932f97905f51cfab9826401ba421992 ow-S0_20090213.21
bb5894ad73c94fb9bdf71cfaec111933 ow-S0_20090213.22
daa4c4752b9aa03d790aace405cfb52a ow-S0_20090213.23
bc1e5771ad462e19164872fd479f5de2 ow-S0_20090214.00
fe79f48a6eca2a973c65c146732d4729 ow-S0_20090214.01
91d477a917e641b68d072afcf354399e ow-S0_20090214.02
9bf77d64b85fdc825e975613416fe024 ow-S0_20090214.03
6b5677775de84e581a5dac47ec663ced ow-S0_20090214.04
329885bed2738080379d0baa5d879c5d ow-S0_20090214.05
6080e3d847e75ca13161bf2c2dc39098 ow-S0_20090214.06
1529bdb495a731daf3f9b3b129a00086 ow-S0_20090214.07
f89462972219648bc345eab949bfdf23 ow-S0_20090214.08
a2ac3c9bf000dc66825d6645b1f12be7 ow-S0_20090214.09
5d764cd38c7b2de9383402993e26959e ow-S0_20090214.10
35315d36b5fb60e9c85b0c1fa9ea0c27 ow-S0_20090214.11
2d08fcad6455ce7529af60213b313032 ow-S0_20090214.12
2a21f16a90675914f5aced6ce9fca9ea ow-S0_20090214.13
c85624dd3de76325f63b2d2d40016958 ow-S0_20090214.14
91d4dfab3b6f1f8508aae7c6ebe7d151 ow-S0_20090214.15
f0b8ebe9ea28a6a579b2f40072e592dc ow-S0_20090214.16
c3c76225bc371115fe7ac044fdbc7b66 ow-S0_20090214.17
2634926eb65cee405d4b60e0aae4f7ff ow-S0_20090214.18
2858e136fc699fee1c2b225c3392e594 ow-S0_20090214.19
289f8b0234cbd07bd2817dd39ade9f91 ow-S0_20090214.20
9b13494e2b99624006aa1ad7b22573d4 ow-S0_20090214.21
9838950e543453bc99241bb57ab6325a ow-S0_20090214.22
c4de9cf5055789c0d40cd5e0133fb8b1 ow-S0_20090214.23
