#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-silk-send.pl 3c7a2eca6a99 2024-06-12 15:06:32Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwuniq = check_silk_app('rwuniq');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'silk', 0, 'polldir');

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     ($ENV{SK_TESTS_LOG_DEBUG} ? "--log-level=debug" : ()),
                     "--sensor-conf=$sensor_conf",
                     "--output-mode=sending",
                     "--copy $file{data}:incoming",
                     "--limit=501876",
                     "--basedir=$tmpdir",
                     "--flush-timeout=5",
                     "--",
                     "--pack-interfaces",
                     "--polling-interval=5",
                     "--file-cache-size=8",
    );

# run it and check the MD5 hash of its output
check_md5_output('a78a286719574389a972724d761c931e', $cmd);


# the following directories should be empty
verify_empty_dirs($tmpdir, qw(error incoming incremental root));

# In sending mode, files are named with a mkstemp()-type extension.
# We must find all files having the same name minus this extension,
# and combine them before checking the output.

# path to the data directory
my $sender_dir = "$tmpdir/sender";
die "ERROR: Missing data directory '$sender_dir'\n"
    unless -d $sender_dir;

# number of files representing a unique {flowtype,sensor,hour} tuple
# to find; i.e., where we have mapped out-S0_20090212.00.LELcDc and
# out-S0_20090212.00.PzEA5g into one entry
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every {flowtype,sensor,hour} we expect to find.
# Read these from the bottom of this file.
my %md5_map;
while (my $line = <DATA>) {
    next if $line =~ /^#/ || $line !~ /\w/;
    chomp $line;
    my ($md5, $path) = split " ", $line;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;
my %file_list = ();

# find the files in the data directory
File::Find::find({wanted => \&find_files, no_chdir => 1}, $sender_dir);

# combine the files and check the output
for my $basename (sort keys %file_list) {
    my $files = $file_list{$basename};
    $cmd = ("$rwuniq --fields=1-5,type,in,out"
            ." --values=records,packets,sTime-Earliest,eTime-Latest"
            ." --ipv6-policy=ignore --sort-output"
            .$files);
    check_md5_output($md5_map{$basename}, $cmd);
    ++$file_count;
}

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files types; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub find_files
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename, stripping the mkstemp
    # extension
    s,^.*/,,;
    s,([^.]+\.\d\d)\..+$,$1,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};

    $file_list{$_} .= " $path";
}


__DATA__
4ca34c46bfe402236e7f53f7b13229df in-S0_20090212.00
1045e87b8c59390bcdfbc7461b8cd26b in-S0_20090212.01
a591bee7e52b19952ba222f7ef994efb in-S0_20090212.02
0c23c4f995186315ebb16f9eec4075ee in-S0_20090212.03
7ba56e2048ecc461b01444fdf5f78651 in-S0_20090212.04
5c134e2372f28a23ef61eaea500f54bd in-S0_20090212.05
aa7fceb673b42b99ce6f810bbc5654d5 in-S0_20090212.06
e9e585e97061acaab39de92482e14ad8 in-S0_20090212.07
a6f1cd8eaa21ccc069cd12625c531ebd in-S0_20090212.08
d953f87333194f3588c807b6af063a05 in-S0_20090212.09
f698eb80531a71a21bbe326982ecbbd4 in-S0_20090212.10
b1c5e1f7e14fe5af75b2bc168dce3bad in-S0_20090212.11
7c7a8fdd7a67fe41b8a30b6eb58a94be in-S0_20090212.12
fef98b9619c2e24c4b02e7e1e65935dc in-S0_20090212.13
50c78d952b647c32c75dca43252a8584 in-S0_20090212.14
f1402f6f0b96d8fffbe63041e8246c4f in-S0_20090212.15
bb4cf3847a425f278fb9b9e412ab846c in-S0_20090212.16
b3b420179925754d2c7ccaa5d85e824e in-S0_20090212.17
d293cb7301c3ebcc0cacaa1bc9583214 in-S0_20090212.18
4932cd0bf781e7bb17400083e40c402a in-S0_20090212.19
2da08db00489871dc3c560dfeca7acbc in-S0_20090212.20
5a897f51e58f2e82f7ac21ce64f26020 in-S0_20090212.21
3de6c61d33079edb31879308ce5dd4fe in-S0_20090212.22
4bac3526aa6e0b99af791739008da227 in-S0_20090212.23
4fc7f90e17cfee87c195128ff6198f01 in-S0_20090213.00
a2d06a270012df1b2ef1e333894f5041 in-S0_20090213.01
c6ce98623db77b0afc8c06438ba79ec6 in-S0_20090213.02
66848e3feb0d5a63b7f5c158d2502a78 in-S0_20090213.03
e028099be4169403b70f00f525719d19 in-S0_20090213.04
7fe8267c8657b5ab048e3ecbb81b5b9a in-S0_20090213.05
b9a6219f50f1b230055a6371a0c6b6d4 in-S0_20090213.06
60385fd03adba0396610f3d9116d57a7 in-S0_20090213.07
5d5d18937dcede68f1ada2630074fc75 in-S0_20090213.08
52739a8f9345c295c0eacea176e8862c in-S0_20090213.09
c9c59e925b027663c43ec45e1c152d13 in-S0_20090213.10
05a4e52ec2439d80cacdece1553fbee1 in-S0_20090213.11
7ab05341c6eb0e933e182e44347a9038 in-S0_20090213.12
952ea9f023889322b58a5c47cdb0d6a2 in-S0_20090213.13
4c61aaa11ef51e02284619b5ceaaff83 in-S0_20090213.14
c0c445fed91d1e2c5860917df17f0d90 in-S0_20090213.15
64d653e954046548e7a9ebfd2d3d64c4 in-S0_20090213.16
71bbc3e365aa1a2256d6c4f2b056e2bc in-S0_20090213.17
3be25706e94db2b51bed1e71ef3ac474 in-S0_20090213.18
1c940320fd061e9ad1fff90e508b692a in-S0_20090213.19
6ad5d277a02200f05a02f29073f69d9e in-S0_20090213.20
250d24667fa77bb8083987a222a81612 in-S0_20090213.21
cc15866a1e980dcaf9b07f8490e08b93 in-S0_20090213.22
1711bd20a9f7780828b57e17953d45fd in-S0_20090213.23
39b80602cbd2f72fb1fb1fde243f90b4 in-S0_20090214.00
f13969c58ad637c847702117ae043505 in-S0_20090214.01
8e32a6c5180d8f9ec07ab17f41a0ad1f in-S0_20090214.02
1f6b18420989d378fab6dae769df08a1 in-S0_20090214.03
5bbeb291ca805de5d20a00c18018fa87 in-S0_20090214.04
4132ee3be0eb0ded3df9a706cd7982da in-S0_20090214.05
c526abf807c2c634fdbd7bcd781fe355 in-S0_20090214.06
b133b0e987f41393fcc509d87fa21c18 in-S0_20090214.07
231651e29b1e833077e609234cd64fbb in-S0_20090214.08
a07cf54cfebe3c25db7b0211897627f3 in-S0_20090214.09
78b6715af0738f016849b89de5bf854c in-S0_20090214.10
759e3d179d23244195357c0f3c02089f in-S0_20090214.11
9d019817c5f213ce5f0db88d1bc7325c in-S0_20090214.12
6c4e044baae1e85c9c38a9291dc1f42b in-S0_20090214.13
930e3a19e03ac951aebe41efe346b760 in-S0_20090214.14
db880f69fd9a7a240e3058ef21641fa1 in-S0_20090214.15
62327f46bf9e4c60c98b3764be460546 in-S0_20090214.16
009e1b748385c77a43c6bcc47d3d0db9 in-S0_20090214.17
eb15064bf2ce159a084c1ae5c67fe982 in-S0_20090214.18
6d78ce0ce54a7e0d73ce39eb917785c1 in-S0_20090214.19
7817777855e1567e00a82daea16fede6 in-S0_20090214.20
2f993d5194a93828e802cc76df879caa in-S0_20090214.21
d2481b0b9ae0753b6302613f336abb8d in-S0_20090214.22
6e109dda3f6d7e4ddefb055034fcd7e5 in-S0_20090214.23
7ebbee610d3985d43ca5b4592cbed3f5 iw-S0_20090212.00
7a25d9f247f5d60fce1f28dcff836c2f iw-S0_20090212.01
8df8fcadf4bc12162deea5ad43ac36a8 iw-S0_20090212.02
09d99eeab5e26d27767e49e884fa13d5 iw-S0_20090212.03
e345ea738370248fbf5ebffc3fc02e63 iw-S0_20090212.04
443c5a27e965cef7c2a053f6e6b6546c iw-S0_20090212.05
65f25734f7f1464fd6e725b787307e76 iw-S0_20090212.06
fa9600430aedaa28bb938e81b49bfd88 iw-S0_20090212.07
d8ca7d3ac512f36bd288eddec2c8d9fc iw-S0_20090212.08
f2080a07c095fb6990bc2723df319c50 iw-S0_20090212.09
5ec49414e8d7a4e2e58d62a7443f5570 iw-S0_20090212.10
7c01525af2be603cf27900461a74a7c9 iw-S0_20090212.11
ae8f76b9e9bbc8f6fb38a323b4f5dbf6 iw-S0_20090212.12
5248e47e4df66a90054cad89243b09e5 iw-S0_20090212.13
87599d12ee1b56f5bfef4b926aa0d184 iw-S0_20090212.14
742aeff735b540d161a44041ef887d4b iw-S0_20090212.15
1608e8a2662dfe12928aba6156da5043 iw-S0_20090212.16
dff4892c5b0cab563e093402d8e7e9b8 iw-S0_20090212.17
e62ef659ee2420237311384eb99016f3 iw-S0_20090212.18
da68c4ac100eae520a81bb6901505d24 iw-S0_20090212.19
69072571f8d44e0340be413627508ea1 iw-S0_20090212.20
52a6c26a1082c2b761daedec8bc05074 iw-S0_20090212.21
e188da58e102ff5d70d9fe1122184527 iw-S0_20090212.22
41d1d4ca0fe8dacb69375fe56d5b8adb iw-S0_20090212.23
d8b42e90303c2788474c85c8f590e9de iw-S0_20090213.00
333e75ad4fc878fa7b700dfb3daa7a0f iw-S0_20090213.01
9dbb46fbe1c0b76773c3a2d64527442b iw-S0_20090213.02
2aec725d52031f5cbacabbd2341ebb67 iw-S0_20090213.03
8e89f25e81b95147eb0286c7bbac1268 iw-S0_20090213.04
7d8f8b652d2036581f0cc7ad341beee5 iw-S0_20090213.05
ea82463376b0a46f3b19cd0cc8c2c270 iw-S0_20090213.06
4b53241ac2ea0083865aeffb67480911 iw-S0_20090213.07
6c8ee69c397344db26380d41dee1dfdc iw-S0_20090213.08
ddfd95f5b305a9182e6f1efe30cc5537 iw-S0_20090213.09
9b0f55744d27fa00a6d159c66a849ea8 iw-S0_20090213.10
54c303d17ed856fbe2e023f2c25db3ec iw-S0_20090213.11
f92e3f77d9dcd347dc4e72b4b1a0908f iw-S0_20090213.12
a98db1ae90565e8214d6a7b6918e1537 iw-S0_20090213.13
0696241f591b21a226b5a61cb85b2e8b iw-S0_20090213.14
cadf42e902693c125b8d0e11bbf9e8bc iw-S0_20090213.15
cf46ea11c56c8385e0d60a8bdb64eaec iw-S0_20090213.16
ba3fb30313cad69f47b66c1c80f32cdb iw-S0_20090213.17
9bc21526cdda1368d9cadd9567ca0481 iw-S0_20090213.18
2445ed81b3055d09415fe3c249269bd8 iw-S0_20090213.19
28ebef342979b0cb1efbe9ae2499256c iw-S0_20090213.20
c1a952cd8230dd57a84921db68278101 iw-S0_20090213.21
5beb587562eab84ef112ddb02dc3c52b iw-S0_20090213.22
4e84f0846b6bdb3cfd12633149c74a9c iw-S0_20090213.23
f8c2d2d81bc4da4c1f0e51f825837753 iw-S0_20090214.00
f9782803f8b05c0b3140e4a4f2a11e50 iw-S0_20090214.01
4218771f0d1014e4ec2d50f3411787ed iw-S0_20090214.02
92b48a485911cf549cfb7b23ee48e273 iw-S0_20090214.03
c008b4779dc5eced1bec2b5db515ab68 iw-S0_20090214.04
339668597206be3d8dfc054e6ea5994e iw-S0_20090214.05
c15b624760878bbeb8b64dcd5b76f866 iw-S0_20090214.06
c69d94f278258b282401d735a1c2a52f iw-S0_20090214.07
d030de673faf828df83cba2b85197097 iw-S0_20090214.08
ce952bbc87f096ec30e86193455ab1ac iw-S0_20090214.09
6a4001efa69bee5a1cd3ff706ad5c570 iw-S0_20090214.10
0976d659ffdd0deb047d3c4deaa64b3b iw-S0_20090214.11
e114e99a8a3e6f3da6a3ae0f4a60a89c iw-S0_20090214.12
c389e96c9a2c6fd3d461fa34659d512c iw-S0_20090214.13
b74c39dfe90dc6bd19dffe4247a4d666 iw-S0_20090214.14
157e0c44342d5f40e32b9e8a90a69256 iw-S0_20090214.15
84f26ba3729ce4c78caf4a935330200e iw-S0_20090214.16
6eb299c18baf27bced9a5cad424349d9 iw-S0_20090214.17
431e01d378838698a4d280132799dd2f iw-S0_20090214.18
13f55180a94d244212547d462caedfa2 iw-S0_20090214.19
65b346f5314cf78b47605c31f7f5b4cc iw-S0_20090214.20
f8638c8fe1ad54cbb8cc0f3e1a99a97c iw-S0_20090214.21
fd73868e53857418a9d68dd91d318e6a iw-S0_20090214.22
fdb45edd7f78737a57e6788284e997fc iw-S0_20090214.23
8c8bc89f2a6fd68e62a09be3ac5a8c6f other-S0_20090212.00
e5db84b3fe39821d1b55d04b680daba8 other-S0_20090212.01
8cd18d3dd85bf2dd46f8824fa06bde60 other-S0_20090212.02
79e1fc5f128521dda18ee497f5d8a8fb other-S0_20090212.03
0c59403b4cdbab8e0c6faa080264c53a other-S0_20090212.04
bb45267bdffcc3e1478cdfbf7eb71c4a other-S0_20090212.05
a98a1b8fd242aca3b36bda78f91e0d55 other-S0_20090212.06
f7a02d15edc65e9700e8feff710e943f other-S0_20090212.07
b95f92fdde83c1db4695e66c924649fb other-S0_20090212.08
7bfb331f3c4fce2ac2eef684589949c2 other-S0_20090212.09
04145483111baf602135771a2cf8fd91 other-S0_20090212.10
ceaaa0f83e79e3d2488d3f0c46bcb692 other-S0_20090212.11
c094cd614687b00f9473f8342ba2049e other-S0_20090212.12
0e7888e8cac6d3964d439711082e5cb9 other-S0_20090212.13
fc0603dafb0f6dd5d4085adcb9f62fff other-S0_20090212.14
dc2fe7daa52e9a89221b0d4087d9eba3 other-S0_20090212.15
90656896d6b88f2c60559ed655769a53 other-S0_20090212.16
450eb413f3b51b75f4e56ebe87cdc0c2 other-S0_20090212.17
9648471a0a1af8d9cb40d9273d2865ba other-S0_20090212.18
78c2b9f9f68ed81ab3740ec48fcd3f1d other-S0_20090212.19
425361220cd30708d5508d2bd62ec9eb other-S0_20090212.20
d82b2f18f03d3ef3a99a96ac861eff4a other-S0_20090212.21
5a570366f337312c843e748c0a8da8ae other-S0_20090212.22
a3385c466c8e6c4f65b63109c9d606bf other-S0_20090212.23
7e8dda54d2fb45e2b190cfcff3b7d98b other-S0_20090213.00
e257038fa86844b9bfbd80da77d0536d other-S0_20090213.01
1924773cfa17b83f702c6bc633f29fcd other-S0_20090213.02
7365e311ddce1f3dbf8ecccdefca70e0 other-S0_20090213.03
40b27f5fcad6eaf82965a860933b5593 other-S0_20090213.04
1e347c3fa1cbce4076a51d62bbddff1d other-S0_20090213.05
83724b6e12e48cc1139140dc227051fa other-S0_20090213.06
ddd16067a81646ae8c7a1110e1f478ff other-S0_20090213.07
4c4161aec682ce003d426980dd0985b1 other-S0_20090213.08
5c0721cdea62dc0d1fc4cc319d264c01 other-S0_20090213.09
188a39b2039ac1027fc5bfd83e61309e other-S0_20090213.10
21ae54c5e2ec501b90d1179d4ff56ade other-S0_20090213.11
8394c6eb08e51f4c83b807b64ac1cedf other-S0_20090213.12
0d17b326d71a205a2eb31a55e605d32a other-S0_20090213.13
f5a7fa8997350eeefcd8682fbc6ff0f6 other-S0_20090213.14
847d6d322dbf40a049fa9e087998b727 other-S0_20090213.15
bfa6b4792208174afa1907ff30c396a5 other-S0_20090213.16
9e5bc63ec754762a1a39d625661c81e1 other-S0_20090213.17
91f830f1052081c8654828fc0d7a5d57 other-S0_20090213.18
4075347383cf9c3772de42f62a0613b2 other-S0_20090213.19
28d7e215e131fe17bd01c3f43b1c837e other-S0_20090213.20
888315ef303b94cfe1fa934244682cd8 other-S0_20090213.21
1d77237623c8558799854883bc823755 other-S0_20090213.22
6a8e58b41bd4d742c5d9f778276d2ddf other-S0_20090213.23
58e03cb68e5f032e855deab6105a2158 other-S0_20090214.00
457071f602ca7597566880790f681cac other-S0_20090214.01
6ab892cebf6708d8435e1aa072266086 other-S0_20090214.02
391bece61ebb3e78d378cb1814677b7a other-S0_20090214.03
39673d7484beea7813bcf2fd1f8a135a other-S0_20090214.04
f9606568ffa1320130e41349b7e92459 other-S0_20090214.05
c33ae8f193fa3aa235552d18da4bd304 other-S0_20090214.06
f8fbbe70bd31876aa2ddf1d5cd8ee301 other-S0_20090214.07
912815d81d187165a2e4f347a4db4b28 other-S0_20090214.08
c5e7cf197d025a00e8603b847f582f64 other-S0_20090214.09
b961a7057f26556928e42b8728c5ee53 other-S0_20090214.10
7fca094b3f1fd3704fbf7c471e674361 other-S0_20090214.11
21e27ea70354fddbe2d931fe6a8cc580 other-S0_20090214.12
5b5b69b4503b957f2acf4709a55f72d9 other-S0_20090214.13
cc858a6603b6da029451a2bb268ef609 other-S0_20090214.14
26260c5227bf4131944ed1456050b48a other-S0_20090214.15
f0165252b97790089ee2c349b98f4cd8 other-S0_20090214.16
b5da0b4a83ff83930c31776e254fe1ab other-S0_20090214.17
bea75b68de278f23e8c6ca3c1a1b708f other-S0_20090214.18
8261ef10579af7b9901fffe4469af733 other-S0_20090214.19
1f1a07e67b0b4036a476b79ba47d3fd6 other-S0_20090214.20
95dc74c5a4997a0efab11ff6df0f9411 other-S0_20090214.21
27d3e689b33dc45213677a7d3fccc880 other-S0_20090214.22
8aff340193809f2892f9d678b2e9b783 other-S0_20090214.23
26e0922515089d078baf87c9ba30c9a4 out-S0_20090212.00
763ed1ed657b81d04e72ea039ddccaab out-S0_20090212.01
8387c46366d8af0a4775857caa4576f1 out-S0_20090212.02
00b1af2563946e904ae5aa6474344779 out-S0_20090212.03
abf8d07e54acfc6f6b3cba67719ba5c5 out-S0_20090212.04
bd3fd0a9a520b76e0502294f9b778566 out-S0_20090212.05
d55632e3a54bcf9ee9a8f6cde0009fe5 out-S0_20090212.06
cea5a3eeaf062ab9e9f04b7f6b89b835 out-S0_20090212.07
dceaf61c970764e5b59d3b115a469e6a out-S0_20090212.08
d05cb35a2255a0d0822d40c320937853 out-S0_20090212.09
be67a4e5ef2a5db1164721b20682e8a2 out-S0_20090212.10
43ab996bfbc021f4bd45f807f96b1bec out-S0_20090212.11
386064b5dac6b10049c39ca1dc71d394 out-S0_20090212.12
082f1f2cf8dccca9c00d7479fa156946 out-S0_20090212.13
3bc7cce5c9491cdbd1a57bfe0c4c62c0 out-S0_20090212.14
1030d98d9187d22e363ed3560551e8a6 out-S0_20090212.15
bab06e726cae986fa16e34e77dec7dcd out-S0_20090212.16
38b9f1b64219f9aebd8618e30bf767b3 out-S0_20090212.17
70e073edf8eed6179d1aa54ee539a587 out-S0_20090212.18
c8238b6741a6782246de5d7ba7f6a1d9 out-S0_20090212.19
0465314d44511fbea44aa872466473ab out-S0_20090212.20
5dbd10b637add6290cd9ad4059692c10 out-S0_20090212.21
3f4a5707ffbd82f59acc8687250d46a1 out-S0_20090212.22
d129cf923c511d76e49af72a601ca5cc out-S0_20090212.23
a3014b74eecf994a84490ab6d9704200 out-S0_20090213.00
ed2ca2505e6c6e417bc51b7284615fc1 out-S0_20090213.01
65368da883db7c910b9211e1c700f703 out-S0_20090213.02
5d42089c74836a5f85858ec2c0811ad6 out-S0_20090213.03
a78b5e2095c59dc1b18fbec728c213af out-S0_20090213.04
1f1a273f3d4e716f173c822f2f0e2297 out-S0_20090213.05
edf8e48c0869f8564c4ea4fd62729b5b out-S0_20090213.06
04a0795832d59f418572bd9e8960ce36 out-S0_20090213.07
1b24ab900add33eb6c3803a9826275fb out-S0_20090213.08
34512057a9761f4fe296ebd674ae7e7e out-S0_20090213.09
e41574fe78c40cf0e3f1e26cc798b995 out-S0_20090213.10
bb8be84219e82a739740dfe8d6f52af1 out-S0_20090213.11
6cafc3385bf0fa18f8b679a11b934c0b out-S0_20090213.12
7fbb76eeac26095698ea7e061494542d out-S0_20090213.13
bb0160090765ba9c302fa874f51e85da out-S0_20090213.14
5236d54556ed29cf4bae4eec99271e7f out-S0_20090213.15
6a92207136c5c07add34a90b526f190d out-S0_20090213.16
d2cebdea5e169a36ff95364e2c54c6ba out-S0_20090213.17
000fd1e169f98077f1a58575f3c5c472 out-S0_20090213.18
8d6db010aa605fc921cc244087c33ddd out-S0_20090213.19
6140bf950c09deb11b3a3ebdae65094c out-S0_20090213.20
ddc8cd4427c2f6dd8ea705eba71e0fcb out-S0_20090213.21
9c21fbf865a452a80dbfab4d43b16d20 out-S0_20090213.22
8c463d51558c5aed1d76f9172ff3a24e out-S0_20090213.23
07cfbe39e70c5699163e75283e93cd8f out-S0_20090214.00
90e0f6ee7e5865aafa5f3ad87bd96259 out-S0_20090214.01
8f56b3ed82066c885ff59fc051ef6197 out-S0_20090214.02
b442820c8a64a3d8ae9a411350c97784 out-S0_20090214.03
930f6ffab202abe52f790a7fd2068a5c out-S0_20090214.04
464a410e7e88afdce7b2df68abc60a8f out-S0_20090214.05
168898c2dbff0d83659b619a0060800f out-S0_20090214.06
28b0cfbc41ca7021c38ae40efd2a2d66 out-S0_20090214.07
731b60b6f74d58b7c8cb45a919f911b4 out-S0_20090214.08
d902901ecf43132398bb165c4b44c222 out-S0_20090214.09
4722530e73e9a740de3964b650132faa out-S0_20090214.10
a310d087a3173a503f603da6a268604b out-S0_20090214.11
2264d3da393e387f2edeb094b7ba52bd out-S0_20090214.12
73e36cc17e64bf93613bad5ceb3bc858 out-S0_20090214.13
3f29b635907f70106a553895d8cecacb out-S0_20090214.14
2e382cd954b0c39ae14953bf02a24718 out-S0_20090214.15
792251b8a1c54ef8c03d73cf958ed77c out-S0_20090214.16
72c36fdeb143db71eca9cdafc91d5eee out-S0_20090214.17
468c5cbaaabbc44c167f4f54908b1c05 out-S0_20090214.18
07e16e61159bf0d658f3a3b4eda19393 out-S0_20090214.19
c571130b1971ff665a189ef12cf3449c out-S0_20090214.20
167c51f20e530f6b27440d8dea85e029 out-S0_20090214.21
4e0df761417d8d94ac6fd6de4ba219e9 out-S0_20090214.22
1c3fdd46b307b12d9cf62c2fa23d76a3 out-S0_20090214.23
5006eab840846c1e091253ce7d9933c6 outnull-S0_20090212.00
2355f0a5f9711393e33626c8694534db outnull-S0_20090212.01
0c808e9a334fd0a2bbbc551e3686329a outnull-S0_20090212.02
912612550d086f09c76a8ac7e915398f outnull-S0_20090212.03
f172834f00c1a6a5ecb32e5ebede8141 outnull-S0_20090212.04
5b869752f664b1129b88359e44d48c7e outnull-S0_20090212.05
e10d5baad6a5dc5d80a94b3936fd2fe8 outnull-S0_20090212.06
1677c1971679123935e7805032bef3ce outnull-S0_20090212.07
bcc8203096c77bcde0967f3708b50025 outnull-S0_20090212.08
e2de0abe5e5f7362883679ff54f51c77 outnull-S0_20090212.09
86eb43f648c2ba060ba2e7865030662b outnull-S0_20090212.10
1f8bb71a34cebe2e28bbcd8faee7aa7c outnull-S0_20090212.11
8c05e79094c9d06d4365875a7ac7edde outnull-S0_20090212.12
98bc47c5cdc26c549b0f3d1e2cf0a270 outnull-S0_20090212.13
f397abec48fdddd017ceb39aead52a2e outnull-S0_20090212.14
ce35cb780b0d838aa51a445815f6ec8e outnull-S0_20090212.15
4348f9c6392594b597843bd7f8eb9533 outnull-S0_20090212.16
0334a0885608d530246f7b31cbe4f8cf outnull-S0_20090212.17
2e81bd1e9e78f69193205a066b9e1a53 outnull-S0_20090212.18
055b9767a753a440af9f725159766d83 outnull-S0_20090212.19
26f45fab781989b679e0f7fe367ab558 outnull-S0_20090212.20
82065d86da191e307808251646f65056 outnull-S0_20090212.21
ae9de21fa7caa548eb6eaea9669c702e outnull-S0_20090212.22
3dea8ce1e5bc10a54c8a546c8a096b2a outnull-S0_20090212.23
e4bd365bc0310e5fd8fe7f215d2fdcef outnull-S0_20090213.00
3e86c9f4e0d4ba8007f0953eb9f7e7f4 outnull-S0_20090213.01
c9307184ae8caea0ef9d61d9296fb96f outnull-S0_20090213.02
46456a6e54fa59224a46878dcbf817ad outnull-S0_20090213.03
2199ffa844efb3e6998b56828477f5df outnull-S0_20090213.04
87ce5e9d4bf6c8053e1726d6fc2668d4 outnull-S0_20090213.05
66f77a421b40ea0c948db1d5c4dec38f outnull-S0_20090213.06
705b2e464f9bc73877eb24e166f9cc1c outnull-S0_20090213.07
15349e7b7eeedfd26e592a0f422c7109 outnull-S0_20090213.08
140beaf7d909fb88a8d0ad6467921002 outnull-S0_20090213.09
a7d2cd5c7a6e65639c52601b4f360a56 outnull-S0_20090213.10
960d652a0dce3d7e29c6358af5b11816 outnull-S0_20090213.11
002960787ab85011c20d83c2336b450c outnull-S0_20090213.12
2b66fe5eab953443cd6a549cc7e411f3 outnull-S0_20090213.13
9ede070c219f6b34f5abcf65dc46fdce outnull-S0_20090213.14
181599c1fb6861b539908939eea68493 outnull-S0_20090213.15
f13f2e95f6fae0bbec53cd5183a34e94 outnull-S0_20090213.16
ac6c19fb7fdbf090fbcc5ad765340a5c outnull-S0_20090213.17
f95dcc1d27b60034310c54242d6978db outnull-S0_20090213.18
3071a8e1482b6f605b8cdeb8b141e9c1 outnull-S0_20090213.19
4e4eb5731ac761852651f87a56a8ad05 outnull-S0_20090213.20
2dc3f2fe74fa89cab940538e5915f25e outnull-S0_20090213.21
380aebcf875d2c8a091b69e6f5b40727 outnull-S0_20090213.22
9cf9321ab3bed491b00f4596e6f07eb2 outnull-S0_20090213.23
af2c39d6852a0180938933ee54ab3e01 outnull-S0_20090214.00
62171eefd0999d672d983db5da17d440 outnull-S0_20090214.01
ffd340721d99eef33b3618bc58aa499e outnull-S0_20090214.02
327945a05632a07e83bff1a6b02936a2 outnull-S0_20090214.03
b7363f7caf57744c0007d38c1f29fbb1 outnull-S0_20090214.04
5860a11cec86f000a800731266a6c942 outnull-S0_20090214.05
a3e3dfa64b0d8f7ad528bb254d9e396d outnull-S0_20090214.06
d2c02f352d9d5f668b996bb3a9e69abf outnull-S0_20090214.07
d9f37f58311c8f3d3f561fa0933d096b outnull-S0_20090214.08
1ee593cc6051d4a0e46feb936b087d20 outnull-S0_20090214.09
92548f7d5745296db2fdb93e6b153a23 outnull-S0_20090214.10
de061c72f271743354fd1ce36247d1a1 outnull-S0_20090214.11
25808848cffa34e70f925c4edaa2e06b outnull-S0_20090214.12
c6bdb2352ce53cc4c57590869b5744da outnull-S0_20090214.13
4f9b47a5f261290536cce9032ed1b067 outnull-S0_20090214.14
e1432113ffc94be44e04268cae03e745 outnull-S0_20090214.15
b69c383f56abee3b0f0300df87b578ba outnull-S0_20090214.16
dc778e0d9e4228a9515f9696581ce39f outnull-S0_20090214.17
ba69eec954a0ef608791ad4cf8b2d7ed outnull-S0_20090214.18
7c5db7abb6afde5732f559f82dce15b8 outnull-S0_20090214.19
60d34f593461c94f688b804ab07f51e1 outnull-S0_20090214.20
d468f2fbc1d62020facde5f81ea763a7 outnull-S0_20090214.21
e0f10948b97cce031f8e1abe96071595 outnull-S0_20090214.22
53b2232b625fe4feaf4974be6752e4df outnull-S0_20090214.23
5a8d6b7834fa0435ee7bd3a37f539d5d ow-S0_20090212.00
aa003e08a8560bb1f180abd99accdde2 ow-S0_20090212.01
304c418d3b2d30c06126b28989aa11d4 ow-S0_20090212.02
c197ab0651c4589c8b201f8cff415cba ow-S0_20090212.03
abe567421e86be411fae67732c59657b ow-S0_20090212.04
76d75aaa9cc48cdb42dedee6ade60398 ow-S0_20090212.05
d10c346151e58e5ce27344abaf5cdec9 ow-S0_20090212.06
9565f033fd39fc8273edb8bda0c4a26d ow-S0_20090212.07
77a1ccc0670ebeda381b2574db21566d ow-S0_20090212.08
fd6f524f11a80e42c0bf16cf2a71e845 ow-S0_20090212.09
6a864ef8cd8e7350523567b31fcd597a ow-S0_20090212.10
a81411ba847bbdd6a62d81e5f757766a ow-S0_20090212.11
42d355867427abdcb2bd3659eae0ce9b ow-S0_20090212.12
a7d85688182691a235b3a0059bfa03c5 ow-S0_20090212.13
fa9c9896df42a6bcfdb1b39ec8df377d ow-S0_20090212.14
cbfb05a5750956c552328afbb76a639f ow-S0_20090212.15
5b05011d3789cb67f89d00ac4a2b68bb ow-S0_20090212.16
6ea3185247ae1ab93fb3e7512c5f1d5c ow-S0_20090212.17
21d19cf2613acfe0cee70142b546e013 ow-S0_20090212.18
6417ab48ab11c911aa78b65f9428ef2a ow-S0_20090212.19
7e46b9a6e1ae3e19bc9b5cd92fdc1f31 ow-S0_20090212.20
42ec87d05bcdf9bfe9a227353f34cc90 ow-S0_20090212.21
fb46ea8ba74b07770c3a793c53373c81 ow-S0_20090212.22
ff56b39dafdd36f6c56c3e17514a98c8 ow-S0_20090212.23
5e1135afca60a00ee452c28615cd1116 ow-S0_20090213.00
1657fec1aacd9ab3b02ebe87e8ba77bb ow-S0_20090213.01
63f7be5680ea8d91b741758f316fb7da ow-S0_20090213.02
5e998754801f8b9cbd92e28e6715bcd8 ow-S0_20090213.03
ae033c0b2767ab2548090d95aa59c3b3 ow-S0_20090213.04
77fd7dac6a674c9086afcc96e74180f1 ow-S0_20090213.05
aebaad9152818c2afc386b744751c6ca ow-S0_20090213.06
f0a958e579bac8ce7de3a791b882470a ow-S0_20090213.07
44c4c9a36b14af3cae37e22494e13da4 ow-S0_20090213.08
83d0dda14f3ff32ff35aedf33aa501bd ow-S0_20090213.09
17786926eb9e6ed4a201216ac8ccef95 ow-S0_20090213.10
cf0672ec795a961d00bc1cebb1450540 ow-S0_20090213.11
9477161a72ff01fbe86e3d5f82253892 ow-S0_20090213.12
c72d8fe60bb7c53514ca282912fe1bb2 ow-S0_20090213.13
3a1c7c468c58b37b0b1aa1e9a8e40349 ow-S0_20090213.14
e5b6fbd80e28b9e4be611e23b5af1e62 ow-S0_20090213.15
4097a7d84cd4e1f9bc7e3f599137accb ow-S0_20090213.16
eeb14db22a7368c199ae183b493fb53b ow-S0_20090213.17
fa04c7372c019fb35eb56826717c45e9 ow-S0_20090213.18
bbdc07269a78efb2118fe6546b3f9ab9 ow-S0_20090213.19
63e643e85e6af8a519b7c7da844d8b5e ow-S0_20090213.20
c5ef8ee58a381c6d46780ca121bbb228 ow-S0_20090213.21
eff3e849e2058119af854df5560ba377 ow-S0_20090213.22
1f4b96dc79adcde28c430868ad53c944 ow-S0_20090213.23
da4987bf67ba15441ff44864b11ed30e ow-S0_20090214.00
722f4c446141b43651f5f87a6e71590d ow-S0_20090214.01
e538cfe67451f2982ae47d6dbab97859 ow-S0_20090214.02
d6c9c45fc578597855180805016177b4 ow-S0_20090214.03
0d9bec42d45de1467ce0a69eb559e917 ow-S0_20090214.04
c2c54487829712acf2a43d1f920f61a6 ow-S0_20090214.05
202586b4fcb642af2fedf5534d58532b ow-S0_20090214.06
697c3b82906e95556c02f42aed174cbd ow-S0_20090214.07
912c32f60a13d39e08980361f888b8f5 ow-S0_20090214.08
2efbc660b35ba8b290d4db05cb5dc10f ow-S0_20090214.09
f744175b978aba1e707a538475f41065 ow-S0_20090214.10
7e7c4fcb4533b7df9fda8663303a2f6f ow-S0_20090214.11
4740a1a0593e4c4a757da882a552b9a7 ow-S0_20090214.12
84684d88d78352d51db24b16b1efe938 ow-S0_20090214.13
34e72b7e44c610c006a3eb3916ae5379 ow-S0_20090214.14
a156d643819044d13333b7da78c44dca ow-S0_20090214.15
438c5af803a5d7b3e7394f6cbbc2849b ow-S0_20090214.16
5080aec646eb5823b6a4786f94b9cae4 ow-S0_20090214.17
75483762b44176155990f99ed77136be ow-S0_20090214.18
92abae85970890bc4481ad96e9bda4a7 ow-S0_20090214.19
ab97291ab89c3a581545648cd3b1c01d ow-S0_20090214.20
863b9ab2173471862ae44c80657f4cd9 ow-S0_20090214.21
bcd293b1f32563177e498b2266ae45b9 ow-S0_20090214.22
22351564b6a31420ee123870a70e6c52 ow-S0_20090214.23
