#! /usr/bin/perl -w
# MD5: multiple
# TEST: ../rwfilter/rwfilter --stime=2009/02/13:20:00-2009/02/13:20 --sensor=S2 --proto=6 --aport=80,8080,443 --pass=stdout ../../tests/data-v6.rwf | ./rwallformats --no-invocation --basename=/tmp/sk-teststmp && md5 /tmp/sk-teststmp*

use strict;
use SiLKTests;
use File::Find;

# name of this script
my $NAME = $0;
$NAME =~ s,.*/,,;

my $rwallformats = check_silk_app('rwallformats');
my $rwcat = check_silk_app('rwcat');
my $rwfilter = check_silk_app('rwfilter');
my $rwfileinfo = check_silk_app('rwfileinfo');
my %file;
$file{v6data} = get_data_or_exit77('v6data');

my $base_name = 'sk-teststmp';

# create our tempdir
my $tmpdir = make_tempdir();

# create subdirs for writing and reading with rwcat
my $write_dir = "$tmpdir/write";
my $rwcat_dir = "$tmpdir/rwcat";
for my $d ($write_dir, $rwcat_dir) {
    mkdir $d
        or die "$NAME: Cannot mkdir '$d': $!";
}

my $cmd = "$rwfilter --stime=2009/02/13:20:00-2009/02/13:20 --sensor=S2 --proto=6 --aport=80,8080,443 --pass=stdout $file{v6data} | $rwallformats --no-invocation --basename='$write_dir/$base_name'";
if (!check_exit_status("$cmd 2>/dev/null")) {
    exit 1;
}

# get list of expected MD5s for each file from the end of this file;
# these lists include the MD5 of the file itself (stored in
# %checksums) and the result of running rwcat on the file (stored in
# %readback).  some files have multiple checksums due to differences
# in LZO or other options
my %checksums;
my %readback;
while (<DATA>) {
    next unless /\w/;
    next if /^\#/;

    my ($expect, $tail_name) = split " ";

    # Handle checksums for reading the data with rwcat
    if ($tail_name !~ /-c\d+-[BL]/) {
        push @{$readback{$tail_name}}, $expect;
        next;
    }

    if (exists $checksums{$tail_name}) {
        push @{$checksums{$tail_name}}, $expect;
    }
    elsif (!$SiLKTests::SK_ENABLE_ZLIB && $tail_name =~ /-c1-/) {
        # skip
    }
    elsif (!$SiLKTests::SK_ENABLE_LZO && $tail_name =~ /-c2-/) {
        # skip
    }
    elsif (!$SiLKTests::SK_ENABLE_SNAPPY && $tail_name =~ /-c3-/) {
        # skip
    }
    else {
        $checksums{$tail_name} = [$expect];
    }
}

# hash to store names of files that had an issue; value is an array of
# possible isues: 'missing', 'file-mismatch', 'read-mismatch',
# 'unknown-file', 'unknown-readback'
#
my %mismatch;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $write_dir);

# handle files we did not find by adding missing entries to %mismatch
for my $missing (keys %checksums) {
    push @{$mismatch{"$base_name-$missing"}}, 'missing';
}

# print results: Files are sorted by the reverse of the name so all
# differences across a wide range of file formats and versions are
# seen first.
#
my @badfiles = sort { reverse($a) cmp reverse($b) } keys %mismatch;
if (@badfiles) {

    my %issues;

    my $msg = sprintf("%s: Found %u file%s with problems:",
                      $NAME, scalar(@badfiles),
                      ((scalar(@badfiles) == 1) ? "" : "s"));
    print STDERR $msg, "\n";

    for my $file (@badfiles) {
        for my $i (@{$mismatch{$file}}) {
            ++$issues{$i};
        }
        print STDERR join(" ", "$NAME:", "$file -", @{$mismatch{$file}}), "\n";
    }

    my $issuemsg = join ", ", map { "$_: ".$issues{$_} } keys %issues;

    die($msg, " ", $issuemsg, "\n");
}

# successful!
exit 0;


# This function is called by File::Find::find.  The full path to the
# file is in the $_ variable.
#
# The function checks for the file in the %checksums hash.  If not
# present, an entry is added to %mismatch for an unknown file and the
# function returns.
#
# Files in %checksums hash have their values removed from the hash so
# mising files may be reported at shutdown.
#
# The function computes the MD5 of the file and compares it to the
# expected value; if the values do not match, the function puts the
# file name into the %mismatch hash with a file-mismatch.
#
# The function runs rwcat on the file and computes the MD5 of the
# result.  If that does not match the expected value, the function
# puts the file name into the %mismatch hash with a read-mismatch.
#
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;

    my $path = $_;

    # file name without $write_dir
    my $file = $_;
    $file =~ s,^\Q$write_dir/\E,,;

    # set $tail_name to be the varying part of the filename; that is,
    # remove the directory and base_name
    my $tail_name = $_;
    $tail_name =~ s,^$write_dir/$base_name-,,;

    # set $read_name to be $tail_name truncated at the version (that
    # is, minus the compresssion and byte-order)
    my $read_name = $tail_name;
    $read_name =~ s,^(\w+-v\d+)-.+,$1,;

    unless (exists $checksums{$tail_name}) {
        push @{$mismatch{$file}}, 'unknown-file';
        return;
    }

    # file checksums are stored as an array; delete the checksums for
    # files we find so that we may report missing files
    my @sums = @{$checksums{$tail_name}};
    delete $checksums{$tail_name};

    # check file itselef
    my $file_md5;
    compute_md5_file(\$file_md5, $path, 0);
    if (! grep {$_ eq $file_md5} @sums ) {
        print STDERR "$NAME: $tail_name: file checksum mismatch [$file_md5]",
            " (expected ", join(",", @sums), ")\n";
        push @{$mismatch{$file}}, 'file-mismatch';
    }

    unless (exists $readback{$read_name}) {
        push @{$mismatch{$file}}, 'unknown-readback';
        return;
    }

    # check reading
    my $read_md5;
    my $read_cmd = ("$rwcat --byte-order=little --compression=none"
                    ." --output-path=$rwcat_dir/$file $path");
    check_exit_status($read_cmd, 1);
    compute_md5_file(\$read_md5, "$rwcat_dir/$file");
    if (! grep {$_ eq $read_md5} @{$readback{$read_name}}) {
        print STDERR "$NAME: $read_name: read checksum mismatch [$read_md5]",
            " (expected ", join(",", @{$readback{$read_name}}), ")\n";
        push @{$mismatch{$file}}, 'read-mismatch';
    }
}



__DATA__
#
#   File formats capable of holding only IPv4 addresses contain only the
#   file header.  On readback, only the column headers appear as output.
#
8833c2f15a7242f83f154e1a757b70e2 FT_FLOWCAP-v2-c0-B.dat
598252c89164a4d1205a449f4cb145a8 FT_FLOWCAP-v2-c0-L.dat
9c39415c6060ffab173ede8e05c8893d FT_FLOWCAP-v2-c1-B.dat
b472730fcd4d17220ea6be01714afbf2 FT_FLOWCAP-v2-c1-L.dat
a0425112595004f0326043fc04f8d7aa FT_FLOWCAP-v2-c2-B.dat
5126de92998996b00f3b2d7a6a350b89 FT_FLOWCAP-v2-c2-L.dat
795c5735af66e63ca7874694e4430176 FT_FLOWCAP-v2-c3-B.dat
d8b3771595269c0d0c02ef7729b9fe1d FT_FLOWCAP-v2-c3-L.dat
c9611092085997070f3aab9cacf34cf2 FT_FLOWCAP-v3-c0-B.dat
67531bec946e1678d93887c7712b217e FT_FLOWCAP-v3-c0-L.dat
e1ed5cc4da1c15ecd09c07963a3fbdd6 FT_FLOWCAP-v3-c1-B.dat
d1ddae4cb7154b298170e0c60df78775 FT_FLOWCAP-v3-c1-L.dat
3347ca61535d0826930df1a7bde1e8a8 FT_FLOWCAP-v3-c2-B.dat
9a64d90e9b94187f67a87dd8691de448 FT_FLOWCAP-v3-c2-L.dat
8f29e727812a837442da46e97239c516 FT_FLOWCAP-v3-c3-B.dat
c0f253342584e9f80cd96f4e7a935e23 FT_FLOWCAP-v3-c3-L.dat
df9a704ff642bb8a184173897c0cb10e FT_FLOWCAP-v4-c0-B.dat
49966c961c58ede5e8b7c67840bf3ccd FT_FLOWCAP-v4-c0-L.dat
cf1bdb1f57568e9182320d946df385f1 FT_FLOWCAP-v4-c1-B.dat
045ea223c5b644f9e5acb6512528f32b FT_FLOWCAP-v4-c1-L.dat
99974c597959b977e6f9d7e869832454 FT_FLOWCAP-v4-c2-B.dat
5e6bd55a451f028473937acb5a51422d FT_FLOWCAP-v4-c2-L.dat
4db5317165a6955d49a9af829f3de727 FT_FLOWCAP-v4-c3-B.dat
a69b0c9ef1a9f8826e898e9aac6b44ce FT_FLOWCAP-v4-c3-L.dat
a1676e54ab1d39d7bb29f0761b24e8ea FT_FLOWCAP-v5-c0-B.dat
aed73481454039f57edcfec583b078ae FT_FLOWCAP-v5-c0-L.dat
8d39b3ffc2905264f88ff92c46b51c77 FT_FLOWCAP-v5-c1-B.dat
e251a21713dccb4ee1735af523a84390 FT_FLOWCAP-v5-c1-L.dat
06384fb435791a33683b35ce724b9e7b FT_FLOWCAP-v5-c2-B.dat
841739481d23a339bb1b0371b28fc892 FT_FLOWCAP-v5-c2-L.dat
0707c8f64404b38a4f37ef066a9616ac FT_FLOWCAP-v5-c3-B.dat
4d7df03bf1cbc3ce0e00803a7d00c146 FT_FLOWCAP-v5-c3-L.dat
a0c7aac6a407cb04394e1ed29b2f9fea FT_FLOWCAP-v6-c0-B.dat
69a7748ea5b2fcc5ee8f16301e4da3ce FT_FLOWCAP-v6-c0-L.dat
2bd1adfc9f3cb577c72da38131e45dc0 FT_FLOWCAP-v6-c1-B.dat
f7e1682b08dcf029c81f848888d3869c FT_FLOWCAP-v6-c1-L.dat
4e48eba9f1a21788dac380538f48cec6 FT_FLOWCAP-v6-c2-B.dat
af52dd5a7dcc600dbc470e420cd1cc0f FT_FLOWCAP-v6-c2-L.dat
9481398139097bc8af8ae7a721c1fece FT_FLOWCAP-v6-c3-B.dat
5e13e2502e25c2c6f4f3760701b78c99 FT_FLOWCAP-v6-c3-L.dat
d1f2bc2c3df9f0c6f042c1414f56a5a4 FT_RWAUGMENTED-v1-c0-B.dat
4709cded22491877bcf7a41de57e1ce4 FT_RWAUGMENTED-v1-c0-L.dat
74fa21bac8f1c57ded7925006185caa8 FT_RWAUGMENTED-v1-c1-B.dat
4715908f38a7de264a544f0b421657f7 FT_RWAUGMENTED-v1-c1-L.dat
91c9bc763d9f6e43065f7b1dc51d9827 FT_RWAUGMENTED-v1-c2-B.dat
992877d48e776a0d9ff623ca2af22e41 FT_RWAUGMENTED-v1-c2-L.dat
74a06e2b8bf58ca4ea285f15e5d19a9c FT_RWAUGMENTED-v1-c3-B.dat
2e977d957a50920d075bbab60ec04b20 FT_RWAUGMENTED-v1-c3-L.dat
b56c93baf08d5f163143e30071f347ee FT_RWAUGMENTED-v2-c0-B.dat
68d0b42ca2d258ae4cce26de856f96ab FT_RWAUGMENTED-v2-c0-L.dat
083d5a99fabe2d5ecd38e98f77ebe593 FT_RWAUGMENTED-v2-c1-B.dat
399bfd810c099c2419ef6ded5835845e FT_RWAUGMENTED-v2-c1-L.dat
51a0978e455441f6b43035e276930100 FT_RWAUGMENTED-v2-c2-B.dat
133fffae4404aea454d826a53a6dee6d FT_RWAUGMENTED-v2-c2-L.dat
e76ddee59a89f529718dac648dfbc629 FT_RWAUGMENTED-v2-c3-B.dat
1b57e6f24b8ae3f2d72a9681080530a0 FT_RWAUGMENTED-v2-c3-L.dat
032676f13ac440adf0ddaa3b4e475f99 FT_RWAUGMENTED-v3-c0-B.dat
8eb4a37f3dc0bcc2c567e46786c4a2df FT_RWAUGMENTED-v3-c0-L.dat
4f44e45135c1af37a7ea7af0de77a966 FT_RWAUGMENTED-v3-c1-B.dat
6bb0e9913189f3b8b00dd4be7edadaf8 FT_RWAUGMENTED-v3-c1-L.dat
3d29dd7a8311a8b10b67c63c68b76a9e FT_RWAUGMENTED-v3-c2-B.dat
090336d0f5016e57a19d64481e468a83 FT_RWAUGMENTED-v3-c2-L.dat
1025f40ffe2be1371283d3fb71ea50d4 FT_RWAUGMENTED-v3-c3-B.dat
0c700e46027141b974dd1e80110aa9a3 FT_RWAUGMENTED-v3-c3-L.dat
658698c924d5e2a4ff46784e16c9d701 FT_RWAUGMENTED-v4-c0-B.dat
4b76c249b6a16872108fc19d8cb1a06a FT_RWAUGMENTED-v4-c0-L.dat
780fb2bdb2202813e22d7eea265c9efa FT_RWAUGMENTED-v4-c1-B.dat
12199a25c9a3061fd2f74faef251ec28 FT_RWAUGMENTED-v4-c1-L.dat
cac354c5b4c94cb742c3dee800ea4969 FT_RWAUGMENTED-v4-c2-B.dat
ca68bdcda5a1097630f67b19d9bbd4cb FT_RWAUGMENTED-v4-c2-L.dat
6d4fba0e36191b7c662a78b0b345ae8a FT_RWAUGMENTED-v4-c3-B.dat
493717f385f25aba0695ce38db0504fd FT_RWAUGMENTED-v4-c3-L.dat
80e17be604d6e18dbeefebad7ab5dd0e FT_RWAUGMENTED-v5-c0-B.dat
7840879abd9ed15c97676c9244045ca7 FT_RWAUGMENTED-v5-c0-L.dat
a97eb1b9aa5f414139624df96c1811c9 FT_RWAUGMENTED-v5-c1-B.dat
c576bd27e1a449d2e2f1829674055aca FT_RWAUGMENTED-v5-c1-L.dat
31580f80b8e6b5a0ed2e199fa45dcacd FT_RWAUGMENTED-v5-c2-B.dat
8732f1d0d93d2a5a7248c64889ca5454 FT_RWAUGMENTED-v5-c2-L.dat
1fd6c78adbae5c5d2c0b449fe80c57da FT_RWAUGMENTED-v5-c3-B.dat
5b11a8a32f9c34ee08addf5211957767 FT_RWAUGMENTED-v5-c3-L.dat
7f803a62641cf09c1d60178e02c21578 FT_RWAUGMENTED-v6-c0-B.dat
a9c832af936ca4e8a813d0d2e01f06bc FT_RWAUGMENTED-v6-c0-L.dat
14a657b9e5a720c9267c749d7f48d4f7 FT_RWAUGMENTED-v6-c1-B.dat
cf3b6dda236bf0c9e607ee7209ef6d8b FT_RWAUGMENTED-v6-c1-L.dat
5ced3f6f161b18fcbb0b49b56fa02b83 FT_RWAUGMENTED-v6-c2-B.dat
ad706ede51f85f30839f36c70b10d0fd FT_RWAUGMENTED-v6-c2-L.dat
59e3736b3b7df4b8e0a67782abc35bef FT_RWAUGMENTED-v6-c3-B.dat
c1e1965a01ad92c1cf4f6288c4030c2f FT_RWAUGMENTED-v6-c3-L.dat
bd79d23b3df43235ea3d4e5290d611e7 FT_RWAUGROUTING-v1-c0-B.dat
f45cd6e551bee7f7e31515c065b57b24 FT_RWAUGROUTING-v1-c0-L.dat
661f2a31e0d5aca1437916bdb3a332e4 FT_RWAUGROUTING-v1-c1-B.dat
2845a15990d046823f275817e95bfa6a FT_RWAUGROUTING-v1-c1-L.dat
6b11326b3cc7f74f87c082dc96cecd1a FT_RWAUGROUTING-v1-c2-B.dat
72c1d5f5148d911318968635a688044b FT_RWAUGROUTING-v1-c2-L.dat
3954070c4dab95bce775a63724c7abe4 FT_RWAUGROUTING-v1-c3-B.dat
b5ee7be47db6591f4792f6757c5e2399 FT_RWAUGROUTING-v1-c3-L.dat
d37db80113d1a7e720c46964ff19c18e FT_RWAUGROUTING-v2-c0-B.dat
3aee36bf253472f54f059aed413e4661 FT_RWAUGROUTING-v2-c0-L.dat
cd016154f73582b60f13866d151de23e FT_RWAUGROUTING-v2-c1-B.dat
ec1ee87099824f24ce031526789d799c FT_RWAUGROUTING-v2-c1-L.dat
570095cc84624f5f64445c7b6321cd8d FT_RWAUGROUTING-v2-c2-B.dat
446fedb0962750c72d8b2d73d2bf6606 FT_RWAUGROUTING-v2-c2-L.dat
110960c19c0789110ddb91a41609264e FT_RWAUGROUTING-v2-c3-B.dat
fe22dece2b3ddb16e28ec473054ce9b0 FT_RWAUGROUTING-v2-c3-L.dat
b1a5acaab0da1b944c54eb9cef9bb8a1 FT_RWAUGROUTING-v3-c0-B.dat
0b944f42f4714e27f45c656261168e67 FT_RWAUGROUTING-v3-c0-L.dat
aa8a04a8a40b9044449d9242707bd867 FT_RWAUGROUTING-v3-c1-B.dat
c6e9ceb651a64a6a469a509c768196af FT_RWAUGROUTING-v3-c1-L.dat
8f371a472e9e7b98f8b85e6f25fadfaf FT_RWAUGROUTING-v3-c2-B.dat
92f8c80c17c48aaae6fc9cb900bd21bb FT_RWAUGROUTING-v3-c2-L.dat
0b67fd103992e0bb3a73531762a723d2 FT_RWAUGROUTING-v3-c3-B.dat
7e7174892ed72d4fea0a55e4e77ec18a FT_RWAUGROUTING-v3-c3-L.dat
d9526204520fd243b0292c1cc66fcbac FT_RWAUGROUTING-v4-c0-B.dat
a177c26ef75a85cf5a0b9ea0fbecb413 FT_RWAUGROUTING-v4-c0-L.dat
b3d19c3a5638093bc077ff15a7a5cfd4 FT_RWAUGROUTING-v4-c1-B.dat
0a8df44e708621697e25c1905ba52e65 FT_RWAUGROUTING-v4-c1-L.dat
c06937ed504e55971e9a9bb02c29f2c3 FT_RWAUGROUTING-v4-c2-B.dat
42193cf22fc22dec5884d5f34d4d5be7 FT_RWAUGROUTING-v4-c2-L.dat
dc189ba469434b5e53749e1cc89b62e4 FT_RWAUGROUTING-v4-c3-B.dat
8eca38f18122d99e7f60d23433c9d4f8 FT_RWAUGROUTING-v4-c3-L.dat
bc80ce3e26b63ebc0742316614a9e3c8 FT_RWAUGROUTING-v5-c0-B.dat
641cf9d2affb39bb47b2ed1c1c6e4fd8 FT_RWAUGROUTING-v5-c0-L.dat
ee1bf6a594d902b61d6c969ba9cca982 FT_RWAUGROUTING-v5-c1-B.dat
3542cc388d5c12bf26e401c4c1b80dfe FT_RWAUGROUTING-v5-c1-L.dat
074fe73e00af9a618e3334e7e4631f19 FT_RWAUGROUTING-v5-c2-B.dat
122112e5a0ea87cf7148f302e3366be0 FT_RWAUGROUTING-v5-c2-L.dat
dacbf20ca277643c464965ff9e935413 FT_RWAUGROUTING-v5-c3-B.dat
93bf434a392d049151b90f1489ea0010 FT_RWAUGROUTING-v5-c3-L.dat
d295dd8173c32d5415d4d4cf5851fe86 FT_RWAUGROUTING-v6-c0-B.dat
363dc46c0b2bf46002982ae6443de8ff FT_RWAUGROUTING-v6-c0-L.dat
249aa9e213a7913362b5992b3f294991 FT_RWAUGROUTING-v6-c1-B.dat
9d1daae69533fd8fbf8bf662499251e0 FT_RWAUGROUTING-v6-c1-L.dat
0e691703ae2d9110a82315c074b69a6d FT_RWAUGROUTING-v6-c2-B.dat
b3eb5c3a97a0a2c2355995ce48d9a093 FT_RWAUGROUTING-v6-c2-L.dat
109eb945e93db1c5dd9c36835d9af0a5 FT_RWAUGROUTING-v6-c3-B.dat
77426d3e2339fb3a88181ea0e78bfd38 FT_RWAUGROUTING-v6-c3-L.dat
cba3c0c98d3059353489561bbf3e9ccb FT_RWAUGSNMPOUT-v1-c0-B.dat
a3c4c8cffb7b79080139dee070e4a6cb FT_RWAUGSNMPOUT-v1-c0-L.dat
ab6a1add60b6433b172410ba0468f18d FT_RWAUGSNMPOUT-v1-c1-B.dat
2e024c94bd1bd7885e18a1a72c136648 FT_RWAUGSNMPOUT-v1-c1-L.dat
5c2f10a8ad57ee7ff0dec3a94dad89be FT_RWAUGSNMPOUT-v1-c2-B.dat
03d5c5dbc15bbbc2c940ceb48c73cb70 FT_RWAUGSNMPOUT-v1-c2-L.dat
9111aec5081d3fb0d8842aeb85aa522c FT_RWAUGSNMPOUT-v1-c3-B.dat
9c89a5541d8fb647d233f7d40f503826 FT_RWAUGSNMPOUT-v1-c3-L.dat
fe2f1b26b46c953f95d537224e2fbee6 FT_RWAUGSNMPOUT-v2-c0-B.dat
a7b744a83830e319fa3af92e51ca1c3a FT_RWAUGSNMPOUT-v2-c0-L.dat
4983cecc7ac0cfbd83161e9dabbcb691 FT_RWAUGSNMPOUT-v2-c1-B.dat
007579b25e9796b440af6d6efd171857 FT_RWAUGSNMPOUT-v2-c1-L.dat
6503041344f6c83a6aab57cc605244ec FT_RWAUGSNMPOUT-v2-c2-B.dat
b56330c0b8b7790700fdfb4f18815fc9 FT_RWAUGSNMPOUT-v2-c2-L.dat
b2803bb511406c079c344706157d8f88 FT_RWAUGSNMPOUT-v2-c3-B.dat
953c6e4b9bf96760cd60554aa8fc6b2b FT_RWAUGSNMPOUT-v2-c3-L.dat
18b33e05cd79609429c7368cfe98970c FT_RWAUGSNMPOUT-v3-c0-B.dat
f5b56b012da991b91b835e2dc2e5aa0c FT_RWAUGSNMPOUT-v3-c0-L.dat
bf8034857adfece4ca7770bbabf80a74 FT_RWAUGSNMPOUT-v3-c1-B.dat
45bda1c9afe83e0a639374e1b87869f5 FT_RWAUGSNMPOUT-v3-c1-L.dat
4b3a98834e77b91710fbb3e32eee3468 FT_RWAUGSNMPOUT-v3-c2-B.dat
dc3c7357fde5b0c43474fc352cf533c2 FT_RWAUGSNMPOUT-v3-c2-L.dat
554e9253399fe6b0229bd4fccc674ba3 FT_RWAUGSNMPOUT-v3-c3-B.dat
30eaae16c04d5445eeec01fe4e3d5314 FT_RWAUGSNMPOUT-v3-c3-L.dat
8ab31704064f30f61aec33cf96a0c21a FT_RWAUGSNMPOUT-v4-c0-B.dat
dd3f35759d02cdb1bf444b8929440043 FT_RWAUGSNMPOUT-v4-c0-L.dat
18389f69cb365019d9dc4a961261805c FT_RWAUGSNMPOUT-v4-c1-B.dat
7fc703ea3e45387b032e1a9602e3d399 FT_RWAUGSNMPOUT-v4-c1-L.dat
148d769dd4210b30d85b0f35afd3479e FT_RWAUGSNMPOUT-v4-c2-B.dat
38a40858d4fe0162c7f19cd4805f3fb6 FT_RWAUGSNMPOUT-v4-c2-L.dat
1b230100cfb919c18a7b90f4771a24e6 FT_RWAUGSNMPOUT-v4-c3-B.dat
1e990161daa65f2ffad81075be4dac3e FT_RWAUGSNMPOUT-v4-c3-L.dat
440c7f68211cd5795e282603e0ac77fc FT_RWAUGSNMPOUT-v5-c0-B.dat
e8721bc3121ce005e3c5605933ac2919 FT_RWAUGSNMPOUT-v5-c0-L.dat
26e43c2c26a440bfce046f1416ac8fd6 FT_RWAUGSNMPOUT-v5-c1-B.dat
692b9ee1ee00c64904e4c6198be15151 FT_RWAUGSNMPOUT-v5-c1-L.dat
040c5f690df819267007806a072b5cab FT_RWAUGSNMPOUT-v5-c2-B.dat
b684fa3b8f010c29cf744afd3ce85c01 FT_RWAUGSNMPOUT-v5-c2-L.dat
0b4166729295ec8ef97363faf1921f1a FT_RWAUGSNMPOUT-v5-c3-B.dat
a1521270d16432245d873616f9402fa8 FT_RWAUGSNMPOUT-v5-c3-L.dat
e2c7275a72b144441eae92c944067fc6 FT_RWAUGWEB-v1-c0-B.dat
3722323df3bb8dc7bd0953c38d0373a3 FT_RWAUGWEB-v1-c0-L.dat
e0f728267487fec99cd407ee7b988c3a FT_RWAUGWEB-v1-c1-B.dat
80b7f651ba897b203cc7bdf5ae43cf1a FT_RWAUGWEB-v1-c1-L.dat
ea51a0b824a6a51ff515e65ec68ffe43 FT_RWAUGWEB-v1-c2-B.dat
0d9835d36cd71ddff103dd74c5baeda4 FT_RWAUGWEB-v1-c2-L.dat
6d9b88a5e915f82a13b541cd2969ebd0 FT_RWAUGWEB-v1-c3-B.dat
55f9d5bf9d0a2d6016105f0e60f808e6 FT_RWAUGWEB-v1-c3-L.dat
aaa451e9a3da7d38da4ac03607754020 FT_RWAUGWEB-v2-c0-B.dat
09dd6393cec91d52a75b0ee2a535399f FT_RWAUGWEB-v2-c0-L.dat
f44d8a70bc3007384756ed40bcf72be2 FT_RWAUGWEB-v2-c1-B.dat
125f2e88ae1e0ca71f8f25aba1cabdb4 FT_RWAUGWEB-v2-c1-L.dat
6b7b4483dc174416d3dd9c9fe15f4daf FT_RWAUGWEB-v2-c2-B.dat
434c7fd32a4f08aff3d1330f498f7ce7 FT_RWAUGWEB-v2-c2-L.dat
6db8aa6a77c5b3c33c6f09e7dbc87628 FT_RWAUGWEB-v2-c3-B.dat
058c16f1cf16eeaeb24b08ee0cc05fb6 FT_RWAUGWEB-v2-c3-L.dat
53c7497fc3ccdfbd6a2bf3047de46ecb FT_RWAUGWEB-v3-c0-B.dat
cba26bcd24968a1529094f6988fdd9df FT_RWAUGWEB-v3-c0-L.dat
8f185d502c2974d8e9f0f96ec3d3355d FT_RWAUGWEB-v3-c1-B.dat
d6015804c6875248fc83ce79676f819a FT_RWAUGWEB-v3-c1-L.dat
b8e5ff386db2a7a8393e650774917958 FT_RWAUGWEB-v3-c2-B.dat
74a973d73bc9e7e957c4196deeee7ccb FT_RWAUGWEB-v3-c2-L.dat
8f6156797296c6cfcdc774ddae4c15fd FT_RWAUGWEB-v3-c3-B.dat
dbe15629398ba2efbe36be9a888ff8e3 FT_RWAUGWEB-v3-c3-L.dat
c18678f123f6396a047cc42f28c9b24f FT_RWAUGWEB-v4-c0-B.dat
a4bbcd01d7428e3bc3b4499c9faf00c4 FT_RWAUGWEB-v4-c0-L.dat
d59233e1ad63076f4157babdbbb1987a FT_RWAUGWEB-v4-c1-B.dat
41fe2aa8aea4ad13b84ca0abe3eef71a FT_RWAUGWEB-v4-c1-L.dat
161ef4ea7f7e33503360ddcabde88072 FT_RWAUGWEB-v4-c2-B.dat
b2297e03225e83ca926a96127732d692 FT_RWAUGWEB-v4-c2-L.dat
e454f8fbe65f640d5a2358da6d882613 FT_RWAUGWEB-v4-c3-B.dat
f14b06c6bbea02992b1ae987a6c0756e FT_RWAUGWEB-v4-c3-L.dat
149b47e8e3071de44e3cd0e05e73120b FT_RWAUGWEB-v5-c0-B.dat
b8cfdd797a909c0c9f311ab3d9a34751 FT_RWAUGWEB-v5-c0-L.dat
f4d01b469b2da073a564ff190bc5b7cb FT_RWAUGWEB-v5-c1-B.dat
e31e97c446309773ec4f9234d7b52aa4 FT_RWAUGWEB-v5-c1-L.dat
6c98d2e7a88958cc91026711234e015b FT_RWAUGWEB-v5-c2-B.dat
1a9efdea9ba793db41e29974ced1fb05 FT_RWAUGWEB-v5-c2-L.dat
27ca25ee03b6e8eaaeb9659de8037097 FT_RWAUGWEB-v5-c3-B.dat
21e3fd4a3910c77f6bef4ba0c113ebd6 FT_RWAUGWEB-v5-c3-L.dat
81416e45bef0a8c9975b6765d1292ccd FT_RWAUGWEB-v6-c0-B.dat
6165b1a38394e602a7f30911cccabf86 FT_RWAUGWEB-v6-c0-L.dat
5b466a4a69b82893378f840b941d27db FT_RWAUGWEB-v6-c1-B.dat
03e90bed5ead6cb03496e269d83398f9 FT_RWAUGWEB-v6-c1-L.dat
5f011e500ca2cb2822696f2ec7c62193 FT_RWAUGWEB-v6-c2-B.dat
c8073eacf95db889f9467f7c1cc6f833 FT_RWAUGWEB-v6-c2-L.dat
2fb954a114874f6a4f10633fe0cc6822 FT_RWAUGWEB-v6-c3-B.dat
14e3633364b4c095fe6054d95e6293ec FT_RWAUGWEB-v6-c3-L.dat
b0b28c4f629f5a11413d67892bcec672 FT_RWFILTER-v1-c0-B.dat
9650289285bc6caebc892916f01448c7 FT_RWFILTER-v1-c0-L.dat
5181f63ab7730a7c1acb3e7c573ffebb FT_RWFILTER-v1-c1-B.dat
af7a559bbb5924c0f5c19c43e992de3c FT_RWFILTER-v1-c1-L.dat
3ee4ca32c0ce56c2de314775240d3892 FT_RWFILTER-v1-c2-B.dat
80e2e94fbd97521f16c88bbbb73dc68e FT_RWFILTER-v1-c2-L.dat
7a5907572e23557a6c4e61e49161c327 FT_RWFILTER-v1-c3-B.dat
06df34c1ae866da222a543146f42560c FT_RWFILTER-v1-c3-L.dat
22ffbd9f9cdd3eec2944bc994574cd3a FT_RWFILTER-v2-c0-B.dat
c8288d93ddf262e2678399875687605e FT_RWFILTER-v2-c0-L.dat
b13b9ee23d97be4c49ea50dbce10c552 FT_RWFILTER-v2-c1-B.dat
d8f1821bad80a0aa23bb92ef0269ab46 FT_RWFILTER-v2-c1-L.dat
77cec57aacc75862d9a11b71c06cb250 FT_RWFILTER-v2-c2-B.dat
370641561cd1bb224e0378eed34174e7 FT_RWFILTER-v2-c2-L.dat
4ba61fed86699ca7151a6659d392453d FT_RWFILTER-v2-c3-B.dat
854d19dfd5df20e2487f7e8abcdbd98e FT_RWFILTER-v2-c3-L.dat
1b9cef852320997d891b733be302f637 FT_RWFILTER-v3-c0-B.dat
6bcf942038b593023ab9c5f331c54163 FT_RWFILTER-v3-c0-L.dat
31a42b8337ee103ac1ca46ac856d7450 FT_RWFILTER-v3-c1-B.dat
d5c93e6b6bdaeb8de3a4fcc403730f14 FT_RWFILTER-v3-c1-L.dat
101444757b6ae4b5862d27890f44b4e9 FT_RWFILTER-v3-c2-B.dat
a575b5b5d69f1e7134148de6e47f168e FT_RWFILTER-v3-c2-L.dat
7670459004905a17bba1406c7187e226 FT_RWFILTER-v3-c3-B.dat
e5768a70d758ee556ff939bbfb966e1d FT_RWFILTER-v3-c3-L.dat
6db416f5c23596ac61feacef896ad280 FT_RWFILTER-v4-c0-B.dat
29bdc2284a246ba759262b8a975139aa FT_RWFILTER-v4-c0-L.dat
fc02db745beaae196ccb8bd6ed4adf15 FT_RWFILTER-v4-c1-B.dat
c37dedfcaf075287c1099844d1ed446c FT_RWFILTER-v4-c1-L.dat
4f7bc6f8690620e6e5ec65426e8b955c FT_RWFILTER-v4-c2-B.dat
74107200e43f9a25e3b1d0dceba29044 FT_RWFILTER-v4-c2-L.dat
b5c922e6fed078f6465a3506f7d48f9f FT_RWFILTER-v4-c3-B.dat
96b6b12b7e34f76b2af1a69903ce40ab FT_RWFILTER-v4-c3-L.dat
3cc03b55e225ebea6522c470e9f94929 FT_RWFILTER-v5-c0-B.dat
4d174c4ff235a3610bff512581140478 FT_RWFILTER-v5-c0-L.dat
ec38e6709233293ad819674864ef4707 FT_RWFILTER-v5-c1-B.dat
2c743906bb1027d8eb0c6fea02a18b06 FT_RWFILTER-v5-c1-L.dat
560db7d423ad413cd1c8d6aee5ebdec4 FT_RWFILTER-v5-c2-B.dat
df0cbdce562b925726d1f96273508bf5 FT_RWFILTER-v5-c2-L.dat
cd393bad5542abea58b877a3ee4703e1 FT_RWFILTER-v5-c3-B.dat
db3db581237ba733086da4ee6b6bcc93 FT_RWFILTER-v5-c3-L.dat
2452b43d09d09dd2d57ccc5fb6f0d9fe FT_RWGENERIC-v0-c0-B.dat
3d76f254bffd886da84884fc262299d3 FT_RWGENERIC-v0-c0-L.dat
9c0751c890aa3fbe5dbec2cf26dfa5d6 FT_RWGENERIC-v0-c1-B.dat
61db508f01a4f6902f173576e5642c62 FT_RWGENERIC-v0-c1-L.dat
58c4f3e5655d470b9025c6970d8b74d4 FT_RWGENERIC-v0-c2-B.dat
d305dad58a1b2971529242946783d565 FT_RWGENERIC-v0-c2-L.dat
901d9e986f6daaca5c024920aa5f4931 FT_RWGENERIC-v0-c3-B.dat
7545b1e13316c1d14fbc559a34594c7d FT_RWGENERIC-v0-c3-L.dat
817fe6468bb1bb75b468ba08dd081050 FT_RWGENERIC-v1-c0-B.dat
60cdfe25511dc661f31b12e4ecb7bb7f FT_RWGENERIC-v1-c0-L.dat
94101f3f90edbc61e755d413ffe9e3de FT_RWGENERIC-v1-c1-B.dat
30d9c9847421426cd1ded14e90aba46b FT_RWGENERIC-v1-c1-L.dat
821be67388d33c999ecafc78be0411a3 FT_RWGENERIC-v1-c2-B.dat
aae990a40b46ba424275104bbd8f0fc7 FT_RWGENERIC-v1-c2-L.dat
fb00083ebba844e0d2ba4aeeea47d17c FT_RWGENERIC-v1-c3-B.dat
694b38e53b43921a2bf99f3ed6f544c9 FT_RWGENERIC-v1-c3-L.dat
f617a8bd0716fdf668e8b09b7ea7cae5 FT_RWGENERIC-v2-c0-B.dat
e1daa4425596b50a728ee4b39a680142 FT_RWGENERIC-v2-c0-L.dat
09054463f976296c8e27bfe4730e77af FT_RWGENERIC-v2-c1-B.dat
6d21766bb9ea2bd31eb310ae6b533016 FT_RWGENERIC-v2-c1-L.dat
2008915818099a3786c4ebd40b6970e0 FT_RWGENERIC-v2-c2-B.dat
aa2369f18ca128884ba8c74b7e3f5574 FT_RWGENERIC-v2-c2-L.dat
5a2c48eae90aaffc35316d16bcb405e9 FT_RWGENERIC-v2-c3-B.dat
5a203d9900c40b0a6a998fedc56b04ca FT_RWGENERIC-v2-c3-L.dat
c44cad33f58a39499ba5aff7fade9fdb FT_RWGENERIC-v3-c0-B.dat
fb79cf418265dab07e320c93fdc34ff1 FT_RWGENERIC-v3-c0-L.dat
34f73ac01ed06972ae8c83cabfba3fff FT_RWGENERIC-v3-c1-B.dat
fa8f5c60caff538beb76e37deae12658 FT_RWGENERIC-v3-c1-L.dat
e34ab91bd12c6bf533454f8150a0fe36 FT_RWGENERIC-v3-c2-B.dat
e78ee2c17e563ee51e4dbd6c8f911b12 FT_RWGENERIC-v3-c2-L.dat
1c80c594f0874d6576d143a9abe636d1 FT_RWGENERIC-v3-c3-B.dat
13d95893df1f11f089bf8e7d3326c7e1 FT_RWGENERIC-v3-c3-L.dat
f7417582a7be8eaab81c2c1b56fb26be FT_RWGENERIC-v4-c0-B.dat
2077d9bb161641bd18513937d72d7d54 FT_RWGENERIC-v4-c0-L.dat
f645bd9783b57f288212b1dc767e4018 FT_RWGENERIC-v4-c1-B.dat
88f3fc47f0b4ca945952e042562b6f1b FT_RWGENERIC-v4-c1-L.dat
74e953abe2af659ef2ab9bbcef5ccda6 FT_RWGENERIC-v4-c2-B.dat
bba84e2d849e55b05d29e8325e47d3f8 FT_RWGENERIC-v4-c2-L.dat
ec8789272caf1a2bd1feeff6a9736c3b FT_RWGENERIC-v4-c3-B.dat
80cb42e05d76d14757390981e2330bd2 FT_RWGENERIC-v4-c3-L.dat
b505264632622f0dd57e086335b991e3 FT_RWGENERIC-v5-c0-B.dat
920f27499ae17fd1a3f109c66b43ea93 FT_RWGENERIC-v5-c0-L.dat
64be40dfabb1af0d79dcc9ed01c0290c FT_RWGENERIC-v5-c1-B.dat
78186d0844d663744ac130d8fdb41bfd FT_RWGENERIC-v5-c1-L.dat
fe5366b62f9799822156bbfa05858cd2 FT_RWGENERIC-v5-c2-B.dat
9aedbf291be85fb0ca4e18cd4ef17bd6 FT_RWGENERIC-v5-c2-L.dat
95ccad7822986a5d8dae668403773556 FT_RWGENERIC-v5-c3-B.dat
b3a40457ff6ecd34c781f12210b59e91 FT_RWGENERIC-v5-c3-L.dat
d0e0094bcfd2d91d2dc78661a08a02b0 FT_RWGENERIC-v6-c0-B.dat
b3f39d7014e99f5335a2d897a1b1b6a9 FT_RWGENERIC-v6-c0-L.dat
bad1bd322d3eb6d491f0ba0556733538 FT_RWGENERIC-v6-c1-B.dat
d6c4578434358e2a85c0ad654ebac062 FT_RWGENERIC-v6-c1-L.dat
68a29aedf79d04cdc7950c6d2b364c77 FT_RWGENERIC-v6-c2-B.dat
c1d0e06b9f7cead0c5f1e6847bb99204 FT_RWGENERIC-v6-c2-L.dat
90c1946764cb49d9f476a539969de2ad FT_RWGENERIC-v6-c3-B.dat
a01ebcdd5be357d4c367bf3836c31b8e FT_RWGENERIC-v6-c3-L.dat
f794108fbc88b00a00069044d93b47e5 FT_RWIPV6-v1-c0-B.dat
14876446e91372565e698fc94b72f198 FT_RWIPV6-v1-c0-L.dat
f4a5f30ec365912d851ac639f0b86991 FT_RWIPV6-v1-c1-B.dat
4fff30edb199d0ae7b9a23d21ef0cd7f FT_RWIPV6-v1-c1-L.dat
d3fbc57a978408470793b5ea098da17f FT_RWIPV6-v1-c2-B.dat
38863e23e5a04b109c096ff1c6584f30 FT_RWIPV6-v1-c2-L.dat
fe5d207797817f35e2f286ae3489b888 FT_RWIPV6-v1-c3-B.dat
2553b70933717731554ef56b0b794537 FT_RWIPV6-v1-c3-L.dat
7d0ac88bc28f5224596160703a7b523b FT_RWIPV6-v2-c0-B.dat
f9b0390395954284a73ee3ec28390a99 FT_RWIPV6-v2-c0-L.dat
63978ad78b45eb494ea133f65ba3b545 FT_RWIPV6-v2-c1-B.dat
7fe19679bc4f269e69666b2780d9ea75 FT_RWIPV6-v2-c1-L.dat
9dd899ddf685c50cfcc2f56cfe096e4e FT_RWIPV6-v2-c2-B.dat
071114b04d877e3400184912c21a98bd FT_RWIPV6-v2-c2-L.dat
1afc26c7676c9477558e043ee8c2ac96 FT_RWIPV6-v2-c3-B.dat
1dd53e701f993138e6b0075189023add FT_RWIPV6-v2-c3-L.dat
aceb98fff0f110120481e2b3c40f558a FT_RWIPV6-v3-c0-B.dat
295078c0db42754f3beecc6aa5ea5963 FT_RWIPV6-v3-c0-L.dat
98728d8000f4d022f1e51cc02dd5cf32 FT_RWIPV6-v3-c1-B.dat
fbf826879c128a5cd9186bc3ef894364 FT_RWIPV6-v3-c1-L.dat
b6679646daa672f19cd0f58b7c4f7574 FT_RWIPV6-v3-c2-B.dat
c85760002b114a4a8ff52ff3a22093a6 FT_RWIPV6-v3-c2-L.dat
5f5facf323a9dcfb58311493ef173fc6 FT_RWIPV6-v3-c3-B.dat
6963fc97627d0dd144426adec0a16dc4 FT_RWIPV6-v3-c3-L.dat
4525546f5cff45deb6973534feff7599 FT_RWIPV6ROUTING-v1-c0-B.dat
c2f55876b3571802a87f2b7460824960 FT_RWIPV6ROUTING-v1-c0-L.dat
af35f8bfa3986c6b4fb71a51949a0451 FT_RWIPV6ROUTING-v1-c1-B.dat
2bc173b8d96de582139a7d322a945245 FT_RWIPV6ROUTING-v1-c1-L.dat
7219c351795d1ac8fab7734740b32c4a FT_RWIPV6ROUTING-v1-c2-B.dat
d2ff3736b86178f4e5ccc71f277d8a5b FT_RWIPV6ROUTING-v1-c2-L.dat
5d93c783f0dd0e0ebcdb032c7c26a886 FT_RWIPV6ROUTING-v1-c3-B.dat
982d0331fafa5a46d3d4c17fb8efb9ff FT_RWIPV6ROUTING-v1-c3-L.dat
664cb157323032152f9f92672e1de3f3 FT_RWIPV6ROUTING-v2-c0-B.dat
57a0b78324ad229122d7c97916149ca8 FT_RWIPV6ROUTING-v2-c0-L.dat
4c06e07cd27d7192ea78de60be4d6373 FT_RWIPV6ROUTING-v2-c1-B.dat
617f63252e9ef59bbe6b36eb2a0faf5e FT_RWIPV6ROUTING-v2-c1-L.dat
8fdf578aac0516681a305e4ab6a7125e FT_RWIPV6ROUTING-v2-c2-B.dat
5f9df0b3a33ae4c001e686f0ba600eb7 FT_RWIPV6ROUTING-v2-c2-L.dat
f363b5b4ccf5e226f8b5a8565b1036d8 FT_RWIPV6ROUTING-v2-c3-B.dat
b2953cb0f291bb21b4914676ac815e24 FT_RWIPV6ROUTING-v2-c3-L.dat
1db6f2d1b48d1344c21e78fee2dc7e59 FT_RWIPV6ROUTING-v3-c0-B.dat
02e70d907e890c4f0f0615a0cb892b7c FT_RWIPV6ROUTING-v3-c0-L.dat
273dbb6619ee1577f82ed21bfd7825b0 FT_RWIPV6ROUTING-v3-c1-B.dat
ac09ec62689d3ef28f28f103626d2ef3 FT_RWIPV6ROUTING-v3-c1-L.dat
15c77029330780a9aace4cc65113fa06 FT_RWIPV6ROUTING-v3-c2-B.dat
4f429ecd42541800accca022c42e570a FT_RWIPV6ROUTING-v3-c2-L.dat
c6015518b688efaaec139c56503fb1a3 FT_RWIPV6ROUTING-v3-c3-B.dat
dfb299073de450ff7d5e43d58e94ff47 FT_RWIPV6ROUTING-v3-c3-L.dat
90835164a7793123ac68de05b90c2d8f FT_RWIPV6ROUTING-v4-c0-B.dat
41fb823548116e541be6360afdd1000d FT_RWIPV6ROUTING-v4-c0-L.dat
caf7268be8b9fabe376cbd9c44ac1e60 FT_RWIPV6ROUTING-v4-c1-B.dat
71dcaeda6f84c09a630eba42fe65522f FT_RWIPV6ROUTING-v4-c1-L.dat
11f47d8697e8bd261edf6d37af394090 FT_RWIPV6ROUTING-v4-c2-B.dat
b94c5772a262a15b5f2e590923a8372e FT_RWIPV6ROUTING-v4-c2-L.dat
d8895084642d6c434be2eaff00ee330b FT_RWIPV6ROUTING-v4-c3-B.dat
62fb1f22f3566457c229b6e12044fef1 FT_RWIPV6ROUTING-v4-c3-L.dat
3f2cddfbb18a42bf333995892b781321 FT_RWNOTROUTED-v1-c0-B.dat
bec026d4136ca1487caade2a186a9d6c FT_RWNOTROUTED-v1-c0-L.dat
508996c85f185b95ba847c5fc91c1935 FT_RWNOTROUTED-v1-c1-B.dat
031bb7d243d1af6c27bc7cf5151ba31f FT_RWNOTROUTED-v1-c1-L.dat
0bb5fbaf37297e1d4c212b3a7b9cf233 FT_RWNOTROUTED-v1-c2-B.dat
dafc874f9f1f901e382900fcc96194ec FT_RWNOTROUTED-v1-c2-L.dat
f14f36f65f106a6fcebf40783aec14f0 FT_RWNOTROUTED-v1-c3-B.dat
0c17c6f338a6e21849a65abbea66f708 FT_RWNOTROUTED-v1-c3-L.dat
087fbee5edaf597e285e7981dfd143ef FT_RWNOTROUTED-v2-c0-B.dat
1170ff7fcd51899c63394a4ee9d7151d FT_RWNOTROUTED-v2-c0-L.dat
5175cd31ec7a922de153e87667b624eb FT_RWNOTROUTED-v2-c1-B.dat
ef699f403e4d51211fa3d6624be44455 FT_RWNOTROUTED-v2-c1-L.dat
f7c03fdf640f4d306576b6212afcd72a FT_RWNOTROUTED-v2-c2-B.dat
c62be560dacf4ff6afd40eea4addc5fc FT_RWNOTROUTED-v2-c2-L.dat
9ad41979e468ffa189c59d45a9ec6df0 FT_RWNOTROUTED-v2-c3-B.dat
5714ad0907a33d04a34d599450e79556 FT_RWNOTROUTED-v2-c3-L.dat
49a9506a1dd103f5c7ad1d64dbfe3bc6 FT_RWNOTROUTED-v3-c0-B.dat
63daf7a9d821ec8744aabcafb0e1b9aa FT_RWNOTROUTED-v3-c0-L.dat
87e06b40650b8c096c8affbbb8cdb630 FT_RWNOTROUTED-v3-c1-B.dat
3377720390e86bc1544c840f502c5f74 FT_RWNOTROUTED-v3-c1-L.dat
6b09a79bfb5811695e4e7f3cea27343a FT_RWNOTROUTED-v3-c2-B.dat
aa72c14c44d8154ddd9b1a9acb76cea6 FT_RWNOTROUTED-v3-c2-L.dat
7797f56ebc72b9a589a7742e32931972 FT_RWNOTROUTED-v3-c3-B.dat
53a0d993969fc10a129f84492d5242ed FT_RWNOTROUTED-v3-c3-L.dat
d8a8acd74b8669c493428c7f66d5592a FT_RWNOTROUTED-v4-c0-B.dat
9a0dccf76ade5ceee0c7415dec4535c2 FT_RWNOTROUTED-v4-c0-L.dat
463fd1555d55e4f112fc273887fff7e9 FT_RWNOTROUTED-v4-c1-B.dat
7c17fed8f80f701c18ffb85feee341eb FT_RWNOTROUTED-v4-c1-L.dat
fe62c0cf00819feff9d20de42939b388 FT_RWNOTROUTED-v4-c2-B.dat
ada454e3c1d08e7ada314ecceacf9ea7 FT_RWNOTROUTED-v4-c2-L.dat
357efa757412a67a767228f301cb96d9 FT_RWNOTROUTED-v4-c3-B.dat
17626006b45cf3d02003181107f6a711 FT_RWNOTROUTED-v4-c3-L.dat
db7a2e3bdc07c9949d355605fc5e6cd3 FT_RWNOTROUTED-v5-c0-B.dat
8e3923e789ca8f5f79dd6007636f768b FT_RWNOTROUTED-v5-c0-L.dat
3606a0be1d301105ee5297ba2442b354 FT_RWNOTROUTED-v5-c1-B.dat
3bf2c6d7a00b9f0612b2e883838a6541 FT_RWNOTROUTED-v5-c1-L.dat
efe925e4c6cc825ad1a7e98ef08513f7 FT_RWNOTROUTED-v5-c2-B.dat
e4d58ccffeb9581cd239cc9e78d7ff42 FT_RWNOTROUTED-v5-c2-L.dat
6741ff19ba194291127b9156545fc8f0 FT_RWNOTROUTED-v5-c3-B.dat
09602931164e2edd4c1ccb443857ab79 FT_RWNOTROUTED-v5-c3-L.dat
fc336be830bfa81c656edf2a8ebd09a1 FT_RWROUTED-v1-c0-B.dat
1cc6994a16cd8bdd18154ec5b2399b6c FT_RWROUTED-v1-c0-L.dat
5b39a0cb4febc3670262196bf7d2c9b8 FT_RWROUTED-v1-c1-B.dat
f598c9ee471123902f9dfb42c6259f3c FT_RWROUTED-v1-c1-L.dat
d72e8cfcfb2b03891ce21177e45b6596 FT_RWROUTED-v1-c2-B.dat
5004b9086b19a6cd095ab4ea50fe6f05 FT_RWROUTED-v1-c2-L.dat
a0de2d452e76e8ab6faee4fa065e87ef FT_RWROUTED-v1-c3-B.dat
c60951e6129ff9c92885565ca426502f FT_RWROUTED-v1-c3-L.dat
7cb38b980c9c9ce3afb5c5eed49c7024 FT_RWROUTED-v2-c0-B.dat
7294a09625f8be066d9eaf2e5118b1b9 FT_RWROUTED-v2-c0-L.dat
60c6540acce95eb3ce0c10e78d0dc988 FT_RWROUTED-v2-c1-B.dat
9f41df323a6cc4772060b3b570a76f18 FT_RWROUTED-v2-c1-L.dat
e802846546da9f29cc811cbf7f877ae1 FT_RWROUTED-v2-c2-B.dat
1db9894634de170f3d3a2d80d4ccb779 FT_RWROUTED-v2-c2-L.dat
042944cb1c9e2894b680d848f880e989 FT_RWROUTED-v2-c3-B.dat
f2c7e938b9da8c6a5ef276d88a90e60d FT_RWROUTED-v2-c3-L.dat
233f8d39cb92b72712812ac99f1a0ee8 FT_RWROUTED-v3-c0-B.dat
2eedd0cdd1dc5f56c27f5aaee1239c69 FT_RWROUTED-v3-c0-L.dat
847b9104bbdc2f56d92c2b4f5039582c FT_RWROUTED-v3-c1-B.dat
6b9e2cab2c37362527529ca57d42ddb1 FT_RWROUTED-v3-c1-L.dat
99cb90de217595f02491f69148316ce9 FT_RWROUTED-v3-c2-B.dat
c504bba1e6dfb6183face0f640a3cb60 FT_RWROUTED-v3-c2-L.dat
b5b95e43e8e2e5b635a6395fd4f98513 FT_RWROUTED-v3-c3-B.dat
ea2ce55bb6c5b45ef42d66d05dd31a3a FT_RWROUTED-v3-c3-L.dat
904295035fda840d8e86b860b36af87a FT_RWROUTED-v4-c0-B.dat
ff03b805d535b11c3aab9234c155e632 FT_RWROUTED-v4-c0-L.dat
d8208af3daf40adb2d44743543ab20cb FT_RWROUTED-v4-c1-B.dat
93d62383b1681473dfcd1d08417fda53 FT_RWROUTED-v4-c1-L.dat
4eab1d8adfde492d3f77ad401c349c83 FT_RWROUTED-v4-c2-B.dat
ad41f710c432a60e5515d8040ea08773 FT_RWROUTED-v4-c2-L.dat
42bbbbf4a7ed7b42a52898cfd4b22ff2 FT_RWROUTED-v4-c3-B.dat
c35b121920658e34857ad51fd13d14c3 FT_RWROUTED-v4-c3-L.dat
25c9c58badc79aad6ebcd7bfec3c2807 FT_RWROUTED-v5-c0-B.dat
a3c4fd1346b696466e9a160f07c379b7 FT_RWROUTED-v5-c0-L.dat
4228805d08e1bf4aae84faad84d42cb5 FT_RWROUTED-v5-c1-B.dat
853e37c8b0859cff0a068669bc37c8f4 FT_RWROUTED-v5-c1-L.dat
2ccba690f3daa7251eeec1793ab01429 FT_RWROUTED-v5-c2-B.dat
54e9016de02914a32d05cff17a34c818 FT_RWROUTED-v5-c2-L.dat
035e13e17216c9ebcd3e8786a3914dcf FT_RWROUTED-v5-c3-B.dat
6aa1507ca920d8fdc43a9a9212c45548 FT_RWROUTED-v5-c3-L.dat
eaa9a8df71606869d615091068e3eb4e FT_RWSPLIT-v1-c0-B.dat
b164e6e0f71799680a5a31c92bcaa144 FT_RWSPLIT-v1-c0-L.dat
ff2337ab79e454dbd823a821b234cb73 FT_RWSPLIT-v1-c1-B.dat
34a30f8ca5a70817828ef394702f30d9 FT_RWSPLIT-v1-c1-L.dat
aea9f2725e4d55e431440ec0f1c7a8ef FT_RWSPLIT-v1-c2-B.dat
2796acd3c0bcedf03623bb5573f14800 FT_RWSPLIT-v1-c2-L.dat
f0e6cf7c986dd12c1e9b00d66626bf14 FT_RWSPLIT-v1-c3-B.dat
8e09457d239ee6236e6b992cb8923bdd FT_RWSPLIT-v1-c3-L.dat
190d042e83ff76fbfa2a9a179053f5a4 FT_RWSPLIT-v2-c0-B.dat
7f1ace8529ff7189572a16fa7b321cc5 FT_RWSPLIT-v2-c0-L.dat
1a3251277dd7ebe3caa4d46d108e73a4 FT_RWSPLIT-v2-c1-B.dat
71ea33173c433ec324cd4098efe5c865 FT_RWSPLIT-v2-c1-L.dat
55cb95971fe7dbcd0d3b7262e177c241 FT_RWSPLIT-v2-c2-B.dat
946e18aa653e7ec21998d4da16c78efb FT_RWSPLIT-v2-c2-L.dat
14b67928c1c07e5cc27a55872d27f992 FT_RWSPLIT-v2-c3-B.dat
3a3a39e4940aba234bc321d4f571e378 FT_RWSPLIT-v2-c3-L.dat
2d7eb223c9a7ff15395b6cd34bdc9a07 FT_RWSPLIT-v3-c0-B.dat
90abed88a3d368e9ec643d477d5a6804 FT_RWSPLIT-v3-c0-L.dat
377c84d6cacc51d25959f421ef1cd309 FT_RWSPLIT-v3-c1-B.dat
a4153efac1725a9fc44ea544b8817eb0 FT_RWSPLIT-v3-c1-L.dat
4fb345f699b2a6c0288dbf96e57e2edb FT_RWSPLIT-v3-c2-B.dat
0b8f960e7857c44e21aeccd445440321 FT_RWSPLIT-v3-c2-L.dat
31a715ffd551376bb7e23942062def49 FT_RWSPLIT-v3-c3-B.dat
81e874ca5d7f8369e5e85b4c83651977 FT_RWSPLIT-v3-c3-L.dat
7d4c3b85d50e8478f521033edd9a6149 FT_RWSPLIT-v4-c0-B.dat
a5f1f02894f5ee5a1258e65bfa25b348 FT_RWSPLIT-v4-c0-L.dat
056ef706e1cc999b7c8bcdf1a5e7efc2 FT_RWSPLIT-v4-c1-B.dat
976079331127ec8a4b08aa9c6d5b1b96 FT_RWSPLIT-v4-c1-L.dat
566347bb4e0eaefda1c7c6485699a312 FT_RWSPLIT-v4-c2-B.dat
a4d72802084c5afba1470bbae9b34ed4 FT_RWSPLIT-v4-c2-L.dat
43b2960b161e9ae8da57fb0896b4f94e FT_RWSPLIT-v4-c3-B.dat
b16f86220c07710dcbe69b7e19249f39 FT_RWSPLIT-v4-c3-L.dat
ce7c5157cdfdcbdf0cff9e099b9d438a FT_RWSPLIT-v5-c0-B.dat
8d39e7086fdb0c274c3481e1a1f32374 FT_RWSPLIT-v5-c0-L.dat
294b395211bf5d0aa53a02125c9c7cc0 FT_RWSPLIT-v5-c1-B.dat
e4db1940d014c48ed264f39757aad985 FT_RWSPLIT-v5-c1-L.dat
e2c6c4b76cc9444aa09d25f95ddb8fad FT_RWSPLIT-v5-c2-B.dat
ffe8abd98281c04630b8af904a3ea235 FT_RWSPLIT-v5-c2-L.dat
ad6d794c90992ac3eb0685e122dcec73 FT_RWSPLIT-v5-c3-B.dat
41e627c8203bff9d14952e4a8eae8e66 FT_RWSPLIT-v5-c3-L.dat
26f8810cc98adc5f5e1f64146b23e880 FT_RWWWW-v1-c0-B.dat
0af34510e05ca302ab1359850d9662ea FT_RWWWW-v1-c0-L.dat
1aa9d790131bed8a1c69e232327c174b FT_RWWWW-v1-c1-B.dat
9511bf03bd005650a785d946dd75bd5f FT_RWWWW-v1-c1-L.dat
83b0799725651a553ec1d1245f706e4d FT_RWWWW-v1-c2-B.dat
0e3f03851539cfff651fb0195f798d03 FT_RWWWW-v1-c2-L.dat
9aa227dde9cd8e760728210a6ab6b84d FT_RWWWW-v1-c3-B.dat
68a345c4c9d02293bcb3fd59d0e095c7 FT_RWWWW-v1-c3-L.dat
bdc49cfe36db23d23609dd57f6797326 FT_RWWWW-v2-c0-B.dat
464c5782dba54fe4f40b89c7cd585244 FT_RWWWW-v2-c0-L.dat
b9dc9a940ccc90744a12c8ec3c04194a FT_RWWWW-v2-c1-B.dat
cf15b09d84f0f17a66e6b553dcb9e7a4 FT_RWWWW-v2-c1-L.dat
d0b52844842d47384e037b568eb846f4 FT_RWWWW-v2-c2-B.dat
a33cadd6741917d611bfe812972ac0f5 FT_RWWWW-v2-c2-L.dat
680e13313767e9477199bc49be470e18 FT_RWWWW-v2-c3-B.dat
4959860d3e687c7b29e89fd4ea91a0e2 FT_RWWWW-v2-c3-L.dat
c799684e667621c733e3b2c3fb0b3b26 FT_RWWWW-v3-c0-B.dat
aa67e5c73a3c702d240de8dc6f327df6 FT_RWWWW-v3-c0-L.dat
5ee3c862a8d7981de0ceb026431bfcba FT_RWWWW-v3-c1-B.dat
a94f539cd4641afab1a73a02ccefc8b4 FT_RWWWW-v3-c1-L.dat
f65062b3847d4c884ee4cabf0047ec54 FT_RWWWW-v3-c2-B.dat
c99a5f63780335bf4dc1ee26869ae545 FT_RWWWW-v3-c2-L.dat
752128d0fc78d1f24443fb437aec4e20 FT_RWWWW-v3-c3-B.dat
c5260c8e538c41e15eb3ce5bbac6a785 FT_RWWWW-v3-c3-L.dat
c4a630ea766f3207c30770920987a75f FT_RWWWW-v4-c0-B.dat
30cd4ea884fb571cf47355802991df83 FT_RWWWW-v4-c0-L.dat
6a718fc229e5987c18fd9892c75b0467 FT_RWWWW-v4-c1-B.dat
32d4286bbf9a67fbdfa20cd0a720b4b6 FT_RWWWW-v4-c1-L.dat
e99dab694f70cd0dc8769eb012e7e8d5 FT_RWWWW-v4-c2-B.dat
0d8f9542aff7a1174dc908c13885448c FT_RWWWW-v4-c2-L.dat
72769c606462a8afcc15b4af2b08baf2 FT_RWWWW-v4-c3-B.dat
0e4fad0e95a91d9af2c0c03ead30d3ff FT_RWWWW-v4-c3-L.dat
728847040532a4acc9389307bb8c9198 FT_RWWWW-v5-c0-B.dat
705fc84769afdb394121e1e456dea816 FT_RWWWW-v5-c0-L.dat
26c81286cb446e7853bceade61817ffb FT_RWWWW-v5-c1-B.dat
cb41e853f9b03a4c2f078a0452a2c8d9 FT_RWWWW-v5-c1-L.dat
33c9c3ef2843b8f60b22cea6a9998df7 FT_RWWWW-v5-c2-B.dat
3150db69cb22a179996363fdedeb3fad FT_RWWWW-v5-c2-L.dat
65c16fa09af771bf644991e210c6dc14 FT_RWWWW-v5-c3-B.dat
421cf74a0497148a95f7ee8d963e5aed FT_RWWWW-v5-c3-L.dat

# Snappy-1.2.1 generates different checksums
29acfd756ac6cfec9dd27ff0f099f3c7 FT_RWIPV6-v1-c3-B.dat
c36599de4f2312df6fe54801f8f144e4 FT_RWIPV6-v1-c3-L.dat
9f0e5c678a9d6be266cb28c39369aed0 FT_RWIPV6-v2-c3-B.dat
50975dacaa8fd5883a09664c21043b5a FT_RWIPV6-v2-c3-L.dat
f1cf2703bf12ab3d42e884122f273859 FT_RWIPV6-v3-c3-B.dat
fbc59debc8e48e9206a9138ec3897576 FT_RWIPV6-v3-c3-L.dat
07612c0e213c11aa47d16651fcac9a87 FT_RWIPV6ROUTING-v1-c3-B.dat
1886274bdfac815b8d586c39177e111a FT_RWIPV6ROUTING-v1-c3-L.dat
5ea4038cc9a99892fbfed63f770b3809 FT_RWIPV6ROUTING-v2-c3-B.dat
384a4fe1dd3e2f4a32f39a7f244dcd6b FT_RWIPV6ROUTING-v2-c3-L.dat
be21c9657e5f18acd717b1680044c081 FT_RWIPV6ROUTING-v3-c3-B.dat
d02d7a13f6425d6505313678c2240509 FT_RWIPV6ROUTING-v3-c3-L.dat
cc5131686c4546da799da93beb563166 FT_RWIPV6ROUTING-v4-c3-B.dat
07229f277dcfd83e223d86cbe6a7017b FT_RWIPV6ROUTING-v4-c3-L.dat

# Snappy-1.2.1 on Apple Silicon
06f9f6aa8c4309602f9fd8f5288509d1 FT_RWIPV6-v1-c3-B.dat
35c7484a5419e5f8e7ff96d83dea35fd FT_RWIPV6-v1-c3-L.dat
bc739ac9a662597becb93262f5176d5b FT_RWIPV6-v2-c3-B.dat
ac18745dd98e29d0b03f4988e1dde84b FT_RWIPV6-v2-c3-L.dat
70dc6db608404378ee8e7d9baa0e78e7 FT_RWIPV6-v3-c3-B.dat
0e21804f2104e72523601278adb56747 FT_RWIPV6-v3-c3-L.dat
06abecbec7c417045bb13e8956f54387 FT_RWIPV6ROUTING-v1-c3-B.dat
45b93b7bfd21fef3088ff0d7ea1674e4 FT_RWIPV6ROUTING-v1-c3-L.dat
e64ca8afbc6150ef22bfb9ba8aa1f875 FT_RWIPV6ROUTING-v2-c3-B.dat
a24ad2beea875c6b45a28df2d938fe07 FT_RWIPV6ROUTING-v2-c3-L.dat
eaf606573a86046ed5157df2d570d02c FT_RWIPV6ROUTING-v3-c3-B.dat
fe85b8299b09f043d52eaaeefc9f5648 FT_RWIPV6ROUTING-v3-c3-L.dat
d54917efed1f68c5825248f2e3f0e7ee FT_RWIPV6ROUTING-v4-c3-B.dat
55b938feb7a118f67e907fd45329a4ae FT_RWIPV6ROUTING-v4-c3-L.dat

#########################################################################
#
#   These are the expected MD5 values when reading the data with rwcat
#
#   Values should not depend on compression or byte-order
#
#   File formats capable of holding only IPv4 addresses contain only the
#   file header.  On readback, only the column headers appear as output.
#
5731e2c44407288dfc97c60f877bc3ed FT_FLOWCAP-v2
5731e2c44407288dfc97c60f877bc3ed FT_FLOWCAP-v3
5731e2c44407288dfc97c60f877bc3ed FT_FLOWCAP-v4
5731e2c44407288dfc97c60f877bc3ed FT_FLOWCAP-v5
5731e2c44407288dfc97c60f877bc3ed FT_FLOWCAP-v6
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGMENTED-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGMENTED-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGMENTED-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGMENTED-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGMENTED-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGMENTED-v6
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGROUTING-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGROUTING-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGROUTING-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGROUTING-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGROUTING-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGROUTING-v6
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGSNMPOUT-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGSNMPOUT-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGSNMPOUT-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGSNMPOUT-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGSNMPOUT-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGWEB-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGWEB-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGWEB-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGWEB-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGWEB-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWAUGWEB-v6
5731e2c44407288dfc97c60f877bc3ed FT_RWFILTER-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWFILTER-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWFILTER-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWFILTER-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWFILTER-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWGENERIC-v0
5731e2c44407288dfc97c60f877bc3ed FT_RWGENERIC-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWGENERIC-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWGENERIC-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWGENERIC-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWGENERIC-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWGENERIC-v6

70bf9717f7b4b68de397a5d17f4f3e27 FT_RWIPV6-v1
179646d912cd948bc5fee4f86e384db0 FT_RWIPV6-v2
179646d912cd948bc5fee4f86e384db0 FT_RWIPV6-v3
f34bb0d3be683a6c917e8816c6806ea6 FT_RWIPV6ROUTING-v1
fd45a2b77e3f63234f7ce2e1814bedc0 FT_RWIPV6ROUTING-v2
f34bb0d3be683a6c917e8816c6806ea6 FT_RWIPV6ROUTING-v3
f34bb0d3be683a6c917e8816c6806ea6 FT_RWIPV6ROUTING-v4

5731e2c44407288dfc97c60f877bc3ed FT_RWNOTROUTED-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWNOTROUTED-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWNOTROUTED-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWNOTROUTED-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWNOTROUTED-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWROUTED-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWROUTED-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWROUTED-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWROUTED-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWROUTED-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWSPLIT-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWSPLIT-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWSPLIT-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWSPLIT-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWSPLIT-v5
5731e2c44407288dfc97c60f877bc3ed FT_RWWWW-v1
5731e2c44407288dfc97c60f877bc3ed FT_RWWWW-v2
5731e2c44407288dfc97c60f877bc3ed FT_RWWWW-v3
5731e2c44407288dfc97c60f877bc3ed FT_RWWWW-v4
5731e2c44407288dfc97c60f877bc3ed FT_RWWWW-v5
