#! /usr/bin/perl -w
# MD5: multiple
# TEST: ../rwfilter/rwfilter --stime=2009/02/13:20:00-2009/02/13:20 --sensor=S2 --proto=6 --aport=80,8080,443 --pass=stdout ../../tests/data.rwf | ./rwallformats --no-invocation --basename=/tmp/sk-teststmp && md5 /tmp/sk-teststmp*

use strict;
use SiLKTests;
use File::Find;

# name of this script
my $NAME = $0;
$NAME =~ s,.*/,,;

my $rwallformats = check_silk_app('rwallformats');
my $rwcut = check_silk_app('rwcut');
my $rwtuc = check_silk_app('rwtuc');

# Fields to test
my $tuc_fields = 'in,out';
my $cut_fields = $tuc_fields;

my $base_name = 'sk-teststmp';

# create our tempdir
my $tmpdir = make_tempdir();

# create subdirs for writing and reading with rwcut
my $write_dir = "$tmpdir/write";
my $rwcut_dir = "$tmpdir/rwcut";
for my $d ($write_dir, $rwcut_dir) {
    mkdir $d
        or die "$NAME: Cannot mkdir '$d': $!";
}

my $tmpfile = make_tempname('rwtuc-input.txt');
my $tmpsilk = make_tempname('rwtuc-output.rw');
open my $tmpfp, '>', $tmpfile
    or die "$NAME: Cannot open '$tmpfile': $!";
print $tmpfp <<'RWTUC_INPUT';
          in|         out|ignore
         192|           0|
         192|          22|
         192|       55555|
         192|     7777777|
         192|121212121212| # line rejected by rwtuc
           0|         168|
          22|         168|
       55555|         168|
     7777777|         168|
121212121212|         168| # line rejected by rwtuc
RWTUC_INPUT
close $tmpfp
    or die "$NAME: Unable to write '$tmpfile': $!";

my $cmd = ("$rwtuc --fields=$tuc_fields,ignore"
           ." --protocol=6 --dport=8080 --application=80"
           ." --bytes=2560 --packets=20"
           ." --output-path=$tmpsilk --verbose -- $tmpfile");
if (!check_exit_status($cmd)) {
    exit 1;
}
$cmd = ("$rwallformats --no-invocation --basename='$write_dir/$base_name'"
        ." $tmpsilk");
if (!check_exit_status($cmd)) {
    exit 1;
}

# get list of expected MD5s for each file from the end of this file;
# these are the MD5 from running rwcut on the file
my %readback;
while (<DATA>) {
    next unless /\w/;
    next if /^\#/;

    my ($expect, $tail_name) = split " ";
    push @{$readback{$tail_name}}, $expect;
}

# hash to store names of files that had an issue; value is an array of
# possible isues: 'missing', 'read-mismatch', 'unknown-file',
# 'unknown-readback'
#
my %mismatch;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $write_dir);

# print results: Files are sorted by the reverse of the name so all
# differences across a wide range of file formats and versions are
# seen first.
#
my @badfiles = sort { reverse($a) cmp reverse($b) } keys %mismatch;
if (@badfiles) {

    my %issues;

    my $msg = sprintf("%s: Found %u file%s with problems:",
                      $NAME, scalar(@badfiles),
                      ((scalar(@badfiles) == 1) ? "" : "s"));
    print STDERR $msg, "\n";

    for my $file (@badfiles) {
        for my $i (@{$mismatch{$file}}) {
            ++$issues{$i};
        }
        print STDERR join(" ", "$NAME:", "$file -", @{$mismatch{$file}}), "\n";
    }

    my $issuemsg = join ", ", map { "$_: ".$issues{$_} } keys %issues;

    die($msg, " ", $issuemsg, "\n");
}

# successful!
exit 0;


# This function is called by File::Find::find.  The full path to the
# file is in the $_ variable.
#
# The function checks for the file in the %checksums hash.  If not
# present, an entry is added to %mismatch for an unknown file and the
# function returns.
#
# Files in %checksums hash have their values removed from the hash so
# mising files may be reported at shutdown.
#
# The function computes the MD5 of the file and compares it to the
# expected value; if the values do not match, the function puts the
# file name into the %mismatch hash with a file-mismatch.
#
# The function runs rwcut on the file and computes the MD5 of the
# result.  If that does not match the expected value, the function
# puts the file name into the %mismatch hash with a read-mismatch.
#
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;

    my $path = $_;

    # file name without $write_dir
    my $file = $_;
    $file =~ s,^\Q$write_dir/\E,,;

    # set $tail_name to be the varying part of the filename; that is,
    # remove the directory and base_name
    my $tail_name = $_;
    $tail_name =~ s,^$write_dir/$base_name-,,;

    # set $read_name to be $tail_name truncated at the version (that
    # is, minus the compresssion and byte-order)
    my $read_name = $tail_name;
    $read_name =~ s,^(\w+-v\d+)-.+,$1,;

    unless (exists $readback{$read_name}) {
        push @{$mismatch{$file}}, 'unknown-readback';
        return;
    }

    # check reading
    my $read_md5;
    my $read_cmd = ("$rwcut --fields=$cut_fields --delimited=,"
                    ." --output-path=$rwcut_dir/$file $path");
    check_exit_status($read_cmd, 1);
    compute_md5_file(\$read_md5, "$rwcut_dir/$file");
    if (! grep {$_ eq $read_md5} @{$readback{$read_name}}) {
        print STDERR "$NAME: $read_name: read checksum mismatch [$read_md5]",
            " (expected ", join(",", @{$readback{$read_name}}), ")\n";
        push @{$mismatch{$file}}, 'read-mismatch';
    }
}



__DATA__
#########################################################################
#
#   These are the expected MD5 values when reading the data with rwcut
#
#   Values should not depend on compression or byte-order
#

##  These formats do not store SNMP interface values.
#
#|in,out
#|0,0
#|0,0
#|0,0
#|0,0
#|0,0
#|0,0
#|0,0
#|0,0
#
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGMENTED-v1
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGMENTED-v2
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGMENTED-v3
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGMENTED-v4
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGMENTED-v5
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGMENTED-v6
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGWEB-v1
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGWEB-v2
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGWEB-v3
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGWEB-v4
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGWEB-v5
10ac805809e8a44ec0a15e799d35d2bb FT_RWAUGWEB-v6
10ac805809e8a44ec0a15e799d35d2bb FT_RWIPV6-v1
10ac805809e8a44ec0a15e799d35d2bb FT_RWIPV6-v2
10ac805809e8a44ec0a15e799d35d2bb FT_RWIPV6-v3
10ac805809e8a44ec0a15e799d35d2bb FT_RWSPLIT-v1
10ac805809e8a44ec0a15e799d35d2bb FT_RWSPLIT-v2
10ac805809e8a44ec0a15e799d35d2bb FT_RWSPLIT-v3
10ac805809e8a44ec0a15e799d35d2bb FT_RWSPLIT-v4
10ac805809e8a44ec0a15e799d35d2bb FT_RWSPLIT-v5
10ac805809e8a44ec0a15e799d35d2bb FT_RWWWW-v1
10ac805809e8a44ec0a15e799d35d2bb FT_RWWWW-v2
10ac805809e8a44ec0a15e799d35d2bb FT_RWWWW-v3
10ac805809e8a44ec0a15e799d35d2bb FT_RWWWW-v4
10ac805809e8a44ec0a15e799d35d2bb FT_RWWWW-v5


##  These formats store 8-bit SNMP interfaces and use the lower 8-bits
##  on values that overflow
#
#|in,out
#|192,0
#|192,22
#|192,3
#|192,241
#|0,168
#|22,168
#|3,168
#|241,168
#
e0ab5de52bf864ea94bbff496a0912da FT_FLOWCAP-v2
e0ab5de52bf864ea94bbff496a0912da FT_FLOWCAP-v3
e0ab5de52bf864ea94bbff496a0912da FT_FLOWCAP-v4


##  These formats store 16-bit SNMP interfaces and use the lower
##  16-bits on values that overflow
#
#|in,out
#|192,0
#|192,22
#|192,55555
#|192,44529
#|0,168
#|22,168
#|55555,168
#|44529,168
#
63d93283362710b90d6679e64a551de1 FT_FLOWCAP-v5
63d93283362710b90d6679e64a551de1 FT_FLOWCAP-v6


##  These formats store 8-bit SNMP interfaces and reject records that
##  overflow
#
#|in,out
#|192,0
#|192,22
#|0,168
#|22,168
#
f5871f89b925450f0040658b7336ac8d FT_RWFILTER-v1
f5871f89b925450f0040658b7336ac8d FT_RWFILTER-v2
f5871f89b925450f0040658b7336ac8d FT_RWGENERIC-v0
f5871f89b925450f0040658b7336ac8d FT_RWGENERIC-v1
f5871f89b925450f0040658b7336ac8d FT_RWROUTED-v1
f5871f89b925450f0040658b7336ac8d FT_RWROUTED-v2


##  These formats store only the ingress SNMP interface value in
##  8-bits and reject records where the ingress value overflows
#
#|in,out
#|192,0
#|192,0
#|192,0
#|192,0
#|0,0
#|22,0
#
2a444c00bcf014341ab34bc80dc56575 FT_RWNOTROUTED-v1
2a444c00bcf014341ab34bc80dc56575 FT_RWNOTROUTED-v2


##  These formats store only the ingress SNMP interface value in
##  16-bits and reject records where the ingress value overflows
#
#|in,out
#|192,0
#|192,0
#|192,0
#|192,0
#|0,0
#|22,0
#|55555,0
#
e189693902a82771c61d0cc3ee072d85 FT_RWNOTROUTED-v3
e189693902a82771c61d0cc3ee072d85 FT_RWNOTROUTED-v4
e189693902a82771c61d0cc3ee072d85 FT_RWNOTROUTED-v5


##  These formats store only the egress SNMP interface value in
##  16-bits and reject records where the egress value overflows
#
#|in,out
#|0,0
#|0,22
#|0,55555
#|0,168
#|0,168
#|0,168
#|0,168
#
9d11c52f92e297a4a5599e0706d3e1ee FT_RWAUGSNMPOUT-v1
9d11c52f92e297a4a5599e0706d3e1ee FT_RWAUGSNMPOUT-v2
9d11c52f92e297a4a5599e0706d3e1ee FT_RWAUGSNMPOUT-v3
9d11c52f92e297a4a5599e0706d3e1ee FT_RWAUGSNMPOUT-v4
9d11c52f92e297a4a5599e0706d3e1ee FT_RWAUGSNMPOUT-v5

##  These formats store 16-bit SNMP interfaces and reject records that
##  overflow
#
#|in,out
#|192,0
#|192,22
#|192,55555
#|0,168
#|22,168
#|55555,168
#
d5dd5c16c124f7f482c9172854e1f0de FT_RWAUGROUTING-v1
d5dd5c16c124f7f482c9172854e1f0de FT_RWAUGROUTING-v2
d5dd5c16c124f7f482c9172854e1f0de FT_RWAUGROUTING-v3
d5dd5c16c124f7f482c9172854e1f0de FT_RWAUGROUTING-v4
d5dd5c16c124f7f482c9172854e1f0de FT_RWAUGROUTING-v5
d5dd5c16c124f7f482c9172854e1f0de FT_RWFILTER-v3
d5dd5c16c124f7f482c9172854e1f0de FT_RWFILTER-v4
d5dd5c16c124f7f482c9172854e1f0de FT_RWFILTER-v5
d5dd5c16c124f7f482c9172854e1f0de FT_RWGENERIC-v2
d5dd5c16c124f7f482c9172854e1f0de FT_RWGENERIC-v3
d5dd5c16c124f7f482c9172854e1f0de FT_RWGENERIC-v4
d5dd5c16c124f7f482c9172854e1f0de FT_RWGENERIC-v5
d5dd5c16c124f7f482c9172854e1f0de FT_RWIPV6ROUTING-v1
d5dd5c16c124f7f482c9172854e1f0de FT_RWIPV6ROUTING-v2
d5dd5c16c124f7f482c9172854e1f0de FT_RWROUTED-v3
d5dd5c16c124f7f482c9172854e1f0de FT_RWROUTED-v4
d5dd5c16c124f7f482c9172854e1f0de FT_RWROUTED-v5


##  These formats support 32-bit SNMP interface values
#
#|in,out
#|192,0
#|192,22
#|192,55555
#|192,7777777
#|0,168
#|22,168
#|55555,168
#|7777777,168
#
172e35d14d3c0d9e2688c7ad7984b8ec FT_RWAUGROUTING-v6
172e35d14d3c0d9e2688c7ad7984b8ec FT_RWGENERIC-v6
172e35d14d3c0d9e2688c7ad7984b8ec FT_RWIPV6ROUTING-v3
172e35d14d3c0d9e2688c7ad7984b8ec FT_RWIPV6ROUTING-v4
