/*
** Copyright (C) 2001-2024 by Carnegie Mellon University.
**
** @OPENSOURCE_LICENSE_START@
**
** SiLK 3.23
**
** Copyright 2024 Carnegie Mellon University.
**
** NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
** INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
** UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
** IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
** FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
** OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
** MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
** TRADEMARK, OR COPYRIGHT INFRINGEMENT.
**
** Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
** contact permission@sei.cmu.edu for full terms.
**
** [DISTRIBUTION STATEMENT A] This material has been approved for public
** release and unlimited distribution.  Please see Copyright notice for
** non-US Government use and distribution.
**
** This Software includes and/or makes use of Third-Party Software each
** subject to its own license.
**
** DM24-1064
**
** @OPENSOURCE_LICENSE_END@
*/

/*
** rwgenericio.c
**
** Suresh L Konda
**      routines to do io stuff with generic records.
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwgenericio.c afa589dbc78d 2024-09-19 17:35:03Z mthomas $");

/* #define RWPACK_BYTES_PACKETS          1 */
/* #define RWPACK_FLAGS_TIMES_VOLUMES    1 */
#define RWPACK_INTERFACES_16BIT       1
#define RWPACK_PKTS_BYTES_32BIT       1
/* #define RWPACK_PROTO_FLAGS            1 */
/* #define RWPACK_SBB_PEF                1 */
/* #define RWPACK_STATE_FLAGS_TIMES_NANO 1 */
/* #define RWPACK_TIME_BYTES_PKTS_FLAGS  1 */
/* #define RWPACK_TIMES_FLAGS_PROTO      1 */
#define RWPACK_TIMES_MILLI            1
#include "rwpack.c"
#include "skstream_priv.h"


/* Version to use when SK_RECORD_VERSION_ANY is specified */
#define DEFAULT_RECORD_VERSION 6


/* ********************************************************************* */

/*
**  RWGENERIC VERSION 6
**
**    int64_t       sTime;           //  0- 7  Flow start time as nanoseconds
**                                   //        since UNIX epoch
**
**    int64_t       eTime;           //  8-15  Flow end time as nanoseconds
**                                   //        since UNIX epoch
**
**    uint16_t      sPort;           // 16-17  Source port
**    uint16_t      dPort;           // 18-19  Destination port
**
**    uint8_t       proto;           // 20     IP protocol
**    uint8_t       flow_type;       // 21     Class & Type info
**    uint16_t      sID;             // 22-23  Sensor ID
**
**    uint8_t       flags;           // 24     OR of all flags (Netflow flags)
**    uint8_t       init_flags;      // 25     TCP flags in first packet
**                                   //        or blank for "legacy" data
**    uint8_t       rest_flags;      // 26     TCP flags on non-initial packet
**                                   //        or blank for "legacy" data
**    uint8_t       tcp_state;       // 27     TCP state machine info (below)
**
**    uint16_t      application;     // 28-29  Indication of type of traffic
**    uint16_t      memo;            // 30-31  Application specific field
**
**    uint32_t      input;           // 32-35  Router incoming SNMP interface
**    uint32_t      output;          // 36-39  Router outgoing SNMP interface
**
**    uint64_t      pkts;            // 40-47  Count of packets
**    uint64_t      bytes;           // 48-55  Count of bytes
**
**    uint32_t      sIP;             // 56-59  Source IP
**    uint32_t      dIP;             // 60-63  Destination IP
**    uint32_t      nhIP;            // 64-67  Router Next Hop IP
**
**
**  68 bytes on disk.
*/

#define RECLEN_RWGENERIC_V6 68


/*
 *    Byte swap the RWGENERIC v6 record 'ar' in place.
 */
#define genericioRecordSwap_V6(ar)                              \
    {                                                           \
        SWAP_DATA64((ar) +  0);   /* sTime */                   \
        SWAP_DATA64((ar) +  8);   /* eTime */                   \
        SWAP_DATA16((ar) + 16);   /* sPort */                   \
        SWAP_DATA16((ar) + 18);   /* dPort */                   \
        /* Two single bytes: (20)proto, (21)flow_type */        \
        SWAP_DATA16((ar) + 22);   /* sID */                     \
        /* Four single bytes: (24)flags, (25)init_flags,        \
         *                    (26)rest_flags, (27)tcp_state */  \
        SWAP_DATA16((ar) + 28);   /* application */             \
        SWAP_DATA16((ar) + 30);   /* memo */                    \
        SWAP_DATA32((ar) + 32);   /* input */                   \
        SWAP_DATA32((ar) + 36);   /* output */                  \
        SWAP_DATA64((ar) + 40);   /* pkts */                    \
        SWAP_DATA64((ar) + 48);   /* bytes */                   \
        SWAP_DATA32((ar) + 56);   /* sIP */                     \
        SWAP_DATA32((ar) + 60);   /* dIP */                     \
        SWAP_DATA32((ar) + 64);   /* nhIP */                    \
    }


/*
 *  Unpack the array of bytes 'ar' into a record 'rwrec'
 */
static int
genericioRecordUnpack_V6(
    skstream_t         *stream,
    rwGenericRec_V6    *rwrec,
    uint8_t            *ar)
{
    uint64_t tmp64;

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V6(ar);
    }

    /* Start time */
    memcpy(&tmp64, &ar[ 0], sizeof(tmp64));
    rwRecSetStartTime(rwrec, sktimeCreateFromNano(0, tmp64));

    /* End time */
    memcpy(&tmp64, &ar[ 8], sizeof(tmp64));
    rwRecSetEndTime(rwrec, sktimeCreateFromNano(0, tmp64));

    rwRecMemSetSPort(rwrec, &ar[16]);
    rwRecMemSetDPort(rwrec, &ar[18]);
    rwRecMemSetProto(rwrec, &ar[20]);
    rwRecMemSetFlowType(rwrec, &ar[21]);
    rwRecMemSetSensor(rwrec, &ar[22]);
    rwRecMemSetFlags(rwrec, &ar[24]);
    rwRecMemSetInitFlags(rwrec, &ar[25]);
    rwRecMemSetRestFlags(rwrec, &ar[26]);
    rwRecMemSetTcpState(rwrec, &ar[27]);
    rwRecMemSetApplication(rwrec, &ar[28]);
    rwRecMemSetMemo(rwrec, &ar[30]);
    rwRecMemSetInput(rwrec, &ar[32]);
    rwRecMemSetOutput(rwrec, &ar[36]);
    rwRecMemSetPkts(rwrec, &ar[40]);
    rwRecMemSetBytes(rwrec, &ar[48]);
    rwRecMemSetSIPv4(rwrec, &ar[56]);
    rwRecMemSetDIPv4(rwrec, &ar[60]);
    rwRecMemSetNhIPv4(rwrec, &ar[64]);

    /*
     * No need for this; file format is post SiLK-3.6.0
     * RWREC_MAYBE_CLEAR_TCPSTATE_EXPANDED(rwrec);
     */

    return SKSTREAM_OK;
}


/*
 *  Pack the record 'rwrec' into an array of bytes 'ar'
 */
static int
genericioRecordPack_V6(
    skstream_t             *stream,
    const rwGenericRec_V6  *rwrec,
    uint8_t                *ar)
{
    uint64_t u64;

    /* Start time */
    u64 = sktimeGetEpochNano(rwRecGetStartTime(rwrec));
    memcpy(&ar[ 0], &u64, sizeof(u64));

    /* End time */
    u64 = sktimeGetEpochNano(sktimeAddInterval(rwRecGetStartTime(rwrec),
                                               rwRecGetElapsed(rwrec)));
    memcpy(&ar[ 8], &u64, sizeof(u64));

    rwRecMemGetSPort(rwrec, &ar[16]);
    rwRecMemGetDPort(rwrec, &ar[18]);
    rwRecMemGetProto(rwrec, &ar[20]);
    rwRecMemGetFlowType(rwrec, &ar[21]);
    rwRecMemGetSensor(rwrec, &ar[22]);
    rwRecMemGetFlags(rwrec, &ar[24]);
    rwRecMemGetInitFlags(rwrec, &ar[25]);
    rwRecMemGetRestFlags(rwrec, &ar[26]);
    rwRecMemGetTcpState(rwrec, &ar[27]);
    rwRecMemGetApplication(rwrec, &ar[28]);
    rwRecMemGetMemo(rwrec, &ar[30]);
    rwRecMemGetInput(rwrec, &ar[32]);
    rwRecMemGetOutput(rwrec, &ar[36]);
    rwRecMemGetPkts(rwrec, &ar[40]);
    rwRecMemGetBytes(rwrec, &ar[48]);
    rwRecMemGetSIPv4(rwrec, &ar[56]);
    rwRecMemGetDIPv4(rwrec, &ar[60]);
    rwRecMemGetNhIPv4(rwrec, &ar[64]);

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V6(ar);
    }

    return SKSTREAM_OK;
}


/* ********************************************************************* */

/*
**  RWGENERIC VERSION 5
**
**    int64_t       sTime;           //  0- 7  Flow start time as milliseconds
**                                   //        since UNIX epoch
**
**    uint32_t      elapsed;         //  8-11  Duration of flow in milliseconds
**                                   //        (Allows for a 49 day flow)
**
**    uint16_t      sPort;           // 12-13  Source port
**    uint16_t      dPort;           // 14-15  Destination port
**
**    uint8_t       proto;           // 16     IP protocol
**    uint8_t       flow_type;       // 17     Class & Type info
**    uint16_t      sID;             // 18-19  Sensor ID
**
**    uint8_t       flags;           // 20     OR of all flags (Netflow flags)
**    uint8_t       init_flags;      // 21     TCP flags in first packet
**                                   //        or blank for "legacy" data
**    uint8_t       rest_flags;      // 22     TCP flags on non-initial packet
**                                   //        or blank for "legacy" data
**    uint8_t       tcp_state;       // 23     TCP state machine info (below)
**
**    uint16_t      application;     // 24-25  Indication of type of traffic
**    uint16_t      memo;            // 26-27  Application specific field
**
**    uint16_t      input;           // 28-29  Router incoming SNMP interface
**    uint16_t      output;          // 30-31  Router outgoing SNMP interface
**
**    uint32_t      pkts;            // 32-35  Count of packets
**    uint32_t      bytes;           // 36-39  Count of bytes
**
**    uint32_t      sIP;             // 40-43  Source IP
**    uint32_t      dIP;             // 44-47  Destination IP
**    uint32_t      nhIP;            // 48-51  Router Next Hop IP
**
**
**  52 bytes on disk.
*/

#define RECLEN_RWGENERIC_V5 52


/*
 *    Byte swap the RWGENERIC v5 record 'ar' in place.
 */
#define genericioRecordSwap_V5(ar)                              \
    {                                                           \
        SWAP_DATA64((ar) +  0);   /* sTime */                   \
        SWAP_DATA32((ar) +  8);   /* elapsed */                 \
        SWAP_DATA16((ar) + 12);   /* sPort */                   \
        SWAP_DATA16((ar) + 14);   /* dPort */                   \
        /* Two single bytes: (16)proto, (17)flow_type */        \
        SWAP_DATA16((ar) + 18);   /* sID */                     \
        /* Four single bytes: (20)flags, (21)init_flags,        \
         *                    (22)rest_flags, (23)tcp_state */  \
        SWAP_DATA16((ar) + 24);   /* application */             \
        SWAP_DATA16((ar) + 26);   /* memo */                    \
        SWAP_DATA16((ar) + 28);   /* input */                   \
        SWAP_DATA16((ar) + 30);   /* output */                  \
        SWAP_DATA32((ar) + 32);   /* pkts */                    \
        SWAP_DATA32((ar) + 36);   /* bytes */                   \
        SWAP_DATA32((ar) + 40);   /* sIP */                     \
        SWAP_DATA32((ar) + 44);   /* dIP */                     \
        SWAP_DATA32((ar) + 48);   /* nhIP */                    \
    }


/*
 *  Unpack the array of bytes 'ar' into a record 'rwrec'
 */
static int
genericioRecordUnpack_V5(
    skstream_t         *stream,
    rwGenericRec_V6    *rwrec,
    uint8_t            *ar)
{
    uint64_t stime;
    uint32_t elapsed;

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V5(ar);
    }

    memcpy(&stime, &ar[0], sizeof(stime));
    rwRecSetStartTime(rwrec, sktimeCreateFromMilli(0, stime));

    memcpy(&elapsed, &ar[8], sizeof(elapsed));
    rwRecSetEndTime(rwrec, sktimeCreateFromMilli(0, stime + elapsed));

    rwRecMemSetSPort(rwrec, &ar[12]);
    rwRecMemSetDPort(rwrec, &ar[14]);
    rwRecMemSetProto(rwrec, &ar[16]);
    rwRecMemSetFlowType(rwrec, &ar[17]);
    rwRecMemSetSensor(rwrec, &ar[18]);
    rwRecMemSetFlags(rwrec, &ar[20]);
    rwRecMemSetInitFlags(rwrec, &ar[21]);
    rwRecMemSetRestFlags(rwrec, &ar[22]);
    rwRecMemSetTcpState(rwrec, &ar[23]);
    rwRecMemSetApplication(rwrec, &ar[24]);
    rwRecMemSetMemo(rwrec, &ar[26]);

    /* input, output */
    rwpackUnpackInterfaces16bit(rwrec, &ar[28], &ar[30]);

    /* packets, bytes */
    rwpackUnpackPacketsBytes32bit(rwrec, &ar[32], &ar[36]);

    rwRecMemSetSIPv4(rwrec, &ar[40]);
    rwRecMemSetDIPv4(rwrec, &ar[44]);
    rwRecMemSetNhIPv4(rwrec, &ar[48]);

    /*
     * No need for this; file format is post SiLK-3.6.0
     * RWREC_MAYBE_CLEAR_TCPSTATE_EXPANDED(rwrec);
     */
    return SKSTREAM_OK;
}


/*
 *  Pack the record 'rwrec' into an array of bytes 'ar'
 */
static int
genericioRecordPack_V5(
    skstream_t             *stream,
    const rwGenericRec_V6  *rwrec,
    uint8_t                *ar)
{
    uint64_t stime;
    uint32_t elapsed;
    ssize_t rv;

    /* Check elapsed */
    if (skIntervalGetMilli(rwRecGetElapsed(rwrec)) > UINT32_MAX) {
        return SKSTREAM_ERR_ELPSD_OVRFLO;
    }

    /* Input, Ouput */
    rv = rwpackPackInterfaces16bit(&ar[28], &ar[30], rwrec);
    if (rv) { return rv; }

    /* Packets, Bytes */
    rv = rwpackPackPacketsBytes32bit(&ar[32], &ar[36], rwrec);
    if (rv) { return rv; }

    stime = sktimeGetEpochMilli(rwRecGetStartTime(rwrec));
    memcpy(&ar[0], &stime, sizeof(stime));

    elapsed = skIntervalGetMilli(rwRecGetElapsed(rwrec));
    memcpy(&ar[8], &elapsed, sizeof(elapsed));

    rwRecMemGetSPort(rwrec, &ar[12]);
    rwRecMemGetDPort(rwrec, &ar[14]);
    rwRecMemGetProto(rwrec, &ar[16]);
    rwRecMemGetFlowType(rwrec, &ar[17]);
    rwRecMemGetSensor(rwrec, &ar[18]);
    rwRecMemGetFlags(rwrec, &ar[20]);
    rwRecMemGetInitFlags(rwrec, &ar[21]);
    rwRecMemGetRestFlags(rwrec, &ar[22]);
    rwRecMemGetTcpState(rwrec, &ar[23]);
    rwRecMemGetApplication(rwrec, &ar[24]);
    rwRecMemGetMemo(rwrec, &ar[26]);

    /* Input &ar[28], Output &ar[30] */
    /* Packets &ar[32], Bytes &ar[36] */

    rwRecMemGetSIPv4(rwrec, &ar[40]);
    rwRecMemGetDIPv4(rwrec, &ar[44]);
    rwRecMemGetNhIPv4(rwrec, &ar[48]);

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V5(ar);
    }

    return SKSTREAM_OK;
}


/* ********************************************************************* */

/*
**  RWGENERIC VERSION 3
**  RWGENERIC VERSION 4
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint32_t      nhIP;            // 12-15  Router Next Hop IP
**    uint16_t      input;           // 16-17  Router incoming SNMP interface
**    uint16_t      output;          // 18-19  Router outgoing SNMP interface
**
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**    uint32_t      elapsed;         // 24-27  Duration of flow
**
**    uint32_t      pkts;            // 28-31  Count of packets
**    uint32_t      bytes;           // 32-35  Count of bytes
**
**    uint8_t       proto;           // 36     IP protocol
**    uint8_t       flow_type;       // 37     Class & Type info
**    uint16_t      sID;             // 38-39  Sensor ID
**
**    uint8_t       flags;           // 40     OR of all flags (Netflow flags)
**    uint8_t       init_flags;      // 41     TCP flags in first packet
**                                   //        or blank for "legacy" data
**    uint8_t       rest_flags;      // 42     TCP flags on non-initial packet
**                                   //        or blank for "legacy" data
**    uint8_t       tcp_state;       // 43     TCP state machine information
**
**    uint32_t      bpp;             // 44-47  Bytes-per-Packet
**
**    uint16_t      sTime_msec;      // 48-49  Start time fraction (millisec)
**    uint16_t      elapsed_msec;    // 50-51  Elapsed time fraction (millisec)
**
**    uint16_t      application;     // 52-53  Type of traffic
**    uint8_t[2]                     // 54-55  PADDING
**
**  56 bytes on disk.
*/

#define RECLEN_RWGENERIC_V3 56
#define RECLEN_RWGENERIC_V4 RECLEN_RWGENERIC_V3


/*
 *    Byte swap the RWGENERIC v3 record 'ar' in place.
 */
#define genericioRecordSwap_V3(ar)                              \
    {                                                           \
        SWAP_DATA32((ar) +  0);   /* sIP */                     \
        SWAP_DATA32((ar) +  4);   /* dIP */                     \
        SWAP_DATA16((ar) +  8);   /* sPort */                   \
        SWAP_DATA16((ar) + 10);   /* dPort */                   \
        SWAP_DATA32((ar) + 12);   /* nhIP */                    \
        SWAP_DATA16((ar) + 16);   /* input */                   \
        SWAP_DATA16((ar) + 18);   /* output */                  \
        SWAP_DATA32((ar) + 20);   /* sTime */                   \
        SWAP_DATA32((ar) + 24);   /* elapsed */                 \
        SWAP_DATA32((ar) + 28);   /* pkts */                    \
        SWAP_DATA32((ar) + 32);   /* bytes */                   \
        /* Two single bytes: (36)proto, (37)flow_type */        \
        SWAP_DATA16((ar) + 38);   /* sID */                     \
        /* Four single bytes: (40)flags, (41)init_flags,        \
         *                    (42)rest_flags, (43)tcp_state */  \
        SWAP_DATA32((ar) + 44);   /* bpp */                     \
        SWAP_DATA16((ar) + 48);   /* sTime_msec */              \
        SWAP_DATA16((ar) + 50);   /* elapsed_msec */            \
        SWAP_DATA16((ar) + 52);   /* application */             \
    }


/*
 *  Unpack the array of bytes 'ar' into a record 'rwrec'
 */
static int
genericioRecordUnpack_V3(
    skstream_t         *stream,
    rwGenericRec_V6    *rwrec,
    uint8_t            *ar)
{
    uint32_t sTime_sec, elapsed_sec;
    uint16_t sTime_mil, elapsed_mil;

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V3(ar);
    }

    /* sIP, dIP, sPort, dPort, nhIP */
    rwRecMemSetSIPv4(rwrec, &ar[0]);
    rwRecMemSetDIPv4(rwrec, &ar[4]);
    rwRecMemSetSPort(rwrec, &ar[8]);
    rwRecMemSetDPort(rwrec, &ar[10]);
    rwRecMemSetNhIPv4(rwrec, &ar[12]);

    /* input, output */
    rwpackUnpackInterfaces16bit(rwrec, &ar[16], &ar[18]);

    /* sTime, sTime_msec */
    memcpy(&sTime_sec, &ar[20], 4);
    memcpy(&sTime_mil, &ar[48], 2);

    /* elapsed, elapsed_msec */
    memcpy(&elapsed_sec, &ar[24], 4);
    memcpy(&elapsed_mil, &ar[50], 2);

    rwpackUnpackTimesMilli(rwrec, sTime_sec, sTime_mil,
                           elapsed_sec, elapsed_mil, sktimeInit());

    /* pkts, bytes */
    rwpackUnpackPacketsBytes32bit(rwrec, &ar[28], &ar[32]);

    /* proto, flowtype, sensor, flags, init_flags, rest_flags, tcp_state */
    rwRecMemSetProto(rwrec, &ar[36]);
    rwRecMemSetFlowType(rwrec, &ar[37]);
    rwRecMemSetSensor(rwrec, &ar[38]);
    rwRecMemSetFlags(rwrec, &ar[40]);
    rwRecMemSetInitFlags(rwrec, &ar[41]);
    rwRecMemSetRestFlags(rwrec, &ar[42]);
    rwRecMemSetTcpState(rwrec, &ar[43]);

    /* bpp field no longer exists */
    /* sTime_msec (above), elapsed_msec (above) */

    /* application */
    rwRecMemSetApplication(rwrec, &ar[52]);

    RWREC_MAYBE_CLEAR_TCPSTATE_EXPANDED(rwrec);

    return SKSTREAM_OK;
}


/*
 *  Pack the record 'rwrec' into an array of bytes 'ar'
 */
static int
genericioRecordPack_V3(
    skstream_t             *stream,
    const rwGenericRec_V6  *rwrec,
    uint8_t                *ar)
{
    uint32_t sTime_sec, elapsed_sec;
    uint32_t sTime_mil, elapsed_mil;
    uint16_t tmp16;
    int rv;

    /* input, output */
    rv = rwpackPackInterfaces16bit(&ar[16], &ar[18], rwrec);
    if (rv) { return rv; }

    /* pkts, bytes */
    rv = rwpackPackPacketsBytes32bit(&ar[28], &ar[32], rwrec);
    if (rv) { return rv; }

    /* sIP, dIP, sPort, dPort, nhIP */
    rwRecMemGetSIPv4(rwrec, &ar[0]);
    rwRecMemGetDIPv4(rwrec, &ar[4]);
    rwRecMemGetSPort(rwrec, &ar[8]);
    rwRecMemGetDPort(rwrec, &ar[10]);
    rwRecMemGetNhIPv4(rwrec, &ar[12]);

    /* Input &ar[16], Output &ar[18] */

    /* sTime, sTime_msec, elapsed, elapsed_msec */
    rwpackPackTimesMilli(&sTime_sec, &sTime_mil, &elapsed_sec, &elapsed_mil,
                         rwrec, sktimeInit());
    memcpy(&ar[20], &sTime_sec, 4);
    tmp16 = sTime_mil;
    memcpy(&ar[48], &tmp16, 2);
    memcpy(&ar[24], &elapsed_sec, 4);
    tmp16 = elapsed_mil;
    memcpy(&ar[50], &tmp16, 2);

    /* Packets &ar[28], Bytes &ar[32] */

    /* proto, flowtype, sensor, flags, init_flags, rest_flags, tcp_state */
    rwRecMemGetProto(rwrec, &ar[36]);
    rwRecMemGetFlowType(rwrec, &ar[37]);
    rwRecMemGetSensor(rwrec, &ar[38]);
    rwRecMemGetFlags(rwrec, &ar[40]);
    rwRecMemGetInitFlags(rwrec, &ar[41]);
    rwRecMemGetRestFlags(rwrec, &ar[42]);
    rwRecMemGetTcpState(rwrec, &ar[43]);

    /* bpp field no longer exists */
    memset(&ar[44], 0, 4);

    /* sTime_msec (above), elapsed_msec (above) */

    /* application */
    rwRecMemGetApplication(rwrec, &ar[52]);

    /* padding */
    memset(&ar[54], 0, 2);

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V3(ar);
    }

    return SKSTREAM_OK;
}


/* ********************************************************************* */

/*
**  RWGENERIC VERSION 2
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint32_t      nhIP;            // 12-15  Router Next Hop IP
**    uint16_t      input;           // 16-17  Router incoming SNMP interface
**    uint16_t      output;          // 18-19  Router outgoing SNMP interface
**
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**    uint32_t      elapsed;         // 24-27  Duration of flow
**
**    uint32_t      pkts;            // 28-31  Count of packets
**    uint32_t      bytes;           // 32-35  Count of bytes
**
**    uint8_t       proto;           // 36     IP protocol
**    uint8_t       flow_type;       // 37     Class & Type info
**    uint16_t      sID;             // 38-39  Sensor ID
**
**    uint8_t       flags;           // 40     OR of all flags (Netflow flags)
**    uint8_t       init_flags;      // 41     TCP flags in first packet
**                                   //        or blank for "legacy" data
**    uint8_t       rest_flags;      // 42     TCP flags on non-initial packet
**                                   //        or blank for "legacy" data
**    uint8_t       tcp_state;       // 43     TCP state machine information
**
**    uint32_t      bpp;             // 44-47  Bytes-per-Packet
**
**
**  48 bytes on disk.
*/

#define RECLEN_RWGENERIC_V2 48


/*
 *    Byte swap the RWGENERIC v2 record 'ar' in place.
 */
#define genericioRecordSwap_V2(ar)                              \
    {                                                           \
        SWAP_DATA32((ar) +  0);   /* sIP */                     \
        SWAP_DATA32((ar) +  4);   /* dIP */                     \
        SWAP_DATA16((ar) +  8);   /* sPort */                   \
        SWAP_DATA16((ar) + 10);   /* dPort */                   \
        SWAP_DATA32((ar) + 12);   /* nhIP */                    \
        SWAP_DATA16((ar) + 16);   /* input */                   \
        SWAP_DATA16((ar) + 18);   /* output */                  \
        SWAP_DATA32((ar) + 20);   /* sTime */                   \
        SWAP_DATA32((ar) + 24);   /* elapsed */                 \
        SWAP_DATA32((ar) + 28);   /* pkts */                    \
        SWAP_DATA32((ar) + 32);   /* bytes */                   \
        /* Two single bytes: (36)proto, (37)flow_type */        \
        SWAP_DATA16((ar) + 38);   /* sID */                     \
        /* Four single bytes: (40)flags, (41)init_flags,        \
         *                    (42)rest_flags, (43)tcp_state */  \
        SWAP_DATA32((ar) + 44);   /* bpp */                     \
    }


/*
 *  Unpack the array of bytes 'ar' into a record 'rwrec'
 */
static int
genericioRecordUnpack_V2(
    skstream_t         *stream,
    rwGenericRec_V6    *rwrec,
    uint8_t            *ar)
{
    uint32_t sTime, elapsed;

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V2(ar);
    }

    /* sIP, dIP, sPort, dPort, nhIP */
    rwRecMemSetSIPv4(rwrec, &ar[0]);
    rwRecMemSetDIPv4(rwrec, &ar[4]);
    rwRecMemSetSPort(rwrec, &ar[8]);
    rwRecMemSetDPort(rwrec, &ar[10]);
    rwRecMemSetNhIPv4(rwrec, &ar[12]);

    /* input, output */
    rwpackUnpackInterfaces16bit(rwrec, &ar[16], &ar[18]);

    /* sTime(sec), elapsed(sec) */
    memcpy(&sTime, &ar[20], 4);
    memcpy(&elapsed, &ar[24], 4);
    rwpackUnpackTimesMilli(rwrec, sTime, 0, elapsed, 0, sktimeInit());

    /* pkts, bytes */
    rwpackUnpackPacketsBytes32bit(rwrec, &ar[28], &ar[32]);

    /* proto, flow_type, sID, flags, init_flags, rest_flags, tcp_state */
    rwRecMemSetProto(rwrec, &ar[36]);
    rwRecMemSetFlowType(rwrec, &ar[37]);
    rwRecMemSetSensor(rwrec, &ar[38]);
    rwRecMemSetFlags(rwrec, &ar[40]);
    rwRecMemSetInitFlags(rwrec, &ar[41]);
    rwRecMemSetRestFlags(rwrec, &ar[42]);
    rwRecMemSetTcpState(rwrec, &ar[43]);

    /* bpp field no longer exists */

    RWREC_MAYBE_CLEAR_TCPSTATE_EXPANDED(rwrec);

    return SKSTREAM_OK;
}


/*
 *  Pack the record 'rwrec' into an array of bytes 'ar'
 */
static int
genericioRecordPack_V2(
    skstream_t             *stream,
    const rwGenericRec_V6  *rwrec,
    uint8_t                *ar)
{
    uint32_t sTime_sec, elapsed_sec;
    int rv;

    /* input, output */
    rv = rwpackPackInterfaces16bit(&ar[16], &ar[18], rwrec);
    if (rv) { return rv; }

    /* pkts, bytes */
    rv = rwpackPackPacketsBytes32bit(&ar[28], &ar[32], rwrec);
    if (rv) { return rv; }

    /* sIP, dIP, sPort, dPort, nhIP */
    rwRecMemGetSIPv4(rwrec, &ar[0]);
    rwRecMemGetDIPv4(rwrec, &ar[4]);
    rwRecMemGetSPort(rwrec, &ar[8]);
    rwRecMemGetDPort(rwrec, &ar[10]);
    rwRecMemGetNhIPv4(rwrec, &ar[12]);

    /* Input &ar[16], Output &ar[18] */

    /* sTime(epoch sec), elapsed(sec) */
    rwpackPackTimesMilli(&sTime_sec, NULL, &elapsed_sec, NULL, rwrec,
                         sktimeInit());
    memcpy(&ar[20], &sTime_sec, 4);
    memcpy(&ar[24], &elapsed_sec, 4);

    /* Packets &ar[28], Bytes &ar[32] */

    /* proto, flow_type, sID, flags, init_flags, rest_flags, tcp_state */
    rwRecMemGetProto(rwrec, &ar[36]);
    rwRecMemGetFlowType(rwrec, &ar[37]);
    rwRecMemGetSensor(rwrec, &ar[38]);
    rwRecMemGetFlags(rwrec, &ar[40]);
    rwRecMemGetInitFlags(rwrec, &ar[41]);
    rwRecMemGetRestFlags(rwrec, &ar[42]);
    rwRecMemGetTcpState(rwrec, &ar[43]);

    /* bpp field no longer exists */
    memset(&ar[44], 0, 4);

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V2(ar);
    }

    return SKSTREAM_OK;
}


/* ********************************************************************* */

/*
**  RWGENERIC VERSION 0
**  RWGENERIC VERSION 1
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint8_t       proto;           // 12     IP protocol
**    uint8_t       flags;           // 13     OR of all TCP flags on all pkts
**    uint8_t       input;           // 14     Router incoming SNMP interface
**    uint8_t       output;          // 15     Router outgoing SNMP interface
**
**    uint32_t      nhIP;            // 16-19  Router Next Hop IP
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**    uint32_t      pkts;            // 24-27  Count of packets
**    uint32_t      bytes;           // 28-31  Count of bytes
**    uint32_t      elapsed;         // 32-35  Duration of flow
**
**    uint8_t       sID;             // 36     Sensor ID
**    uint8_t       padding[3];      // 37-39  Padding
**
**  40 bytes on disk with padding (VERSION 0)
**  37 bytes on disk without padding (VERSION 1)
*/

#define RECLEN_RWGENERIC_V0 40
#define RECLEN_RWGENERIC_V1 37


/*
 *    Byte swap the RWGENERIC v1 record 'ar' in place.
 */
#define genericioRecordSwap_V1(ar)                                      \
    {                                                                   \
        SWAP_DATA32((ar) +  0);   /* sIP */                             \
        SWAP_DATA32((ar) +  4);   /* dIP */                             \
        SWAP_DATA16((ar) +  8);   /* sPort */                           \
        SWAP_DATA16((ar) + 10);   /* dPort */                           \
        /* Four single bytes: (12)proto, (13)flags, (14)input, (15)output */ \
        SWAP_DATA32((ar) + 16);   /* nhIP */                            \
        SWAP_DATA32((ar) + 20);   /* sTime */                           \
        SWAP_DATA32((ar) + 24);   /* pkts */                            \
        SWAP_DATA32((ar) + 28);   /* bytes */                           \
        SWAP_DATA32((ar) + 32);   /* elapsed */                         \
        /* One single bytes: (36)sensorId */                            \
    }


/*
 *  Unpack the array of bytes 'ar' into a record 'rwrec'
 */
static int
genericioRecordUnpack_V1(
    skstream_t         *stream,
    rwGenericRec_V6    *rwrec,
    uint8_t            *ar)
{
    uint32_t sTime, elapsed;

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V1(ar);
    }

    /* sIP, dIP, sPort, dPort */
    rwRecMemSetSIPv4(rwrec, &ar[0]);
    rwRecMemSetDIPv4(rwrec, &ar[4]);
    rwRecMemSetSPort(rwrec, &ar[8]);
    rwRecMemSetDPort(rwrec, &ar[10]);

    /* proto, flags */
    rwRecSetProto(rwrec, ar[12]);
    rwRecSetFlags(rwrec, ar[13]);

    /* input and output are single byte values */
    rwRecSetInput(rwrec, ar[14]);
    rwRecSetOutput(rwrec, ar[15]);

    /* nhIP */
    rwRecMemSetNhIPv4(rwrec, &ar[16]);

    /* sTime(sec) */
    memcpy(&sTime, &ar[20], 4);

    /* pkts, bytes */
    rwpackUnpackPacketsBytes32bit(rwrec, &ar[24], &ar[28]);

    /* elapsed(sec) */
    memcpy(&elapsed, &ar[32], 4);

    rwpackUnpackTimesMilli(rwrec, sTime, 0, elapsed, 0, sktimeInit());

    /* sID */
    rwRecSetSensor(rwrec, ar[36]);

    return SKSTREAM_OK;
}


/*
 *  Pack the record 'rwrec' into an array of bytes 'ar'
 */
static int
genericioRecordPack_V1(
    skstream_t             *stream,
    const rwGenericRec_V6  *rwrec,
    uint8_t                *ar)
{
    uint32_t sTime_sec, elapsed_sec;
    int rv;

    /* Check sizes of fields we've expanded in later versions */
    if (rwRecGetInput(rwrec) > UINT8_MAX || rwRecGetOutput(rwrec) > UINT8_MAX) {
        return SKSTREAM_ERR_SNMP_OVRFLO;
    }
    /* Check sizes of fields we've expanded in later versions */
    if (rwRecGetSensor(rwrec) > UINT8_MAX) {
        return SKSTREAM_ERR_SENSORID_OVRFLO;
    }

    /* pkts, bytes */
    rv = rwpackPackPacketsBytes32bit(&ar[24], &ar[28], rwrec);
    if (rv) { return rv; }

    /* sIP, dIP, sPort, dPort */
    rwRecMemGetSIPv4(rwrec, &ar[0]);
    rwRecMemGetDIPv4(rwrec, &ar[4]);
    rwRecMemGetSPort(rwrec, &ar[8]);
    rwRecMemGetDPort(rwrec, &ar[10]);

    /* proto, flags, input, output */
    ar[12] = rwRecGetProto(rwrec);
    ar[13] = rwRecGetFlags(rwrec);
    ar[14] = (uint8_t)rwRecGetInput(rwrec);
    ar[15] = (uint8_t)rwRecGetOutput(rwrec);

    /* nhIP */
    rwRecMemGetNhIPv4(rwrec, &ar[16]);

    /* sTime(epoch sec), elapsed(sec) */
    rwpackPackTimesMilli(&sTime_sec, NULL, &elapsed_sec, NULL, rwrec,
                         sktimeInit());
    memcpy(&ar[20], &sTime_sec, 4);
    memcpy(&ar[32], &elapsed_sec, 4);

    /* Packets &ar[24], Bytes &ar[28] */

    /* sID */
    ar[36] = (uint8_t)rwRecGetSensor(rwrec);

    /* clear padding if present (for consistent output) */
    if (stream->recLen == 40) {
        memset(&ar[37], 0, 3);
    }

    /* swap if required */
    if (stream->swapFlag) {
        genericioRecordSwap_V1(ar);
    }

    return SKSTREAM_OK;
}


/* ********************************************************************* */

/*
 *  Return length of record of specified version, or 0 if no such
 *  version exists.  See skstream_priv.h for details.
 */
uint16_t
genericioGetRecLen(
    sk_file_version_t   vers)
{
    switch (vers) {
      case 0:
        return RECLEN_RWGENERIC_V0;
      case 1:
        return RECLEN_RWGENERIC_V1;
      case 2:
        return RECLEN_RWGENERIC_V2;
      case 3:
        return RECLEN_RWGENERIC_V3;
      case 4:
        return RECLEN_RWGENERIC_V4;
      case 5:
        return RECLEN_RWGENERIC_V5;
      case 6:
        return RECLEN_RWGENERIC_V6;
      default:
        return 0;
    }
}


/*
 *  status = genericioPrepare(&stream);
 *
 *    Sets the record version to the default if it is unspecified,
 *    checks that the record format supports the requested record
 *    version, sets the record length, and sets the pack and unpack
 *    functions for this record format and version.
 */
int
genericioPrepare(
    skstream_t         *stream)
{
#define FILE_FORMAT "FT_RWGENERIC"
    sk_file_header_t *hdr = stream->silk_hdr;
    int rv = SKSTREAM_OK; /* return value */

    assert(skHeaderGetFileFormat(hdr) == FT_RWGENERIC);

    /* Set version if none was selected by caller */
    if ((stream->io_mode == SK_IO_WRITE)
        && (skHeaderGetRecordVersion(hdr) == SK_RECORD_VERSION_ANY))
    {
        skHeaderSetRecordVersion(hdr, DEFAULT_RECORD_VERSION);
    }

    /* version check; set values based on version */
    switch (skHeaderGetRecordVersion(hdr)) {
      case 6:
        stream->rwUnpackFn = &genericioRecordUnpack_V6;
        stream->rwPackFn   = &genericioRecordPack_V6;
        break;
      case 5:
        stream->rwUnpackFn = &genericioRecordUnpack_V5;
        stream->rwPackFn   = &genericioRecordPack_V5;
        break;
      case 4:
      case 3:
        /* V3 and V4 differ only in that V4 supports compression on
         * read and write; V3 supports compression only on read */
        stream->rwUnpackFn = &genericioRecordUnpack_V3;
        stream->rwPackFn   = &genericioRecordPack_V3;
        break;
      case 2:
        stream->rwUnpackFn = &genericioRecordUnpack_V2;
        stream->rwPackFn   = &genericioRecordPack_V2;
        break;
      case 1:
      case 0:
        /* Version 0 and Version 1 records are nearly the same; the
         * on-disk Version 0 records included the 3 bytes of in-core
         * padding; the on-disk Version 1 records do not include these
         * 3 bytes. */
        stream->rwUnpackFn = &genericioRecordUnpack_V1;
        stream->rwPackFn   = &genericioRecordPack_V1;
        break;
      default:
        rv = SKSTREAM_ERR_UNSUPPORT_VERSION;
        goto END;
    }

    stream->recLen = genericioGetRecLen(skHeaderGetRecordVersion(hdr));

    /* verify lengths */
    if (stream->recLen == 0) {
        skAppPrintErr("Record length not set for %s version %u",
                      FILE_FORMAT, (unsigned)skHeaderGetRecordVersion(hdr));
        skAbort();
    }
    if (stream->recLen != skHeaderGetRecordLength(hdr)) {
        if (0 == skHeaderGetRecordLength(hdr)) {
            skHeaderSetRecordLength(hdr, stream->recLen);
        } else {
            skAppPrintErr(("Record length mismatch for %s version %u\n"
                           "\tcode = %" PRIu16 " bytes;  header = %lu bytes"),
                          FILE_FORMAT, (unsigned)skHeaderGetRecordVersion(hdr),
                          stream->recLen,
                          (unsigned long)skHeaderGetRecordLength(hdr));
            skAbort();
        }
    }

  END:
    return rv;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
