/*
** Copyright (C) 2004-2024 by Carnegie Mellon University.
**
** @OPENSOURCE_LICENSE_START@
**
** SiLK 3.23
**
** Copyright 2024 Carnegie Mellon University.
**
** NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
** INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
** UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
** IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
** FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
** OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
** MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
** TRADEMARK, OR COPYRIGHT INFRINGEMENT.
**
** Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
** contact permission@sei.cmu.edu for full terms.
**
** [DISTRIBUTION STATEMENT A] This material has been approved for public
** release and unlimited distribution.  Please see Copyright notice for
** non-US Government use and distribution.
**
** This Software includes and/or makes use of Third-Party Software each
** subject to its own license.
**
** DM24-1064
**
** @OPENSOURCE_LICENSE_END@
*/

/*
**  Functions to support binary output: opening binary SiLK files for
**  writing or appending.
**
*/


#include <silk/silk.h>

RCSIDENTVAR(rcsID_RWPACK_C, "$SiLK: rwpack.c afa589dbc78d 2024-09-19 17:35:03Z mthomas $");

#ifndef RWPACK_BYTES_PACKETS
#if defined(RWPACK_FLAGS_TIMES_VOLUMES) || defined(RWPACK_SBB_PEF)
#define RWPACK_BYTES_PACKETS 1
#endif
#endif
#ifndef RWPACK_TIMES_MILLI
#if defined(RWPACK_FLAGS_TIMES_VOLUMES) || defined(RWPACK_SBB_PEF) || defined(RWPACK_TIME_BYTES_PKTS_FLAGS) || defined(RWPACK_TIMES_FLAGS_PROTO)
#define RWPACK_TIMES_MILLI 1
#endif
#endif
#include "skstream_priv.h"


#ifdef RWPACK_TIMES_MILLI
static int
rwpackPackTimesMilli(
    uint32_t               *stime_sec,
    uint32_t               *stime_milli,
    uint32_t               *elapsed_sec,
    uint32_t               *elapsed_milli,
    const rwGenericRec_V6  *rwrec,
    sktime_t                file_start_time)
{
    if (!sktimeIsSet(file_start_time)) {
        assert(NULL != stime_sec);
        if (NULL == stime_milli) {
            /* only return seconds */
            rwRecMemGetStartSeconds(rwrec, stime_sec);
        } else {
            sktimeGetSecondsMilli(
                rwRecGetStartTime(rwrec), stime_sec, stime_milli);
        }
    } else {
        sktime_t sTime = rwRecGetStartTime(rwrec);
        skInterval_t ntval;
        if (sktimeCmp(sTime, file_start_time, <)) {
            return SKSTREAM_ERR_STIME_UNDRFLO;
        }
        ntval = sktimeComputeInterval(sTime, file_start_time);
        if (skIntervalCmp(
                ntval, skIntervalSetFromSeconds(MAX_START_TIME), >=))
        {
            return SKSTREAM_ERR_STIME_OVRFLO;
        }

        if (NULL == stime_milli) {
            /* only return seconds */
            if (stime_sec) {
                *stime_sec = skIntervalGetSeconds(ntval);
            }
        } else if (NULL == stime_sec) {
            *stime_milli = skIntervalGetMilli(ntval);
        } else {
            skIntervalGetSecondsMilli(ntval, stime_sec, stime_milli);
        }
    }

    if (NULL == elapsed_milli) {
        if (elapsed_sec) {
            rwRecMemGetElapsedSeconds(rwrec, elapsed_sec);
        }
    } else if (NULL == elapsed_sec) {
        uint64_t dur = skIntervalGetMilli(rwRecGetElapsed(rwrec));
        if (dur > UINT32_MAX) {
            return SKSTREAM_ERR_ELPSD_OVRFLO;
        }
        *elapsed_milli = dur;
    } else {
        skIntervalGetSecondsMilli(rwRecGetElapsed(rwrec),
                                  elapsed_sec, elapsed_milli);
    }

    return 0;
}


/*  Sets the record's start-time and end-time given the file's starting time
 *  from its header, the record's starting time relative to the file's start
 *  time, and the records elapsed time.  The record's stime and elapsed are
 *  broken into whole seconds and milliseconds and one or both parameters may
 *  be specified. */
static void
rwpackUnpackTimesMilli(
    rwGenericRec_V6    *rwrec,
    uint32_t            stime_sec,
    uint32_t            stime_milli,
    uint32_t            elapsed_sec,
    uint32_t            elapsed_milli,
    sktime_t            file_start_time)
{
    if (sktimeIsSet(file_start_time)) {
        rwRecSetStartTime(
            rwrec,
            sktimeAddInterval(file_start_time,
                              skIntervalSetFromMilli(stime_sec, stime_milli)));
    } else {
        rwRecSetStartTime(
            rwrec, sktimeCreateFromMilli(stime_sec, stime_milli));
    }
    rwRecSetEndTime(
        rwrec,
        sktimeAddInterval(rwRecGetStartTime(rwrec),
                          skIntervalSetFromMilli(elapsed_sec, elapsed_milli)));
}
#endif  /* RWPACK_TIMES_MILLI */


#ifdef RWPACK_BYTES_PACKETS
/*  Convert bytes and packets fields in 'rwrec' to the values used in
 *  rwfilter output files and in the packed file formats.  See
 *  skstream_priv.h. */
static int
rwpackPackBytesPackets(
    uint32_t               *bpp_out,
    uint32_t               *pkts_out,
    uint32_t               *pflag_out,
    const rwGenericRec_V6  *rwrec)
{
    imaxdiv_t bpp;
    uint64_t packets;
    uint64_t bytes;

    assert(bpp_out);
    assert(pkts_out);
    assert(pflag_out);
    assert(rwrec);

    packets = rwRecGetPkts(rwrec);
    bytes = rwRecGetBytes(rwrec);

    /* Check for 0 value in 'pkts' field */
    if (packets == 0) {
        return SKSTREAM_ERR_PKTS_ZERO;
    }

    /* Verify that there are more bytes than packets */
    if (packets > bytes) {
        return SKSTREAM_ERR_PKTS_GT_BYTES;
    }

    /* Set packets field; check for overflow */
    if (packets < MAX_PKTS) {
        *pkts_out = packets;
        *pflag_out = 0;
    } else if (packets >= DBL_MAX_PKTS) {
        /* Double overflow in pkts */
        return SKSTREAM_ERR_PKTS_OVRFLO;
    } else {
        *pkts_out = packets / PKTS_DIVISOR;
        *pflag_out = 1;
    }

    /* calculate the bytes-per-packet ratio */
    bpp = imaxdiv(bytes, packets);

    if (bpp.quot > MASKARRAY_14) {
        return SKSTREAM_ERR_BPP_OVRFLO;
    }

    /* compute new value */
    *bpp_out = (((uint32_t)bpp.quot) << 6) |
                ((uint32_t)(bpp.rem * BPP_PRECN / packets));

    return SKSTREAM_OK;
}


/*  Fill in the bytes and packets fields in rwrec by expanding the
 *  values that were read from disk.  See skstream_priv.h for details. */
static void
rwpackUnpackBytesPackets(
    rwGenericRec_V6    *rwrec,
    uint32_t            bpp,
    uint32_t            pkts,
    uint32_t            pflag)
{
    uint64_t bytes;
    long bPPkt, bPPFrac;
    lldiv_t i;

    if (pflag) {
        pkts *= PKTS_DIVISOR;
    }

    /* Unpack the bpp value:  bPPkt:14; bPPFrac:6; */
    bPPkt = GET_MASKED_BITS(bpp, 6, 14);
    bPPFrac = GET_MASKED_BITS(bpp, 0, 6);

    /* convert fraction to whole number */
    i = lldiv((bPPFrac * (long)pkts), BPP_PRECN);

    bytes = ((bPPkt * pkts) + i.quot + ((i.rem >= BPP_PRECN_DIV_2) ? 1 : 0));

    rwRecSetPkts(rwrec, pkts);
    rwRecSetBytes(rwrec, bytes);
}
#endif  /* RWPACK_BYTES_PACKETS */


#ifdef RWPACK_PROTO_FLAGS
/* Pack the protocol, flags, and TCP state fields.  See skstream_priv.h */
static void
rwpackPackProtoFlags(
    uint8_t                *is_tcp_out,
    uint8_t                *prot_flags_out,
    uint8_t                *tcp_state_out,
    uint8_t                *rest_flags_out,
    const rwGenericRec_V6  *rwrec)
{
    *tcp_state_out = rwRecGetTcpState(rwrec);
    if (rwRecGetProto(rwrec) != IPPROTO_TCP) {
        /* Flow is not TCP, so there is no additional TCP info.  Set
         * '*rest_flags_out' to value of rwrec->flags. */
        *is_tcp_out = 0;
        *prot_flags_out = rwRecGetProto(rwrec);
        *rest_flags_out = rwRecGetFlags(rwrec);
    } else {
        /* Flow is TCP */
        *is_tcp_out = 1;
        if (*tcp_state_out & SK_TCPSTATE_EXPANDED) {
            /* There is additional TCP info.  Put the initial TCP
             * flags into the '*prot_flags_out' value. */
            *prot_flags_out = rwRecGetInitFlags(rwrec);
            *rest_flags_out = rwRecGetRestFlags(rwrec);
        } else {
            /* There is no additional TCP info. */
            *prot_flags_out = rwRecGetFlags(rwrec);
            *rest_flags_out = 0;
        }
    }
}


/* Fill in the protocol, flags, and TCP state fields on the rwrec.  See
 * skstream_priv.h */
static void
rwpackUnpackProtoFlags(
    rwGenericRec_V6    *rwrec,
    uint8_t             is_tcp,
    uint8_t             prot_flags,
    uint8_t             tcp_state,
    uint8_t             rest_flags)
{
    /* For some record types (e.g., RWWWW), proto is fixed at 6(TCP)
     * and there may be another value in the 'is_tcp' bit; ignore the
     * 'is_tcp' bit if the protocol is already set to TCP. */
    rwRecSetTcpState(rwrec, tcp_state);
    if ((rwRecGetProto(rwrec) == IPPROTO_TCP) || (is_tcp == 1)) {
        /* Flow is TCP */
        rwRecSetProto(rwrec, IPPROTO_TCP);
        if (tcp_state & SK_TCPSTATE_EXPANDED) {
            /* We have additional flow information; value in
             * prot_flags are the flags on the first packet. */
            rwRecSetInitFlags(rwrec, prot_flags);
            rwRecSetRestFlags(rwrec, rest_flags);
            rwRecSetFlags(rwrec, (uint8_t)(prot_flags | rest_flags));
        } else {
            /* We don't have additional TCP info; 'prot_flags' holds
             * the flags. */
            rwRecSetFlags(rwrec, prot_flags);
        }
    } else {
        /* Flow is not TCP so there can be no additional TCP info.
         * 'prot_flags' holds the protocol.  Although 'flags' has no
         * real meaning here, the 'rest_flags' value has the value
         * that we got from the flow collector, so set 'rwrec->flags'
         * to that value. */
        rwRecSetProto(rwrec, prot_flags);
        rwRecSetFlags(rwrec, rest_flags);
    }
}
#endif  /* RWPACK_PROTO_FLAGS */


#ifdef RWPACK_SBB_PEF
/*  Compute the 'sbb' and 'pef' fields used in packed file formats.
 *  See skstream_priv.h. */
static int
rwpackPackSbbPef(
    uint32_t               *sbb_out,
    uint32_t               *pef_out,
    const rwGenericRec_V6  *rwrec,
    sktime_t                file_start_time)
{
    int rv = SKSTREAM_OK; /* return value */
    uint32_t start_time;
    uint32_t elapsed;
    uint32_t pkts, bpp, pflag;

    rv = rwpackPackTimesMilli(&start_time, NULL, &elapsed, NULL,
                              rwrec, file_start_time);
    if (rv) { goto END; }
    if (elapsed >= MAX_ELAPSED_TIME_OLD) {
        rv = SKSTREAM_ERR_ELPSD_OVRFLO;
        goto END;
    }

    rv = rwpackPackBytesPackets(&bpp, &pkts, &pflag, rwrec);
    if (rv) { goto END; }

    /* sbb: uint32_t sTime:12;  uint32_t bPPkt:14;  uint32_t bPPFrac:6; */
    *sbb_out = (((MASKARRAY_12 & (uint32_t)start_time) << 20)
                | (bpp & MASKARRAY_20));

    /* pef: uint32_t pkts:20; uint32_t elapsed :11; uint32_t pktsFlag:1; */
    *pef_out = ((pkts << 12) | (elapsed << 1) | pflag);

  END:
    return rv;
}


/* Set values in rwrec by expanding the 'sbb' and 'pef' fields that
 * exist in the packed file formats.  See skstream_priv.h for details. */
static void
rwpackUnpackSbbPef(
    rwGenericRec_V6    *rwrec,
    sktime_t            file_start_time,
    const uint32_t     *sbb,
    const uint32_t     *pef)
{
    uint32_t pkts, elapsed, pktsFlag, bpp, start_time;

    /* pef: uint32_t pkts:20; uint32_t elapsed :11; uint32_t pktsFlag:1; */
    pkts = *pef >> 12;
    elapsed = ((*pef >> 1) & MASKARRAY_11);
    pktsFlag = *pef & MASKARRAY_01;

    /* sbb: uint32_t start_time:12; uint32_t bpp:20 */
    bpp = *sbb & MASKARRAY_20;
    start_time = (*sbb >> 20);

    rwpackUnpackTimesMilli(rwrec, start_time, 0, elapsed, 0, file_start_time);
    rwpackUnpackBytesPackets(rwrec, bpp, pkts, pktsFlag);
}
#endif  /* RWPACK_SBB_PEF */


#ifdef RWPACK_TIME_BYTES_PKTS_FLAGS
static int
rwpackPackTimeBytesPktsFlags(
    uint32_t               *pkts_stime_out,
    uint32_t               *bbe_out,
    uint32_t               *msec_flags_out,
    const rwGenericRec_V6  *rwrec,
    sktime_t                file_start_time)
{
    int rv = SKSTREAM_OK; /* return value */
    uint32_t pkts, bpp, pflag, is_tcp;
    uint32_t stime_sec, stime_mil;
    uint32_t elapsed_sec, elapsed_mil;
    uint8_t prot_flags;

    rv = rwpackPackTimesMilli(&stime_sec, &stime_mil, &elapsed_sec,
                              &elapsed_mil, rwrec, file_start_time);
    if (rv) { goto END; }
    if (elapsed_sec >= MAX_ELAPSED_TIME) {
        rv = SKSTREAM_ERR_ELPSD_OVRFLO;
        goto END;
    }

    rv = rwpackPackBytesPackets(&bpp, &pkts, &pflag, rwrec);
    if (rv) { goto END; }

    /* pkts_stime: pkts:20; sTime: 12; */
    *pkts_stime_out = ((pkts << 12)
                       | (MASKARRAY_12 & stime_sec));

    /* bbe: bpp: 20; elapsed: 12 */
    *bbe_out = ((bpp << 12)
                | (MASKARRAY_12 & elapsed_sec));

    /* set is_tcp bit and prot_flags */
    if (rwRecGetProto(rwrec) == IPPROTO_TCP) {
        is_tcp = 1;
        prot_flags = rwRecGetFlags(rwrec);
    } else {
        is_tcp = 0;
        prot_flags = rwRecGetProto(rwrec);
    }

    /* msec_flags: sTime_msec:10; elaps_msec:10; pflag:1;
     *             is_tcp:1; pad:2; prot_flags:8;*/
    *msec_flags_out = (((MASKARRAY_10 & stime_mil) << 22)
                       | ((MASKARRAY_10 & elapsed_mil) << 12)
                       | (pflag ? (1 << 11) : 0)
                       | (is_tcp ? (1 << 10) : 0)
                       | prot_flags);

  END:
    return rv;
}


static void
rwpackUnpackTimeBytesPktsFlags(
    rwGenericRec_V6    *rwrec,
    sktime_t            file_start_time,
    const uint32_t     *pkts_stime,
    const uint32_t     *bbe,
    const uint32_t     *msec_flags)
{
    uint32_t pkts, bpp, is_tcp, pflag;
    uint32_t stime_sec, stime_msec, elapsed_sec, elapsed_msec;
    uint8_t prot_flags;

    /* pkts_stime: pkts:20; sTime: 12; */
    pkts = GET_MASKED_BITS(*pkts_stime, 12, 20);
    stime_sec = GET_MASKED_BITS(*pkts_stime, 0, 12);

    /* msec_flags: sTime_msec:10; elaps_msec:10; pflag:1;
     *             is_tcp:1; pad:2, prot_flags:8;          */
    stime_msec = GET_MASKED_BITS(*msec_flags, 22, 10);
    elapsed_msec = GET_MASKED_BITS(*msec_flags, 12, 10);
    pflag = GET_MASKED_BITS(*msec_flags, 11, 1);
    is_tcp = GET_MASKED_BITS(*msec_flags, 10, 1);
    prot_flags = (uint8_t)GET_MASKED_BITS(*msec_flags, 0, 8);

    /* bbe: bpp: 20; elapsed_sec: 12 */
    bpp = GET_MASKED_BITS(*bbe, 12, 20);
    elapsed_sec = GET_MASKED_BITS(*bbe, 0, 12);

    rwpackUnpackTimesMilli(rwrec, stime_sec, stime_msec, elapsed_sec,
                           elapsed_msec, file_start_time);

    if (rwRecGetProto(rwrec) == IPPROTO_TCP) {
        /* caller has forced record to be TCP */
        rwRecSetFlags(rwrec, prot_flags);
    } else if (is_tcp == 0) {
        /* flow is not TCP */
        rwRecSetProto(rwrec, prot_flags);
    } else {
        /* flow is TCP */
        rwRecSetProto(rwrec, IPPROTO_TCP);
        rwRecSetFlags(rwrec, prot_flags);
    }

    /* unpack the bpp value into bytes and packets */
    rwpackUnpackBytesPackets(rwrec, bpp, pkts, pflag);
}
#endif  /* RWPACK_TIME_BYTES_PKTS_FLAGS */


#ifdef RWPACK_FLAGS_TIMES_VOLUMES
static int
rwpackPackFlagsTimesVolumes(
    uint8_t                *ar,
    const rwGenericRec_V6  *rwrec,
    sktime_t                file_start_time,
    size_t                  len)
{
    uint32_t bpp, tmp, pkts, pflag;
    uint32_t stime_mil;
    uint32_t elapsed_mil;
    uint8_t tcp_state;
    int rv = SKSTREAM_OK;

    rv = rwpackPackTimesMilli(NULL, &stime_mil, NULL, &elapsed_mil,
                              rwrec, file_start_time);
    if (rv) { goto END; }
    if (elapsed_mil >= 1000u * MAX_ELAPSED_TIME) {
        rv = SKSTREAM_ERR_ELPSD_OVRFLO;
        goto END;
    }

    rv = rwpackPackBytesPackets(&bpp, &pkts, &pflag, rwrec);
    if (rv) { goto END; }

/*
**    uint32_t      stime_bb1;       //  0- 3
**    // uint32_t     stime     :22  //        Start time:msec offset from hour
**    // uint32_t     bPPkt1    :10; //        Whole bytes-per-packet (hi 10)
*/
    tmp = (((MASKARRAY_22 & stime_mil) << 10)
           | (GET_MASKED_BITS(bpp, 10, 10)));
    memcpy(&ar[0], &tmp, sizeof(tmp));

/*
**    uint32_t      bb2_elapsed;     //  4- 7
**    // uint32_t     bPPkt2    : 4; //        Whole bytes-per-packet (low 4)
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     elapsed   :22; //        Duration of flow in msec
**
*/
    tmp = ((GET_MASKED_BITS(bpp, 0, 10) << 22)
           | (MASKARRAY_22 & elapsed_mil));
    memcpy(&ar[4], &tmp, sizeof(tmp));

/*
**    uint8_t      tcp_state;        // 12     TCP state machine info
**    uint8_t      rest_flags;       // 13     is_tcp==0: Flow's reported flags
**                                   //        is_tcp==1 &&
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**    uint16_t     application;      // 14-15  Type of traffic
*/
    if (len == 12) {
        tcp_state = 0;
    } else if (len == 16) {
        tcp_state = rwRecGetTcpState(rwrec);
        ar[12] = tcp_state;
        if (rwRecGetProto(rwrec) != IPPROTO_TCP) {
            /* when not TCP, holds whatever flags value we have */
            ar[13] = rwRecGetFlags(rwrec);
        } else if (tcp_state & SK_TCPSTATE_EXPANDED) {
            /* when TCP and extended data, hold the rest flags */
            ar[13] = rwRecGetRestFlags(rwrec);
        } else {
            /* when TCP but no extended data, is empty */
            ar[13] = 0;
        }
        rwRecMemGetApplication(rwrec, &ar[14]);
    } else {
        skAppPrintErr(("Bad length (%lu) to rwpackPackFlagsTimesVolumes"
                       " at %s:%d"),
                      (unsigned long)len, __FILE__, __LINE__);
        skAbort();
    }

/*
**    uint32_t      pro_flg_pkts;    //  8-11
**    // uint32_t     prot_flags: 8; //        is_tcp==0: IP protocol
**                                   //        is_tcp==1 &&
**                                   //          EXPANDED==0:TCPflags/All pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**    // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
**    // uint32_t     padding   : 2; //
**    // uint32_t     pkts      :20; //        Count of packets
*/
    tmp = ((pflag << 23)
           | (MASKARRAY_20 & pkts));
    if (rwRecGetProto(rwrec) != IPPROTO_TCP) {
        tmp |= (rwRecGetProto(rwrec) << 24);
    } else {
        if (tcp_state & SK_TCPSTATE_EXPANDED) {
            tmp |= ((rwRecGetInitFlags(rwrec) << 24)
                    | (1 << 22));
        } else {
            tmp |= ((rwRecGetFlags(rwrec) << 24)
                    | (1 << 22));
        }
    }
    memcpy(&ar[8], &tmp, sizeof(tmp));

  END:
    return rv;
}


static void
rwpackUnpackFlagsTimesVolumes(
    rwGenericRec_V6    *rwrec,
    const uint8_t      *ar,
    sktime_t            file_start_time,
    size_t              len,
    int                 is_tcp)
{
    uint32_t bpp, tmp, pkts, pflag;
    uint8_t tcp_state, rest_flags;

/*
**    uint8_t      tcp_state;        // 12     TCP state machine info
**    uint8_t      rest_flags;       // 13     is_tcp==0: Flow's reported flags
**                                   //        is_tcp==1 &&
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**    uint16_t     application;      // 14-15  Type of traffic
*/
    if (len == 12) {
        tcp_state = 0;
        rest_flags = 0;
    } else if (len == 16) {
        tcp_state = ar[12];
        rest_flags = ar[13];
        rwRecSetTcpState(rwrec, tcp_state);
        rwRecMemSetApplication(rwrec, &ar[14]);
    } else {
        skAppPrintErr(("Bad length (%lu) to rwpackUnpackFlagsTimesVolumes"
                       " at %s:%d"),
                      (unsigned long)len, __FILE__, __LINE__);
        skAbort();
    }

/*
**    uint32_t      pro_flg_pkts;    //  8-11
**    // uint32_t     prot_flags: 8; //        is_tcp==0: IP protocol
**                                   //        is_tcp==1 &&
**                                   //          EXPANDED==0:TCPflags/All pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**    // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
**    // uint32_t     padding   : 2; //
**    // uint32_t     pkts      :20; //        Count of packets
*/
    memcpy(&tmp, &ar[8], sizeof(tmp));
    pkts = GET_MASKED_BITS(tmp, 0, 20);
    pflag = GET_MASKED_BITS(tmp, 23, 1);
    if (!is_tcp) {
        is_tcp = GET_MASKED_BITS(tmp, 22, 1);
    }
    if (!is_tcp) {
        rwRecSetProto(rwrec, GET_MASKED_BITS(tmp, 24, 8));
        rwRecSetFlags(rwrec, rest_flags);
    } else {
        rwRecSetProto(rwrec, IPPROTO_TCP);
        if (tcp_state & SK_TCPSTATE_EXPANDED) {
            rwRecSetRestFlags(rwrec, rest_flags);
            rwRecSetInitFlags(rwrec, GET_MASKED_BITS(tmp, 24, 8));
        }
        rwRecSetFlags(rwrec, (GET_MASKED_BITS(tmp, 24, 8)
                              | rest_flags));
    }

/*
**    uint32_t      bb2_elapsed;     //  4- 7
**    // uint32_t     bPPkt2    : 4; //        Whole bytes-per-packet (low 4)
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     elapsed   :22; //        Duration of flow in msec
**
*/
    memcpy(&tmp, &ar[4], sizeof(tmp));

/*
**    uint32_t      stime_bb1;       //  0- 3
**    // uint32_t     stime     :22  //        Start time:msec offset from hour
**    // uint32_t     bPPkt1    :10; //        Whole bytes-per-packet (hi 10)
*/
    memcpy(&bpp, &ar[0], sizeof(bpp));

    rwpackUnpackTimesMilli(rwrec, 0, GET_MASKED_BITS(bpp, 10, 22),
                           0, GET_MASKED_BITS(tmp, 0, 22), file_start_time);

    bpp = ((GET_MASKED_BITS(bpp, 0, 10) << 10)
           | GET_MASKED_BITS(tmp, 22, 10));

    rwpackUnpackBytesPackets(rwrec, bpp, pkts, pflag);
}
#endif  /* RWPACK_FLAGS_TIMES_VOLUMES */


#ifdef RWPACK_TIMES_FLAGS_PROTO
static int
rwpackPackTimesFlagsProto(
    const rwGenericRec_V6  *rwrec,
    uint8_t                *ar_rflag_stime,
    uint8_t                *ar_elapsed,
    sktime_t                file_start_time)
{
    uint32_t stime_mil;
    uint32_t elapsed_mil;
    uint32_t tmp;
    int rv = SKSTREAM_OK; /* return value */

    rv = rwpackPackTimesMilli(NULL, &stime_mil, NULL, &elapsed_mil,
                              rwrec, file_start_time);
    if (rv) { goto END; }

    memcpy(ar_elapsed, &elapsed_mil, sizeof(elapsed_mil));

/*
**    uint32_t      rflag_stime;     //  0- 3
**    // uint32_t     rest_flags: 8; //        is_tcp==0: Empty; else
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**    // uint32_t     is_tcp    : 1; //        1 if FLOW is TCP; 0 otherwise
**    // uint32_t     unused    : 1; //        Reserved
**    // uint32_t     stime     :22; //        Start time:msec offset from hour
**
**    uint8_t       proto_iflags;    //  4     is_tcp==0: Protocol; else:
**                                   //          EXPANDED==0:TCPflags/ALL pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**    uint8_t       tcp_state;       //  5     TCP state machine info
*/

    /* Start time, Protocol, TCP Flags */
    if (IPPROTO_TCP != rwRecGetProto(rwrec)) {
        /* First 4 bytes only contains stime */
        assert((MASKARRAY_22 & stime_mil) == stime_mil);
        memcpy(&ar_rflag_stime[ 0], &stime_mil, sizeof(stime_mil));
        rwRecMemGetProto(rwrec, &ar_rflag_stime[ 4]);

    } else if (rwRecGetTcpState(rwrec) & SK_TCPSTATE_EXPANDED) {
        tmp = ((rwRecGetRestFlags(rwrec) << 24)
               | (1 << 23)
               | (MASKARRAY_22 & stime_mil));
        memcpy(&ar_rflag_stime[ 0], &tmp, sizeof(tmp));
        rwRecMemGetInitFlags(rwrec, &ar_rflag_stime[ 4]);

    } else {
        tmp = ((1 << 23)
               | (MASKARRAY_22 & stime_mil));
        memcpy(&ar_rflag_stime[ 0], &tmp, sizeof(tmp));
        rwRecMemGetFlags(rwrec, &ar_rflag_stime[ 4]);
    }

    rwRecMemGetTcpState(rwrec, &ar_rflag_stime[ 5]);

  END:
    return rv;
}


static void
rwpackUnpackTimesFlagsProto(
    rwGenericRec_V6    *rwrec,
    const uint8_t      *ar_rflag_stime,
    const uint8_t      *ar_elapsed,
    sktime_t            file_start_time)
{
    uint32_t tmp, elapsed;

/*
**    uint32_t      rflag_stime;     //  0- 3
**    // uint32_t     rest_flags: 8; //        is_tcp==0: Empty; else
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**    // uint32_t     is_tcp    : 1; //        1 if FLOW is TCP; 0 otherwise
**    // uint32_t     unused    : 1; //        Reserved
**    // uint32_t     stime     :22; //        Start time:msec offset from hour
**
**    uint8_t       proto_iflags;    //  4     is_tcp==0: Protocol; else:
**                                   //          EXPANDED==0:TCPflags/ALL pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**    uint8_t       tcp_state;       //  5     TCP state machine info
*/

    memcpy(&tmp, &ar_rflag_stime[0], sizeof(tmp));
    memcpy(&elapsed, &ar_elapsed[0], sizeof(elapsed));

    rwpackUnpackTimesMilli(rwrec, 0, GET_MASKED_BITS(tmp, 0, 22),
                           0, elapsed, file_start_time);

    if (0 == GET_MASKED_BITS(tmp, 23, 1)) {
        /* Not TCP; protocol is in the 'proto_iflags' field */
        rwRecMemSetProto(rwrec, &ar_rflag_stime[4]);

    } else if (ar_rflag_stime[5] & SK_TCPSTATE_EXPANDED) {
        /* Is TCP and have initial-flags and session-flags */
        rwRecSetProto(rwrec, IPPROTO_TCP);
        rwRecSetRestFlags(rwrec, GET_MASKED_BITS(tmp, 24, 8));
        rwRecMemSetInitFlags(rwrec, &ar_rflag_stime[4]);
        rwRecSetFlags(rwrec,
                      (rwRecGetInitFlags(rwrec) | rwRecGetRestFlags(rwrec)));
    } else {
        /* Is TCP; only have combined TCP flags */
        rwRecSetProto(rwrec, IPPROTO_TCP);
        rwRecMemSetFlags(rwrec, &ar_rflag_stime[4]);
    }

    rwRecMemSetTcpState(rwrec, &ar_rflag_stime[5]);
}
#endif  /* RWPACK_TIMES_FLAGS_PROTO */


#ifdef RWPACK_STATE_FLAGS_TIMES_NANO
static int
rwpackPackStateFlagsTimesNano(
    uint8_t                *ar,
    const rwGenericRec_V6  *rwrec,
    sktime_t                file_start_time)
{
    uint64_t state_flag_stime;
    uint64_t file_start_nano;
    uint64_t rflag_etime;
    uint8_t tcp_state;

    file_start_nano = sktimeGetEpochNano(file_start_time);

/*
**    uint64_t      state_flag_stime;//  0- 7
**    // uint64_t     tcp_state : 8; //        TCP state machine info
**    // uint64_t     pro_iflags: 8; //        is_tcp==0: Protocol; else:
**                                   //          EXPANDED==0:TCPflags/ALL pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**    // uint64_t     unused    : 5; //        Reserved
**    // uint64_t     is_tcp    : 1; //        1 if FLOW is TCP; 0 otherwise
**    // uint64_t     stime     :42; //        Start time:nsec offset from hour
**
**    uint64_t      rflag_etime;     //  8-15
**    // uint64_t     rest_flags: 8; //        is_tcp==0: Empty; else
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**    // uint64_t     etime     :56; //        End time:nsec offset from hour
*/

    state_flag_stime = sktimeGetEpochNano(rwRecGetStartTime(rwrec));
    if (state_flag_stime < file_start_nano) {
        return SKSTREAM_ERR_STIME_UNDRFLO;
    }
    state_flag_stime -= file_start_nano;
    if (state_flag_stime >= UINT64_C(1000000000) * MAX_START_TIME) {
        return SKSTREAM_ERR_STIME_OVRFLO;
    }

    rflag_etime = sktimeGetEpochNano(rwRecGetEndTime(rwrec));
    rflag_etime -= file_start_nano;
    if (0 != GET_MASKED_BITS64(rflag_etime, 56, 6)) {
        return SKSTREAM_ERR_ELPSD_OVRFLO;
    }

    assert(0 == GET_MASKED_BITS64(state_flag_stime, 42, 22));

    /* rwRecGetTcpState() masks off the IsIPv6 bit, so add it back */
    tcp_state = rwRecGetTcpState(rwrec) | (rwRecIsIPv6(rwrec) ? 0x80 : 0);

    state_flag_stime |= ((uint64_t)tcp_state) << 56;
    if (IPPROTO_TCP != rwRecGetProto(rwrec)) {
        state_flag_stime |= ((uint64_t)rwRecGetProto(rwrec)) << 48;
    } else if (0 == (tcp_state & SK_TCPSTATE_EXPANDED)) {
        state_flag_stime |= ((((uint64_t)rwRecGetFlags(rwrec)) << 48)
                             | (UINT64_C(1) << 42));
    } else {
        state_flag_stime |= ((((uint64_t)rwRecGetInitFlags(rwrec)) << 48)
                             | (UINT64_C(1) << 42));
        rflag_etime |= (((uint64_t)rwRecGetRestFlags(rwrec)) << 56);
    }

    memcpy(&ar[ 0], &state_flag_stime, sizeof(state_flag_stime));
    memcpy(&ar[ 8], &rflag_etime, sizeof(rflag_etime));

    return SKSTREAM_OK;
}

static void
rwpackUnpackStateFlagsTimesNano(
    rwGenericRec_V6    *rwrec,
    const uint8_t      *ar,
    sktime_t            file_start_time)
{
    uint64_t state_flag_stime;
    uint64_t rflag_etime;
    uint64_t file_start_nano;
    uint8_t tcp_state;
    uint8_t proto_iflags;
    uint8_t rest_flags;
    uint8_t is_tcp;

    file_start_nano = sktimeGetEpochNano(file_start_time);

/*
**    uint64_t      state_flag_stime;//  0- 7
**    // uint64_t     tcp_state : 8; //        TCP state machine info
**    // uint64_t     pro_iflags: 8; //        is_tcp==0: Protocol; else:
**                                   //          EXPANDED==0:TCPflags/ALL pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**    // uint64_t     unused    : 5; //        Reserved
**    // uint64_t     is_tcp    : 1; //        1 if FLOW is TCP; 0 otherwise
**    // uint64_t     stime     :42; //        Start time:nsec offset from hour
*/

    memcpy(&state_flag_stime, &ar[ 0], sizeof(state_flag_stime));

    tcp_state = GET_MASKED_BITS64(state_flag_stime, 56, 8);
    proto_iflags = GET_MASKED_BITS64(state_flag_stime, 48, 8);
    is_tcp = GET_MASKED_BITS64(state_flag_stime, 42, 1);

    rwRecSetStartTime(
        rwrec,
        sktimeCreateFromNano(0,
                             (file_start_nano
                              + GET_MASKED_BITS64(state_flag_stime, 0, 42))));

/*
**    uint64_t      rflag_etime;     //  8-15
**    // uint64_t     rest_flags: 8; //        is_tcp==0: Empty; else
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**    // uint64_t     etime     :56; //        End time:nsec offset from hour
*/

    memcpy(&rflag_etime, &ar[ 8], sizeof(rflag_etime));

    rest_flags = GET_MASKED_BITS64(rflag_etime, 56, 8);

    rwRecSetEndTime(
        rwrec,
        sktimeCreateFromNano(0,
                             (file_start_nano
                              + GET_MASKED_BITS64(rflag_etime, 0, 56))));

#if SK_ENABLE_IPV6
    /* The IsIPv6 bit is a stowaway on the octet holding TcpState */
    if (tcp_state & 0x80) {
        rwRecSetIPv6(rwrec);
    }
#endif  /* SK_ENABLE_IPV6 */

    rwRecSetTcpState(rwrec, tcp_state);
    if (!is_tcp) {
        rwRecSetProto(rwrec, proto_iflags);
    } else if (0 == (tcp_state & SK_TCPSTATE_EXPANDED)) {
        rwRecSetProto(rwrec, IPPROTO_TCP);
        rwRecSetFlags(rwrec, proto_iflags);
    } else {
        rwRecSetProto(rwrec, IPPROTO_TCP);
        rwRecSetInitFlags(rwrec, proto_iflags);
        rwRecSetRestFlags(rwrec, rest_flags);
        rwRecSetFlags(rwrec, proto_iflags | rest_flags);
    }
}
#endif  /* RWPACK_STATE_FLAGS_TIMES_NANO */


#ifdef RWPACK_INTERFACES_16BIT
static int
rwpackPackInterfaces16bit(
    uint8_t                *input,
    uint8_t                *output,
    const rwGenericRec_V6  *rwrec)
{
    uint16_t tmp16;

    if (rwRecGetInput(rwrec) > UINT16_MAX
        || rwRecGetOutput(rwrec) > UINT16_MAX)
    {
        return SKSTREAM_ERR_SNMP_OVRFLO;
    }

    tmp16 = (uint16_t)rwRecGetInput(rwrec);
    memcpy(input, &tmp16, sizeof(uint16_t));

    tmp16 = (uint16_t)rwRecGetOutput(rwrec);
    memcpy(output, &tmp16, sizeof(uint16_t));

    return SKSTREAM_OK;
}

static void
rwpackUnpackInterfaces16bit(
    rwGenericRec_V6    *rwrec,
    const uint8_t      *input,
    const uint8_t      *output)
{
    uint16_t tmp16;

    memcpy(&tmp16, input, sizeof(tmp16));
    rwRecSetInput(rwrec, tmp16);

    memcpy(&tmp16, output, sizeof(tmp16));
    rwRecSetOutput(rwrec, tmp16);
}
#endif  /* #ifdef RWPACK_INTERFACES_16BIT */


#ifdef RWPACK_PKTS_BYTES_32BIT
static int
rwpackPackPacketsBytes32bit(
    uint8_t                *packets,
    uint8_t                *bytes,
    const rwGenericRec_V6  *rwrec)
{
    uint32_t tmp32;

    if (rwRecGetPkts(rwrec) > UINT32_MAX) {
        return SKSTREAM_ERR_PKTS_OVRFLO;
    }
    if (rwRecGetBytes(rwrec) > UINT32_MAX) {
        return SKSTREAM_ERR_BYTES_OVRFLO;
    }

    tmp32 = (uint32_t)rwRecGetPkts(rwrec);
    memcpy(packets, &tmp32, sizeof(tmp32));

    tmp32 = rwRecGetBytes(rwrec);
    memcpy(bytes, &tmp32, sizeof(tmp32));

    return SKSTREAM_OK;
}

/* flowcapio.c only needs the Pack version of this function */
#ifndef RWPACK_HIDE_UNPACK_PKTS_BYTES_32BIT
static void
rwpackUnpackPacketsBytes32bit(
    rwGenericRec_V6    *rwrec,
    const uint8_t      *packets,
    const uint8_t      *bytes)
{
    uint32_t tmp32;

    memcpy(&tmp32, packets, sizeof(tmp32));
    rwRecSetPkts(rwrec, tmp32);

    memcpy(&tmp32, bytes, sizeof(tmp32));
    rwRecSetBytes(rwrec, tmp32);
}
#endif  /* #ifndef RWPACK_HIDE_UNPACK_PKTS_BYTES_32BIT */
#endif  /* #ifdef RWPACK_PKTS_BYTES_32BIT */


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
