/*
** Copyright (C) 2007-2013 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  skipfix.c
**
**    SiLK Flow Record / IPFIX translation core
**
**    Brian Trammell
**    February 2007
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: skipfix.c 2b9a194ba4b5 2013-07-29 20:27:45Z mthomas $");

#include <silk/rwrec.h>
#include <silk/skipaddr.h>
#include <silk/skipfix.h>
#include <silk/sklog.h>
#include <silk/skvector.h>
#include <silk/utils.h>

#ifdef SKIPFIX_TRACE_LEVEL
#define TRACEMSG_LEVEL 1
#endif
#define TRACEMSG(x)  TRACEMSG_TO_TRACEMSGLVL(1, x)
#include <silk/sktracemsg.h>


/* LOCAL DEFINES AND TYPEDEFS */

/* The IPFIX Private Enterprise Number for CERT */
#define IPFIX_CERT_PEN  6871

/* Extenal Template ID used for SiLK Flows written by rwsilk2ipfix.
 * This is defined in skipfix.h. */
/* #define SKI_FLOW_TID        0xAFEA */

/* Internal Template ID for extended SiLK flows. */
#define SKI_EXTFLOW_TID         0xAFEB

/* Internal Template ID for TCP information. */
#define SKI_TCP_STML_TID        0xAFEC

/* Bit in Template ID that yaf sets for templates containing reverse
 * elements */
#define SKI_YAF_REVERSE_BIT     0x0010

/* Template ID used by yaf for a yaf stats option record */
#define SKI_YAF_STATS_TID       0xD000

/* Template ID used by yaf for a subTemplateMultiList containing only
 * forward TCP flags information. */
#define SKI_YAF_TCP_FLOW_TID    0xC003

/* Values to pass to skiSessionNew() */
#define DO_NOT_SAVE_SESSION 0
#define SAVE_SESSION        1

/*
 *  val = CLAMP_VAL(val, max);
 *
 *    If 'val' is greater then 'max', return 'max'.  Otherwise,
 *    return (max & val).
 */
#define CLAMP_VAL(val, max) \
    (((val) > (max)) ? (max) : ((max) & (val)))

/* One more than UINT32_MAX */
#define ROLLOVER32 ((intmax_t)UINT32_MAX + 1)

/*
 *    For NetFlow V9, when the absolute value of the magnitude of the
 *    difference between the sysUpTime and the flowStartSysUpTime is
 *    greater than this value (in milliseconds), assume one of the
 *    values has rolled over.
 */
#define MAXIMUM_FLOW_TIME_DEVIATION  ((intmax_t)INT32_MAX)

/* Define the IPFIX information elements in IPFIX_CERT_PEN space for
 * SiLK */
static fbInfoElement_t ski_info_elements[] = {
    /* Extra fields produced by yaf for SiLK records */
    FB_IE_INIT("initialTCPFlags",              IPFIX_CERT_PEN, 14,  1,
               FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),
    FB_IE_INIT("unionTCPFlags",                IPFIX_CERT_PEN, 15,  1,
               FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),
    FB_IE_INIT("reverseFlowDeltaMilliseconds", IPFIX_CERT_PEN, 21,  4,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkFlowType",                 IPFIX_CERT_PEN, 30,  1,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkFlowSensor",               IPFIX_CERT_PEN, 31,  2,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkTCPState",                 IPFIX_CERT_PEN, 32,  1,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("silkAppLabel",                 IPFIX_CERT_PEN, 33,  2,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("flowAttributes",               IPFIX_CERT_PEN, 40,  2,
               FB_IE_F_ENDIAN | FB_IE_F_REVERSIBLE),

    /* Extra fields produced by yaf for yaf statistics */
    FB_IE_INIT("expiredFragmentCount",         IPFIX_CERT_PEN, 100, 4,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("assembledFragmentCount",       IPFIX_CERT_PEN, 101, 4,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("meanFlowRate",                 IPFIX_CERT_PEN, 102, 4,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("meanPacketRate",               IPFIX_CERT_PEN, 103, 4,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("flowTableFlushEventCount",     IPFIX_CERT_PEN, 104, 4,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("flowTablePeakCount",           IPFIX_CERT_PEN, 105, 4,
               FB_IE_F_ENDIAN),
    FB_IE_NULL
};


/* These are IPFIX information elements in the standard space.
 * However, these elements are not defined in all versions of
 * libfixbuf. */
static fbInfoElement_t ski_std_info_elements[] = {
    /* the following are defined as of libfixbuf-1.1.0 */
    FB_IE_INIT("initiatorOctets",   0, 231, 8, FB_IE_F_ENDIAN),
    FB_IE_INIT("responderOctets",   0, 232, 8, FB_IE_F_ENDIAN),
    FB_IE_INIT("firewallEvent",     0, 233, 1, FB_IE_F_ENDIAN),

#ifndef FB_CISCO_ASA_EVENT_ID
    /* from include/fixbuf/public.h */
#define FB_CISCO_ASA_EVENT_ID                  9998
#define FB_CISCO_ASA_EVENT_XTRA                9997
    FB_IE_INIT("NF_F_FW_EVENT",     0, FB_CISCO_ASA_EVENT_ID,   1,
               FB_IE_F_ENDIAN),
    FB_IE_INIT("NF_F_FW_EXT_EVENT", 0, FB_CISCO_ASA_EVENT_XTRA, 2,
               FB_IE_F_ENDIAN),
#endif

    /* the following are defined as of libfixbuf-1.3.0 */
    FB_IE_INIT("initiatorPackets", 0, 298, 8, FB_IE_F_ENDIAN),
    FB_IE_INIT("responderPackets", 0, 299, 8, FB_IE_F_ENDIAN),
    FB_IE_NULL
};


/* Bytes of padding to add to ski_flow_spec and ski_rwrec_st to get a
 * multiple of 64bits */
#define SKI_FLOW_SPEC_PADDING  6

/*
 * This is an IPFIX encoding of a standard SiLK Flow record (rwRec);
 * it is used for export by rwsilk2ipfix where it has Template ID
 * SKI_FLOW_TID.
 *
 * This also becomes part of the "Extended" record, ski_extflow_spec,
 * that is used for import.
 *
 * Keep this in sync with the ski_rwrec_t below.  Use
 * SKI_FLOW_SPEC_PADDING to pad to 64bits. */
static fbInfoElementSpec_t ski_flow_spec[] = {
    /* Millisecond start and end (epoch) (native time) */
    { (char*)"flowStartMilliseconds",              0, 0 },
    { (char*)"flowEndMilliseconds",                0, 0 },
    /* 4-tuple */
    { (char*)"sourceIPv6Address",                  0, 0 },
    { (char*)"destinationIPv6Address",             0, 0 },
    { (char*)"sourceIPv4Address",                  0, 0 },
    { (char*)"destinationIPv4Address",             0, 0 },
    { (char*)"sourceTransportPort",                0, 0 },
    { (char*)"destinationTransportPort",           0, 0 },
    /* Router interface information */
    { (char*)"ipNextHopIPv4Address",               0, 0 },
    { (char*)"ipNextHopIPv6Address",               0, 0 },
    { (char*)"ingressInterface",                   0, 0 },
    { (char*)"egressInterface",                    0, 0 },
    /* Counters (reduced length encoding for SiLK) */
    { (char*)"packetDeltaCount",                   0, 0 },
    { (char*)"octetDeltaCount",                    0, 0 },
    /* Protocol; sensor information */
    { (char*)"protocolIdentifier",                 0, 0 },
    { (char*)"silkFlowType",                       0, 0 },
    { (char*)"silkFlowSensor",                     0, 0 },
    /* Flags */
    { (char*)"tcpControlBits",                     0, 0 },
    { (char*)"initialTCPFlags",                    0, 0 },
    { (char*)"unionTCPFlags",                      0, 0 },
    { (char*)"silkTCPState",                       0, 0 },
    { (char*)"silkAppLabel",                       0, 0 },
    /* pad record to 64-bit boundary */
#if SKI_FLOW_SPEC_PADDING != 0
    { (char*)"paddingOctets",  SKI_FLOW_SPEC_PADDING, 0 },
#endif
    FB_IESPEC_NULL
};

/* Keep this in sync with the ski_flow_spec above. Pad to 64bits */
typedef struct ski_rwrec_st {
    uint64_t        flowStartMilliseconds;          /*   0-  7 */
    uint64_t        flowEndMilliseconds;            /*   8- 15 */

    uint8_t         sourceIPv6Address[16];          /*  16- 31 */
    uint8_t         destinationIPv6Address[16];     /*  32- 47 */

    uint32_t        sourceIPv4Address;              /*  48- 51 */
    uint32_t        destinationIPv4Address;         /*  52- 55 */

    uint16_t        sourceTransportPort;            /*  56- 57 */
    uint16_t        destinationTransportPort;       /*  58- 59 */

    uint32_t        ipNextHopIPv4Address;           /*  60- 63 */
    uint8_t         ipNextHopIPv6Address[16];       /*  64- 79 */
    uint32_t        ingressInterface;               /*  80- 83 */
    uint32_t        egressInterface;                /*  84- 87 */

    uint64_t        packetDeltaCount;               /*  88- 95 */
    uint64_t        octetDeltaCount;                /*  96-103 */

    uint8_t         protocolIdentifier;             /* 104     */
    flowtypeID_t    silkFlowType;                   /* 105     */
    sensorID_t      silkFlowSensor;                 /* 106-107 */

    uint8_t         tcpControlBits;                 /* 108     */
    uint8_t         initialTCPFlags;                /* 109     */
    uint8_t         unionTCPFlags;                  /* 110     */
    uint8_t         silkTCPState;                   /* 111     */
    uint16_t        silkAppLabel;                   /* 112-113 */
#if SKI_FLOW_SPEC_PADDING != 0
    uint8_t         pad[SKI_FLOW_SPEC_PADDING];     /* 114-119 */
#endif
} ski_rwrec_t;



/* Bytes of padding to add to ski_extflow_spec and ski_extrwrec_st to
 * get a multiple of 64bits */
#define SKI_EXTFLOW_SPEC_PADDING  6

/* These are additional IPFIX fields (or different encodings of the
 * SiLK fields) that we may get from other flowmeters.  They will be
 * appended to the ski_flow_spec (above) to create the complete
 * ski_extrwrec_t (defined below).  This has Template ID
 * SKI_EXTFLOW_TID. */
static fbInfoElementSpec_t ski_extflow_spec[] = {
    /* Total counter support */
    { (char*)"packetTotalCount",                   0, 0 },
    { (char*)"octetTotalCount",                    0, 0 },
    { (char*)"initiatorPackets",                   0, 0 },
    { (char*)"initiatorOctets",                    0, 0 },
    /* Reverse counter support */
    { (char*)"reversePacketDeltaCount",            0, 0 },
    { (char*)"reverseOctetDeltaCount",             0, 0 },
    { (char*)"reversePacketTotalCount",            0, 0 },
    { (char*)"reverseOctetTotalCount",             0, 0 },
    { (char*)"responderPackets",                   0, 0 },
    { (char*)"responderOctets",                    0, 0 },
    /* Microsecond start and end (RFC1305-style) (extended time) */
    { (char*)"flowStartMicroseconds",              0, 0 },
    { (char*)"flowEndMicroseconds",                0, 0 },
    /* SysUpTime, used to handle Netflow v9 SysUpTime offset times */
    { (char*)"systemInitTimeMilliseconds",         0, 0 },
    /* Second start and end (extended time) */
    { (char*)"flowStartSeconds",                   0, 0 },
    { (char*)"flowEndSeconds",                     0, 0 },
    /* Flow durations (extended time) */
    { (char*)"flowDurationMicroseconds",           0, 0 },
    { (char*)"flowDurationMilliseconds",           0, 0 },
    /* Microsecond delta start and end (extended time) */
    { (char*)"flowStartDeltaMicroseconds",         0, 0 },
    { (char*)"flowEndDeltaMicroseconds",           0, 0 },
    /* Initial packet roundtrip */
    { (char*)"reverseFlowDeltaMilliseconds",       0, 0 },
    /* SysUpTime-based fields */
    { (char*)"flowStartSysUpTime",                 0, 0 },
    { (char*)"flowEndSysUpTime",                   0, 0 },
    /* Reverse flags */
    { (char*)"reverseTcpControlBits",              0, 0 },
    { (char*)"reverseInitialTCPFlags",             0, 0 },
    { (char*)"reverseUnionTCPFlags",               0, 0 },
    /* End reason */
    { (char*)"flowEndReason",                      0, 0 },
    /* Flow attributes */
    { (char*)"flowAttributes",                     0, 0 },
    { (char*)"reverseFlowAttributes",              0, 0 },
    /* Vlan IDs */
    { (char*)"vlanId",                             0, 0 },
    { (char*)"postVlanId",                         0, 0 },
    { (char*)"reverseVlanId",                      0, 0 },
    { (char*)"reversePostVlanId",                  0, 0 },
    /* TOS */
    { (char*)"ipClassOfService",                   0, 0 },
    { (char*)"reverseIpClassOfService",            0, 0 },
    /* Firewall events */
    { (char*)"firewallEvent",                      0, 0 },
    { (char*)"NF_F_FW_EVENT",                      0, 0 },
    { (char*)"NF_F_FW_EXT_EVENT",                  0, 0 },
    /* pad record to 64-bit boundary. */
#if SKI_EXTFLOW_SPEC_PADDING != 0
    { (char*)"paddingOctets", SKI_EXTFLOW_SPEC_PADDING, 0 },
#endif
    { (char*)"subTemplateMultiList",               0, 0 },
    FB_IESPEC_NULL
};

/* Keep in sync with the ski_extflow_spec[] above. */
typedef struct ski_extrwrec_st {
    ski_rwrec_t     rw;                             /*   0-119 */

    uint64_t        packetTotalCount;               /* 120-127 */
    uint64_t        octetTotalCount;                /* 128-135 */
    uint64_t        initiatorPackets;               /* 136-143 */
    uint64_t        initiatorOctets;                /* 144-151 */

    uint64_t        reversePacketDeltaCount;        /* 152-159 */
    uint64_t        reverseOctetDeltaCount;         /* 160-167 */
    uint64_t        reversePacketTotalCount;        /* 168-175 */
    uint64_t        reverseOctetTotalCount;         /* 176-183 */
    uint64_t        responderPackets;               /* 184-191 */
    uint64_t        responderOctets;                /* 192-199 */

    /* Time can be represented in many different formats: */

    /* start time as NTP (RFC1305); may either have end Time in same
     * format or as an flowDurationMicroseconds value. */
    uint64_t        flowStartMicroseconds;          /* 200-207 */
    uint64_t        flowEndMicroseconds;            /* 208-215 */

    /* SysUpTime: used for flow{Start,End}SysUpTime calculations.
     * Needed to support Netflow v9 in particular. */
    uint64_t        systemInitTimeMilliseconds;     /* 216-223 */

    /* start time and end times as seconds since UNIX epoch. no
     * flowDuration field */
    uint32_t        flowStartSeconds;               /* 224-227 */
    uint32_t        flowEndSeconds;                 /* 228-231 */

    /* elapsed time as either microsec or millisec.  used when the
     * flowEnd time is not given. */
    uint32_t        flowDurationMicroseconds;       /* 232-235 */
    uint32_t        flowDurationMilliseconds;       /* 236-239 */

    /* start time as delta (negative microsec offsets) from the export
     * time; may either have end time in same format or a
     * flowDurationMicroseconds value */
    uint32_t        flowStartDeltaMicroseconds;     /* 240-243 */
    uint32_t        flowEndDeltaMicroseconds;       /* 244-247 */

    /* start time of reverse flow, as millisec offset from start time
     * of forward flow */
    uint32_t        reverseFlowDeltaMilliseconds;   /* 248-251 */

    /* Start and end time as delta from the system init time.  Needed
     * to support Netflow v9. */
    uint32_t        flowStartSysUpTime;             /* 252-255 */
    uint32_t        flowEndSysUpTime;               /* 256-259 */

    /* Flags for the reverse flow: */
    uint8_t         reverseTcpControlBits;          /* 260     */
    uint8_t         reverseInitialTCPFlags;         /* 261     */
    uint8_t         reverseUnionTCPFlags;           /* 262     */

    uint8_t         flowEndReason;                  /* 263     */

    /* Reverse flow attribute flags */
    uint16_t        flowAttributes;                 /* 264-265 */
    uint16_t        reverseFlowAttributes;          /* 266-267 */

    /* vlan IDs */
    uint16_t        vlanId;                         /* 268-269 */
    uint16_t        postVlanId;                     /* 270-271 */
    uint16_t        reverseVlanId;                  /* 272-273 */
    uint16_t        reversePostVlanId;              /* 274-275 */

    /* TOS */
    uint8_t         ipClassOfService;               /* 276     */
    uint8_t         reverseIpClassOfService;        /* 277     */

    /* Firewall events */
    uint8_t          firewallEvent;                 /* 278     */
    uint8_t          NF_F_FW_EVENT;                 /* 279     */
    uint16_t         NF_F_FW_EXT_EVENT;             /* 280-281 */

    /* padding */
#if SKI_EXTFLOW_SPEC_PADDING != 0
    uint8_t         pad[SKI_EXTFLOW_SPEC_PADDING];  /* 282-287 */
#endif

    /* TCP flags from yaf (when it is run without --silk) */
    fbSubTemplateMultiList_t stml;

} ski_extrwrec_t;

/* Support for reading TCP flags from an IPFIX subTemplateMultiList as
 * exported by YAF.  This has Template ID SKI_TCP_STML_TID.
 *
 * Keep this in sync with the ski_tcp_stml_t defined below. */
static fbInfoElementSpec_t ski_tcp_stml_spec[] = {
    { (char*)"initialTCPFlags",                    0, 0 },
    { (char*)"unionTCPFlags",                      0, 0 },
    { (char*)"reverseInitialTCPFlags",             0, 0 },
    { (char*)"reverseUnionTCPFlags",               0, 0 },
    FB_IESPEC_NULL
};

/* Keep in sync with the ski_tcp_stml_spec[] defined above. */
typedef struct ski_tcp_stml_st {
    uint8_t         initialTCPFlags;
    uint8_t         unionTCPFlags;
    uint8_t         reverseInitialTCPFlags;
    uint8_t         reverseUnionTCPFlags;
} ski_tcp_stml_t;


/* This lists statistics values that yaf may export. */
/* Keep this in sync with ski_yaf_stats_t defined in skipfix.h */
static fbInfoElementSpec_t ski_yaf_stats_option_spec[] = {
    { (char*)"systemInitTimeMilliseconds",         0, 0 },
    { (char*)"exportedFlowRecordTotalCount",       0, 0 },
    { (char*)"packetTotalCount",                   0, 0 },
    { (char*)"droppedPacketTotalCount",            0, 0 },
    { (char*)"ignoredPacketTotalCount",            0, 0 },
    { (char*)"notSentPacketTotalCount",            0, 0 },
    { (char*)"expiredFragmentCount",               0, 0 },
#if 0
    { (char*)"assembledFragmentCount",             0, 0 },
    { (char*)"flowTableFlushEventCount",           0, 0 },
    { (char*)"flowTablePeakCount",                 0, 0 },
    { (char*)"meanFlowRate",                       0, 0 },
    { (char*)"meanPacketRate",                     0, 0 },
    { (char*)"exporterIPv4Address",                0, 0 },
#endif  /* 0 */
    { (char*)"exportingProcessId",                 0, 0 },
#if SKI_YAF_STATS_PADDING != 0
    { (char*)"paddingOctets", SKI_YAF_STATS_PADDING,  0 },
#endif
    FB_IESPEC_NULL
};

/* Values for the flowEndReason. this first set is defined by the
 * IPFIX spec */
#define SKI_END_IDLE            1
#define SKI_END_ACTIVE          2
#define SKI_END_CLOSED          3
#define SKI_END_FORCED          4
#define SKI_END_RESOURCE        5

/* SiLK will ignore flows with a flowEndReason of
 * SKI_END_YAF_INTERMEDIATE_FLOW */
#define SKI_END_YAF_INTERMEDIATE_FLOW 0x1F

/* Mask for the values of flowEndReason: want to ignore the next bit */
#define SKI_END_MASK            0x1f

/* Bits from flowEndReason: whether flow is a continuation */
#define SKI_END_ISCONT          0x80


/* Bits from flowAttributes */
#define SKI_FLOW_ATTRIBUTE_UNIFORM_PACKET_SIZE 0x01

/* These are individual Information Elements whose presence we want to
 * check for via fbTemplateContainsElementByName(). */
static struct ies_st {
    fbInfoElementSpec_t NF_F_FW_EVENT;
    fbInfoElementSpec_t NF_F_FW_EXT_EVENT;
    fbInfoElementSpec_t firewallEvent;
    fbInfoElementSpec_t flowStartSysUpTime;
    fbInfoElementSpec_t postVlanId;
    fbInfoElementSpec_t reverseInitialTCPFlags;
    fbInfoElementSpec_t reverseTcpControlBits;
    fbInfoElementSpec_t reverseVlanId;
    fbInfoElementSpec_t systemInitTimeMilliseconds;
} ies = {
    {(char*)"NF_F_FW_EVENT",                0, 0},
    {(char*)"NF_F_FW_EXT_EVENT",            0, 0},
    {(char*)"firewallEvent",                0, 0},
    {(char*)"flowStartSysUpTime",           0, 0},
    {(char*)"postVlanId",                   0, 0},
    {(char*)"reverseInitialTCPFlags",       0, 0},
    {(char*)"reverseTcpControlBits",        0, 0},
    {(char*)"reverseVlanId",                0, 0},
    {(char*)"systemInitTimeMilliseconds",   0, 0}
};

static fbInfoModel_t *ski_model = NULL;
static sk_vector_t *session_list = NULL;


/* FUNCTION DEFINITIONS */

static fbInfoModel_t *skiInfoModel(
    void)
{
    if (!ski_model) {
        ski_model = fbInfoModelAlloc();
        fbInfoModelAddElementArray(ski_model, ski_info_elements);
        fbInfoModelAddElementArray(ski_model, ski_std_info_elements);
    }

    return ski_model;
}


static void skiInfoModelFree(
    void)
{
    if (ski_model) {
        fbInfoModelFree(ski_model);
        ski_model = NULL;
    }
}


/*
 *     This callback is invoked whenever the session receives a new
 *     template.  The purpose of the callback is the tell fixbuf how
 *     to process items in a subTemplateMultiList.  We tell fixbuf to
 *     map from the two templates that yaf uses for TCP flags (one of
 *     which has reverse elements and one of which does not) to the
 *     struct used in this file.
 *
 *     This function must have the signature defined by the
 *     'fbNewTemplateCallback_fn' typedef.
 */
void skiTemplateCallback(
    fbSession_t            *session,
    uint16_t                tid,
    fbTemplate_t    UNUSED(*tmpl))
{
    TRACEMSG(("Template callback called for template %0x04X", tid));

    if (SKI_YAF_TCP_FLOW_TID == (tid & ~SKI_YAF_REVERSE_BIT)) {
        fbSessionAddTemplatePair(session, tid, SKI_TCP_STML_TID);
    } else {
        /* ignore */
        fbSessionAddTemplatePair(session, tid, 0);
    }
}


static fbSession_t *skiSessionNew(
    int status)
{
    fbInfoModel_t *model;
    fbSession_t *session;

    model = skiInfoModel();
    if (model == NULL) {
        return NULL;
    }
    if (status == SAVE_SESSION && !session_list) {
        session_list = skVectorNew(sizeof(fbSession_t *));
        if (session_list == NULL) {
            return NULL;
        }
    }
    session = fbSessionAlloc(model);
    if (session == NULL) {
        return NULL;
    }
    if (status == SAVE_SESSION
        && skVectorAppendValue(session_list, &session) != 0)
    {
        fbSessionFree(session);
        return NULL;
    }
    return session;
}


static void skiSessionsFree(
    void)
{
    size_t i;
    fbSession_t *session;

    if (session_list) {
        for (i = 0; i < skVectorGetCount(session_list); i++) {
            skVectorGetValue(&session, session_list, i);
            fbSessionFree(session);
        }
        skVectorDestroy(session_list);
        session_list = NULL;
    }
}


/*
 *    Initialize an fbSession object that reads from either the
 *    network or from a file.
 */
static int skiSessionInitReader(
    fbSession_t            *session,
    GError                **err)
{
    fbInfoModel_t   *model = skiInfoModel();
    fbTemplate_t    *tmpl = NULL;

    /* Add the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto ERROR;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_FLOW_TID, tmpl, err)) {
        goto ERROR;
    }

    /* Add the extended record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto ERROR;
    }
    if (!fbTemplateAppendSpecArray(tmpl, ski_extflow_spec, 0, err)) {
        goto ERROR;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_EXTFLOW_TID, tmpl, err)) {
        goto ERROR;
    }

    /* Add the TCP record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_tcp_stml_spec, 0, err)) {
        goto ERROR;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_TCP_STML_TID, tmpl, err)) {
        goto ERROR;
    }

    /* Add the yaf stats record template  */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_yaf_stats_option_spec, 0, err)) {
        goto ERROR;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_YAF_STATS_TID, tmpl, err)) {
        goto ERROR;
    }

    return 1;

  ERROR:
    fbTemplateFreeUnused(tmpl);
    return 0;
}


void skiTeardown(
    void)
{
    skiInfoModelFree();
    skiSessionsFree();
}


/* **************************************************************
 * *****  Support for reading/import
 */


fbListener_t *skiCreateListener(
    fbConnSpec_t           *spec,
    fbListenerAppInit_fn    appinit,
    fbListenerAppFree_fn    appfree,
    GError                **err)
{
    fbSession_t     *session = NULL;

    /* Allocate a session.  Save it for freeing during teardown, as it
     * is not owned by the listener, but is used by it. */
    session = skiSessionNew(SAVE_SESSION);
    if (NULL == session) {
        return NULL;
    }
    /* Initialize session for reading */
    if (!skiSessionInitReader(session, err)) {
        return NULL;
    }

    /* Allocate a listener */
    return fbListenerAlloc(spec, session, appinit, appfree, err);
}


fBuf_t *skiCreateReadBufferForFP(
    FILE           *fp,
    GError        **err)
{
    return skiCreateReadBuffer(fbCollectorAllocFP(NULL, fp), err);
}


fBuf_t *skiCreateReadBuffer(
    fbCollector_t  *collector,
    GError        **err)
{
    fbSession_t *session = NULL;
    fBuf_t *fbuf;

    /* Allocate a session.  The session will be owned by the fbuf, so
     * don't save it for later freeing. */
    session = skiSessionNew(DO_NOT_SAVE_SESSION);
    if (NULL == session) {
        return NULL;
    }
    /* Initialize session for reading */
    if (!skiSessionInitReader(session, err)) {
        fbSessionFree(session);
        return NULL;
    }

    /* Create a buffer with the session and supplied collector */
    fbuf = fBufAllocForCollection(session, collector);
    if (NULL == fbuf) {
        fbSessionFree(session);
        return NULL;
    }

    /* Make certain the fbuf has an internal template */
    if (!fBufSetInternalTemplate(fbuf, SKI_FLOW_TID, err)) {
        fBufFree(fbuf);
        return NULL;
    }

    /* Invoke a callback when a new template arrives that tells fixbuf
     * how to map from the subTemplateMultiList used by YAF for TCP
     * information to our internal strucure. */
    fbSessionAddTemplateCallback(session, &skiTemplateCallback);

    return fbuf;
}


/* Convert an NTP timestamp (RFC1305) to epoch millisecond */
static uint64_t skiNTPDecode(
    uint64_t        ntp)
{
    double          dntp;
    uint64_t        millis;

    if (!ntp) {
        return 0;
    }

    dntp = (ntp & UINT64_C(0xFFFFFFFF00000000)) >> 32;
    dntp += ((ntp & UINT64_C(0x00000000FFFFFFFF)) * 1.0) / (UINT64_C(2) << 32);
    millis = (uint64_t)(dntp * 1000);
    return millis;
}


/* Print a message saying why a flow was ignored */
static void skiFlowIgnored(
    const ski_extrwrec_t   *fixrec,
    const char             *reason)
{
    char sipbuf[64];
    char dipbuf[64];

    if (!SK_IPV6_IS_ZERO(fixrec->rw.sourceIPv6Address)) {
#ifdef SK_HAVE_INET_NTOP
        if (!inet_ntop(AF_INET6, &fixrec->rw.sourceIPv6Address,
                       sipbuf, sizeof(sipbuf)))
#endif
        {
            strcpy(sipbuf, "unknown-v6");
        }
    } else {
        num2dot_r(fixrec->rw.sourceIPv4Address, sipbuf);
    }
    if (!SK_IPV6_IS_ZERO(fixrec->rw.destinationIPv6Address)) {
#ifdef SK_HAVE_INET_NTOP
        if (!inet_ntop(AF_INET6, &fixrec->rw.destinationIPv6Address,
                       dipbuf, sizeof(dipbuf)))
#endif
        {
            strcpy(dipbuf, "unknown-v6");
        }
    } else {
        num2dot_r(fixrec->rw.destinationIPv4Address, dipbuf);
    }

    INFOMSG(("IGNORED|%s|%s|%u|%u|%u|%" PRIu64 "|%" PRIu64 "|%s|"),
            sipbuf, dipbuf, fixrec->rw.sourceTransportPort,
            fixrec->rw.destinationTransportPort,fixrec->rw.protocolIdentifier,
            ((fixrec->rw.packetDeltaCount)
             ? fixrec->rw.packetDeltaCount
             : ((fixrec->packetTotalCount)
                ? fixrec->packetTotalCount
                : fixrec->initiatorPackets)),
            ((fixrec->rw.octetDeltaCount)
             ? fixrec->rw.octetDeltaCount
             : ((fixrec->octetTotalCount)
                ? fixrec->octetTotalCount
                : fixrec->initiatorOctets)),
            reason);
}


/* get the type of the next record */
ski_rectype_t skiGetNextRecordType(
    fBuf_t         *fbuf,
    GError        **err)
{
    fbTemplate_t *tmpl;
    uint16_t tid;

    tmpl = fBufNextCollectionTemplate(fbuf, &tid, err);
    if (tmpl == NULL) {
        return SKI_RECTYPE_ERROR;
    }

    /* Get Options Elements - if not 0 - Stats Template */
    if (fbTemplateGetOptionsScope(tmpl)) {
        if (tid == SKI_YAF_STATS_TID) {
            return SKI_RECTYPE_STATS;
        }
        return SKI_RECTYPE_UNKNOWN;
    }
    return SKI_RECTYPE_FLOW;
}

gboolean skiRwNextStats(
    fBuf_t           *fbuf,
    ski_yaf_stats_t  *stats,
    GError          **err)
{
    size_t len;

    /* Set internal template to read an yaf stats record */
    if (!fBufSetInternalTemplate(fbuf, SKI_YAF_STATS_TID, err)) {
        return FALSE;
    }

    memset(stats, 0, sizeof(*stats));
    len = sizeof(*stats);

    if (!fBufNext(fbuf, (uint8_t *)stats, &len, err)) {
        return FALSE;
    }

    return TRUE;
}

/* Get a record from the libfixbuf buffer 'fbuf' and fill in the
 * forward and reverse flow records, 'rec' and 'revRec'.  Return 1 if
 * the record is uni-flow, 2 if it is bi-flow, 0 if the record is to
 * be ignored, -1 if there is an error. */
int skiRwNextRecord(
    fBuf_t     *fbuf,
    rwRec      *rec,
    rwRec      *revRec,
    GError    **err,
    int         vlan_interfaces)
{
    fbTemplate_t *tmpl = NULL;
    fbSubTemplateMultiListEntry_t *stml;
    ski_extrwrec_t fixrec;
    size_t len;
    uint64_t sTime, eTime;
    uint64_t pkts, bytes;
    uint64_t rev_pkts, rev_bytes;
    uint8_t tcp_state;
    uint8_t tcp_flags;
    int have_tcp_stml = 0;

    /* Clear out the record */
    RWREC_CLEAR(rec);

    /* Set internal template to read an extended flow record */
    if (!fBufSetInternalTemplate(fbuf, SKI_EXTFLOW_TID, err)) {
        return -1;
    }

    /* Get the next record */
    len = sizeof(fixrec);
    if (!fBufNext(fbuf, (uint8_t *)&fixrec, &len, err)) {
        return -1;
    }

    if ((fixrec.flowEndReason & SKI_END_MASK) == SKI_END_YAF_INTERMEDIATE_FLOW)
    {
        TRACEMSG(("Ignored YAF intermediate uniflow"));
        return 0;
    }

#if !SK_ENABLE_IPV6
    /* SiLK was compiled without IPv6 support.  Ignore IPv6 records. */
    if (!SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address)
        || !SK_IPV6_IS_ZERO(fixrec.rw.destinationIPv6Address))
    {
        skiFlowIgnored(&fixrec, "IPv6 record");
        return 0;
    }
#endif  /* !SK_ENABLE_IPV6 */

    /* Get the template used for the last record */
    tmpl = fBufGetCollectionTemplate(fbuf, NULL);

    /* Ignore records that represent firewall status */
    if (fbTemplateContainsElementByName(tmpl, &ies.firewallEvent)
        || fbTemplateContainsElementByName(tmpl, &ies.NF_F_FW_EVENT)
        || fbTemplateContainsElementByName(tmpl, &ies.NF_F_FW_EXT_EVENT))
    {
        char msg[64];
        snprintf(msg, sizeof(msg), "firewallEvent=%u,extended=%u",
                 (fixrec.firewallEvent
                  ? fixrec.firewallEvent : fixrec.NF_F_FW_EVENT),
                 fixrec.NF_F_FW_EXT_EVENT);
        skiFlowIgnored(&fixrec, msg);
        return 0;
    }

    /* FIXME.  What if the record has a flowDirection field that is
     * set to egress (0x01)?  Shouldn't we handle that by reversing
     * the record?  Or has fixbuf done that for us? */

    /* Get the forward and reverse packet and byte counts (run the
     * Gauntlet of Volume). */
    pkts = ((fixrec.rw.packetDeltaCount)
            ? fixrec.rw.packetDeltaCount
            : ((fixrec.packetTotalCount)
               ? fixrec.packetTotalCount
               : fixrec.initiatorPackets));
    bytes = ((fixrec.rw.octetDeltaCount)
             ? fixrec.rw.octetDeltaCount
             : ((fixrec.octetTotalCount)
                ? fixrec.octetTotalCount
                : fixrec.initiatorOctets));

    rev_pkts = ((fixrec.reversePacketDeltaCount)
                ? fixrec.reversePacketDeltaCount
                : ((fixrec.reversePacketTotalCount)
                   ? fixrec.reversePacketTotalCount
                   : fixrec.responderPackets));
    rev_bytes = ((fixrec.reverseOctetDeltaCount)
                 ? fixrec.reverseOctetDeltaCount
                 : ((fixrec.reverseOctetTotalCount)
                    ? fixrec.reverseOctetTotalCount
                    : fixrec.responderOctets));

    if (0 == bytes && 0 == rev_bytes) {
        skiFlowIgnored(&fixrec, "no forward/reverse octets");
        return 0;
    }
    if (0 == pkts && 0 == rev_pkts)
#if !(SK_ENABLE_ASA_ZERO_PACKET_HACK && SK_ENABLE_NETFLOW9)
    {
        /* Ignore records with no volume. */
        skiFlowIgnored(&fixrec, "no forward/reverse packets");
        return 0;
    }
#else
    {
        /* attempt to handle NetFlowV9 records from an ASA router that
         * have no packet count.  The code assumes all records from an
         * ASA have a byte count, though this is not always true. */
        if (!fixrec.rw.sourceIPv4Address
            && SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address))
        {
            /* Record has no volume and no source IP. Ignore it. */
            skiFlowIgnored(&fixrec,"no forward/reverse pkts and no source IP");
            return 0;
        }
        if (bytes) {
            /* there is a forward byte count */
            if (0 == pkts) {
                TRACEMSG(("Setting forward packets to 1"));
                pkts = 1;
            }
        }
        if (rev_bytes) {
            /* there is a reverse byte count */
            if (0 == rev_pkts) {
                TRACEMSG(("Setting reverse packets to 1"));
                rev_pkts = 1;
            }
        }
    }
#endif  /* SK_ENABLE_ASA_ZERO_PACKET_HACK && SK_ENABLE_NETFLOW9 */

    /* If the TCP flags are in a subTemplateMultiList, copy them from
     * the list and into the record.  The fixbuf.stml gets initialized
     * by the call to fBufNext().*/
    stml = NULL;
    while ((stml = fbSubTemplateMultiListGetNextEntry(&fixrec.stml, stml))) {
        if (SKI_TCP_STML_TID != stml->tmplID) {
            fbSubTemplateMultiListEntryNextDataPtr(stml, NULL);
        } else {
            ski_tcp_stml_t *tcp = NULL;
            tcp = ((ski_tcp_stml_t*)
                   fbSubTemplateMultiListEntryNextDataPtr(stml, tcp));
            fixrec.rw.initialTCPFlags = tcp->initialTCPFlags;
            fixrec.rw.unionTCPFlags = tcp->unionTCPFlags;
            fixrec.reverseInitialTCPFlags = tcp->reverseInitialTCPFlags;
            fixrec.reverseUnionTCPFlags = tcp->reverseUnionTCPFlags;
            have_tcp_stml = 1;
        }
    }
    fbSubTemplateMultiListClear(&fixrec.stml);

    if (pkts && bytes) {
        /* We have forward information. */
        TRACEMSG(("Read a forward record"));

        /* Handle the IP addresses */
#if SK_ENABLE_IPV6
        if (!SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address)
            || !SK_IPV6_IS_ZERO(fixrec.rw.destinationIPv6Address))
        {
            /* Values found in IPv6 addresses--use them */
            rwRecSetIPv6(rec);
            rwRecMemSetSIPv6(rec, &fixrec.rw.sourceIPv6Address);
            rwRecMemSetDIPv6(rec, &fixrec.rw.destinationIPv6Address);
            rwRecMemSetNhIPv6(rec, &fixrec.rw.ipNextHopIPv6Address);
        } else
#endif /* SK_ENABLE_IPV6 */
        {
            /* Take values from IPv4 */
            rwRecSetSIPv4(rec, fixrec.rw.sourceIPv4Address);
            rwRecSetDIPv4(rec, fixrec.rw.destinationIPv4Address);
            rwRecSetNhIPv4(rec, fixrec.rw.ipNextHopIPv4Address);
        }

        /* Handle the Protocol and Ports */
        rwRecSetProto(rec, fixrec.rw.protocolIdentifier);
        rwRecSetSPort(rec, fixrec.rw.sourceTransportPort);
        rwRecSetDPort(rec, fixrec.rw.destinationTransportPort);

        /* Handle the SNMP or VLAN interfaces */
        if (vlan_interfaces) {
            rwRecSetInput(rec, fixrec.vlanId);
            rwRecSetOutput(rec, fixrec.postVlanId);
        } else {
            rwRecSetInput(rec,
                          CLAMP_VAL(fixrec.rw.ingressInterface, UINT16_MAX));
            rwRecSetOutput(rec,
                           CLAMP_VAL(fixrec.rw.egressInterface, UINT16_MAX));
        }


        /* Store volume, clamping counts to 32 bits. */
        rwRecSetPkts(rec, CLAMP_VAL(pkts, UINT32_MAX));
        rwRecSetBytes(rec, CLAMP_VAL(bytes, UINT32_MAX));
    } else {
        if (0 == rev_pkts || 0 == rev_bytes) {
            skAbort();
        }

        /* We have no forward information, only reverse.  Write the
         * source and dest values from the IPFIX record to SiLK's dest
         * and source fields, respectively. */
        TRACEMSG(("Read a reversed record"));

        /* Store volume, clamping counts to 32 bits. */
        rwRecSetPkts(rec, CLAMP_VAL(rev_pkts, UINT32_MAX));
        rwRecSetBytes(rec, CLAMP_VAL(rev_bytes, UINT32_MAX));

        /* This cannot be a bi-flow.  Clear rev_pkts and rev_bytes
         * variables now. We check this in the revRec code below. */
        rev_pkts = rev_bytes = 0;

        /* Handle the IP addresses */
#if SK_ENABLE_IPV6
        if (!SK_IPV6_IS_ZERO(fixrec.rw.sourceIPv6Address)
            || !SK_IPV6_IS_ZERO(fixrec.rw.destinationIPv6Address))
        {
            /* Values found in IPv6 addresses--use them */
            rwRecSetIPv6(rec);
            rwRecMemSetSIPv6(rec, &fixrec.rw.destinationIPv6Address);
            rwRecMemSetDIPv6(rec, &fixrec.rw.sourceIPv6Address);
            rwRecMemSetNhIPv6(rec, &fixrec.rw.ipNextHopIPv6Address);
        } else
#endif /* SK_ENABLE_IPV6 */
        {
            /* Take values from IPv4 */
            rwRecSetSIPv4(rec, fixrec.rw.destinationIPv4Address);
            rwRecSetDIPv4(rec, fixrec.rw.sourceIPv4Address);
            rwRecSetNhIPv4(rec, fixrec.rw.ipNextHopIPv4Address);
        }

        /* Handle the Protocol and Ports */
        rwRecSetProto(rec, fixrec.rw.protocolIdentifier);
        if (!rwRecIsICMP(rec)) {
            rwRecSetSPort(rec, fixrec.rw.destinationTransportPort);
            rwRecSetDPort(rec, fixrec.rw.sourceTransportPort);
        } else {
            /* For an ICMP record, put whichever Port field is
             * non-zero into the record's dPort field */
            rwRecSetSPort(rec, 0);
            rwRecSetDPort(rec, (fixrec.rw.destinationTransportPort
                                ? fixrec.rw.destinationTransportPort
                                : fixrec.rw.sourceTransportPort));
        }

        /* Handle the SNMP or VLAN interfaces */
        if (vlan_interfaces) {
            if (fbTemplateContainsElementByName(tmpl, &ies.reverseVlanId))
            {
                /* If we have the reverse elements, use them */
                rwRecSetInput(rec, fixrec.reverseVlanId);
                rwRecSetOutput(rec, fixrec.reversePostVlanId);
            } else if (fbTemplateContainsElementByName(tmpl, &ies.postVlanId)){
                /* If we have a single vlanId, set 'input' to that value;
                 * otherwise, set 'input' to postVlanId and 'output' to
                 * vlanId. */
                rwRecSetInput(rec, fixrec.postVlanId);
                rwRecSetOutput(rec, fixrec.vlanId);
            } else {
                /* we have a single vlanId, so don't swap the values */
                rwRecSetInput(rec, fixrec.vlanId);
            }
        } else {
            rwRecSetInput(rec,
                          CLAMP_VAL(fixrec.rw.egressInterface, UINT16_MAX));
            rwRecSetOutput(rec,
                           CLAMP_VAL(fixrec.rw.ingressInterface, UINT16_MAX));
        }

    }


    /* Run the Gauntlet of Time - convert all the various ways an IPFIX
     * record's time could be represented into start and elapsed times. */
    if (fixrec.rw.flowStartMilliseconds) {
        /* Flow start time in epoch milliseconds */
        rwRecSetStartTime(rec, (sktime_t)fixrec.rw.flowStartMilliseconds);
        if (fixrec.rw.flowEndMilliseconds >= fixrec.rw.flowStartMilliseconds) {
            /* Flow end time in epoch milliseconds */
            rwRecSetElapsed(rec,(uint32_t)(fixrec.rw.flowEndMilliseconds
                                           -fixrec.rw.flowStartMilliseconds));
        } else {
            /* Flow duration in milliseconds */
            rwRecSetElapsed(rec, fixrec.flowDurationMilliseconds);
        }
    } else if (fixrec.flowStartMicroseconds) {
        /* Flow start time in NTP microseconds */
        sTime = skiNTPDecode(fixrec.flowStartMicroseconds);
        rwRecSetStartTime(rec, (sktime_t)sTime);
        if (fixrec.flowEndMicroseconds >= fixrec.flowStartMicroseconds) {
            /* Flow end time in NTP microseconds */
            rwRecSetElapsed(rec,
                            (uint32_t)(skiNTPDecode(fixrec.flowEndMicroseconds)
                                       - sTime));
        } else {
            /* Flow duration in microseconds */
            rwRecSetElapsed(rec, (fixrec.flowDurationMicroseconds / 1000));
        }
    } else if (fixrec.flowStartSeconds) {
        /* Seconds? Sure, why not. */
        rwRecSetStartTime(rec, sktimeCreate(fixrec.flowStartSeconds, 0));
        rwRecSetElapsed(rec, ((uint32_t)1000 * (fixrec.flowEndSeconds
                                                - fixrec.flowStartSeconds)));
    } else if (fixrec.flowStartDeltaMicroseconds) {
        /* Flow start time in delta microseconds */
        sTime = (fBufGetExportTime(fbuf) * 1000
                 - fixrec.flowStartDeltaMicroseconds / 1000);
        rwRecSetStartTime(rec, (sktime_t)sTime);
        if (fixrec.flowEndDeltaMicroseconds
            && (fixrec.flowEndDeltaMicroseconds
                <= fixrec.flowStartDeltaMicroseconds))
        {
            /* Flow end time in delta microseconds */
            eTime = (fBufGetExportTime(fbuf) * 1000
                     - fixrec.flowEndDeltaMicroseconds / 1000);
            rwRecSetElapsed(rec, (uint32_t)(eTime - sTime));
        } else {
            /* Flow duration in microseconds */
            rwRecSetElapsed(rec, (fixrec.flowDurationMicroseconds / 1000));
        }
    } else if (fbTemplateContainsElementByName(tmpl, &ies.flowStartSysUpTime)){
        /* Times based on flow generator system uptimes (Netflow v9) */
        intmax_t uptime, difference;
        sktime_t export_msec;

        /* Set duration.  Our NetFlow v5 code checks the magnitude of
         * the difference between te eTime and sTime; this code is not
         * that complicated---we assume if eTime is less than sTime
         * then eTime has rolled over. */
        if (fixrec.flowStartSysUpTime <= fixrec.flowEndSysUpTime) {
            rwRecSetElapsed(rec, (fixrec.flowEndSysUpTime
                                  - fixrec.flowStartSysUpTime));
        } else {
            /* assume EndTime rolled-over and start did not */
            rwRecSetElapsed(rec, (ROLLOVER32 + fixrec.flowEndSysUpTime
                                  - fixrec.flowStartSysUpTime));
        }

        /* Set start time. */
        export_msec = sktimeCreate(fBufGetExportTime(fbuf), 0);
        if (!fbTemplateContainsElementByName(
                tmpl, &ies.systemInitTimeMilliseconds))
        {
            /* we do not know when the router booted.  assume end-time
             * is same as the record's export time and set start-time
             * accordingly. */
            TRACEMSG((("Setting times for NetFlowV9:"
                       " exportTime %" PRId64 ", bootTime N/A, upTime N/A"
                       ", flowStartSysUpTime %" PRIu32
                       ", flowEndSysUpTime %" PRIu32),
                      (int64_t)export_msec, fixrec.flowStartSysUpTime,
                      fixrec.flowEndSysUpTime));
            rwRecSetStartTime(rec, export_msec - rwRecGetElapsed(rec));
        } else {
            /* systemInitTimeMilliseconds is the absolute router boot
             * time (msec), and libfixbuf sets it by subtracting the
             * NFv9 uptime (msec) from the record's abolute export
             * time (sec). */
            uptime = export_msec - fixrec.systemInitTimeMilliseconds;
            TRACEMSG((("Setting times for NetFlowV9:"
                       " exportTime %" PRId64 ", bootTime %" PRIu64
                       ", upTime %" PRId64 ", flowStartSysUpTime %" PRIu32
                       ", flowEndSysUpTime %" PRIu32),
                      (int64_t)export_msec,
                      fixrec.systemInitTimeMilliseconds, (int64_t)uptime,
                      fixrec.flowStartSysUpTime,
                      fixrec.flowEndSysUpTime));

            difference = uptime - fixrec.flowStartSysUpTime;
            if (difference > MAXIMUM_FLOW_TIME_DEVIATION) {
                /* assume upTime is set before record is composed and
                 * that start-time has rolled over. */
                rwRecSetStartTime(rec, (fixrec.systemInitTimeMilliseconds
                                        + fixrec.flowStartSysUpTime
                                        + ROLLOVER32));
            } else if (-difference > MAXIMUM_FLOW_TIME_DEVIATION) {
                /* assume upTime is set after record is composed and
                 * that upTime has rolled over. */
                rwRecSetStartTime(rec, (fixrec.systemInitTimeMilliseconds
                                        + fixrec.flowStartSysUpTime
                                        - ROLLOVER32));
            } else {
                /* times look reasonable; assume no roll over */
                rwRecSetStartTime(rec, (fixrec.systemInitTimeMilliseconds
                                        + fixrec.flowStartSysUpTime));
            }
        }
    } else {
        /* No per-flow time information.
         * Assume message export is flow end time. */
        if (fixrec.flowDurationMilliseconds) {
            /* Flow duration in milliseconds */
            rwRecSetElapsed(rec, fixrec.flowDurationMilliseconds);
        } else if (fixrec.flowDurationMicroseconds) {
            /* Flow duration in microseconds */
            rwRecSetElapsed(rec, (fixrec.flowDurationMicroseconds / 1000));
        } else {
            /* Presume zero duration flow */
            rwRecSetElapsed(rec, 0);
        }
        /* Set start time based on export and elapsed time */
        rwRecSetStartTime(rec, sktimeCreate((fBufGetExportTime(fbuf)
                                             - rwRecGetElapsed(rec)), 0));
    }

    /* Copy the remainder of the record */
    rwRecSetFlowType(rec, fixrec.rw.silkFlowType);
    rwRecSetSensor(rec, fixrec.rw.silkFlowSensor);
    rwRecSetApplication(rec, fixrec.rw.silkAppLabel);

    tcp_state = fixrec.rw.silkTCPState;
    tcp_flags = (fixrec.rw.initialTCPFlags | fixrec.rw.unionTCPFlags);

    /* Ensure the SK_TCPSTATE_EXPANDED bit is properly set. */
    if (tcp_flags && IPPROTO_TCP == rwRecGetProto(rec)) {
        /* Flow is TCP and init|session flags had a value. */
        rwRecSetFlags(rec, tcp_flags);
        rwRecSetInitFlags(rec, fixrec.rw.initialTCPFlags);
        rwRecSetRestFlags(rec, fixrec.rw.unionTCPFlags);
        tcp_state |= SK_TCPSTATE_EXPANDED;
    } else {
        /* clear bit when not TCP or no separate init/session flags */
        tcp_state &= ~SK_TCPSTATE_EXPANDED;
        /* use whatever all-flags we were given; leave initial-flags
         * and session-flags unset */
        rwRecSetFlags(rec, fixrec.rw.tcpControlBits);
    }

    /* Process the flowEndReason and flowAttributes unless one of
     * those bits is already set (via silkTCPState). */
    if (!(tcp_state
          & (SK_TCPSTATE_FIN_FOLLOWED_NOT_ACK | SK_TCPSTATE_TIMEOUT_KILLED
             | SK_TCPSTATE_TIMEOUT_STARTED | SK_TCPSTATE_UNIFORM_PACKET_SIZE)))
    {
        /* Note active timeout */
        if ((fixrec.flowEndReason & SKI_END_MASK) == SKI_END_ACTIVE) {
            tcp_state |= SK_TCPSTATE_TIMEOUT_KILLED;
        }
        /* Note continuation */
        if (fixrec.flowEndReason & SKI_END_ISCONT) {
            tcp_state |= SK_TCPSTATE_TIMEOUT_STARTED;
        }
        /* Note flows with records of uniform size */
        if (fixrec.flowAttributes & SKI_FLOW_ATTRIBUTE_UNIFORM_PACKET_SIZE) {
            tcp_state |= SK_TCPSTATE_UNIFORM_PACKET_SIZE;
        }
        rwRecSetTcpState(rec, tcp_state);
    }

    rwRecSetTcpState(rec, tcp_state);

    /* Handle the reverse record if the caller provided one and if
     * there is one in the IPFIX record, which is indicated by the
     * value of 'rev_bytes'.*/
    if (0 == rev_bytes) {
        /* No data for reverse direction; just clear the record. */
        if (revRec) {
            RWREC_CLEAR(revRec);
        }
    } else if (revRec) {
        /* We have data for reverse direction. */
        TRACEMSG(("Handling reverse side of bi-flow"));

#define COPY_FORWARD_REC_TO_REVERSE 1
#if COPY_FORWARD_REC_TO_REVERSE
        /* Initialize the reverse record with the forward
         * record  */
        RWREC_COPY(revRec, rec);
#else
        /* instead of copying the forward record and changing
         * nearly everything, we could just set these fields on
         * the reverse record. */
        rwRecSetProto(revRec, fixrec.rw.protocolIdentifier);
        rwRecSetFlowType(revRec, fixrec.rw.silkFlowType);
        rwRecSetSensor(revRec, fixrec.rw.silkFlowSensor);
        rwRecSetTcpState(revRec, fixrec.rw.silkTCPState);
        rwRecSetApplication(revRec, fixrec.rw.silkAppLabel);
        /* does using the forward nexthop IP for the reverse
         * record make any sense?  Shouldn't we check for a
         * reverse next hop address? */
#if SK_ENABLE_IPV6
        if (rwRecIsIPv6(rec)) {
            rwRecSetIPv6(revRec);
            rwRecMemSetNhIPv6(revRec, &fixrec.rw.ipNextHopIPv6Address);
        } else
#endif
        {
            rwRecSetNhIPv4(revRec, &fixrec.rw.ipNextHopIPv4Address);
        }
#endif  /* #else clause of #if COPY_FORWARD_REC_TO_REVERSE */

        /* Reverse the IPs */
#if SK_ENABLE_IPV6
        if (rwRecIsIPv6(rec)) {
            rwRecMemSetSIPv6(revRec, &fixrec.rw.destinationIPv6Address);
            rwRecMemSetDIPv6(revRec, &fixrec.rw.sourceIPv6Address);
        } else
#endif
        {
            rwRecSetSIPv4(revRec, fixrec.rw.destinationIPv4Address);
            rwRecSetDIPv4(revRec, fixrec.rw.sourceIPv4Address);
        }

        /* Reverse the ports unless this is an ICMP record */
        if (!rwRecIsICMP(rec)) {
            rwRecSetSPort(revRec, rwRecGetDPort(rec));
            rwRecSetDPort(revRec, rwRecGetSPort(rec));
        }

        /* Reverse the SNMP or VLAN interfaces */
        if (!vlan_interfaces) {
            rwRecSetInput(revRec, rwRecGetOutput(rec));
            rwRecSetOutput(revRec, rwRecGetInput(rec));
        } else if (fbTemplateContainsElementByName(tmpl, &ies.reverseVlanId)) {
            /* Reverse VLAN values exist.  Use them */
            rwRecSetInput(rec, fixrec.reverseVlanId);
            rwRecSetOutput(rec, fixrec.reversePostVlanId);
        } else if (fbTemplateContainsElementByName(tmpl, &ies.postVlanId)) {
            /* Reverse the forward values */
            rwRecSetInput(rec, fixrec.postVlanId);
            rwRecSetOutput(rec, fixrec.vlanId);
        } else {
            /* we have a single vlanId, so don't swap the values */
            rwRecSetInput(rec, fixrec.vlanId);
        }

        /* Set volume.  We retrieved them above */
        rwRecSetPkts(revRec, CLAMP_VAL(rev_pkts, UINT32_MAX));
        rwRecSetBytes(revRec, CLAMP_VAL(rev_bytes, UINT32_MAX));

        /* Calculate reverse start time from reverse RTT */

        /* Reverse flow's start time must be increased and its
         * duration decreased by its offset from the forward
         * record  */
        rwRecSetStartTime(revRec, (rwRecGetStartTime(rec)
                                   + fixrec.reverseFlowDeltaMilliseconds));
        rwRecSetElapsed(revRec, (rwRecGetElapsed(rec)
                                 - fixrec.reverseFlowDeltaMilliseconds));

        /* Note: the value of the 'tcp_state' variable from above is
         * what is in rwRecGetTcpState(revRec). */

        /* Get reverse TCP flags from the IPFIX record if they are
         * available.  Otherwise, leave the flags unchanged (using
         * those from the forward direction). */
        tcp_flags =(fixrec.reverseInitialTCPFlags|fixrec.reverseUnionTCPFlags);

        if (tcp_flags && IPPROTO_TCP == rwRecGetProto(rec)) {
            /* Flow is TCP and init|session has a value. */
            TRACEMSG(("Using reverse TCP flags (initial|session)"));
            rwRecSetFlags(revRec, tcp_flags);
            rwRecSetInitFlags(revRec, fixrec.reverseInitialTCPFlags);
            rwRecSetRestFlags(revRec, fixrec.reverseUnionTCPFlags);
            tcp_state |= SK_TCPSTATE_EXPANDED;
        } else if (fbTemplateContainsElementByName(
                       tmpl, &ies.reverseTcpControlBits))
        {
            /* Use whatever is in all-flags; clear any init/session
             * flags we got from the forward rec. */
            TRACEMSG(("Using reverse TCP flags (all only)"));
            rwRecSetFlags(revRec, fixrec.reverseTcpControlBits);
            rwRecSetInitFlags(revRec, 0);
            rwRecSetRestFlags(revRec, 0);
            tcp_state &= ~SK_TCPSTATE_EXPANDED;
        } else if (have_tcp_stml || (fbTemplateContainsElementByName(
                                         tmpl, &ies.reverseInitialTCPFlags)))
        {
            /* If a reverseInitialTCPFlags Element existed on the
             * template; use it even though its value is 0. */
            TRACEMSG(("Setting all TCP flags to 0"));
            rwRecSetFlags(revRec, 0);
            rwRecSetInitFlags(revRec, 0);
            rwRecSetRestFlags(revRec, 0);
            tcp_state &= ~SK_TCPSTATE_EXPANDED;
        }
        /* else leave the flags unchanged */

        /* Handle reverse flow attributes */
        if (fixrec.reverseFlowAttributes
            & SKI_FLOW_ATTRIBUTE_UNIFORM_PACKET_SIZE)
        {
            /* ensure it is set */
            tcp_state |= SK_TCPSTATE_UNIFORM_PACKET_SIZE;
        } else {
            /* ensure it it not set */
            tcp_state &= ~SK_TCPSTATE_UNIFORM_PACKET_SIZE;
        }

        rwRecSetTcpState(revRec, tcp_state);

    }

    /* all done */
    return ((rev_bytes > 0) ? 2 : 1);
}


/* **************************************************************
 * *****  Support for writing/export
 */

fBuf_t *skiCreateWriteBufferForFP(
    FILE           *fp,
    uint32_t        domain,
    GError        **err)
{
    return skiCreateWriteBuffer(fbExporterAllocFP(fp), domain, err);
}


fBuf_t *skiCreateWriteBuffer(
    fbExporter_t   *exporter,
    uint32_t        domain,
    GError        **err)
{
    fbInfoModel_t   *model = skiInfoModel();
    fbSession_t     *session = NULL;
    fbTemplate_t    *tmpl = NULL;
    fBuf_t          *fbuf = NULL;

    /* Allocate a session.  The session will be owned by the fbuf, so
     * don't save it for later freeing. */
    session = skiSessionNew(DO_NOT_SAVE_SESSION);

    /* set observation domain */
    fbSessionSetDomain(session, domain);

    /* Add the full record template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, ski_flow_spec, 0, err)) {
        goto ERROR;
    }
    if (!fbSessionAddTemplate(session, TRUE, SKI_FLOW_TID, tmpl, err)) {
        goto ERROR;
    }
    if (!fbSessionAddTemplate(session, FALSE, SKI_FLOW_TID, tmpl, err)) {
        goto ERROR;
    }

    /* Create a buffer with the session and supplied exporter */
    fbuf = fBufAllocForExport(session, exporter);

    /* write RW base flow template */
    if (!fbSessionExportTemplates(session, err)) {
        goto ERROR;
    }

    /* set default templates */
    if (!fBufSetInternalTemplate(fbuf, SKI_FLOW_TID, err)) {
        goto ERROR;
    }
    if (!fBufSetExportTemplate(fbuf, SKI_FLOW_TID, err)) {
        goto ERROR;
    }

    /* done */
    return fbuf;

  ERROR:
    if (fbuf) {
        fBufFree(fbuf);
    } else {
        fbTemplateFreeUnused(tmpl);
        if (session) {
            fbSessionFree(session);
        }
    }
    return NULL;
}


/* Append SiLK Flow 'rec' to the libfixbuf buffer 'fbuf' */
gboolean skiRwAppendRecord(
    fBuf_t      *fbuf,
    const rwRec *rec,
    GError     **err)
{
    ski_rwrec_t fixrec;

    /* Convert time from start/elapsed to start and end epoch millis. */
    fixrec.flowStartMilliseconds = (uint64_t)rwRecGetStartTime(rec);
    fixrec.flowEndMilliseconds = ((uint64_t)fixrec.flowStartMilliseconds
                                  + rwRecGetElapsed(rec));

    /* Handle IP addresses */
#if SK_ENABLE_IPV6
    if (rwRecIsIPv6(rec)) {
        rwRecMemGetSIPv6(rec, fixrec.sourceIPv6Address);
        rwRecMemGetDIPv6(rec, fixrec.destinationIPv6Address);
        rwRecMemGetNhIPv6(rec, fixrec.ipNextHopIPv6Address);
        fixrec.sourceIPv4Address = 0;
        fixrec.destinationIPv4Address = 0;
        fixrec.ipNextHopIPv4Address = 0;
    } else
#endif
    {
        memset(fixrec.sourceIPv6Address, 0,
               sizeof(fixrec.sourceIPv6Address));
        memset(fixrec.destinationIPv6Address, 0,
               sizeof(fixrec.destinationIPv6Address));
        memset(fixrec.ipNextHopIPv6Address, 0,
               sizeof(fixrec.ipNextHopIPv6Address));
        fixrec.sourceIPv4Address = rwRecGetSIPv4(rec);
        fixrec.destinationIPv4Address = rwRecGetDIPv4(rec);
        fixrec.ipNextHopIPv4Address = rwRecGetNhIPv4(rec);
    }

    /* Copy rest of record */
    fixrec.sourceTransportPort = rwRecGetSPort(rec);
    fixrec.destinationTransportPort = rwRecGetDPort(rec);
    fixrec.ingressInterface = rwRecGetInput(rec);
    fixrec.egressInterface = rwRecGetOutput(rec);
    fixrec.packetDeltaCount = rwRecGetPkts(rec);
    fixrec.octetDeltaCount = rwRecGetBytes(rec);
    fixrec.protocolIdentifier = rwRecGetProto(rec);
    fixrec.silkFlowType = rwRecGetFlowType(rec);
    fixrec.silkFlowSensor = rwRecGetSensor(rec);
    fixrec.tcpControlBits = rwRecGetFlags(rec);
    fixrec.initialTCPFlags = rwRecGetInitFlags(rec);
    fixrec.unionTCPFlags = rwRecGetRestFlags(rec);
    fixrec.silkTCPState = rwRecGetTcpState(rec);
    fixrec.silkAppLabel = rwRecGetApplication(rec);

#if SKI_FLOW_SPEC_PADDING != 0
    /* According to RFC5102, the value of the paddingOctets
     * Information Element "is always a sequence of 0x00 values." */
    memset(fixrec.pad, 0, SKI_FLOW_SPEC_PADDING);
#endif

    /* Append the record to the buffer */
    if (!fBufAppend(fbuf, (uint8_t *)&fixrec, sizeof(fixrec), err)) {
        return FALSE;
    }

    /* all done */
    return TRUE;
}


void skiCheckDataStructure(
    FILE       *fh)
{
    unsigned long pos;

#define PRINT_TITLE(s_)                                         \
    fprintf(fh, "===> %s\n%5s|%5s|%5s|%5s|%5s|%s\n", #s_,       \
            "begin", "end", "size", "alerr", "hole", "member")

#define PRINT_OFFSET(pos_, s_, mem_)                                    \
    {                                                                   \
        s_ x;                                                           \
        unsigned long off_ = (unsigned long)offsetof(s_, mem_);         \
        unsigned long sz_  = (unsigned long)sizeof(x.mem_);             \
        unsigned long end_ = off_ + sz_ - 1;                            \
        int align_ = ((off_ % sz_) == 0);                               \
        int hole_ = (pos_ != off_);                                     \
        pos_ += sz_;                                                    \
        fprintf(fh, "%5lu|%5lu|%5lu|%5s|%5s|%s\n",                      \
                off_, end_, sz_, (align_ ? "" : "alerr"),               \
                (hole_ ? "hole" : ""), #mem_);                          \
    }

    pos = 0;
    PRINT_TITLE(ski_rwrec_t);
    PRINT_OFFSET(pos, ski_rwrec_t, flowStartMilliseconds);
    PRINT_OFFSET(pos, ski_rwrec_t, flowEndMilliseconds);
    PRINT_OFFSET(pos, ski_rwrec_t, sourceIPv6Address);
    PRINT_OFFSET(pos, ski_rwrec_t, destinationIPv6Address);
    PRINT_OFFSET(pos, ski_rwrec_t, sourceIPv4Address);
    PRINT_OFFSET(pos, ski_rwrec_t, destinationIPv4Address);
    PRINT_OFFSET(pos, ski_rwrec_t, sourceTransportPort);
    PRINT_OFFSET(pos, ski_rwrec_t, destinationTransportPort);
    PRINT_OFFSET(pos, ski_rwrec_t, ipNextHopIPv4Address);
    PRINT_OFFSET(pos, ski_rwrec_t, ipNextHopIPv6Address);
    PRINT_OFFSET(pos, ski_rwrec_t, ingressInterface);
    PRINT_OFFSET(pos, ski_rwrec_t, egressInterface);
    PRINT_OFFSET(pos, ski_rwrec_t, packetDeltaCount);
    PRINT_OFFSET(pos, ski_rwrec_t, octetDeltaCount);
    PRINT_OFFSET(pos, ski_rwrec_t, protocolIdentifier);
    PRINT_OFFSET(pos, ski_rwrec_t, silkFlowType);
    PRINT_OFFSET(pos, ski_rwrec_t, silkFlowSensor);
    PRINT_OFFSET(pos, ski_rwrec_t, tcpControlBits);
    PRINT_OFFSET(pos, ski_rwrec_t, initialTCPFlags);
    PRINT_OFFSET(pos, ski_rwrec_t, unionTCPFlags);
    PRINT_OFFSET(pos, ski_rwrec_t, silkTCPState);
    PRINT_OFFSET(pos, ski_rwrec_t, silkAppLabel);
#if SKI_FLOW_SPEC_PADDING != 0
    PRINT_OFFSET(pos, ski_rwrec_t, pad);
#endif

    pos = 0;
    PRINT_TITLE(ski_extrwrec_t);
    PRINT_OFFSET(pos, ski_extrwrec_t, rw);
    PRINT_OFFSET(pos, ski_extrwrec_t, packetTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, octetTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, initiatorPackets);
    PRINT_OFFSET(pos, ski_extrwrec_t, initiatorOctets);
    PRINT_OFFSET(pos, ski_extrwrec_t, reversePacketDeltaCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseOctetDeltaCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, reversePacketTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseOctetTotalCount);
    PRINT_OFFSET(pos, ski_extrwrec_t, responderPackets);
    PRINT_OFFSET(pos, ski_extrwrec_t, responderOctets);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowStartMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, systemInitTimeMilliseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowStartSeconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndSeconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowDurationMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowDurationMilliseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowStartDeltaMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndDeltaMicroseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseFlowDeltaMilliseconds);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowStartSysUpTime);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndSysUpTime);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseTcpControlBits);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseInitialTCPFlags);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseUnionTCPFlags);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowEndReason);
    PRINT_OFFSET(pos, ski_extrwrec_t, flowAttributes);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseFlowAttributes);
    PRINT_OFFSET(pos, ski_extrwrec_t, vlanId);
    PRINT_OFFSET(pos, ski_extrwrec_t, postVlanId);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseVlanId);
    PRINT_OFFSET(pos, ski_extrwrec_t, reversePostVlanId);
    PRINT_OFFSET(pos, ski_extrwrec_t, ipClassOfService);
    PRINT_OFFSET(pos, ski_extrwrec_t, reverseIpClassOfService);
    PRINT_OFFSET(pos, ski_extrwrec_t, firewallEvent);
    PRINT_OFFSET(pos, ski_extrwrec_t, NF_F_FW_EVENT);
    PRINT_OFFSET(pos, ski_extrwrec_t, NF_F_FW_EXT_EVENT);
#if SKI_EXTFLOW_SPEC_PADDING != 0
    PRINT_OFFSET(pos, ski_extrwrec_t, pad);
#endif

    pos = 0;
    PRINT_TITLE(ski_yaf_stats_t);
    PRINT_OFFSET(pos, ski_yaf_stats_t, systemInitTimeMilliseconds);
    PRINT_OFFSET(pos, ski_yaf_stats_t, exportedFlowRecordTotalCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, packetTotalCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, droppedPacketTotalCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, ignoredPacketTotalCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, notSentPacketTotalCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, expiredFragmentCount);
#if 0
    PRINT_OFFSET(pos, ski_yaf_stats_t, assembledFragmentCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, flowTableFlushEventCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, flowTablePeakCount);
    PRINT_OFFSET(pos, ski_yaf_stats_t, meanFlowRate);
    PRINT_OFFSET(pos, ski_yaf_stats_t, meanPacketRate);
    PRINT_OFFSET(pos, ski_yaf_stats_t, exporterIPv4Address);
#endif  /* 0 */
    PRINT_OFFSET(pos, ski_yaf_stats_t, exportingProcessId);
#if SKI_YAF_STATS_PADDING != 0
    PRINT_OFFSET(pos, ski_yaf_stats_t, pad);
#endif
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
