/*
** Copyright (C) 2005-2013 by Carnegie Mellon University.
**
** @OPENSOURCE_HEADER_START@
**
** Use of the SILK system and related source code is subject to the terms
** of the following licenses:
**
** GNU Public License (GPL) Rights pursuant to Version 2, June 1991
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
**
** NO WARRANTY
**
** ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
** PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
** PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
** "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
** KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
** LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
** MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
** OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
** SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
** TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
** WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
** LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
** CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
** CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
** DELIVERABLES UNDER THIS LICENSE.
**
** Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
** Mellon University, its trustees, officers, employees, and agents from
** all claims or demands made against them (and any related losses,
** expenses, or attorney's fees) arising out of, or relating to Licensee's
** and/or its sub licensees' negligent use or willful misuse of or
** negligent conduct or willful misconduct regarding the Software,
** facilities, or other rights or assistance granted by Carnegie Mellon
** University under this License, including, but not limited to, any
** claims of product liability, personal injury, death, damage to
** property, or violation of any laws or regulations.
**
** Carnegie Mellon University Software Engineering Institute authored
** documents are sponsored by the U.S. Department of Defense under
** Contract FA8721-05-C-0003. Carnegie Mellon University retains
** copyrights in all material produced under this contract. The U.S.
** Government retains a non-exclusive, royalty-free license to publish or
** reproduce these documents, or allow others to do so, for U.S.
** Government purposes only pursuant to the copyright license under the
** contract clause at 252.227.7013.
**
** @OPENSOURCE_HEADER_END@
*/

/*
**  rwgroupsetup.c
**
**   Application setup, teardown, and option parsing for rwgroup.
**
*/

#include <silk/silk.h>

RCSIDENT("$SiLK: rwgroupsetup.c f35f37a50ba7 2013-02-21 14:04:22Z mthomas $");

#include <silk/sksite.h>
#include <silk/skstringmap.h>
#include <silk/skprefixmap.h>
#include <silk/skcountry.h>
#include "rwgroup.h"


/* TYPEDEFS AND DEFINES */

/* File handle for --help output */
#define USAGE_FH stdout


/* LOCAL VARIABLES */

/* the text the user entered for the --id-fields switch */
static const char *id_fields_arg = NULL;

/* the text the user entered for the --delta-field switch */
static const char *delta_field_arg = NULL;

/* the compression method to use when writing the file.
 * sksiteCompmethodOptionsRegister() will set this to the default or
 * to the value the user specifies. */
static sk_compmethod_t comp_method;

/* where to write a copy of the input */
static skstream_t *copy_input = NULL;

/* whether stdout has been used as an output stream */
static int stdout_used = 0;

/* available key fields; rwAsciiFieldMapAddDefaultFields() fills this */
static sk_stringmap_t *key_field_map = NULL;

/* fields that get defined just like plugins */
static const struct app_static_plugins_st {
    const char         *name;
    skplugin_setup_fn_t setup_fn;
} app_static_plugins[] = {
    {"addrtype",        skAddressTypesAddFields},
    {"ccfilter",        skCountryAddFields},
    {"pmapfilter",      skPrefixMapAddFields},
    {NULL, NULL}        /* sentinel */
};

/* names of plug-ins to attempt to load at startup */
static const char *app_plugin_names[] = {
    SK_PLUGIN_ADD_SUFFIX("silkpython"),
    NULL /* sentinel */
};


/* OPTIONS SETUP */

typedef enum {
    OPT_ID_FIELDS, OPT_PLUGIN,
    OPT_DELTA_FIELD, OPT_DELTA_VALUE,
    OPT_OBJECTIVE, OPT_SUMMARIZE,
    OPT_REC_THRESHOLD, OPT_GROUP_OFFSET,
    OPT_OUTPUT_PATH, OPT_COPY_INPUT
} appOptionsEnum;

static struct option appOptions[] = {
    {"id-fields",           REQUIRED_ARG, 0, OPT_ID_FIELDS},
    {"plugin",              REQUIRED_ARG, 0, OPT_PLUGIN},
    {"delta-field",         REQUIRED_ARG, 0, OPT_DELTA_FIELD},
    {"delta-value",         REQUIRED_ARG, 0, OPT_DELTA_VALUE},
    {"objective",           NO_ARG,       0, OPT_OBJECTIVE},
    {"summarize",           NO_ARG,       0, OPT_SUMMARIZE},
    {"rec-threshold",       REQUIRED_ARG, 0, OPT_REC_THRESHOLD},
    {"group-offset",        REQUIRED_ARG, 0, OPT_GROUP_OFFSET},
    {"output-path",         REQUIRED_ARG, 0, OPT_OUTPUT_PATH},
    {"copy-input",          REQUIRED_ARG, 0, OPT_COPY_INPUT},
    {0,0,0,0}               /* sentinel entry */
};

static const char *appHelp[] = {
    ("List of fields that must be identical among all records\n"
     "\tin a group"),
    ("Load given plug-in to add fields. Switch may be repeated to\n"
     "\t load multiple plug-ins. Def. None"),
    ("Field which can differ within the group; only one allowed"),
    ("Maximum difference allowed between delta-field values\n"
     "\tbefore the records are considered as being in different groups"),
    ("Count values objectively; all delta-field values must be\n"
     "\twithin the delta-value of the first record in the group.  Normally,\n"
     "\tthe delta-field value of consecutive records is compared. Def. No"),
    ("Output a summary (a single record) for each group rather\n"
     "\tthan a all the records in the group. Def. No"),
    ("Minimum number of records a group must have before\n"
     "\tit will be written to the output stream. Def. 1"),
    ("Value to use for first group. Def. 0"),
    ("Write the output to the named location. Def. stdout"),
    ("Copy the input records to the named location. Def. No"),
    (char *)NULL
};


/* LOCAL FUNCTION PROTOTYPES */

static int  appOptionsHandler(clientData cData, int opt_index, char *opt_arg);
static int  parseIdFields(const char *field_string);
static int  parseDeltaField(const char *field_string);
static int  createStringmaps(void);


/* FUNCTION DEFINITIONS */

/*
 *  appUsageLong();
 *
 *    Print complete usage information to USAGE_FH.  Pass this
 *    function to skOptionsSetUsageCallback(); skOptionsParse() will
 *    call this funciton and then exit the program when the --help
 *    option is given.
 */
static void appUsageLong(void)
{
#define USAGE_MSG                                                        \
    ("{--id-fields=KEY | --delta-field=F --delta-value=N} [SWITCHES]\n"  \
     "\tGroups flows together by specified id-fields and delta-field;\n" \
     "\tmarks the group ID in Next Hop IP; requires pre-sorting.\n")

    FILE *fh = USAGE_FH;
    int i;

    /* Create the string map for --fields */
    createStringmaps();

    fprintf(fh, "%s %s", skAppName(), USAGE_MSG);
    fprintf(fh, "\nSWITCHES:\n");
    skOptionsDefaultUsage(fh);
    for (i = 0; appOptions[i].name; i++ ) {
        fprintf(fh, "--%s %s. ", appOptions[i].name,
                SK_OPTION_HAS_ARG(appOptions[i]));
        switch (i) {
          case OPT_ID_FIELDS:
            /* Dynamically build the help */
            fprintf(fh, "Field(s) to use as key:\n");
            skStringMapPrintUsage(key_field_map, fh, 4);
            break;
          default:
            /* Simple help text from the appHelp array */
            assert(appHelp[i]);
            fprintf(fh, "%s\n", appHelp[i]);
            break;
        }
    }

    skOptionsNotesUsage(fh);
    sksiteCompmethodOptionsUsage(fh);
    sksiteOptionsUsage(fh);
    skPluginOptionsUsage(fh);
}


/*
 *  appTeardown()
 *
 *    Teardown all modules, close all files, and tidy up all
 *    application state.
 *
 *    This function is idempotent.
 */
void appTeardown(void)
{
    static int teardownFlag = 0;
    int rv;

    if (teardownFlag) {
        return;
    }
    teardownFlag = 1;

    /* close the copy stream */
    if (copy_input) {
        rv = skStreamClose(copy_input);
        if (rv) {
            skStreamPrintLastErr(copy_input, rv, &skAppPrintErr);
        }
        skStreamDestroy(&copy_input);
    }

    /* close and destroy output */
    if (out_rwios) {
        rv = skStreamDestroy(&out_rwios);
        if (rv) {
            /* only print error when not in signal handler */
            skStreamPrintLastErr(out_rwios, rv, &skAppPrintErr);
        }
        out_rwios = NULL;
    }

    /* close input */
    skStreamDestroy(&in_rwios);

    /* plug-in teardown */
    skPluginRunCleanup(SKPLUGIN_APP_SORT);
    skPluginTeardown();

    /* free variables */
    if (thresh_buf) {
        free(thresh_buf);
        thresh_buf = NULL;
    }
    if (id_fields != NULL) {
        free(id_fields);
    }
    if (key_field_map) {
        skStringMapDestroy(key_field_map);
    }

    skOptionsNotesTeardown();
    skAppUnregister();
}


/*
 *  appSetup(argc, argv);
 *
 *    Perform all the setup for this application include setting up
 *    required modules, parsing options, etc.  This function should be
 *    passed the same arguments that were passed into main().
 *
 *    Returns to the caller if all setup succeeds.  If anything fails,
 *    this function will cause the application to exit with a FAILURE
 *    exit status.
 */
void appSetup(int argc, char **argv)
{
    SILK_FEATURES_DEFINE_STRUCT(features);
    const char *in_path = "stdin";
    int arg_index;
    int j;
    int rv;

    /* verify same number of options and help strings */
    assert((sizeof(appHelp)/sizeof(char *))
           == (sizeof(appOptions)/sizeof(struct option)));

    /* register the application */
    skAppRegister(argv[0]);
    skAppVerifyFeatures(&features, NULL);
    skOptionsSetUsageCallback(&appUsageLong);

    /* initialize variables */
    skipaddrClear(&group_id);

    /* initialize plugin library */
    skPluginSetup(1, SKPLUGIN_APP_GROUP);

    /* register the options */
    if (skOptionsRegister(appOptions, &appOptionsHandler, NULL)
        || skOptionsNotesRegister(NULL)
        || sksiteCompmethodOptionsRegister(&comp_method)
        || sksiteOptionsRegister(SK_SITE_FLAG_CONFIG_FILE))
    {
        skAppPrintErr("Unable to register options");
        exit(EXIT_FAILURE);
    }

    /* register the teardown handler */
    if (atexit(appTeardown) < 0) {
        skAppPrintErr("Unable to register appTeardown() with atexit()");
        exit(EXIT_FAILURE);
    }

    /* try to load hard-coded plugins */
    for (j = 0; app_static_plugins[j].name; ++j) {
        skPluginAddAsPlugin(app_static_plugins[j].name,
                            app_static_plugins[j].setup_fn);
    }
    for (j = 0; app_plugin_names[j]; ++j) {
        skPluginLoadPlugin(app_plugin_names[j], 0);
    }

    /* parse options */
    arg_index = skOptionsParse(argc, argv);
    if (arg_index < 0) {
        skAppUsage();             /* never returns */
    }

    /* try to load site config file; if it fails, we will not be able
     * to resolve flowtype and sensor from input file names */
    sksiteConfigure(0);

    /* create the --fields */
    if (createStringmaps()) {
        exit(EXIT_FAILURE);
    }

    /* verify that some field was specified */
    if (NULL == id_fields_arg && NULL == delta_field_arg) {
        skAppPrintErr("No fields specified; must specify --%s or --%s",
                      appOptions[OPT_ID_FIELDS].name,
                      appOptions[OPT_DELTA_FIELD].name);
        skAppUsage();             /* never returns */
    }

    /* parse the id-fields argument */
    if (id_fields_arg != NULL) {
        if (parseIdFields(id_fields_arg)) {
            exit(EXIT_FAILURE);
        }
    }

    /* parse the delta-field argument and ensure value is specified */
    if (NULL != delta_field_arg) {
        if (parseDeltaField(delta_field_arg)) {
            exit(EXIT_FAILURE);
        }
    } else if (delta_value) {
        skAppPrintErr("The --%s switch only allowed when a --%s is specified",
                      appOptions[OPT_DELTA_VALUE].name,
                      appOptions[OPT_DELTA_FIELD].name);
        skAppUsage();             /* never returns */
    }

    /* check for an input file name on the command line */
    if (arg_index < argc) {
        in_path = argv[arg_index];
        ++arg_index;
    }

    /* check for extra options */
    if (arg_index != argc) {
        skAppPrintErr(("Too many arguments;"
                       " only a single input file is permitted"));
        skAppUsage();             /* never returns */
    }

    /* create threshold buffer */
    if (threshold) {
        thresh_buf = (rwRec*)calloc(threshold, sizeof(rwRec));
        if (NULL == thresh_buf) {
            skAppPrintOutOfMemory(NULL);
            exit(EXIT_FAILURE);
        }
    }

    /* check for an output stream; or default to stdout  */
    if (out_rwios == NULL) {
        if (stdout_used) {
            skAppPrintErr("Only one output stream may use stdout");
            exit(EXIT_FAILURE);
        }
        if ((rv = skStreamCreate(&out_rwios, SK_IO_WRITE,SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(out_rwios, "stdout")))
        {
            skStreamPrintLastErr(out_rwios, rv, NULL);
            skStreamDestroy(&out_rwios);
            exit(EXIT_FAILURE);
        }
    }

    /* open the input stream */
    rv = skStreamOpenSilkFlow(&in_rwios, in_path, SK_IO_READ);
    if (rv) {
        skStreamPrintLastErr(in_rwios, rv, &skAppPrintErr);
        skStreamDestroy(&in_rwios);
        skAppPrintErr("Could not open %s for reading.  Exiting.", in_path);
        exit(EXIT_FAILURE);
    }

    /* set the copy-input stream to get everything we read */
    if (copy_input) {
        rv = skStreamSetCopyInput(in_rwios, copy_input);
        if (rv) {
            skStreamPrintLastErr(in_rwios, rv, &skAppPrintErr);
            exit(EXIT_FAILURE);
        }
    }

    /* set the compmethod on the output */
    rv = skHeaderSetCompressionMethod(skStreamGetSilkHeader(out_rwios),
                                      comp_method);
    if (rv) {
        skAppPrintErr("Error setting header on %s: %s",
                      skStreamGetPathname(out_rwios), skHeaderStrerror(rv));
        exit(EXIT_FAILURE);
    }

    /* copy annotations and command line entries from the input to the
     * output */
    if ((rv = skHeaderCopyEntries(skStreamGetSilkHeader(out_rwios),
                                  skStreamGetSilkHeader(in_rwios),
                                  SK_HENTRY_INVOCATION_ID))
        || (rv = skHeaderCopyEntries(skStreamGetSilkHeader(out_rwios),
                                     skStreamGetSilkHeader(in_rwios),
                                     SK_HENTRY_ANNOTATION_ID)))
    {
        skStreamPrintLastErr(out_rwios, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    /* add invocation and notes to the output */
    if ((rv = skHeaderAddInvocation(skStreamGetSilkHeader(out_rwios),
                                    1, argc, argv))
        || (rv = skOptionsNotesAddToStream(out_rwios)))
    {
        skStreamPrintLastErr(out_rwios, rv, &skAppPrintErr);
        exit(EXIT_FAILURE);
    }

    /* open output */
    rv = skStreamOpen(out_rwios);
    if (rv) {
        skStreamPrintLastErr(out_rwios, rv, &skAppPrintErr);
        skAppPrintErr("Could not open output file.");
        exit(EXIT_FAILURE);
    }

    /* write the header */
    rv = skStreamWriteSilkHeader(out_rwios);
    if (rv) {
        skStreamPrintLastErr(out_rwios, rv, &skAppPrintErr);
        skAppPrintErr("Could not write header to output file.");
        exit(EXIT_FAILURE);
    }

    return;                     /* OK */
}


/*
 *  status = appOptionsHandler(cData, opt_index, opt_arg);
 *
 *    This function is passed to skOptionsRegister(); it will be called
 *    by skOptionsParse() for each user-specified switch that the
 *    application has registered; it should handle the switch as
 *    required---typically by setting global variables---and return 1
 *    if the switch processing failed or 0 if it succeeded.  Returning
 *    a non-zero from from the handler causes skOptionsParse() to return
 *    a negative value.
 *
 *    The clientData in 'cData' is typically ignored; 'opt_index' is
 *    the index number that was specified as the last value for each
 *    struct option in appOptions[]; 'opt_arg' is the user's argument
 *    to the switch for options that have a REQUIRED_ARG or an
 *    OPTIONAL_ARG.
 */
static int appOptionsHandler(
    clientData  UNUSED(cData),
    int         opt_index,
    char       *opt_arg)
{
    int rv;

    switch ((appOptionsEnum)opt_index) {
      case OPT_ID_FIELDS:
        if (id_fields_arg) {
            skAppPrintErr("Invalid %s: Switch used multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        id_fields_arg = opt_arg;
        break;

      case OPT_OUTPUT_PATH:
        if (!opt_arg || 0 == strlen(opt_arg)) {
            skAppPrintErr("Missing file name for --%s option",
                          appOptions[opt_index].name);
            return 1;
        }
        if (out_rwios) {
            skAppPrintErr("Invalid %s: Switch used multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        if ((0 == strcmp(opt_arg, "stdout")) || (0 == strcmp(opt_arg, "-"))) {
            if (stdout_used) {
                skAppPrintErr("Only one output stream may use stdout");
                return 1;
            }
            stdout_used = 1;
        }
        if ((rv = skStreamCreate(&out_rwios, SK_IO_WRITE,SK_CONTENT_SILK_FLOW))
            || (rv = skStreamBind(out_rwios, opt_arg)))
        {
            skStreamPrintLastErr(out_rwios, rv, NULL);
            return 1;
        }
        break;

      case OPT_COPY_INPUT:
        if (!opt_arg || 0 == strlen(opt_arg)) {
            skAppPrintErr("Missing file name for --%s option",
                          appOptions[opt_index].name);
            return 1;
        }
        if (copy_input) {
            skAppPrintErr("Invalid %s: Switch used multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        if ((0 == strcmp(opt_arg, "stdout")) || (0 == strcmp(opt_arg, "-"))) {
            if (stdout_used) {
                skAppPrintErr("Only one output stream may use stdout");
                return 1;
            }
            stdout_used = 1;
        }
        rv = skStreamOpenSilkFlow(&copy_input, opt_arg, SK_IO_WRITE);
        if (rv) {
            skStreamPrintLastErr(copy_input, rv, &skAppPrintErr);
            skStreamDestroy(&copy_input);
            return 1;
        }
        break;

      case OPT_PLUGIN:
        if (skPluginLoadPlugin(opt_arg, 1) != 0) {
            skAppPrintErr("Unable to load %s as a plugin", opt_arg);
            return 1;
        }
        break;

      case OPT_DELTA_FIELD:
        if (delta_field_arg) {
            skAppPrintErr("Invalid %s: Switch used multiple times",
                          appOptions[opt_index].name);
            return 1;
        }
        delta_field_arg = opt_arg;
        break;

      case OPT_DELTA_VALUE:
        rv = skStringParseUint64(&delta_value, opt_arg, 1, 0);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_REC_THRESHOLD:
        rv = skStringParseUint32(&threshold, opt_arg, 0, MAX_THRESHOLD);
        if (rv) {
            goto PARSE_ERROR;
        }
        /* threshold of 1 is effectively 0 */
        if (threshold <= 1) {
            threshold = 0;
        }
        break;

      case OPT_GROUP_OFFSET:
        rv = skStringParseIP(&group_id, opt_arg);
        if (rv) {
            goto PARSE_ERROR;
        }
        break;

      case OPT_SUMMARIZE:
        summarize = 1;
        break;

      case OPT_OBJECTIVE:
        objective = 1;
        break;
    }

    return 0; /* OK */

  PARSE_ERROR:
    skAppPrintErr("Invalid %s '%s': %s",
                  appOptions[opt_index].name, opt_arg,
                  skStringParseStrerror(rv));
    return 1;
}


/*
 *  status = parseIdFields(fields_string);
 *
 *    Parse the user's option for the --fields switch.  Fill the
 *    global 'id_list' with the list of fields, and set 'num_fields'
 *    to the number of fields in 'id_list'.  Also initialize the
 *    plug-ins.  Return 0 on success; -1 on failure.
 */
static int parseIdFields(const char *field_string)
{
    sk_stringmap_iter_t    *iter = NULL;
    sk_stringmap_entry_t   *entry;
    key_field_t            *key;
    skplugin_field_t       *field_handle;
    size_t                  bin_width;
    skplugin_err_t          err;
    int                     rv = -1;
    uint32_t                i;
    char                   *errmsg;

    /* parse the input */
    if (skStringMapParse(key_field_map, field_string, SKSTRINGMAP_DUPES_ERROR,
                         &iter, &errmsg))
    {
        skAppPrintErr("Invalid %s: %s",
                      appOptions[OPT_ID_FIELDS].name, errmsg);
        goto END;
    }

    num_fields = skStringMapIterCountMatches(iter);
    id_fields = (uint32_t*)malloc(num_fields * sizeof(uint32_t));
    if (NULL == id_fields) {
        skAppPrintOutOfMemory(NULL);
        goto END;
    }

    /* fill the array, and initialize any plug-ins */
    for (i = 0; skStringMapIterNext(iter, &entry, NULL)==SK_ITERATOR_OK; ++i) {
        assert(i < num_fields);
        id_fields[i] = entry->id;
        if (NULL != entry->userdata) {
            /* field comes from a plug-in */
            assert(id_fields[i] >=  RWREC_PRINTABLE_FIELD_COUNT);

            if (key_num_fields == MAX_PLUGIN_KEY_FIELDS) {
                skAppPrintErr("Too many fields specified %lu > %u max",
                              (unsigned long)key_num_fields,
                              MAX_PLUGIN_KEY_FIELDS);
                goto END;
            }

            field_handle = (skplugin_field_t*)(entry->userdata);

            /* Activate the plugin (so cleanup knows about it) */
            err = skPluginFieldActivate(field_handle);
            if (err != SKPLUGIN_OK) {
                goto END;
            }

            /* Initialize this field */
            err = skPluginFieldRunInitialize(field_handle);
            if (err != SKPLUGIN_OK) {
                goto END;
            }

            /* get the bin width for this field */
            err = skPluginFieldGetLenBin(field_handle, &bin_width);
            if (err != SKPLUGIN_OK) {
                goto END;
            }

            key = &(key_fields[key_num_fields]);
            key->kf_field_handle = field_handle;
            key->kf_offset       = node_size;
            key->kf_width        = bin_width;

            ++key_num_fields;

            node_size += bin_width;
            if (node_size > MAX_NODE_SIZE) {
                skAppPrintErr("Sort key is too large %lu bytes > %d max",
                              (unsigned long)node_size, MAX_NODE_SIZE);
                goto END;
            }
        }
    }

    /* success */
    rv = 0;

  END:
    if (iter) {
        skStringMapIterDestroy(iter);
    }
    return rv;
}


static int parseDeltaField(const char *field_string)
{
    sk_stringmap_iter_t *iter = NULL;
    sk_stringmap_entry_t *entry;
    char *errmsg;
    uint64_t limit = 1;
    int rv = -1;

    /* verify user gave us input */
    if (field_string == NULL || field_string[0] == '\0') {
        skAppPrintErr("Missing a value for the --%s switch",
                      appOptions[OPT_DELTA_FIELD].name);
        goto END;
    }

    /* parse the input */
    if (skStringMapParse(key_field_map, field_string, SKSTRINGMAP_DUPES_ERROR,
                         &iter, &errmsg))
    {
        skAppPrintErr("Invalid %s: %s",
                      appOptions[OPT_ID_FIELDS].name, errmsg);
        goto END;
    }

    /* we only expect one field */
    if (skStringMapIterCountMatches(iter) > 1) {
        skAppPrintErr("Invalid %s: Only one field may be specified",
                      appOptions[OPT_DELTA_FIELD].name);
        goto END;
    }

    skStringMapIterNext(iter, &entry, NULL);
    delta_field = entry->id;

    /* make certain field is a valid the delta-value is within range */
    switch (delta_field) {
      case RWREC_FIELD_SIP:
      case RWREC_FIELD_DIP:
      case RWREC_FIELD_NHIP:
        /* for IPs, the delta_value is the number of LEAST significant
         * bits to REMOVE */
#if SK_ENABLE_IPV6
        limit = 127;
        if (delta_value <= limit) {
            uint8_t mask[16];
            uint32_t i = (128 - delta_value) >> 3;
            memset(&mask[0], 0xFF, i);
            mask[i] = ~(0xFF >> ((128 - delta_value) & 0x7));
            memset(&mask[i+1], 0, (15 - i));
            skipaddrSetV6(&delta_value_ip, mask);
        }
#else
        limit = 31;
        if (delta_value <= limit) {
            limit = 0;
            delta_value = UINT32_MAX << delta_value;
        }
#endif /* SK_ENABLE_IPV6 */
        break;

      case RWREC_FIELD_STIME:
      case RWREC_FIELD_STIME_MSEC:
      case RWREC_FIELD_ETIME:
      case RWREC_FIELD_ETIME_MSEC:
        /* this is a sktime_t. multiply user's value by 1000 to
         * convert from seconds to milliseconds */
        limit = INT64_MAX / 1000;
        if (delta_value <= limit) {
            limit = 0;
            delta_value *= 1000;
        }
        break;

      case RWREC_FIELD_ELAPSED:
      case RWREC_FIELD_ELAPSED_MSEC:
        /* max elapsed is UINT32_MAX milliseconds.  multiply user's
         * value by 1000 to convert from seconds to milliseconds. */
        limit = UINT32_MAX / 1000;
        if (delta_value <= limit) {
            limit = 0;
            delta_value *= 1000;
        }
        break;

      case RWREC_FIELD_PKTS:
      case RWREC_FIELD_BYTES:
        /* these values hold uint32_t */
        limit = UINT32_MAX - 1;
        break;

      case RWREC_FIELD_SPORT:
      case RWREC_FIELD_DPORT:
      case RWREC_FIELD_APPLICATION:
      case RWREC_FIELD_SID:
      case RWREC_FIELD_INPUT:
      case RWREC_FIELD_OUTPUT:
      case RWREC_FIELD_ICMP_TYPE_CODE:
        /* these values hold uint16_t */
        limit = UINT16_MAX - 1;
        break;

      case RWREC_FIELD_PROTO:
        /* this is a uint8_t */
        limit = UINT8_MAX - 1;
        break;

      case RWREC_FIELD_FLAGS:
      case RWREC_FIELD_INIT_FLAGS:
      case RWREC_FIELD_REST_FLAGS:
      case RWREC_FIELD_TCP_STATE:
      case RWREC_FIELD_FTYPE_CLASS:
      case RWREC_FIELD_FTYPE_TYPE:
      default:
        /* these are nonsense */
        skAppPrintErr("Setting %s to '%s' is not supported",
                      appOptions[OPT_DELTA_FIELD].name, entry->name);
        goto END;
    }

    /* verify delta_value is not too large */
    if (limit != 0 && delta_value > limit) {
        skAppPrintErr("The maximum %s for the '%s' field is %" PRIu64,
                      appOptions[OPT_DELTA_VALUE].name,
                      entry->name, limit);
        goto END;
    }

    /* a delta value is required */
    if (0 == delta_value) {
        skAppPrintErr("Using the --%s switch requires a --%s",
                      appOptions[OPT_DELTA_FIELD].name,
                      appOptions[OPT_DELTA_VALUE].name);
        goto END;
    }

    /* success */
    rv = 0;

   END:
    if (iter) {
        skStringMapIterDestroy(iter);
    }
    return rv;
}


/*
 *  ok = createStringmaps();
 *
 *    Create the string-map to assist in parsing the --fields switch.
 */
static int createStringmaps(void)
{
    skplugin_field_iter_t  iter;
    skplugin_err_t         err;
    skplugin_field_t      *field_handle;
    sk_stringmap_status_t  map_err;
    sk_stringmap_entry_t   entry;
    const char           **field_names;
    const char           **name;
    uint32_t               max_id;

    /* initialize string-map of field identifiers: add default fields;
     * keep the millisecond fields so that SiLK applications take the
     * same switches; the seconds and milliseconds value map to the
     * same code. */
    if (rwAsciiFieldMapAddDefaultFields(&key_field_map)) {
        skAppPrintErr("Unable to setup fields stringmap");
        exit(EXIT_FAILURE);
    }
    max_id = RWREC_PRINTABLE_FIELD_COUNT - 1;

    /* add --fields from plug-ins */
    err = skPluginFieldIteratorBind(&iter, SKPLUGIN_FN_REC_TO_BIN, 1);
    if (err != SKPLUGIN_OK) {
        assert(err == SKPLUGIN_OK);
        skAppPrintErr("Unable to bind plugin field iterator");
        return -1;
    }

    while (skPluginFieldIteratorNext(&iter, &field_handle)) {
        skPluginFieldName(field_handle, &field_names);
        ++max_id;

        /* Add fields to the key_field_map */
        for (name = field_names; *name; name++) {
            memset(&entry, 0, sizeof(entry));
            entry.name = *name;
            entry.id = max_id;
            entry.userdata = field_handle;
            map_err = skStringMapAddEntries(key_field_map, 1, &entry);
            if (map_err != SKSTRINGMAP_OK) {
                const char *plugin_name;
                skPluginFieldGetPluginName(field_handle, &plugin_name);
                skAppPrintErr(("Plug-in cannot add field named '%s': %s."
                               " Plug-in file: %s"),
                              *name, skStringMapStrerror(map_err),plugin_name);
                return -1;
            }
        }
    }

    return 0;
}


/*
** Local Variables:
** mode:c
** indent-tabs-mode:nil
** c-basic-offset:4
** End:
*/
