#  Copyright (C) 2009-2013 by Carnegie Mellon University.
#
#  See end of file for License
#######################################################################
#  SiLKTests.pm
#
#  Mark Thomas
#  March 2009
#
#######################################################################
#  RCSIDENT("$SiLK: SiLKTests.pm af906ffa5f49 2013-05-13 20:32:26Z mthomas $")
#######################################################################
#
#    Perl module used by the scripts that "make check" runs.
#
#######################################################################
#
#    This module is also used by the various make-tests.pl scripts
#    that generate the tests that "make check" runs.
#
#    In make-tests.pl, each test is defined by a tuple, which contains
#    two positional arguments and multiple keyed arguments.  The first
#    two arguments must be (1)the test-name, (2)a variable denoting
#    the type of test:
#
#    -- $SiLKTests::STATUS
#
#       An exit-status-only test.  The command is run and its exit
#       status is compared to see if it matches the expected value.
#       If the values match the test passes; otherwise it fails.
#
#    -- $SiLKTests::MD5
#
#       The command is run and its output is captured.  If the command
#       fails to run or exit with a non-zero exit status, the test
#       fails.  The md5 checksum of the output is computed and
#       compared with the expected value.  If the values match the
#       test passes; otherwise it fails.
#
#    -- $SiLKTests::ERR_MD5
#
#       The same as $SiLKTests::MD5, except that the command MUST exit
#       with a non-zero exit status.  One use of this is to determine
#       whether a command is failing with the correct error message.
#
#    -- $SiLKTests::CMP_MD5
#
#       Runs two different commands and determines whether the MD5
#       checksum of their outputs is identical.  If they are
#       identical, the test passes.  If either command exits with a
#       non-zero exit status, the test fails.
#
#    The remaining members of the tuple are key/value pairs, where the
#    keys and their values are (note that keys include the single
#    leading hyphen):
#
#    -file
#
#        Array reference of data file keys the test uses.  The test
#        should refer to them as $file{key}.  These files must exist
#        in the $top_builddir/tests/. directory.  If these files do
#        not exist when the test is run, the test will be skipped.
#
#    -app
#
#        Array reference of other SiLK applications required by the
#        test.  The test should refer to them by name, e.g., $rwcat.
#        If these applications do not exist when the test is run, the
#        test will be skipped.
#
#    -env
#
#        Hash reference of environment variable names and values that
#        should be set.
#
#    -lib
#
#        Array reference of directories.  The LD_LIBRARY_PATH, or
#        equivalent, will be set to include these directories.  Used
#        to test plug-in support.
#
#    -temp
#
#        Array reference of name keys.  These will be replaced by
#        temporary files, with the same name being mapped to the same
#        file.  The test should refer to them as $temp{key}.  The test
#        should not rely on the file name, since that will differ with
#        each run.
#
#    -features
#
#        Array reference of feature keys.  This adds a call to the
#        check_features() function in the generated script, and uses
#        the elements in the array reference as the arguments to the
#        function.  See the check_features() function for the
#        supported list of keys.
#
#    -pretest
#
#        Adds arbitrary code to the generated test.
#
#    -exit77
#
#        Text to add directly to the test file being created.  When
#        the test is run, this text will be treated as the body of a
#        subroutine to be called with no arguments.  If the sub
#        returns TRUE, the test will exit with exit code 77.  This is
#        a way to skip tests for features that may not have been
#        compiled into SiLK.
#
#    -cmd
#
#        Either a single command string or an array reference
#        containing one or more command strings.
#
#######################################################################
#
#    Environment variables affecting tests
#
#    -- SK_TESTS_VERBOSE
#
#       Print commands before they get executed.  If this value is not
#       specified in the environment, it defaults to TRUE.
#
#    -- SK_TESTS_SAVEOUTPUT
#
#       Normally once the output of the command is used to compute the
#       MD5 checksum, the output is forgotten.  When this variable is
#       set, the output used to compute the MD5 checksum is saved.
#       This variable also prevents the removal of any temporary files
#       that were used by the command
#
#    -- SK_TESTS_MAKEFILE
#
#       Used by make-tests.pl.  When this variable is set, the new
#       TESTS and XFAIL_TESTS variables are appended to the
#       Makefile.am file.  The user should remove the previous values
#       before running automake.
#
#    -- SK_TESTS_CHECK_MAKEFILE
#
#       Similar to SK_TESTS_MAKEFILE, except the Makefile.am file is
#       not updated.  Instead, a message is printed noting how the
#       TESTS and XFAIL_TESTS variables need to be modified.
#
#######################################################################

package SiLKTests;

use strict;
use warnings;
use Carp;
use IO::Socket::INET qw();
use File::Temp;


END {
}


BEGIN {
    our $PWD = `pwd`;
    chomp $PWD;
    our $top_srcdir = $ENV{top_srcdir};
    defined $top_srcdir
        or die "Must set top_srcdir environment variable\n";
    our $top_builddir = $ENV{top_builddir};
    defined $top_builddir
        or die "Must set top_builddir environment variable\n";
    our $testsdir = "$top_builddir/tests";
    if ($top_builddir) {
        require "$testsdir/config-vars.pm";
    }

    our $srcdir = $ENV{srcdir};

    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    # These define the type of tests to run.
    our $STATUS = 0;  # Just check exit status of command
    our $MD5 = 1;     # Check MD5 checksum of output against known value
    our $ERR_MD5 = 2; # Check MD5 checksum, expect cmd to exit non-zero
    our $CMP_MD5 = 3; # Compare the MD5 checksums of two commands

    our $NAME = $0;
    $NAME =~ s,.*/,,;

    # set the version for version checking
    $VERSION     = 1.00;

    @ISA         = qw(Exporter);
    @EXPORT      = qw(&add_plugin_dirs &check_app_switch
                      &check_exit_status &check_features
                      &check_md5_file &check_md5_output
                      &check_python_bin &check_silk_app &compute_md5
                      &get_data_or_exit77 &get_datafile
                      &get_ephemeral_port &make_packer_sensor_conf
                      &make_tempdir &make_tempname &make_test_scripts
                      &print_tests_hash &run_command &skip_test
                      $srcdir $top_srcdir
                      );
    %EXPORT_TAGS = ( );
    @EXPORT_OK   = ( );  #qw($Var1 %Hashit &func3);


    # Default to being verbose
    unless (defined $ENV{SK_TESTS_VERBOSE}) {
        $ENV{SK_TESTS_VERBOSE} = 1;
    }

    eval { require Digest::MD5; Digest::MD5->import; };
    if ($@) {
        skip_test("Digest::MD5 module not available");
    }

    # List of features used by check_features().
    our %feature_hash = (
        gnutls      => sub {
            skip_test("No GnuTLS support")
                unless 1 == $SiLKTests::SK_ENABLE_GNUTLS;
        },
        ipa         => sub {
            skip_test("No IPA support")
                unless 1 == $SiLKTests::SK_ENABLE_IPA;
        },
        ipfix       => sub {
            skip_test("No IPFIX support")
                unless 1 == $SiLKTests::SK_ENABLE_IPFIX;
        },
        inet6       => sub {
            skip_test("No IPv6 networking support")
                unless $SiLKTests::SK_ENABLE_INET6_NETWORKING;
        },
        ipset_v6    => sub {
            skip_test("No IPv6 IPset support")
                unless ($SiLKTests::SK_ENABLE_IPV6
                        && $SiLKTests::SK_ENABLE_SILK3_IPSETS);
        },
        ipv6        => sub {
            skip_test("No IPv6 Flow record support")
                unless $SiLKTests::SK_ENABLE_IPV6;
        },
        netflow9    => sub {
            skip_test("No NetFlow V9 support")
                unless ($SiLKTests::SK_ENABLE_IPFIX
                        && $SiLKTests::SK_ENABLE_NETFLOW9);
        },
        );

}
our @EXPORT_OK;

our $top_builddir;
our $top_srcdir;
our $testsdir;
our $srcdir;
our $STATUS;
our $MD5;
our $ERR_MD5;
our $CMP_MD5;
our $NAME;

# whether to print contents of scripts that get run
our $DEBUG_SCRIPTS = $ENV{SK_TESTS_DEBUG_SCRIPTS};

# how to indent each line of the output
our $INDENT = "    ";

# ensure all commands are run in UTC timezone
$ENV{TZ} = "0";

# specify silk.conf file to use for all tests
$ENV{SILK_CONFIG_FILE} = "$top_builddir/tests/silk.conf";

# do not put the SiLK version number into binary SiLK files
$ENV{SILK_HEADER_NOVERSION} = 1;

# unset several environment variables
for my $e (qw(SILK_IPV6_POLICY SILK_PYTHON_TRACEBACK SILK_RWFILTER_THREADS
              SILK_LOGSTATS_RWFILTER SILK_LOGSTATS SILK_LOGSTATS_DEBUG))
{
    delete $ENV{$e};
}

# run the C locale
$ENV{LANG} = 'C';
$ENV{LC_ALL} = 'C';

# disable creating of *.pyc files in Python 2.6+
$ENV{PYTHONDONTWRITEBYTECODE} = 1;

my %test_files = (
    empty           => "$testsdir/empty.rwf",
    data            => "$testsdir/data.rwf",
    v6data          => "$testsdir/data-v6.rwf",
    scandata        => "$testsdir/scandata.rwf",

    v4set1          => "$testsdir/set1-v4.set",
    v4set2          => "$testsdir/set2-v4.set",
    v4set3          => "$testsdir/set3-v4.set",
    v4set4          => "$testsdir/set4-v4.set",
    v6set1          => "$testsdir/set1-v6.set",
    v6set2          => "$testsdir/set2-v6.set",
    v6set3          => "$testsdir/set3-v6.set",
    v6set4          => "$testsdir/set4-v6.set",

    v4bag1          => "$testsdir/bag1-v4.bag",
    v4bag2          => "$testsdir/bag2-v4.bag",
    v4bag3          => "$testsdir/bag3-v4.bag",
    v6bag1          => "$testsdir/bag1-v6.bag",
    v6bag2          => "$testsdir/bag2-v6.bag",
    v6bag3          => "$testsdir/bag3-v6.bag",

    address_types   => "$testsdir/address_types.pmap",
    fake_cc         => "$testsdir/fake-cc.pmap",
    v6_fake_cc      => "$testsdir/fake-cc-v6.pmap",
    ip_map          => "$testsdir/ip-map.pmap",
    v6_ip_map       => "$testsdir/ip-map-v6.pmap",
    proto_port_map  => "$testsdir/proto-port-map.pmap",

    pysilk_plugin   => "$top_srcdir/tests/pysilk-plugin.py",

    pdu_small       => "$testsdir/small.pdu",
);


#  $path = get_datafile('key')
#
#    Return the path to the data file named by 'key'.  If $key is
#    invalid or if the file does not exist, return undef.
#
sub get_datafile
{
    my ($arg) = @_;

    my $file = $test_files{$arg};
    unless (defined $file) {
        return undef;
    }
    unless (-f $file) {
        return undef;
    }
    return $file;
}


#  $path = get_data_or_exit77('key');
#
#    Like get_datafile(), but exits the program with code 77 if the
#    file does not exist.  This would cause "make check" to skip the
#    test.
#
sub get_data_or_exit77
{
    my ($arg) = @_;

    my $file = get_datafile($arg);
    if (!$file) {
        skip_test("Did not find '$arg' file");
    }
    return $file;
}


#  $env = dump_env();
#
#    Return a string specifying any environment variables that may
#    affect this run
#
sub dump_env
{
    join " ", ((map {"$_=$ENV{$_}"}
                grep {defined $ENV{$_}}
                qw(top_srcdir top_builddir srcdir
                   TZ LANG LC_ALL SILK_HEADER_NOVERSION
                   SILK_DATA_ROOTDIR SILK_CONFIG_FILE
                   SILK_COUNTRY_CODES SILK_ADDRESS_TYPES
                   RWSCANRC PYTHONPATH
                   LD_LIBRARY_PATH DYLD_LIBRARY_PATH LIBPATH SHLIB_PATH)),
               "");
}


#  skip_test($msg);
#
#    Print a message indicating that the test is being skipped due to
#    '$msg' and then exit with status 77.
#
sub skip_test
{
    my ($msg) = @_;
    if ($ENV{SK_TESTS_VERBOSE}) {
        if (!$msg) {
            warn "$NAME: Skipping test\n";
        }
        else {
            warn "$NAME: Skipping test: $msg\n";
        }
    }
    exit 77;
}


#  $dir = make_tempdir();
#
#    Make a temporary directory and return its location.  This will
#    remove the directory on exit unless the appropriate environment
#    variable is set.
#
#    If a temporary directory cannot be created, exit with status 77.
#
sub make_tempdir
{
    my $tmpdir = File::Temp::tempdir(CLEANUP => !$ENV{SK_TESTS_SAVEOUTPUT});
    unless (-d $tmpdir) {
        skip_test("Unable to create temporary directory");
    }
    return $tmpdir;
}


#  $path = make_tempname($key);
#
#    Return a path to a temporary file.  Calls to this function with
#    the same $key return the same name.
#
sub make_tempname
{
    my ($key) = @_;

    our $tmpdir;
    our %TEMP_MAP;

    unless (defined $tmpdir) {
        $tmpdir = File::Temp::tempdir(CLEANUP => !$ENV{SK_TESTS_SAVEOUTPUT});
        unless (-d $tmpdir) {
            skip_test("Unable to create tempdir");
        }
    }

    # change anything other than -, _, ., and alpha-numerics to a
    # single underscore
    $key =~ tr/-_.0-9A-Za-z/_/cs;

    unless (exists $TEMP_MAP{$key}) {
        $TEMP_MAP{$key} = "$tmpdir/$key";
    }
    return $TEMP_MAP{$key};
}


#  run_command($cmd, $callback);
#
#    Run $cmd in a subshell.  $callback should be a function that
#    takes two arguments.  The first argument is a file handle from
#    which the standard output of the $cmd can be read.
#
#    The second argument may be undefined.  If it is defined, the
#    SK_TESTS_SAVEOUTPUT environment variable is set, and the argument
#    contains the name of the (unopened) file to which results should
#    be written.  The individual test can determine which data to
#    write to this file.
#
#    This function returns 0.
#
sub run_command
{
    my ($cmd, $callback) = @_;

    my $save_file;
    if ($ENV{SK_TESTS_SAVEOUTPUT}) {
        $save_file = "tests-$NAME.txt";
    }

    if ($ENV{SK_TESTS_VERBOSE}) {
        print STDERR "RUNNING: ", dump_env(), $cmd, "\n";
    }
    my $io;
    unless (open $io, "$cmd |") {
        die "$NAME: cannot run '$cmd': $!\n";
    }
    binmode($io);
    $callback->($io, $save_file);
    return 0;
}


#  $ok = compute_md5(\$md5, $cmd, $expect_err);
#
#    Run $cmd in a subshell, compute the MD5 of the output, and store
#    the hex-encoded MD5 in $md5.  Dies if $cmd cannot be run.  If
#    $expect_err is false, function dies if the command exits
#    abnormally.  If $expect_err is true, function dies if command
#    exits normally.
#
sub compute_md5
{
    my ($md5_out, $cmd, $expect_err) = @_;

    # make certain $expect_err has a value
    unless (defined $expect_err) {
        $expect_err = 0;
    }

    if ($ENV{SK_TESTS_VERBOSE}) {
        print STDERR "RUNNING: ", dump_env(), $cmd, "\n";
    }
    my $md5 = Digest::MD5->new;
    my $io;
    unless (open $io, "$cmd |") {
        die "$NAME: cannot run '$cmd': $!\n";
    }
    binmode($io);
    if (!$ENV{SK_TESTS_SAVEOUTPUT}) {
        $md5->addfile($io);
    } else {
        my $txt = "tests-$NAME.txt";
        open TXT, ">$txt"
            or die "$NAME: Cannot open output file '$txt': $!\n";
        binmode TXT;
        while(<$io>) {
            print TXT;
            $md5->add($_);
        }
        close TXT
            or die "$NAME: Cannot close file '$txt': $!\n";
    }
    (!!close($io) ^ $expect_err)
        or exit 1;
    (!$? ^ $expect_err)
        or exit 1;
    $$md5_out = $md5->hexdigest;
    return 0;
}


#  check_md5_output($expect_md5, $cmd, $expect_err);
#
#    Die if the MD5 of the output from running $cmd does not equal
#    $expect_md5.  $cmd and $expect_err are passed through to
#    compute_md5().
#
sub check_md5_output
{
    my ($expect, $cmd, $expect_err) = (@_);

    my $md5;
    my $err = compute_md5(\$md5, $cmd, $expect_err);
    if ($expect ne $md5) {
        die "$NAME: checksum mismatch [$md5] ($cmd)\n";
    }
}


#  $ok = check_md5_file($expect_md5, $file);
#
#    Compute the MD5 checksum of $file and compare it to the value in
#    $expect_md5.  Die if the values are not identical.
#
sub check_md5_file
{
    my ($expect, $file) = @_;

    my $md5 = Digest::MD5->new;
    my $io;
    unless (open $io, $file) {
        die "$NAME: cannot open '$file': $!\n";
    }
    binmode($io);
    $md5->addfile($io);
    close($io);

    my $md5_hex = $md5->hexdigest;
    if ($expect ne $md5_hex) {
        die "$NAME: checksum mismatch [$md5_hex] ($file)\n";
    }
    return 0;
}


#  $app = check_silk_app($name)
#
#    Find the SiLK application named $name and return a path to it.
#
#    If an environment variable exists whose name is the uppercase
#    version of $name, that value is returned.  Otherwise, the
#    subroutine looks for $name in "../$name/$name", where the
#    directory name may be altered slightly depending on $name.
#
#    Exit with status 77 if the application does not exist.
#
sub check_silk_app
{
    my ($name) = @_;

    if ($ENV{"\U$name"}) {
        return $ENV{"\U$name"};
    }

    my $path = "../$name/$name";
    if ($name =~ /^(rwset|rwbag|rwipa|rwpmap)/) {
        $path = "../$1/$name";
    }
    elsif ($name =~ /^rwip2cc$/) {
        $path = "../rwpmap/$name";
    }
    elsif ($name =~ /^rwallformats$/) {
        $path = "../rwconvert/$name";
    }
    elsif ($name =~ /^rwrtd2split$/) {
        $path = "../rwconvert/$name";
    }
    elsif ($name =~ /^rwfglob$/) {
        $path = "../rwfilter/$name";
    }
    elsif ($name =~ /^rwipfix2silk$/) {
        $path = "../rwipfix/$name";
    }
    elsif ($name =~ /^rwsilk2ipfix$/) {
        $path = "../rwipfix/$name";
    }
    elsif ($name =~ /^rwdedupe$/) {
        $path = "../rwsort/$name";
    }

    if (-x $path) {
        return $path;
    }
    skip_test("Did not find application '$path'");
}


#  check_features(@list)
#
#    Check whether SiLK was compiled with support for each of the
#    features in '@list'.  If any feature in @list is not present,
#    exit with status 77.
#
#    The acceptable names for '@list' are:
#
#    gnutls    -- verify that GnuTLS support is available
#    ipa       -- verify that support for libipa is available
#    ipfix     -- verify that IPFIX support is available
#    inet6     -- verify that IPv6 networking support is available
#    ipset_v6  -- verify that IPv6 IPsets is available
#    ipv6      -- verify that IPv6 Flow record support is available
#    netflow9  -- verify that support for NetFlow V9 is available
#
#    If any other name is provided, exit with an error.
#
#    TODO: Idea for an extension, which we currently do not need:
#    Preceding a feature name with '!' causes the script to exit with
#    status 77 if the feature IS present.
#
sub check_features
{
    my (@list) = @_;

    our %feature_hash;

    for my $feature (@list) {
        my $check = $feature_hash{$feature};
        if (!$check) {
            die "$NAME: No such feature as '$feature'\n";
        }
        $check->();
    }
}


#  check_app_switch($app, $switch, $re)
#
#    Check the output of the --help switch.  This function invokes
#    "$app --help" and captures the output.  '$app' should be the
#    application name, and it may include switches.  The function
#    searches the output for the switch '--$switch'.  If '$re' is
#    undefined, the function returns true if the switch is found.
#    When '$re' is defined, the help text of the switch is regex
#    matched with '$re', and the result of the match is returned.
#
#    The function returns false if the running the application fails.
#
sub check_app_switch
{
    my ($app, $switch, $re) = @_;
    my $output = `$app --help 2>&1`;
    if ($?) {
        return 0;
    }
    my $text;
    if ($output =~ m/^(--$switch[^-\w].*(\n[^-].+)*)/m) {
        $text = $1;
        unless (defined $re) {
            return 1;
        }
        if ($text =~ $re) {
            return 1;
        }
    }
    return 0;
}


#  check_exit_status($cmd)
#
#    Run $cmd.  Return 1 if the command succeeded, or 0 if it failed.
#    Discards stdout and stderr.
#
sub check_exit_status
{
    my ($cmd) = @_;

    if ($ENV{SK_TESTS_VERBOSE}) {
        print STDERR "RUNNING: ", dump_env(), $cmd, "\n";
    }
    $cmd .= " >/dev/null 2>&1";
    system $cmd;
    if ($? == 0) {
        return 1;
    }
    return 0;
}


#  check_python_bin()
#
#    Check whether we found a python interpreter.  If we did not,
#    exit 77.  Otherwise, prefix any existing PYTHONPATH with the
#    proper directories and return 1.
#
sub check_python_bin
{
    if ($SiLKTests::PYTHON eq "no"
        || $SiLKTests::PYTHON_VERSION !~ /^[23]/
        || $SiLKTests::PYTHON_VERSION =~ /^2.[45]/)
    {
        skip_test("Python unset or not >= 2.6 < 4.0");
    }
    $ENV{PYTHONPATH} = join ":", ($SiLKTests::top_builddir.'/tests',
                                  $SiLKTests::srcdir.'/tests',
                                  $SiLKTests::top_srcdir.'/tests',
                                  ($ENV{PYTHONPATH} ? $ENV{PYTHONPATH} : ()));
    return 1;
}


#  add_plugin_dirs(@libs)
#
#    For each directory in @libs, prefix the directory name and the
#    directory name with "/.libs" appended to it to the
#    LD_LIBRARY_PATH (and platform variations of that environment
#    variable).
#
#    Each directory in @libs should relative to the top of the build
#    tree.
#
sub add_plugin_dirs
{
    my (@dirs) = (@_);

    my $newlibs = join (":",
                        map {"$_:$_/.libs"}
                        map {"$SiLKTests::top_builddir$_"}
                        @dirs);
    for my $L (qw(LD_LIBRARY_PATH DYLD_LIBRARY_PATH LIBPATH SHLIB_PATH)) {
        if ($ENV{$L}) {
            $ENV{$L} = $newlibs.":".$ENV{$L};
        }
        else {
            $ENV{$L} = $newlibs;
        }
    }
}


#  $port = get_ephemeral_port($host, $proto);
#
#    Get an ephemeral port by creating a short-lived server listening
#    on the specified $host and $protocol, and using the fact that
#    binding to port 0 assigns an available ephemeral port.
#
#    If the short-lived server cannot be created, the program exits
#    with status 77.
#
sub get_ephemeral_port
{
    use Socket ();

    my ($host, $proto) = @_;
    my $type = Socket::SOCK_DGRAM;

    unless ($host) {
        $host = '127.0.0.1';
    }
    unless ($proto) {
        $proto = getprotobyname('tcp');
        $type = Socket::SOCK_STREAM;
    }
    else {
        if ($proto =~ /\D/) {
            $proto = getprotobyname($proto);
        }
        if (getprotobyname('tcp') == $proto) {
            $type = Socket::SOCK_STREAM;
        }
    }

    if ($host =~ /:/) {
        # IPv6; run in an eval, in case Socket6 is not available.
        my $have_socket6 = 0;

        my $port = eval <<EOF;
        use Socket  qw(SOL_SOCKET SO_REUSEADDR);
        use Socket6 qw(getaddrinfo getnameinfo AF_INET6
                       NI_NUMERICHOST NI_NUMERICSERV);

        \$have_socket6 = 1;

        my (\$s, \$port);
        unless (socket(\$s, AF_INET6, $type, $proto)) {
            skip_test("Unable to open socket: \$!)";
        }
        setsockopt(\$s, SOL_SOCKET, SO_REUSEADDR, 1);

        my \@res = getaddrinfo('$host', 0, AF_INET6, $type, $proto);
        if (\$#res == 0) {
            skip_test("Unable to resolve '$host'");
        }

        my \$s_addr = \$res[3];
        unless (bind(\$s, \$s_addr)) {
            skip_test("Unable to bind to socket: \$!");
        }

        (undef, \$port) = getnameinfo(getsockname(\$s),
                                      (NI_NUMERICHOST | NI_NUMERICSERV));
        close(\$s);

        return \$port;
EOF
        if (defined $port) {
            return $port;
        }
        if ($@ && $have_socket6) {
            skip_test("$@");
        }
        # Assume failure was due to absence of Socket6 module, and use
        # IPv4 to get a port
        $host = '127.0.0.1';
    }

    # IPv4

    my ($s, $port);
    unless (socket($s, Socket::PF_INET, $type, $proto)) {
        skip_test("Unable to open socket: $!");
    }
    setsockopt($s, Socket::SOL_SOCKET, Socket::SO_REUSEADDR, 1);

    my $h_addr = Socket::inet_aton($host);
    unless (defined $h_addr) {
        skip_test("Unable to resolve '$host'");
    }
    my $s_addr = Socket::sockaddr_in(0, $h_addr);

    unless (bind($s, $s_addr)) {
        skip_test("Unable to bind to socket: $!");
    }

    ($port, ) = Socket::sockaddr_in(getsockname($s));
    close($s);

    return $port;

    # The following does the same as the above
    #
    # my $s = IO::Socket::INET->new(Proto =>     $proto,
    #                               LocalAddr => $host,
    #                               LocalPort=>  0,
    #                               Reuse =>     1,
    #     );
    # unless ($s) {
    #     if ($ENV{SK_TESTS_VERBOSE}) {
    #         warn "$NAME: Cannot create $proto server on $host: $!\n";
    #     }
    #     exit 77;
    # }
    # my $port = $s->sockport;
    # $s->close();
    #
    # return $port;
}


#  santize_cmd(\$cmd);
#
#    Remove any references to the source path or to the temporary
#    directory from '$cmd' prior to $cmd into the top of the source
#    file
#
sub sanitize_cmd
{
    my ($cmd) = @_;

    # don't put source path into test file
    $$cmd =~ s,\Q$top_srcdir,$top_builddir,g;

    # don't put TMPDIR into test file
    my $tmp = $ENV{TMPDIR} || '/tmp';

    # convert "$TMPDIR/foobar/file" to "/tmp/file"
    $$cmd =~ s,\Q$tmp\E/?(\S+/)*,/tmp/,g;
}


sub make_test_scripts
{
    my ($APP, $test_tuples, $tests_list_hash) = @_;

    my @temp_param = ('make-tests-XXXXXXXX',
                      UNLINK => 1,
                      DIR => File::Spec->tmpdir);

    if ($ENV{SK_TESTS_SAVEOUTPUT}) {
        $File::Temp::KEEP_ALL = 1;
    }

    my $APP_PATH;
    if ($APP =~ m,/,) {
        $APP_PATH = $APP;
        $APP =~ s,.*/,,;
    }
    else {
        $APP_PATH = "./$APP";
    }

    my @test_list;
    my @xfail_list;

    our (%global_tests);

  TUPLE:
    while (defined(my $tuple = shift @$test_tuples)) {
        # first two arguments in tuple are positional
        my $test_name = shift @$tuple;
        my $test_type = shift @$tuple;

        # print the name of the file to create
        my $outfile = "$APP-$test_name.pl";
        print "Creating $outfile\n";

        # others are in tuple are by keyword
        my ($file_keys, $app_keys, $env_hash, $lib_list, $temp_keys,
            $feature_list, $exit77, $pretest, @cmd_list);
        while (defined (my $k = shift @$tuple)) {
            if ($k =~ /^-files?/) {
                $file_keys = shift @$tuple;
            }
            elsif ($k =~ /^-apps?/) {
                $app_keys  = shift @$tuple;
            }
            elsif ($k =~ /^-env/) {
                $env_hash  = shift @$tuple;
            }
            elsif ($k =~ /^-libs?/) {
                $lib_list  = shift @$tuple;
            }
            elsif ($k =~ /^-temps?/) {
                $temp_keys = shift @$tuple;
            }
            elsif ($k =~ /^-cmds?/) {
                my $tmp = shift @$tuple;
                if ('ARRAY' eq ref($tmp)) {
                    @cmd_list = @$tmp;
                } else {
                    @cmd_list = ($tmp);
                }
            }
            elsif ($k =~ /^-features?/) {
                $feature_list = shift @$tuple;
            }
            elsif ($k =~ /^-exit77/) {
                $exit77 = shift @$tuple;
            }
            elsif ($k =~ /^-pretest/) {
                $pretest = shift @$tuple;
            }
            elsif ($k =~ /^-/) {
                croak "$NAME: Unknown tuple key '$k'";
            }
            else {
                croak "$NAME: Expected to find key in tuple";
            }
        }

        # add file to create to our output list
        $outfile = "tests/$outfile";
        push @test_list, $outfile;
        $outfile = "$srcdir/$outfile";

        if ($global_tests{$outfile}) {
            carp "\nWARNING!! Duplicate test '$outfile'\n";
        }
        $global_tests{$outfile} = 1;

        # the body of the test file we are writing
        my $test_body = <<EOF;
#! /usr/bin/perl -w
#HEADER
use strict;
use SiLKTests;

my \$$APP = \$ENV{\U$APP\E} || '$APP_PATH';
EOF

        # the body of the string we eval to get the test command
        my $run_body = "my \$$APP = '$APP_PATH';\n";

        # handle any required applications
        if ($app_keys && @$app_keys) {
            for my $key (@$app_keys) {
                my $app = check_silk_app($key);
                if (!$app) {
                    die "$NAME: No $app";
                }
                $run_body .= "my \$$key = '$app';\n";
                $test_body .= "my \$$key = check_silk_app('$key');\n";
            }
        }

        # handle any required data files
        if ($file_keys && @$file_keys) {
            $run_body .= "my \%file;\n";
            $test_body .= "my \%file;\n";
            for my $key (@$file_keys) {
                my $file = get_datafile($key);
                if (!$file) {
                    # Skip V6 when built without V6
                    if ($key eq 'v6data' && $SiLKTests::SK_ENABLE_IPV6 == 0) {
                        warn $INDENT, "Skipping V6 test\n";
                        next TUPLE;
                    }
                    die "$NAME: No file '$key'";
                }
                $run_body .= "\$file{$key} = '$file';\n";
                $test_body .= "\$file{$key} = get_data_or_exit77('$key');\n";
            }
        }

        # handle any necessary temporary files
        if ($temp_keys && @$temp_keys) {
            $run_body .= "my \%temp;\n";
            $test_body .= "my \%temp;\n";
            for my $key (@$temp_keys) {
                my $temp = make_tempname("$APP-$test_name-$key");
                if (!$temp) {
                    die "$NAME: No temp '$APP-$test_name-$key'";
                }
                # make certain to start with a clean slate
                unlink $temp;
                $run_body .= "\$temp{$key} = '$temp';\n";
                $test_body .= "\$temp{$key} = make_tempname('$key');\n";
            }
        }

        # Set any environment variables
        if ($env_hash) {
            while (my ($var, $val) = each %$env_hash) {
                $test_body .= "\$ENV{$var} = $val;\n";
                $run_body .= "\$ENV{$var} = $val;\n";
            }
        }

        # Set the LD_LIBRARY_PATH
        if ($lib_list) {
            my $new_libs = join ", ", map {"'/$_'"} @$lib_list;
            my $libs_expr .= <<EOF;
add_plugin_dirs($new_libs);
EOF

            $test_body .= $libs_expr;
            $run_body .= $libs_expr;
        }

        # Add feature checks
        if ($feature_list && @$feature_list) {
            $test_body .= <<EOF;
check_features(qw(@$feature_list));
EOF
        }
        if ($exit77) {
            $test_body .= <<EOF;

exit 77 if sub { $exit77 }->();

EOF
        }

        # add any extra code
        if ($pretest) {
            $run_body .= "\n$pretest\n";
            $test_body .= "\n$pretest\n";
        }

        # This gets filled in by the various test types
        my $header = "\n";

        # run the test, which depends on its type
        if ($test_type == $STATUS) {
            if (@cmd_list > 1) {
                croak "$NAME: Too many commands\n";
            }
            my $cmd = shift @cmd_list;

            my ($fh, $tmp_cmd) = File::Temp::tempfile(@temp_param);

            # make $fh unbuffered
            select((select($fh), $| = 1)[0]);
            print $fh <<EOF;
use strict;
do "$INC{'SiLKTests.pm'}";
import SiLKTests;
$run_body
exec "$cmd"
EOF

            if ($DEBUG_SCRIPTS) {
                print $INDENT, "****\n";
                seek $fh, 0, 0;
                while (defined (my $line = <$fh>)) {
                    print $INDENT, $line;
                }
                print $INDENT, "****\n";
            }

            # the run_body returns the string containing the test to run
            my %OLD_ENV = (%ENV);
            $run_body .= qq/"$cmd"/;
            my $run_cmd = eval "$run_body"
                or croak "ERROR! '$cmd'\n$@";
            %ENV = (%OLD_ENV);

            print $INDENT, "Invoking $run_cmd\n";
            my $status = check_exit_status("perl $tmp_cmd");
            my ($status_str, $exit_conditions);
            if (!$status) {
                $status_str = 'ERR';
                $exit_conditions = '? 1 : 0';
            }
            else {
                $status_str = 'OK';
                $exit_conditions = '? 0 : 1';
            }
            print $INDENT, "[$status_str]\n";

            sanitize_cmd(\$run_cmd);

            # store the test string in the test itself
            $header = <<EOF;
# STATUS: $status_str
# TEST: $run_cmd
EOF

            $test_body .= <<EOF;
my \$cmd = "$cmd";

exit (check_exit_status(\$cmd) $exit_conditions);
EOF
        }

        elsif ($test_type == $MD5 || $test_type == $ERR_MD5) {
            if (@cmd_list > 1) {
                croak "$NAME: Too many commands\n";
            }
            my $cmd = shift @cmd_list;

            my ($fh, $tmp_cmd) = File::Temp::tempfile(@temp_param);

            # make $fh unbuffered
            select((select($fh), $| = 1)[0]);
            print $fh <<EOF;
use strict;
do "$INC{'SiLKTests.pm'}";
import SiLKTests;
$run_body
exec "$cmd"
EOF

            if ($DEBUG_SCRIPTS) {
                print $INDENT, "****\n";
                seek $fh, 0, 0;
                while (defined (my $line = <$fh>)) {
                    print $INDENT, $line;
                }
                print $INDENT, "****\n";
            }

            # the run_body returns the string containing the test to run
            my %OLD_ENV = (%ENV);
            $run_body .= qq/"$cmd"/;
            my $run_cmd = eval "$run_body"
                or croak "ERROR! '$cmd'\n$@";
            %ENV = (%OLD_ENV);

            my $expect_err = "";
            if ($test_type == $ERR_MD5) {
                $expect_err = ", 1";
            }

            my $test_type_str = (($test_type == $MD5) ? "MD5" : "ERR_MD5");

            print $INDENT, "Invoking $run_cmd\n";
            my $md5;
            compute_md5(\$md5, "perl $tmp_cmd", !!$expect_err);
            print $INDENT, "[$md5]\n";

            sanitize_cmd(\$run_cmd);

            # store the test string in the test itself
            $header = <<EOF;
# $test_type_str: $md5
# TEST: $run_cmd
EOF

            $test_body .= <<EOF;
my \$cmd = "$cmd";
my \$md5 = "$md5";

check_md5_output(\$md5, \$cmd$expect_err);
EOF
        }

        elsif ($test_type == $CMP_MD5) {
            my @expanded_cmd = ();

            my $run_body_orig = $run_body;
            for my $cmd (@cmd_list) {

                my ($fh, $tmp_cmd) = File::Temp::tempfile(@temp_param);

                # make $fh unbuffered
                select((select($fh), $| = 1)[0]);
                print $fh <<EOF;
use strict;
do "$INC{'SiLKTests.pm'}";
import SiLKTests;
$run_body_orig
exec "$cmd"
EOF

                if ($DEBUG_SCRIPTS) {
                    print $INDENT, "****\n";
                    seek $fh, 0, 0;
                    while (defined (my $line = <$fh>)) {
                        print $INDENT, $line;
                    }
                    print $INDENT, "****\n";
                }

                my %OLD_ENV = (%ENV);
                $run_body = $run_body_orig . qq/"$cmd"/;
                my $run_cmd = eval "$run_body"
                    or croak "ERROR! '$cmd'\n$@";
                %ENV = (%OLD_ENV);

                print $INDENT, "Invoking $run_cmd\n";
                my $md5;
                compute_md5(\$md5, "perl $tmp_cmd");
                print $INDENT, "[$md5]\n";

                sanitize_cmd(\$run_cmd);

                push @expanded_cmd, $run_cmd;
            }

            $header = join("\n# TEST: ", '# CMP_MD5', @expanded_cmd)."\n";

            my $cmds_string = '"'.join(qq|",\n|.(' 'x12).'"', @cmd_list).'"';

            $test_body .= <<EOF;
my \@cmds = ($cmds_string);
my \$md5_old;

for my \$cmd (\@cmds) {
    my \$md5;
    compute_md5(\\\$md5, \$cmd);
    if (!defined \$md5_old) {
        \$md5_old = \$md5;
    }
    elsif (\$md5_old ne \$md5) {
        die "$APP-$test_name.pl: checksum mismatch [\$md5] (\$cmd)\\n";
    }
}
EOF
        }

        # fill in the header
        $test_body =~ s/^#HEADER/$header/m;

        open OUTFILE, "> $outfile"
            or die "$NAME: open $outfile: $!";
        print OUTFILE $test_body;
        close(OUTFILE)
            or die "$NAME: close $outfile: $!";
    }


    # Tests are complete.  Either put the values into the hash
    # reference that was passed in, or print the values ourselves

    if ('HASH' ne ref($tests_list_hash)) {
        print_tests_hash({TESTS => \@test_list, XFAIL_TESTS => \@xfail_list});
    }
    else {
        if (@test_list) {
            unless (exists $tests_list_hash->{TESTS}) {
                $tests_list_hash->{TESTS} = [];
            }
            push @{$tests_list_hash->{TESTS}}, @test_list;
        }
        if (@xfail_list) {
            unless (exists $tests_list_hash->{XFAIL_TESTS}) {
                $tests_list_hash->{XFAIL_TESTS} = [];
            }
            push @{$tests_list_hash->{XFAIL_TESTS}}, @xfail_list;
        }
    }
}


sub print_tests_hash
{
    my ($tests_list) = @_;

    for my $t (qw(TESTS XFAIL_TESTS)) {
        if (exists($tests_list->{$t}) && @{$tests_list->{$t}}) {
            print "$t = @{$tests_list->{$t}}\n";
        }
    }

    if ($ENV{SK_TESTS_MAKEFILE}) {
        my $makefile = "$srcdir/Makefile.am";
        if (-f $makefile) {
            print "Modifying $makefile\n";

            open MF, ">> $makefile"
                or croak "$NAME: Opening '$makefile' failed: $!";
            print MF "\n# Added by $NAME on ".localtime()."\n";
            for my $t (qw(TESTS XFAIL_TESTS)) {
                if (exists($tests_list->{$t}) && @{$tests_list->{$t}}) {
                    print MF join(" \\\n\t", "$t =", @{$tests_list->{$t}}),"\n";
                }
            }
            close MF
                or croak "$NAME: Closing '$makefile' failed: $!";
        }
    }

    if ($ENV{SK_TESTS_CHECK_MAKEFILE}) {
        my $makefile = "$srcdir/Makefile.am";
        if (-f $makefile) {
            print "Checking $makefile\n";

            my %make_lists = (TESTS => {}, XFAIL_TESTS => {});

            open MF, "$makefile"
                or croak "$NAME: Opening '$makefile' failed: $!";
            my $t;
            while (defined(my $line = <MF>)) {
                if ($line =~ /^(TESTS|XFAIL_TESTS) *= *\\/) {
                    $t = $1;
                    next;
                }
                next unless $t;
                if ($line =~ /^[ \t]*(\S+)(| \\)$/) {
                    $make_lists{$t}{$1} = 1;
                    if (!$2) {
                        $t = undef;
                    }
                }
            }
            close MF;

            for my $t (qw(TESTS XFAIL_TESTS)) {
                my @missing;
                if (exists($tests_list->{$t})) {
                    for my $i (@{$tests_list->{$t}}) {
                        if (!$make_lists{$t}{$i}) {
                            push @missing, $i;
                        }
                        else {
                            delete $make_lists{$t}{$i};
                        }
                    }
                }
                my @extra = keys %{$make_lists{$t}};
                if (@missing) {
                    print "MISSING $t = @missing\n";
                }
                if (@extra) {
                    print "EXTRA $t = @extra\n";
                }
            }
        }
    }
}


sub make_packer_sensor_conf
{
    my ($sensor_conf, $probe_type, $port, @rest) = @_;

    my $sensor_template = "$srcdir/tests/sensors.conf";

    my %features;

    for my $f (@rest) {
        my $re = "\\#\U$f\\#";
        $features{$f} = qr/$re/;
    }

    unlink $sensor_conf;
    open SENSOR_OUT, ">$sensor_conf"
        or die;
    open SENSOR_IN, $sensor_template
        or die;
    while (defined (my $line = <SENSOR_IN>)) {
        $line =~ s/PROBETYPE/$probe_type/g;
        $line =~ s/RANDOMPORT/$port/g;
        for my $re (values %features) {
            $line =~ s/$re//g;
        }
        print SENSOR_OUT $line;
    }
    close SENSOR_OUT
        or die;
    close SENSOR_IN;
}


1;
__END__

#######################################################################
# Copyright (C) 2009-2013 by Carnegie Mellon University.
#
# @OPENSOURCE_HEADER_START@
#
# Use of the SILK system and related source code is subject to the terms
# of the following licenses:
#
# GNU Public License (GPL) Rights pursuant to Version 2, June 1991
# Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
#
# NO WARRANTY
#
# ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
# PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
# PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
# "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
# KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
# LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
# MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
# OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
# SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
# TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
# WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
# LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
# CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
# CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
# DELIVERABLES UNDER THIS LICENSE.
#
# Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
# Mellon University, its trustees, officers, employees, and agents from
# all claims or demands made against them (and any related losses,
# expenses, or attorney's fees) arising out of, or relating to Licensee's
# and/or its sub licensees' negligent use or willful misuse of or
# negligent conduct or willful misconduct regarding the Software,
# facilities, or other rights or assistance granted by Carnegie Mellon
# University under this License, including, but not limited to, any
# claims of product liability, personal injury, death, damage to
# property, or violation of any laws or regulations.
#
# Carnegie Mellon University Software Engineering Institute authored
# documents are sponsored by the U.S. Department of Defense under
# Contract FA8721-05-C-0003. Carnegie Mellon University retains
# copyrights in all material produced under this contract. The U.S.
# Government retains a non-exclusive, royalty-free license to publish or
# reproduce these documents, or allow others to do so, for U.S.
# Government purposes only pursuant to the copyright license under the
# contract clause at 252.227.7013.
#
# @OPENSOURCE_HEADER_END@
#######################################################################
