#! /usr/bin/perl -w
#
#
# RCSIDENT("$SiLK: rwflowpack-pack-silk-ipv6.pl 49656cafea96 2013-12-05 15:47:49Z mthomas $")

use strict;
use SiLKTests;
use File::Find;

my $rwflowpack = check_silk_app('rwflowpack');

# find the apps we need.  this will exit 77 if they're not available
my $rwcat = check_silk_app('rwcat');
my $rwsort = check_silk_app('rwsort');

# find the data files we use as sources, or exit 77
my %file;
$file{data} = get_data_or_exit77('data');
$file{v6data} = get_data_or_exit77('v6data');

# verify that required features are available
check_features(qw(ipv6));

# prefix any existing PYTHONPATH with the proper directories
check_python_bin();

# set the environment variables required for rwflowpack to find its
# packing logic plug-in
add_plugin_dirs('/site/twoway');

# Skip this test if we cannot load the packing logic
check_exit_status("$rwflowpack --sensor-conf=$srcdir/tests/sensor77.conf"
                  ." --verify-sensor-conf")
    or skip_test("Cannot load packing logic");

# create our tempdir
my $tmpdir = make_tempdir();

# Generate the sensor.conf file
my $sensor_conf = "$tmpdir/sensor-templ.conf";
make_packer_sensor_conf($sensor_conf, 'silk', 0, 'polldir', 'ipv6');

# the command that wraps rwflowpack
my $cmd = join " ", ("$SiLKTests::PYTHON $srcdir/tests/rwflowpack-daemon.py",
                     ($ENV{SK_TESTS_VERBOSE} ? "--verbose" : ()),
                     #"--log-level=debug",
                     "--sensor-conf=$sensor_conf",
                     "--copy $file{data}:incoming",
                     "--copy $file{v6data}:incoming",
                     "--limit=1003752",
                     "--basedir=$tmpdir",
                     "--",
                     "--polling-interval=5",
                     "--flat-archive",
                     "--file-cache-size=8",
    );

# run it and check the MD5 hash of its output
check_md5_output('9b8040c3af8489905de6018edd08cd1b', $cmd);

# the following directories should be empty
my @empty_dirs = qw(error incoming incremental sender);
for my $d (map {"$tmpdir/$_"} @empty_dirs) {
    next unless -d $d;
    unless (opendir D, $d) {
        warn "ERROR: Cannot open directory '$d': $!\n";
        next;
    }
    while (my $f = readdir(D)) {
        next if $f =~ /^\.\.?$/;
        die "ERROR: Found unexpected file '$d/$f'\n";
    }
    closedir D;
}

# input files should now be in the archive directory
my $archive_dir = "$tmpdir/archive/";
unless (opendir D, $archive_dir) {
    die "ERROR: Cannot open directory '$archive_dir': $!\n";
}
my %expected;
for my $f ($file{data}, $file{v6data}) {
    die unless $f =~ m,.*/(.+),;
    $expected{$1} = "$archive_dir/$1";
}
while (my $f = readdir(D)) {
    next if $f =~ /^\.\.?$/;
    if ($expected{$f}) {
        delete $expected{$f};
    }
    else {
        die "ERROR: Found unexpected file '$archive_dir/$f'\n";
    }
}
if (keys %expected) {
    die "ERROR: Missing file '", join("', '", values %expected), "'\n";
}
    closedir D;


# path to the data directory
my $data_dir = "$tmpdir/root";
die "ERROR: Missing data directory '$data_dir'\n"
    unless -d $data_dir;

# number of files to find in the data directory
my $expected_count = 0;
my $file_count = 0;

# read in the MD5s for every packed file we expect to find.  Read
# these from the bottom of this file.
my %md5_map;
while (my $lines = <DATA>) {
    my ($md5, $path) = split " ", $lines;
    $md5_map{$path} = $md5;
    ++$expected_count;
}
close DATA;

# find the files in the data directory and compare their MD5 hashes
File::Find::find({wanted => \&check_file, no_chdir => 1}, $data_dir);

# did we find all our files?
if ($file_count != $expected_count) {
    die "ERROR: Found $file_count files in root; expected $expected_count\n";
}

# successful!
exit 0;


# this is called by File::Find::find.  The full path to the file is in
# the $_ variable
sub check_file
{
    # skip anything that is not a file
    return unless -f $_;
    my $path = $_;
    # set $_ to just be the file basename
    s,^.*/,,;
    die "ERROR: Unexpected file $path\n"
        unless $md5_map{$_};
    ++$file_count;

    # do the MD5 sums match?  Use rwsort since input has two files,
    # and we don't know which was processed first.  IPv6 files do not
    # use the bytes/packet ratio, so no rounding issue to work-around
    check_md5_output($md5_map{$_},
                     ("$rwsort --fields=stime,sip $path "
                      ." | $rwcat --compression=none --byte-order=little"));
}


__DATA__
5c1202799db1611bc2208ba0fc08a3c2 in-S0_20090212.00
e49d9864f8a310b18c992e096e94c90f in-S0_20090212.01
72c0b5d8dbb3d3bfd944217cd49d8a65 in-S0_20090212.02
eb35e9be06a6112db7705ef6e43e01a2 in-S0_20090212.03
c19bff77a31283af3145b4baf51a3576 in-S0_20090212.04
2daebbfa0bc5677497898802edd221d3 in-S0_20090212.05
734a02a9529967f2e6ab30815a4323fa in-S0_20090212.06
e2ec0965c46690ffb1fd347aa9d0d04c in-S0_20090212.07
295a03629ccc512028b1b1175b237bbe in-S0_20090212.08
8278f3317b43d2691fbc215360d3a223 in-S0_20090212.09
2bcd088ec78c121600dadcd507ca91c1 in-S0_20090212.10
094cf8df69d990d8d1b2f09005952e33 in-S0_20090212.11
1d6a1a4ab9961cb3904ccc2fa007a60a in-S0_20090212.12
78b8f4f54cb49e9911b6c2c57fd37f88 in-S0_20090212.13
e86ccc07f2e9313a957e155620398c17 in-S0_20090212.14
bfdb8addaf026be01165796d36fd5fc0 in-S0_20090212.15
726088454164437f1e34ca362881e718 in-S0_20090212.16
d3aa3f68724a63645af25397e4070467 in-S0_20090212.17
8e48bc8ce02f7a63249c83c161b6aefd in-S0_20090212.18
8da19d4d1ef286d3693ee03894f85fd7 in-S0_20090212.19
78b327d3d03d8d0e6e185f4fe1c8edba in-S0_20090212.20
e28697ed0a2312ee1330e318f2ee8445 in-S0_20090212.21
9180df63d28faabebe88ec3a4b1308c6 in-S0_20090212.22
758dff197b5f6fcbbce5a063a4a02b62 in-S0_20090212.23
b7936efb7d3305ae7eb761ec01ecabff in-S0_20090213.00
d769b77a4be7713e29a04c3b977dc556 in-S0_20090213.01
1ec16a4594480d6c62dd27f9f8cd9a0a in-S0_20090213.02
93262ed17a33164b84e5f23460458659 in-S0_20090213.03
b2e0a1c2108acd8d502193370f92236d in-S0_20090213.04
3896d5b39a42cdf58f13ce3549951de0 in-S0_20090213.05
3833039b8b172dcb62d57f9976a72449 in-S0_20090213.06
36e7f21b9f612122012499c6fbf90b52 in-S0_20090213.07
400a5b81fb12d93d79fea4493bd26438 in-S0_20090213.08
dc2bef11e7c9552b80d74f4e77edcc8b in-S0_20090213.09
01eb357868b5e89861a06c340a922239 in-S0_20090213.10
e9daaf3f6c5819800c34777a04a92973 in-S0_20090213.11
2a46c1a9306702569201848ef8c42329 in-S0_20090213.12
5fb0c310a2e21eed5953f57587b8f176 in-S0_20090213.13
bd25c31b2f9d038a1e4b87a9b18ff4fb in-S0_20090213.14
5ebfdbc8959d30ae78663df72854c355 in-S0_20090213.15
81858554995fabb0007747c7a29c65e2 in-S0_20090213.16
1788d4e43cf6b70be1a57f8fdfece65b in-S0_20090213.17
e3489e8488eca5759e25d8fbd0e228f1 in-S0_20090213.18
72c21f34c4aa8f187bfb8173e19c6629 in-S0_20090213.19
18a12790a7b73edc12a0fc596695820f in-S0_20090213.20
b186a316a6fafa3d42dfc9b0191a340c in-S0_20090213.21
bcccb7bd83ebc72b6eb62592dfc34c5b in-S0_20090213.22
cb7b045dfa3c1e4046847978afa00859 in-S0_20090213.23
973015ba7d6cc5d6965db87a8eef0a0a in-S0_20090214.00
38a210b59856718723b9706872f088e7 in-S0_20090214.01
a05278f9e4cd40d5fc48a4baa52fc8f1 in-S0_20090214.02
29537ab708c05502ebfa7792dabd69e2 in-S0_20090214.03
01ad3854af7a5d71b208ea89f8c8a056 in-S0_20090214.04
9eb0a4f7b659f952037e8fff795e2e95 in-S0_20090214.05
f9d2b84106b6cb9660906dd6046e5841 in-S0_20090214.06
749412c7760852c2e8e208cfd917779f in-S0_20090214.07
0224f37e6b416cac5d567c6e9771655f in-S0_20090214.08
95e02bcdef0fca2837441c10c5b92235 in-S0_20090214.09
9bf2de351d40823b89acc9b689d1ecc2 in-S0_20090214.10
c8f48091e7668ff6f83c979dfdd063c9 in-S0_20090214.11
9413ff606305aa922f4bdf966598b1fa in-S0_20090214.12
c93bf08c03398aebadd904fed86413da in-S0_20090214.13
4818ce06ba752c3837a449b731aaa1b7 in-S0_20090214.14
8b79c4a15f14fb1734d59c9a6810abac in-S0_20090214.15
b4e96ea406e2d1355be3cedeecaa2d0d in-S0_20090214.16
1ca7d4e644e55609dc0b13e832b7d98a in-S0_20090214.17
764244a864dd008f999b7d0bb3f7f2c5 in-S0_20090214.18
2642c421166d58c177072a2ef1c27e2b in-S0_20090214.19
d7f95135f006725149494b13154a0290 in-S0_20090214.20
1239c1468163fe8753c218dca04364a1 in-S0_20090214.21
357cb41c32ae395e25d722dab67b4859 in-S0_20090214.22
7079fb8af89912e74f6f85d1350cfddf in-S0_20090214.23
1bb69f6b9eedc88f8bb70a7e79c713da iw-S0_20090212.00
58f84ee3545461519dc010a8d756033f iw-S0_20090212.01
81cb1bd50277331930e2492ceafd41b2 iw-S0_20090212.02
73ad0f3e86ffaa2fd2fea8e213512777 iw-S0_20090212.03
87d563a9dae0f3c1923b2b33daf7bd5c iw-S0_20090212.04
8b98a3e80cb53a1f3497ef332acdd6b1 iw-S0_20090212.05
3f55df39562fcfb37abf6e017a6724eb iw-S0_20090212.06
cc404842551db2295b1212fe014651c9 iw-S0_20090212.07
d44e26d95c7d0528b6ec0f4922d83882 iw-S0_20090212.08
2e8a9b7ec66bb539cb0f37bd7ae7c471 iw-S0_20090212.09
f0d05a7a997f8ea2983f4bf339f7b75c iw-S0_20090212.10
d6a2a46d53d14bc1f159bf43c78069b7 iw-S0_20090212.11
d3c44bfae726f09c919ae51fc1fbf79c iw-S0_20090212.12
4164de27ac2b62df9318b7efbe71f68e iw-S0_20090212.13
839a3281c6dc0a9b0bc83f3e7d916760 iw-S0_20090212.14
20b27e9f285574bff5cc590f60edda07 iw-S0_20090212.15
0e2fe90d905fac8f264f77636a5fb89e iw-S0_20090212.16
cc19586841365d7fe7f8630463f63e5b iw-S0_20090212.17
4eed64cecf1dce34c132855d507109c9 iw-S0_20090212.18
7d40fc41529da3da7551d0d85398e26e iw-S0_20090212.19
cd9eac258b0b56608ed4c5a440598c0e iw-S0_20090212.20
caf803409b982c3b356c0e7613b3ec0d iw-S0_20090212.21
355225c535d7d55d3d67b18c40023839 iw-S0_20090212.22
dabb88498abfeb538d2551312139fb7f iw-S0_20090212.23
8f412cbe1f3952e7df209c028834e4bc iw-S0_20090213.00
c98742c8bb641a3a54abaa7df541c51d iw-S0_20090213.01
5016dac31dbb9fe98159e2e31eb3d57a iw-S0_20090213.02
9a26a9dc7c99d7b26c20c3158bdd5e13 iw-S0_20090213.03
4df30d18b7ec254682c39d07b874f999 iw-S0_20090213.04
c2e145a63e4706866cd959dcf953362a iw-S0_20090213.05
0d873bd800ea7be2ac63c01a0706a917 iw-S0_20090213.06
06f22118768b7ec53c7a9dad51b7a68c iw-S0_20090213.07
15aa878b61df648b12fccb43c3e61bdc iw-S0_20090213.08
8a8110a050555b6625d833f64aee6257 iw-S0_20090213.09
0f15bfc18e16176b407e63a32dbb71d9 iw-S0_20090213.10
8f087951e474dc2af116c7f5febf0d23 iw-S0_20090213.11
e3bb8bac0a671b22930525c3bde176e0 iw-S0_20090213.12
05f5d9aeb47348629fdee0f814f90f69 iw-S0_20090213.13
69e5a57defe28ee957041c077a7e15be iw-S0_20090213.14
4bf8ed575a440c22beb97cdf0553a7f6 iw-S0_20090213.15
b8cb55933a712e2fd3af7c5df7263c6d iw-S0_20090213.16
9401a15cedd78fac15a25de8ead6a12e iw-S0_20090213.17
0227f81ce595d34dd578940f839c85c8 iw-S0_20090213.18
fa63bd1548ae7d5997393db64b63d2ed iw-S0_20090213.19
c4e06248454124e77db33adfc37ce9de iw-S0_20090213.20
cbfa51d8f20e8da4fb9ffa3af7874c28 iw-S0_20090213.21
b7da1fd9b22382856b2dff5502d282cf iw-S0_20090213.22
17125c721d80f4c7ab7d1500c3eda2e5 iw-S0_20090213.23
977daceaefa1056d66048fdcbfb5a9b1 iw-S0_20090214.00
5c710668dad923f4cede80540ddb1734 iw-S0_20090214.01
b50ec55b0358c4b3ba3075a24a7fb5b8 iw-S0_20090214.02
12a80b0cc881e6528b4db23155f267f1 iw-S0_20090214.03
49f42b9e57dd14b859164fe308105319 iw-S0_20090214.04
13d003fabf485bca827ec151a3243c67 iw-S0_20090214.05
3ca9848a51df351eb71b1ecff735ea43 iw-S0_20090214.06
3c76e37e571ac3b534492a70a0542a27 iw-S0_20090214.07
76602d2ab7c7cc754f13a97f8ee6ac27 iw-S0_20090214.08
edecb3665d0d716d6b20cdaabdff446b iw-S0_20090214.09
3f24e1d03e3d2152edf8f313a56831de iw-S0_20090214.10
48ad0d33d8199a14bca9876a9228349d iw-S0_20090214.11
96abcfe83430ba8960a6c9deabffb133 iw-S0_20090214.12
9c98849d728a491e228ec7b051065266 iw-S0_20090214.13
8636587e1211c772a048c7535d4d0fbf iw-S0_20090214.14
3bd08dfe8bae1764192f6bfcd6cb7ae4 iw-S0_20090214.15
d5239b9e9e340b741086a5bafc96595d iw-S0_20090214.16
01b221fe6684f02bc5d20da2ec970484 iw-S0_20090214.17
6707e1fbd0e13f4a4e758ffb2a444b99 iw-S0_20090214.18
b0fe3b773f8a1f809fd4b935b5089e6f iw-S0_20090214.19
df74fdc099d6e33698ceb4356865bda7 iw-S0_20090214.20
7c2a9a40074753dbb53ea579e8986c07 iw-S0_20090214.21
0ff872ea7deff6e21fb0419831050ba8 iw-S0_20090214.22
7b06da464ccebda33ddaf216e8c5d67d iw-S0_20090214.23
ed628ebe77d438f561a67d0b6c125156 other-S0_20090212.00
3bdd8b72fd0e76b8bc6865b40ed6536f other-S0_20090212.01
b6f14c679dac8f46d0c100eb83d69179 other-S0_20090212.02
a2026aa978fb59779d836719cb128146 other-S0_20090212.03
97edd593f69bcd96ea837dbc09bde6f2 other-S0_20090212.04
201a3d7efd2746528319381da248d2a1 other-S0_20090212.05
b7418b2e320c2cee0a430c3df3289625 other-S0_20090212.06
edd0f738cc0e69846a4cc98d58a1723f other-S0_20090212.07
8a8db5e5ac370a39221966e2b3cf6c79 other-S0_20090212.08
1f858d734c26c98b55b719c4a14328da other-S0_20090212.09
5579d052f65d04a9ce9b18a37ba4dff9 other-S0_20090212.10
cce65c625382b3b8a42cd5102c9fed98 other-S0_20090212.11
45ce2b8418ba27413a76b979959aa2fd other-S0_20090212.12
d6279ebda832b430bd5f608f2e34c262 other-S0_20090212.13
203df7e9194d13c3da30ec03e2373ffb other-S0_20090212.14
4d1a771b1b23d7df425177972b0224f4 other-S0_20090212.15
93dfd59c2f4416728ce3c3b2f7a2723e other-S0_20090212.16
4526d4709e13c5411f1b251e73e31f93 other-S0_20090212.17
7826dce0c12849f7abf8566767dbd6dd other-S0_20090212.18
c3682ab7a8b54e7cf8ea5d72a0c5eb29 other-S0_20090212.19
4cfd93b902003b9a898e8f2b3c17a7f7 other-S0_20090212.20
86a3e03d3b377ef390aaeb11494081f0 other-S0_20090212.21
d53d01e4e0f7a485d2e5c9607f5f4fa8 other-S0_20090212.22
3e5005c25dab102838789fd085807dd5 other-S0_20090212.23
7e14fe05a884a76bb0f1e9298b8dd803 other-S0_20090213.00
afceef40dadda2eb0e59116d0263b19a other-S0_20090213.01
b0b19eda7d33bdc7917e385c6e583269 other-S0_20090213.02
c8a77c59fb27007ea42e94cf4920c269 other-S0_20090213.03
7fad3a829772d4f2389a7269d1f9e5a0 other-S0_20090213.04
084400cf826b4ebb96bd3a1641c3f45b other-S0_20090213.05
ba36c5d79b2c120b99f34f621e2e12ed other-S0_20090213.06
4d429049c5669b15be1570e6e150dacc other-S0_20090213.07
72788ccf4cef4dbcc343d180288c7dd2 other-S0_20090213.08
943b1874159399d900559a0ecf32060a other-S0_20090213.09
c2a9d1bb8ddc98ae37814e8d9bcec931 other-S0_20090213.10
68df4c06f6f594ea83d958c0caeef2c9 other-S0_20090213.11
018b3758e1e809daa0758119852dd7fd other-S0_20090213.12
7c5b0e68197e87b6c73cf56844654a39 other-S0_20090213.13
1fec33e6c922df3bf6f12bc9f7233c84 other-S0_20090213.14
2ba3dc63ae1b127c22424cf6ce1fe462 other-S0_20090213.15
5016f6dbddc69e2ff24441ce127d3608 other-S0_20090213.16
cb245c6b2b4df6fd33e8aa98ffeaf540 other-S0_20090213.17
0933c22b555ed69b5f44bad52f6e5100 other-S0_20090213.18
0b10fa0b4728b7a8ccfcf75bcc3b3261 other-S0_20090213.19
e91d955ecae846dae5bf44d8762f0e91 other-S0_20090213.20
022d62115dc8ea0720c8e543fe029824 other-S0_20090213.21
35137290d3712625644b62746dbdc2fd other-S0_20090213.22
e3253dc9e9ef1ead6f23b6005e1d702e other-S0_20090213.23
8029de6e89e47fed1ae4c336ed4441ce other-S0_20090214.00
73260fa6942942e82eadc947c4533ba0 other-S0_20090214.01
314990883eb7982687f092ce41bea8a0 other-S0_20090214.02
c6540cf6af932e53bc6329201d6a6ab9 other-S0_20090214.03
ecfc940a13dbaa1d8224ee4359e9f0cb other-S0_20090214.04
c05f38dcafaf2632cc62745bc0b42c61 other-S0_20090214.05
3813d9f611913e54ebb78096219d577c other-S0_20090214.06
77ee0e9bbbad2828034c49f8cd0992a4 other-S0_20090214.07
1ff5f17bd1cb6ea5b7725dae15dd6f8a other-S0_20090214.08
6c02432ea32910afbd9cf48dd293f0e0 other-S0_20090214.09
2b23cde3be232d81c43b1afa425ccab6 other-S0_20090214.10
822caf299cdfc1e42e25724711b7d12f other-S0_20090214.11
f3f93e037f00673cba0eb25837085685 other-S0_20090214.12
3f74c47289dd58da1347475fcf47afb5 other-S0_20090214.13
89ba9a3af59b2015f5d657cb34db29da other-S0_20090214.14
f0680a2f735a1b7ca4c854a4beddd24a other-S0_20090214.15
da234b8d6b1231bc42d5188cd945f83b other-S0_20090214.16
959d25a20317f2899b530dd34ffb4472 other-S0_20090214.17
d69d758ab3f7528bf69fdee8ee8c1966 other-S0_20090214.18
45a43a3a6889a9cab7d108b0fc217616 other-S0_20090214.19
7f8d63860bfff56b2d411125b3d6bf3a other-S0_20090214.20
90e8453a32f8b459b628345c8fa75740 other-S0_20090214.21
e451d39d70bc5fa3cfa3e79aec5c89e8 other-S0_20090214.22
f145fc928ffac920fd11c3a139406efe other-S0_20090214.23
38b4b361a41b36c099942c2890aece30 out-S0_20090212.00
5531264e6cd7080c2fa931ad14e1a18c out-S0_20090212.01
53a21a3c3cf86d051b647b478bc8a292 out-S0_20090212.02
b4d85d2a7a778d20be863f02b5f53f23 out-S0_20090212.03
13e67a308b520b2a91f54743caa0ff4f out-S0_20090212.04
551dfa47a45413a37c71d7b14f20ace1 out-S0_20090212.05
7d133f7fc9a9ef9992cf4d72c1dc524e out-S0_20090212.06
f76ce676a952de16aae31c0dbddfe581 out-S0_20090212.07
ebbf71f3745e08a75648c65e28d715ab out-S0_20090212.08
cf8768c9c2ea9a2bbf74bdec781d8e2f out-S0_20090212.09
283d1c90e40de6d9fa03096c976ba877 out-S0_20090212.10
a346caa1feb7e482d59f915af5c2eed6 out-S0_20090212.11
720f7e482448cb1e80a4f24e184215df out-S0_20090212.12
0786f5fe84b87eb3fc21e41aa303b33d out-S0_20090212.13
22a814de4263a7afbbde3cb215a0e73b out-S0_20090212.14
472f99ba3bda878aff31bfd79cbe38e1 out-S0_20090212.15
aa75eb5877efddeffd41b6fd1ebe6abd out-S0_20090212.16
1782fa8ac547c16313aa2a7eec9d6fc9 out-S0_20090212.17
6ecb29fde2a636b5c7f9980bfea26055 out-S0_20090212.18
94897e233d82f6d1e568d78353bff810 out-S0_20090212.19
8952ef5d9bdb00eba53986f6f833a6e6 out-S0_20090212.20
e1a6633c3289c41c7f4037e339fb934e out-S0_20090212.21
1d5610eb4a04678fc6bd349c01d2b83c out-S0_20090212.22
7893a7231dcb606de4fceedda2a787f8 out-S0_20090212.23
126bccc96550ec87b103e3ded8af1541 out-S0_20090213.00
33f2d92e13bcb388d91a9a983ddee1c7 out-S0_20090213.01
e7aa165d6c9f1b644fe4063217609d1a out-S0_20090213.02
958d6cf8ddef48f3bb3b7cd3eae4b133 out-S0_20090213.03
abcaa997f77db5a81320d9ebd0bd855b out-S0_20090213.04
3d5a9a3e9f32f14e0961497bc4e3f713 out-S0_20090213.05
b25ff66e1238c25c352d3ce87e4aea98 out-S0_20090213.06
95491042bd13654f06e956475a0c0f47 out-S0_20090213.07
1e5dc52e5a48b4e81973d9164a61d663 out-S0_20090213.08
bfad275cb3ae485a0125f745f45adf0c out-S0_20090213.09
eac13563aaaf0d4eb133cd019f498805 out-S0_20090213.10
2bc39b05e8ea89950dcc808b498657fa out-S0_20090213.11
1f692166a4f569c15628102fe348879a out-S0_20090213.12
a4fb1ef604a0014f004f934f0f1d0ec7 out-S0_20090213.13
5e33b3ce73af126ef09ac974a31e8631 out-S0_20090213.14
49effeb60ba1389c4e26bddd772e9b91 out-S0_20090213.15
ce571cdc971351ba68feae1aa2d214d4 out-S0_20090213.16
82add68ae7f95596bfc2df6a3d63e9c7 out-S0_20090213.17
d587367d38cee70fe639ab5ee3592642 out-S0_20090213.18
e9403ce7a6f8deca1439ff11309280a9 out-S0_20090213.19
e6a0f9fab68785ba34e4cbf7db490732 out-S0_20090213.20
22cd35aa337901813f3593f0e6980cc0 out-S0_20090213.21
a74fe1d6a606b18115ffd5ad89e4e337 out-S0_20090213.22
02d18caf7b4eb7533d83239c5386c763 out-S0_20090213.23
84efc560e368391b41b9e346d2ebb236 out-S0_20090214.00
fc664beb2da2657d2b646b8f0a048871 out-S0_20090214.01
6ba57d57970fd4dc4cff5d8569319307 out-S0_20090214.02
8fe3edce8ca285445cdbff1ead1ec286 out-S0_20090214.03
4f6db403de16cc1927c871d1557f43b4 out-S0_20090214.04
f264d0e9d008207f7d84f43c669c9679 out-S0_20090214.05
bb0abdf6c4a7fa0a333879cfeba70e40 out-S0_20090214.06
18a924410cd8c7e1a3e5f68af0cd517e out-S0_20090214.07
01fbc9c0d4649d78056389d4ff461744 out-S0_20090214.08
4c1511286097c0afbe6ca33360dcd3c4 out-S0_20090214.09
f29f75ff1c271d969bc17e6e332a615c out-S0_20090214.10
580e4d0a34586779a8250f8b7bf09f38 out-S0_20090214.11
8206e0b50e23f46f7052676d60a523cc out-S0_20090214.12
3ac08955af12311b54c831b54e2e41fa out-S0_20090214.13
390873aaa3dc44fe3d29c1164ee51e78 out-S0_20090214.14
e8a42ae77a9bdacfabdfd793a25098cf out-S0_20090214.15
2e553dd3d7701e7bd879052c338ebe55 out-S0_20090214.16
d372eb046b5ccf9c4da2d690dde026c8 out-S0_20090214.17
5af69df9297fb4e97725150c5517f890 out-S0_20090214.18
85445cda501281ba42b461f59b60bfba out-S0_20090214.19
ab83a3468bc4bbf3c7a3ad71ecafafb5 out-S0_20090214.20
69286b615c3cacff375a2f72a1240367 out-S0_20090214.21
cec5a4372aae2da78c9135d548506819 out-S0_20090214.22
91dca756aa95cd43c913d898bf0a394e out-S0_20090214.23
1366aebedadd1015d465ebdfe38e4aa0 outnull-S0_20090212.00
16a6fd26a5feb90334e4804811779bb0 outnull-S0_20090212.01
7bb019bfe97954b43ed7262f25e39b27 outnull-S0_20090212.02
57acd3afcafdc67905dff56cffd7284a outnull-S0_20090212.03
7f815c9149c599e1615b53538567742a outnull-S0_20090212.04
cd8ce95289eca04434c8ed016f8f5772 outnull-S0_20090212.05
2ea8d3ccd37a4ef5d25826b0c2a0fe69 outnull-S0_20090212.06
eceb575bdd32720c28382b75ae6d4106 outnull-S0_20090212.07
fbb6e733df292173580648161615981a outnull-S0_20090212.08
be69355f0c3f7157526270dad816d757 outnull-S0_20090212.09
ddae1b78c149098033f1282b869bace5 outnull-S0_20090212.10
3291ddf0c7f891ba10bfdbe6ea41f0eb outnull-S0_20090212.11
a65b03404825dee8a06aa8c8666f9d8b outnull-S0_20090212.12
dc84ea165bc1dce9ca86e202b3b1d931 outnull-S0_20090212.13
2410ab7296058967c6304ef598183b73 outnull-S0_20090212.14
24af9b083eb845f36ded395dbf98cab3 outnull-S0_20090212.15
2fd14741033c4baf513b1a573f60ec7d outnull-S0_20090212.16
fbae3cc3955e564f6e85c2cfc1026f12 outnull-S0_20090212.17
d72e58d04f9de5310fc78f62f43ba787 outnull-S0_20090212.18
5887010c70ab6db4b20c40fdb70632b0 outnull-S0_20090212.19
5144b67dcb2011d443e192dc4a7d01c4 outnull-S0_20090212.20
40b440ca69337e6504b6106b81ecce20 outnull-S0_20090212.21
9ae5fd3a623c631a09f5c62551661761 outnull-S0_20090212.22
3c00ab70ad65b51e9e924bc81c3c94f5 outnull-S0_20090212.23
fd97c38a17cc1954fbc4b80083ffbd34 outnull-S0_20090213.00
ad45ca9a9651a2eff2d94dc0ee8ce7a5 outnull-S0_20090213.01
3738ab94dc9186998c8f3fa019c7a500 outnull-S0_20090213.02
cf10bb4a48a01b329c67bda918507d06 outnull-S0_20090213.03
43fd42023adfe2e6cf28394b86d2a3a2 outnull-S0_20090213.04
a27776f93a42cfdb28560666de5bc7a8 outnull-S0_20090213.05
a10f85194bd1064896dd734d58d73586 outnull-S0_20090213.06
ad3abb9c2604413b81492497e130b6a9 outnull-S0_20090213.07
86c9e795ed2aabe3fd223e3016f8ba33 outnull-S0_20090213.08
7c7a5bccbf61afc2fb1a95968a457feb outnull-S0_20090213.09
c9f11f80f5b8559f991a64921e916d7d outnull-S0_20090213.10
e52812bc4b8a20cb965e22dba1a7ce77 outnull-S0_20090213.11
632e302ffd52199622d013fb39629ab4 outnull-S0_20090213.12
da3e0906cf2de25d2a164c0e94884440 outnull-S0_20090213.13
92de8a693044ed92f2f2c0e8e7f344c3 outnull-S0_20090213.14
8676d9f3c8078d8b652a6355d3e02951 outnull-S0_20090213.15
1021c8cf5373b8ae065850b76322c1e6 outnull-S0_20090213.16
373d757cdef77d02e2b8895134f6267c outnull-S0_20090213.17
8ebb0e9411a7e1ee8155d43c6c829f27 outnull-S0_20090213.18
9fd1177bbbf2c360a095efb053e3553b outnull-S0_20090213.19
529b509d82d8c9c5998f8722f45b8c76 outnull-S0_20090213.20
118dfdb32ed1e5ec81beaf08608e6408 outnull-S0_20090213.21
cc7a10cc37c3410a87635159108e3362 outnull-S0_20090213.22
5beb1ea50323d7c51c8953f2bf60f0a2 outnull-S0_20090213.23
f58b65067409acfd617eeb15f619a480 outnull-S0_20090214.00
32ade0a7a29e24163a688d085434c121 outnull-S0_20090214.01
4f6de7001f584c5930de25f8084baadf outnull-S0_20090214.02
c6ba6d404736944db5084577d04d83a8 outnull-S0_20090214.03
fc8e2542952454703f20ce987eab82b3 outnull-S0_20090214.04
ee7130db207ae923563716ff07bab631 outnull-S0_20090214.05
63f705b7656b90bfac9997e08e8fa66c outnull-S0_20090214.06
92c96225b5f89e1928a348a2d4a25240 outnull-S0_20090214.07
04cfa784c5c9ea4d619264573774314b outnull-S0_20090214.08
4ea189a24692f6746cdeea05c3676029 outnull-S0_20090214.09
570580e47708892b01a760043e4c4a39 outnull-S0_20090214.10
60fe554dd6f699135d106dcbe6ea24c7 outnull-S0_20090214.11
ed666ed3ba2a1c212b86f51648c4408e outnull-S0_20090214.12
d910a62244a174de481641cb23845200 outnull-S0_20090214.13
8be62d7daf4c204e7adab70975ef39c0 outnull-S0_20090214.14
720e859c2b05705ec6d4d7c4474a7094 outnull-S0_20090214.15
e0d2133bbc95e8f0b2d4e2fdb5ae0585 outnull-S0_20090214.16
3d1dc9afc7901675ad15d9f4d19cbd9a outnull-S0_20090214.17
5663095028060c294270fd463a76baba outnull-S0_20090214.18
3d79a18619ba1e7c8729dc549f5a20a5 outnull-S0_20090214.19
9488c2206fbdf43d6e550b522916cc18 outnull-S0_20090214.20
f1ff04bdab14c66ad261be4e738ec6ab outnull-S0_20090214.21
133e4896c3e148b893d5146af4d38fef outnull-S0_20090214.22
3f7281ffbc1fbba8dbe16d1f868f5c12 outnull-S0_20090214.23
9b3e883113ed1a2ea5ab5273bff42430 ow-S0_20090212.00
6ce42eefe725e0af1b73e86fe9578882 ow-S0_20090212.01
ab29836b33f9e9b1f865f72ce4d35de8 ow-S0_20090212.02
daee83e9ef13578f620000e3049f1d7b ow-S0_20090212.03
a9467801402c0002719c9635a6627a07 ow-S0_20090212.04
c7686ca5aa846073a1bf410176e38158 ow-S0_20090212.05
015c71459a07e9d2bf6494d0ebd71ec0 ow-S0_20090212.06
1c94fedf52fe4e31ccbce3e9645cf309 ow-S0_20090212.07
19029a7194ecad3abe8dd430ce90d92e ow-S0_20090212.08
3546c35a096fdd73e9c25e7d79299cfc ow-S0_20090212.09
f03c2b7a183a1ee78666741d5164337c ow-S0_20090212.10
10e512a3c9b8acc48fa6a3617fd34612 ow-S0_20090212.11
313eb47ba620c174ea8ac640495514a5 ow-S0_20090212.12
ae9a37a35d934d6e76fceb42d09f7f7c ow-S0_20090212.13
32c8d949316d99d557ea37dc2d1ab23b ow-S0_20090212.14
51b038f9c82a05f82084d8c27502ad5b ow-S0_20090212.15
3ada1e67c9a88a30ac22219f798ab48b ow-S0_20090212.16
e273efca152e0b19b37bde55cb096c67 ow-S0_20090212.17
07b261bb9bf31524f0db6e2cfddead9c ow-S0_20090212.18
29f12e162f3cab1507513f1c9c730c55 ow-S0_20090212.19
11c88b25703d68e5673aa37a07a3088e ow-S0_20090212.20
abbacfd05310a1c04a98ded064d09e04 ow-S0_20090212.21
36d2c2d327f8ae6098b3fd96130250ef ow-S0_20090212.22
407e34f251b073336876163029577b62 ow-S0_20090212.23
7e1218a91a4b5f20a4fc5bbe5dd80709 ow-S0_20090213.00
0afed1e0eb42e5fc58141c92aa54b1be ow-S0_20090213.01
8ba82162eb6319e42c628f2f8d958ca3 ow-S0_20090213.02
fc9492172703707ff8b5ce35144d478d ow-S0_20090213.03
ea767d26877143303d81f1a6021d131b ow-S0_20090213.04
c34e3d6a8ced1a1ce5ad17bdd314da4c ow-S0_20090213.05
924de890c216739865099c7e00f5c3a5 ow-S0_20090213.06
e20cfbcf0fe46cc464ada47250d3f29f ow-S0_20090213.07
0b975dab106a370e78307f8a52806e9e ow-S0_20090213.08
4584634c04584ddd9ec422355dfb994f ow-S0_20090213.09
3bf45871730614f0cdf2903ef2d929a7 ow-S0_20090213.10
2fd151450a8a0fadd9bde935f4b487d5 ow-S0_20090213.11
44d857209e3b4d024703b8884f419e37 ow-S0_20090213.12
00acd9aa9f8a9defc158b8be3f4808e0 ow-S0_20090213.13
b91e39944738a1b03732c0daedb527fd ow-S0_20090213.14
83707c2ce99098335a7beb6d73b6a5e8 ow-S0_20090213.15
4184ec4239ea6dc79cd670c5cd300601 ow-S0_20090213.16
5647033b8efd520056652f1bc0169d63 ow-S0_20090213.17
f96b2e2e8495d0765ade1b577ebe260b ow-S0_20090213.18
e5541799034c3b978d33eca8ccb3db93 ow-S0_20090213.19
4cad2f85da583ba8ed82e0f8695721a5 ow-S0_20090213.20
f1d963adef76564ff3e081a9a26eb765 ow-S0_20090213.21
596c5df7b66f883d7ea0767eb82c5398 ow-S0_20090213.22
456ca5f836b9947db64ae53f2ebbf64a ow-S0_20090213.23
63ad3957f8ce86a2124eea9653b345b1 ow-S0_20090214.00
4a9f5ac39314b8036067f49817b31a14 ow-S0_20090214.01
fba2ee06614f81e193520cf6b3e40a8c ow-S0_20090214.02
db5f780f30d8d2fc9634975da2579815 ow-S0_20090214.03
4d141e3e829f3ed06374b7e957f6c351 ow-S0_20090214.04
07df95645e5ebd598fb3b7c98b97b2a9 ow-S0_20090214.05
db3ce312e0e323dbca6df16bdcb548f9 ow-S0_20090214.06
513c62f7cab183920c0422c57a3e8a33 ow-S0_20090214.07
8cd43629e03d62fd8d022aa5b75d0a01 ow-S0_20090214.08
ccca01d212bfb0d0801ff75762cbf306 ow-S0_20090214.09
7b45f3b6ac9f5d4ad1deb5a1a0128683 ow-S0_20090214.10
2488ab5e4ccd16d6871e2181ba11e8e6 ow-S0_20090214.11
5b2fa827382a6079d8a251015c604a23 ow-S0_20090214.12
9914aadf9ab6b8c3e7ead160f7b414b7 ow-S0_20090214.13
54d3a65adaac6329588480b12853b3b6 ow-S0_20090214.14
20122a4077256fecbafdc79e28c31cbe ow-S0_20090214.15
313ab1d90af606db6dcdf6d750116baf ow-S0_20090214.16
6dff3b4cd3968d021b0c261be4cc1ac4 ow-S0_20090214.17
04e18084e7511fcab5d2a3eadcd71770 ow-S0_20090214.18
220627a806de1e4a693d44bc2a89b939 ow-S0_20090214.19
f54e228708f9941b9b386936e1388471 ow-S0_20090214.20
12b869c761d508fb011d13fe1141d186 ow-S0_20090214.21
d6bbd779810ec6132987c7ade68027bc ow-S0_20090214.22
21ce14fa98960505779e1c3d76184f69 ow-S0_20090214.23
