%{
/*
 *  Copyright (C) 2012-2024 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_config_parse.y
 *
 *  YACC source file for super_mediator.conf parser.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  Super Mediator 1.10.0
 *
 *  Copyright 2023 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  GOVERNMENT PURPOSE RIGHTS - Software and Software Documentation
 *  Contract No.: FA8702-15-D-0002
 *  Contractor Name: Carnegie Mellon University
 *  Contractor Address: 4500 Fifth Avenue, Pittsburgh, PA 15213
 *  The Government's rights to use, modify, reproduce, release, perform,
 *  display, or disclose this software are restricted by paragraph (b)(2) of
 *  the Rights in Noncommercial Computer Software and Noncommercial Computer
 *  Software Documentation clause contained in the above identified
 *  contract. No restrictions apply after the expiration date shown
 *  above. Any reproduction of the software or portions thereof marked with
 *  this legend must also reproduce the markings.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM23-2316
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_ctx.h"
#include "mediator_config.h"
#include "mediator_core.h"
#include "mediator_inf.h"
#include "mediator.h"
#include "mediator_filter.h"
#include "mediator_dns.h"
#include "mediator_dedup.h"
#include "mediator_ssl.h"
#if ENABLE_SKTYPESENSOR
#include "probeconf.h"
#endif


#define REQUIRE_NOTNULL(var)                                     \
    if (NULL != (var)) { /* okay */ } else {                     \
        g_error("Programmer error: %s is NULL in %s(), line %d", \
                #var, __func__, __LINE__);                       \
    }

/*Exporter stuff */
/* first in list */
static md_export_node_t  *ebeg = NULL;
/* used for processing various config blocks */
static md_export_node_t  *etemp = NULL;
static mdFlowExporter_t  *exp_temp = NULL;
static mdTransportType_t  md_ipfix_outtransport = NONE;
static gboolean           spread_exporter = FALSE;

/*Collector Stuff */
static md_collect_node_t *ctemp = NULL;
static mdFlowCollector_t *coll_temp = NULL;
static mdTransportType_t  md_ip_intransport = NONE;
#if HAVE_SPREAD
static int                num_in_groups;
#endif
/* Shared */
static md_spread_filter_t *sftemp = NULL;
static md_filter_t        *ftemp = NULL;
static gboolean            and_filter = FALSE;
static gboolean            md5_filter = FALSE;
static gboolean            sha1_filter = FALSE;


static gboolean       default_tables = FALSE;
static gboolean       custom_tables = FALSE;
static gboolean       ssl_dedup_only = FALSE;

static smFieldMap_t  *maptemp = NULL;
static smFieldMap_t  *mapitem = NULL;
static mdFieldList_t *temp_list = NULL;

static int            valueListTemp[MAX_VALUE_LIST];
static int            numValueList = 0;
static int            numCustomList = 0;
static int            numUserElements = 0;
static int            valueListWild = 0;

static struct cfg_dns_dedup_st {
    char          *temp_name;
    int           *type_list;
    smFieldMap_t  *map;
    int            max_hit;
    int            flush_timeout;
    gboolean       lastseen;
    gboolean       exportname;
} cfg_dns_dedup = {NULL, NULL, NULL, 0, 0, FALSE, FALSE};


/* parsing function declarations */
static void
validateConfFile(
    void);
static void
parseCollectorBegin(
    mdTransportType_t   mode,
    char               *name);
static void
parseCollectorEnd(
    void);
static void
parseCollectorPort(
    int   port);
static void
parseCollectorHost(
    char  *host);
static void
parseCollectorPath(
    char  *file);
static void
parseCollectorPollingInterval(
    int   poll_time);
static void
parseCollectorSpreadDaemon(
    char  *daemon_name);
static void
parseCollectorSpreadGroup(
    char  *group);
static void
parseCollectorLock(
    void);
static void
parseCollectorMovePath(
    char  *dir);
static void
parseCollectorDecompressDirectory(
    char  *path);
static void
parseCollectorDelete(
    void);
static void
parseCollectorSilkConfigProbe(
    char                   *probe_name,
    mdAcceptFilterField_t   field);
static void
parseFilterBegin(
    void);
static void
parseFilterEnd(
    void);
static void
parseComparison(
    mdAcceptFilterField_t   field,
    fieldOperator           oper,
    char                   *val,
    int                     val_type);
static void
parseExporterBegin(
    mdTransportType_t   mode,
    char               *name);
static void
parseExporterEnd(
    void);
static void
parseExporterPort(
    int   port);
static void
parseExporterHost(
    char  *host);
static void
parseExporterPath(
    char  *file);
static void
parseExporterTextDelimiter(
    char  *delim);
static void
parseExporterDPIDelimiter(
    char  *delim);
static void
parseExporterSpreadDaemon(
    char  *daemon_name);
static void
parseExporterSpreadGroup(
    char  *group);
static void
parseExporterLock(
    void);
static void
parsePidFile(
    char  *pid_file);
static void
parseExporterRotateSeconds(
    int   secs);
static void
parseExporterUDPTimeout(
    int   mins);
static void
parseExporterFlowOnly(
    void);
static void
parseExporterDPIOnly(
    void);
static void
parseExporterMatchesIPSet(
    char  *file,
    char  *name);
static void
parseExporterPrefixMap(
    char  *file,
    char  *map_name);
static void
parseSpreadGroup(
    char  *name);
static void
parseSpreadGroupEnd(
    void);
static void
parseStatisticsConfig(
    void);
static void
parsePreserveObDomainConfig(
    void);
static void
parseRewriteSslCertsConfig(
    void);
static void
parseGenTombstoneConfig(
    void);
static void
parseTombstoneIdConfig(
    int   configured_id);
static void
parseDNSDeDupConfig(
    void);
static void
parseExporterMovePath(
    char  *dir);
static void
parseDNSDeDupOnly(
    void);
static void
parseSSLDeDupOnly(
    int   mode);
static void
parseExporterPrintHeader(
    void);
static void
parseExporterEscapeChars(
    void);
static void
parseLogConfig(
    char  *log_file);
static void
parseLogDir(
    char  *log_dir);
static void
parseStatsTimeout(
    int   timeout);
static void
parseExporterNoStats(
    void);
static void
parseExporterRemoveEmpty(
    void);
static void
parseExporterAddStats(
    void);
static void
parseValueListItems(
    char  *val);
static void
parseFieldListItems(
    char                   *fint,
    mdAcceptFilterField_t   field);
static void
parseTableList(
    char  *table);
static void
parseTableListBegin(
    char  *index_label);
static void
parseExporterMultiFiles(
    void);
static void
parseExporterNoIndex(
    void);
static void
parseExporterTimestamp(
    void);
static void
parseExporterNoFlowStats(
    void);
static void
parseMySQLParams(
    char  *user,
    char  *pw,
    char  *db,
    char  *host,
    char  *table);
static void
parseExporterRemoveUploaded(
    void);
static void
parseUserInfoElement(
    int    ie_num,
    char  *name,
    int    app_num);
static void
parseExporterJson(
    void);
static void
parseExporterDNSRROnly(
    int   mode);
static void
parseExporterDNSRespOnly(
    void);
static void
parseDnsDedupRecordTypeList(
    void);
static void
parseDnsDedupConfigEnd(
    void);
static smFieldMap_t *
parseMapStmt(
    char  *mapname);
static void
parseSSLConfigBegin(
    char  *name);
static void
parseSSLIssuerTypeList(
    void);
static void
parseSSLSubjectTypeList(
    void);
static void
parseSSLOtherTypeList(
    void);
static void
parseSSLExtensionsTypeList(
    void);
static void
parseExporterDedupPerFlow(
    void);
static void
parseDedupConfig(
    char  *exp_name);
static void
parseDedupFileList(
    char                   *file,
    mdAcceptFilterField_t   field,
    char                   *mapname);
static void
parseSSLCertDedup(
    void);
static void
parseSSLCertFile(
    char  *filename);
static void
parseExporterSSLMD5Hash(
    void);
static void
parseExporterSSLSHA1Hash(
    void);
static void
parseExporterGzipFiles(
    void);
static void
parseExporterDedupOnly(
    void);
static void
parseExporterNoFlow(
    void);
static void
parseMapBegin(
    mdAcceptFilterField_t   map_type,
    char                   *name);
static void
parseMapLine(
    char  *label);
static void
parseMapOther(
    char  *name);
static void
parseMapDiscard(
    void);
static void
parseMapEnd(
    mdAcceptFilterField_t   map_type);
static void
parseExporterMetadataExport(
    void);
static void
parseSilkConfigBeginEnd(
    int   begin_end);
static void
parseSilkConfigStmt(
    char  *sensor_conf,
    char  *silk_conf);
static int
parseNumericValue(
    char  *number,
    int    base);

/*  Tell uncrustify to ignore the next part of the file */
/*  *INDENT-OFF* */
%}

%union {
    char                   *str;
    int                     integer;
    smFieldMap_t           *fieldMap;
    mdParserNumber_t        *number;
    mdTransportType_t       transport;
    mdAcceptFilterField_t   field;
    fieldOperator           oper;
    mdLogLevel_t            log_level;
}

%token EOS

%token COMMA
%token LEFT_SQ_BRACKET
%token RIGHT_SQ_BRACKET
%token LEFT_PAREN
%token RIGHT_PAREN
%token WILD

%token TOK_COLLECTOR TOK_EXPORTER TOK_DNS_DEDUP TOK_DNSDEDUP_ONLY TOK_NO_STATS
%token TOK_PORT TOK_HOST TOK_PATH TOK_DAEMON TOK_DELIM TOK_PRINT_HDR
%token TOK_GROUP TOK_MOVE TOK_DELETE TOK_LOCK TOK_UDP_TIMEOUT
%token TOK_ROTATE TOK_END TOK_FILTER TOK_LOG_FILE
%token TOK_FLOW_ONLY TOK_DPI_ONLY TOK_POLL TOK_MAX_HIT TOK_FLUSH_TIMEOUT
%token TOK_LOG_LEVEL TOK_BASE_64 TOK_LAST_SEEN TOK_RM_EMPTY TOK_STATS_ONLY
%token TOK_TABLE TOK_DPI_CONFIG TOK_MULTI_FILES TOK_NO_INDEX
%token TOK_TIMESTAMP TOK_NO_FLOW_STATS TOK_PID_FILE TOK_MY_REMOVE
%token TOK_MY_USER TOK_MY_PW TOK_MY_DB TOK_MY_HOST TOK_MY_TABLE
%token TOK_FIELDS TOK_DPI_FIELD_LIST TOK_DPI_DELIMITER TOK_STATS_TIMEOUT
%token TOK_USERIE TOK_AND_FILTER TOK_ESCAPE TOK_DNSRR_ONLY TOK_FULL
%token TOK_LOG_DIR TOK_JSON TOK_RECORDS TOK_RESP_ONLY TOK_SSL_CONFIG
%token TOK_ISSUER TOK_SUBJECT TOK_OTHER TOK_EXTENSIONS TOK_DEDUP_PER_FLOW
%token TOK_DEDUP_CONFIG TOK_FILE TOK_MERGE TOK_SSL_DEDUP TOK_CERT_FILE
%token TOK_SSL_DEDUP_ONLY TOK_MD5 TOK_SHA1 TOK_GZIP TOK_DNSRR
%token TOK_DEDUP_ONLY TOK_NO_FLOW TOK_OBID_MAP TOK_VLAN_MAP TOK_MAP
%token TOK_DISCARD TOK_ADD_EXPORTER_NAME TOK_DECOMPRESS TOK_METADATA_EXPORT
%token TOK_GEN_TOMBSTONE TOK_TOMBSTONE_CONFIGURED_ID TOK_TOMBSTONE_CONFIG
%token TOK_PRESERVE_OBDOMAIN TOK_REWRITE_SSL_CERTS TOK_PREFIX_MAP
%token TOK_MATCHES_IPSET TOK_SILKCONFIG_CONFIG TOK_SILKCONFIG_PROBE
%token TOK_SILKCONFIG_SENSOR TOK_SILKCONFIG_SILK

 /* values returned from lex */

%token <str>        VAL_ATOM
%token <str>        VAL_DATETIME
%token <str>        VAL_DOUBLE
%token <str>        VAL_INTEGER
%token <str>        VAL_IP
%token <str>        VAL_QSTRING
%token <transport>  VAL_TRANSPORT
%token <oper>       VAL_OPER
%token <field>      VAL_FIELD
%token <log_level>  VAL_LOGLEVEL

 /* result of parsing statements */

%type <str>             optionalName
%type <str>             atomOrQstring
%type <fieldMap>        mapStmt
%type <integer>         numericValue
%type <integer>         maxHitCount
%type <integer>         flushSeconds

%%
mediatorConfFile:       mediatorConf
{
    validateConfFile();
};

mediatorConf:           stmtList
;

stmtList:               /* empty */
                        | stmtList stmt
;

stmt:                   EOS
                        | collectorBlock
                        | filterBlock
                        | exporterBlock
                        | spreadGroup
                        | statsConfig
                        | preserveObDomainConfig
                        | rewriteSslCertsConfig
                        | tombstoneConfig
                        | logConfig
                        | logLevelConfig
                        | logDirConfig
                        | pidConfig
                        | dnsDedupBlock
                        | dpiConfigBlock
                        | sslConfigBlock
                        | dedupConfigBlock
                        | statsTimeout
                        | userIE
                        | vlanMapBlock
                        | obidMapBlock
                        | silkConfigBlock
;

collectorBlock:         collectorBegin collectorStmtList collectorEnd
;

collectorBegin:         TOK_COLLECTOR VAL_TRANSPORT optionalName EOS
{
    parseCollectorBegin($2, $3);
};

collectorEnd:           TOK_COLLECTOR TOK_END EOS
{
    parseCollectorEnd();
};

collectorStmtList:      /* empty */
                        | collectorStmtList collectorStmt
;

collectorStmt:          EOS
                        | col_port
                        | col_host
                        | col_path
                        | col_polling_interval
                        | col_spread_daemon
                        | col_spread_group
                        | col_lock
                        | col_move_path
                        | col_delete
                        | col_decompress
                        | filter_comparison
                        | filter_and_filter
                        | col_silkconfig_probe
;

col_port:               TOK_PORT numericValue EOS
{
    parseCollectorPort($2);
};

col_host:               TOK_HOST atomOrQstring EOS
{
    parseCollectorHost($2);
};

col_path:               TOK_PATH atomOrQstring EOS
{
    parseCollectorPath($2);
};

col_polling_interval:   TOK_POLL numericValue EOS
{
    parseCollectorPollingInterval($2);
};

col_spread_daemon:      TOK_DAEMON atomOrQstring EOS
{
    parseCollectorSpreadDaemon($2);
};

col_decompress:         TOK_DECOMPRESS atomOrQstring EOS
{
    parseCollectorDecompressDirectory($2);
};

col_spread_group:       TOK_GROUP atomOrQstring EOS
{
    parseCollectorSpreadGroup($2);
};

col_lock:               TOK_LOCK EOS
{
    parseCollectorLock();
};

col_move_path:          TOK_MOVE atomOrQstring EOS
{
    parseCollectorMovePath($2);
};

col_delete:             TOK_DELETE EOS
{
    parseCollectorDelete();
};

col_silkconfig_probe:   TOK_SILKCONFIG_PROBE atomOrQstring VAL_FIELD
{
    parseCollectorSilkConfigProbe($2, $3);
};

filterBlock:            filterBegin filterStmtList filterEnd
;

filterStmtList:         /* empty */
                        | filterStmtList filterStmt
;

filterStmt:             EOS
                        | filter_comparison
                        | filter_and_filter
;

filterBegin:            TOK_FILTER EOS
{
    parseFilterBegin();
};

filterEnd:              TOK_FILTER TOK_END EOS
{
    parseFilterEnd();
};

filter_and_filter:      TOK_AND_FILTER EOS
{
    and_filter = TRUE;
};

exp_dpi_field_list:     TOK_DPI_FIELD_LIST valueList EOS
;

/* '[' <item>, <item>, ..., <item> ']' */
valueList:              valueListStart valueListItems valueListEnd
;

valueListStart:         LEFT_SQ_BRACKET
{
    numValueList = 0;
    valueListWild = 0;
};

valueListEnd:           RIGHT_SQ_BRACKET
;

valueListItems:         VAL_INTEGER
{
    parseValueListItems($1);
}
                        | valueListItems COMMA VAL_INTEGER
{
    parseValueListItems($3);
}
                        | WILD
{
    valueListWild = 1;
};

fieldList:              fieldListItem
                        | fieldList fieldListItem
                        | fieldList COMMA fieldListItem
;

fieldListItem:          VAL_FIELD
{
    parseFieldListItems(NULL, $1);
}
                        | VAL_INTEGER
{
    parseFieldListItems($1, 0);
}
                        | TOK_COLLECTOR
{
    parseFieldListItems(NULL, COLLECTOR);
};

    /* same as filterStmtList except no support for AND_FILTER */
spreadComparisonList:   /* empty */
                        | spreadComparisonList filter_comparison
;

filter_comparison:      VAL_FIELD VAL_OPER VAL_INTEGER EOS
{
    parseComparison($1, $2, $3, VAL_INTEGER);
}
                        | VAL_FIELD VAL_OPER VAL_IP EOS
{
    parseComparison($1, $2, $3, VAL_IP);
}
                        | VAL_FIELD VAL_OPER VAL_QSTRING EOS
{
    /* ANY_IP IN_LIST "my_set.set" */
    parseComparison($1, $2, $3, VAL_QSTRING);
}
                        | TOK_COLLECTOR VAL_OPER VAL_QSTRING EOS
{
    parseComparison(80, $2, $3, VAL_QSTRING);
}
                        | TOK_COLLECTOR VAL_OPER VAL_ATOM EOS
{
    parseComparison(80, $2, $3, VAL_QSTRING);
};


/*                       | VAL_FIELD VAL_OPER valueList EOS
{
    parseComparison($1, $2, NULL, VAL_ATOM, NUM_FIELDS);
};*/

exporterBlock:          exporterBegin exporterStmtList exporterEnd
;

exporterBegin:          TOK_EXPORTER VAL_TRANSPORT optionalName EOS
{
    parseExporterBegin($2, $3);
};

exporterEnd:            TOK_EXPORTER TOK_END EOS
{
    parseExporterEnd();
};

exporterStmtList:       /* empty */
                        | exporterStmtList exporterStmt
;

exporterStmt:           EOS
                        | exp_port
                        | exp_host
                        | exp_path
                        | exp_spread_daemon
                        | exp_spread_group
                        | exp_lock
                        | exp_delim
                        | exp_dpi_delim
                        | exp_rotate
                        | exp_udp_timeout
                        | exp_flow_only
                        | exp_dpi_only
                        | exp_no_stats
                        | exp_stats_only
                        | exp_dns_dedup
                        | exp_dedup_flow
                        | exp_dns_dedup_only
                        | exp_remove_empty
                        | exp_print_headers
                        | exp_multi_files
                        | exp_no_index
                        | exp_timestamp
                        | exp_no_flow_stats
                        | exp_json
                        | filter_comparison
                        | expCustomList
                        | mysqlConfig
                        | filter_and_filter
                        | exp_dpi_field_list
                        | exp_remove_uploaded
                        | exp_escape
                        | exp_dns_rr_only
                        | exp_dns_rr
                        | exp_dns_resp_only
                        | exp_ssl_dedup_only
                        | exp_md5_hash
                        | exp_sha1_hash
                        | exp_gzip_files
                        | exp_move_path
                        | exp_no_flow
                        | exp_dedup_only
                        | exp_metadata_export
                        | exp_matches_ipset
                        | exp_prefix_map
;

exp_md5_hash:           TOK_MD5 EOS
{
    parseExporterSSLMD5Hash();
};

exp_sha1_hash:          TOK_SHA1 EOS
{
    parseExporterSSLSHA1Hash();
};

exp_move_path:          TOK_MOVE atomOrQstring EOS
{
    parseExporterMovePath($2);
};

exp_port:               TOK_PORT numericValue EOS
{
    parseExporterPort($2);
};

exp_host:               TOK_HOST atomOrQstring EOS
{
    parseExporterHost($2);
};

exp_path:               TOK_PATH atomOrQstring EOS
{
    parseExporterPath($2);
};

exp_spread_daemon:      TOK_DAEMON atomOrQstring EOS
{
    parseExporterSpreadDaemon($2);
};

exp_spread_group:       TOK_GROUP atomOrQstring EOS
{
    parseExporterSpreadGroup($2);
};

exp_delim:              TOK_DELIM  atomOrQstring EOS
{
    parseExporterTextDelimiter($2);
};

exp_dpi_delim:          TOK_DPI_DELIMITER atomOrQstring EOS
{
    parseExporterDPIDelimiter($2);
};

exp_lock:               TOK_LOCK EOS
{
    parseExporterLock();
};

exp_rotate:             TOK_ROTATE numericValue EOS
{
    parseExporterRotateSeconds($2);
};

exp_udp_timeout:        TOK_UDP_TIMEOUT numericValue EOS
{
    parseExporterUDPTimeout($2);
};

exp_flow_only:          TOK_FLOW_ONLY EOS
{
    parseExporterFlowOnly();
};

exp_dpi_only:           TOK_DPI_ONLY EOS
{
    parseExporterDPIOnly();
};

exp_no_stats:           TOK_NO_STATS EOS
{
    parseExporterNoStats();
};

exp_stats_only:         TOK_STATS_ONLY EOS
{
    parseExporterAddStats();
};

exp_remove_empty:       TOK_RM_EMPTY EOS
{
    parseExporterRemoveEmpty();
};

exp_multi_files:        TOK_MULTI_FILES EOS
{
    parseExporterMultiFiles();
};

exp_no_flow_stats:      TOK_NO_FLOW_STATS EOS
{
    parseExporterNoFlowStats();
};

exp_json:               TOK_JSON EOS
{
    parseExporterJson();
};

exp_matches_ipset:      TOK_MATCHES_IPSET atomOrQstring EOS
{
    parseExporterMatchesIPSet($2, NULL);
}
                        | TOK_MATCHES_IPSET atomOrQstring atomOrQstring EOS
{
    /* name appears first, but function takes opposite order */
    parseExporterMatchesIPSet($3, $2);
};

exp_prefix_map:         TOK_PREFIX_MAP atomOrQstring EOS
{
    parseExporterPrefixMap($2, NULL);
}
                        | TOK_PREFIX_MAP atomOrQstring atomOrQstring EOS
{
    /* name appears first, but function takes opposite order */
    parseExporterPrefixMap($3, $2);
};

spreadGroup:            spreadBegin spreadComparisonList spreadEnd
;

spreadBegin:            TOK_GROUP atomOrQstring EOS
{
    parseSpreadGroup($2);
};

spreadEnd:              TOK_GROUP TOK_END
{
    parseSpreadGroupEnd();
};

/* turns stats forwarding off*/
statsConfig:            TOK_NO_STATS EOS
{
    parseStatisticsConfig();
};

preserveObDomainConfig: TOK_PRESERVE_OBDOMAIN EOS
{
    parsePreserveObDomainConfig();
};

rewriteSslCertsConfig:  TOK_REWRITE_SSL_CERTS EOS
{
    parseRewriteSslCertsConfig();
};

/* have super mediator create its own tombstone records*/
tombstoneConfig:        tombstone_begin tombstoneStmtList tombstone_end
;

tombstone_begin:        TOK_TOMBSTONE_CONFIG EOS
;

tombstone_end:          TOK_TOMBSTONE_CONFIG TOK_END EOS
;

tombstoneStmtList:      /* empty */
                        | tombstoneStmtList tombstoneStmt
;

tombstoneStmt:          EOS
                        | genTombstoneConfig
                        | tombstoneIdConfig
;

genTombstoneConfig:     TOK_GEN_TOMBSTONE EOS
{
    parseGenTombstoneConfig();
};

tombstoneIdConfig:      TOK_TOMBSTONE_CONFIGURED_ID numericValue EOS
{
    parseTombstoneIdConfig($2);
};


statsTimeout:           TOK_STATS_TIMEOUT numericValue EOS
{
    parseStatsTimeout($2);
};

exp_dns_dedup:          TOK_DNS_DEDUP EOS
{
    parseDNSDeDupConfig();
};

exp_dns_dedup_only:     TOK_DNSDEDUP_ONLY EOS
{
    parseDNSDeDupOnly();
};

exp_ssl_dedup_only:     TOK_SSL_DEDUP_ONLY EOS
{
    parseSSLDeDupOnly(1);
}
                        | TOK_SSL_DEDUP EOS
{
    parseSSLDeDupOnly(0);
};

exp_no_flow:            TOK_NO_FLOW EOS
{
    parseExporterNoFlow();
};

exp_dedup_only:         TOK_DEDUP_ONLY EOS
{
    parseExporterDedupOnly();
};

exp_print_headers:      TOK_PRINT_HDR EOS
{
    parseExporterPrintHeader();
};

exp_no_index:           TOK_NO_INDEX EOS
{
    parseExporterNoIndex();
};

exp_escape:             TOK_ESCAPE EOS
{
    parseExporterEscapeChars();
};

exp_dedup_flow:         TOK_DEDUP_PER_FLOW EOS
{
    parseExporterDedupPerFlow();
};

exp_timestamp:          TOK_TIMESTAMP EOS
{
    parseExporterTimestamp();
};

exp_dns_rr_only:        TOK_DNSRR_ONLY EOS
{
    parseExporterDNSRROnly(1);
}
                        | TOK_DNSRR_ONLY TOK_FULL EOS
{
    parseExporterDNSRROnly(2);
};

exp_dns_rr:             TOK_DNSRR EOS
{
    parseExporterDNSRROnly(3);
}
                        | TOK_DNSRR TOK_FULL EOS
{
    parseExporterDNSRROnly(4);
};

exp_dns_resp_only:      TOK_RESP_ONLY EOS
{
    parseExporterDNSRespOnly();
};

exp_gzip_files:         TOK_GZIP EOS
{
    parseExporterGzipFiles();
};

exp_metadata_export:    TOK_METADATA_EXPORT EOS
{
    parseExporterMetadataExport();
};

expCustomList:          TOK_FIELDS fieldList EOS
{
    numCustomList = 0;
};

/* logging */
logConfig:              TOK_LOG_FILE atomOrQstring EOS
{
    parseLogConfig($2);
};

logDirConfig:           TOK_LOG_DIR atomOrQstring EOS
{
    parseLogDir($2);
};

logLevelConfig:         TOK_LOG_LEVEL VAL_LOGLEVEL EOS
{
    md_log_level = $2;
};

pidConfig:              TOK_PID_FILE atomOrQstring EOS
{
    parsePidFile($2);
};

dnsDedupRecordList:     TOK_RECORDS valueList EOS
{
    parseDnsDedupRecordTypeList();
};

dnsDedupBlock:          dns_dedup_begin dnsDedupStmtList dns_dedup_end
;

dns_dedup_begin:        TOK_DNS_DEDUP optionalName EOS
{
    free(cfg_dns_dedup.temp_name);
    cfg_dns_dedup.temp_name = $2;
};

dns_dedup_end:          TOK_DNS_DEDUP TOK_END
{
    parseDnsDedupConfigEnd();
};

dnsDedupStmtList:       /* empty */
                        | dnsDedupStmtList dnsDedupStmt
;

dnsDedupStmt:           EOS
                        | dnsDedupHitConfig
                        | dnsDedupFlushConfig
                        | dnsDedupBase64Config
                        | dnsDedupLastSeenConfig
                        | dnsDedupRecordList
                        | dnsDedupMapStmt
                        | dnsDedupAddExporterName
;

dedupConfigBlock:       dedup_config_begin dedupStmtList dedup_config_end
;

dedup_config_begin:     TOK_DEDUP_CONFIG optionalName EOS
{
    parseDedupConfig($2);
};

dedup_config_end:       TOK_DEDUP_CONFIG TOK_END
{
    etemp = NULL;
};

dedupStmtList:          /* empty */
                        | dedupStmtList dedupStmt
;

dedupStmt:              EOS
                        | dedupHitConfig
                        | dedupFlushConfig
                        | dedupFileStmt
                        | dedupMergeTruncated
                        | dedupAddExporterName
;

dedupHitConfig:         maxHitCount
{
    /* TOK_MAX_HIT */
    REQUIRE_NOTNULL(etemp);
    md_dedup_configure_state(etemp->dedup, $1, 0, FALSE, FALSE);
};

dedupFlushConfig:       flushSeconds
{
    /* TOK_FLUSH_TIMEOUT */
    REQUIRE_NOTNULL(etemp);
    md_dedup_configure_state(etemp->dedup, 0, $1, FALSE, FALSE);
};

dedupAddExporterName:   TOK_ADD_EXPORTER_NAME EOS
{
    REQUIRE_NOTNULL(etemp);
    md_dedup_configure_state(etemp->dedup, 0, 0, FALSE, TRUE);
};

dedupMergeTruncated:    TOK_MERGE EOS
{
    REQUIRE_NOTNULL(etemp);
    md_dedup_configure_state(etemp->dedup, 0, 0, TRUE, FALSE);
};

dedupFileStmt:          TOK_FILE atomOrQstring VAL_FIELD valueList EOS
{
    /* PREFIX "name" {SIP|DIP|FLOWKEYHASH} [ IE, IE, ... ] */
    parseDedupFileList($2, $3, NULL);
}
                        | TOK_FILE atomOrQstring valueList EOS
{
    /* PREFIX "name" [ IE, IE, ... ] */
    /* uses SIP by default */
    parseDedupFileList($2, SIP_V4, NULL);
}
                        | TOK_FILE atomOrQstring VAL_FIELD TOK_MAP LEFT_PAREN atomOrQstring RIGHT_PAREN valueList EOS
{
    /* PREFIX "name" {SIP|DIP|FLOWKEYHASH} MAP ( "name" ) [ IE, IE, ... ] */
    parseDedupFileList($2, $3, $6);
}
                        | TOK_FILE atomOrQstring TOK_MAP LEFT_PAREN atomOrQstring RIGHT_PAREN valueList EOS
{
    /* PREFIX "name" MAP ( "name" ) [ IE, IE, ... ] */
    parseDedupFileList($2, SIP_V4, $5);
};

mapStmt:                TOK_MAP LEFT_PAREN atomOrQstring RIGHT_PAREN EOS
{
    $$ = parseMapStmt($3);
};

dnsDedupAddExporterName: TOK_ADD_EXPORTER_NAME EOS
{
    cfg_dns_dedup.exportname = TRUE;
};

dnsDedupMapStmt:        mapStmt
{
    /* MAP("name") */
    if (cfg_dns_dedup.map) {
        mediator_config_error(
            "MAP already defined for this DNS_DEDUP config block.");
    }
    cfg_dns_dedup.map = $1;
};

dnsDedupHitConfig:      maxHitCount
{
    if (($1) > UINT16_MAX) {
        mediator_config_error("MAX_HIT_COUNT has max of 65535");
    }
    cfg_dns_dedup.max_hit = $1;
};

dnsDedupFlushConfig:    flushSeconds
{
    if (($1) > UINT16_MAX) {
        mediator_config_error("FLUSH_TIMEOUT has max of 65535");
    }
    cfg_dns_dedup.flush_timeout = $1;
};

dnsDedupBase64Config:   TOK_BASE_64 EOS
{
    md_config.dns_base64_encode = TRUE;
};

dnsDedupLastSeenConfig: TOK_LAST_SEEN EOS
{
    cfg_dns_dedup.lastseen = TRUE;
};

dpiConfigBlock:         dpi_config_begin dpiConfigStmtList dpi_config_end
;

dpi_config_begin:       TOK_DPI_CONFIG optionalName EOS
{
    parseTableListBegin($2);
};

dpi_config_end:         TOK_DPI_CONFIG TOK_END EOS
{
    numValueList = 0;
};

dpiConfigStmtList:      /* empty */
                        | dpiConfigStmtList dpiConfigStmt
;

dpiConfigStmt:          EOS
                        | tableStmt
;

tableStmt:              TOK_TABLE atomOrQstring valueList EOS
{
    parseTableList($2);
};

mysqlConfig:            TOK_MY_USER atomOrQstring EOS
{
    parseMySQLParams($2, NULL, NULL, NULL, NULL);
}
                        | TOK_MY_PW atomOrQstring EOS
{
    parseMySQLParams(NULL, $2, NULL, NULL, NULL);
}
                        | TOK_MY_DB atomOrQstring EOS
{
    parseMySQLParams(NULL, NULL, $2, NULL, NULL);
}
                        | TOK_MY_HOST atomOrQstring EOS
{
    parseMySQLParams(NULL, NULL, NULL, $2, NULL);
}
                        | TOK_MY_TABLE atomOrQstring EOS
{
    parseMySQLParams(NULL, NULL, NULL, NULL, $2);
};

userIE:                 TOK_USERIE numericValue atomOrQstring EOS
{
    parseUserInfoElement($2, $3, 0);
}
                        | TOK_USERIE numericValue atomOrQstring numericValue EOS
{
    parseUserInfoElement($2, $3, $4);
};

exp_remove_uploaded:    TOK_MY_REMOVE EOS
{
    parseExporterRemoveUploaded();
};

sslConfigBlock:         ssl_config_begin sslConfigStmtList ssl_config_end
;

ssl_config_begin:       TOK_SSL_CONFIG atomOrQstring EOS
{
    parseSSLConfigBegin($2);
};

ssl_config_end:         TOK_SSL_CONFIG TOK_END
{
    numValueList = 0;
};

sslConfigStmtList:      /* empty */
                        | sslConfigStmtList sslConfigStmt
;

sslConfigStmt:          EOS
                        | sslIssuerList
                        | sslSubjectList
                        | sslOtherList
                        | sslExtensionList
                        | sslCertDedup
                        | sslDedupHitConfig
                        | sslDedupFlushConfig
                        | sslCertFile
                        | ssldedupAddExporterName
                        | sslMapStmt
;

ssldedupAddExporterName: TOK_ADD_EXPORTER_NAME EOS
{
    REQUIRE_NOTNULL(etemp);
    md_ssl_dedup_configure_state(etemp->ssl_dedup, 0, 0, NULL, NULL, TRUE);
};

sslMapStmt:             mapStmt
{
    /* MAP("name") */
    REQUIRE_NOTNULL(etemp);
    md_ssl_dedup_configure_state(etemp->ssl_dedup, 0, 0, NULL, $1, FALSE);
};

sslIssuerList:          TOK_ISSUER valueList EOS
{
    parseSSLIssuerTypeList();
};

sslSubjectList:         TOK_SUBJECT valueList EOS
{
    parseSSLSubjectTypeList();
};

sslOtherList:           TOK_OTHER valueList EOS
{
    parseSSLOtherTypeList();
};

sslExtensionList:       TOK_EXTENSIONS valueList EOS
{
    parseSSLExtensionsTypeList();
};

sslCertDedup:           TOK_SSL_DEDUP EOS
{
    parseSSLCertDedup();
};

sslDedupHitConfig:      maxHitCount
{
    /* TOK_MAX_HIT_COUNT */
    REQUIRE_NOTNULL(etemp);
    md_ssl_dedup_configure_state(etemp->ssl_dedup, $1, 0, NULL, NULL, FALSE);
};

sslDedupFlushConfig:    flushSeconds
{
    /* TOK_MAX_HIT_COUNT */
    REQUIRE_NOTNULL(etemp);
    md_ssl_dedup_configure_state(etemp->ssl_dedup, 0, $1, NULL, NULL, FALSE);
};

sslCertFile:            TOK_CERT_FILE atomOrQstring EOS
{
    parseSSLCertFile($2);
};

    /* VLAN_MAP and OBID_MAP use the same statement parsing production */
vlanMapBlock:           vlanMapBegin voMapStmtList vlanMapEnd
;

vlanMapBegin:           TOK_VLAN_MAP atomOrQstring EOS
{
    parseMapBegin(VLAN, $2);
};

vlanMapEnd:             TOK_VLAN_MAP TOK_END
{
    parseMapEnd(VLAN);
};

voMapStmtList:          /* empty */
                        | voMapStmtList voMapStmt
;

voMapStmt:              EOS
                        | voMapListItem
                        | voMapListOther
                        | voMapListDiscard
;

voMapListItem:          atomOrQstring valueList EOS
{
    parseMapLine($1);
};

voMapListOther:         atomOrQstring TOK_OTHER EOS
{
    parseMapOther($1);
};

voMapListDiscard:       TOK_DISCARD EOS
{
    parseMapDiscard();
};

    /* VLAN_MAP and OBID_MAP use the same statement parsing production */
obidMapBlock:           obidMapBegin voMapStmtList obidMapEnd
;

obidMapBegin:           TOK_OBID_MAP atomOrQstring EOS
{
    parseMapBegin(OBDOMAIN, $2);
};

obidMapEnd:             TOK_OBID_MAP TOK_END
{
    parseMapEnd(OBDOMAIN);
};

silkConfigBlock:        silkConfigBegin silkConfigStmtList silkConfigEnd
;

silkConfigBegin:        TOK_SILKCONFIG_CONFIG EOS
{
    parseSilkConfigBeginEnd(0);
};

silkConfigEnd:          TOK_SILKCONFIG_CONFIG TOK_END EOS
{
    parseSilkConfigBeginEnd(1);
};

silkConfigStmtList:     /* empty */
                        | silkConfigStmtList silkConfigStmt
;

silkConfigStmt:         TOK_SILKCONFIG_SENSOR atomOrQstring EOS
{
    parseSilkConfigStmt($2, NULL);
}
                        | TOK_SILKCONFIG_SILK atomOrQstring EOS
{
    parseSilkConfigStmt(NULL, $2);
};

maxHitCount:            TOK_MAX_HIT numericValue EOS
{
    if (($2) < 1) {
        mediator_config_error("MAX_HIT_COUNT must be a positive integer");
    }
    $$ = $2;
};

flushSeconds:           TOK_FLUSH_TIMEOUT numericValue EOS
{
    if (($2) < 1) {
        mediator_config_error("FLUSH_TIMEOUT must be a positive integer");
    }
    $$ = $2;
};

optionalName:           /*empty*/
{
    $$ = NULL;
}
                        | atomOrQstring
;

atomOrQstring:          VAL_ATOM | VAL_QSTRING
;

numericValue:           VAL_INTEGER
{
    /* parse into a signed 32 bit integer */
    $$ = parseNumericValue($1, 10);
};

%%
/*  Reenable uncrustify */
/*  *INDENT-ON* */


/*
 *  Finds the exporter whose name is `name` when `name` is not NULL, or
 *  reports a fatal error if no such exporter is found.  If `name` is NULL and
 *  only one exporter exists, returns it; otherwise, reports a fatal error.
 *  Parameter `block_type` is the current block and is used in the error.
 */
static md_export_node_t *
findExporter(
    const char  *exp_name,
    const char  *block_type)
{
    (void)block_type;

    if (exp_name) {
        md_export_node_t *exp;

        for (exp = ebeg; exp; exp = exp->next) {
            if (mdExporterCompareNames(exp->exp, exp_name)) {
                return exp;
            }
        }
    } else if (NULL != ebeg && NULL == ebeg->next) {
        return ebeg;
    }

    /* ERROR */
    if (NULL == ebeg) {
        mediator_config_error("Cannot find an exporter. "
                              "No exporters have been defined");
    }
    if (exp_name) {
        mediator_config_error("Cannot find an exporter with the given name");
    }
    mediator_config_error("Cannot find an exporter. Must specify"
                          " exporter name when multiple exporters exist");

    abort();                    /* UNREACHABLE */
}

/*
 *  Finds the map whose name is `mapname`, which must not be NULL.  Reports a
 *  fatal error if no such map is found unless 'no_error' is TRUE.
 */
static smFieldMap_t *
findFieldMap(
    const char  *mapname,
    gboolean     no_error)
{
    smFieldMap_t *map;

    REQUIRE_NOTNULL(mapname);

    for (map = maptemp; map; map = map->next) {
        if (strcmp(map->name, mapname) == 0) {
            return map;
        }
    }
    if (no_error) {
        return NULL;
    }

    /* ERROR */
    if (NULL == maptemp) {
        mediator_config_error("Cannot find the MAP. No MAPS previously"
                              " defined in configuration file");
    }
    mediator_config_error("Cannot find the named MAP");

    abort();                    /* UNREACHABLE */
}

static void
validateConfFile(
    void)
{
    if (ebeg == NULL) {
        mediator_config_error("No Exporter Information Given. "
                              " Need an Exporter or DEDUP File.");
    }

    md_config.flowexit = ebeg;
    md_config.flowsrc = ctemp;
    md_config.maps = maptemp;
}

static void
parseCollectorBegin(
    mdTransportType_t   mode,
    char               *name)
{
    switch (mode) {
      case UDP:
        md_ip_intransport = UDP;
        break;
      case TCP:
        md_ip_intransport = TCP;
        break;
      case SPREAD:
        md_ip_intransport = SPREAD;
        break;
      case FILEHANDLER:
        md_ip_intransport = FILEHANDLER;
        break;
      case DIRECTORY:
        md_ip_intransport = DIRECTORY;
        break;
      default:
        mediator_config_error("Unacceptable transport mode for exporter");
    }

    coll_temp = mdNewFlowCollector(mode, name);
    free(name);
}

static void
parseCollectorPort(
    int   port)
{
    char  str[32];

    if (!((md_ip_intransport == TCP) || (md_ip_intransport == UDP))) {
        mediator_config_error("PORT only valid for TCP or UDP Collectors");
    }
    if (port < 1024) {
        mediator_config_error("Invalid Port.  Port must be above 1024.");
    }

    snprintf(str, sizeof(str), "%d", port);
    mdCollectorSetPort(coll_temp, str);
}

static void
parseCollectorHost(
    char  *host)
{
    if (!((md_ip_intransport == TCP) || (md_ip_intransport == UDP) ||
          (md_ip_intransport == SPREAD)))
    {
        mediator_config_error(
            "HOSTNAME only valid for TCP or UDP Collectors");
    }

    mdCollectorSetInSpec(coll_temp, host);
    free(host);
}


static void
parseCollectorPath(
    char  *file)
{
    if ((md_ip_intransport == TCP) || (md_ip_intransport == UDP) ||
        (md_ip_intransport == SPREAD))
    {
        mediator_config_error(
            "PATH and FILE only valid for FILEHANDLER Collectors");
    }

    if (md_ip_intransport == DIRECTORY) {
        mdCollectorSetPollTime(coll_temp, "30");
        mdCollectorSetInSpec(coll_temp, file);
    } else if (g_file_test(file, G_FILE_TEST_IS_DIR)) {
        mdCollectorSetPollTime(coll_temp, "30");
        mdCollectorSetInSpec(coll_temp, file);
    } else {
        mdCollectorSetInSpec(coll_temp, file);
    }

    free(file);
}

static void
parseCollectorPollingInterval(
    int   poll_time)
{
    char  str[32];

    if ((md_ip_intransport == TCP || md_ip_intransport == UDP) ||
        (md_ip_intransport == SPREAD))
    {
        mediator_config_error("Invalid Keyword: POLL only valid for "
                              "FILEHANDLER or DIR  Collectors");
    }
    if (poll_time > 65535) {
        mediator_config_error("POLL has max of 65535");
    }

    snprintf(str, sizeof(str), "%d", poll_time);
    mdCollectorSetPollTime(coll_temp, str);
}

static void
parseCollectorSpreadDaemon(
    char  *daemon_name)
{
    if ((md_ip_intransport != SPREAD)) {
        mediator_config_error(
            "Invalid Keyword: Collector NOT configured for SPREAD.");
    }
#if HAVE_SPREAD
    mdCollectorSetInSpec(coll_temp, daemon_name);
#else
    mediator_config_error("Mediator not configured with Spread Support. "
                          "Confirm Spread is installed.");
#endif
    free(daemon_name);
}

static void
parseCollectorSpreadGroup(
    char  *group)
{
    if (md_ip_intransport != SPREAD) {
        mediator_config_error(
            "Invalid keyword: Collector NOT configured for SPREAD.");
        return;
    }
#if HAVE_SPREAD
    mdCollectorAddSpreadGroup(coll_temp, group, num_in_groups);
    num_in_groups++;
#else
    mediator_config_error("Mediator not configured with Spread Support. "
                          "Confirm Spread is installed.");
#endif /* if HAVE_SPREAD */
    free(group);
}

static void
parseCollectorLock(
    void)
{
    if (md_ip_intransport != DIRECTORY) {
        mediator_config_error(
            "Invalid Keyword: LOCK must be used with DIR Collector");
    } else {
        mdCollectorSetLockMode(coll_temp, TRUE);
    }
}

static void
parseCollectorDecompressDirectory(
    char  *path)
{
    if ((md_ip_intransport != FILEHANDLER) &&
        (md_ip_intransport != DIRECTORY))
    {
        mediator_config_error("Invalid Keyword: DECOMPRESS must be used"
                              " with a FILEHANDLER or DIR Collector");
    }

    mdCollectorSetDecompressDir(coll_temp, path);
    free(path);
}

static void
parseCollectorMovePath(
    char  *path)
{
    if ((md_ip_intransport != FILEHANDLER) &&
        (md_ip_intransport != DIRECTORY))
    {
        mediator_config_error("Invalid Keyword: MOVE must be used"
                              " with a FILEHANDLER or DIR Collector");
    }

    if (!g_file_test(path, G_FILE_TEST_IS_DIR)) {
        mediator_config_error("MOVE expects a valid file directory");
    }

    mdCollectorSetMoveDir(coll_temp, path);
    free(path);
}

static void
parseCollectorDelete(
    void)
{
    if ((md_ip_intransport != FILEHANDLER) &&
        (md_ip_intransport != DIRECTORY))
    {
        mediator_config_error("Invalid Keyword: DELETE must be used "
                              "with FILEHANDLER or DIR Collector");
    }

    mdCollectorSetDeleteFiles(coll_temp, TRUE);
}

static void
parseCollectorEnd(
    void)
{
    md_collect_node_t *new_node;

    REQUIRE_NOTNULL(coll_temp);

    if (!mdCollectorVerifySetup(coll_temp, NULL)) {
        exit(-1);
    }

#if HAVE_SPREAD
    if (md_ip_intransport == SPREAD) {
        if (num_in_groups == 0) {
            mediator_config_error(
                "SPREAD Collector Requires AT LEAST ONE group.");
        }
    }
#endif /* if HAVE_SPREAD */

    new_node = md_new_collect_node();
    new_node->coll = coll_temp;
    new_node->filter = ftemp;
    if (new_node->filter) {
        md_filter_t *nnode = NULL;
        md_filter_t *fnode = new_node->filter;

        while (fnode) {
            nnode = fnode->next;
            if (fnode->field == COLLECTOR) {
                mediator_config_error(
                    "FILTER invalid: COLLECTOR field not valid"
                    " in COLLECTOR block.");
            }
            fnode = nnode;
        }
    }

    new_node->and_filter = and_filter;

    attachHeadToSLL((mdSLL_t **)&(ctemp), (mdSLL_t *)new_node);

    temp_list = NULL;
    coll_temp = NULL;
    ftemp = NULL;
    and_filter = FALSE;
    numValueList = 0;
    md_ip_intransport = NONE;
#if HAVE_SPREAD
    num_in_groups = 0;
#endif
}

/* COLLECTOR block: SILK_PROBE [PROBE_NAME] [VLAN_OR_INGRESS] */
static void
parseCollectorSilkConfigProbe(
    char                   *probe_name,
    mdAcceptFilterField_t   field)
{
    gboolean  use_vlan = FALSE;
    GError   *err = NULL;

    switch (field) {
      case VLAN:
      case VLANINT:
        use_vlan = TRUE;
        break;
      case INGRESS:
      case EGRESS:
        use_vlan = FALSE;
        break;
      default:
        mediator_config_error("Second argument to SILK_PROBE must be one of"
                              " VLAN, VLANINT, INGRESS, EGRESS");
        break;
    }

    if (!mdCollectorSetSilkProbe(coll_temp, probe_name, use_vlan, &err)) {
        mediator_config_error("Error setting SILK_PROBE to '%s' %s: %s",
                              probe_name, (use_vlan ? "VLAN" : "INGRESS"),
                              err->message);
    }

    free(probe_name);
}

static void
parseFilterBegin(
    void)
{
    g_warning("Filter blocks outside of COLLECTOR or EXPORTER blocks will "
              "apply to all COLLECTORS.");
}

static void
parseFilterEnd(
    void)
{
    md_collect_node_t *cnode = NULL;

    if (NULL == ftemp) {
        mediator_config_error("FILTER block contains no comparisons");
    }

    for (cnode = ctemp; cnode; cnode = cnode->next) {
        if (!cnode->filter) {
            cnode->filter = ftemp;
        } else {
            md_filter_t *new_filter = md_new_filter_node();
            memcpy(new_filter, ftemp, sizeof(md_filter_t));
            attachHeadToSLL((mdSLL_t **)&(cnode->filter),
                            (mdSLL_t *)new_filter);
            /* remove next reference */
        }
    }

    md_config.shared_filter = TRUE;

    ftemp = NULL;
}


static void
parseExporterBegin(
    mdTransportType_t   mode,
    char               *name)
{
    switch (mode) {
      case TEXT:
        md_ipfix_outtransport = TEXT;
        break;
      case UDP:
        md_ipfix_outtransport = UDP;
        break;
      case TCP:
        md_ipfix_outtransport = TCP;
        break;
      case SPREAD:
        md_ipfix_outtransport = SPREAD;
        if (!spread_exporter) {
            spread_exporter = TRUE;
        } else {
            mediator_config_error("Error: Only ONE Spread Exporter Permitted");
        }
        break;
      case DIRECTORY:
      case FILEHANDLER:
        md_ipfix_outtransport = FILEHANDLER;
        break;
      default:
        mediator_config_error("Unacceptable transport mode for exporter");
    }

    exp_temp = mdNewFlowExporter(mode);
    if (name) {
        mdExporterSetName(exp_temp, name);
        free(name);
    }
    ssl_dedup_only = FALSE;
}

static void
parseExporterPort(
    int   port)
{
    char  str[32];

    REQUIRE_NOTNULL(exp_temp);

    if (!((md_ipfix_outtransport == TCP) || (md_ipfix_outtransport == UDP))) {
        mediator_config_error("Invalid Keyword: PORT only valid for "
                              "TCP or UDP Exporter");
    }
    if (port < 1024) {
        mediator_config_error("Invalid Export Port. Port must be above 1024.");
    }

    snprintf(str, sizeof(str), "%d", port);
    mdExporterSetPort(exp_temp, str);
}

static void
parseExporterHost(
    char  *host)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!((md_ipfix_outtransport == TCP) || (md_ipfix_outtransport == UDP))) {
        mediator_config_error("Invalid Keyword: HOSTNAME only valid "
                              "for TCP or UDP Collectors");
    }

    mdExporterSetHost(exp_temp, host);
    free(host);
}

static void
parseExporterPath(
    char  *file)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!(md_ipfix_outtransport == TEXT ||
          md_ipfix_outtransport == FILEHANDLER))
    {
        mediator_config_error("Invalid Keyword PATH.  Not a defined "
                              "keyword for this Exporter");
    }

    mdExporterSetFileSpec(exp_temp, file);
    free(file);
}

static void
parseExporterSpreadDaemon(
    char  *daemon_name)
{
    REQUIRE_NOTNULL(exp_temp);

    if ((md_ipfix_outtransport != SPREAD)) {
        mediator_config_error("Invalid keyword: DAEMON only valid for "
                              "SPREAD Exporter");
    }
#if HAVE_SPREAD
    mdExporterSetFileSpec(exp_temp, daemon_name);
#else
    mediator_config_error("Spread is not enabled. "
                          "Confirm Spread is Installed");
#endif

    free(daemon_name);
}

static void
parseExporterSpreadGroup(
    char  *group)
{
    if (md_ipfix_outtransport != SPREAD) {
        mediator_config_error("Invalid keyword: Exporter NOT "
                              "configured for SPREAD.");
    }

#if HAVE_SPREAD
    if (md_config.out_spread.groups != NULL) {
        int  n = 0;
        n = num_out_groups + 2;
        md_out_groups = (char **)g_renew(char *, md_out_groups, n);
        md_out_groups[num_out_groups] = g_strdup(group);
        md_out_groups[n - 1] = (char *)'\0';
        md_config.out_spread.groups = md_out_groups;
        sftemp = md_new_spread_node();
        sftemp->group = md_out_groups[num_out_groups];
        attachHeadToSLL((mdSLL_t **)&(md_config.mdspread),
                        (mdSLL_t *)sftemp);
        sftemp = NULL;
        num_out_groups++;
    } else {
        md_out_groups = g_new0(char *, 2);
        md_out_groups[0] = g_strdup(group);
        md_out_groups[1] = (char *)'\0';
        md_config.out_spread.groups = md_out_groups;
        sftemp = md_new_spread_node();
        sftemp->group = md_out_groups[num_out_groups];
        attachHeadToSLL((mdSLL_t **)&(md_config.mdspread),
                        (mdSLL_t *)sftemp);
        sftemp = NULL;
        num_out_groups++;
    }
#else  /* if HAVE_SPREAD */
    mediator_config_error("Mediator not configured with Spread Support. "
                          "Confirm Spread is installed.");
#endif /* if HAVE_SPREAD */
    free(group);
}

static void
parseExporterLock(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != FILEHANDLER && md_ipfix_outtransport != TEXT) {
        mediator_config_error("Invalid Keyword: LOCK only valid for "
                              "TEXT or FILEHANDLER Exporters.");
    }

    mdExporterSetLock(exp_temp);
}

static void
parseExporterNoFlowStats(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterSetNoFlowStats(exp_temp);
}

static void
parseExporterJson(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterSetJson(exp_temp);
}

static void
parseExporterRotateSeconds(
    int   secs)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != FILEHANDLER && md_ipfix_outtransport != TEXT) {
        mediator_config_error("Invalid Keyword: ROTATE only valid for "
                              "TEXT or FILEHANDLER Exporters.");
    }

    mdExporterSetRotate(exp_temp, secs);
}

static void
parseExporterUDPTimeout(
    int   mins)
{
    if (md_ipfix_outtransport != UDP) {
        mediator_config_error("Invalid Keyword: UDP TEMPLATE TIMEOUT "
                              "only valid for UDP Exporters.");
    }

    md_config.udp_template_timeout = mins;
}

static void
parseExporterEnd(
    void)
{
    md_export_node_t *new_node;
    gboolean          dns_dedup;
    int               i;

    REQUIRE_NOTNULL(exp_temp);

    dns_dedup = mdExporterGetDNSDedupStatus(exp_temp);

    new_node = md_new_export_node(dns_dedup, FALSE);
    new_node->exp = exp_temp;
    new_node->filter = ftemp;
    new_node->and_filter = and_filter;
    if (ssl_dedup_only) {
        new_node->ssl_dedup = md_ssl_new_dedup_state();
    }
    attachHeadToSLL((mdSLL_t **)&(ebeg),
                    (mdSLL_t *)new_node);

    if (temp_list) {
        mdExportCustomList(new_node->exp, temp_list);
    }

    if (numValueList > 0) {
        if (md_ipfix_outtransport != TEXT) {
            mediator_config_error("DPI_FIELD_LIST only valid for TEXT "
                                  "exporters.");
        }
        for (i = 0; i < numValueList; i++) {
            mdInsertDPIFieldItem(exp_temp, valueListTemp[i]);
        }
    }

    /* if (!mdExporterVerifySetup(exp_temp)) { */
    /*     exit(-1); */
    /* } */

    new_node->md5_hash = md5_filter;
    new_node->sha1_hash = sha1_filter;

    temp_list = NULL;
    exp_temp = NULL;
    ftemp = NULL;
    and_filter = FALSE;
    numValueList = 0;
    numCustomList = 0;
    md5_filter = FALSE;
    sha1_filter = FALSE;
    md_ipfix_outtransport = NONE;
    ssl_dedup_only = FALSE;
}

static void
parseExporterTextDelimiter(
    char  *delim)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("Invalid Keyword.  DELIMITER requires "
                              "TEXT Exporter.");
    }
    if (strlen(delim) != 1) {
        mediator_config_error("Invalid Text Delimiter.  Text Delimiter "
                              "may only be 1 character.");
    }

    mdExporterSetDelim(exp_temp, delim);
    free(delim);
}

static void
parseExporterDPIDelimiter(
    char  *delim)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("Invalid Keyword.  DELIMITER requires "
                              "TEXT Exporter.");
    }
    if (strlen(delim) != 1) {
        mediator_config_error("Invalid Text Delimiter.  Text Delimiter "
                              "may only be 1 character.");
    }

    mdExporterSetDPIDelim(exp_temp, delim);
    free(delim);
}

static void
parseExporterFlowOnly(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterSetFlowOnly(exp_temp)) {
        mediator_config_error("DPI_ONLY, DNS_DEDUP, SSL_DEDUP, "
                              " DEDUP_ONLY, or DNS_RR_ONLY also specified. "
                              " Only one can be listed for an exporter");
    }
}

static void
parseExporterDPIOnly(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterSetDPIOnly(exp_temp)) {
        mediator_config_error("FLOW_ONLY or DNS_DEDUP_ONLY or"
                              " SSL_DEDUP_ONLY also specified.  "
                              "Only one can be listed for an exporter");
    }
}


static void
parseExporterRemoveEmpty(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT && md_ipfix_outtransport != FILEHANDLER) {
        mediator_config_error("REMOVE_EMPTY_FILES only valid for TEXT "
                              "or FILEHANDLER Exporter");
    }

    mdExporterSetRemoveEmpty(exp_temp);
}

static void
parseExporterNoStats(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterSetStats(exp_temp, 1);
}

static void
parseExporterAddStats(
    void)
{
    REQUIRE_NOTNULL(exp_temp);
    mdExporterSetStats(exp_temp, 2);
}

static void
parseExporterPrintHeader(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("PRINT_HEADER Keyword only available "
                              "for TEXT Exporter");
    }

    mdExporterSetPrintHeader(exp_temp);
}

static void
parseExporterEscapeChars(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("ESCAPE_CHARS keyword only available "
                              "for TEXT Exporters");
    }

    mdExporterSetEscapeChars(exp_temp);
}

static void
parseExporterDNSRROnly(
    int   mode)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport == TEXT) {
        mediator_config_error("DNS_RR_ONLY keyword only available "
                              "for IPFIX (TCP, UDP, SPREAD, FILEHANDLER) "
                              "Exporters");
    }

    if (!mdExporterSetDNSRROnly(exp_temp, mode)) {
        mediator_config_error("FLOW_ONLY, DPI_ONLY, SSL_DEDUP_ONLY, "
                              "DNS_DEDUP_ONLY, or DEDUP_ONLY"
                              " not permitted with DNS_RR_ONLY");
    }
}

static void
parseExporterDNSRespOnly(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterSetDNSRespOnly(exp_temp);
}

static void
parseExporterDedupPerFlow(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterDedupPerFlow(exp_temp);
}

static void
parseExporterNoIndex(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("NO_INDEX Keyword only valid for "
                              "TEXT Exporters");
    }

    mdExporterSetNoIndex(exp_temp, TRUE);
}

static void
parseExporterNoFlow(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterSetNoFlow(exp_temp);
}

static void
parseExporterTimestamp(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("TIMESTAMP_FILES Keyword only valid for "
                              "TEXT Exporters");
    }

    mdExporterSetTimestampFiles(exp_temp);
}


static void
parseExporterMultiFiles(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("MULTI_FILES keyword only valid for TEXT "
                              "Exporters");
    }

    if (!mdExportMultiFiles(exp_temp)) {
        mediator_config_error("MULTI_FILES configuration error.");
    }
}

static void
parseExporterMetadataExport(
    void)
{
    REQUIRE_NOTNULL(exp_temp);
#if SM_ENABLE_METADATA_EXPORT
    mdExporterSetMetadataExport(exp_temp);
#else
    mediator_config_error(
        "Mediator not configured with metadata(type) export enabled. ");
#endif
}

static void
parseSpreadGroup(
    char  *name)
{
    gboolean  found = FALSE;

    if (!spread_exporter) {
        mediator_config_error("Invalid Group Block.  One Exporter must be "
                              "configured for Spread Transport");
    }

    for (sftemp = md_config.mdspread; sftemp; sftemp = sftemp->next) {
        if (strcmp(name, sftemp->group) == 0) {
            found = TRUE;
            break;
        }
    }

    if (!found) {
        mediator_config_error("Group must exist in EXPORTER GROUP.");
    }

    ftemp = sftemp->filterList;

    free(name);
}

static void
parseSpreadGroupEnd(
    void)
{
    if (ftemp == NULL) {
        mediator_config_error("No Spread Exporter Filters Found");
    }

    sftemp->filterList = ftemp;

    ftemp = NULL;
    sftemp = NULL;
}

static void
parseStatsTimeout(
    int   timeout)
{
    md_stats_timeout = timeout;
}

static void
parseLogConfig(
    char  *log_file)
{
    md_logfile = g_strdup(log_file);
    free(log_file);
}

static void
parseLogDir(
    char  *log_dir)
{
    md_logdest = g_strdup(log_dir);
    free(log_dir);
}

static void
parsePidFile(
    char  *pid_file)
{
    md_pidfile = g_strdup(pid_file);
    free(pid_file);
}

static void
parseStatisticsConfig(
    void)
{
    md_config.no_stats = TRUE;
}

static void
parsePreserveObDomainConfig(
    void)
{
    md_config.preserve_obdomain = TRUE;
}

static void
parseRewriteSslCertsConfig(
    void)
{
    md_config.rewrite_ssl_certs = TRUE;
}

static void
parseGenTombstoneConfig(
    void)
{
    md_config.gen_tombstone = TRUE;
}

static void
parseTombstoneIdConfig(
    int   configured_id)
{
    md_config.tombstone_configured_id = configured_id;
}

static void
parseDNSDeDupConfig(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterSetDNSDeDup(exp_temp);
}

static void
parseDNSDeDupOnly(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterSetDNSDeDupOnly(exp_temp)) {
        mediator_config_error("FLOW_ONLY OR DPI_ONLY not permitted with "
                              "DNS_DEDUP_ONLY");
    }
}

static void
parseComparison(
    mdAcceptFilterField_t   field,
    fieldOperator           oper,
    char                   *val,
    int                     val_type)
{
#if ENABLE_SKIPSET
    GError            *err = NULL;
#endif
    uint32_t           ip;
    uint8_t            ip6[16];
    md_collect_node_t *cnode = NULL;
    md_filter_t       *filter = md_new_filter_node();

    if (field > 15) {
        if (field != INGRESS && field != EGRESS && field != COLLECTOR) {
            mediator_config_error("Invalid Filter Field.  "
                                  "Please refer to documentation for "
                                  "acceptable filter fields.");
        }
    }

    filter->field = field;

    filter->oper = oper;

    if (val_type == VAL_INTEGER) {
        filter->val[filter->num_in_list] = atoi(val);
    } else if (val_type == VAL_IP) {
        switch (field) {
          case SIP_V4:
          case DIP_V4:
          case ANY_IP:
          case SIP_ANY:
          case DIP_ANY:
            if ((inet_pton(AF_INET, val, (struct in_addr *)&ip) <= 0)) {
                mediator_config_error("Invalid IPv4 Address");
            }
            filter->val[filter->num_in_list] = g_ntohl(ip);
            break;
          case SIP_V6:
          case DIP_V6:
          case ANY_IP6:
            if (inet_pton(AF_INET6, val, ip6) <= 0) {
                mediator_config_error("Invalid IPv6 Address");
            }
            memcpy(filter->val, ip6, 16);
            break;
          default:
            mediator_config_error("Invalid Filter Field for IP Address");
        }
    } else if ((field == COLLECTOR) && !(oper == EQUAL || oper == NOT_EQUAL)) {
        mediator_config_error(
            "COLLECTOR Filter must use EQUAL/NOT_EQUAL operator");
    } else if (val_type == VAL_QSTRING &&
               (oper == IN_LIST || oper == NOT_IN_LIST))
    {
        switch (field) {
          case SIP_V4:
          case DIP_V4:
          case ANY_IP:
          case SIP_V6:
          case DIP_V6:
          case ANY_IP6:
          case SIP_ANY:
          case DIP_ANY:
            break;
          default:
            mediator_config_error("Invalid Filter Field for "
                                  "IN_LIST/NOT_IN_LIST operator");
        }

#if !defined(ENABLE_SKIPSET) || !ENABLE_SKIPSET
        mediator_config_error("Unable to compare to an IPSet: super_mediator"
                              " was built without support for SiLK IPSets");
#else
        filter->ipset = mdUtilIPSetOpen(val, &err);
        if (!filter->ipset) {
            mediator_config_error("Error loading IPSET '%s': %s",
                                  val, err->message);
        }

        switch (field) {
          case SIP_V6:
          case DIP_V6:
          case ANY_IP6:
#if !HAVE_SKIPADDRSETV6
            g_warning("IPv6 address test is always false due to missing"
                      " IPv6 support in SiLK");
#else
            if (!skIPSetIsV6(filter->ipset->ipset)) {
                g_warning("IPSet configured for IPv4 but filtering on IPv6 "
                          "Address.");
            }
#endif  /* HAVE_SKIPADDRSETV6 */
            break;

          case SIP_V4:
          case DIP_V4:
            if (skIPSetIsV6(filter->ipset->ipset)) {
                g_warning("IPSet configured for IPv6 but filtering on IPv4 "
                          "Address.");
            }
            break;

          default:
            break;
        }
#endif  /* ENABLE_SKIPSET */
    } else if (field == COLLECTOR) {
        for (cnode = ctemp; cnode; cnode = cnode->next) {
            char *name = mdCollectorGetName(cnode);
            if (!(strcmp(name, val))) {
                filter->val[filter->num_in_list] = mdCollectorGetID(cnode);
            }
        }
        if (filter->val[filter->num_in_list] == 0) {
            mediator_config_error("No COLLECTOR with given name.");
        }
    } else {
        mediator_config_error("Invalid comparison for filter.");
    }

    filter->num_in_list++;

    attachHeadToSLL((mdSLL_t **)&(ftemp),
                    (mdSLL_t *)filter);

    free(val);
}

static smFieldMap_t *
parseMapStmt(
    char  *mapname)
{
    smFieldMap_t *map;

    map = findFieldMap(mapname, FALSE);
    free(mapname);

    return map;
}


static void
parseTableListBegin(
    char  *index_label)
{
    void *currentTable = NULL;

    if (default_tables) {
        mediator_config_error("Error: Default Tables already defined. "
                              "Remove application label from USER_IE line "
                              "to build custom tables.");
    }

    custom_tables = TRUE;

    if (index_label == NULL) {
        currentTable = mdNewTable(INDEX_DEFAULT);
    } else {
        currentTable = mdNewTable(index_label);
    }

    if (!mdInsertTableItem(currentTable, 0)) {
        mediator_config_error("Error Creating Index Table for DPI Config.");
    }

    g_free(index_label);
}


static void
parseTableList(
    char  *table)
{
    int   i = 0;
    void *currentTable = NULL;

    if (numValueList == 0) {
        mediator_config_error("No items in List.");
    }

    currentTable = mdNewTable(table);

    for (i = 0; i < numValueList; i++) {
        if (!mdInsertTableItem(currentTable, valueListTemp[i])) {
            mediator_config_error("Item can not be present in another list.");
        }
    }

    free(table);
}


static void
parseDnsDedupRecordTypeList(
    void)
{
    int  i;

    if (numValueList == 0) {
        mediator_config_error("No items in DNS_DEDUP RECORDS list.");
    }
    if (cfg_dns_dedup.type_list) {
        mediator_config_error("DNS_DEDUP RECORDS previously set.");
    }

    cfg_dns_dedup.type_list = g_new0(int, 1 + MD_DNS_DD_MAX_TYPE);
    for (i = 0; i < numValueList; i++) {
        /* turn types of records "on" */
        if (valueListTemp[i] > MD_DNS_DD_MAX_TYPE) {
            mediator_config_error("Invalid RECORD Type. "
                                  "Valid Types: 0,1,2,5,6,12,15,16,28,33");
        }
        cfg_dns_dedup.type_list[valueListTemp[i]] = 1;
    }

    numValueList = 0;
}

static void
parseValueListItems(
    char  *val)
{
    int  value = atoi(val);

    if (value < 0 || value > 65535) {
        mediator_config_error("Item too large for list. "
                              " Must be between 1 - 65535.");
    }
    if (numValueList == sizeof(valueListTemp) / sizeof(valueListTemp[0])) {
        mediator_config_error("Attempted to add too many items to list.");
    }

    valueListTemp[numValueList] = value;
    numValueList++;

    free(val);
}

static void
parseFieldListItems(
    char                   *fint,
    mdAcceptFilterField_t   field)
{
    mdFieldList_t *item = NULL;

    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("Custom List FIELDS only valid for "
                              "TEXT exporters.");
    }

    numCustomList++;

    if (fint) {
        field = atoi(fint);
    }

    if (field == DPI) {
        mdExporterCustomListDPI(exp_temp);
        free(fint);
        return;
    }

    item = mdCreateFieldList(field);
    if (!item) {
        if (fint) {
            mediator_config_error(
                "Invalid Custom Field Item \"%s\" at position %d in list",
                fint, numCustomList);
        } else {
            mediator_config_error(
                "Invalid Custom Field Item ID %d at position %d in list",
                field, numCustomList);
        }
    }

    if (temp_list == NULL) {
        temp_list = item;
    } else {
        /* append item to list */
        mdFieldList_t *f = temp_list;
        while (f->next) {
            f = f->next;
            continue;
        }
        f->next = item;
    }

    free(fint);
}


static void
parseMySQLParams(
    char  *user,
    char  *pw,
    char  *db,
    char  *host,
    char  *table)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterAddMySQLInfo(exp_temp, user, pw, db, host, table)) {
        exit(-1);
    }
    free(user);
    free(pw);
    free(db);
    free(host);
    free(table);
}

static void
parseExporterRemoveUploaded(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterSetRemoveUploaded(exp_temp);
}


static void
parseUserInfoElement(
    int    ie_num,
    char  *name,
    int    app_num)
{
    void            *table = NULL;
    fbInfoElement_t  add_element;

    if (ie_num > 16383) {
        mediator_config_error("Invalid Information Element ID number. "
                              "Number must be between 1 and 16383");
    }
    if (app_num > 65535) {
        mediator_config_error("Invalid Information Element ID number. "
                              "Number must be between 0 and 65535");
    }

    if (user_elements == NULL) {
        user_elements = g_new0(fbInfoElement_t, 50);
    } else if (numUserElements > 50) {
        mediator_config_error("Max Limit reached on adding user-defined"
                              " Information Elements");
    }

    memset(&add_element, 0, sizeof(fbInfoElement_t));

    add_element.num = ie_num;
    add_element.ent = 6871;
    add_element.len = FB_IE_VARLEN;
    add_element.ref.name = g_strdup(name);
    add_element.midx = 0;
    add_element.flags = 0;

    memcpy((user_elements + numUserElements), &add_element,
           sizeof(fbInfoElement_t));

    numUserElements++;

    if (app_num) {
        if (custom_tables) {
            mediator_config_error("Invalid application label for USER_IE "
                                  "Add Information Element Number"
                                  " to DPI_CONFIG tables.");
        }

        if (!default_tables) {
            mdBuildDefaultTableHash();
            default_tables = TRUE;
        }

        table = mdGetTable(app_num);

        if (!table) {
            mediator_config_error("Not a valid application label for USER_IE");
        }

        if (!mdInsertTableItem(table, ie_num)) {
            mediator_config_error("Information Element already defined.");
        }
    }

    free(name);
}


static void
parseDnsDedupConfigEnd(
    void)
{
    md_export_node_t *cnode = NULL;

    if (cfg_dns_dedup.temp_name) {
        cnode = findExporter(cfg_dns_dedup.temp_name, "DNS_DEDUP");
        if (!cnode->dns_dedup) {
            mediator_config_error("Exporter for DNS_DEDUP config"
                                  " block does not have DNS_DEDUP enabled.");
        }
    } else {
        /* no name, find most recent exporter with DNS_DEDUP enabled */
        for (cnode = ebeg; cnode && !cnode->dns_dedup; cnode = cnode->next) {
            /* empty */
        }
        if (NULL == cnode) {
            mediator_config_error("No exporter has DNS_DEDUP enabled.");
        }
    }

    md_ipfix_outtransport = mdExporterGetType(cnode->exp);

    if (cnode->dedup && (md_ipfix_outtransport == TEXT) &&
        !(mdExporterGetJson(exp_temp)))
    {
        mediator_config_error("Exporter already configured for DEDUP. "
                              "Define a separate TEXT EXPORTER for DNS_DEDUP");
    }

    /* ownership of type_list passes to dns_dedup */
    md_dns_dedup_configure_state(
        cnode->dns_dedup, cfg_dns_dedup.type_list, cfg_dns_dedup.max_hit,
        cfg_dns_dedup.flush_timeout, cfg_dns_dedup.lastseen,
        cfg_dns_dedup.map, cfg_dns_dedup.exportname);

    free(cfg_dns_dedup.temp_name);
    memset(&cfg_dns_dedup, 0, sizeof(cfg_dns_dedup));
}

static void
parseSSLConfigBegin(
    char  *exp_name)
{
    md_export_node_t *cnode = NULL;

    cnode = findExporter(exp_name, "SSL_CONFIG");
    etemp = cnode;
    exp_temp = cnode->exp;

    md_ipfix_outtransport = mdExporterGetType(exp_temp);

    numValueList = 0;

    free(exp_name);
}

static void
parseSSLIssuerTypeList(
    void)
{
    int       i;
    uint32_t *sslIssuerlist;

    REQUIRE_NOTNULL(exp_temp);

    sslIssuerlist = g_new0(uint32_t, MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));

    if (valueListWild) {
        /* TURN THEM ALL ON */
        memset(sslIssuerlist, 0xff,
               sizeof(uint32_t) * MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));
        mdExporterSetSSLConfig(exp_temp, sslIssuerlist, 1);
        return;
    }

    if (numValueList == 0) {
        mediator_config_error("No items in ISSUER list.");
    }

    for (i = 0; i < numValueList; i++) {
        if (valueListTemp[i] >= MD_SSL_CONFIG_MAX_ID) {
            mediator_config_error("SSL Issuer List takes only values 0-"
                                  MD_SSL_CONFIG_MAX_ID_STR);
        }
        /* turn types of records "on" */
        MD_BMAP_SET(sslIssuerlist, valueListTemp[i]);
    }

    mdExporterSetSSLConfig(exp_temp, sslIssuerlist, 1);
}

static void
parseSSLSubjectTypeList(
    void)
{
    int       i;
    uint32_t *sslSubjectlist;

    REQUIRE_NOTNULL(exp_temp);

    sslSubjectlist = g_new0(uint32_t, MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));

    if (valueListWild) {
        /* TURN THEM ALL ON */
        memset(sslSubjectlist, 0xff,
               sizeof(uint32_t) * MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));
        mdExporterSetSSLConfig(exp_temp, sslSubjectlist, 2);
        return;
    }

    if (numValueList == 0) {
        mediator_config_error("No items in SUBJECT list.");
    }

    for (i = 0; i < numValueList; i++) {
        if (valueListTemp[i] >= MD_SSL_CONFIG_MAX_ID) {
            mediator_config_error("SSL Subject List takes only values 0-"
                                  MD_SSL_CONFIG_MAX_ID_STR);
        }
        /* turn types of records "on" */
        MD_BMAP_SET(sslSubjectlist, valueListTemp[i]);
    }

    mdExporterSetSSLConfig(exp_temp, sslSubjectlist, 2);
}


static void
parseSSLOtherTypeList(
    void)
{
    int       i;
    uint32_t *sslOtherList;

    REQUIRE_NOTNULL(exp_temp);

    sslOtherList = g_new0(uint32_t, MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));

    if (valueListWild) {
        /* TURN THEM ALL ON */
        memset(sslOtherList, 0xff,
               sizeof(uint32_t) * MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));
        mdExporterSetSSLConfig(exp_temp, sslOtherList, 3);
        return;
    }

    if (numValueList == 0) {
        mediator_config_error("No items in OTHER list.");
    }

    for (i = 0; i < numValueList; i++) {
        if (valueListTemp[i] >= MD_SSL_CONFIG_MAX_ID) {
            mediator_config_error("SSL Other List takes only values 0-"
                                  MD_SSL_CONFIG_MAX_ID_STR);
        }
        MD_BMAP_SET(sslOtherList, valueListTemp[i]);
    }

    mdExporterSetSSLConfig(exp_temp, sslOtherList, 3);
}

static void
parseSSLExtensionsTypeList(
    void)
{
    int       i;
    uint32_t *sslExtList;

    REQUIRE_NOTNULL(exp_temp);

    /* yaf only exports id-ce 14-37 */
    sslExtList = g_new0(uint32_t, MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));

    if (valueListWild) {
        /* TURN THEM ALL ON */
        memset(sslExtList, 0xff,
               sizeof(uint32_t) * MD_BMAP_ARRAYSIZE(MD_SSL_CONFIG_MAX_ID));
        mdExporterSetSSLConfig(exp_temp, sslExtList, 4);
        return;
    }

    if (numValueList == 0) {
        mediator_config_error("No items in Extensions list.");
    }

    for (i = 0; i < numValueList; i++) {
        if (valueListTemp[i] >= MD_SSL_CONFIG_MAX_ID) {
            mediator_config_error("SSL Extensions List takes only values 0-"
                                  MD_SSL_CONFIG_MAX_ID_STR);
        }
        MD_BMAP_SET(sslExtList, valueListTemp[i]);
        /* switch (valueListTemp[i]) { */
        /*   case 14: */
        /*   case 15: */
        /*   case 16: */
        /*   case 17: */
        /*   case 18: */
        /*   case 29: */
        /*   case 31: */
        /*   case 32: */
        /*     sslExtList[valueListTemp[i]] = 1; */
        /*     continue; */
        /*   default: */
        /*     mediator_config_error( */
        /*         "Invalid Extension in SSL EXTENSIONS List." */
        /*         " super_mediator accepts 14-18, 29, 31, 32"); */
        /* } */
    }

    mdExporterSetSSLConfig(exp_temp, sslExtList, 4);
}

static void
parseDedupConfig(
    char  *exp_name)
{
    md_export_node_t *cnode = NULL;

    cnode = findExporter(exp_name, "DEDUP_CONFIG");
    etemp = cnode;
    exp_temp = cnode->exp;

    md_ipfix_outtransport = mdExporterGetType(exp_temp);

    if (cnode->dns_dedup && (md_ipfix_outtransport == TEXT) &&
        !(mdExporterGetJson(exp_temp)))
    {
        mediator_config_error("Exporter already configured for DNS_DEDUP."
                              " Define a separate TEXT EXPORTER for DEDUP");
    } else if (cnode->ssl_dedup && (md_ipfix_outtransport == TEXT) &&
               !(mdExporterGetJson(exp_temp)))
    {
        mediator_config_error("Exporter already configured for SSL_DEDUP."
                              " Define a separate TEXT EXPORTER for DEDUP");
    }

    etemp->dedup = md_dedup_new_dedup_state();

    mdExporterSetDeDupConfig(etemp->exp);

    free(exp_name);

    numValueList = 0;
}

static void
parseDedupFileList(
    char                   *file,
    mdAcceptFilterField_t   field,
    char                   *mapname)
{
    int            i = 0;
    int            sip;
    md_dedup_ie_t *ietab = NULL;
    smFieldMap_t  *map = NULL;

    if (numValueList == 0) {
        mediator_config_error("No items in FILE List.");
    }

    switch (field) {
      case SIP_V4:
      case SIP_ANY:
        sip = 1;
        break;
      case DIP_V4:
      case DIP_ANY:
        sip = 0;
        break;
      case FLOWKEYHASH:
        sip = 2;
        break;
      default:
        mediator_config_error(
            "Invalid Field in DEDUP_CONFIG."
            "  SIP, DIP, and FLOWKEYHASH are only valid fields.");
    }

    if (mapname) {
        map = findFieldMap(mapname, FALSE);
        free(mapname);
    }

    if (md_ipfix_outtransport != TEXT) {
        /* create a table for each element in the list bc it needs a template
         * for each element in the list */
        for (i = 0; i < numValueList; i++) {
            ietab = md_dedup_add_ie_table(etemp->dedup, file, map,
                                          valueListTemp[i], sip);
            if (!ietab) {
                mediator_config_error(
                    "Information Element already in FILE Table.");
            }
        }
    } else {
        ietab = md_dedup_add_ie_table(etemp->dedup, file, map,
                                      valueListTemp[0], sip);
        if (!ietab) {
            mediator_config_error("Information Element already in FILE Table.");
        }
        if ((valueListTemp[0] == 244) && (numValueList > 1)) {
            mediator_config_error("244 (SSL) must exist in a list by itself.");
        }
        for (i = 1; i < numValueList; i++) {
            if (valueListTemp[i] == 244) {
                mediator_config_error(
                    "244 (SSL) must exist in a list by itself.");
            }
            md_dedup_add_ie(etemp->dedup, ietab, valueListTemp[i]);
        }
    }

    free(file);
    numValueList = 0;
}

static void
parseSSLCertDedup(
    void)
{
    REQUIRE_NOTNULL(etemp);

    if (etemp->dns_dedup && (md_ipfix_outtransport == TEXT) &&
        !(mdExporterGetJson(exp_temp)))
    {
        mediator_config_error("Exporter already configured for DNS_DEDUP."
                              " Define a separate TEXT EXPORTER for SSL_DEDUP");
    } else if (etemp->dedup && (md_ipfix_outtransport == TEXT) &&
               !(mdExporterGetJson(exp_temp)))
    {
        mediator_config_error("Exporter already configured for DEDUP."
                              " Define a separate TEXT EXPORTER for SSL_DEDUP");
    }

    /* may have already been enabled with SSL_DEDUP_ONLY */
    if (etemp->ssl_dedup == NULL) {
        etemp->ssl_dedup = md_ssl_new_dedup_state();
        mdExporterSetSSLDeDupConfig(etemp->exp);
    }
}

static void
parseSSLCertFile(
    char  *filename)
{
    if (md_ipfix_outtransport != TEXT) {
        mediator_config_error("CERT_FILE only valid for TEXT Exporters.");
    }

    if (mdExporterGetJson(exp_temp)) {
        mediator_config_error("CERT_FILE not valid with JSON");
    }

    md_ssl_dedup_configure_state(etemp->ssl_dedup, 0, 0, filename, NULL, FALSE);

    free(filename);
}

static void
parseSSLDeDupOnly(
    int   mode)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterSetSSLDeDupOnly(exp_temp, mode)) {
        mediator_config_error("FLOW_ONLY or DNS_DEDUP_ONLY or"
                              " DEDUP_ONLY or DNS_RR_ONLY also specified."
                              " Only one can be used per exporter.");
    }

    ssl_dedup_only = TRUE;
}


static void
parseExporterDedupOnly(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterDedupOnly(exp_temp)) {
        mediator_config_error("FLOW_ONLY or DNS_DEDUP_ONLY or"
                              " SSL_DEDUP_ONLY or DNS_RR_ONLY "
                              "also specified.  Only "
                              " can be used per exporter.");
    }
}

static void
parseExporterSSLMD5Hash(
    void)
{
    REQUIRE_NOTNULL(exp_temp);
#if HAVE_OPENSSL
    if (!mdExporterSetSSLMD5Hash(exp_temp)) {
        mediator_config_error(
            "ERROR MD5: Exporter already configured with conflicting settings");
    }
    md5_filter = TRUE;

#else  /* if HAVE_OPENSSL */
    mediator_config_error("Super_mediator not configured with OpenSSL support");
#endif /* if HAVE_OPENSSL */
}

static void
parseExporterSSLSHA1Hash(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

#if HAVE_OPENSSL
    if (!mdExporterSetSSLSHA1Hash(exp_temp)) {
        mediator_config_error("ERROR SHA_1: Exporter already configured"
                              " with conflicting settings");
    }

    sha1_filter = TRUE;
#else  /* if HAVE_OPENSSL */
    mediator_config_error("Super_mediator not configured with OpenSSL support");
#endif /* if HAVE_OPENSSL */
}

static void
parseExporterGzipFiles(
    void)
{
    REQUIRE_NOTNULL(exp_temp);

    mdExporterGZIPFiles(exp_temp);
}

static void
parseExporterMovePath(
    char  *path)
{
    REQUIRE_NOTNULL(exp_temp);

    if ((md_ipfix_outtransport != FILEHANDLER)
        && (md_ipfix_outtransport != TEXT))
    {
        mediator_config_error("Invalid Keyword: MOVE must be used with a "
                              "FILEHANDLER or TEXT EXPORTER");
    }

    if (!g_file_test(path, G_FILE_TEST_IS_DIR)) {
        mediator_config_error("MOVE expects a valid file directory");
    }

    mdExporterSetMovePath(exp_temp, path);
    free(path);
}

static void
parseExporterMatchesIPSet(
    char  *path,
    char  *name)
{
    GError *err = NULL;

    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterSetMatchesIPSet(exp_temp, path, name, &err)) {
        mediator_config_error("Error setting MATCHES_IPSET to '%s': %s",
                              path, err->message);
    }

    free(name);
    free(path);
}

static void
parseExporterPrefixMap(
    char  *path,
    char  *map_name)
{
    GError *err = NULL;

    REQUIRE_NOTNULL(exp_temp);

    if (!mdExporterSetPrefixMap(exp_temp, path, map_name, &err)) {
        mediator_config_error("Error setting PREFIX_MAP to '%s': %s",
                              path, err->message);
    }

    free(map_name);
    free(path);
}

/* SILK_CONFIG and SILK_CONFIG END */
static void
parseSilkConfigBeginEnd(
    int   begin_end)
{
#if !defined(ENABLE_SKTYPESENSOR) || !ENABLE_SKTYPESENSOR
    (void)begin_end;
    mediator_config_error(("%s was built without support for"
                           " SiLK Type+Sensor Labeling"), g_get_prgname());
#else  /* if !defined(ENABLE_SKTYPESENSOR) || !ENABLE_SKTYPESENSOR */
    if (0 == begin_end) {
        /* SILK_CONFIG (begin) */
        if (mdSkpcSilkTypeSensorEnabled()) {
            mediator_config_error(
                "A SILK_CONFIG block was previously seen");
        }
    } else {
        /* SILK_CONFIG END */
        GError *err = NULL;

        if (!mdSkpcSilkTypeSensorLoad(&err)) {
            mediator_config_error("Error in SILK_CONFIG block: %s",
                            err->message);
        }
    }
#endif  /* ENABLE_SKTYPESENSOR */
}

/* SILK_CONFIG block: SILK_CONF_PATH and SENSOR_CONF_PATH */
static void
parseSilkConfigStmt(
    char  *sensor_conf,
    char  *silk_conf)
{
#if ENABLE_SKTYPESENSOR
    GError *err = NULL;

    if (!mdSkpcSilkTypeSensorSet(silk_conf, sensor_conf, &err)) {
        mediator_config_error("%s", err->message);
    }
#endif  /* ENABLE_SKTYPESENSOR */

    free(sensor_conf);
    free(silk_conf);
}

static void
parseMapLine(
    char  *label)
{
    smFieldMapKV_t *value;
    smFieldMapKV_t *key;
    int             i;

    REQUIRE_NOTNULL(mapitem);

    if (numValueList == 0) {
        mediator_config_error("No items in MAP list.");
    }

    if (mapitem->labels == NULL) {
        mapitem->labels = g_new0(char *, MAX_MAPS);
    } else if (mapitem->count + 1 >= MAX_MAPS) {
        mediator_config_error("Too many labels in the map");
    }

    ++mapitem->count;
    mapitem->labels[mapitem->count] = g_strdup(label);
    free(label);

    for (i = 0; i < numValueList; i++) {
        key = g_slice_new0(smFieldMapKV_t);
        key->val = (uint32_t)valueListTemp[i];
        value = g_slice_new0(smFieldMapKV_t);
        value->val = mapitem->count;
        smHashTableInsert(mapitem->table, (uint8_t *)key, (uint8_t *)value);
    }

    numValueList = 0;
}

/* Handle OBID_MAP or VLAN_MAP */
static void
parseMapBegin(
    mdAcceptFilterField_t   map_type,
    char                   *name)
{
    if (mapitem) {
        mediator_config_error("Programmer error: mapitem is not null");
    }
    if (!(VLAN == map_type || OBDOMAIN == map_type)) {
        mediator_config_error("Unexpected map type value");
    }
    if (NULL != findFieldMap(name, TRUE)) {
        mediator_config_error(
            "Cannot create MAP since name already in use by another map");
    }

    mapitem = g_slice_new0(smFieldMap_t);
    mapitem->field = map_type;
    mapitem->name = g_strdup(name);
    mapitem->table = smCreateHashTable(sizeof(uint32_t),
                                       md_free_hash_key, md_free_hash_key);
    numValueList = 0;
    attachHeadToSLL((mdSLL_t **)&(maptemp), (mdSLL_t *)mapitem);
    free(name);
}

static void
parseMapEnd(
    mdAcceptFilterField_t   map_type)
{
    REQUIRE_NOTNULL(mapitem);

    if (map_type != mapitem->field) {
        mediator_config_error("Unexpected map type value");
    }
    if (mapitem->labels == NULL) {
        mediator_config_error("No labels were created in MAP block.");
    }
    if ((mapitem->labels[0] == NULL) && !mapitem->discard) {
        mediator_config_error(
            "Must specify either OTHER Map List or DISCARD_OTHER");
    }

    mapitem = NULL;
}

static void
parseMapOther(
    char  *name)
{
    REQUIRE_NOTNULL(mapitem);

    if (mapitem->discard) {
        mediator_config_error("DISCARD_OTHER not valid with OTHER list");
    }
    if (mapitem->labels == NULL) {
        mapitem->labels = g_new0(char *, MAX_MAPS);
    }
    mapitem->labels[0] = g_strdup(name);
    free(name);
    mapitem->count++;
}

static void
parseMapDiscard(
    void)
{
    REQUIRE_NOTNULL(mapitem);

    if (mapitem->labels[0] != NULL) {
        mediator_config_error("OTHER is not valid with DISCARD_OTHER");
    }
    mapitem->discard = TRUE;
}

/**
 *  Parses 'number' as a number in 'base', frees 'number' and returns the
 *  result.  Exits the program if parsing fails or the value is negative or
 *  greater than INT_MAX.
 */
static int
parseNumericValue(
    char  *number,
    int    base)
{
    long  val;

    errno = 0;
    val = strtol(number, NULL, base);
    if (val < 0) {
        mediator_config_error("Negative values are not allowed");
    }
    if (val > INT_MAX) {
        mediator_config_error("Value exceeds maximum allowed");
    }
    free(number);
    return (int)val;
}


int
yyerror(
    const char  *s)
{
    /* mediator config error subtracts one */
    lineNumber++;
    mediator_config_error("%s", s);
    return 0;
}
