/*
 *  Copyright (C) 2012-2024 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_ctx.h
 *
 *  IPFIX mediator for filtering, DNS deduplication, and other mediator-like
 *  things
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  Super Mediator 1.10.0
 *
 *  Copyright 2023 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  GOVERNMENT PURPOSE RIGHTS - Software and Software Documentation
 *  Contract No.: FA8702-15-D-0002
 *  Contractor Name: Carnegie Mellon University
 *  Contractor Address: 4500 Fifth Avenue, Pittsburgh, PA 15213
 *  The Government's rights to use, modify, reproduce, release, perform,
 *  display, or disclose this software are restricted by paragraph (b)(2) of
 *  the Rights in Noncommercial Computer Software and Noncommercial Computer
 *  Software Documentation clause contained in the above identified
 *  contract. No restrictions apply after the expiration date shown
 *  above. Any reproduction of the software or portions thereof marked with
 *  this legend must also reproduce the markings.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM23-2316
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#ifndef MD_CTX
#define MD_CTX

#include "mediator_autohdr.h"
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h>
#include <assert.h>
#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_FCNTL_H
#include <sys/fcntl.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif
#include <glib.h>
#include <fixbuf/public.h>


#define FBSTLNEXT(a, b) fbSubTemplateListGetNextPtr((a), (b))
#define FBSTMLNEXT(a, b) fbSubTemplateMultiListEntryNextDataPtr((a), (b))
#define FBSTMLNEXTENTRY(a, b) fbSubTemplateMultiListGetNextEntry((a), (b))

#define CERT_PEN 6871


#define MD_UNUSED_PARAM(x_)  (void)(x_)


#define MAX_LIST 10
/* 30 sec */
#define MD_RESTART_MS 30000
#define PRINT_TIME_FMT "%04u-%02u-%02u %02u:%02u:%02u"
#define MD_MSGLEN_STD 65535
#define MAX_MAPS 100
typedef enum mdTransportType_en {
    NONE,
    TCP,
    UDP,
    SPREAD,
    FILEHANDLER,
    TEXT,
    DIRECTORY
} mdTransportType_t;

typedef enum fieldOperator_en {
    OPER_UNTOUCHED,
    IN_LIST,
    NOT_IN_LIST,
    EQUAL,
    NOT_EQUAL,
    LESS_THAN,
    LESS_THAN_OR_EQUAL,
    GREATER_THAN,
    GREATER_THAN_OR_EQUAL
} fieldOperator;

typedef enum mdAcceptFilterField_en {
    SIP_ANY,
    DIP_ANY,
    SIP_V4,
    DIP_V4,
    SPORT,
    DPORT,
    PROTOCOL,
    APPLICATION,
    SIP_V6,
    DIP_V6,
    ANY_IP6,
    ANY_IP,
    ANY_PORT,
    OBDOMAIN,
    IPVERSION,
    VLAN,

    /* Most of the following keywords are NOT supported in FILTERS */
    FLOWKEYHASH,
    DURATION,
    STIME,
    ENDTIME,
    STIMEMS,
    ETIMEMS,
    SIP_INT,
    DIP_INT,
    RTT,
    PKTS,
    RPKTS,
    BYTES,
    RBYTES,
    IFLAGS,
    RIFLAGS,
    UFLAGS,
    RUFLAGS,
    ATTRIBUTES,
    RATTRIBUTES,
    MAC,
    DSTMAC,
    TCPSEQ,
    RTCPSEQ,
    ENTROPY,
    RENTROPY,
    END,
    OSNAME,
    OSVERSION,
    ROSNAME,
    ROSVERSION,
    FINGERPRINT,
    RFINGERPRINT,
    DHCPFP,
    DHCPVC,
    RDHCPFP,
    RDHCPVC,
    INGRESS,
    EGRESS,
    DATABYTES,
    RDATABYTES,
    ITIME,
    RITIME,
    STDITIME,
    RSTDITIME,
    TCPURG,
    RTCPURG,
    SMALLPKTS,
    RSMALLPKTS,
    LARGEPKTS,
    RLARGEPKTS,
    NONEMPTYPKTS,
    RNONEMPTYPKTS,
    MAXSIZE,
    RMAXSIZE,
    STDPAYLEN,
    RSTDPAYLEN,
    FIRSTEIGHT,
    DPI,
    VLANINT,
    TOS,
    RTOS,
    MPLS1,
    MPLS2,
    MPLS3,
    COLLECTOR,
    FIRSTNONEMPTY,
    RFIRSTNONEMPTY,
    MPTCPSEQ,
    MPTCPTOKEN,
    MPTCPMSS,
    MPTCPID,
    MPTCPFLAGS,
    PAYLOAD,
    RPAYLOAD,
    DHCPOPTIONS,
    RDHCPOPTIONS,
    NDPI_MASTER,
    NDPI_SUB,
    NONE_FIELD
} mdAcceptFilterField_t;

typedef enum mdLogLevel_en {
    MD_DEBUG,
    MESSAGE,
    WARNING,
    ERROR,
    QUIET
} mdLogLevel_t;

typedef struct mdConfig_st mdConfig_t;

/* configuration options */
extern int              myVersion;
extern int              md_stats_timeout;
extern mdConfig_t       md_config;
#if HAVE_SPREAD
extern char           **md_out_groups;
extern int              num_out_groups;
#endif
extern char            *md_logfile;
extern char            *md_logdest;
extern char            *md_pidfile;
extern mdLogLevel_t     md_log_level;
extern uint16_t         dns_flush_timeout;
extern gboolean         multi_file_mode;
extern fbInfoElement_t *user_elements;
#if ENABLE_SKIPSET || ENABLE_SKPREFIXMAP || ENABLE_SKTYPESENSOR
extern int              app_registered;
#endif


struct mdFlowCollector_st;
typedef struct mdFlowCollector_st mdFlowCollector_t;

struct mdFlowExporter_st;
typedef struct mdFlowExporter_st mdFlowExporter_t;


#if ENABLE_SKIPSET
/* From mediator_utils.h: A wrapper over the SiLK IPSet structure to avoid
 * opening the same file multiple times. */
typedef struct mdIPSet_st mdIPSet_t;
#endif  /* ENABLE_SKIPSET */

/* Double Linked List */
typedef struct mdDLL_st mdDLL_t;
struct mdDLL_st {
    mdDLL_t  *next;
    mdDLL_t  *prev;
};

/* Single Linked List */
typedef struct mdSLL_st mdSLL_t;
struct mdSLL_st {
    mdSLL_t  *next;
};

typedef struct mdQueue_st {
    mdDLL_t  *head;
    mdDLL_t  *tail;
} mdQueue_t;

typedef struct smHashTable_st {
    size_t       len;
    GHashTable  *table;
} smHashTable_t;

typedef struct smFieldMap_st smFieldMap_t;
struct smFieldMap_st {
    smFieldMap_t           *next;
    mdAcceptFilterField_t   field;
    smHashTable_t          *table;
    char                   *name;
    char                  **labels;
    size_t                  count;
    gboolean                discard;
};

typedef struct smFieldMapKV_st {
    uint32_t   val;
} smFieldMapKV_t;

typedef struct md_dns_dedup_state_st md_dns_dedup_state_t;

typedef struct md_dedup_state_st md_dedup_state_t;
typedef struct md_dedup_str_node_st md_dedup_str_node_t;

typedef struct md_ssl_dedup_state_st md_ssl_dedup_state_t;

typedef struct md_filter_st md_filter_t;
struct md_filter_st {
    md_filter_t            *next;
#if ENABLE_SKIPSET
    mdIPSet_t              *ipset;
#endif
    fieldOperator           oper;
    mdAcceptFilterField_t   field;
    uint8_t                 num_in_list;
    uint32_t                val[MAX_LIST];
};

typedef struct md_spread_filter_st md_spread_filter_t;
struct md_spread_filter_st {
    md_spread_filter_t  *next;
    char                *group;
    md_filter_t         *filterList;
};

typedef struct md_export_node_st md_export_node_t;
struct md_export_node_st {
    md_export_node_t      *next;
    mdFlowExporter_t      *exp;
    md_filter_t           *filter;
    md_dns_dedup_state_t  *dns_dedup;
    md_dedup_state_t      *dedup;
    md_ssl_dedup_state_t  *ssl_dedup;
    gboolean               and_filter;
    gboolean               md5_hash;
    gboolean               sha1_hash;
};

typedef struct md_stats_st {
    uint64_t   recvd_flows;
    uint64_t   dns;
    uint64_t   recvd_filtered;
    uint64_t   recvd_stats;
    uint64_t   nonstd_flows;
    uint64_t   uniflows;
    uint32_t   files;
    uint16_t   restarts;
} md_stats_t;

typedef struct md_collect_node_st md_collect_node_t;
struct md_collect_node_st {
    md_collect_node_t  *next;
    mdFlowCollector_t  *coll;
    md_filter_t        *filter;
    fBuf_t             *fbuf;
    md_stats_t         *stats;
    pthread_cond_t      cond;
    pthread_mutex_t     mutex;
    gboolean            and_filter;
    gboolean            active;
};

typedef struct mdBuf_st {
    char    *cp;
    char    *buf;
    size_t   buflen;
} mdBuf_t;

struct mdConfig_st {
    md_collect_node_t   *flowsrc;
    md_export_node_t    *flowexit;
    smFieldMap_t        *maps;
    FILE                *log;
    md_spread_filter_t  *mdspread;
    char                *collector_name;
    pthread_cond_t       log_cond;
    pthread_mutex_t      log_mutex;
    gboolean             no_stats;
    gboolean             ipfixSpreadTrans;
    gboolean             lockmode;
    gboolean             dns_base64_encode;
    gboolean             dns_print_lastseen;
    gboolean             shared_filter;
    gboolean             preserve_obdomain;
    gboolean             gen_tombstone;
    gboolean             rewrite_ssl_certs;
    uint16_t             tombstone_configured_id;
    uint64_t             udp_template_timeout;
    uint64_t             ctime;
    uint32_t             current_domain;
    unsigned int         usec_sleep;
    uint8_t              num_listeners;
    uint8_t              collector_id;
#ifdef HAVE_SPREAD
    fbSpreadParams_t     out_spread;
#endif
#if    ENABLE_SKTYPESENSOR
    /* The CURRENT collector's setting for the name of SILK_PROBE */
    const char          *collector_silk_probe_name;
    /* The CURRENT collector's setting for the vlan setting of SILK_PROBE */
    gboolean             collector_silk_probe_vlan;
#endif  /* ENABLE_SKTYPESENSOR */
};

#ifdef MEDIATOR_MAIN_SOURCE
static mdConfig_t  MD_CONFIG_INIT = {
    NULL,                       /* flowsrc */
    NULL,                       /* flowexit */
    NULL,                       /* maps */
    NULL,                       /* log */
    NULL,                       /* mdspread */
    NULL,                       /* collector_name */
    PTHREAD_COND_INITIALIZER,   /* log_cond */
    PTHREAD_MUTEX_INITIALIZER,  /* log_mutex */
    FALSE,                      /* no_stats */
    FALSE,                      /* ipfixSpreadTrans */
    FALSE,                      /* lockmode */
    FALSE,                      /* dns_base64_encode */
    FALSE,                      /* dns_print_lastseen */
    FALSE,                      /* shared_filter */
    FALSE,                      /* preserve_obdomain */
    FALSE,                      /* gen_tombstone */
    FALSE,                      /* rewrite_ssl_certs */
    0,                          /* tombstone_configured_id */
    600,                        /* udp_template_timeout */
    0,                          /* ctime */
    0,                          /* current_domain */
    0,                          /* usec_sleep */
    0,                          /* num_listeners */
    0                           /* collector_id */
#ifdef HAVE_SPREAD
    , FB_SPREADPARAMS_INIT       /* out_spread */
#endif
#if    ENABLE_SKTYPESENSOR
    , NULL                       /* collector_silk_probe_name */
    , FALSE                      /* collector_silk_probe_vlan */
#endif  /* ENABLE_SKTYPESENSOR */
};
#endif  /* MEDIATOR_MAIN_SOURCE */


typedef struct mdContext_st {
    mdConfig_t  *cfg;
    md_stats_t  *stats;
    GError      *err;
} mdContext_t;

#define MD_CTX_INIT { NULL, NULL, NULL }


/* Function to parse the config file, defined in mediator_config_lex.l */
gboolean
mediator_config_load(
    const char  *config_file);


#endif  /* MD_CTX */
