/*
 *  Copyright (C) 2012-2024 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_log.c
 *
 *  glib-based logging support for super_mediator (taken from libairfame)
 *
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  Super Mediator 1.10.0
 *
 *  Copyright 2023 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  GOVERNMENT PURPOSE RIGHTS - Software and Software Documentation
 *  Contract No.: FA8702-15-D-0002
 *  Contractor Name: Carnegie Mellon University
 *  Contractor Address: 4500 Fifth Avenue, Pittsburgh, PA 15213
 *  The Government's rights to use, modify, reproduce, release, perform,
 *  display, or disclose this software are restricted by paragraph (b)(2) of
 *  the Rights in Noncommercial Computer Software and Noncommercial Computer
 *  Software Documentation clause contained in the above identified
 *  contract. No restrictions apply after the expiration date shown
 *  above. Any reproduction of the software or portions thereof marked with
 *  this legend must also reproduce the markings.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM23-2316
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_log.h"
#include "mediator_util.h"

static gint
md_logc_syslog_level(
    GLogLevelFlags   level)
{
    if (level & G_LOG_LEVEL_DEBUG) {return LOG_DEBUG;}
    if (level & G_LOG_LEVEL_INFO) {return LOG_INFO;}
    if (level & G_LOG_LEVEL_MESSAGE) {return LOG_NOTICE;}
    if (level & G_LOG_LEVEL_WARNING) {return LOG_WARNING;}
    if (level & G_LOG_LEVEL_CRITICAL) {return LOG_ERR;}
    if (level & G_LOG_LEVEL_ERROR) {return LOG_ERR;}

    return LOG_DEBUG;
}

GLogLevelFlags
md_parse_log_level(
    mdLogLevel_t   log_level,
    gboolean       debug,
    gboolean       quiet)
{
    GLogLevelFlags  level;

    if (log_level == ERROR) {
        level = (G_LOG_FLAG_RECURSION |
                 G_LOG_FLAG_FATAL |
                 G_LOG_LEVEL_ERROR);
    } else if (log_level == WARNING) {
        level = (G_LOG_FLAG_RECURSION |
                 G_LOG_FLAG_FATAL |
                 G_LOG_LEVEL_ERROR |
                 G_LOG_LEVEL_CRITICAL |
                 G_LOG_LEVEL_WARNING);
    } else if (log_level == MD_DEBUG) {
        level = (G_LOG_FLAG_RECURSION |
                 G_LOG_FLAG_FATAL |
                 G_LOG_LEVEL_ERROR |
                 G_LOG_LEVEL_CRITICAL |
                 G_LOG_LEVEL_WARNING |
                 G_LOG_LEVEL_MESSAGE |
                 G_LOG_LEVEL_INFO |
                 G_LOG_LEVEL_DEBUG);
    } else if (log_level == QUIET) {
        level = 0;
    } else {
        level = (G_LOG_FLAG_RECURSION |
                 G_LOG_FLAG_FATAL |
                 G_LOG_LEVEL_ERROR |
                 G_LOG_LEVEL_CRITICAL |
                 G_LOG_LEVEL_WARNING |
                 G_LOG_LEVEL_MESSAGE |
                 G_LOG_LEVEL_INFO);
    }

    if (debug) {
        level = (G_LOG_FLAG_RECURSION |
                 G_LOG_FLAG_FATAL |
                 G_LOG_LEVEL_ERROR |
                 G_LOG_LEVEL_CRITICAL |
                 G_LOG_LEVEL_WARNING |
                 G_LOG_LEVEL_MESSAGE |
                 G_LOG_LEVEL_INFO |
                 G_LOG_LEVEL_DEBUG);
    }

    if (quiet) {
        level = 0;
    }

    return level;
}


static gboolean
md_parse_syslog_facility(
    const char  *facstr,
    gint        *facility)
{
#ifdef LOG_AUTH
    if (strcmp("auth", facstr) == 0) {
        *facility = LOG_AUTH;
        return TRUE;
    }
#endif /* ifdef LOG_AUTH */

#ifdef LOG_AUTHPRIV
    if (strcmp("authpriv", facstr) == 0) {
        *facility = LOG_AUTHPRIV;
        return TRUE;
    }
#endif /* ifdef LOG_AUTHPRIV */

#ifdef LOG_CONSOLE
    if (strcmp("console", facstr) == 0) {
        *facility = LOG_CONSOLE;
        return TRUE;
    }
#endif /* ifdef LOG_CONSOLE */

#ifdef LOG_CRON
    if (strcmp("cron", facstr) == 0) {
        *facility = LOG_CRON;
        return TRUE;
    }
#endif /* ifdef LOG_CRON */

#ifdef LOG_DAEMON
    if (strcmp("daemon", facstr) == 0) {
        *facility = LOG_DAEMON;
        return TRUE;
    }
#endif /* ifdef LOG_DAEMON */

#ifdef LOG_FTP
    if (strcmp("ftp", facstr) == 0) {
        *facility = LOG_FTP;
        return TRUE;
    }
#endif /* ifdef LOG_FTP */

#ifdef LOG_LPR
    if (strcmp("lpr", facstr) == 0) {
        *facility = LOG_LPR;
        return TRUE;
    }
#endif /* ifdef LOG_LPR */

#ifdef LOG_MAIL
    if (strcmp("mail", facstr) == 0) {
        *facility = LOG_MAIL;
        return TRUE;
    }
#endif /* ifdef LOG_MAIL */

#ifdef LOG_NEWS
    if (strcmp("news", facstr) == 0) {
        *facility = LOG_NEWS;
        return TRUE;
    }
#endif /* ifdef LOG_NEWS */

#ifdef LOG_SECURITY
    if (strcmp("security", facstr) == 0) {
        *facility = LOG_SECURITY;
        return TRUE;
    }
#endif /* ifdef LOG_SECURITY */

#ifdef LOG_USER
    if (strcmp("user", facstr) == 0) {
        *facility = LOG_USER;
        return TRUE;
    }
#endif /* ifdef LOG_USER */

#ifdef LOG_UUCP
    if (strcmp("uucp", facstr) == 0) {
        *facility = LOG_UUCP;
        return TRUE;
    }
#endif /* ifdef LOG_UUCP */

#ifdef LOG_LOCAL0
    if (strcmp("local0", facstr) == 0) {
        *facility = LOG_LOCAL0;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL0 */

#ifdef LOG_LOCAL1
    if (strcmp("local1", facstr) == 0) {
        *facility = LOG_LOCAL1;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL1 */

#ifdef LOG_LOCAL2
    if (strcmp("local2", facstr) == 0) {
        *facility = LOG_LOCAL2;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL2 */

#ifdef LOG_LOCAL3
    if (strcmp("local3", facstr) == 0) {
        *facility = LOG_LOCAL3;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL3 */

#ifdef LOG_LOCAL4
    if (strcmp("local4", facstr) == 0) {
        *facility = LOG_LOCAL4;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL4 */

#ifdef LOG_LOCAL5
    if (strcmp("local5", facstr) == 0) {
        *facility = LOG_LOCAL5;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL5 */

#ifdef LOG_LOCAL6
    if (strcmp("local6", facstr) == 0) {
        *facility = LOG_LOCAL6;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL6 */

#ifdef LOG_LOCAL7
    if (strcmp("local7", facstr) == 0) {
        *facility = LOG_LOCAL7;
        return TRUE;
    }
#endif /* ifdef LOG_LOCAL7 */

    return FALSE;
}

static void
md_logger_syslog(
    const char      *domain,
    GLogLevelFlags   log_level,
    const char      *message,
    gpointer         user_data)
{
    MD_UNUSED_PARAM(domain);
    MD_UNUSED_PARAM(user_data);

    syslog(md_logc_syslog_level(log_level), "%s", message);
}

static void
md_logger_null(
    const char      *domain,
    GLogLevelFlags   log_level,
    const char      *message,
    gpointer         user_data)
{
    MD_UNUSED_PARAM(domain);
    MD_UNUSED_PARAM(log_level);
    MD_UNUSED_PARAM(message);
    MD_UNUSED_PARAM(user_data);

    return;
}

GIOChannel *
md_log_setup(
    char            *md_log_file,
    GLogLevelFlags   levels,
    int             *ret,
    gboolean         daemon_mode,
    GError         **err)
{
    GIOChannel *logfile = NULL;
    int         facility;

    if (!md_log_file || (strcmp(md_log_file, "stderr") == 0)) {
        if (daemon_mode) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "Can't log to stderr as daemon.");
            return NULL;
        }

        /* set log file to stderr */
        logfile = g_io_channel_unix_new(fileno(stderr));
    } else if (md_parse_syslog_facility(md_log_file, &facility)) {
        /* open log socket */
        openlog("super_mediator", LOG_CONS | LOG_PID, facility);

        /* use syslog logger */
        g_log_set_handler(G_LOG_DOMAIN, levels, md_logger_syslog, NULL);

        *ret = 1;
    } else {
        /* open log file */
        logfile = g_io_channel_new_file(md_log_file, "a", err);
        if (logfile == NULL) {
            fprintf(stderr, "Can't open log file '%s' or syslog(3) facility "
                    "not recognized: %s\n", md_log_file, (*err)->message);
            return NULL;
        }
    }

    /* set default log handler to eat messages */
    g_log_set_default_handler(md_logger_null, NULL);

    return logfile;
}

void
md_log_compress(
    char  *file)
{
    if (file == NULL) {
        g_warning("md_log_compress passed NULL pointer");
        return;
    }

    md_util_compress_file(file, NULL);
}
