/*
 *  Copyright (C) 2012-2024 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_util.h
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  Super Mediator 1.10.0
 *
 *  Copyright 2023 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  GOVERNMENT PURPOSE RIGHTS - Software and Software Documentation
 *  Contract No.: FA8702-15-D-0002
 *  Contractor Name: Carnegie Mellon University
 *  Contractor Address: 4500 Fifth Avenue, Pittsburgh, PA 15213
 *  The Government's rights to use, modify, reproduce, release, perform,
 *  display, or disclose this software are restricted by paragraph (b)(2) of
 *  the Rights in Noncommercial Computer Software and Noncommercial Computer
 *  Software Documentation clause contained in the above identified
 *  contract. No restrictions apply after the expiration date shown
 *  above. Any reproduction of the software or portions thereof marked with
 *  this legend must also reproduce the markings.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM23-2316
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#ifndef MD_UTIL
#define MD_UTIL

#include "mediator_templates.h"

#if ENABLE_SKIPSET
#include SKIPSET_HEADER_NAME
#endif
#if ENABLE_SKPREFIXMAP
#include <silk/skprefixmap.h>
#endif

typedef struct smVarHashKey_st {
    size_t    len;
    uint8_t  *val;
} smVarHashKey_t;

typedef struct md_asn_tlv_st {
    uint8_t   class : 2;
    uint8_t   p_c   : 1;
    uint8_t   tag   : 5;
} md_asn_tlv_t;

#if ENABLE_SKIPSET
/* A wrapper over the SiLK IPSet structure to avoid opening the same file
 * multiple times. */
typedef struct mdIPSet_st {
    skipset_t     *ipset;
    gchar         *path;
    /* pthread_mutex_t     mutex; */
    unsigned int   ref;
} mdIPSet_t;
#endif  /* ENABLE_SKIPSET */

#if ENABLE_SKPREFIXMAP
/* A wrapper over the SiLK PMAP structure to avoid opening the same file
 * multiple times. */
typedef struct mdPMap_st {
    skPrefixMap_t         *pmap;
    /* path to the file, as given by the user */
    gchar                 *path;
    /* name of the map if it contains one; else the basename of 'path'; value
     * references other memory and it is not allocated */
    const gchar           *mapname;
    /* pthread_mutex_t         mutex; */
    unsigned int           ref;
    skPrefixMapContent_t   content;
} mdPMap_t;
#endif  /* ENABLE_SKPREFIXMAP */

int
md_util_hexdump_append(
    char           *dest,
    size_t         *drem,
    const uint8_t  *src,
    size_t          len);

int
md_util_hexdump_append_nospace(
    char           *dest,
    size_t         *drem,
    const uint8_t  *src,
    size_t          len);

uint32_t
md_util_hexdump_g_string_append_line(
    GString        *str,
    const char     *lpfx,
    const uint8_t  *cp,
    uint32_t        lineoff,
    uint32_t        buflen);

void
md_util_hexdump_g_string_append(
    GString        *str,
    const char     *lpfx,
    const uint8_t  *buf,
    uint32_t        len);

void
md_util_print_tcp_flags(
    GString  *str,
    uint8_t   flags);

void
md_util_print_ip6_addr(
    char           *ipaddr_buf,
    const uint8_t  *ipaddr);

void
md_util_print_ip4_addr(
    char      *ipaddr_buf,
    uint32_t   ip);

uint32_t
md_util_flow_key_hash(
    md_main_template_t  *rec);

uint32_t
md_util_rev_flow_key_hash(
    md_main_template_t  *rec);

void
md_util_time_g_string_append(
    GString     *str,
    time_t       c_time,
    const char  *format);

gboolean
md_util_time_buf_append(
    mdBuf_t     *buf,
    size_t      *brem,
    time_t       c_time,
    const char  *format);

uint16_t
md_util_decode_asn1_length(
    uint8_t **buffer,
    size_t   *len);

uint16_t
md_util_decode_asn1_sequence(
    uint8_t **buffer,
    size_t   *len);

void *
detachFromEndOfDLL(
    mdDLL_t **head,
    mdDLL_t **tail);

void
detachThisEntryOfDLL(
    mdDLL_t **head,
    mdDLL_t **tail,
    mdDLL_t  *entry);

void
attachHeadToDLL(
    mdDLL_t **head,
    mdDLL_t **tail,
    mdDLL_t  *newEntry);

guint
sm_octet_array_hash(
    gconstpointer   v);

gboolean
sm_octet_array_equal(
    gconstpointer   v1,
    gconstpointer   v2);

void
sm_octet_array_key_destroy(
    gpointer   data);

smVarHashKey_t *
sm_new_hash_key(
    uint8_t  *val,
    size_t    len);

size_t
md_util_write_buffer(
    FILE        *fp,
    mdBuf_t     *buf,
    const char  *exp_name,
    GError     **err);

gboolean
md_util_append_buffer(
    mdBuf_t        *buf,
    size_t         *brem,
    const uint8_t  *var,
    size_t          len);

gboolean
md_util_append_gstr(
    mdBuf_t        *buf,
    size_t         *brem,
    const GString  *str);

gboolean
md_util_append_varfield(
    mdBuf_t             *buf,
    size_t              *brem,
    const fbVarfield_t  *var);


uint16_t
md_util_decode_length(
    uint8_t   *buffer,
    uint16_t  *offset);

uint16_t
md_util_decode_tlv(
    md_asn_tlv_t  *tlv,
    uint8_t       *buffer,
    uint16_t      *offset);

uint8_t
md_util_asn1_sequence_count(
    uint8_t   *buffer,
    uint16_t   seq_len);

void
md_util_compress_file(
    const char  *file,
    const char  *dest);

smHashTable_t *
smCreateHashTable(
    size_t           length,
    GDestroyNotify   freeKeyfn,
    GDestroyNotify   freeValfn);

gpointer
smHashLookup(
    smHashTable_t  *table,
    uint8_t        *key);

void
smHashTableInsert(
    smHashTable_t  *table,
    uint8_t        *key,
    uint8_t        *value);

void
smHashTableFree(
    smHashTable_t  *table);

void
smHashTableRemove(
    smHashTable_t  *table,
    uint8_t        *key);

uint32_t
smFieldMapTranslate(
    smFieldMap_t  *map,
    mdFullFlow_t  *flow);

void
md_free_hash_key(
    gpointer   v1);

GString *
sm_util_move_file(
    const char  *file,
    const char  *new_dir);

#if ENABLE_SKIPSET
mdIPSet_t *
mdUtilIPSetOpen(
    const char  *path,
    GError     **err);
void
mdUtilIPSetClose(
    mdIPSet_t  *ipset);
#endif  /* ENABLE_SKIPSET */

#if ENABLE_SKPREFIXMAP
mdPMap_t *
mdUtilPMapOpen(
    const char  *path,
    GError     **err);
void
mdUtilPMapClose(
    mdPMap_t  *pmap);
#endif  /* ENABLE_SKPREFIXMAP */

#endif  /* MD_UTIL */
