/*
 *  Copyright (C) 2012-2024 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_dns.c
 *
 *  IPFIX mediator for filtering, DNS deduplication, and other mediator-like
 *  things: contains DNS deduplication code.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.11
 *
 *  Copyright 2024 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM24-1038
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 *
 *
 *    Any DNS entry marked as a query (dnsQueryResponse==0) is skipped and is
 *    not counted.
 *
 *    Responses with an empty dnsQName are ignored but do count as being
 *    filtered.
 *
 *    DNS Types (dnsQRType) greater than 33 (MD_DNS_DEDUP_MAX_TYPE) are not
 *    handled by this code.  The user may choose to limit which Types are
 *    exported via the DNS_DEDUP block in the config file.
 *
 *    If the response code (dnsNXDomain) in a DNS entry is 3 and the DNS
 *    Section (dnsRRSection) is 0, the entry is considered an NX record and is
 *    stored in its own table.  These are exported with the Type set to 0 and
 *    the original type of the query is lost.
 *
 *    Other than when determining the NX setting, DNS Sections values are
 *    ignored by this code and all Resource Records are included regardless of
 *    section.
 *
 *    Resource Records (RRs) are stored in separate tables by Type, where they
 *    are first keyed by the query name and then by the response.  If a VLAN
 *    or Observation ID map is set, that ID is included as part of key.
 *
 */

#include "mediator_dns.h"
#include "mediator_inf.h"
#include "mediator_util.h"


/* Local macro to enable debugging */
#ifndef DNS_DEBUG
/* Defines a function that nothing calls */
#define DNS_DEBUG 0
#endif
#ifndef DNS_ADD_DEBUG
/* Prints status of md_dns_dedup_add_node(). */
#define DNS_ADD_DEBUG 0
#endif


/*
 *  The struct to hold an exported DNS Dedup record is md_dns_dedup_t (defined
 *  in mediator_templates.h) and its template is md_dns_dedup_spec
 *  (mediator_core.c).
 */

/* node is an item in a linked list of md_dns_dedup_t (records) */
typedef struct md_dns_node_st md_dns_node_t;
struct md_dns_node_st {
    md_dns_node_t  *next;
    md_dns_node_t  *prev;
    md_dns_dedup_t  dns_node;
};

/* Similar to md_dns_node_t except without the dnsName and
 * observationDomainName which are stored in the md_hashtab_node_t. */
typedef struct md_cache_node_st md_cache_node_t;
struct md_cache_node_st {
    md_cache_node_t  *next;
    md_cache_node_t  *prev;
    yfTime_t          ftime;
    yfTime_t          ltime;
    uint32_t          ttl;
    uint16_t          rrtype;
    uint16_t          hitcount;
    size_t            caplen;
    uint32_t          ip;
    uint8_t           ipv6[16];
    uint8_t          *rrdata;
};

/* Groups all records that have the same dnsName and observation id. */
typedef struct md_hashtab_node_st md_hashtab_node_t;
struct md_hashtab_node_st {
    md_hashtab_node_t  *next;
    md_hashtab_node_t  *prev;
    md_cache_node_t    *head;
    md_cache_node_t    *tail;
    smVarHashKey_t     *rkey;
    int                 mapindex;
    size_t              rrname_len;
    uint8_t            *rrname;
};

/* Groups all records that have the name DNS Type */
/* typedef struct md_type_hashtab_st md_type_hashtab_t; */
struct md_type_hashtab_st {
    smHashTable_t      *table;
    yfTime_t            last_flush;
    uint32_t            count;
    md_hashtab_node_t  *head;
    md_hashtab_node_t  *tail;
};

/* dns close queue */
typedef struct md_dns_cqueue_st {
    md_dns_node_t  *head;
    md_dns_node_t  *tail;
} md_dns_cqueue_t;

typedef struct md_dns_dedup_stats_st {
    uint64_t   dns_recvd;
    uint64_t   dns_filtered;
    uint64_t   dns_flushed;
} md_dns_dedup_stats_t;


/*  Defined in mediator_ctx.h
 *
 * typedef struct md_dns_dedup_state_st md_dns_dedup_state_t; */
struct md_dns_dedup_state_st {
    md_dns_dedup_stats_t   stats;
    md_type_hashtab_t     *a_table;
    md_type_hashtab_t     *ns_table;
    md_type_hashtab_t     *cname_table;
    md_type_hashtab_t     *soa_table;
    md_type_hashtab_t     *ptr_table;
    md_type_hashtab_t     *mx_table;
    md_type_hashtab_t     *txt_table;
    md_type_hashtab_t     *aaaa_table;
    md_type_hashtab_t     *srv_table;
    md_type_hashtab_t     *nx_table;
    md_dns_cqueue_t       *cq;
    smFieldMap_t          *map;
    int                   *dedup_type_list;
    yfDiffTime_t           dedup_flush_to;
    uint32_t               dedup_max_hit_count;
    gboolean               print_lastseen;
    gboolean               export_name;
};


static void
md_dns_dedup_emit_record(
    md_dns_dedup_state_t  *state,
    md_dns_cqueue_t       *cq,
    md_hashtab_node_t     *hn,
    md_cache_node_t       *cn);

static void
md_dns_dedup_flush_tab(
    md_type_hashtab_t     *nodeTab,
    md_dns_dedup_state_t  *state,
    const yfTime_t        *cur_time,
    gboolean               flush_all);



/**
 * allocTypeTab
 *
 *
 */
static md_type_hashtab_t *
allocTypeTab(
    const yfTime_t  *cur_time)
{
    md_type_hashtab_t *md_type_tab;

    md_type_tab = g_slice_new0(md_type_hashtab_t);
    /* md_type_tab->table = g_hash_table_new((GHashFunc)g_str_hash, */
    /*                                       (GEqualFunc)g_str_equal); */
    md_type_tab->table = smCreateHashTable(0xFF,
                                           sm_octet_array_key_destroy, NULL);
    if (md_type_tab->table == NULL) {
        return NULL;
    }

    md_type_tab->last_flush = *cur_time;

    return md_type_tab;
}

/**
 * md_dns_dedup_print_stats
 *
 * Prints stats to the log.
 *
 *
 */
void
md_dns_dedup_print_stats(
    md_dns_dedup_state_t  *state,
    char                  *exp_name)
{
    if (state->stats.dns_recvd == 0) {
        return;
    }

    g_message("Exporter %s: %" PRIu64 " DNS records, %" PRIu64 " filtered"
              ", %" PRIu64 " flushed (%2.2f%% compression)",
              exp_name, state->stats.dns_recvd, state->stats.dns_filtered,
              state->stats.dns_flushed,
              (100.0 * (1.0 - (((double)state->stats.dns_flushed) /
                               ((double)state->stats.dns_recvd)))));
}

/**
 * md_dns_dedup_reset
 *
 * Flushes all Hash Tables.
 *
 */
void
md_dns_dedup_reset(
    md_dns_dedup_state_t  *state,
    const yfTime_t        *cur_time)
{
    g_warning("Out of Memory Error.  Resetting all Hash Tables");
    md_dns_dedup_flush_all_tab(state, cur_time, TRUE);
}

static void
md_dns_dedup_attempt_flush_tab(
    md_type_hashtab_t     *md_type_tab,
    md_dns_dedup_state_t  *state,
    const yfTime_t        *cur_time)
{
    if (md_type_tab
        && yfTimeCheckElapsed(*cur_time, md_type_tab->last_flush,
                              state->dedup_flush_to))
    {
        md_dns_dedup_flush_tab(md_type_tab, state, cur_time, FALSE);
    }
}

static void
md_dns_dedup_attempt_all_flush(
    md_dns_dedup_state_t  *state,
    const yfTime_t        *cur_time)
{
    md_dns_dedup_attempt_flush_tab(state->a_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->ns_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->cname_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->soa_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->ptr_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->mx_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->txt_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->aaaa_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->srv_table, state, cur_time);
    md_dns_dedup_attempt_flush_tab(state->nx_table, state, cur_time);
}



/**
 * md_dns_dedup_destroy_tab
 *
 * destroys all hash tables
 *
 */
static void
md_dns_dedup_destroy_tab(
    md_dns_dedup_state_t  *state)
{
    if (state->a_table && state->a_table->table) {
        smHashTableFree(state->a_table->table);
    }
    if (state->ns_table && state->ns_table->table) {
        smHashTableFree(state->ns_table->table);
    }
    if (state->cname_table && state->cname_table->table) {
        smHashTableFree(state->cname_table->table);
    }
    if (state->soa_table && state->soa_table->table) {
        smHashTableFree(state->soa_table->table);
    }
    if (state->ptr_table && state->ptr_table->table) {
        smHashTableFree(state->ptr_table->table);
    }
    if (state->mx_table && state->mx_table->table) {
        smHashTableFree(state->mx_table->table);
    }
    if (state->txt_table && state->txt_table->table) {
        smHashTableFree(state->txt_table->table);
    }
    if (state->aaaa_table && state->aaaa_table->table) {
        smHashTableFree(state->aaaa_table->table);
    }
    if (state->nx_table && state->nx_table->table) {
        smHashTableFree(state->nx_table->table);
    }
    if (state->srv_table && state->srv_table->table) {
        smHashTableFree(state->srv_table->table);
    }
}

gboolean
md_dns_dedup_free_state(
    mdConfig_t        *cfg,
    md_export_node_t  *exp,
    GError           **err)
{
    md_dns_dedup_state_t *state = exp->dns_dedup;

    md_dns_dedup_flush_all_tab(state, &cfg->ctime, TRUE);

    if (!md_dns_dedup_flush_queue(exp, cfg, err)) {
        return FALSE;
    }

    md_dns_dedup_destroy_tab(state);
    if (state->dedup_type_list) {
        g_free(state->dedup_type_list);
    }

    g_slice_free1(sizeof(md_dns_cqueue_t), state->cq);
    return TRUE;
}

/**
 * md_debug_table
 *
 *
 */
#if DNS_DEBUG == 1
static void
md_debug_table(
    md_type_hashtab_t  *nodeTab)
{
    md_cache_node_t   *cq;
    md_hashtab_node_t *hn;

    for (hn = nodeTab->head; hn; hn = hn->next) {
        for (cq = hn->head; cq; cq = cq->next) {
            g_debug("%d %p rrname %s", cq->rrtype, cq,
                    hn->rrname);
            g_debug("cq->next is %p", cq->next);
        }
    }
}
#endif /* if DNS_DEBUG == 1 */

/**
 * md_new_dns_queue
 *
 * creates a new close queue for dns-dedup
 */
static md_dns_cqueue_t *
md_dns_dedup_new_queue(
    void)
{
    md_dns_cqueue_t *cq = g_slice_new0(md_dns_cqueue_t);

    cq->head = NULL;
    cq->tail = NULL;

    return cq;
}

md_dns_dedup_state_t *
md_dns_dedup_new_state(
    void)
{
    md_dns_dedup_state_t *state = g_slice_new0(md_dns_dedup_state_t);

    state->cq = md_dns_dedup_new_queue();

    /* set defaults */
    state->dedup_max_hit_count = MD_DNS_DEDUP_DEFAULT_HIT;
    yfDiffTimeFromSeconds(&state->dedup_flush_to, MD_DNS_DEDUP_DEFAULT_FLUSH);
    state->print_lastseen = FALSE;

    return state;
}

void
md_dns_dedup_configure_state(
    md_dns_dedup_state_t  *state,
    int                   *dedup_list,
    int                    max_hit,
    int                    flush_timeout,
    gboolean               lastseen,
    smFieldMap_t          *map,
    gboolean               export_name)
{
    if (!state) {
        return;
    }

    state->dedup_type_list = dedup_list;
    state->print_lastseen = lastseen;
    state->export_name = export_name;
    if (max_hit > 0) {
        state->dedup_max_hit_count = max_hit;
    }
    if (flush_timeout > 0) {
        yfDiffTimeFromSeconds(&state->dedup_flush_to, flush_timeout);
    }
    if (map) {
        state->map = map;
    }
}

/**
 * md_dns_dedup_flush_queue
 *
 * Flushes all records in the close queue.
 *
 */
gboolean
md_dns_dedup_flush_queue(
    md_export_node_t  *exp,
    mdConfig_t        *cfg,
    GError           **err)
{
    md_dns_node_t        *node;
    md_dns_dedup_state_t *state = exp->dns_dedup;
    md_dns_cqueue_t      *cq = exp->dns_dedup->cq;
    uint16_t              tid = MD_DNS_DEDUP_OUT;
    uint16_t              wtid;

    if (cq == NULL) {
        return TRUE;
    }

    if (state->print_lastseen) {
        tid |= MD_LAST_SEEN;
    }

    wtid = tid;

    while ((node = detachFromEndOfDLL((mdDLL_t **)&(cq->head),
                                      (mdDLL_t **)&(cq->tail))))
    {
        wtid = tid;
        if (node->dns_node.rrtype == 1) {
            wtid |= MD_DNS_AREC;
        } else if (node->dns_node.rrtype == 28) {
            wtid |= MD_DNS_AAAAREC;
        } else {
            wtid |= MD_DNS_OREC;
        }

        if (state->export_name && (node->dns_node.mapname.len == 0)) {
            node->dns_node.mapname.buf = (uint8_t *)mdExporterGetName(exp->exp);
            node->dns_node.mapname.len = strlen(mdExporterGetName(exp->exp));
        }

        if (!mdExporterWriteDnsDedup(cfg, exp->exp, wtid,
                                     (uint8_t *)&(node->dns_node),
                                     sizeof(md_dns_dedup_t), err))
        {
            return FALSE;
        }

        state->stats.dns_flushed++;
        g_slice_free1(node->dns_node.rrdata.len, node->dns_node.rrdata.buf);
        g_slice_free1(node->dns_node.rrname.len, node->dns_node.rrname.buf);
        g_slice_free(md_dns_node_t, node);
    }

    /* free the node we just sent out */

    return TRUE;
}



/**
 * nodeClose
 *
 * closes the HASHnode, this means that there is no more
 * cache nodes that belong to this "hash node."  Basically
 * this means that we flushed all information associated
 * with this query name.
 *
 * @param struct that contains node hash table
 * @param pointer to the node entry that we want to close
 *
 */
static void
nodeClose(
    md_type_hashtab_t  *nodeTab,
    md_hashtab_node_t  *hnode)
{
    /*Remove it from list*/

    /*    g_hash_table_remove(nodeTab->table, hnode->rrname);*/
    smHashTableRemove(nodeTab->table, (uint8_t *)hnode->rkey);

    detachThisEntryOfDLL((mdDLL_t **)&(nodeTab->head),
                         (mdDLL_t **)&(nodeTab->tail), (mdDLL_t *)hnode);

    /* free the rrname */

    /*     g_slice_free1(hnode->rrname_len, hnode->rrname); */
    g_slice_free(md_hashtab_node_t, hnode);

    --(nodeTab->count);
}

/**
 * newCacheNode
 *
 * creates a new cache node which will go into
 * a linked list by hash node.  Basically this
 * has the same query name, but a different type
 * or rrdata
 */
static md_cache_node_t *
newCacheNode(
    const yfTime_t         *start_time,
    const md_cache_node_t  *find)
{
    md_cache_node_t *cn;

    cn = g_slice_new0(md_cache_node_t);
    cn->hitcount = 1;
    cn->ftime = *start_time;
    cn->ltime = *start_time;
    cn->ip = find->ip;
    cn->rrtype = find->rrtype;
    if (cn->rrtype == 28) {
        memcpy(cn->ipv6, find->ipv6, sizeof(cn->ipv6));
    } else if (find->caplen) {
        cn->rrdata = g_slice_alloc0(find->caplen);
        memcpy(cn->rrdata, find->rrdata, find->caplen);
        cn->caplen = find->caplen;
    }

    return cn;
}

/**
 * hashTick
 *
 * advances a node to the head of the
 * queue - bottom of queue gets examined
 * for flush timeouts
 *
 * @param pointer to table
 * @param pointer to node
 *
 */
static void
hashTick(
    md_type_hashtab_t  *nodeTab,
    md_hashtab_node_t  *entry)
{
    if (nodeTab->head != entry) {
        if (entry->prev != NULL) {
            detachThisEntryOfDLL((mdDLL_t **)&(nodeTab->head),
                                 (mdDLL_t **)&(nodeTab->tail),
                                 (mdDLL_t *)entry);
        }
        attachHeadToDLL((mdDLL_t **)&(nodeTab->head),
                        (mdDLL_t **)&(nodeTab->tail),
                        (mdDLL_t *)entry);
    }

    /*    md_debug_table(nodeTab);*/
}

/**
 * cacheNodeClose
 *
 * creates a new md_dns_node_t for output,
 * attaches it to the close queue, and frees the
 * cache node associated with the domain name.
 *
 *
 * @param hashNode
 * @param CacheNode to close
 * @param filepointers
 */
static void
cacheNodeClose(
    md_type_hashtab_t     *nodeTab,
    md_hashtab_node_t     *hn,
    md_cache_node_t       *cn,
    md_dns_dedup_state_t  *state)
{
    md_dns_cqueue_t *cq = state->cq;

    if (state->print_lastseen) {
        md_dns_dedup_emit_record(state, cq, hn, cn);
    }

    detachThisEntryOfDLL((mdDLL_t **)&(hn->head),
                         (mdDLL_t **)&(hn->tail),
                         (mdDLL_t *)cn);

    if (cn->rrtype != 28) {
        g_slice_free1(cn->caplen, cn->rrdata);
    }
    g_slice_free(md_cache_node_t, cn);

    if (!hn->head) {
        /*last cacheNode in hashTabNode - remove from hashtable*/
        nodeClose(nodeTab, hn);
    }
}

/**
 * md_dns_dedup_emit_record
 *
 * Adds the record to the close queue without removing
 * the node.
 *
 * @param cq - the close queue to add it to
 * @param cn - the node to add
 *
 */
static void
md_dns_dedup_emit_record(
    md_dns_dedup_state_t  *state,
    md_dns_cqueue_t       *cq,
    md_hashtab_node_t     *hn,
    md_cache_node_t       *cn)
{
    md_dns_node_t *node = g_slice_new0(md_dns_node_t);

    node->dns_node.flowStartMilliseconds = yfTimeToMilli(cn->ftime);
    node->dns_node.flowEndMilliseconds = yfTimeToMilli(cn->ltime);
    yfTimeToNTP(&node->dns_node.flowStartNanoseconds, cn->ftime);
    yfTimeToNTP(&node->dns_node.flowEndNanoseconds, cn->ltime);
    node->dns_node.sourceIPv4Address = cn->ip;
    node->dns_node.rrtype = cn->rrtype;
    if (cn->rrtype == 28) {
        memcpy(node->dns_node.sourceIPv6Address, cn->ipv6,
               sizeof(cn->ipv6));
    } else if (cn->caplen) {
        node->dns_node.rrdata.buf = g_slice_alloc0(cn->caplen);
        memcpy(node->dns_node.rrdata.buf, cn->rrdata, cn->caplen);
        node->dns_node.rrdata.len = cn->caplen;
    }
    node->dns_node.dnsHitCount = cn->hitcount;
    node->dns_node.dnsTTL = cn->ttl;
    if (hn->mapindex < 0) {
        node->dns_node.rrname.buf = g_slice_alloc0(hn->rkey->len);
        memcpy(node->dns_node.rrname.buf, hn->rkey->val, hn->rkey->len);
        node->dns_node.rrname.len = hn->rkey->len;
        node->dns_node.mapname.len = 0;
    } else {
        node->dns_node.rrname.buf =
            g_slice_alloc0(hn->rkey->len - sizeof(uint32_t));
        memcpy(node->dns_node.rrname.buf, hn->rkey->val + sizeof(uint32_t),
               hn->rkey->len - sizeof(uint32_t));
        node->dns_node.rrname.len = hn->rkey->len - sizeof(uint32_t);
        node->dns_node.mapname.buf =
            (uint8_t *)(state->map->labels[hn->mapindex]);
        node->dns_node.mapname.len = strlen(state->map->labels[hn->mapindex]);
    }

    /* node->dns_node.rrname.buf = g_slice_alloc0(hn->rrname_len); */
    /* memcpy(node->dns_node.rrname.buf, hn->rrname, hn->rrname_len); */
    /* node->dns_node.rrname.len = hn->rrname_len; */

    attachHeadToDLL((mdDLL_t **)&(cq->head),
                    (mdDLL_t **)&(cq->tail),
                    (mdDLL_t *)node);
}


/**
 * hashCacheTick
 *
 * advances a node to the head of the cache queue
 * bottom gets examined for flush timeouts
 *
 * @param pointer to head of table
 * @param pointer to node
 *
 */
static void
hashCacheTick(
    md_dns_dedup_state_t  *state,
    md_type_hashtab_t     *nodeTab,
    md_hashtab_node_t     *hn,
    md_cache_node_t       *cn)
{
    if (hn->head != cn) {
        if (cn->prev != NULL) {
            detachThisEntryOfDLL((mdDLL_t **)&(hn->head),
                                 (mdDLL_t **)&(hn->tail),
                                 (mdDLL_t *)cn);
        }
        attachHeadToDLL((mdDLL_t **)&(hn->head),
                        (mdDLL_t **)&(hn->tail),
                        (mdDLL_t *)cn);
    }

    while (hn->tail
           && yfTimeCheckElapsed(cn->ltime, hn->tail->ltime,
                                 state->dedup_flush_to))
    {
        cacheNodeClose(nodeTab, hn, hn->tail, state);
    }
}


/**
 * md_dns_dedup_flush_tab
 *
 * Checks entries in the hash table to see if they are past the
 * flush limit.  If so, it outputs to the appropriate file and deallocates
 * the memory
 *
 * @param the struct that contains the hash table and linked list
 * @param cq - the close queue.
 * @param cur_time to keep track of how often we're flushing
 * @param flush_all (if TRUE -> close all)
 *
 */
static void
md_dns_dedup_flush_tab(
    md_type_hashtab_t     *nodeTab,
    md_dns_dedup_state_t  *state,
    const yfTime_t        *cur_time,
    gboolean               flush_all)
{
    if (nodeTab == NULL) {
        return;
    }

    nodeTab->last_flush = *cur_time;

    if (flush_all) {
        while (nodeTab->tail) {
            cacheNodeClose(nodeTab, nodeTab->tail, nodeTab->tail->tail, state);
        }
        return;
    }

    while (nodeTab->tail &&
           yfTimeCheckElapsed(nodeTab->last_flush, nodeTab->tail->tail->ltime,
                              state->dedup_flush_to))
    {
        cacheNodeClose(nodeTab, nodeTab->tail, nodeTab->tail->tail, state);
    }
}


/**
 * md_dns_dedup_flush_all_tab
 *
 * Flushes all entries from all hash tables
 *
 * @param cq
 *
 */
void
md_dns_dedup_flush_all_tab(
    md_dns_dedup_state_t  *state,
    const yfTime_t        *cur_time,
    gboolean               flush_all)
{
    md_dns_dedup_flush_tab(state->a_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->ns_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->cname_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->soa_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->ptr_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->mx_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->txt_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->aaaa_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->srv_table, state, cur_time, flush_all);
    md_dns_dedup_flush_tab(state->nx_table, state, cur_time, flush_all);
}

/**
 * md_add_dns_node
 *
 * add the dns node to the appropriate hash table
 * this is the main part of deduplication.
 *
 * @param ctx
 * @param mdflow
 *
 */
void
md_dns_dedup_add_node(
    mdContext_t       *ctx,
    md_export_node_t  *exp,
    mdFullFlow_t      *flow)
{
    const uint8_t         zeroip6[16] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };
    md_dns_dedup_state_t *state = exp->dns_dedup;
    yaf_dns_t            *dnsflow = (yaf_dns_t *)flow->app;
    yaf_dnsQR_t          *dnsqrflow = NULL;
    md_cache_node_t      *cn = NULL, *tn = NULL;
    int                  *type_list = state->dedup_type_list;
    md_cache_node_t       find;
    md_hashtab_node_t    *hn = NULL;
    md_type_hashtab_t    *md_type_tab = NULL;
    uint8_t               namebuf[1024];
    uint16_t              name_offset = 0;
    size_t                namelen = 0;
    gboolean              found = FALSE;
    gboolean              nx = FALSE;
    smVarHashKey_t        key;
    uint32_t              mapkey = 0;
#if DNS_ADD_DEBUG
    GString              *str = g_string_sized_new(512);
#endif

    while ((dnsqrflow = (yaf_dnsQR_t *)FBSTLNEXT(
                &dnsflow->dnsQRList, dnsqrflow)))
    {
        if (0 == dnsqrflow->dnsQName.len) {
            if (1 == dnsqrflow->dnsQueryResponse) {
                /* count responses but not queries */
                state->stats.dns_filtered++;
#if DNS_ADD_DEBUG
                g_debug(
                    "\nDNS,,%u,%u,%u,%u,%d,filtered line %d",
                    dnsqrflow->dnsQRType, dnsqrflow->dnsQueryResponse,
                    dnsqrflow->dnsNXDomain, dnsqrflow->dnsRRSection, 0,
                    __LINE__);
#endif  /* DNS_ADD_DEBUG */
            }
            continue;
        }

        find.ip = 0;
        find.caplen = 0;
        find.rrdata = NULL;
        memset(find.ipv6, 0, sizeof(find.ipv6));
        namelen = 0;
        name_offset = 0;
        found = FALSE;
        nx = FALSE;
        find.rrtype = dnsqrflow->dnsQRType;
        find.ttl = dnsqrflow->dnsTTL;

        if (dnsqrflow->dnsNXDomain == 3 && dnsqrflow->dnsRRSection == 0) {
            find.rrtype = 0;
            nx = TRUE;
        } else if (0 == dnsqrflow->dnsQueryResponse) {
            /* don't do queries */
            continue;
        }

        if (find.rrtype > MD_DNS_DEDUP_MAX_TYPE) {
            /* not a valid DNS type for super_mediator dedup */
            state->stats.dns_filtered++;
            continue;
        }
        if (type_list && type_list[find.rrtype] == 0) {
            /* filtered out*/
            state->stats.dns_filtered++;
            continue;
        }

#if DNS_ADD_DEBUG
        /* g_string_assign() clears old content */
        g_string_assign(str, "\nDNS,");
        g_string_append_len(
            str, (char *)dnsqrflow->dnsQName.buf, dnsqrflow->dnsQName.len);
        g_string_append_printf(
            str, ",%u,%u,%u,%u,%d,",
            dnsqrflow->dnsQRType, dnsqrflow->dnsQueryResponse,
            dnsqrflow->dnsNXDomain, dnsqrflow->dnsRRSection, nx);
#endif  /* DNS_ADD_DEBUG */

        if (nx) {
            /* NXDomain */
            if (state->nx_table == NULL) {
                state->nx_table = allocTypeTab(&ctx->cfg->ctime);
            }
            md_type_tab = state->nx_table;
        } else {
            switch (dnsqrflow->dnsQRType) {
              case 1:
                {
                    yaf_dnsA_t *aflow = NULL;
                    if (state->a_table == NULL) {
                        state->a_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((aflow = (yaf_dnsA_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, aflow)))
                    {
                        md_type_tab = state->a_table;
                        find.ip = aflow->sourceIPv4Address;
                    }
                    break;
                }
              case 2:
                {
                    yaf_dnsNS_t *nsflow  = NULL;
                    if (state->ns_table == NULL) {
                        state->ns_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((nsflow = (yaf_dnsNS_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, nsflow)))
                    {
                        md_type_tab = state->ns_table;
                        find.caplen = nsflow->dnsNSDName.len;
                        find.rrdata = nsflow->dnsNSDName.buf;
                    }
                    break;
                }
              case 5:
                {
                    yaf_dnsCNAME_t *cflow = NULL;
                    if (state->cname_table == NULL) {
                        state->cname_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((cflow = (yaf_dnsCNAME_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, cflow)))
                    {
                        md_type_tab = state->cname_table;
                        find.caplen = cflow->dnsCName.len;
                        find.rrdata = cflow->dnsCName.buf;
                    }
                    break;
                }
              case 12:
                {
                    yaf_dnsPTR_t *ptrflow = NULL;
                    if (state->ptr_table == NULL) {
                        state->ptr_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((ptrflow = (yaf_dnsPTR_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, ptrflow)))
                    {
                        md_type_tab = state->ptr_table;
                        find.caplen = ptrflow->dnsPTRDName.len;
                        find.rrdata = ptrflow->dnsPTRDName.buf;
                    }
                    break;
                }
              case 15:
                {
                    yaf_dnsMX_t *mx = NULL;
                    if (state->mx_table == NULL) {
                        state->mx_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((mx = (yaf_dnsMX_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, mx)))
                    {
                        md_type_tab = state->mx_table;
                        find.caplen = mx->dnsMXExchange.len;
                        find.rrdata = mx->dnsMXExchange.buf;
                    }
                    break;
                }
              case 28:
                {
                    yaf_dnsAAAA_t *aa = NULL;
                    if (state->aaaa_table == NULL) {
                        state->aaaa_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((aa = (yaf_dnsAAAA_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, aa)))
                    {
                        md_type_tab = state->aaaa_table;
                        memcpy(find.ipv6, aa->sourceIPv6Address,
                               sizeof(find.ipv6));
                    }
                    break;
                }
              case 16:
                {
                    yaf_dnsTXT_t *txt = NULL;
                    if (state->txt_table == NULL) {
                        state->txt_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((txt = (yaf_dnsTXT_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, txt)))
                    {
                        md_type_tab = state->txt_table;
                        find.caplen = txt->dnsTXTData.len;
                        find.rrdata = txt->dnsTXTData.buf;
                    }
                    break;
                }
              case 33:
                {
                    yaf_dnsSRV_t *srv = NULL;
                    if (state->srv_table == NULL) {
                        state->srv_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((srv = (yaf_dnsSRV_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, srv)))
                    {
                        md_type_tab = state->srv_table;
                        find.rrdata = srv->dnsSRVTarget.buf;
                        find.caplen = srv->dnsSRVTarget.len;
                    }
                    break;
                }
              case 6:
                {
                    yaf_dnsSOA_t *soa = NULL;
                    if (state->soa_table == NULL) {
                        state->soa_table = allocTypeTab(&ctx->cfg->ctime);
                    }
                    while ((soa = (yaf_dnsSOA_t *)FBSTLNEXT(
                                &dnsqrflow->dnsRRList, soa)))
                    {
                        md_type_tab = state->soa_table;
                        find.rrdata = soa->dnsSOAMName.buf;
                        find.caplen = soa->dnsSOAMName.len;
                    }
                    break;
                }
              default:
                /* we don't do this one */
                state->stats.dns_filtered++;
#if DNS_ADD_DEBUG
                g_string_append_printf(str, "filtered line %d", __LINE__);
                g_debug("%s", str->str);
#endif
                continue;
            }

            /* ignore if there is no data */
            if (find.rrtype == 1) {
                if (find.ip == 0) {
                    state->stats.dns_filtered++;
#if DNS_ADD_DEBUG
                    g_string_append_printf(str, "filtered line %d", __LINE__);
                    g_debug("%s", str->str);
#endif
                    continue;
                }
#if DNS_ADD_DEBUG
                g_string_append_printf(
                    str, "%u.%u.%u.%u",
                    (find.ip >> 24), ((find.ip >> 16) & 0xff),
                    ((find.ip >> 8) & 0xff), (find.ip & 0xff));
#endif

            } else if (find.rrtype == 28) {
                if (memcmp(find.ipv6, zeroip6, sizeof(find.ipv6)) == 0) {
                    state->stats.dns_filtered++;
#if DNS_ADD_DEBUG
                    g_string_append_printf(str, "filtered line %d", __LINE__);
                    g_debug("%s", str->str);
#endif
                    continue;
                }
#if DNS_ADD_DEBUG
                g_string_append_printf(
                    str, ("%02x%02x:%02x%02x:%02x%02x:%02x%02x:"
                          "%02x%02x:%02x%02x:%02x%02x:%02x%02x"),
                    find.ipv6[0], find.ipv6[1], find.ipv6[2], find.ipv6[3],
                    find.ipv6[4], find.ipv6[5], find.ipv6[6], find.ipv6[7],
                    find.ipv6[8], find.ipv6[9], find.ipv6[10], find.ipv6[11],
                    find.ipv6[12], find.ipv6[13], find.ipv6[14], find.ipv6[15]);
#endif  /* DNS_ADD_DEBUG */

            } else {
                if (find.caplen == 0) {
                    state->stats.dns_filtered++;
#if DNS_ADD_DEBUG
                    g_string_append_printf(str, "filtered line %d", __LINE__);
                    g_debug("%s", str->str);
#endif
                    continue;
                }
#if DNS_ADD_DEBUG
                g_string_append_len(str, (char *)find.rrdata, find.caplen);
#endif
            }
        }

        /* update stats */
        state->stats.dns_recvd++;

        if (state->map) {
            mapkey = smFieldMapTranslate(state->map, flow);
            if (state->map->discard && mapkey == 0) {
                return;
            }
            memcpy(namebuf, &mapkey, sizeof(uint32_t));
            name_offset += sizeof(uint32_t);
            namelen += sizeof(uint32_t);
        }

        memcpy(namebuf + name_offset, dnsqrflow->dnsQName.buf,
               dnsqrflow->dnsQName.len);
        namelen += dnsqrflow->dnsQName.len;
        key.val = namebuf;
        key.len = namelen;

        if ((hn = smHashLookup(md_type_tab->table, (uint8_t *)&key))) {
            for (tn = hn->head; tn; tn = cn) {
                cn = tn->next;

#if DNS_ADD_DEBUG
                /* each type is a table; these should always match */
                g_assert(find.rrtype == tn->rrtype);
#endif

                /* 'continue' if there is not a match */
                if (find.rrtype == 1) {
                    if (find.ip != tn->ip) {
                        continue;
                    }
                } else if (find.rrtype == 28) {
                    if (memcmp(find.ipv6, tn->ipv6, sizeof(find.ipv6)) != 0) {
                        continue;
                    }
                } else if (find.caplen != tn->caplen ||
                           memcmp(find.rrdata, tn->rrdata, find.caplen) != 0)
                {
                    continue;
                }

                /* match */
                ++tn->hitcount;
                tn->ltime = ctx->cfg->ctime;
                if (find.ttl > tn->ttl) {
                    tn->ttl = find.ttl;
                }
                if (tn->hitcount == state->dedup_max_hit_count) {
#if DNS_ADD_DEBUG
                    g_string_append_printf(
                        str, ",max hit count %u %p", tn->hitcount, hn);
#endif
                    cacheNodeClose(md_type_tab, hn, tn, state);
                } else {
#if DNS_ADD_DEBUG
                    g_string_append_printf(
                        str, ",hash tick %u %p", tn->hitcount, hn);
#endif
                    hashCacheTick(state, md_type_tab, hn, tn);
                    hashTick(md_type_tab, hn);
                }
                found = TRUE;
                break;
            }
            if (!found) {
#if DNS_ADD_DEBUG
                g_string_append_printf(str, ",in table %p but no match", hn);
#endif
            }
        } else {
            hn = g_slice_new0(md_hashtab_node_t);
#if DNS_ADD_DEBUG
            g_string_append(str, ",new hash node");
#endif

            /* copy key over */
            hn->rkey = sm_new_hash_key(key.val, key.len);
            if (state->map) {
                hn->mapindex = mapkey;
            } else {
                hn->mapindex = -1;
            }

            /* Insert into hash table */
            smHashTableInsert(md_type_tab->table, (uint8_t *)hn->rkey,
                              (uint8_t *)hn);
            ++md_type_tab->count;
        }

        if (!found) {
            cn = newCacheNode(&ctx->cfg->ctime, &find);
            cn->ttl = find.ttl;
            if (!state->print_lastseen) {
                md_dns_dedup_emit_record(state, state->cq, hn, cn);
            }
#if DNS_ADD_DEBUG
            g_string_append_printf(str, " new node %u %p", cn->hitcount, hn);
#endif
            hashCacheTick(state, md_type_tab, hn, cn);
            hashTick(md_type_tab, hn);
        }

#if DNS_ADD_DEBUG
        g_debug("%s", str->str);
#endif

    }

#if DNS_ADD_DEBUG
    g_string_free(str, TRUE);
#endif

    /* attempt a flush on all tables */
    md_dns_dedup_attempt_all_flush(state, &ctx->cfg->ctime);
}
