/*
 *  Copyright (C) 2004-2024 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  probeconf.c
 *
 *  Helper file for sensor.conf parsing. Taken from SiLK.
 *
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.11
 *
 *  Copyright 2024 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM24-1038
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */


#include "probeconf.h"
#include <silk/skipaddr.h>
#include <silk/skipset.h>
#include <silk/sksite.h>
#include "probeconfscan.h"


static gboolean
mdSkpcFlowGetSIP(
    const mdFullFlow_t  *flow,
    skipaddr_t          *ipaddr)
{
    if (!flow->tmpl_attr.is_ipv6) {
        skipaddrSetV4(ipaddr, &flow->rec->sourceIPv4Address);
    } else {
#if HAVE_SKIPADDRSETV6
        skipaddrSetV6(ipaddr, flow->rec->sourceIPv6Address);
#else
        return FALSE;
#endif
    }
    return TRUE;
}

static gboolean
mdSkpcFlowGetDIP(
    const mdFullFlow_t  *flow,
    skipaddr_t          *ipaddr)
{
    if (!flow->tmpl_attr.is_ipv6) {
        skipaddrSetV4(ipaddr, &flow->rec->destinationIPv4Address);
    } else {
#if HAVE_SKIPADDRSETV6
        skipaddrSetV6(ipaddr, flow->rec->destinationIPv6Address);
#else
        return FALSE;
#endif
    }
    return TRUE;
}


/* requires glib 2.22 due to g_array_get_element_size() */


/*
 *  *****  Flow Type  **************************************************
 *
 *  The probe is used to determine the flow-type---as defined by in
 *  the silk.conf file---of a flow record (mdFullFlow_t) read from that
 *  probe.
 *
 *  The mdSkpcProbeDetermineFlowtype() function is defined in the
 *  probeconf-<$SILK_SITE>.c file.
 *
 */

/* LOCAL DEFINES AND TYPEDEFS */

/* Minimum version of libfixbuf required for IPFIX */
#define SKPC_LIBFIXBUF_VERSION_IPFIX        "1.7.0"

/* Minimum version of libfixbuf required for NetFlow V9 */
#define SKPC_LIBFIXBUF_VERSION_NETFLOWV9    SKPC_LIBFIXBUF_VERSION_IPFIX

/* Minimum version of libfixbuf required for sFlow */
#define SKPC_LIBFIXBUF_VERSION_SFLOW        SKPC_LIBFIXBUF_VERSION_IPFIX

/* Maximum valid value for a port 2^16 - 1 */
#define PORT_VALID_MAX 0xFFFF

/* Set ports to this invalid value initially */
#define PORT_NOT_SET   0xFFFFFFFF

/* Value to use for remaining IPs to say that it hasn't been set */
#define REMAINDER_NOT_SET  INT8_MAX


/* a map between probe types and printable names */
static const struct probe_type_name_map_st {
    const char           *name;
    md_skpc_probetype_t   value;
} probe_type_name_map[] = {
    {"ipfix",       MD_PROBE_ENUM_IPFIX},
    {"netflow-v5",  MD_PROBE_ENUM_NETFLOW_V5},
    {"netflow-v9",  MD_PROBE_ENUM_NETFLOW_V9},
    {"sflow",       MD_PROBE_ENUM_SFLOW},
    {"silk",        MD_PROBE_ENUM_SILK},

    /* legacy name for netflow-v5 */
    {"netflow",     MD_PROBE_ENUM_NETFLOW_V5},

    /* sentinel */
    {NULL,          MD_PROBE_ENUM_INVALID}
};



/* LOCAL VARIABLES */

/* The probes that have been created and verified as an array of
 * md_skpc_probe_t*. */
static GArray *md_skpc_probes = NULL;

/* The sensors that have been created and verified as an array of
 * md_skpc_sensor_t*. */
static GArray *md_skpc_sensors = NULL;

/* The networks that have been created as an array of md_skpc_network_t. */
static GArray *md_skpc_networks = NULL;

/* The groups that have been created as an array of md_skpc_group_t*. */
static GArray *md_skpc_groups = NULL;

/* The CIDR blocks that are added to the groups as an array of skcidr_t*. */
static GArray *md_skpc_cidrs = NULL;

/* Group containing the default non-routed NetFlow interface */
static md_skpc_group_t *nonrouted_group = NULL;


/* LOCAL FUNCTION PROTOTYPES */

static int
mdSkpcGroupCheckInterface(
    const md_skpc_group_t  *group,
    uint32_t                interface);
static int
mdSkpcGroupCheckIPblock(
    const md_skpc_group_t  *group,
    const skipaddr_t       *ip);
static int
mdSkpcGroupCheckIPset(
    const md_skpc_group_t  *group,
    const skipaddr_t       *ip);
static uint32_t
mdSkpcGroupGetItemCount(
    const md_skpc_group_t  *group);


/* FUNCTION DEFINITIONS */


/*
 *  *****  Probe configuration  **************************************
 */

/* setup the probes */
int
mdSkpcSetup(
    void)
{
    if (NULL == md_skpc_probes) {
        md_skpc_probes = g_array_sized_new(FALSE, TRUE,
                                           sizeof(md_skpc_probe_t *),
                                           DEFAULT_ARRAY_SIZE);
    }

    if (NULL == md_skpc_sensors) {
        md_skpc_sensors = g_array_sized_new(FALSE, TRUE,
                                            sizeof(md_skpc_sensor_t *),
                                            DEFAULT_ARRAY_SIZE);
    }

    if (NULL == md_skpc_networks) {
        md_skpc_networks = g_array_sized_new(FALSE, TRUE,
                                             sizeof(md_skpc_network_t),
                                             DEFAULT_ARRAY_SIZE);
    }

    if (NULL == md_skpc_groups) {
        md_skpc_groups = g_array_sized_new(FALSE, TRUE,
                                           sizeof(md_skpc_group_t *),
                                           DEFAULT_ARRAY_SIZE);
    }

    if (mdSkpcParseSetup()) {
        goto ERROR;
    }

    return 0;

  ERROR:
    if (md_skpc_probes) {
        g_array_free(md_skpc_probes, TRUE);
    }
    if (md_skpc_sensors) {
        g_array_free(md_skpc_sensors, TRUE);
    }
    if (md_skpc_networks) {
        g_array_free(md_skpc_networks, TRUE);
    }
    if (md_skpc_groups) {
        g_array_free(md_skpc_groups, TRUE);
    }
    return -1;
}


/* destroy everything */
void
mdSkpcTeardown(
    void)
{
    md_skpc_network_t *nwp;
    md_skpc_probe_t  **probe;
    md_skpc_sensor_t **sensor;
    md_skpc_group_t  **group;
    skcidr_t          *cidr;
    size_t             i;

    /* clean up the parser */
    mdSkpcParseTeardown();

    /* Free all the networks */
    if (md_skpc_networks) {
        for (i = 0; i < md_skpc_networks->len; ++i) {
            nwp = &g_array_index(md_skpc_networks, md_skpc_network_t, i);
            g_free(nwp->name);
        }
        g_array_free(md_skpc_networks, TRUE);
        md_skpc_networks = NULL;
    }

    /* Free all the groups */
    if (md_skpc_groups) {
        for (i = 0; i < md_skpc_groups->len; ++i) {
            group = &g_array_index(md_skpc_groups, md_skpc_group_t *, i);
            mdSkpcGroupDestroy(group);
        }
        g_array_free(md_skpc_groups, TRUE);
        md_skpc_groups = NULL;
    }

    /* Free all the sensors */
    if (md_skpc_sensors) {
        for (i = 0; i < md_skpc_sensors->len; ++i) {
            sensor = &g_array_index(md_skpc_sensors, md_skpc_sensor_t *, i);
            mdSkpcSensorDestroy(sensor);
        }
        /* destroy the vector itself */
        g_array_free(md_skpc_sensors, TRUE);
        md_skpc_sensors = NULL;
    }

    /* Free all the probes */
    if (md_skpc_probes) {
        for (i = 0; i < md_skpc_probes->len; ++i) {
            probe = &g_array_index(md_skpc_probes, md_skpc_probe_t *, i);
            mdSkpcProbeDestroy(probe);
        }
        /* destroy the vector itself */
        g_array_free(md_skpc_probes, TRUE);
        md_skpc_probes = NULL;
    }

    /* Free all the CIDR blocks */
    if (md_skpc_cidrs) {
        for (i = 0; i < md_skpc_cidrs->len; ++i) {
            cidr = g_array_index(md_skpc_cidrs, skcidr_t *, i);
            g_slice_free(skcidr_t, cidr);
        }
        /* destroy the vector itself */
        g_array_free(md_skpc_cidrs, TRUE);
        md_skpc_cidrs = NULL;
    }
}


/* return a count of verified probes */
size_t
mdSkpcCountProbes(
    void)
{
    assert(md_skpc_probes);
    return md_skpc_probes->len;
}


int
mdSkpcProbeIteratorBind(
    md_skpc_probe_iter_t  *probe_iter)
{
    if (probe_iter == NULL || md_skpc_probes == NULL) {
        return -1;
    }
    probe_iter->cur = 0;
    return 0;
}


int
mdSkpcProbeIteratorNext(
    md_skpc_probe_iter_t   *probe_iter,
    const md_skpc_probe_t **probe)
{
    if (probe_iter == NULL || probe == NULL) {
        return -1;
    }

    if (probe_iter->cur >= md_skpc_probes->len) {
        return 0;
    }

    *probe = g_array_index(md_skpc_probes, const md_skpc_probe_t *,
                           probe_iter->cur);
    ++probe_iter->cur;
    return 1;
}


/* return a probe having the given probe-name. */
const md_skpc_probe_t *
mdSkpcProbeLookupByName(
    const char  *probe_name)
{
    const md_skpc_probe_t *probe;
    size_t                 i;

    assert(md_skpc_probes);

    /* check input */
    if (probe_name == NULL) {
        return NULL;
    }

    /* loop over all probes until we find one with given name */
    for (i = 0; i < md_skpc_probes->len; ++i) {
        probe = g_array_index(md_skpc_probes, const md_skpc_probe_t *, i);
        if (0 == strcmp(probe_name, probe->probe_name)) {
            return probe;
        }
    }

    return NULL;
}


/* return a count of verified sensors */
size_t
mdSkpcCountSensors(
    void)
{
    assert(md_skpc_sensors);
    return md_skpc_sensors->len;
}


int
mdSkpcSensorIteratorBind(
    md_skpc_sensor_iter_t  *sensor_iter)
{
    if (sensor_iter == NULL || md_skpc_sensors == NULL) {
        return -1;
    }
    sensor_iter->cur = 0;
    return 0;
}


int
mdSkpcSensorIteratorNext(
    md_skpc_sensor_iter_t   *sensor_iter,
    const md_skpc_sensor_t **sensor)
{
    if (sensor_iter == NULL || sensor == NULL) {
        return -1;
    }

    if (sensor_iter->cur >= md_skpc_sensors->len) {
        return 0;
    }

    *sensor = g_array_index(md_skpc_sensors, const md_skpc_sensor_t *,
                            sensor_iter->cur);
    ++sensor_iter->cur;
    return 1;
}


/* append to sensor_vec sensors having the given sensor-name. */
int
mdSkpcSensorLookupByName(
    const char  *sensor_name,
    GArray      *sensor_vec)
{
    const md_skpc_sensor_t *s;
    int                     count = 0;
    size_t                  i;

    assert(md_skpc_sensors);

    /* check input */
    if (sensor_name == NULL || sensor_vec == NULL) {
        return -1;
    }
    if (g_array_get_element_size(sensor_vec) != sizeof(md_skpc_sensor_t *)) {
        return -1;
    }

    /* loop over all sensors looking for the given name */
    for (i = 0; i < md_skpc_sensors->len; ++i) {
        s = g_array_index(md_skpc_sensors, const md_skpc_sensor_t *, i);
        if (0 == strcmp(sensor_name, s->sensor_name)) {
            g_array_append_val(sensor_vec, s);
            ++count;
        }
    }

    return count;
}


/* append to sensor_vec sensors having the given sensor-name. */
int
mdSkpcSensorLookupByID(
    sk_sensor_id_t   sensor_id,
    GArray          *sensor_vec)
{
    const md_skpc_sensor_t *s;
    int                     count = 0;
    size_t                  i;

    assert(md_skpc_sensors);

    /* check input */
    if (sensor_vec == NULL) {
        return -1;
    }
    if (g_array_get_element_size(sensor_vec) != sizeof(md_skpc_sensor_t *)) {
        return -1;
    }

    /* loop over all sensors looking for the given id */
    for (i = 0; i < md_skpc_sensors->len; ++i) {
        s = g_array_index(md_skpc_sensors, const md_skpc_sensor_t *, i);
        if (sensor_id == s->sensor_id) {
            g_array_append_val(sensor_vec, s);
            ++count;
        }
    }

    return count;
}


/*
 *  *****  Network  ****************************************************
 */

/* Add a new id->name pair to the list of networks */
void
mdSkpcNetworkAdd(
    md_skpc_network_id_t   id,
    const char            *name)
{
    md_skpc_network_t *nwp;
    md_skpc_network_t  nw;
    size_t             i;

    assert(md_skpc_networks);

    if (id > MD_SKPC_NETWORK_ID_MAX) {
        fprintf(stderr, "Cannot add network %zu => '%s':"
                "Network id is greater than maximum %lu\n",
                id, name, MD_SKPC_NETWORK_ID_MAX);
        exit(1);
    }

    /* check for  */
    for (i = 0; i < md_skpc_networks->len; ++i) {
        nwp = &g_array_index(md_skpc_networks, md_skpc_network_t, i);
        if (id == nwp->id) {
            /* duplicate id */
            fprintf(stderr, "Cannot add network %zu => '%s':"
                    "Network id is already in use with name '%s'\n",
                    id, name, nwp->name);
            exit(1);
        }
        if (0 == strcmp(name, nwp->name)) {
            /* duplicate name */
            fprintf(stderr, "Cannot add network %zu => '%s':"
                    "Network name is already in use with id %zu\n",
                    id, name, nwp->id);
            exit(1);
        }
    }

    /* create network and add it to the vector */
    nw.id = id;
    nw.name = g_strdup(name);
    g_array_append_val(md_skpc_networks, nw);
}


/* Find the network that has the given name 'name' */
const md_skpc_network_t *
mdSkpcNetworkLookupByName(
    const char  *name)
{
    md_skpc_network_t *nwp;
    size_t             i;

    assert(md_skpc_networks);

    for (i = 0; i < md_skpc_networks->len; ++i) {
        nwp = &g_array_index(md_skpc_networks, md_skpc_network_t, i);
        if (0 == strcmp(name, nwp->name)) {
            return nwp;
        }
    }
    return NULL;
}


/* Find the network that has the given ID 'id' */
const md_skpc_network_t *
mdSkpcNetworkLookupByID(
    md_skpc_network_id_t   network_id)
{
    md_skpc_network_t *nwp;
    size_t             i;

    assert(md_skpc_networks);
    assert(network_id <= MD_SKPC_NETWORK_ID_INVALID);

    for (i = 0; i < md_skpc_networks->len; ++i) {
        nwp = &g_array_index(md_skpc_networks, md_skpc_network_t, i);
        if (network_id == nwp->id) {
            return nwp;
        }
    }
    return NULL;
}



/*
 *  *****  Probes  *****************************************************
 */

/* Create a probe */
int
mdSkpcProbeCreate(
    md_skpc_probe_t     **probe,
    md_skpc_probetype_t   probe_type)
{
    assert(probe);

    if (NULL == mdSkpcProbetypeEnumtoName(probe_type)) {
        return -1;
    }

    (*probe) = g_slice_new0(md_skpc_probe_t);
    (*probe)->probe_type = probe_type;

    return 0;
}


/* Destroy a probe and free all memory associated with it */
void
mdSkpcProbeDestroy(
    md_skpc_probe_t **probe)
{
    if (!probe || !(*probe)) {
        return;
    }

    if ((*probe)->sensor_list) {
        g_array_free((*probe)->sensor_list, TRUE);
    }
    if ((*probe)->probe_name) {
        g_free((void *)((*probe)->probe_name));
    }

    g_slice_free(md_skpc_probe_t, *probe);
    *probe = NULL;
}


/* Get and set the name of probe */

#ifndef mdSkpcProbeGetName
const char *
mdSkpcProbeGetName(
    const md_skpc_probe_t  *probe)
{
    assert(probe);
    return probe->probe_name;
}
#endif  /* mdSkpcProbeGetName */

int
mdSkpcProbeSetName(
    md_skpc_probe_t  *probe,
    const char       *name)
{
    const char *cp;
    char       *copy;

    assert(probe);
    if (name == NULL || name[0] == '\0') {
        return -1;
    }

    /* check for illegal characters */
    cp = name;
    while (*cp) {
        if (*cp == '/' || isspace((int)*cp)) {
            return -1;
        }
        ++cp;
    }

    copy = g_strdup(name);
    if (probe->probe_name) {
        g_free((void *)probe->probe_name);
    }
    probe->probe_name = copy;
    return 0;
}


/* Get and set the probe type */
#ifndef mdSkpcProbeGetType
md_skpc_probetype_t
mdSkpcProbeGetType(
    const md_skpc_probe_t  *probe)
{
    assert(probe);
    return probe->probe_type;
}
#endif  /* mdSkpcProbeGetType */


/* Get and set type of data in the input/output fields */

#ifndef mdSkpcProbeGetInterfaceValueType
md_skpc_ifvaluetype_t
mdSkpcProbeGetInterfaceValueType(
    const md_skpc_probe_t  *probe)
{
    assert(probe);
    return probe->ifvaluetype;
}
#endif  /* mdSkpcProbeGetInterfaceValueType */

int
mdSkpcProbeSetInterfaceValueType(
    md_skpc_probe_t        *probe,
    md_skpc_ifvaluetype_t   interface_value_type)
{
    assert(probe);
    switch (interface_value_type) {
      case MD_SKPC_IFVALUE_SNMP:
      case MD_SKPC_IFVALUE_VLAN:
        probe->ifvaluetype = interface_value_type;
        break;
      default:
        return -1;              /* NOTREACHED */
    }
    return 0;
}


#ifndef mdSkpcProbeGetSensorCount
size_t
mdSkpcProbeGetSensorCount(
    const md_skpc_probe_t  *probe)
{
    assert(probe);
    return probe->sensor_count;
}
#endif  /* mdSkpcProbeGetSensorCount */


static int
mdSkpcProbeAddSensor(
    md_skpc_probe_t   *probe,
    md_skpc_sensor_t  *sensor)
{
#if 0
    size_t  i;

    /* 2011.12.09 Allow the same sensor to appear on a probe multiple
     * times, and assume the user is using a filter (e.g.,
     * discard-when) to avoid packing the flow record multiple
     * times. */
    /* each sensor may only appear on a probe one time */
    for (i = 0; i < probe->sensor_count; ++i) {
        if (mdSkpcSensorGetID(probe->sensor_list[i])
            == mdSkpcSensorGetID(sensor))
        {
            return -1;
        }
    }
#endif  /* 0 */

    if (probe->sensor_list == NULL) {
        /* create the sensor list; large enough to hold one sensor */
        probe->sensor_list = g_array_new(FALSE, TRUE,
                                         sizeof(md_skpc_sensor_t *));
    }
    g_array_append_val(probe->sensor_list, sensor);

    return 0;
}


/*
 *  *****  Verification  ***********************************************
 */


/* Has probe been verified? */
int
mdSkpcProbeIsVerified(
    const md_skpc_probe_t  *probe)
{
    assert(probe);
    return probe->verified;
}


/*
 *    Verify that 'p' is a valid probe.
 */
gboolean
mdSkpcProbeVerify(
    md_skpc_probe_t  *probe,
    int               is_ephemeral)
{
    assert(probe);
    assert(md_skpc_probes);

    /* check name */
    if ('\0' == probe->probe_name[0]) {
        fprintf(stderr, "Error verifying probe:\n\tProbe has no name.\n");
        return FALSE;
    }

    /* verify type is not invalid */
    if (probe->probe_type == MD_PROBE_ENUM_INVALID) {
        fprintf(stderr,
                "Error verifying probe '%s':\n\tProbe's type is INVALID.\n",
                probe->probe_name);
        return FALSE;
    }

    /* make certain no other probe has this name */
    if (mdSkpcProbeLookupByName(probe->probe_name)) {
        fprintf(stderr, ("Error verifying probe '%s':\n"
                         "\tA probe with this name is already defined\n"),
                probe->probe_name);
        return FALSE;
    }

    /* if is_ephemeral is specified, add it to the global list of
     * probes but don't mark it as verified */
    if (is_ephemeral) {
        g_array_append_val(md_skpc_probes, probe);
        return TRUE;
    }

    /* probe is valid; add it to the global vector of probes */
    g_array_append_val(md_skpc_probes, probe);

    probe->verified = 1;
    return TRUE;
}


void
mdSkpcProbePrint(
    const md_skpc_probe_t  *probe,
    sk_msg_fn_t             printer)
{
    char  name[PATH_MAX];

    /* fill 'name' with name and type of probe */
    snprintf(name, sizeof(name), "'%s': %s probe;",
             probe->probe_name ? probe->probe_name : "<EMPTY_NAME>",
             mdSkpcProbetypeEnumtoName(probe->probe_type));

    printer("%s", name);
}


/*
 *  *****  Sensors  *****************************************************
 */

/* Create a sensor */
int
mdSkpcSensorCreate(
    md_skpc_sensor_t **sensor)
{
    assert(sensor);

    (*sensor) = g_slice_new0(md_skpc_sensor_t);

    (*sensor)->sensor_id = SK_INVALID_SENSOR;
    (*sensor)->fixed_network[0] = MD_SKPC_NETWORK_ID_INVALID;
    (*sensor)->fixed_network[1] = MD_SKPC_NETWORK_ID_INVALID;

    /* create the decider array; one for each network.  call set_size() since
     * the code expects them to exist. */
    (*sensor)->decider = g_array_sized_new(FALSE, TRUE,
                                           sizeof(md_skpc_netdecider_t),
                                           md_skpc_networks->len);
    g_array_set_size((*sensor)->decider, md_skpc_networks->len);

    return 0;
}


void
mdSkpcSensorDestroy(
    md_skpc_sensor_t **sensor)
{
    size_t  i;

    if (!sensor || !(*sensor)) {
        return;
    }

    /* set the sensor's deciders' group to NULL, then destroy the
     * deciders */
    for (i = 0; i < (*sensor)->decider->len; ++i) {
        g_array_index((*sensor)->decider, md_skpc_netdecider_t, i).nd_group =
            NULL;
    }
    g_array_free((*sensor)->decider, TRUE);
    (*sensor)->decider = NULL;

    /* destroy the probe list */
    if ((*sensor)->probe_list) {
        g_array_free((*sensor)->probe_list, TRUE);
        (*sensor)->probe_list = NULL;
    }

    /* set the 'group' reference on all filters to NULL, then destroy
     * the filters array. */
    if ((*sensor)->filter) {
        for (i = 0; i < (*sensor)->filter->len; ++i) {
            g_array_index((*sensor)->filter, md_skpc_filter_t, i).f_group =
                NULL;
        }
        g_array_free((*sensor)->filter, TRUE);
        (*sensor)->filter = NULL;
    }

    /* destroy other attributes of the sensor */
    if ((*sensor)->sensor_name) {
        g_free((*sensor)->sensor_name);
    }

    /* destroy the sensor itself */
    g_slice_free(md_skpc_sensor_t, *sensor);
    *sensor = NULL;
}


/* Get and set the name of this sensor. */
#ifndef mdSkpcSensorGetID
sk_sensor_id_t
mdSkpcSensorGetID(
    const md_skpc_sensor_t  *sensor)
{
    assert(sensor);
    return sensor->sensor_id;
}
#endif  /* mdSkpcSensorGetID */

#ifndef mdSkpcSensorGetName
const char *
mdSkpcSensorGetName(
    const md_skpc_sensor_t  *sensor)
{
    assert(sensor);
    return sensor->sensor_name;
}
#endif  /* mdSkpcSensorGetName */

int
mdSkpcSensorSetName(
    md_skpc_sensor_t  *sensor,
    const char        *name)
{
    char *copy;

    assert(sensor);
    if (name == NULL || name[0] == '\0') {
        return -1;
    }

    copy = g_strdup(name);
    if (sensor->sensor_name) {
        g_free(sensor->sensor_name);
    }
    sensor->sensor_name = copy;

    sensor->sensor_id = sksiteSensorLookup(name);
    return 0;
}


/* Count the number of SNMP interfaces that have been mapped to a
 * flowtype on 'sensor' excluding the network 'ignored_network_id' */
uint32_t
mdSkpcSensorCountNetflowInterfaces(
    const md_skpc_sensor_t  *sensor,
    int                      ignored_network_id)
{
    const md_skpc_netdecider_t *decider;
    uint32_t                    ifcount = 0;
    size_t                      i;

    for (i = 0; i < sensor->decider->len; ++i) {
        if (ignored_network_id == (int)i) {
            continue;
        }
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        if (((decider->nd_type == MD_SKPC_INTERFACE)
             || (decider->nd_type == MD_SKPC_REMAIN_INTERFACE))
            && (decider->nd_group != NULL))
        {
            ifcount += mdSkpcGroupGetItemCount(decider->nd_group);
        }
    }
    return ifcount;
}


/* Test rwrec against the interfaces (either SNMP or IP block) on the
 * sensor, or used the fixed network value if set. */
int
mdSkpcSensorTestFlowInterfaces(
    const md_skpc_sensor_t  *sensor,
    const mdFullFlow_t      *flow,
    md_skpc_network_id_t     network_id,
    md_skpc_direction_t      rec_dir)
{
#if !defined(ENABLE_SKTYPESENSOR) || !ENABLE_SKTYPESENSOR
    /* handle case when disabled in config.h */
    return 1;
#else

    const md_skpc_netdecider_t *decider;
    uint32_t                    ingress;
    uint32_t                    egress;
    skipaddr_t                  ip;
    int                         found = 0;

    assert(sensor);
    assert(flow);
    assert(network_id <= MD_SKPC_NETWORK_ID_MAX);
    assert(rec_dir == MD_SKPC_DIR_SRC || rec_dir == MD_SKPC_DIR_DST);

    /* use the fixed value if provided */
    if (sensor->fixed_network[rec_dir] == network_id) {
        return 1;
    }

    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, network_id);
    switch (decider->nd_type) {
      case MD_SKPC_UNSET:
        break;

      case MD_SKPC_REMAIN_INTERFACE:
        /* want to find whether the interface is NOT in the list */
        found = 1;
        /* FALLTHROUGH */

      case MD_SKPC_INTERFACE:
        /* An SNMP interface list was set for the network_id.  Test
         * the record's SNMP value against it.  Whether incoming or
         * outgoing depends on 'rec_dir'. */
        if (flow->collector_silk_probe_vlan) {
            ingress = flow->rec->vlanId;
            egress = flow->rec->reverseVlanId;
        } else {
            ingress = flow->rec->ingressInterface;
            egress = flow->rec->egressInterface;
        }
        if (rec_dir == MD_SKPC_DIR_SRC) {
            /* look where the record is coming from: its input SNMP
             * interface */
            if (mdSkpcGroupCheckInterface(decider->nd_group, ingress)) {
                found = !found;
            }
        } else {
            /* look where the record is going to: output SNMP */
            assert(rec_dir == MD_SKPC_DIR_DST);
            if (mdSkpcGroupCheckInterface(decider->nd_group, egress)) {
                found = !found;
            }
        }
        return ((found == 0) ? 1 : -1);

      case MD_SKPC_NEG_IPBLOCK:
      case MD_SKPC_REMAIN_IPBLOCK:
        /* want to find whether the IP is NOT in the list */
        found = 1;
        /* FALLTHROUGH */

      case MD_SKPC_IPBLOCK:
        /* An IP block was set for 'network_id'.  Test the record's IP
         * against it.  Whether source IP or dest IP depends on
         * 'rec_dir'. */
        if (rec_dir == MD_SKPC_DIR_SRC) {
            /* look where the record is coming from: its source IP */
            if (mdSkpcFlowGetSIP(flow, &ip) &&
                mdSkpcGroupCheckIPblock(decider->nd_group, &ip))
            {
                found = !found;
            }
        } else {
            /* look where the record is going to: destination IP */
            assert(rec_dir == MD_SKPC_DIR_DST);
            if (mdSkpcFlowGetDIP(flow, &ip) &&
                mdSkpcGroupCheckIPblock(decider->nd_group, &ip))
            {
                found = !found;
            }
        }
        return ((found == 0) ? -1 : 1);

      case MD_SKPC_NEG_IPSET:
      case MD_SKPC_REMAIN_IPSET:
        found = 1;
        /* FALLTHROUGH */

      case MD_SKPC_IPSET:
        if (rec_dir == MD_SKPC_DIR_SRC) {
            /* look where the record is coming from: its source IP */
            if (mdSkpcFlowGetSIP(flow, &ip) &&
                mdSkpcGroupCheckIPset(decider->nd_group, &ip))
            {
                found = !found;
            }
        } else {
            /* look where the record is going to: destination IP */
            assert(rec_dir == MD_SKPC_DIR_DST);
            if (mdSkpcFlowGetDIP(flow, &ip) &&
                mdSkpcGroupCheckIPset(decider->nd_group, &ip))
            {
                found = !found;
            }
        }
        return ((found == 0) ? -1 : 1);
    }

    return 0;
#endif  /* ENABLE_SKTYPESENSOR */
}


/* Return non-zero if 'flow' matches ANY of the "discard-when"
 * filters on 'sensor' or if it does not match ALL of the
 * "discard-unless" filters. */
int
mdSkpcSensorCheckFilters(
    const md_skpc_sensor_t  *sensor,
    const mdFullFlow_t      *flow)
{
#if !defined(ENABLE_SKTYPESENSOR) || !ENABLE_SKTYPESENSOR
    /* handle case when disabled in config.h */
    return 1;
#else

    skipaddr_t              sip;
    skipaddr_t              dip;
    uint32_t                ingress;
    uint32_t                egress;
    const md_skpc_filter_t *filter;
    size_t                  discard;
    size_t                  j;

    assert(sensor);
    assert(flow);

    if (flow->collector_silk_probe_vlan) {
        ingress = flow->rec->vlanId;
        egress = flow->rec->reverseVlanId;
    } else {
        ingress = flow->rec->ingressInterface;
        egress = flow->rec->egressInterface;
    }

    for (j = 0; j < sensor->filter->len; ++j) {
        filter = &g_array_index(sensor->filter, md_skpc_filter_t, j);
        discard = !filter->f_discwhen;
        switch (filter->f_group_type) {
          case MD_SKPC_GROUP_UNSET:
            skAbortBadCase(filter->f_group_type);

          case MD_SKPC_GROUP_IPBLOCK:
            switch (filter->f_type) {
              case MD_SKPC_FILTER_SOURCE:
                if (mdSkpcFlowGetSIP(flow, &sip) &&
                    mdSkpcGroupCheckIPblock(filter->f_group, &sip))
                {
                    discard = !discard;
                }
                break;

              case MD_SKPC_FILTER_DESTINATION:
                if (mdSkpcFlowGetDIP(flow, &dip) &&
                    mdSkpcGroupCheckIPblock(filter->f_group, &dip))
                {
                    discard = !discard;
                }
                break;

              case MD_SKPC_FILTER_ANY:
                if ((mdSkpcFlowGetSIP(flow, &sip)
                     && mdSkpcGroupCheckIPblock(filter->f_group, &sip)) ||
                    (mdSkpcFlowGetDIP(flow, &dip)
                     && mdSkpcGroupCheckIPblock(filter->f_group, &dip)))
                {
                    discard = !discard;
                }
                break;
            }
            break;

          case MD_SKPC_GROUP_IPSET:
            switch (filter->f_type) {
              case MD_SKPC_FILTER_SOURCE:
                if (mdSkpcFlowGetSIP(flow, &sip) &&
                    mdSkpcGroupCheckIPset(filter->f_group, &sip))
                {
                    discard = !discard;
                }
                break;

              case MD_SKPC_FILTER_DESTINATION:
                if (mdSkpcFlowGetDIP(flow, &dip) &&
                    mdSkpcGroupCheckIPset(filter->f_group, &dip))
                {
                    discard = !discard;
                }
                break;

              case MD_SKPC_FILTER_ANY:
                if ((mdSkpcFlowGetSIP(flow, &sip) &&
                     mdSkpcGroupCheckIPset(filter->f_group, &sip)) ||
                    (mdSkpcFlowGetDIP(flow, &dip) &&
                     mdSkpcGroupCheckIPset(filter->f_group, &dip)))
                {
                    discard = !discard;
                }
                break;
            }
            break;

          case MD_SKPC_GROUP_INTERFACE:
            switch (filter->f_type) {
              case MD_SKPC_FILTER_SOURCE:
                if (mdSkpcGroupCheckInterface(filter->f_group, ingress)) {
                    discard = !discard;
                }
                break;

              case MD_SKPC_FILTER_DESTINATION:
                if (mdSkpcGroupCheckInterface(filter->f_group, egress)) {
                    discard = !discard;
                }
                break;

              case MD_SKPC_FILTER_ANY:
                if (mdSkpcGroupCheckInterface(filter->f_group, ingress)
                    || mdSkpcGroupCheckInterface(filter->f_group, egress))
                {
                    discard = !discard;
                }
                break;
            }
            break;
        }
        if (discard) {
            return 1;
        }
    }

    return 0;
#endif  /* ENABLE_SKTYPESENSOR */
}


int
mdSkpcSensorSetNetworkDirection(
    md_skpc_sensor_t      *sensor,
    md_skpc_network_id_t   network_id,
    md_skpc_direction_t    dir)
{
    const md_skpc_network_t    *network;
    const md_skpc_netdecider_t *decider;
    const char                 *prev_decider = NULL;

    assert(sensor);
    assert(network_id <= MD_SKPC_NETWORK_ID_INVALID);
    assert(dir == MD_SKPC_DIR_SRC || dir == MD_SKPC_DIR_DST);

    /* get network */
    network = mdSkpcNetworkLookupByID(network_id);
    if (network == NULL) {
        return -1;
    }

    /* verify that value not previously set */
    if (sensor->fixed_network[dir] != MD_SKPC_NETWORK_ID_INVALID) {
        fprintf(stderr, ("Error setting %s-network on sensor '%s':\n"
                         "\tCannot overwrite existing value\n"),
                ((dir == 0) ? "source" : "destination"), sensor->sensor_name);
        return -1;
    }

    /* verify that no ipblocks or interfaces have been set for this
     * network */
    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             network->id);
    switch (decider->nd_type) {
      case MD_SKPC_UNSET:
        /* valid */
        break;

      case MD_SKPC_INTERFACE:
      case MD_SKPC_REMAIN_INTERFACE:
        prev_decider = "interface";
        break;

      case MD_SKPC_NEG_IPBLOCK:
      case MD_SKPC_REMAIN_IPBLOCK:
      case MD_SKPC_IPBLOCK:
        prev_decider = "ipblock";
        break;

      case MD_SKPC_NEG_IPSET:
      case MD_SKPC_REMAIN_IPSET:
      case MD_SKPC_IPSET:
        prev_decider = "ipset";
        break;
    }

    if (prev_decider) {
        fprintf(stderr, ("Error setting %s-network on sensor '%s':\n"
                         "\tA %s-%s value has already been set\n"),
                ((dir == 0) ? "source" : "destination"),
                sensor->sensor_name, network->name, prev_decider);
        return -1;
    }

    sensor->fixed_network[dir] = network->id;
    return 0;
}


/* Set the list of interfaces/IPs that represent a network */
int
mdSkpcSensorSetNetworkGroup(
    md_skpc_sensor_t       *sensor,
    md_skpc_network_id_t    network_id,
    const md_skpc_group_t  *group)
{
    const md_skpc_network_t *network;
    md_skpc_netdecider_t    *decider;
    size_t                   i;

    /* check input */
    assert(sensor);
    assert(network_id <= MD_SKPC_NETWORK_ID_INVALID);
    assert(group);
    assert(mdSkpcGroupGetType(group) != MD_SKPC_GROUP_UNSET);

    /* check that group has data and that it is the correct type */
    if (group == NULL) {
        return -1;
    }
    if (!mdSkpcGroupIsFrozen(group)
        || (mdSkpcGroupGetItemCount(group) == 0))
    {
        return -1;
    }

    /* get network */
    network = mdSkpcNetworkLookupByID(network_id);
    if (network == NULL) {
        return -1;
    }
    assert(network->id < sensor->decider->len);

    /* cannot set group when the source/destination network has
     * been fixed to this network_id. */
    for (i = 0; i < 2; ++i) {
        if (sensor->fixed_network[i] == network_id) {
            fprintf(stderr,
                    ("Error setting %ss on sensor '%s':\n"
                     "\tAll flows are assumed to be %s the %s network\n"),
                    mdSkpcGrouptypeEnumtoName(mdSkpcGroupGetType(group)),
                    sensor->sensor_name,
                    ((i == 0) ? "coming from" : "going to"), network->name);
            return -1;
        }
    }

    /* check that we're not attempting to change an existing value */
    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             network->id);
    if (decider->nd_type != MD_SKPC_UNSET) {
        fprintf(stderr, ("Error setting %ss on sensor '%s':\n"
                         "\tCannot overwrite existing %s network value\n"),
                mdSkpcGrouptypeEnumtoName(mdSkpcGroupGetType(group)),
                sensor->sensor_name, network->name);
        return -1;
    }

    decider->nd_group = group;
    switch (mdSkpcGroupGetType(group)) {
      case MD_SKPC_GROUP_INTERFACE:
        decider->nd_type = MD_SKPC_INTERFACE;
        break;
      case MD_SKPC_GROUP_IPBLOCK:
        decider->nd_type = MD_SKPC_IPBLOCK;
        break;
      case MD_SKPC_GROUP_IPSET:
        decider->nd_type = MD_SKPC_IPSET;
        break;
      case MD_SKPC_GROUP_UNSET:
        skAbortBadCase(mdSkpcGroupGetType(group));
    }

    return 0;
}


/* Set the specified network to all values not covered by other networks */
int
mdSkpcSensorSetNetworkRemainder(
    md_skpc_sensor_t      *sensor,
    md_skpc_network_id_t   network_id,
    md_skpc_group_type_t   group_type)
{
    const md_skpc_network_t *network;
    md_skpc_netdecider_t    *decider;
    int                      i;

    assert(sensor);
    assert(network_id <= MD_SKPC_NETWORK_ID_INVALID);
    assert(group_type != MD_SKPC_GROUP_UNSET);

    /* get network */
    network = mdSkpcNetworkLookupByID(network_id);
    if (network == NULL) {
        return -1;
    }

    assert(network->id < sensor->decider->len);

    /* cannot set network when the source/destination network has
     * been fixed to this network_id. */
    for (i = 0; i < 2; ++i) {
        if (sensor->fixed_network[i] == network_id) {
            fprintf(stderr,
                    ("Error setting %ss on sensor '%s':\n"
                     "\tAll flows are assumed to be %s the %s network\n"),
                    mdSkpcGrouptypeEnumtoName(group_type), sensor->sensor_name,
                    ((i == 0) ? "coming from" : "going to"), network->name);
            return -1;
        }
    }
    /* check that we're not attempting to change an existing value */
    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             network->id);
    if (decider->nd_type != MD_SKPC_UNSET) {
        fprintf(stderr, ("Error setting %ss on sensor '%s':\n"
                         "\tCannot overwrite existing %s network value\n"),
                mdSkpcGrouptypeEnumtoName(group_type),
                sensor->sensor_name, network->name);
        return -1;
    }

    switch (group_type) {
      case MD_SKPC_GROUP_INTERFACE:
        decider->nd_type = MD_SKPC_REMAIN_INTERFACE;
        break;
      case MD_SKPC_GROUP_IPBLOCK:
        decider->nd_type = MD_SKPC_REMAIN_IPBLOCK;
        break;
      case MD_SKPC_GROUP_IPSET:
        decider->nd_type = MD_SKPC_REMAIN_IPSET;
        break;
      case MD_SKPC_GROUP_UNSET:
        skAbortBadCase(group_type);
    }

    return 0;
}


int
mdSkpcSensorSetDefaultNonrouted(
    md_skpc_sensor_t      *sensor,
    md_skpc_network_id_t   network_id)
{
    GArray         *ifvec = NULL;
    const uint32_t  default_nonrouted = 0;
    int             rv = -1;

    assert(sensor);
    assert(network_id <= MD_SKPC_NETWORK_ID_INVALID);

    if (NULL == nonrouted_group) {
        ifvec = g_array_sized_new(FALSE, TRUE, sizeof(uint32_t),
                                  DEFAULT_ARRAY_SIZE);
        g_array_append_val(ifvec, default_nonrouted);

        mdSkpcGroupCreate(&nonrouted_group);
        mdSkpcGroupSetType(nonrouted_group, MD_SKPC_GROUP_INTERFACE);
        if (mdSkpcGroupAddValues(nonrouted_group, ifvec)) {
            fprintf(stderr, "%s: Memory allocation error\n", g_get_prgname());
            exit(-1);
        }
        mdSkpcGroupFreeze(nonrouted_group);
    }

    rv = mdSkpcSensorSetNetworkGroup(sensor, network_id, nonrouted_group);
    if (ifvec) {
        g_array_free(ifvec, TRUE);
    }
    return rv;
}


static int
mdSkpcSensorComputeRemainingInterfaces(
    md_skpc_sensor_t  *sensor)
{
    size_t                remain_network = MD_SKPC_NETWORK_ID_INVALID;
    size_t                i;
    md_skpc_group_t      *group = NULL;
    md_skpc_netdecider_t *decider;

    /* determine which network has claimed the 'remainder' */
    for (i = 0; i < sensor->decider->len; ++i) {
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        if (decider->nd_type == MD_SKPC_REMAIN_INTERFACE) {
            if (remain_network != MD_SKPC_NETWORK_ID_INVALID) {
                /* cannot have more than one "remainder" */
                fprintf(stderr,
                        ("Cannot verify sensor '%s':\n"
                         "\tMultiple network values claim 'remainder'\n"),
                        sensor->sensor_name);
                return -1;
            }
            remain_network = i;
        }
    }

    if (remain_network == MD_SKPC_NETWORK_ID_INVALID) {
        /* no one is set to remainder; return */
        return 0;
    }

    /* create a new group */
    mdSkpcGroupCreate(&group);
    mdSkpcGroupSetType(group, MD_SKPC_GROUP_INTERFACE);

    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             remain_network);
    decider->nd_group = group;

    /* add all existing groups to the new group */
    for (i = 0; i < sensor->decider->len; ++i) {
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        if (decider->nd_type == MD_SKPC_INTERFACE) {
            if (mdSkpcGroupAddGroup(group, decider->nd_group)) {
                fprintf(stderr, "%s: Memory allocation error\n",
                        g_get_prgname());
                exit(-1);
            }
        }
    }

    /* freeze the group */
    mdSkpcGroupFreeze(group);

    return 0;
}


static int
mdSkpcSensorComputeRemainingIpBlocks(
    md_skpc_sensor_t  *sensor)
{
    size_t                remain_network = MD_SKPC_NETWORK_ID_INVALID;
    int                   has_ipblocks = 0;
    md_skpc_group_t      *group = NULL;
    md_skpc_netdecider_t *decider;
    size_t                i;

    /* determine which network has claimed the 'remainder'. At the
     * same time, verify that at least one network has 'ipblocks' */
    for (i = 0; i < sensor->decider->len; ++i) {
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        if (decider->nd_type == MD_SKPC_REMAIN_IPBLOCK) {
            if (remain_network != MD_SKPC_NETWORK_ID_INVALID) {
                /* cannot have more than one "remainder" */
                fprintf(stderr,
                        ("Cannot verify sensor '%s':\n"
                         "\tMultiple network values claim 'remainder'\n"),
                        sensor->sensor_name);
                return -1;
            }
            remain_network = i;
        } else if (decider->nd_type == MD_SKPC_IPBLOCK) {
            has_ipblocks = 1;
        }
    }

    if (remain_network == MD_SKPC_NETWORK_ID_INVALID) {
        /* no one is set to remainder; return */
        return 0;
    }

    /* need to have existing IPblocks to set a remainder */
    if (has_ipblocks == 0) {
        const md_skpc_network_t *network =
            mdSkpcNetworkLookupByID(remain_network);
        fprintf(stderr,
                ("Cannot verify sensor '%s':\n"
                 "\tCannot set %s-ipblocks to remaining IP because\n"
                 "\tno other interfaces hold IP blocks\n"),
                sensor->sensor_name, network->name);
        return -1;
    }

    /* create a new group */
    mdSkpcGroupCreate(&group);
    mdSkpcGroupSetType(group, MD_SKPC_GROUP_IPBLOCK);

    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             remain_network);
    decider->nd_group = group;

    /* add all existing groups to the new group */
    for (i = 0; i < sensor->decider->len; ++i) {
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        if (decider->nd_type == MD_SKPC_IPBLOCK) {
            if (mdSkpcGroupAddGroup(group, decider->nd_group)) {
                fprintf(stderr, "%s: Memory allocation error\n",
                        g_get_prgname());
                exit(-1);
            }
        }
    }

    /* freze the group */
    mdSkpcGroupFreeze(group);

    return 0;
}


static int
mdSkpcSensorComputeRemainingIpSets(
    md_skpc_sensor_t  *sensor)
{
    size_t                remain_network = MD_SKPC_NETWORK_ID_INVALID;
    int                   has_ipsets = 0;
    md_skpc_group_t      *group = NULL;
    md_skpc_netdecider_t *decider;
    size_t                i;

    /* determine which network has claimed the 'remainder'. At the
     * same time, verify that at least one network has 'ipsets' */
    for (i = 0; i < sensor->decider->len; ++i) {
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        if (decider->nd_type == MD_SKPC_REMAIN_IPSET) {
            if (remain_network != MD_SKPC_NETWORK_ID_INVALID) {
                /* cannot have more than one "remainder" */
                fprintf(stderr,
                        ("Cannot verify sensor '%s':\n"
                         "\tMultiple network values claim 'remainder'\n"),
                        sensor->sensor_name);
                return -1;
            }
            remain_network = i;
        } else if (decider->nd_type == MD_SKPC_IPSET) {
            has_ipsets = 1;
        }
    }

    if (remain_network == MD_SKPC_NETWORK_ID_INVALID) {
        /* no one is set to remainder; return */
        return 0;
    }

    /* need to have existing IPsets to set a remainder */
    if (has_ipsets == 0) {
        const md_skpc_network_t *network =
            mdSkpcNetworkLookupByID(remain_network);
        fprintf(stderr,
                ("Cannot verify sensor '%s':\n"
                 "\tCannot set %s-ipsets to remaining IP because\n"
                 "\tno other interfaces hold IP sets\n"),
                sensor->sensor_name, network->name);
        return -1;
    }

    /* create a new group */
    mdSkpcGroupCreate(&group);
    mdSkpcGroupSetType(group, MD_SKPC_GROUP_IPSET);

    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             remain_network);
    decider->nd_group = group;

    /* add all existing groups to the new group */
    for (i = 0; i < sensor->decider->len; ++i) {
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        if (decider->nd_type == MD_SKPC_IPSET) {
            if (mdSkpcGroupAddGroup(group, decider->nd_group)) {
                fprintf(stderr, "%s: Memory allocation error\n",
                        g_get_prgname());
                exit(-1);
            }
        }
    }

    /* freze the group */
    mdSkpcGroupFreeze(group);

    return 0;
}


/* add a new discard-{when,unless} list to 'sensor' */
int
mdSkpcSensorAddFilter(
    md_skpc_sensor_t       *sensor,
    const md_skpc_group_t  *group,
    md_skpc_filter_type_t   filter_type,
    int                     is_discardwhen_list,
    md_skpc_group_type_t    group_type)
{
    const char             *filter_name;
    const md_skpc_filter_t *filter;
    md_skpc_filter_t        new_filter;
    size_t                  j;
    int                     rv = -1;

    assert(sensor);

    /* check that group has data and that it is the correct type */
    if (group == NULL) {
        return -1;
    }
    if (!mdSkpcGroupIsFrozen(group)
        || (mdSkpcGroupGetItemCount(group) == 0)
        || (mdSkpcGroupGetType(group) != group_type))
    {
        return -1;
    }

    /* if this is the first filter, allocate space for all the filters
     * on this sensor */
    if (NULL == sensor->filter) {
        /* allow room for both interface-filters and ipblock-filters */
        sensor->filter = g_array_sized_new(FALSE, TRUE,
                                           sizeof(md_skpc_filter_t),
                                           (MD_SKPC_NUM_GROUP_TYPES *
                                            MD_SKPC_NUM_FILTER_TYPES));
    }

    /* verify we are not attempting to overwrite a value */
    for (j = 0; j < sensor->filter->len; ++j) {
        filter = &g_array_index(sensor->filter, md_skpc_filter_t, j);
        if (filter->f_type == filter_type
            && filter->f_group_type == group_type)
        {
            /* error */
            switch (filter_type) {
              case MD_SKPC_FILTER_ANY:
                filter_name = "any";
                break;
              case MD_SKPC_FILTER_DESTINATION:
                filter_name = "destination";
                break;
              case MD_SKPC_FILTER_SOURCE:
                filter_name = "source";
                break;
              default:
                skAbortBadCase(filter_type);
            }
            fprintf(stderr,
                    ("Error setting discard-%s list on sensor '%s':\n"
                     "\tMay not overwrite existing %s-%ss list\n"),
                    (is_discardwhen_list ? "when" : "unless"),
                    sensor->sensor_name, filter_name,
                    mdSkpcGrouptypeEnumtoName(group_type));
            return -1;
        }
    }

    assert(sensor->filter->len < (2 * MD_SKPC_NUM_FILTER_TYPES));

    memset(&new_filter, 0, sizeof(new_filter));
    new_filter.f_group = group;
    new_filter.f_type = filter_type;
    new_filter.f_group_type = group_type;
    new_filter.f_discwhen = (is_discardwhen_list ? 1 : 0);

    g_array_append_val(sensor->filter, new_filter);

    rv = 0;

    return rv;
}


uint32_t
mdSkpcSensorGetProbes(
    const md_skpc_sensor_t  *sensor,
    GArray                  *out_probe_vec)
{
    assert(sensor);

    if (NULL == sensor->probe_list) {
        return 0;
    }
    if (sensor->probe_list->len != 0 && out_probe_vec != NULL) {
        g_array_append_vals(out_probe_vec, sensor->probe_list->data,
                            sensor->probe_list->len);
    }
    return sensor->probe_list->len;
}

int
mdSkpcSensorSetProbes(
    md_skpc_sensor_t  *sensor,
    const GArray      *probe_vec)
{
    /* check input */
    assert(sensor);
    if (probe_vec == NULL || 0 == probe_vec->len) {
        return -1;
    }

    /* copy the values out of the vector */
    if (NULL == sensor->probe_list) {
        /* create a new array */
        sensor->probe_list = g_array_sized_new(FALSE, TRUE, sizeof(void *),
                                               probe_vec->len);
    }
    g_array_append_vals(sensor->probe_list, probe_vec->data, probe_vec->len);

    return 0;
}


gboolean
mdSkpcSensorVerify(
    md_skpc_sensor_t  *sensor)
{
    md_skpc_probe_t *probe;
    uint32_t         i;

    assert(sensor);

    if (sensor->sensor_id == SK_INVALID_SENSOR) {
        fprintf(stderr,
                ("Error verifying sensor '%s'\n"
                 "\tSensor is not defined in site file silk.conf\n"),
                sensor->sensor_name);
        return FALSE;
    }

    if (!mdSkpcVerifySensorLabeler(sensor)) {
        return FALSE;
    }

    /* if any network decider is set to 'remainder', update the sensor */
    if (mdSkpcSensorComputeRemainingInterfaces(sensor)) {
        return FALSE;
    }
    if (mdSkpcSensorComputeRemainingIpBlocks(sensor)) {
        return FALSE;
    }
    if (mdSkpcSensorComputeRemainingIpSets(sensor)) {
        return FALSE;
    }

    if (sensor->probe_list) {
        /* add a link on each probe to this sensor */
        for (i = 0; i < sensor->probe_list->len; ++i) {
            probe = g_array_index(sensor->probe_list, md_skpc_probe_t *, i);
            if (mdSkpcProbeAddSensor(probe, sensor)) {
                fprintf(stderr,
                        ("Error verifying sensor '%s':\n"
                         "\tCannot link probe '%s' to this sensor\n"),
                        sensor->sensor_name, probe->probe_name);
                return FALSE;
            }
        }
    }

    /* sensor is valid; add it to the global vector of sensors */
    g_array_append_val(md_skpc_sensors, sensor);

    return TRUE;
}


/*
 *  *****  Groups  ***********************************************************
 */

/* create a new group */
int
mdSkpcGroupCreate(
    md_skpc_group_t **group)
{
    md_skpc_group_t *g;

    assert(group);

    g = g_slice_new0(md_skpc_group_t);
    g->g_type = MD_SKPC_GROUP_UNSET;

    *group = g;

    return 0;
}

/* destroy a group */
void
mdSkpcGroupDestroy(
    md_skpc_group_t **group)
{
    if (!group || !(*group)) {
        return;
    }

    switch ((*group)->g_type) {
      case MD_SKPC_GROUP_UNSET:
        break;
      case MD_SKPC_GROUP_INTERFACE:
        if ((*group)->g_value.vec) {
            /* vector holds uint32_t */
            g_array_free((*group)->g_value.vec, TRUE);
            (*group)->g_value.vec = NULL;
        }
        break;
      case MD_SKPC_GROUP_IPBLOCK:
        /* vector holds skcidr_t* and ipcidr_list owns them */
        if ((*group)->g_value.vec) {
            g_array_free((*group)->g_value.vec, TRUE);
            (*group)->g_value.vec = NULL;
        }
        break;
      case MD_SKPC_GROUP_IPSET:
        /* vector hold skipset_t* unique to the group */
        if ((*group)->g_value.ipset) {
            skIPSetDestroy(&(*group)->g_value.ipset);
            (*group)->g_value.ipset = NULL;
        }
        break;
    }

    if ((*group)->g_name) {
        g_free((*group)->g_name);
        (*group)->g_name = NULL;
    }

    g_slice_free(md_skpc_group_t, *group);
    *group = NULL;
}


/* mark group as unchangeable, convert ipblock vector to array, and
 * count number of items  */
int
mdSkpcGroupFreeze(
    md_skpc_group_t  *group)
{
    uint64_t  ip_count;

    assert(group);
    if (group->g_is_frozen) {
        return 0;
    }

    switch (group->g_type) {
      case MD_SKPC_GROUP_UNSET:
        /* nothing else do */
        break;
      case MD_SKPC_GROUP_INTERFACE:
        group->g_itemcount = group->g_value.vec->len;
        break;
      case MD_SKPC_GROUP_IPBLOCK:
        group->g_itemcount = group->g_value.vec->len;
        break;
      case MD_SKPC_GROUP_IPSET:
        if (skIPSetClean(group->g_value.ipset)) {
            return -1;
        }
        ip_count = skIPSetCountIPs(group->g_value.ipset, NULL);
        if (ip_count > UINT32_MAX) {
            group->g_itemcount = UINT32_MAX;
        } else {
            group->g_itemcount = (uint32_t)ip_count;
        }
        break;
    }

    group->g_is_frozen = 1;
    g_array_append_val(md_skpc_groups, group);

    return 0;
}


/* get or set the name of the group */
const char *
mdSkpcGroupGetName(
    const md_skpc_group_t  *group)
{
    assert(group);
    return group->g_name;
}

int
mdSkpcGroupSetName(
    md_skpc_group_t  *group,
    const char       *name)
{
    const char *cp;
    char       *copy;

    assert(group);
    if (group->g_is_frozen) {
        return -1;
    }

    if (name == NULL || name[0] == '\0') {
        return -1;
    }

    /* check for illegal characters */
    cp = name;
    while (*cp) {
        if (*cp == '/' || isspace((int)*cp)) {
            return -1;
        }
        ++cp;
    }

    copy = g_strdup(name);
    if (group->g_name) {
        g_free(group->g_name);
    }
    group->g_name = copy;
    return 0;
}


/* get or set the type of the group */
md_skpc_group_type_t
mdSkpcGroupGetType(
    const md_skpc_group_t  *group)
{
    assert(group);
    return group->g_type;
}

int
mdSkpcGroupSetType(
    md_skpc_group_t       *group,
    md_skpc_group_type_t   group_type)
{
    assert(group);
    if (group->g_is_frozen) {
        return -1;
    }
    if (MD_SKPC_GROUP_UNSET != group->g_type) {
        return -1;
    }

    switch (group_type) {
      case MD_SKPC_GROUP_UNSET:
        return -1;

      case MD_SKPC_GROUP_INTERFACE:
        group->g_value.vec = g_array_sized_new(FALSE, TRUE, sizeof(uint32_t),
                                               DEFAULT_ARRAY_SIZE);
        break;

      case MD_SKPC_GROUP_IPBLOCK:
        group->g_value.vec = g_array_sized_new(FALSE, TRUE,
                                               sizeof(skcidr_t *),
                                               DEFAULT_ARRAY_SIZE);
        break;

      case MD_SKPC_GROUP_IPSET:
        if (skIPSetCreate(&group->g_value.ipset, 0)) {
            return -1;
        }
        break;
    }

    group->g_type = group_type;
    return 0;
}


/* add the values from 'vec' to 'group' */
int
mdSkpcGroupAddValues(
    md_skpc_group_t  *group,
    const GArray     *vec)
{
    size_t           count;
    size_t           i;
    const skipset_t *ipset;
    int              rv;

    assert(group);
    if (group->g_is_frozen) {
        return -1;
    }

    if (NULL == vec) {
        return 0;
    }

    count = vec->len;
    if (0 == count) {
        return 0;
    }

    switch (group->g_type) {
      case MD_SKPC_GROUP_UNSET:
        return -1;

      case MD_SKPC_GROUP_INTERFACE:
        /* check that vector has data of the correct type (size) */
        if (g_array_get_element_size((GArray *)vec) != sizeof(uint32_t)) {
            return -1;
        }
        g_array_append_vals(group->g_value.vec, vec->data, vec->len);
        break;

      case MD_SKPC_GROUP_IPBLOCK:
        /* check that vector has data of the correct type (size) */
        if (g_array_get_element_size((GArray *)vec) != sizeof(skcidr_t *)) {
            return -1;
        }
        /* add CIDR blocks to the group */
        g_array_append_vals(group->g_value.vec, vec->data, vec->len);
        /* store the CIDR in md_skpc_cidrs for memory cleanup */
        if (!md_skpc_cidrs) {
            md_skpc_cidrs = g_array_sized_new(FALSE, TRUE,
                                              sizeof(skcidr_t *),
                                              MAX(DEFAULT_ARRAY_SIZE,
                                                  vec->len));
        }
        g_array_append_vals(md_skpc_cidrs, vec->data, vec->len);
        break;

      case MD_SKPC_GROUP_IPSET:
        /* check that vector has data of the correct type (size) */
        if (g_array_get_element_size((GArray *)vec) != sizeof(skipset_t *)) {
            return -1;
        }
        for (i = 0; i < count; ++i) {
            ipset = g_array_index(vec, skipset_t *, i);
            rv = skIPSetUnion(group->g_value.ipset, ipset);
            if (rv) {
                fprintf(stderr, "%s: Memory allocation error\n",
                        g_get_prgname());
                exit(-1);
            }
        }
        rv = skIPSetClean(group->g_value.ipset);
        if (rv) {
            return -1;
        }
        break;
    }

    return 0;
}


/* add the values from the existing group 'g' to group 'group' */
int
mdSkpcGroupAddGroup(
    md_skpc_group_t        *group,
    const md_skpc_group_t  *g)
{
    assert(group);
    if (group->g_is_frozen) {
        return -1;
    }

    /* verify group 'g' */
    if (NULL == g) {
        return 0;
    }
    if (!g->g_is_frozen) {
        return -1;
    }
    if (0 == g->g_itemcount) {
        return 0;
    }

    if (g->g_type != group->g_type) {
        return -1;
    }

    switch (group->g_type) {
      case MD_SKPC_GROUP_UNSET:
        return -1;

      case MD_SKPC_GROUP_INTERFACE:
        g_array_append_vals(group->g_value.vec, g->g_value.vec->data,
                            g->g_value.vec->len);
        break;

      case MD_SKPC_GROUP_IPBLOCK:
        g_array_append_vals(group->g_value.vec, g->g_value.vec->data,
                            g->g_value.vec->len);
        break;

      case MD_SKPC_GROUP_IPSET:
        if (skIPSetUnion(group->g_value.ipset, g->g_value.ipset)) {
            return -1;
        }
        if (skIPSetClean(group->g_value.ipset)) {
            return -1;
        }
        break;
    }

    return 0;
}


/* is group frozen? */
int
mdSkpcGroupIsFrozen(
    const md_skpc_group_t  *group)
{
    assert(group);
    return group->g_is_frozen;
}


/* find the group named 'group_name' */
md_skpc_group_t *
mdSkpcGroupLookupByName(
    const char  *group_name)
{
    md_skpc_group_t *group;
    size_t           i;

    assert(md_skpc_groups);

    /* check input */
    if (group_name == NULL) {
        return NULL;
    }

    /* loop over all groups until we find one with given name */
    for (i = 0; i < md_skpc_groups->len; ++i) {
        group = g_array_index(md_skpc_groups, md_skpc_group_t *, i);
        if ((NULL != group->g_name)
            && (0 == strcmp(group_name, group->g_name)))
        {
            if (mdSkpcGroupFreeze(group)) {
                return NULL;
            }
            return group;
        }
    }

    return NULL;
}


/*
 *  found = mdSkpcGroupCheckInterface(group, interface);
 *
 *    Return 1 if 'group' contains the value 'interface'; 0 otherwise.
 */
static int
mdSkpcGroupCheckInterface(
    const md_skpc_group_t  *group,
    uint32_t                interface)
{
    size_t  i;

    assert(group->g_type == MD_SKPC_GROUP_INTERFACE);
    for (i = 0; i < group->g_value.vec->len; ++i) {
        if (g_array_index(group->g_value.vec, uint32_t, i) == interface) {
            return 1;
        }
    }
    return 0;
}


/*
 *  found = mdSkpcGroupCheckIPblock(group, ip);
 *
 *    Return 1 if 'group' contains the IP Address 'ip'; 0 otherwise.
 */
static int
mdSkpcGroupCheckIPblock(
    const md_skpc_group_t  *group,
    const skipaddr_t       *ip)
{
    const skcidr_t *cidr;
    size_t          i;

    assert(group->g_type == MD_SKPC_GROUP_IPBLOCK);
    for (i = 0; i < group->g_value.vec->len; ++i) {
        cidr = g_array_index(group->g_value.vec, const skcidr_t *, i);
        if (skcidrCheckIP(cidr, ip)) {
            return 1;
        }
    }
    return 0;
}


/*
 *  found = mdSkpcGroupCheckIPset(group, ip);
 *
 *    Return 1 if 'group' contains the IP Address 'ip'; 0 otherwise.
 */
static int
mdSkpcGroupCheckIPset(
    const md_skpc_group_t  *group,
    const skipaddr_t       *ip)
{
    assert(group->g_type == MD_SKPC_GROUP_IPSET);
    return skIPSetCheckAddress(group->g_value.ipset, ip);
}


/*
 *  count = mdSkpcGroupGetItemCount(group);
 *
 *    Return the count of the number of items in 'group'.  Returns 0
 *    if 'group' is not frozen.
 */
static uint32_t
mdSkpcGroupGetItemCount(
    const md_skpc_group_t  *group)
{
    assert(group);
    return group->g_itemcount;
}


/*
 *  *****  Probes Types  *****************************************************
 */

/* return an enum value given a probe type name */
md_skpc_probetype_t
mdSkpcProbetypeNameToEnum(
    const char  *name)
{
    const struct probe_type_name_map_st *entry;

    if (name) {
        for (entry = probe_type_name_map; entry->name; ++entry) {
            if (0 == strcmp(name, entry->name)) {
                return entry->value;
            }
        }
    }
    return MD_PROBE_ENUM_INVALID;
}


/* return the name given a probe type number */
const char *
mdSkpcProbetypeEnumtoName(
    md_skpc_probetype_t   type)
{
    const struct probe_type_name_map_st *entry;

    for (entry = probe_type_name_map; entry->name; ++entry) {
        if (type == entry->value) {
            return entry->name;
        }
    }
    return NULL;
}


/* return a name given a group type */
const char *
mdSkpcGrouptypeEnumtoName(
    md_skpc_group_type_t   type)
{
    switch (type) {
      case MD_SKPC_GROUP_INTERFACE:
        return "interface";
      case MD_SKPC_GROUP_IPBLOCK:
        return "ipblock";
      case MD_SKPC_GROUP_IPSET:
        return "ipset";
      case MD_SKPC_GROUP_UNSET:
        break;
    }
    return NULL;
}
