/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_util.c
 *
 *  Contains the basic utility functions for super_mediator
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.11
 *
 *  Copyright 2024 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM24-1038
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_util.h"
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#if ENABLE_SKIPSET && HAVE_SILK_UTILS_H
#include <silk/utils.h>
#endif

#if !GLIB_CHECK_VERSION(2, 68, 0)
/* g_memdup2() uses gsize in place of guint */
#define g_memdup2(_mem, _size)   g_memdup((_mem), (_size))
#endif

#define MD_COMPRESSOR "gzip"

uint32_t
hashword(
    const uint32_t  *k,
    size_t           length,
    uint32_t         initval);
void
hashword2(
    const uint32_t  *k,
    size_t           length,
    uint32_t        *pc,
    uint32_t        *pb);
uint32_t
hashlittle(
    const void  *key,
    size_t       length,
    uint32_t     initval);
void
hashlittle2(
    const void  *key,
    size_t       length,
    uint32_t    *pc,
    uint32_t    *pb);
uint32_t
hashbig(
    const void  *key,
    size_t       length,
    uint32_t     initval);

#include "lookup3.c"


static ssize_t
md_util_time_helper(
    char                *buf,
    size_t               len,
    const yfTime_t      *t,
    mdTimePrintFormat_t  format);

/**
 *    Size to use when growing mdBuf_t.
 */
#define MDBUF_CHUNK  4096

/**
 *    Minimum mdBuf_t size
 */
#define MDBUF_MIN    1024

/**
 *    "Padding" to add to end of the buffer by subtracting this value when
 *    computing that number of remaining octets.
 */
#define MDBUF_PAD    2

/**
 *    Return TRUE if `md_buf` has space to hold `required` octets.
 */
#define MDBUF_CHECK(md_buf, required)           \
    ((size_t)(required) < (md_buf)->b_rem)

/**
 *    Grow `mdBuf` until it is capable of holding `required` ADDITIONAL
 *    octets.
 */
static void
mdBufGrow(
    mdBuf_t    *mdBuf,
    size_t      required)
{
    /* record length so can re-position b_cp */
    size_t len = mdBufLen(mdBuf);

    while (!MDBUF_CHECK(mdBuf, required)) {
        mdBuf->b_buflen += MDBUF_CHUNK;
        mdBuf->b_rem += MDBUF_CHUNK;
    }

    mdBuf->b_buf = g_renew(char, mdBuf->b_buf, mdBuf->b_buflen);
    mdBuf->b_cp = &mdBuf->b_buf[len];
}

/**
 *    Adjust the current position and the remainder so the current length is
 *    `len`.  Assumes `len` is less than the capacity.
 */
static void
mdBufSetLen(
    mdBuf_t    *mdBuf,
    size_t      len)
{
    mdBuf->b_rem = mdBuf->b_buflen - len - MDBUF_PAD;
    mdBuf->b_cp = &mdBuf->b_buf[len];
    *mdBuf->b_cp = '\0';
}


mdBuf_t *
mdBufNew(
    size_t  initial)
{
    mdBuf_t  *mdBuf = g_new(mdBuf_t, 1);

    mdBuf->b_buflen = ((initial) ? initial : MDBUF_CHUNK);
    mdBuf->b_buf = g_new0(char, mdBuf->b_buflen);

    mdBufSetLen(mdBuf, 0);

    return mdBuf;
}

void
mdBufAppend(
    mdBuf_t    *mdBuf,
    const char *str)
{
    if (str) {
        mdBufAppendLen(mdBuf, (const uint8_t *)str, strlen(str));
    }
}

void
mdBufAppendBase64(
    mdBuf_t        *mdBuf,
    const uint8_t  *data,
    size_t          datalen)
{
    if (datalen) {
        gchar *b64 = g_base64_encode(data, datalen);
        mdBufAppendLen(mdBuf, (uint8_t *)b64, strlen(b64));
        g_free(b64);
    }
}

void
mdBufAppendChar(
    mdBuf_t    *mdBuf,
    char        c)
{
    if (!MDBUF_CHECK(mdBuf, 2)) {
        mdBufGrow(mdBuf, MDBUF_CHUNK);
    }

    --mdBuf->b_rem;
    *mdBuf->b_cp = c;
    ++mdBuf->b_cp;
    *mdBuf->b_cp = '\0';
}

void
mdBufAppendGString(
    mdBuf_t        *mdBuf,
    const GString  *str)
{
    mdBufAppendLen(mdBuf, (const uint8_t *)str->str, str->len);
}

void
mdBufAppendHexdump(
    mdBuf_t        *mdBuf,
    const uint8_t  *src,
    size_t          len,
    gboolean        no_space)
{
    size_t i;

    if (0 == len) {
        return;
    }

    mdBufAppendPrintf(mdBuf, "%s%02hhx", (no_space ? "0x" : ""), src[0]);
    if (no_space) {
        for (i = 1; i < len; ++i) {
            mdBufAppendPrintf(mdBuf, "%02hhx", src[i]);
        }
    } else {
        for (i = 1; i < len; ++i) {
            mdBufAppendPrintf(mdBuf, " %02hhx", src[i]);
        }
    }
}

void
mdBufAppendIP4(
    mdBuf_t        *mdBuf,
    uint32_t        ip4)
{
    mdBufAppendPrintf(mdBuf, "%hhu.%hhu.%hhu.%hhu",
                      ((ip4 >> 24) & 0xff), ((ip4 >> 16) & 0xff),
                      ((ip4 >> 8) & 0xff), (ip4 & 0xff));
}

void
mdBufAppendIP6(
    mdBuf_t        *mdBuf,
    const uint8_t  *ip6)
{
    char ipstr[64];

    md_util_print_ip6_addr(ipstr, ip6);
    mdBufAppendLen(mdBuf, (uint8_t *)ipstr, strlen(ipstr));
}

void
mdBufAppendLen(
    mdBuf_t        *mdBuf,
    const uint8_t  *str,
    size_t          len)
{
    if (0 == len) {
        return;
    }

    if (!MDBUF_CHECK(mdBuf, len)) {
        mdBufGrow(mdBuf, len);
    }

    memcpy(mdBuf->b_cp, str, len);
    mdBuf->b_rem -= len;
    mdBuf->b_cp += len;
    *mdBuf->b_cp = '\0';
}


void
mdBufAppendPrintf(
    mdBuf_t    *mdBuf,
    const char *format,
    ...)
{
    va_list   ap;
    ssize_t   sz;

    for (;;) {
        va_start(ap, format);
        sz = vsnprintf(mdBuf->b_cp, mdBuf->b_rem, format, ap);
        va_end(ap);
        if (MDBUF_CHECK(mdBuf, sz)) {
            mdBuf->b_rem -= sz;
            mdBuf->b_cp += sz;
            return;
        }

        mdBufGrow(mdBuf, sz);
    }
}

void
mdBufAppendTime(
    mdBuf_t             *mdBuf,
    const yfTime_t      *t,
    mdTimePrintFormat_t  format)
{
    char    timestr[256];
    ssize_t ret;

    ret = md_util_time_helper(timestr, sizeof(timestr), t, format);
    mdBufAppendLen(mdBuf, (uint8_t *)timestr, ret);
}

void
mdBufAppendVarfield(
    mdBuf_t            *mdBuf,
    const fbVarfield_t *vf)
{
    mdBufAppendLen(mdBuf, vf->buf, vf->len);
}

//void
//mdBufAssign(
//    mdBuf_t    *mdBuf,
//    const char *format,
//    ...)

void
mdBufChop(
    mdBuf_t    *mdBuf,
    size_t      num_to_remove)
{
    mdBufSetLen(mdBuf, ((num_to_remove < mdBufLen(mdBuf))
                        ? (mdBufLen(mdBuf) - num_to_remove) : 0));
}

void
mdBufClear(
    mdBuf_t    *mdBuf)
{
    mdBufSetLen(mdBuf, 0);
}

void
mdBufFree(
    mdBuf_t    *mdBuf)
{
    if (mdBuf) {
        g_free(mdBuf->b_buf);
        g_free(mdBuf);
    }
}

char
mdBufGetChar(
    const mdBuf_t  *mdBuf,
    size_t          pos)
{
    if (pos < mdBufLen(mdBuf)) {
        return mdBuf->b_buf[pos];
    }
    return '\0';
}

char *
mdBufGetSubstring(
    const mdBuf_t  *mdBuf,
    size_t          start_pos,
    size_t          end_pos)
{
    size_t last_pos = mdBufLen(mdBuf);
    char *out;

    if (start_pos > end_pos) {
        return NULL;
    }
    out = g_new0(char, 1 + end_pos - start_pos);
    if (start_pos < last_pos) {
        memcpy(out, &mdBuf->b_buf[start_pos],
               (MIN(end_pos, last_pos) - start_pos));
    }
    return out;
}

char *
mdBufMakeCopy(
    const mdBuf_t  *mdBuf,
    size_t         *len)
{
    if (len) {
        *len = mdBufLen(mdBuf);
    }
    return g_memdup2(mdBuf->b_buf, 1 + mdBufLen(mdBuf));
}

void
mdBufTruncate(
    mdBuf_t    *mdBuf,
    size_t      len)
{
    if (len < mdBufLen(mdBuf)) {
        mdBufSetLen(mdBuf, len);
    }
}

size_t
mdBufWrite(
    mdBuf_t     *mdBuf,
    FILE        *fp,
    const char  *exporter_name,
    GError     **err)
{
    size_t  buflen = mdBuf->b_cp - mdBuf->b_buf;
    size_t  rc;

    rc = fwrite(mdBuf->b_buf, 1, buflen, fp);
    if (rc != buflen) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                    "%s: Error writing to file: %s\n",
                    exporter_name, strerror(errno));
        return 0;
    }

    /* clear the buffer */
    mdBufSetLen(mdBuf, 0);

    return rc;
}



/**
 * md_util_hexdump_append
 *
 */
int
md_util_hexdump_append(
    char           *dest,
    size_t         *drem,
    const uint8_t  *src,
    size_t          len)
{
    size_t  i = 0;
    int     r;
    int     tot = 0;

    if (len) {
        /* first one shouldn't have a space */
        r = snprintf(dest + tot, *drem, "%02hhx", src[i]);
        if ((size_t)r < *drem) {
            *drem -= r;
        } else {
            return 0;
        }
        tot += r;
    }

    for (i = 1; i < len; i++) {
        r = snprintf(dest + tot, *drem, " %02hhx", src[i]);
        if ((size_t)r < *drem) {
            *drem -= r;
        } else {
            return 0;
        }
        tot += r;
    }

    return tot;
}

/**
 * md_util_hexdump_append_nospace
 *
 */
int
md_util_hexdump_append_nospace(
    char           *dest,
    size_t         *drem,
    const uint8_t  *src,
    size_t          len)
{
    int  i = 0;
    int  r;
    int  tot = 0;

    if (len) {
        r = snprintf(dest + tot, *drem, "0x%02hhx", src[i]);
        if ((size_t)r < *drem) {
            *drem -= r;
        } else {
            return 0;
        }
        tot += r;
    }

    for (i = 1; i < (int)len; i++) {
        r = snprintf(dest + tot, *drem, "%02hhx", src[i]);
        if ((size_t)r < *drem) {
            *drem -= r;
        } else {
            return 0;
        }
        tot += r;
    }

    return tot;
}


/**
 * md_util_hexdump_g_string_append_line
 *
 * stolen from airframe to print yaf payloads
 *
 */
static uint32_t
md_util_hexdump_g_string_append_line(
    GString        *str,
    const char     *lpfx,
    const uint8_t  *cp,
    uint32_t        lineoff,
    uint32_t        buflen)
{
    uint32_t  cwr = 0, twr = 0;

    /* stubbornly refuse to print nothing */
    if (!buflen) {
        return 0;
    }

    /* print line header */
    g_string_append_printf(str, "%s %04x:", lpfx, lineoff);

    /* print hex characters */
    if (buflen >= 16) {
        g_string_append_printf(
            str,
            " %02hhx %02hhx %02hhx %02hhx %02hhx %02hhx %02hhx %02hhx"
            " %02hhx %02hhx %02hhx %02hhx %02hhx %02hhx %02hhx %02hhx",
            cp[ 0], cp[ 1], cp[ 2], cp[ 3], cp[ 4], cp[ 5], cp[ 6], cp[ 7],
            cp[ 8], cp[ 9], cp[10], cp[11], cp[12], cp[13], cp[14], cp[15]);
        cwr += 16; buflen -= 16;
    } else {
        for (twr = 0; twr < 16; twr++) {
            if (buflen) {
                g_string_append_printf(str, " %02hhx", cp[twr]);
                cwr++; buflen--;
            } else {
                g_string_append(str, "   ");
            }
        }
    }

    /* print characters */
    g_string_append_c(str, ' ');
    for (twr = 0; twr < cwr; twr++) {
        if (cp[twr] >= ' ' && cp[twr] <= '~') {
            g_string_append_c(str, cp[twr]);
        } else {
            g_string_append_c(str, '.');
        }
    }
    g_string_append_c(str, '\n');

    return cwr;
}

/**
 * md_util_hexdump_g_string_append
 *
 * stolen from airframe to print hex
 *
 */
void
md_util_hexdump_g_string_append(
    GString        *str,
    const char     *lpfx,
    const uint8_t  *buf,
    uint32_t        len)
{
    uint32_t  cwr = 0, lineoff = 0;

    do {
        cwr = md_util_hexdump_g_string_append_line(str, lpfx, buf, lineoff,
                                                   len);
        buf += cwr; len -= cwr; lineoff += cwr;
    } while (cwr == 16);
}

/**
 * md_util_print_tcp_flags
 *
 * prints TCP flags
 *
 */
void
md_util_print_tcp_flags(
    GString  *str,
    uint8_t   flags)
{
    if (flags & 0x40) {g_string_append_c(str, 'E');}
    if (flags & 0x80) {g_string_append_c(str, 'C');}
    if (flags & 0x20) {g_string_append_c(str, 'U');}
    if (flags & 0x10) {g_string_append_c(str, 'A');}
    if (flags & 0x08) {g_string_append_c(str, 'P');}
    if (flags & 0x04) {g_string_append_c(str, 'R');}
    if (flags & 0x02) {g_string_append_c(str, 'S');}
    if (flags & 0x01) {g_string_append_c(str, 'F');}
    if (!flags) {g_string_append_c(str, '0');}
}

/**
 * md_util_print_ip6_addr
 *
 *
 */
void
md_util_print_ip6_addr(
    char           *ipaddr_buf,
    const uint8_t  *ipaddr)
{
    char     *cp = ipaddr_buf;
    uint16_t *aqp = (uint16_t *)ipaddr;
    uint16_t  aq;
    gboolean  colon_start = FALSE;
    gboolean  colon_end = FALSE;

    for (; (uint8_t *)aqp < ipaddr + 16; aqp++) {
        aq = g_ntohs(*aqp);
        if (aq || colon_end) {
            if ((uint8_t *)aqp < ipaddr + 14) {
                snprintf(cp, 6, "%04hx:", aq);
                cp += 5;
            } else {
                snprintf(cp, 5, "%04hx", aq);
                cp += 4;
            }
            if (colon_start) {
                colon_end = TRUE;
            }
        } else if (!colon_start) {
            if ((uint8_t *)aqp == ipaddr) {
                snprintf(cp, 3, "::");
                cp += 2;
            } else {
                snprintf(cp, 2, ":");
                cp += 1;
            }
            colon_start = TRUE;
        }
    }
}


/**
 * md_util_print_ip4_addr
 *
 *
 */
void
md_util_print_ip4_addr(
    char      *ipaddr_buf,
    uint32_t   ip)
{
    uint32_t  mask = 0xff000000U;
    uint8_t   dqp[4];

    /* split the address */
    dqp[0] = (ip & mask) >> 24;
    mask >>= 8;
    dqp[1] = (ip & mask) >> 16;
    mask >>= 8;
    dqp[2] = (ip & mask) >> 8;
    mask >>= 8;
    dqp[3] = (ip & mask);

    /* print to it */
    snprintf(ipaddr_buf, 16, "%hhu.%hhu.%hhu.%hhu",
             dqp[0], dqp[1], dqp[2], dqp[3]);
}


/**
 * md_util_flow_key_hash
 *
 *
 */
uint32_t
md_util_flow_key_hash(
    md_main_template_t  *rec)
{
    uint32_t  hash = 0;
    uint32_t *v6p;

    if (rec->sourceIPv4Address || rec->destinationIPv4Address) {
        hash = ((rec->sourceTransportPort << 16) ^
                (rec->destinationTransportPort) ^
                (rec->protocolIdentifier << 12) ^ (4 << 4) ^
                (rec->vlanId << 20) ^ (rec->sourceIPv4Address) ^
                (rec->destinationIPv4Address));
        return hash;
    } else {
        v6p = (uint32_t *)rec->sourceIPv6Address;
        hash = ((rec->sourceTransportPort << 16) ^
                (rec->destinationTransportPort) ^
                (rec->protocolIdentifier << 12) ^ (6 << 4) ^
                (rec->vlanId << 20) ^ *v6p);
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p = (uint32_t *)rec->destinationIPv6Address;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        return hash;
    }
}

uint32_t
md_util_rev_flow_key_hash(
    md_main_template_t  *rec)
{
    uint32_t  hash = 0;
    uint32_t *v6p;

    if (rec->sourceIPv4Address || rec->destinationIPv4Address) {
        hash = ((rec->destinationTransportPort << 16) ^
                (rec->sourceTransportPort) ^
                (rec->protocolIdentifier << 12) ^ (4 << 4) ^
                (rec->vlanId << 20) ^ (rec->destinationIPv4Address) ^
                (rec->sourceIPv4Address));
        return hash;
    } else {
        v6p = (uint32_t *)rec->destinationIPv6Address;
        hash = ((rec->destinationTransportPort << 16) ^
                (rec->sourceTransportPort) ^
                (rec->protocolIdentifier << 12) ^ (6 << 4) ^
                (rec->vlanId << 20) ^ *v6p);
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p = (uint32_t *)rec->sourceIPv6Address;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        v6p++;
        hash ^= *v6p;
        return hash;
    }
}


/*
 *    Fills `buf`, having length `len`, with the representation of the time
 *    `t` as specified by `format`.  Returns the number of characters the
 *    function were written to `buf` (or that it attempted to write) not
 *    counting the trailing NUL.
 *
 *    A helper function for md_util_time_g_string_append() and
 *    mdBufAppendTime().
 */
static ssize_t
md_util_time_helper(
    char                *buf,
    size_t               len,
    const yfTime_t      *t,
    mdTimePrintFormat_t  format)
{
    /*
     *    Formats `time_tm` using "%04u-%02u-%02u %02u:%02u:%02u" with
     *    `frac_fmt` appended to the format string using `frac_value` as the
     *    final argument which should match `frac_fmt`.
     *
     *    Uses local variables time_fm, buf, and len
     */
#define FORMAT_TIME_BASE(frac_fmt, frac_value)                          \
    snprintf(buf, len, ("%04u-%02u-%02u %02u:%02u:%02u" frac_fmt),      \
             (time_tm.tm_year + 1900), (time_tm.tm_mon + 1),            \
             time_tm.tm_mday,                                           \
             time_tm.tm_hour, time_tm.tm_min, time_tm.tm_sec,           \
             frac_value)

    struct tm        time_tm;
    struct timespec  tspec;
    ssize_t          sz;

    yfTimeToTimespec(&tspec, *t);
    gmtime_r(&tspec.tv_sec, &time_tm);

    switch (format) {
      case MD_TIME_FMT_FILENAME:
        /* "yyyymmddHHMMSS" */
        sz = snprintf(buf, len, "%04u%02u%02u%02u%02u%02u",
                      time_tm.tm_year + 1900, time_tm.tm_mon + 1,
                      time_tm.tm_mday,
                      time_tm.tm_hour, time_tm.tm_min, time_tm.tm_sec);
        break;
      case MD_TIME_FMT_TIMEONLY:
        /* TIME ONLY "HH:MM:SS" */
        sz = snprintf(buf, len, "%02u:%02u:%02u",
                      time_tm.tm_hour, time_tm.tm_min, time_tm.tm_sec);
        break;
      case MD_TIME_FMT_NOFRAC:
        /* append an empty string to the datetimestamp */
        sz = FORMAT_TIME_BASE("%s", "");
        break;
      case MD_TIME_FMT_MILLI:
        sz = FORMAT_TIME_BASE(".%03ld", tspec.tv_nsec / 1000000);
        break;
      case MD_TIME_FMT_MICRO:
        sz = FORMAT_TIME_BASE(".%06ld", tspec.tv_nsec / 1000);
        break;
      case MD_TIME_FMT_NANO:
        sz = FORMAT_TIME_BASE(".%09ld", tspec.tv_nsec);
        break;
      default:
        g_error("Unknown time format %d", format);
    }

    return sz;

#undef FORMAT_TIME_BASE
}

/**
 *
 * add a formated time string to the str.
 *
 */
void
md_util_time_g_string_append(
    GString             *str,
    const yfTime_t      *t,
    mdTimePrintFormat_t  format)
{
    char timebuf[256];

    md_util_time_helper(timebuf, sizeof(timebuf), t, format);
    g_string_append(str, timebuf);
}

uint16_t
md_util_decode_length(
    uint8_t   *buffer,
    uint16_t  *offset)
{
    uint16_t  obj_len;

    obj_len = *(buffer + *offset);
    if (obj_len == 0x81) {
        (*offset)++;
        obj_len = *(buffer + *offset);
    } else if (obj_len == 0x82) {
        (*offset)++;
        obj_len = ntohs(*(uint16_t *)(buffer + *offset));
        (*offset)++;
    }

    return obj_len;
}



uint16_t
md_util_decode_tlv(
    md_asn_tlv_t  *tlv,
    uint8_t       *buffer,
    uint16_t      *offset)
{
    uint8_t   val = *(buffer + *offset);
    uint16_t  len = 0;

    tlv->class = (val & 0xD0) >> 6;
    tlv->p_c = (val & 0x20) >> 5;
    tlv->tag = (val & 0x1F);

    (*offset)++;

    len = md_util_decode_length(buffer, offset);
    (*offset)++;

    if (tlv->tag == 0x05) { /*CERT_NULL 0x05 */
        *offset += len;
        return md_util_decode_tlv(tlv, buffer, offset);
    }

    return len;
}



uint16_t
md_util_decode_asn1_length(
    uint8_t **buffer,
    size_t   *len)
{
    uint16_t  obj_len;

    obj_len = **buffer;

    if (obj_len == 0x81) {
        (*buffer)++;
        obj_len = (uint16_t)**buffer;
        (*buffer)++;
        *len -= 2;
    } else if (obj_len == 0x82) {
        (*buffer)++;
        obj_len = ntohs(*(uint16_t *)(*buffer));
        (*buffer) += 2;
        *len -= 3;
    } else if ((obj_len & 0x80) == 0) {
        /* first byte describes length */
        obj_len = (uint16_t)**buffer;
        (*buffer)++;
        *len -= 1;
    }

    return obj_len;
}

uint8_t
md_util_asn1_sequence_count(
    uint8_t   *buffer,
    uint16_t   seq_len)
{
    uint16_t      offsetptr = 0;
    uint16_t      len = 0;
    uint16_t      obj_len;
    uint8_t       count = 0;
    md_asn_tlv_t  tlv;

    obj_len = md_util_decode_tlv(&tlv, buffer, &offsetptr);
    while (tlv.tag == 0x11 && len < seq_len) {
        len += obj_len + 2;
        count++;
        offsetptr += obj_len;
        obj_len = md_util_decode_tlv(&tlv, buffer, &offsetptr);
    }

    return count;
}



/* moves buffer to next item and returns length
 */
uint16_t
md_util_decode_asn1_sequence(
    uint8_t **buffer,
    size_t   *len)
{
    uint8_t   val = **buffer;
    uint16_t  newlen = 0;

    if (*len == 0) {
        return 0;
    }

    if (val == 0x30) {
        (*buffer)++;
        *len -= 1;
        newlen = md_util_decode_asn1_length(buffer, len);
    }

    if (newlen > *len) {
        return 0;
    }

    val = **buffer;
    if ((val & 0x80) == 0x80) {
        (*buffer)++;
        *len -= 1;
        newlen = md_util_decode_asn1_length(buffer, len);
    } else if (val == 0x30) {
        /* sequence of sequence */
        (*buffer)++;
        *len -= 1;
        newlen = md_util_decode_asn1_length(buffer, len);
    }

    return newlen;
}

/**
 *  Function: attachHeadToDLL
 *  Description: attach a new entry to the head of a doubly
 *      linked list
 *  Params: **head - double pointer to the head of the DLL.  The
 *                head will point to the new head at the end.
 *          **tail - double pointer to the tail of the DLL.
 *                NULL if tail not used
 *          *newEntry - a pointer to the entry to add as the new head
 *  Return:
 */
void
attachHeadToDLL(
    mdDLL_t **head,
    mdDLL_t **tail,
    mdDLL_t  *newEntry)
{
    assert(newEntry);
    assert(head);

    /*  if this is NOT the first entry in the list */
    if (*head) {
        /*  typical linked list attachements */
        newEntry->next = *head;
        newEntry->prev = NULL;
        (*head)->prev = newEntry;
        *head = newEntry;
    } else {
        /*  the new entry is the only entry now, set head to it */
        *head = newEntry;
        newEntry->prev = NULL;
        newEntry->next = NULL;
        /*  if we're keeping track of tail, assign that too */
        if (tail) {
            *tail = newEntry;
        }
    }
}

/**
 * detachFromEndOfDLL
 *
 * detach a node from the end of a doubly linked list
 *
 */
void *
detachFromEndOfDLL(
    mdDLL_t **head,
    mdDLL_t **tail)
{
    mdDLL_t *node = NULL;

    assert(head);
    assert(tail);

    node = *tail;

    if (*tail) {
        *tail = (*tail)->prev;
        if (*tail) {
            (*tail)->next = NULL;
        } else {
            *head = NULL;
        }
    }

    return node;
}

/**
 * detachThisEntryOfDLL
 *
 * detach this specific node of the DLL
 *
 */
void
detachThisEntryOfDLL(
    mdDLL_t **head,
    mdDLL_t **tail,
    mdDLL_t  *entry)
{
    assert(entry);
    assert(head);

    /*  entry already points to the entry to remove, so we're good
     *  there */
    /*  if it's NOT the head of the list, patch up entry->prev */
    if (entry->prev != NULL) {
        entry->prev->next = entry->next;
    } else {
        /*  if it's the head, reassign the head */
        *head = entry->next;
    }
    /*  if it's NOT the tail of the list, patch up entry->next */
    if (entry->next != NULL) {
        entry->next->prev = entry->prev;
    } else {
        /*  it is the last entry in the list, if we're tracking the
         *  tail, reassign */
        if (tail) {
            *tail = entry->prev;
        }
    }

    /*  finish detaching by setting the next and prev pointers to
     *  null */
    entry->prev = NULL;
    entry->next = NULL;
}

/**
 * Hash Functions
 *
 *
 */
guint
sm_octet_array_hash(
    gconstpointer   v)
{
    smVarHashKey_t *key = (smVarHashKey_t *)v;
    uint32_t        h = 0;
    uint16_t        i = 0;

    if (key->len == 0) {
        return 0;
    }

    h = key->val[0];
    for (i = 1; i < key->len; i++) {
        h = (h << 5) - h + key->val[i];
    }

    return h;
}

gboolean
sm_octet_array_equal(
    gconstpointer   v1,
    gconstpointer   v2)
{
    smVarHashKey_t *var1 = (smVarHashKey_t *)v1;
    smVarHashKey_t *var2 = (smVarHashKey_t *)v2;

    if (var1->len != var2->len) {
        return FALSE;
    }

    if (memcmp(var1->val, var2->val, var1->len) == 0) {
        return TRUE;
    }

    return FALSE;
}

void
sm_octet_array_key_destroy(
    gpointer   data)
{
    smVarHashKey_t *key = data;

    if (data) {
        g_slice_free1(key->len, key->val);
        g_slice_free(smVarHashKey_t, key);
    }
}

smVarHashKey_t *
sm_new_hash_key(
    uint8_t  *val,
    size_t    len)
{
    smVarHashKey_t *key = g_slice_new0(smVarHashKey_t);

    key->val = g_slice_alloc0(len);
    memcpy(key->val, val, len);
    key->len = len;

    return key;
}


/*
 * compress and optionally relocate the compressed file
 * parameters:
 *   file - the input file name for the compressor
 *   dest - the path to the destination directory of the
 *          compressed file
 */
void
md_util_compress_file(
    const char  *file,
    const char  *dest)
{
    pid_t    pid;
    int      status = 0;
    GString *new_name = NULL;
    GString *mv_name = NULL;  /* allocated by sm_util_move_file */

#ifndef MD_COMPRESSOR
    g_warning("gzip is not defined - will not compress file");
    return;
#endif

    /* fork a child to spawn a completely detached gzip process.
     * Monitor the child until it has successfully forked the detached child.
     */
    pid = fork();
    if (pid == -1) {
        g_warning("Could not fork for %s command: %s", MD_COMPRESSOR,
                  strerror(errno));
        return;
    }

    /* In parent, top-level SM process, wait until child has forked then
     * return */
    if (pid != 0) {
        waitpid(pid, NULL, 0);
        return;
    }

    setpgid(0, 0);

    /* Create the grandchild which will be completely detached from the parent
     * SM process.
     */
    pid = fork();
    if (pid == -1) {
        g_warning("Child could not fork for %s command: %s\n",
                  MD_COMPRESSOR, strerror(errno));
        _exit(EXIT_FAILURE);
    }
    /* in the child, exit immediately so top-level SM may resume operation */
    if (pid != 0) {
        _exit(EXIT_SUCCESS);
    }
    /*
     * If we are moving the file, we have to wait for compression to be
     * completed.  Since we don't want to hold up the primary SM process, we
     * can't wait for it in the child process, and since execlp will replace
     * the forked process we must create ANOTHER forked process where we can
     * compress, monitor, and move
     */
    if (dest != NULL) {
        /* fork compress and move */
        pid = fork();
        if (pid == -1) {
            g_warning("Could not fork for %s and move: %s",
                      MD_COMPRESSOR, strerror(errno));
            return;
        }

        /* In the grandchild process, wait for grandgrandchild (gzip) to exit,
         * and check status */
        if (pid != 0) {
            waitpid(pid, &status, 0);
            if (WIFEXITED(status) && WEXITSTATUS(status) == 0) {
                /* compression child exited with success, move the compressed
                 * output */
                new_name = g_string_sized_new(PATH_MAX);
                g_string_append_printf(new_name, "%s.gz", file);
                mv_name = sm_util_move_file(new_name->str, dest);
                if (!mv_name) {
                    g_warning("Unable to move file %s to %s",
                              new_name->str, dest);
                }
                g_string_free(new_name, TRUE);
                g_string_free(mv_name, TRUE);
                _exit(EXIT_SUCCESS);
            } else {
                g_warning("Abnormal termination of gzip process,"
                          " not moving output file %s", file);
                _exit(EXIT_FAILURE);
            }
        }
    }

    /* Replace this (grand)grandchild with gzip. */
    if (execlp(MD_COMPRESSOR, MD_COMPRESSOR, "-f", file, (char *)NULL) == -1) {
        g_warning("Error invoking '%s': %s", MD_COMPRESSOR, strerror(errno));
        _exit(EXIT_FAILURE);
    }
}

static guint
sm_fixed_hash4(
    gconstpointer   v)
{
    return hashlittle(v, 4, 4216);
}

static gboolean
sm_fixed_equal4(
    gconstpointer   v1,
    gconstpointer   v2)
{
    if (memcmp(v1, v2, 4) == 0) {
        return TRUE;
    }
    return FALSE;
}

void
md_free_hash_key(
    gpointer   v1)
{
    g_slice_free(smFieldMapKV_t, v1);
}

static guint
sm_fixed_hash6(
    gconstpointer   v)
{
    return hashlittle(v, 6, 4216);
}

static gboolean
sm_fixed_equal6(
    gconstpointer   v1,
    gconstpointer   v2)
{
    if (memcmp(v1, v2, 6) == 0) {
        return TRUE;
    }
    return FALSE;
}

static guint
sm_fixed_hash8(
    gconstpointer   v)
{
    return hashlittle(v, 8, 4216);
}

static gboolean
sm_fixed_equal8(
    gconstpointer   v1,
    gconstpointer   v2)
{
    if (memcmp(v1, v2, 8) == 0) {
        return TRUE;
    }
    return FALSE;
}

static guint
sm_fixed_hash12(
    gconstpointer   v)
{
    return hashlittle(v, 12, 4216);
}

static gboolean
sm_fixed_equal12(
    gconstpointer   v1,
    gconstpointer   v2)
{
    if (memcmp(v1, v2, 12) == 0) {
        return TRUE;
    }
    return FALSE;
}

static guint
sm_fixed_hash16(
    gconstpointer   v)
{
    return hashlittle(v, 16, 4216);
}

static gboolean
sm_fixed_equal16(
    gconstpointer   v1,
    gconstpointer   v2)
{
    if (memcmp(v1, v2, 16) == 0) {
        return TRUE;
    }
    return FALSE;
}

static guint
sm_fixed_hash18(
    gconstpointer   v)
{
    return hashlittle(v, 18, 4216);
}

static gboolean
sm_fixed_equal18(
    gconstpointer   v1,
    gconstpointer   v2)
{
    if (memcmp(v1, v2, 18) == 0) {
        return TRUE;
    }
    return FALSE;
}

static guint
sm_fixed_hash20(
    gconstpointer   v)
{
    return hashlittle(v, 20, 4216);
}

static gboolean
sm_fixed_equal20(
    gconstpointer   v1,
    gconstpointer   v2)
{
    if (memcmp(v1, v2, 20) == 0) {
        return TRUE;
    }
    return FALSE;
}

smHashTable_t *
smCreateHashTable(
    size_t           length,
    GDestroyNotify   freeKeyfn,
    GDestroyNotify   freeValfn)
{
    smHashTable_t *hTable = g_slice_new0(smHashTable_t);

    hTable->len = length;
    if (length == 4) {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_fixed_hash4,
                                              (GEqualFunc)sm_fixed_equal4,
                                              freeKeyfn, freeValfn);
    } else if (length == 6) {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_fixed_hash6,
                                              (GEqualFunc)sm_fixed_equal6,
                                              freeKeyfn, freeValfn);
    } else if (length == 8) {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_fixed_hash8,
                                              (GEqualFunc)sm_fixed_equal8,
                                              freeKeyfn, freeValfn);
    } else if (length == 12) {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_fixed_hash12,
                                              (GEqualFunc)sm_fixed_equal12,
                                              freeKeyfn, freeValfn);
    } else if (length == 16) {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_fixed_hash16,
                                              (GEqualFunc)sm_fixed_equal16,
                                              freeKeyfn, freeValfn);
    } else if (length == 18) {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_fixed_hash18,
                                              (GEqualFunc)sm_fixed_equal18,
                                              freeKeyfn, freeValfn);
    } else if (length == 20) {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_fixed_hash20,
                                              (GEqualFunc)sm_fixed_equal20,
                                              freeKeyfn, freeValfn);
    } else {
        hTable->table = g_hash_table_new_full((GHashFunc)sm_octet_array_hash,
                                              (GEqualFunc)sm_octet_array_equal,
                                              freeKeyfn, freeValfn);
    }

    return hTable;
}

gpointer
smHashLookup(
    smHashTable_t  *table,
    uint8_t        *key)
{
    return g_hash_table_lookup(table->table, key);
}

void
smHashTableInsert(
    smHashTable_t  *table,
    uint8_t        *key,
    uint8_t        *value)
{
    g_hash_table_insert(table->table, (gpointer)key, (gpointer)value);
}

void
smHashTableFree(
    smHashTable_t  *table)
{
    g_hash_table_destroy(table->table);
    g_slice_free(smHashTable_t, table);
}

void
smHashTableRemove(
    smHashTable_t  *table,
    uint8_t        *key)
{
    g_hash_table_remove(table->table, (gpointer)key);
}

uint32_t
smFieldMapTranslate(
    const smFieldMap_t  *map,
    const mdFullFlow_t  *flow)
{
    smFieldMapKV_t *value;
    smFieldMapKV_t  key;

    switch (map->field) {
      case OBDOMAIN:
        key.val = flow->rec->observationDomainId;
        break;
      case VLAN:
        key.val = flow->rec->vlanId;
        break;
      default:
        break;
    }

    value = smHashLookup(map->table, (uint8_t *)&key);

    if (value) {
        return value->val;
    } else {
        return 0;
    }
}

/* move file from *file to *new_dir */
GString *
sm_util_move_file(
    const char  *file,
    const char  *new_dir)
{
    GString    *new_file = NULL;
    const char *filename;

    filename = g_strrstr(file, "/");
    if (filename == NULL) {
        /* if no slash, use entire filename */
        filename = file;
    }

    new_file = g_string_sized_new(PATH_MAX);

    g_string_append_printf(new_file, "%s", new_dir);
    g_string_append_printf(new_file, "%s", filename);
    if (g_rename(file, new_file->str) != 0) {
        g_string_free(new_file, TRUE);
        return NULL;
    }

    return new_file;
}


#if ENABLE_SKIPSET
static GHashTable      *md_ipset;
static pthread_mutex_t  md_ipset_mutex = PTHREAD_MUTEX_INITIALIZER;

mdIPSet_t *
mdUtilIPSetOpen(
    const char  *path,
    GError     **err)
{
    mdIPSet_t *ipset;
    ssize_t    rv;

    pthread_mutex_lock(&md_ipset_mutex);

    if (!app_registered) {
        skAppRegister(g_get_prgname());
        app_registered++;
    }

    if (!md_ipset) {
        md_ipset = g_hash_table_new(g_str_hash, g_str_equal);
    } else if ((ipset = (mdIPSet_t *)g_hash_table_lookup(md_ipset, path))
               != NULL)
    {
        /* pthread_mutex_lock(&ipset->mutex); */
        ++ipset->ref;
        /* pthread_mutex_unlock(&ipset->mutex); */
        pthread_mutex_unlock(&md_ipset_mutex);
        return ipset;
    }

    ipset = g_slice_new0(mdIPSet_t);
    rv = skIPSetLoad(&ipset->ipset, path);
    if (SKIPSET_OK != rv) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Unable to load IPSet '%s': %s",
                    path, skIPSetStrerror(rv));
        g_slice_free(mdIPSet_t, ipset);
        pthread_mutex_unlock(&md_ipset_mutex);
        return NULL;
    }

    ++ipset->ref;
    ipset->path = g_strdup(path);
    /* pthread_mutex_init(&ipset->mutex, NULL); */
    g_hash_table_insert(md_ipset, (gpointer)ipset->path, (gpointer)ipset);
    pthread_mutex_unlock(&md_ipset_mutex);
    return ipset;
}

void
mdUtilIPSetClose(
    mdIPSet_t  *ipset)
{
    if (NULL == ipset) {
        return;
    }
    pthread_mutex_lock(&md_ipset_mutex);
    /* pthread_mutex_lock(&ipset->mutex); */
    if (ipset->ref > 1) {
        --ipset->ref;
        /* pthread_mutex_unlock(&ipset->mutex); */
        pthread_mutex_unlock(&md_ipset_mutex);
        return;
    }
    if (ipset->ref == 0) {
        /* pthread_mutex_unlock(&ipset->mutex); */
        pthread_mutex_unlock(&md_ipset_mutex);
        return;
    }
    ipset->ref = 0;

    g_hash_table_remove(md_ipset, (gpointer)ipset->path);

    skIPSetDestroy(&ipset->ipset);
    g_free(ipset->path);
    /* pthread_mutex_unlock(&ipset->mutex); */
    /* pthread_mutex_destroy(&ipset->mutex); */
    g_slice_free(mdIPSet_t, ipset);

    pthread_mutex_unlock(&md_ipset_mutex);
}
#endif  /* ENABLE_SKIPSET */

#if ENABLE_SKPREFIXMAP
static GHashTable      *md_pmap;
static pthread_mutex_t  md_pmap_mutex = PTHREAD_MUTEX_INITIALIZER;


mdPMap_t *
mdUtilPMapOpen(
    const char  *path,
    GError     **err)
{
    mdPMap_t *pmap;
    ssize_t   rv;

    pthread_mutex_lock(&md_pmap_mutex);

    if (!app_registered) {
        skAppRegister(g_get_prgname());
        app_registered++;
    }

    if (!md_pmap) {
        md_pmap = g_hash_table_new(g_str_hash, g_str_equal);
    } else if ((pmap = (mdPMap_t *)g_hash_table_lookup(md_pmap, path))
               != NULL)
    {
        /* pthread_mutex_lock(&pmap->mutex); */
        ++pmap->ref;
        /* pthread_mutex_unlock(&pmap->mutex); */
        pthread_mutex_unlock(&md_pmap_mutex);
        return pmap;
    }

    pmap = g_slice_new0(mdPMap_t);
    rv = skPrefixMapLoad(&pmap->pmap, path);
    if (SKPREFIXMAP_OK != rv) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Unable to load PMap '%s': %s",
                    path, skPrefixMapStrerror(rv));
        g_slice_free(mdPMap_t, pmap);
        pthread_mutex_unlock(&md_pmap_mutex);
        return NULL;
    }

    ++pmap->ref;
    pmap->path = g_strdup(path);
    pmap->content = skPrefixMapGetContentType(pmap->pmap);
    pmap->mapname = skPrefixMapGetMapName(pmap->pmap);
    if (NULL == pmap->mapname) {
        const char *slash = strrchr(pmap->path, '/');
        if (slash) {
            pmap->mapname = slash + 1;
        } else {
            pmap->mapname = pmap->path;
        }
    }

    /* pthread_mutex_init(&pmap->mutex, NULL); */
    g_hash_table_insert(md_pmap, (gpointer)pmap->path, (gpointer)pmap);
    pthread_mutex_unlock(&md_pmap_mutex);
    return pmap;
}

void
mdUtilPMapClose(
    mdPMap_t  *pmap)
{
    if (NULL == pmap) {
        return;
    }
    pthread_mutex_lock(&md_pmap_mutex);
    /* pthread_mutex_lock(&pmap->mutex); */
    if (pmap->ref > 1) {
        --pmap->ref;
        /* pthread_mutex_unlock(&pmap->mutex); */
        pthread_mutex_unlock(&md_pmap_mutex);
        return;
    }
    if (pmap->ref == 0) {
        /* pthread_mutex_unlock(&pmap->mutex); */
        pthread_mutex_unlock(&md_pmap_mutex);
        return;
    }
    pmap->ref = 0;

    g_hash_table_remove(md_pmap, (gpointer)pmap->path);

    skPrefixMapDelete(pmap->pmap);
    g_free(pmap->path);
    /* pthread_mutex_unlock(&pmap->mutex); */
    /* pthread_mutex_destroy(&pmap->mutex); */
    g_slice_free(mdPMap_t, pmap);

    pthread_mutex_unlock(&md_pmap_mutex);
}
#endif  /* ENABLE_SKPREFIXMAP */
