/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator.c
 *
 *  IPFIX mediator for filtering, DNS deduplication, and other mediator-like
 *  things.  Contains main() and contains functions for program set-up.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.11
 *
 *  Copyright 2024 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM24-1038
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define MEDIATOR_MAIN_SOURCE 1
#include "mediator.h"
#include "mediator_inf.h"
#include "mediator_core.h"
#include "mediator_filter.h"
#include "mediator_dns.h"
#include "mediator_stat.h"
#include "mediator_log.h"
#include "mediator_dedup.h"
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_GRP_H
#include <grp.h>
#endif

#if ENABLE_SKTYPESENSOR
#include <silk/utils.h>
#include "probeconf.h"
#elif ENABLE_SKPREFIXMAP || (ENABLE_SKIPSET && HAVE_SILK_UTILS_H)
#include <silk/utils.h>
#endif /* if ENABLE_SKTYPESENSOR */

/*#define MDLOG_TESTING_LOG 0*/

/* Maximum timeout (in seconds) for flushing internal state (5 min) */
#define TIMEOUT   300

/* Default UDP template timeout in seconds (10 min).  The templates are resent
 * three times per timeout interval. */
#define DEFAULT_UDP_TEMP_TIMEOUT  600

/* Default read (input) and write (output) ports */
#define DEFAULT_MD_IPORT  "18000"
#define DEFAULT_MD_EPORT  "18001"

mdConfig_t         md_config;
int                md_quit = 0;
char              *md_logfile = NULL;
char              *md_logdest = NULL;
char              *md_pidfile = NULL;
int                md_stats_timeout = TIMEOUT;
int                udp_temp_timeout = 0;
uint16_t           dns_flush_timeout = MD_DNS_DEDUP_DEFAULT_FLUSH;
gboolean           dns_dedup = FALSE;
mdLogLevel_t       md_log_level = WARNING;
unsigned int       md_top_level_time_units = MD_TUNIT_UNSET;
fbInfoElement_t   *user_elements = NULL;
#if HAVE_SPREAD
char             **md_in_groups;
char             **md_out_groups;
int                num_out_groups = 0;
#endif  /* HAVE_SPREAD */
#if ENABLE_SKIPSET || ENABLE_SKPREFIXMAP || ENABLE_SKTYPESENSOR
int  app_registered = 0;
#endif

static char              *last_logfile = NULL;
static char              *cur_logfile = NULL;
static time_t             md_log_rolltime;
static uint16_t           dns_max_hit_count = MD_DNS_DEDUP_DEFAULT_HIT;
static mdTransportType_t  md_ipfix_intransport = NONE;

static gboolean           md_daemon = FALSE;
static char              *md_iport = NULL;
static char              *md_eport = NULL;
static char              *md_conf = NULL;
static char              *md_in_str = NULL;
static char              *md_out_str = NULL;
static char             **md_field_list;
static char              *md_time_units_arg = NULL;
static int                rotate = 0;
static int                sleep_usecs = 0;
static volatile int       shutdown_signal = 0;
static char              *poll_time = NULL;
static mdTransportType_t  md_ipfix_outtransport = NONE;
static char              *outspec = NULL;
static char              *inspec = NULL;
static char              *move_dir = NULL;
static GIOChannel        *logfile;
static GLogLevelFlags     level;
static gboolean           md_verbose = FALSE;
static gboolean           md_quiet = FALSE;
static gboolean           md_print_headers = FALSE;
static char              *become_user = NULL;
static char              *become_group = NULL;
static uid_t              new_user = 0;
static gid_t              new_group = 0;
static gboolean           did_become = FALSE;
static gboolean           md_metadata_export = FALSE;


static void
mdPrintVersion(
    const gchar  *option_name,
    const gchar  *option_arg,
    gpointer      v_context,
    GError      **error);

#define MD_HELP_SUMMARY                                 \
    "Reads IPFIX by listening on sockets, by"           \
    " periodically matching a file glob,\n"             \
    "or by reading a file. Writes IPFIX to"             \
    " sockets or writes IPFIX, JSON, or TEXT\n"         \
    "to a file or to rotating files in"                 \
    " a directory. The default is to read from\n"       \
    "stdin and write to stdout."

#define WRAP_OPT "\n                "
static GOptionEntry  md_core_option[] = {
    {"config", 'c', 0, G_OPTION_ARG_STRING, &md_conf,
     WRAP_OPT "Specify the path to the optional configuration file",
     "file_name"},
    {"in", 'i', 0, G_OPTION_ARG_STRING, &inspec,
     WRAP_OPT "Specify the file to read, file glob to poll, or host/IP-address"
     WRAP_OPT "to listen on [-]",
     "file|glob-pattern|host|IP"},
    {"ipfix-input", 0, 0, G_OPTION_ARG_STRING, &md_in_str,
     WRAP_OPT "When input is a host or IP, select the mode of transport"
     WRAP_OPT "(TCP, UDP, SPREAD)",
     "mode"},
    {"ipfix-port", 'p', 0, G_OPTION_ARG_STRING, &md_iport,
     WRAP_OPT "When input is a host or IP, change listening port from the"
     WRAP_OPT "default [18000]",
     "port"},
    {"watch", 'w', 0, G_OPTION_ARG_STRING, &poll_time,
     WRAP_OPT "When input is a file glob, set the interval to polling for"
     WRAP_OPT "files matching the pattern",
     "seconds"},
    {"move", 0, 0, G_OPTION_ARG_STRING, &move_dir,
     WRAP_OPT "When input is a file glob, move the files to this directory"
     WRAP_OPT "after processing. If not specified, the files are deleted",
     "dir_path"},
    {"lock", 0, 0, G_OPTION_ARG_NONE, &md_config.lockmode,
     WRAP_OPT "When input is file glob, do not read files that are locked",
     NULL},
#if HAVE_SPREAD
    {"groups", 'g', 0, G_OPTION_ARG_STRING_ARRAY, &md_in_groups,
     WRAP_OPT "Set names of the Spread groups to subscribe to",
     "groups"},
#endif
    {"out", 'o', 0, G_OPTION_ARG_STRING, &outspec,
     WRAP_OPT "Name the file to create, path-and-file prefix for multiple"
     WRAP_OPT "output files, or host/IP-address to receive output [-]",
     "file|path-prefix|host|IP"},
    {"output-mode", 'm', 0, G_OPTION_ARG_STRING, &md_out_str,
     WRAP_OPT "Select the mode of export. For file-based export, specify"
     WRAP_OPT "TEXT or JSON to override IPFIX. For network export, specify"
     WRAP_OPT "TCP, UDP, or SPREAD",
     "mode"},
    {"export-port", 0, 0, G_OPTION_ARG_STRING, &md_eport,
     WRAP_OPT "When output is a host, change export port from default [18001]",
     "port"},
    {"udp-temp-timeout", 0, 0, G_OPTION_ARG_INT, &udp_temp_timeout,
     WRAP_OPT "When exporting via UDP, set the Template Timeout Period [600]",
     "seconds"},
    {"rotate", 0, 0, G_OPTION_ARG_INT, &rotate,
     WRAP_OPT "When output is a path-prefix, set the interval for rotating the"
     WRAP_OPT "the output files [3600]",
     "seconds"},
    {"no-stats", 0, 0, G_OPTION_ARG_NONE, &md_config.no_stats,
     WRAP_OPT "Disable decoding and export of YAF process stat records",
     NULL},
    {"preserve-obdomain", 0, 0, G_OPTION_ARG_NONE, &md_config.preserve_obdomain,
     WRAP_OPT "Do not overwrite observationDomainId elements read in the input",
     NULL},
    {"dns-dedup", 0, 0, G_OPTION_ARG_NONE, &dns_dedup,
     WRAP_OPT "Perform DNS Deduplication on all DNS Resource Records",
     NULL},
#if SM_ENABLE_METADATA_EXPORT
    {"metadata-export", 0, 0, G_OPTION_ARG_NONE, &md_metadata_export,
     WRAP_OPT "Include information element (RFC 5610) and template metadata"
     WRAP_OPT "records in the IPFIX output",
     NULL},
#endif
    {"rewrite-ssl-certs", 0, 0, G_OPTION_ARG_NONE, &md_config.rewrite_ssl_certs,
     WRAP_OPT "Rewrite SSL certs for IPFIX exporters and allow reading of"
     WRAP_OPT "rewritten SSL certs from an upstream mediator",
     NULL},
    {"fields", 'f', 0, G_OPTION_ARG_STRING_ARRAY, &md_field_list,
     WRAP_OPT "When exporting in TEXT mode, specify a comma-separated list of"
     WRAP_OPT "flow fields to print",
     "fields"},
    {"print-headers", 'h', 0, G_OPTION_ARG_NONE, &md_print_headers,
     WRAP_OPT "When exporting in TEXT mode, print column headers",
     NULL},
    {"time-units", 0, 0, G_OPTION_ARG_STRING, &md_time_units_arg,
     WRAP_OPT "Sets unit(s) of timestamps for TEXT or JSON exporters [incoming]"
     WRAP_OPT "Choices: best,incoming,micro,milli,nano,nofrac",
     "units,..."},
    {"sleep", 's', 0, G_OPTION_ARG_INT, &sleep_usecs,
     WRAP_OPT "Set the number of microseconds to sleep between exporting IPFIX"
     WRAP_OPT "messages",
     "microsecords"},
    {"log", 'l', 0, G_OPTION_ARG_STRING, &md_logfile,
     WRAP_OPT "Specify the file in which to log errors, messages, etc",
     "logfile"},
    {"log-dir", 0, 0, G_OPTION_ARG_STRING, &md_logdest,
     WRAP_OPT "Specify an existing directory where the log files are written",
     "log_path"},
    {"verbose", 'v', 0, G_OPTION_ARG_NONE, &md_verbose,
     WRAP_OPT "Change the logging level from default WARNING to DEBUG",
     NULL},
    {"quiet", 'q', 0, G_OPTION_ARG_NONE, &md_quiet,
     WRAP_OPT "Change the logging level from default WARNING to QUIET",
     NULL},
    {"daemonize", 'd', 0, G_OPTION_ARG_NONE, &md_daemon,
     WRAP_OPT "Have super_mediator run as a daemon",
     NULL},
    {"pidfile", 0, 0, G_OPTION_ARG_STRING, &md_pidfile,
     WRAP_OPT "Name the complete path to the process ID file",
     "file_path"},
    {"become-user", 'U', 0, G_OPTION_ARG_STRING, &become_user,
     WRAP_OPT "Become user after setup if started as root",
     "user"},
    {"become-group", 0, 0, G_OPTION_ARG_STRING, &become_group,
     WRAP_OPT "Become group after setup if started as root",
     "group"},
    {"version", 'V', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK,
     &mdPrintVersion,
     WRAP_OPT "Print application version and exit",
     NULL},
    { NULL, 0, 0, G_OPTION_ARG_NONE, NULL, NULL, NULL }
};


static void
mdDaemonize(
    void);

static void
mdParseOptions(
    int   *argc,
    char **argv[]);

static void
sigHandler(
    void);

static gboolean
mdPrivc_Setup(
    GError **err);

static gboolean
mdPrivc_Become(
    GError **err);

static void
smFreeMaps(
    mdConfig_t  *cfg);

/**
 * mdLogger
 *
 */
static void
mdLogger(
    const char      *domain __attribute__((unused)),
    GLogLevelFlags   log_level,
    const char      *message,
    gpointer         user_data __attribute__((unused)))
{
    gsize      sz;
    char       timebuf[64];
    struct tm  time_tm;
    time_t     cur_time = time(NULL);

    MD_UNUSED_PARAM(log_level);

    gmtime_r(&cur_time, &time_tm);
    snprintf(timebuf, sizeof(timebuf), "[%04u-%02u-%02u %02u:%02u:%02u] ",
             (time_tm.tm_year + 1900), (time_tm.tm_mon + 1), time_tm.tm_mday,
             time_tm.tm_hour, time_tm.tm_min, time_tm.tm_sec);

    g_io_channel_write_chars(logfile, timebuf, -1, &sz, NULL);
    g_io_channel_write_chars(logfile, message, -1, &sz, NULL);
    g_io_channel_write_chars(logfile, "\n", 1, &sz, NULL);
    g_io_channel_flush(logfile, NULL);
}

/**
 * mdPrintHelp
 *
 */
static void
mdPrintHelp(
    const gchar  *option_name,
    const gchar  *option_arg,
    gpointer      v_context,
    GError      **error)
{
    GOptionContext *ctx;
    char *help;

    (void)option_name;
    (void)option_arg;
    (void)error;

    ctx = (GOptionContext *)v_context;
    help = g_option_context_get_help(ctx, FALSE, NULL);

    printf("%s", help);
    g_free(help);
    exit(0);
}

/**
 * mdPrintVersion
 *
 *
 */
static void
mdPrintVersion(
    const gchar  *option_name,
    const gchar  *option_arg,
    gpointer      v_context,
    GError      **error)
{
    GString *resultString;
    GString *silk;

    MD_UNUSED_PARAM(option_name);
    MD_UNUSED_PARAM(option_arg);
    MD_UNUSED_PARAM(v_context);
    MD_UNUSED_PARAM(error);

    resultString = g_string_sized_new(1024);

    g_string_append_printf(resultString, "super_mediator version %s\n"
                           "Build Configuration: \n",  VERSION);
#ifdef FIXBUF_VERSION
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Fixbuf version:",
                           FIXBUF_VERSION);
#endif

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "MySQL support:",
#if HAVE_MYSQL
                           "YES"
#else
                           "NO"
#endif
                           );

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Spread support:",
#if HAVE_SPREAD
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "OpenSSL support:",
#if HAVE_OPENSSL
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Metadata export support:",
#if SM_ENABLE_METADATA_EXPORT
                           "YES"
#else
                           "NO"
#endif
                           );

    silk = g_string_sized_new(128);
#if ENABLE_SKIPSET
    g_string_append(silk, ", IPSet");
#endif
#if ENABLE_SKPREFIXMAP
    g_string_append(silk, ", PrefixMap");
#endif
#if ENABLE_SKTYPESENSOR
    g_string_append(silk, ", Type+Sensor");
#endif
    if (silk->len > 2) {
        g_string_append_printf(resultString, "    * %-32s  %s (%s)\n",
                               "SiLK support:", "YES", silk->str + 2);
    } else {
        g_string_append_printf(resultString, "    * %-32s  %s\n",
                               "SiLK support:", "NO");
    }
    g_string_free(silk, TRUE);

    g_string_append(resultString,
                    "Copyright 2012-2025 Carnegie Mellon University"
                    "\nGNU General Public License (GPL) Rights "
                    "pursuant to Version 2, June 1991\n");
    g_string_append(resultString,
                    "Government Purpose License Rights (GPLR) "
                    "pursuant to DFARS 252.227-7013\n");
    g_string_append(resultString,
                    "Send bug reports, feature requests, and comments"
                    " to netsa-help@cert.org.\n");

    fprintf(stdout, "%s", resultString->str);

    g_string_free(resultString, TRUE);
    exit(0);
}

static GIOChannel *
mdRotateLog(
    GError **err)
{
    char       path[500];
    char       date[32];
    time_t     t;
    struct tm  ts;
    int        slog = 0;

    /* get current time */
    t = time(NULL);
    localtime_r(&t, &ts);
    strftime(date, sizeof(date), "%Y%m%d", &ts);

#ifndef MDLOG_TESTING_LOG
    ts.tm_hour = 23;
    ts.tm_min = 59;
    ts.tm_sec = 59;
    md_log_rolltime = mktime(&ts) + 1;
#else  /* ifndef MDLOG_TESTING_LOG */
    strftime(date, sizeof(date), "%Y%m%d:%H:%M", &ts);

    if (ts.tm_sec > 55) {
        ++ts.tm_min;
    }
    ts.tm_sec = 0;
    ++ts.tm_min;
    md_log_rolltime = mktime(&ts);
#endif /* ifndef MDLOG_TESTING_LOG */
    snprintf(path, sizeof(path), "%s/sm-%s.log", md_logdest, date);
    if (cur_logfile) {
        if (last_logfile) {
            g_free(last_logfile);
        }
        last_logfile = cur_logfile;
    }
    cur_logfile = g_strdup(path);
    return md_log_setup(path, level, &slog, md_daemon, err);
}


/**
 * mdSetEmitTimer
 *
 * Separate thread that runs when the mediator is to run
 * forever.  This will emit the buffer and flush any
 * export files every 5 minutes in the case that we are not
 * receiving any data.
 * It will also print mediator stats if the mediator is
 * configured to do so.
 *
 */
static void *
mdSetEmitTimer(
    void  *data)
{
    mdContext_t     *ctx = (mdContext_t *)data;
    mdConfig_t      *cfg = ctx->cfg;
    struct timespec  to;
    struct timeval   tp;
    time_t           now;
    time_t           then;
    time_t           sectime;
    int              seconds, rc;
    /* we need to flush at least every 5 minutes */
    const long       timeout = MIN(md_stats_timeout, TIMEOUT);
    GError          *err = NULL;

    then = sectime = time(NULL);

    gettimeofday(&tp, NULL);
    to.tv_sec = tp.tv_sec + timeout;
    to.tv_nsec = tp.tv_usec * 1000;

    pthread_mutex_lock(&cfg->log_mutex);

    while (!md_quit) {
        pthread_cond_timedwait(&cfg->log_cond, &cfg->log_mutex, &to);
        gettimeofday(&tp, NULL);
        to.tv_sec = tp.tv_sec + timeout;
        to.tv_nsec = tp.tv_usec * 1000;
        now = time(NULL);
        /* only flush every 5 minutes */
        seconds = difftime(now, then);
        if (seconds >= TIMEOUT) {
            then = time(NULL);
            mdExporterConnectionReset(cfg, &(ctx->err));
        }
        seconds = difftime(now, sectime);
        if (seconds >= md_stats_timeout) {
            sectime = time(NULL);
            if (!cfg->no_stats) {
                mdStatDump(cfg, ctx->stats);
                if (cfg->gen_tombstone) {
                    mdSendTombstoneRecord(ctx, &err);
                }
            }
        }
        /* rotate log */
        if (md_logdest && md_log_rolltime < now) {
            g_message("Rotating Log File");
            rc = g_io_channel_shutdown(logfile, TRUE, &err);
            if (!rc) {
                g_warning("Unable to rotate log");
            } else {
                /* open new logfile */
                logfile = mdRotateLog(&err);
                /* compress old logfile */
                md_log_compress(last_logfile);
                if (!logfile) {
                    g_warning("Unable to open new log file: %s", err->message);
                }
            }
        }
    }

    pthread_mutex_unlock(&cfg->log_mutex);
    return NULL;
}


static void
mdQuit(
    int   sig)
{
    md_quit++;

    /* typically do not want to call logger from within sighandler */
    /* g_debug("yaf received %s", strsignal(s)); */
    if (0 == shutdown_signal) {
        shutdown_signal = sig;
    }
    mdInterruptListeners(&md_config);
}


/**
 * main
 *
 *
 */
int
main(
    int    argc,
    char  *argv[])
{
    mdContext_t  ctx = MD_CTX_INIT;
    GError      *error = NULL;
    pthread_t    to_thread;
    char        *errmsg = NULL;

    md_config = MD_CONFIG_INIT;
    ctx.cfg = &md_config;

    /* parse all the options */
    mdParseOptions(&argc, &argv);

    g_message("super_mediator starting");

    mdStatInit();

    /* set up mediator statistics */
    ctx.stats = g_slice_new0(md_stats_t);

    /**
     * this program runs forever, until interrupted, handle
     * the interrupt gracefully and exit by installing these
     * handlers
     */

    sigHandler();

    /* open input */

    if (!mdCollectorsInit(ctx.cfg, ctx.cfg->flowsrc, &error)) {
        fprintf(stderr, "Fatal: %s\n", error->message);
        exit(1);
    }

    if (pthread_create(&to_thread, NULL, mdSetEmitTimer, &ctx)) {
        fprintf(stderr, "Fatal error starting statistics thread\n");
        exit(1);
    }

    /* set up output */

    if (!mdExportersInit(ctx.cfg, ctx.cfg->flowexit, &error)) {
        g_warning("Fatal: %s\n", error->message);
        exit(1);
    }

    g_debug("Initialization Successful, starting...");

    /** wait for connections*/
    while (!md_quit) {
        if (!mdCollectorStartListeners(ctx.cfg, ctx.cfg->flowsrc, &error)) {
            fprintf(stderr, "Couldn't start listener threads %s\n",
                    error->message);
            md_quit = 1;
            break;
        }

        if (!mdPrivc_Become(&error)) {
            if (g_error_matches(error, MD_ERROR_DOMAIN, MD_ERROR_NODROP)) {
                g_warning("running as root in --live mode, "
                          "but not dropping privilege");
                g_clear_error(&error);
            } else {
                md_quit = 1;
                g_warning("Cannot drop privilege: %s", error->message);
                break;
            }
        }

        /* manage threads, and process flow data */
        if (!(mdCollectorWait(&ctx, &error))) {
            break;
        }
    }

    if (error) {
        errmsg = g_strdup_printf("super_mediator terminating on error: %s",
                                 error->message);
        g_clear_error(&error);
    }
    if (shutdown_signal) {
        g_message("super_mediator received signal %s",
                  strsignal(shutdown_signal));
    }

    pthread_cancel(to_thread);
    pthread_join(to_thread, NULL);

    mdCollectorDestroy(ctx.cfg, TRUE);
    mdExporterDestroy(ctx.cfg, &error);
    smFreeMaps(ctx.cfg);
#if ENABLE_SKTYPESENSOR
    mdSkpcSilkTypeSensorFree();
#endif  /* ENABLE_SKTYPESENSOR */

#if ENABLE_SKIPSET || ENABLE_SKPREFIXMAP || ENABLE_SKTYPESENSOR
    if (app_registered) {
        skAppUnregister();
    }
#endif

    if (user_elements) {
        fbInfoElement_t *tie = user_elements;
        for (; tie->ref.name; tie++) {
            /* free each name */
            g_free((char *)(tie->ref.name));
        }
        g_free(user_elements);
    }

    fbInfoModelFree(mdInfoModel());

    pthread_mutex_destroy(&ctx.cfg->log_mutex);
    pthread_cond_destroy(&ctx.cfg->log_cond);
    mdStatUpdate(ctx.stats);

    g_slice_free1(sizeof(md_stats_t), ctx.stats);

    if (errmsg) {
        g_warning("%s", errmsg);
        g_free(errmsg);
    } else {
        g_debug("super_mediator Terminating");
    }

    /** finished with no problems */
    return 0;
}

/**
 * md_new_export_node
 *
 */
md_export_node_t *
md_new_export_node(
    gboolean   dnsdedup,
    gboolean   dedup)
{
    md_export_node_t *en = g_slice_new0(md_export_node_t);
    en->next = NULL;

    if (dnsdedup) {
        en->dns_dedup = md_dns_dedup_new_state();
    }

    if (dedup) {
        en->dedup = md_dedup_new_dedup_state();
    }

    return en;
}

/**
 * md_new_collect_node
 *
 *
 */
md_collect_node_t *
md_new_collect_node(
    void)
{
    md_collect_node_t *cn = g_slice_new0(md_collect_node_t);
    cn->next = NULL;

    return cn;
}


static mdFlowCollector_t *
mdConfigureCollector(
    void)
{
    mdFlowCollector_t *collector = NULL;

    collector = mdNewFlowCollector(md_ipfix_intransport,
                                   NULL);

    if (md_ipfix_intransport == SPREAD) {
#if HAVE_SPREAD
        int     n = 0;
        gchar **sa;

        /* input should be Spread Daemon */
        /* test connections and subscribe to groups */
        if (md_in_groups == NULL) {
            fprintf(stderr, "Required: At Least 1 Spread Group Name to "
                    "Subscribe to, exiting...\n");
            exit(1);
        } else {
            sa = g_strsplit(*md_in_groups, ",", -1);
            while (sa[n] && *sa[n]) {
                ++n;
            }
            n = 0;
            while (sa[n] && *sa[n]) {
                if (isspace(sa[n][0])) {
                    mdCollectorAddSpreadGroup(collector, sa[n] + 1, n);
                } else {
                    mdCollectorAddSpreadGroup(collector, sa[n], n);
                }
                ++n;
            }
            g_strfreev(sa);
        }
        mdCollectorSetInSpec(collector, inspec);
#else  /* if HAVE_SPREAD */
        fprintf(stderr, "Spread not enabled, reconfigure with Spread...\n");
        exit(1);
#endif /* if HAVE_SPREAD */
    } else if (md_ipfix_intransport == TCP || md_ipfix_intransport == UDP) {
        mdCollectorSetInSpec(collector, inspec);
        if (NULL == md_iport) {
            mdCollectorSetPort(collector, DEFAULT_MD_IPORT);
        } else if (atoi(md_iport) >= 1024 && atoi(md_iport) <= UINT16_MAX) {
            mdCollectorSetPort(collector, md_iport);
            g_free(md_iport);
        } else {
            g_warning("Fatal: listening port for TCP traffic must be "
                      "between 1024 and 65535 inclusive, %s is invalid\n",
                      md_iport);
            exit(1);
        }
    } else {
        /* it's either a file or directory */
        if ((strlen(inspec) == 1) && inspec[0] == '-') {
            if (isatty(fileno(stdin))) {
                g_warning("Refusing to read from terminal on stdin");
                exit(1);
            }
        } else if (poll_time || move_dir) {
            /* input is a directory */
            if (poll_time) {
                mdCollectorSetPollTime(collector, poll_time);
            } else {
                mdCollectorSetPollTime(collector, "30");
            }
            if (move_dir) {
                if (!g_file_test(move_dir, G_FILE_TEST_IS_DIR)) {
                    g_warning("--move expects a valid file directory");
                    exit(1);
                }
                mdCollectorSetMoveDir(collector, move_dir);
            } else {
                g_warning("No Move Directory Specified.");
                g_warning("Incoming files will be deleted after processing.");
            }
        } else {
            if (g_file_test(inspec, G_FILE_TEST_EXISTS)) {
                /* input file */
                md_ipfix_intransport = FILEHANDLER;
            } else {
                g_warning("File %s does not exist.", inspec);
                exit(1);
            }
        }
        mdCollectorSetInSpec(collector, inspec);
    }

    return collector;
}

static mdFlowExporter_t *
mdConfigureExporter(
    mdConfig_t  *cfg,
    gboolean     json)
{
    mdFlowExporter_t *exporter = NULL;
    gchar           **sa;
    int               n = 0;

    exporter =  mdNewFlowExporter(md_ipfix_outtransport);

    if (dns_dedup) {
        mdExporterSetDNSDeDup(exporter);
    }

    if (md_ipfix_outtransport == SPREAD) {
#if HAVE_SPREAD
        n = 0;
        mdExporterSetHost(md_config.flowexit->exp, outspec);
        g_free(outspec);
        /* output is spread */
        /* need spread groups */
        if (md_out_groups == NULL) {
            g_warning("Required: At Least 1 Spread Group Name to "
                      "Subscribe to, exit...");
            exit(1);
        } else if (md_config.out_spread.groups == NULL) {
            sa = g_strsplit(*md_out_groups, ",", -1);
            while (sa[n] && *sa[n]) {
                ++n;
            }
            md_out_groups = (char **)g_malloc0((sizeof(char *) * (n + 1)));
            n = 0;
            while (sa[n] && *sa[n]) {
                if (isspace(sa[n][0])) {
                    md_out_groups[n] = g_strdup(sa[n] + 1);
                } else {
                    md_out_groups[n] = g_strdup(sa[n]);
                }
                ++n;
            }
            md_config.out_spread.groups = md_out_groups;
            g_strfreev(sa);
        }
#else  /* if HAVE_SPREAD */
        g_warning("Spread not enabled, reconfigure with Spread...");
        exit(1);
#endif /* if HAVE_SPREAD */
    } else if (md_ipfix_outtransport == TEXT) {
        mdExporterSetFileSpec(exporter, outspec);
        if (rotate > 0) {
            mdExporterSetRotate(exporter, rotate);
        }
        /* FIELDS parameter in conf file. */
        if (md_field_list) {
            gboolean               dpi = FALSE;
            mdFieldList_t         *item = NULL;
            mdFieldList_t         *list = NULL;
            mdFieldList_t         *first_item = NULL;
            mdAcceptFilterField_t  field;
            sa = g_strsplit(*md_field_list, ",", -1);
            while (sa[n] && *sa[n]) {
                /* remove any leading whitespace */
                g_strchug(sa[n]);
                /* remove any trailing whitespace */
                g_strchomp(sa[n]);
                field = atoi(sa[n]);
                if (field == DPI) {
                    mdExporterCustomListDPI(exporter);
                    ++n;
                    dpi = TRUE;
                    continue;
                }
                item = mdCreateFieldList((mdAcceptFilterField_t)field, FALSE);
                if (!item) {
                    g_warning("Invalid field item %s\n", sa[n]);
                    exit(1);
                }
                if (first_item == NULL) {
                    first_item = item;
                    list = item;
                } else {
                    list->next = item;
                    list = list->next;
                }
                ++n;
            }
            if (dpi && !item) {
                /* just DPI was chosen - create list and set to None */
                item = mdCreateFieldList(NONE_FIELD, TRUE);
                first_item = item;
                mdExporterSetDPIOnly(exporter);
            }
            mdExportCustomList(exporter, first_item);
            g_strfreev(sa);
        }

        if (json) {
            mdExporterSetJson(exporter);
        }
    } else if (md_ipfix_outtransport == TCP || md_ipfix_outtransport == UDP) {
        if (NULL == md_eport) {
            mdExporterSetPort(exporter, DEFAULT_MD_EPORT);
        } else if (atoi(md_eport) >= 1024 && atoi(md_eport) <= UINT16_MAX) {
            mdExporterSetPort(exporter, md_eport);
            g_free(md_eport);
        } else {
            g_warning("Fatal: exporting port for TCP/UDP traffic must be "
                      " between 1024 and 65535 inclusive, %s is invalid\n",
                      md_eport);
            exit(1);
        }
        mdExporterSetHost(exporter, outspec);

        /* Templates are resent three times per timeout interval */
        udp_temp_timeout /= 3;
        if (udp_temp_timeout <= 0) {
            udp_temp_timeout = DEFAULT_UDP_TEMP_TIMEOUT / 3;
        }
        yfDiffTimeFromSeconds(&cfg->udp_template_interval, udp_temp_timeout);
    } else if (outspec) {
        /* file or a directory or stdout */
        if ((strlen(outspec) == 1) && outspec[0] == '-') {
            if (isatty(fileno(stdout))) {
                g_warning("Refusing to write to terminal on stdout");
                exit(1);
            }
        }

        mdExporterSetFileSpec(exporter, outspec);

        if (rotate > 0) {
            mdExporterSetRotate(exporter, rotate);
        }
    }

    if (md_print_headers) {
        if (!md_field_list) {
            mdExporterSetPrintHeader(exporter);
        } else {
            g_warning("Not printing column headers for field list.");
        }
    }

    if (md_metadata_export) {
        mdExporterSetMetadataExport(exporter);
    }

    if (md_top_level_time_units != MD_TUNIT_UNSET) {
        mdExporterSetTimeUnits(exporter, md_top_level_time_units);
    }

    return exporter;
}


static void
mdParseTimeUnits(
    char   *arg)
{
    const struct tunit_map_st {
        const char     *tname;
        mdTimeUnits_t   tunits;
    } tunit_map[] = {
        {"best",          MD_TUNIT_BEST},
        {"incoming",      MD_TUNIT_INCOMING},
        {"micro",         MD_TUNIT_MICRO},
        {"microsecond",   MD_TUNIT_MICRO},
        {"microseconds",  MD_TUNIT_MICRO},
        {"milli",         MD_TUNIT_MILLI},
        {"millisecond",   MD_TUNIT_MILLI},
        {"milliseconds",  MD_TUNIT_MILLI},
        {"nano",          MD_TUNIT_NANO},
        {"nanosecond",    MD_TUNIT_NANO},
        {"nanoseconds",   MD_TUNIT_NANO},
        {"no-frac",       MD_TUNIT_NOFRAC},
        {"no_frac",       MD_TUNIT_NOFRAC},
        {"nofrac",        MD_TUNIT_NOFRAC},
        {NULL, MD_TUNIT_UNSET}
    };
    unsigned int i, j;
    gchar **tokens;
    uint8_t tunits = MD_TUNIT_UNSET;

    if (md_top_level_time_units != MD_TUNIT_UNSET) {
        fprintf(stderr, "Invalid time-units: Value was previously set\n");
        exit(1);
    }

    tokens = g_strsplit(arg, ",", -1);
    for (i = 0; tokens[i] != NULL; ++i) {
        for (j = 0; tunit_map[j].tname != NULL; ++j) {
            if (0 == strcmp(tokens[i], tunit_map[j].tname)) {
                tunits |= tunit_map[j].tunits;
                break;
            }
        }
        /* check for best or incoming included with another */
        if ((tunits & MD_TUNIT_BEST) && (tunits & ~MD_TUNIT_BEST)) {
            fprintf(stderr, ("Invalid time-units:"
                             " 'best' may not included in a list\n"));
            exit(1);
        }
        if ((tunits & MD_TUNIT_INCOMING) && (tunits & ~MD_TUNIT_INCOMING)) {
            fprintf(stderr, ("Invalid time-units:"
                             " 'incoming' may not included in a list\n"));
            exit(1);
        }
        /* error if not found unless token is empty string */
        if (NULL == tunit_map[j].tname && *tokens[i]) {
            fprintf(stderr, "Invalid time-units item '%s'\n", tokens[i]);
            exit(1);
        }
    }
    if (MD_TUNIT_UNSET == tunits) {
        fprintf(stderr,
                "Invalid time-units: argument contains no valid values\n");
        exit(1);
    }
    g_strfreev(tokens);
    g_free(arg);

    md_top_level_time_units = tunits;
}


/**
 * mdParseOptions
 *
 * parses the command line options
 *
 */
static void
mdParseOptions(
    int   *argc,
    char **argv[])
{
    GOptionEntry help_option[] = {
        {"help", 'h', G_OPTION_FLAG_NO_ARG,
         G_OPTION_ARG_CALLBACK, &mdPrintHelp,
         WRAP_OPT "Show this brief description of the options and exit",
         NULL},
        { NULL, 0, 0, G_OPTION_ARG_NONE, NULL, NULL, NULL }
    };
    GOptionContext   *ctx = NULL;
    GOptionGroup     *group = NULL;
    GError           *err = NULL;
    int               slog = 0;
    gboolean          json = FALSE;
    md_export_node_t *cnode = NULL;

    ctx = g_option_context_new("- Mediator Options");

    /* Create the main group so we can set user_data to ctx */
    group = g_option_group_new("main", "Options", NULL, ctx, NULL);
    g_option_context_set_main_group(ctx, group);
    g_option_context_set_summary(ctx, MD_HELP_SUMMARY);

    /* Create the help option */
    g_option_context_set_help_enabled(ctx, FALSE);
    g_option_group_add_entries(group, help_option);
    g_option_group_add_entries(group, md_core_option);

    if (!g_option_context_parse(ctx, argc, argv, &err)) {
        fprintf(stderr, "option parsing failed: %s\n", err->message);
        g_clear_error(&err);
        exit(1);
    }

    if (!mdPrivc_Setup(&err)) {
        fprintf(stderr, "Error: %s\n", err->message);
        g_clear_error(&err);
        exit(1);
    }

    /* Configuration file has precedence over all */
    /* FIXME: Does it *REALLY* have precedence and does it make sense?
     * Shouldn't command line overrule? */
    if (md_conf) {
        if (!mediator_config_load(md_conf)) {
            exit(1);
        }
    }

    if (md_in_str) {
        if (!g_ascii_strcasecmp(md_in_str, "tcp")) {
            md_ipfix_intransport = TCP;
        } else if (!(g_ascii_strcasecmp(md_in_str, "udp"))) {
            md_ipfix_intransport = UDP;
        } else if (!(g_ascii_strcasecmp(md_in_str, "spread"))) {
            md_ipfix_intransport = SPREAD;
        }
    } else {
        md_ipfix_intransport = FILEHANDLER;
    }

    if (md_out_str) {
        if (!g_ascii_strcasecmp(md_out_str, "tcp")) {
            md_ipfix_outtransport = TCP;
        } else if (!(g_ascii_strcasecmp(md_out_str, "udp"))) {
            md_ipfix_outtransport = UDP;
        } else if (!(g_ascii_strcasecmp(md_out_str, "spread"))) {
            fprintf(stderr, "Spread is only available if configured using the "
                    "configuration file\n");
            exit(1);
        } else if (!(g_ascii_strcasecmp(md_out_str, "text"))) {
            md_ipfix_outtransport = TEXT;
        } else if (!(g_ascii_strcasecmp(md_out_str, "json"))) {
            md_ipfix_outtransport = TEXT;
            json = TRUE;
        }
    } else {
        md_ipfix_outtransport = FILEHANDLER;
    }

    /* Logging options */

    level = md_parse_log_level(md_log_level, md_verbose, md_quiet);

    if (md_logdest) {
        logfile = mdRotateLog(&err);
    } else {
        logfile = md_log_setup(md_logfile, level, &slog, md_daemon, &err);
    }

    if (!logfile && (slog == 0)) {
        fprintf(stderr, "Log option parsing failed: %s\n", err->message);
        exit(1);
    }

    if (!slog) {
        /* if not syslog, set default handler */
        g_log_set_handler(G_LOG_DOMAIN, level, mdLogger, NULL);
    }

    if (md_stats_timeout == 0) {
        g_warning("Turning off stats export.");
        md_stats_timeout = TIMEOUT;
        md_config.no_stats = TRUE;
    }

    /* COLLECTOR OPTIONS */

    if (md_config.flowsrc == NULL || inspec != NULL) {
        if (md_config.flowsrc) {
            g_warning("WARNING: Overriding Collectors in configuration file "
                      "due to presence of command line arguments.");
            mdCollectorDestroy(&md_config, FALSE);
        }
        md_config.flowsrc = md_new_collect_node();
        if (inspec == NULL) {
            inspec = g_strdup("-");
            md_ipfix_intransport = FILEHANDLER;
        }
        md_config.flowsrc->coll = mdConfigureCollector();
    }
    /* g_free(inspec); */

    if (sleep_usecs < 1000000) {
        md_config.usec_sleep = sleep_usecs;
    } else {
        g_warning("Maximum sleep time is 1000000");
        md_config.usec_sleep = sleep_usecs;
    }

    if (md_time_units_arg) {
        mdParseTimeUnits(md_time_units_arg);
        md_time_units_arg = NULL;
    }

    /* NOW TO EXPORT */
    /* If no exporters are defined in the config file, configure one based on
     * command line options. */
    if (md_config.flowexit == NULL) {
        md_config.flowexit = md_new_export_node(dns_dedup, FALSE);
        if (outspec == NULL) {
            outspec = g_strdup("-");
            md_ipfix_outtransport = FILEHANDLER;
        }
        md_config.flowexit->exp = mdConfigureExporter(&md_config, json);
    } else {
        if (outspec) {
            g_warning(
                "WARNING: WILL NOT Override Exporters in configuration file.");
        }
    }

    if (dns_dedup) {
        if (dns_max_hit_count == 0) {
            fprintf(stderr, "Invalid Hit Count of 0 (Must be > 0)\n");
            exit(1);
        }
        if (dns_flush_timeout == 0) {
            fprintf(stderr, "Invalid Flush Timeout of 0 (Must be > 0)\n");
            exit(1);
        }

        md_dns_dedup_configure_state(md_config.flowexit->dns_dedup, NULL,
                                     dns_max_hit_count, dns_flush_timeout,
                                     FALSE, NULL, FALSE);
    }

    if (!mdCollectorVerifySetup(md_config.flowsrc->coll, &err)) {
        fprintf(stderr, "Error Verifying Collector Setup\n");
        exit(1);
    }

    for (cnode = md_config.flowexit; cnode; cnode = cnode->next) {
        if (!mdExporterVerifySetup(cnode->exp)) {
            fprintf(stderr, "Error Verifying Export Setup\n");
            exit(1);
        }
    }

    if (md_daemon) {
        mdDaemonize();
    }

    g_option_context_free(ctx);
}



/**
 * smExit
 *
 * exit handler for super_mediator
 *
 */
static void
mdExit(
    void)
{
    if (md_pidfile) {
        unlink(md_pidfile);
    }
}


/**
 * sigHandler
 *
 * this gets called from various system signal handlers.  It is used to
 * provide a way to exit this program cleanly when the user wants to
 * kill this program
 *
 * @param signalNumber the number of the signal that this handler is
 *        getting called for
 *
 */
static void
sigHandler(
    void)
{
    struct sigaction  sa, osa;

    sa.sa_handler = mdQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGINT, &sa, &osa)) {
        g_error("sigaction(SIGINT) failed: %s", strerror(errno));
    }

    sa.sa_handler = mdQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGTERM, &sa, &osa)) {
        g_error("sigaction(SIGTERM) failed: %s", strerror(errno));
    }
}

static void
mdDaemonize(
    void)
{
    pid_t  pid;
    int    rv = -1;
    char   str[256];
    int    fp;

    if (chdir("/") == -1) {
        rv = errno;
        g_warning("Cannot change directory: %s", strerror(rv));
        exit(-1);
    }

    if ((pid = fork()) == -1) {
        rv = errno;
        g_warning("Cannot fork for daemon: %s", strerror(rv));
        exit(-1);
    } else if (pid != 0) {
        g_debug("Forked child %ld.  Parent exiting", (long)pid);
        _exit(EXIT_SUCCESS);
    }

    setsid();

    umask(0022);

    rv = atexit(mdExit);
    if (rv == -1) {
        g_warning("Unable to register function with atexit(): %s",
                  strerror(rv));
        exit(-1);
    }

    /* Close out the standard file descriptors */
    close(STDIN_FILENO);

    if (md_pidfile) {
        fp = open(md_pidfile, O_RDWR | O_CREAT, 0640);
        if (fp < 0) {
            g_warning("Unable to open pid file %s", md_pidfile);
            exit(1);
        }
        sprintf(str, "%d\n", getpid());
        if (!write(fp, str, strlen(str))) {
            g_warning("Unable to write pid to file");
        }
    } else {
        g_debug("pid: %d", getpid());
    }
}

gboolean
mdListenerConnect(
    fbListener_t     *listener,
    void            **ctx,
    int               fd,
    struct sockaddr  *peer,
    size_t            peerlen,
    GError          **err)
{
    md_collect_node_t *collector;

    MD_UNUSED_PARAM(fd);
    MD_UNUSED_PARAM(peerlen);
    MD_UNUSED_PARAM(err);

    if (!peer) {
        /* this is UDP */
        return TRUE;
    }

    /* set context based on which listener this is */
    collector = mdCollectorFindListener(md_config.flowsrc, listener);

    if (!collector) {
        return FALSE;
    }

    if (peer->sa_family == AF_INET) {
        char *ip = inet_ntoa((((struct sockaddr_in *)peer)->sin_addr));
        pthread_mutex_lock(&md_config.log_mutex);
        g_message("%s: accepting connection from %s:%d",
                  mdCollectorGetName(collector), ip,
                  ((struct sockaddr_in *)peer)->sin_port);
        pthread_mutex_unlock(&md_config.log_mutex);
    } else if (peer->sa_family == AF_INET6) {
        char  straddr[INET6_ADDRSTRLEN];
        inet_ntop(AF_INET6, &(((struct sockaddr_in6 *)peer)->sin6_addr),
                  straddr, sizeof(straddr));
        pthread_mutex_lock(&md_config.log_mutex);
        g_message("%s: accepting connection from %s:%d",
                  mdCollectorGetName(collector), straddr,
                  ((struct sockaddr_in6 *)peer)->sin6_port);
        pthread_mutex_unlock(&md_config.log_mutex);
    }

    collector->stats->restarts++;

    *ctx = (void *)collector;

    return TRUE;
}

static void
smFreeMaps(
    mdConfig_t  *cfg)
{
    unsigned int  i = 0;

    if (cfg->maps) {
        smFieldMap_t *cmap = NULL;
        smFieldMap_t *nmap = NULL;
        for (cmap = cfg->maps; cmap; cmap = cmap->next) {
            smHashTableFree(cmap->table);
            g_free(cmap->name);
            for (i = 0; i < cmap->count; i++) {
                g_free(cmap->labels[i]);
            }
            free(cmap->labels);
        }
        cmap = cfg->maps;
        while (cmap) {
            detachHeadOfSLL((mdSLL_t **)&(cfg->maps),
                            (mdSLL_t **)&cmap);
            nmap = cmap->next;
            g_slice_free(smFieldMap_t, cmap);
            cmap = nmap;
        }
    }
}

static gboolean
mdPrivc_Setup(
    GError **err)
{
    struct passwd *pwe = NULL;
    struct group  *gre = NULL;

    if (geteuid() == 0) {
        /* We're root. Parse user and group names. */
        if (become_user) {
            if (!(pwe = getpwnam(become_user))) {
                g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                            "Cannot become user %s: %s.",
                            become_user, strerror(errno));
                return FALSE;
            }

            /* By default, become new user's user and group. */
            new_user = pwe->pw_uid;
            new_group = pwe->pw_gid;
            if (become_group) {
                if (!(gre = getgrnam(become_group))) {
                    g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                                "Cannot become group %s: %s.",
                                become_group, strerror(errno));
                    return FALSE;
                }

                /* Override new group if set */
                new_group = gre->gr_gid;
            }
        }
    } else {
        /* We're not root. If we have options, the user is confused, and
         * we should straighten him out by killing the process. */
        if (become_user) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "Cannot become user %s: not root.",
                        become_user);
            return FALSE;
        }
        if (become_group) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "Cannot become group %s: not root.",
                        become_group);
            return FALSE;
        }
    }

    /* All done. */
    return TRUE;
}

static gboolean
mdPrivc_Become(
    GError **err)
{
    /* Die if we've already become */
    if (did_become) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "not dropping privileges, already did so");
        return FALSE;
    }

    /* Short circuit if we're not root */
    if (geteuid() != 0) {return TRUE;}

    /* Allow app to warn if not dropping */
    if (new_user == 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_NODROP,
                    "not dropping privileges (use --become-user to do so)");
        return FALSE;
    }

    /* Okay. Do the drop. */

    /* Drop ancillary group privileges while we're still root */
    if (setgroups(1, &new_group) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "couldn't drop ancillary groups: %s", strerror(errno));
        return FALSE;
    }
#if LINUX_PRIVHACK
    /* Change to group */
    if (setregid(new_group, new_group) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "couldn't become group %u: %s", new_group, strerror(errno));
        return FALSE;
    }

    /* Lose root privileges */
    if (setreuid(new_user, new_user) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "couldn't become user %u: %s", new_user, strerror(errno));
        return FALSE;
    }
#else  /* if LINUX_PRIVHACK */
    /* Change to group */
    if (setgid(new_group) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "couldn't become group %u: %s", new_group, strerror(errno));
        return FALSE;
    }

    /* Lose root privileges */
    if (setuid(new_user) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "couldn't become user %u: %s", new_user, strerror(errno));
        return FALSE;
    }
#endif /* if LINUX_PRIVHACK */
    /* All done. */
    did_become = TRUE;
    return TRUE;
}
