/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_json.c
 *
 *  Contains most of the JSON-y functions.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.11
 *
 *  Copyright 2024 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM24-1038
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_ctx.h"
#include "mediator_util.h"
#include "mediator_inf.h"
#include "mediator_print.h"
#include "mediator_json.h"


/*
 *    2024.03.14 Note for timestamps: When printing SM generated records (DNS
 *    DEDUP, etc) these functions take the milliseconds values and convert to
 *    a yfTime_t.  In the individual SM generators (mediator_dedup.c), the
 *    milliseconds value is filled from a yfTime_t when the record was
 *    prepared for export.  To avoid this double conversion, it would be
 *    better to either move the printing implementation of these records into
 *    individual files or make the yfTime_t available on the records.
 *
 *    One potential issue is that we also need to handle the case where SM is
 *    forwarding these records read from a previous SM.
 */


/* RFC 4627 -
 * Any character may be escaped.  If the character is in the Basic
 * Multilingual Plane (U+0000 through U+FFFF), then it may be
 * represented as a six-character sequence: a reverse solidus, followed
 * by the lowercase letter u, followed by four hexadecimal digits that
 * encode the character's code point.  The hexadecimal letters A though
 * F can be upper or lowercase.  So, for example, a string containing
 * only a single reverse solidus character may be represented as
 * "\u005C".
 */
gboolean
mdJsonifyEscapeChars(
    mdBuf_t        *mdbuf,
    const uint8_t  *data,
    size_t          datalen)
{
    size_t   i;
    uint8_t  ch;

    for (i = 0; i < datalen; i++) {
        ch = data[i];
        if (ch < 32 || ch >= 127) {
            mdBufAppendPrintf(mdbuf, "\\u%04x", ch);
        } else if (ch == '\\') {
            mdBufAppend(mdbuf, "\\\\");
        } else if (ch == '"') {
            mdBufAppend(mdbuf, "\\\"");
        } else {
            mdBufAppendChar(mdbuf, ch);
        }
    }

    return TRUE;
}


/**
 *  Quotes any non-printable characters in 'cp' using UTF-8 style
 *  encoding and appends the result to 'str'.
 */
gboolean
mdJsonifyEscapeCharsGStringAppend(
    GString        *str,
    const uint8_t  *cp,
    size_t          len)
{
    while (len) {
        if (*cp < 32 || *cp >= 127) {
            g_string_append_printf(str, "\\u%04x", *cp);
        } else if (*cp == '\\') {
            g_string_append(str, "\\\\");
        } else if (*cp == '"') {
            g_string_append(str, "\\\"");
        } else {
            g_string_append_c(str, *cp);
        }
        ++cp;
        --len;
    }
    return TRUE;
}


gboolean
mdJsonifyDNSRRRecord(
    md_dnsrr_t   *rec,
    gboolean      is_ipv6,
    mdBuf_t      *buf)
{
    size_t    buftest;
    yfTime_t  yftime;
    char      sabuf[40];
    char      dabuf[40];
    gboolean  did_rrdata;

    mdBufAppend(buf, "{\"dns\":{\"flowStartMilliseconds\":\"");
    yfTimeFromMilli(&yftime, rec->flowStartMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppend(buf, "\",\"flowStartNanoseconds\":\"");
    yfTimeFromNTP(&yftime, rec->flowStartNanoseconds, FALSE);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_NANO);
    mdBufAppend(buf, "\",");

    if (is_ipv6) {
        const char  testsip[16] = {
            '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0',
            '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0'
        };
        if (memcmp(rec->sourceIPv6Address, testsip,
                   sizeof(rec->sourceIPv6Address)))
        {
            md_util_print_ip6_addr(sabuf, rec->sourceIPv6Address);
            md_util_print_ip6_addr(dabuf, rec->destinationIPv6Address);
            mdBufAppendPrintf(buf, ("\"sourceIPv6Address\":\"%s\","
                                    "\"destinationIPv6Address\":\"%s\","),
                              sabuf, dabuf);
        }
    } else {
        md_util_print_ip4_addr(sabuf, rec->sourceIPv4Address);
        md_util_print_ip4_addr(dabuf, rec->destinationIPv4Address);
        mdBufAppendPrintf(buf, ("\"sourceIPv4Address\":\"%s\","
                                "\"destinationIPv4Address\":\"%s\","),
                          sabuf, dabuf);
    }

    if (rec->protocolIdentifier) {
        mdBufAppendPrintf(buf, "\"protocolIdentifier\":%d,",
                          rec->protocolIdentifier);
    }

    if (rec->vlanId) {
        mdBufAppendPrintf(buf, "\"vlanId\":\"%u,", rec->vlanId);
    }

    if (rec->sourceTransportPort) {
        mdBufAppendPrintf(buf, "\"sourceTransportPort\":%d,",
                          rec->sourceTransportPort);
    }

    if (rec->destinationTransportPort) {
        mdBufAppendPrintf(buf, "\"destinationTransportPort\":%d,",
                          rec->destinationTransportPort);
    }

    if (rec->yafFlowKeyHash) {
        mdBufAppendPrintf(buf, "\"yafFlowKeyHash\":%u,",
                          rec->yafFlowKeyHash);
    }

    if (rec->observationDomainId) {
        mdBufAppendPrintf(buf, "\"observationDomainId\":%u,",
                          rec->observationDomainId);
    }


    mdBufAppendPrintf(buf, ("\"dnsRRSection\":%d,\"dnsNXDomain\":%d,"
                            "\"dnsAuthoritative\":\"%s\","
                            "\"dnsQRType\":%d, \"dnsTTL\":%u, \"dnsID\":%d,"),
                      rec->dnsRRSection, rec->dnsNXDomain,
                      ((rec->dnsAuthoritative) ? "True" : "False"),
                      rec->dnsQRType, rec->dnsTTL, rec->dnsID);

    if (rec->rrname.buf) {
        mdBufAppend(buf, "\"dnsQName\":\"");
        mdJsonifyEscapeChars(buf, rec->rrname.buf, rec->rrname.len);
        mdBufAppend(buf, "\",");
    } /* else - query may be for the root server which is NULL */

    if (rec->dnsQueryResponse == 0) {
        /* query */
        mdBufChop(buf, 1);
        mdBufAppend(buf, "}}\n");
        return TRUE;
    }

    /* Print the type of the DNS record in the switch().  For A and AAAA
     * records, also print the data and set `did_rrdata` to TRUE.  The data
     * for all other record types is handled after the switch.  `buftest`
     * checks whether any rrdata info is printed. */
    buftest = mdBufLen(buf);
    did_rrdata = FALSE;

    switch (rec->dnsQRType) {
      case 1:
        did_rrdata = TRUE;
        if (rec->rrdata.len) {
            uint32_t  sip;
            memcpy(&sip, rec->rrdata.buf, sizeof(uint32_t));
            md_util_print_ip4_addr(sabuf, sip);
            mdBufAppendPrintf(buf, "\"A\":\"%s\"", sabuf);
        }
        break;
      case 2:
        mdBufAppend(buf, "\"dnsNSDName\":\"");
        break;
      case 5:
        mdBufAppend(buf, "\"dnsCName\":\"");
        break;
      case 6:
        mdBufAppend(buf, "\"dnsSOAMName\":\"");
        break;
      case 12:
        mdBufAppend(buf, "\"dnsPTRDName\":\"");
        break;
      case 15:
        mdBufAppend(buf, "\"dnsMXExchange\":\"");
        break;
      case 16:
        mdBufAppend(buf, "\"dnsTXTData\":\"");
        break;
      case 28:
        did_rrdata = TRUE;
        if (rec->rrdata.len) {
            md_util_print_ip6_addr(sabuf, rec->rrdata.buf);
            mdBufAppendPrintf(buf, "\"AAAA\":\"%s\"", sabuf);
        }
        break;
      case 33:
        mdBufAppend(buf, "\"dnsSRVTarget\":\"");
        break;
      case 46:
        mdBufAppend(buf, "\"dnsSigner\":\"");
        break;
      case 47:
        mdBufAppend(buf, "\"dnsHashData\":\"");
        break;
    }

    if (mdBufLen(buf) == buftest) {
        /* no rrname/rrdata */
        /* remove the comma at the end of dnsQName */
        mdBufChop(buf, 1);
    } else if (!did_rrdata) {
        /* print the data */
        mdJsonifyEscapeChars(buf, rec->rrdata.buf, rec->rrdata.len);
        mdBufAppendChar(buf, '\"');
    }

    mdBufAppend(buf, "}}\n");
    return TRUE;
}



gboolean
mdJsonifyDNSRecord(
    yaf_dnsQR_t  *dns,
    mdBuf_t      *buf)
{
    size_t  buftest;

    mdBufAppendPrintf(buf, ("\"dnsRRSection\":%d,\"dnsNXDomain\":%d,"
                            "\"dnsAuthoritative\":\"%s\","
                            "\"dnsQRType\":%d, \"dnsTTL\":%u, \"dnsID\":%d,"),
                      dns->dnsRRSection, dns->dnsNXDomain,
                      ((dns->dnsAuthoritative) ? "True" : "False"),
                      dns->dnsQRType, dns->dnsTTL, dns->dnsID);

    if (dns->dnsQName.buf) {
        mdBufAppend(buf, "\"dnsQName\":\"");
        mdJsonifyEscapeChars(buf, dns->dnsQName.buf, dns->dnsQName.len);
        mdBufAppend(buf, "\",");
    } /* else - query may be for the root server which is NULL*/

    /* To check whether data is printed */
    buftest = mdBufLen(buf);

    if (dns->dnsQRType == 1) {
        yaf_dnsA_t *aflow = NULL;
        char        ipaddr[20];
        while ((aflow = (yaf_dnsA_t *)FBSTLNEXT(&(dns->dnsRRList), aflow))) {
            if (aflow->sourceIPv4Address) {
                md_util_print_ip4_addr(ipaddr, aflow->sourceIPv4Address);
                mdBufAppendPrintf(buf, "\"A\":\"%s\"", ipaddr);
            }
        }
    } else if (dns->dnsQRType == 2) {
        yaf_dnsNS_t *ns = NULL;
        while ((ns = (yaf_dnsNS_t *)FBSTLNEXT(&(dns->dnsRRList), ns))) {
            mdBufAppend(buf, "\"dnsNSDName\":\"");
            mdJsonifyEscapeChars(buf, ns->dnsNSDName.buf,
                                 ns->dnsNSDName.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 5) {
        yaf_dnsCNAME_t *c = NULL;
        while ((c = (yaf_dnsCNAME_t *)FBSTLNEXT(&(dns->dnsRRList), c))) {
            mdBufAppend(buf, "\"dnsCName\":\"");
            mdJsonifyEscapeChars(buf, c->dnsCName.buf,
                                 c->dnsCName.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 12) {
        yaf_dnsPTR_t *ptr = NULL;
        while ((ptr = (yaf_dnsPTR_t *)FBSTLNEXT(&(dns->dnsRRList), ptr))) {
            mdBufAppend(buf, "\"dnsPTRDName\":\"");
            mdJsonifyEscapeChars(buf, ptr->dnsPTRDName.buf,
                                 ptr->dnsPTRDName.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 15) {
        yaf_dnsMX_t *mx = NULL;
        while ((mx = (yaf_dnsMX_t *)FBSTLNEXT(&(dns->dnsRRList), mx))) {
            mdBufAppend(buf, "\"dnsMXExchange\":\"");
            mdJsonifyEscapeChars(buf, mx->dnsMXExchange.buf,
                                 mx->dnsMXExchange.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 28) {
        yaf_dnsAAAA_t *aa = NULL;
        char           ipaddr[40];
        while ((aa = (yaf_dnsAAAA_t *)FBSTLNEXT(&(dns->dnsRRList), aa))) {
            md_util_print_ip6_addr(ipaddr, (uint8_t *)&(aa->sourceIPv6Address));
            mdBufAppendPrintf(buf, "\"AAAA\":\"%s\"", ipaddr);
        }
    } else if (dns->dnsQRType == 16) {
        yaf_dnsTXT_t *txt = NULL;
        while ((txt = (yaf_dnsTXT_t *)FBSTLNEXT(&(dns->dnsRRList), txt))) {
            mdBufAppend(buf, "\"dnsTXTData\":\"");
            mdJsonifyEscapeChars(buf, txt->dnsTXTData.buf,
                                 txt->dnsTXTData.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 33) {
        yaf_dnsSRV_t *srv = NULL;
        while ((srv = (yaf_dnsSRV_t *)FBSTLNEXT(&(dns->dnsRRList), srv))) {
            mdBufAppend(buf, "\"dnsSRVTarget\":\"");
            mdJsonifyEscapeChars(buf, srv->dnsSRVTarget.buf,
                                 srv->dnsSRVTarget.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 6) {
        yaf_dnsSOA_t *soa = NULL;
        while ((soa = (yaf_dnsSOA_t *)FBSTLNEXT(&(dns->dnsRRList), soa))) {
            mdBufAppend(buf, "\"dnsSOAMName\":\"");
            mdJsonifyEscapeChars(buf, soa->dnsSOAMName.buf,
                                 soa->dnsSOAMName.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 46) {
        yaf_dnsRRSig_t *rr = NULL;
        while ((rr = (yaf_dnsRRSig_t *)FBSTLNEXT(&(dns->dnsRRList), rr))) {
            mdBufAppend(buf, "\"dnsSigner\":\"");
            mdJsonifyEscapeChars(buf, rr->dnsSigner.buf,
                                 rr->dnsSigner.len);
            mdBufAppendChar(buf, '\"');
        }
    } else if (dns->dnsQRType == 47) {
        yaf_dnsNSEC_t *nsec = NULL;
        while ((nsec = (yaf_dnsNSEC_t *)FBSTLNEXT(&(dns->dnsRRList), nsec))) {
            mdBufAppend(buf, "\"dnsHashData\":\"");
            mdJsonifyEscapeChars(buf, nsec->dnsHashData.buf,
                                 nsec->dnsHashData.len);
            mdBufAppendChar(buf, '\"');
        }
    }

    /* no rrname/rrdata */
    if (mdBufLen(buf) == buftest) {
        /* remove the comma at the end of dnsQName */
        mdBufChop(buf, 1);
    }

    return TRUE;
}

size_t
mdPrintJsonStats(
    yaf_process_stats_t  *stats,
    const char           *name,
    FILE                 *lfp,
    GError              **err)
{
    GString *str = NULL;
    char     ipaddr[20];
    size_t   rc;

    md_util_print_ip4_addr(ipaddr, stats->exporterIPv4Address);
    str = g_string_sized_new(512);

    g_string_append(str, "{\"stats\":{");

    g_string_append_printf(str, "\"exportedFlowTotalCount\":%" PRIu64 ",",
                           stats->exportedFlowRecordTotalCount);
    g_string_append_printf(str, "\"packetTotalCount\":%" PRIu64 ",",
                           stats->packetTotalCount);
    g_string_append_printf(str, "\"droppedPacketTotalCount\":%" PRIu64 ",",
                           stats->droppedPacketTotalCount);
    g_string_append_printf(str, "\"ignoredPacketTotalCount\":%" PRIu64 ",",
                           stats->ignoredPacketTotalCount);
    g_string_append_printf(str, "\"expiredFragmentCount\":%u,",
                           stats->expiredFragmentCount);
    g_string_append_printf(str, "\"assembledFragmentCount\":%u,",
                           stats->assembledFragmentCount);
    g_string_append_printf(str, "\"flowTableFlushEvents\":%u,",
                           stats->flowTableFlushEvents);
    g_string_append_printf(str, "\"flowTablePeakCount\":%u,",
                           stats->flowTablePeakCount);
    g_string_append_printf(str, "\"exporterIPv4Address\":\"%s\",", ipaddr);
    g_string_append_printf(str, "\"exportingProcessId\":%d,",
                           stats->exportingProcessId);
    g_string_append_printf(str, "\"meanFlowRate\":%u,",
                           stats->meanFlowRate);
    g_string_append_printf(str, "\"meanPacketRate\":%u,",
                           stats->meanPacketRate);
    g_string_append_printf(str, "\"observationDomainId\":%d,",
                           stats->observationDomainId);
    g_string_append_printf(str, "\"exporterName\":\"%s\"", name);

    g_string_append(str, "}}\n");

    rc = fwrite(str->str, 1, str->len, lfp);

    if (rc != str->len) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                    "Error writing %d b ytes to file: %s\n",
                    (unsigned int)str->len, strerror(errno));
        return 0;
    }

    g_string_free(str, TRUE);

    return rc;
}

size_t
mdPrintJsonTombstone(
    yaf_tombstone_t  *tombstone,
    const char       *name,
    FILE             *lfp,
    GError          **err)
{
    GString *str = NULL;
    size_t   rc;
    int      firstIter = 1;
    void    *entry = NULL;

    MD_UNUSED_PARAM(name);

    str = g_string_sized_new(512);

    g_string_append(str, "{\"tombstone\":{");

    g_string_append_printf(str, "\"observationDomainId\":%" PRIu32 ",",
                           tombstone->observationDomainId);
    g_string_append_printf(str, "\"exportingProcessId\":%" PRIu32 ",",
                           tombstone->exportingProcessId);
    g_string_append_printf(str, "\"exporterConfiguredId\":%" PRIu16 ",",
                           tombstone->exporterConfiguredId);
    g_string_append_printf(str, "\"tombstoneId\":%" PRIu32 ",",
                           tombstone->tombstoneId);
    g_string_append_printf(str, "\"observationTimeSeconds\":%" PRIu32 ",",
                           tombstone->observationTimeSeconds);

    g_string_append(str, "\"tombstoneAccessList\":[");

    while ((entry = fbSubTemplateListGetNextPtr(&(tombstone->accessList),
                                                entry)))
    {
        if (firstIter) {
            firstIter = 0;
            g_string_append_printf(
                str, ("{\"tombstoneAccessEntry\":{"
                      "\"certToolId\":%" PRIu32
                      ",\"observationTimeSeconds\":%" PRIu32 "}}"),
                ((yaf_tombstone_access_t *)entry)->certToolId,
                ((yaf_tombstone_access_t *)entry)->observationTimeSeconds);
        } else {
            g_string_append_printf(
                str, (",{\"tombstoneAccessEntry\":{"
                      "\"certToolId\":%" PRIu32
                      ",\"observationTimeSeconds\":%" PRIu32 "}}"),
                ((yaf_tombstone_access_t *)entry)->certToolId,
                ((yaf_tombstone_access_t *)entry)->observationTimeSeconds);
        }
    }

    g_string_append(str, "]");

    g_string_append(str, "}}\n");

    rc = fwrite(str->str, 1, str->len, lfp);

    if (rc != str->len) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                    "Error writing %d bytes to file: %s\n",
                    (unsigned int)str->len, strerror(errno));
        return 0;
    }

    g_string_free(str, TRUE);

    return rc;
}

int
mdJsonifyDNSDedupRecord(
    FILE            *fp,
    mdBuf_t         *buf,
    const uint8_t   *rec,
    gboolean         print_last,
    gboolean         base64,
    GError         **err)
{
    size_t          rc = 0;
    char            sabuf[40];
    md_dns_dedup_t *record = (md_dns_dedup_t *)rec;
    yfTime_t        yftime;
    gboolean        did_rrdata;

    mdBufAppend(buf, "{\"dns\":{\"flowStartMilliseconds\":\"");
    yfTimeFromMilli(&yftime, record->flowStartMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppend(buf, "\",");

    if (print_last) {
        mdBufAppend(buf, "\"flowEndMilliseconds\":\"");
        yfTimeFromMilli(&yftime, record->flowEndMilliseconds);
        mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
        mdBufAppend(buf, "\",");
    }

    mdBufAppend(buf, "\"flowStartNanoseconds\":\"");
    yfTimeFromNTP(&yftime, record->flowStartNanoseconds, FALSE);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_NANO);
    mdBufAppend(buf, "\",");

    if (print_last) {
        mdBufAppend(buf, "\"flowEndNanoseconds\":\"");
        yfTimeFromNTP(&yftime, record->flowEndNanoseconds, FALSE);
        mdBufAppendTime(buf, &yftime, MD_TIME_FMT_NANO);
        mdBufAppend(buf, "\",");
    }

    mdBufAppendPrintf(buf, "\"dnsQRType\":%d,", record->rrtype);

    if (print_last) {
        mdBufAppendPrintf(buf, "\"dnsHitCount\":%d,\"dnsTTL\":%d,",
                          record->dnsHitCount, record->dnsTTL);
    }

    if (record->rrname.len) {
        /* this is a dns dedup record so we have to subtract one from the name
         * since we added one for the hash table (string hash requires null
         * char at end of string) */
        mdBufAppend(buf, "\"dnsQName\":\"");
        if (base64) {
            mdBufAppendBase64(buf, record->rrname.buf, record->rrname.len - 1);
        } else {
            mdJsonifyEscapeChars(buf, (uint8_t *)record->rrname.buf,
                                 record->rrname.len - 1);
        }
        mdBufAppend(buf, "\",");
    }

    /* Print the type of the DNS record in the switch.  For A and AAAA
     * records, also print the data and set `did_rrdata` to TRUE.  The data
     * for all other record types is handled after the switch. */
    did_rrdata = FALSE;

    switch (record->rrtype) {
      case 1:
        did_rrdata = TRUE;
        if (record->sourceIPv4Address) {
            md_util_print_ip4_addr(sabuf, record->sourceIPv4Address);
            mdBufAppendPrintf(buf, "\"A\":\"%s\"", sabuf);
        }
        break;
      case 2:
        mdBufAppend(buf, "\"dnsNSDName\":\"");
        break;
      case 5:
        mdBufAppend(buf, "\"dnsCName\":\"");
        break;
      case 6:
        mdBufAppend(buf, "\"dnsSOAMName\":\"");
        break;
      case 12:
        mdBufAppend(buf, "\"dnsPTRDName\":\"");
        break;
      case 15:
        mdBufAppend(buf, "\"dnsMXExchange\":\"");
        break;
      case 28:
        did_rrdata = TRUE;
        md_util_print_ip6_addr(sabuf, record->sourceIPv6Address);
        mdBufAppendPrintf(buf, "\"AAAA\":\"%s\"", sabuf);
        break;
      case 16:
        mdBufAppend(buf, "\"dnsTXTData\":\"");
        break;
      case 33:
        mdBufAppend(buf, "\"dnsSRVTarget\":\"");
        break;
      case 46:
        mdBufAppend(buf, "\"dnsSigner\":\"");
        break;
      case 47:
        mdBufAppend(buf, "\"dnsHashData\":\"");
        break;
      default:
        /* if we found no rrData then we need to snip the trailing comma from
         * the previous field.  */
        did_rrdata = TRUE;
        mdBufChop(buf, 1);
        break;
    }

    if (did_rrdata) {
        /* already handled; nothing to do */
    } else if (base64) {
        mdBufAppendBase64(buf, record->rrdata.buf, record->rrdata.len - 1);
        mdBufAppendChar(buf, '\"');
    } else {
        mdJsonifyEscapeChars(buf, record->rrdata.buf, record->rrdata.len);
        mdBufAppendChar(buf, '\"');
    }

    if (record->mapname.len) {
        mdBufAppend(buf, ",\"observationDomainName\":\"");
        mdBufAppendVarfield(buf, &(record->mapname));
        mdBufAppendChar(buf, '\"');
    }

    mdBufAppend(buf, "}}\n");

    rc = mdBufWrite(buf, fp, "", err);
    if (!rc) {
        return -1;
    }

    return rc;
}

int
mdJsonifySSLDedupRecord(
    FILE           *fp,
    mdBuf_t        *buf,
    const uint8_t  *rec,
    GError        **err)
{
    size_t          rc = 0;
    const md_ssl_dedup_t *ssl = (md_ssl_dedup_t *)rec;
    yfTime_t        yftime;

    mdBufAppend(buf, "{\"ssl\":{\"firstSeen\":\"");

    yfTimeFromNTP(&yftime, ssl->flowStartNanoseconds, FALSE);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_NANO);
    mdBufAppend(buf, "\",\"lastSeen\":\"");

    yfTimeFromNTP(&yftime, ssl->flowEndNanoseconds, FALSE);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_NANO);

    mdBufAppend(buf,  "\",\"sslCertSerialNumber\":\"");

    mdBufAppendHexdump(buf, ssl->sslCertSerialNumber.buf,
                       ssl->sslCertSerialNumber.len, TRUE);

    if (ssl->mapname.len) {
        mdBufAppend(buf, "\",\"observationDomainName\":\"");
        mdBufAppendVarfield(buf, &(ssl->mapname));
    }

    mdBufAppendPrintf(buf, ("\",\"observedDataTotalCount\":%" PRIu64
                            ",\"sslCertIssuerCommonName\":\""),
                      ssl->observedDataTotalCount);

    mdBufAppendVarfield(buf, &(ssl->sslCertIssuerCommonName));

    mdBufAppend(buf, "\"}}\n");


    rc = mdBufWrite(buf, fp, "", err);
    if (!rc) {
        return -1;
    }

    return rc;
}

int
mdJsonifyDedupRecord(
    FILE        *fp,
    mdBuf_t     *buf,
    const char  *prefix,
    md_dedup_t  *rec,
    GError     **err)
{
    size_t      rc = 0;
    yfTime_t    yftime;

    mdBufAppend(buf, "{\"dedup\":{\"firstSeen\":\"");

    yfTimeFromMilli(&yftime, rec->monitoringIntervalStartMilliSeconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppend(buf, "\",\"lastSeen\":\"");

    yfTimeFromMilli(&yftime, rec->monitoringIntervalEndMilliSeconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);

    if (rec->sourceIPv4Address != rec->yafFlowKeyHash) {
        /* deduped on IP, so print it */
        if (rec->sourceIPv4Address == 0) {
            mdBufAppend(buf, "\",\"sourceIPv6Address\":\"");
            mdBufAppendIP6(buf, rec->sourceIPv6Address);
        } else {
            mdBufAppend(buf, "\",\"sourceIPv4Address\":\"");
            mdBufAppendIP4(buf, rec->sourceIPv4Address);
        }
    }
    /* else deduped on hash, not IP, so don't print IP */

    mdBufAppendPrintf(buf, ("\",\"yafFlowKeyHash\":%u,"
                            "\"observedDataTotalCount\":%" PRIu64 ","
                            "\"flowStartMilliseconds\":\""),
                      rec->yafFlowKeyHash, rec->observedDataTotalCount);

    /* Flow's start time */
    yfTimeFromMilli(&yftime, rec->flowStartMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);

    mdBufAppend(buf, ",\"flowStartNanoseconds\":\"");
    yfTimeFromNTP(&yftime, rec->flowStartNanoseconds, FALSE);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_NANO);

    if (rec->mapname.len) {
        mdBufAppend(buf, "\",\"observationDomainName\":\"");
        mdBufAppendVarfield(buf, &(rec->mapname));
    }

    mdBufAppendChar(buf, '\"');

    if (rec->observedData.len) {
        mdBufAppendPrintf(buf, ",\"%s\":\"", prefix);
        mdBufAppendVarfield(buf, &(rec->observedData));
        mdBufAppendChar(buf, '\"');
    } else if (rec->sslCertSerialNumber1.len) {
        mdBufAppendPrintf(buf, ",\"sslCertificateChain\":[{\""
                          "sslCertSerialNumber\":\"");
        mdBufAppendHexdump(buf, rec->sslCertSerialNumber1.buf,
                           rec->sslCertSerialNumber1.len, TRUE);
        mdBufAppend(buf, "\", \"sslCertIssuerCommonName\":\"");
        mdBufAppendVarfield(buf, &(rec->sslCertIssuerCommonName1));
        mdBufAppend(buf, "\"}");
        if (rec->sslCertSerialNumber2.len) {
            mdBufAppend(buf, ",{\"sslCertSerialNumber\":\"");
            mdBufAppendHexdump(buf, rec->sslCertSerialNumber2.buf,
                               rec->sslCertSerialNumber2.len, TRUE);
            mdBufAppend(buf, "\", \"sslCertIssuerCommonName\":\"");
            mdBufAppendVarfield(buf, &(rec->sslCertIssuerCommonName2));
            mdBufAppend(buf, "\"}]");
        } else {
            mdBufAppendChar(buf, ']');
        }
    }

    mdBufAppend(buf, "}}\n");

    rc = mdBufWrite(buf, fp, "", err);
    if (!rc) {
        return -1;
    }

    return rc;
}

void
mdJsonifySSLCertBase64(
    mdBuf_t             *buf,
    const fbVarfield_t  *cert)
{
    gchar  *base1;

    /* remove '},' */
    mdBufChop(buf, 2);

    base1 = g_base64_encode((const guchar *)cert->buf, cert->len);
    mdBufAppendPrintf(buf, ",\"sslCertificate\":\"%s\"},", base1);
    g_free(base1);
}
