/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_stat.c
 *
 *  Handles mediator/yaf stats
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.11
 *
 *  Copyright 2024 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM24-1038
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_inf.h"
#include "mediator_core.h"
#include "mediator_util.h"
#include "mediator_stat.h"

static GTimer *md_start = NULL;

/**
 * mdStatInit
 *
 *
 *
 */
void
mdStatInit(
    void)
{
    md_start = g_timer_new();
    g_timer_start(md_start);
}


/**
 * mdStatGetTimer
 *
 *
 */
GTimer *
mdStatGetTimer(
    void)
{
    return md_start;
}


/**
 * mdLogStats
 *
 * Log YAF process statistics
 *
 */
void
mdLogStats(
    yaf_process_stats_t  *stats,
    const char           *colname)
{
    char      ipaddr[20];
    time_t    cur_time = time(NULL);
    uint64_t  ms = cur_time * 1000;
    uint64_t  uptime = 0;
    uint64_t  days = 0;
    uint64_t  hours = 0;
    uint64_t  mins = 0;

    if (ms > stats->systemInitTimeMilliseconds) {
        uptime = (ms - stats->systemInitTimeMilliseconds) / 1000;
        days = uptime / 86400;
        uptime -= (days * 86400);
        hours = (uptime / 3600);
        uptime -= (hours * 3600);
        mins = (uptime / 60);
        uptime -= (mins * 60);
    }

    md_util_print_ip4_addr(ipaddr, stats->exporterIPv4Address);
    g_message(
        ("%s: YAF ID: %d IP: %s Uptime: %" PRIu64 "d"
         ":%" PRIu64 "h:%" PRIu64 "m:%" PRIu64 "s"),
        colname, stats->exportingProcessId, ipaddr, days, hours, mins, uptime);
#if 0
    if (stats->sysInitTime) {
        g_message(
            ("%s: YAF Uptime: %" PRIu64 " Days, %" PRIu64 " Hours,"
             " %" PRIu64 " Minutes, %" PRIu64 " Seconds"),
            colname, days, hours, mins, uptime);
    }
#endif  /* 0 */

    g_message(
        ("%s: YAF Flows: %" PRIu64 " Packets: %" PRIu64 " Dropped: %" PRIu64
         " Ignored: %" PRIu64 " Out of Sequence: %" PRIu64
         " Expired Frags: %u Assembled Frags: %u"),
        colname, stats->exportedFlowRecordTotalCount,
        stats->packetTotalCount, stats->droppedPacketTotalCount,
        stats->ignoredPacketTotalCount, stats->notSentPacketTotalCount,
        stats->expiredFragmentCount, stats->assembledFragmentCount);

#if 0
    g_message("Exported Flows: %" PRIu64, stats->exportedFlowTotalCount);
    g_message("Packets Processed: %" PRIu64, stats->packetTotalCount);
    g_message("Dropped Packets: %" PRIu64, stats->droppedPacketTotalCount);
    g_message("Ignored Packets: %" PRIu64, stats->ignoredPacketTotalCount);
    g_message("Rejected Out of Sequence Packets: %" PRIu64,
              stats->rejectedPacketTotalCount);
    g_message("Expired Fragments: %u", stats->expiredFragmentCount);
    g_message("Assembled Fragments: %u", stats->assembledFragmentCount);
#endif  /* 0 */
}


/**
 * mdStatUpdate
 *
 * Log Mediator process statistics
 *
 */
void
mdStatUpdate(
    md_stats_t  *stats)
{
    uint64_t  seconds = g_timer_elapsed(md_start, NULL);
    uint64_t  uptime = seconds;
    uint64_t  days, hours, mins;

    days = uptime / 86400;
    uptime -= (days * 86400);
    hours = uptime / 3600;
    uptime -= (hours * 3600);
    mins = uptime / 60;
    uptime -= (mins * 60);

    g_message(
        ("SM: Uptime: %" PRIu64 "d:%" PRIu64 "h:%" PRIu64 "m:"
         "%" PRIu64 "s, Total Flows: %" PRIu64 ", Filtered: %" PRIu64 ", "
         "Stats: %" PRIu64 ", DNS: %" PRIu64 ", Other: %" PRIu64
         ", UDP-uniflows: %" PRIu64),
        days, hours, mins, uptime, stats->recvd_flows, stats->recvd_filtered,
        stats->recvd_stats, stats->dns, stats->nonstd_flows, stats->uniflows);
}


void
mdStatDump(
    mdConfig_t  *cfg,
    md_stats_t  *stats)
{
    mdStatUpdate(stats);
    mdExporterUpdateStats(cfg, TRUE);
    mdCollectorUpdateStats(cfg);
}
