/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_util.h
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.12
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-0935
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#ifndef MD_UTIL
#define MD_UTIL

#include "mediator_templates.h"

#if ENABLE_SKIPSET
#include SKIPSET_HEADER_NAME
#endif
#if ENABLE_SKPREFIXMAP
#include <silk/skprefixmap.h>
#endif

/* Formats for printing timestamps */
typedef enum mdTimePrintFormat_en {
    /* Used for file names: "yyyymmddHHMMSS" */
    MD_TIME_FMT_FILENAME,
    /* Time only (UNUSED): "HH:MM:SS: */
    MD_TIME_FMT_TIMEONLY,
    /* Date-time without fractional seconds: "yyyy-mm-dd HH:MM:SS" */
    MD_TIME_FMT_NOFRAC,
    /* Date-time with milliseconds (msec): "yyyy-mm-dd HH:MM:SS.sss" */
    MD_TIME_FMT_MILLI,
    /* Date-time with microseconds (usec): "yyyy-mm-dd HH:MM:SS.ssssss" */
    MD_TIME_FMT_MICRO,
    /* Date-time with nanoseconds (nsec): "yyyy-mm-dd HH:MM:SS.sssssssss" */
    MD_TIME_FMT_NANO
} mdTimePrintFormat_t;

#define PRINT_TIME_FMT  MD_TIME_FMT_MILLI

typedef struct smVarHashKey_st {
    size_t    len;
    uint8_t  *val;
} smVarHashKey_t;

typedef struct md_asn_tlv_st {
    uint8_t   class : 2;
    uint8_t   p_c   : 1;
    uint8_t   tag   : 5;
} md_asn_tlv_t;

#if ENABLE_SKIPSET
/* A wrapper over the SiLK IPSet structure to avoid opening the same file
 * multiple times. */
typedef struct mdIPSet_st {
    skipset_t     *ipset;
    gchar         *path;
    /* pthread_mutex_t     mutex; */
    unsigned int   ref;
} mdIPSet_t;
#endif  /* ENABLE_SKIPSET */

#if ENABLE_SKPREFIXMAP
/* A wrapper over the SiLK PMAP structure to avoid opening the same file
 * multiple times. */
typedef struct mdPMap_st {
    skPrefixMap_t         *pmap;
    /* path to the file, as given by the user */
    gchar                 *path;
    /* name of the map if it contains one; else the basename of 'path'; value
     * references other memory and it is not allocated */
    const gchar           *mapname;
    /* pthread_mutex_t         mutex; */
    unsigned int           ref;
    skPrefixMapContent_t   content;
} mdPMap_t;
#endif  /* ENABLE_SKPREFIXMAP */


mdBuf_t *
mdBufNew(
    size_t   initial);

void
mdBufAppend(
    mdBuf_t     *mdBuf,
    const char  *str);

/**
 *    Appends a Base-64 Encoding of the first `datalen` octets of `data` to
 *    `mdBuf`.  If `datalen` is 0, `mdBuf` is unchanged.
 */
void
mdBufAppendBase64(
    mdBuf_t        *mdBuf,
    const uint8_t  *data,
    size_t          datalen);

void
mdBufAppendChar(
    mdBuf_t  *mdBuf,
    char      c);

void
mdBufAppendGString(
    mdBuf_t        *mdBuf,
    const GString  *str);

/**
 *    Appends a hex-string representation the first `datalen` octets of `data`
 *    to `mdBuf`.  If `datalen` is 0, `mdBuf` is unchanged.
 *
 *    Each octet is converted to a two-digit hexadecimal number and appended
 *    to `mdBuf` with an optional separator.
 *
 *    If no_space is FALSE, the numbers are separated by spaces.  For example,
 *    "foo\n" becomes "66 6f 6f 0a".
 *
 *    If no_space is TRUE, a leading 0x is added nothing separates the
 *    numbers.  For example, "foo\n" becomes "0x666f6f0a".
 */
void
mdBufAppendHexdump(
    mdBuf_t        *mdBuf,
    const uint8_t  *data,
    size_t          datalen,
    gboolean        no_space);

void
mdBufAppendIP4(
    mdBuf_t   *mdBuf,
    uint32_t   ip4);

void
mdBufAppendIP6(
    mdBuf_t        *mdBuf,
    const uint8_t  *ip6);

void
mdBufAppendLen(
    mdBuf_t        *mdBuf,
    const uint8_t  *str,
    size_t          len);

void
mdBufAppendPrintf(
    mdBuf_t     *mdBuf,
    const char  *format,
    ...)
    __attribute__((format(printf, 2, 3)));

void
mdBufAppendTime(
    mdBuf_t              *mdBuf,
    const yfTime_t       *t,
    mdTimePrintFormat_t   format);

void
mdBufAppendVarfield(
    mdBuf_t             *mdBuf,
    const fbVarfield_t  *vf);

/**
 *    Removes `num_to_remove` octets from the end of `mdBuf`.
 */
void
mdBufChop(
    mdBuf_t  *mdBuf,
    size_t    num_to_remove);

void
mdBufClear(
    mdBuf_t  *mdBuf);

void
mdBufFree(
    mdBuf_t  *mdBuf);

/**
 *    Returns the octet as position `pos` in `mdBuf`.  Returns NUL if `pos` is
 *    greater than mdBufLen().
 */
char
mdBufGetChar(
    const mdBuf_t  *mdBuf,
    size_t          pos);

/**
 *    Returns a newly allocated char array containing the octets in `mdBuf`
 *    from `start_pos` to one less than `end_pos`.  The returned array is NUL
 *    terminated.  The caller must use g_free() to free the returned arrray.
 *
 *    The returned array contains (1 + end_pos - start_pos) octets with any
 *    unfilled octets set to NUL.
 *
 *    Returns NULL if `end_pos` is less than `start_pos`.
 */
char *
mdBufGetSubstring(
    const mdBuf_t  *mdBuf,
    size_t          start_pos,
    size_t          end_pos);

/**
 *    Returns the length of string not including the terminating NUL.
 */
/*    size_t
 *    mdBufLen(
 *        const mdBuf_t  *mdBuf);
 */
#define mdBufLen(md_buf)  ((size_t)((md_buf)->b_cp - (md_buf)->b_buf))

/**
 *    Returns a newly allocated char array containing the contents of `mdBuf`.
 *    The returned array is NUL terminated.  If `len` is non-NULL, its
 *    referent is set to the number of octets in the returned array not
 *    including the terminating NUL.
 */
char *
mdBufMakeCopy(
    const mdBuf_t  *mdBuf,
    size_t         *len);

/**
 *    Sets the length of the octets in `mdBuf` to `newlen`.  Has no effect if
 *    `newlen` is greater than mdBufLen().
 */
void
mdBufTruncate(
    mdBuf_t  *mdBuf,
    size_t    newlen);

/**
 *    Writes the contents of `mdBuf` to `fp`.  On success, clears `mdBuf` and
 *    returns the number of octets written.  On failure, sets `err`, using
 *    `exporter_name` in the error message, leaves `mdBuf` unchanged, and
 *    returns 0.
 */
size_t
mdBufWrite(
    mdBuf_t     *mdBuf,
    FILE        *fp,
    const char  *exporter_name,
    GError     **err);

int
md_util_hexdump_append(
    char           *dest,
    size_t         *drem,
    const uint8_t  *src,
    size_t          len);

int
md_util_hexdump_append_nospace(
    char           *dest,
    size_t         *drem,
    const uint8_t  *src,
    size_t          len);

/**
 *    Appends the first `datalen` octets of `data` to `str` in the style of
 *    hexdump(1) with a block of hexadecimal numbers and their ASCII
 *    interpretations to the side.  `lpfx` is a title to write along the left
 *    hand margin on all lines.
 */
void
md_util_hexdump_g_string_append(
    GString        *str,
    const char     *lpfx,
    const uint8_t  *data,
    uint32_t        datalen);

void
md_util_print_tcp_flags(
    GString  *str,
    uint8_t   flags);

void
md_util_print_ip6_addr(
    char           *ipaddr_buf,
    const uint8_t  *ipaddr);

void
md_util_print_ip4_addr(
    char      *ipaddr_buf,
    uint32_t   ip);

uint32_t
md_util_flow_key_hash(
    md_main_template_t  *rec);

uint32_t
md_util_rev_flow_key_hash(
    md_main_template_t  *rec);

void
md_util_time_g_string_append(
    GString              *str,
    const yfTime_t       *t,
    mdTimePrintFormat_t   format);

uint16_t
md_util_decode_asn1_length(
    uint8_t **buffer,
    size_t   *len);

uint16_t
md_util_decode_asn1_sequence(
    uint8_t **buffer,
    size_t   *len);

void *
detachFromEndOfDLL(
    mdDLL_t **head,
    mdDLL_t **tail);

void
detachThisEntryOfDLL(
    mdDLL_t **head,
    mdDLL_t **tail,
    mdDLL_t  *entry);

void
attachHeadToDLL(
    mdDLL_t **head,
    mdDLL_t **tail,
    mdDLL_t  *newEntry);

guint
sm_octet_array_hash(
    gconstpointer   v);

gboolean
sm_octet_array_equal(
    gconstpointer   v1,
    gconstpointer   v2);

void
sm_octet_array_key_destroy(
    gpointer   data);

smVarHashKey_t *
sm_new_hash_key(
    uint8_t  *val,
    size_t    len);

uint16_t
md_util_decode_length(
    uint8_t   *buffer,
    uint16_t  *offset);

uint16_t
md_util_decode_tlv(
    md_asn_tlv_t  *tlv,
    uint8_t       *buffer,
    uint16_t      *offset);

uint8_t
md_util_asn1_sequence_count(
    uint8_t   *buffer,
    uint16_t   seq_len);

void
md_util_compress_file(
    const char  *file,
    const char  *dest);

smHashTable_t *
smCreateHashTable(
    size_t           length,
    GDestroyNotify   freeKeyfn,
    GDestroyNotify   freeValfn);

gpointer
smHashLookup(
    smHashTable_t  *table,
    uint8_t        *key);

void
smHashTableInsert(
    smHashTable_t  *table,
    uint8_t        *key,
    uint8_t        *value);

void
smHashTableFree(
    smHashTable_t  *table);

void
smHashTableRemove(
    smHashTable_t  *table,
    uint8_t        *key);

uint32_t
smFieldMapTranslate(
    const smFieldMap_t  *map,
    const mdFullFlow_t  *flow);

void
md_free_hash_key(
    gpointer   v1);

GString *
sm_util_move_file(
    const char  *file,
    const char  *new_dir);

#if ENABLE_SKIPSET
mdIPSet_t *
mdUtilIPSetOpen(
    const char  *path,
    GError     **err);
void
mdUtilIPSetClose(
    mdIPSet_t  *ipset);
#endif  /* ENABLE_SKIPSET */

#if ENABLE_SKPREFIXMAP
mdPMap_t *
mdUtilPMapOpen(
    const char  *path,
    GError     **err);
void
mdUtilPMapClose(
    mdPMap_t  *pmap);
#endif  /* ENABLE_SKPREFIXMAP */

#endif  /* MD_UTIL */
