/*
 *  Copyright 2004-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  probeconflabel.c
 *
 *  Applies labels from sensor.conf file to IPFIX records, similar to SiLK's
 *  rwflowpack tool.
 *
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.12
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-0935
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "probeconf.h"
#include <silk/skipaddr.h>
#include <silk/skipset.h>
#include <silk/sksite.h>
#include <silk/utils.h>


#define NETWORK_NULL      0
#define NETWORK_EXTERNAL  1
#define NETWORK_INTERNAL  2

#define NUM_NETWORKS      3

static const char *net_names[NUM_NETWORKS] = {
    "null",
    "external",
    "internal"
};


static uint8_t     RW_IN;
static uint8_t     RW_OUT;
static uint8_t     RW_IN_WEB;
static uint8_t     RW_OUT_WEB;
static uint8_t     RW_IN_NULL;
static uint8_t     RW_OUT_NULL;
static uint8_t     RW_INT2INT;
static uint8_t     RW_EXT2EXT;

/* The name of the silk.conf file. */
static const char *silktypesensor_silk_conf;

/* The name of the sensor.conf file. */
static const char *silktypesensor_sensor_conf;

/* Whether Type+Sensor Labeling is enabled (that is, was a SILK_CONFIG block
 * seen by the parser) */
static gboolean  silktypesensor_enabled;

/* To avoid a lookup for every record, create arrays that map an ID (the array
 * index) to its name or description. */
static fbVarfield_t *sensor_name = NULL;
static fbVarfield_t *sensor_desc = NULL;
static fbVarfield_t *type_name = NULL;

static void
mdSkpcStringCachesBuild(
    void)
{
    char                sn[SK_MAX_STRLEN_SENSOR + 1];
    char                ftn[SK_MAX_STRLEN_FLOWTYPE + 1];
    const char         *sd;
    sk_sensor_iter_t    siter;
    sk_flowtype_iter_t  ftiter;
    sk_sensor_id_t      sid;
    sk_flowtype_id_t    ftid;

    sid = sksiteSensorGetMaxID();
    ftid = sksiteFlowtypeGetMaxID();

    if (sid == (sk_sensor_id_t)(-1)) {
        g_error("No SiLK sensors have been defined");
    }
    if (ftid == (sk_flowtype_id_t)(-1)) {
        g_error("No SiLK flowtypes have been defined");
    }

    sensor_name = g_new0(fbVarfield_t, sid + 1);
    sensor_desc = g_new0(fbVarfield_t, sid + 1);
    type_name = g_new0(fbVarfield_t, ftid + 1);

    sksiteSensorIterator(&siter);
    while (sksiteSensorIteratorNext(&siter, &sid)) {
        sd = sksiteSensorGetDescription(sid);
        if (sd) {
            sensor_desc[sid].buf = (uint8_t *)sd;
            sensor_desc[sid].len = strlen(sd);
        }

        sensor_name[sid].len = sksiteSensorGetName(sn, sizeof(sn), sid);
        sensor_name[sid].buf = (uint8_t *)g_strdup(sn);
    }

    sksiteFlowtypeIterator(&ftiter);
    while (sksiteFlowtypeIteratorNext(&ftiter, &ftid)) {
        type_name[ftid].len = sksiteFlowtypeGetType(ftn, sizeof(ftn), ftid);
        type_name[ftid].buf = (uint8_t *)g_strdup(ftn);
    }
}

static void
mdSkpcStringCachesFree(
    void)
{
    sk_sensor_id_t    sid;
    sk_flowtype_id_t  ftid;
    unsigned int      i;

    /* descriptions strings are not copied */
    g_free(sensor_desc);

    /* free sensor name copies then the array */
    if (sensor_name) {
        sid = sksiteSensorGetMaxID();
        if (sid != (sk_sensor_id_t)(-1)) {
            for (i = 0; i <= sid; ++i) {
                g_free(sensor_name[i].buf);
            }
        }
        g_free(sensor_name);
    }

    /* free type name copies then the array */
    if (type_name) {
        ftid = sksiteFlowtypeGetMaxID();
        if (ftid != (sk_flowtype_id_t)(-1)) {
            for (i = 0; i <= ftid; ++i) {
                g_free(type_name[i].buf);
            }
        }
        g_free(type_name);
    }
}


/*
 *    Verify that enough information is present on the sensor to categorize a
 *    flow record.
 */
gboolean
mdSkpcVerifySensorLabeler(
    md_skpc_sensor_t  *sensor)
{
    unsigned int          nd_type_count[MD_SKPC_NUM_NETDECIDER_TYPES];
    md_skpc_netdecider_t *decider;
    unsigned int          block_count;
    unsigned int          set_count;
    unsigned int          if_count;
    unsigned int          i;

    /* There is a single class, so no per-class verification is
     * necessary.  Make certain each sensor has snmp interface values,
     * ipblocks, or IPsets to categorize each flow. */

    /* If the source and destination networks are set, we're
     * good to go. */
    if ((sensor->fixed_network[0] != MD_SKPC_NETWORK_ID_INVALID)
        && (sensor->fixed_network[1] != MD_SKPC_NETWORK_ID_INVALID))
    {
        return TRUE;
    }

    /*
     * Verify that we have enough information to determine the
     * flowtype for every flow.  These are the rules:
     *
     * 1. One of NET-interface, NET-ipblock, or NET-ipset must be
     * specified, where NET is either "internal" or "external".
     *
     * 2. A single sensor may not mix NET-interface, NET-ipblock, or
     * NET-ipset statements, with the exception that a null-interface
     * is always allowed.
     *
     * 3. Only one network may claim the remainder.
     *
     * 4. Using 'remainder' requires that another NET is set.
     *
     * 5. If only one of internal-* or external-* is set, set the
     * other to the remaining values, unless the 'remainder' is
     * already claiming them.
     */

    /* for this sensor, count how many of each decider type (e.g.,
     * MD_SKPC_IBLOCK (NET-ipblock)) have been specified */
    memset(nd_type_count, 0, sizeof(nd_type_count));
    for (i = 0; i < NUM_NETWORKS; ++i) {
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t, i);
        assert(decider->nd_type < MD_SKPC_NUM_NETDECIDER_TYPES);
        ++nd_type_count[decider->nd_type];
    }

    /* get number of deciders for ipblocks and for interfaces */
    if_count = (nd_type_count[MD_SKPC_INTERFACE]
                + nd_type_count[MD_SKPC_REMAIN_INTERFACE]);
    block_count = (nd_type_count[MD_SKPC_IPBLOCK]
                   + nd_type_count[MD_SKPC_REMAIN_IPBLOCK]);
    set_count = (nd_type_count[MD_SKPC_IPSET]
                 + nd_type_count[MD_SKPC_REMAIN_IPSET]);

    if (nd_type_count[MD_SKPC_NEG_IPBLOCK]) {
        /* this should never happen, since there is no way to set this
         * from the sensor.conf file  */
        g_error("Negated IPblock logic not implemented");
    }
    if (nd_type_count[MD_SKPC_NEG_IPSET]) {
        /* this should never happen, since there is no way to set this
         * from the sensor.conf file  */
        g_error("Negated IPset logic not implemented");
    }

    /* make certain ipblocks, ipsets, or interfaces are specified, and
     * make certain something in addition to null-* is specified */
    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             NETWORK_NULL);
    if ((block_count + if_count + set_count == 0)
        || ((block_count + if_count + set_count == 1)
            && (decider->nd_type != MD_SKPC_UNSET)))
    {
        fprintf(stderr, ("Cannot verify sensor %s:\n"
                         "\tMust specify source-network and"
                         " destination-network, or at least one\n"
                         "\tof %s- and %s-interface, %s- and %s-ipblock,"
                         " or %s- and %s-ipset\n"),
                mdSkpcSensorGetName(sensor),
                net_names[NETWORK_EXTERNAL], net_names[NETWORK_INTERNAL],
                net_names[NETWORK_EXTERNAL], net_names[NETWORK_INTERNAL],
                net_names[NETWORK_EXTERNAL], net_names[NETWORK_INTERNAL]);
        return FALSE;
    }

    /* only one 'remainder' is allowed */
    if ((nd_type_count[MD_SKPC_REMAIN_IPBLOCK]
         + nd_type_count[MD_SKPC_REMAIN_INTERFACE]
         + nd_type_count[MD_SKPC_REMAIN_IPSET])
        > 1)
    {
        fprintf(stderr, ("Cannot verify sensor '%s':\n"
                         "\tOnly one network value may use 'remainder'\n"),
                mdSkpcSensorGetName(sensor));
        return FALSE;
    }

    /* handle case where NET-ipblocks are set */
    if (block_count) {
        if (block_count == NUM_NETWORKS) {
            /* all networks were specified. nothing else to check */
            assert(if_count == 0);
            assert(set_count == 0);
            return TRUE;
        }
        /* block_count is either 1 or 2 */
        assert(block_count <= 2);

        if (set_count) {
            fprintf(stderr, ("Cannot verify sensor '%s':\n"
                             "\tCannot mix <NET>-ipblock and <NET>-ipset\n"),
                    mdSkpcSensorGetName(sensor));
            return FALSE;
        }

        /* only valid mix of NET-ipblock and NET-interface is for the
         * interfaces to be on the NULL network */
        if (if_count) {
            decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                                     NETWORK_NULL);
            switch (decider->nd_type) {
              case MD_SKPC_INTERFACE:
              case MD_SKPC_REMAIN_INTERFACE:
                --if_count;
                break;

              default:
                break;
            }
            if (if_count) {
                fprintf(stderr, ("Cannot verify sensor '%s':\n"
                                 "\tCannot mix <NET>-interface"
                                 " and <NET>-ipblock\n"),
                        mdSkpcSensorGetName(sensor));
                return FALSE;
            }
        }

        /* if an ipblock has claimed the 'remainder', verify we have
         * IPs specified elsewhere and return */
        if (nd_type_count[MD_SKPC_REMAIN_IPBLOCK] == 1) {
            /* need at least one IP address to be specified for
             * 'remainder' to work */
            if (block_count == 1) {
                fprintf(stderr, ("Cannot verify sensor '%s':\n"
                                 "\tCannot set ipblocks to remainder when"
                                 " no other networks hold IP blocks\n"),
                        mdSkpcSensorGetName(sensor));
                return FALSE;
            }
            return TRUE;
        }

        /* if either EXTERNAL or INTERNAL is unset, set to remainder */
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_EXTERNAL);
        if (decider->nd_type == MD_SKPC_UNSET) {
            assert(g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_INTERNAL).nd_type == MD_SKPC_IPBLOCK);
            decider->nd_type = MD_SKPC_REMAIN_IPBLOCK;
        }
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_INTERNAL);
        if (decider->nd_type == MD_SKPC_UNSET) {
            assert(g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_EXTERNAL).nd_type == MD_SKPC_IPBLOCK);
            decider->nd_type = MD_SKPC_REMAIN_IPBLOCK;
        }

        return TRUE;
    }

    /* handle case where NET-ipsets are set */
    if (set_count) {
        if (set_count == NUM_NETWORKS) {
            /* all networks were specified. nothing else to check */
            assert(if_count == 0);
            assert(block_count == 0);
            return TRUE;
        }
        /* set_count is either 1 or 2 */
        assert(set_count <= 2);

        if (block_count) {
            fprintf(stderr, ("Cannot verify sensor '%s':\n"
                             "\tCannot mix <NET>-ipset and <NET>-ipblock\n"),
                    mdSkpcSensorGetName(sensor));
            return FALSE;
        }

        /* only valid mix of NET-ipset and NET-interface is for the
         * interfaces to be on the NULL network */
        if (if_count) {
            decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                                     NETWORK_NULL);
            switch (decider->nd_type) {
              case MD_SKPC_INTERFACE:
              case MD_SKPC_REMAIN_INTERFACE:
                --if_count;
                break;

              default:
                break;
            }
            if (if_count) {
                fprintf(stderr, ("Cannot verify sensor '%s':\n"
                                 "\tCannot mix <NET>-interface"
                                 " and <NET>-ipset\n"),
                        mdSkpcSensorGetName(sensor));
                return FALSE;
            }
        }

        /* if an ipset has claimed the 'remainder', verify we have
         * IPs specified elsewhere and return */
        if (nd_type_count[MD_SKPC_REMAIN_IPSET] == 1) {
            /* need at least one IP address to be specified for
             * 'remainder' to work */
            if (set_count == 1) {
                fprintf(stderr, ("Cannot verify sensor '%s':\n"
                                 "\tCannot set ipsets to remainder when"
                                 " no other networks hold IP sets\n"),
                        mdSkpcSensorGetName(sensor));
                return FALSE;
            }
            return TRUE;
        }

        /* if either EXTERNAL or INTERNAL is unset, set to remainder */
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_EXTERNAL);
        if (decider->nd_type == MD_SKPC_UNSET) {
            assert(g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_INTERNAL).nd_type == MD_SKPC_IPSET);
            decider->nd_type = MD_SKPC_REMAIN_IPSET;
        }
        decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_INTERNAL);
        if (decider->nd_type == MD_SKPC_UNSET) {
            assert(g_array_index(sensor->decider, md_skpc_netdecider_t,
                                 NETWORK_EXTERNAL).nd_type == MD_SKPC_IPSET);
            decider->nd_type = MD_SKPC_REMAIN_IPSET;
        }

        return TRUE;
    }

    /* handle case where NET-interfaces are set */
    if (0 == if_count || block_count > 0 || set_count > 0) {
        g_error("Programmer error: NET-interfaces are set");
    }

    if (if_count == NUM_NETWORKS) {
        /* all networks were specified. nothing else to check */
        return TRUE;
    }
    /* if_count is either 1 or 2. */
    assert(if_count <= 2);

    /* if network has claimed the 'remainder', verify we have interfaces
     * specified elsewhere and return */
    if (nd_type_count[MD_SKPC_REMAIN_INTERFACE] == 1) {
        /* need at least one interface to be specified for 'remainder' to
         * work */
        if (if_count == 1) {
            fprintf(stderr, ("Cannot verify sensor '%s':\n"
                             "\tCannot set interfaces to remainder when"
                             " no other networks have set interfaces\n"),
                    mdSkpcSensorGetName(sensor));
            return FALSE;
        }
        return TRUE;
    }

    /* if either EXTERNAL or INTERNAL is unset, set to remainder */
    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             NETWORK_EXTERNAL);
    if (decider->nd_type == MD_SKPC_UNSET) {
        assert(g_array_index(sensor->decider, md_skpc_netdecider_t,
                             NETWORK_INTERNAL).nd_type == MD_SKPC_INTERFACE);
        decider->nd_type = MD_SKPC_REMAIN_INTERFACE;
    }
    decider = &g_array_index(sensor->decider, md_skpc_netdecider_t,
                             NETWORK_INTERNAL);
    if (decider->nd_type == MD_SKPC_UNSET) {
        assert(g_array_index(sensor->decider, md_skpc_netdecider_t,
                             NETWORK_EXTERNAL).nd_type == MD_SKPC_INTERFACE);
        decider->nd_type = MD_SKPC_REMAIN_INTERFACE;
    }

    return TRUE;
}


static void
mdSkpcAppendTypeSensor(
    GArray           **array,
    sk_flowtype_id_t   ftid,
    sk_sensor_id_t     sid)
{
    md_silk_type_sensor_t *typ_sen_rec;

    if (NULL == *array) {
        /* create an array to return the results */
        *array = g_array_sized_new(FALSE, TRUE, sizeof(*typ_sen_rec),
                                   DEFAULT_ARRAY_SIZE);
    }

    /* increment the array and get a handle to the new element */
    g_array_set_size(*array, 1 + (*array)->len);
    typ_sen_rec = &g_array_index(*array, md_silk_type_sensor_t,
                                 (*array)->len - 1);

    /* initialize the element */
    typ_sen_rec->silkSensorName = sensor_name[sid];
    typ_sen_rec->silkSensorDescription = sensor_desc[sid];
    typ_sen_rec->silkTypeName = type_name[ftid];
    typ_sen_rec->silkSensorId = sid;
    typ_sen_rec->silkFlowtypeId = ftid;
}


/**
 *    Whether the Type and Sensor Labeling is Enabled.  That is, was there a
 *    SILK_CONFIG block in the super_mediator config file and were the
 *    silk.conf and sensor.conf files successfully read?
 */
gboolean
mdSkpcSilkTypeSensorEnabled(
    void)
{
    return silktypesensor_enabled;
}


GArray *
mdSkpcSilkTypeSensorLabelFlow(
    mdFullFlow_t  *flow)
{
#if !defined(ENABLE_SKTYPESENSOR) || !ENABLE_SKTYPESENSOR
    /* handle case when disabled in config.h */
    return NULL;
#else
    const md_skpc_probe_t  *probe;
    const md_skpc_sensor_t *sensor;
    GArray                 *array;
    sk_sensor_id_t          sid;
    unsigned int            i;

    /* get the SILK_PROBE specified on the COLLECTOR */
    probe = NULL;
    if (flow->collector_silk_probe_name) {
        probe = mdSkpcProbeLookupByName(flow->collector_silk_probe_name);
    }
    if (NULL == probe) {
        return NULL;
    }

    array = NULL;

    /* visit each sensor that uses the probe */
    for (i = 0; i < probe->sensor_list->len; ++i) {
        sensor = g_array_index(probe->sensor_list, md_skpc_sensor_t *, i);

        /* check whether to discard the flow */
        if (sensor->filter && sensor->filter->len &&
            mdSkpcSensorCheckFilters(sensor, flow))
        {
            continue;
        }

        sid = mdSkpcSensorGetID(sensor);

        /* Check flow's destination against monitored network */
        if (1 == mdSkpcSensorTestFlowInterfaces(sensor, flow,
                                                NETWORK_INTERNAL,
                                                MD_SKPC_DIR_DST))
        {
            /* Flow's DESTINATION is the monitored network: must be
             * either incoming or internal-to-internal.  Determine
             * which by comparing flow's source against the monitored
             * network. */
            if (1 == mdSkpcSensorTestFlowInterfaces(sensor, flow,
                                                    NETWORK_INTERNAL,
                                                    MD_SKPC_DIR_SRC))
            {
                /* Flow reached the monitoring point from the inside
                 * of network, must be internal-to-internal. */
                mdSkpcAppendTypeSensor(&array, RW_INT2INT, sid);
            } else {
                /* Otherwise the flow is incoming */
                mdSkpcAppendTypeSensor(&array, RW_IN, sid);
            }

            /* Check flow's source against the monitored network */
        } else if (1 == mdSkpcSensorTestFlowInterfaces(sensor, flow,
                                                       NETWORK_INTERNAL,
                                                       MD_SKPC_DIR_SRC))
        {
            /* Flow's SOURCE is the monitored network: must be
             * outgoing or out-null.  (We would have caught int2int
             * traffic above.)  Determine which by comparing flow's
             * destination against the null network. */
            if (1 == mdSkpcSensorTestFlowInterfaces(sensor, flow,
                                                    NETWORK_NULL,
                                                    MD_SKPC_DIR_DST))
            {
                /* Flow went to the null destination */
                mdSkpcAppendTypeSensor(&array, RW_OUT_NULL, sid);
            } else {
                /* Otherwise the flow is outgoing */
                mdSkpcAppendTypeSensor(&array, RW_OUT, sid);
            }

            /* Assume flow's SOURCE is the external network. Determine
             * whether it is ext2ext or in-null by comparing flow's
             * destination against the null network. */
        } else if (1 == mdSkpcSensorTestFlowInterfaces(sensor, flow,
                                                       NETWORK_NULL,
                                                       MD_SKPC_DIR_DST))
        {
            /* Flow went to the null destination */
            mdSkpcAppendTypeSensor(&array, RW_IN_NULL, sid);
        } else {
            mdSkpcAppendTypeSensor(&array, RW_EXT2EXT, sid);
        }
    } /* for (sensors-per-probe) */

    return array;
#endif  /* ENABLE_SKTYPESENSOR */
}


gboolean
mdSkpcSilkTypeSensorSet(
    const char  *silk_conf,
    const char  *sensor_conf,
    GError     **err)
{
    if (silk_conf) {
        if (!g_file_test(silk_conf, G_FILE_TEST_IS_REGULAR)) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "SILK_CONF_PATH expects a valid file");
            return FALSE;
        }
        if (silktypesensor_silk_conf) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "SILK_CONFIG already has a SILK_CONF_PATH setting");
            return FALSE;
        }
        silktypesensor_silk_conf = g_strdup(silk_conf);
    }
    if (sensor_conf) {
        if (!g_file_test(sensor_conf, G_FILE_TEST_IS_REGULAR)) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "SENSOR_CONF_PATH expects a valid file");
            return FALSE;
        }
        if (silktypesensor_sensor_conf) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "SILK_CONFIG already has a SENSOR_CONF_PATH setting");
            return FALSE;
        }
        silktypesensor_sensor_conf = g_strdup(sensor_conf);
    }

    return TRUE;
}

void
mdSkpcSilkTypeSensorFree(
    void)
{
    if (silktypesensor_enabled) {
        mdSkpcTeardown();
        silktypesensor_enabled = FALSE;
    }
    mdSkpcStringCachesFree();
    g_free((char *)silktypesensor_sensor_conf);
    g_free((char *)silktypesensor_silk_conf);
    silktypesensor_sensor_conf = NULL;
    silktypesensor_silk_conf = NULL;
}

/*
 *  Bootstrap the SiLK world and load the sensor.conf file.
 */
gboolean
mdSkpcSilkTypeSensorLoad(
    GError **err)
{
    if (silktypesensor_enabled) {
        /* Previously initialized */
        return TRUE;
    }

    if (NULL == silktypesensor_silk_conf) {
        if (NULL == silktypesensor_sensor_conf) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "The SILK_CONFIG block requires both"
                        "SILK_CONF_PATH and SENSOR_CONF_PATH");
        } else {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "The SILK_CONFIG block requires SILK_CONF_PATH");
        }
        return FALSE;
    } else if (NULL == silktypesensor_sensor_conf) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "The SILK_CONFIG block requires SENSOR_CONF_PATH");
        return FALSE;
    }

    if (!app_registered) {
        skAppRegister(g_get_prgname());
        app_registered++;
    }

    /* silk.conf */
    if (0 != sksiteSetConfigPath(silktypesensor_silk_conf)) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Unable to set SILK_CONF_PATH to '%s': Path too long?",
                    silktypesensor_silk_conf);
        return FALSE;
    }
    if (0 != sksiteConfigure(1)) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Error configuring SiLK with SILK_CONF_PATH \"%s\"",
                    silktypesensor_silk_conf);
        return FALSE;
    }

    RW_IN       = sksiteFlowtypeLookup("in");
    RW_OUT      = sksiteFlowtypeLookup("out");
    RW_IN_WEB   = sksiteFlowtypeLookup("iw");
    RW_OUT_WEB  = sksiteFlowtypeLookup("ow");
    RW_IN_NULL  = sksiteFlowtypeLookup("innull");
    RW_OUT_NULL = sksiteFlowtypeLookup("outnull");
    RW_INT2INT  = sksiteFlowtypeLookup("int2int");
    RW_EXT2EXT  = sksiteFlowtypeLookup("ext2ext");

    /* sensor.conf */
    if (0 != mdSkpcSetup()) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Unable to setup probe config file parser");
        return FALSE;
    }

    mdSkpcNetworkAdd(NETWORK_NULL,     net_names[NETWORK_NULL]);
    mdSkpcNetworkAdd(NETWORK_EXTERNAL, net_names[NETWORK_EXTERNAL]);
    mdSkpcNetworkAdd(NETWORK_INTERNAL, net_names[NETWORK_INTERNAL]);

    if (0 != mdSkpcParse(silktypesensor_sensor_conf)) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Error configuring SiLK with SENSOR_CONF_PATH \"%s\"",
                    silktypesensor_sensor_conf);
        return FALSE;
    }

    mdSkpcStringCachesBuild();

    silktypesensor_enabled = TRUE;
    return TRUE;
}
