/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_core.c
 *
 *  IPFIX mediator for filtering, DNS deduplication, and other mediator-like
 *  things: Contains the primary processing functions.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso, Matt Coates
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.12
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-0935
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_core.h"
#include "mediator_filter.h"
#include "mediator_util.h"
#include "mediator_stat.h"
#include "mediator_dns.h"
#include "mediator_dedup.h"
#include "mediator_print.h"
#include "mediator_ssl.h"
#include "infomodel.h"

/* fixbuf 2.x uses char* as the type of the name of info elements in
 * fbInfoElementSpec_t; wrap this around string literals to quiet compiler
 * warnings */
#define C(String) (char *)String

static fbTemplate_t *sm_sub_ssl_tmpl = NULL;

/* The flow record template.  Uses md_main_template_t to hold a record and
 * YAF_SILK_FLOW_TID as the TID. */
static fbInfoElementSpec_t  md_main_template_spec[] = {
    {C("flowStartMilliseconds"),               8,  YTF_MILLI },
    {C("flowEndMilliseconds"),                 8,  YTF_MILLI },

    {C("octetTotalCount"),                     8,  YTF_TOTAL },
    {C("reverseOctetTotalCount"),              8,  YTF_TOTAL | YTF_REV },
    {C("packetTotalCount"),                    8,  YTF_TOTAL },
    {C("reversePacketTotalCount"),             8,  YTF_TOTAL | YTF_REV },

    {C("octetDeltaCount"),                     8,  YTF_DELTA },
    {C("reverseOctetDeltaCount"),              8,  YTF_DELTA | YTF_REV },
    {C("packetDeltaCount"),                    8,  YTF_DELTA },
    {C("reversePacketDeltaCount"),             8,  YTF_DELTA | YTF_REV },

    {C("sourceIPv6Address"),                   16, YTF_IP6 },
    {C("destinationIPv6Address"),              16, YTF_IP6 },
    {C("sourceIPv4Address"),                   4,  YTF_IP4 },
    {C("destinationIPv4Address"),              4,  YTF_IP4 },

    {C("sourceTransportPort"),                 2,  0 },
    {C("destinationTransportPort"),            2,  0 },
    {C("flowAttributes"),                      2,  0 },
    {C("reverseFlowAttributes"),               2,  YTF_REV },
    {C("protocolIdentifier"),                  1,  0 },
    {C("flowEndReason"),                       1,  0 },
    {C("silkAppLabel"),                        2,  0 },
    {C("reverseFlowDeltaMilliseconds"),        4,  YTF_MILLI | YTF_REV },

    {C("tcpSequenceNumber"),                   4,  YTF_TCP },
    {C("reverseTcpSequenceNumber"),            4,  YTF_TCP | YTF_REV },
    {C("initialTCPFlags"),                     1,  YTF_TCP },
    {C("unionTCPFlags"),                       1,  YTF_TCP },
    {C("reverseInitialTCPFlags"),              1,  YTF_TCP | YTF_REV },
    {C("reverseUnionTCPFlags"),                1,  YTF_TCP | YTF_REV },

    /* MAC-specific information */
    {C("vlanId"),                              2,  0 },
    {C("reverseVlanId"),                       2,  YTF_REV },
    {C("ingressInterface"),                    4,  YTF_DAGIF },
    {C("egressInterface"),                     4,  YTF_DAGIF },
    {C("ipClassOfService"),                    1,  0 },
    {C("reverseIpClassOfService"),             1,  YTF_REV },
    {C("mplsTopLabelStackSection"),            3,  YTF_MPLS },
    {C("mplsLabelStackSection2"),              3,  YTF_MPLS },
    {C("mplsLabelStackSection3"),              3,  YTF_MPLS },
    /* add an obs id */
    {C("paddingOctets"),                       1,  YTF_PAD },
    {C("observationDomainId"),                 4,  0 },
    {C("yafFlowKeyHash"),                      4,  0 },

    /* nDPI elements will only be exported when present upon collection */
    {C("nDPIL7Protocol"),                      2,  YTF_NDPI },
    {C("nDPIL7SubProtocol"),                   2,  YTF_NDPI },

    {C("flowStartMicroseconds"),               8,  YTF_MICRO },
    {C("flowEndMicroseconds"),                 8,  YTF_MICRO },
    {C("reverseFlowDeltaMicroseconds"),        8,  YTF_MICRO | YTF_REV },

    {C("flowStartNanoseconds"),                8,  YTF_NANO },
    {C("flowEndNanoseconds"),                  8,  YTF_NANO },
    {C("reverseFlowDeltaNanoseconds"),         8,  YTF_NANO | YTF_REV },

    {C("subTemplateMultiList"),                FB_IE_VARLEN, YTF_LIST },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_process_stats_spec[] = {
    {C("observationDomainId"),                4, 0 },
    {C("exportingProcessId"),                 4, 0 },
    {C("exporterIPv4Address"),                4, 0 },
    {C("observationTimeSeconds"),             4, 0 },
    {C("systemInitTimeMilliseconds"),         8, 0 },
    {C("exportedFlowRecordTotalCount"),       8, 0 },
    {C("packetTotalCount"),                   8, 0 },
    {C("droppedPacketTotalCount"),            8, 0 },
    {C("ignoredPacketTotalCount"),            8, 0 },
    {C("notSentPacketTotalCount"),            8, 0 },
    {C("expiredFragmentCount"),               4, 0 },
    {C("assembledFragmentCount"),             4, 0 },
    {C("flowTableFlushEventCount"),           4, 0 },
    {C("flowTablePeakCount"),                 4, 0 },
    {C("meanFlowRate"),                       4, 0 },
    {C("meanPacketRate"),                     4, 0 },
    FB_IESPEC_NULL
};

/* tombstone records from YAF 2.10.0 only */
static fbInfoElementSpec_t  yaf_tombstone_210_spec[] = {
    {C("exporterConfiguredId"),               2, 0 },
    {C("exporterUniqueId"),                   2, 0 },
    {C("tombstoneId"),                        4, 0 },
    {C("subTemplateList"),                    FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* tombstone records from YAF 2.10.0 only */
static fbInfoElementSpec_t  yaf_tombstone_access_210_spec[] = {
    {C("exportingProcessId"),                 4, 0 },
    {C("observationTimeSeconds"),             4, 0 },
    FB_IESPEC_NULL
};

/* tombstone records as of YAF 2.11.0 */
static fbInfoElementSpec_t  yaf_tombstone_spec[] = {
    {C("observationDomainId"),                4, 0 },
    {C("exportingProcessId"),                 4, 0 },
    {C("exporterConfiguredId"),               2, 0 },
    {C("paddingOctets"),                      6, 0 },
    {C("tombstoneId"),                        4, 0 },
    {C("observationTimeSeconds"),             4, 0 },
    {C("tombstoneAccessList"),                FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* tombstone records as of YAF 2.11.0 */
static fbInfoElementSpec_t  yaf_tombstone_access_spec[] = {
    {C("certToolId"),                         4, 0 },
    {C("observationTimeSeconds"),             4, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_entropy_spec[] = {
    {C("payloadEntropy"),                     1, 0 },
    {C("reversePayloadEntropy"),              1, YTF_REV },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_tcp_spec[] = {
    /* TCP-specific information */
    {C("tcpSequenceNumber"),                  4, 0 },
    {C("initialTCPFlags"),                    1, 0 },
    {C("unionTCPFlags"),                      1, 0 },
    {C("reverseInitialTCPFlags"),             1, YTF_REV },
    {C("reverseUnionTCPFlags"),               1, YTF_REV },
    {C("reverseTcpSequenceNumber"),           4, YTF_REV },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_mptcp_spec[] = {
    {C("mptcpInitialDataSequenceNumber"),     8, 0 },
    {C("mptcpReceiverToken"),                 4, 0 },
    {C("mptcpMaximumSegmentSize"),            2, 0 },
    {C("mptcpAddressID"),                     1, 0 },
    {C("mptcpFlags"),                         1, 0 },
    FB_IESPEC_NULL
};

/* MAC-specific information */
static fbInfoElementSpec_t  yaf_mac_spec[] = {
    {C("sourceMacAddress"),                   6, 0 },
    {C("destinationMacAddress"),              6, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_p0f_spec[] = {
    {C("osName"),                             FB_IE_VARLEN, 0 },
    {C("osVersion"),                          FB_IE_VARLEN, 0 },
    {C("osFingerPrint"),                      FB_IE_VARLEN, 0 },
    {C("reverseOsName"),                      FB_IE_VARLEN, YTF_REV },
    {C("reverseOsVersion"),                   FB_IE_VARLEN, YTF_REV },
    {C("reverseOsFingerPrint"),               FB_IE_VARLEN, YTF_REV },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_fpexport_spec[] = {
    {C("firstPacketBanner"),                  FB_IE_VARLEN, 0 },
    {C("secondPacketBanner"),                 FB_IE_VARLEN, 0 },
    {C("reverseFirstPacketBanner"),           FB_IE_VARLEN, YTF_REV },
    FB_IESPEC_NULL
};

/* Variable-length payload fields */
static fbInfoElementSpec_t  yaf_payload_spec[] = {
    {C("payload"),                            FB_IE_VARLEN, 0 },
    {C("reversePayload"),                     FB_IE_VARLEN, YTF_REV },
    FB_IESPEC_NULL
};

/* Template containing a single basicList; used by multiple DPIs (e.g, IRC,
 * POP3, TLS Binary Certificate) */
static fbInfoElementSpec_t  yaf_singleBL_spec[] = {
    {C("basicList"),       FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_pop3_spec[] = {
    {C("basicList"),              FB_IE_VARLEN, 0 },
    {C("subTemplateList"),        FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_tftp_spec[] = {
    {C("tftpFilename"),          FB_IE_VARLEN, 0 },
    {C("tftpMode"),              FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_flowstats_spec[] = {
    {C("dataByteCount"),                      8, 0 },
    {C("averageInterarrivalTime"),            8, 0 },
    {C("standardDeviationInterarrivalTime"),  8, 0 },
    {C("tcpUrgTotalCount"),                   4, 0 },
    {C("smallPacketCount"),                   4, 0 },
    {C("nonEmptyPacketCount"),                4, 0 },
    {C("largePacketCount"),                   4, 0 },
    {C("firstNonEmptyPacketSize"),            2, 0 },
    {C("maxPacketSize"),                      2, 0 },
    {C("standardDeviationPayloadLength"),     2, 0 },
    {C("firstEightNonEmptyPacketDirections"), 1, 0 },
    {C("paddingOctets"),                      1, 1 },
    {C("reverseDataByteCount"),               8, YTF_REV },
    {C("reverseAverageInterarrivalTime"),     8, YTF_REV },
    {C("reverseStandardDeviationInterarrivalTime"), 8, YTF_REV },
    {C("reverseTcpUrgTotalCount"),            4, YTF_REV },
    {C("reverseSmallPacketCount"),            4, YTF_REV },
    {C("reverseNonEmptyPacketCount"),         4, YTF_REV },
    {C("reverseLargePacketCount"),            4, YTF_REV },
    {C("reverseFirstNonEmptyPacketSize"),     2, YTF_REV },
    {C("reverseMaxPacketSize"),               2, YTF_REV },
    {C("reverseStandardDeviationPayloadLength"), 2, YTF_REV },
    {C("paddingOctets"),                      2, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_slp_spec[] = {
    {C("basicList"),             FB_IE_VARLEN, 0 },
    {C("slpVersion"),            1, 0 },
    {C("slpMessageType"),        1, 0 },
    {C("paddingOctets"),         6, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_http_spec[] = {
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};
static fbInfoElementSpec_t  yaf_ftp_spec[] = {
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_imap_216_spec[] = {
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_imap_spec[] = {
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("basicList"),        FB_IE_VARLEN, 0 },
    {C("subTemplateList"),   FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_rtsp_spec[] = {
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_sip_spec[] = {
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* SMTP record up to yaf 2.11.0 inclusive */
static fbInfoElementSpec_t  yaf_smtp_211_spec[] = {
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    {C("basicList"),         FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* SMTP record after yaf 2.11.0; pairs with yaf_smtp_t, YAF_SMTP_TID */
static fbInfoElementSpec_t  yaf_smtp_spec[] = {
    {C("smtpHello"),         FB_IE_VARLEN, 0 },
    {C("smtpEnhanced"),      FB_IE_VARLEN, 0 },
    {C("smtpMessageSize"),   4, 0 },
    {C("smtpStartTLS"),      1, 0 },
    {C("paddingOctets"),     3, 1 },
    {C("smtpResponseList"),  FB_IE_VARLEN, 0 },
    /* STL of yaf_smtp_message_spec */
    {C("smtpMessageList"),   FB_IE_VARLEN, 0 },
    {C("subTemplateList"),   FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* sub-record of yaf_smtp_spec; pairs with yaf_smtp_message_t,
 * YAF_SMTP_MESSAGE_TID */
static fbInfoElementSpec_t  yaf_smtp_message_spec[] = {
    /* a string */
    {C("smtpSubject"),       FB_IE_VARLEN, 0 },
    /* BL of smptTo */
    {C("smtpToList"),        FB_IE_VARLEN, 0 },
    /* BL of smptFrom */
    {C("smtpFromList"),      FB_IE_VARLEN, 0 },
    /* BL of smptFilename */
    {C("smtpFilenameList"),  FB_IE_VARLEN, 0 },
    /* BL of smptURL */
    {C("smtpURLList"),       FB_IE_VARLEN, 0 },
    /* STL of yaf_smtp_header_spec */
    {C("smtpHeaderList"),    FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* sub-record of yaf_smtp_message_spec, sub-sub-rec of yaf_smtp_spec; pairs
 * with yaf_smtp_header_t, YAF_SMTP_HEADER_TID */
static fbInfoElementSpec_t  yaf_smtp_header_spec[] = {
    /* header field name, a string */
    {C("smtpKey"),           FB_IE_VARLEN, 0 },
    /* header field body, a string */
    {C("smtpValue"),         FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_nntp_spec[] = {
    {C("basicList"),       FB_IE_VARLEN, 0 },
    {C("basicList"),       FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dns_spec[] = {
    {C("subTemplateList"),    FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};
static fbInfoElementSpec_t  yaf_dnsQR_spec[] = {
    {C("subTemplateList"),     FB_IE_VARLEN, 0 }, /*based on type of RR */
    {C("dnsQName"),            FB_IE_VARLEN, 0 }, /*name - varfield*/
    {C("dnsTTL"),              4, 0 },
    {C("dnsQRType"),           2, 0 },  /*Type - uint8*/
    {C("dnsQueryResponse"),    1, 0 },  /*Q or R - uint8*/
    {C("dnsAuthoritative"),    1, 0 }, /* authoritative response (1)*/
    {C("dnsNXDomain"),         1, 0 }, /* nxdomain (1) */
    {C("dnsRRSection"),        1, 0 }, /*0, 1, 2 (ans, auth, add'l) */
    {C("dnsID"),               2, 0 },
    {C("paddingOctets"),       4, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsA_spec[] = {
    {C("sourceIPv4Address"),         4, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsAAAA_spec[] = {
    {C("sourceIPv6Address"),         16, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsCNAME_spec[] = {
    {C("dnsCName"),                  FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsMX_spec[] = {
    {C("dnsMXExchange"),             FB_IE_VARLEN, 0 },
    {C("dnsMXPreference"),           2, 0 },
    {C("paddingOctets"),             6, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsNS_spec[] = {
    {C("dnsNSDName"),                FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsPTR_spec[] = {
    {C("dnsPTRDName"),               FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsTXT_spec[] = {
    {C("dnsTXTData"),                FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsSOA_spec[] = {
    {C("dnsSOAMName"),               FB_IE_VARLEN, 0 },
    {C("dnsSOARName"),               FB_IE_VARLEN, 0 },
    {C("dnsSOASerial"),              4, 0 },
    {C("dnsSOARefresh"),             4, 0 },
    {C("dnsSOARetry"),               4, 0 },
    {C("dnsSOAExpire"),              4, 0 },
    {C("dnsSOAMinimum"),             4, 0 },
    {C("paddingOctets"),             4, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsSRV_spec[] = {
    {C("dnsSRVTarget"),              FB_IE_VARLEN, 0 },
    {C("dnsSRVPriority"),            2, 0 },
    {C("dnsSRVWeight"),              2, 0 },
    {C("dnsSRVPort"),                2, 0 },
    {C("paddingOctets"),             2, 1 },
    FB_IESPEC_NULL
};


static fbInfoElementSpec_t  yaf_dnsDS_spec[] = {
    {C("dnsDigest"),                 FB_IE_VARLEN, 0 },
    {C("dnsKeyTag"),                 2, 0 },
    {C("dnsAlgorithm"),              1, 0 },
    {C("dnsDigestType"),             1, 0 },
    {C("paddingOctets"),             4, 1 },
    FB_IESPEC_NULL
};


static fbInfoElementSpec_t  yaf_dnsRRSig_spec[] = {
    {C("dnsSigner"),                 FB_IE_VARLEN, 0 },
    {C("dnsSignature"),              FB_IE_VARLEN, 0 },
    {C("dnsSignatureInception"),     4, 0 },
    {C("dnsSignatureExpiration"),    4, 0 },
    {C("dnsTTL"),                    4, 0 },
    {C("dnsKeyTag"),                 2, 0 },
    {C("dnsTypeCovered"),            2, 0 },
    {C("dnsAlgorithm"),              1, 0 },
    {C("dnsLabels"),                 1, 0 },
    {C("paddingOctets"),             6, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsNSEC_spec[] = {
    {C("dnsHashData"),               FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsKey_spec[] = {
    {C("dnsPublicKey"),              FB_IE_VARLEN, 0 },
    {C("dnsFlags"),                  2, 0 },
    {C("protocolIdentifier"),        1, 0 },
    {C("dnsAlgorithm"),              1, 0 },
    {C("paddingOctets"),             4, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnsNSEC3_spec[] = {
    {C("dnsSalt"),                   FB_IE_VARLEN, 0 },
    {C("dnsHashData"),               FB_IE_VARLEN, 0 },
    {C("dnsIterations"),             2, 0 },
    {C("dnsAlgorithm"),              1, 0 },
    {C("paddingOctets"),             5, 1 },
    FB_IESPEC_NULL
};


/* Level 1 TLS/SSL Record as of YAF 2.3.0; contains connection info; pairs
 * with yaf_ssl_t, YAF_SSL_TID */
static fbInfoElementSpec_t  yaf_ssl_spec[] = {
    /* list of ciphers 32bit */
    {C("basicList"),                 FB_IE_VARLEN, 0 },
    /* cipher suite in server hello */
    {C("sslServerCipher"),           4, 0 },
    {C("sslClientVersion"),          1, 0 },
    /* compression method in serv hello */
    {C("sslCompressionMethod"),      1, 0 },
    {C("sslRecordVersion"),          2, 0 },
    /* list of certs */
    {C("subTemplateList"),           FB_IE_VARLEN, 0 },
    {C("sslServerName"),             FB_IE_VARLEN, 0 },
    {C("sslClientJA3"),              16, 0 },
    {C("sslServerJA3S"),             16, 0 },
    {C("sslClientJA3Fingerprint"),   FB_IE_VARLEN, 0 },
    {C("sslServerJA3SFingerprint"),  FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* Level 2 TLS/SSL Record as of YAF 2.3.0; contains certificate info; normally
 * sub-record of yaf_ssl_spec but is top-level in ssl_dedup output; pairs
 * with yaf_ssl_cert_t, YAF_SSL_CERT_TID */
static fbInfoElementSpec_t  yaf_ssl_cert_spec[] = {
    /* STL of yaf_ssl_subcert_spec for issuer */
    {C("subTemplateList"),             FB_IE_VARLEN, 0 },
    /* STL of yaf_ssl_subcert_spec for subject */
    {C("subTemplateList"),             FB_IE_VARLEN, 0 },
    /* STL of yaf_ssl_subcert_spec for extensions */
    {C("subTemplateList"),             FB_IE_VARLEN, 0 },
    {C("sslCertSignature"),            FB_IE_VARLEN, 0 },
    {C("sslCertSerialNumber"),         FB_IE_VARLEN, 0 },
    {C("sslCertValidityNotBefore"),    FB_IE_VARLEN, 0 },
    {C("sslCertValidityNotAfter"),     FB_IE_VARLEN, 0 },
    {C("sslPublicKeyAlgorithm"),       FB_IE_VARLEN, 0 },
    {C("sslPublicKeyLength"),          2, 0 },
    {C("sslCertVersion"),              1, 0 },
    {C("paddingOctets"),               5, 1 },
    {C("sslCertificateHash"),          FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* Level 3 TLS/SSL Record as of YAF 2.3.0; sub-record of yaf_ssl_cert_spec,
 * sub-sub-rec of yaf_ssl_spec; pairs with yaf_ssl_subcert_t,
 * YAF_SSL_SUBCERT_TID */
static fbInfoElementSpec_t  yaf_ssl_subcert_spec[] = {
    {C("sslObjectValue"),              FB_IE_VARLEN, 0 },
    {C("sslObjectType"),               1, 0 },
    {C("paddingOctets"),               7, 1 },
    FB_IESPEC_NULL
};


/* YAF-2.2.2 and earlier: pairs with yaf_ssl_y22_t, YAF_SSL_Y22_TID */
static fbInfoElementSpec_t  yaf_ssl_y22_spec[] = {
    /* list of ciphers 32bit */
    {C("basicList"),                 FB_IE_VARLEN, 0 },
    /* cipher suite in server hello */
    {C("sslServerCipher"),           4, 0 },
    {C("sslClientVersion"),          1, 0 },
    /* compression method in serv hello */
    {C("sslCompressionMethod"),      1, 0 },
    {C("paddingOctets"),             2, 1 },
    FB_IESPEC_NULL
};

/* YAF-2.2.2 and earlier: pairs with yaf_ssl_y22_cert_t,
 * YAF_SSL_Y22_CERT_TID */
static fbInfoElementSpec_t  yaf_ssl_y22_cert_spec[] = {
    {C("sslCertSignature"),            FB_IE_VARLEN, 0 },
    {C("sslCertIssuerCountryName"),    FB_IE_VARLEN, 0 },
    {C("sslCertIssuerOrgName"),        FB_IE_VARLEN, 0 },
    {C("sslCertIssuerOrgUnitName"),    FB_IE_VARLEN, 0 },
    {C("sslCertIssuerZipCode"),        FB_IE_VARLEN, 0 },
    {C("sslCertIssuerState"),          FB_IE_VARLEN, 0 },
    {C("sslCertIssuerCommonName"),     FB_IE_VARLEN, 0 },
    {C("sslCertIssuerLocalityName"),   FB_IE_VARLEN, 0 },
    {C("sslCertIssuerStreetAddress"),  FB_IE_VARLEN, 0 },
    {C("sslCertSubCountryName"),       FB_IE_VARLEN, 0 },
    {C("sslCertSubOrgName"),           FB_IE_VARLEN, 0 },
    {C("sslCertSubOrgUnitName"),       FB_IE_VARLEN, 0 },
    {C("sslCertSubZipCode"),           FB_IE_VARLEN, 0 },
    {C("sslCertSubState"),             FB_IE_VARLEN, 0 },
    {C("sslCertSubCommonName"),        FB_IE_VARLEN, 0 },
    {C("sslCertSubLocalityName"),      FB_IE_VARLEN, 0 },
    {C("sslCertSubStreetAddress"),     FB_IE_VARLEN, 0 },
    {C("sslCertVersion"),              1, 0 },
    FB_IESPEC_NULL
};

/* The flattened SSL record representing a single SSL Certificate; pairs with
 * md_ssl_certificate_t, MD_SSL_CERTIFICATE_TID */
static fbInfoElementSpec_t  md_ssl_certificate_spec[] = {
    /** Issuer **/
    /* id-at-commonName {id-at 3} ("sslCertIssuerCommonName") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-countryName {id-at 6} */
    {C("sslCertIssuerCountryName"),        FB_IE_VARLEN, 4},
    /* id-at-localityName {id-at 7} */
    {C("sslCertIssuerLocalityName"),       FB_IE_VARLEN, 4},
    /* id-at-stateOrProvidenceName {id-at 8} */
    {C("sslCertIssuerState"),              FB_IE_VARLEN, 4},
    /* id-at-streetAddress {id-at 9} ("sslCertIssuerStreetAddress") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-organizationName {id-at 10} ("sslCertIssuerOrgName") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-organizationUnitName {id-at 11} ("sslCertIssuerOrgUnitName") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-postalCode {id-at 17} */
    {C("sslCertIssuerZipCode"),            FB_IE_VARLEN, 4},
    /* id-at-title {id-at 12} */
    {C("sslCertIssuerTitle"),              FB_IE_VARLEN, 4},
    /* id-at-name {id-at 41} */
    {C("sslCertIssuerName"),               FB_IE_VARLEN, 4},
    /* pkcs-9-emailAddress {pkcs-9 1} */
    {C("sslCertIssuerEmailAddress"),       FB_IE_VARLEN, 4},
    /* 0.9.2342.19200300.100.1.25 {dc 25} ("sslCertIssuerDomainComponent") */
    {C("basicList"),                       FB_IE_VARLEN, 4},

    /** Subject **/
    /* id-at-commonName {id-at 3} ("sslCertSubCommonName") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-countryName {id-at 6} */
    {C("sslCertSubCountryName"),           FB_IE_VARLEN, 4},
    /* id-at-localityName {id-at 7} */
    {C("sslCertSubLocalityName"),          FB_IE_VARLEN, 4},
    /* id-at-stateOrProvidenceName {id-at 8} */
    {C("sslCertSubState"),                 FB_IE_VARLEN, 4},
    /* id-at-streetAddress {id-at 9} ("sslCertSubStreetAddress") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-organizationName {id-at 10} ("sslCertSubOrgName") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-organizationUnitName {id-at 11} ("sslCertSubOrgUnitName") */
    {C("basicList"),                       FB_IE_VARLEN, 4},
    /* id-at-postalCode {id-at 17} */
    {C("sslCertSubZipCode"),               FB_IE_VARLEN, 4},
    /* id-at-title {id-at 12} */
    {C("sslCertSubTitle"),                 FB_IE_VARLEN, 4},
    /* id-at-name {id-at 41} */
    {C("sslCertSubName"),                  FB_IE_VARLEN, 4},
    /* pkcs-9-emailAddress {pkcs-9 1} */
    {C("sslCertSubEmailAddress"),          FB_IE_VARLEN, 4},
    /* 0.9.2342.19200300.100.1.25 {dc 25} ("sslCertSubDomainComponent") */
    {C("basicList"),                       FB_IE_VARLEN, 4},

    /** Extensions **/
    /* id-ce-subjectKeyIdentifier {id-ce 14} */
    {C("sslCertExtSubjectKeyIdent"),       FB_IE_VARLEN, 4},
    /* id-ce-keyUsage {id-ce 15} */
    {C("sslCertExtKeyUsage"),              FB_IE_VARLEN, 4},
    /* id-ce-privateKeyUsagePeriod {id-ce 16} */
    {C("sslCertExtPrivKeyUsagePeriod"),    FB_IE_VARLEN, 4},
    /* id-ce-subjectAltName {id-ce 17} */
    {C("sslCertExtSubjectAltName"),        FB_IE_VARLEN, 4},
    /* id-ce-issuerAltName {id-ce 18} */
    {C("sslCertExtIssuerAltName"),         FB_IE_VARLEN, 4},
    /* id-ce-certificateIssuer {id-ce 29} */
    {C("sslCertExtCertIssuer"),            FB_IE_VARLEN, 4},
    /* id-ce-cRLDistributionPoints {id-ce 31} */
    {C("sslCertExtCrlDistribution"),       FB_IE_VARLEN, 4},
    /* id-ce-certificatePolicies {id-ce 32} */
    {C("sslCertExtCertPolicies"),          FB_IE_VARLEN, 4},
    /* id-ce-authorityKeyIdentifier {id-ce 35} */
    {C("sslCertExtAuthorityKeyIdent"),     FB_IE_VARLEN, 4},
    /* id-ce-extKeyUsage {id-ce 37} */
    {C("sslCertExtExtendedKeyUsage"),      FB_IE_VARLEN, 4},

    /* Values from the YAF SSL record */
    {C("sslCertSignature"),                FB_IE_VARLEN, 4},
    {C("sslCertSerialNumber"),             FB_IE_VARLEN, 4},
    {C("sslCertValidityNotBefore"),        FB_IE_VARLEN, 4},
    {C("sslCertValidityNotAfter"),         FB_IE_VARLEN, 4},
    {C("sslPublicKeyAlgorithm"),           FB_IE_VARLEN, 4},
    {C("sslPublicKeyLength"),              2,            4},
    {C("sslCertVersion"),                  1,            4},
    {C("paddingOctets"),                   5,            2},
    {C("sslCertificateHash"),              FB_IE_VARLEN, 4},

    FB_IESPEC_NULL
};


static fbInfoElementSpec_t  yaf_mysql_spec[] = {
    {C("subTemplateList"),            FB_IE_VARLEN, 0 },
    {C("mysqlUsername"),              FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_mysql_txt_spec[] = {
    {C("mysqlCommandText"),           FB_IE_VARLEN, 0 },
    {C("mysqlCommandCode"),           1, 0 },
    {C("paddingOctets"),              7, 1 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dhcp_fp_spec[] = {
    {C("dhcpFingerPrint"),             FB_IE_VARLEN, 0 },
    {C("dhcpVendorCode"),              FB_IE_VARLEN, 0 },
    {C("reverseDhcpFingerPrint"),      FB_IE_VARLEN, YTF_REV },
    {C("reverseDhcpVendorCode"),       FB_IE_VARLEN, YTF_REV },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dhcp_options_spec[] = {
    {C("basicList"),                   FB_IE_VARLEN, 0 },
    {C("dhcpVendorCode"),              FB_IE_VARLEN, 0 },
    {C("basicList"),                   FB_IE_VARLEN, YTF_REV},
    {C("reverseDhcpVendorCode"),       FB_IE_VARLEN, YTF_REV },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_rtp_spec[] = {
    {C("rtpPayloadType"),           1, 0 },
    {C("reverseRtpPayloadType"),    1, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnp_rec_spec[] = {
    {C("dnp3SourceAddress"),        2, 0 },
    {C("dnp3DestinationAddress"),   2, 0 },
    {C("dnp3Function"),             1, 0 },
    {C("paddingOctets"),            3, 1 },
    {C("dnp3ObjectData"),           FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  yaf_dnp_spec[] = {
    {C("subTemplateList"),     FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* Added in SM 1.10.0; previously SSH used the yaf_singleBL_spec */
static fbInfoElementSpec_t  yaf_ssh_spec [] = {
    {C("sshVersion"),                   FB_IE_VARLEN, 0 },
    {C("sshServerVersion"),             FB_IE_VARLEN, 0 },
    {C("sshKeyExchangeAlgorithm"),      FB_IE_VARLEN, 0 },
    {C("sshHostKeyAlgorithm"),          FB_IE_VARLEN, 0 },
    {C("sshServerHostKey"),             16,           0 },
    {C("sshCipher"),                    FB_IE_VARLEN, 0 },
    {C("sshMacAlgorithm"),              FB_IE_VARLEN, 0 },
    {C("sshCompressionMethod"),         FB_IE_VARLEN, 0 },
    {C("sshHassh"),                     16,           0 },
    {C("sshServerHassh"),               16,           0 },
    {C("sshHasshAlgorithms"),           FB_IE_VARLEN, 0 },
    {C("sshServerHasshAlgorithms"),     FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* DNS Dedup; pairs with md_dns_dedup_t, MD_DNS_DEDUP_OUT, MD_DNS_DEDUP_FULL */
static fbInfoElementSpec_t  md_dns_dedup_spec[] = {
    /* Millisecond first seen and last seen (epoch) (native time) */
    {C("flowStartMilliseconds"),              8,  0 },
    {C("flowEndMilliseconds"),                8,  MD_LAST_SEEN },
    {C("flowStartNanoseconds"),               8,  0 },
    {C("flowEndNanoseconds"),                 8,  MD_LAST_SEEN },
    /* AAAA-record IP */
    {C("sourceIPv6Address"),                  16, MD_DNS_AAAAREC },
    /* A-record IP */
    {C("sourceIPv4Address"),                  4,  MD_DNS_AREC },
    /** Max TTL */
    {C("dnsTTL"),                             4,  MD_LAST_SEEN },
    /* rrType */
    {C("dnsQRType"),                          2,  0 },
    /* how many times we saw it */
    {C("dnsHitCount"),                        2,  MD_LAST_SEEN },
    {C("paddingOctets"),                      4,  1 },
    /* rrData */
    {C("dnsQName"),                           FB_IE_VARLEN, 0 },
    {C("dnsRName"),                           FB_IE_VARLEN, MD_DNS_OREC },
    {C("observationDomainName"),              FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  md_dnsrr_spec[] = {
    {C("flowStartMilliseconds"),              8,  0 },
    {C("flowStartNanoseconds"),               8,  0 },
    {C("sourceIPv6Address"),                  16, YTF_IP6 | MD_DNSRR_FULL },
    {C("destinationIPv6Address"),             16, YTF_IP6 | MD_DNSRR_FULL },
    {C("sourceIPv4Address"),                  4,  YTF_IP4 | MD_DNSRR_FULL },
    {C("destinationIPv4Address"),             4,  YTF_IP4 | MD_DNSRR_FULL },
    {C("dnsTTL"),                             4,  0 },
    {C("observationDomainId"),                4,  0 },
    {C("yafFlowKeyHash"),                     4,  0 },
    {C("dnsQRType"),                          2,  0 },
    {C("sourceTransportPort"),                2,  MD_DNSRR_FULL },
    {C("destinationTransportPort"),           2,  MD_DNSRR_FULL },
    {C("vlanId"),                             2,  MD_DNSRR_FULL },
    {C("dnsID"),                              2,  0 },
    {C("protocolIdentifier"),                 1,  MD_DNSRR_FULL },
    {C("dnsQueryResponse"),                   1,  0 }, /*Q or R - uint8*/
    {C("dnsAuthoritative"),                   1,  0 }, /*auth response (1)*/
    {C("dnsNXDomain"),                        1,  0 }, /*nxdomain (1)*/
    {C("dnsRRSection"),                       1,  0 }, /*(qry,ans,auth,add'l)*/
    {C("paddingOctets"),                      5,  1 },
    {C("dnsQName"),                           FB_IE_VARLEN, 0 },
    {C("dnsRName"),                           FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  md_dedup_spec[] = {
    /* Millisecond first seen and last seen (epoch) (native time) */
    {C("monitoringIntervalStartMilliSeconds"),  8,  0 },
    {C("monitoringIntervalEndMilliSeconds"),    8,  0 },
    {C("flowStartMilliseconds"),                8,  0 },
    {C("flowStartNanoseconds"),                 8,  0 },
    {C("observedDataTotalCount"),               8,  0 },
    {C("sourceIPv6Address"),                    16, 0 },
    {C("sourceIPv4Address"),                    4,  0 },
    {C("yafFlowKeyHash"),                       4,  0 },
    {C("observationDomainName"),                FB_IE_VARLEN, 0 },
    {C("observedData"),                         FB_IE_VARLEN, 0 },
    {C("sslCertSerialNumber"),                  FB_IE_VARLEN, MD_DEDUP_SSL },
    {C("sslCertIssuerCommonName"),              FB_IE_VARLEN, MD_DEDUP_SSL },
    {C("sslCertSerialNumber"),                  FB_IE_VARLEN, MD_DEDUP_SSL },
    {C("sslCertIssuerCommonName"),              FB_IE_VARLEN, MD_DEDUP_SSL },
    FB_IESPEC_NULL
};

/* SSL dedup record; pairs with md_ssl_dedup_t, MD_SSL_DEDUP_TID */
static fbInfoElementSpec_t  md_ssl_dedup_spec[] = {
    {C("flowStartMilliseconds"),              8, 0 },
    {C("flowEndMilliseconds"),                8, 0 },
    {C("flowStartNanoseconds"),               8, 0 },
    {C("flowEndNanoseconds"),                 8, 0 },
    {C("observedDataTotalCount"),             8, 0 },
    {C("sslCertSerialNumber"),                FB_IE_VARLEN, 0 },
    {C("sslCertIssuerCommonName"),            FB_IE_VARLEN, 0 },
    {C("observationDomainName"),              FB_IE_VARLEN, 0 },
    FB_IESPEC_NULL
};

/* SiLK IPSet matching sub-record; pairs with md_ipset_rec_t,
 * md_ipset_rec_tmpl, MD_IPSET_REC_TID */
static fbInfoElementSpec_t  md_ipset_rec_spec[] = {
    {C("smIPSetName"),                        FB_IE_VARLEN, 0 },
    {C("smIPSetMatchesSource"),               1, 0 },
    {C("smIPSetMatchesDestination"),          1, 0 },
    FB_IESPEC_NULL
};

/* SiLK Prefix Map Label sub-record; pairs with md_pmap_rec_t,
 * md_pcap_rec_tmpl, MD_PMAP_REC_TID */
static fbInfoElementSpec_t  md_pmap_rec_spec[] = {
    {C("smPrefixMapName"),                    FB_IE_VARLEN, 0 },
    {C("smPrefixMapLabelSource"),             FB_IE_VARLEN, 0 },
    {C("smPrefixMapLabelDestination"),        FB_IE_VARLEN, 0 },
    {C("smPrefixMapTypeId"),                  1, 0 },
    FB_IESPEC_NULL
};

static fbInfoElementSpec_t  md_silk_type_sensor_spec[] = {
    {C("silkSensorName"),                     FB_IE_VARLEN, 0 },
    {C("silkSensorDescription"),              FB_IE_VARLEN, 0 },
    {C("silkTypeName"),                       FB_IE_VARLEN, 0 },
    {C("silkSensorId"),                       2, 0 },
    {C("silkFlowtypeId"),                     1, 0 },
    FB_IESPEC_NULL
};



static void
mdMainDecode(
    mdContext_t   *ctx,
    mdFullFlow_t  *flow);

static void
mdCleanUP(
    mdFullFlow_t  *flow);

static void
mdCleanUpSSLCert(
    yaf_ssl_cert_t  *cert);

static gboolean
mdForwardTombstone(
    mdContext_t        *ctx,
    md_collect_node_t  *collector,
    uint16_t            tid,
    GError            **err);

static gboolean
mdForwardStats(
    mdContext_t        *ctx,
    md_collect_node_t  *collector,
    GError            **err);


/**
 * mdInfoModel
 *
 * create an appropriate info model
 * ================================
 * alloc the default (with IANA elements pre-populated) via fbInfoModelAlloc
 * add in the CERT/NetSA added elements using infomodelAddGlobalElements
 *
 */
fbInfoModel_t *
mdInfoModel(
    void)
{
    static fbInfoModel_t *md_info_model = NULL;

    if (!md_info_model) {
        md_info_model = fbInfoModelAlloc();

        infomodelAddGlobalElements(md_info_model);

        if (user_elements) {
            fbInfoModelAddElementArray(md_info_model, user_elements);
        }
    }

    return md_info_model;
}

static void
templateFree(
    void  *tmpl_ctx,
    void  *app_ctx)
{
    MD_UNUSED_PARAM(app_ctx);
    g_free(tmpl_ctx);
}


static void
mdTemplateCallback(
    fbSession_t           *session,
    uint16_t               tid,
    fbTemplate_t          *tmpl,
    void                  *app_ctx,
    void                 **tmpl_ctx,
    fbTemplateCtxFree_fn  *fn)
{
    /* Standard (IANA) individual elements to search for */
    static fbInfoElement_t  ingressInterface =
        FB_IE_INIT(C("ingressInterface"),         0,        10,   4, 0);
    static fbInfoElement_t  mplsTopLabelStackSection =
        FB_IE_INIT(C("mplsTopLabelStackSection"), 0,        70,   3, 0);
    static fbInfoElement_t  flowStartMilliseconds =
        FB_IE_INIT(C("flowStartMicroseconds"),    0,        152,  8, 0);
    static fbInfoElement_t  flowStartMicroseconds =
        FB_IE_INIT(C("flowStartMicroseconds"),    0,        154,  8, 0);
    static fbInfoElement_t  flowStartNanoseconds =
        FB_IE_INIT(C("flowStartNanoseconds"),     0,        156,  8, 0);

    /* CERT individual elements to search for */
    static fbInfoElement_t  nDPIL7Protocol =
        FB_IE_INIT(C("nDPIL7Protocol"),           CERT_PEN, 300,  2, 0);
    static fbInfoElement_t  sslCertExtKeyUsage =
        FB_IE_INIT(C("sslCertExtKeyUsage"),       CERT_PEN, 317,  0, 0);
    static fbInfoElement_t  templateName =
        FB_IE_INIT(C("templateName"),             CERT_PEN, 1000, 0, 0);

    /* Reverse standard individual elements to search for */
    static fbInfoElement_t  reversePacketDeltaCount =
        FB_IE_INIT(C("reversePacketDeltaCount"),  FB_IE_PEN_REVERSE, 2,  8, 0);
    static fbInfoElement_t  reversePacketTotalCount =
        FB_IE_INIT(C("reversePacketTotalCount"),  FB_IE_PEN_REVERSE, 86, 8, 0);

    /* Sets of elements to search for */
    static fbInfoElementSpec_t  dedupRecordCheck[] = {
        {C("observedDataTotalCount"),   8,  0},
        {C("yafFlowKeyHash"),           4,  0},
        FB_IESPEC_NULL
    };
    static fbInfoElementSpec_t  deltaCounters[] = {
        {C("packetDeltaCount"),         8,  0},
        {C("octetDeltaCount"),          8,  0},
        FB_IESPEC_NULL
    };
    static fbInfoElementSpec_t  ipv6Addresses[] = {
        {C("sourceIPv6Address"),        16, 0},
        {C("destinationIPv6Address"),   16, 0},
        FB_IESPEC_NULL
    };

    fbInfoElement_t            *ie;
    GError                     *err = NULL;
    uint16_t                    ntid;
    mdTmplContext_t            *myctx = g_new0(mdTmplContext_t, 1);

    MD_UNUSED_PARAM(app_ctx);

    myctx->tid = tid;
    myctx->num_elem = fbTemplateCountElements(tmpl);

    /* FIXME: Consider just looping over the template's elements instead of
     * repeatedly querying the template */

    if (fbTemplateContainsElement(tmpl, &templateName)) {
        myctx->is_metadata_template = TRUE;
    } else if (fbInfoModelTypeInfoRecord(tmpl)) {
        myctx->is_metadata_element = TRUE;
    }

    myctx->attr.has_interface =
        fbTemplateContainsElement(tmpl, &ingressInterface);
    myctx->attr.has_mpls =
        fbTemplateContainsElement(tmpl, &mplsTopLabelStackSection);
    myctx->attr.has_ndpi =
        fbTemplateContainsElement(tmpl, &nDPIL7Protocol);

    myctx->attr.has_milli =
        fbTemplateContainsElement(tmpl, &flowStartMilliseconds);
    myctx->attr.has_micro =
        fbTemplateContainsElement(tmpl, &flowStartMicroseconds);
    myctx->attr.has_nano =
        fbTemplateContainsElement(tmpl, &flowStartNanoseconds);

    myctx->attr.is_ipv6 =
        fbTemplateContainsAllElementsByName(tmpl, ipv6Addresses);
    myctx->attr.is_delta =
        fbTemplateContainsAllElementsByName(tmpl, deltaCounters);

    myctx->attr.is_reverse =
        (fbTemplateContainsElement(tmpl, &reversePacketTotalCount)
         || fbTemplateContainsElement(tmpl, &reversePacketDeltaCount));

    if (tid == YAF_SSL_TID) {
        /* mthomas.2022.03.10. Previously we used a different internal TID
         * here, but that is not needed as long as the collector defines
         * YAF_SSL_TID as an internal template first. */
        fbSessionAddTemplatePair(session, tid, tid);
    } else if (tid == YAF_SSL_CERT_TID) {
        /* This tid is used for both the SSL structure exported by YAF and the
         * rewritten SSL certificate (md_ssl_certificate_spec).  Check whether
         * the template contains an element from the rewritten certificate; if
         * it does, use that spec if available or abort otherwise.  */
        if (!fbTemplateContainsElement(tmpl, &sslCertExtKeyUsage)) {
            /* it does not */
            /* mthomas.2022.03.10. Ditto the above note. */
            fbSessionAddTemplatePair(session, tid, tid);
        } else if (fbSessionGetTemplate(session, TRUE, MD_SSL_CERTIFICATE_TID,
                                        NULL))
        {
            /* Map this template to itself since session has it */
            fbSessionAddTemplatePair(session, tid, MD_SSL_CERTIFICATE_TID);
        } else {
            /* Abort */
            g_warning("Attempting to reprocess rewritten SSL certificates."
                      " Run super_mediator with --rewrite-ssl-certs"
                      " to properly process this data.");
            g_error("Input with rewritten SSL certificates cannot"
                    " be processed. Aborting");
        }
    } else if (tid == MD_DEDUP_FULL) {
        /* standard ssl dedup rec */
        /* md_dedup_rec */
        fbSessionAddTemplatePair(session, tid, tid);
    } else if (tid == MD_SSL_DEDUP_TID) {
        /* standard cert dedup rec */
        /* md_ssl_dedup_spec */
        fbSessionAddTemplatePair(session, tid, tid);
    } else if (tid == YAF_PROCESS_STATS_210_TID) {
        /* add support for old YAF stats records */
        fbSessionAddTemplatePair(session, tid, YAF_PROCESS_STATS_TID);
    } else if (fbTemplateContainsAllElementsByName(tmpl, dedupRecordCheck)) {
        /* this is a dedup record */
        myctx->is_dedup = TRUE;
        ntid = fbSessionAddTemplate(session, TRUE, tid, tmpl, &err);
        if (ntid == 0) {
            g_warning("Unable to add incoming template %#06x to session: %s",
                      tid, err->message);
            g_clear_error(&err);
        }
        myctx->tid = ntid;
        /* Get the last element in the template */
        ie = fbTemplateGetIndexedIE(tmpl, (myctx->num_elem) - 1);
        /* get the id of the IE */
        myctx->dedup_ie = ie->num;
        /* need to set this so internal template matches up with struct */
    } else {
        fbSessionAddTemplatePair(session, tid, tid);
    }

    *tmpl_ctx = myctx;
    *fn = templateFree;

    return;
}


#ifdef HAVE_SPREAD
static fbTemplate_t *
mdAddSpreadTmpl(
    fbSession_t          *session,
    fbInfoElementSpec_t  *spec,
    uint16_t              tid,
    gboolean              rev,
    GError              **err)
{
    fbInfoModel_t *model = mdInfoModel();
    fbTemplate_t  *tmpl = NULL;
    uint16_t       rtid = rev ? tid | YTF_REV : tid;

    tmpl = fbTemplateAlloc(model);

    if (!fbTemplateAppendSpecArray(tmpl, spec, 0xffffffff, err)) {
        return NULL;
    }

    if (!(fbSessionAddTemplate(session, TRUE, rtid, tmpl, err))) {
        return NULL;
    }

    if (!fbSessionAddTemplatesMulticast(session, md_config.out_spread.groups,
                                        FALSE, rtid, tmpl, err))
    {
        return NULL;
    }

    if (rev) {
        tmpl = fbTemplateAlloc(model);

        if (!fbTemplateAppendSpecArray(tmpl, spec, 0, err)) {
            return NULL;
        }

        if (!fbSessionAddTemplatesMulticast(session,
                                            md_config.out_spread.groups,
                                            FALSE, tid, tmpl, err))
        {
            return NULL;
        }
    }

    return tmpl;
}

/**
 * mdInitSpreadExporterSession
 *
 *
 *
 **/
fbSession_t *
mdInitSpreadExporterSession(
    fbSession_t  *session,
    gboolean      dns_dedup,
    GError      **err)
{
    fbTemplate_t *tmpl = NULL;

    /* YAF_SILK_FLOW_TID is only internal */
    tmpl = fbTemplateAlloc(mdInfoModel());
    if (!fbTemplateAppendSpecArray(tmpl, md_main_template_spec, ~0, err) ||
        !fbSessionAddTemplate(session, TRUE, YAF_SILK_FLOW_TID, tmpl, err))
    {
        fbTemplateFreeUnused(tmpl);
        return NULL;
    }

    if (dns_dedup &&
        !mdAddSpreadTmpl(
            session, md_dns_dedup_spec, MD_DNS_DEDUP_FULL, FALSE, err))
    {
        return NULL;
    }

    /* dns rr only template */
    if (!mdAddSpreadTmpl(session, md_dnsrr_spec, MD_DNSRR_TID, FALSE, err)) {
        return NULL;
    }

    /* dedup template */
    if (!mdAddSpreadTmpl(session, md_dedup_spec, MD_DEDUP_FULL, FALSE, err)) {
        return NULL;
    }

    /* ssl dedup template */
    if (!mdAddSpreadTmpl(
            session, md_ssl_dedup_spec, MD_SSL_DEDUP_TID, FALSE, err))
    {
        return NULL;
    }

    /* add export template */
    tmpl = fbTemplateAlloc(mdInfoModel());
    if (!fbTemplateAppendSpecArray(tmpl, md_dnsrr_spec, 0, err)) {
        return NULL;
    }
    if (!fbSessionAddTemplatesMulticast(session, md_config.out_spread.groups,
                                        FALSE, MD_DNSRR_TID, tmpl, err))
    {
        return NULL;
    }

    /* Options Stats Template */
    tmpl = mdAddSpreadTmpl(session, yaf_process_stats_spec,
                           YAF_PROCESS_STATS_210_TID, FALSE, err);
    if (!tmpl) {
        return NULL;
    }
    fbTemplateSetOptionsScope(tmpl, 3);

    /* Options Tombstone Template */
    tmpl = mdAddSpreadTmpl(session, yaf_tombstone_spec,
                           YAF_TOMBSTONE_TID, FALSE, err);
    if (!tmpl) {
        return NULL;
    }
    fbTemplateSetOptionsScope(tmpl, 3);

    tmpl = mdAddSpreadTmpl(session, yaf_tombstone_access_spec,
                           YAF_TOMBSTONE_ACCESS_TID, FALSE, err);
    if (!tmpl) {
        return NULL;
    }

#if 0
    /* No need to export the old tombstone template */
    tmpl = mdAddSpreadTmpl(session, yaf_tombstone_210_spec,
                           YAF_TOMBSTONE_210_TID, FALSE, err);
    if (!tmpl) {
        return NULL;
    }
    fbTemplateSetOptionsScope(tmpl, 2);

    tmpl = mdAddSpreadTmpl(session, yaf_tombstone_access_210_spec,
                           YAF_TOMBSTONE_ACCESS_210_TID, FALSE, err);
    if (!tmpl) {
        return NULL;
    }
#endif  /* 0 */

    /* flow stats template */
    if (!mdAddSpreadTmpl(session, yaf_flowstats_spec, YAF_FLOWSTATS_TID,
                         TRUE, err))
    {
        return NULL;
    }

    /* Entropy Template */
    if (!mdAddSpreadTmpl(session, yaf_entropy_spec, YAF_ENTROPY_TID,
                         TRUE, err))
    {
        return NULL;
    }

    /* TCP Template */
    if (!mdAddSpreadTmpl(session, yaf_tcp_spec, YAF_TCP_TID, TRUE, err)) {
        return NULL;
    }

    /* MPTCP Template */
    if (!mdAddSpreadTmpl(session, yaf_mptcp_spec, YAF_MPTCP_TID, FALSE, err)) {
        return NULL;
    }

    /* MAC Template */
    if (!mdAddSpreadTmpl(session, yaf_mac_spec, YAF_MAC_TID, FALSE, err)) {
        return NULL;
    }

    /* p0f Template */
    if (!mdAddSpreadTmpl(session, yaf_p0f_spec, YAF_P0F_TID, TRUE, err)) {
        return NULL;
    }

    /* dhcp Template */
    if (!mdAddSpreadTmpl(session, yaf_dhcp_fp_spec, YAF_DHCP_FP_TID,
                         TRUE, err))
    {
        return NULL;
    }

    if (!mdAddSpreadTmpl(session, yaf_dhcp_options_spec, YAF_DHCP_OPTIONS_TID,
                         TRUE, err))
    {
        return NULL;
    }

    /* fpExport Template */
    if (!mdAddSpreadTmpl(session, yaf_fpexport_spec, YAF_FPEXPORT_TID,
                         TRUE, err))
    {
        return NULL;
    }

    /* Payload Template */
    if (!mdAddSpreadTmpl(session, yaf_payload_spec, YAF_PAYLOAD_TID,
                         TRUE, err))
    {
        return NULL;
    }

    /* DPI TEMPLATES - HTTP*/
    if (!mdAddSpreadTmpl(session, yaf_http_spec, YAF_HTTP_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* IRC Template */
    if (!mdAddSpreadTmpl(session, yaf_singleBL_spec, YAF_IRC_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SSH Template */
    if (!mdAddSpreadTmpl(session, yaf_singleBL_spec, YAF_SSH_214_TID,
                         FALSE, err))
    {
        return NULL;
    }

    if (!mdAddSpreadTmpl(session, yaf_ssh_spec, YAF_SSH_TID,
                         FALSE, err))
    {
        return NULL;
    }

    if (!mdAddSpreadTmpl(session, yaf_pop3_spec, YAF_POP3_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* TFTP Template */
    if (!mdAddSpreadTmpl(session, yaf_tftp_spec, YAF_TFTP_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SLP Template */
    if (!mdAddSpreadTmpl(session, yaf_slp_spec, YAF_SLP_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* FTP Template */
    if (!mdAddSpreadTmpl(session, yaf_ftp_spec, YAF_FTP_TID, FALSE, err)) {
        return NULL;
    }

    /* IMAP Templates */
    if (!mdAddSpreadTmpl(session, yaf_imap_216_spec, YAF_IMAP_216_TID,
                         FALSE, err))
    {
        return NULL;
    }
    if (!mdAddSpreadTmpl(session, yaf_imap_spec, YAF_IMAP_TID, FALSE, err)) {
        return NULL;
    }

    /* RTSP Template */
    if (!mdAddSpreadTmpl(session, yaf_rtsp_spec, YAF_RTSP_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SIP Template */
    if (!mdAddSpreadTmpl(session, yaf_sip_spec, YAF_SIP_TID, FALSE, err)) {
        return NULL;
    }

    /* NNTP Template */
    if (!mdAddSpreadTmpl(session, yaf_nntp_spec, YAF_NNTP_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SMTP Templates */
    if (!mdAddSpreadTmpl(session, yaf_smtp_211_spec, YAF_SMTP_211_TID,
                         FALSE, err))
    {
        return NULL;
    }
    if (!mdAddSpreadTmpl(session, yaf_smtp_spec, YAF_SMTP_TID,
                         FALSE, err))
    {
        return NULL;
    }
    if (!mdAddSpreadTmpl(session, yaf_smtp_message_spec, YAF_SMTP_MESSAGE_TID,
                         FALSE, err))
    {
        return NULL;
    }
    if (!mdAddSpreadTmpl(session, yaf_smtp_header_spec, YAF_SMTP_HEADER_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS Template */
    if (!mdAddSpreadTmpl(session, yaf_dns_spec, YAF_DNS_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS QR Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsQR_spec, YAF_DNSQR_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS A Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsA_spec, YAF_DNSA_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS AAAA Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsAAAA_spec, YAF_DNSAAAA_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS CNAME Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsCNAME_spec, YAF_DNSCNAME_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS MX Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsMX_spec, YAF_DNSMX_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS NS Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsNS_spec, YAF_DNSNS_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS PTR Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsPTR_spec, YAF_DNSPTR_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS TXT Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsTXT_spec, YAF_DNSTXT_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS SOA Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsSOA_spec, YAF_DNSSOA_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS SRV Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsSRV_spec, YAF_DNSSRV_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS DS Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsDS_spec, YAF_DNSDS_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS RRSig Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsRRSig_spec, YAF_DNSRRSIG_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS NSEC Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsNSEC_spec, YAF_DNSNSEC_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS Key Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsKey_spec, YAF_DNSKEY_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNS NSEC3 Template */
    if (!mdAddSpreadTmpl(session, yaf_dnsNSEC3_spec, YAF_DNSNSEC3_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SSL Template from yaf 2.2.2 and earlier */
    if (!mdAddSpreadTmpl(session, yaf_ssl_y22_spec, YAF_SSL_Y22_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SSL Cert Template from yaf 2.2.2 and earlier */
    if (!mdAddSpreadTmpl(session, yaf_ssl_y22_cert_spec, YAF_SSL_Y22_CERT_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SSL Level 1 Template as of yaf 2.3.0 */
    if (!mdAddSpreadTmpl(session, yaf_ssl_spec, YAF_SSL_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SSL Level 2 Template as of yaf 2.3.0 */
    if (!mdAddSpreadTmpl(session, yaf_ssl_cert_spec, YAF_SSL_CERT_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* SSL Level 3 Template as of yaf 2.3.0 */
    if (!mdAddSpreadTmpl(session, yaf_ssl_subcert_spec, YAF_SSL_SUBCERT_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* Full Cert SSL Template */
    if (!mdAddSpreadTmpl(session, yaf_singleBL_spec, YAF_SSL_BINARY_CERT_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* MySQL Template */
    if (!mdAddSpreadTmpl(session, yaf_mysql_spec, YAF_MYSQL_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* MYSQL TXT Template */
    if (!mdAddSpreadTmpl(session, yaf_mysql_txt_spec, YAF_MYSQL_TXT_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* DNP 3.0 */
    if (!mdAddSpreadTmpl(session, yaf_dnp_spec, YAF_DNP_TID, FALSE, err)) {
        return NULL;
    }

    if (!mdAddSpreadTmpl(session, yaf_dnp_rec_spec, YAF_DNP_REC_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* Modbus */
    if (!mdAddSpreadTmpl(session, yaf_singleBL_spec, YAF_MODBUS_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /* ENIP */
    if (!mdAddSpreadTmpl(session, yaf_singleBL_spec, YAF_ENIP_TID,
                         FALSE, err))
    {
        return NULL;
    }

    /** RTP */
    if (!mdAddSpreadTmpl(session, yaf_rtp_spec, YAF_RTP_TID, FALSE, err)) {
        return NULL;
    }

    return session;
}
#endif  /* HAVE_SPREAD */

/**
 * mdAddTmpl
 *
 * Creates a new template using `spec` and adds the template to `session` (as
 * both internal and external) with template ID `tid` and the specified `name`
 * and `description` when metadata-export is enabled.
 *
 * Specifying TRUE for the `rev` parameter indicates that `spec` contains both
 * forward and reverse elements (e.g., the TCP flags template in YAF).  When
 * `rev` is TRUE, an additional template that does not contain the reverse
 * elements is added to the session (as an external template).  The `flags`
 * member of the specs in `spec` is expected to be non-zero for the reverse
 * elements and zero for the non-reverse elements.
 */
static fbTemplate_t *
mdAddTmpl(
    fbSession_t          *session,
    fbInfoElementSpec_t  *spec,
    uint16_t              tid,
    gboolean              rev,
    const gchar          *name,
    const gchar          *description,
    GError              **err)
{
    fbInfoModel_t *model = mdInfoModel();
    fbTemplate_t  *tmpl = NULL;
    uint16_t       rtid = rev ? tid | YTF_REV : tid;
    uint16_t       rv;

    tmpl = fbTemplateAlloc(model);

    /* Internal template - biflow */
    if (!fbTemplateAppendSpecArray(tmpl, spec, UINT32_MAX, err) ||
        !fbSessionAddTemplate(session, TRUE, rtid, tmpl, err))
    {
        fbTemplateFreeUnused(tmpl);
        return NULL;
    }

    /* External template - biflow */
#if SM_ENABLE_METADATA_EXPORT
    rv = fbSessionAddTemplateWithMetadata(session, FALSE, rtid, tmpl,
                                          name, description, err);
#else
    MD_UNUSED_PARAM(name);
    MD_UNUSED_PARAM(description);
    rv = fbSessionAddTemplate(session, FALSE, rtid, tmpl, err);
#endif /* if SM_ENABLE_METADATA_EXPORT */
    if (!rv) {
        return NULL;
    }

    if (rev) {
        tmpl = fbTemplateAlloc(model);

        /* External template - uniflow */
        if (!fbTemplateAppendSpecArray(tmpl, spec, 0, err)) {
            fbTemplateFreeUnused(tmpl);
            return NULL;
        }

#if SM_ENABLE_METADATA_EXPORT
        rv = fbSessionAddTemplateWithMetadata(session, FALSE, tid, tmpl,
                                              name, description, err);
#else
        rv = fbSessionAddTemplate(session, FALSE, tid, tmpl, err);
#endif
        if (!rv) {
            return NULL;
        }
    }

    return tmpl;
}


/*
 *  Helper function for functions mdInitExporterSession*() below.
 *
 *  Allocates a session (unless 'session' is non-NULL), enables
 *  metadata-export on the session when 'metadata_export' is TRUE, and
 *  adds the Option Templates for YAF stats and both Tombstone records
 *  when 'stats' is not equal to 1.
 */
static fbSession_t *
mdInitSessionHelper(
    fbSession_t  *session,
    GError      **err,
    uint8_t       stats,
    gboolean      metadata_export)
{
    /*Allocate the session */
    if (!session) {
        session = fbSessionAlloc(mdInfoModel());
    }
#if SM_ENABLE_METADATA_EXPORT
    if (metadata_export) {
        if (!fbSessionSetMetadataExportElements(
                session, TRUE, YAF_ELEMENT_METADATA_TID, err) ||
            !fbSessionSetMetadataExportTemplates(
                session, TRUE, YAF_TEMPLATE_METADATA_TID, err))
        {
            goto error;
        }
    }
#else  /* if SM_ENABLE_METADATA_EXPORT */
    MD_UNUSED_PARAM(metadata_export);
#endif /* if SM_ENABLE_METADATA_EXPORT */

    if (stats != 1) {
        fbTemplate_t *tmpl;

        tmpl = mdAddTmpl(session, yaf_process_stats_spec,
                         YAF_PROCESS_STATS_TID, FALSE,
                         "yaf_process_stats", NULL, err);
        if (!tmpl) {
            goto error;
        }
        fbTemplateSetOptionsScope(tmpl, 3);

        tmpl = mdAddTmpl(session, yaf_tombstone_spec, YAF_TOMBSTONE_TID, FALSE,
                         "tombstone_record", NULL, err);
        if (!tmpl) {
            goto error;
        }
        fbTemplateSetOptionsScope(tmpl, 3);

        tmpl = mdAddTmpl(session, yaf_tombstone_access_spec,
                         YAF_TOMBSTONE_ACCESS_TID, FALSE,
                         "tombstone_access", NULL, err);
        if (!tmpl) {
            goto error;
        }

#if 0
        /* No need to export the old tombstone template */
        tmpl = mdAddTmpl(session, yaf_tombstone_210_spec,
                         YAF_TOMBSTONE_210_TID, FALSE,
                         "tombstone_record", NULL, err);
        if (!tmpl) {
            goto error;
        }
        fbTemplateSetOptionsScope(tmpl, 2);

        tmpl = mdAddTmpl(session, yaf_tombstone_access_210_spec,
                         YAF_TOMBSTONE_ACCESS_210_TID, FALSE,
                         "tombstone_access", NULL, err);
        if (!tmpl) {
            goto error;
        }
#endif  /* 0 */
    }

    return session;

  error:
    fbSessionFree(session);
    return NULL;
}

/**
 *    Callback passed to mdExporterfBufSetup() whose signature must match
 *    md_sess_init_fn.
 */
fbSession_t *
mdInitExporterSessionDNSDedupOnly(
    fbSession_t  *session,
    GError      **err,
    uint8_t       stats,
    gboolean      metadata_export)
{
    fbTemplate_t *tmpl;

    session = mdInitSessionHelper(session, err, stats, metadata_export);
    if (!session) {
        return NULL;
    }

    tmpl = fbTemplateAlloc(mdInfoModel());

    if (!fbTemplateAppendSpecArray(tmpl, md_dns_dedup_spec, UINT32_MAX, err) ||
        !fbSessionAddTemplate(session, TRUE, MD_DNS_DEDUP_FULL, tmpl, err))
    {
        fbTemplateFreeUnused(tmpl);
        return NULL;
    }

    return session;
}

/**
 *    Callback passed to mdExporterfBufSetup() whose signature must match
 *    md_sess_init_fn.
 */
fbSession_t *
mdInitExporterSessionDNSRROnly(
    fbSession_t  *session,
    GError      **err,
    uint8_t       stats,
    gboolean      metadata_export)
{
    fbInfoModel_t *model = mdInfoModel();
    fbTemplate_t  *tmpl = NULL;
    uint16_t       rv;

    session = mdInitSessionHelper(session, err, stats, metadata_export);
    if (!session) {
        return NULL;
    }

    /* Internal template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, md_dnsrr_spec, 0xffffffff, err)
        || !fbSessionAddTemplate(session, TRUE, MD_DNSRR_TID, tmpl, err))
    {
        fbTemplateFreeUnused(tmpl);
        return NULL;
    }

    /* external template */
    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, md_dnsrr_spec, 0, err)) {
        return NULL;
    }

#if SM_ENABLE_METADATA_EXPORT
    rv = fbSessionAddTemplateWithMetadata(session, FALSE, MD_DNSRR_TID, tmpl,
                                          "md_dns_rr_external", NULL, err);
#else
    rv = fbSessionAddTemplate(session, FALSE, MD_DNSRR_TID, tmpl, err);
#endif
    if (!rv) {
        return NULL;
    }

    return session;
}

/**
 *    Callback passed to mdExporterfBufSetup() whose signature must match
 *    md_sess_init_fn.
 */
fbSession_t *
mdInitExporterSessionFlowOnly(
    fbSession_t  *session,
    GError      **err,
    uint8_t       stats,
    gboolean      metadata_export)
{
    session = mdInitSessionHelper(session, err, stats, metadata_export);
    if (!session) {
        return NULL;
    }

    /* YAF_SILK_FLOW_TID is only internal */
    {
        fbTemplate_t *tmpl = fbTemplateAlloc(mdInfoModel());

        if (!fbTemplateAppendSpecArray(tmpl, md_main_template_spec, ~0, err) ||
            !fbSessionAddTemplate(session, TRUE, YAF_SILK_FLOW_TID, tmpl, err))
        {
            fbTemplateFreeUnused(tmpl);
            return NULL;
        }
    }

    return session;
}

/**
 *    Callback passed to mdExporterfBufSetup() whose signature must match
 *    md_sess_init_fn.
 */
fbSession_t *
mdInitExporterSessionDedupOnly(
    fbSession_t  *session,
    GError      **err,
    uint8_t       stats,
    gboolean      metadata_export)
{
    session = mdInitSessionHelper(session, err, stats, metadata_export);
    if (!session) {
        return NULL;
    }

    if (!mdAddTmpl(session, md_dedup_spec, MD_DEDUP_FULL, FALSE,
                   "md_dedup_full", NULL, err))
    {
        return NULL;
    }

    return session;
}

/**
 *    Callback passed to mdExporterfBufSetup() whose signature must match
 *    md_sess_init_fn.
 */
fbSession_t *
mdInitExporterSessionSSLDedupOnly(
    fbSession_t  *session,
    GError      **err,
    uint8_t       stats,
    gboolean      metadata_export)
{
    session = mdInitSessionHelper(session, err, stats, metadata_export);
    if (!session) {
        return NULL;
    }

    /* SSL dedup spec 4 Export */
    if (!mdAddTmpl(session, md_ssl_dedup_spec, MD_SSL_DEDUP_TID, FALSE,
                   "md_ssl_dedup", NULL, err))
    {
        return NULL;
    }

    /* SSL certificate template */
    if (!mdAddTmpl(session, yaf_ssl_cert_spec, YAF_SSL_CERT_TID,
                   FALSE, "yaf_ssl_cert", NULL, err))
    {
        return NULL;
    }

    /* SSL Sub Template */
    if (!mdAddTmpl(session, yaf_ssl_subcert_spec, YAF_SSL_SUBCERT_TID,
                   FALSE, "yaf_ssl_subcert", NULL, err))
    {
        return NULL;
    }

    return session;
}

/**
 *    Callback passed to mdExporterfBufSetup() whose signature must match
 *    md_sess_init_fn.
 */
fbSession_t *
mdInitExporterSession(
    fbSession_t  *session,
    GError      **err,
    uint8_t       stats,
    gboolean      metadata_export)
{
    session = mdInitSessionHelper(session, err, stats, metadata_export);
    if (!session) {
        return NULL;
    }

    /* YAF_SILK_FLOW_TID is only internal */
    {
        fbTemplate_t *tmpl = fbTemplateAlloc(mdInfoModel());

        if (!fbTemplateAppendSpecArray(tmpl, md_main_template_spec, ~0, err) ||
            !fbSessionAddTemplate(session, TRUE, YAF_SILK_FLOW_TID, tmpl, err))
        {
            fbTemplateFreeUnused(tmpl);
            return NULL;
        }
    }

    /* dns dedup */
    if (!mdAddTmpl(session, md_dns_dedup_spec, MD_DNS_DEDUP_FULL,
                   FALSE, "md_dns_full", NULL, err))
    {
        return NULL;
    }

    /* dedup */
    if (!mdAddTmpl(session, md_dedup_spec, MD_DEDUP_FULL,
                   FALSE, "md_dedup_full", NULL, err))
    {
        return NULL;
    }

    /* ssl dedup */
    if (!mdAddTmpl(session, md_ssl_dedup_spec, MD_SSL_DEDUP_TID,
                   FALSE, "md_ssl_dedup", NULL, err))
    {
        return NULL;
    }

    /* flow stats template */
    if (!mdAddTmpl(session, yaf_flowstats_spec, YAF_FLOWSTATS_TID,
                   TRUE, "yaf_flow_stats", NULL, err))
    {
        return NULL;
    }

    /* Entropy Template */
    if (!mdAddTmpl(session, yaf_entropy_spec, YAF_ENTROPY_TID,
                   TRUE, "yaf_entropy", NULL, err))
    {
        return NULL;
    }

    /* TCP Template */
    if (!mdAddTmpl(session, yaf_tcp_spec, YAF_TCP_TID,
                   TRUE, "yaf_tcp", NULL, err))
    {
        return NULL;
    }

    /* MPTCP Template */
    if (!mdAddTmpl(session, yaf_mptcp_spec, YAF_MPTCP_TID,
                   FALSE, "yaf_mptcp", NULL, err))
    {
        return NULL;
    }

    /* MAC Template */
    if (!mdAddTmpl(session, yaf_mac_spec, YAF_MAC_TID,
                   FALSE, "yaf_mac", NULL, err))
    {
        return NULL;
    }

    /* p0f Template */
    if (!mdAddTmpl(session, yaf_p0f_spec, YAF_P0F_TID,
                   TRUE, "yaf_p0f", NULL, err))
    {
        return NULL;
    }

    /* dhcp Template */
    if (!mdAddTmpl(session, yaf_dhcp_fp_spec, YAF_DHCP_FP_TID,
                   TRUE, "yaf_dhcp", NULL, err))
    {
        return NULL;
    }

    /* dhcp Options Template */
    if (!mdAddTmpl(session, yaf_dhcp_options_spec, YAF_DHCP_OPTIONS_TID,
                   TRUE, "yaf_dhcp_options", NULL, err))
    {
        return NULL;
    }

    /* fpExport Template */
    if (!mdAddTmpl(session, yaf_fpexport_spec, YAF_FPEXPORT_TID,
                   TRUE, "yaf_fpexport", NULL, err))
    {
        return NULL;
    }

    /* Payload Template */
    if (!mdAddTmpl(session, yaf_payload_spec, YAF_PAYLOAD_TID,
                   TRUE, "yaf_payload", NULL, err))
    {
        return NULL;
    }

    /* DPI TEMPLATES - HTTP*/
    if (!mdAddTmpl(session, yaf_http_spec, YAF_HTTP_TID,
                   FALSE, "yaf_http", NULL, err))
    {
        return NULL;
    }

    /* IRC Template */
    if (!mdAddTmpl(session, yaf_singleBL_spec, YAF_IRC_TID,
                   FALSE, "yaf_irc", NULL, err))
    {
        return NULL;
    }

    /* SSH Template */
    if (!mdAddTmpl(session, yaf_singleBL_spec, YAF_SSH_214_TID,
                   FALSE, "yaf_ssh", NULL, err))
    {
        return NULL;
    }

    if (!mdAddTmpl(session, yaf_ssh_spec, YAF_SSH_TID,
                   FALSE, "yaf_ssh", NULL, err))
    {
        return NULL;
    }

    /* POP3 Template */
    if (!mdAddTmpl(session, yaf_pop3_spec, YAF_POP3_TID,
                   FALSE, "yaf_pop3", NULL, err))
    {
        return NULL;
    }

    /* TFTP Template */
    if (!mdAddTmpl(session, yaf_tftp_spec, YAF_TFTP_TID,
                   FALSE, "yaf_tftp", NULL, err))
    {
        return NULL;
    }

    /* SLP Template */
    if (!mdAddTmpl(session, yaf_slp_spec, YAF_SLP_TID,
                   FALSE, "yaf_slp", NULL, err))
    {
        return NULL;
    }

    /* FTP Template */
    if (!mdAddTmpl(session, yaf_ftp_spec, YAF_FTP_TID,
                   FALSE, "yaf_ftp", NULL, err))
    {
        return NULL;
    }

    /* IMAP Templates */
    if (!mdAddTmpl(session, yaf_imap_216_spec, YAF_IMAP_216_TID,
                   FALSE, "yaf_imap", NULL, err))
    {
        return NULL;
    }
    if (!mdAddTmpl(session, yaf_imap_spec, YAF_IMAP_TID,
                   FALSE, "yaf_imap", NULL, err))
    {
        return NULL;
    }

    /* RTSP Template */
    if (!mdAddTmpl(session, yaf_rtsp_spec, YAF_RTSP_TID,
                   FALSE, "yaf_rtsp", NULL, err))
    {
        return NULL;
    }

    /* SIP Template */
    if (!mdAddTmpl(session, yaf_sip_spec, YAF_SIP_TID,
                   FALSE, "yaf_sip", NULL, err))
    {
        return NULL;
    }

    /* NNTP Template */
    if (!mdAddTmpl(session, yaf_nntp_spec, YAF_NNTP_TID,
                   FALSE, "yaf_nntp", NULL, err))
    {
        return NULL;
    }

    /* SMTP Templates */
    if (!mdAddTmpl(session, yaf_smtp_211_spec, YAF_SMTP_211_TID,
                   FALSE, "yaf_smtp", NULL, err))
    {
        return NULL;
    }
    if (!mdAddTmpl(session, yaf_smtp_spec, YAF_SMTP_TID,
                   FALSE, "yaf_smtp", NULL, err))
    {
        return NULL;
    }
    if (!mdAddTmpl(session, yaf_smtp_message_spec, YAF_SMTP_MESSAGE_TID,
                   FALSE, "yaf_smtp_message", NULL, err))
    {
        return NULL;
    }
    if (!mdAddTmpl(session, yaf_smtp_header_spec, YAF_SMTP_HEADER_TID,
                   FALSE, "yaf_smtp_header", NULL, err))
    {
        return NULL;
    }

    /* DNS Template */
    if (!mdAddTmpl(session, yaf_dns_spec, YAF_DNS_TID,
                   FALSE, "yaf_dns", NULL, err))
    {
        return NULL;
    }

    /* DNS QR Template */
    if (!mdAddTmpl(session, yaf_dnsQR_spec, YAF_DNSQR_TID,
                   FALSE, "yaf_dns_qr", NULL, err))
    {
        return NULL;
    }

    /* DNS A Template */
    if (!mdAddTmpl(session, yaf_dnsA_spec, YAF_DNSA_TID,
                   FALSE, "yaf_dns_a", NULL, err))
    {
        return NULL;
    }

    /* DNS AAAA Template */
    if (!mdAddTmpl(session, yaf_dnsAAAA_spec, YAF_DNSAAAA_TID,
                   FALSE, "yaf_dns_aaaa", NULL, err))
    {
        return NULL;
    }

    /* DNS CNAME Template */
    if (!mdAddTmpl(session, yaf_dnsCNAME_spec, YAF_DNSCNAME_TID,
                   FALSE, "yaf_dns_cname", NULL, err))
    {
        return NULL;
    }

    /* DNS MX Template */
    if (!mdAddTmpl(session, yaf_dnsMX_spec, YAF_DNSMX_TID,
                   FALSE, "yaf_dns_mx", NULL, err))
    {
        return NULL;
    }

    /* DNS NS Template */
    if (!mdAddTmpl(session, yaf_dnsNS_spec, YAF_DNSNS_TID,
                   FALSE, "yaf_dns_ns", NULL, err))
    {
        return NULL;
    }

    /* DNS PTR Template */
    if (!mdAddTmpl(session, yaf_dnsPTR_spec, YAF_DNSPTR_TID,
                   FALSE, "yaf_dns_ptr", NULL, err))
    {
        return NULL;
    }

    /* DNS TXT Template */
    if (!mdAddTmpl(session, yaf_dnsTXT_spec, YAF_DNSTXT_TID,
                   FALSE, "yaf_dns_txt", NULL, err))
    {
        return NULL;
    }

    /* DNS SOA Template */
    if (!mdAddTmpl(session, yaf_dnsSOA_spec, YAF_DNSSOA_TID,
                   FALSE, "yaf_dns_soa", NULL, err))
    {
        return NULL;
    }

    /* DNS SRV Template */
    if (!mdAddTmpl(session, yaf_dnsSRV_spec, YAF_DNSSRV_TID,
                   FALSE, "yaf_dns_srv", NULL, err))
    {
        return NULL;
    }

    /* DNS DS Template */
    if (!mdAddTmpl(session, yaf_dnsDS_spec, YAF_DNSDS_TID,
                   FALSE, "yaf_dns_ds", NULL, err))
    {
        return NULL;
    }

    /* DNS RRSig Template */
    if (!mdAddTmpl(session, yaf_dnsRRSig_spec, YAF_DNSRRSIG_TID,
                   FALSE, "yaf_dns_sig", NULL, err))
    {
        return NULL;
    }

    /* DNS NSEC Template */
    if (!mdAddTmpl(session, yaf_dnsNSEC_spec, YAF_DNSNSEC_TID,
                   FALSE, "yaf_dns_nsec", NULL, err))
    {
        return NULL;
    }

    /* DNS Key Template */
    if (!mdAddTmpl(session, yaf_dnsKey_spec, YAF_DNSKEY_TID,
                   FALSE, "yaf_dns_key", NULL, err))
    {
        return NULL;
    }

    /* DNS NSEC3 Template */
    if (!mdAddTmpl(session, yaf_dnsNSEC3_spec, YAF_DNSNSEC3_TID,
                   FALSE, "yaf_dns_nsec3", NULL, err))
    {
        return NULL;
    }

    /* Full CERT SSL Template */
    if (!mdAddTmpl(session, yaf_singleBL_spec, YAF_SSL_BINARY_CERT_TID,
                   FALSE, "yaf_ssl_cert_full", NULL, err))
    {
        return NULL;
    }

    /* SSL Template from yaf 2.2.2 and earlier */
    if (!mdAddTmpl(session, yaf_ssl_y22_spec, YAF_SSL_Y22_TID,
                   FALSE, "yaf_ssl", NULL, err))
    {
        return NULL;
    }

    /* SSL Cert Template from yaf 2.2.2 and earlier */
    if (!mdAddTmpl(session, yaf_ssl_y22_cert_spec, YAF_SSL_Y22_CERT_TID,
                   FALSE, "yaf_ssl_cert", NULL, err))
    {
        return NULL;
    }

    /* SSL Level 1 Template as of yaf 2.3.0 */
    if (!mdAddTmpl(session, yaf_ssl_spec, YAF_SSL_TID,
                   FALSE, "yaf_ssl", NULL, err))
    {
        return NULL;
    }

    if (md_config.rewrite_ssl_certs) {
        /* Flattened SSL CERT Template replaces the existing template */
        if (!mdAddTmpl(session, md_ssl_certificate_spec, YAF_SSL_CERT_TID,
                       FALSE, "yaf_ssl_cert", NULL, err))
        {
            return NULL;
        }
    } else {
        /* SSL Level 2 Template as of yaf 2.3.0 */
        if (!mdAddTmpl(session, yaf_ssl_cert_spec, YAF_SSL_CERT_TID,
                       FALSE, "yaf_ssl_cert", NULL, err))
        {
            return NULL;
        }

        /* SSL Level 3 Template as of yaf 2.3.0 */
        if (!mdAddTmpl(session, yaf_ssl_subcert_spec, YAF_SSL_SUBCERT_TID,
                       FALSE, "yaf_ssl_subcert", NULL, err))
        {
            return NULL;
        }
    }

    /* MySQL Template */
    if (!mdAddTmpl(session, yaf_mysql_spec, YAF_MYSQL_TID,
                   FALSE, "yaf_mysql", NULL, err))
    {
        return NULL;
    }

    /* MYSQL TXT Template */
    if (!mdAddTmpl(session, yaf_mysql_txt_spec, YAF_MYSQL_TXT_TID,
                   FALSE, "yaf_mysql_txt", NULL, err))
    {
        return NULL;
    }

    /* DNP 3.0 */
    if (!mdAddTmpl(session, yaf_dnp_spec, YAF_DNP_TID,
                   FALSE, "yaf_dnp", NULL, err))
    {
        return NULL;
    }

    if (!mdAddTmpl(session, yaf_dnp_rec_spec, YAF_DNP_REC_TID,
                   FALSE, "yaf_dnp_rec", NULL, err))
    {
        return NULL;
    }

    /* Modbus */
    if (!mdAddTmpl(session, yaf_singleBL_spec, YAF_MODBUS_TID,
                   FALSE, "yaf_modbus", NULL, err))
    {
        return NULL;
    }

    /* ENIP */
    if (!mdAddTmpl(session, yaf_singleBL_spec, YAF_ENIP_TID,
                   FALSE, "yaf_enip", NULL, err))
    {
        return NULL;
    }

    /* RTP */
    if (!mdAddTmpl(session, yaf_rtp_spec, YAF_RTP_TID,
                   FALSE, "yaf_rtp", NULL, err))
    {
        return NULL;
    }

    /* SiLK IPSet Matching */
    if (!mdAddTmpl(session, md_ipset_rec_spec, MD_IPSET_REC_TID,
                   FALSE, "md_ipset", NULL, err))
    {
        return NULL;
    }

    /* SiLK Prefix Map Label */
    if (!mdAddTmpl(session, md_pmap_rec_spec, MD_PMAP_REC_TID,
                   FALSE, "md_pmap", NULL, err))
    {
        return NULL;
    }

    /* SiLK Type and Sensor Infomation */
    if (!mdAddTmpl(session, md_silk_type_sensor_spec, MD_SILK_TYPE_SENSOR_TID,
                   FALSE, "md_silk_type_sensor", NULL, err))
    {
        return NULL;
    }

    return session;
}


/**
 *  Define a template using all the elements of 'spec' and add it as an
 *  internal template to 'session' with template ID 'tid'.
 */
static gboolean
mdCollectorAddTemplate(
    fbInfoModel_t        *model,
    fbSession_t          *session,
    fbInfoElementSpec_t   spec[],
    uint16_t              tid,
    GError              **err)
{
    fbTemplate_t *tmpl;

    tmpl = fbTemplateAlloc(model);
    if (!fbTemplateAppendSpecArray(tmpl, spec, UINT32_MAX, err) ||
        !fbSessionAddTemplate(session, TRUE, tid, tmpl, err))
    {
        fbTemplateFreeUnused(tmpl);
        return FALSE;
    }

    return TRUE;
}


/**
 * mdInitCollectorSession
 *
 *  =============================================
 * create a session (fbSession_t) and attach it to the info model via
 * fbSessionAlloc create a collector session and then pass that session
 * to the listener to finish building the collector
 *
 **/
fbSession_t *
mdInitCollectorSession(
    GError **err)
{
    fbInfoModel_t *model = mdInfoModel();
    fbSession_t   *session = fbSessionAlloc(model);

    if (!mdCollectorAddTemplate(model, session, md_main_template_spec,
                                YAF_SILK_FLOW_TID, err))
    {
        return NULL;
    }

    if (!mdCollectorAddTemplate(model, session, yaf_process_stats_spec,
                                YAF_PROCESS_STATS_TID, err))
    {
        return NULL;
    }

    if (!mdCollectorAddTemplate(model, session, yaf_tombstone_spec,
                                YAF_TOMBSTONE_TID, err))
    {
        return NULL;
    }

    if (!mdCollectorAddTemplate(model, session, yaf_tombstone_access_spec,
                                YAF_TOMBSTONE_ACCESS_TID, err))
    {
        return NULL;
    }

    if (!mdCollectorAddTemplate(model, session, yaf_tombstone_210_spec,
                                YAF_TOMBSTONE_210_TID, err))
    {
        return NULL;
    }

    if (!mdCollectorAddTemplate(model, session, yaf_tombstone_access_210_spec,
                                YAF_TOMBSTONE_ACCESS_210_TID, err))
    {
        return NULL;
    }

    /* read dns rr only template */
    if (!mdCollectorAddTemplate(model, session, md_dnsrr_spec,
                                MD_DNSRR_TID, err))
    {
        return NULL;
    }

    /* dns dedup */
    if (!mdCollectorAddTemplate(model, session, md_dns_dedup_spec,
                                MD_DNS_DEDUP_FULL, err))
    {
        return NULL;
    }

    /* dedup */
    if (!mdCollectorAddTemplate(model, session, md_dedup_spec,
                                MD_DEDUP_FULL, err))
    {
        return NULL;
    }

    /* ssl dedup */
    if (!mdCollectorAddTemplate(model, session, md_ssl_dedup_spec,
                                MD_SSL_DEDUP_TID, err))
    {
        return NULL;
    }

    /* ssl cert; is top level in ssl_dedup input */
    if (!mdCollectorAddTemplate(model, session, yaf_ssl_cert_spec,
                                YAF_SSL_CERT_TID, err))
    {
        return NULL;
    }

    if (md_config.rewrite_ssl_certs) {
        /* for processing data from an upstream SM */
        if (!mdCollectorAddTemplate(model, session, md_ssl_certificate_spec,
                                    MD_SSL_CERTIFICATE_TID, err))
        {
            return NULL;
        }
    }

    /* mthomas.2022.03.10. define YAF_SSL_TID in the collector to ensure
     * incoming data is transcoded how we expect; previously this used a
     * different internal TID */
    if (!mdCollectorAddTemplate(model, session, yaf_ssl_spec,
                                YAF_SSL_TID, err))
    {
        return NULL;
    }

    /* Add templates for protocols that support Mid-Encryption where the TID
     * was not changed to reflect the presence of the SSL subTemplateList. */
    if (!mdCollectorAddTemplate(model, session, yaf_pop3_spec,
                                YAF_POP3_TID, err))
    {
        return NULL;
    }
    if (!mdCollectorAddTemplate(model, session, yaf_smtp_spec,
                                YAF_SMTP_TID, err))
    {
        return NULL;
    }
    /* IMAP uses different TIDs */

    /** we will want to add a template callback eventually */
    fbSessionAddNewTemplateCallback(session, mdTemplateCallback, NULL);

    return session;
}



/**
 * mdSetExportTemplate
 *
 * Sets the export template on the fbuf.
 *
 * Returns TRUE if `tid` is already defined for the `fbuf`'s session.  Returns
 * FALSE if the attempt to set the TID returns any error other then template
 * not found.
 *
 * If the template is not found, the function creates it and adds it to the
 * `fbuf`, returning the result of that action.
 */
gboolean
mdSetExportTemplate(
    fBuf_t    *fbuf,
    uint32_t   tid,
    GError   **err)
{
    fbSession_t  *session = NULL;
    fbTemplate_t *tmpl = NULL;
    GString      *template_name = NULL;
    uint16_t      rv;

    if (fBufSetExportTemplate(fbuf, (uint16_t)tid, err)) {
        return TRUE;
    }
    if (!g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_TMPL)) {
        return FALSE;
    }
    g_clear_error(err);

    session = fBufGetSession(fbuf);
    tmpl = fbTemplateAlloc(mdInfoModel());
#if SM_ENABLE_METADATA_EXPORT
    template_name = g_string_sized_new(64);
#endif

    if (tid == YAF_PROCESS_STATS_TID) {
        if (!fbTemplateAppendSpecArray(tmpl, yaf_process_stats_spec,
                                       YAF_PROCESS_STATS_TID, err))
        {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
        fbTemplateSetOptionsScope(tmpl, 3);
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, "yaf_stats_options");
#endif
    } else if (tid == YAF_TOMBSTONE_TID) {
        if (!fbTemplateAppendSpecArray(tmpl, yaf_tombstone_spec,
                                       YAF_TOMBSTONE_TID, err))
        {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
        fbTemplateSetOptionsScope(tmpl, 3);
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, "tombstone_record");
#endif
    } else if (tid == YAF_TOMBSTONE_ACCESS_TID) {
        if (!fbTemplateAppendSpecArray(tmpl, yaf_tombstone_access_spec,
                                       YAF_TOMBSTONE_ACCESS_TID, err))
        {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, "tombstone_access");
#endif
    } else if (MD_CHECK_TID_IS_DNS_DEDUP(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_dns_dedup_spec,
                                       (tid & (~MD_DNS_DEDUP_OUT)), err))
        {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, MD_DNS_DEDUP_OUT_NAME);
        if (tid & MD_LAST_SEEN) {
            g_string_append(template_name, MD_LAST_SEEN_NAME);
        }
        if (tid & MD_DNS_AREC) {
            g_string_append(template_name, MD_DNS_AREC_NAME);
        }
        if (tid & MD_DNS_OREC) {
            g_string_append(template_name, MD_DNS_OREC_NAME);
        }
        if (tid & MD_DNS_AAAAREC) {
            g_string_append(template_name, MD_DNS_AAAAREC_NAME);
        }
#endif /* if SM_ENABLE_METADATA_EXPORT */
    } else if (MD_CHECK_TID_IS_DNSRR(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_dnsrr_spec,
                                       (tid & (~MD_DNSRR_TID)), err))
        {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, MD_DNSRR_NAME);
        if (tid & YTF_IP6) {
            g_string_append(template_name, YTF_IP6_NAME);
        } else {
            g_string_append(template_name, YTF_IP4_NAME);
        }
        if (tid & MD_DNSRR_FULL) {
            g_string_append(template_name, MD_DNSRR_FULL_NAME);
        }
#endif /* if SM_ENABLE_METADATA_EXPORT */
    } else if (MD_CHECK_TID_IS_DEDUP(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_dedup_spec,
                                       (tid & (~MD_DEDUP_TID)), err))
        {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, MD_DEDUP_NAME);
        if (tid & MD_DEDUP_SSL) {
            g_string_append(template_name, MD_DEDUP_SSL_NAME);
        }
#endif /* if SM_ENABLE_METADATA_EXPORT */
    } else if (MD_CHECK_TID_IS_SSL_DEDUP(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_ssl_dedup_spec, tid, err)) {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, MD_SSL_NAME);
#endif
    } else {
        if (!fbTemplateAppendSpecArray(tmpl, md_main_template_spec,
                                       tid, err))
        {
            g_string_free(template_name, TRUE);
            return FALSE;
        }
#if SM_ENABLE_METADATA_EXPORT
        g_string_assign(template_name, MD_MAIN_TEMPLATE_NAME);
        if (tid & YTF_DELTA) {
            g_string_append(template_name, YTF_DELTA_NAME);
        } else {
            g_string_append(template_name, YTF_TOTAL_NAME);
        }
        if (tid & YTF_REV) {
            g_string_append(template_name, YTF_REV_NAME);
        }
        if (tid & YTF_IP6) {
            g_string_append(template_name, YTF_IP6_NAME);
        } else {
            g_string_append(template_name, YTF_IP4_NAME);
        }
        if (tid & YTF_TCP) {
            g_string_append(template_name, YTF_TCP_NAME);
        }
        if (tid & YTF_DAGIF) {
            g_string_append(template_name, YTF_DAGIF_NAME);
        }
        if (tid & YTF_MPLS) {
            g_string_append(template_name, YTF_MPLS_NAME);
        }
        if (tid & YTF_NDPI) {
            g_string_append(template_name, YTF_NDPI_NAME);
        }
#endif /* if SM_ENABLE_METADATA_EXPORT */
    }

#if SM_ENABLE_METADATA_EXPORT
    rv = fbSessionAddTemplateWithMetadata(session, FALSE, tid, tmpl,
                                          template_name->str, NULL, err);
    g_string_free(template_name, TRUE);
#else
    rv = fbSessionAddTemplate(session, FALSE, tid, tmpl, err);
#endif  /* #else of #if SM_ENABLE_METADATA_EXPORT */
    if (!rv) {
        return FALSE;
    }

    return fBufSetExportTemplate(fbuf, tid, err);
}


#if HAVE_SPREAD
/**
 * mdSetSpreadExportTemplate
 *
 * set the template on the fbuf and groups
 *
 */
gboolean
mdSetSpreadExportTemplate(
    fBuf_t            *fbuf,
    fbSpreadParams_t  *sp,
    uint32_t           tid,
    char             **groups,
    int                num_groups,
    GError           **err)
{
    fbSession_t  *session = NULL;
    fbTemplate_t *tmpl = NULL;

    if (fBufSetExportTemplate(fbuf, tid, err)) {
        return TRUE;
    }

    if (!g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_TMPL)) {
        return FALSE;
    }

    g_clear_error(err);

    session = fBufGetSession(fbuf);
    tmpl = fbTemplateAlloc(mdInfoModel());

    if (tid == YAF_PROCESS_STATS_210_TID) {
        if (!fbTemplateAppendSpecArray(tmpl, yaf_process_stats_spec,
                                       YAF_PROCESS_STATS_210_TID, err))
        {
            return FALSE;
        }
        fbTemplateSetOptionsScope(tmpl, 3);
    } else if (tid == YAF_TOMBSTONE_TID) {
        if (!fbTemplateAppendSpecArray(tmpl, yaf_tombstone_spec,
                                       YAF_TOMBSTONE_TID, err))
        {
            return FALSE;
        }
        fbTemplateSetOptionsScope(tmpl, 3);
    } else if (tid == YAF_TOMBSTONE_ACCESS_TID) {
        if (!fbTemplateAppendSpecArray(tmpl, yaf_tombstone_access_spec,
                                       YAF_TOMBSTONE_ACCESS_TID, err))
        {
            return FALSE;
        }
    } else if (MD_CHECK_TID_IS_DNS_DEDUP(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_dns_dedup_spec,
                                       (tid & (~MD_DNS_DEDUP_OUT)), err))
        {
            return FALSE;
        }
    } else if (MD_CHECK_TID_IS_DNSRR(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_dnsrr_spec,
                                       (tid & (~MD_DNSRR_TID)), err))
        {
            return FALSE;
        }
    } else if (MD_CHECK_TID_IS_DEDUP(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_dedup_spec,
                                       (tid & (~MD_DEDUP_TID)), err))
        {
            return FALSE;
        }
    } else if (MD_CHECK_TID_IS_SSL_DEDUP(tid)) {
        if (!fbTemplateAppendSpecArray(tmpl, md_ssl_dedup_spec, tid, err)) {
            return FALSE;
        }
    } else {
        if (!fbTemplateAppendSpecArray(tmpl, md_main_template_spec,
                                       (tid & (~YAF_SILK_FLOW_TID)), err))
        {
            return FALSE;
        }
    }

    if (!fbSessionAddTemplatesMulticast(session, sp->groups, FALSE, tid,
                                        tmpl, err))
    {
        return FALSE;
    }

    fBufSetSpreadExportGroup(fbuf, groups, num_groups, err);

    return fBufSetExportTemplate(fbuf, tid, err);
}

#endif  /* HAVE_SPREAD */



/**
 * mdOptionsCheck
 *
 *  Gets the template for the next record to be read from the input buffer.
 *  Sets the referent of `tmpl` and `tid` to the template and its ID, and
 *  returns TRUE if the template is an options template and FALSE if it is
 *  not.
 *
 *  If the record should be ignored (for example, it is a meta-data record),
 *  an error is set, the referent of `tid` is set to 0, and TRUE is retured.
 *
 *  If there is a transitory error reading from the buffer, an error is set
 *  the referent of `tid` is set to 0, and TRUE is returned.  When a fatal
 *  error occurs reading from the buffer (including end of file), an error is
 *  set, the buffer is freed, and FALSE is returned.
 */
gboolean
mdOptionsCheck(
    fBuf_t          **fbuf,
    uint16_t         *tid,
    fbTemplate_t    **tmpl,
    mdTmplContext_t **tctx,
    GError          **err)
{
    static mdTmplContext_t  empty_tctx;

    *tmpl = fBufNextCollectionTemplate(*fbuf, tid, err);
    if (NULL == *tmpl) {
        /* No template.  If transitory error, clear tid and return TRUE; else
         * free fbuf and return FALSE. */
        if ((g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOM)) ||
            (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_NLREAD)) ||
            (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_IPFIX)))
        {
            *tid = 0;
            return TRUE;
        }
        fBufFree(*fbuf);
        *fbuf = NULL;
        return FALSE;
    }

    *tctx = fbTemplateGetContext(*tmpl);
    if (NULL == *tctx) {
        g_debug("No template context defined for template %#06x", *tid);
        *tctx = &empty_tctx;
    }
    if (fbTemplateGetOptionsScope(*tmpl)) {
        /* options message; set `tid` to 0 if the record should be ignored */
        if ((*tctx)->is_metadata_element) {
            /* ignore template metadata records */
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_TMPL,
                        "Got a template stats message - ignore");
            *tid = 0;
        } else if ((*tctx)->is_metadata_template) {
            /* ignore template metadata records */
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_TMPL,
                        "Got a info element type  message - ignore");
            *tid = 0;
        }

        return TRUE;
    }

    /* Valid non-options record */
    return FALSE;
}


/*
 *    If the export time of the most recent message from `fbuf` is more recent
 *    than the current time in `ctx`, update the current time.
 */
static void
mdSetCurrentTimeFromExport(
    mdContext_t  *ctx,
    fBuf_t       *fbuf)
{
    yfTime_t  export;

    yfTimeFromSeconds(&export, fBufGetExportTime(fbuf));
    if (yfTimeCmpOp(export, ctx->cfg->ctime, >)) {
        ctx->cfg->ctime = export;
    }
}


gboolean
mdIgnoreRecord(
    mdContext_t  *ctx,
    fBuf_t       *fbuf,
    GError      **err)
{
    yaf_tombstone_access_t  ignore;
    size_t                  len = sizeof(ignore);

    MD_UNUSED_PARAM(ctx);

    /*
     * Since we are simply discarding the record, we can use whatever template
     * we want. Use the smallest template (with no sublists) known to exist in
     * the collector's session.
     */

    if (!fBufSetInternalTemplate(fbuf, YAF_TOMBSTONE_ACCESS_TID, err)) {
        /* log an error */
        return FALSE;
    }

    if (!fBufNext(fbuf, (uint8_t *)&ignore, &len, err)) {
        /* End of Set. */
        g_clear_error(err);
    }

    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}

/**
 * mdForwardOptions
 *
 * forward the options record to the exporters that
 * are configured to receive YAF stats
 *
 */
gboolean
mdForwardOptions(
    mdContext_t        *ctx,
    md_collect_node_t  *collector,
    uint16_t            tid,
    GError            **err)
{
    switch (tid) {
      case YAF_PROCESS_STATS_210_TID:
      case YAF_PROCESS_STATS_TID:
        return mdForwardStats(ctx, collector, err);
      case YAF_TOMBSTONE_TID:
      case YAF_TOMBSTONE_210_TID:
        return mdForwardTombstone(ctx, collector, tid, err);
      default:
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_TMPL,
                    "Unknown Options message with TID 0x%x - ignore", tid);
        break;
    }

    return FALSE;
}

/**
 * mdForwardTombstone
 *
 * forward the tombstone record to the exporters that
 * are configured to receive YAF stats
 *
 */
static gboolean
mdForwardTombstone(
    mdContext_t        *ctx,
    md_collect_node_t  *collector,
    uint16_t            tid,
    GError            **err)
{
    fBuf_t                 *fbuf = collector->fbuf;
    fbSession_t            *session;
    fbTemplate_t           *accessTemplate;
    gboolean                rc;
    yaf_tombstone_t         tombstone;
    yaf_tombstone_210_t     oldTombstone;
    yaf_tombstone_access_t *accessListPtr;
    size_t                  tombstone_len = sizeof(tombstone);
    size_t                  old_tombstone_len = sizeof(oldTombstone);
    md_export_node_t       *cnode = NULL;

    if (!fBufSetInternalTemplate(fbuf, tid, err)) {
        /* log an error */
        return FALSE;
    }

    memset(&tombstone, 0, tombstone_len);
    memset(&oldTombstone, 0, old_tombstone_len);

    if (tid == YAF_TOMBSTONE_TID) {
        rc = fBufNext(fbuf, (uint8_t *)&tombstone, &tombstone_len, err);
    } else if (tid == YAF_TOMBSTONE_210_TID) {
        rc = fBufNext(fbuf, (uint8_t *)&oldTombstone, &old_tombstone_len, err);
    } else {
        return FALSE;
    }

    if (FALSE == rc) {
        /* End of Set. */
        g_clear_error(err);
        goto end;
    }

    /*  We use no_stats for tombstone as well */
    if (ctx->cfg->no_stats) {
        goto end;
    }

    /* If we are using the old format, populate the new format */
    if (tid == YAF_TOMBSTONE_210_TID) {
        tombstone.tombstoneId = oldTombstone.tombstoneId;
        tombstone.exporterConfiguredId = oldTombstone.exporterConfiguredId;
        tombstone.exportingProcessId = 0;
        accessListPtr = (yaf_tombstone_access_t *)fbSubTemplateListGetDataPtr(
            &(oldTombstone.accessList));
        tombstone.observationTimeSeconds =
            accessListPtr->observationTimeSeconds;
        tombstone.observationDomainId = ctx->cfg->current_domain;
        tombstone.accessList = oldTombstone.accessList;
        session = fBufGetSession(fbuf);
        accessTemplate = fbSessionGetTemplate(session, TRUE,
                                              YAF_TOMBSTONE_ACCESS_TID, err);
        if (!accessTemplate) {
            goto end;
        }
        tombstone.accessList.tmpl = accessTemplate;
        tombstone.accessList.tmplID = YAF_TOMBSTONE_ACCESS_TID;
    }

    g_message("Received Tombstone record: observationDomain:%d, "
              "exporterId:%d:%d, tombstoneId: %d",
              tombstone.observationDomainId, tombstone.exporterConfiguredId,
              tombstone.exportingProcessId, tombstone.tombstoneId);

    /* This works because the old and new access templates are the same size */
    accessListPtr = fbSubTemplateListAddNewElements(&(tombstone.accessList), 1);
    accessListPtr->certToolId = 2;
    accessListPtr->observationTimeSeconds = (int)time(NULL);

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }
        if (!mdExporterWriteOptions(ctx->cfg, cnode->exp, (uint8_t *)&tombstone,
                                    sizeof(yaf_tombstone_t),
                                    YAF_TOMBSTONE_TID, err))
        {
            return FALSE;
        }
    }

  end:
    /* free subTemplateList */
    fbSubTemplateListClear(&(tombstone.accessList));

    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}

/**
 * mdForwardStats
 *
 * forward the option stats record to the exporters that
 * are configured to receive YAF stats
 *
 */
static gboolean
mdForwardStats(
    mdContext_t        *ctx,
    md_collect_node_t  *collector,
    GError            **err)
{
    fBuf_t              *fbuf = collector->fbuf;
    yaf_process_stats_t  stats;
    size_t               stats_len = sizeof(stats);
    md_export_node_t    *cnode = NULL;
    char                *colname = mdCollectorGetName(collector);

    ctx->stats->recvd_stats++;
    collector->stats->recvd_stats++;

    if (!fBufSetInternalTemplate(fbuf, YAF_PROCESS_STATS_TID, err)) {
        /* log an error */
        return FALSE;
    }

    if (!fBufNext(fbuf, (uint8_t *)&stats, &stats_len, err)) {
        /* End of Set. */
        g_clear_error(err);
        goto end;
    }

    if (ctx->cfg->no_stats) {
        goto end;
    }

    mdLogStats(&stats, colname);

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }
        if (!mdExporterWriteOptions(ctx->cfg, cnode->exp, (uint8_t *)&stats,
                                    sizeof(yaf_process_stats_t),
                                    YAF_PROCESS_STATS_TID, err))
        {
            return FALSE;
        }
    }

  end:
    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}

/**
 * mdForwardDNSDedup
 *
 */
gboolean
mdForwardDNSDedup(
    mdContext_t            *ctx,
    const mdTmplContext_t  *tctx,
    fBuf_t                 *fbuf,
    GError                **err)
{
    md_dns_dedup_t    dns;
    size_t            dns_len = sizeof(dns);
    md_export_node_t *cnode = NULL;
    uint16_t          tid;

    if (!fBufSetInternalTemplate(fbuf, MD_DNS_DEDUP_FULL, err)) {
        return FALSE;
    }
    if (!fBufNext(fbuf, (uint8_t *)&dns, &dns_len, err)) {
        /* End of Set. */
        g_clear_error(err);
        goto end;
    }

    /* handle older records without nanoseconds */
    if (!tctx->attr.has_nano) {
        yfTime_t  t;
        yfTimeFromMilli(&t, dns.flowStartMilliseconds);
        yfTimeToNTP(&dns.flowStartNanoseconds, t);
        yfTimeFromMilli(&t, dns.flowEndMilliseconds);
        yfTimeToNTP(&dns.flowEndNanoseconds, t);
    }

    mdSetCurrentTimeFromExport(ctx, fbuf);

    tid = tctx->tid;

    /* Convert IPv6 dedup records from SM-1.9.x and earlier to the SM-1.10
     * layout by copying the rrdata into the sourceIPv6Address (added in 1.10)
     * and adjust the TID.
     *
     * Assume the record is from pre-SM-1.10 when the rrtype is 28, the TID is
     * 0xCEE8 or 0xCEEA, and the length of the rrdata is 16. */
    if ((28 == dns.rrtype)
        && ((tid & ~MD_LAST_SEEN) == (MD_DNS_DEDUP_OUT | MD_DNS_OREC))
        && (sizeof(dns.sourceIPv6Address) == dns.rrdata.len))
    {
        memcpy(dns.sourceIPv6Address, dns.rrdata.buf,
               sizeof(dns.sourceIPv6Address));
        tid = ((tid & ~MD_DNS_OREC) | MD_DNS_AAAAREC);
    }

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }
        if (!mdExporterWriteDnsDedup(ctx->cfg, cnode->exp, tid,
                                     (uint8_t *)&dns, dns_len, err))
        {
            return FALSE;
        }
    }

  end:
    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}



/**
 * mdForwardDNSRR
 *
 * forward the dns rr-only record to the exporters
 * that are configured to receive them
 *
 */
gboolean
mdForwardDNSRR(
    mdContext_t            *ctx,
    const mdTmplContext_t  *tctx,
    fBuf_t                 *fbuf,
    GError                **err)
{
    md_dnsrr_t        dnsrr;
    size_t            rr_len = sizeof(dnsrr);
    md_export_node_t *cnode = NULL;

    if (!fBufSetInternalTemplate(fbuf, MD_DNSRR_TID, err)) {
        return FALSE;
    }

    if (!fBufNext(fbuf, (uint8_t *)&dnsrr, &rr_len, err)) {
        /* End of Set. */
        g_clear_error(err);
        goto end;
    }

    /* handle older records without nanoseconds */
    if (!tctx->attr.has_nano) {
        yfTime_t  t;
        yfTimeFromMilli(&t, dnsrr.flowStartMilliseconds);
        yfTimeToNTP(&dnsrr.flowStartNanoseconds, t);
    }

    mdSetCurrentTimeFromExport(ctx, fbuf);

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }
        if (!mdExporterWriteDNSRRRecord(ctx->cfg, cnode->exp, tctx->tid,
                                        tctx->attr.is_ipv6,
                                        (uint8_t *)&dnsrr, rr_len, err))
        {
            return FALSE;
        }
    }

  end:

    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}

/**
 * mdForwardDedupCustom
 *
 *
 */
gboolean
mdForwardDedupCustom(
    mdContext_t            *ctx,
    const mdTmplContext_t  *tctx,
    fBuf_t                 *fbuf,
    GError                **err)
{
    md_dedup_t        dedup;
    md_dedup_old_t    odedup;
    size_t            dedup_len = sizeof(dedup);
    size_t            odedup_len = sizeof(odedup);
    md_export_node_t *cnode = NULL;
    uint16_t          tid;

    if (!fBufSetInternalTemplate(fbuf, tctx->tid, err)) {
        return FALSE;
    }

    if (tctx->num_elem < 8) {
        if (!fBufNext(fbuf, (uint8_t *)&odedup, &odedup_len, err)) {
            /* End of Set. */
            g_clear_error(err);
            goto end;
        }
        dedup.monitoringIntervalStartMilliSeconds = odedup.fseen;
        dedup.monitoringIntervalEndMilliSeconds = odedup.lseen;
        dedup.flowStartMilliseconds = 0;
        dedup.observedDataTotalCount = odedup.count;
        memcpy(dedup.sourceIPv6Address, odedup.sip6, 16);
        dedup.sourceIPv4Address = odedup.sip;
        dedup.yafFlowKeyHash = odedup.hash;
        dedup.observedData.buf = odedup.data.buf;
        dedup.observedData.len = odedup.data.len;
        dedup.mapname.buf = NULL;
        dedup.mapname.len = 0;
    } else {
        if (!fBufNext(fbuf, (uint8_t *)&dedup, &dedup_len, err)) {
            /* End of Set. */
            g_clear_error(err);
            goto end;
        }
    }

    /* handle older records without nanoseconds */
    if (!tctx->attr.has_nano) {
        yfTime_t  t;
        yfTimeFromMilli(&t, dedup.flowStartMilliseconds);
        yfTimeToNTP(&dedup.flowStartNanoseconds, t);
    }

    mdSetCurrentTimeFromExport(ctx, fbuf);

    fBufGetCollectionTemplate(fbuf, &tid);

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }
        if (cnode->dedup) {
            if (!md_dedup_write_dedup(
                    ctx, cnode, &dedup, tctx->dedup_ie, err))
            {
                return FALSE;
            }
        } else {
            if (!mdExporterWriteDedupRecord(ctx->cfg, cnode, NULL,
                                            &dedup, "dedup", 0, tid, err))
            {
                return FALSE;
            }
        }
    }

  end:
    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}



/**
 * mdForwardDedup
 *
 */
gboolean
mdForwardDedup(
    mdContext_t            *ctx,
    const mdTmplContext_t  *tctx,
    fBuf_t                 *fbuf,
    GError                **err)
{
    md_dedup_t        dedup;
    size_t            dedup_len = sizeof(dedup);
    md_export_node_t *cnode = NULL;

    if (!fBufSetInternalTemplate(fbuf, MD_DEDUP_FULL, err)) {
        return FALSE;
    }

    if (!fBufNext(fbuf, (uint8_t *)&dedup, &dedup_len, err)) {
        /* End of Set. */
        g_clear_error(err);
        goto end;
    }

    /* handle older records without nanoseconds */
    if (!tctx->attr.has_nano) {
        yfTime_t  t;
        yfTimeFromMilli(&t, dedup.flowStartMilliseconds);
        yfTimeToNTP(&dedup.flowStartNanoseconds, t);
    }

    mdSetCurrentTimeFromExport(ctx, fbuf);

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }
        if (cnode->dedup) {
            /* ssl is the only way that should make it here */
            if (!md_dedup_write_dedup(ctx, cnode, &dedup, 244,
                                      err))
            {
                return FALSE;
            }
        } else {
            if (!mdExporterWriteDedupRecord(ctx->cfg, cnode, NULL, &dedup,
                                            "dedup", 0, tctx->tid, err))
            {
                return FALSE;
            }
        }
    }

  end:
    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}

/**
 * mdForwardSSLDedup
 *
 */
gboolean
mdForwardSSLDedup(
    mdContext_t            *ctx,
    const mdTmplContext_t  *tctx,
    fBuf_t                 *fbuf,
    GError                **err)
{
    md_ssl_dedup_t    ssl;
    size_t            ssl_len = sizeof(ssl);
    md_export_node_t *cnode = NULL;

    if (!fBufSetInternalTemplate(fbuf, MD_SSL_DEDUP_TID, err)) {
        return FALSE;
    }

    if (!fBufNext(fbuf, (uint8_t *)&ssl, &ssl_len, err)) {
        /* End of Set. */
        g_clear_error(err);
        goto end;
    }

    /* handle older records without nanoseconds */
    if (!tctx->attr.has_nano) {
        yfTime_t  t;
        yfTimeFromMilli(&t, ssl.flowStartMilliseconds);
        yfTimeToNTP(&ssl.flowStartNanoseconds, t);
        yfTimeFromNano(&t, ssl.flowEndMilliseconds);
        yfTimeToNTP(&ssl.flowEndNanoseconds, t);
    }

    mdSetCurrentTimeFromExport(ctx, fbuf);

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }
        if (!mdExporterWriteSSLDedupRecord(ctx->cfg, cnode->exp,
                                           MD_SSL_DEDUP_TID,
                                           (uint8_t *)&ssl, ssl_len, err))
        {
            return FALSE;
        }
    }

  end:

    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}

/**
 * mdForwardSSLCert
 *
 */
gboolean
mdForwardSSLCert(
    mdContext_t            *ctx,
    const mdTmplContext_t  *tctx,
    fBuf_t                 *fbuf,
    GError                **err)
{
    yaf_ssl_cert_t    cert;
    size_t            cert_len = sizeof(cert);
    md_export_node_t *cnode = NULL;

    MD_UNUSED_PARAM(tctx);

    if (!fBufSetInternalTemplate(fbuf, YAF_SSL_CERT_TID, err)) {
        return FALSE;
    }

    memset(&cert, 0, cert_len);

    if (!fBufNext(fbuf, (uint8_t *)&cert, &cert_len, err)) {
        /* End of Set. */
        g_clear_error(err);
        goto end;
    }

    mdSetCurrentTimeFromExport(ctx, fbuf);

    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        if (cnode->filter
            && FALSE == mdFilter(cnode->filter, NULL, ctx->cfg->current_domain,
                                 cnode->and_filter, ctx->cfg->collector_id))
        {
            continue;
        }

        md_ssl_export_ssl_cert(ctx, cnode, &cert, err);
    }

    mdCleanUpSSLCert(&cert);

  end:
    /* set internal template back to SiLK Flow */
    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}



/**
 * mdConvertToSiLK
 *
 * convert a normal YAF record to SiLK record
 * put TCP info back in the normal IPFIX record
 *
 * Tests for the presence of some IEs (with value) in the record and
 * sets flags for the external export template as needed.
 * For MPLS and NDPI, these values are transfered from the template ID
 * obtained during collection.
 *
 * @return the template id of the record
 */
uint32_t
mdConvertToSiLK(
    const mdFullFlow_t  *flow)
{
    uint32_t  wtid = YAF_SILK_FLOW_TID;

    if (flow->tmpl_attr.is_reverse) {
        wtid |= YTF_REV;
    }
    if (flow->tmpl_attr.is_ipv6) {
        wtid |= YTF_IP6;
    } else {
        wtid |= YTF_IP4;
    }

    if (flow->rec->protocolIdentifier == 6) {
        wtid |= YTF_TCP;
    }

    if (flow->tmpl_attr.has_interface) {
        wtid |= YTF_DAGIF;
    }
    if (flow->tmpl_attr.is_delta) {
        wtid |= YTF_DELTA;
    } else {
        wtid |= YTF_TOTAL;
    }

    if (flow->tmpl_attr.has_milli) {
        wtid |= YTF_MILLI;
    }
    if (flow->tmpl_attr.has_micro) {
        wtid |= YTF_MICRO;
    }
    if (flow->tmpl_attr.has_nano) {
        wtid |= YTF_NANO;
    }

    if (flow->tmpl_attr.has_ndpi) {
        wtid |= YTF_NDPI;
    }
    if (flow->tmpl_attr.has_mpls) {
        wtid |= YTF_MPLS;
    }

    return wtid;
}


/**
 * mdDecodeAndClear
 *
 * Do the STML decode and clean up
 *
 */
void
mdDecodeAndClear(
    mdContext_t   *ctx,
    mdFullFlow_t  *flow)
{
    mdMainDecode(ctx, flow);

    mdCleanUP(flow);

    fbSubTemplateMultiListClear(&flow->rec->subTemplateMultiList);
}

/**
 * mdForwardFlow
 *
 * Forward a normal flow record to the exporters
 * that are configured to receive it
 *
 */
gboolean
mdForwardFlow(
    mdContext_t   *ctx,
    mdFullFlow_t  *flow,
    GError       **err)
{
    gboolean          rc;
    int               wf = 0;
    md_export_node_t *cnode = NULL;

    flow->rec->yafFlowKeyHash = md_util_flow_key_hash(flow->rec);

    mdMainDecode(ctx, flow);
    if (!md_config.preserve_obdomain) {
        flow->rec->observationDomainId = ctx->cfg->current_domain;
    }

    /* copy collector name */
    flow->collector_name = ctx->cfg->collector_name;
    flow->collector_id = ctx->cfg->collector_id;
#if ENABLE_SKTYPESENSOR
    flow->collector_silk_probe_name = ctx->cfg->collector_silk_probe_name;
    flow->collector_silk_probe_vlan = ctx->cfg->collector_silk_probe_vlan;
#endif

    /* Iterate through the exporters and emit the flow */
    for (cnode = ctx->cfg->flowexit; cnode; cnode = cnode->next) {
        rc = TRUE;
        if (cnode->filter) {
            rc = mdFilter(cnode->filter, flow,
                          ctx->cfg->current_domain, cnode->and_filter,
                          ctx->cfg->collector_id);
        }
        if (rc) {
            if (cnode->dns_dedup && (flow->app_tid == YAF_DNS_TID)) {
                md_dns_dedup_add_node(ctx, cnode, flow);
            }
            if (cnode->ssl_dedup && ((flow->app_tid == YAF_SSL_TID)
                                     || flow->fullcert))
            {
                md_ssl_add_node(ctx, cnode, flow);
            }

            if (cnode->dedup) {
                if ((flow->app_tid == 0) && !flow->dhcpfpIndex &&
                    !flow->p0f)
                {
                    /* continue; */
                } else {
                    md_dedup_lookup_node(ctx, cnode, flow, err);
                }
                /* continue; */
            }

            /* Flow will be emitted from within mdExporterWriteFlow */
            wf = mdExporterWriteFlow(ctx->cfg, cnode->exp, flow, err);
            if (wf < 0) {
                rc = FALSE;
                goto done;
            }
        }

        if (cnode->dns_dedup || cnode->dedup) {
            /* only flush queue every 50 flows */
            if ((ctx->stats->recvd_flows % 50) == 0) {
                if (cnode->dns_dedup) {
                    if (!md_dns_dedup_flush_queue(cnode, ctx->cfg, err)) {
                        rc = FALSE;
                        goto done;
                    }
                }
                if (cnode->dedup) {
                    if (!md_dedup_flush_queue(cnode, ctx->cfg, err)) {
                        rc = FALSE;
                        goto done;
                    }
                }
            }
        }
    }

    rc = TRUE;

  done:
    mdCleanUP(flow);

    fbSubTemplateMultiListClear(&flow->rec->subTemplateMultiList);

    return rc;
}

/**
 * mdMainDecode
 *
 * loop through the STML and store the pointers
 * to specific records in the mdFullFlow for quick
 * retrieval later
 *
 */
static void
mdMainDecode(
    mdContext_t   *ctx,
    mdFullFlow_t  *flow)
{
    fbSubTemplateMultiListEntry_t *stml = NULL;
    unsigned int                   nextIndex = 0;

    if (flow->rec->packetTotalCount == 0) {
        flow->rec->packetTotalCount = flow->rec->packetDeltaCount;
        flow->rec->reversePacketTotalCount = flow->rec->reversePacketDeltaCount;
    }
    if (flow->rec->octetTotalCount == 0) {
        flow->rec->octetTotalCount = flow->rec->octetDeltaCount;
        flow->rec->reverseOctetTotalCount = flow->rec->reverseOctetDeltaCount;
    }

    while ((stml = (fbSubTemplateMultiListGetNextEntry(
                        &flow->rec->subTemplateMultiList, stml))))
    {
        /* if we need to store a handle to the STML Entry, use a value one
         * greater than its index in the STML (so zero is not-found) */
        ++nextIndex;
        switch ((stml->tmplID & YTF_BIF)) {
          case YAF_TCP_TID:
            /* TCP Data moves into the main record */
            {
                yaf_tcp_t *tcp = NULL;
                tcp = (yaf_tcp_t *)FBSTMLNEXT(stml, tcp);
                flow->rec->tcpSequenceNumber = tcp->tcpSequenceNumber;
                flow->rec->initialTCPFlags = tcp->initialTCPFlags;
                flow->rec->unionTCPFlags = tcp->unionTCPFlags;
                if ((stml->tmplID & YTF_REV)) {
                    flow->rec->reverseTcpSequenceNumber =
                        tcp->reverseTcpSequenceNumber;
                    flow->rec->reverseInitialTCPFlags =
                        tcp->reverseInitialTCPFlags;
                    flow->rec->reverseUnionTCPFlags =
                        tcp->reverseUnionTCPFlags;
                }
                break;
            }
          case YAF_ENTROPY_TID:
            flow->entropy =
                (yaf_entropy_t *)FBSTMLNEXT(stml, flow->entropy);
            break;
          case YAF_FLOWSTATS_TID:
            flow->stats = (yaf_flowstats_t *)FBSTMLNEXT(stml, flow->stats);
            break;
          case YAF_MAC_TID:
            flow->mac = (yaf_mac_t *)FBSTMLNEXT(stml, flow->mac);
            break;
          case YAF_PAYLOAD_TID:
            flow->pay = (yaf_payload_t *)FBSTMLNEXT(stml, flow->pay);
            break;
          case YAF_P0F_TID:
            flow->p0f = (yaf_p0f_t *)FBSTMLNEXT(stml, flow->p0f);
            break;
          case YAF_FPEXPORT_TID:
            flow->fp = (yaf_fpexport_t *)FBSTMLNEXT(stml, flow->fp);
            break;
          case YAF_MPTCP_TID:
            flow->mptcp = (yaf_mptcp_t *)FBSTMLNEXT(stml, flow->mptcp);
            break;
          case MD_IPSET_REC_TID:
            flow->ipsetrecIndex = nextIndex;
            break;
          case MD_PMAP_REC_TID:
            flow->pmaprecIndex = nextIndex;
            break;
          case MD_SILK_TYPE_SENSOR_TID:
            flow->silkTypeSensorIndex = nextIndex;
            break;
          case YAF_HTTP_TID:
          case YAF_FTP_TID:
          case YAF_SIP_TID:
          case YAF_SMTP_211_TID:
          case YAF_SSH_214_TID:
          case YAF_NNTP_TID:
          case YAF_IMAP_216_TID:
          case YAF_RTSP_TID:
            flow->app = (void *)FBSTMLNEXT(stml, flow->app);
            flow->app_tid = stml->tmplID;
            flow->app_elements = fbTemplateCountElements(stml->tmpl);
            break;
          case YAF_IMAP_TID:
          case YAF_IRC_TID:
          case YAF_TFTP_TID:
          case YAF_SLP_TID:
          case YAF_MYSQL_TID:
          case YAF_DNP_TID:
          case YAF_MODBUS_TID:
          case YAF_ENIP_TID:
          case YAF_RTP_TID:
            flow->app = (void *)FBSTMLNEXT(stml, flow->app);
            flow->app_tid = stml->tmplID;
            break;
          case YAF_POP3_TID:
            {
                /* When transcoding incoming POP3 data from a YAF before
                 * Mid-Encryption was added, the STL is left with a NULL
                 * template and 0 template ID.  Set them here if needed. */
                yaf_pop3_t *pop3;
                flow->app = (void *)FBSTMLNEXT(stml, flow->app);
                flow->app_tid = stml->tmplID;
                pop3 = (yaf_pop3_t *)flow->app;
                if (0 == fbSubTemplateListGetTemplateID(&pop3->sslCertList)) {
                    fbSession_t  *s = fBufGetSession(ctx->cfg->flowsrc->fbuf);
                    fbTemplate_t *t = fbSessionGetTemplate(
                        s, TRUE, YAF_SSL_TID, NULL);
                    fbSubTemplateListInit(
                        &pop3->sslCertList, 3, YAF_SSL_TID, t, 0);
                }
            }
            break;
          case YAF_SSH_TID:
            flow->app = (void *)FBSTMLNEXT(stml, flow->app);
            flow->app_tid = stml->tmplID;
            break;
          case YAF_SSL_TID:
            flow->app = (void *)FBSTMLNEXT(stml, flow->app);
            flow->app_tid = stml->tmplID;
            /* store value one greater than index of this STML Entry */
            flow->certIndex = nextIndex;
            break;
          case YAF_SSL_Y22_TID:
            /* SSL from YAF 2.2.0. STML contained one or two SSL entries:
             * YAF_SSL_Y22_TID contained connection info; YAF_SSL_Y22_CERT_TID
             * contained the certificate info and was not always present */
            flow->app = (void *)FBSTMLNEXT(stml, flow->app);
            flow->app_tid = stml->tmplID;
            break;
          case YAF_SSL_Y22_CERT_TID:
            /* store value one greater than index of this STML Entry */
            flow->certIndex = nextIndex;
            break;
          case YAF_SMTP_TID:
            {
                /* When transcoding incoming SMTP data from a YAF before
                 * Mid-Encryption was added, the STL is left with a NULL
                 * template and 0 template ID.  Set them here if needed. */
                yaf_smtp_t *smtp;
                flow->app = (void *)FBSTMLNEXT(stml, flow->app);
                flow->app_tid = stml->tmplID;
                smtp = (yaf_smtp_t *)flow->app;
                if (0 == fbSubTemplateListGetTemplateID(&smtp->sslCertList)) {
                    fbSession_t  *s = fBufGetSession(ctx->cfg->flowsrc->fbuf);
                    fbTemplate_t *t = fbSessionGetTemplate(
                        s, TRUE, YAF_SSL_TID, NULL);
                    fbSubTemplateListInit(
                        &smtp->sslCertList, 3, YAF_SSL_TID, t, 0);
                }
                flow->app_tid = stml->tmplID;
            }
            break;
          case YAF_DNS_TID:
            {
                yaf_dns_t   *dnsflow = NULL;
                yaf_dnsQR_t *dnsqrflow = NULL;
                flow->app = (void *)FBSTMLNEXT(stml, flow->app);
                flow->app_tid = stml->tmplID;
                dnsflow = (yaf_dns_t *)flow->app;
                dnsqrflow = FBSTLNEXT(&(dnsflow->dnsQRList), dnsqrflow);
                if (dnsqrflow) {
                    if (dnsqrflow->dnsQueryResponse || dnsqrflow->dnsNXDomain) {
                        /* if the first one is a response & it's UDP_FORCE,
                         * reversify flow key hash */
                        if (flow->rec->flowEndReason == YAF_END_UDPFORCE) {
                            flow->rec->yafFlowKeyHash =
                                md_util_rev_flow_key_hash(flow->rec);
                        }
                    }
                    ctx->stats->dns++;
                }
                if (flow->rec->flowEndReason == YAF_END_UDPFORCE) {
                    ctx->stats->uniflows++;
                }
            }
            break;
          case YAF_DHCP_FP_TID:
          case YAF_DHCP_OPTIONS_TID:
            /* store value one greater than index of this STML Entry */
            flow->dhcpfpIndex = nextIndex;
            flow->app_tid = stml->tmplID;
            break;
          case YAF_SSL_BINARY_CERT_TID:
            {
                int           i = 0;
                fbVarfield_t *ct = NULL;

                if (!sm_sub_ssl_tmpl) {
                    fbInfoModel_t *model = mdInfoModel();
                    sm_sub_ssl_tmpl = fbTemplateAlloc(model);
                    if (!fbTemplateAppendSpecArray(
                            sm_sub_ssl_tmpl, yaf_ssl_subcert_spec,
                            0xffffffff, &(ctx->err)))
                    {
                        g_warning(
                            "error creating template for ssl cert decode %s",
                            ctx->err->message);
                        break;
                    }
                }

                flow->fullcert =
                    (yaf_ssl_binary_cert_t *)FBSTMLNEXT(stml, NULL);
                flow->sslcerts =
                    g_new0(yaf_ssl_cert_t *,
                           (flow->fullcert->certBL.numElements + 1));
                if (flow->app_tid == 0) {
                    flow->app_tid = stml->tmplID;
                }
                while ((ct = ((fbVarfield_t *)fbBasicListGetNextPtr(
                                  &(flow->fullcert->certBL), ct))))
                {
                    flow->sslcerts[i] =
                        md_ssl_cert_decode(ct->buf, ct->len, sm_sub_ssl_tmpl);
                    i++;
                }
                break;
            }
          case 0:
            /* No internal template for the STML-Entry, perhaps because
             * fbSessionAddTemplatePair() said to ignore it */
            break;
          default:
            g_debug("Received Unknown Template ID %#06x in STML;"
                    " Parent Template ID is %#0x6",
                    stml->tmplID, flow->tid);
            break;
        }
    }
}

static void
Mid_Encrption_CleanUp(
    yaf_ssl_t  *rec)
{
    if (fbSubTemplateListGetTemplateID(&rec->sslCertList)
        == MD_SSL_CERTIFICATE_TID)
    {
        md_ssl_certificate_t *cert = NULL;
        while ((cert = fbSubTemplateListGetNextPtr(&rec->sslCertList, cert))) {
            fbBasicListClear(&cert->sslCertIssuerCommonName);
            fbBasicListClear(&cert->sslCertIssuerStreetAddress);
            fbBasicListClear(&cert->sslCertIssuerOrgName);
            fbBasicListClear(&cert->sslCertIssuerOrgUnitName);
            fbBasicListClear(&cert->sslCertIssuerDomainComponent);
            fbBasicListClear(&cert->sslCertSubCommonName);
            fbBasicListClear(&cert->sslCertSubStreetAddress);
            fbBasicListClear(&cert->sslCertSubOrgName);
            fbBasicListClear(&cert->sslCertSubOrgUnitName);
            fbBasicListClear(&cert->sslCertSubDomainComponent);
        }
    } else {
        yaf_ssl_cert_t *cert = NULL;
        while ((cert = fbSubTemplateListGetNextPtr(&rec->sslCertList, cert))) {
            fbSubTemplateListClear(&(cert->issuer));
            fbSubTemplateListClear(&(cert->subject));
            fbSubTemplateListClear(&(cert->extension));
        }
    }

    fbSubTemplateListClear(&(rec->sslCertList));
    fbBasicListClear(&(rec->sslCipherList));
}


/**
 * mdCleanUP
 *
 * clean up after the messy dynamic lists...
 *
 */
static void
mdCleanUP(
    mdFullFlow_t  *flow)
{
    fbBasicList_t *bl = NULL;
    int            loop;

    switch (flow->app_tid & YTF_BIF) {
      case YAF_HTTP_TID:
      case YAF_SSH_214_TID:
      case YAF_RTSP_TID:
      case YAF_SMTP_211_TID:
      case YAF_SIP_TID:
      case YAF_FTP_TID:
      case YAF_IMAP_216_TID:
      case YAF_NNTP_TID:
        bl = (fbBasicList_t *)flow->app;
        for (loop = 0; loop < flow->app_elements; loop++) {
            fbBasicListClear(bl);
            bl++;
        }
        break;
      case YAF_IMAP_TID:
        {
            yaf_imap_t *rec = (yaf_imap_t *)flow->app;
            fbBasicListClear(&rec->imapCapability);
            fbBasicListClear(&rec->imapLogin);
            fbBasicListClear(&rec->imapStartTLS);
            fbBasicListClear(&rec->imapAuthenticate);
            fbBasicListClear(&rec->imapCommand);
            fbBasicListClear(&rec->imapExists);
            fbBasicListClear(&rec->imapRecent);

            if (fbSubTemplateListGetDataPtr(&rec->sslCertList) != NULL) {
                Mid_Encrption_CleanUp(
                    fbSubTemplateListGetDataPtr(&rec->sslCertList));
            }
            fbSubTemplateListClear(&rec->sslCertList);
            break;
        }
      case YAF_POP3_TID:
        {
            yaf_pop3_t *rec = (yaf_pop3_t *)flow->app;
            fbBasicListClear(&(rec->pop3msg));
            if (fbSubTemplateListGetDataPtr(&rec->sslCertList) != NULL) {
                Mid_Encrption_CleanUp(
                    fbSubTemplateListGetDataPtr(&rec->sslCertList));
            }
            fbSubTemplateListClear(&rec->sslCertList);
            break;
        }
      case YAF_MODBUS_TID:
        {
            yaf_modbus_t *rec = (yaf_modbus_t *)flow->app;
            fbBasicListClear(&(rec->mbmsg));
            break;
        }
      case YAF_ENIP_TID:
        {
            yaf_enip_t *rec = (yaf_enip_t *)flow->app;
            fbBasicListClear(&(rec->enipmsg));
            break;
        }
      case YAF_IRC_TID:
        {
            yaf_irc_t *rec = (yaf_irc_t *)flow->app;
            fbBasicListClear(&(rec->ircMsg));
            break;
        }
      case YAF_SLP_TID:
        {
            yaf_slp_t *rec = (yaf_slp_t *)flow->app;
            fbBasicListClear(&(rec->slpString));
            break;
        }
      case YAF_SSL_Y22_TID:
        {
            yaf_ssl_y22_t *rec = (yaf_ssl_y22_t *)flow->app;
            fbBasicListClear(&(rec->sslCipherList));
            break;
        }
      case YAF_SSL_TID:
        {
            yaf_ssl_t *rec = (yaf_ssl_t *)flow->app;
            Mid_Encrption_CleanUp(rec);
        }
        break;
      case YAF_MYSQL_TID:
        {
            yaf_mysql_t *rec = (yaf_mysql_t *)flow->app;
            fbSubTemplateListClear(&(rec->mysqlList));
            break;
        }
      case YAF_SMTP_TID:
        {
            yaf_smtp_t         *smtp = (yaf_smtp_t *)flow->app;
            yaf_smtp_message_t *smtp_msg = NULL;
            fbBasicListClear(&smtp->smtpFailedCodes);
            while ((smtp_msg = ((yaf_smtp_message_t *)FBSTLNEXT(
                                    &smtp->smtpMessageList, smtp_msg))))
            {
                fbBasicListClear(&smtp_msg->smtpToList);
                fbBasicListClear(&smtp_msg->smtpFromList);
                fbBasicListClear(&smtp_msg->smtpFilenameList);
                fbBasicListClear(&smtp_msg->smtpURLList);
                fbSubTemplateListClear(&smtp_msg->smtpHeaderList);
            }
            fbSubTemplateListClear(&smtp->smtpMessageList);
            if (fbSubTemplateListGetDataPtr(&smtp->sslCertList) != NULL) {
                Mid_Encrption_CleanUp(
                    fbSubTemplateListGetDataPtr(&smtp->sslCertList));
            }
            fbSubTemplateListClear(&smtp->sslCertList);
            break;
        }
      case YAF_DNS_TID:
        {
            yaf_dns_t   *rec = (yaf_dns_t *)flow->app;
            yaf_dnsQR_t *dns = NULL;
            while ((dns = fbSubTemplateListGetNextPtr(&(rec->dnsQRList),
                                                      dns)))
            {
                fbSubTemplateListClear(&(dns->dnsRRList));
            }

            fbSubTemplateListClear(&(rec->dnsQRList));
            break;
        }
      case YAF_DHCP_OPTIONS_TID:
        {
            fbSubTemplateMultiListEntry_t *entry;
            yaf_dhcp_options_t            *dhcp = NULL;
            entry = fbSubTemplateMultiListGetIndexedEntry(
                &flow->rec->subTemplateMultiList, flow->dhcpfpIndex - 1);
            dhcp = (yaf_dhcp_options_t *)FBSTMLNEXT(entry, dhcp);
            fbBasicListClear(&(dhcp->options));
            if (flow->app_tid & YTF_REV) {
                fbBasicListClear(&(dhcp->revOptions));
            }
            break;
        }
      default:
        break;
    }

    if (flow->fullcert) {
        yaf_ssl_cert_t *cert;
        int             i = 0;
        while ((cert = flow->sslcerts[i])) {
            fbSubTemplateListClear(&(cert->issuer));
            fbSubTemplateListClear(&(cert->subject));
            fbSubTemplateListClear(&(cert->extension));
            g_slice_free(yaf_ssl_cert_t, cert);
            i++;
        }
        g_free(flow->sslcerts);
        fbBasicListClear(&(flow->fullcert->certBL));
    }
}

static void
mdCleanUpSSLCert(
    yaf_ssl_cert_t  *cert)
{
    fbSubTemplateListClear(&(cert->issuer));
    fbSubTemplateListClear(&(cert->subject));
    fbSubTemplateListClear(&(cert->extension));
}


/**
 * mdNewFieldList
 *
 */
static mdFieldList_t *
mdNewFieldList(
    void)
{
    return g_slice_new0(mdFieldList_t);
}

/**
 * mdCreateFieldList
 *
 * add custom field to field list
 *
 */
mdFieldList_t *
mdCreateFieldList(
    mdAcceptFilterField_t   field,
    gboolean                internal)
{
    mdFieldList_t *item = NULL;

    if (!internal && field >= FIELDS_INTERNAL_ONLY) {
        return NULL;
    }

    item = mdNewFieldList();

    item->field = field;

    switch (field) {
      case FLOWKEYHASH:
        item->print_fn = mdPrintFlowKeyHash;
        break;
      case SIP_ANY:
        item->print_fn = mdPrintSIPText;
        break;
      case DIP_ANY:
        item->print_fn = mdPrintDIPText;
        break;
      case SIP_INT:
        item->print_fn = mdPrintSIPINT;
        break;
      case DIP_INT:
        item->print_fn = mdPrintDIPINT;
        break;
      case STIME_EPOCH:
        item->print_fn = mdPrintSTimeEpoch;
        break;
      case ETIME_EPOCH:
        item->print_fn = mdPrintETimeEpoch;
        break;
      case STIME_EPOCH_MS:
        item->print_fn = mdPrintSTimeEpochMS;
        break;
      case ETIME_EPOCH_MS:
        item->print_fn = mdPrintETimeEpochMS;
        break;
      case STIME_PLACEHOLDER:
      case ETIME_PLACEHOLDER:
      case DURATION_PLACEHOLDER:
      case RTT_PLACEHOLDER:
        /* these will get replaced */
        item->print_fn = mdPrintNone;
        break;
      case STIME_BEST:
        item->print_fn = mdPrintSTimeBest;
        break;
      case ETIME_BEST:
        item->print_fn = mdPrintETimeBest;
        break;
      case DURATION_BEST:
        item->print_fn = mdPrintDurationBest;
        break;
      case RTT_BEST:
        item->print_fn = mdPrintRTTBest;
        break;
      case STIME_INCOMING:
        item->print_fn = mdPrintSTimeIncomingText;
        break;
      case ETIME_INCOMING:
        item->print_fn = mdPrintETimeIncomingText;
        break;
      case DURATION_INCOMING:
        item->print_fn = mdPrintDurationIncomingText;
        break;
      case RTT_INCOMING:
        item->print_fn = mdPrintRTTIncomingText;
        break;
      case STIME_MICRO:
        item->print_fn = mdPrintSTimeMicro;
        break;
      case ETIME_MICRO:
        item->print_fn = mdPrintETimeMicro;
        break;
      case DURATION_MICRO:
        item->print_fn = mdPrintDurationMicro;
        break;
      case RTT_MICRO:
        item->print_fn = mdPrintRTTMicro;
        break;
      case STIME_MILLI:
        item->print_fn = mdPrintSTimeMilli;
        break;
      case ETIME_MILLI:
        item->print_fn = mdPrintETimeMilli;
        break;
      case DURATION_MILLI:
        item->print_fn = mdPrintDurationMilli;
        break;
      case RTT_MILLI:
        item->print_fn = mdPrintRTTMilli;
        break;
      case STIME_NANO:
        item->print_fn = mdPrintSTimeNano;
        break;
      case ETIME_NANO:
        item->print_fn = mdPrintETimeNano;
        break;
      case DURATION_NANO:
        item->print_fn = mdPrintDurationNano;
        break;
      case RTT_NANO:
        item->print_fn = mdPrintRTTNano;
        break;
      case STIME_NOFRAC:
        item->print_fn = mdPrintSTimeNoFrac;
        break;
      case ETIME_NOFRAC:
        item->print_fn = mdPrintETimeNoFrac;
        break;
      case DURATION_NOFRAC:
        item->print_fn = mdPrintDurationNoFrac;
        break;
      case RTT_NOFRAC:
        item->print_fn = mdPrintRTTNoFrac;
        break;
      case SPORT:
        item->print_fn = mdPrintSPort;
        break;
      case DPORT:
        item->print_fn = mdPrintDPort;
        break;
      case PROTOCOL:
        item->print_fn = mdPrintProto;
        break;
      case APPLICATION:
        item->print_fn = mdPrintAppLabel;
        break;
      case OBDOMAIN:
        item->print_fn = mdPrintOBDomain;
        break;
      case VLAN:
        item->print_fn = mdPrintVLAN;
        break;
      case VLANINT:
        item->print_fn = mdPrintVLANINT;
        break;
      case PKTS:
        item->print_fn = mdPrintPackets;
        break;
      case RPKTS:
        item->print_fn = mdPrintPacketsRev;
        break;
      case BYTES:
        item->print_fn = mdPrintBytes;
        break;
      case RBYTES:
        item->print_fn = mdPrintBytesRev;
        break;
      case IFLAGS:
        item->print_fn = mdPrintInitFlags;
        break;
      case RIFLAGS:
        item->print_fn = mdPrintInitFlagsRev;
        break;
      case UFLAGS:
        item->print_fn = mdPrintUnionFlags;
        break;
      case RUFLAGS:
        item->print_fn = mdPrintUnionFlagsRev;
        break;
      case ATTRIBUTES:
        item->print_fn = mdPrintAttributes;
        break;
      case RATTRIBUTES:
        item->print_fn = mdPrintAttributesRev;
        break;
      case MAC:
        item->print_fn = mdPrintMAC;
        break;
      case DSTMAC:
        item->print_fn = mdPrintDSTMAC;
        break;
      case TCPSEQ:
        item->print_fn = mdPrintTCPSeq;
        break;
      case RTCPSEQ:
        item->print_fn = mdPrintTCPSeqRev;
        break;
      case ENTROPY:
        item->print_fn = mdPrintEntropy;
        break;
      case RENTROPY:
        item->print_fn = mdPrintEntropyRev;
        break;
      case ENDREASON:
        item->print_fn = mdPrintEndReason;
        break;
      case DHCPFP:
        item->print_fn = mdPrintDHCPFP;
        break;
      case RDHCPFP:
        item->print_fn = mdPrintDHCPFPRev;
        break;
      case DHCPVC:
        item->print_fn = mdPrintDHCPVC;
        break;
      case RDHCPVC:
        item->print_fn = mdPrintDHCPVCRev;
        break;
      case OSNAME:
        item->print_fn = mdPrintOSNAME;
        break;
      case OSVERSION:
        item->print_fn = mdPrintOSVersion;
        break;
      case ROSNAME:
        item->print_fn = mdPrintOSNAMERev;
        break;
      case ROSVERSION:
        item->print_fn = mdPrintOSVersionRev;
        break;
      case FINGERPRINT:
        item->print_fn = mdPrintOSFingerprint;
        break;
      case RFINGERPRINT:
        item->print_fn = mdPrintOSFingerprintRev;
        break;
      case INGRESS:
        item->print_fn = mdPrintIngress;
        break;
      case EGRESS:
        item->print_fn = mdPrintEgress;
        break;
      case DATABYTES:
        item->print_fn = mdPrintDataBytes;
        break;
      case RDATABYTES:
        item->print_fn = mdPrintDataBytesRev;
        break;
      case ITIME:
        item->print_fn = mdPrintInterArriveTime;
        break;
      case RITIME:
        item->print_fn = mdPrintInterArriveTimeRev;
        break;
      case STDITIME:
        item->print_fn = mdPrintSTDITime;
        break;
      case RSTDITIME:
        item->print_fn = mdPrintSTDITimeRev;
        break;
      case TCPURG:
        item->print_fn = mdPrintTCPURG;
        break;
      case RTCPURG:
        item->print_fn = mdPrintTCPURGRev;
        break;
      case SMALLPKTS:
        item->print_fn = mdPrintSmallPkts;
        break;
      case RSMALLPKTS:
        item->print_fn = mdPrintSmallPktsRev;
        break;
      case LARGEPKTS:
        item->print_fn = mdPrintLargePkts;
        break;
      case RLARGEPKTS:
        item->print_fn = mdPrintLargePktsRev;
        break;
      case NONEMPTYPKTS:
        item->print_fn = mdPrintNonEmptyPkts;
        break;
      case RNONEMPTYPKTS:
        item->print_fn = mdPrintNonEmptyPktsRev;
        break;
      case MAXSIZE:
        item->print_fn = mdPrintMaxPacketSize;
        break;
      case RMAXSIZE:
        item->print_fn = mdPrintMaxPacketSizeRev;
        break;
      case STDPAYLEN:
        item->print_fn = mdPrintSTDPayLen;
        break;
      case RSTDPAYLEN:
        item->print_fn = mdPrintSTDPayLenRev;
        break;
      case FIRSTEIGHT:
        item->print_fn = mdPrintFirstEight;
        break;
      case TOS:
        item->print_fn = mdPrintTOS;
        break;
      case RTOS:
        item->print_fn = mdPrintTOSRev;
        break;
      case MPLS1:
        item->print_fn = mdPrintMPLS1;
        break;
      case MPLS2:
        item->print_fn = mdPrintMPLS2;
        break;
      case MPLS3:
        item->print_fn = mdPrintMPLS3;
        break;
      case COLLECTOR:
        item->print_fn = mdPrintCollectorName;
        break;
      case FIRSTNONEMPTY:
        item->print_fn = mdPrintFirstNonEmpty;
        break;
      case RFIRSTNONEMPTY:
        item->print_fn = mdPrintFirstNonEmptyRev;
        break;
      case MPTCPSEQ:
        item->print_fn = mdPrintMPTCPSeq;
        break;
      case MPTCPTOKEN:
        item->print_fn = mdPrintMPTCPToken;
        break;
      case MPTCPMSS:
        item->print_fn = mdPrintMPTCPMss;
        break;
      case MPTCPID:
        item->print_fn = mdPrintMPTCPId;
        break;
      case MPTCPFLAGS:
        item->print_fn = mdPrintMPTCPFlags;
        break;
      case NONE_FIELD:
        item->print_fn = mdPrintNone;
        break;
      case PAYLOAD:
        item->print_fn = mdPrintPayloadText;
        break;
      case RPAYLOAD:
        item->print_fn = mdPrintPayloadTextRev;
        break;
      case DHCPOPTIONS:
        item->print_fn = mdPrintDHCPOptions;
        break;
      case RDHCPOPTIONS:
        item->print_fn = mdPrintDHCPOptionsRev;
        break;
      case NDPI_MASTER:
        item->print_fn = mdPrintNDPIMaster;
        break;
      case NDPI_SUB:
        item->print_fn = mdPrintNDPISub;
        break;
      case FIELDS_INTERNAL_ONLY:
      case ANY_IP6:
      case ANY_IP:
      case ANY_PORT:
      case DIP_V4:
      case DIP_V6:
      case DPI:
      case IPVERSION:
      case SIP_V4:
      case SIP_V6:
        g_slice_free(mdFieldList_t, item);
        return NULL;
    }

    if (NULL == item->print_fn) {
        g_slice_free(mdFieldList_t, item);
        return NULL;
    }

    return item;
}


/**
 * mdSetFieldListDecoratorCustom
 *
 * create custom printer for CSV
 *
 */
void
mdSetFieldListDecoratorCustom(
    mdFieldList_t  *list,
    char            delimiter)
{
    mdFieldList_t *item = NULL;

    for (item = list; item; item = item->next) {
        if (item->decorator) {
            /* decorator already set */
            return;
        }
        switch (item->field) {
          case NONE_FIELD:
          case PAYLOAD:
          case RPAYLOAD:
            item->decorator = g_string_new("");
            break;
          case BYTES:
          case DATABYTES:
          case ETIME_EPOCH_MS:
          case MPTCPSEQ:
          case PKTS:
          case RBYTES:
          case RDATABYTES:
          case RPKTS:
          case STIME_EPOCH_MS:
            item->decorator = g_string_new("%" PRIu64);
            break;
          case DURATION_NOFRAC:
          case RTT_NOFRAC:
            item->decorator = g_string_new("%.0f");
            break;
          case DURATION_MILLI:
          case ITIME:
          case RITIME:
          case RSTDITIME:
          case RTT_MILLI:
          case STDITIME:
            item->decorator = g_string_new("%.3f");
            break;
          case DURATION_MICRO:
          case RTT_MICRO:
            item->decorator = g_string_new("%.6f");
            break;
          case DURATION_NANO:
          case RTT_NANO:
            item->decorator = g_string_new("%.9f");
            break;
          case ATTRIBUTES:
          case FIRSTEIGHT:
          case MPTCPFLAGS:
          case RATTRIBUTES:
          case RTOS:
          case TOS:
            item->decorator = g_string_new("%02x");
            break;
          case VLAN:
            item->decorator = g_string_new("%03x");
            break;
          case RTCPSEQ:
          case TCPSEQ:
            item->decorator = g_string_new("%08x");
            break;
          case APPLICATION:
          case DPORT:
          case FIRSTNONEMPTY:
          case MAXSIZE:
          case MPLS1:
          case MPLS2:
          case MPLS3:
          case MPTCPID:
          case MPTCPMSS:
          case NDPI_MASTER:
          case NDPI_SUB:
          case PROTOCOL:
          case RFIRSTNONEMPTY:
          case RMAXSIZE:
          case RSTDPAYLEN:
          case SPORT:
          case STDPAYLEN:
            item->decorator = g_string_new("%d");
            break;
          case COLLECTOR:
          case DHCPFP:
          case DHCPOPTIONS:
          case DHCPVC:
          case DIP_ANY:
          case DSTMAC:
          case DURATION_BEST:
          case DURATION_INCOMING:
          case ENDREASON:
          case ETIME_BEST:
          case ETIME_INCOMING:
          case ETIME_EPOCH:
          case ETIME_MICRO:
          case ETIME_MILLI:
          case ETIME_NANO:
          case ETIME_NOFRAC:
          case FINGERPRINT:
          case IFLAGS:
          case MAC:
          case OSNAME:
          case OSVERSION:
          case RDHCPFP:
          case RDHCPOPTIONS:
          case RDHCPVC:
          case RFINGERPRINT:
          case RIFLAGS:
          case ROSNAME:
          case ROSVERSION:
          case RTT_BEST:
          case RTT_INCOMING:
          case RUFLAGS:
          case SIP_ANY:
          case STIME_BEST:
          case STIME_INCOMING:
          case STIME_EPOCH:
          case STIME_MICRO:
          case STIME_MILLI:
          case STIME_NANO:
          case STIME_NOFRAC:
          case UFLAGS:
            item->decorator = g_string_new("%s");
            break;
          case DIP_INT:
          case EGRESS:
          case ENTROPY:
          case FLOWKEYHASH:
          case INGRESS:
          case LARGEPKTS:
          case MPTCPTOKEN:
          case NONEMPTYPKTS:
          case OBDOMAIN:
          case RENTROPY:
          case RLARGEPKTS:
          case RNONEMPTYPKTS:
          case RSMALLPKTS:
          case RTCPURG:
          case SIP_INT:
          case SMALLPKTS:
          case TCPURG:
          case VLANINT:
            item->decorator = g_string_new("%u");
            break;
          case ANY_IP6:
          case ANY_IP:
          case ANY_PORT:
          case DIP_V4:
          case DIP_V6:
          case DPI:
          case DURATION_PLACEHOLDER:
          case ETIME_PLACEHOLDER:
          case FIELDS_INTERNAL_ONLY:
          case IPVERSION:
          case RTT_PLACEHOLDER:
          case SIP_V4:
          case SIP_V6:
          case STIME_PLACEHOLDER:
            g_warning("Unexpected value %u seen when setting TEXT decorator",
                      item->field);
            return;
        }

        if (NULL == item->decorator) {
            g_warning("Value %u did not set a TEXT decorator", item->field);
            return;
        }

        g_string_append_c(item->decorator, delimiter);
    }
}

/**
 * mdSetFieldListDecoratorJSON
 *
 * create the custom printer for JSON
 *
 */
void
mdSetFieldListDecoratorJSON(
    mdFieldList_t  *list)
{
    mdFieldList_t *item = NULL;

    /* Wrap the `key_` in quotes, add a colon and final comma */
#define ITEM_DECOR(key_, decorator_)                                    \
    item->decorator = g_string_new("\"" key_ "\":" decorator_ ",")

    for (item = list; item; item = item->next) {
        if (item->decorator) {
            /* decorator already set */
            return;
        }
        switch (item->field) {
          case FLOWKEYHASH:
            ITEM_DECOR("yafFlowKeyHash", "%u");
            break;
          case SIP_ANY:
            /* the function adds the key(s), value(s), and final comma */
            item->print_fn = mdPrintSIPJson;
            item->decorator = g_string_new("%s");
            break;
          case DIP_ANY:
            /* the function adds the key(s), value(s), and final comma */
            item->print_fn = mdPrintDIPJson;
            item->decorator = g_string_new("%s");
            break;
          case SIP_INT:
            /* always v4; do not have int128 support */
            ITEM_DECOR("sourceIPv4Address", "%u");
            break;
          case DIP_INT:
            /* always v4; do not have int128 support */
            ITEM_DECOR("destinationIPv4Address", "%u");
            break;
          case STIME_EPOCH_MS:
            ITEM_DECOR("flowStartMilliseconds", "\"%" PRIu64 "\"");
            break;
          case ETIME_EPOCH_MS:
            ITEM_DECOR("flowEndMilliseconds", "\"%" PRIu64 "\"");
            break;
          case STIME_EPOCH:
            /* the function fills the key and value */
            ITEM_DECOR("%s", "\"%s\"");
            break;
          case ETIME_EPOCH:
            /* the function adds the key and value */
            ITEM_DECOR("%s", "\"%s\"");
            break;
          case SPORT:
            ITEM_DECOR("sourceTransportPort", "%d");
            break;
          case DPORT:
            ITEM_DECOR("destinationTransportPort", "%d");
            break;
          case PROTOCOL:
            ITEM_DECOR("protocolIdentifier", "%d");
            break;
          case APPLICATION:
            ITEM_DECOR("silkAppLabel", "%d");
            break;
          case OBDOMAIN:
            ITEM_DECOR("observationDomainId", "%u");
            break;
          case VLAN:
            ITEM_DECOR("vlanId", "\"0x%03x\"");
            break;
          case VLANINT:
            ITEM_DECOR("vlanId", "%u");
            break;
          case DURATION_MILLI:
            ITEM_DECOR("flowDurationMilliseconds", "%.3f");
            break;
          case STIME_MILLI:
            ITEM_DECOR("flowStartMilliseconds", "\"%s\"");
            break;
          case ETIME_MILLI:
            ITEM_DECOR("flowEndMilliseconds", "\"%s\"");
            break;
          case RTT_MILLI:
            ITEM_DECOR("reverseFlowDeltaMilliseconds", "%.3f");
            break;
          case DURATION_MICRO:
            ITEM_DECOR("flowDurationMicroseconds", "%.6f");
            break;
          case STIME_MICRO:
            ITEM_DECOR("flowStartMicroseconds", "\"%s\"");
            break;
          case ETIME_MICRO:
            ITEM_DECOR("flowEndMicroseconds", "\"%s\"");
            break;
          case RTT_MICRO:
            ITEM_DECOR("reverseFlowDeltaMicroseconds", "%.6f");
            break;
          case DURATION_NANO:
            ITEM_DECOR("flowDurationNanoseconds", "%.9f");
            break;
          case STIME_NANO:
            ITEM_DECOR("flowStartNanoseconds", "\"%s\"");
            break;
          case ETIME_NANO:
            ITEM_DECOR("flowEndNanoseconds", "\"%s\"");
            break;
          case RTT_NANO:
            ITEM_DECOR("reverseFlowDeltaNanoseconds", "%.9f");
            break;
          case DURATION_NOFRAC:
            ITEM_DECOR("flowDurationSeconds", "%.0f");
            break;
          case STIME_NOFRAC:
            ITEM_DECOR("flowStartSeconds", "\"%s\"");
            break;
          case ETIME_NOFRAC:
            ITEM_DECOR("flowEndSeconds", "\"%s\"");
            break;
          case RTT_NOFRAC:
            ITEM_DECOR("reverseFlowDeltaSeconds", "%.0f");
            break;
          case DURATION_BEST:
            /* the function fills the key and value */
            ITEM_DECOR("%s", "\"%s\"");
            break;
          case STIME_BEST:
            /* the function fills the key and value */
            ITEM_DECOR("%s", "\"%s\"");
            break;
          case ETIME_BEST:
            /* the function fills the key and value */
            ITEM_DECOR("%s", "\"%s\"");
            break;
          case RTT_BEST:
            /* the function fills the key and value */
            ITEM_DECOR("%s", "\"%s\"");
            break;
          case DURATION_INCOMING:
            /* the function adds the key(s), value(s), and final comma */
            item->print_fn = mdPrintDurationIncomingJson;
            item->decorator = g_string_new("%s");
            break;
          case STIME_INCOMING:
            /* the function adds the key(s), value(s), and final comma */
            item->print_fn = mdPrintSTimeIncomingJson;
            item->decorator = g_string_new("%s");
            break;
          case ETIME_INCOMING:
            /* the function adds the key(s), value(s), and final comma */
            item->print_fn = mdPrintETimeIncomingJson;
            item->decorator = g_string_new("%s");
            break;
          case RTT_INCOMING:
            /* the function adds the key(s), value(s), and final comma */
            item->print_fn = mdPrintRTTIncomingJson;
            item->decorator = g_string_new("%s");
            break;
          case PKTS:
            ITEM_DECOR("packetTotalCount", "%" PRIu64);
            break;
          case RPKTS:
            ITEM_DECOR("reversePacketTotalCount", "%" PRIu64);
            break;
          case BYTES:
            ITEM_DECOR("octetTotalCount", "%" PRIu64);
            break;
          case RBYTES:
            ITEM_DECOR("reverseOctetTotalCount", "%" PRIu64);
            break;
          case IFLAGS:
            ITEM_DECOR("initialTCPFlags", "\"%s\"");
            break;
          case RIFLAGS:
            ITEM_DECOR("reverseInitialTCPFlags", "\"%s\"");
            break;
          case UFLAGS:
            ITEM_DECOR("unionTCPFlags", "\"%s\"");
            break;
          case RUFLAGS:
            ITEM_DECOR("reverseUnionTCPFlags", "\"%s\"");
            break;
          case ATTRIBUTES:
            ITEM_DECOR("flowAttributes", "\"%02x\"");
            break;
          case RATTRIBUTES:
            ITEM_DECOR("reverseFlowAttributes", "\"%02x\"");
            break;
          case MAC:
            ITEM_DECOR("sourceMacAddress", "\"%s\"");
            break;
          case DSTMAC:
            ITEM_DECOR("destinationMacAddress", "\"%s\"");
            break;
          case TCPSEQ:
            ITEM_DECOR("tcpSequenceNumber", "\"0x%08x\"");
            break;
          case RTCPSEQ:
            ITEM_DECOR("reverseTcpSequenceNumber", "\"0x%08x\"");
            break;
          case ENTROPY:
            ITEM_DECOR("payloadEntropy", "%u");
            break;
          case RENTROPY:
            ITEM_DECOR("reversePayloadEntropy", "%u");
            break;
          case ENDREASON:
            ITEM_DECOR("flowEndReason", "\"%s\"");
            break;
          case DHCPFP:
            ITEM_DECOR("dhcpFingerPrint", "\"%s\"");
            break;
          case RDHCPFP:
            ITEM_DECOR("reverseDhcpFingerPrint", "\"%s\"");
            break;
          case DHCPVC:
            ITEM_DECOR("dhcpVendorCode", "\"%s\"");
            break;
          case RDHCPVC:
            ITEM_DECOR("reverseDhcpVendorCode", "\"%s\"");
            break;
          case DHCPOPTIONS:
            ITEM_DECOR("dhcpOptionsList", "[%s]");
            break;
          case RDHCPOPTIONS:
            ITEM_DECOR("reverseDhcpOptionsList", "[%s]");
            break;
          case OSNAME:
            ITEM_DECOR("osName", "\"%s\"");
            break;
          case OSVERSION:
            ITEM_DECOR("osVersion", "\"%s\"");
            break;
          case ROSNAME:
            ITEM_DECOR("reverseOsName", "\"%s\"");
            break;
          case ROSVERSION:
            ITEM_DECOR("reverseOsVersion", "\"%s\"");
            break;
          case FINGERPRINT:
            ITEM_DECOR("osFingerPrint", "\"%s\"");
            break;
          case RFINGERPRINT:
            ITEM_DECOR("reverseOsFingerPrint", "\"%s\"");
            break;
          case INGRESS:
            ITEM_DECOR("ingressInterface", "%u");
            break;
          case EGRESS:
            ITEM_DECOR("egressInterface", "%u");
            break;
          case DATABYTES:
            ITEM_DECOR("dataByteCount", "%" PRIu64);
            break;
          case RDATABYTES:
            ITEM_DECOR("reverseDataByteCount", "%" PRIu64);
            break;
          case ITIME:
            ITEM_DECOR("averageInterarrivalTime", "%.3f");
            break;
          case RITIME:
            ITEM_DECOR("reverseAverageInterArrivalTime", "%.3f");
            break;
          case STDITIME:
            ITEM_DECOR("standardDeviationInterarrivalTime", "%.3f");
            break;
          case RSTDITIME:
            ITEM_DECOR("reverseStandardDeviationInterarrivalTime", "%.3f");
            break;
          case TCPURG:
            ITEM_DECOR("tcpUrgentCount", "%u");
            break;
          case RTCPURG:
            ITEM_DECOR("reverseTcpUrgentCount", "%u");
            break;
          case SMALLPKTS:
            ITEM_DECOR("smallPacketCount", "%u");
            break;
          case RSMALLPKTS:
            ITEM_DECOR("reverseSmallPacketCount", "%u");
            break;
          case LARGEPKTS:
            ITEM_DECOR("largePacketCount", "%u");
            break;
          case RLARGEPKTS:
            ITEM_DECOR("reverseLargePacketCount", "%u");
            break;
          case NONEMPTYPKTS:
            ITEM_DECOR("nonEmptyPacketCount", "%u");
            break;
          case RNONEMPTYPKTS:
            ITEM_DECOR("reverseNonEmptyPacketCount", "%u");
            break;
          case MAXSIZE:
            ITEM_DECOR("maxPacketSize", "%d");
            break;
          case RMAXSIZE:
            ITEM_DECOR("reverseMaxPacketSize", "%d");
            break;
          case STDPAYLEN:
            ITEM_DECOR("standardDeviationPayloadLength", "%d");
            break;
          case RSTDPAYLEN:
            ITEM_DECOR("reverseStandardDeviationPayloadLength", "%d");
            break;
          case FIRSTEIGHT:
            ITEM_DECOR("firstEightNonEmptyPacketDirections", "\"%02x\"");
            break;
          case TOS:
            ITEM_DECOR("ipClassOfService", "\"0x%02x\"");
            break;
          case RTOS:
            ITEM_DECOR("reverseIpClassOfService", "\"0x%02x\"");
            break;
          case MPLS1:
            ITEM_DECOR("mplsTopLabelStackSection", "%d");
            break;
          case MPLS2:
            ITEM_DECOR("mplsTopLabelStackSection2", "%d");
            break;
          case MPLS3:
            ITEM_DECOR("mplsTopLabelStackSection3", "%d");
            break;
          case COLLECTOR:
            ITEM_DECOR("collectorName", "\"%s\"");
            break;
          case FIRSTNONEMPTY:
            ITEM_DECOR("firstNonEmptyPacketSize", "%d");
            break;
          case RFIRSTNONEMPTY:
            ITEM_DECOR("reverseFirstNonEmptyPacketSize", "%d");
            break;
          case MPTCPSEQ:
            ITEM_DECOR("mptcpInitialDataSequenceNumber", "%" PRIu64);
            break;
          case MPTCPTOKEN:
            ITEM_DECOR("mptcpReceiverToken", "%u");
            break;
          case MPTCPMSS:
            ITEM_DECOR("mptcpMaximumSegmentSize", "%d");
            break;
          case MPTCPID:
            ITEM_DECOR("mptcpAddressID", "%d");
            break;
          case MPTCPFLAGS:
            ITEM_DECOR("mptcpFlags", "\"%02x\"");
            break;
          case PAYLOAD:
            item->print_fn = mdPrintPayloadJson;
            ITEM_DECOR("payload", "\"%s\"");
            break;
          case RPAYLOAD:
            item->print_fn = mdPrintPayloadJsonRev;
            ITEM_DECOR("reversePayload", "\"%s\"");
            break;
          case NDPI_MASTER:
            ITEM_DECOR("nDPIL7Protocol", "%d");
            break;
          case NDPI_SUB:
            ITEM_DECOR("nDPIL7SubProtocol", "%d");
            break;
          case NONE_FIELD:
            item->decorator = NULL;
            break;
          case ANY_IP6:
          case ANY_IP:
          case ANY_PORT:
          case DIP_V4:
          case DIP_V6:
          case DPI:
          case DURATION_PLACEHOLDER:
          case ETIME_PLACEHOLDER:
          case FIELDS_INTERNAL_ONLY:
          case IPVERSION:
          case RTT_PLACEHOLDER:
          case SIP_V4:
          case SIP_V6:
          case STIME_PLACEHOLDER:
            g_warning("Unexpected value %u seen when setting JSON decorator",
                      item->field);
            break;
        }
    }

#undef ITEM_DECOR
}


/**
 * mdSetFieldListDecoratorBasic
 *
 * create basic printer for CSV
 *
 * Differs from a custom printer in that the output is columnar.
 *
 */
void
mdSetFieldListDecoratorBasic(
    mdFieldList_t  *list,
    char            delimiter)
{
    mdFieldList_t *item = NULL;

    for (item = list; item; item = item->next) {
        if (item->decorator) {
            /* decorator already set */
            return;
        }
        switch (item->field) {
          case SIP_ANY:
            item->decorator = g_string_new("%40s");
            break;
          case DIP_ANY:
            item->decorator = g_string_new("%40s");
            break;
          case SPORT:
            item->decorator = g_string_new("%5d");
            break;
          case DPORT:
            item->decorator = g_string_new("%5d");
            break;
          case PROTOCOL:
            item->decorator = g_string_new("%3d");
            break;
          case NDPI_MASTER:
          case NDPI_SUB:
          case APPLICATION:
            item->decorator = g_string_new("%5d");
            break;
          case VLAN:
            item->decorator = g_string_new("%03x");
            break;
          case DURATION_MILLI:
            item->decorator = g_string_new("%8.3f");
            break;
          case STIME_MILLI:
            item->decorator = g_string_new("%s");
            break;
          case ETIME_MILLI:
            item->decorator = g_string_new("%s");
            break;
          case RTT_MILLI:
            item->decorator = g_string_new("%8.3f");
            break;
          case PKTS:
            item->decorator = g_string_new("%8" PRIu64);
            break;
          case RPKTS:
            item->decorator = g_string_new("%8" PRIu64);
            break;
          case BYTES:
            item->decorator = g_string_new("%8" PRIu64);
            break;
          case RBYTES:
            item->decorator = g_string_new("%8" PRIu64);
            break;
          case IFLAGS:
            item->decorator = g_string_new("%8s");
            break;
          case RIFLAGS:
            item->decorator = g_string_new("%8s");
            break;
          case UFLAGS:
            item->decorator = g_string_new("%8s");
            break;
          case RUFLAGS:
            item->decorator = g_string_new("%8s");
            break;
          case ATTRIBUTES:
            item->decorator = g_string_new("%02x");
            break;
          case RATTRIBUTES:
            item->decorator = g_string_new("%02x");
            break;
          case MAC:
            item->decorator = g_string_new("%s");
            break;
          case DSTMAC:
            item->decorator = g_string_new("%s");
            break;
          case TCPSEQ:
            item->decorator = g_string_new("%08x");
            break;
          case RTCPSEQ:
            item->decorator = g_string_new("%08x");
            break;
          case ENTROPY:
            item->decorator = g_string_new("%3u");
            break;
          case RENTROPY:
            item->decorator = g_string_new("%3u");
            break;
          case ENDREASON:
            item->decorator = g_string_new("%6s");
            break;
          case INGRESS:
            item->decorator = g_string_new("%5u");
            break;
          case EGRESS:
            item->decorator = g_string_new("%5u");
            break;
          case TOS:
            item->decorator = g_string_new(" %02x");
            break;
          case RTOS:
            item->decorator = g_string_new("%02x");
            break;
          case COLLECTOR:
            /* collector is last field so no delimiter, add newline */
            item->decorator = g_string_new("%s\n");
            continue;
          case PAYLOAD:
          case RPAYLOAD:
            item->decorator = g_string_new("");
            continue;
          default:
            g_warning("Invalid field (ID=%d) for Basic Flow Print.",
                      item->field);
            break;
        }

        g_string_append_c(item->decorator, delimiter);
    }
}


mdFieldList_t *
mdCreateBasicFieldList(
    gboolean   payload,
    gboolean   obdomain)
{
    mdFieldList_t         *start = NULL;
    mdFieldList_t        **item = &start;
    mdAcceptFilterField_t *f;

    mdAcceptFilterField_t  fields[] = {
        STIME_PLACEHOLDER,
        ETIME_PLACEHOLDER,
        DURATION_PLACEHOLDER,
        RTT_PLACEHOLDER,
        PROTOCOL,
        SIP_ANY,
        SPORT,
        PKTS,
        BYTES,
        ATTRIBUTES,
        MAC,
        DIP_ANY,
        DPORT,
        RPKTS,
        RBYTES,
        RATTRIBUTES,
        DSTMAC,
        IFLAGS,
        UFLAGS,
        RIFLAGS,
        RUFLAGS,
        TCPSEQ,
        RTCPSEQ,
        INGRESS,
        EGRESS,
        VLAN,
        APPLICATION,
        TOS,
        ENDREASON,
        COLLECTOR,
        /* end of list marker */
        NONE_FIELD
    };
    mdAcceptFilterField_t  payload_fields[] = {
        PAYLOAD,
        RPAYLOAD,
        /* end of list marker */
        NONE_FIELD
    };
    mdAcceptFilterField_t  obdomain_fields[] = {
        OBDOMAIN,
        /* end of list marker */
        NONE_FIELD
    };

    for (f = fields; *f != NONE_FIELD; ++f) {
        *item = mdCreateFieldList(*f, TRUE);
        item = &((*item)->next);
    }

    if (payload) {
        for (f = payload_fields; *f != NONE_FIELD; ++f) {
            *item = mdCreateFieldList(*f, TRUE);
            item = &((*item)->next);
        }
    }

    if (obdomain) {
        for (f = obdomain_fields; *f != NONE_FIELD; ++f) {
            *item = mdCreateFieldList(*f, TRUE);
            item = &((*item)->next);
        }
    }

    return start;
}

mdFieldList_t *
mdCreateIndexFieldList(
    void)
{
    mdFieldList_t         *start = NULL;
    mdFieldList_t        **item = &start;
    mdAcceptFilterField_t *f;
    mdAcceptFilterField_t  fields[] = {
        STIME_BEST,
        PROTOCOL,
        SIP_ANY,
        SPORT,
        DIP_ANY,
        DPORT,
        VLAN,
        OBDOMAIN,
        /* end of list marker */
        NONE_FIELD
    };

    for (f = fields; *f != NONE_FIELD; ++f) {
        *item = mdCreateFieldList(*f, TRUE);
        item = &((*item)->next);
    }

    return start;
}

/**
 *  Function: attachHeadToSLL
 *  Description: attach a new entry to the front of a singly linked list
 *  Params: **head - double pointer to the current head.  *head will point
 *                to that new element at the end of this function
 *          *newEntry - a pointer to the previously allocated entry to be added
 *  Return: NONE
 */
void
attachHeadToSLL(
    mdSLL_t **head,
    mdSLL_t  *newEntry)
{
    assert(head);
    assert(newEntry);

    /*  works even if *head starts out null, being no elements attach
     *  the new entry to the head */
    newEntry->next = *head;
    /*  reassign the head pointer to the new entry */
    *head = newEntry;
}

/**
 *  Function: detachHeadOfSLL
 *  Description: remove the head entry from a singly linked list, set
 *      the head pointer to the next one in the list, and return the
 *      old head
 *  Params: **head - double pointer to the head node of the list.  After
 *                      this function, (*head) will point to the
 *                      new head (*(originalhead)->next)
 *          **toRemove - double pointer to use to return the old head
 *  Return: NONE
 */
void
detachHeadOfSLL(
    mdSLL_t **head,
    mdSLL_t **toRemove)
{
    assert(toRemove);
    assert(head);
    assert(*head);

    /*  set the outgoing pointer to point to the head listing */
    *toRemove = *head;
    /*  move the head pointer down one */
    *head = (*head)->next;
}
