/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_print.c
 *
 *  Contains all printing functions for custom field printers
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.12
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-0935
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_ctx.h"
#include "mediator_util.h"
#include "mediator_inf.h"
#include "mediator_print.h"


/*
 *    **********************************************************************
 */
/*
 *    Callback functions (or helper functions for those callbacks) used by
 *    mdFieldList_t->print_fn.
 */

#if   defined(__clang__)
_Pragma("clang diagnostic push")
_Pragma("clang diagnostic ignored \"-Wformat-nonliteral\"")
#elif defined(__GCC__)
_Pragma("GCC diagnostic push")
_Pragma("GCC diagnostic ignored \"-Wformat-nonliteral\"")
#endif /* if   defined(__clang__) */


/* Helper */
static gboolean
mdPrintInteger(
    int          value,
    mdBuf_t     *buf,
    size_t      *bufsize,
    const char  *decorator)
{
    MD_UNUSED_PARAM(bufsize);
    mdBufAppendPrintf(buf, decorator, value);
    return TRUE;
}

/* Helper */
static gboolean
mdPrintUnsigned32(
    uint32_t     val32,
    mdBuf_t     *buf,
    size_t      *bufsize,
    const char  *decorator)
{
    MD_UNUSED_PARAM(bufsize);
    mdBufAppendPrintf(buf, decorator, val32);
    return TRUE;
}

/* Helper */
static gboolean
mdPrintUnsigned64(
    uint64_t     val64,
    mdBuf_t     *buf,
    size_t      *bufsize,
    const char  *decorator)
{
    MD_UNUSED_PARAM(bufsize);
    mdBufAppendPrintf(buf, decorator, val64);
    return TRUE;
}

/* FLOWKEYHASH */
gboolean
mdPrintFlowKeyHash(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    uint32_t  hash = flow->rec->yafFlowKeyHash;

    if (hash == 0) {
        hash = md_util_flow_key_hash(flow->rec);
    }
    return mdPrintUnsigned32(hash, buf, bufsize, decorator);
}

/* NONE_FIELD */
gboolean
mdPrintNone(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    MD_UNUSED_PARAM(flow);
    MD_UNUSED_PARAM(buf);
    MD_UNUSED_PARAM(bufsize);
    MD_UNUSED_PARAM(decorator);

    return TRUE;
}

/* SIP_INT */
gboolean
mdPrintSIPINT(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->sourceIPv4Address,
                             buf, bufsize, decorator);
}

/* DIP_INT */
gboolean
mdPrintDIPINT(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->destinationIPv4Address,
                             buf, bufsize, decorator);
}

/* STIME_EPOCH_MS */
gboolean
mdPrintSTimeEpochMS(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    MD_UNUSED_PARAM(bufsize);
    mdBufAppendPrintf(buf, decorator, yfTimeToMilli(flow->stime));
    return TRUE;
}

/* ETIME_EPOCH_MS */
gboolean
mdPrintETimeEpochMS(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    MD_UNUSED_PARAM(bufsize);
    mdBufAppendPrintf(buf, decorator, yfTimeToMilli(flow->etime));
    return TRUE;
}

/* SIP_ANY */
gboolean
mdPrintSIPText(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    char  sabuf[40];

    MD_UNUSED_PARAM(bufsize);
    if (!flow->tmpl_attr.is_ipv6) {
        md_util_print_ip4_addr(sabuf, flow->rec->sourceIPv4Address);
    } else {
        md_util_print_ip6_addr(sabuf, flow->rec->sourceIPv6Address);
    }

    mdBufAppendPrintf(buf, decorator, sabuf);
    return TRUE;
}

/* SIP_ANY */
gboolean
mdPrintSIPJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp;
    char     sabuf[40];

    MD_UNUSED_PARAM(bufsize);
    tmp = g_string_sized_new(256);

    if (!flow->tmpl_attr.is_ipv6) {
        md_util_print_ip4_addr(sabuf, flow->rec->sourceIPv4Address);
        g_string_printf(tmp, "\"sourceIPv4Address\":\"%s\",", sabuf);
    } else {
        md_util_print_ip6_addr(sabuf, flow->rec->sourceIPv6Address);
        g_string_printf(tmp, ("\"sourceIPv6Address\":\"%s\","
                              "\"sourceIPv4Address\":\"%s\","),
                        sabuf, sabuf);
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* DIP_ANY */
gboolean
mdPrintDIPText(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    char  dabuf[40];

    MD_UNUSED_PARAM(bufsize);
    if (!flow->tmpl_attr.is_ipv6) {
        md_util_print_ip4_addr(dabuf, flow->rec->destinationIPv4Address);
    } else {
        md_util_print_ip6_addr(dabuf, flow->rec->destinationIPv6Address);
    }

    mdBufAppendPrintf(buf, decorator, dabuf);
    return TRUE;
}

/* DIP_ANY */
gboolean
mdPrintDIPJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp;
    char     sabuf[40];

    MD_UNUSED_PARAM(bufsize);
    tmp = g_string_sized_new(256);

    if (!flow->tmpl_attr.is_ipv6) {
        md_util_print_ip4_addr(sabuf, flow->rec->destinationIPv4Address);
        g_string_printf(tmp, "\"destinationIPv4Address\":\"%s\",", sabuf);
    } else {
        md_util_print_ip6_addr(sabuf, flow->rec->destinationIPv6Address);
        g_string_printf(tmp, ("\"destinationIPv6Address\":\"%s\","
                              "\"destinationIPv4Address\":\"%s\","),
                        sabuf, sabuf);
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* Helper */
static gboolean
mdPrintTimeEpochHelper(
    const mdFullFlow_t  *flow,
    mdBuf_t             *buf,
    size_t              *bufsize,
    const char          *decorator,
    gboolean             is_start)
{
    const yfTime_t *t = (is_start ? &flow->stime : &flow->etime);
    const char     *label;
    char            str[64];

    /* epoch seconds with fractional seconds based on precision */
    MD_UNUSED_PARAM(bufsize);

    if (flow->tmpl_attr.has_nano) {
        struct timespec  ts;

        yfTimeToTimespec(&ts, *t);
        snprintf(str, sizeof(str), "%lld.%09ld",
                 (long long)ts.tv_sec, ts.tv_nsec);
        label = (is_start ? "flowStartNanoseconds" : "flowEndNanoseconds");
    } else if (flow->tmpl_attr.has_micro) {
        struct timeval  tv;

        yfTimeToTimeval(&tv, *t);
        snprintf(str, sizeof(str), "%lld.%06ld",
                 (long long)tv.tv_sec, (long)tv.tv_usec);
        label = (is_start ? "flowStartMicroseconds" : "flowEndMicroseconds");
    } else {
        yf_time_milli_t  tv;

        yfTimeToTimemilli(&tv, *t);
        snprintf(str, sizeof(str), "%lld.%03ld", tv.tv_sec, tv.tv_msec);
        label = (is_start ? "flowStartMilliseconds" : "flowEndMilliseconds");
    }

    if (strchr(decorator, ':')) {
        mdBufAppendPrintf(buf, decorator, label, str);
    } else {
        mdBufAppendPrintf(buf, decorator, str);
    }
    return TRUE;
}

/* STIME_EPOCH */
gboolean
mdPrintSTimeEpoch(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    /* epoch seconds with fractional seconds based on precision */
    return mdPrintTimeEpochHelper(flow, buf, bufsize, decorator, TRUE);
}

/* ETIME_EPOCH */
gboolean
mdPrintETimeEpoch(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    /* epoch seconds with fractional seconds based on precision */
    return mdPrintTimeEpochHelper(flow, buf, bufsize, decorator, FALSE);
}

/* Helper */
static gboolean
mdPrintTimeIncomingJsonHelper(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator,
    gboolean       is_start)
{
    GString        *tmp;
    const yfTime_t *t;
    const char     *start_end;

    MD_UNUSED_PARAM(bufsize);
    tmp = g_string_sized_new(256);
    start_end = (is_start ? "Start" : "End");
    t = (is_start ? &flow->stime : &flow->etime);

    if (flow->tmpl_attr.has_nano) {
        g_string_append_printf(tmp, "\"flow%sNanoseconds\":\"", start_end);
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_NANO);
        g_string_append(tmp, "\",");
    }
    if (flow->tmpl_attr.has_micro) {
        g_string_append_printf(tmp, "\"flow%sMicroseconds\":\"", start_end);
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_MICRO);
        g_string_append(tmp, "\",");
    }
    if (flow->tmpl_attr.has_milli) {
        g_string_append_printf(tmp, "\"flow%sMilliseconds\":\"", start_end);
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_MILLI);
        g_string_append(tmp, "\",");
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* STIME_INCOMING */
gboolean
mdPrintSTimeIncomingJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeIncomingJsonHelper(flow, buf, bufsize, decorator, TRUE);
}

/* ETIME_INCOMING */
gboolean
mdPrintETimeIncomingJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeIncomingJsonHelper(flow, buf, bufsize, decorator, FALSE);
}

/* Helper */
static gboolean
mdPrintTimeIncomingTextHelper(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator,
    gboolean       is_start)
{
    GString        *tmp;
    const yfTime_t *t;
    char            delimiter;

    MD_UNUSED_PARAM(bufsize);

    /* determine the delimiter in use */
    delimiter = decorator[strlen(decorator) - 1];

    tmp = g_string_sized_new(256);
    t = (is_start ? &flow->stime : &flow->etime);

    if (flow->tmpl_attr.has_nano) {
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_NANO);
    }
    if (flow->tmpl_attr.has_micro) {
        if (tmp->len) {
            g_string_append_c(tmp, delimiter);
        }
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_MICRO);
    }
    if (flow->tmpl_attr.has_milli) {
        if (tmp->len) {
            g_string_append_c(tmp, delimiter);
        }
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_MILLI);
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* STIME_INCOMING */
gboolean
mdPrintSTimeIncomingText(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeIncomingTextHelper(flow, buf, bufsize, decorator, TRUE);
}

/* ETIME_INCOMING */
gboolean
mdPrintETimeIncomingText(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeIncomingTextHelper(flow, buf, bufsize, decorator, FALSE);
}

/* Helper */
static gboolean
mdPrintTimeBestHelperJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator,
    gboolean       is_start)
{
    GString        *tmp;
    const yfTime_t *t;
    const char     *label;

    MD_UNUSED_PARAM(bufsize);
    tmp = g_string_sized_new(64);
    t = (is_start ? &flow->stime : &flow->etime);

    if (flow->tmpl_attr.has_nano) {
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_NANO);
        label = (is_start ? "flowStartNanoseconds" : "flowEndNanoseconds");
    } else if (flow->tmpl_attr.has_micro) {
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_MICRO);
        label = (is_start ? "flowStartMicroseconds" : "flowEndMicroseconds");
    } else {
        md_util_time_g_string_append(tmp, t, MD_TIME_FMT_MILLI);
        label = (is_start ? "flowStartMilliseconds" : "flowEndMilliseconds");
    }

    mdBufAppendPrintf(buf, decorator, label, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}


/* Helper */
static gboolean
mdPrintTimeHelper(
    const yfTime_t       *t,
    mdTimePrintFormat_t   t_fmt,
    mdBuf_t              *buf,
    size_t               *bufsize,
    const char           *decorator)
{
    GString *tmp = g_string_sized_new(64);

    MD_UNUSED_PARAM(bufsize);
    md_util_time_g_string_append(tmp, t, t_fmt);
    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* STIME_BEST */
gboolean
mdPrintSTimeBest(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    /* use highest precision available */
    if (strchr(decorator, ':')) {
        return mdPrintTimeBestHelperJson(flow, buf, bufsize, decorator, TRUE);
    }
    return mdPrintTimeHelper(&flow->stime, (flow->tmpl_attr.has_nano
                                            ? MD_TIME_FMT_NANO
                                            : (flow->tmpl_attr.has_micro
                                               ? MD_TIME_FMT_MICRO
                                               : MD_TIME_FMT_MILLI)),
                             buf, bufsize, decorator);
}

/* ETIME_BEST */
gboolean
mdPrintETimeBest(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    /* use highest precision available */
    if (strchr(decorator, ':')) {
        return mdPrintTimeBestHelperJson(flow, buf, bufsize, decorator, FALSE);
    }
    return mdPrintTimeHelper(&flow->etime, (flow->tmpl_attr.has_nano
                                            ? MD_TIME_FMT_NANO
                                            : (flow->tmpl_attr.has_micro
                                               ? MD_TIME_FMT_MICRO
                                               : MD_TIME_FMT_MILLI)),
                             buf, bufsize, decorator);
}

/* STIME_MICRO */
gboolean
mdPrintSTimeMicro(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->stime, MD_TIME_FMT_MICRO,
                             buf, bufsize, decorator);
}

/* ETIME_MICRO */
gboolean
mdPrintETimeMicro(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->etime, MD_TIME_FMT_MICRO,
                             buf, bufsize, decorator);
}

/* STIME_MILLI */
gboolean
mdPrintSTimeMilli(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->stime, MD_TIME_FMT_MILLI,
                             buf, bufsize, decorator);
}

/* ETIME_MILLI */
gboolean
mdPrintETimeMilli(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->etime, MD_TIME_FMT_MILLI,
                             buf, bufsize, decorator);
}

/* STIME_NANO */
gboolean
mdPrintSTimeNano(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->stime, MD_TIME_FMT_NANO,
                             buf, bufsize, decorator);
}

/* ETIME_NANO */
gboolean
mdPrintETimeNano(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->etime, MD_TIME_FMT_NANO,
                             buf, bufsize, decorator);
}

/* STIME_NOFRAC */
gboolean
mdPrintSTimeNoFrac(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->stime, MD_TIME_FMT_NOFRAC,
                             buf, bufsize, decorator);
}

/* ETIME_NOFRAC */
gboolean
mdPrintETimeNoFrac(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTimeHelper(&flow->etime, MD_TIME_FMT_NOFRAC,
                             buf, bufsize, decorator);
}

/* DURATION */
gboolean
mdPrintDurationBest(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    yfDiffTime_t  dur;
    const char   *label;
    char          dur_str[32];

    MD_UNUSED_PARAM(bufsize);
    yfTimeDifference(&dur, flow->etime, flow->stime);
    if (flow->tmpl_attr.has_nano) {
        int64_t  dur_ns = yfDiffTimeToNano(dur);
        snprintf(dur_str, sizeof(dur_str), "%.9f", (double)dur_ns / 1e9);
        label = "flowDurationNanoseconds";
    } else if (flow->tmpl_attr.has_micro) {
        int64_t  dur_us = yfDiffTimeToMicro(dur);
        snprintf(dur_str, sizeof(dur_str), "%.6f", (double)dur_us / 1e6);
        label = "flowDurationMicroseconds";
    } else {
        int64_t  dur_ms = yfDiffTimeToMilli(dur);
        snprintf(dur_str, sizeof(dur_str), "%.3f", (double)dur_ms / 1e3);
        label = "flowDurationMilliseconds";
    }

    if (strchr(decorator, ':')) {
        mdBufAppendPrintf(buf, decorator, label, dur_str);
    } else {
        mdBufAppendPrintf(buf, decorator, dur_str);
    }
    return TRUE;
}

/* DURATION_MILLI */
gboolean
mdPrintDurationMilli(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    yfDiffTime_t  dur;
    int64_t       dur_ms;

    MD_UNUSED_PARAM(bufsize);
    yfTimeDifference(&dur, flow->etime, flow->stime);
    dur_ms = yfDiffTimeToMilli(dur);

    mdBufAppendPrintf(buf, decorator, (double)dur_ms / 1e3);
    return TRUE;
}

/* DURATION_MICRO */
gboolean
mdPrintDurationMicro(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    yfDiffTime_t  dur;
    int64_t       dur_us;

    MD_UNUSED_PARAM(bufsize);
    yfTimeDifference(&dur, flow->etime, flow->stime);
    dur_us = yfDiffTimeToMicro(dur);

    mdBufAppendPrintf(buf, decorator, (double)dur_us / 1e6);
    return TRUE;
}

/* DURATION_NANO */
gboolean
mdPrintDurationNano(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    yfDiffTime_t  dur;
    int64_t       dur_ns;

    MD_UNUSED_PARAM(bufsize);
    yfTimeDifference(&dur, flow->etime, flow->stime);
    dur_ns = yfDiffTimeToNano(dur);

    mdBufAppendPrintf(buf, decorator, (double)dur_ns / 1e9);
    return TRUE;
}

/* DURATION_INCOMING */
gboolean
mdPrintDurationIncomingJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    yfDiffTime_t  dur;
    GString      *str = g_string_sized_new(256);

    MD_UNUSED_PARAM(bufsize);
    yfTimeDifference(&dur, flow->etime, flow->stime);
    if (flow->tmpl_attr.has_nano) {
        int64_t  dur_ns = yfDiffTimeToNano(dur);
        g_string_append_printf(str, "\"flowDurationNanoseconds\":%.9f,",
                               (double)dur_ns / 1e9);
    }
    if (flow->tmpl_attr.has_micro) {
        int64_t  dur_us = yfDiffTimeToMicro(dur);
        g_string_append_printf(str, "\"flowDurationMicroseconds\":%.6f,",
                               (double)dur_us / 1e6);
    }
    if (flow->tmpl_attr.has_milli) {
        int64_t  dur_ms = yfDiffTimeToMilli(dur);
        g_string_append_printf(str, "\"flowDurationMilliseconds\":%.3f,",
                               (double)dur_ms / 1e3);
    }

    mdBufAppendPrintf(buf, decorator, str->str);
    g_string_free(str, TRUE);

    return TRUE;
}

/* DURATION_INCOMING */
gboolean
mdPrintDurationIncomingText(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    yfDiffTime_t  dur;
    GString      *str = g_string_sized_new(256);
    char          delimiter;

    MD_UNUSED_PARAM(bufsize);

    /* determine the delimiter in use */
    delimiter = decorator[strlen(decorator) - 1];

    yfTimeDifference(&dur, flow->etime, flow->stime);
    if (flow->tmpl_attr.has_nano) {
        int64_t  dur_ns = yfDiffTimeToNano(dur);
        g_string_append_printf(str, "%.9f", (double)dur_ns / 1e9);
    }
    if (flow->tmpl_attr.has_micro) {
        int64_t  dur_us = yfDiffTimeToMicro(dur);
        if (str->len) {
            g_string_append_c(str, delimiter);
        }
        g_string_append_printf(str, "%.6f", (double)dur_us / 1e6);
    }
    if (flow->tmpl_attr.has_milli) {
        int64_t  dur_ms = yfDiffTimeToMilli(dur);
        if (str->len) {
            g_string_append_c(str, delimiter);
        }
        g_string_append_printf(str, "%.3f", (double)dur_ms / 1e3);
    }

    mdBufAppendPrintf(buf, decorator, str->str);
    g_string_free(str, TRUE);

    return TRUE;
}

/* RTT_BEST */
gboolean
mdPrintRTTBest(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const char *label;
    char        delta_str[32];

    MD_UNUSED_PARAM(bufsize);
    if (flow->tmpl_attr.has_nano) {
        snprintf(delta_str, sizeof(delta_str), "%.9f",
                 (double)flow->rec->reverseFlowDeltaNanoseconds / 1e9);
        label = "reverseFlowDeltaNanoseconds";
    } else if (flow->tmpl_attr.has_micro) {
        snprintf(delta_str, sizeof(delta_str), "%.6f",
                 (double)flow->rec->reverseFlowDeltaMicroseconds / 1e6);
        label = "reverseFlowDeltaMicroseconds";
    } else {
        snprintf(delta_str, sizeof(delta_str), "%.3f",
                 (double)flow->rec->reverseFlowDeltaMilliseconds / 1e3);
        label = "reverseFlowDeltaMilliseconds";
    }

    if (strchr(decorator, ':')) {
        mdBufAppendPrintf(buf, decorator, label, delta_str);
    } else {
        mdBufAppendPrintf(buf, decorator, delta_str);
    }
    return TRUE;
}

/* RTT_MILLI */
gboolean
mdPrintRTTMilli(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    double  delta;

    MD_UNUSED_PARAM(bufsize);
    if (flow->tmpl_attr.has_milli) {
        delta = (double)flow->rec->reverseFlowDeltaMilliseconds / 1e3;
    } else if (flow->tmpl_attr.has_micro) {
        delta = (double)flow->rec->reverseFlowDeltaMicroseconds / 1e6;
    } else if (flow->tmpl_attr.has_nano) {
        delta = (double)flow->rec->reverseFlowDeltaNanoseconds / 1e9;
    } else {
        delta = 0.0;
    }

    mdBufAppendPrintf(buf, decorator, delta);
    return TRUE;
}

/* RTT_MICRO */
gboolean
mdPrintRTTMicro(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    double  delta;

    MD_UNUSED_PARAM(bufsize);
    if (flow->tmpl_attr.has_micro) {
        delta = (double)flow->rec->reverseFlowDeltaMicroseconds / 1e6;
    } else if (flow->tmpl_attr.has_nano) {
        delta = (double)flow->rec->reverseFlowDeltaNanoseconds / 1e9;
    } else if (flow->tmpl_attr.has_milli) {
        delta = (double)flow->rec->reverseFlowDeltaMilliseconds / 1e3;
    } else {
        delta = 0.0;
    }

    mdBufAppendPrintf(buf, decorator, delta);
    return TRUE;
}

/* RTT_NANO */
gboolean
mdPrintRTTNano(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    double  delta;

    MD_UNUSED_PARAM(bufsize);
    if (flow->tmpl_attr.has_nano) {
        delta = (double)flow->rec->reverseFlowDeltaNanoseconds / 1e9;
    } else if (flow->tmpl_attr.has_micro) {
        delta = (double)flow->rec->reverseFlowDeltaMicroseconds / 1e6;
    } else if (flow->tmpl_attr.has_milli) {
        delta = (double)flow->rec->reverseFlowDeltaMilliseconds / 1e3;
    } else {
        delta = 0.0;
    }

    mdBufAppendPrintf(buf, decorator, delta);
    return TRUE;
}

/* RTT_INCOMING */
gboolean
mdPrintRTTIncomingJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    double   delta;
    GString *str = g_string_sized_new(256);

    MD_UNUSED_PARAM(bufsize);
    if (flow->tmpl_attr.has_nano) {
        delta = (double)flow->rec->reverseFlowDeltaNanoseconds / 1e9;
        g_string_append_printf(str, "\"reverseFlowDeltaNanoseconds\":%.9f,",
                               delta);
    }
    if (flow->tmpl_attr.has_micro) {
        delta = (double)flow->rec->reverseFlowDeltaMicroseconds / 1e6;
        g_string_append_printf(str, "\"reverseFlowDeltaMicroseconds\":%.6f,",
                               delta);
    }
    if (flow->tmpl_attr.has_milli) {
        delta = (double)flow->rec->reverseFlowDeltaMilliseconds / 1e3;
        g_string_append_printf(str, "\"reverseFlowDeltaMilliseconds\":%.3f,",
                               delta);
    }

    mdBufAppendPrintf(buf, decorator, str->str);
    g_string_free(str, TRUE);

    return TRUE;
}

/* RTT_INCOMING */
gboolean
mdPrintRTTIncomingText(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    double   delta;
    GString *str = g_string_sized_new(128);
    char     delimiter;

    MD_UNUSED_PARAM(bufsize);

    /* determine the delimiter in use */
    delimiter = decorator[strlen(decorator) - 1];

    if (flow->tmpl_attr.has_nano) {
        delta = (double)flow->rec->reverseFlowDeltaNanoseconds / 1e9;
        g_string_append_printf(str, "%.9f", delta);
    }
    if (flow->tmpl_attr.has_micro) {
        delta = (double)flow->rec->reverseFlowDeltaMicroseconds / 1e6;
        if (str->len) {
            g_string_append_c(str, delimiter);
        }
        g_string_append_printf(str, "%.6f", delta);
    }
    if (flow->tmpl_attr.has_milli) {
        delta = (double)flow->rec->reverseFlowDeltaMilliseconds / 1e3;
        if (str->len) {
            g_string_append_c(str, delimiter);
        }
        g_string_append_printf(str, "%.3f", delta);
    }

    mdBufAppendPrintf(buf, decorator, str->str);
    g_string_free(str, TRUE);

    return TRUE;
}

/* PROTOCOL */
gboolean
mdPrintProto(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->protocolIdentifier,
                          buf, bufsize, decorator);
}

/* SPORT */
gboolean
mdPrintSPort(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    uint16_t  sp = flow->rec->sourceTransportPort;

    /* FIXME: Should include 58, ICMPv6 */
    if (flow->rec->protocolIdentifier == 1) {
        sp = (flow->rec->destinationTransportPort >> 8);
    }

    return mdPrintInteger(sp, buf, bufsize, decorator);
}

/* DPORT */
gboolean
mdPrintDPort(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    uint16_t  dp = flow->rec->destinationTransportPort;

    /* FIXME: Should include 58, ICMPv6 */
    if (flow->rec->protocolIdentifier == 1) {
        dp = (flow->rec->destinationTransportPort & 0xFF);
    }

    return mdPrintInteger(dp, buf, bufsize, decorator);
}

/* PKTS */
gboolean
mdPrintPackets(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned64(flow->rec->packetTotalCount,
                             buf, bufsize, decorator);
}

/* RPKTS */
gboolean
mdPrintPacketsRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned64(flow->rec->reversePacketTotalCount,
                             buf, bufsize, decorator);
}

/* BYTES */
gboolean
mdPrintBytes(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned64(flow->rec->octetTotalCount,
                             buf, bufsize, decorator);
}

/* RBYTES */
gboolean
mdPrintBytesRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned64(flow->rec->reverseOctetTotalCount,
                             buf, bufsize, decorator);
}

/* TOS */
gboolean
mdPrintTOS(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->ipClassOfService,
                          buf, bufsize, decorator);
}

/* RTOS */
gboolean
mdPrintTOSRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->reverseIpClassOfService,
                          buf, bufsize, decorator);
}

/* MPLS1 */
gboolean
mdPrintMPLS1(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    uint32_t  label = 0;

    MD_UNUSED_PARAM(bufsize);
    memcpy(&label, flow->rec->mplsTopLabelStackSection, 3);
    mdBufAppendPrintf(buf, decorator, label);
    return TRUE;
}

/* MPLS2 */
gboolean
mdPrintMPLS2(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    uint32_t  label = 0;

    MD_UNUSED_PARAM(bufsize);
    memcpy(&label, flow->rec->mplsLabelStackSection2, 3);
    mdBufAppendPrintf(buf, decorator, label);
    return TRUE;
}

/* MPLS3 */
gboolean
mdPrintMPLS3(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    uint32_t  label = 0;

    MD_UNUSED_PARAM(bufsize);
    memcpy(&label, flow->rec->mplsLabelStackSection3, 3);
    mdBufAppendPrintf(buf, decorator, label);
    return TRUE;
}

/* Helper */
static gboolean
mdPrintTcpFlags(
    uint8_t      flags,
    mdBuf_t     *buf,
    size_t      *bufsize,
    const char  *decorator)
{
    char  tmp[10] = "\0";
    char *cp = tmp;

    MD_UNUSED_PARAM(bufsize);

    if (0 == flags) {
        *cp = '0';
        ++cp;
    } else {
        if (flags & 0x40) {*cp = 'E'; ++cp;}
        if (flags & 0x80) {*cp = 'C'; ++cp;}
        if (flags & 0x20) {*cp = 'U'; ++cp;}
        if (flags & 0x10) {*cp = 'A'; ++cp;}
        if (flags & 0x08) {*cp = 'P'; ++cp;}
        if (flags & 0x04) {*cp = 'R'; ++cp;}
        if (flags & 0x02) {*cp = 'S'; ++cp;}
        if (flags & 0x01) {*cp = 'F'; ++cp;}
    }
    *cp = '\0';

    mdBufAppendPrintf(buf, decorator, tmp);
    return TRUE;
}

/* IFLAGS */
gboolean
mdPrintInitFlags(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTcpFlags(flow->rec->initialTCPFlags,
                           buf, bufsize, decorator);
}

/* UFLAGS */
gboolean
mdPrintUnionFlags(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTcpFlags(flow->rec->unionTCPFlags,
                           buf, bufsize, decorator);
}

/* RIFLAGS */
gboolean
mdPrintInitFlagsRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTcpFlags(flow->rec->reverseInitialTCPFlags,
                           buf, bufsize, decorator);
}

/* RUFLAGS */
gboolean
mdPrintUnionFlagsRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintTcpFlags(flow->rec->reverseUnionTCPFlags,
                           buf, bufsize, decorator);
}

/* ATTRIBUTES */
gboolean
mdPrintAttributes(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->flowAttributes,
                          buf, bufsize, decorator);
}

/* RATTRIBUTES */
gboolean
mdPrintAttributesRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->reverseFlowAttributes,
                          buf, bufsize, decorator);
}

/* Helper */
static gboolean
mdPrintHelperMAC(
    const uint8_t   mac[],
    mdBuf_t        *buf,
    size_t         *bufsize,
    const char     *decorator)
{
    char  tmp[30];

    MD_UNUSED_PARAM(bufsize);
    snprintf(tmp, sizeof(tmp), "%02x:%02x:%02x:%02x:%02x:%02x",
             mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
    mdBufAppendPrintf(buf, decorator, tmp);
    return TRUE;
}

/* MAC */
gboolean
mdPrintMAC(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const uint8_t  noMac[6] = {0, 0, 0, 0, 0, 0};

    if (flow->mac) {
        return mdPrintHelperMAC(flow->mac->sourceMacAddress,
                                buf, bufsize, decorator);
    }
    return mdPrintHelperMAC(noMac, buf, bufsize, decorator);
}

/* DSTMAC */
gboolean
mdPrintDSTMAC(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const uint8_t  noMac[6] = {0, 0, 0, 0, 0, 0};

    if (flow->mac) {
        return mdPrintHelperMAC(flow->mac->destinationMacAddress,
                                buf, bufsize, decorator);
    }
    return mdPrintHelperMAC(noMac, buf, bufsize, decorator);
}

/* TCPSEQ */
gboolean
mdPrintTCPSeq(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->tcpSequenceNumber,
                             buf, bufsize, decorator);
}

/* RTCPSEQ */
gboolean
mdPrintTCPSeqRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->reverseTcpSequenceNumber,
                             buf, bufsize, decorator);
}

/* VLAN */
gboolean
mdPrintVLAN(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->vlanId,
                          buf, bufsize, decorator);
}

/* VLANINT */
gboolean
mdPrintVLANINT(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->vlanId,
                             buf, bufsize, decorator);
}

/* APPLICATION */
gboolean
mdPrintAppLabel(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->silkAppLabel,
                          buf, bufsize, decorator);
}

/* NDPI_MASTER */
gboolean
mdPrintNDPIMaster(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->nDPIL7Protocol,
                          buf, bufsize, decorator);
}

/* NDPI_SUB */
gboolean
mdPrintNDPISub(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintInteger(flow->rec->nDPIL7SubProtocol,
                          buf, bufsize, decorator);
}

/* OBDOMAIN */
gboolean
mdPrintOBDomain(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->observationDomainId,
                             buf, bufsize, decorator);
}

/* INGRESS */
gboolean
mdPrintIngress(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->ingressInterface,
                             buf, bufsize, decorator);
}

/* EGRESS */
gboolean
mdPrintEgress(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    return mdPrintUnsigned32(flow->rec->egressInterface,
                             buf, bufsize, decorator);
}

/* ENTROPY */
gboolean
mdPrintEntropy(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->entropy) {
        return mdPrintUnsigned32(flow->entropy->payloadEntropy,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* RENTROPY */
gboolean
mdPrintEntropyRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->entropy && flow->rec->reverseOctetTotalCount) {
        return mdPrintUnsigned32(flow->entropy->reversePayloadEntropy,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* ENDREASON */
gboolean
mdPrintEndReason(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const char *tmp = "";

    MD_UNUSED_PARAM(bufsize);
    switch (flow->rec->flowEndReason & YAF_END_MASK) {
      case YAF_END_IDLE:
        tmp = "idle";
        break;
      case YAF_END_ACTIVE:
        tmp = "active";
        break;
      case YAF_END_FORCED:
        tmp = "eof";
        break;
      case YAF_END_RESOURCE:
        tmp = "rsrc";
        break;
      case YAF_END_UDPFORCE:
        tmp = "force";
        break;
    }

    mdBufAppendPrintf(buf, decorator, tmp);
    return TRUE;
}

/* DHCPFP */
gboolean
mdPrintDHCPFP(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString                       *tmp = g_string_sized_new(512);
    yaf_dhcp_fp_t                 *dhcp = NULL;
    fbSubTemplateMultiListEntry_t *entry;

    MD_UNUSED_PARAM(bufsize);
    if (flow->dhcpfpIndex) {
        entry = fbSubTemplateMultiListGetIndexedEntry(
            &flow->rec->subTemplateMultiList, flow->dhcpfpIndex - 1);
        if (entry->tmplID == YAF_DHCP_FP_TID) {
            dhcp = (yaf_dhcp_fp_t *)FBSTMLNEXT(entry, dhcp);
            g_string_append_len(tmp, (char *)dhcp->dhcpFingerPrint.buf,
                                dhcp->dhcpFingerPrint.len);
        }
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* RDHCPFP */
gboolean
mdPrintDHCPFPRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString                       *tmp = g_string_sized_new(512);
    yaf_dhcp_fp_t                 *dhcp = NULL;
    fbSubTemplateMultiListEntry_t *entry;

    MD_UNUSED_PARAM(bufsize);
    if (flow->dhcpfpIndex) {
        entry = fbSubTemplateMultiListGetIndexedEntry(
            &flow->rec->subTemplateMultiList, flow->dhcpfpIndex - 1);
        if (entry->tmplID == (YAF_DHCP_FP_TID | YTF_REV)) {
            dhcp = (yaf_dhcp_fp_t *)FBSTMLNEXT(entry, dhcp);
            g_string_append_len(tmp, (char *)dhcp->reverseDhcpFingerPrint.buf,
                                dhcp->reverseDhcpFingerPrint.len);
        }
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* DHCPVC */
gboolean
mdPrintDHCPVC(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString                       *tmp = g_string_sized_new(512);
    fbSubTemplateMultiListEntry_t *entry;

    MD_UNUSED_PARAM(bufsize);
    if (flow->dhcpfpIndex) {
        entry = fbSubTemplateMultiListGetIndexedEntry(
            &flow->rec->subTemplateMultiList, flow->dhcpfpIndex - 1);
        if (entry->tmplID == YAF_DHCP_FP_TID) {
            yaf_dhcp_fp_t *dhcp = NULL;
            dhcp = (yaf_dhcp_fp_t *)FBSTMLNEXT(entry, dhcp);
            g_string_append_len(tmp, (char *)dhcp->dhcpVendorCode.buf,
                                dhcp->dhcpVendorCode.len);
        } else if (entry->tmplID == YAF_DHCP_OPTIONS_TID) {
            yaf_dhcp_options_t *dhcp = NULL;
            dhcp = (yaf_dhcp_options_t *)FBSTMLNEXT(entry, dhcp);
            g_string_append_len(tmp, (char *)dhcp->dhcpVendorCode.buf,
                                dhcp->dhcpVendorCode.len);
        }
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* RDHCPVC */
gboolean
mdPrintDHCPVCRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString                       *tmp = g_string_sized_new(512);
    fbSubTemplateMultiListEntry_t *entry;

    MD_UNUSED_PARAM(bufsize);
    if (flow->dhcpfpIndex) {
        entry = fbSubTemplateMultiListGetIndexedEntry(
            &flow->rec->subTemplateMultiList, flow->dhcpfpIndex - 1);
        if (entry->tmplID == (YAF_DHCP_FP_TID | YTF_REV)) {
            yaf_dhcp_fp_t *dhcp = NULL;
            dhcp = (yaf_dhcp_fp_t *)FBSTMLNEXT(entry, dhcp);
            g_string_append_len(tmp, (char *)dhcp->reverseDhcpVendorCode.buf,
                                dhcp->reverseDhcpVendorCode.len);
        } else if (entry->tmplID == (YAF_DHCP_OPTIONS_TID | YTF_REV)) {
            yaf_dhcp_options_t *dhcp = NULL;
            dhcp = (yaf_dhcp_options_t *)FBSTMLNEXT(entry, dhcp);
            g_string_append_len(tmp, (char *)dhcp->reverseDhcpVendorCode.buf,
                                dhcp->reverseDhcpVendorCode.len);
        }
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* DHCPOPTIONS */
gboolean
mdPrintDHCPOptions(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    int                            w;
    GString                       *tmp = g_string_sized_new(512);
    yaf_dhcp_options_t            *dhcp = NULL;
    uint8_t                       *option;
    fbSubTemplateMultiListEntry_t *entry;

    MD_UNUSED_PARAM(bufsize);
    if (flow->dhcpfpIndex) {
        entry = fbSubTemplateMultiListGetIndexedEntry(
            &flow->rec->subTemplateMultiList, flow->dhcpfpIndex - 1);
        if (entry->tmplID == YAF_DHCP_OPTIONS_TID) {
            dhcp = (yaf_dhcp_options_t *)FBSTMLNEXT(entry, dhcp);
            for (w = 0;
                 (option = (uint8_t *)fbBasicListGetIndexedDataPtr(
                      &(dhcp->options), w));
                 w++)
            {
                g_string_append_printf(tmp, "%d, ", *option);
            }
        }
    }

    if (tmp->len > 2) {
        g_string_truncate(tmp, tmp->len - 2);
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* RDHCPOPTIONS */
gboolean
mdPrintDHCPOptionsRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    int                            w;
    GString                       *tmp = g_string_sized_new(512);
    yaf_dhcp_options_t            *dhcp = NULL;
    uint8_t                       *option;
    fbSubTemplateMultiListEntry_t *entry;

    MD_UNUSED_PARAM(bufsize);
    if (flow->dhcpfpIndex) {
        entry = fbSubTemplateMultiListGetIndexedEntry(
            &flow->rec->subTemplateMultiList, flow->dhcpfpIndex - 1);
        if (entry->tmplID == (YAF_DHCP_OPTIONS_TID | YTF_REV)) {
            dhcp = (yaf_dhcp_options_t *)FBSTMLNEXT(entry, dhcp);
            for (w = 0;
                 (option = ((uint8_t *)fbBasicListGetIndexedDataPtr(
                                &(dhcp->revOptions), w)));
                 w++)
            {
                g_string_append_printf(tmp, "%d, ", *option);
            }
        }
    }

    if (tmp->len > 2) {
        g_string_truncate(tmp, tmp->len - 2);
    }

    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* OSNAME */
gboolean
mdPrintOSNAME(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp = g_string_sized_new(512);

    MD_UNUSED_PARAM(bufsize);
    if (flow->p0f) {
        g_string_append_len(tmp, (char *)flow->p0f->osName.buf,
                            flow->p0f->osName.len);
    }
    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* ROSNAME */
gboolean
mdPrintOSNAMERev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp = g_string_sized_new(512);

    MD_UNUSED_PARAM(bufsize);
    if (flow->p0f && flow->rec->reverseOctetTotalCount) {
        g_string_append_len(tmp, (char *)flow->p0f->reverseOsName.buf,
                            flow->p0f->reverseOsName.len);
    }
    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* OSVERSION */
gboolean
mdPrintOSVersion(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp = g_string_sized_new(512);

    MD_UNUSED_PARAM(bufsize);
    if (flow->p0f) {
        g_string_append_len(tmp, (char *)flow->p0f->osVersion.buf,
                            flow->p0f->osVersion.len);
    }
    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* ROSVERSION */
gboolean
mdPrintOSVersionRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp = g_string_sized_new(512);

    MD_UNUSED_PARAM(bufsize);
    if (flow->p0f && flow->rec->reverseOctetTotalCount) {
        g_string_append_len(tmp, (char *)flow->p0f->reverseOsVersion.buf,
                            flow->p0f->reverseOsVersion.len);
    }
    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* FINGERPRINT */
gboolean
mdPrintOSFingerprint(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp = g_string_sized_new(512);

    MD_UNUSED_PARAM(bufsize);
    if (flow->p0f) {
        g_string_append_len(tmp, (char *)flow->p0f->osFingerPrint.buf,
                            flow->p0f->osFingerPrint.len);
    }
    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* RFINGERPRINT */
gboolean
mdPrintOSFingerprintRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    GString *tmp = g_string_sized_new(512);

    MD_UNUSED_PARAM(bufsize);
    if (flow->p0f && flow->rec->reverseOctetTotalCount) {
        g_string_append_len(tmp, (char *)flow->p0f->reverseOsFingerPrint.buf,
                            flow->p0f->reverseOsFingerPrint.len);
    }
    mdBufAppendPrintf(buf, decorator, tmp->str);
    g_string_free(tmp, TRUE);

    return TRUE;
}

/* DATABYTES */
gboolean
mdPrintDataBytes(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintUnsigned64(flow->stats->dataByteCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned64(0, buf, bufsize, decorator);
}

/* ITIME */
gboolean
mdPrintInterArriveTime(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    float  none = 0;

    MD_UNUSED_PARAM(bufsize);
    if (flow->stats) {
        mdBufAppendPrintf(buf, decorator,
                          flow->stats->averageInterarrivalTime / 1000.0);
    } else {
        mdBufAppendPrintf(buf, decorator, none);
    }

    return TRUE;
}

/* RITIME */
gboolean
mdPrintInterArriveTimeRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    float  none = 0;

    MD_UNUSED_PARAM(bufsize);
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        mdBufAppendPrintf(buf, decorator,
                          flow->stats->reverseAverageInterarrivalTime / 1000.0);
    } else {
        mdBufAppendPrintf(buf, decorator, none);
    }

    return TRUE;
}

/* STDITIME */
gboolean
mdPrintSTDITime(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    float  none = 0;

    MD_UNUSED_PARAM(bufsize);
    if (flow->stats) {
        mdBufAppendPrintf(buf, decorator,
                          flow->stats->standardDeviationInterarrivalTime);
    } else {
        mdBufAppendPrintf(buf, decorator, none);
    }

    return TRUE;
}

/* TCPURG */
gboolean
mdPrintTCPURG(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintUnsigned32(flow->stats->tcpUrgTotalCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* SMALLPKTS */
gboolean
mdPrintSmallPkts(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintUnsigned32(flow->stats->smallPacketCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* NONEMPTYPKTS */
gboolean
mdPrintNonEmptyPkts(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintUnsigned32(flow->stats->nonEmptyPacketCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* LARGEPKTS */
gboolean
mdPrintLargePkts(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintUnsigned32(flow->stats->largePacketCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* FIRSTNONEMPTY */
gboolean
mdPrintFirstNonEmpty(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintInteger(flow->stats->firstNonEmptyPacketSize,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* MAXSIZE */
gboolean
mdPrintMaxPacketSize(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintInteger(flow->stats->maxPacketSize,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* STDPAYLEN */
gboolean
mdPrintSTDPayLen(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintInteger(flow->stats->standardDeviationPayloadLength,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* FIRSTEIGHT */
gboolean
mdPrintFirstEight(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats) {
        return mdPrintInteger(flow->stats->firstEightNonEmptyPacketDirections,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* RDATABYTES */
gboolean
mdPrintDataBytesRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintUnsigned64(flow->stats->reverseDataByteCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned64(0, buf, bufsize, decorator);
}

/* RSTDITIME */
gboolean
mdPrintSTDITimeRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    float  none = 0;

    MD_UNUSED_PARAM(bufsize);
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        mdBufAppendPrintf(
            buf, decorator,
            flow->stats->reverseStandardDeviationInterarrivalTime);
    } else {
        mdBufAppendPrintf(buf, decorator, none);
    }

    return TRUE;
}

/* RTCPURG */
gboolean
mdPrintTCPURGRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintUnsigned32(flow->stats->reverseTcpUrgTotalCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* RSMALLPKTS */
gboolean
mdPrintSmallPktsRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintUnsigned32(flow->stats->reverseSmallPacketCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* RNONEMPTYPKTS */
gboolean
mdPrintNonEmptyPktsRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintUnsigned32(flow->stats->reverseNonEmptyPacketCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* RLARGEPKTS */
gboolean
mdPrintLargePktsRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintUnsigned32(flow->stats->reverseLargePacketCount,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* RFIRSTNONEMPTY */
gboolean
mdPrintFirstNonEmptyRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintInteger(flow->stats->reverseFirstNonEmptyPacketSize,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* RMAXSIZE */
gboolean
mdPrintMaxPacketSizeRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintInteger(flow->stats->reverseMaxPacketSize,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* COLLECTOR */
gboolean
mdPrintCollectorName(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    MD_UNUSED_PARAM(bufsize);
    mdBufAppendPrintf(buf, decorator, flow->collector_name);
    return TRUE;
}

/* RSTDPAYLEN */
gboolean
mdPrintSTDPayLenRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->stats && flow->rec->reverseOctetTotalCount) {
        return mdPrintInteger(
            flow->stats->reverseStandardDeviationPayloadLength,
            buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* MPTCPSEQ */
gboolean
mdPrintMPTCPSeq(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->mptcp) {
        return mdPrintUnsigned64(flow->mptcp->mptcpInitialDataSequenceNumber,
                                 buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* MPTCPTOKEN */
gboolean
mdPrintMPTCPToken(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->mptcp) {
        return mdPrintUnsigned32(flow->mptcp->mptcpReceiverToken,
                                 buf, bufsize, decorator);
    }
    return mdPrintUnsigned32(0, buf, bufsize, decorator);
}

/* MPTCPMSS */
gboolean
mdPrintMPTCPMss(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->mptcp) {
        return mdPrintInteger(flow->mptcp->mptcpMaximumSegmentSize,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* MPTCPID */
gboolean
mdPrintMPTCPId(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->mptcp) {
        return mdPrintInteger(flow->mptcp->mptcpAddressID,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* MPTCPFLAGS */
gboolean
mdPrintMPTCPFlags(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    if (flow->mptcp) {
        return mdPrintInteger(flow->mptcp->mptcpFlags,
                              buf, bufsize, decorator);
    }
    return mdPrintInteger(0, buf, bufsize, decorator);
}

/* Helper */
static gboolean
mdPrintPayloadHexdump(
    const uint8_t  *payload,
    size_t          paylen,
    mdBuf_t        *buf,
    size_t         *bufsize,
    const char     *decorator,
    const char     *arrow)
{
    const char  none[] = "";

    MD_UNUSED_PARAM(bufsize);

    if (payload) {
        GString *str = g_string_new("\n");
        md_util_hexdump_g_string_append(str, arrow, payload, paylen);
        mdBufAppendGString(buf, str);
        g_string_free(str, TRUE);
    } else {
        mdBufAppendPrintf(buf, decorator, none);
    }

    return TRUE;
}

/* Helper */
static gboolean
mdPrintPayloadBase64(
    const uint8_t  *payload,
    size_t          paylen,
    mdBuf_t        *buf,
    size_t         *bufsize,
    const char     *decorator)
{
    const char  none[] = "";

    MD_UNUSED_PARAM(bufsize);

    if (payload) {
        gchar *base1;
        base1 = g_base64_encode((const guchar *)payload, paylen);
        mdBufAppendPrintf(buf, decorator, base1);
        g_free(base1);
    } else {
        mdBufAppendPrintf(buf, decorator, none);
    }

    return TRUE;
}

/* PAYLOAD */
gboolean
mdPrintPayloadText(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const uint8_t *payload = (flow->pay) ? flow->pay->payload.buf : NULL;
    size_t         paylen = (flow->pay) ? flow->pay->payload.len : 0;

    return mdPrintPayloadHexdump(payload, paylen, buf, bufsize,
                                 decorator, "  -> ");
}

/* RPAYLOAD */
gboolean
mdPrintPayloadTextRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const uint8_t *payload = NULL;
    size_t         paylen = 0;

    if (flow->pay && flow->rec->reverseOctetTotalCount) {
        payload = flow->pay->reversePayload.buf;
        paylen = flow->pay->reversePayload.len;
    }

    return mdPrintPayloadHexdump(payload, paylen, buf, bufsize,
                                 decorator, "  <- ");
}

/* PAYLOAD */
gboolean
mdPrintPayloadJson(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const uint8_t *payload = (flow->pay) ? flow->pay->payload.buf : NULL;
    size_t         paylen = (flow->pay) ? flow->pay->payload.len : 0;

    return mdPrintPayloadBase64(payload, paylen, buf, bufsize, decorator);
}

/* RPAYLOAD */
gboolean
mdPrintPayloadJsonRev(
    mdFullFlow_t  *flow,
    mdBuf_t       *buf,
    size_t        *bufsize,
    const char    *decorator)
{
    const uint8_t *payload = NULL;
    size_t         paylen = 0;

    if (flow->pay && flow->rec->reverseOctetTotalCount) {
        payload = flow->pay->reversePayload.buf;
        paylen = flow->pay->reversePayload.len;
    }

    return mdPrintPayloadBase64(payload, paylen, buf, bufsize, decorator);
}


#if   defined(__clang__)
_Pragma("clang diagnostic pop")
#elif defined(__GCC__)
_Pragma("GCC diagnostic pop")
#endif

/*
 *    Callback functions appear above this line
 */
/*
 *    **********************************************************************
 */



/**
 * mdPrintBasicFlow
 *
 * print the given flow to the given FILE.
 * this only prints basic flow information, as well
 * as p0f and payload if they are available.
 *
 */
size_t
mdPrintBasicFlow(
    mdFullFlow_t  *flow,
    FILE          *fp,
    char           delimiter,
    GError       **err)
{
    md_main_template_t *rec = flow->rec;
    GString            *str = NULL;
    GString            *tstr = NULL;
    char                sabuf[40];
    char                dabuf[40];
    double              delta;
    yfDiffTime_t        dur;
    int64_t             dur_ms;
    size_t              rc = 0;
    int                 loop;

    str = g_string_sized_new(1024);

    md_util_time_g_string_append(str, &flow->stime, MD_TIME_FMT_MILLI);

    g_string_append_c(str, delimiter);
    md_util_time_g_string_append(str, &flow->etime, MD_TIME_FMT_MILLI);

    /* compute the flow's duration */
    yfTimeDifference(&dur, flow->etime, flow->stime);
    dur_ms = yfDiffTimeToMilli(dur);

    g_string_append_printf(str, "%c%8.3f", delimiter, (double)dur_ms / 1000.0);

    /* determine which reverse-flow-delta time to use */
    if (flow->tmpl_attr.has_milli) {
        delta = (double)flow->rec->reverseFlowDeltaMilliseconds / 1e3;
    } else if (flow->tmpl_attr.has_micro) {
        delta = (double)flow->rec->reverseFlowDeltaMicroseconds / 1e6;
    } else if (flow->tmpl_attr.has_nano) {
        delta = (double)flow->rec->reverseFlowDeltaNanoseconds / 1e9;
    } else {
        delta = 0.0;
    }

    g_string_append_printf(str, "%c%8.3f", delimiter, delta);
    if (!flow->tmpl_attr.is_ipv6) {
        md_util_print_ip4_addr(sabuf, rec->sourceIPv4Address);
        md_util_print_ip4_addr(dabuf, rec->destinationIPv4Address);
    } else {
        md_util_print_ip6_addr(sabuf, (uint8_t *)&(rec->sourceIPv6Address));
        md_util_print_ip6_addr(dabuf,
                               (uint8_t *)&(rec->destinationIPv6Address));
    }
    g_string_append_printf(str, "%c%3d", delimiter, rec->protocolIdentifier);
    if (rec->protocolIdentifier == 1) {
        g_string_append_printf(str, "%c%40s%c%5u%c%8llu%c%8llu%c%02x",
                               delimiter, sabuf, delimiter,
                               rec->destinationTransportPort >> 8, delimiter,
                               (long long unsigned int)rec->packetTotalCount,
                               delimiter,
                               (long long unsigned int)rec->octetTotalCount,
                               delimiter, rec->flowAttributes);
        if (flow->mac) {
            g_string_append_c(str, delimiter);
            for (loop = 0; loop < 5; loop++) {
                g_string_append_printf(str, "%02x:",
                                       flow->mac->sourceMacAddress[loop]);
            }
            g_string_append_printf(str, "%02x",
                                   flow->mac->sourceMacAddress[loop]);
        } else {
            g_string_append_printf(str, "%c00:00:00:00:00:00", delimiter);
        }

        g_string_append_printf(str, "%c%40s%c%5u%c%8llu%c%8llu%c%02x",
                               delimiter, dabuf, delimiter,
                               rec->destinationTransportPort & 0xFF, delimiter,
                               ((long long unsigned int)
                                rec->reversePacketTotalCount),
                               delimiter,
                               ((long long unsigned int)
                                rec->reverseOctetTotalCount),
                               delimiter, rec->reverseFlowAttributes);
        if (flow->mac) {
            g_string_append_c(str, delimiter);
            for (loop = 0; loop < 5; loop++) {
                g_string_append_printf(str, "%02x:",
                                       flow->mac->destinationMacAddress[loop]);
            }
            g_string_append_printf(str, "%02x",
                                   flow->mac->destinationMacAddress[loop]);
        } else {
            g_string_append_printf(str, "%c00:00:00:00:00:00", delimiter);
        }
    } else {
        g_string_append_printf(str, "%c%40s%c%5u%c%8llu%c%8llu%c%02x",
                               delimiter, sabuf, delimiter,
                               rec->sourceTransportPort, delimiter,
                               (long long unsigned int)rec->packetTotalCount,
                               delimiter,
                               (long long unsigned int)rec->octetTotalCount,
                               delimiter, rec->flowAttributes);
        if (flow->mac) {
            g_string_append_c(str, delimiter);
            for (loop = 0; loop < 5; loop++) {
                g_string_append_printf(str, "%02x:",
                                       flow->mac->sourceMacAddress[loop]);
            }
            g_string_append_printf(str, "%02x",
                                   flow->mac->sourceMacAddress[loop]);
        } else {
            g_string_append_printf(str, "%c00:00:00:00:00:00", delimiter);
        }

        g_string_append_printf(str, "%c%40s%c%5u%c%8llu%c%8llu%c%02x",
                               delimiter, dabuf, delimiter,
                               rec->destinationTransportPort, delimiter,
                               ((long long unsigned int)
                                rec->reversePacketTotalCount),
                               delimiter,
                               ((long long unsigned int)
                                rec->reverseOctetTotalCount),
                               delimiter, rec->reverseFlowAttributes);
        if (flow->mac) {
            g_string_append_c(str, delimiter);
            for (loop = 0; loop < 5; loop++) {
                g_string_append_printf(str, "%02x:",
                                       flow->mac->destinationMacAddress[loop]);
            }
            g_string_append_printf(str, "%02x",
                                   flow->mac->destinationMacAddress[loop]);
        } else {
            g_string_append_printf(str, "%c00:00:00:00:00:00", delimiter);
        }
    }

    tstr = g_string_sized_new(32);
    md_util_print_tcp_flags(tstr, rec->initialTCPFlags);
    g_string_append_printf(str, "%c%8s", delimiter, tstr->str);
    g_string_truncate(tstr, 0);
    md_util_print_tcp_flags(tstr, rec->unionTCPFlags);
    g_string_append_printf(str, "%c%8s", delimiter, tstr->str);
    g_string_truncate(tstr, 0);
    md_util_print_tcp_flags(tstr, rec->reverseInitialTCPFlags);
    g_string_append_printf(str, "%c%8s", delimiter, tstr->str);
    g_string_truncate(tstr, 0);
    md_util_print_tcp_flags(tstr, rec->reverseUnionTCPFlags);
    g_string_append_printf(str, "%c%8s", delimiter, tstr->str);
    g_string_free(tstr, TRUE);

    g_string_append_printf(str, "%c%08x%c%08x", delimiter,
                           rec->tcpSequenceNumber,
                           delimiter,
                           rec->reverseTcpSequenceNumber);
    /* g_string_append_printf(str, "%c%04x", */
    /*                        delimiter, rec->ingressInterface); */
    /* g_string_append_printf(str, "%c%04x", delimiter, */
    /*                        rec->egressInterface); */
    g_string_append_printf(str, "%c%03x", delimiter, rec->vlanId);
    g_string_append_printf(str, "%c%5u", delimiter, rec->silkAppLabel);
    if (flow->entropy) {
        g_string_append_printf(str, "%c%3u", delimiter,
                               flow->entropy->payloadEntropy);
        if (rec->reverseOctetTotalCount) {
            g_string_append_printf(str, "%c%3u", delimiter,
                                   flow->entropy->reversePayloadEntropy);
        } else {
            g_string_append_printf(str, "%c000", delimiter);
        }
    } else {
        g_string_append_printf(str, "%c000%c000",
                               delimiter, delimiter);
    }
    g_string_append_c(str, delimiter);
    /* end reason flags */
    switch (rec->flowEndReason & YAF_END_MASK) {
      case YAF_END_IDLE:
        g_string_append_printf(str, "idle  ");
        break;
      case YAF_END_ACTIVE:
        g_string_append_printf(str, "active");
        break;
      case YAF_END_FORCED:
        g_string_append_printf(str, "eof   ");
        break;
      case YAF_END_RESOURCE:
        g_string_append_printf(str, "rsrc  ");
        break;
      case YAF_END_UDPFORCE:
        g_string_append_printf(str, "force ");
        break;
    }

    if (flow->collector_name) {
        g_string_append_printf(str, "%c%s", delimiter, flow->collector_name);
    } else {
        g_string_append_c(str, delimiter);
    }

    g_string_append_printf(str, "\n");

    if (flow->pay) {
        md_util_hexdump_g_string_append(str, "  -> ", flow->pay->payload.buf,
                                        flow->pay->payload.len);
        if (rec->reversePacketTotalCount) {
            md_util_hexdump_g_string_append(str, "  <- ",
                                            flow->pay->reversePayload.buf,
                                            flow->pay->reversePayload.len);
        }
    }

    rc = fwrite(str->str, 1, str->len, fp);

    if (rc != str->len) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                    "Error printing flow: %s\n", strerror(errno));
        rc = 0;
    }

    g_string_free(str, TRUE);

    return rc;
}

/**
 * mdPrintStats
 *
 * print a YAF stats message to the given exporter
 *
 */
size_t
mdPrintStats(
    yaf_process_stats_t  *stats,
    char                 *name,
    FILE                 *lfp,
    char                  delim,
    gboolean              no_stats,
    GError              **err)
{
    GString *str = NULL;
    char     ipaddr[20];
    size_t   rc;

    md_util_print_ip4_addr(ipaddr, stats->exporterIPv4Address);
    str = g_string_sized_new(512);

    if (no_stats != 2) {
        g_string_append_printf(
            str, "stats%c%" PRIu64 "%c%" PRIu64 "%c%" PRIu64 "%c%" PRIu64 "%c",
            delim, stats->exportedFlowRecordTotalCount, delim,
            stats->packetTotalCount, delim,
            stats->droppedPacketTotalCount, delim,
            stats->ignoredPacketTotalCount, delim);
    } else {
        /* stats only */
        g_string_append_printf(
            str, "\\N%c%" PRIu64 "%c%" PRIu64 "%c%" PRIu64 "%c%" PRIu64 "%c",
            delim, stats->exportedFlowRecordTotalCount, delim,
            stats->packetTotalCount, delim,
            stats->droppedPacketTotalCount, delim,
            stats->ignoredPacketTotalCount, delim);
    }

    g_string_append_printf(str, "%u%c%u%c%u%c%u%c%s%c",
                           stats->expiredFragmentCount, delim,
                           stats->assembledFragmentCount, delim,
                           stats->flowTableFlushEvents, delim,
                           stats->flowTablePeakCount, delim,
                           ipaddr, delim);
    g_string_append_printf(str, "%d%c%u%c%u%c%s\n",
                           stats->exportingProcessId, delim,
                           stats->meanFlowRate, delim,
                           stats->meanPacketRate, delim,
                           name);

    rc = fwrite(str->str, 1, str->len, lfp);

    if (rc != str->len) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                    "Error writing %d bytes to file: %s\n",
                    (unsigned int)str->len, strerror(errno));
        rc = 0;
    }

    g_string_free(str, TRUE);

    return rc;
}


/**
 * mdPrintBasicHeader
 *
 * appends a format header to the given GString
 *
 */
void
mdPrintBasicHeader(
    GString  *rstr,
    char      delimiter)
{
    g_string_append_printf(rstr, "start-time%14c", delimiter);
    g_string_append_printf(rstr, "end-time%16c", delimiter);
    g_string_append_printf(rstr, "dur%6c", delimiter);
    g_string_append_printf(rstr, "rtt%6c", delimiter);
    g_string_append_printf(rstr, "pro%c", delimiter);
    g_string_append_printf(rstr, "sip%38c", delimiter);
    g_string_append_printf(rstr, "sp%4c", delimiter);
    g_string_append_printf(rstr, "pkt%6c", delimiter);
    g_string_append_printf(rstr, "oct%6c", delimiter);
    g_string_append_printf(rstr, "at%c", delimiter);
    g_string_append_printf(rstr, "srcMacAddr%8c", delimiter);
    g_string_append_printf(rstr, "dip%38c", delimiter);
    g_string_append_printf(rstr, "dp%4c", delimiter);
    g_string_append_printf(rstr, "rpkt%5c", delimiter);
    g_string_append_printf(rstr, "roct%5c", delimiter);
    g_string_append_printf(rstr, "ra%c", delimiter);
    g_string_append_printf(rstr, "destMacAddr%7c", delimiter);
    g_string_append_printf(rstr, "iflags%3c", delimiter);
    g_string_append_printf(rstr, "uflags%3c", delimiter);
    g_string_append_printf(rstr, "riflags%2c", delimiter);
    g_string_append_printf(rstr, "ruflags%2c", delimiter);
    g_string_append_printf(rstr, "isn%6c", delimiter);
    g_string_append_printf(rstr, "risn%5c", delimiter);
    g_string_append_printf(rstr, "in%4c", delimiter);
    g_string_append_printf(rstr, "out%3c", delimiter);
    g_string_append_printf(rstr, "tag%c", delimiter);
    g_string_append_printf(rstr, "app%3c", delimiter);
    g_string_append_printf(rstr, "tos%c", delimiter);
    g_string_append_printf(rstr, "end%4c", delimiter);
    g_string_append_printf(rstr, "collector");
    g_string_append(rstr, "\n");
}

int
mdPrintDNSRecord(
    FILE      *fp,
    mdBuf_t   *buf,
    char       delimiter,
    uint8_t   *rec,
    gboolean   base64,
    gboolean   print_last,
    gboolean   escape_chars,
    GError   **err)
{
    char            sabuf[40];
    md_dns_dedup_t *record = (md_dns_dedup_t *)rec;
    yfTime_t        yftime;
    size_t          rc;

    yfTimeFromMilli(&yftime, record->flowStartMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppendChar(buf, delimiter);

    if (print_last) {
        yfTimeFromMilli(&yftime, record->flowEndMilliseconds);
        mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
        mdBufAppendChar(buf, delimiter);
    }

    mdBufAppendPrintf(buf, "%d%c", record->rrtype, delimiter);

    if (record->rrname.len) {
        /* this is a dns dedup record so we have to subtract one from the name
         * since we added one for the hash table (string hash requires null
         * char at end of string) */
        if (base64) {
            mdBufAppendBase64(buf, record->rrname.buf, record->rrname.len - 1);
        } else if (escape_chars) {
            mdPrintEscapeChars(buf, record->rrname.buf,
                               record->rrname.len - 1, delimiter);
        } else {
            mdBufAppendLen(buf, record->rrname.buf, record->rrname.len - 1);
        }
        mdBufAppendChar(buf, delimiter);
    }

    if (print_last) {
        mdBufAppendPrintf(buf, "%d%c", record->dnsHitCount, delimiter);
    }

    if (record->sourceIPv4Address) {
        md_util_print_ip4_addr(sabuf, record->sourceIPv4Address);
        mdBufAppendPrintf(buf, "%s", sabuf);
    } else if (record->rrtype == 28) {
        md_util_print_ip6_addr(sabuf, record->sourceIPv6Address);
        mdBufAppendPrintf(buf, "%s", sabuf);
    } else if (record->rrdata.len) {
        if (base64) {
            mdBufAppendBase64(buf, record->rrdata.buf, record->rrdata.len);
        } else if (escape_chars) {
            mdPrintEscapeChars(buf, record->rrdata.buf,
                               record->rrdata.len, delimiter);
        } else {
            mdBufAppendLen(buf, record->rrdata.buf, record->rrdata.len);
        }
    }
    if (record->mapname.len) {
        mdBufAppendChar(buf, delimiter);
        mdBufAppendVarfield(buf, &record->mapname);
    }

    mdBufAppendChar(buf, '\n');

    rc = mdBufWrite(buf, fp, "", err);
    if (!rc) {
        return -1;
    }

    return rc;
}

int
mdPrintDNSRRRecord(
    mdBuf_t   *buf,
    FILE      *fp,
    char       delimiter,
    uint8_t   *rec,
    gboolean   is_ipv6,
    gboolean   base64,
    gboolean   escape_chars,
    GError   **err)
{
    md_dnsrr_t *record = (md_dnsrr_t *)rec;
    yfTime_t    yftime;
    size_t      rc;

    yfTimeFromMilli(&yftime, record->flowStartMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppendPrintf(buf, "%c%u%c%u%c", delimiter,
                      record->yafFlowKeyHash, delimiter,
                      record->observationDomainId, delimiter);

    if (FALSE == is_ipv6) {
        mdBufAppendIP4(buf, record->sourceIPv4Address);
    } else {
        mdBufAppendIP6(buf, record->sourceIPv6Address);
    }
    mdBufAppendChar(buf, delimiter);

    if (FALSE == is_ipv6) {
        mdBufAppendIP4(buf, record->destinationIPv4Address);
    } else {
        mdBufAppendIP6(buf, record->sourceIPv6Address);
    }

    mdBufAppendPrintf(buf, ("%c" "%d%c" "%d%c" "%d%c" "%d%c" "%c%c" /* Q|R */
                            "%d%c" "%d%c" "%d%c" "%d%c" "%d%c" "%u%c"),
                      delimiter,
                      record->protocolIdentifier, delimiter,
                      record->sourceTransportPort, delimiter,
                      record->destinationTransportPort, delimiter,
                      record->vlanId, delimiter,
                      ((record->dnsQueryResponse) ? 'R' : 'Q'), delimiter,
                      record->dnsID, delimiter,
                      record->dnsRRSection, delimiter,
                      record->dnsNXDomain, delimiter,
                      record->dnsAuthoritative, delimiter,
                      record->dnsQRType, delimiter,
                      record->dnsTTL, delimiter);

    if (record->rrname.len) {
        if (base64) {
            mdBufAppendBase64(buf, record->rrname.buf, record->rrname.len);
        } else if (escape_chars) {
            mdPrintEscapeChars(buf, record->rrname.buf,
                               record->rrname.len, delimiter);
        } else {
            mdBufAppendLen(buf, record->rrname.buf, record->rrname.len);
        }
        mdBufAppendChar(buf, delimiter);
    }

    if (record->rrdata.len) {
        if (record->dnsQRType == 1) {
            uint32_t  sip;
            memcpy(&sip, record->rrdata.buf, sizeof(uint32_t));
            mdBufAppendIP4(buf, sip);
        } else if (record->dnsQRType == 28) {
            mdBufAppendIP6(buf, record->rrdata.buf);
        } else if (base64) {
            mdBufAppendBase64(buf, record->rrdata.buf, record->rrdata.len);
        } else if (escape_chars) {
            mdPrintEscapeChars(buf, record->rrdata.buf,
                               record->rrdata.len, delimiter);
        } else {
            mdBufAppendVarfield(buf, &record->rrdata);
        }
    }

    mdBufAppendChar(buf, '\n');

    rc = mdBufWrite(buf, fp, "", err);
    if (!rc) {
        return -1;
    }

    return rc;
}


void
mdPrintEscapeChars(
    mdBuf_t        *mdbuf,
    const uint8_t  *data,
    size_t          datalen,
    char            delimiter)
{
    size_t  i;

    for (i = 0; i < datalen; i++) {
        if (data[i] < ' ' || data[i] > '~') {
            mdBufAppendPrintf(mdbuf, "\\%03o", data[i]);
        } else if (data[i] == '\\') {
            mdBufAppend(mdbuf, "\\\\");
        } else if (data[i] == delimiter) {
            mdBufAppendPrintf(mdbuf, "\\%c", data[i]);
        } else {
            mdBufAppendChar(mdbuf, data[i]);
        }
    }
}

gboolean
mdPrintBasicList(
    mdBuf_t        *buf,
    const GString  *index_str,
    fbBasicList_t  *bl,
    char            delimiter,
    gboolean        hex,
    gboolean        escape)
{
    uint16_t      w = 0;
    fbVarfield_t *var = NULL;

    for (w = 0;
         (var = (fbVarfield_t *)fbBasicListGetIndexedDataPtr(bl, w));
         w++)
    {
        if (var->len == 0) {
            continue;
        }

        if (index_str) {
            mdBufAppendGString(buf, index_str);
        }

        if (hex) {
            mdBufAppendHexdump(buf, var->buf, var->len, FALSE);
        } else if (escape) {
            mdPrintEscapeChars(buf, var->buf, var->len, delimiter);
        } else {
            mdBufAppendVarfield(buf, var);
        }
        mdBufAppendChar(buf, '\n');
    }
    return TRUE;
}

gboolean
mdPrintVariableLength(
    mdBuf_t        *mdbuf,
    const uint8_t  *data,
    size_t          datalen,
    char            delimiter,
    gboolean        hex,
    gboolean        escape)
{
    if (!datalen || !data) {
        return TRUE;
    }

    if (hex) {
        mdBufAppendHexdump(mdbuf, data, datalen, FALSE);
    } else if (escape) {
        mdPrintEscapeChars(mdbuf, data, datalen, delimiter);
    } else {
        mdBufAppendLen(mdbuf, data, datalen);
    }

    return TRUE;
}

int
mdPrintDedupRecord(
    FILE        *fp,
    mdBuf_t     *buf,
    md_dedup_t  *rec,
    char         delimiter,
    GError     **err)
{
    char      sabuf[40];
    yfTime_t  yftime;
    size_t    rc;

    yfTimeFromMilli(&yftime, rec->monitoringIntervalStartMilliSeconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppendChar(buf, delimiter);

    yfTimeFromMilli(&yftime, rec->monitoringIntervalEndMilliSeconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppendChar(buf, delimiter);

    if (rec->sourceIPv4Address != rec->yafFlowKeyHash) {
        if (rec->sourceIPv4Address == 0) {
            md_util_print_ip6_addr(sabuf, rec->sourceIPv6Address);
        } else {
            md_util_print_ip4_addr(sabuf, rec->sourceIPv4Address);
        }
        mdBufAppendPrintf(buf, "%s%c", sabuf, delimiter);
    } else {
        /* configured to dedup on hash (not IP) */
        mdBufAppendPrintf(buf, "%u%c",
                          rec->sourceIPv4Address, delimiter);
    }

    /*stime for flow - with hash makes unique key */
    yfTimeFromMilli(&yftime, rec->flowStartMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppendChar(buf, delimiter);

    /* hash, count */
    mdBufAppendPrintf(buf, "%u%c%" PRIu64 "%c",
                      rec->yafFlowKeyHash, delimiter,
                      rec->observedDataTotalCount, delimiter);

    if (rec->observedData.len) {
        mdBufAppendVarfield(buf, &(rec->observedData));
    } else if (rec->sslCertSerialNumber1.len) {
        mdBufAppendHexdump(buf, rec->sslCertSerialNumber1.buf,
                           rec->sslCertSerialNumber1.len, TRUE);
        mdBufAppendChar(buf, delimiter);
        mdBufAppendVarfield(buf, &(rec->sslCertIssuerCommonName1));
        mdBufAppendChar(buf, delimiter);

        if (rec->sslCertSerialNumber2.len) {
            mdBufAppendHexdump(buf, rec->sslCertSerialNumber2.buf,
                               rec->sslCertSerialNumber2.len, TRUE);
            mdBufAppendChar(buf, delimiter);
            mdBufAppendVarfield(buf, &(rec->sslCertIssuerCommonName2));
        } else {
            mdBufAppendChar(buf, delimiter);
        }
    }

    /* PRINT MAPNAME if available */
    if (rec->mapname.len) {
        mdBufAppendChar(buf, delimiter);
        mdBufAppendVarfield(buf, &(rec->mapname));
    }

    mdBufAppendChar(buf, '\n');

    rc = mdBufWrite(buf, fp, "", err);
    if (!rc) {
        return -1;
    }

    return rc;
}

int
mdPrintSSLDedupRecord(
    FILE     *fp,
    mdBuf_t  *buf,
    uint8_t  *rec,
    char      delimiter,
    GError  **err)
{
    const md_ssl_dedup_t *ssl = (md_ssl_dedup_t *)rec;
    yfTime_t              yftime;
    size_t                rc;

    yfTimeFromMilli(&yftime, ssl->flowStartMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppendChar(buf, delimiter);

    yfTimeFromMilli(&yftime, ssl->flowEndMilliseconds);
    mdBufAppendTime(buf, &yftime, MD_TIME_FMT_MILLI);
    mdBufAppendChar(buf, delimiter);

    mdBufAppendHexdump(buf, ssl->sslCertSerialNumber.buf,
                       ssl->sslCertSerialNumber.len, TRUE);

    mdBufAppendPrintf(buf, "%c%" PRIu64 "%c",
                      delimiter, ssl->observedDataTotalCount, delimiter);
    mdBufAppendVarfield(buf, &(ssl->sslCertIssuerCommonName));

    if (ssl->mapname.len) {
        mdBufAppendChar(buf, delimiter);
        mdBufAppendVarfield(buf, &(ssl->mapname));
    }

    mdBufAppendChar(buf, '\n');

    rc = mdBufWrite(buf, fp, "", err);
    if (!rc) {
        return -1;
    }

    return rc;
}
