/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_ssl.h
 *
 *  header file for mediator_ssl.c
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.12
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-0935
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#include "mediator_templates.h"
#include "mediator_util.h"

#if HAVE_OPENSSL
#include <openssl/evp.h>
#if OPENSSL_VERSION_NUMBER < 0x30000000
#include <openssl/md5.h>
#include <openssl/sha.h>
#endif
#endif  /* HAVE_OPENSSL */


/* In mdSSLConfig_t, the size of the bitmap. */
#define MD_SSL_CONFIG_MAX_ID      1024
#define MD_SSL_CONFIG_MAX_ID_STR  "1023"

/* In a bitmap (array[uint32_t]) return the index in the array of the bit at
 * position 'p' */
#define MD_BMAP_INDEX(p)      ((p) >> 5)
/* Given a uint32_t found by MD_BMAP_INDEX, return a mask for the bit at
 * position 'p' */
#define MD_BMAP_OFFSET(p)     (1u << ((p) & 0x1F))

/* In bit-array `name`, turn on the bit at position `p` */
#define MD_BMAP_SET(name, p)                             \
    do {                                                 \
        ((name)[MD_BMAP_INDEX(p)] |= MD_BMAP_OFFSET(p)); \
    } while (0)
/* In bit-array `name`, turn off the bit at position `p` */
#define MD_BMAP_UNSET(name, p)                            \
    do {                                                  \
        ((name)[MD_BMAP_INDEX(p)] &= ~MD_BMAP_OFFSET(p)); \
    } while (0)
/* In bit-array `name`, return the value at position `p` */
#define MD_BMAP_GET(name, p)                                    \
    (((name)[MD_BMAP_INDEX(p)] & MD_BMAP_OFFSET(p)) ? 1 : 0)
/* Return the number of uint32_t required to hold a 'numbits' bitmap */
#define MD_BMAP_ARRAYSIZE(numbits)   ((numbits + 0x1f) >> 5)


/* also appears in mediator_ctx.h */
typedef struct md_ssl_dedup_state_st md_ssl_dedup_state_t;


md_ssl_dedup_state_t *
md_ssl_new_dedup_state(
    void);

void
md_ssl_dedup_configure_state(
    md_ssl_dedup_state_t  *state,
    int                    max_hit,
    int                    flush_timeout,
    char                  *filename,
    smFieldMap_t          *map,
    gboolean               export_name);


gboolean
md_ssl_flush_queue(
    md_export_node_t  *exp,
    mdConfig_t        *md_config,
    GError           **err);

gboolean
md_ssl_dedup_free_state(
    mdConfig_t        *cfg,
    md_export_node_t  *state,
    GError           **err);

void
md_ssl_dedup_print_stats(
    md_ssl_dedup_state_t  *state,
    char                  *exp_name);

void
md_ssl_flush_tab(
    md_ssl_dedup_state_t  *state,
    const yfTime_t        *cur_time,
    gboolean               flush_all);

gboolean
md_ssl_add_node(
    mdContext_t       *ctx,
    md_export_node_t  *exp,
    mdFullFlow_t      *flow);

yaf_ssl_cert_t *
md_ssl_cert_decode(
    uint8_t       *cert,
    size_t         cert_len,
    fbTemplate_t  *tmpl);

gboolean
md_ssl_export_ssl_cert(
    mdContext_t       *ctx,
    md_export_node_t  *exp,
    yaf_ssl_cert_t    *cert,
    GError           **err);

#if HAVE_OPENSSL
void
md_ssl_md5_hash(
    unsigned char  *hash,
    uint8_t        *cert,
    size_t          cert_len);
void
md_ssl_sha1_hash(
    unsigned char  *hash,
    uint8_t        *cert,
    size_t          cert_len);
#endif /* if HAVE_OPENSSL */
