/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_ctx.h
 *
 *  IPFIX mediator for filtering, DNS deduplication, and other mediator-like
 *  things
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.13
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1447
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#ifndef MD_CTX
#define MD_CTX

#include "mediator_autohdr.h"
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h>
#include <assert.h>
#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_FCNTL_H
#include <sys/fcntl.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif
#include <glib.h>
#include <fixbuf/public.h>
#include "mediator_time.h"


#define FBSTLNEXT(a, b) fbSubTemplateListGetNextPtr((a), (b))
#define FBSTMLNEXT(a, b) fbSubTemplateMultiListEntryNextDataPtr((a), (b))
#define FBSTMLNEXTENTRY(a, b) fbSubTemplateMultiListGetNextEntry((a), (b))

#define CERT_PEN 6871


#define MD_UNUSED_PARAM(x_)  (void)(x_)


/* Erorr constants for g_set_error() */
#define MD_ERROR_DOMAIN     g_quark_from_string("MediatorError")
/* Template Issue - Not Critical*/
#define MD_ERROR_TMPL   1
/* IO Error - Critical */
#define MD_ERROR_IO     2
/* Setup Error */
#define MD_ERROR_SETUP  3
/* memory problem */
#define MD_ERROR_MEM    4
/* Error to ignore */
#define MD_ERROR_NODROP 5

/* Maximum number of lists */
#define MAX_LIST 10

/* Maximum number of maps */
#define MAX_MAPS 100

/* Do not restart if the previous (re)start time is less this this number if
 * milliseconds ago (30 sec) */
#define MD_RESTART_MS 30000

/* Initial size for the text export buffer. */
#define MD_MSGLEN_STD 65535


typedef enum mdTransportType_en {
    NONE,
    TCP,
    UDP,
    SPREAD,
    FILEHANDLER,
    TEXT,
    DIRECTORY
} mdTransportType_t;

typedef enum fieldOperator_en {
    OPER_UNTOUCHED,
    IN_LIST,
    NOT_IN_LIST,
    EQUAL,
    NOT_EQUAL,
    LESS_THAN,
    LESS_THAN_OR_EQUAL,
    GREATER_THAN,
    GREATER_THAN_OR_EQUAL
} fieldOperator;

/*
 *    Identifiers for fields used by TEXT and JSON exporters and for filtering
 *    expressions.
 *
 *    The text names for these fields are defined in mediator_config_lex.l.
 *
 *    DO NOT MODIFY THE ORDER OF FIELDS.
 *
 *    The parser supports (and the --fields option requires) numeric IDs for
 *    fields which depend on the order presented here.
 *
 *    DO NOT MODIFY THE ORDER OF FIELDS.
 */
typedef enum mdAcceptFilterField_en {
    SIP_ANY,
    DIP_ANY,
    /* Not a printable field */
    SIP_V4,
    /* Not a printable field */
    DIP_V4,
    SPORT,
    DPORT,
    PROTOCOL,
    APPLICATION,

    /* 8 */
    /* Not a printable field */
    SIP_V6,
    /* Not a printable field */
    DIP_V6,
    /* Not a printable field */
    ANY_IP6,
    /* Not a printable field */
    ANY_IP,
    /* Not a printable field */
    ANY_PORT,

    OBDOMAIN,

    /* Not a printable field */
    IPVERSION,

    VLAN,

    /* Most of the following keywords are NOT supported in FILTERS */

    /* 16 */
    FLOWKEYHASH,
    /* Gets replaced by one or more of DURATION_BEST, DURATION_INCOMING,
     * DURATION_MICRO, DURATION_MILLI, DURATION_NANO, DURATION_NOFRAC */
    DURATION_PLACEHOLDER,

    /* Gets replaced by one or more of STIME_BEST, STIME_INCOMING,
     * STIME_MICRO, STIME_MILLI, STIME_NANO, STIME_NOFRAC */
    STIME_PLACEHOLDER,
    /* Gets replaced by one or more of ETIME_BEST, ETIME_INCOMING,
     * ETIME_MICRO, ETIME_MILLI, ETIME_NANO, ETIME_NOFRAC */
    ETIME_PLACEHOLDER,

    /* Times as epoch milliseconds */
    STIME_EPOCH_MS,
    ETIME_EPOCH_MS,

    /* Integer IPv4 addresses */
    SIP_INT,
    DIP_INT,

    /* 24 */
    /* Gets replaced by one or more of RTT_BEST, RTT_INCOMING, RTT_MICRO,
     * RTT_MILLI, RTT_NANO, RTT_NOFRAC */
    RTT_PLACEHOLDER,
    PKTS,
    RPKTS,
    BYTES,
    RBYTES,
    IFLAGS,
    RIFLAGS,
    UFLAGS,

    /* 32 */
    RUFLAGS,
    ATTRIBUTES,
    RATTRIBUTES,
    MAC,
    DSTMAC,
    TCPSEQ,
    RTCPSEQ,
    ENTROPY,

    /* 40 */
    RENTROPY,
    ENDREASON,
    OSNAME,
    OSVERSION,
    ROSNAME,
    ROSVERSION,
    FINGERPRINT,
    RFINGERPRINT,

    /* 48 */
    DHCPFP,
    DHCPVC,
    RDHCPFP,
    RDHCPVC,
    INGRESS,
    EGRESS,
    DATABYTES,
    RDATABYTES,

    /* 56 */
    ITIME,
    RITIME,
    STDITIME,
    RSTDITIME,
    TCPURG,
    RTCPURG,
    SMALLPKTS,
    RSMALLPKTS,

    /* 64 */
    LARGEPKTS,
    RLARGEPKTS,
    NONEMPTYPKTS,
    RNONEMPTYPKTS,
    MAXSIZE,
    RMAXSIZE,
    STDPAYLEN,
    RSTDPAYLEN,

    /* 72 */
    FIRSTEIGHT,
    /* Not a printable field */
    DPI,
    VLANINT,
    TOS,
    RTOS,
    MPLS1,
    MPLS2,
    MPLS3,

    /* 80 */
    COLLECTOR,
    FIRSTNONEMPTY,
    RFIRSTNONEMPTY,
    MPTCPSEQ,
    MPTCPTOKEN,
    MPTCPMSS,
    MPTCPID,
    MPTCPFLAGS,

    /* 88 */
    PAYLOAD,
    RPAYLOAD,
    DHCPOPTIONS,
    RDHCPOPTIONS,
    NDPI_MASTER,
    NDPI_SUB,
    /* Times as epoch seconds with fractional seconds as determined by the
     * highest precision available on the record */
    STIME_EPOCH,
    ETIME_EPOCH,

    /* 96 */

    /* Start of internal / non-public values */
    FIELDS_INTERNAL_ONLY = 160,

    /* 200 */
    /* A single time with the highest precision */
    DURATION_BEST = 200,
    ETIME_BEST,
    STIME_BEST,
    RTT_BEST,
    /* Times as they are on the incoming record */
    DURATION_INCOMING,
    ETIME_INCOMING,
    RTT_INCOMING,
    STIME_INCOMING,

    /* 208 */
    /* Force times to microseconds */
    DURATION_MICRO,
    ETIME_MICRO,
    RTT_MICRO,
    STIME_MICRO,
    /* Force times to milliseconds */
    DURATION_MILLI,
    ETIME_MILLI,
    RTT_MILLI,
    STIME_MILLI,

    /* 216 */
    /* Force times to nanoseconds */
    DURATION_NANO,
    ETIME_NANO,
    RTT_NANO,
    STIME_NANO,
    /* Force times to seconds with no fractional part */
    DURATION_NOFRAC,
    ETIME_NOFRAC,
    RTT_NOFRAC,
    STIME_NOFRAC,

    /* 224 */
    NONE_FIELD
} mdAcceptFilterField_t;

typedef enum mdLogLevel_en {
    MD_DEBUG,
    MESSAGE,
    WARNING,
    ERROR,
    QUIET
} mdLogLevel_t;

typedef enum mdTimeUnits_en {
    MD_TUNIT_UNSET = 0,
    MD_TUNIT_DEFAULT = 1u,
    MD_TUNIT_BEST = 2u,
    MD_TUNIT_INCOMING = 4u,
    /* values >= 16 are individual bits */
    MD_TUNIT_NOFRAC = 0x10u,
    MD_TUNIT_MILLI = 0x20u,
    MD_TUNIT_MICRO = 0x40u,
    MD_TUNIT_NANO = 0x80u
} mdTimeUnits_t;

typedef struct mdConfig_st mdConfig_t;

/* configuration options */
extern int              myVersion;
extern int              md_stats_timeout;
extern mdConfig_t       md_config;
#if HAVE_SPREAD
extern char           **md_out_groups;
extern int              num_out_groups;
#endif
extern char            *md_logfile;
extern char            *md_logdest;
extern char            *md_pidfile;
extern mdLogLevel_t     md_log_level;
extern unsigned int     md_top_level_time_units;
/* user's value for template timeout, in seconds */
extern int              udp_temp_timeout;
extern uint16_t         dns_flush_timeout;
extern gboolean         multi_file_mode;
extern fbInfoElement_t *user_elements;
#if ENABLE_SKIPSET || ENABLE_SKPREFIXMAP || ENABLE_SKTYPESENSOR
extern int              app_registered;
#endif


struct mdFlowCollector_st;
typedef struct mdFlowCollector_st mdFlowCollector_t;

struct mdFlowExporter_st;
typedef struct mdFlowExporter_st mdFlowExporter_t;


#if ENABLE_SKIPSET
/* From mediator_utils.h: A wrapper over the SiLK IPSet structure to avoid
 * opening the same file multiple times. */
typedef struct mdIPSet_st mdIPSet_t;
#endif  /* ENABLE_SKIPSET */

/* Double Linked List */
typedef struct mdDLL_st mdDLL_t;
struct mdDLL_st {
    mdDLL_t  *next;
    mdDLL_t  *prev;
};

/* Single Linked List */
typedef struct mdSLL_st mdSLL_t;
struct mdSLL_st {
    mdSLL_t  *next;
};

typedef struct smHashTable_st {
    size_t       len;
    GHashTable  *table;
} smHashTable_t;

typedef struct smFieldMap_st smFieldMap_t;
struct smFieldMap_st {
    smFieldMap_t           *next;
    mdAcceptFilterField_t   field;
    smHashTable_t          *table;
    char                   *name;
    char                  **labels;
    size_t                  count;
    gboolean                discard;
};

typedef struct smFieldMapKV_st {
    uint32_t   val;
} smFieldMapKV_t;

typedef struct md_dns_dedup_state_st md_dns_dedup_state_t;

typedef struct md_dedup_state_st md_dedup_state_t;

typedef struct md_ssl_dedup_state_st md_ssl_dedup_state_t;

/* Defined in mediator_filter.h */
typedef struct md_filter_st md_filter_t;

/* Defined in mediator_filter.h */
typedef struct md_spread_filter_st md_spread_filter_t;

typedef struct md_export_node_st md_export_node_t;
struct md_export_node_st {
    md_export_node_t      *next;
    mdFlowExporter_t      *exp;
    md_filter_t           *filter;
    md_dns_dedup_state_t  *dns_dedup;
    md_dedup_state_t      *dedup;
    md_ssl_dedup_state_t  *ssl_dedup;
    gboolean               and_filter;
    gboolean               md5_hash;
    gboolean               sha1_hash;
};

typedef struct md_stats_st {
    uint64_t   recvd_flows;
    uint64_t   dns;
    uint64_t   recvd_filtered;
    uint64_t   recvd_stats;
    uint64_t   nonstd_flows;
    uint64_t   uniflows;
    uint32_t   files;
    uint16_t   restarts;
} md_stats_t;

typedef struct md_collect_node_st md_collect_node_t;
struct md_collect_node_st {
    md_collect_node_t  *next;
    mdFlowCollector_t  *coll;
    md_filter_t        *filter;
    fBuf_t             *fbuf;
    md_stats_t         *stats;
    pthread_cond_t      cond;
    pthread_mutex_t     mutex;
    gboolean            and_filter;
    gboolean            active;
};

typedef struct mdBuf_st {
    /* buffer being appended to */
    char    *b_buf;
    /* current write position in `b_buf` */
    char    *b_cp;
    /* total length of `b_buf` */
    size_t   b_buflen;
    /* available octets in `b_buf` */
    size_t   b_rem;
} mdBuf_t;

struct mdConfig_st {
    md_collect_node_t   *flowsrc;
    md_export_node_t    *flowexit;
    smFieldMap_t        *maps;
    FILE                *log;
    md_spread_filter_t  *mdspread;
    /* linked list of filter comparisons defined in the FILTER block; see also
     * shared_filter_and below */
    md_filter_t         *shared_filter;
    char                *collector_name;
    pthread_cond_t       log_cond;
    pthread_mutex_t      log_mutex;
    /* how often to resend templates when using UDP; this will be set to 1/3
     * of the UDP template timeout */
    yfDiffTime_t         udp_template_interval;
    /* "current" time */
    yfTime_t             ctime;
#ifdef HAVE_SPREAD
    fbSpreadParams_t     out_spread;
#endif
    uint32_t             current_domain;
    unsigned int         usec_sleep;
    uint16_t             tombstone_configured_id;
    uint16_t             collector_id;
    gboolean             no_stats;
    gboolean             ipfixSpreadTrans;
    gboolean             lockmode;
    gboolean             dns_base64_encode;
    gboolean             dns_print_lastseen;
    /* whether the shared_filter above is an AND filter */
    gboolean             shared_filter_and;
    gboolean             preserve_obdomain;
    gboolean             gen_tombstone;
    gboolean             rewrite_ssl_certs;
#if    ENABLE_SKTYPESENSOR
    /* The CURRENT collector's setting for the vlan setting of SILK_PROBE */
    gboolean             collector_silk_probe_vlan;
    /* The CURRENT collector's setting for the name of SILK_PROBE */
    const char          *collector_silk_probe_name;
#endif  /* ENABLE_SKTYPESENSOR */
};

#ifdef MEDIATOR_MAIN_SOURCE
static mdConfig_t  MD_CONFIG_INIT = {
    NULL,                       /* flowsrc */
    NULL,                       /* flowexit */
    NULL,                       /* maps */
    NULL,                       /* log */
    NULL,                       /* mdspread */
    NULL,                       /* shared_filter */
    NULL,                       /* collector_name */
    PTHREAD_COND_INITIALIZER,   /* log_cond */
    PTHREAD_MUTEX_INITIALIZER,  /* log_mutex */
    YF_DIFFTIME_INIT,           /* udp_template_interval */
    YF_TIME_INIT,               /* ctime */
#ifdef HAVE_SPREAD
    FB_SPREADPARAMS_INIT,       /* out_spread */
#endif
    0,                          /* current_domain */
    0,                          /* usec_sleep */
    0,                          /* tombstone_configured_id */
    0,                          /* collector_id */
    FALSE,                      /* no_stats */
    FALSE,                      /* ipfixSpreadTrans */
    FALSE,                      /* lockmode */
    FALSE,                      /* dns_base64_encode */
    FALSE,                      /* dns_print_lastseen */
    FALSE,                      /* shared_filter_and */
    FALSE,                      /* preserve_obdomain */
    FALSE,                      /* gen_tombstone */
    FALSE                       /* rewrite_ssl_certs */
#if    ENABLE_SKTYPESENSOR
    , FALSE                      /* collector_silk_probe_vlan */
    , NULL                       /* collector_silk_probe_name */
#endif  /* ENABLE_SKTYPESENSOR */
};
#endif  /* MEDIATOR_MAIN_SOURCE */


typedef struct mdContext_st {
    mdConfig_t  *cfg;
    md_stats_t  *stats;
    GError      *err;
} mdContext_t;

#define MD_CTX_INIT { NULL, NULL, NULL }


/* Function to parse the config file, defined in mediator_config_lex.l */
gboolean
mediator_config_load(
    const char  *config_file);


#endif  /* MD_CTX */
