/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator_open.c
 *
 *  All IPFIX collector functionality
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.13
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1447
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */


#define LINE_BUF_SIZE 4096
/*RFC 1950 */
#define ZLIB_HEADER 0x9C78
/* RFC 1952 */
#define GZIP_HEADER 0x8B1F
#define SM_CHUNK 16384

#include "mediator_inf.h"
#include "mediator_core.h"
#include "mediator.h"
#include "mediator_filter.h"
#ifdef HAVE_GLOB_H
#include <glob.h>
#endif

#if SM_ENABLE_ZLIB
#include <zlib.h>
#endif

extern int              md_quit;

static pthread_mutex_t  global_listener_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t   global_listener_cond = PTHREAD_COND_INITIALIZER;

static unsigned int     num_collectors = 0;

/* typedef struct mdFlowCollector_st mdFlowCollector_t; */
struct mdFlowCollector_st {
    fbCollector_t      *collector;
    fbListener_t       *listener;
    FILE               *lfp;
    fbSession_t        *session;
    char               *name;
    char               *inspec;
    char               *move_dir;
    char               *decompress;
    GString            *fname_in;
    GString            *fname_lock;
    GError             *err;
    fbConnSpec_t        connspec;
    mdTransportType_t   type;
    pthread_t           thread;
#if HAVE_SPREAD
    fbSpreadParams_t    spread;
#endif
#if ENABLE_SKTYPESENSOR
    char               *silk_probe_name;
    gboolean            silk_probe_vlan;
#endif
    uint32_t            domain;
    uint16_t            poll_time;
    uint16_t            id;
    gboolean            active;
    gboolean            data;
    gboolean            restart;
    gboolean            std_in;
    /* Note: lockmode is always active regardless of this setting */
    gboolean            lockmode;
    gboolean            delete_files;
};


/**
 *    Object to hold the name and modification time of a file found by glob().
 *    Used by a polling collector to process the oldest files first instead of
 *    in name order.
 */
typedef struct mdCollectorDirEntry_st {
    /* pointer to the file-name held by the glob-data */
    const char *name;
    /* modification time in epoch nanoseconds */
    uint64_t    mtime;
} mdCollectorDirEntry_t;


static void
mdfBufFree(
    md_collect_node_t  *collector)
{
    fBufFree(collector->fbuf);
    collector->fbuf = NULL;
}


void
mdInterruptListeners(
    mdConfig_t  *cfg)
{
    md_collect_node_t *cnode = NULL;

    for (cnode = cfg->flowsrc; cnode; cnode = cnode->next) {
        if (cnode->active) {
#if HAVE_SPREAD
            if (cnode->coll->spread.session && cnode->coll->collector) {
                fbCollectorClose(cnode->coll->collector);
                continue;
            }
#endif /* if HAVE_SPREAD */
            if (cnode->coll->listener) {
                fbListenerInterrupt(cnode->coll->listener);
                pthread_cond_signal(&cnode->cond);
            }
        }
    }
    pthread_cond_signal(&global_listener_cond);
}

mdFlowCollector_t *
mdNewFlowCollector(
    mdTransportType_t   mode,
    const char         *name)
{
    mdFlowCollector_t *collector;

    collector = g_slice_new0(mdFlowCollector_t);

    collector->type = mode;

    if (mode == TCP) {
        collector->connspec.transport = FB_TCP;
    } else if (mode == UDP) {
        collector->connspec.transport = FB_UDP;
    }

    num_collectors++;
    collector->id = num_collectors;

    if (name) {
        collector->name = g_strdup(name);
    } else {
        collector->name = g_strdup_printf("C%d", collector->id);
    }

    return collector;
}

#if SM_ENABLE_ZLIB
static FILE *
mdFileDecompress(
    FILE        *src,
    const char  *tmp_file_path)
{
    int            ret;
    z_stream       strm;
    unsigned int   leftover;
    unsigned char  in[SM_CHUNK];
    unsigned char  out[SM_CHUNK];
    FILE          *dst = NULL;
    int            fd;
    char           tmpname[SM_CHUNK];
    char           temp_suffix[] = ".XXXXXX";

    /*allocate state */
    strm.zalloc = Z_NULL;
    strm.zfree = Z_NULL;
    strm.opaque = Z_NULL;
    strm.avail_in = 0;
    strm.next_in = Z_NULL;
    if (tmp_file_path) {
        snprintf(tmpname, SM_CHUNK, "%s/sm_def_tmp%s", tmp_file_path,
                 temp_suffix);
    } else if (getenv("TMPDIR")) {
        const char *env = getenv("TMPDIR");
        snprintf(tmpname, SM_CHUNK, "%s/sm_def_tmp%s", env, temp_suffix);
    } else {
        snprintf(tmpname, SM_CHUNK, "/tmp/sm_def_tmp%s", temp_suffix);
    }

    g_debug("Input file is compressed, attempting decompression");

    fd = mkstemp(tmpname);
    if (fd == -1) {
        g_warning("Unable to open decompression tmp file '%s': %s",
                  tmpname, strerror(errno));
        return NULL;
    } else {
        dst = fdopen(fd, "wb+");
        if (!dst) {
            g_warning("Unable to open decompression tmp file '%s': %s",
                      tmpname, strerror(errno));
            return NULL;
        }
    }

    ret = inflateInit2(&strm, 16 + MAX_WBITS);
    if (ret != Z_OK) {
        return NULL;
    }
    do {
        strm.avail_in = fread(in, 1, SM_CHUNK, src);
        if (ferror(src)) {
            (void)inflateEnd(&strm);
            return NULL;
        }

        if (strm.avail_in == 0) {
            break;
        }
        strm.next_in = in;

        do {
            strm.avail_out = SM_CHUNK;
            strm.next_out = out;

            ret = inflate(&strm, Z_NO_FLUSH);
            if (ret == Z_STREAM_ERROR) { return NULL; }
            leftover = SM_CHUNK - strm.avail_out;
            if (fwrite(out, 1, leftover, dst) != leftover || ferror(dst)) {
                (void)inflateEnd(&strm);
                return NULL;
            }
        } while (strm.avail_out == 0);
    } while (ret != Z_STREAM_END);

    (void)inflateEnd(&strm);

    rewind(dst);
    unlink(tmpname);

    return dst;
}
#endif /* if SM_ENABLE_ZLIB */


void
mdCollectorSetInSpec(
    mdFlowCollector_t  *collector,
    const char         *inspec)
{
    collector->inspec = g_strdup(inspec);

    if (collector->type == TCP || collector->type == UDP) {
        collector->connspec.host = collector->inspec;
    }
}

void
mdCollectorSetPollTime(
    mdFlowCollector_t  *collector,
    const char         *poll_time)
{
    collector->poll_time = atoi(poll_time);
    collector->type = DIRECTORY;
}

void
mdCollectorSetMoveDir(
    mdFlowCollector_t  *collector,
    const char         *move_dir)
{
    collector->move_dir = g_strdup(move_dir);
}

void
mdCollectorSetLockMode(
    mdFlowCollector_t  *collector,
    gboolean            lockmode)
{
    collector->lockmode = lockmode;
}

void
mdCollectorSetPort(
    mdFlowCollector_t  *collector,
    const char         *port)
{
    collector->connspec.svc = g_strdup(port);
}

char *
mdCollectorGetName(
    md_collect_node_t  *node)
{
    return node->coll->name;
}

void
mdCollectorSetDecompressDir(
    mdFlowCollector_t  *collector,
    const char         *path)
{
    collector->decompress = g_strdup(path);
}

uint16_t
mdCollectorGetID(
    md_collect_node_t  *node)
{
    return node->coll->id;
}

void
mdCollectorSetDeleteFiles(
    mdFlowCollector_t  *collector,
    gboolean            delete)
{
    collector->delete_files = delete;
}

#if HAVE_SPREAD
void
mdCollectorAddSpreadGroup(
    mdFlowCollector_t  *collector,
    const char         *group,
    int                 group_no)
{
    if (group_no) {
        collector->spread.groups = (char **)g_renew(char *,
                                                    collector->spread.groups,
                                                    group_no + 2);
        collector->spread.groups[group_no] = g_strdup(group);
        /*collector->spread.groups[group_no+1] = '\0';*/
    } else {
        collector->spread.groups = (char **)g_new0(char *, 2);
        collector->spread.groups[0] = g_strdup(group);
        /*collector->spread.groups[1] = '\0';*/
    }
}
#endif /* if HAVE_SPREAD */

gboolean
mdCollectorSetSilkProbe(
    mdFlowCollector_t  *collector,
    const char         *probe_name,
    gboolean            use_vlan,
    GError            **err)
{
#if !defined(ENABLE_SKTYPESENSOR) || !ENABLE_SKTYPESENSOR
    MD_UNUSED_PARAM(collector);
    MD_UNUSED_PARAM(probe_name);
    MD_UNUSED_PARAM(use_vlan);
    g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                "%s was built without support for SiLK Flowsource labeling",
                g_get_prgname());
    return FALSE;
#else  /* if !defined(ENABLE_SKTYPESENSOR) || !ENABLE_SKTYPESENSOR */
    if (collector->silk_probe_name) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "COLLECTOR already has SILK_PROBE set to %s",
                    collector->silk_probe_name);
        return FALSE;
    }

    collector->silk_probe_name = g_strdup(probe_name);
    collector->silk_probe_vlan = use_vlan;

    return TRUE;
#endif  /* ENABLE_SKTYPESENSOR */
}

md_collect_node_t *
mdCollectorGetNode(
    fBuf_t  *fbuf)
{
    fbCollector_t *collector = NULL;

    collector = fBufGetCollector(fbuf);

    return (md_collect_node_t *)fbCollectorGetContext(collector);
}

gboolean
mdCollectorVerifySetup(
    mdFlowCollector_t  *collector,
    GError            **err)
{
    MD_UNUSED_PARAM(err);

    if (!collector->name) {
        collector->name = g_strdup_printf("C%d", collector->id);
    }

    switch (collector->type) {
      case SPREAD:
#if HAVE_SPREAD
        if (collector->inspec == NULL) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "Missing DAEMON name for SPREAD collector %s",
                        collector->name);
            return FALSE;
        }
#endif /* if HAVE_SPREAD */
        break;
      case UDP:
      case TCP:
        if (!collector->connspec.svc) {
            collector->connspec.svc = g_strdup_printf("18000");
        }
        break;
      case FILEHANDLER:
        if (collector->inspec == NULL) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "No input specifier for FILEHANDLER collector %s",
                        collector->name);
            return FALSE;
        }
        if (collector->move_dir) {
            g_message("MOVE for collector %s is ignored when reading"
                      " a single file", collector->name);
            g_free(collector->move_dir);
            collector->move_dir = NULL;
        }
        if (collector->delete_files) {
            g_message("DELETE for collector %s is ignored when reading"
                      " a single file", collector->name);
            collector->delete_files = FALSE;
        }
        break;
      case DIRECTORY:
        if (collector->inspec == NULL) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "No input specifier for DIRECTORY collector %s",
                        collector->name);
            return FALSE;
        }
        if (g_file_test(collector->inspec, G_FILE_TEST_EXISTS)) {
            /* Error if inspec is a directory; otherwise note that inspec is
             * an existing file which may not be what was intended */
            if (g_file_test(collector->inspec, G_FILE_TEST_IS_DIR)) {
                g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                            "Input specifier for DIRECTORY collector %s"
                            " expects a glob pattern but '%s' is a directory",
                            collector->name, collector->inspec);
                return FALSE;
            }
            g_message("Input specifier for DIRECTORY collector %s"
                      " expects a glob pattern but '%s' is an existing file;"
                      " was this intentional?",
                      collector->name, collector->inspec);
        }
        if (collector->move_dir) {
            if (collector->delete_files) {
                g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                            "Both MOVE and DELETE are present"
                            " in DIRECTORY collector %s; may use only one",
                            collector->name);
                return FALSE;
            }
        } else if (!collector->delete_files) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "Either MOVE or DELETE must be present"
                        " in DIRECTORY collector %s",
                        collector->name);
            return FALSE;
        }
      default:
        break;
    }

    return TRUE;
}

/**
 * mdFlowSourceClose
 *
 * close the file we were reading
 *
 */
static void
mdFlowSourceClose(
    mdFlowCollector_t  *collector)
{
    if (collector->lfp) {
        fclose(collector->lfp);
        collector->lfp = NULL;
    }
}


/**
 * mdFindListener
 *
 *
 */
md_collect_node_t *
mdCollectorFindListener(
    md_collect_node_t  *collector,
    fbListener_t       *listener)
{
    md_collect_node_t *cnode = NULL;

    for (cnode = collector; cnode; cnode = cnode->next) {
        if (cnode->coll->listener == listener) {
            cnode->active = TRUE;
            return cnode;
        }
    }

    return NULL;
}

/**
 * mdCollectorOpenFile
 *
 * open an IPFIX file to read
 *
 */
static fBuf_t *
mdCollectorOpenFile(
    mdFlowCollector_t  *collector,
    const char         *path,
    GError            **err)
{
    fBuf_t *buf;

    if (collector->lfp || collector->std_in) {
        /* file is already open - close it & done */
        mdFlowSourceClose(collector);
        return NULL;
    }

    if ((strlen(path) == 1) && path[0] == '-') {
        collector->collector = fbCollectorAllocFile(NULL, path, err);
        collector->std_in = TRUE;
    } else {
#if SM_ENABLE_ZLIB
        FILE *tmp = fopen(path, "rb");
        if (tmp) {
            uint16_t  header = 0;
            fread(&header, 1, 2, tmp);
            if ((header == ZLIB_HEADER) || (header == GZIP_HEADER)) {
                rewind(tmp);
                collector->lfp = mdFileDecompress(tmp, collector->decompress);
                fclose(tmp);
            } else {
                fclose(tmp);
                collector->lfp = fopen(path, "rb");
            }
        } else {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                        "%s: Cannot open file '%s' for reading: %s",
                        collector->name, path, strerror(errno));
            return NULL;
        }
#else  /* if SM_ENABLE_ZLIB */
        collector->lfp = fopen(path, "rb");
#endif /* if SM_ENABLE_ZLIB */
        if (collector->lfp == NULL) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                        "%s: Cannot open file '%s' for reading: %s",
                        collector->name, path, strerror(errno));
            return NULL;
        }

        collector->collector = fbCollectorAllocFP(NULL, collector->lfp);
    }

    if (collector->collector == NULL) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                    "Error creating the collector");
        return NULL;
    }

    collector->session = mdInitCollectorSession(err);

    if (collector->session == NULL) {
        return NULL;
    }

    buf = fBufAllocForCollection(collector->session, collector->collector);

    return buf;
}

/**
 * mdCollectorMoveFile
 *
 * move a file once we are done with it
 *
 */
static gboolean
mdCollectorMoveFile(
    const char  *file,
    const char  *new_dir,
    GError     **err)
{
    GString *new_file = NULL;
    char    *filename;

    filename = g_strrstr(file, "/");

    new_file = g_string_sized_new(PATH_MAX);

    g_string_append_printf(new_file, "%s", new_dir);
    g_string_append_printf(new_file, "%s", filename);

    if (g_rename(file, new_file->str) != 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO, "Unable to move file "
                    "to %s", new_file->str);
        g_string_free(new_file, TRUE);
        return FALSE;
    }

    g_string_free(new_file, TRUE);

    return TRUE;
}

/**
 *    Orders two mdCollectorDirEntry_t objects so the entry with the older
 *    timestamp is first.  If the objects have the same timestamp, they are
 *    ordered by strcmp().  A callback function used by GArray.
 */
static gint
mdCollectorDirEntryCompare(
    gconstpointer  v_entry_a,
    gconstpointer  v_entry_b)
{
    const mdCollectorDirEntry_t  *a, *b;

    a = (const mdCollectorDirEntry_t *)v_entry_a;
    b = (const mdCollectorDirEntry_t *)v_entry_b;

    if (a->mtime == b->mtime) {
        return strcmp(a->name, b->name);
    }
    return ((a->mtime < b->mtime) ? -1 : 1);
}


/**
 * mdCollectorFileNext
 *
 *    Does the polling for new files matching a glob pattern.
 *
 *    Thread entry point for collector->thread.
 *
 */
static void *
mdCollectorFileNext(
    void  *v_collector_node)
{

    md_collect_node_t *node = (md_collect_node_t *)v_collector_node;
    mdFlowCollector_t *collector = node->coll;
    GArray            *files = NULL;
    glob_t             gbuf;
    unsigned int       i;
    int                grc;
    gboolean           error = FALSE;
    const char        *lock_ext = ".lock";
    const size_t       lock_ext_len = strlen(lock_ext);

    /* create the array to hold file names and modification times */
    files = g_array_sized_new(FALSE, FALSE, sizeof(mdCollectorDirEntry_t),
                              256);

    memset(&gbuf, 0, sizeof(gbuf));

    while (!md_quit) {
        grc = glob(collector->inspec, 0, NULL, &gbuf);
        if (grc == GLOB_NOSPACE) {
            g_set_error(&collector->err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                        "Out of memory: glob allocation failure");
            globfree(&gbuf);
            break;
        }
        if (grc == GLOB_NOMATCH) {
            /* should already be zero */
            gbuf.gl_pathc = 0;
        }

        /* Iterate over the glob paths, storing the modification time and name
         * of files to be processed in a GArray */
        for (i = 0; i < gbuf.gl_pathc; i++) {
            mdCollectorDirEntry_t  new_entry;
            const char  *name;
            size_t       len;
            struct stat  st;
            int          rc;

            /* get path and length */
            name = gbuf.gl_pathv[i];
            len = strlen(name);

            /* stat the file, skip if a stat error or not a file */
            rc = stat(name, &st);
            if (-1 == rc) {
                /* No access to `cfg` in this function. */
                /* pthread_mutex_lock(&cfg->log_mutex); */
                /* g_warning("Unable to stat '%s': %s", */
                /*           name, strerror(errno)); */
                /* pthread_mutex_unlock(&cfg->log_mutex); */
                continue;
            }
            if (0 == (st.st_mode & S_IFREG)) {
                continue;
            }

            /* skip a file whose name ends in ".lock" */
            if (len > lock_ext_len
                && (0 == strcmp(lock_ext, (name + len - lock_ext_len))))
            {
                continue;
            }

            /* skip a file "foo" if a file "foo.lock" is also present */
            if (!collector->fname_lock) {
                collector->fname_lock = g_string_sized_new(PATH_MAX);
            }
            g_string_printf(collector->fname_lock, "%s%s", name, lock_ext);
            if (g_file_test(collector->fname_lock->str,
                            G_FILE_TEST_IS_REGULAR))
            {
                continue;
            }

            /* prep new_entry and append to array: set the file name */
            new_entry.name = name;
            /* set the new_entry's timestamp in nanoseconds if available */
#if   defined(HAVE_STRUCT_STAT_ST_MTIM_MEMBER)
            /* st_mtim - Linux, OpenBSD, FreeBSD */
            new_entry.mtime = (UINT64_C(1000000000) * st.st_mtim.tv_sec
                               + st.st_mtim.tv_nsec);
#elif defined(HAVE_STRUCT_STAT_ST_MTIMESPEC_MEMBER)
            /* sm_mtimespec - macOS */
            new_entry.mtime = (UINT64_C(1000000000) * st.st_mtimespec.tv_sec
                               + st.st_mtimespec.tv_nsec);
#else
            /* use traditional time_t */
            new_entry.mtime = (UINT64_C(1000000000) * st.st_mtime);
#endif  /* HAVE_STRUCT_STAT_ST_MTIM_MEMBER */
            /* append to the array */
            g_array_append_val(files, new_entry);
        }

        /* sort the array's entries by modification time */
        g_array_sort(files, mdCollectorDirEntryCompare);

        /* process the array's entries */
        for (i = 0; i < files->len; ++i) {
            mdCollectorDirEntry_t *entry;

            entry = &g_array_index(files, mdCollectorDirEntry_t, i);

#if 0
            /* this section repeats the tests from above in case the file's
             * status changes while building the array of mod-times.  It is
             * #if0'ed out since we assume building the array is
             * fast-enough */

            /* check the file's status again */
            if (!g_file_test(entry->name, G_FILE_TEST_IS_REGULAR)) {
                continue;
            }
            /* check again for a lock file */
            if (!collector->fname_lock) {
                collector->fname_lock = g_string_sized_new(PATH_MAX);
            }
            g_string_printf(collector->fname_lock, "%s%s",
                            entry->name, lock_ext);
            if (g_file_test(collector->fname_lock->str,
                            G_FILE_TEST_IS_REGULAR))
            {
                continue;
            }
#endif  /* 0 */

            /* store the file name on the collector */
            if (!collector->fname_in) {
                collector->fname_in = g_string_new(entry->name);
            } else {
                g_string_assign(collector->fname_in, entry->name);
            }

            /* open the file as an fBuf */
            pthread_mutex_lock(&node->mutex);
            node->fbuf = mdCollectorOpenFile(collector,
                                             collector->fname_in->str,
                                             &collector->err);
            if (!node->fbuf) {
                pthread_mutex_unlock(&node->mutex);
                /* No access to `cfg` in this function. */
                /* pthread_mutex_lock(&cfg->log_mutex); */
                /* g_warning("%s", collector->err->message); */
                /* pthread_mutex_unlock(&cfg->log_mutex); */
                error = TRUE;
                break;
            }
            node->coll->data = TRUE;
            node->stats->files++;

            /* signal to main thread we have a file */
            pthread_mutex_lock(&global_listener_mutex);
            pthread_cond_signal(&global_listener_cond);
            pthread_mutex_unlock(&global_listener_mutex);

            /* wait for main thread to finish with the file */
            while (node->fbuf) {
                pthread_cond_wait(&node->cond, &node->mutex);
            }
            pthread_mutex_unlock(&node->mutex);
        }

        /* finished with the glob data */
        globfree(&gbuf);

        /* clear all the entries in the array */
        g_array_set_size(files, 0);

        if (error) {
            break;
        }

        sleep(collector->poll_time);
    }

    /* free the array */
    g_array_free(files, TRUE);

    if (collector->fname_lock) {
        g_string_free(collector->fname_lock, TRUE);
        collector->fname_lock = NULL;
    }

    collector->active = FALSE;
    node->active = FALSE;
    pthread_mutex_lock(&global_listener_mutex);
    pthread_cond_signal(&global_listener_cond);
    pthread_mutex_unlock(&global_listener_mutex);

    /* thread has ended */
    return NULL;
}

#if HAVE_SPREAD
/**
 * mdCollectorInitSpread
 *
 * subscribe to the spread daemon for a group(s)
 *
 */
static fBuf_t *
mdCollectorInitSpread(
    mdConfig_t         *md,
    mdFlowCollector_t  *collector,
    GError            **err)
{
    fBuf_t *fbuf = NULL;

    MD_UNUSED_PARAM(md);
    collector->session = mdInitCollectorSession(err);

    if (collector->session == NULL) {
        return NULL;
    }

    collector->spread.session = collector->session;

    collector->spread.daemon = collector->inspec;

    collector->collector = fbCollectorAllocSpread(0, &(collector->spread),
                                                  err);

    if (collector->collector == NULL) {
        return NULL;
    }

    fbuf = fBufAllocForCollection(collector->session, collector->collector);

    return fbuf;
}

#endif /* if HAVE_SPREAD */

/**
 * mdCollectorOpenListener
 *
 *
 */
static void *
mdCollectorOpenListener(
    void  *data)
{
    md_collect_node_t *node = (md_collect_node_t *)data;
    mdFlowCollector_t *collector = node->coll;

    if (collector->type == UDP) {
        if (!fbListenerGetCollector(collector->listener,
                                    &collector->collector,
                                    &collector->err))
        {
            return NULL;
        }
        fbCollectorSetUDPMultiSession(collector->collector, TRUE);
    }

    while (!md_quit) {
        pthread_mutex_lock(&node->mutex);
        node->fbuf = NULL;
        node->fbuf = fbListenerWait(collector->listener, &collector->err);
        if (md_quit) {
            g_clear_error(&collector->err);
            pthread_mutex_unlock(&node->mutex);
            /* exit immediately if interrupted*/
            break;
        }
        if (node->fbuf) {
            node->coll->data = TRUE;
            fBufSetAutomaticMode(node->fbuf, FALSE);
        }
        /* signal to main thread that we have an active fbuf */
        pthread_mutex_lock(&global_listener_mutex);
        pthread_cond_signal(&global_listener_cond);
        pthread_mutex_unlock(&global_listener_mutex);

        pthread_cond_wait(&node->cond, &node->mutex);
        pthread_mutex_unlock(&node->mutex);
    }
    node->active = FALSE;
    collector->active = FALSE;

    return NULL;
}


/**
 * mdCollectorPrepareFBuf
 *
 *
 */
static gboolean
mdCollectorPrepareFBuf(
    mdConfig_t         *cfg,
    fBuf_t             *fbuf,
    mdFlowCollector_t  *collector,
    GError            **err)
{
    if (collector->type == UDP || collector->type == TCP ||
        collector->type == SPREAD)
    {
        fBufSetAutomaticMode(fbuf, FALSE);
    } else if (collector->type == FILEHANDLER) {
        pthread_mutex_lock(&cfg->log_mutex);
        g_message("%s: Opening file: %s", collector->name, collector->inspec);
        pthread_mutex_unlock(&cfg->log_mutex);
    } else if (collector->type == DIRECTORY) {
        pthread_mutex_lock(&cfg->log_mutex);
        g_message("%s: Opening file: %s", collector->name,
                  collector->fname_in->str);
        pthread_mutex_unlock(&cfg->log_mutex);
    }

    if (!fBufSetInternalTemplate(fbuf, YAF_SILK_FLOW_TID, err)) {
        return FALSE;
    }

    return TRUE;
}


/**
 * mdCollectorsInit
 *
 * open a TCP or UDP IPFIX listener via fixbuf or open a file
 *
 */
gboolean
mdCollectorsInit(
    mdConfig_t         *md,
    md_collect_node_t  *collector,
    GError            **err)
{
    md_collect_node_t *cnode = NULL;

    MD_UNUSED_PARAM(md);

    for (cnode = collector; cnode; cnode = cnode->next) {
        if (cnode->coll->type == TCP || cnode->coll->type == UDP) {
            cnode->coll->session = mdInitCollectorSession(err);

            if (cnode->coll->session == NULL) {
                return FALSE;
            }

            cnode->coll->listener = fbListenerAlloc(&(cnode->coll->connspec),
                                                    cnode->coll->session,
                                                    mdListenerConnect, NULL,
                                                    err);

            if (cnode->coll->listener == NULL) {
                return FALSE;
            }

            pthread_mutex_init(&cnode->mutex, NULL);
            pthread_cond_init(&cnode->cond, NULL);
        } else if (cnode->coll->type == FILEHANDLER) {
            cnode->fbuf = mdCollectorOpenFile(cnode->coll, cnode->coll->inspec,
                                              err);
            if (cnode->fbuf == NULL) {
                return FALSE;
            }
            /* set active here because we don't start up a thread for files */
            cnode->active = TRUE;
            cnode->coll->data = TRUE;
            pthread_mutex_init(&cnode->mutex, NULL);
            pthread_cond_init(&cnode->cond, NULL);
        } else if (cnode->coll->type == SPREAD) {
#if HAVE_SPREAD
            cnode->coll->listener =
                (fbListener_t *)mdCollectorInitSpread(md, cnode->coll, err);
            if (cnode->coll->listener == NULL) {
                return FALSE;
            }

            pthread_mutex_init(&cnode->mutex, NULL);
            pthread_cond_init(&cnode->cond, NULL);
#endif /* if HAVE_SPREAD */
        } else if (cnode->coll->type == DIRECTORY) {
            pthread_mutex_init(&cnode->mutex, NULL);
            pthread_cond_init(&cnode->cond, NULL);
        }

        cnode->stats = g_slice_new0(md_stats_t);
    }

    return TRUE;
}

static int
mdOpenCollectors(
    md_collect_node_t  *collector)
{
    md_collect_node_t *cnode = NULL;
    int                active = 0;

    for (cnode = collector; cnode; cnode = cnode->next) {
        if (cnode->active) {
            active++;
        }
    }

    return active;
}

/**
 * mdCollectFBuf()
 *
 *  The primary mechanism to collect and export flows.
 */
static gboolean
mdCollectFBuf(
    mdContext_t        *ctx,
    md_collect_node_t  *collector,
    GError            **err)
{
    uint16_t            tid;
    mdFullFlow_t        md_flow;
    md_main_template_t  ipfixFullFlow;
    size_t              length;
    fbSession_t        *session;
    gboolean            reset, rv, rc;
    fbTemplate_t       *tmpl = NULL;
    mdTmplContext_t    *tmpl_ctx = NULL;

    memset(&ipfixFullFlow, 0, sizeof(ipfixFullFlow));

    rv = TRUE;

    reset = FALSE;

    if (collector->coll->data == FALSE) {
        /* no data yet - don't call mdOptionsCheck */
        return TRUE;
    }

    if (!mdCollectorPrepareFBuf(ctx->cfg, collector->fbuf, collector->coll,
                                err))
    {
        return FALSE;
    }

    /* set the current collector's name */
    ctx->cfg->collector_name = collector->coll->name;
    ctx->cfg->collector_id = collector->coll->id;
#if ENABLE_SKTYPESENSOR
    ctx->cfg->collector_silk_probe_name = collector->coll->silk_probe_name;
    ctx->cfg->collector_silk_probe_vlan = collector->coll->silk_probe_vlan;
#endif

    for (;;) {
        /*
         *  Get the template for the next record from fixbuf.
         *
         *  A return value of TRUE with a non-zero `tid` indicates the next
         *  record is an Options record.  TRUE with a `tid` of zero indicates
         *  an ignored record (e.g., metadata) or a transitory error.
         *
         *  A return value of FALSE with a non-NULL `fbuf` is a valid
         *  non-Options record.  FALSE with a NULL `fbuf` indicates end of
         *  stream or a fatal error.
         */
        if (mdOptionsCheck(&(collector->fbuf), &tid, &tmpl, &tmpl_ctx, err)) {
            if (tid != 0) {
                pthread_mutex_lock(&ctx->cfg->log_mutex);
                if (!mdForwardOptions(ctx, collector, tid, err)) {
                    g_warning("Error Forwarding Options:");
                    g_warning("Error: %s", (*err)->message);
                    rv = FALSE;
                    pthread_mutex_unlock(&ctx->cfg->log_mutex);
                    goto end;
                }

                pthread_mutex_unlock(&ctx->cfg->log_mutex);
            } else {
                if (g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_IPFIX)) {
                    pthread_mutex_lock(&ctx->cfg->log_mutex);
                    g_warning("%s: Ignoring Packet: %s", collector->coll->name,
                              (*err)->message);
                    pthread_mutex_unlock(&ctx->cfg->log_mutex);
                    collector->fbuf = NULL;
                } else if (g_error_matches(*err, FB_ERROR_DOMAIN,
                                           FB_ERROR_NLREAD))
                {
                    pthread_mutex_lock(&ctx->cfg->log_mutex);
                    g_warning("%s: Ignoring Connection: %s",
                              collector->coll->name, (*err)->message);
                    pthread_mutex_unlock(&ctx->cfg->log_mutex);
                    collector->fbuf = NULL;
                } else if (g_error_matches(*err, MD_ERROR_DOMAIN,
                                           MD_ERROR_TMPL))
                {
                    pthread_mutex_lock(&ctx->cfg->log_mutex);
                    mdIgnoreRecord(ctx, collector->fbuf, err);
                    /* g_warning("%s: Ignoring Options Record: %s", */
                    /*           collector->coll->name, (*err)->message); */

                    pthread_mutex_unlock(&ctx->cfg->log_mutex);
                    g_clear_error(err);
                    continue;
                }
                g_clear_error(err);
                collector->coll->data = FALSE;
                goto end;
            }
            continue;
        }

        if (collector->fbuf == NULL) {
            pthread_mutex_lock(&ctx->cfg->log_mutex);
            if (!(g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOF))) {
                g_warning("%s: Closing Connection: %s",
                          collector->coll->name, (*err)->message);
            } else {
                g_message("%s: Closing Connection: %s",
                          collector->coll->name, (*err)->message);
            }
            if (collector->coll->type == SPREAD) {
                collector->coll->listener = NULL;
                g_message("%s: Retrying connection", collector->coll->name);
            }
            collector->coll->data = FALSE;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            g_clear_error(err);
            rv = TRUE;
            reset = TRUE;
            break;
        }

        /* have a valid non-Options record */

        pthread_mutex_lock(&ctx->cfg->log_mutex);

        session = fBufGetSession(collector->fbuf);
        ctx->cfg->current_domain = fbSessionGetDomain(session);

        if (tmpl_ctx->is_dedup) {
            rc = mdForwardDedupCustom(ctx, tmpl_ctx, collector->fbuf, err);
            if (!rc) {
                g_warning("%s: Error Forwarding DEDUP Rec:",
                          collector->coll->name);
                g_warning("Error: %s", (*err)->message);
                rv = FALSE;
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                goto end;
            }
            collector->stats->nonstd_flows++;
            ctx->stats->nonstd_flows++;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            continue;
        } else if (MD_CHECK_TID_IS_DNS_DEDUP(tid)) {
            rc = mdForwardDNSDedup(ctx, tmpl_ctx, collector->fbuf, err);
            if (!rc) {
                g_warning("%s: Error Forwarding DNS Dedup Rec:",
                          collector->coll->name);
                g_warning("Error: %s", (*err)->message);
                rv = FALSE;
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                goto end;
            }
            collector->stats->nonstd_flows++;
            ctx->stats->nonstd_flows++;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            continue;
        } else if (MD_CHECK_TID_IS_DNSRR(tid)) {
            rc = mdForwardDNSRR(ctx, tmpl_ctx, collector->fbuf, err);
            if (!rc) {
                g_warning("%s: Error Forwarding DNS RR Rec:",
                          collector->coll->name);
                g_warning("Error: %s", (*err)->message);
                rv = FALSE;
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                goto end;
            }
            collector->stats->nonstd_flows++;
            ctx->stats->nonstd_flows++;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            continue;
        } else if (MD_CHECK_TID_IS_SSL_DEDUP(tid)) {
            rc = mdForwardSSLDedup(ctx, tmpl_ctx, collector->fbuf, err);
            if (!rc) {
                g_warning("%s: Error Forwarding SSL DEDUP Rec:",
                          collector->coll->name);
                g_warning("Error: %s", (*err)->message);
                rv = FALSE;
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                goto end;
            }
            collector->stats->nonstd_flows++;
            ctx->stats->nonstd_flows++;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            continue;
        } else if (MD_CHECK_TID_IS_DEDUP(tid)) {
            rc = mdForwardDedup(ctx, tmpl_ctx, collector->fbuf, err);
            if (!rc) {
                g_warning("%s: Error Forwarding DEDUP Rec:",
                          collector->coll->name);
                g_warning("Error: %s", (*err)->message);
                rv = FALSE;
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                goto end;
            }
            collector->stats->nonstd_flows++;
            ctx->stats->nonstd_flows++;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            continue;
        } else if (tid == YAF_SSL_CERT_TID) {
            rc = mdForwardSSLCert(ctx, tmpl_ctx, collector->fbuf, err);
            if (!rc) {
                g_warning("%s: Error Forwarding SSL CERT Rec:",
                          collector->coll->name);
                g_warning("Error: %s", (*err)->message);
                rv = FALSE;
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                goto end;
            }
            collector->stats->nonstd_flows++;
            ctx->stats->nonstd_flows++;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            continue;
        }
        /* else, assume it is a flow record */

        memset(&md_flow, 0, sizeof(md_flow));
        md_flow.rec = &ipfixFullFlow;

        length = sizeof(*md_flow.rec);
        rc = fBufNext(collector->fbuf, (uint8_t *)md_flow.rec, &length,
                      err);
        if (FALSE == rc) {
            if (!(g_error_matches(*err, FB_ERROR_DOMAIN, FB_ERROR_EOM))) {
                mdfBufFree(collector);
                g_warning("Error Receiving Flow %s", (*err)->message);
                reset = TRUE;
                rv = FALSE;
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                collector->coll->data = FALSE;
                break;
            }
            g_clear_error(err);
            rv = TRUE;
            collector->coll->data = FALSE;
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            break;
        }
        ctx->stats->recvd_flows++;

        if (tmpl_ctx->attr.has_nano && md_flow.rec->flowEndNanoseconds) {
            yfTimeFromNTP(&md_flow.stime, md_flow.rec->flowStartNanoseconds,
                          FALSE);
            yfTimeFromNTP(&md_flow.etime, md_flow.rec->flowEndNanoseconds,
                          FALSE);
        } else if (tmpl_ctx->attr.has_micro
                   && md_flow.rec->flowEndMicroseconds)
        {
            yfTimeFromNTP(&md_flow.stime, md_flow.rec->flowStartMicroseconds,
                          TRUE);
            yfTimeFromNTP(&md_flow.etime, md_flow.rec->flowEndMicroseconds,
                          TRUE);
        } else if (tmpl_ctx->attr.has_milli) {
            yfTimeFromMilli(&md_flow.stime, md_flow.rec->flowStartMilliseconds);
            yfTimeFromMilli(&md_flow.etime, md_flow.rec->flowEndMilliseconds);
        } else {
            g_debug("No time values found for record using tmpl %#06x", tid);
        }

        /* update current time */
        if (yfTimeCmpOp(md_flow.etime, ctx->cfg->ctime, >)) {
            ctx->cfg->ctime = md_flow.etime;
        }

        md_flow.tid = tid;
        md_flow.tmpl_attr = tmpl_ctx->attr;

        /* check whether the shared filter or the collector's filter rejects
         * this flow */
        if ((ctx->cfg->shared_filter
             && FALSE == mdFilter(ctx->cfg->shared_filter, &md_flow,
                                  ctx->cfg->current_domain,
                                  ctx->cfg->shared_filter_and, 0)) ||
            (collector->filter
             && FALSE == mdFilter(collector->filter, &md_flow,
                                  ctx->cfg->current_domain,
                                  collector->and_filter, 0)))
        {
            ctx->stats->recvd_filtered++;
            collector->stats->recvd_filtered++;
            mdDecodeAndClear(ctx, &md_flow);
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            continue;
        }

        /* only count flows that the collector actually receives */
        collector->stats->recvd_flows++;

        /* mdForwardFlow is where the majority of non-special case exporters
         * will begin to emit the flow. */
        if (!mdForwardFlow(ctx, &md_flow, err)) {
            g_warning("Error Forwarding Flow...");
            g_warning("Error: %s", (*err)->message);
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            rv = FALSE;
            goto end;
        }

        pthread_mutex_unlock(&ctx->cfg->log_mutex);
    }

    if (reset) {
        if (!mdExporterConnectionReset(ctx->cfg, err)) {
            pthread_mutex_lock(&ctx->cfg->log_mutex);
            g_warning("Error resetting %s\n", (*err)->message);
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            rv = FALSE;
        }
    }

  end:
    if (collector->coll->type == DIRECTORY) {
        /* Delete or move file for DIRECTORY collectors */
        if (collector->coll->lfp) {
            const char *action = "Deleting";
            if (collector->coll->move_dir && collector->coll->fname_in) {
                action = "Moving";
                if (!mdCollectorMoveFile(collector->coll->fname_in->str,
                                         collector->coll->move_dir, err))
                {
                    g_string_free(collector->coll->fname_in, TRUE);
                    collector->coll->fname_in = NULL;
                    return FALSE;
                }
            }
            mdFlowSourceClose(collector->coll);
            g_remove(collector->coll->fname_in->str);

            pthread_mutex_lock(&ctx->cfg->log_mutex);
            g_message("%s file %s", action, collector->coll->fname_in->str);
            pthread_mutex_unlock(&ctx->cfg->log_mutex);
            g_string_free(collector->coll->fname_in, TRUE);
            collector->coll->fname_in = NULL;
        }
    }

    return rv;
}


/**
 *  mdCollectorWait()
 *
 *  Overall thread and collector handling loop.  Data/flow processing loop
 *  is called below in mdCollectFBuf()
 */
gboolean
mdCollectorWait(
    mdContext_t  *ctx,
    GError      **err)
{
    md_collect_node_t *clist = ctx->cfg->flowsrc;
    md_collect_node_t *cnode = NULL;
    gboolean           active;
    int                rv;
    int                collectors = 0;
    struct timeval     tp;
    struct timespec    to;

    collectors = mdOpenCollectors(clist);

    if (!collectors) {return FALSE;}

    while (collectors && !md_quit) {
        active = FALSE;
        for (cnode = ctx->cfg->flowsrc; cnode; cnode = cnode->next) {
            if (cnode->active) {
                if (cnode->fbuf && cnode->coll->data) {
                    active = TRUE;
                    rv = pthread_mutex_trylock(&cnode->mutex);
                    if (rv != 0) {continue;}
                    if (!mdCollectFBuf(ctx, cnode, err)) {
                        pthread_cond_signal(&cnode->cond);
                        pthread_mutex_unlock(&cnode->mutex);
                        return FALSE;
                    }
                    if (cnode->coll->type == FILEHANDLER) {
                        cnode->active = FALSE;
                        cnode->coll->restart = TRUE;
                        collectors--;
                    }
                    pthread_cond_signal(&cnode->cond);
                    pthread_mutex_unlock(&cnode->mutex);
                }
            } else if (!cnode->coll->restart) {
                /* start this guy back up */
                pthread_mutex_lock(&ctx->cfg->log_mutex);
                g_warning("Collector \"%s\" Error: %s",
                          cnode->coll->name, cnode->coll->err->message);
                cnode->coll->restart = TRUE;
                g_clear_error(&cnode->coll->err);
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                if (!mdCollectorRestartListener(ctx->cfg, cnode, err)) {
                    pthread_mutex_lock(&ctx->cfg->log_mutex);
                    g_warning("Error restarting collector %s: %s",
                              cnode->coll->name, (*err)->message);
                    pthread_mutex_unlock(&ctx->cfg->log_mutex);
                    g_clear_error(err);
                    collectors--;
                }
            } else if (cnode->coll->err) {
                /* this collector is permanently inactive */
                pthread_mutex_lock(&ctx->cfg->log_mutex);
                g_warning("Could not restart collector %s: %s",
                          cnode->coll->name, cnode->coll->err->message);
                g_clear_error(&cnode->coll->err);
                g_warning(
                    "Collector \"%s\" is now inactive until program restart.",
                    cnode->coll->name);
                pthread_mutex_unlock(&ctx->cfg->log_mutex);
                collectors--;
            }
        }
        if (!active) {
            /* wait on listeners to collect something (timeout of 1 sec)*/
            gettimeofday(&tp, NULL);
            to.tv_sec = tp.tv_sec + 1;
            to.tv_nsec = tp.tv_usec * 1000;
            pthread_mutex_lock(&global_listener_mutex);
            pthread_cond_timedwait(&global_listener_cond,
                                   &global_listener_mutex, &to);
            pthread_mutex_unlock(&global_listener_mutex);
        }
    }

    return FALSE;
}


#if HAVE_SPREAD
static void *
mdCollectorSpreadSubscribe(
    void  *data)
{
    md_collect_node_t *node = (md_collect_node_t *)data;
    mdFlowCollector_t *collector = node->coll;
    fbTemplate_t      *tmpl = NULL;
    uint16_t           tid;
    int                restarts = 0;

    while (!md_quit) {
        pthread_mutex_lock(&node->mutex);

        if ((collector->listener == NULL) && (node->fbuf == NULL)) {
            /* Some type of error occurred (connection down?) */
            collector->restart = TRUE;
        }

        if (collector->restart) {
            /* try to connect to the spread daemon 10 times -
             * after that forget it. */
            while (restarts < 10) {
                collector->listener = (fbListener_t *)mdCollectorInitSpread(
                    NULL, collector, &collector->err);
                if (collector->listener) {
                    collector->restart = FALSE;
                    break;
                }
                sleep(30);
                restarts++;
                if (restarts == 5) {
                    collector->active = FALSE;
                    node->active = FALSE;
                    pthread_mutex_unlock(&node->mutex);
                    pthread_mutex_lock(&global_listener_mutex);
                    pthread_cond_signal(&global_listener_cond);
                    pthread_mutex_unlock(&global_listener_mutex);
                    return NULL;
                }
                g_clear_error(&collector->err);
            }
        }

        /* Spread is weird bc there is no listener. It basically hangs out
         * in read until a message arrives, we don't want to use fBufNext
         * because that will advance the buffer and we'll miss a record
         * when we call fBufNext again */
        tmpl = fBufNextCollectionTemplate((fBuf_t *)node->coll->listener,
                                          &tid, &collector->err);

        if (!tmpl) {
            /* ignore eom */
            if (g_error_matches(collector->err, FB_ERROR_DOMAIN,
                                FB_ERROR_EOM))
            {
                g_clear_error(&collector->err);
                pthread_mutex_unlock(&node->mutex);
                continue;
            } else if (g_error_matches(collector->err, FB_ERROR_DOMAIN,
                                       FB_ERROR_IO))
            {
                fBufFree((fBuf_t *)node->coll->listener);
                node->fbuf = NULL;
                pthread_mutex_unlock(&node->mutex);
                break;
            } else if (g_error_matches(collector->err, FB_ERROR_DOMAIN,
                                       FB_ERROR_CONN))
            {
                fBufFree((fBuf_t *)node->coll->listener);
                node->fbuf = NULL;
                pthread_mutex_unlock(&node->mutex);
                break;
            }
        }

        node->fbuf = (fBuf_t *)node->coll->listener;
        node->coll->data = TRUE;
        if (md_quit) {
            /* exit immediately if interrupted*/
            break;
        }

        /* signal to main thread that we have an active fbuf */
        pthread_mutex_lock(&global_listener_mutex);
        pthread_cond_signal(&global_listener_cond);
        pthread_mutex_unlock(&global_listener_mutex);

        pthread_cond_wait(&node->cond, &node->mutex);
        pthread_mutex_unlock(&node->mutex);
    }

    collector->active = FALSE;
    node->active = FALSE;
    pthread_mutex_lock(&global_listener_mutex);
    pthread_cond_signal(&global_listener_cond);
    pthread_mutex_unlock(&global_listener_mutex);

    return NULL;
}
#endif /* if HAVE_SPREAD */

/**
 * mdCollectorRestartListener
 *
 */
gboolean
mdCollectorRestartListener(
    mdConfig_t         *md,
    md_collect_node_t  *collector,
    GError            **err)
{
    md_collect_node_t *cnode = collector;

    if (cnode->active) {
        return TRUE;
    }

    if (cnode->coll->type == FILEHANDLER) {
        /* file is already open */
        return TRUE;
    }
    if (cnode->coll->type == DIRECTORY) {
        pthread_mutex_lock(&md->log_mutex);
        g_message("%s: Restarting Directory Poller on %s",
                  cnode->coll->name, cnode->coll->inspec);
        pthread_mutex_unlock(&md->log_mutex);
        cnode->coll->active = TRUE;
        if (pthread_create(&(cnode->coll->thread), NULL,
                           mdCollectorFileNext, cnode))
        {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                        "Couldn't open polling thread.");
            return FALSE;
        }
    } else if (cnode->coll->type == SPREAD) {
#if HAVE_SPREAD
        cnode->coll->active = TRUE;
        pthread_mutex_lock(&md->log_mutex);
        g_message("%s: Restarting Spread Collector for Daemon %s",
                  cnode->coll->name, cnode->coll->inspec);
        pthread_mutex_unlock(&md->log_mutex);
        if (pthread_create(&(cnode->coll->thread), NULL,
                           mdCollectorSpreadSubscribe, cnode))
        {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                        "Couldn't open Spread Subscriber.");
            return FALSE;
        }
#endif /* if HAVE_SPREAD */
    } else {
        cnode->coll->active = TRUE;
        pthread_mutex_lock(&md->log_mutex);
        g_message("%s: Restarting Listener on %s:%s",
                  cnode->coll->name, cnode->coll->connspec.host,
                  cnode->coll->connspec.svc);
        pthread_mutex_unlock(&md->log_mutex);

        if (pthread_create(&(cnode->coll->thread), NULL,
                           mdCollectorOpenListener, cnode))
        {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                        "Couldn't open listening thread.");
            return FALSE;
        }
    }
    cnode->active = TRUE;

    return TRUE;
}



/**
 * mdCollectorStartListeners
 *
 */
gboolean
mdCollectorStartListeners(
    mdConfig_t         *md,
    md_collect_node_t  *collector,
    GError            **err)
{
    md_collect_node_t *cnode = NULL;

    for (cnode = collector; cnode; cnode = cnode->next) {
        if (!cnode->active) {
            if (cnode->coll->type == FILEHANDLER) {
                /* file is already open */
                continue;
            }
            if (cnode->coll->type == DIRECTORY) {
                pthread_mutex_lock(&md->log_mutex);
                g_message("%s: Starting Directory Poller on '%s'",
                          cnode->coll->name, cnode->coll->inspec);
                pthread_mutex_unlock(&md->log_mutex);
                cnode->coll->active = TRUE;
                if (pthread_create(&(cnode->coll->thread), NULL,
                                   mdCollectorFileNext, cnode))
                {
                    g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                                "Couldn't open polling thread.");
                    return FALSE;
                }
            } else if (cnode->coll->type == SPREAD) {
#if HAVE_SPREAD
                cnode->coll->active = TRUE;
                pthread_mutex_lock(&md->log_mutex);
                g_message("%s: Starting Spread Collector for Daemon %s",
                          cnode->coll->name, cnode->coll->inspec);
                pthread_mutex_unlock(&md->log_mutex);
                if (pthread_create(&(cnode->coll->thread), NULL,
                                   mdCollectorSpreadSubscribe, cnode))
                {
                    g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                                "Couldn't open Spread Subscriber.");
                    return FALSE;
                }
#endif /* if HAVE_SPREAD */
            } else {
                cnode->coll->active = TRUE;
                pthread_mutex_lock(&md->log_mutex);
                g_message("%s: Starting Listener on %s:%s",
                          cnode->coll->name, cnode->coll->connspec.host,
                          cnode->coll->connspec.svc);
                pthread_mutex_unlock(&md->log_mutex);
                if (pthread_create(&(cnode->coll->thread), NULL,
                                   mdCollectorOpenListener, cnode))
                {
                    g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_IO,
                                "Couldn't open listening thread.");
                    return FALSE;
                }
            }
            cnode->active = TRUE;
        }
    }

    return TRUE;
}



/**
 * mdDestroyCollector
 *
 * remove the collector
 *
 */
static void
mdDestroyCollector(
    mdFlowCollector_t  *coll,
    gboolean            active)
{
#if HAVE_SPREAD
    int  n = 0;
#endif

    if (coll->type == TCP || coll->type == UDP) {
        if (coll->session) {
            fbSessionFree(coll->session);
        }
    }

    if (coll->lfp) {
        mdFlowSourceClose(coll);
    }

    if (coll->inspec) {
        g_free(coll->inspec);
    }

    if (coll->decompress) {
        g_free(coll->decompress);
    }

    if (coll->type == TCP || coll->type == UDP) {
        g_free(coll->connspec.svc);
    }

    if (coll->move_dir) {
        g_free(coll->move_dir);
    }

    if (coll->active) {
        if (pthread_cancel(coll->thread)) {
            fprintf(stderr, "Error canceling %s collector thread\n",
                    coll->name);
        }
    }

    /* if destroyed due to presence of command line options,
     * calling pthread_join will cause segfault. */
    if (coll->type != FILEHANDLER && active) {
        pthread_join(coll->thread, NULL);
    }

    if (coll->name) {
        g_free(coll->name);
    }

#if HAVE_SPREAD
    if (coll->spread.session) {
        while (coll->spread.groups[n] && *coll->spread.groups[n]) {
            g_free(coll->spread.groups[n]);
            n++;
        }
        g_free(coll->spread.groups);
    }
#endif /* if HAVE_SPREAD */
}

static void
mdCollectorFree(
    mdFlowCollector_t  *collector)
{
    if (collector && collector->fname_in) {
        g_string_free(collector->fname_in, TRUE);
        collector->fname_in = NULL;
    }
    g_slice_free(mdFlowCollector_t, collector);
}


void
mdCollectorDestroy(
    mdConfig_t  *cfg,
    gboolean     active)
{
    md_collect_node_t *cnode = NULL;
    md_filter_t       *fnode;

    mdCollectorUpdateStats(cfg);

    for (cnode = cfg->flowsrc; cnode; cnode = cnode->next) {
        if (cnode->coll->active) {
            pthread_cond_signal(&cnode->cond);
        }
        mdDestroyCollector(cnode->coll, active);

        if (cnode->fbuf) {
            fBufFree(cnode->fbuf);
        }

        /* this collector's listener could be NULL when called during
         * configuration to deconflict CLI options and config file collectors.
         */
        if (cnode->coll->type == TCP && cnode->coll->listener != NULL) {
            fbListenerFree(cnode->coll->listener);
        }

        pthread_cond_destroy(&cnode->cond);
        pthread_mutex_destroy(&cnode->mutex);

        /* free the filters from the collector */
        while (cnode->filter) {
            detachHeadOfSLL((mdSLL_t **)&(cnode->filter),
                            (mdSLL_t **)&fnode);
            md_free_filter_node(fnode);
        }
        mdCollectorFree(cnode->coll);
    }

    /* Now free the shared filters */
    while (cfg->shared_filter) {
        detachHeadOfSLL((mdSLL_t **)&(cfg->shared_filter),
                        (mdSLL_t **)&fnode);
        md_free_filter_node(fnode);
    }
    cfg->shared_filter = NULL;

    /* free md_collect_node_t */
}


void
mdCollectorUpdateStats(
    mdConfig_t  *cfg)
{
    md_collect_node_t *cnode = NULL;
    char               active[10];

    for (cnode = cfg->flowsrc; cnode; cnode = cnode->next) {
        if (cnode->active) {
            active[0] = '\0';
        } else {
            sprintf(active, "INACTIVE ");
        }
        if (cnode->stats == NULL) {
            continue;
        }
        if (cnode->coll->type == FILEHANDLER ||
            cnode->coll->type == DIRECTORY)
        {
            g_message(
                ("%sCollector %s: %" PRIu64 " flows, %" PRIu64 " other flows, "
                 "%" PRIu64 " stats, %" PRIu64 " filtered, %u files"),
                active, cnode->coll->name, cnode->stats->recvd_flows,
                cnode->stats->nonstd_flows,
                cnode->stats->recvd_stats, cnode->stats->recvd_filtered,
                cnode->stats->files);
        } else if (cnode->coll->type == SPREAD) {
            g_message(
                ("%sCollector %s: %" PRIu64 " flows, %" PRIu64 " other flows, "
                 "%" PRIu64 " stats, %" PRIu64 " filtered"),
                active, cnode->coll->name, cnode->stats->recvd_flows,
                cnode->stats->nonstd_flows,
                cnode->stats->recvd_stats, cnode->stats->recvd_filtered);
        } else {
            g_message(
                ("%sCollector %s: %" PRIu64 " flows, %" PRIu64 " other flows, "
                 "%" PRIu64 " stats, %" PRIu64 " filtered, %d connections"),
                active, cnode->coll->name, cnode->stats->recvd_flows,
                cnode->stats->nonstd_flows,
                cnode->stats->recvd_stats, cnode->stats->recvd_filtered,
                cnode->stats->restarts);
        }
    }
}
