/*
 *  Copyright 2012-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  mediator.c
 *
 *  IPFIX mediator for filtering, DNS deduplication, and other mediator-like
 *  things.  Contains main() and contains functions for program set-up.
 *
 *  ------------------------------------------------------------------------
 *  Authors: Emily Sarneso
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.13
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1447
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

#define MEDIATOR_MAIN_SOURCE 1
#include "mediator.h"
#include "mediator_inf.h"
#include "mediator_core.h"
#include "mediator_filter.h"
#include "mediator_dns.h"
#include "mediator_stat.h"
#include "mediator_log.h"
#include "mediator_dedup.h"
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_GRP_H
#include <grp.h>
#endif

#if ENABLE_SKTYPESENSOR
#include <silk/utils.h>
#include "probeconf.h"
#elif ENABLE_SKPREFIXMAP || (ENABLE_SKIPSET && HAVE_SILK_UTILS_H)
#include <silk/utils.h>
#endif /* if ENABLE_SKTYPESENSOR */

/*#define MDLOG_TESTING_LOG 0*/

/* Maximum timeout (in seconds) for flushing internal state (5 min) */
#define TIMEOUT   300

/* Maximum --sleep (sleep_usecs) value is 1 second */
#define SLEEP_USECS_MAX  1000000

/* Default UDP template timeout in seconds (10 min).  The templates are resent
 * three times per timeout interval. */
#define DEFAULT_UDP_TEMP_TIMEOUT  600

/* Default read (input) and write (output) ports */
#define DEFAULT_MD_IPORT  "18000"
#define DEFAULT_MD_EPORT  "18001"

/* Default polling interval (--watch) when file-glob polling is enabled
 * because --move is specified without --watch. */
#define DEFAULT_POLL_TIME "30"

mdConfig_t                md_config;
int                       md_quit = 0;
char                     *md_logfile = NULL;
char                     *md_logdest = NULL;
char                     *md_pidfile = NULL;
int                       md_stats_timeout = TIMEOUT;
int                       udp_temp_timeout = 0;
uint16_t                  dns_flush_timeout = MD_DNS_DEDUP_DEFAULT_FLUSH;
gboolean                  dns_dedup = FALSE;
mdLogLevel_t              md_log_level = WARNING;
unsigned int              md_top_level_time_units = MD_TUNIT_UNSET;
fbInfoElement_t          *user_elements = NULL;
#if HAVE_SPREAD
static char              *in_groups;
char                    **md_out_groups;
int                       num_out_groups = 0;
#endif  /* HAVE_SPREAD */
#if ENABLE_SKIPSET || ENABLE_SKPREFIXMAP || ENABLE_SKTYPESENSOR
int                       app_registered = 0;
#endif

static char              *last_logfile = NULL;
static char              *cur_logfile = NULL;
static time_t             md_log_rolltime;
static uint16_t           dns_max_hit_count = MD_DNS_DEDUP_DEFAULT_HIT;
static mdTransportType_t  md_ipfix_intransport = NONE;

static gboolean           md_daemon = FALSE;
static char              *md_iport = NULL;
static char              *md_eport = NULL;
static char              *md_conf = NULL;
static char              *md_in_str = NULL;
static char              *md_out_str = NULL;
static char              *md_field_list;
static char              *md_time_units_arg = NULL;
static int                rotate = 0;
static int                sleep_usecs = 0;
static volatile int       shutdown_signal = 0;
static char              *poll_time = NULL;
static mdTransportType_t  md_ipfix_outtransport = NONE;
static char              *outspec = NULL;
static char              *inspec = NULL;
static char              *move_dir = NULL;
static gboolean           delete_inspec = FALSE;
static GIOChannel        *logfile;
static GLogLevelFlags     level;
static gboolean           md_verbose = FALSE;
static gboolean           md_quiet = FALSE;
static gboolean           md_test_config = FALSE;
static gboolean           md_print_headers = FALSE;
static char              *become_user = NULL;
static char              *become_group = NULL;
static uid_t              new_user = 0;
static gid_t              new_group = 0;
static gboolean           did_become = FALSE;
static gboolean           md_metadata_export = FALSE;


static void
mdPrintVersion(
    const gchar  *option_name,
    const gchar  *option_arg,
    gpointer      v_context,
    GError      **error);

#define MD_HELP_SUMMARY                                 \
    "Reads IPFIX by listening on sockets, by"           \
    " periodically matching a file glob,\n"             \
    "or by reading a file. Writes IPFIX to"             \
    " sockets or writes IPFIX, JSON, or TEXT\n"         \
    "to a file or to rotating files in"                 \
    " a directory. The default is to read from\n"       \
    "stdin and write to stdout."

#define WRAP_OPT "\n                "
static GOptionEntry  md_core_option[] = {
    {"config", 'c', 0, G_OPTION_ARG_STRING, &md_conf,
     WRAP_OPT "Specifies the path to the optional configuration file",
     "file_name"},
    {"in", 'i', 0, G_OPTION_ARG_STRING, &inspec,
     WRAP_OPT "Specifies the file to read, file glob to poll, Spread daemon"
     WRAP_OPT "to contact, or host/IP-address to listen on [-]",
     "file|glob-pattern|host|IP"},
    {"ipfix-input", 0, 0, G_OPTION_ARG_STRING, &md_in_str,
     WRAP_OPT "Enables input over the network via the specified the mode of"
     WRAP_OPT "transport: TCP,UDP,SPREAD. --in must name a host or IP address"
     WRAP_OPT "or Spread daemon",
     "mode"},
    {"ipfix-port", 'p', 0, G_OPTION_ARG_STRING, &md_iport,
     WRAP_OPT "When --ipfix-input is TCP or UDP, changes the listening port"
     WRAP_OPT "from the default [18000]",
     "port"},
    {"watch", 'w', 0, G_OPTION_ARG_STRING, &poll_time,
     WRAP_OPT "Enables polling for files matching the file glob pattern set"
     WRAP_OPT "by --in and sets the polling interval. Requires use of --move"
     WRAP_OPT "or --delete",
     "seconds"},
    {"move", 0, 0, G_OPTION_ARG_STRING, &move_dir,
     WRAP_OPT "Enables polling for files matching the file glob pattern set"
     WRAP_OPT "by --in and names the directory where the files are moved"
     WRAP_OPT "after processing",
     "dir_path"},
    {"delete", 0, 0, G_OPTION_ARG_NONE, &delete_inspec,
     WRAP_OPT "When polling for files matching a file glob pattern, enables"
     WRAP_OPT "deleting the input files after processing",
     NULL},
    {"lock", 0, 0, G_OPTION_ARG_NONE, &md_config.lockmode,
     WRAP_OPT "When polling for files matching a file glob pattern, disables"
     WRAP_OPT "reading of files that are locked",
     NULL},
#if HAVE_SPREAD
    {"groups", 'g', 0, G_OPTION_ARG_STRING, &in_groups,
     WRAP_OPT "Sets the names of the Spread groups to subscribe to",
     "groups"},
#endif
    {"out", 'o', 0, G_OPTION_ARG_STRING, &outspec,
     WRAP_OPT "Names the file to create, path-and-file prefix for multiple"
     WRAP_OPT "output files, or host/IP-address to receive output [-]",
     "file|path-prefix|host|IP"},
    {"output-mode", 'm', 0, G_OPTION_ARG_STRING, &md_out_str,
     WRAP_OPT "Selects the mode of export. For file-based export, specify"
     WRAP_OPT "TEXT or JSON to override IPFIX. For network export, specify"
     WRAP_OPT "TCP, UDP, or SPREAD",
     "mode"},
    {"export-port", 0, 0, G_OPTION_ARG_STRING, &md_eport,
     WRAP_OPT "When --output-mode is TCP or UDP, changes the export port"
     WRAP_OPT "from the default [18001]",
     "port"},
    {"udp-temp-timeout", 0, 0, G_OPTION_ARG_INT, &udp_temp_timeout,
     WRAP_OPT "When exporting via UDP, sets the Template Timeout Period [600]",
     "seconds"},
    {"rotate", 0, 0, G_OPTION_ARG_INT, &rotate,
     WRAP_OPT "Enables rolling output files and sets the interval for opening"
     WRAP_OPT "a new file. --in names the directory and file prefix where the"
     WRAP_OPT "files are written",
     "seconds"},
    {"no-stats", 0, 0, G_OPTION_ARG_NONE, &md_config.no_stats,
     WRAP_OPT "Disables decoding and exporting of YAF process stat records",
     NULL},
    {"preserve-obdomain", 0, 0, G_OPTION_ARG_NONE, &md_config.preserve_obdomain,
     WRAP_OPT "Disables overwriting any existing values of observationDomainId"
     WRAP_OPT "elements in the incoming IPFIX records with the domain ID"
     WRAP_OPT "present in the IPFIX message header",
     NULL},
    {"dns-dedup", 0, 0, G_OPTION_ARG_NONE, &dns_dedup,
     WRAP_OPT "Performs DNS Deduplication on all DNS Resource Records",
     NULL},
#if SM_ENABLE_METADATA_EXPORT
    {"metadata-export", 0, 0, G_OPTION_ARG_NONE, &md_metadata_export,
     WRAP_OPT "Includes information element (RFC 5610) and template metadata"
     WRAP_OPT "records in the IPFIX output",
     NULL},
#endif /* if SM_ENABLE_METADATA_EXPORT */
    {"rewrite-ssl-certs", 0, 0, G_OPTION_ARG_NONE, &md_config.rewrite_ssl_certs,
     WRAP_OPT "Rewrites SSL certs for IPFIX exporters and allows reading of"
     WRAP_OPT "rewritten SSL certs from an upstream mediator",
     NULL},
    {"fields", 'f', 0, G_OPTION_ARG_STRING, &md_field_list,
     WRAP_OPT "When exporting in TEXT mode, specifies a comma-separated list"
     WRAP_OPT "of the numeric IDs of the flow fields to print",
     "fields"},
    {"print-headers", 'h', 0, G_OPTION_ARG_NONE, &md_print_headers,
     WRAP_OPT "When exporting in TEXT mode, prints column headers. May not"
     WRAP_OPT "be used with --fields",
     NULL},
    {"time-units", 0, 0, G_OPTION_ARG_STRING, &md_time_units_arg,
     WRAP_OPT "Sets unit(s) of timestamps for TEXT or JSON exporters [incoming]"
     WRAP_OPT "Choices: best,incoming,micro,milli,nano,nofrac",
     "units,..."},
    {"sleep", 's', 0, G_OPTION_ARG_INT, &sleep_usecs,
     WRAP_OPT "Sets the number of microseconds to sleep between exporting"
     WRAP_OPT "IPFIX messages. Argument may not exceed 1000000 [0]",
     "microseconds"},
    {"log", 'l', 0, G_OPTION_ARG_STRING, &md_logfile,
     WRAP_OPT "Specifies the syslog facility or file to which log and error"
     WRAP_OPT "messages are written",
     "syslog-name|logfile"},
    {"log-dir", 0, 0, G_OPTION_ARG_STRING, &md_logdest,
     WRAP_OPT "Specifies an existing directory where the log files are"
     WRAP_OPT "written",
     "log_path"},
    {"verbose", 'v', 0, G_OPTION_ARG_NONE, &md_verbose,
     WRAP_OPT "Changes the logging level from default WARNING to DEBUG",
     NULL},
    {"quiet", 'q', 0, G_OPTION_ARG_NONE, &md_quiet,
     WRAP_OPT "Changes the logging level from default WARNING to QUIET",
     NULL},
    {"daemonize", 'd', 0, G_OPTION_ARG_NONE, &md_daemon,
     WRAP_OPT "Tells super_mediator to run as a daemon",
     NULL},
    {"pidfile", 0, 0, G_OPTION_ARG_STRING, &md_pidfile,
     WRAP_OPT "Names the complete path to the process ID file",
     "file_path"},
    {"become-user", 'U', 0, G_OPTION_ARG_STRING, &become_user,
     WRAP_OPT "Sets the user ID (uid) after setup if started as root",
     "user"},
    {"become-group", 0, 0, G_OPTION_ARG_STRING, &become_group,
     WRAP_OPT "Sets the group ID (gid) after setup if started as root",
     "group"},
    {"test-config", '\0', 0, G_OPTION_ARG_NONE, &md_test_config,
     WRAP_OPT "Exits after parsing the config file",
     NULL},
    {"version", 'V', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_CALLBACK,
     &mdPrintVersion,
     WRAP_OPT "Prints application version and exits",
     NULL},
    { NULL, 0, 0, G_OPTION_ARG_NONE, NULL, NULL, NULL }
};

static void
mdFatal(
    const char  *format,
    ...)
    __attribute__((format (printf, 1, 2)))
    __attribute__((__noreturn__));

static void
mdDaemonize(
    void);

static void
mdParseOptions(
    int   *argc,
    char **argv[]);

static void
sigHandler(
    void);

static gboolean
mdPrivc_Setup(
    GError **err);

static gboolean
mdPrivc_Become(
    GError **err);

static void
smFreeMaps(
    mdConfig_t  *cfg);

/**
 * mdLogger
 *
 */
static void
mdLogger(
    const char      *domain __attribute__((unused)),
    GLogLevelFlags   log_level,
    const char      *message,
    gpointer         user_data __attribute__((unused)))
{
    gsize      sz;
    char       timebuf[64];
    struct tm  time_tm;
    time_t     cur_time = time(NULL);

    MD_UNUSED_PARAM(log_level);

    gmtime_r(&cur_time, &time_tm);
    snprintf(timebuf, sizeof(timebuf), "[%04u-%02u-%02u %02u:%02u:%02u] ",
             (time_tm.tm_year + 1900), (time_tm.tm_mon + 1), time_tm.tm_mday,
             time_tm.tm_hour, time_tm.tm_min, time_tm.tm_sec);

    g_io_channel_write_chars(logfile, timebuf, -1, &sz, NULL);
    g_io_channel_write_chars(logfile, message, -1, &sz, NULL);
    g_io_channel_write_chars(logfile, "\n", 1, &sz, NULL);
    g_io_channel_flush(logfile, NULL);
}

/**
 * mdPrintHelp
 *
 */
static void
mdPrintHelp(
    const gchar  *option_name,
    const gchar  *option_arg,
    gpointer      v_context,
    GError      **error)
{
    GOptionContext *ctx;
    char           *help;

    (void)option_name;
    (void)option_arg;
    (void)error;

    ctx = (GOptionContext *)v_context;
    help = g_option_context_get_help(ctx, FALSE, NULL);

    printf("%s", help);
    g_free(help);
    exit(0);
}

/**
 * mdPrintVersion
 *
 *
 */
static void
mdPrintVersion(
    const gchar  *option_name,
    const gchar  *option_arg,
    gpointer      v_context,
    GError      **error)
{
    GString *resultString;
    GString *silk;

    MD_UNUSED_PARAM(option_name);
    MD_UNUSED_PARAM(option_arg);
    MD_UNUSED_PARAM(v_context);
    MD_UNUSED_PARAM(error);

    resultString = g_string_sized_new(1024);

    g_string_append_printf(resultString, "super_mediator version %s\n"
                           "Build Configuration: \n",  VERSION);
#ifdef FIXBUF_VERSION
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Fixbuf version:",
                           FIXBUF_VERSION);
#endif

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "MySQL support:",
#if HAVE_MYSQL
                           "YES"
#else
                           "NO"
#endif
                           );

    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Spread support:",
#if HAVE_SPREAD
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "OpenSSL support:",
#if HAVE_OPENSSL
                           "YES"
#else
                           "NO"
#endif
                           );
    g_string_append_printf(resultString, "    * %-32s  %s\n",
                           "Metadata export support:",
#if SM_ENABLE_METADATA_EXPORT
                           "YES"
#else
                           "NO"
#endif
                           );

    silk = g_string_sized_new(128);
#if ENABLE_SKIPSET
    g_string_append(silk, ", IPSet");
#endif
#if ENABLE_SKPREFIXMAP
    g_string_append(silk, ", PrefixMap");
#endif
#if ENABLE_SKTYPESENSOR
    g_string_append(silk, ", Type+Sensor");
#endif
    if (silk->len > 2) {
        g_string_append_printf(resultString, "    * %-32s  %s (%s)\n",
                               "SiLK support:", "YES", silk->str + 2);
    } else {
        g_string_append_printf(resultString, "    * %-32s  %s\n",
                               "SiLK support:", "NO");
    }
    g_string_free(silk, TRUE);

    g_string_append(resultString,
                    "Copyright 2012-2025 Carnegie Mellon University"
                    "\nGNU General Public License (GPL) Rights "
                    "pursuant to Version 2, June 1991\n");
    g_string_append(resultString,
                    "Government Purpose License Rights (GPLR) "
                    "pursuant to DFARS 252.227-7013\n");
    g_string_append(resultString,
                    "Send bug reports, feature requests, and comments"
                    " to netsa-help@cert.org.\n");

    fprintf(stdout, "%s", resultString->str);

    g_string_free(resultString, TRUE);
    exit(0);
}

static GIOChannel *
mdRotateLog(
    GError **err)
{
    char       path[500];
    char       date[32];
    time_t     t;
    struct tm  ts;
    int        slog = 0;

    /* get current time */
    t = time(NULL);
    localtime_r(&t, &ts);
    strftime(date, sizeof(date), "%Y%m%d", &ts);

#ifndef MDLOG_TESTING_LOG
    ts.tm_hour = 23;
    ts.tm_min = 59;
    ts.tm_sec = 59;
    md_log_rolltime = mktime(&ts) + 1;
#else  /* ifndef MDLOG_TESTING_LOG */
    strftime(date, sizeof(date), "%Y%m%d:%H:%M", &ts);

    if (ts.tm_sec > 55) {
        ++ts.tm_min;
    }
    ts.tm_sec = 0;
    ++ts.tm_min;
    md_log_rolltime = mktime(&ts);
#endif /* ifndef MDLOG_TESTING_LOG */
    snprintf(path, sizeof(path), "%s/sm-%s.log", md_logdest, date);
    if (cur_logfile) {
        if (last_logfile) {
            g_free(last_logfile);
        }
        last_logfile = cur_logfile;
    }
    cur_logfile = g_strdup(path);
    return md_log_setup(path, level, &slog, md_daemon, err);
}


/**
 * mdSetEmitTimer
 *
 * Separate thread that runs when the mediator is to run
 * forever.  This will emit the buffer and flush any
 * export files every 5 minutes in the case that we are not
 * receiving any data.
 * It will also print mediator stats if the mediator is
 * configured to do so.
 *
 */
static void *
mdSetEmitTimer(
    void  *data)
{
    mdContext_t     *ctx = (mdContext_t *)data;
    mdConfig_t      *cfg = ctx->cfg;
    struct timespec  to;
    struct timeval   tp;
    time_t           now;
    time_t           then;
    time_t           sectime;
    int              seconds, rc;
    /* we need to flush at least every 5 minutes */
    const long       timeout = MIN(md_stats_timeout, TIMEOUT);
    GError          *err = NULL;

    then = sectime = time(NULL);

    gettimeofday(&tp, NULL);
    to.tv_sec = tp.tv_sec + timeout;
    to.tv_nsec = tp.tv_usec * 1000;

    pthread_mutex_lock(&cfg->log_mutex);

    while (!md_quit) {
        pthread_cond_timedwait(&cfg->log_cond, &cfg->log_mutex, &to);
        gettimeofday(&tp, NULL);
        to.tv_sec = tp.tv_sec + timeout;
        to.tv_nsec = tp.tv_usec * 1000;
        now = time(NULL);
        /* only flush every 5 minutes */
        seconds = difftime(now, then);
        if (seconds >= TIMEOUT) {
            then = time(NULL);
            mdExporterConnectionReset(cfg, &(ctx->err));
        }
        seconds = difftime(now, sectime);
        if (seconds >= md_stats_timeout) {
            sectime = time(NULL);
            if (!cfg->no_stats) {
                mdStatDump(cfg, ctx->stats);
                if (cfg->gen_tombstone) {
                    mdSendTombstoneRecord(ctx, &err);
                }
            }
        }
        /* rotate log */
        if (md_logdest && md_log_rolltime < now) {
            g_message("Rotating Log File");
            rc = g_io_channel_shutdown(logfile, TRUE, &err);
            if (!rc) {
                g_warning("Unable to rotate log");
            } else {
                /* open new logfile */
                logfile = mdRotateLog(&err);
                /* compress old logfile */
                md_log_compress(last_logfile);
                if (!logfile) {
                    g_warning("Unable to open new log file: %s", err->message);
                }
            }
        }
    }

    pthread_mutex_unlock(&cfg->log_mutex);
    return NULL;
}


static void
mdQuit(
    int   sig)
{
    md_quit++;

    /* typically do not want to call logger from within sighandler */
    /* g_debug("yaf received %s", strsignal(s)); */
    if (0 == shutdown_signal) {
        shutdown_signal = sig;
    }
    mdInterruptListeners(&md_config);
}


/*
 *  Format and print an error message to stderr with an additional newline and
 *  exit the program.
 *
 *  This is intended to be used while checking command line arguments.
 */
static void
mdFatal(
    const char  *format,
    ...)
{
    va_list  ap;

    va_start(ap, format);
    vfprintf(stderr, format, ap);
    fprintf(stderr, "\n");
    va_end(ap);
    exit(1);
}


/**
 * main
 *
 *
 */
int
main(
    int    argc,
    char  *argv[])
{
    mdContext_t  ctx = MD_CTX_INIT;
    GError      *error = NULL;
    pthread_t    to_thread;
    char        *errmsg = NULL;

    md_config = MD_CONFIG_INIT;
    ctx.cfg = &md_config;

    /* parse all the options */
    mdParseOptions(&argc, &argv);

    g_message("super_mediator starting");

    mdStatInit();

    /* set up mediator statistics */
    ctx.stats = g_slice_new0(md_stats_t);

    /**
     * this program runs forever, until interrupted, handle
     * the interrupt gracefully and exit by installing these
     * handlers
     */

    sigHandler();

    /* open input */

    if (!mdCollectorsInit(ctx.cfg, ctx.cfg->flowsrc, &error)) {
        g_warning("Fatal error starting collectors: %s", error->message);
        mdFatal("Fatal error starting collectors: %s", error->message);
    }

    if (pthread_create(&to_thread, NULL, mdSetEmitTimer, &ctx)) {
        g_warning("Fatal error starting statistics thread");
        mdFatal("Fatal error starting statistics thread");
    }

    /* set up output */

    if (!mdExportersInit(ctx.cfg, ctx.cfg->flowexit, &error)) {
        g_warning("Fatal error starting exporters: %s", error->message);
        mdFatal("Fatal error starting exporters: %s", error->message);
    }

    g_debug("Initialization Successful, starting...");

    /** wait for connections*/
    while (!md_quit) {
        if (!mdCollectorStartListeners(ctx.cfg, ctx.cfg->flowsrc, &error)) {
            g_warning("Could not start listener threads %s",
                      error->message);
            fprintf(stderr, "Could not start listener threads %s\n",
                    error->message);
            md_quit = 1;
            break;
        }

        if (!mdPrivc_Become(&error)) {
            if (g_error_matches(error, MD_ERROR_DOMAIN, MD_ERROR_NODROP)) {
                g_warning("Note: Running as root in --live mode "
                          "but not dropping privilege");
                g_clear_error(&error);
            } else {
                md_quit = 1;
                g_warning("Unable to drop privileges: %s", error->message);
                fprintf(stderr, "Unable to drop privileges: %s\n",
                        error->message);
                break;
            }
        }

        /* manage threads, and process flow data */
        if (!(mdCollectorWait(&ctx, &error))) {
            break;
        }
    }

    if (error) {
        errmsg = g_strdup_printf("super_mediator terminating on error: %s",
                                 error->message);
        g_clear_error(&error);
    }
    if (shutdown_signal) {
        g_message("super_mediator received signal %s",
                  strsignal(shutdown_signal));
    }

    pthread_cancel(to_thread);
    pthread_join(to_thread, NULL);

    mdCollectorDestroy(ctx.cfg, TRUE);
    mdExporterDestroy(ctx.cfg, &error);
    smFreeMaps(ctx.cfg);
#if ENABLE_SKTYPESENSOR
    mdSkpcSilkTypeSensorFree();
#endif  /* ENABLE_SKTYPESENSOR */

#if ENABLE_SKIPSET || ENABLE_SKPREFIXMAP || ENABLE_SKTYPESENSOR
    if (app_registered) {
        skAppUnregister();
    }
#endif

    if (user_elements) {
        fbInfoElement_t *tie = user_elements;
        for (; tie->ref.name; tie++) {
            /* free each name */
            g_free((char *)(tie->ref.name));
        }
        g_free(user_elements);
    }

    fbInfoModelFree(mdInfoModel());

    pthread_mutex_destroy(&ctx.cfg->log_mutex);
    pthread_cond_destroy(&ctx.cfg->log_cond);
    mdStatUpdate(ctx.stats);

    g_slice_free1(sizeof(md_stats_t), ctx.stats);

    if (errmsg) {
        g_warning("%s", errmsg);
        g_free(errmsg);
    } else {
        g_debug("super_mediator Terminating");
    }

    /** finished with no problems */
    return 0;
}

/**
 * md_new_export_node
 *
 */
md_export_node_t *
md_new_export_node(
    gboolean   dnsdedup,
    gboolean   dedup)
{
    md_export_node_t *en = g_slice_new0(md_export_node_t);
    en->next = NULL;

    if (dnsdedup) {
        en->dns_dedup = md_dns_dedup_new_state();
    }

    if (dedup) {
        en->dedup = md_dedup_new_dedup_state();
    }

    return en;
}

/**
 * md_new_collect_node
 *
 *
 */
md_collect_node_t *
md_new_collect_node(
    void)
{
    md_collect_node_t *cn = g_slice_new0(md_collect_node_t);
    cn->next = NULL;

    return cn;
}


static mdFlowCollector_t *
mdConfigureCollector(
    void)
{
    mdFlowCollector_t *collector = NULL;
    const char        *opt_conflict = NULL;
    const char        *opt_forget = "";

    if (NONE == md_ipfix_intransport) {
        md_ipfix_intransport = FILEHANDLER;
    }
    collector = mdNewFlowCollector(md_ipfix_intransport, NULL);

    if (md_ipfix_intransport == SPREAD) {
#if !HAVE_SPREAD
        mdFatal("Spread support not enabled; reconfigure with Spread");
#else  /* HAVE_SPREAD */
        int     count;
        int     n;
        gchar **sa;

        /* input should be Spread Daemon */
        if (NULL == inspec) {
            mdFatal("Using --ipfix-input=%s requires --in to specify"
                    " Spread daemon to connect to", md_in_str);
        }

        /* check for conflicting options */
        if (md_iport) {
            opt_conflict = "ipfix-port";
        } else if (poll_time) {
            opt_conflict = "watch";
        } else if (move_dir) {
            opt_conflict = "move";
        } else if (delete_inspec) {
            opt_conflict = "delete";
        } else if (md_config.lockmode) {
            opt_conflict = "lock";
        }
        if (opt_conflict) {
            mdFatal("May not use --%s with --ipfix-input=%s",
                    opt_conflict, md_in_str);
        }

        /* test connections and subscribe to groups */
        if (NULL == in_groups) {
            mdFatal("The --group option must specify at least"
                    " 1 Spread Group Name to subscribe to");
        }
        count = 0;
        sa = g_strsplit(in_groups, ",", -1);
        for (n = 0; sa[n]; ++n) {
            g_strstrip(sa[n]);
            if (sa[n][0]) {
                mdCollectorAddSpreadGroup(collector, sa[n], count);
                ++count;
            }
        }
        g_strfreev(sa);
        if (0 == group) {
            mdFatal("No groups found in the --group option");
        }
        mdCollectorSetInSpec(collector, inspec);

        return collector;
#endif /* #else of #if !HAVE_SPREAD */
    }

    if (md_ipfix_intransport == TCP || md_ipfix_intransport == UDP) {
        /* input is a hostname or an IP address */
        if (NULL == inspec) {
            mdFatal("Using --ipfix-input=%s requires --in to specify"
                    " hostname or IP address to connect to", md_in_str);
        }

        /* check for conflicting options */
        if (poll_time) {
            opt_conflict = "watch";
        } else if (move_dir) {
            opt_conflict = "move";
        } else if (delete_inspec) {
            opt_conflict = "delete";
        } else if (md_config.lockmode) {
            opt_conflict = "lock";
#if HAVE_SPREAD
        } else if (in_groups) {
            opt_conflict = "groups";
#endif  /* HAVE_SPREAD */
        }
        if (opt_conflict) {
            mdFatal("May not use --%s with --ipfix-input=%s",
                    opt_conflict, md_in_str);
        }

        mdCollectorSetInSpec(collector, inspec);
        if (NULL == md_iport) {
            mdCollectorSetPort(collector, DEFAULT_MD_IPORT);
        } else {
            char *ep;
            long  p;

            errno = 0;
            p = strtol(md_iport, &ep, 0);
            if (ep == md_iport
                || *ep != '\0'
                || (0 == p && EINVAL == errno))
            {
                mdFatal("Invalid --ipfix-port '%s'", md_iport);
            }
            if (p < 1024 || p > UINT16_MAX) {
                mdFatal("Invalid --ipfix-port '%s': Listening port for"
                        " TCP/UDP must be between 1024 and 65535 inclusive",
                        md_iport);
            }
            mdCollectorSetPort(collector, md_iport);
            g_free(md_iport);
        }

        return collector;
    }

    /* it's either a file or a file-glob (which we call directory) */
    if (poll_time || move_dir) {
        /* input is expected to be a file glob */
        if (NULL == inspec) {
            mdFatal("Using --watch or --move requires"
                    " --in to specify a file glob pattern");
        }

        /* check for conflicting options */
        if (md_iport) {
            opt_conflict = "ipfix-port";
#if HAVE_SPREAD
        } else if (in_groups) {
            opt_conflict = "groups";
#endif  /* HAVE_SPREAD */
        }
        if (opt_conflict) {
            mdFatal("May not use --%s with --watch or --move",
                    opt_conflict);
        }

        if (poll_time) {
            mdCollectorSetPollTime(collector, poll_time);
        } else {
            mdCollectorSetPollTime(collector, DEFAULT_POLL_TIME);
        }
        if (move_dir) {
            if (delete_inspec) {
                mdFatal("May not specify both --move and --delete");
            }
            if (!g_file_test(move_dir, G_FILE_TEST_IS_DIR)) {
                mdFatal("Invalid argument to --move '%s':"
                        " Argument is not a valid file directory",
                        move_dir);
            }
            mdCollectorSetMoveDir(collector, move_dir);
        } else if (delete_inspec) {
            mdCollectorSetDeleteFiles(collector, TRUE);
        } else {
            mdFatal("Must specify --move or --delete when periodically"
                    " checking a file glob");
        }

        mdCollectorSetInSpec(collector, inspec);
        return collector;
    }

    /* Either a file or stdin */
    g_assert(NULL == md_in_str);
    if (NULL == inspec || 0 == strcmp("-", inspec)) {
        if (isatty(fileno(stdin))) {
            mdFatal("Refusing to read from terminal on stdin");
        }
        if (NULL == inspec) {
            inspec = g_strdup("-");
        }
    } else if (!g_file_test(inspec, G_FILE_TEST_EXISTS)) {
        mdFatal("Input file '%s' does not exist", inspec);
    }

    md_ipfix_intransport = FILEHANDLER;

    /* check for conflicting options */
    if (md_iport) {
        opt_conflict = "ipfix-port";
        opt_forget = "; did you forget to specify --ipfix-input?";
    } else if (poll_time) {
        opt_conflict = "watch";
    } else if (move_dir) {
        opt_conflict = "move";
    } else if (delete_inspec) {
        opt_conflict = "delete";
    } else if (md_config.lockmode) {
        opt_conflict = "lock";
#if HAVE_SPREAD
    } else if (in_groups) {
        opt_conflict = "groups";
        opt_forget = "; did you forget to specify --ipfix-input?";
#endif  /* HAVE_SPREAD */
    }
    if (opt_conflict) {
        mdFatal("May not use --%s when reading from a single stream%s",
                opt_conflict, opt_forget);
    }

    mdCollectorSetInSpec(collector, inspec);
    return collector;
}

static mdFlowExporter_t *
mdConfigureExporter(
    mdConfig_t  *cfg,
    gboolean     json)
{
    mdFlowExporter_t *exporter = NULL;
    const char       *opt_conflict = NULL;
    gchar           **sa;
    int               n = 0;

    if (NONE == md_ipfix_outtransport) {
        md_ipfix_outtransport = FILEHANDLER;
    }
    exporter = mdNewFlowExporter(md_ipfix_outtransport);

    if (dns_dedup) {
        mdExporterSetDNSDeDup(exporter);
    }

    if (md_ipfix_outtransport == SPREAD) {
#if !HAVE_SPREAD
        mdFatal("Spread support not enabled; reconfigure with Spread");
#else  /* HAVE_SPREAD */

        /* input should be Spread Daemon */
        if (NULL == outspec) {
            mdFatal("Using --output-mode=%s requires --out to specify"
                    " the Spread daemon to connect to", md_out_str);
        }

        /* check for conflicting options */
        if (md_iport) {
            opt_conflict = "export-port";
        } else if (rotate) {
            opt_conflict = "rotate";
        } else if (md_field_list) {
            opt_conflict = "fields";
        } else if (md_print_headers) {
            opt_conflict = "print-headers";
        } else if (md_top_level_time_units != MD_TUNIT_UNSET) {
            opt_conflict = "time-units";
        }
        if (opt_conflict) {
            mdFatal("May not use --%s with --output-mode=%s",
                    opt_conflict, md_out_str);
        }

        mdExporterSetHost(md_config.flowexit->exp, outspec);
        g_free(outspec);
        /* output is spread */
        /* need spread groups */
        if (md_out_groups == NULL) {
            g_warning("Use the configuration file to set export groups");
            mdFatal("Use the configuration file to set Spread export groups");
        }
#endif /* if HAVE_SPREAD */
    } else if (md_ipfix_outtransport == TEXT) {
        if (NULL == outspec) {
            if (rotate > 0) {
                mdFatal("Using --rotate requires --out to specify"
                        " the output file prefix (with optional directory)");
            }
            outspec = g_strdup("-");
        }

        if (md_eport) {
            opt_conflict = "export-port";
        } else if (json) {
            if (md_field_list) {
                opt_conflict = "fields";
            } else if (md_print_headers) {
                opt_conflict = "print-headers";
            }
        }
        if (opt_conflict) {
            mdFatal("May not use --%s with --output-mode=%s",
                    opt_conflict, md_out_str);
        }

        mdExporterSetFileSpec(exporter, outspec);
        g_free(outspec);

        if (rotate > 0) {
            mdExporterSetRotateSeconds(exporter, rotate);
        }

        /* The --fields command line option. */
        if (md_field_list) {
            gboolean               dpi = FALSE;
            mdFieldList_t         *first_item = NULL;
            mdFieldList_t        **item = &first_item;
            mdAcceptFilterField_t  field;

            sa = g_strsplit(md_field_list, ",", -1);
            for (n = 0; sa[n]; ++n) {
                /* strip leading/trailing whitespace */
                g_strstrip(sa[n]);
                if (!g_ascii_isdigit(sa[n][0])) {
                    if (!sa[n][0]) {
                        continue;
                    }
                    mdFatal("Invalid field item '%s':"
                            " Field items must be numeric",
                            sa[n]);
                }
                field = atoi(sa[n]);
                if (field == DPI) {
                    mdExporterCustomListDPI(exporter);
                    dpi = TRUE;
                    continue;
                }
                *item = mdCreateFieldList((mdAcceptFilterField_t)field, FALSE);
                if (!*item) {
                    mdFatal("Invalid field item %s: No such field",
                            sa[n]);
                }
                item = &((*item)->next);
            }
            if (dpi && !first_item) {
                /* just DPI was chosen - create list and set to None */
                first_item = mdCreateFieldList(NONE_FIELD, TRUE);
                mdExporterSetDPIOnly(exporter);
            }
            if (!first_item) {
                mdFatal("Invalid fields list '%s': No valid fields given",
                        md_field_list);
            }
            mdExportCustomList(exporter, first_item);
            g_strfreev(sa);
        }

        if (md_print_headers) {
            if (!md_field_list) {
                mdExporterSetPrintHeader(exporter);
            } else {
                g_warning("Printing of column headers is disabled"
                          " when using a custom field list");
            }
        }
        if (json) {
            mdExporterSetJson(exporter);
        }
    } else if (md_ipfix_outtransport == TCP || md_ipfix_outtransport == UDP) {
        if (NULL == outspec) {
            mdFatal("Using --output-mode=%s requires --out to specify"
                    " a hostname or IP address to connect to", md_out_str);
        }

        if (rotate) {
            opt_conflict = "rotate";
        } else if (md_field_list) {
            opt_conflict = "fields";
        } else if (md_print_headers) {
            opt_conflict = "print-headers";
        } else if (md_top_level_time_units != MD_TUNIT_UNSET) {
            opt_conflict = "time-units";
        }
        if (opt_conflict) {
            mdFatal("May not use --%s with --output-mode=%s",
                    opt_conflict, md_out_str);
        }

        if (NULL == md_eport) {
            mdExporterSetPort(exporter, DEFAULT_MD_EPORT);
        } else {
            char *ep;
            long  p;

            errno = 0;
            p = strtol(md_eport, &ep, 0);
            if (ep == md_eport
                || *ep != '\0'
                || (0 == p && EINVAL == errno))
            {
                mdFatal("Invalid --export-port '%s'", md_eport);
            }
            if (p < 1024 || p > UINT16_MAX) {
                mdFatal("Invalid --export-port '%s': Exporting port for"
                        " TCP/UDP must be between 1024 and 65535 inclusive",
                        md_eport);
            }
            mdExporterSetPort(exporter, md_eport);
            g_free(md_eport);
        }
        mdExporterSetHost(exporter, outspec);

        /* Templates are resent three times per timeout interval */
        udp_temp_timeout /= 3;
        if (udp_temp_timeout <= 0) {
            udp_temp_timeout = DEFAULT_UDP_TEMP_TIMEOUT / 3;
        }
        yfDiffTimeFromSeconds(&cfg->udp_template_interval, udp_temp_timeout);
    } else {
        /* Exporting IPFIX to stdout, a file, or rotating output files */

        if (rotate > 0) {
            /* IPFIX output to a directory */
            if (NULL == outspec) {
                mdFatal("Using --rotate requires --out to specify"
                        " the output file prefix (with optional directory)");
            }
        } else if (NULL == outspec || 0 == strcmp("-", outspec)) {
            if (isatty(fileno(stdout))) {
                mdFatal("Refusing to write to terminal on stdout");
            }
            if (NULL == outspec) {
                outspec = g_strdup("-");
            }
        }

        if (md_eport) {
            opt_conflict = "export-port";
        } else if (md_field_list) {
            opt_conflict = "fields";
        } else if (md_print_headers) {
            opt_conflict = "print-headers";
        } else if (md_top_level_time_units != MD_TUNIT_UNSET) {
            opt_conflict = "time-units";
        }
        if (opt_conflict) {
            mdFatal("May not use --%s when exporting IPFIX to files;"
                    " did you forget to specify --output-mode?", opt_conflict);
        }

        mdExporterSetFileSpec(exporter, outspec);
        if (rotate > 0) {
            mdExporterSetRotateSeconds(exporter, rotate);
        }
    }

    if (md_metadata_export) {
        if (TEXT == md_ipfix_outtransport) {
            g_debug("Ignoring --metadata-export for TEXT exporter");
        } else {
            mdExporterSetMetadataExport(exporter);
        }
    }

    if (md_top_level_time_units != MD_TUNIT_UNSET) {
        mdExporterSetTimeUnits(exporter, md_top_level_time_units);
    }

    return exporter;
}


static void
mdParseTimeUnits(
    char  *arg)
{
    const struct tunit_map_st {
        const char     *tname;
        mdTimeUnits_t   tunits;
    } tunit_map[] = {
        {"best",          MD_TUNIT_BEST},
        {"incoming",      MD_TUNIT_INCOMING},
        {"micro",         MD_TUNIT_MICRO},
        {"microsecond",   MD_TUNIT_MICRO},
        {"microseconds",  MD_TUNIT_MICRO},
        {"milli",         MD_TUNIT_MILLI},
        {"millisecond",   MD_TUNIT_MILLI},
        {"milliseconds",  MD_TUNIT_MILLI},
        {"nano",          MD_TUNIT_NANO},
        {"nanosecond",    MD_TUNIT_NANO},
        {"nanoseconds",   MD_TUNIT_NANO},
        {"no-frac",       MD_TUNIT_NOFRAC},
        {"no_frac",       MD_TUNIT_NOFRAC},
        {"nofrac",        MD_TUNIT_NOFRAC},
        {NULL, MD_TUNIT_UNSET}
    };
    unsigned int  i, j;
    gchar       **tokens;
    uint8_t       tunits = MD_TUNIT_UNSET;

    if (md_top_level_time_units != MD_TUNIT_UNSET) {
        mdFatal("Invalid time-units: Value was previously set");
    }

    tokens = g_strsplit(arg, ",", -1);
    for (i = 0; tokens[i] != NULL; ++i) {
        for (j = 0; tunit_map[j].tname != NULL; ++j) {
            if (0 == strcmp(tokens[i], tunit_map[j].tname)) {
                tunits |= tunit_map[j].tunits;
                break;
            }
        }
        /* check for best or incoming included with another */
        if ((tunits & MD_TUNIT_BEST) && (tunits & ~MD_TUNIT_BEST)) {
            mdFatal("Invalid time-units:"
                    " 'best' may not be included in a list");
        }
        if ((tunits & MD_TUNIT_INCOMING) && (tunits & ~MD_TUNIT_INCOMING)) {
            mdFatal("Invalid time-units:"
                    " 'incoming' may not be included in a list");
        }
        /* error if not found unless token is empty string */
        if (NULL == tunit_map[j].tname && *tokens[i]) {
            mdFatal("Invalid time-units item '%s'", tokens[i]);
        }
    }
    if (MD_TUNIT_UNSET == tunits) {
        mdFatal("Invalid time-units: argument contains no valid values");
    }
    g_strfreev(tokens);
    g_free(arg);

    md_top_level_time_units = tunits;
}


/**
 * mdParseOptions
 *
 * parses the command line options
 *
 */
static void
mdParseOptions(
    int   *argc,
    char **argv[])
{
    GOptionEntry      help_option[] = {
        {"help", 'h', G_OPTION_FLAG_NO_ARG,
         G_OPTION_ARG_CALLBACK, &mdPrintHelp,
         WRAP_OPT "Shows this brief description of the options and exits",
         NULL},
        { NULL, 0, 0, G_OPTION_ARG_NONE, NULL, NULL, NULL }
    };
    GOptionContext   *ctx = NULL;
    GOptionGroup     *group = NULL;
    GError           *err = NULL;
    int               slog = 0;
    gboolean          json = FALSE;
    md_export_node_t *cnode = NULL;

    ctx = g_option_context_new("- Mediator Options");

    /* Create the main group so we can set user_data to ctx */
    group = g_option_group_new("main", "Options", NULL, ctx, NULL);
    g_option_context_set_main_group(ctx, group);
    g_option_context_set_summary(ctx, MD_HELP_SUMMARY);

    /* Create the help option */
    g_option_context_set_help_enabled(ctx, FALSE);
    g_option_group_add_entries(group, help_option);
    g_option_group_add_entries(group, md_core_option);

    if (!g_option_context_parse(ctx, argc, argv, &err)) {
        mdFatal("Option parsing failed: %s", err->message);
    }

    if (!mdPrivc_Setup(&err)) {
        mdFatal("%s", err->message);
    }

    /* Configuration file has precedence over all */
    /* FIXME: Does it *REALLY* have precedence and does it make sense?
     * Shouldn't command line overrule? */
    if (md_conf) {
        if (!mediator_config_load(md_conf)) {
            exit(1);
        }
    } else if (md_test_config) {
        mdFatal("Must specify --config when using --test-config");
    }

    /* Parse --ipfix-input */
    if (NULL == md_in_str) {
        md_ipfix_intransport = FILEHANDLER;
    } else if (0 == g_ascii_strcasecmp(md_in_str, "tcp")) {
        md_ipfix_intransport = TCP;
    } else if (0 == g_ascii_strcasecmp(md_in_str, "udp")) {
        md_ipfix_intransport = UDP;
    } else if (0 == g_ascii_strcasecmp(md_in_str, "spread")) {
        md_ipfix_intransport = SPREAD;
    } else {
        mdFatal("Invalid --ipfix-input '%s'", md_in_str);
    }

    /* Parse --output-mode */
    if (NULL == md_out_str) {
        md_ipfix_outtransport = FILEHANDLER;
    } else if (0 == g_ascii_strcasecmp(md_out_str, "tcp")) {
        md_ipfix_outtransport = TCP;
    } else if (0 == g_ascii_strcasecmp(md_out_str, "udp")) {
        md_ipfix_outtransport = UDP;
    } else if (0 == g_ascii_strcasecmp(md_out_str, "text")) {
        md_ipfix_outtransport = TEXT;
    } else if (0 == g_ascii_strcasecmp(md_out_str, "json")) {
        md_ipfix_outtransport = TEXT;
        json = TRUE;
    } else if (0 == g_ascii_strcasecmp(md_out_str, "file")
               || 0 == g_ascii_strcasecmp(md_out_str, "filehandler"))
    {
        md_ipfix_outtransport = FILEHANDLER;
    } else if (0 == g_ascii_strcasecmp(md_out_str, "spread")) {
        mdFatal("Spread export is only available if configured using the "
                "configuration file");
    } else {
        mdFatal("Invalid --output-mode '%s'", md_out_str);
    }

    /* Logging options */

    level = md_parse_log_level(md_log_level, md_verbose, md_quiet);

    /* Do not set up the log file when --test-config is given */
    if (NULL == md_conf || !md_test_config) {
        if (md_logdest) {
            if (md_logfile) {
                mdFatal("May not use both --log and --log-dir");
            }
            logfile = mdRotateLog(&err);
        } else {
            logfile = md_log_setup(md_logfile, level, &slog, md_daemon, &err);
        }

        if (!logfile && (slog == 0)) {
            mdFatal("Log option parsing failed: %s", err->message);
        }

        if (!slog) {
            /* if not syslog, set default handler */
            g_log_set_handler(G_LOG_DOMAIN, level, mdLogger, NULL);
        }
    }

    if (md_stats_timeout == 0) {
        g_warning("Turning off stats export.");
        md_stats_timeout = TIMEOUT;
        md_config.no_stats = TRUE;
    }

    /* COLLECTOR OPTIONS */

    /* If --in is specified, the command-line configured collector takes
     * precedence over thoese in config file. */
    if (md_config.flowsrc == NULL || inspec != NULL) {
        if (md_config.flowsrc) {
            g_warning("WARNING: Overriding Collectors in configuration file "
                      "due to presence of command line arguments.");
            mdCollectorDestroy(&md_config, FALSE);
        }
        md_config.flowsrc = md_new_collect_node();
        md_config.flowsrc->coll = mdConfigureCollector();
    }

    /* Handle --sleep */
    if (sleep_usecs < SLEEP_USECS_MAX) {
        md_config.usec_sleep = sleep_usecs;
    } else {
        g_warning("Overriding requested --sleep=%d with maximum allowed %d",
                  sleep_usecs, SLEEP_USECS_MAX);
        md_config.usec_sleep = SLEEP_USECS_MAX;
    }

    if (md_time_units_arg) {
        mdParseTimeUnits(md_time_units_arg);
        md_time_units_arg = NULL;
    }

    /* NOW TO EXPORT */
    /* If no exporters are defined in the config file, configure one based on
     * command line options. */
    if (md_config.flowexit == NULL) {
        md_config.flowexit = md_new_export_node(dns_dedup, FALSE);
        md_config.flowexit->exp = mdConfigureExporter(&md_config, json);
    } else if (outspec) {
        g_warning(
            "WARNING: WILL NOT Override Exporters in configuration file.");
    }

    if (dns_dedup) {
        if (dns_max_hit_count == 0) {
            mdFatal("Invalid Hit Count of 0 (Must be > 0)");
        }
        if (dns_flush_timeout == 0) {
            mdFatal("Invalid Flush Timeout of 0 (Must be > 0)");
        }

        md_dns_dedup_configure_state(md_config.flowexit->dns_dedup, NULL,
                                     dns_max_hit_count, dns_flush_timeout,
                                     FALSE, NULL, FALSE);
    }

    if (!mdCollectorVerifySetup(md_config.flowsrc->coll, &err)) {
        mdFatal("Error Verifying Collectors: %s", err->message);
    }

    for (cnode = md_config.flowexit; cnode; cnode = cnode->next) {
        if (!mdExporterVerifySetup(cnode->exp)) {
            mdFatal("Error Verifying Exporters");
        }
    }

    /* If --config and --test-config are both given, exit */
    if (md_conf && md_test_config) {
        if (!md_quiet) {
            fprintf(stderr, "Successfully parsed configuration file \"%s\"\n",
                    md_conf);
        }
        exit(0);
    }

    if (md_daemon) {
        mdDaemonize();
    }

    g_option_context_free(ctx);
}



/**
 * smExit
 *
 * exit handler for super_mediator
 *
 */
static void
mdExit(
    void)
{
    if (md_pidfile) {
        unlink(md_pidfile);
    }
}


/**
 * sigHandler
 *
 * this gets called from various system signal handlers.  It is used to
 * provide a way to exit this program cleanly when the user wants to
 * kill this program
 *
 * @param signalNumber the number of the signal that this handler is
 *        getting called for
 *
 */
static void
sigHandler(
    void)
{
    struct sigaction  sa, osa;

    sa.sa_handler = mdQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGINT, &sa, &osa)) {
        g_error("sigaction(SIGINT) failed: %s", strerror(errno));
    }

    sa.sa_handler = mdQuit;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;
    if (sigaction(SIGTERM, &sa, &osa)) {
        g_error("sigaction(SIGTERM) failed: %s", strerror(errno));
    }
}

static void
mdDaemonize(
    void)
{
    pid_t  pid;
    int    rv = -1;
    char   str[256];
    int    fp;

    if (chdir("/") == -1) {
        rv = errno;
        g_warning("Cannot change directory: %s", strerror(rv));
        mdFatal("Unable to become daemon; see the log file");
    }

    if ((pid = fork()) == -1) {
        rv = errno;
        g_warning("Cannot fork for daemon: %s", strerror(rv));
        mdFatal("Unable to become daemon; see the log file");
    } else if (pid != 0) {
        g_debug("Forked child %ld.  Parent exiting", (long)pid);
        _exit(EXIT_SUCCESS);
    }

    setsid();

    umask(0022);

    rv = atexit(mdExit);
    if (rv == -1) {
        g_warning("Unable to register function with atexit(): %s",
                  strerror(rv));
        mdFatal("Unable to become daemon; see the log file");
    }

    /* Close out the standard file descriptors */
    close(STDIN_FILENO);

    if (md_pidfile) {
        fp = open(md_pidfile, O_RDWR | O_CREAT, 0640);
        if (fp < 0) {
            g_warning("Unable to open pid file %s", md_pidfile);
            mdFatal("Unable to become daemon; see the log file");
        }
        sprintf(str, "%d\n", getpid());
        if (!write(fp, str, strlen(str))) {
            g_warning("Unable to write pid to file");
        }
    } else {
        g_debug("pid: %d", getpid());
    }
}

gboolean
mdListenerConnect(
    fbListener_t     *listener,
    void            **ctx,
    int               fd,
    struct sockaddr  *peer,
    size_t            peerlen,
    GError          **err)
{
    md_collect_node_t *collector;

    MD_UNUSED_PARAM(fd);
    MD_UNUSED_PARAM(peerlen);
    MD_UNUSED_PARAM(err);

    if (!peer) {
        /* this is UDP */
        return TRUE;
    }

    /* set context based on which listener this is */
    collector = mdCollectorFindListener(md_config.flowsrc, listener);

    if (!collector) {
        return FALSE;
    }

    if (peer->sa_family == AF_INET) {
        char *ip = inet_ntoa((((struct sockaddr_in *)peer)->sin_addr));
        pthread_mutex_lock(&md_config.log_mutex);
        g_message("%s: accepting connection from %s:%d",
                  mdCollectorGetName(collector), ip,
                  ((struct sockaddr_in *)peer)->sin_port);
        pthread_mutex_unlock(&md_config.log_mutex);
    } else if (peer->sa_family == AF_INET6) {
        char  straddr[INET6_ADDRSTRLEN];
        inet_ntop(AF_INET6, &(((struct sockaddr_in6 *)peer)->sin6_addr),
                  straddr, sizeof(straddr));
        pthread_mutex_lock(&md_config.log_mutex);
        g_message("%s: accepting connection from %s:%d",
                  mdCollectorGetName(collector), straddr,
                  ((struct sockaddr_in6 *)peer)->sin6_port);
        pthread_mutex_unlock(&md_config.log_mutex);
    }

    collector->stats->restarts++;

    *ctx = (void *)collector;

    return TRUE;
}

static void
smFreeMaps(
    mdConfig_t  *cfg)
{
    unsigned int  i = 0;

    if (cfg->maps) {
        smFieldMap_t *cmap = NULL;
        smFieldMap_t *nmap = NULL;
        for (cmap = cfg->maps; cmap; cmap = cmap->next) {
            smHashTableFree(cmap->table);
            g_free(cmap->name);
            for (i = 0; i < cmap->count; i++) {
                g_free(cmap->labels[i]);
            }
            free(cmap->labels);
        }
        cmap = cfg->maps;
        while (cmap) {
            detachHeadOfSLL((mdSLL_t **)&(cfg->maps),
                            (mdSLL_t **)&cmap);
            nmap = cmap->next;
            g_slice_free(smFieldMap_t, cmap);
            cmap = nmap;
        }
    }
}

static gboolean
mdPrivc_Setup(
    GError **err)
{
    struct passwd *pwe = NULL;
    struct group  *gre = NULL;

    if (geteuid() == 0) {
        /* We're root. Parse user and group names. */
        if (become_user) {
            if (!(pwe = getpwnam(become_user))) {
                g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                            "Cannot become user %s: %s.",
                            become_user, strerror(errno));
                return FALSE;
            }

            /* By default, become new user's user and group. */
            new_user = pwe->pw_uid;
            new_group = pwe->pw_gid;
            if (become_group) {
                if (!(gre = getgrnam(become_group))) {
                    g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                                "Cannot become group %s: %s.",
                                become_group, strerror(errno));
                    return FALSE;
                }

                /* Override new group if set */
                new_group = gre->gr_gid;
            }
        }
    } else {
        /* We're not root. If we have options, the user is confused, and
         * we should straighten him out by killing the process. */
        if (become_user) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "Cannot become user %s: not root.",
                        become_user);
            return FALSE;
        }
        if (become_group) {
            g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                        "Cannot become group %s: not root.",
                        become_group);
            return FALSE;
        }
    }

    /* All done. */
    return TRUE;
}

static gboolean
mdPrivc_Become(
    GError **err)
{
    /* Die if we've already become */
    if (did_become) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "not dropping privileges, already did so");
        return FALSE;
    }

    /* Short circuit if we're not root */
    if (geteuid() != 0) {return TRUE;}

    /* Allow app to warn if not dropping */
    if (new_user == 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_NODROP,
                    "not dropping privileges (use --become-user to do so)");
        return FALSE;
    }

    /* Okay. Do the drop. */

    /* Drop ancillary group privileges while we're still root */
    if (setgroups(1, &new_group) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Could not drop ancillary groups: %s", strerror(errno));
        return FALSE;
    }
#if LINUX_PRIVHACK
    /* Change to group */
    if (setregid(new_group, new_group) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Could not become group %u: %s",
                    new_group, strerror(errno));
        return FALSE;
    }

    /* Lose root privileges */
    if (setreuid(new_user, new_user) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Could not become user %u: %s", new_user, strerror(errno));
        return FALSE;
    }
#else  /* if LINUX_PRIVHACK */
    /* Change to group */
    if (setgid(new_group) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Could not become group %u: %s",
                    new_group, strerror(errno));
        return FALSE;
    }

    /* Lose root privileges */
    if (setuid(new_user) < 0) {
        g_set_error(err, MD_ERROR_DOMAIN, MD_ERROR_SETUP,
                    "Could not become user %u: %s", new_user, strerror(errno));
        return FALSE;
    }
#endif /* if LINUX_PRIVHACK */
    /* All done. */
    did_become = TRUE;
    return TRUE;
}
