%{
/*
 *  Copyright 2005-2025 Carnegie Mellon University
 *  See license information in LICENSE.txt.
 */
/*
 *  probeconfparse.y
 *
 *  YACC source file for sensor.conf parser. Taken from SiLK.
 *
 *  ------------------------------------------------------------------------
 *  @DISTRIBUTION_STATEMENT_BEGIN@
 *  super_mediator-1.13
 *
 *  Copyright 2025 Carnegie Mellon University.
 *
 *  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
 *  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
 *  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
 *  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
 *  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
 *  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
 *  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
 *  INFRINGEMENT.
 *
 *  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
 *  contact permission@sei.cmu.edu for full terms.
 *
 *  [DISTRIBUTION STATEMENT A] This material has been approved for public
 *  release and unlimited distribution.  Please see Copyright notice for
 *  non-US Government use and distribution.
 *
 *  This Software includes and/or makes use of Third-Party Software each
 *  subject to its own license.
 *
 *  DM25-1447
 *  @DISTRIBUTION_STATEMENT_END@
 *  ------------------------------------------------------------------------
 */

/*
**  Parser for probe configuration file
**
*/

#include "mediator_ctx.h"
#include <silk/utils.h>
#include <silk/skipaddr.h>
#include <silk/skipset.h>
#include "probeconf.h"
#include "probeconfscan.h"


/* LOCAL DEFINES AND TYPEDEFS */

/* Set DEBUG to 1 to enable debugging printf messasges, 0 otherwise.
 * Generally best to leave this commented out so gcc -DDEBUG=1 will
 * work */
/* #define DEBUG 1 */

/* Verify DEBUG is set */
#ifndef DEBUG
#  define DEBUG 0
#endif

/* For printing messages when DEBUG is non-zero.  Use as:
 *    DEBUG_PRINTF(("x is %d\n", x));
 * Note ((double parens)) */
#if DEBUG
#  define DEBUG_PRINTF(x) printf x
#else
#  define DEBUG_PRINTF(x)
#endif


/* magic value used to denote that a uint32_t---which are stored in
 * uint32_t's in the parser---has not yet been given a value. */
#define UINT32_NO_VALUE  UINT32_MAX

/*
 * sk_vector_t of IDs are created as they are needed, but to avoid
 * repeatedly creating and destroying the vectors, "deleted" vectors are
 * added to a pool.  When creating a vector, the code will check the
 * pool before allocating.  This macro is the size of the pool; if a
 * vector is "deleted" and the pool is full, the vector is free()ed.
 */

#define DEFAULT_ARRAY_SIZE    16

typedef union skcidr_un skcidr_t;

/* LOCAL VARIABLES */

/* number of errors in current defn */
static int  defn_errors = 0;

/* The parser works on a single global probe, sensor, or group */
static md_skpc_probe_t  *probe = NULL;
static md_skpc_sensor_t *sensor = NULL;
static md_skpc_group_t  *group = NULL;

/* Place to stash listen_as address and port until end of probe is
 * reached */
static char *listen_as_address = NULL;
static char *listen_port = NULL;

/* LOCAL FUNCTION PROTOTYPES */

static void
missing_value(
    void);


/* include a file */
static void
include_file(
    char  *name);

/* functions to set attributes of a probe_attr_t */
static void
probe_begin(
    char  *probe_name,
    char  *probe_type);
static void
probe_end(
    void);
static void
probe_priority(
    GArray  *v);
static void
probe_protocol(
    GArray  *v);
static void
probe_listen_as_host(
    GArray  *v);
static void
probe_listen_on_port(
    GArray  *v);
static void
probe_listen_on_usocket(
    GArray  *v);
static void
probe_read_from_file(
    GArray  *v);
static void
probe_poll_directory(
    GArray  *v);
static void
probe_accept_from_host(
    GArray  *v);
static void
probe_log_flags(
    GArray  *v);
static void
probe_interface_values(
    GArray  *v);
static void
probe_quirks(
    GArray  *v);

static void
sensor_begin(
    char  *sensor_name);
static void
sensor_end(
    void);
static void
sensor_isp_ip(
    GArray  *v);
static void
sensor_interface(
    char    *name,
    GArray  *list);
static void
sensor_ipblock(
    char    *name,
    GArray  *wl);
static void
sensor_ipset(
    char    *name,
    GArray  *wl);
static void
sensor_filter(
    md_skpc_filter_t   filter,
    GArray            *v,
    int                is_files);
static void
sensor_network(
    md_skpc_direction_t   direction,
    char                 *name);
static void
sensor_probes(
    char    *probe_type,
    GArray  *v);

static void
group_begin(
    char  *group_name);
static void
group_end(
    void);
static void
group_add_data(
    GArray                *v,
    md_skpc_group_type_t   g_type);

static md_skpc_group_t *
get_group(
    const char            *g_name,
    md_skpc_group_type_t   g_type);
static int
add_values_to_group(
    md_skpc_group_t       *g,
    GArray                *v,
    md_skpc_group_type_t   g_type);


/* functions to convert string input to another form */
static uint32_t
parse_int_u32(
    char  *s);
static skipset_t *
parse_ipset_filename(
    char  *s);
static skcidr_t *
parse_cidr_addr(
    char  *s);

/*  Tell uncrustify to ignore the next part of the file */
/*  *INDENT-OFF* */
%}
%union {
    char               *string;
    GArray             *vector;
    uint32_t            u32;
    md_skpc_direction_t    net_dir;
    md_skpc_filter_t       filter;
}

%token ACCEPT_FROM_HOST_T
%token COMMA
%token END_GROUP_T
%token END_PROBE_T
%token END_SENSOR_T
%token EOL
%token GROUP_T
%token INCLUDE_T
%token INTERFACES_T
%token INTERFACE_VALUES_T
%token IPBLOCKS_T
%token IPSETS_T
%token ISP_IP_T
%token LISTEN_AS_HOST_T
%token LISTEN_ON_PORT_T
%token LISTEN_ON_USOCKET_T
%token LOG_FLAGS_T
%token POLL_DIRECTORY_T
%token PRIORITY_T
%token PROBE_T
%token PROTOCOL_T
%token QUIRKS_T
%token READ_FROM_FILE_T
%token REMAINDER_T
%token SENSOR_T
%token <string> ID
%token <string> NET_NAME_INTERFACE
%token <string> NET_NAME_IPBLOCK
%token <string> NET_NAME_IPSET
%token <string> PROBES
%token <string> QUOTED_STRING
%token <net_dir> NET_DIRECTION
%token <filter> FILTER;

%token ERR_STR_TOO_LONG

%type <vector>      id_list
%type <vector>      filename_list
%type <string>      filename


%%

    /*
     * ******************  GRAMMAR RULES  ***********************************
     */

input:                                    /* nothing */
                                        | input probe_defn
                                        | input sensor_defn
                                        | input group_defn
                                        | input include_stmt
                                        | error
{
    mdSkpcParseErr("Misplaced or unrecognized keyword");
    ++pcscan_errors;
};


    /*
     * Include <FILE>
     */

include_stmt:                             INCLUDE_T QUOTED_STRING EOL
{
    include_file($2);
}
                                        | INCLUDE_T EOL
{
    missing_value();
};


    /*
     * A probe
     */

probe_defn:                               probe_begin probe_stmts probe_end
;

probe_begin:                              PROBE_T ID ID EOL
{
    probe_begin($2, $3);
}
                                        | PROBE_T ID EOL
{
    /* error */
    probe_begin(NULL, $2);
}
                                        | PROBE_T EOL
{
    /* error */
    probe_begin(NULL, NULL);
};

probe_end:                                END_PROBE_T EOL
{
    probe_end();
}
                                        | END_GROUP_T EOL
{
    ++defn_errors;
    mdSkpcParseErr("%s used to close probe", pcscan_clause);
    probe_end();
};
                                        | END_SENSOR_T EOL
{
    ++defn_errors;
    mdSkpcParseErr("%s used to close probe", pcscan_clause);
    probe_end();
};


probe_stmts:                              /* empty */
                                        | probe_stmts probe_stmt
;

probe_stmt:                               stmt_probe_priority
                                        | stmt_probe_protocol
                                        | stmt_probe_listen_host
                                        | stmt_probe_listen_port
                                        | stmt_probe_listen_usocket
                                        | stmt_probe_read_file
                                        | stmt_probe_poll_directory
                                        | stmt_probe_accept_host
                                        | stmt_probe_log_flags
                                        | stmt_probe_interface_values
                                        | stmt_probe_quirks
                                        | error
{
    ++defn_errors;
    mdSkpcParseErr(("Error in probe %s:"
                    " Missing \"end probe\" or invalid keyword or value"),
                   (probe ? mdSkpcProbeGetName(probe) : "block"));
};


stmt_probe_priority:                      PRIORITY_T id_list EOL
{
    probe_priority($2);
}
                                        | PRIORITY_T EOL
{
    missing_value();
};

stmt_probe_protocol:                      PROTOCOL_T id_list EOL
{
    probe_protocol($2);
}
                                        | PROTOCOL_T EOL
{
    missing_value();
};

stmt_probe_listen_host:                   LISTEN_AS_HOST_T id_list EOL
{
    probe_listen_as_host($2);
}
                                        | LISTEN_AS_HOST_T EOL
{
    missing_value();
};

stmt_probe_listen_port:                   LISTEN_ON_PORT_T id_list EOL
{
    probe_listen_on_port($2);
}
                                        | LISTEN_ON_PORT_T EOL
{
    missing_value();
};

stmt_probe_listen_usocket:                LISTEN_ON_USOCKET_T id_list EOL
{
    probe_listen_on_usocket($2);
}
                                        | LISTEN_ON_USOCKET_T EOL
{
    missing_value();
};

stmt_probe_read_file:                     READ_FROM_FILE_T id_list EOL
{
    probe_read_from_file($2);
}
                                        | READ_FROM_FILE_T EOL
{
    missing_value();
};

stmt_probe_poll_directory:                POLL_DIRECTORY_T id_list EOL
{
    probe_poll_directory($2);
}
                                        | POLL_DIRECTORY_T EOL
{
    missing_value();
};

stmt_probe_accept_host:                   ACCEPT_FROM_HOST_T id_list EOL
{
    probe_accept_from_host($2);
}
                                        | ACCEPT_FROM_HOST_T EOL
{
    missing_value();
};


stmt_probe_log_flags:                     LOG_FLAGS_T id_list EOL
{
    probe_log_flags($2);
}
                                        | LOG_FLAGS_T EOL
{
    missing_value();
};

stmt_probe_interface_values:              INTERFACE_VALUES_T id_list EOL
{
    probe_interface_values($2);
}
                                        | INTERFACE_VALUES_T EOL
{
    missing_value();
};

stmt_probe_quirks:                        QUIRKS_T id_list EOL
{
    probe_quirks($2);
}
                                        | QUIRKS_T EOL
{
    missing_value();
};


    /*
     * A sensor
     */

sensor_defn:                              sensor_begin sensor_stmts sensor_end
;

sensor_stmts:                           /* empty */
                                        | sensor_stmts sensor_stmt
;

sensor_stmt:                              stmt_sensor_isp_ip
                                        | stmt_sensor_interface
                                        | stmt_sensor_ipblock
                                        | stmt_sensor_ipset
                                        | stmt_sensor_filter
                                        | stmt_sensor_network
                                        | stmt_sensor_probes
                                        | error
{
    ++defn_errors;
    mdSkpcParseErr(("Error in sensor %s:"
                    " Missing \"end sensor\" or invalid keyword or value"),
                   (sensor ? mdSkpcSensorGetName(sensor) : "block"));
};


sensor_begin:                             SENSOR_T ID EOL
{
    sensor_begin($2);
}
                                        | SENSOR_T EOL
{
    sensor_begin(NULL);
};

sensor_end:                               END_SENSOR_T EOL
{
    sensor_end();
}
                                        | END_GROUP_T EOL
{
    ++defn_errors;
    mdSkpcParseErr("%s used to close sensor", pcscan_clause);
    sensor_end();
}
                                        | END_PROBE_T EOL
{
    ++defn_errors;
    mdSkpcParseErr("%s used to close sensor", pcscan_clause);
    sensor_end();
};

stmt_sensor_isp_ip:                       ISP_IP_T id_list EOL
{
    sensor_isp_ip($2);
}
                                        | ISP_IP_T EOL
{
    missing_value();
};

stmt_sensor_interface:                    NET_NAME_INTERFACE id_list EOL
{
    sensor_interface($1, $2);
}
                                        | NET_NAME_INTERFACE REMAINDER_T EOL
{
    sensor_interface($1, NULL);
}
                                        | NET_NAME_INTERFACE EOL
{
    missing_value();
    if ($1) {
        free($1);
    }
};

stmt_sensor_ipblock:                      NET_NAME_IPBLOCK id_list EOL
{
    sensor_ipblock($1, $2);
}
                                        | NET_NAME_IPBLOCK REMAINDER_T EOL
{
    sensor_ipblock($1, NULL);
}
                                        | NET_NAME_IPBLOCK EOL
{
    missing_value();
    if ($1) {
        free($1);
    }
};

stmt_sensor_ipset:                        NET_NAME_IPSET id_list EOL
{
    sensor_ipset($1, $2);
}
                                        | NET_NAME_IPSET filename_list EOL
{
    sensor_ipset($1, $2);
}
                                        | NET_NAME_IPSET REMAINDER_T EOL
{
    sensor_ipset($1, NULL);
}
                                        | NET_NAME_IPSET EOL
{
    missing_value();
    if ($1) {
        free($1);
    }
};

stmt_sensor_filter:                       FILTER id_list EOL
{
    /* discard-{when,unless}
     * {source,destination,any}-{interfaces,ipblocks,ipsets} */
    sensor_filter($1, $2, 0);
}
                                        | FILTER filename_list EOL
{
    /* discard-{when,unless}
     * {source,destination,any}-ipsets */
    sensor_filter($1, $2, 1);
}
                                        | FILTER EOL
{
    missing_value();
};

stmt_sensor_network:                      NET_DIRECTION ID EOL
{
    sensor_network($1, $2);
}
                                        | NET_DIRECTION EOL
{
    missing_value();
};

stmt_sensor_probes:                       PROBES id_list EOL
{
    sensor_probes($1, $2);
}
                                        | PROBES EOL
{
    missing_value();
    if ($1) {
        free($1);
    }
};


    /*
     *  A group
     */

group_defn:                               group_begin group_stmts group_end
;

group_stmts:                            /* empty */
                                        | group_stmts group_stmt
;

group_stmt:                               stmt_group_interfaces
                                        | stmt_group_ipblocks
                                        | stmt_group_ipsets
                                        | error
{
    ++defn_errors;
    mdSkpcParseErr(("Error in group %s:"
                    " Missing \"end group\" or invalid keyword or value"),
                   (group ? mdSkpcGroupGetName(group) : "block"));
};


group_begin:                              GROUP_T ID EOL
{
    group_begin($2);
}
                                        | GROUP_T EOL
{
    group_begin(NULL);
};

group_end:                                END_GROUP_T EOL
{
    group_end();
}
                                        | END_PROBE_T EOL
{
    ++defn_errors;
    mdSkpcParseErr("%s used to close group", pcscan_clause);
    group_end();
};
                                        | END_SENSOR_T EOL
{
    ++defn_errors;
    mdSkpcParseErr("%s used to close group", pcscan_clause);
    group_end();
};

stmt_group_interfaces:                    INTERFACES_T id_list EOL
{
    group_add_data($2, MD_SKPC_GROUP_INTERFACE);
}
                                        | INTERFACES_T EOL
{
    missing_value();
};

stmt_group_ipblocks:                      IPBLOCKS_T id_list EOL
{
    group_add_data($2, MD_SKPC_GROUP_IPBLOCK);
}
                                        | IPBLOCKS_T EOL
{
    missing_value();
};

stmt_group_ipsets:                        IPSETS_T id_list EOL
{
    group_add_data($2, MD_SKPC_GROUP_IPSET);
}
                                        | IPSETS_T filename_list EOL
{
    group_add_data($2, MD_SKPC_GROUP_IPSET);
}
                                        | IPSETS_T EOL
{
    missing_value();
};


    /*
     *  Parse a list of strings, which are separated by whitespace
     *  and/or commas.  Return as a GArray*
     */

id_list:                                  ID
{
    GArray *v = g_array_sized_new(FALSE, TRUE, sizeof(void *), DEFAULT_ARRAY_SIZE);
    char *s = $1;
    g_array_append_val(v, s);
    $$ = v;
}
                                        | id_list ID
{
    char *s = $2;
    g_array_append_val($1, s);
    $$ = $1;
}
                                        | id_list COMMA ID
{
    char *s = $3;
    g_array_append_val($1, s);
    $$ = $1;
};


    /*
     *  Parse a list of strings where at least one of the items is a
     *  double-quoted string.  The items are separated by whitespace
     *  and/or commas.  Return as a GArray*
     */

filename_list:                            QUOTED_STRING
{
    GArray *v = g_array_sized_new(FALSE, TRUE, sizeof(void *), DEFAULT_ARRAY_SIZE);
    char *s = $1;
    g_array_append_val(v, s);
    $$ = v;
}
                                        | id_list QUOTED_STRING
{
    char *s = $2;
    g_array_append_val($1, s);
    $$ = $1;
}
                                        | id_list COMMA QUOTED_STRING
{
    char *s = $3;
    g_array_append_val($1, s);
    $$ = $1;
}
                                        | filename_list filename
{
    char *s = $2;
    g_array_append_val($1, s);
    $$ = $1;
}
                                        | filename_list COMMA filename
{
    char *s = $3;
    g_array_append_val($1, s);
    $$ = $1;
};

filename:                                 ID | QUOTED_STRING;


%%
/*  Reenable uncrustify */
/*  *INDENT-ON* */

/*
 * *******************   SUPPORTING CODE  ******************************
 */


static void
probeconf_free_garray_string(
    GArray  *v)
{
    guint  i;
    char  *s;

    for (i = 0; i < v->len; ++i) {
        s = g_array_index(v, char *, i);
        free(s);
    }
    g_array_free(v, TRUE);
}


static void
missing_value(
    void)
{
    ++defn_errors;
    mdSkpcParseErr("Missing arguments for %s statement", pcscan_clause);
}


static void
include_file(
    char  *filename)
{
    mdSkpcParseIncludePush(filename);
}



/*
 *  *****  Probes  *****************************************************
 */


/* complete the current probe */
static void
probe_end(
    void)
{
    if (!probe) {
        mdSkpcParseErr("No active probe in %s statement", pcscan_clause);
        goto END;
    }

    if (defn_errors) {
        goto END;
    }

    if (!mdSkpcProbeVerify(probe, 0)) {
        mdSkpcParseErr("Unable to verify probe '%s'",
                       mdSkpcProbeGetName(probe));
        ++defn_errors;
        goto END;
    }

    /* Probe is valid and now owned by probeconf.  Set to NULL so it
     * doesn't get free()'ed. */
    probe = NULL;

  END:
    if (defn_errors) {
        fprintf(stderr, "Encountered %d error%s while processing probe '%s'\n",
                defn_errors, ((defn_errors == 1) ? "" : "s"),
                (probe ? mdSkpcProbeGetName(probe) : ""));
        pcscan_errors += defn_errors;
        defn_errors = 0;
    }
    if (probe) {
        mdSkpcProbeDestroy(&probe);
        probe = NULL;
    }
    if (listen_as_address) {
        free(listen_as_address);
        listen_as_address = NULL;
    }
    if (listen_port) {
        free(listen_port);
        listen_port = NULL;
    }
}


/* Begin a new probe by setting the memory of the global probe_attr_t
 * into its initial state. */
static void
probe_begin(
    char  *probe_name,
    char  *probe_type)
{
    const char          *dummy_name = "<NONAME>";
    md_skpc_probetype_t  t;

    if (probe) {
        mdSkpcParseErr("Found active probe in %s statement", pcscan_clause);
        mdSkpcProbeDestroy(&probe);
        probe = NULL;
    }
    defn_errors = 0;

    /* probe_name will only be NULL on bad input from user */
    if (NULL == probe_name) {
        mdSkpcParseErr("%s requires a name and a type", pcscan_clause);
        ++defn_errors;
        t = MD_PROBE_ENUM_NETFLOW_V5;
    } else {
        if (mdSkpcProbeLookupByName(probe_name)) {
            mdSkpcParseErr("A probe named '%s' already exists", probe_name);
            ++defn_errors;
        }

        t = mdSkpcProbetypeNameToEnum(probe_type);
        if (t == MD_PROBE_ENUM_INVALID) {
            mdSkpcParseErr("Do not recognize probe type '%s'", probe_type);
            ++defn_errors;
            t = MD_PROBE_ENUM_NETFLOW_V5;
        }
    }

    if (mdSkpcProbeCreate(&probe, t)) {
        mdSkpcParseErr("Fatal: Unable to create probe");
        exit(EXIT_FAILURE);
    }

    /* probe_name will only be NULL on bad input from user */
    if (probe_name == NULL) {
        if (probe_type == NULL) {
            mdSkpcProbeSetName(probe, dummy_name);
        } else if (mdSkpcProbeSetName(probe, probe_type)) {
            mdSkpcParseErr("Error setting probe name to %s", probe_type);
            ++defn_errors;
        }
        goto END;
    }

    if (mdSkpcProbeSetName(probe, probe_name)) {
        mdSkpcParseErr("Error setting probe name to %s", probe_name);
        ++defn_errors;
    }
    free(probe_name);

    if (listen_as_address != NULL) {
        free(listen_as_address);
        listen_as_address = NULL;
    }
    if (listen_port) {
        free(listen_port);
        listen_port = NULL;
    }

  END:
    free(probe_type);
}


/*
 *  probe_priority(s);
 *
 *    Set the priority of the global probe to s.
 */
static void
probe_priority(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_protocol(s);
 *
 *    Set the probe-type of the global probe to 's'.
 */
static void
probe_protocol(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_listen_as_host(s);
 *
 *    Set the global probe to listen for flows as the host IP 's'.
 */
static void
probe_listen_as_host(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_listen_on_port(s);
 *
 *    Set the global probe to listen for flows on port 's'.
 */
static void
probe_listen_on_port(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_listen_on_usocket(s);
 *
 *    Set the global probe to listen for flows on the unix domain socket 's'.
 */
static void
probe_listen_on_usocket(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_read_from_file(s);
 *
 *    Set the global probe to read flows from the file 's'.
 */
static void
probe_read_from_file(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_poll_directory(s);
 *
 *    Set the global probe to read flows from files that appear in the
 *    directory 's'.
 */
static void
probe_poll_directory(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_accept_from_host(list);
 *
 *    Set the global probe to accept data from the hosts in 'list'.
 */
static void
probe_accept_from_host(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_log_flags(list);
 *
 *    Set the log flags on the probe to 'n';
 */
static void
probe_log_flags(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  probe_interface_values(s);
 *
 *    Set the global probe to store either snmp or vlan values in the
 *    'input' and 'output' interface fields on SiLK Flow records.
 */
static void
probe_interface_values(
    GArray  *v)
{
    md_skpc_ifvaluetype_t  ifvalue = MD_SKPC_IFVALUE_SNMP;
    char                  *s;

    if (1 != v->len) {
        mdSkpcParseErr("The %s clause takes a single argument", pcscan_clause);
        ++defn_errors;
        goto END;
    }

    s = g_array_index(v, char *, 0);
    if (0 == strcmp(s, "snmp")) {
        ifvalue = MD_SKPC_IFVALUE_SNMP;
    } else if (0 == strcmp(s, "vlan")) {
        ifvalue = MD_SKPC_IFVALUE_VLAN;
    } else {
        mdSkpcParseErr("Invalid %s value '%s'",
                       pcscan_clause, s);
        ++defn_errors;
        goto END;
    }

    if (mdSkpcProbeSetInterfaceValueType(probe, ifvalue)) {
        mdSkpcParseErr("Unable to set %s value '%s'",
                       pcscan_clause, s);
        ++defn_errors;
        goto END;
    }

  END:
    probeconf_free_garray_string(v);
}


/*
 *  probe_quirks(list);
 *
 *    Set the "quirks" field on the global probe to the values in list.
 */
static void
probe_quirks(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  *****  Sensors  ****************************************************
 */


static void
sensor_end(
    void)
{
    if (!sensor) {
        mdSkpcParseErr("No active sensor in %s statement", pcscan_clause);
        goto END;
    }

    if (defn_errors) {
        goto END;
    }

    if (!mdSkpcSensorVerify(sensor)) {
        mdSkpcParseErr("Unable to verify sensor '%s'",
                       mdSkpcSensorGetName(sensor));
        ++defn_errors;
        goto END;
    }

    /* Sensor is valid and now owned by probeconf.  Set to NULL so it
     * doesn't get free()'ed. */
    sensor = NULL;

  END:
    if (defn_errors) {
        fprintf(stderr, "Encountered %d error%s while processing sensor '%s'\n",
                defn_errors, ((defn_errors == 1) ? "" : "s"),
                (sensor ? mdSkpcSensorGetName(sensor) : ""));
        pcscan_errors += defn_errors;
        defn_errors = 0;
    }
    if (sensor) {
        mdSkpcSensorDestroy(&sensor);
        sensor = NULL;
    }
}


/* Begin a new sensor by setting the memory of the global probe_attr_t
 * into its initial state. */
static void
sensor_begin(
    char  *sensor_name)
{
    const char *dummy_name = "<ERROR>";

    if (sensor) {
        mdSkpcParseErr("Found active sensor in %s statement", pcscan_clause);
        mdSkpcSensorDestroy(&sensor);
        sensor = NULL;
    }
    defn_errors = 0;

    if (mdSkpcSensorCreate(&sensor)) {
        mdSkpcParseErr("Fatal: Unable to create sensor");
        exit(EXIT_FAILURE);
    }

    /* sensor_name will only be NULL on bad input from user */
    if (sensor_name == NULL) {
        mdSkpcParseErr("%s requires a sensor name", pcscan_clause);
        ++defn_errors;
        mdSkpcSensorSetName(sensor, dummy_name);
    } else {
#if 0
        if (mdSkpcSensorLookupByName(sensor_name)) {
            mdSkpcParseErr("A sensor named '%s' already exists", sensor_name);
            ++defn_errors;
        }
#endif /* if 0 */

        if (mdSkpcSensorSetName(sensor, sensor_name)) {
            mdSkpcParseErr("Error setting sensor name to %s", sensor_name);
            ++defn_errors;
        }

        if (SK_INVALID_SENSOR == mdSkpcSensorGetID(sensor)) {
            mdSkpcParseErr("There is no known sensor named %s", sensor_name);
            ++defn_errors;
        }

        free(sensor_name);
    }
}


/*
 *  sensor_isp_ip(list);
 *
 *    Set the isp-ip's on the global sensor to 'list'.
 */
static void
sensor_isp_ip(
    GArray  *v)
{
    probeconf_free_garray_string(v);
}


/*
 *  sensor_interface(name, list);
 *
 *    Set the interface list for the network whose name is 'name' on
 *    the global sensor to 'list'.
 *
 *    If 'list' is NULL, set the interface list to all the indexes NOT
 *    listed on other interfaces---set it to the 'remainder' of the
 *    interfaces.
 */
static void
sensor_interface(
    char    *name,
    GArray  *v)
{
    const md_skpc_network_t *network = NULL;
    const size_t             count = (v ? v->len : 0);
    md_skpc_group_t         *g;
    char                    *s;
    size_t                   i;

    if (name == NULL) {
        mdSkpcParseErr("Interface list '%s' gives a NULL name", pcscan_clause);
        skAbort();
    }

    /* convert the name to a network */
    network = mdSkpcNetworkLookupByName(name);
    if (network == NULL) {
        mdSkpcParseErr(("Cannot set %s for sensor '%s' because\n"
                        "\tthe '%s' network is not defined"),
                       pcscan_clause, mdSkpcSensorGetName(sensor), name);
        ++defn_errors;
        goto END;
    }

    /* NULL vector indicates want to set network to 'remainder' */
    if (v == NULL) {
        if (mdSkpcSensorSetNetworkRemainder(
                sensor, network->id, MD_SKPC_GROUP_INTERFACE))
        {
            ++defn_errors;
        }
    } else {
        /* determine if we are using a single existing group */
        if (1 == count) {
            s = g_array_index(v, char *, 0);
            if ('@' == *s) {
                g = get_group(s + 1, MD_SKPC_GROUP_INTERFACE);
                if (NULL == g) {
                    goto END;
                }
                if (mdSkpcSensorSetNetworkGroup(sensor, network->id, g)) {
                    ++defn_errors;
                }
                goto END;
            }
        }

        /* not a single group, so need to create a new group */
        if (mdSkpcGroupCreate(&g)) {
            mdSkpcParseErr("Allocation error near %s", pcscan_clause);
            ++defn_errors;
            goto END;
        }
        mdSkpcGroupSetType(g, MD_SKPC_GROUP_INTERFACE);

        /* parse the strings and add them to the group */
        if (add_values_to_group(g, v, MD_SKPC_GROUP_INTERFACE)) {
            v = NULL;
            goto END;
        }
        v = NULL;

        /* add the group to the sensor */
        if (mdSkpcGroupFreeze(g)) {
            ++defn_errors;
            goto END;
        }
        if (mdSkpcSensorSetNetworkGroup(sensor, network->id, g)) {
            ++defn_errors;
        }
    }

  END:
    if (name) {
        free(name);
    }
    if (v) {
        for (i = 0; i < count; ++i) {
            s = g_array_index(v, char *, i);
            free(s);
        }
        g_array_free(v, TRUE);
    }
}


/*
 *  sensor_ipblock(name, ip_list);
 *
 *    When 'ip_list' is NULL, set a flag for the network 'name' noting
 *    that its ipblock list should be set to any IP addresses not
 *    covered by other IP blocks; ie., the remaining ipblocks.
 *
 *    Otherwise, set the ipblocks for the 'name'
 *    network of the global sensor to the inverse of 'ip_list'.
 */
static void
sensor_ipblock(
    char    *name,
    GArray  *v)
{
    const size_t             count = (v ? v->len : 0);
    const md_skpc_network_t *network = NULL;
    md_skpc_group_t         *g;
    size_t                   i;
    char                    *s;

    if (name == NULL) {
        mdSkpcParseErr("IP Block list '%s' gives a NULL name", pcscan_clause);
        skAbort();
    }

    /* convert the name to a network */
    network = mdSkpcNetworkLookupByName(name);
    if (network == NULL) {
        mdSkpcParseErr(("Cannot set %s for sensor '%s' because\n"
                        "\tthe '%s' network is not defined"),
                       pcscan_clause, mdSkpcSensorGetName(sensor), name);
        ++defn_errors;
        goto END;
    }

    /* NULL vector indicates want to set network to 'remainder' */
    if (v == NULL) {
        if (mdSkpcSensorSetNetworkRemainder(
                sensor, network->id, MD_SKPC_GROUP_IPBLOCK))
        {
            ++defn_errors;
        }
    } else {
        /* determine if we are using a single existing group */
        if (1 == count) {
            s = g_array_index(v, char *, 0);
            if ('@' == *s) {
                g = get_group(s + 1, MD_SKPC_GROUP_IPBLOCK);
                if (NULL == g) {
                    goto END;
                }
                if (mdSkpcSensorSetNetworkGroup(sensor, network->id, g)) {
                    ++defn_errors;
                }
                goto END;
            }
        }

        /* not a single group, so need to create a new group */
        if (mdSkpcGroupCreate(&g)) {
            mdSkpcParseErr("Allocation error near %s", pcscan_clause);
            ++defn_errors;
            goto END;
        }
        mdSkpcGroupSetType(g, MD_SKPC_GROUP_IPBLOCK);

        /* parse the strings and add them to the group */
        if (add_values_to_group(g, v, MD_SKPC_GROUP_IPBLOCK)) {
            v = NULL;
            goto END;
        }
        v = NULL;

        /* add the group to the sensor */
        if (mdSkpcGroupFreeze(g)) {
            ++defn_errors;
            goto END;
        }
        if (mdSkpcSensorSetNetworkGroup(sensor, network->id, g)) {
            ++defn_errors;
            goto END;
        }
    }

  END:
    if (name) {
        free(name);
    }
    if (v) {
        for (i = 0; i < count; ++i) {
            s = g_array_index(v, char *, i);
            free(s);
        }
        g_array_free(v, TRUE);
    }
}


/*
 *  sensor_ipset(name, ip_list);
 *
 *    When 'ip_list' is NULL, set a flag for the network 'name' noting
 *    that its ipset list should be set to any IP addresses not
 *    covered by other IP sets; ie., the remaining ipsets.
 *
 *    Otherwise, set the ipsets for the 'name' network of the global
 *    sensor to the inverse of 'ip_list'.
 */
static void
sensor_ipset(
    char    *name,
    GArray  *v)
{
    const size_t             count = (v ? v->len : 0);
    const md_skpc_network_t *network = NULL;
    md_skpc_group_t         *g;
    size_t                   i;
    char                    *s;

    if (name == NULL) {
        mdSkpcParseErr("IP Set list '%s' gives a NULL name", pcscan_clause);
        skAbort();
    }

    /* convert the name to a network */
    network = mdSkpcNetworkLookupByName(name);
    if (network == NULL) {
        mdSkpcParseErr(("Cannot set %s for sensor '%s' because\n"
                        "\tthe '%s' network is not defined"),
                       pcscan_clause, mdSkpcSensorGetName(sensor), name);
        ++defn_errors;
        goto END;
    }

    /* NULL vector indicates want to set network to 'remainder' */
    if (v == NULL) {
        if (mdSkpcSensorSetNetworkRemainder(
                sensor, network->id, MD_SKPC_GROUP_IPSET))
        {
            ++defn_errors;
        }
    } else {
        /* determine if we are using a single existing group */
        if (1 == count) {
            s = g_array_index(v, char *, 0);
            if ('@' == *s) {
                g = get_group(s + 1, MD_SKPC_GROUP_IPSET);
                if (NULL == g) {
                    goto END;
                }
                if (mdSkpcSensorSetNetworkGroup(sensor, network->id, g)) {
                    ++defn_errors;
                }
                goto END;
            }
        }

        /* not a single group, so need to create a new group */
        if (mdSkpcGroupCreate(&g)) {
            mdSkpcParseErr("Allocation error near %s", pcscan_clause);
            ++defn_errors;
            goto END;
        }
        mdSkpcGroupSetType(g, MD_SKPC_GROUP_IPSET);

        /* parse the strings and add them to the group */
        if (add_values_to_group(g, v, MD_SKPC_GROUP_IPSET)) {
            v = NULL;
            goto END;
        }
        v = NULL;

        /* add the group to the sensor */
        if (mdSkpcGroupFreeze(g)) {
            ++defn_errors;
            goto END;
        }
        if (mdSkpcSensorSetNetworkGroup(sensor, network->id, g)) {
            ++defn_errors;
            goto END;
        }
    }

  END:
    if (name) {
        free(name);
    }
    if (v) {
        for (i = 0; i < count; ++i) {
            s = g_array_index(v, char *, i);
            free(s);
        }
        g_array_free(v, TRUE);
    }
}


static void
sensor_filter(
    md_skpc_filter_t   filter,
    GArray            *v,
    int                is_files)
{
    const size_t     count = (v ? v->len : 0);
    md_skpc_group_t *g;
    size_t           i;
    char            *s;

    if (count < 1) {
        mdSkpcParseErr("Missing arguments for %s on sensor '%s'",
                       pcscan_clause, mdSkpcSensorGetName(sensor));
        ++defn_errors;
        goto END;
    }

    if (is_files && MD_SKPC_GROUP_IPSET != filter.f_group_type) {
        mdSkpcParseErr(("Error in %s on sensor '%s':"
                        " Only IPset filenames may be quoted"),
                       pcscan_clause, mdSkpcSensorGetName(sensor));
        ++defn_errors;
        goto END;
    }

    /* determine if we are using a single existing group */
    if (1 == count) {
        s = g_array_index(v, char *, 0);
        if ('@' == *s) {
            g = get_group(s + 1, filter.f_group_type);
            if (NULL == g) {
                goto END;
            }
            if (mdSkpcSensorAddFilter(sensor, g, filter.f_type,
                                      filter.f_discwhen, filter.f_group_type))
            {
                ++defn_errors;
            }
            goto END;
        }
    }

    /* not a single group, so need to create a new group */
    if (mdSkpcGroupCreate(&g)) {
        mdSkpcParseErr("Allocation error near %s", pcscan_clause);
        ++defn_errors;
        goto END;
    }
    mdSkpcGroupSetType(g, filter.f_group_type);

    /* parse the strings in 'v' and add them to the group */
    if (add_values_to_group(g, v, filter.f_group_type)) {
        v = NULL;
        goto END;
    }
    v = NULL;

    /* add the group to the filter */
    if (mdSkpcGroupFreeze(g)) {
        ++defn_errors;
        goto END;
    }
    if (mdSkpcSensorAddFilter(sensor, g, filter.f_type, filter.f_discwhen,
                              filter.f_group_type))
    {
        ++defn_errors;
    }

  END:
    if (v) {
        for (i = 0; i < count; ++i) {
            s = g_array_index(v, char *, i);
            free(s);
        }
        g_array_free(v, TRUE);
    }
}


static void
sensor_network(
    md_skpc_direction_t   direction,
    char                 *name)
{
    const md_skpc_network_t *network = NULL;

    if (name == NULL) {
        mdSkpcParseErr("Missing network name in %s on sensor '%s'",
                       pcscan_clause, mdSkpcSensorGetName(sensor));
        ++defn_errors;
        goto END;
    }

    /* convert the name to a network */
    network = mdSkpcNetworkLookupByName(name);
    if (network == NULL) {
        mdSkpcParseErr(("Cannot set %s for sensor '%s' because\n"
                        "\tthe '%s' network is not defined"),
                       pcscan_clause, mdSkpcSensorGetName(sensor), name);
        ++defn_errors;
        goto END;
    }

    if (mdSkpcSensorSetNetworkDirection(sensor, network->id, direction)) {
        mdSkpcParseErr("Cannot set %s for sensor '%s' to %s",
                       pcscan_clause, mdSkpcSensorGetName(sensor), name);
        ++defn_errors;
        goto END;
    }

  END:
    if (name) {
        free(name);
    }
}


static void
sensor_probes(
    char    *probe_type,
    GArray  *v)
{
    GArray                *pl;
    size_t                 i = 0;
    char                  *s;
    const md_skpc_probe_t *p;
    md_skpc_probetype_t    t;

    /* get a vector to use for the probe objects */
    pl = g_array_sized_new(FALSE, TRUE, sizeof(void *), DEFAULT_ARRAY_SIZE);

    /* get the probe-type */
    t = mdSkpcProbetypeNameToEnum(probe_type);
    if (t == MD_PROBE_ENUM_INVALID) {
        mdSkpcParseErr("Do not recognize probe type '%s'", probe_type);
        ++defn_errors;
        goto END;
    }

    for (i = 0; i < v->len; ++i) {
        s = g_array_index(v, char *, i);
        p = mdSkpcProbeLookupByName(s);
        if (p) {
            if (mdSkpcProbeGetType(p) != t) {
                mdSkpcParseErr("Attempt to use %s probe '%s' in a %s statement",
                               mdSkpcProbetypeEnumtoName(mdSkpcProbeGetType(p)),
                               s, pcscan_clause);
                ++defn_errors;
            }
        } else {
            /* Create a new probe with the specified name and type */
            md_skpc_probe_t *new_probe;
            if (mdSkpcProbeCreate(&new_probe, t)) {
                mdSkpcParseErr("Fatal: Unable to create ephemeral probe");
                exit(EXIT_FAILURE);
            }
            if (mdSkpcProbeSetName(new_probe, s)) {
                mdSkpcParseErr("Error setting ephemeral probe name to %s", s);
                ++defn_errors;
                goto END;
            }
            if (!mdSkpcProbeVerify(new_probe, 1)) {
                mdSkpcParseErr("Error verifying ephemeral probe '%s'", s);
                ++defn_errors;
                goto END;
            }
            p = mdSkpcProbeLookupByName(s);
            if (p == NULL) {
                mdSkpcParseErr("Cannot find newly created ephemeral probe '%s'",
                               s);
                skAbort();
            }
        }
        g_array_append_val(pl, p);
    }

    if (mdSkpcSensorSetProbes(sensor, pl)) {
        ++defn_errors;
    }

  END:
    free(probe_type);
    for (i = 0; i < v->len; ++i) {
        s = g_array_index(v, char *, i);
        free(s);
    }
    g_array_free(v, TRUE);
    g_array_free(pl, TRUE);
}


/*
 *  *****  Groups  ****************************************************
 */


static void
group_end(
    void)
{
    if (!group) {
        mdSkpcParseErr("No active group in %s statement", pcscan_clause);
        goto END;
    }

    if (defn_errors) {
        goto END;
    }

    if (mdSkpcGroupFreeze(group)) {
        mdSkpcParseErr("Unable to freeze group '%s'",
                       mdSkpcGroupGetName(group));
        ++defn_errors;
        goto END;
    }

    /* Group is valid and now owned by probeconf.  Set to NULL so it
     * doesn't get free()'ed. */
    group = NULL;

  END:
    if (defn_errors) {
        fprintf(stderr, "Encountered %d error%s while processing group '%s'\n",
                defn_errors, ((defn_errors == 1) ? "" : "s"),
                (group ? mdSkpcGroupGetName(group) : ""));
        pcscan_errors += defn_errors;
        defn_errors = 0;
    }
    if (group) {
        mdSkpcGroupDestroy(&group);
        group = NULL;
    }
}


/* Begin a new group by setting the memory of the global probe_attr_t
 * into its initial state. */
static void
group_begin(
    char  *group_name)
{
    const char *dummy_name = "<ERROR>";

    if (group) {
        mdSkpcParseErr("Found active group in %s statement", pcscan_clause);
        mdSkpcGroupDestroy(&group);
        group = NULL;
    }
    defn_errors = 0;

    if (mdSkpcGroupCreate(&group)) {
        mdSkpcParseErr("Fatal: Unable to create group");
        exit(EXIT_FAILURE);
    }

    /* group_name will only be NULL on bad input from user */
    if (group_name == NULL) {
        mdSkpcParseErr("%s requires a group name", pcscan_clause);
        ++defn_errors;
        mdSkpcGroupSetName(group, dummy_name);
    } else {
        if (mdSkpcGroupLookupByName(group_name)) {
            mdSkpcParseErr("A group named '%s' already exists", group_name);
            ++defn_errors;
        }
        if (mdSkpcGroupSetName(group, group_name)) {
            mdSkpcParseErr("Error setting group name to %s", group_name);
            ++defn_errors;
        }
        free(group_name);
    }
}


/*
 *  group_add_data(v, g_type);
 *
 *   Verify that the global group has a type of 'g_type'.  If so,
 *   parse the string values in 'v' and add the values to the global
 *   group.
 *
 *   If the global group's type is not set, the value to 'g_type' and
 *   append the values.
 *
 *   Used by 'stmt_group_interfaces', 'stmt_group_ipblocks', and
 *   'stmt_group_ipsets'
 */
static void
group_add_data(
    GArray                *v,
    md_skpc_group_type_t   g_type)
{
    const char *g_type_str = "unknown data";
    size_t      i;
    char       *s;

    switch (mdSkpcGroupGetType(group)) {
      case MD_SKPC_GROUP_UNSET:
        mdSkpcGroupSetType(group, g_type);
        break;
      case MD_SKPC_GROUP_INTERFACE:
        g_type_str = "interface values";
        break;
      case MD_SKPC_GROUP_IPBLOCK:
        g_type_str = "ipblocks";
        break;
      case MD_SKPC_GROUP_IPSET:
        g_type_str = "ipsets";
        break;
    }

    if (g_type != mdSkpcGroupGetType(group)) {
        mdSkpcParseErr(("Cannot add %s to group because\n"
                        "\tthe group already contains %s"),
                       pcscan_clause, g_type_str);
        ++defn_errors;
        goto END;
    }

    add_values_to_group(group, v, g_type);
    v = NULL;

  END:
    if (v) {
        for (i = 0; i < v->len; ++i) {
            s = g_array_index(v, char *, i);
            free(s);
        }
        g_array_free(v, TRUE);
    }
}


static int
add_values_to_group(
    md_skpc_group_t       *g,
    GArray                *v,
    md_skpc_group_type_t   g_type)
{
    const size_t     count = (v ? v->len : 0);
    md_skpc_group_t *named_group;
    GArray          *vec = NULL;
    char            *s;
    size_t           i = 0;
    uint32_t         n;
    skcidr_t        *cidr;
    skipset_t       *ipset;
    int              rv = -1;

    /* determine the GArray to use for the parsed values */
    if (MD_SKPC_GROUP_INTERFACE == g_type) {
        /* parse numbers and/or groups */
        vec = g_array_sized_new(FALSE, TRUE, sizeof(uint32_t),
                                DEFAULT_ARRAY_SIZE);
    } else if (MD_SKPC_GROUP_IPBLOCK == g_type) {
        /* parse ipblocks and/or groups */
        vec = g_array_sized_new(FALSE, TRUE, sizeof(void *),
                                DEFAULT_ARRAY_SIZE);
    } else if (MD_SKPC_GROUP_IPSET == g_type) {
        /* parse ipsets and/or groups */
        vec = g_array_sized_new(FALSE, TRUE, sizeof(void *),
                                DEFAULT_ARRAY_SIZE);
    } else {
        skAbortBadCase(g_type);
    }

    /* process the strings in the vector 'v' */
    for (i = 0; i < count; ++i) {
        s = g_array_index(v, char *, i);

        /* is this a group? */
        if ('@' == *s) {
            named_group = get_group(s + 1, g_type);
            free(s);
            if (NULL == named_group) {
                ++i;
                goto END;
            }
            if (mdSkpcGroupAddGroup(g, named_group)) {
                ++defn_errors;
                ++i;
                goto END;
            }
        } else if (g_type == MD_SKPC_GROUP_IPBLOCK) {
            cidr = parse_cidr_addr(s);
            if (cidr == NULL) {
                ++defn_errors;
                ++i;
                goto END;
            }
            g_array_append_val(vec, cidr);
        } else if (g_type == MD_SKPC_GROUP_IPSET) {
            ipset = parse_ipset_filename(s);
            if (ipset == NULL) {
                ++defn_errors;
                ++i;
                goto END;
            }
            g_array_append_val(vec, ipset);
        } else if (MD_SKPC_GROUP_INTERFACE == g_type) {
            assert(g_type == MD_SKPC_GROUP_INTERFACE);
            n = parse_int_u32(s);
            if (n == UINT32_NO_VALUE) {
                ++defn_errors;
                ++i;
                goto END;
            }
            g_array_append_val(vec, n);
        }
    }

    /* add values to the group */
    if (mdSkpcGroupAddValues(g, vec)) {
        ++defn_errors;
    }

    rv = 0;

  END:
    for ( ; i < count; ++i) {
        s = g_array_index(v, char *, i);
        free(s);
    }
    if (v) {
        g_array_free(v, TRUE);
    }
    if (vec) {
        if (g_type == MD_SKPC_GROUP_IPSET) {
            for (i = 0; i < vec->len; ++i) {
                ipset = g_array_index(vec, skipset_t *, i);
                if (ipset) {
                    skIPSetDestroy(&ipset);
                }
            }
        }
        g_array_free(vec, TRUE);
    }
    return rv;
}


static md_skpc_group_t *
get_group(
    const char            *g_name,
    md_skpc_group_type_t   g_type)
{
    md_skpc_group_t *g;

    g = mdSkpcGroupLookupByName(g_name);
    if (!g) {
        mdSkpcParseErr("Error in %s: group '%s' is not defined",
                       pcscan_clause, g_name);
        ++defn_errors;
        return NULL;
    }
    if (mdSkpcGroupGetType(g) != g_type) {
        mdSkpcParseErr(("Error in %s: the '%s' group does not contain %ss"),
                       pcscan_clause, g_name,
                       mdSkpcGrouptypeEnumtoName(g_type));
        ++defn_errors;
        return NULL;
    }
    return g;
}


/*
 *  *****  Parsing Utilities  ******************************************
 */


/*
 *  val = parse_int_u32(s);
 *
 *    Parse 's' as a integer from 0 to 0xFFFFFFFF inclusive.  Returns the
 *    value on success.  Prints an error and returns UINT32_NO_VALUE
 *    if parsing is unsuccessful or value is out of range.
 */
static uint32_t
parse_int_u32(
    char  *s)
{
    uint32_t       num;
    unsigned long  val;

    /* parse the string */
    errno = 0;
    val = strtoul(s, NULL, 10);
    if (ULONG_MAX == val && errno == ERANGE) {
        /* overflow */
        mdSkpcParseErr("Invalid %s '%s': Number overflows the parser",
                       pcscan_clause, s);
        num = UINT32_NO_VALUE;
    } else if (val > UINT32_MAX) {
        mdSkpcParseErr("Invalid %s '%s': Value is larger than %" PRIu32,
                       pcscan_clause, s, UINT32_MAX);
        num = UINT32_NO_VALUE;
    } else {
        num = (uint32_t)val;
    }

    free(s);
    return num;
}


/*
 *  cidr = parse_cidr_addr(ip);
 *
 *    Parse 'ip' as an IP address with optional CIDR length.
 *
 *    Return the set of ips as an skcidr_t*, or NULL on error.
 */
static skcidr_t *
parse_cidr_addr(
    char  *s)
{
    skipaddr_t  ipaddr;
    skcidr_t   *cidr;
    uint32_t    mask;
    int         rv;

    rv = skStringParseCIDR(&ipaddr, &mask, s);
    if (0 != rv) {
        mdSkpcParseErr("Invalid IP address block '%s': %s",
                       s, skStringParseStrerror(rv));
        cidr = NULL;
    } else {
        cidr = g_slice_new0(skcidr_t);
        skcidrSetFromIPAddr(cidr, &ipaddr, mask);
    }

    free(s);
    return cidr;
}


/*
 *  ipset = parse_ipset_filename(filename);
 *
 *    Treat 'filename' as the name of an IPset file.  Load the file
 *    and return a pointer to it.  Return NULL on failure.
 */
static skipset_t *
parse_ipset_filename(
    char  *s)
{
    skipset_t *ipset;
    ssize_t    rv;

    /* reject standard input */
    if (0 == strcmp(s, "-") || (0 == strcmp(s, "stdin"))) {
        mdSkpcParseErr("May not read an IPset from the standard input");
        ipset = NULL;
        goto END;
    }

    rv = skIPSetLoad(&ipset, s);
    if (rv) {
        mdSkpcParseErr("Unable to read IPset from '%s': %s",
                       s, skIPSetStrerror(rv));
        ipset = NULL;
    }
    if (skIPSetCountIPs(ipset, NULL) == 0) {
        mdSkpcParseErr("May not use the IPset in '%s': IPset is empty", s);
        skIPSetDestroy(&ipset);
        ipset = NULL;
    }

  END:
    free(s);
    return ipset;
}


int
yyerror(
    const char  *s)
{
    SK_UNUSED_PARAM(s);
    return 0;
}


int
mdSkpcParseSetup(
    void)
{
    return 0;
}


void
mdSkpcParseTeardown(
    void)
{
    if (probe) {
        ++defn_errors;
        mdSkpcParseErr("Missing \"end probe\" statement");
        mdSkpcProbeDestroy(&probe);
        probe = NULL;
    }
    if (sensor) {
        ++defn_errors;
        mdSkpcParseErr("Missing \"end sensor\" statement");
        mdSkpcSensorDestroy(&sensor);
        sensor = NULL;
    }
    if (group) {
        ++defn_errors;
        mdSkpcParseErr("Missing \"end group\" statement");
        mdSkpcGroupDestroy(&group);
        group = NULL;
    }

    pcscan_errors += defn_errors;
}
