#! /bin/sh
#
##  Copyright 2021-2025 Carnegie Mellon University
##  See license information in LICENSE.txt.
##
usage() {
  cat <<__HERE__
$0 OUTPUT_FILE

This script uses YAF to create an IPFIX file for use in testing
super_mediator.  To build the IPFIX file, the script uses four PCAP
files that are part of the Flaming Cupcake Challenge data available
from https://tools.netsa.cert.org/silk/referencedata.html#FCCX15

The script looks for the "yaf" program on your PATH.  Alternatively,
set the YAF environmnet variable to the full path to the "yaf" program
to use.

To use the script, set the FCCX_PCAP_DIR environmnet variable to the
full path of the "FCCX-data" directory that is created when the
"FCCX-pcap.tar.gz" file is expanded.

The script requires the MERGECAP tool from wireshark to join the pcap
files together.  The script looks for the "mergecap" program on your
PATH.  Alternatively, set the MERGECAP environmnet variable to the
full path to the "mergecap" program to use.

Arguments:

    OUTPUT_FILE
        The path to the output file to produce.

ENVIRONMENT

    This program requires or uses the following variables in your
    environment as described above.

    FCCX_DATA_DIR
        Path holding the Flaming Cupcake Challenge PCAP files.

    MERGECAP
        The location of the "mergecap" program ; overrides location on
        your PATH.

    YAF
        The location of the "yaf" program; overrides location on your
        PATH.

__HERE__
}

if [ $# -lt 1 ]; then
    usage
    exit 1
elif [ "$1" = "-h" ] || [ "$1" = "--help" ]; then
    usage
    exit 0
fi

#  Exit on error
set -e

OUTPUT_FILE="$1"
shift


#    Check for yaf
YAF="${YAF-yaf}"

if ! "${YAF}" --help-all >/dev/null ; then
    echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
    echo "$0: Unable to run '${YAF} --help-all'; check the MERGECAP environment variable" 1>&2
    exit 1;
fi


#    Check for mergecap
MERGECAP="${MERGECAP-mergecap}"

if ! "${MERGECAP}" -h >/dev/null 2>&1 ; then
    echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
    echo "$0: Unable to run '${MERGECAP} -h'; check the MERGECAP environment variable" 1>&2
    exit 1
fi


#    Check the pcap directory
if test -z "${FCCX_PCAP_DIR}" ; then
    echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
    echo "$0: FCCX_PCAP_DIR is unset; set it to the location of the Flaming Cupcake Challenge data" 1>&2
    exit 1
fi
if test ! -d "${FCCX_PCAP_DIR}" ; then
    echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
    echo "$0: The FCCX_PCAP_DIR location (${FCCX_PCAP_DIR}) is not a directory" 1>&2
    exit 1
fi


#    Find pcap files in the directory, also checking for compressed versions

#   The pcap files from the Flaming Cupcake Challenge
PCAP_FILES=("gatewaySensor-2.pcap"
            "gatewaySensor-3.pcap"
            "marSensor.pcap"
            "intel2Sensor.pcap")

#   Allow the pcap files to be compressed with what wireshark supports
SUFFIXES=("" ".gz" ".zst" ".lz4")

###echo "PCAP_FILES = ${PCAP_FILES[@]}"

PCAP_PATHS=()
for p in "${PCAP_FILES[@]}" ; do
    found=0
    for suffix in "${SUFFIXES[@]}" ; do
        if test -f "${FCCX_PCAP_DIR}/$p$suffix" ; then
            PCAP_PATHS+=("${FCCX_PCAP_DIR}/$p$suffix")
            found=1
            break
        fi
    done
    if [ 0 -eq "$found" ]; then
        echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
        echo "$0: Cannot find file '$p' in '${FCCX_PCAP_DIR}'" 1>&2
        echo "$0: Also tried with these suffixes:${SUFFIXES[@]}" 1>&2
        exit 1
    fi
done

####echo "PATH_PATHS = ${PCAP_PATHS[@]}"


#   Run mergecap, creating the outputpcap file
outputpcap=$( mktemp )

####echo Running "${MERGECAP}" -F pcap -w "${outputpcap}" "${PCAP_PATHS[@]}" 1>&2
if ! "${MERGECAP}" -F pcap -w "${outputpcap}" "${PCAP_PATHS[@]}" ; then
    echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
    echo "$0: Failed to run $MERGECAP" 1>&2
    exit 1
fi


#   Build the YAF command

YAF_ARGS="--max-payload=65500 --applabel"
YAF3_ARGS="--dpi"
YAF2_ARGS="--plugin-name=dpacketplugin.la"

if "${YAF}" --help-all | grep -e --dpi >/dev/null 2>&1 ; then
    echo "$0: Warning: Using yaf-3.x for super_mediator-1.x" 1>&2
    YAF_CMD="${YAF} ${YAF_ARGS} ${YAF3_ARGS}"
elif "${YAF}" --help-all | grep -e --metadata-export >/dev/null 2>&1 ; then
    YAF_CMD="${YAF} ${YAF_ARGS} ${YAF2_ARGS} --metadata-export"
elif "${YAF}" --help-all | grep -e --template-info >/dev/null 2>&1 ; then
    YAF_CMD="${YAF} ${YAF_ARGS} ${YAF2_ARGS} --template-info"
else
    YAF_CMD="${YAF} ${YAF_ARGS} ${YAF2_ARGS}"
fi


#   Run yaf, reading outputpcap and creating yafout
yaflog=$( mktemp )
yafout=$( mktemp )

####echo Running ${YAF_CMD} --verbose --log="${yaflog}" --out="${yafout}" 1>&2
if ! ${YAF_CMD} --verbose --in="${outputpcap}" --log="${yaflog}" --out="${yafout}" ; then
    echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
    echo "$0: Failed to run $YAF" 1>&2
    exit 1
fi

if [ ! -s "$yafout" ]; then
    echo "$0: Unable to create $OUTPUT_FILE; tests may be skipped" 1>&2
    echo "$0: Output file created by $YAF has size 0" 1>&2
    exit 1
fi


#   Move file into place
mv "${yafout}" "${OUTPUT_FILE}"

#   Cleanup
rm -f "$outputpcap" "$yaflog"

exit


##  ------------------------------------------------------------------------
##  @DISTRIBUTION_STATEMENT_BEGIN@
##  super_mediator-1.13
##
##  Copyright 2025 Carnegie Mellon University.
##
##  NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
##  INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
##  UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
##  AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS FOR
##  PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF
##  THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF
##  ANY KIND WITH RESPECT TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT
##  INFRINGEMENT.
##
##  Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or
##  contact permission@sei.cmu.edu for full terms.
##
##  [DISTRIBUTION STATEMENT A] This material has been approved for public
##  release and unlimited distribution.  Please see Copyright notice for
##  non-US Government use and distribution.
##
##  This Software includes and/or makes use of Third-Party Software each
##  subject to its own license.
##
##  DM25-1447
##  @DISTRIBUTION_STATEMENT_END@
##  ------------------------------------------------------------------------
