/*
** Copyright (C) 2004-2020 by Carnegie Mellon University.
**
** @OPENSOURCE_LICENSE_START@
** See license information in ../../LICENSE.txt
** @OPENSOURCE_LICENSE_END@
*/

/*
**  probeconf.h
**
**    Functions to parse a probe configuration file and use the
**    results.
**
*/

#ifndef _PROBECONF_H
#define _PROBECONF_H

#include <mediator/mediator_ctx.h>
#include <mediator/templates.h>
#include <silk/silk_types.h>


/**
 *  @file
 *
 *    Functions to parse a probe configuration file and use the
 *    results.
 *
 *    This file is part of libflowsource.
 *
 *
 *    Lifecycle:
 *
 *    The application calls mdSkpcSetup() to initialize the
 *    skpc data structures and memory.
 *
 *    The application should call mdSkpcParse() to parse the
 *    application's configuration file.  mdSkpcParse() will create
 *    sensors (if any) and probes.  The probes are created and checked
 *    for validity--this means they have all the data they require.
 *    If valid they are added to the list maintained by the skpc.
 *    If not valid, they are destroyed.
 *
 *    Once the probes have been created, the application can use
 *    mdSkpcProbeIteratorBind() and mdSkpcProbeIteratorNext() to process
 *    each probe.
 *
 *    Finally, the application calls mdSkpcTeardown() to destroy
 *    the probes, sensors, and to fee all memory.
 *
 *    Note that skpc allows one to create a "temporary" sensor;
 *    i.e., a sensor that will only exist as long as the application
 *    is running; this is useful for testing a new sensor without
 *    requiring a complete recompile of SiLK.  However, "temporary"
 *    sensors will NOT be available to the analysis applications.  For
 *    the analysis applications to know about a sensor, it MUST be
 *    listed in the sensorInfo[] array.
 */


#define DEFAULT_ARRAY_SIZE  16

/**
 *    Values for the type of a probe.
 */
typedef enum {
    MD_PROBE_ENUM_INVALID = 0,
    MD_PROBE_ENUM_IPFIX = 10,
    MD_PROBE_ENUM_NETFLOW_V5 = 5,
    MD_PROBE_ENUM_NETFLOW_V9 = 9,
    MD_PROBE_ENUM_SFLOW = 16,
    MD_PROBE_ENUM_SILK = 15
} md_skpc_probetype_t;


/**
 *    The type for network ids
 */
typedef size_t md_skpc_network_id_t;

/**
 *    The maximum possible network ID
 */
#define MD_SKPC_NETWORK_ID_MAX     ((md_skpc_network_id_t)254)

/**
 *    The invalid network ID
 */
#define MD_SKPC_NETWORK_ID_INVALID ((md_skpc_network_id_t)255)


/**
 *   Which "side" of the record we look at when testing its flow
 *   interfaces, whether
 *
 *   -- its source is a particular network; i.e., it is COMING FROM an
 *      internet cloud.  For this case, look at its source IP or input
 *      SNMP interface.
 *
 *   -- its destination is a particular network; i.e., it is GOING TO
 *      a cloud.  For this case, look at the destination IP or output
 *      SNMP interface.
 */
typedef enum {
    MD_SKPC_DIR_SRC = 0, MD_SKPC_DIR_DST = 1
} md_skpc_direction_t;


/**
 *    The "type" of value that the probe stores in the input and
 *    output fields.
 *
 *    A value of 'MD_SKPC_IFVALUE_SNMP' signifies that those fields hold
 *    the index of the interface (ifIndex) where the flows entered and
 *    left the router, respectively.
 *
 *    A value of 'MD_SKPC_IFVALUE_VLAN' signifies that those fields hold
 *    the vlanIds for the source and destination networks,
 *    respectively.  If only vlan Id is available, the 'input' is set
 *    to that value and the 'output' is set to 0.
 */
typedef enum {
    MD_SKPC_IFVALUE_SNMP = 0,
    MD_SKPC_IFVALUE_VLAN = 1
} md_skpc_ifvaluetype_t;

/*  Forward declaration */
typedef struct md_skpc_sensor_st md_skpc_sensor_t;


/**
 *    The network definition.
 *
 *    Maps a name to an ID.
 */
typedef struct md_skpc_network_st {
    char                   *name;
    md_skpc_network_id_t    id;
} md_skpc_network_t;


/**
 *    The probe definition.
 *
 *    A probe tells how to collect data and the type of data.  For
 *    example, IPFIX data from machine 10.10.10.10 as TCP to port
 *    9999.  A probe is associated with one or more sensors.
 */
typedef struct md_skpc_probe_st {

    /** List of sensors (md_skpc_sensor_t*) to which this probe belongs, and a
     * count of those sensors */
    GArray                 *sensor_list;

    /** the name of the probe */
    const char             *probe_name;

    /** Probe type */
    md_skpc_probetype_t     probe_type;

    /** Type of the interface value */
    md_skpc_ifvaluetype_t   ifvaluetype;

    /** Has probe been verified */
    unsigned                verified :1;

} md_skpc_probe_t;


/**
 *  A 'group'
 *
 *    A 'group' may contain one of the following: (1)a list of
 *    interface numbers, (2)a list of CIDR Blocks, (3)an IPset.
 *
 *    A group is created by giving it a list containing values or
 *    previously defined groups.
 */
typedef enum {
    MD_SKPC_GROUP_UNSET,
    MD_SKPC_GROUP_INTERFACE,
    MD_SKPC_GROUP_IPBLOCK,
    MD_SKPC_GROUP_IPSET
} md_skpc_group_type_t;

/**
 *    Number of different types of groups.
 */
#define MD_SKPC_NUM_GROUP_TYPES 3

typedef struct md_skpc_group_st {
    /** groups have an optional name */
    char               *g_name;
    /** the contents of the group */
    union md_skpc_group_value_un {
        /** Array of uin32_t for interfaces or skcidr_t* for ipblocks. */
        GArray             *vec;
        /** An IPset */
        skipset_t          *ipset;
    }                   g_value;
    /** number of items in the group */
    uint32_t            g_itemcount;
    /** the type of the group */
    md_skpc_group_type_t   g_type;
    /** once frozen, a group cannot be changed */
    int8_t              g_is_frozen;
} md_skpc_group_t;



/**
 *  The 'decider'.
 *
 *    This describes the logic that the sensor will use to determine
 *    (decide) the flowtype (class/type) of each flow.  The type will
 *    depend on whether the sensor.conf file lists interfaces,
 *    ipblocks, or IPsets for the sensor.
 */
typedef enum {
    /** no interface, ipblock, or ipset values seen */
    MD_SKPC_UNSET,
    /** *-interface (SNMP) value seen */
    MD_SKPC_INTERFACE,
    /** *-ipblock value seen */
    MD_SKPC_IPBLOCK,
    /** ipblock is inverted */
    MD_SKPC_NEG_IPBLOCK,
    /** *-ipset value seen */
    MD_SKPC_IPSET,
    /** ipset is inverted */
    MD_SKPC_NEG_IPSET,
    /** sensor.conf has "*-interface remainder" line */
    MD_SKPC_REMAIN_INTERFACE,
    /** sensor.conf has "*-ipblock remainder" line */
    MD_SKPC_REMAIN_IPBLOCK,
    /** sensor.conf has "*-ipset remainder" line */
    MD_SKPC_REMAIN_IPSET
} md_skpc_netdecider_type_t;

typedef struct md_skpc_netdecider_st {
    md_skpc_netdecider_type_t  nd_type;
    const md_skpc_group_t     *nd_group;
} md_skpc_netdecider_t;

/**    number of 'decider' types */
#define MD_SKPC_NUM_NETDECIDER_TYPES  9


/**
 *  A filter
 *
 *    A filter is similar to the decider in that it accepts a list of
 *    interfaces, ipblocks, or IPsets.  However, instead of being used
 *    to decide the flowtype, a filter is used to determine whether
 *    rwflowpack should even consider the flow.  A filter can match
 *    the 'source' (either source IP or input interface), the
 *    'destination' (either destination IP or output interface), or
 *    'any' (any of the above).  Filters are set in the sensor.conf
 *    file by using the 'discard-when' and 'discard-unless'
 *    statements.
 */
typedef enum {
    MD_SKPC_FILTER_SOURCE, MD_SKPC_FILTER_DESTINATION, MD_SKPC_FILTER_ANY
} md_skpc_filter_type_t;

typedef struct md_skpc_filter_st {
    /** the value to use as the filter */
    const md_skpc_group_t  *f_group;
    /** the part of the flow record to use */
    md_skpc_filter_type_t   f_type;
    /** the type of the group in 'f_group' */
    md_skpc_group_type_t    f_group_type;
    /** if non-zero, discard flows that match the value in 'f_group'.
     * if zero, discard flows that do NOT match the value */
    unsigned                f_discwhen :1;
} md_skpc_filter_t;

/**    number of 'filter' types */
#define MD_SKPC_NUM_FILTER_TYPES 3



/**
 *  The sensor definition.
 *
 *    The sensor takes the flows from one or more probes and
 *    determines how to pack them---i.e., their flowtype or
 *    class/type.
 */
struct md_skpc_sensor_st {

    /**
     * An array of network-deciders (md_skpc_netdecider_t), one for each of
     * the networks defined for this site.  For example, a normal border
     * router that has the INTERNAL, EXTERNAL, and NULL networks would have 3
     * valid elements. */
    GArray                 *decider;

    /**
     * An array of probes (md_skpc_probe_t *) associated with this sensor and
     * the number of entries in that list */
    GArray                 *probe_list;

    /**
     * The name of the sensor */
    char                   *sensor_name;

    /**
     * An array of filters (md_skpc_filter_t). */
    GArray                 *filter;

    /**
     * The source and destination networks, if they have been set to a
     * fixed value. */
    md_skpc_network_id_t    fixed_network[2];

    /**
     * The sensor ID as defined in the silk.conf file. */
    sk_sensor_id_t          sensor_id;
};


/**
 *  Iterators over probes and sensors
 */
typedef struct md_skpc_probe_iter_st {
    size_t cur;
} md_skpc_probe_iter_t;

typedef struct md_skpc_sensor_iter_st {
    size_t cur;
} md_skpc_sensor_iter_t;



/*
 *  *****  Probe configuration  **************************************
 */


/**
 *    Initialize the probe configuration data structures.
 */
int
mdSkpcSetup(
    void);


/**
 *    Destroy all probes and sensors and free all memory used by the
 *    probe configuration.
 */
void
mdSkpcTeardown(
    void);


/**
 *    Parse the probe configuration file 'filename'.  This should only
 *    be called one time.
 *
 *    This function will parse the configuration file and create
 *    sensors and probes.
 */
int
mdSkpcParse(
    const char             *filename);


/**
 *    Return the count of created and verified probes.
 */
size_t
mdSkpcCountProbes(
    void);


/**
 *    Bind 'probe_iter' to loop over all the probes that have been
 *    defined.  Returns 0 on success, or -1 on error.
 */
int
mdSkpcProbeIteratorBind(
    md_skpc_probe_iter_t   *probe_iter);


/**
 *    If the probe iterator 'probe_iter' has exhausted all probes,
 *    leave 'probe' untouched and return 0; otherwise, fill 'probe'
 *    with a pointer to the next verified probe and return 1.  Returns
 *    -1 on error (such as NULL input).  The caller should not modify
 *    or free the probe.
 */
int
mdSkpcProbeIteratorNext(
    md_skpc_probe_iter_t   *probe_iter,
    const md_skpc_probe_t **probe);


/**
 *    Returns the probe named 'probe_name'.  Returns NULL if not
 *    found.  The caller should not modify nor free the return value.
 */
const md_skpc_probe_t *
mdSkpcProbeLookupByName(
    const char             *probe_name);


/**
 *    Return the count of created and verified sensors.
 */
size_t
mdSkpcCountSensors(
    void);


/**
 *    Bind 'sensor_iter' to loop over all the sensors that have been
 *    defined.  Returns 0 on success, or -1 on error.
 */
int
mdSkpcSensorIteratorBind(
    md_skpc_sensor_iter_t  *sensor_iter);


/**
 *    If the sensor iterator 'sensor_iter' has exhausted all sensors,
 *    leave 'sensor' untouched and return 0; otherwise, fill 'sensor'
 *    with a pointer to the next verified sensor and return 1.  Returns
 *    -1 on error (such as NULL input).  The caller should not modify
 *    or free the sensor.
 */
int
mdSkpcSensorIteratorNext(
    md_skpc_sensor_iter_t      *sensor_iter,
    const md_skpc_sensor_t    **sensor);


/**
 *    Appends to 'sensor_vec' the sensors whose ID is 'sensor_id'.
 *    Returns the number of sensors added to 'sensor_vec'.  Returns -1
 *    to indicate invalid input or memory error appending to the
 *    vector.  'sensor_vec' should be a vector having elements of size
 *    sizeof(md_skpc_sensor_t*).  The caller should not modify nor free
 *    the values appended to the vector.
 */
int
mdSkpcSensorLookupByID(
    sk_sensor_id_t          sensor_id,
    GArray                 *sensor_vec);


/**
 *    Appends to 'sensor_vec' the sensors whose name is 'sensor_name'.
 *    Returns the number of sensors added to 'sensor_vec'.  Returns -1
 *    to indicate invalid input or memory error appending to the
 *    vector.  'sensor_vec' should be a vector having elements of size
 *    sizeof(md_skpc_sensor_t*).  The caller should not modify nor free
 *    the values appended to the vector.
 */
int
mdSkpcSensorLookupByName(
    const char             *sensor_name,
    GArray                 *sensor_vec);


/**
 *    Given a printable representation of a probe, return the probe
 *    type.
 *
 *    Return PROBE_ENUM_INVALID when given an unrecognized name.
 */
md_skpc_probetype_t
mdSkpcProbetypeNameToEnum(
    const char             *name);

/**
 *    Return the printable respresentation of the probe type.
 *
 *    Return NULL when given an illegal value.
 */
const char *
mdSkpcProbetypeEnumtoName(
    md_skpc_probetype_t     type);


/*
 *  *****  Networks  ***************************************************
 */


/**
 *    Add a (id, name) pair to the list of networks used when
 *    determining the flowtype (class/type) of a flow record.
 */
void
mdSkpcNetworkAdd(
    md_skpc_network_id_t    network_id,
    const char             *name);

/**
 *    Return the network object that was created with the name
 *    attribute set to 'name'.  Returns NULL if no such network
 *    exists.
 */
const md_skpc_network_t *
mdSkpcNetworkLookupByName(
    const char             *name);


/**
 *    Return the network object that was created with the id attribute
 *    set to 'id'.  Returns NULL if no such network exists.
 */
const md_skpc_network_t *
mdSkpcNetworkLookupByID(
    md_skpc_network_id_t    network_id);



/*
 *  *****  Probes  *****************************************************
 *
 *
 *  Flows are stored by SENSOR; a SENSOR is a logical construct made
 *  up of one or more physical PROBES.
 *
 *  A probe collects flows in one of three ways:
 *
 *  1.  The probe can listen to network traffic.  For this case,
 *  mdSkpcProbeGetListenOnSockaddr() will return the port on which to
 *  listen for traffic and the IP address that the probe should bind()
 *  to.  In addition, the mdSkpcProbeGetAcceptFromHost() method will
 *  give the IP address from which the probe should accept
 *  connections.
 *
 *  2.  The probe can listen on a UNIX domain socket.  The
 *  mdSkpcProbeGetListenOnUnixDomainSocket() method returns the pathname
 *  to the socket.
 *
 *  3.  The probe can read from a file.  The mdSkpcProbeGetFileSource()
 *  method returns the name of the file.
 *
 *  A probe is not valid it has been set to use one and only one of
 *  these collection methods.
 *
 *  Once the probe has collected a flow, it needs to determine whether
 *  the flow represents incoming traffic, outgoing traffic, ACL
 *  traffic, etc.  The packLogicDetermineFlowtype() will take an
 *  'flow' and the probe where the record was collected and use the
 *  external, internal, and null interface values and the list of ISP
 *  IPs to update the 'flow_type' field on the flow.  The flow's
 *  sensor id ('sID') field is also updated.
 *
 */


/**
 *    Create a new probe of type 'probe_type' and set the referent of
 *    'probe' to the newly allocated probe.  Return 0 on success.
 *    Return -1 if 'probe_type' is unknown or if the allocation fails.
 */
int
mdSkpcProbeCreate(
    md_skpc_probe_t       **probe,
    md_skpc_probetype_t     probe_type);


/**
 *    Destroy the probe at '**probe' and free all memory.  Sets *probe
 *    to NULL.  Does nothing if 'probe' or the location it points to
 *    is NULL.
 */
void
mdSkpcProbeDestroy(
    md_skpc_probe_t       **probe);


/**
 *    Return the name of a probe.  The caller should not modify the
 *    name, and does not need to free() it.
 */
#define mdSkpcProbeGetName(m_probe)       ((m_probe)->probe_name)
#ifndef mdSkpcProbeGetName
const char *
mdSkpcProbeGetName(
    const md_skpc_probe_t  *probe);
#endif  /* mdSkpcProbeGetName */


/**
 *    Set the name of a probe.  The probe name must
 *    meet all the requirements of a sensor name.  Each probe that is
 *    a collection point for a single sensor must have a unique name.
 *
 *    The function makes a copy of 'name' and returns 0 on
 *    success, non-zero on memory allocation failure.
 */
int
mdSkpcProbeSetName(
    md_skpc_probe_t        *probe,
    const char             *name);


/**
 *    Return the type of the probe.  Before it is set by the user, the
 *    probe's type is PROBE_ENUM_INVALID.
 */
#define mdSkpcProbeGetType(m_probe)       ((m_probe)->probe_type)
#ifndef mdSkpcProbeGetType
md_skpc_probetype_t
mdSkpcProbeGetType(
    const md_skpc_probe_t  *probe);
#endif  /* mdSkpcProbeGetType */


/**
 *    Determine whether the probe is currently configured to store
 *    SNMP interfaces or VLAN tags.
 */
#define mdSkpcProbeGetInterfaceValueType(m_probe) \
    ((m_probe)->ifvaluetype)
#ifndef mdSkpcProbeGetInterfaceValueType
md_skpc_ifvaluetype_t
mdSkpcProbeGetInterfaceValueType(
    const md_skpc_probe_t  *probe);
#endif  /* mdSkpcProbeGetInterfaceValueType */


/**
 *    Set the type of value that the probe stores in the 'input' and
 *    'output' fields on the SiLK flow records; specifically SNMP
 *    values or VLAN tags.
 *
 *    If not set by the user, the probe stores the SNMP interfaces in
 *    the input and output fields.
 */
int
mdSkpcProbeSetInterfaceValueType(
    md_skpc_probe_t        *probe,
    md_skpc_ifvaluetype_t   interface_value_type);


/**
 *    Return a count of sensors that are using this probe.
 */
#define mdSkpcProbeGetSensorCount(m_probe)    ((m_probe)->sensor_count)
#ifndef mdSkpcProbeGetSensorCount
size_t
mdSkpcProbeGetSensorCount(
    const md_skpc_probe_t  *probe);
#endif  /* mdSkpcProbeGetSensorCount */


/**
 *    Return TRUE if probe has been verified; FALSE otherwise.
 */
gboolean
mdSkpcProbeIsVerified(
    const md_skpc_probe_t  *probe);


/**
 *    Verify the 'probe' is valid.  For example, that it's name is
 *    unique among all probes, and that if it is an IPFIX probe,
 *    verify that a listen-on-port has been specified.
 *
 *    When 'is_ephemeral' is specified, the function only verifies
 *    that is name is unique.  If the name is unique, the probe will
 *    be added to the global list of probes, but mdSkpcProbeIsVerified()
 *    on the probe will return FALSE.
 *
 *    If valid, add the probe to the list of probes and return TRUE.
 *    Otherwise return FALSE.
 */
int
mdSkpcProbeVerify(
    md_skpc_probe_t        *probe,
    int                     is_ephemeral);


/**
 *    Print a one line summary of 'probe' using the print function
 *    'printer'.
 */
void
mdSkpcProbePrint(
    const md_skpc_probe_t  *probe,
    sk_msg_fn_t             printer);




/*
 *  *****  Sensors  ****************************************************
 */


/**
 *    Create a new sensor and fill in 'sensor' with the address of
 *    the newly allocated sensor.
 */
int
mdSkpcSensorCreate(
    md_skpc_sensor_t      **sensor);


/**
 *    Destroy the sensor at '**sensor' and free all memory.  Sets
 *    *sensor to NULL.  Does nothing if 'sensor' or the location it
 *    points to is NULL.
 */
void
mdSkpcSensorDestroy(
    md_skpc_sensor_t      **sensor);


/**
 *    Get the numeric ID of the sensor, as determined by the silk.conf
 *    file.
 */
#define mdSkpcSensorGetID(m_sensor)       ((m_sensor)->sensor_id)
#ifndef mdSkpcSensorGetID
sk_sensor_id_t
mdSkpcSensorGetID(
    const md_skpc_sensor_t *sensor);
#endif  /* mdSkpcSensorGetID */

/**
 *    Get the name of the sensor.  The caller should not modify the
 *    name, and does not need to free() it.
 */
#define mdSkpcSensorGetName(m_sensor)     ((m_sensor)->sensor_name)
#ifndef mdSkpcSensorGetName
const char *
mdSkpcSensorGetName(
    const md_skpc_sensor_t *sensor);
#endif  /* mdSkpcSensorGetName */

/**
 *    Set the name of a sensor.  The function makes a copy of 'name'
 *    and returns 0 on success, non-zero on memory allocation failure.
 */
int
mdSkpcSensorSetName(
    md_skpc_sensor_t       *sensor,
    const char             *name);


/**
 *    Add a new filter (discard-when, discard-unless) to 'sensor'.
 *    'filter_type' specifies what part of the record to match
 *    (source, destination, or any).
 *
 *    If 'is_discardwhen_list' is non-zero, the caller is adding a
 *    filter that, when matched, causes rwflowpack to discard the
 *    flow.  Otherwise, not matching the filter causes the flow to be
 *    discarded.
 *
 *    The 'group' must be frozen and it must contain values, otherwise the
 *    function returns -1.
 *
 *    Return 0 on success.  Return -1 on memory allocation error or
 *    when a filter for the specified 'filter_type' already exists.
 */
int
mdSkpcSensorAddFilter(
    md_skpc_sensor_t       *sensor,
    const md_skpc_group_t  *group,
    md_skpc_filter_type_t   filter_type,
    int                     is_discardwhen_list,
    md_skpc_group_type_t    group_type);


/**
 *    Specify that for all traffic seen at 'sensor' the direction
 *    'dir' of all traffic is assigned to the network 'network_id'.
 *    The list of 'network_id's is defined by the
 *    packing logic plug-in that is specified on the rwflowpack
 *    command line.
 *
 *    Here, "network" refers to one of the domains that are being
 *    monitored by the router or other flow collection software.  For
 *    example, a border router joins the internal and external
 *    networks, and a flow whose source IP is specified in the list of
 *    external network addresses will be considered incoming.
 *
 *    For example, to configure 'sensor' so that all its traffic is
 *    incoming (coming from the external network and going to the
 *    internal network), one would call this function twice, once to
 *    set the MD_SKPC_DIR_SRC to 'external' and again to set the
 *    MD_SKPC_DIR_DST to 'internal'.
 *
 *    This function conflicts with mdSkpcSensorSetNetworkGroup().
 */
int
mdSkpcSensorSetNetworkDirection(
    md_skpc_sensor_t       *sensor,
    md_skpc_network_id_t    network_id,
    md_skpc_direction_t     dir);


/**
 *    Function to set the list of interfaces or IPs associated with
 *    the network 'network_id' to those in the group 'group'.  The
 *    list of 'network_id's is defined by the packing logic plug-in
 *    that is specified on the rwflowpack command line.
 *
 *    Here, "network" refers to one of the domains that are being
 *    monitored by the router or other flow collection software.  For
 *    example, a border router joins the internal and external
 *    networks, and a flow whose source IP is specified in the list of
 *    external network addresses will be considered incoming.
 *
 *    The 'group' must be frozen.
 *
 *    The function returns 0 on success, or -1 if the group is NULL,
 *    not frozen, or empty.
 *
 *    The mdSkpcSensorSetNetworkGroup() function will return an error if
 *    it is called multiple times for the same network or if
 *    the mdSkpcSensorSetNetworkDirection() function has already assigned all
 *    source or destination traffic to this 'network_id'.
 */
int
mdSkpcSensorSetNetworkGroup(
    md_skpc_sensor_t       *sensor,
    md_skpc_network_id_t    network_id,
    const md_skpc_group_t  *ip_group);


/**
 *    Sets the list of interfaces or IPs that are part of the network
 *    'network_id' to all those that not assigned to another
 *    interface.  The list of 'network_id's is defined by the packing
 *    logic plug-in that is specified on the rwflowpack command line.
 */
int
mdSkpcSensorSetNetworkRemainder(
    md_skpc_sensor_t       *sensor,
    md_skpc_network_id_t    network_id,
    md_skpc_group_type_t    group_type);


/**
 *    Sets the group of SNMP interfaces that connect to the network
 *    whose ID is 'network_id' to 0, the SNMP interface value used by
 *    Cisco to designate a non-routed flow.  The network may not have
 *    been previously set.  Return 0 on success, or non-zero on
 *    failure.
 */
int
mdSkpcSensorSetDefaultNonrouted(
    md_skpc_sensor_t       *sensor,
    md_skpc_network_id_t    network_id);


/**
 *    Appends to 'out_probe_vec' all the probes defined on 'sensor'.
 *    Returns the number of probes defined on 'sensor'.  Returns 0 if
 *    no probes are defined or if there is a memory error appending
 *    the probes to the vector.  If 'out_probe_vec' is NULL, the count
 *    of probes on sensors is returned.  'out_probe_vec' should be a
 *    vector having elements of size sizeof(md_skpc_probe_t*).  The
 *    caller should not modify nor free the values appended to the
 *    vector.
 */
uint32_t
mdSkpcSensorGetProbes(
    const md_skpc_sensor_t *sensor,
    GArray                 *out_probe_vec);


/**
 *    Copy the probes listed in 'probe_vec' onto 'sensor'.
 *
 *    Return 0 on success, or -1 if 'probe_vec' is NULL or empty, or
 *    if memory allocation fails.
 */
int
mdSkpcSensorSetProbes(
    md_skpc_sensor_t       *sensor,
    const GArray           *probe_vec);


/**
 *    Count the number of SNMP interfaces that have been mapped to a
 *    flowtype on 'sensor'.  Will exclude the network ID
 *    'ignored_network_id'; pass MD_SKPC_NETWORK_ID_INVALID or a negative
 *    value in that parameter to ensure that all SNMP interfaces are
 *    counted.
 */
uint32_t
mdSkpcSensorCountNetflowInterfaces(
    const md_skpc_sensor_t *sensor,
    int                     ignored_network_id);


/**
 *    Test 'flow' against the 'network_id' interfaces---either the
 *    SNMP values or the IP-block values---on the 'sensor'.  The value
 *    'rec_dir' tells the function whether to check if the flow was
 *    coming from the specified 'network_id' or going to that network.
 *
 *    The function returns 1 if there is a match, -1 if there was not
 *    a match, and 0 if neither an IP block list nor an SNMP interface
 *    list was defined for the 'network_id'.
 *
 *    If 'rec_dir' is MD_SKPC_DIR_SRC, the function checks the record's
 *    sIP against the list of IP blocks for the 'network_id'.  If no
 *    IP blocks are defined for the specified 'network_id' on this
 *    'sensor', the function checks the record's SNMP input interface
 *    against the list of SNMP interfaces for the 'network_id'.  When
 *    'rec_dir' is MD_SKPC_DIR_DST, the record's dIP and SNMP output
 *    values are checked.
 */
int
mdSkpcSensorTestFlowInterfaces(
    const md_skpc_sensor_t *sensor,
    const mdFullFlow_t     *flow,
    md_skpc_network_id_t    network_id,
    md_skpc_direction_t     rec_dir);


/**
 *    Check whether 'flow' matches the filters specified on 'sensor'.
 *    Return 0 if the flow should be packed, or non-zero to discard
 *    the flow.
 *
 *    When 'flow' matches any "discard-when" filter on 'sensor', this
 *    function returns non-zero.  Otherwise, if no "discard-unless"
 *    filters exist, the function returns 0.
 *
 *    If any "discard-unless" filters exist on 'sensor', 'flow' must
 *    match EVERY one of them for the function to return 0.
 */
int
mdSkpcSensorCheckFilters(
    const md_skpc_sensor_t *sensor,
    const mdFullFlow_t     *flow);


/**
 *    Verify that 'sensor' is valid.  For example, that if its probe
 *    is an IPFIX probe, the sensor has ipblocks are defined, etc.  If
 *    'site_sensor_verify_fn' is defined, it will be called to verify
 *    the sensor for the current site.  That function should return 0
 *    if the sensor is valid, or non-zero if not valid.
 *
 *    Returns TRUE if it is valid, FALSE otherwise.
 */
gboolean
mdSkpcSensorVerify(
    md_skpc_sensor_t       *sensor);



/*
 *  *****  Groups  *****************************************************
 */


/**
 *    Create a new group and fill in 'group' with the address of the
 *    newly allocated group.
 */
int
mdSkpcGroupCreate(
    md_skpc_group_t       **group);


/**
 *    Destroy the group at '**group' and free all memory.  Sets *group
 *    to NULL.  Does nothing if 'group' or the location it points to
 *    is NULL.
 */
void
mdSkpcGroupDestroy(
    md_skpc_group_t       **group);


/**
 *    Specifies that no changes can be made to the group (other than
 *    destroying it).  Returns 0 on success, or -1 if there is a
 *    memory allocation failure.  (Freezing a group may allocate
 *    memory as data is rearranged in the group.)
 *
 *    Freezing a frozen group is no-op, and the function returns 0.
 */
int
mdSkpcGroupFreeze(
    md_skpc_group_t        *group);


/**
 *    Get the name of a group.
 *
 *    A group can be anonymous---that is, not have a name.  These
 *    groups are created when an interface or ipblock list is specified
 *    outside of a "group" block; for example, by listing integers (or
 *    multiple groups) on in "internal-interfaces" statement.  For
 *    these groups, the name is NULL.
 *
 *    The function returns the name of the group.  The caller
 *    should not modify the name, and does not need to free() it.
 */
const char *
mdSkpcGroupGetName(
    const md_skpc_group_t  *group);

/**
 *    Set the name of a group.  A group name must
 *    meet all the requirements of a sensor name.
 *
 *    The set function makes a copy of 'name' and returns 0 on
 *    success, non-zero on memory allocation failure.  The set
 *    function also returns non-zero if the group is frozen (see
 *    mdSkpcGroupFreeze()).
 */
int
mdSkpcGroupSetName(
    md_skpc_group_t        *group,
    const char             *group_name);


/**
 *    Get the groups's type; that is, whether it stores IP-blocks or
 *    interface values.
 *
 *    Until it has been set, the group's type is MD_SKPC_GROUP_UNSET.
 */
md_skpc_group_type_t
mdSkpcGroupGetType(
    const md_skpc_group_t  *group);

/**
 *    Set the groups's type.
 *
 *    The set function returns -1 if the type of the group has been
 *    previously set or if the group is frozen (see
 *    mdSkpcGroupFreeze()).
 */
int
mdSkpcGroupSetType(
    md_skpc_group_t        *group,
    md_skpc_group_type_t    group_type);


/**
 *    Add the values in 'vec' to the group 'group'.  If the type of
 *    'group' is MD_SKPC_GROUP_INTERFACE, 'vec' should contain
 *    uint32_t's.  If the type of 'group' is MD_SKPC_GROUP_IPBLOCK, 'vec'
 *    should contain pointers to skcidr_t.
 *
 *    When 'vec' contains skcidr_t*, this function assumes
 *    ownership of the data and will free the skcidr_t when
 *    mdSkpcTeardown() is called.
 *
 *    The function returns 0 on success.  It also returns 0 when 'vec'
 *    is NULL or contains no elements.
 *
 *    Return -1 if the group is frozen, if the group's type is
 *    MD_SKPC_GROUP_UNSET, if the size of elements in the 'vec' is not
 *    consistent with the group's type, or if there is a memory
 *    allocation error.
 */
int
mdSkpcGroupAddValues(
    md_skpc_group_t        *group,
    const GArray           *vec);


/**
 *    Add the contents of group 'g' to group 'group'.
 *
 *    Return 0 on success.  Also return 0 if 'g' is NULL or if
 *    contains no items.
 *
 *    Return -1 if 'group' is frozen, if 'g' is NOT frozen, if the
 *    groups' types are different, or if there is a memory allocation
 *    error.
 */
int
mdSkpcGroupAddGroup(
    md_skpc_group_t        *group,
    const md_skpc_group_t  *g);


/**
 *    Return 1 if 'group' is frozen; 0 otherwise.
 */
int
mdSkpcGroupIsFrozen(
    const md_skpc_group_t  *group);


/**
 *    Returns the group named 'group_name'.  Returns NULL if not found
 *    of if 'group_name' is NULL.  The returned group is frozen.
 */
md_skpc_group_t *
mdSkpcGroupLookupByName(
    const char             *group_name);


/**
 *    Return the printable respresentation of the group type.
 *
 *    Return NULL when given an illegal value.
 */
const char *
mdSkpcGrouptypeEnumtoName(
    md_skpc_group_type_t    type);


/**
 *    Whether the Type and Sensor Labeling is Enabled.  That is, was there a
 *    SILK_CONFIG block in the super_mediator config file and were the
 *    silk.conf and sensor.conf files successfully read?
 */
gboolean
mdSkpcSilkTypeSensorEnabled(
    void);

/**
 *    Set the location of the silk.conf and/or sensor.conf file.  Return TRUE
 *    on success.  Set `err` and return FALSE for the following errors: An
 *    argument is not a file (and not NULL), a value is provided for a file
 *    that was previously set set.
 */
gboolean
mdSkpcSilkTypeSensorSet(
    const char       *silk_conf,
    const char       *sensor_conf,
    GError          **err);


/**
 *    Confirm that both silk.conf and sensor.conf have been set via
 *    mdSkpcSilkTypeSensorSet(), bootstrap the SiLK world, and load the
 *    sensor.conf file.  Return TRUE on success; set `err` and return FALSE on
 *    error.
 */
gboolean
mdSkpcSilkTypeSensorLoad(
    GError        **err);


/**
 *    Destroy settings and free memory allocated by mdSkpcSetSilkTypeSensor()
 *    and mdSkpcLoadSilkTypeSensor().
 */
void
mdSkpcSilkTypeSensorFree(
    void);


/**
 *    Determine the sensor-type pair(s) for `flow` and return an array of ``
 *    containing them.  Return NULL if no values are set.
 */
GArray *
mdSkpcSilkTypeSensorLabelFlow(
    mdFullFlow_t   *flow);


/**
 *    Verify sensor according to the defined networks.  This is a helper
 *    function for mdSkpcSensorVerify().
 */
gboolean
mdSkpcVerifySensorLabeler(
    md_skpc_sensor_t   *sensor);

#endif /* _PROBECONF_H */
