dnl -*- autoconf -*-
dnl Copyright (C) 2004-2023 by Carnegie Mellon University.
dnl
dnl @OPENSOURCE_HEADER_START@
dnl
dnl Use of the SILK system and related source code is subject to the terms
dnl of the following licenses:
dnl
dnl GNU Public License (GPL) Rights pursuant to Version 2, June 1991
dnl Government Purpose License Rights (GPLR) pursuant to DFARS 252.227.7013
dnl
dnl NO WARRANTY
dnl
dnl ANY INFORMATION, MATERIALS, SERVICES, INTELLECTUAL PROPERTY OR OTHER
dnl PROPERTY OR RIGHTS GRANTED OR PROVIDED BY CARNEGIE MELLON UNIVERSITY
dnl PURSUANT TO THIS LICENSE (HEREINAFTER THE "DELIVERABLES") ARE ON AN
dnl "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY
dnl KIND, EITHER EXPRESS OR IMPLIED AS TO ANY MATTER INCLUDING, BUT NOT
dnl LIMITED TO, WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE,
dnl MERCHANTABILITY, INFORMATIONAL CONTENT, NONINFRINGEMENT, OR ERROR-FREE
dnl OPERATION. CARNEGIE MELLON UNIVERSITY SHALL NOT BE LIABLE FOR INDIRECT,
dnl SPECIAL OR CONSEQUENTIAL DAMAGES, SUCH AS LOSS OF PROFITS OR INABILITY
dnl TO USE SAID INTELLECTUAL PROPERTY, UNDER THIS LICENSE, REGARDLESS OF
dnl WHETHER SUCH PARTY WAS AWARE OF THE POSSIBILITY OF SUCH DAMAGES.
dnl LICENSEE AGREES THAT IT WILL NOT MAKE ANY WARRANTY ON BEHALF OF
dnl CARNEGIE MELLON UNIVERSITY, EXPRESS OR IMPLIED, TO ANY PERSON
dnl CONCERNING THE APPLICATION OF OR THE RESULTS TO BE OBTAINED WITH THE
dnl DELIVERABLES UNDER THIS LICENSE.
dnl
dnl Licensee hereby agrees to defend, indemnify, and hold harmless Carnegie
dnl Mellon University, its trustees, officers, employees, and agents from
dnl all claims or demands made against them (and any related losses,
dnl expenses, or attorney's fees) arising out of, or relating to Licensee's
dnl and/or its sub licensees' negligent use or willful misuse of or
dnl negligent conduct or willful misconduct regarding the Software,
dnl facilities, or other rights or assistance granted by Carnegie Mellon
dnl University under this License, including, but not limited to, any
dnl claims of product liability, personal injury, death, damage to
dnl property, or violation of any laws or regulations.
dnl
dnl Carnegie Mellon University Software Engineering Institute authored
dnl documents are sponsored by the U.S. Department of Defense under
dnl Contract FA8721-05-C-0003. Carnegie Mellon University retains
dnl copyrights in all material produced under this contract. The U.S.
dnl Government retains a non-exclusive, royalty-free license to publish or
dnl reproduce these documents, or allow others to do so, for U.S.
dnl Government purposes only pursuant to the copyright license under the
dnl contract clause at 252.227.7013.
dnl
dnl @OPENSOURCE_HEADER_END@


# ---------------------------------------------------------------------------
# AX_LIB_SKIPSET
#
#    Determine how to use silk or skipset.
#    Output variable: SKIPSET_LDFLAGS, SKIPSET_CFLAGS
#    Output definition: SKIPSET_HEADER_NAME, ENABLE_SKIPSET,
#                       ENABLE_SKPREFIXMAP, ENABLE_SKTYPESENSOR,
#                       HAVE_SKIPADDRSETV6,
#                       HAVE_SILK_SKIPADDR_H, HAVE_SILK_UTILS_H
#    Make conditionals: BUILD_PROBECONF
#
#    Note: If the SiLK library is used, the IPSet, PrefixMap, and
#    Type+Sensor labeling are all enabled.  The builder should be able
#    to enable these individually.
#
AC_DEFUN([AX_LIB_SKIPSET],[
    AC_SUBST(SKIPSET_LDFLAGS)
    AC_SUBST(SKIPSET_CFLAGS)

    AC_ARG_WITH([skipset],
    [AS_HELP_STRING(
        [--with-skipset@<:@=SILK_DIR@:>@],
        [include SiLK support [default=auto], optionally specify location of SiLK or SiLK-IPSet Library; find "silk/silk.h" or "silk-ipset/silk-ipset.h" in SILK_DIR/include/; find "libsilk.so" or "libskipset.so" in SILK_DIR/lib/]
    )],[
        if test "x${with_skipset}" != "xyes" -a "x${with_skipset}" != "xno" 
        then
            skipset_includes="${with_skipset}/include"
            skipset_libs="${with_skipset}/lib"
        fi
    ])

    AC_ARG_WITH([libsilk],
    [AS_HELP_STRING(
        [--without-libsilk],
        [include SiLK support but ONLY via the stand-only SiLK-IPSet library, not by using SiLK]
    )])

    AC_ARG_WITH([silk-ipset],
    [AS_HELP_STRING(
        [--without-silk-ipset],
        [when building with SiLK, disable SiLK-IPSet filtering and IPSet flow labeling]
    )],,[
        with_silk_ipset=yes
    ])

    AC_ARG_WITH([silk-pmap],
    [AS_HELP_STRING(
        [--without-silk-pmap],
        [when building with SiLK, disable SiLK Prefix Map (pmap) flow labeling]
    )],,[
        with_silk_pmap=yes
    ])

    AC_ARG_WITH([silk-type-sensor],
    [AS_HELP_STRING(
        [--without-silk-type-sensor],
        [when building with SiLK, disable SiLK Type and Sensor flow labeling]
    )],,[
        with_silk_type_sensor=yes
    ])


    search_lib='SiLK'
    if test "x${with_libsilk}" = xno
    then
        search_lib='SiLK-IPSet'
    fi


    # Check whether to disable the check for SiLK/SiLK-IPSet because
    # labeling has been disabled
    if test "x${with_skipset}" = xno
    then
        # All of silk is disabled
        :
    elif test "x${with_libsilk}" = xno
    then
        # libsilk is disabled; check for --without-silk-ipset
        if test "x${with_silk_ipset}" = xno
        then
            if test "x${with_skipset}" != x
            then
                AC_MSG_ERROR([conflicting options: --with-skipset specified but --without-libsilk --without-silk-ipset disables searching for it])
            fi
            labeling_disabled=yes
            with_skipset=no
            AC_MSG_NOTICE([not checking for SiLK-IPSet since IPSet labeling has been disabled])
        fi
    elif test "x${with_silk_ipset}-${with_silk_pmap}-${with_silk_type_sensor}" = "xno-no-no"
    then
        if test "x${with_skipset}" != x
        then
            AC_MSG_ERROR([conflicting options: --with-skipset specified but all labeling is disabled (--without-silk-ipset, etc)])
        fi
        AC_MSG_NOTICE([not checking for SiLK since all labeling has been disabled])
        labeling_disabled=yes
        with_skipset=no
    fi

    if test "x${with_skipset}" != "xno"
    then
        AC_MSG_NOTICE([checking for ${search_lib} library])

        # Cache current compiler and library flags
        skip_save_LDFLAGS="${LDFLAGS}"
        skip_save_LIBS="${LIBS}"
        skip_save_CFLAGS="${CFLAGS}"
        skip_save_CPPFLAGS="${CPPFLAGS}"

        if test "x${skipset_libs}" != "x"
        then
            SKIPSET_LDFLAGS="-L${skipset_libs}"
            LDFLAGS="${SKIPSET_LDFLAGS} ${skip_save_LDFLAGS}"
        fi

        if test "x${skipset_includes}" != "x"
        then
            SKIPSET_CFLAGS="-I${skipset_includes}"
            CPPFLAGS="${SKIPSET_CFLAGS} ${skip_save_CPPFLAGS}"
        fi

	# Check for libraries that SiLK may be using.  We should fix
	# all of this to use silk_config instead.
	sk_extra_libs=
	AC_CHECK_LIB([z], [gzopen],
	    [sk_extra_libs="-lz ${sk_extra_libs}"])
	AC_CHECK_LIB([snappy], [snappy_compress],
	    [sk_extra_libs="-lsnappy ${sk_extra_libs}"])
        AC_CHECK_LIB([lzo2], [lzo1x_1_15_compress],
	    [sk_extra_libs="-llzo2 ${sk_extra_libs}"],
            [AC_CHECK_LIB([lzo], [lzo1x_1_15_compress],
	        [sk_extra_libs="-llzo ${sk_extra_libs}"])])

        have_silk=no
        have_skipset=no

        if test "x${with_libsilk}" != xno
        then
            # Look for complete SiLK install first by checking for the
            # SiLK library and headers
            sk_ip_hdr=silk/skipset.h
            sk_ip_lib=silk
            AC_CHECK_LIB([${sk_ip_lib}], [skStreamReadRecord],
            [
                have_silk=yes
                LIBS="-l${sk_ip_lib} ${sk_extra_libs} ${LIBS}"
                AC_CHECK_FUNCS([skAppRegister skIPSetLoad skPrefixMapLoad sksiteConfigure],,
                [
                    have_silk=no
                ])
                if test "x${have_silk}" = xyes
                then
                    AC_CHECK_HEADERS([silk/silk_types.h silk/sksite.h silk/skipset.h silk/skprefixmap.h silk/skipaddr.h silk/utils.h],,
                    [
                        have_silk=no
                    ])
                fi
                if test "x${have_silk}" = xno
                then
                    AC_MSG_NOTICE([found ${sk_ip_lib} library but functions or headers are missing])
                fi
            ],,[
                ${sk_extra_libs}
            ])
        fi

        if test "x${have_silk}" = xyes
        then
            :
        elif test "x${with_libsilk}" != xno -a "x${with_silk_ipset}" = xno
        then
            AC_MSG_NOTICE([not checking for stand-alone SiLK-IPSet library due to --without-silk-ipset])
            if test "x${with_skipset}" = x
            then
                # only ignore the failure to find silk when
                # --with-skipset is not given
                labeling_disabled=yes
            fi
        else
            if test "x${with_libsilk}" != xno
            then
                AC_MSG_NOTICE([checking for stand-alone SiLK-IPSet library])
            fi

            # Look for the standalone ipset library
	    LIBS="${skip_save_LIBS}"

            sk_ip_hdr=silk-ipset/silk-ipset.h
            sk_ip_lib=skipset
            AC_CHECK_LIB([${sk_ip_lib}],[skIPSetLoad],
            [
                have_skipset=yes
                LIBS="-l${sk_ip_lib} ${sk_extra_libs} ${LIBS}"
                AC_CHECK_FUNCS([skAppRegister],,
                [
                    have_skipset=no
                ])
                if test "x${have_skipset}" = xyes
                then
                    AC_CHECK_HEADERS([silk-ipset/silk-ipset.h],,
                    [
                        have_skipset=no
                    ])
                fi
                if test "x${have_skipset}" = xno
		then
                    AC_MSG_NOTICE([found ${sk_ip_lib} library but functions or headers are missing])
                fi
            ],,[
                ${sk_extra_libs}
            ])
        fi

        if test "x${have_silk}" = xyes -o "x${have_skipset}" = xyes
        then
            SKIPSET_LDFLAGS="${SKIPSET_LDFLAGS} -l${sk_ip_lib} ${sk_extra_libs}"

            # Does silk include IPv6 support?
            sk_ip_hdr="<${sk_ip_hdr}>"
            AC_MSG_CHECKING([for skipaddrSetV6])
            AC_LINK_IFELSE([
                AC_LANG_PROGRAM([[
#include ${sk_ip_hdr}
#if HAVE_SILK_SKIPADDR_H
#include <silk/skipaddr.h>
#endif
                    ]],[[
skipaddr_t ip;
char ip6[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
skipaddrSetV6(&ip, ip6);
                    ]])
            ],[
                AC_MSG_RESULT([yes])
                AC_DEFINE([HAVE_SKIPADDRSETV6],[1],
                    [Define to 1 if `skipaddrSetV6' is available.])
            ],[
                AC_MSG_RESULT([no])
                AC_MSG_WARN([SiLK library ${sk_ip_lib} does not have IPv6 support; functionality may be limited])
            ])
        fi

        # Restore cached values
        LDFLAGS="${skip_save_LDFLAGS}"
        LIBS="${skip_save_LIBS}"
        CFLAGS="${skip_save_CFLAGS}"
        CPPFLAGS="${skip_save_CPPFLAGS}"
    fi


    # Report the results

    ENABLE_SKIPSET=0
    ENABLE_SKPREFIXMAP=0
    ENABLE_SKTYPESENSOR=0

    support_type=
    if test "x${have_skipset}" = xyes
    then
        support_type="SiLK-IPSet"
        ENABLE_SKIPSET=1
        silk_features="${silk_features} ipset"
    elif test "x${have_silk}" = xyes
    then
        support_type="SiLK"
        silk_features=
        if test "x${with_silk_ipset}" != xno
        then
            ENABLE_SKIPSET=1
            silk_features="${silk_features} ipset"
        fi
        if test "x${with_silk_pmap}" != xno
        then
            ENABLE_SKPREFIXMAP=1
            silk_features="${silk_features} pmap"
        fi
        if test "x${with_silk_type_sensor}" != xno
        then
            ENABLE_SKTYPESENSOR=1
            silk_features="${silk_features} type-sensor"
        fi
    fi
    # remove leading whitespace
    silk_features=`echo ${silk_features}`

    if test "x${support_type}" != "x"
    then
        AC_MSG_NOTICE([building with ${support_type} support; enabled features: ${silk_features}])
    elif test "x${labeling_disabled}" = "xyes"
    then
        # message printed above
        SKIPSET_LDFLAGS=
        SKIPSET_CFLAGS=
    elif test "x${with_skipset}" = "xno"
    then
        AC_MSG_NOTICE([not building SiLK support at user request])
    elif test "x${with_skipset}" = "x"
    then
        AC_MSG_NOTICE([not building ${search_lib} support due to missing headers or libraries])
        SKIPSET_LDFLAGS=
        SKIPSET_CFLAGS=
    elif test "x${with_skipset}" = "xyes"
    then
        AC_MSG_FAILURE([${search_lib} support requested but unable to find required files])
    else
        AC_MSG_FAILURE([${search_lib} support requested but unable to find required files under ${with_skipset}])
    fi


    # Set output variables, definitions, etc

    if test "x${ENABLE_SKIPSET}" = "x1"
    then
        AC_DEFINE(ENABLE_SKIPSET, [1],
            [Define to 1 if SiLK IPSet Labeling and Filtering is supported])
        AC_DEFINE_UNQUOTED([SKIPSET_HEADER_NAME],[${sk_ip_hdr}],
            [When ENABLE_SKIPSET is set, this is the path to the skipset.h header file])
    fi
    
    if test "x${ENABLE_SKPREFIXMAP}" = "x1"
    then
        AC_DEFINE(ENABLE_SKPREFIXMAP, [1],
            [Define to 1 if SiLK Prefix Map Labeling is supported])
    fi
    
    if test "x${ENABLE_SKTYPESENSOR}" = "x1"
    then
        AC_DEFINE(ENABLE_SKTYPESENSOR, [1],
            [Define to 1 if SiLK Type+Sensor Labeling is supported])
    fi

    AM_CONDITIONAL([BUILD_PROBECONF], [test "x${ENABLE_SKTYPESENSOR}" = "x1"])
])
