/*
** airlock.c
** Airframe lockfile interface
**
** ------------------------------------------------------------------------
** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
** ------------------------------------------------------------------------
** Authors: Brian Trammell <bht@cert.org>
** ------------------------------------------------------------------------
** GNU Lesser GPL Rights pursuant to Version 2.1, February 1999
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
** ------------------------------------------------------------------------
*/

/**
 * @file 
 *
 * Airframe lockfile interface. Used to acquire lockfiles compatible with 
 * the Airframe filedaemon.
 */
 
/* idem hack */
#ifndef _AIR_AIRLOCK_H_
#define _AIR_AIRLOCK_H_

#include <airframe/autoinc.h>

/** GError domain for lock errors */
#define LOCK_ERROR_DOMAIN g_quark_from_string("airframeLockError")
/** 
 * A lock could not be acquired.
 */
#define LOCK_ERROR_LOCK  1

/** 
 * A lock structure. Must be maintained by a caller from lock acquisition
 * to release. Should be initialized by AIR_LOCK_INIT or memset(0) or bzero(). 
 */
typedef struct _AirLock {
    /** Path to .lock file */
    GString     *lpath;
    /** File descriptor of open .lock file */
    int         lfd;
    /** TRUE if this lock is held, and lpath and lfd are valid. */
    gboolean    held;
} AirLock;

/** Convenience initializer for AirLock structures */
#define AIR_LOCK_INIT { NULL, 0, FALSE }

/**
 * Acquire a lock. Creates a lockfile and returns TRUE if the lockfile was
 * created (and is now held). 
 *
 * @param lock AirLock structure to store lockfile information in.
 * @param path path of file to lock (without .lock extension).
 * @param err an error descriptor
 * @return TRUE if lockfile created, FALSE if lock not available
 */
gboolean air_lock_acquire(
    AirLock     *lock,
    const char  *path,
    GError      **err);

/**
 * Release an acquired lock.
 * 
 * @param lock AirLock structure filled in by air_lock_acquire()
 */
void air_lock_release(
    AirLock     *lock);

/**
 * Free storage used by an AirLock structure. Does not free the structure
 * itself.
 *
 * @param lock AirLock to free
 */
void air_lock_cleanup(
    AirLock     *lock);

/* end idem */
#endif
