/*
 ** daeconfig.h
 ** Generic daemon configuration support
 **
 ** ------------------------------------------------------------------------
 ** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
 ** ------------------------------------------------------------------------
 ** Authors: Brian Trammell <bht@cert.org>
 **          Tony Cebzanov <tonyc@cert.org>
 ** ------------------------------------------------------------------------
 ** GNU Lesser GPL Rights pursuant to Version 2.1, February 1999 
 ** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
 ** ------------------------------------------------------------------------
 */

/**
 * @file
 *
 * Airframe Daemon Configuration Support. Supplies automatic daemonization 
 * and the command line option processing necessary to use it. Use this when
 * your application can run as a daemon and you want to give your users control 
 * over whether it does via the command line.
 */

/* idem hack */
#ifndef _AIR_DAECONFIG_H_
#define _AIR_DAECONFIG_H_

#include <airframe/autoinc.h>
#include <airframe/airopt.h>

/** GError domain for daeconfig errors */
#define DAEC_ERROR_DOMAIN g_quark_from_string("airframeDaemonError")
/** 
 * Daeconfig setup error. Signifies that daemonization failed due to an
 * underlying operating system error.
 */
#define DAEC_ERROR_SETUP  1

/**
 * Set up daemon configuration. Call this after parsing an options context 
 * including a GOptionGroup returned from daec_option_group(). This sets
 * up internal state used by the other daeconfig calls and daemonizes the 
 * application, if necessary.
 *
 * @param err an error description
 * @return TRUE on success, FALSE otherwise
 */
 
gboolean daec_setup(
    GError          **err);

/**
 * Add an option group for daemon configuration to the given optoin context.
 * This option group defines two options: --daemon (-d) to become a daemon, and
 * --foreground to run in daemon mode without forking.
 *
 * @param aoctx airframe option context
 * @return TRUE if successful, FALSE otherwise
 */
gboolean daec_add_option_group(AirOptionCtx *aoctx);

/**
 * Return daemon mode state. Returns true if --daemon was passed in on the 
 * command line, regardless of whether --foreground was also present. If an 
 * application's logic is different for daemon and non-daemon mode, the 
 * application should use this call to determine which mode to run in.
 *
 * @return TRUE if in daemon mode, FALSE otherwise. 
 */

gboolean daec_is_daemon();

/**
 * Return future fork state. Returns true if --daemon and not --foreground. Use 
 * this call to determine whether a call to daec_setup() will cause the 
 * application for fork to the background. This is primarily designed for 
 * interoperation with logconfig, which must know whether daeconfig will 
 * fork without requiring said fork to occur before logging is set up.
*
 * @return TRUE if subsequent call to daec_setup() will fork, FALSE otherwise.
 */
 
gboolean daec_will_fork();

/**
 * Return forked state. Returns true if a prior call to daec_setup() caused 
 * the application to fork to the background. 
 *
 * @return TRUE if the daemon has forked, FALSE otherwise
 */
 
gboolean daec_did_fork();

/**
 * Return quit flag state. Returns FALSE until daec_quit() has been called, 
 * then returns TRUE. Provided as a convenience, so applications don't have 
 * to track their own quit flag.
 *
 * @return TRUE if daec_quit() has been called.
 */
 
gboolean daec_did_quit();

/**
 * Set the quit flag.
 */
 
void daec_quit();

/* end idem */
#endif
