/*
** airlock.c
** Airframe lockfile interface
**
** ------------------------------------------------------------------------
** Copyright (C) 2005-2007 Carnegie Mellon University. All Rights Reserved.
** ------------------------------------------------------------------------
** Authors: Brian Trammell <bht@cert.org>
** ------------------------------------------------------------------------
** GNU Lesser GPL Rights pursuant to Version 2.1, February 1999
** Government Purpose License Rights (GPLR) pursuant to DFARS 252.225-7013
** ------------------------------------------------------------------------
*/

#define _AIRFRAME_SOURCE_
#include <airframe/airlock.h>

static char *RCSID __attribute__((unused)) =
    "$Id: airutil.c 4728 2006-08-30 14:41:01Z bht $";

gboolean air_lock_acquire(
    AirLock     *lock,
    const char  *path,
    GError      **err)
{

    /* Lazily create scratch path */
    if (!lock->lpath) {
        lock->lpath = g_string_new("");
    }

    /* Generate lock path */
    g_string_printf(lock->lpath, "%s.lock", path);
    
    /* Open lock file */
    lock->lfd = open(lock->lpath->str, O_WRONLY | O_CREAT | O_EXCL, 0664);
    if (lock->lfd < 0) {
        g_set_error(err, LOCK_ERROR_DOMAIN, LOCK_ERROR_LOCK,
                    "Cannot lock file %s: %s",
                    path, strerror(errno));
        unlink(lock->lpath->str);
        return FALSE;
    }
    
    /* Note lock held */
    lock->held = TRUE;
    
    return TRUE;
}
    
void air_lock_release(
    AirLock     *lock)
{
    /* Lock release is no-op if lock not held */
    if (!lock->held) {
        return;
    }
    
    /* Verify lockfile still exists */
    if (!g_file_test(lock->lpath->str, G_FILE_TEST_IS_REGULAR)) {
        g_warning("Lock collision warning: %s missing", 
                  lock->lpath->str);
    }
 
    /* Close and unlink lockfile */
    close(lock->lfd);
    unlink(lock->lpath->str);
    
    /* clean up the lock */
    lock->held = FALSE;
}

void air_lock_cleanup(
    AirLock     *lock)
{
    if (lock->lpath) {
        g_string_free(lock->lpath, TRUE);
    }
}
